# -*- coding: utf-8 -*- #

import pymel.core as core
import maya.cmds as cmds
import re

import pymel.api as api

import transform
import json

import wizart.anim_utils.network

def getAnimCurveData(animCurve):
    '''
    Get an animation curve (animCurve) data as a list.  
    '''
    animCurve = core.PyNode(animCurve)
    curveType = core.nodeType(animCurve)
    
    data = []
    for i in range(core.keyframe(animCurve, q=True, kc=True)):
        if curveType in ["animCurveTA", "animCurveTL", "animCurveTT", "animCurveTU"]:
            time = core.keyframe(animCurve, index=i, q=True, tc=True) # time based curves
        else:            
            time = core.keyframe(animCurve, index=i, q=True, fc=True) # unit based curves

        value = core.keyframe(animCurve, index=i, q=True, vc=True)
            
        tangentNames = core.keyTangent(animCurve, index=[i], q=True, itt=True, ott=True)
        tangentValues = core.keyTangent(animCurve, index=[i], q=True, ix=True, ox=True, iy=True, oy=True)
        locks = core.keyTangent(animCurve, index=[i], q=True, wt=True, l=True, wl=True)
        
        data.append([time[0], value[0], tangentNames, tangentValues, locks])

    return curveType, animCurve.preInfinity.get(), animCurve.postInfinity.get(), data

def makeAnimCurve(data, name=""):
    '''
    Make an animation curve with data privided.
    '''
    curveType, preinf, postinf, keyData = data
    
    animCurve = core.createNode(curveType, n="%s_%s"%(name, curveType) if name else curveType)
    animCurve.preInfinity.set(preinf)
    animCurve.postInfinity.set(postinf)
    
    for i, frameData in enumerate(keyData):
        time, value, tangentNames, tangentValues, locks = frameData
        itt, ott = tangentNames
        ix, ox, iy, oy = tangentValues
        wt, l, wl = locks
        
        if curveType in ["animCurveTA", "animCurveTL", "animCurveTT", "animCurveTU"]:
            core.setKeyframe(animCurve, t=[time], v=value)
        else:          
            core.setKeyframe(animCurve, f=[time], v=value)
            
        core.keyTangent(animCurve, index=[i], wt=wt)
        core.keyTangent(animCurve, index=[i], wl=wl, l=l)        
        core.keyTangent(animCurve, index=[i], itt=itt, ott=ott)
        core.keyTangent(animCurve, index=[i], ix=ix, ox=ox, iy=iy, oy=oy)

    return animCurve 
    
def setPolevectorPositon(j1, j2, j3, offsetCoeff, polevector):
    start = core.xform(j1, q=1, ws=1, t=1)
    mid = core.xform(j2, q=1, ws=1, t=1)
    end = core.xform(j3, q=1, ws=1, t=1)

    startV = api.MVector(start[0], start[1], start[2])
    midV = api.MVector(mid[0], mid[1], mid[2])
    endV = api.MVector(end[0], end[1], end[2])

    startEnd = endV - startV
    startMid = midV - startV

    dotP = startMid * startEnd
    proj = float(dotP) / float(startEnd.length())
    startEndN = startEnd.normal()
    projV = startEndN * proj

    arrowV = startMid - projV
    arrowV *= offsetCoeff
    finalV = arrowV + midV

    core.xform(polevector, ws=1, t=(finalV.x, finalV.y, finalV.z))
    
def makeScaleCoeffs(joints, driver, stretch, squash):
    '''
    Доводка стречта/сквоша костей.

    ВХОД
    joints (список core.nt.Joint) - входящие кости
    driver (core.Attribute) - атрибут, управляющий стретчем/сквошем
    stretch (список core.Attribute) - список коэффициентов(или атрибутов) на каждую кость при растяжении
    squash (список core.Attribute) - список коэффициентов(или атрибутов) на каждую кость при сжатии
    '''
    if not stretch and not squash:
        return

    for i, j in enumerate(joints):
        sy = j.sy.listConnections(s=True, d=False, scn=True, p=True)
        sz = j.sz.listConnections(s=True, d=False, scn=True, p=True)

        cond = core.createNode("condition", n=j + "_scaleCoeff_condition")
        driver >> cond.firstTerm
        cond.secondTerm.set(1)
        cond.operation.set(2) # >

        if core.objExists(stretch[i]):
            stretch[i] >> cond.colorIfTrueR
            stretch[i] >> cond.colorIfTrueG
        else:
            cond.colorIfTrueR.set(stretch[i])
            cond.colorIfTrueG.set(stretch[i])

        if core.objExists(squash[i]):
            squash[i] >> cond.colorIfFalseR
            squash[i] >> cond.colorIfFalseG
        else:
            cond.colorIfFalseR.set(squash[i])
            cond.colorIfFalseG.set(squash[i])

        if sy or sz:
            b2a = core.createNode("blendTwoAttr", n=j + "_scaleCoeff_yz_blendTwoAttr")
            cond.outColorR >> b2a.attributesBlender
            b2a.i[0].set(1)
            sy[0] >> b2a.i[1]
            if sy: b2a.output >> j.sy
            if sz: b2a.output >> j.sz

def getWeights(skin):
    '''
    Получение весов у skinCluster.

    ВХОД
    skin (core.nt.skinCluster) - опрашиваемый скинКластер

    ВЫХОД
    {"инфлюенс":[вес1,вес2,...]}
    '''

    geom = skin.getOutputGeometry()[0] # получаем выходную геометрию
    geom.name() # без этого почему-то вылазит ошибка

    if isinstance(geom, core.nt.Mesh): # работает для меша
        attr = "vtx"
        count = geom.numVertices()

    elif isinstance(geom, core.nt.NurbsCurve): # для кривой
        attr = "cv"
        count = geom.numCVs()

    elif isinstance(geom, core.nt.NurbsSurface): # для поверхности
        attr = "cv"
        count = geom.numCVsInU() * geom.numCVsInV()

    weights = {}

    for inf in skin.influenceObjects(): # пройтись по всем инфлюенсам
        for comp in range(count): # по всем компонентам
            w = core.skinPercent(skin, geom + "." + attr + "[" + str(comp) + "]", q=True, transform=inf) # считать вес
            if not weights.has_key(inf.name()):
                weights[inf.name()] = [] # подготовить список

            weights[inf.name()].append(w) # сохранить вес

    return json.dumps(weights)

def defaultWeights(skin, dropoff):
    '''
    Установка весов по умолчанию.

    ВХОД
    skin (core.nt.skinCluster) - скинКластер
    dropoff (float) - максимальное расстояние для отсечки весов
    '''
    geom = skin.getOutputGeometry()[0]

    if isinstance(geom, core.nt.Mesh): # для меша
        attr = "vtx"
        count = geom.numVertices()

    elif isinstance(geom, core.nt.NurbsCurve): # для кривой
        attr = "cv"
        count = geom.numCVs()

    elif isinstance(geom, core.nt.NurbsSurface): # для поверхности
        attr = "cv"
        count = geom.numCVsInU() * geom.numCVsInV()

    data = []
    for id in range(count):
        vtx = []
        for inf in skin.influenceObjects():
            dist = core.xform(geom + "." + attr + "[" + str(id) + "]", q=True, ws=True, t=True) - inf.getTranslation("world")
            vtx.append([inf.name(), dist.length()])

        data.append(vtx)

    weights = {}

    for vtx in data:
        sumDist = sum(map(lambda x: x[1], vtx))
        for inf, d in vtx:

            if inf not in weights:
                weights[inf] = []

            w = (sumDist - d) / sumDist if d < dropoff else 0

            weights[inf].append(w)

    setWeights(skin, weights)

def setWeights(skin, weights):
    '''
    Установка весов, сохраненных функцией getWeights.

    ВХОД
    skin (core.nt.skinCluster) - скинКластер
    weights (список) - ассоциативный список, возвращаемый функцией getWeights
    '''
    geom = skin.getOutputGeometry()[0] # получаем выходную геометрию
    geom.name() # без этого почему-то вылазит ошибка

    if isinstance(geom, core.nt.Mesh): # для меша
        attr = "vtx"
        count = geom.numVertices()

    elif isinstance(geom, core.nt.NurbsCurve): # для кривой
        attr = "cv"
        count = geom.numCVs()

    elif isinstance(geom, core.nt.NurbsSurface): # для поверхности
        attr = "cv"
        count = geom.numCVsInU() * geom.numCVsInV()

    if count != len(weights.items()[0][1]): # если не совпадает кол-во компонентов
        core.displayWarning("setWeights: different number of components. Skipped")
        return

    nw = skin.normalizeWeights.get()
    skin.normalizeWeights.set(2) # post

    for inf in weights: # пройтись по сохраненным инфлюенсам
        for comp, w in enumerate(weights[inf]): # и по весам на каждый компонент
            if inf in skin.influenceObjects(): # если инфлюенс найден у скинКластера
                core.skinPercent(skin, geom + "." + attr + "[" + str(comp) + "]", transformValue=[inf, w])
            else:
                core.displayWarning("setWeights: cannot find '" + inf + "' influence in '" + skin + "'. Skipped")

    skin.normalizeWeights.set(nw)

def createPathCurve(name, objs, degree=3, spans=4):
    '''
    Создание пути-кривой.

    ВХОД
    name (String) - базовое имя создаваемых нод
    objs (список core.nt.Transform) - объекты, которые требуется "соединить" кривой
    degree (int) - degree для rebuildCurve
    spans (int) - spans для rebuildCurve

    ВЫХОД
    кривая-путь
    '''
    if not objs:
        return

    poses = [core.PyNode(o).getTranslation("world") for o in objs]
    crv = core.curve(d=1, ep=poses, n=name + "_curve")

    if degree and spans:
        core.rebuildCurve(crv, ch=1, rpo=1, rt=0, end=1, kr=0, kcp=0, kep=1, kt=0, d=degree, s=spans, tol=0.01)

    return crv

def createFollicle(name, surface, uv=(0.5, 0.5)):
    '''
    Создать фоликулу на поверхности.

    ВХОД
    name (String) - базовое имя создаваемых нод
    surface (core.nt.NurbsSurface) - поверхность, к которой нужно прицепить фоликулу
    uv (список из 2-х элементов) - координаты UV фоликулы

    ВЫХОД
    [трансформ_фоликулы, шейп_фоликулы]
    '''
    follicle = core.createNode("follicle", n=name + "_follicleShape1")
    follicleParent = follicle.listRelatives(p=True)[0]
    follicleParent.rename(name + "_follicle")

    follicle.outTranslate >> follicleParent.t
    follicle.outRotate >> follicleParent.r

    try:
        surface.worldSpace >> follicle.inputSurface
    except:
        surface.worldMesh >> follicle.inputMesh

    follicle.parameterU.set(uv[0])
    follicle.parameterV.set(uv[1])

    return [follicleParent, follicle]

def createTransformOnSurface(name, surface, uv=(0.5, 0.5)):
    '''
    Создать трансформ привязанный к поверхности.

    ВХОД
    name (String) - базовое имя создаваемых нод
    surface (core.nt.NurbsSurface) - поверхность, к которой нужно прицепить трансформу
    uv (список из 2-х элементов) - координаты UV трансформы

    ВЫХОД
    [трансформ]
    '''
    surfaceInfo = core.createNode("pointOnSurfaceInfo", n=name + "_pointOnSurfaceInfo")
    vectorProd = core.createNode("vectorProduct", n=name + "_vectorProduct")
    fourByFourMatrx = core.createNode("fourByFourMatrix", n=name + "_fourByFourMatrix")
    decomposeMatrx = core.createNode("decomposeMatrix", n=name + "_decomposeMatrix")

    #set vector product node options
    vectorProd.operation.set(2)
    vectorProd.normalizeOutput.set(1)

    #set pointOnSurfaceInfo uv
    surfaceInfo.parameterU.set(uv[0])
    surfaceInfo.parameterV.set(uv[1])
    surfShape = surface.getShape()
    surfShape.worldSpace[0] >> surfaceInfo.inputSurface

    #product tangentU vs normal
    surfaceInfo.normalizedTangentU >> vectorProd.input1
    surfaceInfo.normalizedNormal >> vectorProd.input2

    #set Rotate Matrix
    surfaceInfo.normalizedTangentUX >> fourByFourMatrx.in00
    surfaceInfo.normalizedTangentUY >> fourByFourMatrx.in01
    surfaceInfo.normalizedTangentUZ >> fourByFourMatrx.in02
    fourByFourMatrx.in03.set(0.0)

    surfaceInfo.normalizedNormalX >> fourByFourMatrx.in10
    surfaceInfo.normalizedNormalY >> fourByFourMatrx.in11
    surfaceInfo.normalizedNormalZ >> fourByFourMatrx.in12
    fourByFourMatrx.in13.set(0.0)

    vectorProd.outputX >> fourByFourMatrx.in20
    vectorProd.outputY >> fourByFourMatrx.in21
    vectorProd.outputZ >> fourByFourMatrx.in22
    fourByFourMatrx.in23.set(0.0)

    fourByFourMatrx.in30.set(0.0)
    fourByFourMatrx.in31.set(0.0)
    fourByFourMatrx.in32.set(0.0)
    fourByFourMatrx.in33.set(1.0)

    #decompose
    fourByFourMatrx.output >> decomposeMatrx.inputMatrix

    #create transform
    transf = core.createNode("transform", n=name + "_transform")
    surfaceInfo.position >> transf.translate
    decomposeMatrx.outputRotate >> transf.rotate

    return transf

def getDistance(transform1, transform2):
    '''
    Получить расстояние между двумя трансформами.
    '''
    p1 = transform1.getTranslation("world")
    p2 = transform2.getTranslation("world")

    return p1.distanceTo(p2)

def createDistance(name, point1, point2, parentTo=None, main_control=None):
    '''
    Создание рига для измерения расстояния.

    ВХОД
    name (String) - базовое имя создаваемых нод
    point1 (core.nt.Transform) - начальная точка
    point2 (core.nt.Transform) - конечная точка
    parentTo (core.nt.Transform) - куда парентить результирующую группу
    main_control (core.nt.Transform) - main_control для компенсации скалирования

    ВЫХОД
    список из 3-х элементов (distanceBetween, трансфорт 1-ой точки, трансформ 2-ой точки)
    '''
    dd = core.createNode("distanceBetween", n=name + "_distanceBetween")

    pnt1 = core.createNode("transform", n=name + "_point_1_transform")
    pnt2 = core.createNode("transform", n=name + "_point_2_transform")

    grp = core.group(pnt1, pnt2, dd, n=name + "_group")
    grp.v.set(False)

    if parentTo: # если parentTo задан, то
        if core.objExists(parentTo):
            core.parent(grp, parentTo) # выполнить необходимые действия, если он существует
        else:
            core.displayWarning("createDistance: '" + parentTo + "' doesn't exist. ParentTo is skipped.") # иначе вывести предупреждение

    core.pointConstraint(point1, pnt1)
    core.pointConstraint(point2, pnt2)

    pnt1.t >> dd.point1
    pnt2.t >> dd.point2

    if main_control: # если mainControl задан, то
        if core.objExists(main_control): # учитывать его скалирование
            main_control.inverseMatrix >> dd.inMatrix1
            main_control.inverseMatrix >> dd.inMatrix2
        else:
            core.displayWarning("createDistance: '" + main_control + "' doesn't exist. mainControl is skipped.") # иначе вывести предупреждение

    transform.lockAttrs(grp, [1, 1, 1], [1, 1, 1], [1, 1, 1], 1)
    transform.lockAttrs(pnt1, [1, 1, 1], [1, 1, 1], [1, 1, 1], 1)
    transform.lockAttrs(pnt2, [1, 1, 1], [1, 1, 1], [1, 1, 1], 1)

    return [dd, pnt1, pnt2]

def makeStretchable(name, objects, attr, stretch=1, squash=1, saveVolume=0, minValue=0, scales=(True, True, True), useTx=False):
    '''
    Создание стретча.

    ВХОД
    name (String) - базовое имя создаваемых нод
    objects (список core.nt.Transform) - стретчируемые кости (или трансформы)
    attr (core.PyNode) - управляющий стретчем атрибут
    stretch (core.PyNode) - атрибут, включающий/отключающий стретч, либо числовое значение
    squash (core.PyNode) - атрибут, управляющий сжатием, либо числовое значение
    saveVolume (core.PyNode) - атрибут, управляющий сохраненим объема, либо числовое значение
    minValue (core.PyNode или float) - атрибут, задающий минимальный дистанс, либо числовое значение
    scales (список из 3-х элементов) - назначенные каналы скейла (True-назначить, False-пропустить)
    useTx (True или False) - использовать ли translateX вместо scaleX

    ВЫХОД
    список из 2-х элементов (финальное скалирование по Х, финальное скалирование по У)

    ПРИМЕРЫ
    makeStretchable("L_arm_ik",
                    [core.PyNode("L_arm_1_joint"),
                    core.PyNode("L_arm_3_joint"),
                    core.PyNode("L_arm_3_joint")],
                    core.PyNode("distanceDimension1").distance
                    ) - стретч без атрибутов управления

    makeStretchable("L_arm_ik",
                    [core.PyNode("L_arm_1_joint"),
                    core.PyNode("L_arm_3_joint"),
                    core.PyNode("L_arm_3_joint")],
                    core.PyNode("distanceDimension1").distance,
                    core.PyNode("L_arm_ik_control").stretch,
                    core.PyNode("L_arm_ik_control").squash,
                    core.PyNode("L_arm_ik_control").saveVolume,
                    10,
                    [True,False,False]
                    ) - управляемый стретч, с точкой срабатывания растягивания от 10 по атрибуту distanceDimension1.distance, работающий только по оси Х
    '''

    mult = core.createNode("multiplyDivide", n=name + "_multiplyDivide")
    mult.attr("operation").set(2) # деление

    attr >> mult.input1X
    attr >> mult.input2Y

    if core.objExists(minValue):
        minValue >> mult.input1Y
        minValue >> mult.input2X
    elif minValue == 0:
        mult.attr("input1Y").set(attr.get()) # использовать значение управляющего атрибута attr
        mult.attr("input2X").set(attr.get())
    else:
        mult.attr("input1Y").set(minValue)
        mult.attr("input2X").set(minValue)

    finalX = mult.outputX # переходим дальше по цепочке. final - итоговый атрибут на стретч
    finalY = mult.outputY

    # stretch
    b2aX = core.createNode("blendTwoAttr", n=name + "_stretch_x_blendTwoAttr")

    if core.objExists(stretch):
        stretch >> b2aX.attributesBlender
    else:
        b2aX.attributesBlender.set(stretch)

    b2aX.i[0].set(1)
    mult.outputX >> b2aX.i[1]

    b2aY = core.createNode("blendTwoAttr", n=name + "_stretch_yz_blendTwoAttr")
    if core.objExists(stretch):
        stretch >> b2aY.attributesBlender
    else:
        b2aY.attributesBlender.set(stretch)

    b2aY.i[0].set(1)
    mult.outputY >> b2aY.i[1]

    finalX = b2aX.output
    finalY = b2aY.output

    # squash
    squashCond = core.createNode("condition", n=name + "_squash_condition")
    squashCond.attr("operation").set(2) # >
    finalX >> squashCond.firstTerm

    if core.objExists(squash):
        setRange = core.createNode("setRange", n=name + "_squash_setRange")
        squash >> setRange.valueX
        setRange.oldMinX.set(0)
        setRange.oldMaxX.set(1)
        setRange.minX.set(1)
        setRange.maxX.set(0.1)

        setRange.outValueX >> squashCond.secondTerm
        setRange.outValueX >> mult.input2Z
        setRange.outValueX >> squashCond.colorIfFalseR
    else: # если задано числовое значение
        squashCond.secondTerm.set(squash)
        mult.input2Z.set(squash)
        squashCond.colorIfFalseR.set(squash)

    finalX >> squashCond.colorIfTrueR
    finalY >> squashCond.colorIfTrueG

    mult.input1Z.set(1)
    mult.outputZ >> squashCond.colorIfFalseG

    finalX = squashCond.outColorR
    finalY = squashCond.outColorG

    # saveVolume
    b2a = core.createNode("blendTwoAttr", n=name + "_saveVolume_blendTwoAttr")
    b2a.i[0].set(1)
    finalY >> b2a.i[1]

    if core.objExists(saveVolume):
        saveVolume >> b2a.attributesBlender
    else:
        b2a.attributesBlender.set(saveVolume)

    finalY = b2a.output

    for obj in objects: # итоговое назначение на каналы
        if scales[0]: # x
            if useTx: # использовать translateX
                obj = core.PyNode(obj)

                t = obj.t.listConnections(p=True, s=True, d=False)
                if t:
                    mult = core.createNode("multiplyDivide", n=obj + "_t_multiplyDivide")
                    t[0] >> mult.input1
                    finalX >> mult.input2X
                    mult.output >> obj.t
                else:
                    tx = obj.tx.listConnections(p=True, s=True, d=False)

                    mult = core.createNode("multDoubleLinear", n=obj + "_tx_multDoubleLinear")

                    if tx:
                        tx[0] >> mult.input1
                    else:
                        mult.input1.set(obj.tx.get())

                    finalX >> mult.input2
                    mult.output >> obj.tx
            else:
                finalX >> core.PyNode(obj).sx

        if scales[1]: # y
            finalY >> core.PyNode(obj).sy

        if scales[2]: # z
            finalY >> core.PyNode(obj).sz

    return [finalX, finalY] # вернуть финальный атрибут для скалирования по Х и по У

def makeHelpLine(name, objects, visibleAttr=None, parent=None):
    '''
    Создание вспомогательной линии между объектами.

    ВХОД
    name (String) - базовое имя создаваемых нод
    objects (список core.nt.Transform) - соединяемые линией объекты
    visibleAttr (core.PyNode) - атрибут, управляющий видимостью линии
    parent (core.nt.Transform) - куда парентить результат (обычно others)
    '''
    poses = [[0, 0, 0] for o in objects] # создать кривую с нужным кол-вом вершин
    crv = core.curve(n=name + "_helpLine_curve", d=1, p=poses)
    crv.template.set(True)

    grp = core.createNode("transform", n=name + "_helpLine_group", p=parent)
    grp | crv

    if visibleAttr: # если задан атрибут visibleA
        if core.objExists(visibleAttr):
            visibleAttr >> grp.v
        else:
            core.displayWarning("makeHelpLine: '" + visibleAttr + "' doesn't exist. Used default value")

    for i, obj in enumerate(objects):
        pnt = core.createNode("transform", n=name + "_helpLine_" + str(i + 1) + "_transform", p=grp)
        pnt.t >> crv.controlPoints[i]
        core.pointConstraint(obj, pnt)

        transform.lockAttrs(pnt, [1, 1, 1], [1, 1, 1], [1, 1, 1], 1)

    transform.lockAttrs(crv, [1, 1, 1], [1, 1, 1], [1, 1, 1], 1)
    transform.lockAttrs(grp, [1, 1, 1], [1, 1, 1], [1, 1, 1], 1)

def _genOffsets(dest, src, rough=0.01):
    '''
    Генерация maintain offset для parent/orient констрейнтов для всех rotateOrder.

    ВХОД
    dest (core.nt.Transform) - принимающий трансформ для констрейнта
    src (core.nt.Transform) - исходный трансформ
    rough (float) - пороговая длина вектора maintain offset. Все, что ниже считается отсутствием изменений.

    ВЫХОД
    None либо атрибут, принимающий различные значения-смещения в зависимости от dest.rotateOrder
    '''
    oldRo = dest.ro.get()
    offsets = [] # вектора maintain offset

    for i in range(6):
        dest.ro.set(i)
        dest.r.set([0, 0, 0])

        pc = core.parentConstraint(src, dest, mo=True)
        v = pc.target[0].targetOffsetRotate.get()
        offsets.append(v)

        core.delete(pc)

    dest.ro.set(oldRo)
    dest.r.set([0, 0, 0])

    if not [x for x in offsets if x.length() > rough]:
        return # Нет смещений

    choice = core.createNode("choice", n=dest + "_rotateOrder_choice")
    dest.ro >> choice.selector

    for i, o in enumerate(offsets):
        attr = "choice_ro" + str(i)
        if not dest.hasAttr(attr):
            dest.addAttr(attr, dt="double3")

        dest.attr(attr).set(o)
        dest.attr(attr) >> choice.input[i]

    outAttr = "choice_output"
    if not dest.hasAttr(outAttr):
        dest.addAttr(outAttr, at="double3")
        dest.addAttr(outAttr + "X", at="double", p=outAttr)
        dest.addAttr(outAttr + "Y", at="double", p=outAttr)
        dest.addAttr(outAttr + "Z", at="double", p=outAttr)

    choice.output >> dest.attr(outAttr)
    return dest.attr(outAttr)

def makeSeamlessSwitching(network, name, control, influence):
    '''
    Создание сетапа бесшовного переключение контрола.

    ВХОД
    network (core.nt.Network) - нетворк для атрибутов
    name (String) - базовое имя атрибутов
    control (core.nt.Transform) - настраиваемый контрол
    influence (core.nt.Transform) - управляющий объект

    ПРИМЕРЫ
    makeSeamlessSwitching(core.PyNode("L_arm_network"),
                        "L_arm_ik",
                        core.PyNode("L_arm_ik_control"),
                        core.PyNode("L_arm_fk_3_joint")) - теперь для переключения кинематики L_arm_ik_control используется L_arm_fk_3_joint
    '''
    switchTransform = control.duplicate(po=True)[0] # объект-переключатель
    switchTransform.rename(control.name() + "_seamless_kinematic")
    if isinstance(switchTransform, core.nt.Joint):
        control.jo >> switchTransform.jo

    transform.lockAttrs(switchTransform, [0, 0, 0], [0, 0, 0], [1, 1, 1], 0)
    switchTransform.v.set(False)

    outAttr = _genOffsets(switchTransform, influence)

    control.rotateOrder >> switchTransform.rotateOrder
    pc = core.parentConstraint(influence, switchTransform, mo=True)

    if outAttr:
        outAttr >> pc.target[0].targetOffsetRotate
    
    network = wizart.anim_utils.network.Network(network, True)
    network.setAttr(name, control + ".message", "connection")
    network.setAttr(name + "_seamless", switchTransform + ".message", "connection")

    transform.lockAttrs(switchTransform, [1, 1, 1], [1, 1, 1], [1, 1, 1], 1)

def makeDynamicParent(controlParent, control):
    '''
    Создание динамического парента.

    ВХОД
    controlParent (core.nt.Transform) - на какой трансформ цеплять dynParent группу?
    control (core.nt.Transform) - контрол, которому делать динамический парент

    Меню по умолчанию
    Dynamic parents
        Add selected
    '''
    parent = controlParent.getParent()
    dynParent = core.createNode("transform", n=control.name() + "_dynParent_transform", p=parent)

    dynParent | controlParent
    transform.lockAttrs(dynParent, [], [], [1, 1, 1], 1)

    control.addAttr("dynParent", at="message")
    dynParent.message >> control.dynParent

def getDeformers(shape, types=None):
    '''
    Get GEO deformers. If TYPES is specified then return nodes of these types only.
    '''
    geoType = cmds.objectType(shape)
    
    if geoType == "mesh":
        attr = shape + ".inMesh"
    elif geoType in ["nurbsCurve", "nurbsSurface"]:
        attr = shape + ".create"
    else:
        cmds.error("Unsupported type: %s"%geoType)
    
    deformers = []
    while True:
        conn = cmds.listConnections(attr, s=True, d=False, p=True)
        if not conn:
            break
        
        plug = conn[0]
        
        node, cattr = plug.split(".")
        nodeType = cmds.objectType(node)        

        if nodeType == "groupParts":
            attr = node +".inputGeometry"
        elif cattr.startswith("outputGeometry") and "input.inputGeometry" in cmds.listAttr(node):
            id = re.search("\\[(\\d+)\\]", cattr).group(1)
            attr = node + ".input[%s].inputGeometry"%id
        else:
            break

        if not types or (types and nodeType in types):        
            deformers.append(node)
        
    return [core.PyNode(d) for d in deformers[::-1]]
