import pymel.core as core
import pymel.api as api
import os
import re
import glob

crvPath = os.path.realpath(os.path.dirname(__file__) + "/curves")

def shapeParent(objects):
    if not objects or len(objects) == 1:
        core.error("Select at least 2 object to make parenting")
        return

    parent = objects[-1]
    for obj in objects[:-1]:
        for a in ["tx", "ty", "tz", "t", "rx", "ry", "rz", "r", "sx", "sy", "sz", "s"]:
            core.setAttr(obj.attr(a), k=True, l=False)

        parent | obj

        objShapes = core.listRelatives(obj, f=True, s=True)
        core.makeIdentity(obj, apply=True, t=1, r=1, s=1, n=0)

        for i, s in enumerate(objShapes):
            core.parent(s, parent, r=True, shape=True)
            s.rename("%sShape%s"%(parent, i+1))

        core.delete(obj)

    return parent

def getCurveData(fileName):
    with open(fileName, "r") as f:
        text = f.read()
        cvs_num = int(re.search("CVs Num:([0-9]+)", text).group(1))
        knots_num = int(re.search("Knots Num:([0-9]+)", text).group(1))
        cvs = re.search("CVs:([^(Knots)]*)", text).group(1)
        knots = re.search("Knots:([^(Degree)]*)", text).group(1)
        degree = int(re.search("Degree:([0-9]+)", text).group(1))
        form = int(re.search("Form:([0-9]+)", text).group(1))

        knots = [float(k) for k in re.split("\n", knots.strip())]

        cvs_data = []
        for cv in re.split("\n", cvs.strip()):
            r = re.search("x:([^ ]+) y:([^ ]+) z:([^ ]+)", cv)
            x, y, z = r.groups()
            x, y, z = float(x), float(y), float(z)
            cvs_data.append([x,y,z])

        cvs = cvs_data

        return {"cvs_num":cvs_num, "knots_num":knots_num, "cvs": cvs, "knots": knots, "degree": degree, "form": form}

def loadCurveFromFile(fileName):
    curveData = getCurveData(fileName)
    return core.curve(p=curveData["cvs"], d=curveData["degree"], k=curveData["knots"], per=False)

def makeCurveByHelper(parent, helper, _=None):
    '''
    Make curve by helper.
    '''
    return shapeParent(helper.duplicate() + [parent]).getShapes()

def makeHelper(name, curveType):
    '''
    Create helper by curveType.
    '''    
    curves = [loadCurveFromFile(f) for f in glob.glob("{dir}/{type}/*.crv".format(dir=crvPath, type=curveType))]
    
    crv = shapeParent(curves) if len(curves) > 1 else curves[0]
    
    color = 17 # yellow by default
    if name.startswith("L_"):
        color = 13 # red
    elif name.startswith("R_"):
        color = 6 # blue
        
    crv.rename(name)        
    for sh in crv.getShapes():     
        sh.overrideEnabled.set(True)
        sh.overrideColor.set(color)
        
    return crv