# -*- coding: utf-8 -*- #

import pymel.core as core
import pymel.api as api

import maya.cmds as cmds

def freezeJoints(joints):
    '''
    Фриз ротейтов костей.
    
    ВХОД
    joints (список из core.nt.Joint) - список костей
    '''
    for j in joints:
        r = j.getRotation().asMatrix()
        jo = j.getOrientation().asMatrix()

        ro = r * jo
        tr = api.MTransformationMatrix(ro)
        quat = tr.eulerRotation().asQuaternion()

        j.setRotation([0, 0, 0])
        j.setOrientation(quat)

def lockAttr(attr, val):
    '''
    Залочить/скрыть атрибут в channelBox.
    
    ВХОД
    attr (core.PyNode) - атрибут для локинга
    val (0,0.5,1) - значение
    '''
    if val == 1: # убрать полностью (залочить, скрыть и сделать неанимируемым)
        attr.setKeyable(False)
        attr.setLocked(True)
        attr.showInChannelBox(False)

    if val == 0.5: # сделать неанимируемым, разлочить и показать
        attr.setKeyable(False)
        attr.setLocked(False)
        attr.showInChannelBox(True)

    if val == 0: # сделать анимируемым и разлочить
        attr.setLocked(False)
        attr.showInChannelBox(True)
        attr.setKeyable(True)

def lockAttrs(object, t, r, s, v):
    '''
    Разлочить/скрыть атрибуты трансформации у transform объекта.
    
    ВХОД
    object (core.nt.Transform) - выбранный объект
    t (список из 3-х элементов) - значения локинга для каждого из каналов translate
    r (список из 3-х элементов) - ... rotate
    s (список из 3-х элементов) - ... scale
    v (0,0.5,1) - значение локинга для канала visibility
    '''
    if not core.objExists(object):
        return

    if not (isinstance(object, core.nt.Transform) or isinstance(object, core.nt.Joint)):
        return # пропустить все ноды, кроме трансформов и суставов

    translates = ["tx", "ty", "tz"]
    rotates = ["rx", "ry", "rz"]
    scales = ["sx", "sy", "sz"]
    visibility = ["v"]

    listAttrs = ([translates, t], [rotates, r], [scales, s], [visibility, [v]]) # соответствующие атрибуты и значения

    for attrs, vals in listAttrs: # пройтись по атрибутам и значениям
        for attr, val in zip(attrs, vals):
            lockAttr(object.attr(attr), val)

def setPivot(source, destination):
    '''
    Установка пивота в нужное место.
    
    ВХОД
    source (core.nt.Transform) - отсюда берем пивот (пивот берется из мирового траснлейта объекта)
    destination (core.nt.Transform) - копируем его сюда (меняем как rotatePivot так и scalePivot)
    '''
    t = source.getTranslation("world")
    cmds.move(t[0], t[1], t[2], [destination + ".scalePivot", destination + ".rotatePivot"], a=True)

def matchJoint(sourceJoint):
    """
    Создать joint, совпадающий матрицей с исходным.
    
    ВХОД
    sourceJoint (core.nt.Joint) - исходный сустав
    
    ВЫХОД
    core.nt.Joint - совпадающий с исходным новый сустав
    """

    j = core.createNode("joint")
    core.delete(core.parentConstraint(sourceJoint, j))
    j.preferredAngle.set(sourceJoint.preferredAngle.get())
    freezeJoints([j])

    return j

def matchTransform(transform, srcTransform, translate=True, rotate=True, translateOffset=(0, 0, 0), rotateOffset=(0, 0, 0)):
    """
    Приравнять трансформы.
    
    ВХОД
    transform (core.nt.Transform) - принимающий трансформ
    srcTransform (core.nt.Transform) - исходный трансформ
    translate (Bool) - учитывать перемещение?
    rotate (Bool) - учитывать вращение?
    translateOffset (список [0,0,0]) - смещение по транслейту относительно srcTransform
    rotateOffset (список [0,0,0]) - смещение по ротейту относительно srcTransform    
    """
    st = [] if translate else ["x", "y", "z"]
    sr = [] if rotate else ["x", "y", "z"]

    if translate or rotate:
        pc = core.parentConstraint(srcTransform, transform, st=st, sr=sr)

        pc.target[0].targetOffsetTranslate.set(translateOffset)
        pc.target[0].targetOffsetRotate.set(rotateOffset)
        core.delete(pc)
    else:
        transform.t.set(transform.t.get() + translateOffset)
        transform.r.set(transform.r.get() + rotateOffset)

def setTranslateRelative(tfm, parent, offset=(0, 0, 0)):
    """
    Установка транслейта в специфических родительских координатах
    
    ВХОД
    tfm (core.nt.Transform) - на этот трансформ устанавливаем координаты
    parent (core.nt.Transform) - родитель
    offset (список из 3-х элементов) - смещение
    """
    point = core.dt.Point(offset[0], offset[1], offset[2])
    point = point * parent.__apimdagpath__().inclusiveMatrix()

    tfm.setTranslation(point)
