import pymel.core as core
import json

def setup_shapes_visibility():
    cache_set = core.PyNode("cache_set")
    geometry = core.PyNode('geometry')
    members = cache_set.members()
    for member in members:
        if not core.listRelatives(member, noIntermediate = True, shapes = True):
            continue

        shapes = core.listRelatives(member, noIntermediate = True, shapes = True)
        shape = shapes[0] if shapes else None
        if shape:
            if not shape.type() in ['mesh','nurbsSurface']:
                 core.displayError('not supported shape type %s' % member.name() )
                 continue
            #force visibility ON on original if we its not connected but we hide on another handy reason
            if not member.v.inputs():
                member.v.set(1)
            #visibility constraint
            parents = member.getAllParents()
            #slice by geometry group we dont want to nodes above geometry group(including it)
            # to control visibility of new node
            if geometry in parents:
                index = parents.index(geometry)
                parents = parents[:index]

            if len(parents) > 0:
                vis_list = parents
                vis_list_names = [obj.v.name() for obj in vis_list]
                if member.visibility.inputs():
                    input_attr = member.visibility.inputs(plugs = True)[0]
                    vis_list_names.insert(0, input_attr.name())
                    member.visibility.disconnect()
                expr = " && ".join(vis_list_names)
                exprFull = "{out} = {expr};".format(out = member.visibility.name(), expr = expr)
                core.expression(s = exprFull)

def __get_cache_set_names_from_file(cache_set_file):
    with open(cache_set_file, 'r') as f:
		data = json.loads(f.read())
    if data:
        result = data.get('cache_set', [])
        if len(result) < 1:
            core.error("failed retrieving cache_set from file '%s'" % cache_set_file)
        return result

def __get_cache_set_nodes_from_file( cache_set_file, namespace = None):
    names = __get_cache_set_names_from_file(cache_set_file)
    result = []
    for name in names:
        namespaced_name = namespace + ":" + name if namespace else name
        if not core.objExists(namespaced_name):
            core.error("'%s' do not exists in maya scene, failed retrieving cache_set from file '%s' " % (namespaced_name, cache_set_file))
        result.append( core.PyNode(namespaced_name))
    return result

def create_cachable_geometry(namespace = None, no_compensation_nodes = None, cache_set_file = None):
    """
    Method to make geometry "cacheble" for alembic caching
    we create new "cache_geometry" group that must have
    same geometry names and flat hierarchy (all identical to model source),

    at input we have "cache_set" object set with rigged geometry, we process that
    and conenct geometry shapes to new location and trying to keep all transforms.
    At output we generate "cache_set" swapped with new location (pointing to new location) geometry,
    at "cache_geometry" group.
    Note that cache_geometry group and parent to scale group i.e. transformed by main_control,
    but source "geometry" group are not. It is desirable to make more optimized abc caches (with less numbers on P primvar but on xform).
    So to keep correct transforming of geometry i.e. WYSWIG then you must add to AbcExport -worldSpace flag.

    We also swapping mesh shape nodes to keep all installed attributes for AbcExport

    """
    # if namespace:
    #     orig_namespace = core.namespaceInfo(currentNamespace = True)
    #     core.namespace(set = ':' + namespace)
    if no_compensation_nodes is None:
        no_compensation_nodes = []
    if cache_set_file:
        members = __get_cache_set_nodes_from_file(cache_set_file, namespace = namespace)
        cache_set = core.sets(name = namespace + ":cache_set" if namespace else "cache_set", empty = True)
        for member in members:
            core.sets(cache_set, edit = True, addElement = member)
    else:
        cache_set = core.PyNode(namespace + ":cache_set" if namespace else "cache_set")
        members = cache_set.members()
    group = core.createNode("transform", n = "cache_geometry")
    geometry = core.PyNode(namespace + ':geometry' if namespace else 'geometry')
    core.PyNode(namespace + ":scale" if namespace else 'scale') | group

    for member in members:
        #skip any nodes without shapes
        if not core.listRelatives(member, noIntermediate = True, shapes = True):
            continue
        new_tfm = core.createNode("transform", n = member.nodeName(), p = group)

        #force visibility ON on original if we its not connected but we hide on another handy reason
        if not member.v.inputs():
            member.v.set(1)

        #visibility constraint
        parents = member.getAllParents()
        #slice by geometry group we dont want to nodes above geometry group(including it)
        # to control visibility of new node
        if geometry in parents:
            index = parents.index(geometry)
            parents = parents[:index]
        vis_list = parents + [member]
        vis_list_names = [obj.v.name() for obj in vis_list]
        if member.visibility.inputs():
            input_attr = member.visibility.inputs(plugs = True)[0]
            vis_list_names.insert(0, input_attr.name())
            member.visibility.disconnect()
        expr = " && ".join(vis_list_names)
        exprFull = "{out} = {expr};".format(out = new_tfm.v.name(), expr = expr)
        core.expression(s = exprFull)


        #swap transforms in set
        core.sets(cache_set, edit = True, remove = member)
        core.sets(cache_set, edit = True, addElement = new_tfm)

        shapes = core.listRelatives(member, noIntermediate = True, shapes = True)
        shape = shapes[0] if shapes else None
        if shape:
            if not shape.type() in ['mesh','nurbsSurface']:
                 core.displayError('not supported shape type %s' % member.name() )
                 continue
            inputGeometryAttrName = "inMesh" if shape.type() == 'mesh' else "create"
            outputGeometryAttrName = "outMesh" if shape.type() == 'mesh' else "local"
            have_skin = core.mel.eval('findRelatedSkinCluster %s' % member.name())
            #swap shapes
            core.parent(shape, new_tfm, shape = True, relative = True)
            new_shape = core.createNode(shape.type(), p = member, n = member.nodeName() + 'Shape')
            tfm_connection_mask = [member.attr(attr_name).isConnected() for attr_name in ('tx', 'ty', 'tz', 'rx', 'ry', 'rz', 'sx', 'sy', 'sz')]
            if member in no_compensation_nodes:
                """
                Special case where we assume node already have correct transformation against scale group,
                so we just swap shape node, copy transformation without constraining
                """
                shape.attr(outputGeometryAttrName) >> new_shape.attr(inputGeometryAttrName)


                new_tfm.translate.set(member.translate.get())
                new_tfm.rotate.set(member.rotate.get())
                new_tfm.scale.set(member.scale.get())
                #NOTE: parentConstraint do not copy any of transform pivots even with maintainOffset == False
                #so we need to copy these beforehand, or else you cannot get always equal transformations
                new_tfm.rotatePivot.set(member.rotatePivot.get())
                new_tfm.rotatePivotTranslate.set(member.rotatePivotTranslate.get())
                new_tfm.scalePivot.set(member.scalePivot.get())
                new_tfm.scalePivotTranslate.set(member.scalePivotTranslate.get())
                new_tfm.transMinusRotatePivot.set(member.transMinusRotatePivot.get())

            elif have_skin and any(tfm_connection_mask) is False:
                tfm_geometry = core.createNode("transformGeometry")
                shape.attr(inputGeometryAttrName).inputs(plugs = True)[0] >> tfm_geometry.inputGeometry
                mult_matrix = core.createNode("multMatrix")

                # compensate for geomMatrix that skinned geometry may have
                # so we first need go to world
                # 'substract' cache_geometry matrix (parentInverseMatrix)
                # then go back
                new_tfm.matrix >> mult_matrix.matrixIn[0]
                new_tfm.parentInverseMatrix[0] >> mult_matrix.matrixIn[1]
                new_tfm.inverseMatrix >> mult_matrix.matrixIn[2]

                mult_matrix.matrixSum >> tfm_geometry.transform
                tfm_geometry.outputGeometry >> shape.attr(inputGeometryAttrName)
                shape.attr(inputGeometryAttrName).inputs(plugs = True)[0] >> new_shape.attr(inputGeometryAttrName)
                #copy transformation to new transform
                new_tfm.translate.set(member.translate.get())
                new_tfm.rotate.set(member.rotate.get())
                new_tfm.scale.set(member.scale.get())
            else:
                shape.attr(outputGeometryAttrName) >> new_shape.attr(inputGeometryAttrName)


                new_tfm.translate.set(member.translate.get())
                new_tfm.rotate.set(member.rotate.get())
                new_tfm.scale.set(member.scale.get())
                #NOTE: parentConstraint do not copy any of transform pivots even with maintainOffset == False
                #so we need to copy these beforehand, or else you cannot get always equal transformations
                new_tfm.rotatePivot.set(member.rotatePivot.get())
                new_tfm.rotatePivotTranslate.set(member.rotatePivotTranslate.get())
                new_tfm.scalePivot.set(member.scalePivot.get())
                new_tfm.scalePivotTranslate.set(member.scalePivotTranslate.get())
                new_tfm.transMinusRotatePivot.set(member.transMinusRotatePivot.get())

                crt = core.parentConstraint(member, new_tfm)
                crt.visibility.set(0)
                crt = core.scaleConstraint(member, new_tfm)
                crt.visibility.set(0)
            #transfer materials
            core.mel.eval("deformerAfterObjectSetMod %s %s" % ( shape.name(), new_shape.name())  )
        else:
            core.error('not found shape on %s' % member.name() )

    #hide source geometry
    core.PyNode(namespace + ":geometry" if namespace else geometry).visibility.set(0)
    # if namespace:
    #     core.namespace(set = orig_namespace)
