#
# Copyright 2017 Pixar
#
# Licensed under the Apache License, Version 2.0 (the "Apache License")
# with the following modification; you may not use this file except in
# compliance with the Apache License and the following modification to it:
# Section 6. Trademarks. is deleted and replaced with:
#
# 6. Trademarks. This License does not grant permission to use the trade
#    names, trademarks, service marks, or product names of the Licensor
#    and its affiliates, except as required to comply with Section 4(c) of
#    the License and to reproduce the content of the NOTICE file.
#
# You may obtain a copy of the Apache License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the Apache License with the above modification is
# distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
# KIND, either express or implied. See the Apache License for the specific
# language governing permissions and limitations under the Apache License.
#

from __future__ import absolute_import

from ._Qt import QtCore, QtWidgets, QtGui
from pxr import Sdf, Tf, Usd

if False:
    from typing import *

from dcc.code_editor.code_editor import CodeEditor
import wizart.dcc.core as dcc_core


class LayerTextEditor(QtWidgets.QWidget):
    """A basic text widget for viewing/editing the contents of a layer."""
    # Emitted when the layer is saved by this editor.
    layerSaved = QtCore.Signal(Sdf.Layer)

    def __init__(self, layer, readOnly=False, parent=None):
        # type: (Sdf.Layer, bool, Optional[QtWidgets.QWidget]) -> None
        """
        Parameters
        ----------
        layer : Sdf.Layer
        readOnly : bool
        parent : Optional[QtWidgets.QWidget]
        """
        super(LayerTextEditor, self).__init__(parent=parent)

        self._layer = layer
        self.readOnly = readOnly

        self.textArea = CodeEditor(self, lang='usd', highlight_line=True, highlight_selected=False, line_numbers=True)
        
        layout = QtWidgets.QVBoxLayout(self)
        layout.setContentsMargins(0, 0, 0, 0)
        layout.setSpacing(0)

        toolbar = QtWidgets.QToolBar(self)
        toolbar.setIconSize(QtCore.QSize(20, 20))
        toolbar.addAction(QtGui.QIcon(':/icons/refresh'), 'Reload', self.Refresh)

        layout.addWidget(toolbar)

        if not readOnly:
            editableCheck_action = QtWidgets.QAction(QtGui.QIcon(':/icons/unlock'), 'Unlock for Editing', self)
            editableCheck_action.setCheckable(True)
            editableCheck_action.setChecked(False)
            editableCheck_action.toggled.connect(self.SetEditable)
            toolbar.addAction(editableCheck_action)
            self.save_action = QtWidgets.QAction(QtGui.QIcon(':/icons/save'), 'Save', self)
            self.save_action.triggered.connect(self.Save)
            toolbar.addAction(self.save_action)

        layout.addWidget(self.textArea)

        self.SetEditable(False)
        self.Refresh()

    def SetEditable(self, editable):
        # type: (bool) -> None
        """
        Parameters
        ----------
        editable : bool
        """
        if editable:
            if self.readOnly:
                return
            self.textArea.code_editor.setUndoRedoEnabled(True)
            self.textArea.code_editor.setReadOnly(False)
            self.save_action.setEnabled(True)
        else:
            self.textArea.code_editor.setUndoRedoEnabled(False)
            self.textArea.code_editor.setReadOnly(True)
            if not self.readOnly:
                self.save_action.setEnabled(False)

    def Refresh(self):
        self.textArea.code_editor.setPlainText(self._layer.ExportToString())

    def Save(self):
        if self.readOnly:
            raise RuntimeError('Cannot save layer when readOnly is set')
        try:
            success = self._layer.ImportFromString(self.textArea.code_editor.toPlainText())
        except Tf.ErrorException as e:
            if Usd.GetVersion() >= (0, 22, 5):
                QtWidgets.QMessageBox.warning(self, 'Layer Syntax Error',
                                              'Failed to apply modified layer '
                                              'contents:\n\n{0}'.format(str(e)))
            else:
                QtWidgets.QMessageBox.warning(self, 'Layer Syntax Error',
                              'Failed to apply modified layer '
                              'contents:\n\n{0}'.format(e.message))
        else:
            if success:
                self.layerSaved.emit(self._layer)
                self.Refresh()  # To standardize formatting


class LayerTextEditorDialog(QtWidgets.QDialog):
    """Dialog version of LayerTextEditor that enables easy sharing of instances.
    """
    _sharedInstances = {}
    def __init__(self, layer, readOnly=False, parent=None):
        # type: (Sdf.Layer, bool, Optional[QtWidgets.QWidget]) -> None
        """
        Parameters
        ----------
        layer : Sdf.Layer
        readOnly : bool
        parent : Optional[QtWidgets.QWidget]
        """
        super(LayerTextEditorDialog, self).__init__(parent=parent)

        app = dcc_core.Application.instance()
        self.settings = app.get_ui_settings()
        self.geometry_key = "LayerTextEditorDialog_geometry"

        layout = QtWidgets.QVBoxLayout(self)
        layout.setContentsMargins(0, 0, 0, 0)
        self.editor = LayerTextEditor(layer, readOnly=readOnly, parent=self)
        layout.addWidget(self.editor)

        self.setWindowTitle('Layer: %s' % layer.identifier)
        self.resize(800, 600)

    def save_geometry(self):
        self.settings.setValue(self.geometry_key, self.saveGeometry())

    def load_geometry(self):
        self.restoreGeometry(
            self.settings.value(self.geometry_key, self.saveGeometry()))

    def showEvent(self, event):
        self.load_geometry()
        super(LayerTextEditorDialog, self).showEvent(event)

    def closeEvent(self, event):
        self.save_geometry()
        super(LayerTextEditorDialog, self).closeEvent(event)

    @classmethod
    def _OnSharedInstanceFinished(cls, layer):
        dialog = cls._sharedInstances.pop(layer, None)
        if dialog:
            dialog.deleteLater()

    @classmethod
    def GetSharedInstance(cls, layer, readOnly=False, parent=None):
        # type: (Sdf.Layer, bool, Optional[QtWidgets.QWidget]) -> LayerTextEditorDialog
        """Convenience method to get or create a shared editor dialog instance.

        Parameters
        ----------
        layer : Sdf.Layer
        readOnly : bool
        parent : Optional[QtWidgets.QWidget]

        Returns
        -------
        LayerTextEditorDialog
        """
        dialog = cls._sharedInstances.get(layer)
        if dialog is None:
            dialog = cls(layer, readOnly=readOnly, parent=parent)
            cls._sharedInstances[layer] = dialog
            dialog.finished.connect(
                lambda result: cls._OnSharedInstanceFinished(layer))
        return dialog


if __name__ == "__main__":
    import sys

    stage = Usd.Stage.Open(
        '../usdQt/testenv/testUsdQtOpinionModel/simple.usda')
    layer = stage.GetLayerStack(includeSessionLayers=True)[1]

    app = QtWidgets.QApplication(sys.argv)

    e = LayerTextEditorDialog(layer, readOnly=True)
    e.show()
    sys.exit(app.exec_())
