import os
import time
import six
from six.moves.urllib import parse as urlparse

from ...i18n import i18n

from pxr.UsdQtEditors._Qt import QtCore, QtGui, QtWidgets
from pxr import Usd, Sdf, Tf, Ar

AR_VERSION = 1
if Usd.GetMinorVersion() >= 22: #assume that with USD 22.05 we build with AR_VERSION=2
    AR_VERSION = 2


NULL_INDEX = QtCore.QModelIndex()
FONT_BOLD = QtGui.QFont()
FONT_BOLD.setBold(True)

LayerMatchRole = QtCore.Qt.UserRole+1

from six.moves import range

class LayerItem:
    def __init__(self, layer, parent_item=None, is_ref=False):
        self.layer = layer
        self.children = []
        self.parent = parent_item
        self.row = 0
        self.is_ref = is_ref
        self.update_date = None
        self.update_user = None
        if parent_item:
            self.row = len(self.parent.children)

            self.parent.add_child(self)

    def add_child(self, child):
        if not child.layer.identifier in [c.layer.identifier for c in self.children]:
            self.children.append(child)


    def remove_children(self):
        """remove children and children of children explicitly"""
        for child in self.children:
            child.remove_children()
        self.children = []


class LayerStackModel(QtCore.QAbstractItemModel):
    """Layer stack model for the outliner's edit target selection dialog."""
    Name = i18n("layer_widget", "Name")
    Path = i18n("layer_widget", "Path")
    Resolved_path = i18n("layer_widget", "Resolved Path")
    Muted = i18n("layer_widget", "Muted")
    Dirty = i18n("layer_widget", "Dirty")
    Version = i18n("layer_widget", "Version")
    UpdateDate = i18n("layer_widget", "Update Date")
    UpdateUser = i18n("layer_widget", "Update User")

    headerLabels = ["Name", "Path", "Resolved Path", "Muted", "Dirty", "Version", "Update Date", "Update User"]
    headerLabelNames = [Name, Path, Resolved_path, Muted, Dirty, Version, UpdateDate, UpdateUser]
   
    EditTargetParentsColor = QtGui.QBrush(QtGui.QColor(150, 200, 250, 40))
    CurrentEditTargetColor = QtGui.QBrush(QtGui.QColor(150, 200, 250, 70))

    def __init__(self, stage=None, includeSessionLayers=True, parent=None):
        # type: (Usd.Stage, bool, Optional[QtCore.QObject]) -> None
        """
        Parameters
        ----------
        stage : Usd.Stage
        includeSessionLayers : bool
        parent : Optional[QtCore.QObject]
        """
        QtCore.QAbstractItemModel.__init__(self, parent=parent)
        self._stage = stage
        self._includeSessionLayers = includeSessionLayers

        self.set_icons()
        self.root_items = []
        self.edit_target_parents = []
        self.ResetStage(stage)
        self._layer_list = []

    def set_icons(self):
        self.muted_icon = QtGui.QPixmap(":/layer_icons/muted")
        self.unmuted_icon = QtGui.QPixmap(":/layer_icons/unmuted")
        self.checkmark_icon = QtGui.QPixmap(":/icons/checkmark_white")
        self.layers_icons = {'sublayer': QtGui.QPixmap(":/icons/layers"),
                             'reference': QtGui.QPixmap(":/icons/create_reference")}
        self.in_memory_title = '_in_memory'
        add_pixmap = QtGui.QPixmap(":/layer_icons/in_memory_layer")
        # for each icon in list add another icon with flag on it
        for title, pixmap in list(self.layers_icons.items()):
            new_title = title + self.in_memory_title
            new_pixmap = QtGui.QPixmap(pixmap)
            painter = QtGui.QPainter(new_pixmap)
            painter.drawPixmap(0, 0, 10, 10, add_pixmap)
            self.layers_icons[new_title] = new_pixmap

    def columnCount(self, parentIndex):
        return len(self.headerLabels)

    def data(self, modelIndex, role=QtCore.Qt.DisplayRole):
        if not modelIndex.isValid():
            return
        column = modelIndex.column()
        item = modelIndex.internalPointer()

        if role == QtCore.Qt.DisplayRole:
            if column == self.headerLabels.index(self.Name):
                return item.layer.identifier.split('/')[-1]
            elif column == self.headerLabels.index(self.Path):
                return item.layer.identifier
            elif column == self.headerLabels.index(self.Resolved_path):
                return item.layer.realPath
            elif column == self.headerLabels.index(self.Version):
                return item.layer.version
            elif column == self.headerLabels.index(self.UpdateDate):
                if item.update_date:
                    return item.update_date
                real_path = item.layer.realPath
                if os.path.exists(real_path):
                    edit_time = time.strftime("%d.%m.%y %H:%M", time.gmtime(os.path.getmtime(real_path)))
                    item.update_date = edit_time
                    return item.update_date
            elif column == self.headerLabels.index(self.UpdateUser):
                if not item.update_user:
                    item.update_user = self.get_update_user_for_layer(item.layer)
                return item.update_user

        elif role == QtCore.Qt.FontRole:
            if item.layer == self._stage.GetEditTarget().GetLayer():
                return FONT_BOLD

        elif role == QtCore.Qt.DecorationRole:
            if column == self.headerLabels.index(self.Name):
                if item.layer.anonymous:
                    sub = self.in_memory_title
                else:
                    sub = ''
                if item.is_ref:
                    return self.layers_icons['reference' + sub]
                return self.layers_icons['sublayer' + sub]
            elif column == self.headerLabels.index(self.Muted):
                if self._stage.GetRootLayer().identifier == item.layer.identifier:
                    return None
                if self._stage.IsLayerMuted(item.layer.identifier):
                    return self.muted_icon
                else:
                    return self.unmuted_icon
            elif column == self.headerLabels.index(self.Dirty):
                if item.layer.dirty or item.layer.anonymous:
                    return self.checkmark_icon

        elif role == QtCore.Qt.BackgroundRole:
            if item.layer == self._stage.GetEditTarget().GetLayer():
                return self.CurrentEditTargetColor
            if item in self.edit_target_parents:
                return self.EditTargetParentsColor

        elif role == LayerMatchRole:
            return item.layer

    def get_layer(self, identifier, parent):
        layer = Sdf.Layer.FindOrOpen(identifier)
        if layer:
            return layer
        resolver = Ar.GetResolver()

        if AR_VERSION == 2:
            if parent:
                path = resolver.CreateIdentifier(identifier, Ar.ResolvedPath(parent.layer.identifier))
                layer = Sdf.Layer.FindOrOpen(path)
                if layer:
                    return layer
        else:
            if resolver.IsRelativePath(identifier) and parent:
                path = resolver.AnchorRelativePath(parent.layer.identifier, identifier)
                layer = Sdf.Layer.FindOrOpen(path)
                if layer:
                    return layer

    def ResetStage(self, stage=None):
        # type: (Usd.Stage) -> None
        """Reset the model from a new stage.

        Parameters
        ----------
        stage : Usd.Stage
        """
        self.beginResetModel()
        self.Invalidate()
        self._layer_list = []

        if stage:
            def addLayer(identifier, parent_id=0, parent=None, is_ref=False):
                self._layer_list.append(identifier)
                # we need to reopen layer,
                # because if we mute one of stage layers it doesn't stay in layer stack and get destroyed
                layer = self.get_layer(identifier, parent)

                if layer:
                    layerItem = LayerItem(layer, parent_item=parent, is_ref=is_ref)
                    children = list(dict(path=path, is_ref=False) for path in list(layer.subLayerPaths))

                    child_refs = list(dict(path=ref, is_ref=True) for ref in layer.externalReferences
                                if ref and ref not in layer.subLayerPaths)
                    child_refs.sort(key=lambda ref: (not Sdf.Layer.IsAnonymousLayerIdentifier(ref["path"]), ref["path"]))
                
                    children.extend(child_refs)
                    if layer == self._stage.GetEditTarget().GetLayer():
                        self.update_edit_target_parents(layerItem)
                    for child in children:
                        if not child["path"] in self._layer_list:
                            addLayer(child["path"], parent=layerItem, is_ref=child.get("is_ref"))

                    return layerItem

            def addLayerTree(layerTree, parent=None):
                item = addLayer(layerTree.layer.identifier, parent)
                if not parent:
                    self.root_items.append(item)
                for childTree in layerTree.childTrees:
                    addLayerTree(childTree, parent=item)

            root = stage.GetPseudoRoot()
            if root:
                self._stage = stage
                if self._includeSessionLayers:
                    sessionLayer = stage.GetSessionLayer()
                    if sessionLayer:
                        sessionLayerItem = LayerItem(sessionLayer)
                        self.root_items.append(sessionLayerItem)
                        for path in sessionLayer.subLayerPaths:
                            new_item = LayerItem(Sdf.Layer.FindOrOpen(path), sessionLayerItem)
                layerTree = root.GetPrimIndex().rootNode.layerStack.layerTree
                addLayerTree(layerTree)
        self.endResetModel()

    def update_edit_target_parents(self, item=None):
        if not item:
            index = self.find_item_for_current_edit_target()
            if index:
                item = index.internalPointer()
        while item and item.parent:
            self.edit_target_parents.append(item.parent)
            item = item.parent

    def find_item_for_current_edit_target(self):
        def check_index(index):
            if index.isValid() \
                    and index.internalPointer() \
                    and index.internalPointer().layer == self._stage.GetEditTarget().GetLayer():
                return index
            else:
                rows_num = self.rowCount(index)
                for row in range(rows_num):
                    res = check_index(self.index(row, 0, index))
                    if res:
                        return res

        return check_index(QtCore.QModelIndex())

    def Invalidate(self):
        self._stage = None
        # remove items more explicitly to prevent memory leak
        for item in self.root_items:
            item.remove_children()
        self.root_items = []
        self.edit_target_parents = []

    def index(self, row, column, parent=NULL_INDEX):
        if not parent.isValid():
            return self.createIndex(row, column, self.root_items[row])
        else:
            if row < self.rowCount(parent):
                return self.createIndex(row, column,
                                        parent.internalPointer().children[row])
        return NULL_INDEX

    def rowCount(self, parent=QtCore.QModelIndex()):
        if not parent.isValid():
            return len(self.root_items)
        internalPointer = parent.internalPointer()
        return len(internalPointer.children)

    def parent(self, index):
        if not index.isValid():
            return NULL_INDEX
        internalPointer = index.internalPointer()
        if internalPointer.parent:
            return self.createIndex(internalPointer.parent.row, 0,
                                        internalPointer.parent)
        else:
            return NULL_INDEX

    def isModelValid(self):
        return True if self._stage and self.root_items else False

    def headerData(self, section, orientation, role=QtCore.Qt.DisplayRole):
        if orientation == QtCore.Qt.Horizontal and role == QtCore.Qt.DisplayRole:
            return self.headerLabelNames[section]

    def get_update_user_for_layer(self, layer):
        try:
            import wref_resolver
            from wizart.desktop import auth_session
        except ImportError:
            return
        # get current version
        current_version = self._parse_version(layer.version)

        edited_path = urlparse.urlparse(layer.identifier)._replace(query='').geturl()
        component_path = wref_resolver.resolve(edited_path)
        if not os.path.isdir(component_path):
            return

        # get versions list
        assetvcs_session = auth_session.get_assetvcs_session()
        if not assetvcs_session or not assetvcs_session.is_asset(assetvcs_session, component_path):
            return
        asset = assetvcs_session.get_asset(component_path)
        try:
            version = asset.get_version(current_version)
        except:
            return
        if version:
            version.update()
            user = version._json_data.get('update_user')
            return user

    def _parse_version(self, str_version):
        import re

        match = re.search('\D*(\d+)', str_version)
        if match and match.groups():
            try:
                return int(match.groups()[0])
            except TypeError:
                return

    def supportedDropActions(self):
        return QtCore.Qt.CopyAction

    def mimeTypes(self):
        return [u'text/plain']

    def mimeData(self, indexes):
        mimedata = QtCore.QMimeData()
        items_to_move = []
        for index in indexes:
            if index.column() != 0:
                continue
            item = index.internalPointer()
            items_to_move.append(item)
        mimedata.setData(u'text/plain', str([item.layer.identifier for item in items_to_move]))
        mimedata.items_to_move = items_to_move
        return mimedata

    def get_stage(self):
        return self._stage


class FilterModel(QtCore.QSortFilterProxyModel):
    def __init__(self, parent=None):
        QtCore.QSortFilterProxyModel.__init__(self, parent)
        self._show_references = True
        self._show_only_dirty = False
        self._show_only_anon = False
        self._show_only_empty = False

    def filterAcceptsRow(self, row_num, parent):
        """QSortFilterProxyModel method"""
        if parent is None:
            return False

        model = self.sourceModel()
        source_index = model.index(row_num, 0, parent)
        item = source_index.internalPointer()

        has_accepted_children = self.has_accepted_children(row_num, parent)
        if not item:
            return False
        if not self._show_references and item.is_ref:
            return False or has_accepted_children
        if self._show_only_dirty and not (item.layer.dirty or item.layer.anonymous):
            return False or has_accepted_children
        if self._show_only_anon and not item.layer.anonymous:
            return False or has_accepted_children
        if self._show_only_empty and not item.layer.empty:
            return False or has_accepted_children

        if QtCore.QSortFilterProxyModel.filterAcceptsRow(self, row_num, parent):
            return True

        return has_accepted_children

    def has_accepted_children(self, row_num, parent):
        """Starting from the current node as root, traverse all
            the descendants and test if any of the children match"""
        model = self.sourceModel()
        source_index = model.index(row_num, 0, parent)

        children_count = model.rowCount(source_index)
        for i in range(children_count):
            if self.filterAcceptsRow(i, source_index):
                return True
        return False

    def set_filter(self, filter_name, state):
        if filter_name == 'references':
            self._show_references = state
        if filter_name == 'dirty':
            self._show_only_dirty = state
        if filter_name == 'anon':
            self._show_only_anon = state
        if filter_name == 'empty':
            self._show_only_empty = state
        self.invalidateFilter()

    def ResetStage(self, stage):
        self.sourceModel().ResetStage(stage)

    def isModelValid(self):
        return self.sourceModel().isModelValid()

    def getPersistentIndexList(self):
        self.sourceModel().getPersistentIndexList()

    def supportedDropActions(self):
        return self.sourceModel().supportedDropActions()

    def mimeTypes(self):
        return self.sourceModel().mimeTypes()

    def mimeData(self, indexes):
        model = self.sourceModel()
        source_indexes = [self.mapToSource(index) for index in indexes]
        if source_indexes:
            return model.mimeData(source_indexes)

    def flags(self, modelIndex):
        source_index = self.mapToSource(modelIndex)
        return source_index.flags() | QtCore.Qt.ItemIsDragEnabled | QtCore.Qt.ItemIsDropEnabled

    def get_item(self, index):
        source_index = self.mapToSource(index)
        return source_index.internalPointer()

    def get_stage(self):
        model = self.sourceModel()
        return model.get_stage()
