from .layer_widget import LayerWidget, MenuAction, LayerView as CommonLayerView, OpenWith, \
    ShowLayerText, CopyLayerPath, ReloadLayer, LocaliseLayer, \
    MenuSeparator, ChangeVersionAction, SaveLayer, SaveAsLayer, \
    CreateSubLayer, CreateAnonymousSubLayer, AddExistingSubLayer, RemoveLayer
from pxr.UsdQtEditors._Qt import QtWidgets, QtCore
import wizart.dcc.core as dcc_core
from ...i18n import i18n


FEATURE_ANIM_ENGINE = False

try:
    import wizart.anim_engine
    FEATURE_ANIM_ENGINE = True
except:
    pass

class BakeAnimation(MenuAction):
    defaultText = i18n("layer_widget", "Bake Animation")

    def Do(self):
        context = self.GetCurrentContext()
        if context.selectedLayers:
            from dcc.anim_engine.bake_animation_window import BakeAnimationWindow
            app = dcc_core.Application.instance()
            bake_window = BakeAnimationWindow(context.selectedLayers[0], app.get_main_window() )
            bake_window.show()

        
class BakeExpressions(MenuAction):
    defaultText = i18n("layer_widget", "Bake Expressions")

    def Do(self):
        context = self.GetCurrentContext()
        if context.selectedLayers:
            from dcc.expressions.bake_expressions_window import BakeExpressionsWindow
            app = dcc_core.Application.instance()
            bake_window = BakeExpressionsWindow(context.selectedLayers[0], app.get_main_window() )
            bake_window.show()

class DccOpenWith(OpenWith):
    def _open_with_dcc(self):
        context = self.GetCurrentContext()
        if not context.selectedLayers:
            return

        app = dcc_core.Application.instance()
        session = app.get_session()
        session.open_stage(context.selectedLayers[0].identifier)

    def _actions(self, menu):
        actions = OpenWith._actions(self, menu)
        separator = QtWidgets.QAction(menu)
        separator.setSeparator(True)
        actions.append(separator)
        open_with_dcc = QtWidgets.QAction(i18n("layer_widget", "Open as Current Stage"), menu)
        open_with_dcc.triggered.connect(self._open_with_dcc)
        actions.append(open_with_dcc)
        return actions

    def Build(self, context):
        if len(context.selectedLayers) != 1:
            return

        menu = QtWidgets.QMenu(i18n("layer_widget", "Open With"), context.qtParent)

        for action in self._actions(menu):
            menu.addAction(action)
        return menu.menuAction()


class DccLayerWidgetRole():
    @classmethod
    def GetContextMenuActions(cls):
        actions_list = [ShowLayerText, CopyLayerPath, ReloadLayer, LocaliseLayer,
                MenuSeparator, CreateSubLayer, CreateAnonymousSubLayer, AddExistingSubLayer, RemoveLayer,
                MenuSeparator, DccOpenWith, SaveLayer, SaveAsLayer,
                MenuSeparator, ChangeVersionAction]

        if FEATURE_ANIM_ENGINE:
            actions_list.append(MenuSeparator)
            actions_list.append(BakeAnimation) 
            actions_list.append(BakeExpressions)                

        return actions_list


class DccLayerWidget(LayerWidget):
    def __init__(self, stage=None, role=DccLayerWidgetRole, editTargetChangeCallback=None, parent=None):
        self._role = role
        LayerWidget.__init__(self, stage, role, editTargetChangeCallback, parent=parent)
        self.app = dcc_core.Application.instance()
        self.add_callbacks()

        self.data_role = QtCore.Qt.UserRole+1 # todo: fix this because it's horrible and I will burn in hell for it
        self.view.selectionModel().selectionChanged.connect(self.selection_changed)
        self.layer_selected()

    def get_selected_layers(self):
        rows = self.view.selectionModel().selectedRows()
        indexes = [self._filterModel.mapToSource(row) for row in rows]
        layers = [self._dataModel.data(index, self.data_role) for index in indexes]
        return layers

    def selection_changed(self, selected, deselected):
        self.app.set_layer_selection(self.get_selected_layers())
    
    def closeEvent(self, event):
        self.remove_callbacks()
        event.accept()

    def update_stage(self):
        stage = self.app.get_session().get_current_stage()
        self.set_stage(stage)

    def add_callbacks(self):
        """Add callbacks to keep ui in sync while visible"""
        self.stage_changed_callback_id = self.app.register_event_callback(
            dcc_core.Application.EventType.CURRENT_STAGE_CHANGED,
            self.update_stage)
        self.layer_selection_changed_callback_id = self.app.register_event_callback(
            dcc_core.Application.EventType.LAYER_SELECTION_CHANGED, 
            self.layer_selected)
        self.update_stage()

    def layer_selected(self):
        selected_layers = set(self.get_selected_layers())
        layers = set(self.app.get_layer_selection())
        if selected_layers != layers:
            selection_model = self.view.selectionModel()
            selection_model.blockSignals(True)
            selection_model.clearSelection()

            def item_action(index, selection_index):
                layer = self._dataModel.data(index, self.data_role)
                if layer in layers:
                    selection_model.select(selection_index, QtCore.QItemSelectionModel.Select | QtCore.QItemSelectionModel.Rows)

            def traverse(index, selection_index, model, item_selection_model, fun):
                if index.isValid():
                    fun(index, selection_index)
                rows = model.rowCount(index)
                for row in range(rows):
                    traverse(model.index(row, 0, index), item_selection_model.index(row, 0, selection_index), model, item_selection_model, fun)

            traverse(QtCore.QModelIndex(), QtCore.QModelIndex(), self._dataModel, selection_model.model(), item_action)

            selection_model.blockSignals(False)
            self.update()

    def remove_callbacks(self):
        """Remove callbacks on ui hide"""
        if self.stage_changed_callback_id:
            self.app.unregister_event_callback(
                dcc_core.Application.EventType.CURRENT_STAGE_CHANGED, self.stage_changed_callback_id)
            self.stage_changed_callback_id = None

        if self.layer_selection_changed_callback_id:
            self.app.unregister_event_callback(
                dcc_core.Application.EventType.LAYER_SELECTION_CHANGED, self.layer_selection_changed_callback_id
            )
            self.layer_selection_changed_callback_id = None
