from Qt import QtCore, QtGui, QtWidgets
import os.path
from .arc_edit_widgets import EditPayloadWidget, EditReferenceWidget, \
    ArcDialog
from dcc.undo import UsdEditsUndoBlock


def format_path(assetPath, primPath, short_path=False):
    payload_line = ''
    if assetPath:
        if short_path:
            assetPath = os.path.split(assetPath)[1]
        payload_line += '@{}@'.format(assetPath)
    if primPath:
        if short_path:
            primPath = primPath.name
        payload_line += '<{}>'.format(primPath)
    return payload_line


def is_prim_spec_on_edit_target(prim, layer):
    path = prim.GetPath()

    if path.IsTargetPath():
        x = layer.GetObjectAtPath(path.GetParentPath())
        if x:
            return x.targetAttributes[path.targetPath].IsValid()
        return False
    return bool(layer.GetObjectAtPath(path))


def item_has_ancestor(item, ancestor):
    item_parent = item.parent()
    if not item_parent:
        return False
    else:
        if item_parent == ancestor:
            return True
        else:
            return item_has_ancestor(item_parent, ancestor)


class CompositionMetadataTree(QtWidgets.QTreeWidget):
    selection_changed = QtCore.Signal(QtWidgets.QTreeWidgetItem)

    def __init__(self, prim=None, parent=None):
        super(CompositionMetadataTree, self).__init__(parent)
        self.setColumnCount(3)
        self.setHeaderLabels(['', '', ''])

        self.update(prim)
        self.itemSelectionChanged.connect(self.update_selection)

        self.setContextMenuPolicy(QtCore.Qt.CustomContextMenu)
        self.customContextMenuRequested.connect(self.open_context_menu)

    def _add_item(self, parent, data, widget_type=None,
                  **kwargs):
        item = QtWidgets.QTreeWidgetItem(parent, data)
        item._val = kwargs.get('val', None)
        item._is_editable = kwargs.get('is_editable', False)
        item._is_add_enabled = kwargs.get('is_add_enabled', False)
        item._has_edits = kwargs.get('has_edits', False)
        item._is_root = kwargs.get('is_root', False)
        item._is_removable = kwargs.get('is_removable', False)
        if not item._is_editable:
            for i in range(item.columnCount()):
                item.setForeground(i, QtCore.Qt.darkGray)
            item._edit_widget_type = None
        else:
            item._edit_widget_type = widget_type

        return item

    def _rebuild(self):
        self.clear()

        if not self._prim:
            return

        # Populate the treeview with items from the prim metadata
        self._rebuild_payloads()
        self._rebuild_references()
        self._rebuild_variants()

    def add_listop(self, parent, title, listop, widget, prim_spec_list):
        if listop:
            parent_item = self._add_item(parent, [title],
                                         listop_type=title.lower(),
                                         is_editable=self._has_prim_spec_on_layer)
            for path in listop:
                is_removable = is_editable = bool(self._prim_spec) \
                               and bool(prim_spec_list) \
                               and prim_spec_list.ContainsItemEdit(path)

                item = self._add_item(parent_item, [format_path(path.assetPath, path.primPath, short_path=True),
                                                    path.assetPath, str(path.primPath)],
                                      widget_type=widget,
                                      val=path,
                                      is_editable=is_editable,
                                      is_removable=is_removable)

    def _rebuild_payloads(self):
        payloads = self._prim.GetMetadata('payload')

        # We leave ability to add payload only when there are no one
        # If payload already set, we can edit it only if it set on current edit target
        is_editable = False
        if (self._has_prim_spec_on_layer and self._prim.HasPayload()) or (not self._prim.HasPayload()):
            is_editable = True

        has_edits = self._has_prim_spec_on_layer and self._prim_spec.payloadList
        self.payload_parent_item = self._add_item(self, ['Payload'],
                                                  widget_type=EditPayloadWidget,
                                                  is_editable=is_editable,
                                                  is_add_enabled=is_editable,
                                                  has_edits=has_edits,
                                                  is_root=True)
        if self._prim.HasPayload():
            payload_spec_list = None
            if self._prim_spec:
                payload_spec_list = self._prim_spec.payloadList
            self.add_listop(self.payload_parent_item, 'Added', payloads.addedItems, EditPayloadWidget, payload_spec_list)
            self.add_listop(self.payload_parent_item, 'Appended', payloads.appendedItems, EditPayloadWidget, payload_spec_list)
            self.add_listop(self.payload_parent_item, 'Deleted', payloads.deletedItems, EditPayloadWidget, payload_spec_list)
            self.add_listop(self.payload_parent_item, 'Explicit', payloads.explicitItems, EditPayloadWidget, payload_spec_list)
            self.add_listop(self.payload_parent_item, 'Ordered', payloads.orderedItems, EditPayloadWidget, payload_spec_list)
            self.add_listop(self.payload_parent_item, 'Prepended', payloads.prependedItems, EditPayloadWidget, payload_spec_list)

    def _rebuild_references(self):
        has_edits = self._has_prim_spec_on_layer and self._prim_spec.hasReferences
        self.references_parent_item = self._add_item(self, ['References'],
                                                     widget_type=EditReferenceWidget,
                                                     is_editable=True,
                                                     is_add_enabled=True,
                                                     has_edits=has_edits,
                                                     is_root=True)
        references = self._prim.GetMetadata('references')

        if references:
            reference_spec_list = None
            if self._prim_spec:
                reference_spec_list = self._prim_spec.referenceList
            self.add_listop(self.references_parent_item, 'Added', references.addedItems, EditReferenceWidget, reference_spec_list)
            self.add_listop(self.references_parent_item, 'Appended', references.appendedItems, EditReferenceWidget, reference_spec_list)
            self.add_listop(self.references_parent_item, 'Deleted', references.deletedItems, EditReferenceWidget, reference_spec_list)
            self.add_listop(self.references_parent_item, 'Explicit', references.explicitItems, EditReferenceWidget, reference_spec_list)
            self.add_listop(self.references_parent_item, 'Ordered', references.orderedItems, EditReferenceWidget, reference_spec_list)
            self.add_listop(self.references_parent_item, 'Prepended', references.prependedItems, EditReferenceWidget, reference_spec_list)

    def _rebuild_variants(self):
        has_edits = self._has_prim_spec_on_layer and len(self._prim_spec.variantSelections) > 0
        variant_sets = self._prim.GetVariantSets().GetNames()
        flag = len(variant_sets) > 0
        self.variants_parent_item = self._add_item(self, ['Variants'],
                                                   is_editable=flag,
                                                   has_edits=has_edits,
                                                   is_root=True)
        for set_name in variant_sets:
            variant_set = self._prim.GetVariantSet(set_name)
            variant_names = variant_set.GetVariantNames()
            variant_selection = variant_set.GetVariantSelection()

            combo = QtWidgets.QComboBox(self)
            # First index is always empty to indicate no (or invalid)
            # variant selection.
            combo.addItem('')
            for name in variant_names:
                combo.addItem(name)
            combo.variantSetName = set_name
            selected_index = combo.findText(variant_selection)
            combo.setCurrentIndex(selected_index)

            item = self._add_item(self.variants_parent_item, [set_name], is_editable=True)
            self.setItemWidget(item, 1, combo)
            combo.currentIndexChanged.connect(self.update_variant_selection)

    def update_variant_selection(self, index):
        selected = self.selectedItems()
        if not selected:
            return
        selected_item = selected[0]
        combobox = self.itemWidget(selected_item, 1)
        variant_set = self._prim.GetVariantSet(combobox.variantSetName)
        current = variant_set.GetVariantSelection()
        new = str(combobox.currentText())
        if current != new:
            with UsdEditsUndoBlock():
                res = variant_set.SetVariantSelection(new)

    def update(self, prim):
        self._prim = prim
        if prim:
            stage = prim.GetStage()
            layer = stage.GetEditTarget().GetLayer()
            self._has_prim_spec_on_layer = is_prim_spec_on_edit_target(self._prim, layer)
            self._prim_spec = layer.GetPrimAtPath(prim.GetPath())
        self._rebuild()

        self.expandAll()
        self.resizeColumnToContents(0)

    def update_selection(self):
        lst = self.selectedItems()
        for item in lst:
            self.selection_changed.emit(item)

    def open_context_menu(self, position):
        menu = QtWidgets.QMenu()
        selected = self.selectedItems()
        if not selected:
            return
        selected_item = selected[0]

        # no widget - no ability to add or edit
        if selected_item._edit_widget_type:
            title = 'Add {}'.format(selected_item.data(0, QtCore.Qt.DisplayRole))
            func = lambda : self.edit_arc(True)

            if not selected_item._is_add_enabled:
                title = 'Edit {}'.format(selected_item.data(0, QtCore.Qt.DisplayRole))
                func = lambda : self.edit_arc(False)

            add_act = QtWidgets.QAction(title)
            add_act.triggered.connect(func)
            menu.addAction(add_act)

            if not selected_item._is_editable:
                add_act.setEnabled(False)

        if selected_item._is_removable:
            remove_act = QtWidgets.QAction('Remove {}'.format(selected_item.data(0, QtCore.Qt.DisplayRole)))
            remove_act.triggered.connect(self.remove_arc)
            menu.addAction(remove_act)

        if selected_item._is_root:
            clear_act = QtWidgets.QAction('Clear {}'.format(selected_item.data(0, QtCore.Qt.DisplayRole)))
            clear_act.triggered.connect(self.clear_arc)
            menu.addAction(clear_act)
            if not selected_item._has_edits:
                clear_act.setEnabled(False)

        if not menu.isEmpty():
            menu.exec_(self.mapToGlobal(position))

    def edit_arc(self, is_for_adding=True):
        selected = self.selectedItems()
        if selected:
            selected_item = selected[0]

            dialog = ArcDialog(self._prim, is_for_adding=is_for_adding)
            dialog.set_new_editor(selected_item._edit_widget_type, val=selected_item._val, parent=self)
            res = dialog.exec_()
            if res:
                self.update(self._prim)

    def clear_arc(self):
        selected = self.selectedItems()
        if not selected:
            return

        selected_item = selected[0]
        with UsdEditsUndoBlock():
            if selected_item == self.payload_parent_item:
                self._prim.ClearPayload()
            elif selected_item == self.references_parent_item:
                self._prim.GetReferences().ClearReferences()
            elif selected_item == self.variants_parent_item:
                variant_sets = self._prim.GetVariantSets().GetNames()
                for set_name in variant_sets:
                    variant_set = self._prim.GetVariantSet(set_name)
                    variant_set.ClearVariantSelection()

        self.update(self._prim)

    def remove_arc(self):
        selected = self.selectedItems()
        if not selected or not self._prim_spec:
            return

        selected_item = selected[0]
        with UsdEditsUndoBlock():
            if item_has_ancestor(selected_item, self.references_parent_item):
                references = self._prim_spec.referenceList
                if references and selected_item._val:
                    references.Erase(selected_item._val)
                    self.update(self._prim)
            elif item_has_ancestor(selected_item, self.payload_parent_item):
                payloads = self._prim_spec.payloadList
                if payloads and selected_item._val:
                    payloads.Erase(selected_item._val)
                    self.update(self._prim)




