from pxr.UsdQtEditors._Qt import QtCore, QtWidgets, QtGui
from pxr import Sdf, UsdGeom, UsdShade
from pxr.UsdQt.hierarchyModel import HierarchyBaseModel

from ._func import is_prim_spec_on_edit_target, rename_prim, IsOnEditTargetRole, \
    HasPcpErrorsRole, has_prim_pcp_errors

from .view_delegates import VariantsViewDelegate, MaterialViewDelegate, RefineLevelDelegate, DrawModeViewDelegate

try:
    from dcc.undo import UsdEditsUndoBlock as UndoBlock
except ImportError:
    from pxr.UsdQt._usdQt import UndoBlock


SHOW_HEADER_TEXT = 1
SHOW_HEADER_ICON = 2

icons_cache = {}
edit_target_flag = '_is_not_on_edit_target'
pcp_error_flag = '_has_pcp_error'
_icons_init = False
_type_icons_list = ['mesh', 'xform',
                    'camera', 'material',
                    'cylinderlight', 'disklight', 'distantlight', 'domelight',
                    'rectlight', 'spherelight',
                    'wgenclumping', 'wgencut', 'wgengrow', 'wgeninterpolation',
                    'wgenloadparamsfrommesh', 'wgenscatter', 'wgenshape',
                    'wgenskinning','wgenanimwires', 'wgenaddchannel', 
                    'wgenbundlesscatter', 'wgenoperator', 'wgenvolumesculpt', 'wgenbend',
                    'wgenbender', 'wgencleanup',
                    'pointinstancer', 'scope', 'basiscurves', 'shader',
                    'cube', 'capsule', 'cone', 'cylinder', 'sphere',
                    'scenelibusdin', 
                    'scenelibedit', 'scenelibobjectsettings', 'scenelibluascript', 
                    'scenelibsetattribute', 'scenelibmerge', 'scenelibprune',
                    'scenelibisolate', 'scenelibcreatelocation',
                    'scenelibnodegraph', 'scenelibvariableswitch' , 'scenelibcyclesrendersettings', 
                    'scenelibrenderoutput', 'scenelibrendersettings', 'sceneliblightrig',
                    'nodegraph',
                    'rendersettings', 'renderproduct', 'rendervar']


def create_icons():
    global _icons_init
    if not _icons_init:
        _icons_init = True
        icons_lst = ["eye_open",
                     "eye_closed",
                     "eye_open_faded",
                     "eye_closed_faded"]
        icons_lst += ["group", "component", "asset", "group", "subcomponent", 
            "default", "withouttype", "xform_instance", "mesh_instance"] + _type_icons_list

        for icon in icons_lst:
            add_icon(icon, ":/icons/{0}".format(icon), defined=True)


def mix_pixmaps(pixmap, add_pixmap):
    new_pixmap = QtGui.QPixmap(pixmap)
    painter = QtGui.QPainter(new_pixmap)
    painter.drawPixmap(0, 0, 10, 10, add_pixmap)
    return new_pixmap


def gen_error(pixmap):
    new_pixmap = QtGui.QPixmap(pixmap)
    painter = QtGui.QPainter(new_pixmap)
    painter.setPen(QtGui.QPen(QtCore.Qt.darkRed, 2))
    painter.drawRoundedRect(0, 0, new_pixmap.width(), new_pixmap.height(), 5, 5)
    return new_pixmap


def add_icon(title, pixmap_name, defined=False):
    global icons_cache

    title = title.lower()

    pixmap = QtGui.QPixmap(pixmap_name).scaled(20, 20)
    add_pixmap = QtGui.QPixmap(":/icons/is_not_on_edit_target")
    edit_target_pixmap = mix_pixmaps(pixmap, add_pixmap)

    icons_cache[title] = pixmap
    icons_cache[title + edit_target_flag] = edit_target_pixmap
    icons_cache[title + pcp_error_flag] = gen_error(pixmap)
    icons_cache[title + edit_target_flag + pcp_error_flag] = gen_error(edit_target_pixmap)

    if not defined:
        global _type_icons_list
        if not title in _type_icons_list:
            _type_icons_list.append(title)


class BaseColumn:
    """
    Column with basic functions common for all
    """
    header_name = ''
    visibility = True
    header_icon = None
    view_mode = SHOW_HEADER_TEXT
    has_delegate = False

    HighlightedBackColor = QtGui.QBrush(QtGui.QColor(150, 200, 250, 60))

    def __init__(self, header_name, display_name):
        self.header_name = header_name
        self.visible = True
        self.display_name = display_name

    def set_icon(self, icon):
        self.header_icon = icon

    def set_visibility(self, flag):
        self.visible = flag

    def set_view_mode(self, view_mode):
        self.view_mode = view_mode

    def headerData(self, role):
        if role == QtCore.Qt.DecorationRole and self.view_mode == SHOW_HEADER_ICON:
            return self.header_icon
        if role == QtCore.Qt.DisplayRole:
            if self.view_mode == SHOW_HEADER_TEXT:
                return self.display_name

    def data(self, modelIndex, role=QtCore.Qt.DisplayRole):
        prim = modelIndex.model()._GetPrimForIndex(modelIndex)
        if role == QtCore.Qt.ForegroundRole:
            #color = qtUtils.LIGHT_BLUE
            #if prim.HasVariantSets():
            #    color = qtUtils.DARK_ORANGE
            #brush = QtGui.QBrush(color)
            #if not prim.IsActive():
            #    brush.setColor(brush.color().darker(
            #        UsdQtModel.InactiveDarker))
            #return brush
            return
        elif role == QtCore.Qt.BackgroundRole:
            return HierarchyBaseModel.data(modelIndex.model(), modelIndex, role)
        elif role == QtCore.Qt.FontRole:
            font = QtGui.QFont()
            if not prim.IsActive():
                font.setStrikeOut(True)
            if prim.IsAbstract() or not prim.IsDefined():
                font.setStyle(QtGui.QFont.StyleItalic)
            return font
        elif role == QtCore.Qt.TextAlignmentRole:
            return QtCore.Qt.AlignLeft
        elif role == QtCore.Qt.ToolTipRole:
            specifier = prim.GetSpecifier()
            primType = prim.GetTypeName()
            # documentation = prim.GetDocumentation()
            if specifier == Sdf.SpecifierDef:
                toolTipString = "Defined %s"
            elif specifier == Sdf.SpecifierOver:
                toolTipString = "Undefined %s"
            elif specifier == Sdf.SpecifierClass:
                toolTipString = "Abstract %s"
            else:
                raise Exception("Unhandled specifier for tooltip.")

            if not primType:
                toolTipString = toolTipString % "Prim"
            else:
                toolTipString = toolTipString % primType

            # if documentation:
            #     toolTipString = "%s\n%s" % (toolTipString, documentation)

            if prim.IsInstanceProxy():
                toolTipString = "Instance Proxy of %s" % toolTipString
            if prim.IsInstanceable():
                toolTipString = "Instance Root of %s" % toolTipString

            has_pcp_errors = BaseColumn.data(self, modelIndex, role=HasPcpErrorsRole)
            if has_pcp_errors:
                errors = prim.GetPrimIndex().localErrors
                toolTipString += """" 
"""
                toolTipString += """" 
""".join(str(error) for error in errors)
            return toolTipString
        elif role == QtCore.Qt.DecorationRole:
            return
        elif role == IsOnEditTargetRole:
            model = modelIndex.model()
            stage = model._stage
            return is_prim_spec_on_edit_target(prim, stage.GetEditTarget())
        elif role == HasPcpErrorsRole:
            return has_prim_pcp_errors(prim)
        return

    def flags(self, modelIndex):
        if not modelIndex.isValid():
            return QtCore.Qt.NoItemFlags

        flags = QtCore.Qt.ItemIsEnabled | QtCore.Qt.ItemIsSelectable | QtCore.Qt.ItemIsDragEnabled

        # if modelIndex.data(IsOnEditTargetRole):
        flags = flags | QtCore.Qt.ItemIsDropEnabled

        return flags

    def setData(self, modelIndex, value, role=QtCore.Qt.EditRole):
        return False


class NameColumn(BaseColumn):

    def __init__(self, header_name, display_name):
        BaseColumn.__init__(self, header_name, display_name)
        create_icons()
        self.is_changing = False

    def data(self, modelIndex, role=QtCore.Qt.DisplayRole):
        prim = modelIndex.model()._GetPrimForIndex(modelIndex)

        if role in (QtCore.Qt.DisplayRole, QtCore.Qt.EditRole):
            return prim.GetName()

        if role == QtCore.Qt.DecorationRole:
            kind = prim.GetMetadata('kind')
            kind = kind.lower() if kind is not None else ''
            type_name = prim.GetTypeName()
            type_name = type_name.lower() if type_name is not None else ''

            is_not_on_edit_target = not BaseColumn.data(self, modelIndex, role=IsOnEditTargetRole)
            suff = ''
            if is_not_on_edit_target:
                suff = edit_target_flag
            if modelIndex.model()._show_pcp_errors:
                has_pcp_errors = BaseColumn.data(self, modelIndex, role=HasPcpErrorsRole)
                if has_pcp_errors:
                    suff = suff + pcp_error_flag

            if kind == 'assembly':
                icon = 'asset' + suff
                if icon in icons_cache:
                    return icons_cache[icon]
            elif kind == 'group':
                icon = 'group' + suff
                if icon in icons_cache:
                    return icons_cache[icon]
            elif kind == 'component':
                icon = 'component' + suff
                if icon in icons_cache:
                    return icons_cache[icon]
            elif kind == 'subcomponent':
                icon = 'subcomponent' + suff
                if icon in icons_cache:
                    return icons_cache[icon]

            if type_name == '':
                icon = 'withouttype' + suff
                if icon in icons_cache:
                    return icons_cache[icon]

            if type_name == 'xform':
                if prim.IsInstanceable():
                    icon = 'xform_instance' + suff
                    if icon in icons_cache:
                        return icons_cache[icon]

            if type_name == 'mesh':
                if prim.IsInstance() or prim.IsInstanceProxy(): 
                    icon = 'mesh_instance' + suff
                    if icon in icons_cache:
                        return icons_cache[icon]

            if type_name in _type_icons_list:
                icon = type_name + suff
                if icon in icons_cache:
                    return icons_cache[icon]
            else: 
                icon = 'default' + suff
                if icon in icons_cache:
                    return icons_cache[icon]

        elif role == QtCore.Qt.BackgroundRole:
            # TODO not appropriate highlighting in name-column.
            # Tree navigation not highlighted
            index = modelIndex.model().index(modelIndex.row(), 0, modelIndex.parent())
            if index in modelIndex.model().highlighted:
                return self.HighlightedBackColor
            else:
                return BaseColumn.data(self, modelIndex, role)
        else:
            return BaseColumn.data(self, modelIndex, role)

    def setData(self, modelIndex, value, role=QtCore.Qt.EditRole):
        self.is_changing = True
        res = False
        if role == QtCore.Qt.EditRole:
            prim = modelIndex.model()._GetPrimForIndex(modelIndex)
            if value and prim.GetName() != value:
                try:
                    from wizart.dcc import cmds
                    cmds.rename_prim(value, prim = prim)
                    modelIndex.model().repaintView.emit()
                except ImportError:
                    with UndoBlock():
                        res, new_path = rename_prim(prim, value)
                        if res:
                            modelIndex.model().needSelectIndexes.emit([new_path])
                            modelIndex.model().repaintView.emit()
        self.is_changing = False
        return res

    def flags(self, modelIndex):
        if not modelIndex.isValid():
            return QtCore.Qt.NoItemFlags

        flags = BaseColumn.flags(self, modelIndex)

        # if self.is_changing or modelIndex.data(IsOnEditTargetRole):
        flags = flags | QtCore.Qt.ItemIsEditable | QtCore.Qt.ItemIsDropEnabled

        return flags


class VisibilityColumn(BaseColumn):

    def __init__(self, header_name, display_name):
        BaseColumn.__init__(self, header_name, display_name)
        create_icons()
        self.header_icon = icons_cache['eye_open']
        self.view_mode = SHOW_HEADER_TEXT
        self.display_name = display_name

    def headerData(self, role):
        if role == QtCore.Qt.DecorationRole \
                and self.view_mode == SHOW_HEADER_ICON:
            return self.header_icon
        return BaseColumn.headerData(self, role)

    def data(self, modelIndex, role=QtCore.Qt.DisplayRole):
        prim = modelIndex.model()._GetPrimForIndex(modelIndex)
        model = modelIndex.model()

        if role == QtCore.Qt.DisplayRole:
            return
        elif role == QtCore.Qt.DecorationRole:
            vis = prim.GetAttribute('visibility')
            if not vis.IsValid(): return
            suf = '' if self.is_visibility_editable(prim, model._time_code) else '_faded'
            if vis.Get(model._time_code) == UsdGeom.Tokens.invisible:
                return icons_cache['eye_closed' + suf]
            else:
                return icons_cache['eye_open' + suf]
        elif role == QtCore.Qt.TextAlignmentRole:
            return QtCore.Qt.AlignHCenter
        return BaseColumn.data(self, modelIndex, role)

    def is_visibility_editable(self, prim, time_code):
        """
        If some parent is invisible - return False, otherwise return True
        TODO: if problems with productivity rewrite
        to use additional dict with visEditability
        """
        if prim.GetName() == '/' or prim.GetParent().GetName() == '/':
            return True
        parent = prim.GetParent()

        vis = parent.GetAttribute('visibility')
        if not vis.IsValid(): return False
        if vis.Get(time_code) == UsdGeom.Tokens.invisible:
            return False
        else:
            return self.is_visibility_editable(parent, time_code)

    def setData(self, modelIndex, value, role=QtCore.Qt.EditRole):
        if role == QtCore.Qt.EditRole:
            prim = modelIndex.model()._GetPrimForIndex(modelIndex)
            if prim.IsPseudoRoot(): return

            vis = prim.GetAttribute('visibility')
            if not vis.IsValid(): return

            if value:
                with UndoBlock():
                    vis.Set(UsdGeom.Tokens.inherited)
                modelIndex.model().dataChanged.emit(modelIndex, QtCore.QModelIndex())
                return True
            else:
                with UndoBlock():
                    vis.Set(UsdGeom.Tokens.invisible)
                modelIndex.model().dataChanged.emit(modelIndex, QtCore.QModelIndex())
                return True
        return False


class CompositionColumn(BaseColumn):
    pass


class AttributeColumn(BaseColumn):
    def __init__(self, header_name, display_name, attribute_name=''):
        BaseColumn.__init__(self, header_name, display_name)
        self.attribute_name = attribute_name

    def data(self, modelIndex, role=QtCore.Qt.DisplayRole):
        if role==QtCore.Qt.DisplayRole:
            model = modelIndex.model()
            prim = modelIndex.model()._GetPrimForIndex(modelIndex)
            return prim.GetAttribute(self.attribute_name).Get(model._time_code)
        else:
            return BaseColumn.data(self, modelIndex, role)


class RelationshipColumn(BaseColumn):
    def __init__(self, header_name, display_name, model=None, relationship_name=''):
        BaseColumn.__init__(self, header_name, display_name)
        self.relationship_name = relationship_name

    def data(self, modelIndex, role=QtCore.Qt.DisplayRole):
        if role == QtCore.Qt.DisplayRole:
            prim = modelIndex.model()._GetPrimForIndex(modelIndex)
            targets = prim.GetRelationship(self.relationship_name).GetTargets()
            return ', '.join([str(path) for path in targets])
        else:
            return BaseColumn.data(self, modelIndex, role)


class MetaDataColumn(BaseColumn):
    def __init__(self, header_name, display_name, metadata_name=''):
        BaseColumn.__init__(self, header_name, display_name)
        self.metadata_name = metadata_name

    def data(self, modelIndex, role=QtCore.Qt.DisplayRole):
        prim = modelIndex.model()._GetPrimForIndex(modelIndex)
        if role == QtCore.Qt.DisplayRole:
            return prim.GetMetadata(self.metadata_name)
        else:
            return BaseColumn.data(self, modelIndex, role)


class VariantColumn(BaseColumn):
    def __init__(self, header_name, display_name, model_variant_name=''):
        BaseColumn.__init__(self, header_name, display_name)
        self.model_variant_name = model_variant_name
        self.has_delegate = True
        self.delegate_instance = None

    def delegate(self, model):
        if not self.delegate_instance:
            self.delegate_instance = VariantsViewDelegate(model,
                                        variants_set_name=self.model_variant_name)
        return self.delegate_instance

    def data(self, modelIndex, role=QtCore.Qt.DisplayRole):
        if role == QtCore.Qt.DisplayRole:
            prim = modelIndex.model()._GetPrimForIndex(modelIndex)
            if not prim.HasVariantSets():
                return ''
            variantSet = prim.GetVariantSet(self.model_variant_name)
            current = variantSet.GetVariantSelection()
            return current
        return BaseColumn.data(self, modelIndex, role)

    def flags(self, modelIndex):
        if modelIndex.isValid():
            return QtCore.Qt.ItemIsEnabled | QtCore.Qt.ItemIsSelectable \
                   | QtCore.Qt.ItemIsEditable
        return BaseColumn.flags(self, modelIndex)


class DrawModeColumn(BaseColumn):
    def __init__(self, header_name, display_name):
        BaseColumn.__init__(self, header_name, display_name)
        self.has_delegate = True
        self.delegate_instance = None

    def delegate(self, model):
        if not self.delegate_instance:
            self.delegate_instance = DrawModeViewDelegate(model)
        return self.delegate_instance

    def data(self, modelIndex, role=QtCore.Qt.DisplayRole):
        if role == QtCore.Qt.DisplayRole:
            prim = modelIndex.model()._GetPrimForIndex(modelIndex)
            if prim:
                model = UsdGeom.ModelAPI(prim)
                return model.ComputeModelDrawMode()
        return BaseColumn.data(self, modelIndex, role)

    def flags(self, modelIndex):
        if modelIndex.isValid():
            return QtCore.Qt.ItemIsEnabled | QtCore.Qt.ItemIsSelectable \
                   | QtCore.Qt.ItemIsEditable
        return BaseColumn.flags(self, modelIndex)

class MaterialColumn(BaseColumn):
    def __init__(self, header_name, display_name, token=UsdShade.Tokens.full):
        BaseColumn.__init__(self, header_name, display_name)
        self.token = token
        self.has_delegate = True
        self.delegate_instance = None

    def delegate(self, model):
        if not self.delegate_instance:
            self.delegate_instance = MaterialViewDelegate(model)
        return self.delegate_instance

    def data(self, modelIndex, role=QtCore.Qt.DisplayRole):
        model = modelIndex.model()
        if role == QtCore.Qt.DisplayRole:
            prim = model._GetPrimForIndex(modelIndex)
            material_bind = UsdShade.MaterialBindingAPI(prim)

            material, rel = material_bind.ComputeBoundMaterial(self.token)
            res_str = ''
            if (material.GetPrim().IsValid()):
                res_str += material.GetPrim().GetPath().pathString

            return res_str
        else:
            return BaseColumn.data(self, modelIndex, role)


class RefineLevelColumn(BaseColumn):
    def __init__(self, header_name, display_name):
        BaseColumn.__init__(self, header_name, display_name)
        from dcc_core import UsdViewportRefineManager
        self.refine_manager = UsdViewportRefineManager.instance()
        self.has_delegate = True
        self.delegate_instance = None

    def delegate(self, model):
        if not self.delegate_instance:
            self.delegate_instance = RefineLevelDelegate(model)
        return self.delegate_instance

    def data(self, modelIndex, role=QtCore.Qt.DisplayRole):
        if role == QtCore.Qt.DisplayRole:
            prim = modelIndex.model()._GetPrimForIndex(modelIndex)
            stage = modelIndex.model()._stage
            return self.refine_manager.get_refine_level(stage, prim.GetPath())
        else:
            return BaseColumn.data(self, modelIndex, role)

    def setData(self, modelIndex, value, role=QtCore.Qt.EditRole):
        self.is_changing = True
        res = False
        if role == QtCore.Qt.EditRole:
            prim = modelIndex.model()._GetPrimForIndex(modelIndex)
            stage = modelIndex.model()._stage
            if self.refine_manager.get_refine_level(stage, prim.GetPath()) != int(value):
                with UndoBlock():
                    self.refine_manager.set_refine_level(stage, prim.GetPath(), int(value))
                    res = True
        self.is_changing = False
        return res

    def flags(self, modelIndex):
        if not modelIndex.isValid():
            return QtCore.Qt.NoItemFlags

        flags = BaseColumn.flags(self, modelIndex) | QtCore.Qt.ItemIsEditable

        return flags


class ColumnController(object):
    """
    Class to aggregate all columns of the outliner and manage them
    Ex. get column by id, get columns count,
    iterate through columns and others
    """
    def __init__(self):
        self.column_dict = {}
        self.order_lst = []
        self.tree_index = 0

    def add_column(self, cls, name, display_name, *args, **qwargs):
        self.column_dict[name] = cls(name, display_name, *args, **qwargs)
        self.order_lst.append(name)
        return self.column_dict[name]

    def remove_column(self, index):
        self.column_dict.pop(self.order_lst[index])
        self.order_lst.pop(index)

    def remove_column_by_name(self, header_name):
        self.column_dict.pop(header_name)
        self.order_lst.remove(header_name)

    def __getattr__(self, item):
        """
        Little hack: imitate that columns instances become attributes
        of the controller available by their header name.
        Ex.: for name like 'Modeling Variant' name of attribute
        will be .modeling_variant
        :param item:
        :return:
        """
        for col in self.order_lst:
            if item == col.replace(' ', '_').lower():
                return self.column_dict[col]

    def __getitem__(self, i):
        """
        To make columns available by their order index
        :param i: column index
        type: int
        :return: column instance
        """
        return self.column_dict[self.order_lst[i]]

    def __iter__(self):
        """
        To iterate through columns
        :return: column instance
        """
        for col in self.order_lst:
            yield self.column_dict[col]

    def count(self):
        """
        :return: number of columns
        """
        return len(self.column_dict)

    def index(self, name):
        """
        Get index by name
        :param name: header_name of the column
        :return: index of the column
        """
        return self.order_lst.index(name)

    def set_tree_index(self, index):
        self.tree_index = index

    def __repr__(self):
        tmp = []
        for i in range(len(self.order_lst)):
            tmp.append(str((self.order_lst[i], self.column_dict[self.order_lst[i]].__class__)))
        return ', '.join(tmp)
