"""
Module contains customized wizOutliner which has additional opportunities in Maya, such as
- synchronized selection
"""
import os
import re
from pxr.UsdQtEditors._Qt import QtCore, QtGui, QtWidgets
from pxr import Sdf, Tf, Usd, UsdGeom
from .outliner import UsdOutliner, OutlinerTreeView, OutlinerRole
from pxr.UsdQtEditors.outliner import SaveState, MenuBuilder, SaveEditLayer, ShowEditTargetLayerText,\
    ShowEditTargetDialog, MenuSeparator
from pxr.UsdQt.qtUtils import MenuAction
from .widgets.layer_stack_widget.layer_widget import LayerWidget
from .widgets.layer_stack_widget.maya_layer_widget import MayaLayerWidget
from .widgets.composition_editor.maya_composition_editor import MayaCompositionEditor
import pymel.core as pm
from .resources import resources
from collections import namedtuple
from . import maya_undo

HAVE_UFE = False
try:
    import ufe
    HAVE_UFE = True
except:
    pass


AL_PROXY_SHAPE_TYPE = 'AL_usdmaya_ProxyShape'


def setStageDataDirty(proxy_shape_name, flag):
    """
    Set attribute stageDataDirty = flag. Usually set True in order update stage
    :param flag: bool
    """
    pm.setAttr('{0}.stageDataDirty'.format(proxy_shape_name), flag)


class ALMayaOutlinerTreeView(OutlinerTreeView):

    primSelectionChanged = QtCore.Signal(list, list)

    expandedIndexes = []
    highlightedParents = []

    def __init__(self, dataModel, contextMenuActions, proxy_shape_name=None, contextProvider=None,
                 parent=None):
        # type: (QtCore.QAbstractItemModel, List[MenuAction], Optional[ContextProvider], Optional[QtWidgets.QWidget]) -> None
        """
        Parameters
        ----------
        dataModel : QtCore.QAbstractItemModel
        contextMenuActions : List[MenuAction]
        contextProvider : Optional[ContextProvider]
        parent : Optional[QtWidgets.QWidget]
        """
        OutlinerTreeView.__init__(self,
                                        dataModel,
                                        contextMenuActions=contextMenuActions,
                                        contextProvider=contextProvider,
                                        parent=parent)

        self._proxy_shape_name = proxy_shape_name

        self._blockSelectionCallback = False

    def set_proxy_shape_name(self, proxy_shape_name):
        self._proxy_shape_name = proxy_shape_name

    def connectSignals(self):
        """
        Connect signals. Relevant mostly when the model settled
        """
        OutlinerTreeView.connectSignals(self)
        self.primSelectionChanged.connect(self.change_selection_into_maya)
        self._dataModel.needUpdateStage.connect(self.setStageDataDirty)

    def change_selection_into_maya(self, selected, deselected):
        """
        Synchronize outliner selection with Maya
        Connected to QTreeView.selectionChanged signal
        :param selected: list of prims selected in Outliner
        :param deselected: list of prims deselected in Outliner
        """
        # Block selection in Maya while updating selection state by hands
        self._blockSelectionCallback = True

        if self._proxy_shape_name is None:
            return
        # In AL_usdmaya said "Don't use -i flag". But i'm a rebel
        # TODO this can break if AL_usdmaya will be changed
        print("change_selection_into_maya", selected, deselected)
        if len(selected) > 0:

            if HAVE_UFE:
                #with ufe with can use standard maya selection command
                sel_list = []
                for prim in selected:
                    sel_list.append( "{},{}".format(pm.PyNode(self._proxy_shape_name).fullPath(), str(prim.GetPath())))
                pm.select(sel_list, add=True)
                print("change_selection_into_maya:selected",sel_list)
            else:
                # flag -a means append into selection list
                pm.AL_usdmaya_ProxyShapeSelect(p=self._proxy_shape_name,
                                            pp=[str(prim.GetPath()) for prim in selected],
                                            a=True,
                                            i=True)

        if len(deselected) > 0:
            if HAVE_UFE:
                #with ufe with can use standard maya selection command
                sel_list = []
                for prim in deselected:
                    sel_list.append( "{},{}".format(pm.PyNode(self._proxy_shape_name).fullPath(), str(prim.GetPath())))
                pm.select(sel_list, d=True)
                print("change_selection_into_maya:deselected",sel_list)
            else:
                # flag -d means deselect
                pm.AL_usdmaya_ProxyShapeSelect(p=self._proxy_shape_name,
                                            pp=[str(prim.GetPath()) for prim in deselected],
                                            d=True,
                                            i=True)
        
        self._blockSelectionCallback = False

    def update_selection_from_maya(self):
        """
        Get list of selected in Maya and update selection in Outliner
        Driven by Maya-signal 'SelectionChanged'
        Callbask to that signal connected each time the Outliner shown
        """
        if self._blockSelectionCallback:
            return
        if HAVE_UFE:
            paths = []
            for item in ufe.GlobalSelection.get():
                if len(item.path().segments) == 2:
                    paths.append( str(item.path().segments[1]) )
            self.select_tree_items_by_path(paths)
        else:
            selected = pm.ls(type=pm.nt.AL_usdmaya_Transform, selection=1)
            qSelection = QtCore.QItemSelection()
            paths = [pm.getAttr(dagNode+'.pp').replace('|', '/') for dagNode in selected ]
            self.select_tree_items_by_path(paths)

    def setStageDataDirty(self, *args):
        setStageDataDirty(self._proxy_shape_name, True)


class ALProxyShapeModel(QtCore.QAbstractItemModel):
    """Model for ProxyShape Combobox.
    Stores maya nodes for AL_usdmaya_ProxyShape."""
    def __init__(self, parent=None):
        QtCore.QAbstractItemModel.__init__(self, parent)
        self._items = []

    def rowCount(self, parent):
        return len(self._items)

    def columnCount(self, parent):
        return 1

    def data(self, modelIndex, role=QtCore.Qt.DisplayRole):
        if role == QtCore.Qt.DisplayRole:
            if len(self._items) > 0 and self._items[modelIndex.row()]:
                return self._items[modelIndex.row()].name()

    def index(self, row, column, parent=QtCore.QModelIndex()):
        if -1 < row < len(self._items) and column == 0:
            return self.createIndex(row, column, self._items[row])
        return QtCore.QModelIndex()

    def parent(self, modelIndex):
        return QtCore.QModelIndex()

    # ---------------------Custom Methods--------------------- #

    def addItem(self, node):
        self._items.append(node)

    def item(self, row):
        if -1 < row < len(self._items):
            return self._items[row]

    def itemToRow(self, item):
        if item:
            return self._items.index(item)

    def clear(self):
        self._items = []

def find_model_in_children_nodes( node):
    if node.stripNamespace() == 'model':
        return node

    for node in node.getChildren():
        model = find_model_in_children_nodes(node)
        if model:
            return model


def find_model_node_for_prim(prim):
    model = None
    stage = prim.GetStage()
    attr_maya_node_name = ''

    # the right way to find model is:
    # find prim rig under asset prim
    # get metadata "customData" and field 'maya_associatedReferenceNode' from this custom data.
    kind = prim.GetMetadata('kind')
    if kind == 'assembly':
        rig_prim = stage.GetPrimAtPath(prim.GetPath().AppendChild('group_rig')) \
                   or stage.GetPrimAtPath(prim.GetPath().AppendChild('rig'))
    else:
        rig_prim = stage.GetPrimAtPath(prim.GetPath().AppendChild('rig'))
    if rig_prim:
        custom_data = rig_prim.GetMetadata('customData')
        if custom_data:
            attr_maya_node_name = custom_data.get('maya_associatedReferenceNode')

    if attr_maya_node_name:
        # If everything exists, we can get reference node, associatedNamespace
        # and under this namespace should be our model
        refs = pm.ls(attr_maya_node_name)
        if not refs:
            print("Couldn't find node with name " + str(attr_maya_node_name))
            return

        ref = refs[0]

        namespace = ref.associatedNamespace(True)
        nodes = pm.ls(namespace + '::model') #recursive search for namespaces
        if not nodes:
            print("Couldn't find model node")
            return
        if kind == 'assembly':
            model = nodes
        else:
            model = nodes[0]
    else:
        # if right way to find model didn't work
        # find node with name same as asset
        # find first appearing node with name model in children
        nodes = pm.ls(prim.GetName())
        if not nodes:
            print("Couldn't find node with name '{}'".format(prim.GetName()))
            return

        model = find_model_in_children_nodes(nodes[0])
        if not model:
            print("Couldn't find model node")
            return

    return model

class WIPCache(MenuAction):
    '''
    The "Create wip_cache" context menu item that calls command
    that will export current animation to local file
    for subsequent export in order to create a new animDisplayVariant,
    and includes this Variant.
    '''
    defaultText = 'Create wip_cache'

    def Update(self, action, context):
        prims = context.selectedPrims
        enable = len(prims) == 1 and prims[0].GetVariantSets().HasVariantSet('animDisplayVariant')
        action.setEnabled(enable)

    def Do(self):
        context = self.GetCurrentContext()
        if not context.selectedPrims:
            return
        prim = context.selectedPrims[0]
        
        animDisplayVariant = prim.GetVariantSets().AddVariantSet('animDisplayVariant')
        animDisplayVariant.SetVariantSelection('rig')
        rig = [child for child in prim.GetChildren() if child.GetName() == "rig"][0]

        file_name = context.proxy_shape.firstParent().name()
        file_name +=prim.GetPath().pathString.replace('/','_')+'.usd'
        project_dir = pm.workspace( q=True, rd=True )
        cache_path = os.path.join(project_dir, 'cache', file_name)

        model = find_model_node_for_prim(prim)
        if not model:
            return

        current_quality = None
        if 'quality' in pm.listAttr(pm.ls(model.namespace()+'main_control')[0]):
            quality = pm.ls(model.namespace()+'main_control.quality')[0]
            current_quality = quality.get()
        if current_quality:
            quality.set(0)

        import wizart.usd_utils.export_utils as export_utils
        export_utils.usd_export_filtered(model, prim.GetName(), cache_path)

        if current_quality is not None and current_quality != quality.get():
                quality.set(current_quality)

        animDisplayVariant.AddVariant('wip_cache')
        animDisplayVariant.SetVariantSelection('wip_cache')
        with animDisplayVariant.GetVariantEditContext():
            prim.GetReferences().AddReference(cache_path)
            rig.SetActive(False)
        

class ExportAssetToFile(MenuAction):
    defaultText = 'Export Asset Cache To File ...'

    def Update(self, action, context):
        action.setEnabled(len(context.selectedPrims) == 1)

    def Do(self):
        context = self.GetCurrentContext()
        if not context.selectedPrims:
            return
        prim = context.selectedPrims[0]
        kind = prim.GetMetadata('kind')
        save_path, ext = \
            QtWidgets.QFileDialog.getSaveFileName(context.qtParent, "Export All", '',
                                                  "*.usd *.usda *.usdc *.usdz;;*.usd;;*.usda;;*.usdc;;*.usdz")
        if not save_path:
            return

        model = find_model_node_for_prim(prim)
        if model:
            import wizart.usd_utils.export_utils as export_utils
            if kind == 'assembly':
                export_utils.usd_export_filtered_compound(model, prim.GetName(), save_path)
            else:
                export_utils.usd_export_filtered(model, prim.GetName(), save_path)


class ExportAsset(MenuAction):
    """
    Export of asset anim cache specially for animation department.
    """
    defaultText = 'Export Asset'

    def Update(self, action, context):
        workspace = os.environ.get('WORKSPACE')
        action.setEnabled(len(context.selectedPrims) == 1 and bool(workspace))

    def Do(self):
        context = self.GetCurrentContext()
        if not context.selectedPrims:
            return
        from wizart.fs_ctrl import fs_schema

        schema = fs_schema.ProjectFileSchema(os.environ["PROJECT_NAME"], os.environ["PROJECT_ROOT_PATH"])
        schema.load_components()
        prim = context.selectedPrims[0]
        kind = prim.GetMetadata('kind')

        workspace = os.environ.get('WORKSPACE')
        component_name = 'cache.asset.anim'
        asset = prim.GetName()

        _, resolve_dict = schema.find_node_by_path(prim.GetStage().GetRootLayer().realPath, return_resolved_vars=True)

        resolve_dict['asset'] = asset

        wref_path, file_name = schema.get_wref_path_to_file(component_name, **resolve_dict)

        save_path = os.path.join(workspace, file_name)

        if not save_path:
            return

        model = find_model_node_for_prim(prim)
        if model:
            import wizart.usd_utils.export_utils as export_utils

            if kind == 'assembly':
                export_utils.usd_export_filtered_compound(model, prim.GetName(), save_path)
            else:
                export_utils.usd_export_filtered(model, prim.GetName(), save_path)

            import wizart.desktop.workspace_utils as wutils
            wutils.repath(wref_path, os.path.join('$WORKSPACE', file_name))


class PushToMaya(MenuAction):
    defaultText = 'Push to Maya'

    def Update(self, action, context):
        action.setEnabled(len(context.selectedPrims) >= 1)

    def Do(self):
        context = self.GetCurrentContext()
        if not context.selectedPrims:
            return
        import AL.usdmaya
        prims = context.selectedPrims

        proxy_shape = AL.usdmaya.ProxyShape.getByName(context.proxy_shape.name())
        proxy_node = context.proxy_shape.parent(i=0)
        #collect in set in case we traverse duplicate paths
        mesh_prim_set = set()
        for prim in prims:
            for child_prim in Usd.PrimRange(prim): #recursive traverse
                if child_prim.GetTypeName() == "Mesh":
                    mesh_prim_set.add(child_prim)
        
        for prim in mesh_prim_set:
            pm.select(proxy_node.name())
            pm.AL_usdmaya_TranslatePrim(proxy=str(proxy_node.name()), fi=True, ip=str(prim.GetPath()))

            # check are points or topology animated
            points_attr = prim.GetAttribute('points')
            vertex_counts_attr = prim.GetAttribute('faceVertexCounts')
            vertex_indixes_attr = prim.GetAttribute('faceVertexIndices')
            is_points_animated = points_attr.GetNumTimeSamples() > 0
            is_topology_animated = (vertex_counts_attr.GetNumTimeSamples() > 0) \
                or (vertex_indixes_attr.GetNumTimeSamples() > 0)
            if not is_topology_animated and not is_points_animated:
                continue

            transform_node_name = proxy_shape.getMayaPathFromUsdPrim(prim)
            if not transform_node_name:
                print("Can't find node {}".format(transform_node_name))
                continue
            mesh_nodes = pm.listRelatives(transform_node_name, type='mesh')
            if not mesh_nodes:
                print("Translation failed, can't find the mesh")
                continue
            mesh_node = mesh_nodes[0]
            cs = pm.listConnections('%s.inStageData' % transform_node_name)[0]

            # if only points animated - add deformer AL_usdmaya_MeshAnimDeformer
            if is_points_animated and not is_topology_animated:
                dup_mesh = pm.duplicate(mesh_node, addShape=True)[0]
                deformer_node = pm.createNode('AL_usdmaya_MeshAnimDeformer')
                pm.setAttr('%s.primPath' % deformer_node, prim.GetPrimPath(),
                           type='string')
                pm.connectAttr('%s.outStageData' % cs,
                               '%s.inStageData' % deformer_node)
                pm.connectAttr('%s.outMesh' % dup_mesh.name(),
                               '%s.inMesh' % deformer_node)
                pm.connectAttr('%s.outMesh' % deformer_node,
                               '%s.inMesh' % mesh_node)
                pm.connectAttr('%s.outTime' % proxy_node.name(),
                               '%s.inTime' % deformer_node)
                pm.setAttr('%s.intermediateObject' % dup_mesh.name(), 1)

            # if animated topology
            if is_topology_animated:
                creator_node = pm.createNode('AL_usdmaya_MeshAnimCreator')
                pm.setAttr('%s.primPath' % creator_node, prim.GetPrimPath(),
                           type='string')
                pm.connectAttr('%s.outStageData' % cs,
                               '%s.inStageData' % creator_node)
                pm.connectAttr('%s.outMesh' % creator_node,
                               '%s.inMesh' % mesh_node.name())


def show_ensure_edit_target_message(context):
    current_edit_target = context.stage.GetEditTarget().GetLayer().GetDisplayName()
    buttons = QtWidgets.QMessageBox.Yes | QtWidgets.QMessageBox.Cancel
    answer = QtWidgets.QMessageBox.question(
        context.qtParent,
        'Attention! Confirm Edit Target',
        'Current edit target: {}. Do you want to store all changes on this layer?'.format(current_edit_target),
        buttons=buttons,
        defaultButton=QtWidgets.QMessageBox.Yes)
    if answer == QtWidgets.QMessageBox.Cancel:
        return False
    elif answer == QtWidgets.QMessageBox.Yes:
        return True


class PushToUsd(MenuAction):
    defaultText = 'Push to Usd'

    def Update(self, action, context):
        action.setEnabled(len(context.selectedPrims) >= 1)

    def Do(self):
        context = self.GetCurrentContext()
        if not context.selectedPrims:
            return
        answer = show_ensure_edit_target_message(context)
        if not answer:
            return
        prims = context.selectedPrims
        #collect in set in case we traverse duplicate paths
        mesh_prim_set = set()
        for prim in prims:
            for child_prim in Usd.PrimRange(prim): #recursive traverse
                if child_prim.GetTypeName() == "Mesh":
                    mesh_prim_set.add(child_prim)
        

        proxy_node = pm.listRelatives(context.proxy_shape.name(), p=True)[0]
        prims_to_translate = ','.join([str(prim.GetPath()) for prim in mesh_prim_set])
        pm.AL_usdmaya_TranslatePrim(tp=prims_to_translate, proxy=str(proxy_node.name()))


class PushFramerangeToUsd(MenuAction):
    defaultText = 'Push Framerange to Usd'

    def Update(self, action, context):
        action.setEnabled(len(context.selectedPrims) >= 1)

    def Do(self):
        context = self.GetCurrentContext()
        if not context.selectedPrims:
            return
        answer = show_ensure_edit_target_message(context)
        if not answer:
            return
        prims = context.selectedPrims
        proxy_node = pm.listRelatives(context.proxy_shape.name(), p=True)[0]
        start = context.stage.GetStartTimeCode()
        end = context.stage.GetEndTimeCode()

        #collect in set in case we traverse duplicate paths
        mesh_prim_set = set()
        for prim in prims:
            for child_prim in Usd.PrimRange(prim): #recursive traverse
                if child_prim.GetTypeName() == "Mesh":
                    mesh_prim_set.add(child_prim)

        prims_to_translate = ','.join([str(prim.GetPath()) for prim in mesh_prim_set])
        pm.AL_usdmaya_TranslatePrim(tp=prims_to_translate,
                                    fr=(start, end),
                                    frs=(-0.2, 0.0, 0.2),
                                    proxy=str(proxy_node.name()))


class SetEditTarget(MenuAction):
    defaultText = 'Set Edit Target'

    def Update(self, action, context):
        action.setEnabled(len(context.selectedPrims) == 1)

    def Do(self):
        context = self.GetCurrentContext()
        if not context.selectedPrims:
            return
        context.qtParent.select_widget = MayaCompositionEditor(context.selectedPrims[0], context.stage)
        context.qtParent.select_widget.show()

class BlendTPoseRig(MenuAction):
    defaultText = 'Blend T Pose for Rig'

    def Update(self, action, context):
        import wizart.anim_utils.bake_anim
        if context.selectedPrims:
            ref_prim = [child for child in context.selectedPrims[0].GetChildren() if child.GetName() == "rig" ]
            if ref_prim:
                ref_prim = ref_prim[0]
                if ref_prim.GetTypeName() == "ALMayaReference":
                    custom_data = ref_prim.GetMetadata("customData")
                    if custom_data and custom_data.get('maya_associatedReferenceNode'):
                        ref_node = pm.ls(custom_data.get('maya_associatedReferenceNode'))
                        if ref_node:
                            ref = pm.FileReference(ref_node[0])
                            res = wizart.anim_utils.bake_anim.is_t_pose_possible(ref)
                            action.setEnabled(res)
                            return
        action.setEnabled(False)

    def Do(self):
        import wizart.anim_utils.bake_anim
        context = self.GetCurrentContext()
        if not context.selectedPrims:
            return
        if context.selectedPrims:
            for prim in context.selectedPrims:
                ref_prim = [child for child in prim.GetChildren() if child.GetName() == "rig" ]
                if ref_prim and ref_prim[0].GetTypeName() == "ALMayaReference":
                    custom_data = ref_prim[0].GetMetadata("customData")
                    if custom_data and custom_data.get('maya_associatedReferenceNode'):
                        ref_node = pm.ls(custom_data.get('maya_associatedReferenceNode'))
                        if ref_node:
                            ref = pm.FileReference(ref_node[0])
                            start_time = pm.playbackOptions(q=True, min=True)
                            end_time = pm.playbackOptions(q=True, max=True)
                            wizart.anim_utils.bake_anim.add_t_pose_blend(ref, start_time, end_time)

class MayaOutlinerRole(OutlinerRole):
    """Helper which provides standard hooks for defining the context menu
    actions and menu bar menus that should be added to an outliner.
    """
    @classmethod
    def GetContextMenuActions(cls, outliner):
        # type: (UsdOutliner) -> List[Union[MenuAction, Type[MenuAction]]]
        """
        Parameters
        ----------
        outliner : UsdOutliner

        Returns
        -------
        List[Union[MenuAction, Type[MenuAction]]]
        """
        if os.environ.get('WIZART_FREELANCE') == '1':
            return OutlinerRole.GetContextMenuActions(outliner)
        else:
            role = OutlinerRole.GetContextMenuActions(outliner)
        return role + [MenuSeparator, WIPCache] + \
            [MenuSeparator, ExportAssetToFile, ExportAsset, BlendTPoseRig] + \
            [MenuSeparator, SetEditTarget] + \
            [MenuSeparator, PushToMaya, PushToUsd, PushFramerangeToUsd]


class MayaUsdOutliner(UsdOutliner):
    """UsdStage editing application which displays the hierarchy of a stage."""
    def __init__(self, stage=None, role = None, parent=None):
        """
        Parameters
        ----------
        stage : Usd.Stage
        role : Optional[Union[Type[OutlinerRole], OutlinerRole]]
        parent : Optional[QtGui.QWidget]
        """
        self._layer_widget = LayerWidget()
        UsdOutliner.__init__(self, stage, role, parent)
        self.setStyle()
        self._splitter = QtWidgets.QSplitter(QtCore.Qt.Vertical, self)
        self.layout.addWidget(self._splitter)

        self._splitter.addWidget(self.view)

        self._splitter.addWidget(self._layer_widget)
        self._layer_widget.setVisible(False)
        self._create_menu_bar()

    def _CreateView(self, role):
        """Create the hierarchy view for the outliner.

        This is provided as a convenience for subclass implementations.

        Parameters
        ----------
        stage : Usd.Stage
        role : Union[Type[OutlinerRole], OutlinerRole]

        Returns
        -------
        QtWidgets.QAbstractItemView
        """
        return OutlinerTreeView(
            self._dataModel,
            contextMenuActions=role.GetContextMenuActions(self),
            contextProvider=self,
            parent=self)
    
    def SetStage(self, stage = None):
        UsdOutliner.SetStage(self, stage)
        self._layer_widget.set_stage(stage)

    def _create_menu_bar(self):
        UsdOutliner._create_menu_bar(self)
        self._menubar = self.layout.menuBar()
        if not self._menubar:
            self._menubar = QtWidgets.QMenuBar()
            self.layout.setMenuBar(self._menubar)

        self._menu_tools = QtWidgets.QMenu("&Tools", self._menubar)
        self._menubar.addMenu(self._menu_tools)


        act_show_lt = QtWidgets.QWidgetAction(self)
        checkBox = QtWidgets.QCheckBox('Layer Editor', self)
        act_show_lt.setDefaultWidget(checkBox)
        checkBox.setChecked(False)
        checkBox.toggled.connect(self._layer_widget.setVisible)

        self._menu_tools.addAction(act_show_lt)


MayaOutlinerContext = namedtuple('MayaOutlinerContext',
                             ['qtParent', 'outliner', 'stage',
                              'editTargetLayer', 'selectedPrim',
                              'selectedPrims', 'proxy_shape'])
    

class ALMayaUsdOutliner(UsdOutliner):
    """UsdStage editing application which displays the hierarchy of a stage."""
    def __init__(self, role=None, parent=None):
        """
        Parameters
        ----------
        stage : Usd.Stage
        role : Optional[Union[Type[OutlinerRole], OutlinerRole]]
        parent : Optional[QtGui.QWidget]
        """
        maya_undo.init_usd_qt_undo()
        self._undo_edit_target_listener = None
        self._edit_target_listener = None
        self._maya_proxy_node = None
        self._layer_widget = MayaLayerWidget()
        if not role:
            role = MayaOutlinerRole
        UsdOutliner.__init__(self, None, role, parent)

        self._create_menu_bar()
        self.add_callbacks()
        self.mayaSelectionChanged()
        self.setStyle()

        self._proxy_layout = QtWidgets.QHBoxLayout(self)
        self.layout.insertLayout(0, self._proxy_layout)
        self._proxy_combobox = QtWidgets.QComboBox()
        self._proxy_layout.addWidget(self._proxy_combobox)

        combobox_size_policy = self._proxy_combobox.sizePolicy()
        combobox_size_policy.setHorizontalStretch(1)
        self._proxy_combobox.setSizePolicy(combobox_size_policy)

        self._refresh_button = QtWidgets.QPushButton()
        self._refresh_button.setIcon(QtGui.QIcon(":/icons/refresh"))
        self._refresh_button.setToolTip('Refresh')
        self._refresh_button.pressed.connect(self._refresh)
        self._proxy_layout.addWidget(self._refresh_button)
        self._refresh_button.setFlat(True)

        splitter = QtWidgets.QSplitter(QtCore.Qt.Horizontal)
        splitter_size_policy = splitter.sizePolicy()
        splitter_size_policy.setHorizontalStretch(2)
        splitter.setSizePolicy(splitter_size_policy)
        self._proxy_layout.addWidget(splitter)

        self.update_proxies_list()
        self.destroyed.connect(lambda: ALMayaUsdOutliner.remove_callbacks(self))

        self._splitter = QtWidgets.QSplitter(QtCore.Qt.Vertical, self)
        self.layout.addWidget(self._splitter)

        self._splitter.addWidget(self.view)

        self._splitter.addWidget(self._layer_widget)
        self._layer_widget.setVisible(False)

    def update_proxies_list(self, *args):
        """Update combobox for selecting ProxyShapes
        by getting all nodes with type AL_usdmaya_ProxyShape and filling the model"""
        current_index = self._proxy_combobox.currentIndex()
        current_node = self._proxy_combobox.model().item(current_index)
        self._proxy_combobox.clear()
        proxyShape = pm.ls(type=AL_PROXY_SHAPE_TYPE)

        model = ALProxyShapeModel()
        for proxy in proxyShape:
            model.addItem(proxy)

        try:
            self._proxy_combobox.currentIndexChanged.disconnect(self._set_proxy)
        except:
            # then it was not connected
            pass

        self._proxy_combobox.setModel(model)

        new_current_index = self._proxy_combobox.model().itemToRow(current_node)
        if (new_current_index is None or (isinstance(new_current_index, int) and new_current_index < 0)  ) \
                and len(proxyShape) > 0:
            new_current_index = 0

        if new_current_index >= 0:
            self._proxy_combobox.setCurrentIndex(new_current_index)
            self._set_proxy(new_current_index)

        self._proxy_combobox.currentIndexChanged.connect(self._set_proxy)

    def mayaSelectionChanged(self, *args):
        self.view.update_selection_from_maya()

    @classmethod
    def get_stage_to_proxy(self, maya_proxy_node):
        import AL.usdmaya
        proxy = pm.listRelatives(maya_proxy_node.name(), parent=True, type='transform')
        if proxy:
            # commonly there's only one, so takes first
            proxy_shape = AL.usdmaya.ProxyShape.getByName(proxy[0].name())
            if proxy_shape:
                return proxy_shape.getUsdStage()

    def _set_proxy(self, index):
        proxy = self._proxy_combobox.model().item(index)
        if proxy:
            stage = self.get_stage_to_proxy(proxy)
            UsdOutliner.SetStage(self, stage)
            self.view.set_proxy_shape_name(proxy.name())
            self._layer_widget.set_stage(stage)
            if stage:
                maya_undo.init_undo_for_stage(stage)
                self._undo_edit_target_listener = Tf.Notice.Register(Usd.Notice.StageEditTargetChanged,
                                                                     maya_undo._edit_target_changed_callback, stage)
                self._edit_target_listener = Tf.Notice.Register(Usd.Notice.StageEditTargetChanged,
                                                                self.edit_target_changed, stage)
            else:
                self._undo_edit_target_listener = None
                self._edit_target_listener = None
        else:
            UsdOutliner.SetStage(self, None)
            self._layer_widget.set_stage(None)
            self._undo_edit_target_listener = None
            self._edit_target_listener = None

    def _refresh(self, *args):
        """Refresh proxies combobox in case of renames
        and update hierarchy model after stage was loaded"""
        self.update_proxies_list()

    def _parse_scene_path(self, path):
        path = os.path.normpath(path)
        path = path.replace('\\', '/')
        from wizart import prodtrack_api
        expr = prodtrack_api.get_project_config()['scene_name_exp']
        match = re.search(expr, path)
        if match:
            if os.environ.get("PROJECT_IS_SERIES", "0") == "1":
                return dict( series = match.groupdict().get('series'),  episode = match.groupdict().get('episode'), scene = match.groupdict().get('scene') )
            else:
                return dict( episode = match.groupdict().get('episode'), scene = match.groupdict().get('scene') )
        else:
            print("Error parsing episode and scene from path {}".format(path))
            return None
    
    def _publish_layers(self):
        import pyblish.api
        ctx = pyblish.api.Context()
        ctx.data['explicit_context'] = True
        import pyblish_lite
        pyblish_lite.show(context = ctx)
    
    def _import_edit_camera(self):
        # get path to current scene, parse and get episode and scene
        index = self._proxy_combobox.currentIndex()
        proxy = self._proxy_combobox.model().item(index)
        if not proxy:
            return

        stage = self.get_stage_to_proxy(proxy)
        stage_path = stage.GetRootLayer().identifier

        shot_name_bits = self._parse_scene_path(stage_path)
        if not shot_name_bits:
            return

        # find path on camera
        from wizart.fs_ctrl import fs_schema
        self.schema = fs_schema.ProjectFileSchema(os.environ["PROJECT_NAME"], os.environ["PROJECT_ROOT_PATH"])
        self.schema.load_components()

        if os.environ["PROJECT_NAME"] == "sq":
            episode = "ep_{}".format(episode)
            scene = "sc_{}".format(scene)

        camera_wref_path, file_name = self.schema.get_wref_path_to_file(component_name='cam', **shot_name_bits)
        namespace, _ = os.path.splitext(file_name)

        nodes = pm.importFile(camera_wref_path, namespace="EDIT_CAMERA", returnNewNodes = True)
        for node in nodes:
            if node.type() == "camera":
                #highlight top_level_transform in outiliner
                top_camera_transform = node.getAllParents().pop()
                top_camera_transform.useOutlinerColor.set(True)
                top_camera_transform.outlinerColor.set((1,0,0))
                break

    def _invalidate(self, *args):
        self._set_proxy(-1)
        self._proxy_combobox.model().clear()
        self.remove_node_callbacks()

    def _CreateView(self, role):
        """Create the hierarchy view for the outliner.

        This is provided as a convenience for subclass implementations.

        Parameters
        ----------
        stage : Usd.Stage
        role : Union[Type[OutlinerRole], OutlinerRole]

        Returns
        -------
        QtWidgets.QAbstractItemView
        """
        return ALMayaOutlinerTreeView(
            self._dataModel,
            proxy_shape_name=None,
            contextMenuActions=role.GetContextMenuActions(self),
            contextProvider=self,
            parent=self)

    def add_callbacks(self):
        # on selection
        self.selectionCallbackId = \
            pm.api.MEventMessage.addEventCallback("UFESelectionChanged" if HAVE_UFE else "SelectionChanged",
                                                  self.mayaSelectionChanged)

        self.add_node_callbacks()

        # on open new maya scene
        self.beforeOpenCallbackId = pm.api.MSceneMessage.addCallback(pm.api.MSceneMessage.kBeforeOpen,
                                                                     self._invalidate)
        self.beforeNewCallbackId = pm.api.MSceneMessage.addCallback(pm.api.MSceneMessage.kBeforeNew,
                                                                    self._invalidate)

        # after open new maya scene, we need to recreate callbacks on adding maya node
        self.afterOpenCallbackId = pm.api.MSceneMessage.addCallback(pm.api.MSceneMessage.kAfterOpen,
                                                                     self.add_node_callbacks)
        self.afterNewCallbackId = pm.api.MSceneMessage.addCallback(pm.api.MSceneMessage.kAfterNew,
                                                                    self.add_node_callbacks)

    def add_node_callbacks(self, *args):
        # create or remove maya node Al_usdmaya_ProxyShape
        self.proxyAddedCallbackId = \
            pm.api.MDGMessage.addNodeAddedCallback(self.update_proxies_list, AL_PROXY_SHAPE_TYPE)

        self.proxyRemovedCallbackId = \
            pm.api.MDGMessage.addNodeRemovedCallback(self.update_proxies_list, AL_PROXY_SHAPE_TYPE)

    def remove_node_callbacks(self):
        if self.proxyAddedCallbackId:
            pm.api.MMessage.removeCallback(self.proxyAddedCallbackId)
            self.proxyAddedCallbackId = None
        if self.proxyRemovedCallbackId:
            pm.api.MMessage.removeCallback(self.proxyRemovedCallbackId)
            self.proxyRemovedCallbackId = None

    @staticmethod
    def remove_callbacks(self):
        """Remove callbacks"""
        if self.selectionCallbackId:
            pm.api.MMessage.removeCallback(self.selectionCallbackId)
            self.selectionCallbackId = None

        ALMayaUsdOutliner.remove_node_callbacks(self)

        if self.beforeOpenCallbackId:
            pm.api.MMessage.removeCallback(self.beforeOpenCallbackId)
            self.beforeOpenCallbackId = None
        if self.beforeNewCallbackId:
            pm.api.MMessage.removeCallback(self.beforeNewCallbackId)
            self.beforeNewCallbackId = None

        if self.afterOpenCallbackId:
            pm.api.MMessage.removeCallback(self.afterOpenCallbackId)
            self.afterOpenCallbackId = None
        if self.afterNewCallbackId:
            pm.api.MMessage.removeCallback(self.afterNewCallbackId)
            self.afterNewCallbackId = None

        self._undo_edit_target_listener = None
        self._edit_target_listener = None

        self._dataModel._Invalidate()

    def _create_menu_bar(self):
        self._filterModel.set_show_flags(False, False, False)
        UsdOutliner._create_menu_bar(self)
        self._menubar = self.layout.menuBar()
        if not self._menubar:
            self._menubar = QtWidgets.QMenuBar()
            self.layout.setMenuBar(self._menubar)

        self._menu_tools = QtWidgets.QMenu("&Tools", self._menubar)
        self._menubar.addMenu(self._menu_tools)

        act_show_lt = QtWidgets.QWidgetAction(self)
        checkBox = QtWidgets.QCheckBox('Layer Editor', self)
        act_show_lt.setDefaultWidget(checkBox)
        checkBox.setChecked(False)
        checkBox.toggled.connect(self._layer_widget.setVisible)
        self._menu_tools.addAction(act_show_lt)

        self._menu_actions = QtWidgets.QMenu("&Actions", self._menubar)
        self._menubar.addMenu(self._menu_actions)
        act_import_camera = QtWidgets.QAction('Import Editable Camera', self)
        act_import_camera.triggered.connect(self._import_edit_camera)
        self._menu_actions.addAction(act_import_camera)
        self._menu_actions.addSeparator()
        act_publish_layers = QtWidgets.QAction('Publish Layers', self)
        act_publish_layers.triggered.connect(self._publish_layers)
        self._menu_actions.addAction(act_publish_layers)
        
    def GetMenuContext(self):
        # type: () -> OutlinerContext
        """
        Returns
        -------
        OutlinerContext
        """
        selectedPrims = self.view.SelectedPrims()
        selectedPrim = selectedPrims[0] if selectedPrims else None
        return MayaOutlinerContext(qtParent=self, outliner=self, stage=self._stage,
                               editTargetLayer=self.GetEditTargetLayer(),
                               selectedPrim=selectedPrim,
                               selectedPrims=selectedPrims,
                               proxy_shape=self._proxy_combobox.model().item(self._proxy_combobox.currentIndex()))

    def edit_target_changed(self, *args):
        if self._stage:
            edit_target_layer = self._stage.GetEditTarget().GetLayer()
            self.UpdateTitle(edit_target_layer.identifier)
        else:
            self.UpdateTitle(None)


def run_from_maya_alusd(maya_proxy_node=None):
    """
    In case driven in maya use this to make Dockable Maya Window
    :param stage: Usd.Stage
    :param proxy: proxyShape
    :return: ALMayaUsdDockableOutliner instance
    """
    if not pm.pluginInfo("AL_USDMayaPlugin", query=True, loaded=True):
        pm.loadPlugin("AL_USDMayaPlugin")
    if not pm.pluginInfo("AL_USDMayaPxrTranslators", query=True, loaded=True):
        pm.loadPlugin("AL_USDMayaPxrTranslators")

    from maya.app.general.mayaMixin import MayaQWidgetDockableMixin

    class ALMayaUsdDockableOutliner(MayaQWidgetDockableMixin, ALMayaUsdOutliner):
        def __init__(self, parent=None):
            ALMayaUsdOutliner.__init__(self, parent)
            MayaQWidgetDockableMixin._initForMaya(self, parent)
            self.setAttribute(QtCore.Qt.WA_DeleteOnClose, True)

        def dockCloseEventTriggered(self):
            ALMayaUsdOutliner.close(self)


    window = ALMayaUsdDockableOutliner()
    window.show(dockable=True)

    return window

def run_from_maya(stage=None, role = None):
    """
    In case driven in maya use this to make Dockable Maya Window
    :param stage: Usd.Stage
    :return: MayaUsdDockableOutliner instance
    """

    from maya.app.general.mayaMixin import MayaQWidgetDockableMixin

    class MayaUsdDockableOutliner(MayaQWidgetDockableMixin, MayaUsdOutliner):
        def __init__(self, stage = None, role = None, parent=None):
            MayaUsdOutliner.__init__(self, stage, role, parent)
            MayaQWidgetDockableMixin._initForMaya(self, parent)
            self.setAttribute(QtCore.Qt.WA_DeleteOnClose, True)


    window = MayaUsdDockableOutliner(stage, role)
    window.show(dockable=True)

    return window

if __name__ == '__main__':
    # simple test
    import sys

    app = QtWidgets.QApplication(sys.argv)

    usdFileArg = sys.argv[1]
    #usdFileArg = 'd:/Work/USD/Kitchen_set/Kitchen_set.usd'

    window = ALMayaUsdOutliner.FromUsdFile(usdFileArg)
    window.show()
    sys.exit(app.exec_())
