"""
Module contains customized wizOutliner which has additional options in Max
"""
from pxr.UsdQtEditors._Qt import QtCore, QtGui, QtWidgets
from pxr import Sdf, Tf, Usd, UsdGeom
from outliner import UsdOutliner, OutlinerTreeView, OutlinerRole, CopyPrimPath
from pxr.UsdQtEditors.outliner import SaveState, MenuBuilder, SaveEditLayer, ShowEditTargetLayerText,\
    ShowEditTargetDialog, MenuSeparator
from pxr.UsdQt.qtUtils import MenuAction
from widgets.layer_stack_widget.layer_widget import LayerWidget
import resources.resources
import usd_max.import_utils as max_utils
import pymxs


class TranslateSelectedction(MenuAction):
    defaultText = 'Translate Selected'

    def Update(self, action, context):
        action.setEnabled(bool(context.selectedPrims))

    def Do(self):
        context = self.GetCurrentContext()
        for prim in context.selectedPrims:
            max_utils.translate_subtree(prim)
#
# class AttachSelectedAction(MenuAction):
#     defaultText = 'Attach to Max Objects'
#
#     def Update(self, action, context):
#         action.setEnabled(bool(context.selectedPrims))
#
#     def Do(self):
#         context = self.GetCurrentContext()
#         max_utils.attach_subtree(context.selectedPrims[0])


class AttachSelectedSinglePrimAction(MenuAction):
    defaultText = 'Attach to Single Max Object'

    def Update(self, action, context):
        action.setEnabled(bool(context.selectedPrims) and bool( list(pymxs.runtime.selection)) )

    def Do(self):
        context = self.GetCurrentContext()
        max_utils.attach_single_prim(context.selectedPrims[0], pymxs.runtime.selection[0])


class AttachSubtreeToSelected(MenuAction):
    defaultText = 'Attach to Max Hierarchy'

    def Update(self, action, context):
        action.setEnabled(bool(context.selectedPrims) and bool(list(pymxs.runtime.selection)))

    def Do(self):
        context = self.GetCurrentContext()
        max_utils.attach_subtree_to_selected(context.selectedPrims[0], pymxs.runtime.selection[0])


class MaxOutlinerRole(OutlinerRole):
    """Helper which provides standard hooks for defining the context menu
    actions and menu bar menus that should be added to an outliner.
    """
    @classmethod
    def GetContextMenuActions(cls, outliner):
        # type: (UsdOutliner) -> List[Union[MenuAction, Type[MenuAction]]]
        """
        Parameters
        ----------
        outliner : UsdOutliner

        Returns
        -------
        List[Union[MenuAction, Type[MenuAction]]]
        """
        return [CopyPrimPath, MenuSeparator, TranslateSelectedction,
                AttachSelectedSinglePrimAction, AttachSubtreeToSelected]


class MaxUsdOutliner(UsdOutliner):
    """UsdStage editing application which displays the hierarchy of a stage."""
    def __init__(self, role=None, parent=None):
        """
        Parameters
        ----------
        stage : Usd.Stage
        role : Optional[Union[Type[OutlinerRole], OutlinerRole]]
        parent : Optional[QtGui.QWidget]
        """
        self._layer_widget = LayerWidget()
        if not role:
            role = MaxOutlinerRole
        UsdOutliner.__init__(self, None, role, parent)
        self._create_menu_bar()

        self._splitter = QtWidgets.QSplitter(QtCore.Qt.Vertical, self)
        self.layout.addWidget(self._splitter)

        self._splitter.addWidget(self.view)

        self._splitter.addWidget(self._layer_widget)
        self._layer_widget.setVisible(False)

    def _CreateView(self, role):
        """Create the hierarchy view for the outliner.

        This is provided as a convenience for subclass implementations.

        Parameters
        ----------
        stage : Usd.Stage
        role : Union[Type[OutlinerRole], OutlinerRole]

        Returns
        -------
        QtWidgets.QAbstractItemView
        """
        return OutlinerTreeView(
            self._dataModel,
            contextMenuActions=role.GetContextMenuActions(self),
            contextProvider=self,
            parent=self)

    def _create_menu_bar(self):
        UsdOutliner._create_menu_bar(self)
        self._menubar = self.layout.menuBar()
        if not self._menubar:
            self._menubar = QtWidgets.QMenuBar()
            self.layout.setMenuBar(self._menubar)

        self._menu_file = QtWidgets.QMenu("&File", self._menubar)
        self._menubar.addMenu(self._menu_file)
        act_open_stage = self._menu_file.addAction('Open Stage', self.OpenStage)

        self._menu_tools = QtWidgets.QMenu("&Tools", self._menubar)
        self._menubar.addMenu(self._menu_tools)


        act_show_lt = QtWidgets.QWidgetAction(self)
        checkBox = QtWidgets.QCheckBox('Layer Editor', self)
        act_show_lt.setDefaultWidget(checkBox)
        checkBox.setChecked(False)
        checkBox.toggled.connect(self._layer_widget.setVisible)

        self._menu_tools.addAction(act_show_lt)
       
    def OpenStage(self):
        #PySide2 return tuple ?
        scene_path , _ = QtWidgets.QFileDialog.getOpenFileName(self, "Open Stage", "", "*.usd *.usda *.usdc *.usdz;;*.usd;;*.usda;;*.usdc;;*.usdz")
        if scene_path:
            with Usd.StageCacheContext(Usd.BlockStageCaches):
                stage = Usd.Stage.Open(scene_path)
                assert stage, 'Failed to open stage'
                stage.SetEditTarget(stage.GetSessionLayer())
                self.SetStage(stage)
    def SetStage(self, stage):
        UsdOutliner.SetStage(self, stage)
        self._layer_widget.set_stage(stage)


def run_from_max():
    """
    In case driven in max
    """
    import MaxPlus
    mainWindow = MaxPlus.GetQMaxMainWindow()

    dockWidget = QtWidgets.QDockWidget(mainWindow)
    outliner = MaxUsdOutliner()
    dockWidget.setWidget(outliner)
    dockWidget.setObjectName("USD_Outliner")
    dockWidget.setWindowTitle("USD Outliner")
    mainWindow.addDockWidget(QtCore.Qt.LeftDockWidgetArea, dockWidget)
    dockWidget.setFloating(True)
    dockWidget.show()
    return dockWidget

if __name__ == '__main__':
    # simple test
    import sys

    app = QtWidgets.QApplication(sys.argv)

    usdFileArg = sys.argv[1]
    #usdFileArg = 'd:/Work/USD/Kitchen_set/Kitchen_set.usd'

    window = MaxUsdOutliner.FromUsdFile(usdFileArg)
    window.show()
    sys.exit(app.exec_())