"""
Module contains customized dccOutliner which has additional opportunities in DCC, such as
- synchronized selection
"""
from pxr.UsdQtEditors._Qt import QtCore, QtGui, QtWidgets
from pxr.UsdQt.qtUtils import MenuAction, MenuSeparator
from pxr import Sdf, Tf, Usd, UsdGeom

import wizart.dcc.core as dcc_core
from wizart.dcc import cmds
from dcc_core import UsdViewportRefineManager

from .i18n import i18n
from .outliner import UsdOutliner, OutlinerTreeView, OutlinerRole
from .modelColumn import RefineLevelColumn, DrawModeColumn


class ClearRefineAtPrim(MenuAction):
    defaultText = i18n("outliner", "Clear refine for prims")

    def Update(self, action, context):
        action.setEnabled(bool(context.selectedPrims))

    def Do(self):
        context = self.GetCurrentContext()

        refine_manager = UsdViewportRefineManager.instance()
        app = dcc_core.Application.instance()
        session = app.get_session()
        stage = session.get_current_stage()
        for prim in context.selectedPrims:
            refine_manager.clear_refine_level(stage, prim.GetPath())


class ClearAllRefine(MenuAction):
    defaultText = i18n("dcc_outliner", "Clear all refine")

    def Do(self):
        context = self.GetCurrentContext()

        refine_manager = UsdViewportRefineManager.instance()
        app = dcc_core.Application.instance()
        session = app.get_session()
        stage = session.get_current_stage()
        refine_manager.clear_stage(stage)


class DccOutlinerTreeView(OutlinerTreeView):

    primSelectionChanged = QtCore.Signal(list, list)

    def __init__(self, dataModel, contextMenuActions, contextProvider=None,
                 parent=None):
        # type: (QtCore.QAbstractItemModel, List[MenuAction], Optional[ContextProvider], Optional[QtWidgets.QWidget]) -> None
        """
        Parameters
        ----------
        dataModel : QtCore.QAbstractItemModel
        contextMenuActions : List[MenuAction]
        contextProvider : Optional[ContextProvider]
        parent : Optional[QtWidgets.QWidget]
        """
        OutlinerTreeView.__init__(self,
                                        dataModel,
                                        contextMenuActions=contextMenuActions,
                                        contextProvider=contextProvider,
                                        parent=parent)
        refine_column = self.model().add_column(RefineLevelColumn, "Refine", i18n("dcc_outliner", "Refine"))
        self.hideColumn(self.model().columns_controller().index("Refine"))

        refine_column = self.model().add_column(DrawModeColumn, "DrawMode", i18n("dcc_outliner", "DrawMode"))
        self.hideColumn(self.model().columns_controller().index("DrawMode"))

        self._blockSelectionCallback = False

        self.primSelectionChanged.connect(self.change_selection_into_dcc)

        self.app = dcc_core.Application.instance()
        self._sharedLayerTextEditors = {}

    def change_selection_into_dcc(self, selected, deselected):
        """
        Synchronize outliner selection with DCC
        Connected to QTreeView.selectionChanged signal
        :param selected: list of prims selected in Outliner
        :param deselected: list of prims deselected in Outliner
        """
        if self._blockSelectionCallback:
            return
        self._blockSelectionCallback = True
        selected = self.selectedIndexes()
        paths = [self.model()._GetPrimForIndex(index).GetPath() for index in selected
                 if index.column() == self.treePosition()]

        #add try/finally block here cause if command raise runtime error we never exit selection block
        try:
            if paths:
                cmds.select(dcc_core.SelectionList(paths), replace=True)
            else:
                cmds.select(dcc_core.SelectionList(), clear=True)
        finally:
            self._blockSelectionCallback = False

    def update_selection_from_dcc(self):
        """
        Get list of selected in dcc and update selection in Outliner
        Driven by dcc-signal 'SelectionChanged'
        Callbask to that signal connected each time the Outliner shown
        """
        if self._blockSelectionCallback or not self.model()._IsStageValid():
            return
        self._blockSelectionCallback = True

        selected_prim_paths = self.app.get_prim_selection()
        self.select_tree_items_by_path(selected_prim_paths)

        self._blockSelectionCallback = False

    def dropEvent(self, event):
        """"Reimplemented reparent command specially for dcc"""
        pos = event.pos()
        index = self.indexAt(pos)
        indicator_position = self.dropIndicatorPosition()

        if not index.isValid():
            new_parent_prim = self.model().get_stage().GetPrimAtPath('/')
        else:
            if indicator_position != self.OnItem:
                return
            new_parent_prim = self.model()._GetPrimForIndex(index)

        mime_data = event.mimeData()

        settings = dcc_core.Application.instance().get_settings()
        preserve_transform = settings.get_bool(
            "parent_command.preserve_transform", True
        )

        cmds.parent_prim(new_parent_prim.GetPath(), paths=mime_data.paths_list, preserve_transform=preserve_transform)


class DccOutlinerRole(OutlinerRole):
    @classmethod
    def GetContextMenuActions(cls, outliner):
        # type: (UsdOutliner) -> List[Union[MenuAction, Type[MenuAction]]]
        """
        Parameters
        ----------
        outliner : UsdOutliner

        Returns
        -------
        List[Union[MenuAction, Type[MenuAction]]]
        """
        return OutlinerRole.GetContextMenuActions(outliner) + [MenuSeparator, ClearRefineAtPrim, ClearAllRefine]


class DccUsdOutliner(UsdOutliner):
    """UsdStage editing application which displays the hierarchy of a stage."""
    def __init__(self, role=None, parent=None):
        # type: (Usd.Stage, Optional[Union[Type[OutlinerRole], OutlinerRole]], Optional[QtGui.QWidget]) -> None
        """
        Parameters
        ----------
        stage : Usd.Stage
        role : Optional[Union[Type[OutlinerRole], OutlinerRole]]
        parent : Optional[QtGui.QWidget]
        """
        self.app = dcc_core.Application.instance()
        self.session = self.app.get_session()
        stage = self.session.get_current_stage()
        role = DccOutlinerRole if role is None else role
        UsdOutliner.__init__(self, stage, role, parent)

        self.selection_callback_id = None
        self.current_time_callback_id = None
        self.stage_changed_callback_id = None

        self.add_callbacks()

    def closeEvent(self, event):
        self.remove_callbacks()
        event.accept()
    
    def _CreateView(self, role):
        # type: (Usd.Stage, Union[Type[OutlinerRole], OutlinerRole]) -> QtWidgets.QAbstractItemView
        """Create the hierarchy view for the outliner.

        This is provided as a convenience for subclass implementations.

        Parameters
        ----------
        stage : Usd.Stage
        role : Union[Type[OutlinerRole], OutlinerRole]

        Returns
        -------
        QtWidgets.QAbstractItemView
        """
        return DccOutlinerTreeView(
            self._dataModel,
            contextMenuActions=role.GetContextMenuActions(self),
            contextProvider=self,
            parent=self)

    def update_stage(self):
        stage = self.session.get_current_stage()
        self.SetStage(stage)

    def set_current_time(self):
        timecode = Usd.TimeCode(self.app.get_current_time())
        self._dataModel.set_time_code(timecode)

    def add_callbacks(self):
        """Add callbacks to keep ui in sync while visible"""
        self.selection_callback_id = self.app.register_event_callback(dcc_core.Application.EventType.SELECTION_CHANGED, self.view.update_selection_from_dcc)
        self.current_time_callback_id = self.app.register_event_callback(dcc_core.Application.EventType.CURRENT_TIME_CHANGED,
                                                                         self.set_current_time)
        self.stage_changed_callback_id = self.app.register_event_callback(dcc_core.Application.EventType.CURRENT_STAGE_CHANGED,
                                                                          self.update_stage)
        # make sure selection starts in correct state
        self.view.update_selection_from_dcc()
        self.set_current_time()

    def remove_callbacks(self):
        """Remove callbacks on ui hide"""
        if self.selection_callback_id:
            self.app.unregister_event_callback(dcc_core.Application.EventType.SELECTION_CHANGED, self.selection_callback_id)
            self.selection_callback_id = None
        if self.current_time_callback_id:
            self.app.unregister_event_callback(dcc_core.Application.EventType.CURRENT_TIME_CHANGED, self.current_time_callback_id)
            self.current_time_callback_id = None
        if self.stage_changed_callback_id:
            self.app.unregister_event_callback(dcc_core.Application.EventType.CURRENT_STAGE_CHANGED, self.stage_changed_callback_id)
            self.stage_changed_callback_id = None

    def apply_filter(self, string):
        UsdOutliner.apply_filter(self, string)
        self.view.update_selection_from_dcc()


if __name__ == '__main__':
    # simple test
    import sys

    app = QtWidgets.QApplication(sys.argv)
    usdFileArg = sys.argv[1]
    window = DccUsdOutliner.FromUsdFile(usdFileArg)
    window.show()
    sys.exit(app.exec_())