import os
import sys
from pxr.UsdQt._bindings import _HierarchyCache
from pxr import Sdf, Tf, UsdGeom, Gf, Usd
from pxr.UsdQtEditors._Qt import QtWidgets, QtCore

from .i18n import i18n

import re

IsOnEditTargetRole = QtCore.Qt.UserRole + 1
HasPcpErrorsRole = QtCore.Qt.UserRole + 2


def is_prim_spec_on_edit_target(prim, edittarget):
    if prim:
        stack = prim.GetPrimStack()
        layer = edittarget.GetLayer()

        for spec in stack:
            if spec.layer == layer:
                return True

    return False


def has_prim_pcp_errors(prim):
    pcp_index = prim.GetPrimIndex()
    if pcp_index.localErrors:
        return True
    return False


def join_sdfpath(parent_path, name):
    if parent_path == '/':
        return Sdf.Path('/' + name)
    else:
        return Sdf.Path(str(parent_path) + '/' + name)


def edit_prim_path(new_edit, stage):
    if new_edit.newPath.isEmpty:
        return False

    edit = Sdf.BatchNamespaceEdit()
    edit.Add(new_edit)

    layer = stage.GetEditTarget().GetLayer()

    def hasObject(path):
        if path.IsTargetPath():
            x = layer.GetObjectAtPath(path.GetParentPath())
            if x:
                return x.targetAttributes[path.targetPath].IsValid()
            return False
        return bool(layer.GetObjectAtPath(path))

    canEdit = lambda e: True

    res, detail = edit.Process(hasObject, canEdit)
    if not res:
        print("Edit can't be done. {}".format(detail[0].reason))
        return

    res = layer.Apply(edit)
    return res


def rename_prim(prim, new_name):
    parent_prim = prim.GetParent()
    stage = prim.GetStage()
    path = prim.GetPath()
    if path.name == new_name:
        return False, None

    new_path = join_sdfpath(parent_prim.GetPath(), new_name)

    if has_parent_same_child(parent_prim, new_name):
        new_path = find_new_name_for_prim(new_path, parent_prim)
        new_name = new_path.name

    new_edit = Sdf.NamespaceEdit().Rename(path, new_name)
    return edit_prim_path(new_edit, stage), new_path


def change_prim_path(path, new_path, stage):
    if path.GetParentPath() == new_path:
        return
    new_edit = Sdf.NamespaceEdit().Reparent(path, new_path, 0)
    return edit_prim_path(new_edit, stage)


def has_parent_same_child(parent_prim, child_name):
    neighbours = parent_prim.GetAllChildren()
    for prim in neighbours:
        if prim.GetName() == child_name:
            return True

    return False


def find_new_name_for_prim(path, parent_prim):
    parent_path = path.GetParentPath()
    prim_name = path.name

    neighbours = parent_prim.GetAllChildren()

    prog = re.compile('[0-9]*$')

    cutted_prim_name = prog.sub('', prim_name)
    max_num = 0
    for prim in neighbours:
        cutted_neighbour_name = prog.sub('', prim.GetName())
        if cutted_neighbour_name == cutted_prim_name:
            num_suffix = prim.GetName().lstrip(cutted_neighbour_name)
            if num_suffix:
                max_num = max(max_num, int(num_suffix))

    new_prim_name = cutted_prim_name + str(max_num + 1)

    return join_sdfpath(parent_path, new_prim_name)


def reparent_paths(new_parent_prim, paths, preserve_position=False, time_code=Usd.TimeCode.Default()):
    new_parent_path = new_parent_prim.GetPath()
    stage = new_parent_prim.GetStage()
    layer = stage.GetEditTarget().GetLayer()

    res = False
    new_paths = {}
    with Sdf.ChangeBlock():
        for path in paths:
            if path.GetParentPath() == new_parent_path:
                print("Edit can't be done. The same parent")
                continue

            new_transform = None
            if preserve_position:
                prim = stage.GetPrimAtPath(path)
                if not UsdGeom.Xformable(prim).TransformMightBeTimeVarying():
                    xform_cache = UsdGeom.XformCache(time_code)
                    xform_prim = xform_cache.GetLocalToWorldTransform(prim)
                    xform_new_parent_prim = xform_cache.GetLocalToWorldTransform(new_parent_prim)
                    new_transform = xform_prim * xform_new_parent_prim.GetInverse()

            if bool(layer.GetObjectAtPath(new_parent_path)) == False and stage.GetPrimAtPath(new_parent_path).IsValid():
                Sdf.CreatePrimInLayer(layer, new_parent_path)
                print("Over prim %s defined in other layers" % new_parent_path)
            name = path.name
            if has_parent_same_child(new_parent_prim, name):
                new_path = find_new_name_for_prim(path, new_parent_prim)
                name = new_path.name
            new_edit = Sdf.NamespaceEdit().ReparentAndRename(path, new_parent_path, name, 0)
            res = edit_prim_path(new_edit, stage)

            if res:
                new_paths[join_sdfpath(new_parent_path, name)] = new_transform

    if preserve_position:
        with Sdf.ChangeBlock():
            for new_path, transform in new_paths.items():
                prim = stage.GetPrimAtPath(new_path)
                if prim and not UsdGeom.Xformable(prim).TransformMightBeTimeVarying():
                    xform_common_api = UsdGeom.XformCommonAPI(prim)
                    new_transform = Gf.Transform(transform)
                    is_close_enough = Gf.IsClose(new_transform.GetPivotOrientation().angle, 0, 0.001)
                    if xform_common_api and is_close_enough:                        
                        xform_common_api.SetTranslate(new_transform.GetTranslation())
                        euler_angles = new_transform.GetRotation().Decompose(Gf.Vec3d.ZAxis(), Gf.Vec3d.YAxis(),
                                                                             Gf.Vec3d.XAxis())
                        rotation = Gf.Vec3f(euler_angles[2], euler_angles[1], euler_angles[0])
                        xform_common_api.SetRotate(rotation)
                        xform_common_api.SetScale(Gf.Vec3f(new_transform.GetScale()))
                    else:
                        UsdGeom.Xformable(prim).MakeMatrixXform().Set(transform)

    # res is True if one of the paths was reparented
    return res, new_paths.keys()


def remove_prims(prims_paths, stage, ask=True, messagebox_parent=None):
    buttons = QtWidgets.QMessageBox.Yes | QtWidgets.QMessageBox.Cancel
    if len(prims_paths) > 1:
        buttons |= QtWidgets.QMessageBox.YesToAll
    with Sdf.ChangeBlock():
        for path in prims_paths:
            if ask:
                answer = QtWidgets.QMessageBox.question(
                    messagebox_parent,
                    'Confirm Prim Removal',
                    'Remove prim/prim edits (and any children) at {0}?'.format(path),
                    buttons=buttons,
                    defaultButton=QtWidgets.QMessageBox.Yes)
                if answer == QtWidgets.QMessageBox.Cancel:
                    return
                elif answer == QtWidgets.QMessageBox.YesToAll:
                    ask = False
            stage.RemovePrim(path)


def reload_layers(layers, widget_parent):
    if any([layer.dirty for layer in layers]):
        buttons = QtWidgets.QMessageBox.Yes | QtWidgets.QMessageBox.Cancel
        answer = QtWidgets.QMessageBox.question(
            widget_parent,
            "Confirm Reload Layer",
            "Layer you want to reload has changes. Do you still want to reload it and discard changes?",
            buttons=buttons,
            defaultButton=QtWidgets.QMessageBox.Yes)
        if answer == QtWidgets.QMessageBox.Cancel:
            return
    for layer in layers:
        layer.Reload()


def localise_layer(layer, stage, widget_parent):
    import json
    from shutil import copy
    workspace = os.environ.get('WORKSPACE')
    workspace_json = os.environ.get('WREF_RESOLVER_WORKSPACE')
    file_name = os.path.basename(layer.realPath)
    if os.path.exists(os.path.join(workspace, file_name)):
        QtWidgets.QMessageBox.information(
            widget_parent,
            i18n("warnings", "Cannot copy file"),
            i18n("warnings", "File {} already exists.").format(os.path.join(workspace, file_name)))
        return

    copy(layer.realPath, workspace)

    import wizart.desktop.workspace_utils as wutils
    wutils.repath(layer.identifier, os.path.join('$WORKSPACE', file_name))

    print('File {} localised successfully'.format(layer.realPath))

    with open(workspace_json) as f:
        try:
            config = json.load(f)
        except ValueError:
            config = {}

    if os.path.basename(sys.executable).lower() == 'maya.exe':
        import pymel.core as pm
        try:
            file_info = json.loads(pm.system.fileInfo.get("wref_resolver_config", '{}'))
        except ValueError:
            file_info = {}

        config.get('repath', {}).update(file_info.get('repath', {}))
        config.get('pin', {}).update(file_info.get('pin', {}))

    import usd_wref_resolver
    from pxr import Ar
    ctx = stage.GetPathResolverContext()
    ctx.config = json.dumps(config)
    with Ar.ResolverContextBinder(ctx):
        layer.UpdateAssetInfo()
        layer.Reload()
