import os, sys
from Qt import QtCore, QtGui, QtWidgets
import wizart.dcc.core as dcc_core
from .composition_tree import CompositionTreeWidget
from itertools import groupby
from pxr import Sdf, Usd, UsdGeom
from pxr.UsdQtEditors.dcc_layerTextEditor import LayerTextEditorDialog
import six
from wizart.dcc.color_theme import get_color_theme, ColorTheme
from wizart.dcc.i18n import i18n


class VariantComboBox(QtWidgets.QComboBox):
    def __init__(self, parent, prim, variantSetName):
        QtWidgets.QComboBox.__init__(self, parent)
        self.prim = prim
        self.variantSetName = variantSetName

    def updateVariantSelection(self, index):
        variantSet = self.prim.GetVariantSet(self.variantSetName)
        currentVariantSelection = variantSet.GetVariantSelection()
        newVariantSelection = str(self.currentText())
        if currentVariantSelection != newVariantSelection:
            variantSet.SetVariantSelection(newVariantSelection)

def GetValueAtFrame(prop, frame):
    if isinstance(prop, Usd.Relationship):
        return prop.GetTargets()
    elif isinstance(prop, Usd.Attribute):
        return prop.Get(frame)
    elif isinstance(prop, Sdf.AttributeSpec):
        if frame == Usd.TimeCode.Default():
            return prop.default
        else:
            numTimeSamples = -1
            if prop.HasInfo('timeSamples'):
                numTimeSamples = prop.layer.GetNumTimeSamplesForPath(prop.path)
            if numTimeSamples == -1:
                return prop.default
            elif numTimeSamples == 1:
                return "1 time sample"
            else:
                return str(numTimeSamples) + " time samples"
    elif isinstance(prop, Sdf.RelationshipSpec):
        return prop.targetPathList

    return None

def GetScalarTypeFromAttr(attr):
    '''
    returns the (scalar, isArray) where isArray is True if it was an array type
    '''
    # Usd.Attribute and customAttributes.CustomAttribute have a
    # GetTypeName function, while Sdf.AttributeSpec has a typeName attr.
    if hasattr(attr, 'GetTypeName'):
        typeName = attr.GetTypeName()
    elif hasattr(attr, 'typeName'):
        typeName = attr.typeName
    else:
        typeName = ""

    if isinstance(typeName, Sdf.ValueTypeName):
        return typeName.scalarType, typeName.isArray
    else:
        return None, False

def GetShortStringForValue(prop, val):
    if isinstance(prop, Usd.Relationship):
        val = ", ".join(str(p) for p in val)
    elif isinstance(prop, Sdf.RelationshipSpec):
        return str(prop.targetPathList)

    # If there is no value opinion, we do not want to display anything,
    # since python 'None' has a different meaning than usda-authored None,
    # which is how we encode attribute value blocks (which evaluate to 
    # Sdf.ValueBlock)
    if val is None:
        return ''
    
    scalarType, isArray = GetScalarTypeFromAttr(prop)
    result = ''
    if isArray and not isinstance(val, Sdf.ValueBlock):
        def arrayToStr(a):
            from itertools import chain
            elems = a if len(a) <= 6 else chain(a[:3], ['...'], a[-3:])
            return '[' + ', '.join(map(str, elems)) + ']'
        if val is not None and len(val):
            result = "%s[%d]: %s" % (scalarType, len(val), arrayToStr(val))
        else:
            result = "%s[]" % scalarType
    else:
        result = str(val)

    return result[:500]

# Gathers information about a layer used as a subLayer, including its
# position in the layerStack hierarchy.
class SubLayerInfo(object):
    def __init__(self, sublayer, offset, containingLayer, prefix):
        self.layer = sublayer
        self.offset = offset
        self.parentLayer = containingLayer
        self._prefix = prefix

    def GetOffsetString(self):
        o = self.offset.offset
        s = self.offset.scale
        if o == 0:
            if s == 1:
                return ""
            else:
                return str.format("(scale = {})", s)
        elif s == 1:
            return str.format("(offset = {})", o)
        else:
            return str.format("(offset = {0}; scale = {1})", o, s)

    def GetHierarchicalDisplayString(self):
        return self._prefix + self.layer.GetDisplayName()

def _AddSubLayers(layer, layerOffset, prefix, parentLayer, layers):
    offsets = layer.subLayerOffsets
    layers.append(SubLayerInfo(layer, layerOffset, parentLayer, prefix))
    for i, l in enumerate(layer.subLayerPaths):
        offset = offsets[i] if offsets is not None and len(offsets) > i else Sdf.LayerOffset()
        subLayer = Sdf.Layer.FindRelativeToLayer(layer, l)
        # Due to an unfortunate behavior of the Pixar studio resolver,
        # FindRelativeToLayer() may fail to resolve certain paths.  We will
        # remove this extra Find() call as soon as we can retire the behavior;
        # in the meantime, the extra call does not hurt (but should not, in
        # general, be necessary)
        if not subLayer:
            subLayer = Sdf.Layer.Find(l)

        if subLayer:
            # This gives a 'tree'-ish presentation, but it looks sad in
            # a QTableWidget.  Just use spaces for now
            # addedPrefix = "|-- " if parentLayer is None else "|    "
            addedPrefix = "     "
            _AddSubLayers(subLayer, offset, addedPrefix + prefix, layer, layers)
        else:
            print("Could not find layer " + l)

def GetRootLayerStackInfo(layer):
    layers = []
    _AddSubLayers(layer, Sdf.LayerOffset(), "", None, layers)
    return layers

class UsdDetailsView(QtWidgets.QWidget):
    def __init__(self, parent=None):
        QtWidgets.QWidget.__init__(self, parent=parent)

        self.layers_icon = QtGui.QIcon(":/icons/layers")

        self._layout = QtWidgets.QVBoxLayout(self)
        self._layout.setContentsMargins(0, 0, 0, 0)

        self._tabs = QtWidgets.QTabWidget()

        style = """
QTabWidget::pane {
    background: palette(light);
    border-top-color: palette(light);
}

QTabBar::tab:selected, QTabBar::tab:hover {
    background: palette(light);
    color: palette(foreground);
}

QTabBar::tab:!selected {
    background: TAB_BACKGROUND;
}

QToolBar[att~="main"] {
    background: palette(light);
    margin-top: 1px;
    margin-bottom: 1px;
}

QTabBar::tab {
    color: TAB_COLOR;
    background: TAB_BACKGROUND;

    padding-left: 12px;
    padding-right: 12px;
    padding-top: 4px;
    padding-bottom: 5px;

    border-radius: 0px;
    border-left: 0px;

    border: 1px solid;
    border-width: 0px 1px 0px 0px;
    border-color: palette(base) palette(light) palette(light) palette(base);
}

QTabBar::tab:left {
    padding-left: 4px;
    padding-right: 5px;
    padding-top: 12px;
    padding-bottom: 12px;
    border-width: 0px 0px 1px 0px;
    border-color: palette(light) palette(base) palette(light) palette(light);
}

QTabBar::tab:last {
    border: 0px;
}
""" 
        if get_color_theme() == ColorTheme.LIGHT:
            style = style.replace("TAB_BACKGROUND", "#d6d6d6")
            style = style.replace("TAB_COLOR", "#3b3b3b")
        else:
            style = style.replace("TAB_BACKGROUND", "rgb(55, 55, 55)")
            style = style.replace("TAB_COLOR", "palette(dark)")

        self._tabs.setStyleSheet(style)

        self._metadata_view = QtWidgets.QTableWidget()
        self._metadata_view.setFrameShape(QtWidgets.QFrame.NoFrame)
        self._metadata_view.setFrameShadow(QtWidgets.QFrame.Plain)
        self._metadata_view.setLineWidth(0)
        self._metadata_view.setEditTriggers(QtWidgets.QAbstractItemView.NoEditTriggers)
        self._metadata_view.setAlternatingRowColors(True)
        self._metadata_view.setSelectionMode(QtWidgets.QAbstractItemView.SingleSelection)
        self._metadata_view.setSelectionBehavior(QtWidgets.QAbstractItemView.SelectRows)
        self._metadata_view.setShowGrid(False)
        self._metadata_view.setGridStyle(QtCore.Qt.SolidLine)
        self._metadata_view.setColumnCount(2)
        self._metadata_view.setObjectName("metadataView")
        self._metadata_view.setColumnCount(2)
        self._metadata_view.setRowCount(0)
        item = QtWidgets.QTableWidgetItem()
        self._metadata_view.setHorizontalHeaderItem(0, item)
        item = QtWidgets.QTableWidgetItem()
        self._metadata_view.setHorizontalHeaderItem(1, item)
        self._metadata_view.horizontalHeader().setCascadingSectionResizes(True)
        self._metadata_view.horizontalHeader().setDefaultSectionSize(220)
        self._metadata_view.horizontalHeader().setMinimumSectionSize(30)
        self._metadata_view.horizontalHeader().setSortIndicatorShown(False)
        self._metadata_view.horizontalHeader().setStretchLastSection(True)
        self._metadata_view.horizontalHeader().setHighlightSections(False)
        self._metadata_view.verticalHeader().setVisible(False)
        self._metadata_view.verticalHeader().setDefaultSectionSize(30)
        self._metadata_view.verticalHeader().setMinimumSectionSize(20)
        self._metadata_view.verticalHeader().setStretchLastSection(False)
        self._metadata_view.horizontalHeaderItem(0).setText(i18n("panels.usd_details_view", "Field Name"))
        self._metadata_view.horizontalHeaderItem(1).setText(i18n("panels.usd_details_view", "Value"))

        self._layer_stack = QtWidgets.QTableWidget()
        self._layer_stack.setFrameShape(QtWidgets.QFrame.NoFrame)
        self._layer_stack.setFrameShadow(QtWidgets.QFrame.Plain)
        self._layer_stack.setEditTriggers(QtWidgets.QAbstractItemView.NoEditTriggers)
        self._layer_stack.setAlternatingRowColors(True)
        self._layer_stack.setSelectionMode(QtWidgets.QAbstractItemView.SingleSelection)
        self._layer_stack.setSelectionBehavior(QtWidgets.QAbstractItemView.SelectRows)
        self._layer_stack.setShowGrid(False)
        self._layer_stack.setGridStyle(QtCore.Qt.SolidLine)
        self._layer_stack.setColumnCount(3)
        self._layer_stack.setRowCount(0)
        item = QtWidgets.QTableWidgetItem()
        self._layer_stack.setHorizontalHeaderItem(0, item)
        item = QtWidgets.QTableWidgetItem()
        self._layer_stack.setHorizontalHeaderItem(1, item)
        item = QtWidgets.QTableWidgetItem()
        self._layer_stack.setHorizontalHeaderItem(2, item)
        self._layer_stack.horizontalHeader().setCascadingSectionResizes(True)
        self._layer_stack.horizontalHeader().setDefaultSectionSize(220)
        self._layer_stack.horizontalHeader().setMinimumSectionSize(30)
        self._layer_stack.horizontalHeader().setSortIndicatorShown(False)
        self._layer_stack.horizontalHeader().setStretchLastSection(True)
        self._layer_stack.horizontalHeader().setHighlightSections(False)
        self._layer_stack.verticalHeader().setVisible(False)
        self._layer_stack.verticalHeader().setDefaultSectionSize(30)
        self._layer_stack.verticalHeader().setMinimumSectionSize(20)
        self._layer_stack.verticalHeader().setStretchLastSection(False)
        self._layer_stack.horizontalHeaderItem(0).setText(i18n("panels.usd_details_view", "Layer"))
        self._layer_stack.horizontalHeaderItem(1).setText(i18n("panels.usd_details_view", "Path"))
        self._layer_stack.horizontalHeaderItem(2).setText(i18n("panels.usd_details_view", "Value"))
        self._layer_stack.setIconSize(QtCore.QSize(20, 20))

        self._layer_stack.setContextMenuPolicy(QtCore.Qt.CustomContextMenu)
        self._layer_stack.customContextMenuRequested.connect(self.open_context_menu)
        self._layer_stack.itemSelectionChanged.connect(self._selection_changed)

        self._layer_stack.setSelectionMode(QtWidgets.QAbstractItemView.ExtendedSelection)

        self._composition = CompositionTreeWidget()

        self._tabs.addTab(self._metadata_view, i18n("panels.usd_details_view", "Meta Data"))
        self._tabs.addTab(self._layer_stack, i18n("panels.usd_details_view", "Layer Stack"))
        self._tabs.addTab(self._composition, i18n("panels.usd_details_view", "Composition"))

        self._layout.addWidget(self._tabs)

        self.selection_callback_id = None

        self.app = dcc_core.Application.instance()
        self.register_callbacks()

    def _get_selected_layers(self):
        selected_items = self._layer_stack.selectedItems()
        layers = []
        for item in selected_items:
            layer = Sdf.Layer.Find(item.identifier)
            layers.append(layer)
        return layers

    def _set_layer_selection(self):
        selected_layers = set(self._get_selected_layers())
        layers = set(self.app.get_layer_selection())
        if selected_layers != layers:
            self._layer_stack.blockSignals(True)
            self._layer_stack.clearSelection()
            for layer in layers:
                for row in range(self._layer_stack.rowCount()):
                    item = self._layer_stack.item(row, 0)
                    if item:
                        layer = Sdf.Layer.Find(item.identifier)
                        if layer in layers:
                            self._layer_stack.selectRow(item.row())
            self._layer_stack.blockSignals(False)

    def _selection_changed(self):
        self.app.set_layer_selection(self._get_selected_layers())

    def layer_selected(self):
        self._set_layer_selection()
        self._composition.layer_selected()

    def register_callbacks(self):
        self.selection_callback_id = self.app.register_event_callback(
            dcc_core.Application.EventType.SELECTION_CHANGED,
            self.update_data)
        self.stage_changed_callback_id = self.app.register_event_callback(
            dcc_core.Application.EventType.CURRENT_STAGE_CHANGED,
            self.update_data)
        self.layer_selection_changed_callback_id = self.app.register_event_callback(
            dcc_core.Application.EventType.LAYER_SELECTION_CHANGED, 
            self.layer_selected)

        # make sure editor starts in correct state
        self.update_data()

    def unregister_callbacks(self):
        if self.selection_callback_id:
            self.app.unregister_event_callback(
                dcc_core.Application.EventType.SELECTION_CHANGED, 
                self.selection_callback_id)
            self.selection_callback_id = None
        if self.stage_changed_callback_id:
            self.app.unregister_event_callback(
                dcc_core.Application.EventType.CURRENT_STAGE_CHANGED, 
                self.stage_changed_callback_id)
            self.stage_changed_callback_id = None
        if self.layer_selection_changed_callback_id:
            self.app.unregister_event_callback(
                dcc_core.Application.EventType.LAYER_SELECTION_CHANGED, 
                self.layer_selection_changed_callback_id)
            self.layer_selection_changed_callback_id = None

    def closeEvent(self, event):
        self.unregister_callbacks()
        event.accept()
    
    def update_data(self):
        session = self.app.get_session()
        stage = session.get_current_stage()
        if not stage:
            self.update_meta_data()
            self.update_layer_stack()
            self._composition.update()
            return

        selected_prim_path = self.app.get_prim_selection()

        prims = []
        for prim_path in selected_prim_path:
            prim = stage.GetPrimAtPath(prim_path)
            if prim.IsValid():
                prims.append(prim)

        prim = None
        if prims:
            prim = prims[0]

        self._composition.update(prim, stage)

        obj = prim # because it can be a prim or a property

        if obj:
            selection = self.app.get_selection()
            selection_data = selection.get_selection_data(prim.GetPath())
            if selection_data.properties:
                attr_name = selection_data.properties[-1]
                obj = prim.GetProperty(attr_name)
                if not obj.IsValid():
                    # Check if it is an inherited primvar.
                    inherited_primvar = UsdGeom.PrimvarsAPI(
                            prim).FindPrimvarWithInheritance(attr_name)
                    if inherited_primvar:
                        obj = inherited_primvar.GetAttr()

        self.update_meta_data(obj, stage)
        self.update_layer_stack(obj, stage)
        self._set_layer_selection()

    def update_layer_stack(self, obj=None, stage=None):
        tableWidget = self._layer_stack

        # Setup table widget
        tableWidget.clearContents()
        tableWidget.setRowCount(0)

        if not stage:
            return

        if not obj:
            obj = stage.GetPrimAtPath(Sdf.Path.absoluteRootPath)

        path = obj.GetPath()

        # The pseudoroot is different enough from prims and properties that
        # it makes more sense to process it separately
        if path == Sdf.Path.absoluteRootPath:
            layers = GetRootLayerStackInfo(
                stage.GetRootLayer())
            tableWidget.setColumnCount(2)
            tableWidget.horizontalHeaderItem(1).setText(i18n("panels.usd_details_view", "Layer Offset"))

            tableWidget.setRowCount(len(layers))

            for i, layer in enumerate(layers):
                layerItem = QtWidgets.QTableWidgetItem(self.layers_icon, layer.GetHierarchicalDisplayString())
                layerItem.layerPath = layer.layer.realPath
                layerItem.identifier = layer.layer.identifier
                toolTip = "<b>identifier:</b> @%s@ <br> <b>resolved path:</b> %s" % \
                    (layer.layer.identifier, layerItem.layerPath)
                toolTip = self._limitToolTipSize(toolTip)
                layerItem.setToolTip(toolTip)
                tableWidget.setItem(i, 0, layerItem)

                offsetItem = QtWidgets.QTableWidgetItem(layer.GetOffsetString())
                offsetItem.layerPath = layer.layer.realPath
                offsetItem.identifier = layer.layer.identifier
                toolTip = self._limitToolTipSize(str(layer.offset))
                offsetItem.setToolTip(toolTip)
                tableWidget.setItem(i, 1, offsetItem)

            tableWidget.resizeColumnToContents(0)
        else:
            specs = []
            tableWidget.setColumnCount(3)
            header = tableWidget.horizontalHeader()
            header.setSectionResizeMode(0, QtWidgets.QHeaderView.ResizeToContents)
            header.setSectionResizeMode(1, QtWidgets.QHeaderView.Stretch)
            header.setSectionResizeMode(2, QtWidgets.QHeaderView.ResizeToContents)
            tableWidget.horizontalHeaderItem(1).setText(i18n("panels.usd_details_view", "Path"))

            if path.IsPropertyPath():
                prop = obj
                specs = prop.GetPropertyStack(Usd.TimeCode(self.app.get_current_time()))
                c3 = "Value" if (len(specs) == 0 or
                                 isinstance(specs[0], Sdf.AttributeSpec)) else "Target Paths"
                tableWidget.setHorizontalHeaderItem(2,
                                                    QtWidgets.QTableWidgetItem(c3))
            else:
                specs = obj.GetPrimStack()
                tableWidget.setHorizontalHeaderItem(2,
                    QtWidgets.QTableWidgetItem(i18n("panels.usd_details_view", "Metadata")))

            tableWidget.setRowCount(len(specs))

            for i, spec in enumerate(specs):
                layerItem = QtWidgets.QTableWidgetItem(self.layers_icon, spec.layer.GetDisplayName())
                layerItem.setToolTip(self._limitToolTipSize(spec.layer.realPath))
                tableWidget.setItem(i, 0, layerItem)

                pathItem = QtWidgets.QTableWidgetItem(spec.path.pathString)
                pathItem.setToolTip(self._limitToolTipSize(spec.path.pathString))
                tableWidget.setItem(i, 1, pathItem)

                if path.IsPropertyPath():
                    val = GetValueAtFrame(spec, Usd.TimeCode(self.app.get_current_time()))
                    valStr = GetShortStringForValue(spec, val)
                    ttStr = valStr
                    valueItem = QtWidgets.QTableWidgetItem(valStr)
                    sampleBased = (spec.HasInfo('timeSamples') and
                        spec.layer.GetNumTimeSamplesForPath(path) != -1)
                    if sampleBased:
                        valueItem.setForeground(QtGui.QColor(177, 207, 153))
                    valueItem.setToolTip(ttStr)

                else:
                    metadataKeys = spec.GetMetaDataInfoKeys()
                    metadataDict = {}
                    for mykey in metadataKeys:
                        if spec.HasInfo(mykey):
                            metadataDict[mykey] = spec.GetInfo(mykey)
                    valStr, ttStr = self._formatMetadataValueView(metadataDict)
                    valueItem = QtWidgets.QTableWidgetItem(valStr)
                    valueItem.setToolTip(ttStr)

                tableWidget.setItem(i, 2, valueItem)
                # Add the data the context menu needs
                for j in range(3):
                    item = tableWidget.item(i, j)
                    item.layerPath = spec.layer.realPath
                    item.path = spec.path.pathString
                    item.identifier = spec.layer.identifier

    def update_meta_data(self, obj=None, stage=None):
        tableWidget = self._metadata_view
        # self._propertiesDict = self._getPropertiesDict()

        # Setup table widget
        tableWidget.clearContents()
        tableWidget.setRowCount(0)

        if not stage or not obj:
            return

        m = obj.GetAllMetadata()

        # We have to explicitly add in metadata related to composition arcs
        # and value clips here, since GetAllMetadata prunes them out.
        #
        # XXX: Would be nice to have some official facility to query
        # this.
        compKeys = [# composition related metadata
                    "references", "inheritPaths", "specializes",
                    "payload", "subLayers",

                    # non-template clip metadata
                    "clipAssetPaths", "clipTimes", "clipManifestAssetPath",
                    "clipActive", "clipPrimPath",

                    # template clip metadata
                    "clipTemplateAssetPath",
                    "clipTemplateStartTime", "clipTemplateEndTime",
                    "clipTemplateStride"]


        for k in compKeys:
            v = obj.GetMetadata(k)
            if not v is None:
                m[k] = v

        clipMetadata = obj.GetMetadata("clips")
        if clipMetadata is None:
            clipMetadata = {}
        numClipRows = 0
        for (clip, data) in clipMetadata.items():
            numClipRows += len(data)
        m["clips"] = clipMetadata
        
        numMetadataRows = (len(m) - 1) + numClipRows

        # Variant selections that don't have a defined variant set will be 
        # displayed as well to aid debugging. Collect them separately from
        # the variant sets.
        variantSets = {}
        setlessVariantSelections = {}
        if (isinstance(obj, Usd.Prim)):
            # Get all variant selections as setless and remove the ones we find
            # sets for.
            setlessVariantSelections = obj.GetVariantSets().GetAllVariantSelections()

            variantSetNames = obj.GetVariantSets().GetNames()
            for variantSetName in variantSetNames:
                variantSet = obj.GetVariantSet(variantSetName)
                variantNames = variantSet.GetVariantNames()
                variantSelection = variantSet.GetVariantSelection()
                combo = VariantComboBox(None, obj, variantSetName)
                # First index is always empty to indicate no (or invalid)
                # variant selection.
                combo.addItem('')
                for variantName in variantNames:
                    combo.addItem(variantName)
                indexToSelect = combo.findText(variantSelection)
                combo.setCurrentIndex(indexToSelect)
                variantSets[variantSetName] = combo
                # Remove found variant set from setless.
                setlessVariantSelections.pop(variantSetName, None)

        tableWidget.setRowCount(numMetadataRows + len(variantSets) + 
                                len(setlessVariantSelections) + 2)

        rowIndex = 0

        # Although most metadata should be presented alphabetically,the most 
        # user-facing items should be placed at the beginning of the  metadata 
        # list, these consist of [object type], [path], variant sets, active, 
        # assetInfo, and kind.
        def populateMetadataTable(key, val, rowIndex):
            attrName = QtWidgets.QTableWidgetItem(str(key))
            tableWidget.setItem(rowIndex, 0, attrName)

            valStr, ttStr = self._formatMetadataValueView(val)
            attrVal = QtWidgets.QTableWidgetItem(valStr)
            attrVal.setToolTip(ttStr)

            tableWidget.setItem(rowIndex, 1, attrVal)

        sortedKeys = sorted(m.keys())
        reorderedKeys = ["kind", "assetInfo", "active"]

        for key in reorderedKeys:
            if key in sortedKeys:
                sortedKeys.remove(key)
                sortedKeys.insert(0, key)

        object_type = "Attribute" if type(obj) is Usd.Attribute \
               else "Prim" if type(obj) is Usd.Prim \
               else "Relationship" if type(obj) is Usd.Relationship \
               else "Unknown"
        populateMetadataTable("[object type]", object_type, rowIndex)
        rowIndex += 1
        populateMetadataTable("[path]", str(obj.GetPath()), rowIndex)
        rowIndex += 1

        for variantSetName, combo in six.iteritems(variantSets):
            attrName = QtWidgets.QTableWidgetItem(str(variantSetName+ ' variant'))
            tableWidget.setItem(rowIndex, 0, attrName)
            tableWidget.setCellWidget(rowIndex, 1, combo)
            combo.currentIndexChanged.connect(
                lambda i, combo=combo: combo.updateVariantSelection(i))
            rowIndex += 1

        # Add all the setless variant selections directly after the variant 
        # combo boxes
        for variantSetName, variantSelection in six.iteritems(setlessVariantSelections):
            attrName = QtWidgets.QTableWidgetItem(str(variantSetName+ ' variant'))
            tableWidget.setItem(rowIndex, 0, attrName)

            valStr, ttStr = self._formatMetadataValueView(variantSelection)
            # Italicized label to stand out when debugging a scene.
            label = QtWidgets.QLabel('<i>' + valStr + '</i>')
            label.setIndent(3)
            label.setToolTip(ttStr)
            tableWidget.setCellWidget(rowIndex, 1, label)

            rowIndex += 1

        for key in sortedKeys:
            if key == "clips":
                for (clip, metadataGroup) in m[key].items():
                    attrName = QtWidgets.QTableWidgetItem(str('clip:' + clip))
                    tableWidget.setItem(rowIndex, 0, attrName)
                    for metadata in metadataGroup.keys():
                        dataPair = (metadata, metadataGroup[metadata])
                        valStr, ttStr = self._formatMetadataValueView(dataPair)
                        attrVal = QtWidgets.QTableWidgetItem(valStr)
                        attrVal.setToolTip(ttStr)
                        tableWidget.setItem(rowIndex, 1, attrVal)
                        rowIndex += 1
            elif key == "customData":
                populateMetadataTable(key, obj.GetCustomData(), rowIndex)
                rowIndex += 1
            else:
                populateMetadataTable(key, m[key], rowIndex)
                rowIndex += 1


        tableWidget.resizeColumnToContents(0)

    def _findIndentPos(self, s):
        for index, char in enumerate(s):
            if char != ' ':
                return index

        return len(s) - 1

    def _maxToolTipWidth(self):
        return 90

    def _maxToolTipHeight(self):
        return 32

    def _trimWidth(self, s, isList=False):
        # We special-case the display offset because list
        # items will have </li> tags embedded in them.
        offset = 10 if isList else 5

        if len(s) >= self._maxToolTipWidth():
            # For strings, well do special ellipsis behavior
            # which displays the last 5 chars with an ellipsis
            # in between. For other values, we simply display a
            # trailing ellipsis to indicate more data.
            if s[0] == '\'' and s[-1] == '\'':
                return (s[:self._maxToolTipWidth() - offset]
                        + '...'
                        + s[len(s) - offset:])
            else:
                return s[:self._maxToolTipWidth()] + '...'
        return s

    def _limitToolTipSize(self, s, isList=False):
        ttStr = ''

        lines = s.split('<br>')
        for index, line in enumerate(lines):
            if index+1 > self._maxToolTipHeight():
                break
            ttStr += self._trimWidth(line, isList)
            if not isList and index != len(lines)-1:
                ttStr += '<br>'

        if (len(lines) > self._maxToolTipHeight()):
            ellipsis = ' '*self._findIndentPos(line) + '...'
            if isList:
                ellipsis = '<li>' + ellipsis + '</li>'
            else:
                ellipsis += '<br>'

            ttStr += ellipsis
            ttStr += self._trimWidth(lines[len(lines)-2], isList)

        return ttStr

    def _addRichTextIndicators(self, s):
        # - We'll need to use html-style spaces to ensure they are respected
        # in the toolTip which uses richtext formatting.
        # - We wrap the tooltip as a paragraph to ensure &nbsp; 's are
        # respected by Qt's rendering engine.
        return '<p>' + s.replace(' ', '&nbsp;') + '</p>'

    def _limitValueDisplaySize(self, s):
        maxValueChars = 300
        return s[:maxValueChars]

    def _cleanStr(self, s, repl):
        # Remove redundant char seqs and strip newlines.
        replaced = str(s).replace('\n', repl)
        filtered = [u for (u, _) in groupby(replaced.split())]
        return ' '.join(filtered)

    def _formatMetadataValueView(self, val):
        from pprint import pformat, pprint

        valStr = self._cleanStr(val, ' ')
        ttStr  = ''
        isList = False

        # For iterable things, like VtArrays and lists, we want to print
        # a nice numbered list.
        if isinstance(val, list) or getattr(val, "_isVtArray", False):
            isList = True

            # We manually supply the index for our list elements
            # because Qt's richtext processor starts the <ol> numbering at 1.
            for index, value in enumerate(val):
                last = len(val) - 1
                trimmed = self._cleanStr(value, ' ')
                ttStr += ("<li>" + str(index) + ":  " + trimmed + "</li><br>")

        elif isinstance(val, dict):
            # We stringify all dict elements so they display more nicely.
            # For example, by default, the pprint operation would print a
            # Vt Array as Vt.Array(N, (E1, ....). By running it through
            # str(..). we'd get [(E1, E2), ....] which is more useful to
            # the end user trying to examine their data.
            for k, v in val.items():
                val[k] = str(v)

            # We'll need to strip the quotes generated by the str' operation above
            stripQuotes = lambda s: s.replace('\'', '').replace('\"', "")

            valStr = stripQuotes(self._cleanStr(val, ' '))

            formattedDict = pformat(val)
            formattedDictLines = formattedDict.split('\n')
            for index, line in enumerate(formattedDictLines):
                ttStr += (stripQuotes(line)
                    + ('' if index == len(formattedDictLines) - 1 else '<br>'))
        else:
            ttStr = self._cleanStr(val, '<br>')

        valStr = self._limitValueDisplaySize(valStr)
        ttStr = self._addRichTextIndicators(
                    self._limitToolTipSize(ttStr, isList))
        return valStr, ttStr

    def open_context_menu(self, position):
        menu = QtWidgets.QMenu(self)

        act_layer_text = QtWidgets.QAction(i18n("panels.usd_details_view", "Show Layer Text"), menu)
        act_layer_text.triggered.connect(self._show_layer_text)

        act_copy_layer_identifier = QtWidgets.QAction(i18n("panels.usd_details_view", "Copy Layer Identifier"), menu)
        act_copy_layer_identifier.triggered.connect(self._copy_layer_identifier)

        act_reload_layer = QtWidgets.QAction(i18n("panels.usd_details_view", "Reload layer"), menu)
        act_reload_layer.triggered.connect(self._reload_layer)

        act_localise_layer = QtWidgets.QAction(i18n("panels.usd_details_view", "Localise Layer"), menu)
        workspace = os.environ.get('WORKSPACE')
        if not workspace:
            act_localise_layer.setDisabled(True)
        act_localise_layer.triggered.connect(self._localise_layer)

        menu.addAction(act_layer_text)
        menu.addAction(act_copy_layer_identifier)
        menu.addAction(act_reload_layer)
        menu.addAction(act_localise_layer)
        menu.exec_(self._layer_stack.viewport().mapToGlobal(position))

    def _get_selected_layer(self):
        row = self._layer_stack.currentRow()
        item = self._layer_stack.item(row, 0)
        if not item:
            return
        layer = Sdf.Layer.Find(item.identifier)
        return layer

    def _show_layer_text(self):
        layer = self._get_selected_layer()
        if layer:
            dialog = LayerTextEditorDialog.GetSharedInstance(
                layer,
                parent=self)
            dialog.show()
            dialog.raise_()
            dialog.activateWindow()

    def _copy_layer_identifier(self):
        layer = self._get_selected_layer()
        if layer:
            cb = QtWidgets.QApplication.clipboard()
            cb.setText(layer.identifier)

    def _reload_layer(self):
        layer = self._get_selected_layer()
        if layer:
            reload_layers([layer], self)

    def _localise_layer(self):
        layer = self._get_selected_layer()
        workspace = os.environ.get('WORKSPACE')
        if layer and workspace:
            localise_layer(layer, self._stage, self)