import os
import subprocess
from pxr.UsdQtEditors.outliner import EditTargetEditor, LayerStackTreeView, LayerTextEditorDialog, \
    CopyToClipboard, MenuAction, MenuSeparator
from pxr.UsdQt.qtUtils import ContextMenuMixin
from pxr.UsdQtEditors._Qt import QtWidgets, QtCore, QtGui
from ...i18n import i18n
from .layer_model import LayerStackModel, FilterModel, LayerMatchRole
from ..header_visibility_menu import HeaderVisibilityMenu
from ...add_branches import add_branches
from ..._func import reload_layers, localise_layer
from pxr import Usd, Tf, Sdf, Ar
from functools import partial
from collections import namedtuple

import six
from six.moves.urllib import parse as urlparse

try:
    from dcc.undo import UsdEditsUndoBlock as UndoBlock
except ImportError:
    from pxr.UsdQt._usdQt import UndoBlock

_DCC = False
try:
    from wizart.common_widgets import SearchWidget
    _DCC = True
except ImportError:
    _DCC = False

# 5 mb
MAX_LAYER_SIZE_TO_OPEN = 5*1024*1024


LayerStackDialogContext = namedtuple('LayerStackDialogContext',
                                     ['qtParent', 'layerDialog', 'stage',
                                      'selectedLayers', 'editTargetLayer', 'selectedItems'])


class ShowLayerText(MenuAction):
    defaultText = i18n("layer_widget", "Show Layer Text")

    def Update(self, action, context):
        if len(context.selectedLayers) != 1:
            action.setText(i18n("layer_widget", "Show Layer Text"))
            action.setEnabled(False)
        else:
            action.setEnabled(True)
            layer = context.selectedLayers[0]
            if os.path.isfile(layer.realPath):
                size = os.path.getsize(layer.realPath)
                str_size = ''
                if size < 1024:
                    str_size = i18n("layer_widget", "{} B").format(size)
                elif size < 1024*1024:
                    str_size = i18n("layer_widget", "{:.2f} KB").format(float(size) / 1024)
                else:
                    str_size = i18n("layer_widget", "{:.2f} MB").format(float(size) / (1024 * 1024))
                action.setText(i18n("layer_widget", "Show Layer Text ({})").format(str_size))

    def Do(self):
        do_open = True

        context = self.GetCurrentContext()
        if not context.selectedLayers:
            return
        layer = context.selectedLayers[0]
        path = layer.realPath
        if os.path.isfile(path) and os.path.getsize(path) > MAX_LAYER_SIZE_TO_OPEN:
            answer = QtWidgets.QMessageBox.question(
                context.qtParent,
                "Confirm Open Layer",
                "The USD file is too big and opening it will take a lot of time or even block the program. Are you sure you want to continue?",
                buttons=QtWidgets.QMessageBox.Yes | QtWidgets.QMessageBox.Cancel,
                defaultButton=QtWidgets.QMessageBox.Cancel)
            if answer == QtWidgets.QMessageBox.Cancel:
                do_open = False

        if do_open:
            qtParent = context.qtParent
            if qtParent and hasattr(qtParent, 'ShowLayerTextDialog'):
                # a parent dialog is in charge of tracking layers
                qtParent.ShowLayerTextDialog(layer)
            else:
                # use global shared instance registry
                dialog = LayerTextEditorDialog.GetSharedInstance(
                    layer,
                    parent=qtParent or context.layerDialog)
                dialog.show()
                dialog.raise_()
                dialog.activateWindow()


class CopyLayerPath(MenuAction):
    defaultText = i18n("layer_widget", "Copy Layer Identifier")

    def Update(self, action, context):
        action.setEnabled(bool(context.selectedLayers))

    def Do(self):
        context = self.GetCurrentContext()
        if context.selectedLayers:
            if len(context.selectedLayers) == 1:
                text = context.selectedLayers[0].identifier
            else:
                text = str([layer.identifier for layer in context.selectedLayers])
            CopyToClipboard(text)


class ReloadLayer(MenuAction):
    defaultText = i18n("layer_widget", "Reload layer")

    def Do(self):
        context = self.GetCurrentContext()
        if context.selectedLayers:
            reload_layers(context.selectedLayers, context.qtParent)

class LocaliseLayer(MenuAction):
    defaultText = i18n("layer_widget", "Localise Layer")

    def Update(self, action, context):
        workspace = os.environ.get('WORKSPACE')
        action.setEnabled(bool(context.selectedLayers) and bool(workspace))

    def Do(self):
        context = self.GetCurrentContext()
        if context.selectedLayers:
            for layer in context.selectedLayers:
                localise_layer(layer, context.stage, context.qtParent)


class CreateSubLayer(MenuAction):
    defaultText = i18n("layer_widget", "Create Sublayer")

    def Update(self, action, context):
        action.setEnabled(len(context.selectedLayers) == 1)

    def Do(self):
        context = self.GetCurrentContext()
        if context.selectedLayers:
            context.layerDialog.add_sublayer()


class CreateAnonymousSubLayer(MenuAction):
    defaultText = i18n("layer_widget", "Create Anonymous Sublayer")

    def Update(self, action, context):
        action.setEnabled(len(context.selectedLayers) == 1)

    def Do(self):
        context = self.GetCurrentContext()
        if context.selectedLayers:
            parent_layer = context.selectedLayers[0]
            with UndoBlock():
                new_layer = Sdf.Layer.CreateAnonymous()
                if new_layer:
                    parent_layer.subLayerPaths.append(new_layer.identifier)
                else:
                    return
            context.layerDialog.refresh_after_adding(parent_layer)


class AddExistingSubLayer(MenuAction):
    defaultText = i18n("layer_widget", "Add Existing Sublayer")

    def Update(self, action, context):
        action.setEnabled(len(context.selectedLayers) == 1)

    def open_file_dialog(self):
        file_name = QtWidgets.QFileDialog.getOpenFileName(self.GetCurrentContext().layerDialog, i18n("layer_widget", "Select file"), "",
                                                          i18n("layer_widget", "Image files (*.usd *.usda *.usdc)"))
        if file_name and file_name[0]:
            return file_name[0]

    def Do(self):
        context = self.GetCurrentContext()
        if context.selectedLayers:
            parent_layer = context.selectedLayers[0]
            path = self.open_file_dialog()
            if not path:
                return
            with UndoBlock():
                new_layer = Sdf.Layer.FindOrOpen(path)
                if new_layer:
                    parent_layer.subLayerPaths.append(new_layer.identifier)
                else:
                    return
            context.layerDialog.refresh_after_adding(parent_layer)


class SaveLayer(MenuAction):
    defaultText = i18n("layer_widget", "Save Layer")

    def Update(self, action, context):
        is_any_not_anonymous = any([not layer.anonymous for layer in context.selectedLayers])
        if bool(context.selectedLayers) and is_any_not_anonymous:
            action.setEnabled(True)
        else:
            action.setEnabled(False)

    def Do(self):
        context = self.GetCurrentContext()
        for layer in context.selectedLayers:
            if not layer.anonymous:
                layer.Save()


class SaveAsLayer(MenuAction):
    defaultText = i18n("layer_widget", "Save Layer As")

    def Update(self, action, context):
        # for now rename only sublayers
        if bool(context.selectedItems) and len(context.selectedItems) == 1 and not context.selectedItems[0].is_ref:
            action.setEnabled(True)
        else:
            action.setEnabled(False)

    def Do(self):
        context = self.GetCurrentContext()
        if not context.selectedItems:
            return
        item = context.selectedItems[0]
        layer = item.layer
        if not item.is_ref:
            parent_item = item.parent
            # path = EditIdentifierDialog.show()
            import wizart.dcc.core as dcc_core
            app = dcc_core.Application.instance()
            path, _ = QtWidgets.QFileDialog.getSaveFileName(
                app.get_main_window(),
                i18n("layer_widget", "Save Layer As"),
                layer.realPath,
                "*.usd *.usda *.usdc *.usdz;;*.usd;;*.usda;;*.usdc;;*.usdz" if not _DCC else dcc_core.Application.get_file_extensions(),
            )

            if not path:
                return

            if parent_item and layer.identifier in parent_item.layer.subLayerPaths:
                old_identifier = layer.identifier
                layer.identifier = path
                parent_item.layer.subLayerPaths.replace(old_identifier, layer.identifier)
            else:
                layer.identifier = path

            layer.Save()
            context.layerDialog.refresh()


class RemoveLayer(MenuAction):
    defaultText = i18n("layer_widget", "Remove Layer")

    def Update(self, action, context):
        # for now delete only sublayers
        # and don't delete root layer
        is_root_layer_in_list = any([not bool(item.parent) for item in context.selectedItems])
        is_any_sublayer_in_list = any([not item.is_ref for item in context.selectedItems])

        if bool(context.selectedItems) and not is_root_layer_in_list and is_any_sublayer_in_list:
            action.setEnabled(True)
        else:
            action.setEnabled(False)

    def Do(self):
        context = self.GetCurrentContext()
        if not context.selectedItems:
            return

        for item in context.selectedItems:
            if not item.is_ref:
                parent_item = item.parent
                if parent_item and item.layer.identifier in parent_item.layer.subLayerPaths:
                    with UndoBlock():
                        parent_item.layer.subLayerPaths.remove(item.layer.identifier)
                    context.layerDialog.refresh()


class ChangeVersionAction(MenuAction):
    def _parse_version(self, str_version):
        import re

        match = re.search('\D*(\d+)', str_version)
        if match and match.groups():
            try:
                return int(match.groups()[0])
            except TypeError:
                return

    def _set_version(self, version, layer):
        import usd_wref_resolver
        import json
        parsed_path = urlparse.urlparse(layer.identifier)
        filename_from_query = urlparse.parse_qs(parsed_path.query).get('path')
        if filename_from_query:
            new_query = 'path={}'.format(filename_from_query[0])
        else:
            new_query = ''

        pin_path = os.path.expandvars(parsed_path._replace(query=new_query).geturl())
        ctx_dict = {"pin": {pin_path: version}}
        ctx = usd_wref_resolver.WrefResolverContext()
        ctx.config = json.dumps(ctx_dict)
        with Ar.ResolverContextBinder(ctx):
            layer.UpdateAssetInfo()
            layer.Reload(True)

    def Build(self, context):
        try:
            import wref_resolver
            from wizart.desktop import auth_session
        except ImportError:
            return
        context = self.GetCurrentContext()

        if len(context.selectedLayers) != 1:
            return
        layer = context.selectedLayers[0]
        # get current version
        current_version = self._parse_version(layer.version)

        edited_path = urlparse.urlparse(layer.identifier)._replace(query='').geturl()
        component_path = wref_resolver.resolve(edited_path)
        if not os.path.isdir(component_path):
            return

        # get versions list
        assetvcs_session = auth_session.get_assetvcs_session()
        if not assetvcs_session or not assetvcs_session.is_asset(assetvcs_session, component_path):
            return
        asset = assetvcs_session.get_asset(component_path)
        versions = asset.get_versions()
        if not versions:
            return
        versions = [version.num for version in versions]

        # build menu
        menu = QtWidgets.QMenu('Versions', context.qtParent)
        for num in versions:
            version_menu = menu.addAction(str(num))
            if num == current_version:
                version_menu.setIcon(QtGui.QIcon(':/icons/checkmark'))
            else:
                version_menu.triggered.connect(partial(self._set_version,
                                                       num,
                                                       layer))
        return menu.menuAction()


class OpenWith(MenuAction):
    def _open_with_wizart_desktop(self, app):
        context = self.GetCurrentContext()
        if not context.selectedLayers:
            return

        _env = os.environ.copy()
        if 'QT_PREFERRED_BINDING' in _env:
            _env.pop('QT_PREFERRED_BINDING')
        if 'PYTHONPATH' in _env:
            _env.pop("PYTHONPATH")
        wd_path = os.path.join(os.environ.get('WIZART_DESKTOP'), 'wizart_desktop.exe')
        cmds = [wd_path, '--launch_app', app, context.selectedLayers[0].realPath]
        exitcode = subprocess.call(cmds,
                                   shell=True,
                                   startupinfo=subprocess.STARTUPINFO(),
                                   env=_env)

    def _open_with_usdview(self):
        context = self.GetCurrentContext()
        if not context.selectedLayers:
            return

        exitcode = subprocess.Popen(['usdview', context.selectedLayers[0].realPath],
                                    shell=True,
                                    startupinfo=subprocess.STARTUPINFO())

    def _open_with_os(self):
        context = self.GetCurrentContext()
        if not context.selectedLayers:
            return
 
        url = QtCore.QUrl("file:///%s" % context.selectedLayers[0].realPath)
        QtGui.QDesktopServices.openUrl(url)

    def _actions(self, menu):
        actions = []

        if os.environ.get('WIZART_DESKTOP'):
            open_with_usdview = QtWidgets.QAction('UsdView', menu)
            open_with_usdview.triggered.connect(
                lambda: self._open_with_wizart_desktop('usdview'))
            actions.append(open_with_usdview)

            open_with_text_editor = QtWidgets.QAction(i18n("layer_widget", "Text Editor"), menu)
            open_with_text_editor.triggered.connect(
                lambda: self._open_with_wizart_desktop('code'))
            actions.append(open_with_text_editor)
        else:
            open_with_usdview = QtWidgets.QAction('UsdView', menu)
            open_with_usdview.triggered.connect(self._open_with_usdview)
            actions.append(open_with_usdview)

            open_with_text_editor = QtWidgets.QAction(i18n("layer_widget", "Text Editor"), menu)
            open_with_text_editor.triggered.connect(self._open_with_os)
            actions.append(open_with_text_editor)

        return actions

    def Build(self, context):
        if len(context.selectedLayers) != 1:
            return

        menu = QtWidgets.QMenu(i18n("layer_widget", "Open With"), context.qtParent)

        for action in self._actions(menu):
            menu.addAction(action)
        return menu.menuAction()


class LayerWidgetRole():
    """Helper which provides standard hooks for defining the context menu
    actions and menu bar menus that should be added to an Layer Widget.
    """

    @classmethod
    def GetContextMenuActions(cls):
        """
        Returns
        -------
        List[Union[MenuAction, Type[MenuAction]]]
        """
        return [ShowLayerText, CopyLayerPath, ReloadLayer, LocaliseLayer,
                MenuSeparator, CreateSubLayer, CreateAnonymousSubLayer, AddExistingSubLayer, RemoveLayer,
                MenuSeparator, OpenWith, SaveLayer, SaveAsLayer,
                MenuSeparator, ChangeVersionAction]


@add_branches(0)
class LayerView(LayerStackTreeView):
    def __init__(self, contextProvider, role=None, parent=None):
        if not role:
            role = LayerWidgetRole()
        contextMenuActions = role.GetContextMenuActions()
        ContextMenuMixin.__init__(self,
            contextMenuActions=contextMenuActions,
            contextProvider=contextProvider,
            parent=parent)
        self._expanded_layers = []
        self.setAllColumnsShowFocus(True)
        self.setSelectionMode(QtWidgets.QAbstractItemView.ExtendedSelection)
        self.setState(self.DraggingState)
        self.setDragEnabled(True)
        self.setAcceptDrops(True)
        self.setDragDropMode(QtWidgets.QAbstractItemView.DragDrop)
        self.setDropIndicatorShown(True)

        self.setProperty("unfocusedKeyEvent_enable", True)

    def keyPressEvent(self, event):
        """Overloaded function for hotkeys"""
        if event.key() == QtCore.Qt.Key_F and event.modifiers() == QtCore.Qt.NoModifier:
            self.frame_selection()
        else:
            super(LayerView, self).keyPressEvent(event)

    def frame_selection(self):
        selectionModel = self.selectionModel()
        indexes = selectionModel.selectedRows()
        for index in indexes:
            source_index = self.model().mapToSource(index)
            self.expand_parents(source_index)
        if indexes:
            self.scrollTo(indexes[0])

    def expand_parents(self, index):
        while True:
            index = index.parent()
            if index.isValid():
                layer = index.data(LayerMatchRole)
                self.set_expanded_layer(layer, True)
            else:
                break

    def GetSelectedLayers(self):
        # type: () -> Optional[Sdf.Layer]
        """
        Returns
        -------
        Optional[Sdf.Layer]
        """
        selectionModel = self.selectionModel()
        indexes = selectionModel.selectedRows()
        layers = []
        for index in indexes:
            source_index = self.model().mapToSource(index)
            if source_index.isValid():
                layers.append(source_index.internalPointer().layer)
        return layers

    def GetSelectedItems(self):
        selectionModel = self.selectionModel()
        indexes = selectionModel.selectedRows()
        items = []
        for index in indexes:
            source_index = self.model().mapToSource(index)
            if source_index.isValid():
                items.append(source_index.internalPointer())
        return items

    def save_expanded_state(self):
        self._expanded_layers = []
        for index in self.model().persistentIndexList():
            if self.isExpanded(index):
                self._expanded_layers.append(index.data(LayerMatchRole))

    def restore_expanded_state(self):
        for layer in self._expanded_layers:
            self.set_expanded_layer(layer, True)

    def set_expanded_layer(self, layer, state):
        indexes = self.model().match(self.model().index(0, 0), LayerMatchRole, layer, flags=QtCore.Qt.MatchRecursive)
        for index in indexes:
            self.setExpanded(index, state)

    def select_layer(self, layer):
        indexes = self.model().match(self.model().index(0, 0), LayerMatchRole, layer, flags=QtCore.Qt.MatchRecursive)
        if indexes:
            self.selectionModel().select(indexes[0], QtCore.QItemSelectionModel.ClearAndSelect |
                                                     QtCore.QItemSelectionModel.Rows)

    def startDrag(self, supportedActions):
        selected = self.selectedIndexes()
        mouse_state = QtWidgets.QApplication.mouseButtons()
        if selected and mouse_state == QtCore.Qt.MiddleButton:
            mime_data = self.model().mimeData(selected)
            dragQDrag = QtGui.QDrag(self)
            # dragQDrag.setPixmap(QtGui.QPixmap('test.jpg')) # <- For put your custom image here
            dragQDrag.setMimeData(mime_data)
            defaultDropAction = QtCore.Qt.IgnoreAction
            if ((supportedActions & QtCore.Qt.CopyAction) and (
                    self.dragDropMode() != QtWidgets.QAbstractItemView.InternalMove)):
                defaultDropAction = QtCore.Qt.CopyAction
            dragQDrag.exec_(supportedActions, defaultDropAction)

    def dropEvent(self, event):
        pos = event.pos()
        index = self.indexAt(pos)
        indicator_position = self.dropIndicatorPosition()

        row = 0
        if not index.isValid():
            new_parent_item = self.model().get_item(self.model().index(0, 0, QtCore.QModelIndex()))
            new_parent_layer = self.model().get_stage().GetRootLayer()
        else:
            item = self.model().get_item(index)
            if indicator_position == self.OnItem:
                new_parent_item = item
            elif indicator_position == self.AboveItem:
                new_parent_item = item.parent
                row = new_parent_item.layer.subLayerPaths.index(item.layer.identifier) - 1
            elif indicator_position == self.BelowItem:
                new_parent_item = item.parent
                row = new_parent_item.layer.subLayerPaths.index(item.layer.identifier)
            else:
                return

        new_parent_layer = new_parent_item.layer

        def has_item_ancestor(item, ancestor):
            ancestor = ancestor
            while ancestor.parent:
                if item == ancestor:
                    return True
                ancestor = ancestor.parent
            return False

        mime_data = event.mimeData()
        with UndoBlock():
            for item in mime_data.items_to_move:
                old_parent_tem = item.parent
                if not old_parent_tem:
                    print("Can't move root layer")
                    continue
                if has_item_ancestor(item, new_parent_item):
                    print("Can't move layer '{}' into child '{}'".format(item.layer.identifier,
                                                                     new_parent_item.layer.identifier))
                    continue
                old_parent = old_parent_tem.layer
                old_parent.subLayerPaths.remove(item.layer.identifier)
                new_parent_layer.subLayerPaths.insert(row, item.layer.identifier)

        self.parent().refresh()


class EditIdentifierDialog(QtWidgets.QDialog):
    def __init__(self, parent=None):
        QtWidgets.QDialog.__init__(self, parent,
                                   QtCore.Qt.WindowSystemMenuHint |
                                   QtCore.Qt.WindowTitleHint |
                                   QtCore.Qt.WindowCloseButtonHint)

        self.setModal(False)
        self.setWindowTitle(i18n("layer_widget", "Enter path for layer identifier"))
        self.resize(600, 120)
        self.layout = QtWidgets.QVBoxLayout(self)
        self.layout.addWidget(QtWidgets.QSplitter())

        path_layout = QtWidgets.QHBoxLayout()
        path_layout.setSpacing(2)
        self.layout.addLayout(path_layout)
        self.sublayer_path_le = QtWidgets.QLineEdit(self)
        self.sublayer_path_le.setPlaceholderText(i18n("layer_widget", "Type new path..."))
        path_layout.addWidget(self.sublayer_path_le)
        self.file_dialog_btn = QtWidgets.QPushButton('...', self)
        self.file_dialog_btn.pressed.connect(self.open_file_dialog)
        path_layout.addWidget(self.file_dialog_btn)

        self.layout.addWidget(QtWidgets.QSplitter())

        buttons = QtWidgets.QDialogButtonBox(
            QtWidgets.QDialogButtonBox.Ok | QtWidgets.QDialogButtonBox.Cancel,
            QtCore.Qt.Horizontal, self)
        buttons.accepted.connect(self.accept)
        buttons.rejected.connect(self.reject)
        self.layout.addWidget(buttons)

    def open_file_dialog(self):
        file_name = QtWidgets.QFileDialog.getSaveFileName(self, i18n("layer_widget", "Select file"), "",
                                                          i18n("layer_widget", "Image files (*.usd *.usda *.usdc)"))
        if file_name and file_name[0]:
            self.sublayer_path_le.setText(file_name[0])

    @staticmethod
    def show():
        dialog = EditIdentifierDialog()
        res = dialog.exec_()
        if res:
            return dialog.sublayer_path_le.text()


class LayerWidget(EditTargetEditor):
    def __init__(self, stage=None, role=None, editTargetChangeCallback=None, parent=None):
        QtWidgets.QWidget.__init__(self, parent=parent)
        self._stage = stage
        self._role = role
        self._dataModel = LayerStackModel(stage, parent=self)
        self._filterModel = FilterModel(parent=self)
        self._filterModel.setSourceModel(self._dataModel)
        self._filterModel.setFilterKeyColumn(self._dataModel.headerLabels.index(self._dataModel.Name))
        self._editTargetChangeCallback = editTargetChangeCallback

        # Widget and other Qt setup
        self.view = self.create_view()
        self.view.setModel(self._filterModel)
        self.view.doubleClicked.connect(self.ChangeEditTarget)
        self.view.clicked.connect(lambda index: self.view_item_clicked(index), QtCore.Qt.UniqueConnection)
        self.view.setExpandsOnDoubleClick(False)

        layout = QtWidgets.QVBoxLayout(self)
        layout.setContentsMargins(0, 0, 0, 0)
        layout.setSpacing(2)

        header_layout = QtWidgets.QHBoxLayout()
        layout.addLayout(header_layout)

        if _DCC:
            self._filter_le = SearchWidget()
        else:
            self._filter_le = QtWidgets.QLineEdit()
        self._filter_le.setPlaceholderText(i18n("layer_widget", "Search..."))
        self._filter_le.textChanged.connect(self.apply_filter)

        self._edit_sublayers_btn = QtWidgets.QPushButton(QtGui.QIcon(':/layer_icons/add_layer'), '')
        self._edit_sublayers_btn.setIconSize(QtCore.QSize(20, 20))
        self._edit_sublayers_btn.setFlat(True)
        self._edit_sublayers_btn.setToolTip(i18n("layer_widget", "Create Sublayer"))
        self._edit_sublayers_btn.pressed.connect(self.add_sublayer)

        self._refresh_btn = QtWidgets.QPushButton(QtGui.QIcon(':/icons/refresh'), '')
        self._refresh_btn.setIconSize(QtCore.QSize(20, 20))
        self._refresh_btn.setFlat(True)
        self._refresh_btn.setToolTip(i18n("layer_widget", "Refresh"))
        self._refresh_btn.pressed.connect(self.refresh)

        header_layout.addWidget(self._filter_le)
        header_layout.addWidget(self._refresh_btn)
        header_layout.addWidget(self._edit_sublayers_btn)
        layout.addWidget(self.view)

        self.view.header().setContextMenuPolicy(QtCore.Qt.CustomContextMenu)
        self.view.header().customContextMenuRequested.connect(self.open_header_menu)

        self.view.hideColumn(self._dataModel.headerLabels.index('Path'))
        self.view.hideColumn(self._dataModel.headerLabels.index('Resolved Path'))

        self.view.header().resizeSection(self._dataModel.headerLabels.index('Name'), 250)
        self.view.header().resizeSection(self._dataModel.headerLabels.index('Muted'), 45)
        self.view.header().resizeSection(self._dataModel.headerLabels.index('Dirty'), 45)
        self.view.header().resizeSection(self._dataModel.headerLabels.index('Version'), 45)
        self.view.expandAll()

        self._create_menu_bar()

        self.destroyed.connect(lambda: LayerWidget.on_destroyed(self))

    def create_view(self):
        return LayerView(self, role=self._role, parent=self)

    def set_stage(self, stage):
        self._stage = stage
        self._dataModel.ResetStage(stage)
        if self._stage:
            self._listener_obj_chgd = Tf.Notice.Register(Usd.Notice.ObjectsChanged,
                                                         self._OnEditTargetChanged, stage)
            self._listener_ed_tgt_chgd = Tf.Notice.Register(Usd.Notice.StageEditTargetChanged,
                                                         self._OnEditTargetChanged, stage)
        else:
            self._listener_obj_chgd = None
            self._listener_ed_tgt_chgd = None

    def _OnEditTargetChanged(self, *args):
        self.view.dataChanged(QtCore.QModelIndex(), QtCore.QModelIndex())
        self._dataModel.update_edit_target_parents()

    def view_item_clicked(self, index):
        model = self._dataModel
        if model.headerLabels.index(model.Muted) == index.column():
            self.change_muted(index)

    def change_muted(self, index):
        source_index = self._filterModel.mapToSource(index)
        layer = source_index.internalPointer().layer

        if self._stage.GetRootLayer().identifier == layer.identifier:
            return
        if self._stage.IsLayerMuted(layer.identifier):
            self._stage.UnmuteLayer(layer.identifier)
        else:
            self._stage.MuteLayer(layer.identifier)

    def open_header_menu(self, position):
        menu = HeaderVisibilityMenu(self.view)
        menu.set_column_permanent(0, True)
        menu.exec_(self.view.header().mapToGlobal(position))

    @staticmethod
    def on_destroyed(self):
        # invalidate model, so it will not receive callbacks
        self.view.model().ResetStage(None)
        self._editTargetChangeCallback = None
        self._listener_obj_chgd = None
        self._listener_ed_tgt_chgd = None

    def _create_menu_bar(self):
        self._menubar = QtWidgets.QMenuBar()
        self.layout().setMenuBar(self._menubar)
        self._menu_show = QtWidgets.QMenu(i18n("layer_widget", "&Show"), self._menubar)
        self._menubar.addMenu(self._menu_show)

        act_show_refs = QtWidgets.QWidgetAction(self._menu_show)
        checkbox_refs = QtWidgets.QCheckBox(i18n("layer_widget", "References"), self)
        act_show_refs.setDefaultWidget(checkbox_refs)
        checkbox_refs.setChecked(self._filterModel._show_references)
        checkbox_refs.toggled.connect(lambda state: self._filterModel.set_filter('references', state))
        self._menu_show.addAction(act_show_refs)

        act_show_dirty = QtWidgets.QWidgetAction(self._menu_show)
        checkbox_dirty = QtWidgets.QCheckBox(i18n("layer_widget", "Only Dirty Layers"), self)
        act_show_dirty.setDefaultWidget(checkbox_dirty)
        checkbox_dirty.setChecked(self._filterModel._show_only_dirty)
        checkbox_dirty.toggled.connect(lambda state: self._filterModel.set_filter('dirty', state))
        self._menu_show.addAction(act_show_dirty)

        act_show_anon = QtWidgets.QWidgetAction(self._menu_show)
        checkbox_anon = QtWidgets.QCheckBox(i18n("layer_widget", "Anonymous Layers"), self)
        act_show_anon.setDefaultWidget(checkbox_anon)
        checkbox_anon.setChecked(self._filterModel._show_only_anon)
        checkbox_anon.toggled.connect(lambda state: self._filterModel.set_filter('anon', state))
        self._menu_show.addAction(act_show_anon)

        act_show_empty = QtWidgets.QWidgetAction(self._menu_show)
        checkbox_empty = QtWidgets.QCheckBox(i18n("layer_widget", "Empty Layers"), self)
        act_show_empty.setDefaultWidget(checkbox_empty)
        checkbox_empty.setChecked(self._filterModel._show_only_empty)
        checkbox_empty.toggled.connect(lambda state: self._filterModel.set_filter('empty', state))
        self._menu_show.addAction(act_show_empty)

    def apply_filter(self, string):
        self.view._blockSelectionCallback = True
        regexp = QtCore.QRegExp('^'+string)
        regexp.setCaseSensitivity(QtCore.Qt.CaseInsensitive)
        self._filterModel.setFilterRegExp(regexp)
        self.view._blockSelectionCallback = False

    @QtCore.Slot(QtCore.QModelIndex)
    def ChangeEditTarget(self, modelIndex):
        # type: (QtCore.QModelIndex) -> None
        """
        Parameters
        ----------
        modelIndex : QtCore.QModelIndex
        """
        if not modelIndex.isValid():
            return
        source_index = self._filterModel.mapToSource(modelIndex)
        item = source_index.internalPointer()
        newLayer = item.layer

        if self._stage.IsLayerMuted(newLayer.identifier):
            print("Can't change edit target on muted layer")
            return

        if item.is_ref:
            #print("Can't change edit target on reference. Use Composition Editor for this purpose")
            print("Can't change edit target on reference")
            return

        if self._editTargetChangeCallback is None \
                or self._editTargetChangeCallback(newLayer):
            self._stage.SetEditTarget(newLayer)

    def find_new_layer_identifier(self, parent_layer):
        orig_name = 'new_layer'
        new_identifier = os.path.join(os.path.dirname(parent_layer.realPath), orig_name + '.usd')
        num = 0
        while True:
            tryed_layer = Sdf.Layer.Find(new_identifier)
            if not tryed_layer:
                return new_identifier
            num += 1
            new_identifier = os.path.join(os.path.dirname(parent_layer.realPath), orig_name  + str(num) + '.usd')

    def add_sublayer(self):
        layers = self.view.GetSelectedLayers()
        if not layers and self._stage:
            layer = self._stage.GetRootLayer()
        else:
            layer = layers[0]
        if layer:
            new_layer_name = self.find_new_layer_identifier(layer)
            with UndoBlock():
                new_layer = Sdf.Layer.New(Sdf.FileFormat.FindById('usd'), new_layer_name)
                if new_layer.identifier not in layer.subLayerPaths:
                    layer.subLayerPaths.append(new_layer.identifier)
            self.refresh_after_adding(parent_layer=layer)

    def refresh_after_adding(self, parent_layer=None):
        self.refresh()
        if parent_layer:
            self.view.set_expanded_layer(parent_layer, True)
            self.view.select_layer(parent_layer)

    def refresh(self):
        self.view.save_expanded_state()
        self.set_stage(self._stage)
        self.view.restore_expanded_state()

    def GetMenuContext(self):
        stage = self._stage
        if not stage:
            return
        return LayerStackDialogContext(
            qtParent=self.parent() or self,
            layerDialog=self,
            stage=stage,
            selectedLayers=self.view.GetSelectedLayers(),
            editTargetLayer=stage.GetEditTarget().GetLayer(),
            selectedItems=self.view.GetSelectedItems())
