from Qt import QtCore, QtGui, QtWidgets
from pxr import Sdf, Usd
import os
from dcc.undo import UsdEditsUndoBlock


def get_reference(file_path, prim_path):
    if not file_path and not prim_path:
        return None
    elif not prim_path:
        new_ref = Sdf.Reference(file_path)
    elif not file_path:
        new_ref = Sdf.Reference(primPath=prim_path)
    else:
        new_ref = Sdf.Reference(file_path, prim_path)
    return new_ref


class PrimSpecTreeWidget(QtWidgets.QTreeWidget):
    """Simple Tree to represent SdfPrimSpec hierarchy for current layer"""
    def __init__(self, layer=None, parent=None):
        QtCore.QAbstractItemModel.__init__(self, parent)
        self.setColumnCount(1)
        self.setHeaderHidden(True)
        self.update(layer)

    def update(self, layer):
        if not layer:
            return

        def traverse_prim(parent, prim):
            item = QtWidgets.QTreeWidgetItem(parent, [prim.name])
            item._path = prim.path
            for child_prim in prim.nameChildren:
                traverse_prim(item, child_prim)

        for prim in layer.rootPrims:
            traverse_prim(self, prim)


class PrimPathDialog(QtWidgets.QDialog):
    """Dialog for selecting prim path in the layer"""
    def __init__(self, layer, parent=None):
        QtWidgets.QDialog.__init__(self, parent=parent)
        self._layout = QtWidgets.QVBoxLayout(self)
        self._layout.setContentsMargins(3, 3, 3, 3)

        self._tree = PrimSpecTreeWidget(layer)
        self._layout.addWidget(self._tree)

        # OK and Cancel buttons
        buttons = QtWidgets.QDialogButtonBox(
            QtWidgets.QDialogButtonBox.Ok | QtWidgets.QDialogButtonBox.Cancel,
            QtCore.Qt.Horizontal, self)
        buttons.accepted.connect(self.accept)
        buttons.rejected.connect(self.reject)

        self._tree.itemDoubleClicked.connect(self.accept)
        self._layout.addWidget(buttons)

    def selected_path(self):
        items = self._tree.selectedItems()
        if len(items) > 0:
            item = items[0]
            return item._path
        return None

    @staticmethod
    def get_prim_path(path):
        dialog = PrimPathDialog(path)
        res = dialog.exec_()
        if res:
            return dialog.selected_path()
        else:
            return None


class PathEditor(QtWidgets.QWidget):
    data_changed = QtCore.Signal()
    accept_signal = QtCore.Signal()
    reject_signal = QtCore.Signal()

    def __init__(self, prim=None, empty_file_path=True, btn_title='Add', parent=None):
        QtWidgets.QWidget.__init__(self, parent)

        self._prim = prim
        self._empty_file_path = empty_file_path

        self._main_layout = QtWidgets.QVBoxLayout(self)
        self._main_layout.setContentsMargins(3, 0, 3, 0)
        self._main_layout.setAlignment(QtCore.Qt.AlignTop | QtCore.Qt.AlignLeft)

        self._layout = QtWidgets.QGridLayout()
        self._main_layout.addLayout(self._layout)
        self._layout.setContentsMargins(0, 0, 3, 0)
        self._layout.setSpacing(2)
        self._layout.setAlignment(QtCore.Qt.AlignTop | QtCore.Qt.AlignLeft)

        self._file_path_lbl = QtWidgets.QLabel('File Path')
        self._prim_path_lbl = QtWidgets.QLabel('Prim Path')

        self._file_path_le = QtWidgets.QLineEdit()
        self._prim_path_le = QtWidgets.QLineEdit()

        self._file_dlg_btn = QtWidgets.QPushButton('...', self)
        self._file_dlg_btn.setMaximumWidth(20)
        self._file_dlg_btn.pressed.connect(self.open_file_dialog)
        self._file_path_le.editingFinished.connect(self.show_hide_prim_btn)
        self._file_path_le.textChanged.connect(self.show_hide_prim_btn)

        self._prim_dlg_btn = QtWidgets.QPushButton('...', self)
        self._prim_dlg_btn.setMaximumWidth(20)
        self._prim_dlg_btn.pressed.connect(self.open_prim_dialog)
        self.show_hide_prim_btn()

        self._layout.addWidget(self._file_path_lbl, 0, 0)
        self._layout.addWidget(self._prim_path_lbl, 1, 0)
        self._layout.addWidget(self._file_path_le, 0, 1)
        self._layout.addWidget(self._prim_path_le, 1, 1)
        self._layout.addWidget(self._file_dlg_btn, 0, 2)
        self._layout.addWidget(self._prim_dlg_btn, 1, 2)

        self._controls_layout = QtWidgets.QHBoxLayout()
        self._controls_layout.setContentsMargins(3, 0, 3, 0)
        self._main_layout.addLayout(self._controls_layout)
        self._btn = QtWidgets.QPushButton(btn_title, self)
        self._controls_layout.addWidget(self._btn)
        self._controls_layout.addWidget(QtWidgets.QSplitter())
        self._btn_cnsl = QtWidgets.QPushButton('Cancel', self)
        self._controls_layout.addWidget(self._btn_cnsl)

    def set_button_text(self, text):
        self._btn.setText(text)

    def set_file_path(self, path):
        self._file_path_le.setText(path)

    def file_path(self):
        return self._file_path_le.text()

    def set_prim_path(self, path):
        self._prim_path_le.setText(str(path))

    def prim_path(self):
        return self._prim_path_le.text()

    def connect_button(self, func):
        self._btn.pressed.connect(func)
        self._btn.pressed.connect(self.accept_signal)
        self._btn_cnsl.pressed.connect(self.reject_signal)

    def open_file_dialog(self):
        file_name = QtWidgets.QFileDialog.getOpenFileName(self, 'Select file',
                                            self._file_path_le.text(), "Image files (*.usd *.usda *.usdc)")
        if file_name and file_name[0]:
            self._file_path_le.setText(file_name[0])

    def show_hide_prim_btn(self):
        visible = False
        file_path = self.file_path()
        # is file path valid - open stage from file,
        # if it's empty and empty file path is permitted - select prim in current stage
        if os.path.isfile(file_path) or (self.is_file_path_empty() and self._empty_file_path):
            visible = True

        self._prim_dlg_btn.setVisible(visible)

    def open_prim_dialog(self):
        if self.is_file_path_empty() and self._empty_file_path:
            layer = self._prim.GetStage().GetRootLayer()
        else:
            file_path = self.file_path()
            if not os.path.isfile(file_path):
                return
            layer = Sdf.Layer.FindOrOpen(file_path)

        if not layer:
            return

        res = PrimPathDialog.get_prim_path(layer)
        if res:
            self._prim_path_le.setText(str(res))

    def is_path_empty(self):
        if not str(self.file_path()) and not str(self.prim_path()):
            return True
        return False

    def is_file_path_empty(self):
        if not str(self.file_path()):
            return True
        return False

    def is_prim_path_empty(self):
        if not str(self.prim_path()):
            return True
        return False

    def clear_edits(self):
        self._file_path_le.setText('')
        self._prim_path_le.setText('')


class EditPayloadWidget(PathEditor):
    data_changed = QtCore.Signal()
    accept_signal = QtCore.Signal()
    reject_signal = QtCore.Signal()

    def __init__(self, prim, path=None, for_add=True, parent=None):
        PathEditor.__init__(self, prim, empty_file_path=False, parent=parent)
        if not for_add:
            self.set_button_text('Apply')

        self._prim = prim
        self._for_add = for_add

        self._current_payload = path
        if self._current_payload and not for_add:
            self.set_file_path(self._current_payload.assetPath)
            self.set_prim_path(self._current_payload.primPath)

        if for_add:
            self.connect_button(self.add_new_path)
        else:
            self.connect_button(self.edit_path)

    def get_payload(self):
        new_payload = None
        if self.is_prim_path_empty():
            new_payload = Sdf.Payload(self.file_path())
        elif not self.is_prim_path_empty():
            new_payload = Sdf.Payload(self.file_path(), self.prim_path())
        return new_payload

    def add_new_path(self):
        if self._current_payload and self._current_payload.assetPath == self.file_path()\
                and self._current_payload.primPath == self.prim_path():
            return

        if self.is_path_empty():
            print("Couldn't set empty payload")
            return

        new_payload = self.get_payload()

        if new_payload:
            payloads = self._prim.GetPayloads()
            with UsdEditsUndoBlock():
                res = payloads.AddPayload(new_payload)

                if not res:
                    print("Payload couldn't be set")
                else:
                    self.data_changed.emit()

    def edit_path(self):
        if not self._prim and not self._current_payload:
            return

        new_payload = self.get_payload()
        if not new_payload:
            print("Couldn't change reference on empty")
            return

        stage = self._prim.GetStage()
        layer = stage.GetEditTarget().GetLayer()
        prim_spec = layer.GetPrimAtPath(self._prim.GetPath())
        if not prim_spec:
            print("Couldn't change reference on current edit target")
            return

        payload_spec = prim_spec.payloadList
        with UsdEditsUndoBlock():
            payload_spec.ReplaceItemEdits(self._current_payload, new_payload)


class EditReferenceWidget(PathEditor):
    data_changed = QtCore.Signal()
    accept_signal = QtCore.Signal()
    reject_signal = QtCore.Signal()

    def __init__(self, prim, path=None, for_add=True, parent=None):
        PathEditor.__init__(self, prim, parent)
        self._prim = prim
        self._ref = path

        if not for_add:
            self.set_button_text('Apply')
            self.connect_button(self.edit_ref)

            if self._ref:
                self.set_file_path(self._ref.assetPath)
                self.set_prim_path(self._ref.primPath)
        else:
            self.connect_button(self.add_new_ref)

    def get_reference(self):
        return get_reference(self.file_path(), self.prim_path())

    def add_new_ref(self):
        new_ref = self.get_reference()
        res = False
        if new_ref:
            with UsdEditsUndoBlock():
                res = self._prim.GetReferences().AddReference(new_ref)

                if res:
                    self.data_changed.emit()

    def edit_ref(self):
        if not self._prim and not self._ref:
            return

        new_ref = self.get_reference()
        if not new_ref:
            print("Couldn't change reference on empty")
            return

        stage = self._prim.GetStage()
        layer = stage.GetEditTarget().GetLayer()
        prim_spec = layer.GetPrimAtPath(self._prim.GetPath())
        if not prim_spec:
            print("Couldn't change reference on current edit target")
            return

        refs_spec = prim_spec.referenceList
        with UsdEditsUndoBlock():
            refs_spec.ReplaceItemEdits(self._ref, new_ref)


class ArcDialog(QtWidgets.QDialog):
    data_changed = QtCore.Signal()

    def __init__(self, prim, is_for_adding=True, parent=None):
        QtWidgets.QWidget.__init__(self, parent)

        self._prim = prim
        self._is_for_adding = is_for_adding

        self._layout = QtWidgets.QVBoxLayout(self)
        self._layout.setAlignment(QtCore.Qt.AlignTop)

        self._editor = QtWidgets.QWidget()

    def set_new_editor(self, editor_type=None, **kwargs):
        self._layout.removeWidget(self._editor)
        self._editor.deleteLater()

        if editor_type == PathEditor:
            self._editor = PathEditor(parent=kwargs.get('parent'))
        elif editor_type in (EditPayloadWidget, EditReferenceWidget):
            self._editor = editor_type(self._prim,
                                             path=kwargs.get('val'),
                                             for_add=self._is_for_adding,
                                             parent=kwargs.get('parent'))
        else:
            self._editor = QtWidgets.QWidget()

        if type(self._editor) != QtWidgets.QWidget:
            self._editor.data_changed.connect(self.data_changed)
            self._editor.accept_signal.connect(self.accept)
            self._editor.accept_signal.connect(self.close)
            self._editor.reject_signal.connect(self.reject)
            self._editor.reject_signal.connect(self.close)

        self._layout.addWidget(self._editor)
