from pxr.UsdQtEditors._Qt import QtCore, QtGui, QtWidgets
from pxr.UsdQt.usdUtils import GetPrimVariants
from functools import partial
from pxr import Sdf, Tf, Usd, UsdGeom
from .hierarchyModel import HierarchyModel, FilterModel
from .modelColumn import NameColumn, VisibilityColumn, MetaDataColumn, VariantColumn, \
    ColumnController, AttributeColumn, RelationshipColumn, IsOnEditTargetRole

from pxr.UsdQt.qtUtils import MenuAction, MenuSeparator, \
    MenuBuilder, ContextMenuMixin, MenuBarBuilder
from pxr.UsdQtEditors.outliner import UsdOutliner as UsdQtOutliner, LayerStackModel, \
    SaveEditLayer, ShowEditTargetLayerText, ShowEditTargetDialog, SaveState, \
    RemovePrim as RemovePrimUsdQt
from .i18n import i18n
from ._func import is_prim_spec_on_edit_target, change_prim_path, \
    reparent_paths, remove_prims
from .widgets.header_visibility_menu import HeaderVisibilityMenu
import os

try:
    from dcc.undo import UsdEditsUndoBlock as UndoBlock
except ImportError:
    from pxr.UsdQt._usdQt import UndoBlock

from .add_branches import add_branches


_DCC = False
try:
    from wizart.common_widgets import SearchWidget
    _DCC = True
except ImportError:
    _DCC = False


class SelectVariants(MenuAction):

    NO_VARIANT_SELECTION = '<No Variant Selected>'

    @staticmethod
    def _ApplyVariant(prim, variantSetName, variantValue):
        if prim:
            variantSet = prim.GetVariantSet(variantSetName)
            if variantValue == SelectVariants.NO_VARIANT_SELECTION:
                variantSet.ClearVariantSelection()
            else:
                variantSet.SetVariantSelection(variantValue)

    def _ApplyVariantToMultiple(self, prims, variantSetName, variantValue):
        for prim in prims:
            self._ApplyVariant(prim, variantSetName, variantValue)

    def _GetIntersectionData(self, prims):
        compareData = {}
        for prim in prims:
            primData = self._GetSingleData(prim)
            if primData is None:
                return
            if not compareData:
                compareData = primData
                continue
            difference = set(compareData.keys()).symmetric_difference(set(primData))
            for differentVariant in difference:
                if differentVariant in compareData:
                    compareData.pop(differentVariant)

            for variant in compareData.keys():
                sameVariantNames = [x for x in compareData[variant] if x in primData[variant]]
                if sameVariantNames:
                    compareData.update({variant: sameVariantNames})
                else:
                    compareData.pop(variant)

        return compareData

    def _GetSingleData(self, prim):
        if not prim.HasVariantSets():
            return
        primData = {}
        for setName, currentValue in GetPrimVariants(prim):
            primData.update({setName: prim.GetVariantSet(setName).GetVariantNames()})
        return primData

    def _CreateMenu(self, context, prims, primsData):
        if prims:
            checkedValues = {}
            for setName, currentValue in GetPrimVariants(prims[0]):
                checkedValues.update({setName: currentValue})
            variantsMenu = QtWidgets.QMenu(i18n("outliner.select_variants", "Variants"), context.qtParent)
            for variant in primsData:
                menu = variantsMenu.addMenu(variant)
                for variantName in [SelectVariants.NO_VARIANT_SELECTION] + primsData[variant]:
                    a = menu.addAction(variantName)
                    a.setCheckable(True)
                    if variantName == checkedValues[variant]:
                        a.setChecked(True)
                    a.triggered.connect(partial(self._ApplyVariantToMultiple, prims, variant, variantName))

            return variantsMenu.menuAction()

    def Build(self, context):
        primsData = {}
        prims = context.selectedPrims
        if len(prims) > 1:
            primsData = self._GetIntersectionData(prims)
        elif len(prims) == 1:
            primsData = self._GetSingleData(prims[0])

        if not primsData:
            return

        return self._CreateMenu(context, prims, primsData)

def SetVisibility(index, flag):
    index.model().setData(index, flag, QtCore.Qt.EditRole)
    

class SetPrimsVisible(MenuAction):
    defaultText = i18n("outliner", "Set Visible")

    def Update(self, action, context):
        action.setEnabled(bool(context.selectedPrims))

    def Do(self):
        context = self.GetCurrentContext()
        with Sdf.ChangeBlock(), UndoBlock():
            for prim in context.selectedPrims:
                if prim.IsPseudoRoot(): return
                vis = prim.GetAttribute('visibility')
                if vis.IsValid():
                    vis.Set(UsdGeom.Tokens.inherited)


class SetPrimsInvisible(MenuAction):
    defaultText = i18n("outliner", "Set Invisible")

    def Update(self, action, context):
        action.setEnabled(bool(context.selectedPrims))

    def Do(self):
        context = self.GetCurrentContext()
        with Sdf.ChangeBlock(), UndoBlock():
            for prim in context.selectedPrims:
                if prim.IsPseudoRoot(): return
                vis = prim.GetAttribute('visibility')
                if vis.IsValid():
                    vis.Set(UsdGeom.Tokens.invisible)


class ActivatePrims(MenuAction):
    defaultText = i18n("outliner", "Activate")

    def Update(self, action, context):
        action.setEnabled(bool(context.selectedPrims))

    def Do(self):
        context = self.GetCurrentContext()
        with Sdf.ChangeBlock(), UndoBlock():
            for prim in context.selectedPrims:
                if prim.IsPseudoRoot(): continue
                prim.SetActive(True)


class DeactivatePrims(MenuAction):
    defaultText = i18n("outliner", "Deactivate")

    def Update(self, action, context):
        action.setEnabled(bool(context.selectedPrims))

    def Do(self):
        context = self.GetCurrentContext()
        with Sdf.ChangeBlock(), UndoBlock():
            for prim in context.selectedPrims:
                if prim.IsPseudoRoot(): continue
                prim.SetActive(False)


class RemovePrim(RemovePrimUsdQt):
    def Update(self, action, context):
        flag = False
        for prim in context.selectedPrims:
            if is_prim_spec_on_edit_target(prim, context.stage.GetEditTarget()):
                flag = True
                break
        if not context.stage or not flag:
            action.setEnabled(False)
        else:
            RemovePrimUsdQt.Update(self, action, context)

    def Do(self):
        context = self.GetCurrentContext()
        paths = [prim.GetPath() for prim in context.selectedPrims]
        with Sdf.ChangeBlock(), UndoBlock():
            remove_prims(paths, context.stage, ask=False)


class CopyPrimPath(MenuAction):
    defaultText = i18n("outliner", "Copy Prim Path")

    def Update(self, action, context):
        action.setEnabled(bool(context.selectedPrims))

    def Do(self):
        context = self.GetCurrentContext()
        clipboard = QtWidgets.QApplication.clipboard()
        prim_paths = []

        for prim in context.selectedPrims:
            if prim.IsPseudoRoot(): continue
            prim_paths.append(str(prim.GetPath()))

        if len(prim_paths) == 1:
            clipboard.setText(prim_paths[0])
        elif len(prim_paths) > 1:
            clipboard.setText(str(prim_paths))


class HeaderMenu(QtWidgets.QMenu):
    def __init__(self, parent):
        QtWidgets.QMenu.__init__(self, parent=parent)
        self.action_dict = {}
        self.view = parent

        self.aboutToShow.connect(self.update_actions_list)

    def update_actions_list(self):
        self.clear()
        self.select_deselect_action = self.add_action(i18n("outliner", "Select all/Deselect all"),
                                                      False,
                                                      lambda checked_flag:
                                                      self.select_deselect_all(checked_flag))
        self.addSeparator()

        titles = self.view.model().columns_controller().order_lst
        for title in titles:
            column_index = self.view.model().columns_controller().index(title)
            is_hidden = self.view.header().isSectionHidden(column_index)
            func = (lambda arg: lambda checked_flag:
                    self.change_state(checked_flag, arg))\
                   (column_index)
            self.action_dict[title] = self.add_action(title,
                                                      not is_hidden,
                                                      func)

        self.block_update_state = False
        self.update_aggr_chbx_state()

    def add_action(self, name, checked, func):
        action = QtWidgets.QWidgetAction(self)
        checkBox = QtWidgets.QCheckBox(name, self)
        action.setDefaultWidget(checkBox)
        checkBox.setChecked(checked)
        checkBox.toggled.connect(func)
        checkBox.toggled.connect(self.update_aggr_chbx_state)
        self.addAction(action)
        return action

    def change_state(self, checked, column_index):
        if checked:
            self.view.showColumn(column_index)
        if not checked:
            self.view.hideColumn(column_index)

    def select_deselect_all(self, checked_flag):
        if self.block_update_state:
            return
        self.block_update_state = True
        for action in self.action_dict.values():
            action.defaultWidget().setChecked(checked_flag)
        self.block_update_state = False

    def update_aggr_chbx_state(self):
        if self.block_update_state:
            return
        self.block_update_state = True
        shown = True
        hidden = True
        for action in self.action_dict.values():
            chbx_state = action.defaultWidget().isChecked()
            shown = shown and chbx_state
            hidden = hidden and not chbx_state

        if shown:
            self.select_deselect_action.defaultWidget().setChecked(True)
        elif hidden:
            self.select_deselect_action.defaultWidget().setChecked(False)
        else:
            self.select_deselect_action.defaultWidget().setCheckState(QtCore.Qt.PartiallyChecked)
        self.block_update_state = False


@add_branches(-1)
class OutlinerTreeView(ContextMenuMixin, QtWidgets.QTreeView):
    # Emitted with lists of selected and deselected prims
    primSelectionChanged = QtCore.Signal(list, list)
    primHighlightingChanged = QtCore.Signal(list)

    def __init__(self, dataModel, contextMenuActions, contextProvider=None,
                 parent=None):
        # type: (QtCore.QAbstractItemModel, List[MenuAction], Optional[ContextProvider], Optional[QtWidgets.QWidget]) -> None
        """
        Parameters
        ----------
        dataModel : QtCore.QAbstractItemModel
        contextMenuActions : List[MenuAction]
        contextProvider : Optional[ContextProvider]
        parent : Optional[QtWidgets.QWidget]
        """
        ContextMenuMixin.__init__(self,
            contextMenuActions=contextMenuActions,
            contextProvider=contextProvider,
            parent=parent)

        self.setAlternatingRowColors(True)
        self.setSelectionBehavior(self.SelectRows)
        self.setSelectionMode(self.ExtendedSelection)
        self.setEditTriggers(self.DoubleClicked)
        self.setState(self.DraggingState)

        self.setUniformRowHeights(True)

        dataModel.needSelectIndexes.connect(self.select_tree_items_by_path)
        self.setModel(dataModel)
        self._blockSelectionCallback = False

        self.clicked.connect(lambda index: self.changeVisibilityClicked(index), QtCore.Qt.UniqueConnection)

        self.header().setContextMenuPolicy(QtCore.Qt.CustomContextMenu)
        self.header().customContextMenuRequested.connect(self.open_header_menu)

        self.setDragEnabled(True)
        self.setAcceptDrops(True)
        self.setDragDropMode(QtWidgets.QAbstractItemView.DragDrop)
        self.setDropIndicatorShown(True)

        self.setAllColumnsShowFocus(True)

        self.connectSignals()

        self.expanded.connect(self.expand)
        self.collapsed.connect(self.collapse)
        self.setProperty("unfocusedKeyEvent_enable", True)


    def setExpandedRecursively(self, index, state): 
        self.blockSignals(True)
        parents = [index]
        while parents:
            parent = parents.pop()
            for row in range(self.model().rowCount(parent)):
                child = self.model().index(row, 0, parent)
                if not child.isValid():
                    break
                parents.append(child)
                self.setExpanded(child, state)
        self.blockSignals(False)

    def expand(self, index):
        app = QtWidgets.QApplication.instance()
        if (app.keyboardModifiers() & QtCore.Qt.ShiftModifier) == QtCore.Qt.ShiftModifier:
            self.setExpandedRecursively(index, True)

    def collapse(self, index):
        app = QtWidgets.QApplication.instance()
        if (app.keyboardModifiers() & QtCore.Qt.ShiftModifier) == QtCore.Qt.ShiftModifier:
            self.setExpandedRecursively(index, False)

    def set_column_settings(self):
        header = self.header()
        vis_title = self.model().columns_controller().vis.header_name
        vis_index = self.model().columns_controller().index(vis_title)

        header.resizeSection(vis_index, 20)
        header.setSectionResizeMode(vis_index, QtWidgets.QHeaderView.Fixed)
        header.setStretchLastSection(True)
        header.setDefaultAlignment(QtCore.Qt.AlignCenter)
        self.setTreePosition(self.model().columns_controller().tree_index)

        for col in self.model().columns_controller():
            self.connect_delegate_to_column(col)
            if not col.visible:
                self.hideColumn(self.model().columns_controller().index(col.header_name))

    def connect_delegate_to_column(self, column):
        if column.has_delegate:
            delegate = column.delegate(self._dataModel)
            self.setItemDelegateForColumn(
                self.model().columns_controller().index(column.header_name), delegate)

    def connectSignals(self):
        """
        Connect signals. Relevant mostly when the model settled
        """

        # This can't be a one-liner because of a PySide refcount bug.
        selectionModel = self.selectionModel()
        selectionModel.selectionChanged.connect(self._SelectionChanged)

        self.primHighlightingChanged.connect(self._dataModel.updateHighlighted)

        self._dataModel.repaintView.connect(lambda: self.dataChanged(QtCore.QModelIndex(), QtCore.QModelIndex()))

    def selection_to_prims(self, qSelection):
        return [self.model()._GetPrimForIndex(index)
                for index in qSelection.indexes()
                if index.column() == self.model().getColumnIndex(self.model().columns_controller().name.header_name)]

    @QtCore.Slot(QtCore.QItemSelection, QtCore.QItemSelection)
    def _SelectionChanged(self, selected, deselected):
        """Connected to selectionChanged"""

        self.primSelectionChanged.emit(self.selection_to_prims(selected), self.selection_to_prims(deselected))

        self.update_highlighted_parents()
        self.primHighlightingChanged.emit(self.highlightedParents)

    def update_highlighted_parents(self):
        """
        Update list of highlighted parents after changing selection
        """
        self.highlightedParents = []
        for index in self.selectedIndexes():
            if self._proxyModel:
                index = self._proxyModel.mapToSource(index)
            self.add_highlighted_parents(index)

    def add_highlighted_parents(self, index):
        parent = index.parent()
        while True:
            if not parent.isValid():
                return

            if parent not in self.highlightedParents:
                self.highlightedParents.append(parent)
                parent = parent.parent()
            else:
                return

    def SelectedPrims(self):
        # type: () -> List[Usd.Prim]
        """
        Returns
        -------
        List[Usd.Prim]
        """
        result = []
        for index in self.selectionModel().selectedRows():
            prim = self.model()._GetPrimForIndex(index)
            if prim:
                result.append(prim)
        return result

    def ChangeVis(self, index):
        prim = self.model()._GetPrimForIndex(index)
        if prim.IsPseudoRoot():
            return

        vis_attr = prim.GetAttribute('visibility')
        vis = vis_attr.Get()
        if vis:
            SetVisibility(index, not vis == UsdGeom.Tokens.inherited)

    def changeVisibilityClicked(self, index):
        """
        Change visibility of a prim my clicking on the Vis column
        :param index: QModelIndex
        """
        if self.model().columns_controller().vis == self.model().columns_controller()[index.column()]:
            self.ChangeVis(index)
            # update view from current row to the bottom
            # to change visibility column for children
            # TODO: if necessary rewrite to go only through children
            # and not to the lowest bottom
            QtWidgets.QTreeView.dataChanged(self, index, QtCore.QModelIndex())

    def setModel(self, model):
        """
        Overwrite QTreeView method to save reference to model in local namespace
        and connect signals. For it only needed when model updated
        """
        QtWidgets.QTreeView.setModel(self, model)
        self._dataModel = model
        self.setRoot()
        self.set_column_settings()

    def setProxyModel(self, proxy):
        """
        Squeeze sortFilterProxyModel
        :param proxy:
        :return:
        """
        _dataModel = self.model()
        self._proxyModel = proxy
        self._proxyModel.setSourceModel(self._dataModel)
        self.setModel(self._proxyModel)
        self._dataModel = _dataModel
        self.selectionModel().selectionChanged.connect(self._SelectionChanged)

    def setRoot(self):
        """
        TODO Temporary, while i didn't get how to get rid of pseudo root otherwise
        """
        self.setRootIndex(self.model().index(0, 0))

    def open_header_menu(self, position):
        menu = HeaderVisibilityMenu(self)
        menu.set_column_permanent(self.model().columns_controller().tree_index, True)
        menu.exec_(self.header().mapToGlobal(position))

    def startDrag(self, supportedActions):
        selected = self.selectedIndexes()
        mouse_state = QtWidgets.QApplication.mouseButtons()
        if selected and mouse_state == QtCore.Qt.MiddleButton:
            mime_data = self.model().mimeData(selected)
            dragQDrag = QtGui.QDrag(self)
            # dragQDrag.setPixmap(QtGui.QPixmap('test.jpg')) # <- For put your custom image here
            dragQDrag.setMimeData(mime_data)
            defaultDropAction = QtCore.Qt.IgnoreAction
            if ((supportedActions & QtCore.Qt.CopyAction) and (
                    self.dragDropMode() != QtWidgets.QAbstractItemView.InternalMove)):
                defaultDropAction = QtCore.Qt.CopyAction
            dragQDrag.exec_(supportedActions, defaultDropAction)

    def dropEvent(self, event):
        pos = event.pos()
        index = self.indexAt(pos)
        indicator_position = self.dropIndicatorPosition()

        if not index.isValid():
            new_parent_prim = self.model().get_stage().GetPrimAtPath('/')
        else:
            if indicator_position != self.OnItem:
                return
            new_parent_prim = self.model()._GetPrimForIndex(index)

        mime_data = event.mimeData()
        with UndoBlock():
            res, new_paths = reparent_paths(new_parent_prim, mime_data.paths_list)

            if res:
                self.select_tree_items_by_path(new_paths)

    def select_tree_items_by_path(self, paths):
        """Select items of TreeView by Prim paths.
        Works a bit faster because aggregate QModelIndexes into ranges"""

        def divide_to_segments(num_list):
            """Divide list of numbers.
            Each segment will contain continuous progressive list of numbers"""
            res_lst = []
            num_list.sort()

            segment = [num_list[0]]
            for ind in range(1, len(num_list)):
                if num_list[ind] == num_list[ind - 1]:
                    continue
                if num_list[ind]-1 != num_list[ind-1]:
                    # save old segment and start new segment
                    res_lst.append(segment)
                    segment = []
                segment.append(num_list[ind])

            # add last segment
            res_lst.append(segment)
            return res_lst

        # make dictionary of indexes
        d = {}
        for prim_path in paths:
            index = self.model().GetIndexForPath(prim_path)

            # if no index found, load all parents by model
            if index is None:
                self._dataModel.force_load_children_to_path(prim_path)
                index = self.model().GetIndexForPath(prim_path)

            # if finally found map to parent (to make ranges further)
            if index is not None:
                indexes = d.get(index.parent())
                if indexes:
                    indexes.append(index.row())
                else:
                    d[index.parent()] = [index.row()]

        # modify indexes to make ranges
        for key, val in d.items():
            d[key] = divide_to_segments(val)

        qSelection = QtCore.QItemSelection()
        right_column = self.model().columnCount(QtCore.QModelIndex()) - 1
        for key, ranges in d.items():
            for row_range in ranges:
                top_left = self.model().index(row_range[0], 0, key)
                bottom_right = self.model().index(row_range[-1], right_column, key)
                qSelection.select(top_left, bottom_right)

        selModel = self.selectionModel()
        if qSelection.isEmpty():
            selModel.clearSelection()
        else:
            # Better not to use flag QItemSelectionModel.Rows
            # because it's slower than if you send all indexes in row in one range
            selModel.select(qSelection,
                            QtCore.QItemSelectionModel.SelectCurrent |
                            QtCore.QItemSelectionModel.Clear)

        self.primSelectionChanged.emit(self.selection_to_prims(qSelection), [])

    def select_tree_items_by_path_second_way(self, paths):
        """This easier to test, but a bit slower"""
        right_column = self.model().columnCount(QtCore.QModelIndex()) - 1

        qSelection = QtCore.QItemSelection()
        for prim_path in paths:
            index = self.model().GetIndexForPath(prim_path)

            if index is None:
                self._dataModel.force_load_children_to_path(prim_path)

                index = self.model().GetIndexForPath(prim_path)
            if index is not None:
                top_left = self.model().index(index.row(), 0, index.parent())
                bottom_right = self.model().index(index.row(), right_column, index.parent())
                qSelection.select(top_left, bottom_right)

        selModel = self.selectionModel()
        if qSelection.isEmpty():
            selModel.clearSelection()
        else:
            selModel.select(qSelection,
                            QtCore.QItemSelectionModel.SelectCurrent |
                            QtCore.QItemSelectionModel.Clear)

    def frame_selection(self):
        selected = self.selectedIndexes()
        for index in selected:
            self.expand_parents(index)
        if selected:
            self.scrollTo(selected[0])

    def keyPressEvent(self, event):
        """Overloaded function for hotkeys"""
        if event.key() == QtCore.Qt.Key_F and event.modifiers() == QtCore.Qt.NoModifier:
            self.frame_selection()
        else:
            QtWidgets.QTreeView.keyPressEvent(self, event)

    def expand_parents(self, index):
        while True:
            index = index.parent()
            if index.isValid():
                self.expand(index)
            else:
                break


class OutlinerRole(object):
    """Helper which provides standard hooks for defining the context menu
    actions and menu bar menus that should be added to an outliner.
    """
    @classmethod
    def GetContextMenuActions(cls, outliner):
        # type: (UsdOutliner) -> List[Union[MenuAction, Type[MenuAction]]]
        """
        Parameters
        ----------
        outliner : UsdOutliner

        Returns
        -------
        List[Union[MenuAction, Type[MenuAction]]]
        """
        return [CopyPrimPath,
                MenuSeparator, ActivatePrims, DeactivatePrims, SetPrimsVisible, SetPrimsInvisible, SelectVariants,
                MenuSeparator, RemovePrim]


class UsdOutliner(UsdQtOutliner, QtWidgets.QWidget):
    """UsdStage editing application which displays the hierarchy of a stage."""
    def __init__(self, stage=None, role=None, parent=None):
        """
        Parameters
        ----------
        stage : Usd.Stage
        role : Optional[Union[Type[OutlinerRole], OutlinerRole]]
        parent : Optional[QtGui.QWidget]
        """
        #assert isinstance(stage, Usd.Stage), 'A Stage instance is required'
        QtWidgets.QWidget.__init__(self, parent=parent)

        # load icons from local resources only if it's independent outliner
        if type(self) == UsdOutliner:
            import resources.resources
        self.resize(900, 600)

        self._stage = stage
        self._dataModel = HierarchyModel(stage=stage, parent=self)
        self.role = OutlinerRole if role is None else role

        view = self._CreateView(self.role)
        view.setColumnWidth(0, 360)
        self.view = view
        self.layout = QtWidgets.QVBoxLayout(self)
        self.layout.setContentsMargins(4, 4, 4, 4)
        self.layout.setSpacing(2)

        # apply filter settings
        self._filterModel = FilterModel(parent=self)

        self._create_menu_bar()

        if stage is None:
            self.UpdateTitle(i18n("outliner", "No stage"))
        else:
            self.UpdateTitle()

        # Instances of child dialogs (for reference-counting purposes)

        self._sharedLayerTextEditors = {}
        self.editTargetDialog = None
        self.variantEditorDialog = None
        self._need_update_target_dialog = False

        self.view.setProxyModel(self._filterModel)
        if _DCC:
            self._filter_le = SearchWidget()
        else:
            self._filter_le = QtWidgets.QLineEdit()
        self._filter_le.setPlaceholderText(i18n("outliner", "Search..."))
        self._filter_le.textChanged.connect(self.apply_filter)
        self.layout.addWidget(self._filter_le)

        self.layout.addWidget(view)
        self.setAttribute(QtCore.Qt.WA_DeleteOnClose, True)
        self.destroyed.connect(lambda : UsdOutliner.on_destroyed(self))

    def set_columns_controller(self, columns):
        self._dataModel.set_columns(columns)
        self.view.setRoot()
        self.view.set_column_settings()

    def add_column(self, column_type, header_name, *args, **kwargs):
        column = self._dataModel.add_column(column_type, header_name, *args, **kwargs)
        self.view.connect_delegate_to_column(column)
        self.view.setRoot()

    def remove_column(self, index):
        self._dataModel.remove_column(index)

    def remove_column_by_name(self, header_name):
        self._dataModel.remove_column_by_name(header_name)
        self.view.setRoot()

    def setStyle(self):
        """Set styles for outliner"""
        resourceDir = os.path.dirname(os.path.realpath(__file__)) + "/"
        sheet = open(os.path.join(resourceDir, 'usdviewstyle.qss'), 'r')

        sheetString = sheet.read()

        self.setStyleSheet(sheetString)

    def SetStage(self, stage):
        """Update stage, create a new model, activate menubar, recreate columns"""
        if self._stage == stage:
            return
        self._stage = stage
        self._dataModel.ResetStage(stage)
        self._need_update_target_dialog = True
        if stage:
            self.UpdateTitle()
            self.view.setRoot()
        else:
            self.UpdateTitle(i18n("outliner", "No stage"))

    def _CreateView(self, role):
        # type: (Usd.Stage, Union[Type[OutlinerRole], OutlinerRole]) -> QtWidgets.QAbstractItemView
        """Create the hierarchy view for the outliner.

        This is provided as a convenience for subclass implementations.

        Parameters
        ----------
        stage : Usd.Stage
        role : Union[Type[OutlinerRole], OutlinerRole]

        Returns
        -------
        QtWidgets.QAbstractItemView
        """
        return OutlinerTreeView(
            self._dataModel,
            contextMenuActions=role.GetContextMenuActions(self),
            contextProvider=self,
            parent=self)

    def UpdateTitle(self, identifier=None):
        # type: (Optional[str]) -> None
        """
        Parameters
        ----------
        identifier : Optional[str]
            If not provided, it is acquired from the curent edit target.
        """
        if not identifier and self.GetEditTargetLayer():
            identifier = self.GetEditTargetLayer().identifier
        self.setWindowTitle(i18n("outliner", "Usd Outliner - %s") % identifier)

    @classmethod
    def FromUsdFile(cls, usdFile, role=None, parent=None):
        # type: (str, Optional[Union[Type[OutlinerRole], OutlinerRole]], Optional[QtGui.QWidget]) -> UsdOutliner
        """
        Parameters
        ----------
        usdFile : str
        role : Optional[Union[Type[OutlinerRole], OutlinerRole]]
        parent : Optional[QtGui.QWidget]

        Returns
        -------
        UsdOutliner
        """
        with Usd.StageCacheContext(Usd.BlockStageCaches):
            stage = Usd.Stage.Open(usdFile, Usd.Stage.LoadNone)
            assert stage, 'Failed to open stage'
            stage.SetEditTarget(stage.GetSessionLayer())
        return cls(stage, role=role, parent=parent)

    def ShowEditTargetDialog(self):
        if self._need_update_target_dialog:
            self.editTargetDialog = None
        UsdQtOutliner.ShowEditTargetDialog(self)

    @staticmethod
    def on_destroyed(self):
        # invalidate model, so it will not receive callbacks
        self.view.model().ResetStage(None)

    def apply_filter(self, string):
        self.view._blockSelectionCallback = True
        self._filterModel.SetPathStartsWithFilter(self._filter_le.text())
        self.view.setRootIndex(self._filterModel.index(0, 0))
        self.view._blockSelectionCallback = False

    def _create_menu_bar(self):
        self._menubar = QtWidgets.QMenuBar()
        self.layout.setMenuBar(self._menubar)

        self._menu_show = QtWidgets.QMenu(i18n("outliner", "&Show"), self._menubar)
        self._menubar.addMenu(self._menu_show)

        act_show_pcp_errors = QtWidgets.QWidgetAction(self._menu_show)
        checkbox_show_pcp_errors = QtWidgets.QCheckBox(i18n("outliner", "Show Errors of Composing"), self)
        act_show_pcp_errors.setDefaultWidget(checkbox_show_pcp_errors)
        checkbox_show_pcp_errors.toggled.connect(self._dataModel.change_show_pcp_errors_flag)
        checkbox_show_pcp_errors.setChecked(self._dataModel._show_pcp_errors)
        self._menu_show.addAction(act_show_pcp_errors)
        self._menu_show.addSeparator()

        act_show_inactive = QtWidgets.QWidgetAction(self._menu_show)
        checkbox_inactive = QtWidgets.QCheckBox(i18n("outliner", "Inactive Prims"), self)
        act_show_inactive.setDefaultWidget(checkbox_inactive)
        checkbox_inactive.toggled.connect(self._filterModel.TogglePrimInactive)
        checkbox_inactive.setChecked(self._filterModel._show_inactive)

        act_show_undefined = QtWidgets.QWidgetAction(self._menu_show)
        checkbox_undefined = QtWidgets.QCheckBox(i18n("outliner", "Undefined Prims (Overs)"), self)
        act_show_undefined.setDefaultWidget(checkbox_undefined)
        checkbox_undefined.toggled.connect(self._filterModel.TogglePrimUndefined)
        checkbox_undefined.setChecked(self._filterModel._show_undefined)

        act_show_abstact = QtWidgets.QWidgetAction(self._menu_show)
        checkbox_abstract = QtWidgets.QCheckBox(i18n("outliner", "Abstract Prims (Classes)"), self)
        act_show_abstact.setDefaultWidget(checkbox_abstract)
        checkbox_abstract.toggled.connect(self._filterModel.TogglePrimAbstract)
        checkbox_abstract.setChecked(self._filterModel._show_abstract)

        self._menu_show.addAction(act_show_inactive)
        self._menu_show.addAction(act_show_undefined)
        self._menu_show.addAction(act_show_abstact)

    def add_filter(self, filter_key, state):
        self._filterModel.beginResetModel()
        self._filterModel.set_filter('inactive', state)
        self._filterModel.endResetModel()
        self.view.setRoot()

def columns_for_outliner():
    """
    For example. Create list of columns. The order in a list is significant to visualisation in the same order
    :return:
    """
    columns = ColumnController()
    columns.add_column(VisibilityColumn, i18n("outliner", "Vis"))
    columns.add_column(NameColumn, i18n("outliner", "Name"))
    columns.add_column(MetaDataColumn, i18n("outliner", "Type"), "typeName")
    columns.add_column(AttributeColumn, i18n("outliner", "orientation"), "orientation")
    columns.add_column(RelationshipColumn, i18n("outliner", "material:binding"), "material:binding")
    columns.add_column(MetaDataColumn, i18n("outliner", "Kind"), "kind")
    columns.add_column(VariantColumn, i18n("outliner", "Modeling Variant"), "modelingVariant")
    columns.set_tree_index(columns.index(columns.name.header_name))
    return columns

if __name__ == '__main__':
    # simple test
    import sys

    app = QtWidgets.QApplication(sys.argv)

    #usdFileArg = sys.argv[1]
    usdFileArg = 'd:/Work/USD/Kitchen_set/Kitchen_set.usd'
    #usdFileArg = r'd:\hg\convert_scn_to_usd\Episode_03_Scene_02\scene.usd'

    #window = UsdOutliner(None, None)
    window = UsdOutliner.FromUsdFile(usdFileArg)

    window.show()


    sys.exit(app.exec_())
