import os
import json
from pxr import Ar
from Qt import QtCore, QtWidgets, QtGui
import wref_resolver
import pymel.core as pm
from .utils import get_basepath, get_from_usd_refs

from wizart.usd_anim_utils.utils import check_alusdmaya_plugin

def get_ref_data(scene_path):
    refs = []

    get_from_usd_refs(scene_path, refs)

    file_info = pm.system.fileInfo.get("wref_resolver_config")
    if not file_info:
        return refs
    file_info = file_info.replace(r'\"','"') #escape double quoutes returned by maya
    ctx = json.loads(file_info)
    if not ctx or not ctx.get('pin'):
        return refs

    for ref in refs:
        if ctx.get('pin').get(get_basepath(ref.path)):
            ref.pin = ctx.get('pin').get(get_basepath(ref.path))
    return refs


class VersionsEditor(QtWidgets.QComboBox):
    def __init__(self, max_version, pin=None, parent=None):
        QtWidgets.QComboBox.__init__(self, parent)

        self.addItems([None]+[str(item) for item in range(1, max_version+1)])
        model = self.model()
        firstIndex = model.index(0, self.modelColumn(), self.rootModelIndex())
        firstItem = model.itemFromIndex(firstIndex)
        firstItem.setSelectable(True)

        current_index = self.findText(str(pin))
        self.setCurrentIndex(current_index)


class VersionComboBoxDelegate(QtWidgets.QItemDelegate):
    item_changed = QtCore.Signal(QtWidgets.QTreeWidgetItem)

    def __init__(self, tree):
        QtWidgets.QItemDelegate.__init__(self)
        self._tree = tree

    def createEditor(self, parent, options, index):
        """When editing starting, create editor"""
        if index.column() != 1:
            return
        item = self._tree.itemFromIndex(index)
        wref_path = item.wref_path
        comboBox = VersionsEditor(wref_path.max_version, wref_path.pin, parent=parent)
        self._tree.setItemWidget(item, 1, comboBox)
        return comboBox

    def setModelData(self, editor, model, index):
        """When editing finished, apply changes to model"""
        value = int(editor.currentText()) if editor.currentText() else None
        model.setData(index, value, QtCore.Qt.EditRole)
        item = self._tree.itemFromIndex(index)
        item.wref_path.pin = value
        self.item_changed.emit(item)
        self.destroyEditor(editor, index)

    def sizeHint(self, options, index):
        size = QtWidgets.QItemDelegate.sizeHint(self, options, index)
        size.setHeight(20)
        return size


class VersionLineEditDelegate(QtWidgets.QItemDelegate):
    item_changed = QtCore.Signal(QtCore.QModelIndex)

    def __init__(self):
        QtWidgets.QItemDelegate.__init__(self)

    def createEditor(self, parent, options, index):
        """When editing starting, create editor"""
        editable_column = index.model().header_labels.index('Version')
        if index.column() != editable_column:
            return

        ref = index.model().ref_by_index(index)
        edit = QtWidgets.QLineEdit(parent=parent)
        edit.setText(str(ref.pin))
        edit.setInputMask('00000')
        return edit

    def setModelData(self, editor, model, index):
        """When editing finished, apply changes to model"""
        value = int(editor.text()) if editor.text() else None
        # if value > item.wref_path.max_version:
        #     value = item.wref_path.max_version
        #     editor.setText(str(value))
        model.setData(index, value, QtCore.Qt.EditRole)
        self.item_changed.emit(index)

    def sizeHint(self, options, index):
        size = QtWidgets.QItemDelegate.sizeHint(self, options, index)
        size.setHeight(20)
        return size


class RefVersionsModel(QtCore.QAbstractItemModel):
    header_labels = ('Pin', 'Path', 'Version', 'Latest', 'Resolved')

    def __init__(self, stage=None, parent=None):
        QtCore.QAbstractItemModel.__init__(self, parent=parent)
        self._pin_icon = QtGui.QPixmap(":pinned.png")
        self._no_pin_icon = QtGui.QPixmap(":unpinned.png")

        self._usd_icon = QtGui.QPixmap(os.path.join(os.path.dirname(__file__), "icons/usd.png")).scaled(18, 18)
        self._maya_icon = QtGui.QPixmap(os.path.join(os.path.dirname(__file__), "icons/mayaico_2018.png")).scaled(18, 18)
        self._latest_icon = QtGui.QPixmap(os.path.join(os.path.dirname(__file__), "icons/checked.png")).scaled(16, 16)

        self.invalidate()
        if stage:
            self._refs = []
            self._stage = stage
            self._root_layer = self._stage.GetRootLayer()
            self.update_items()

    def invalidate(self):
        self.beginResetModel()
        self._refs = []
        self._stage = None
        self._root_layer = None
        self.endResetModel()

    def columnCount(self, parentIndex):
        return len(self.header_labels)

    def rowCount(self, parentIndex):
        if not parentIndex.isValid():
            return len(self._refs)
        return 0

    def headerData(self, section, orientation, role=QtCore.Qt.DisplayRole):
        if orientation == QtCore.Qt.Horizontal and role == QtCore.Qt.DisplayRole:
            return self.header_labels[section]

    def data(self, modelIndex, role=QtCore.Qt.DisplayRole):
        if not modelIndex.isValid():
            return
        ref = self.ref_by_index(modelIndex)
        if not ref:
            return

        if modelIndex.column() == self.header_labels.index('Pin'):
            if role == QtCore.Qt.DisplayRole:
                return None
            elif role == QtCore.Qt.DecorationRole:
                if ref.pin:
                    return self._pin_icon
                else:
                    return self._no_pin_icon
        elif modelIndex.column() == self.header_labels.index('Path'):
            if role == QtCore.Qt.DisplayRole:
                return ref.path
            if role == QtCore.Qt.DecorationRole:
                if ref.is_usd_file():
                    return self._usd_icon
                if ref.is_maya_file():
                    return self._maya_icon
        elif modelIndex.column() == self.header_labels.index('Version'):
            if role == QtCore.Qt.DisplayRole:
                return ref.current_version
        elif modelIndex.column() == self.header_labels.index('Latest'):
            if role == QtCore.Qt.DisplayRole:
                return ''
            elif role == QtCore.Qt.DecorationRole:
                if ref.is_latest():
                    return self._latest_icon
        elif modelIndex.column() == self.header_labels.index('Resolved'):
            if role == QtCore.Qt.DisplayRole:
                return ref.resolved_path

    def setData(self, modelIndex, value, role=QtCore.Qt.EditRole):
        if role == QtCore.Qt.EditRole:
            if modelIndex.column() == self.header_labels.index('Version'):
                self._refs[modelIndex.row()].pin = value
                return True
        return False

    def reset_model(self, stage):
        self._stage = stage
        self._root_layer = self._stage.GetRootLayer()
        self.update_items()

    def update_items(self):
        scene_path = None
        if self._root_layer:
            scene_path = self._stage.GetRootLayer().identifier

        if not scene_path:
            return

        self.beginResetModel()
        self._refs = get_ref_data(wref_resolver.resolve(scene_path))
        self.endResetModel()

    def index(self, row, column, parent=QtCore.QModelIndex()):
        if not parent.isValid():
            # We assume the root has already been registered.
            return self.createIndex(row, column, self._refs[column])
        return QtCore.QModelIndex()

    def parent(self, modelIndex):
        return QtCore.QModelIndex()

    def ref_by_index(self, modelIndex):
        if modelIndex.row() < len(self._refs):
            return self._refs[modelIndex.row()]

    def flags(self, modelIndex):
        if modelIndex.column() == self.header_labels.index('Version'):
            return QtCore.QAbstractItemModel.flags(self, modelIndex) | QtCore.Qt.ItemIsEditable
        return QtCore.QAbstractItemModel.flags(self, modelIndex)

    def get_pin_config(self):
        config_pin = {}
        for ref in self._refs:
            if ref.pin:
                config_pin[os.path.expandvars(ref.path.split('&')[0])] = ref.pin
        return config_pin

    def get_paths_to_reload(self):
        reload_usd_paths = []
        reload_maya_paths = []
        for ref in self._refs:
            if ref.need_reload:
                if ref.is_usd_file():
                    reload_usd_paths.append(ref.path)
                elif ref.is_maya_file():
                    reload_maya_paths.append(ref.path)
                ref.need_reload = False
        return reload_usd_paths, reload_maya_paths

    def actualize_refs(self):
        self.beginResetModel()
        for ref in self._refs:
            if ref.need_reload:
                ref.update_current_version()
        self.endResetModel()


class FilterModel(QtCore.QSortFilterProxyModel):
    needSelectIndexes = QtCore.Signal(list)

    def __init__(self, parent=None):
        QtCore.QSortFilterProxyModel.__init__(self, parent)
        self._show_usd = True
        self._show_maya = True
        self._show_others = True
        self.header_labels = RefVersionsModel.header_labels

    def filterAcceptsRow(self, row_num, parent):
        """QSortFilterProxyModel method"""
        if parent is None:
            return False

        model = self.sourceModel()
        source_index = model.index(row_num, 0, parent)
        ref = model.ref_by_index(source_index)

        if ref.is_usd_file() and not self._show_usd:
            return False
        if ref.is_maya_file() and not self._show_maya:
            return False
        if not ref.is_usd_file() and not ref.is_maya_file() and not self._show_others:
            return False

        return True

    def set_filter(self, filter_name, state):
        if filter_name == 'usd':
            self._show_usd = state
        elif filter_name == 'maya':
            self._show_maya = state
        elif filter_name == 'others':
            self._show_others = state

    def reset_model(self, stage):
        self.sourceModel().reset_model(stage)

    def ref_by_index(self, index):
        source_index = self.mapToSource(index)
        if source_index:
            return self.sourceModel().ref_by_index(source_index)

    def get_pin_config(self):
        return self.sourceModel().get_pin_config()

    def get_paths_to_reload(self):
        return self.sourceModel().get_paths_to_reload()

    def actualize_refs(self):
        return self.sourceModel().actualize_refs()


class Header(QtWidgets.QHeaderView):
    def contextMenuEvent(self, event):
        menu = QtWidgets.QMenu(self)
        act = menu.addAction("Show/Hide Resolved")
        act.triggered.connect(self._show_resolved_column)
        menu.exec_(self.mapToGlobal(event.pos()))

    def _show_resolved_column(self):
        is_hidden = self.isSectionHidden(RefVersionsModel.header_labels.index('Resolved'))
        self.setSectionHidden(RefVersionsModel.header_labels.index('Resolved'), not is_hidden)


class VersionsTreeView(QtWidgets.QTreeView):
    def __init__(self, parent=None):
        QtWidgets.QTreeView.__init__(self, parent)
        self.setHeader(Header(QtCore.Qt.Horizontal, self))

    def contextMenuEvent(self, event):
        menu = QtWidgets.QMenu(self)
        act = menu.addAction("Pin")
        act.triggered.connect(self.pin_action)
        act = menu.addAction("Unpin")
        act.triggered.connect(self.unpin_action)
        act = menu.addAction("Update to latest")
        act.triggered.connect(self.update_to_latest_action)
        menu.exec_(self.mapToGlobal(event.pos()))

    def pin_action(self):
        self._is_modifing = True
        selected = self.selectedIndexes()

        if not selected:
            current = self.currentIndex()
            selected = [current]

        for index in selected:
            if index.column() == self.model().header_labels.index('Version'):
                ref = self.model().ref_by_index(index)
                ref.need_reload = True
                if not ref.pin:
                    self.model().setData(index, ref.current_version, QtCore.Qt.EditRole)
        self._is_modifing = False
        self.parent().update_context()

    def update_to_latest_action(self):
        selected = self.selectedIndexes()
        self._is_modifing = True
        for index in selected:
            if index.column() == self.model().header_labels.index('Version'):
                ref = self.model().ref_by_index(index)
                ref.need_reload = True
                self.model().setData(index, ref.get_latest_version(), QtCore.Qt.EditRole)
        self._is_modifing = False
        self.parent().update_context()

    def unpin_action(self):
        self._is_modifing = True
        selected = self.selectedIndexes()

        if not selected:
            current = self.currentIndex()
            selected = [current]

        for index in selected:
            if index.column() == self.model().header_labels.index('Version'):
                ref = self.model().ref_by_index(index)
                ref.need_reload = True
                self.model().setData(index, None, QtCore.Qt.EditRole)
        self._is_modifing = False
        self.parent().update_context()


class WrefContextManager(QtWidgets.QWidget):
    def __init__(self, parent=None):
        QtWidgets.QWidget.__init__(self, parent)
        self.setWindowFlags(QtCore.Qt.Window)
        self.setWindowTitle('Wref Context Manager')
        self.resize(600, 600)
        self._stage = None
        self._root_layer = None

        self._main_layout = QtWidgets.QVBoxLayout(self)
        self._main_layout.setContentsMargins(0, 0, 0, 0)
        self._main_layout.setSpacing(0)

        self._menubar = QtWidgets.QMenuBar()
        self._main_layout.setMenuBar(self._menubar)

        self._menu_show = QtWidgets.QMenu("&Show", self._menubar)
        self._menubar.addMenu(self._menu_show)
        act_refresh = QtWidgets.QAction('Refresh', self._menubar)
        self._menubar.addAction(act_refresh)
        act_refresh.setIcon(QtGui.QIcon(":refresh.png"))
        act_refresh.triggered.connect(self.update_stage)

        self._tree = VersionsTreeView(self)
        self._tree.setSelectionMode(QtWidgets.QAbstractItemView.ExtendedSelection)
        self._tree.setEditTriggers(QtWidgets.QAbstractItemView.DoubleClicked)
        self._main_layout.addWidget(self._tree)

        self._model = RefVersionsModel(self._stage)
        self._proxy_model = FilterModel()
        self._proxy_model.setSourceModel(self._model)
        self._proxy_model.setFilterKeyColumn(1)
        self._tree.setModel(self._proxy_model)

        self._tree.header().setSectionHidden(self._model.header_labels.index('Resolved'), True)

        self._tree.header().setSectionResizeMode(0, QtWidgets.QHeaderView.Fixed)
        self._tree.setColumnWidth(self._model.header_labels.index('Pin'), 30)
        self._tree.setColumnWidth(self._model.header_labels.index('Path'), 400)
        self._tree.setColumnWidth(self._model.header_labels.index('Version'), 70)
        self._tree.setColumnWidth(self._model.header_labels.index('Latest'), 70)
        self._tree.setColumnWidth(self._model.header_labels.index('Resolved'), 400)
        self._tree_delegate = VersionLineEditDelegate()
        self._tree.setItemDelegateForColumn(self._model.header_labels.index('Version'), self._tree_delegate)
        self._tree.setTreePosition(self._model.header_labels.index('Path'))
        self._tree._is_modifing = False
        self._tree_delegate.item_changed.connect(self.version_changed)

        act_show_usd = QtWidgets.QWidgetAction(self._menu_show)
        checkbox_usd = QtWidgets.QCheckBox('Usd files', self)
        act_show_usd.setDefaultWidget(checkbox_usd)
        checkbox_usd.setChecked(True)
        checkbox_usd.toggled.connect(lambda state: self.add_filter('usd', state))

        act_show_maya = QtWidgets.QWidgetAction(self._menu_show)
        checkbox_maya = QtWidgets.QCheckBox('Maya files', self)
        act_show_maya.setDefaultWidget(checkbox_maya)
        checkbox_maya.setChecked(self._proxy_model._show_maya)
        checkbox_maya.toggled.connect(lambda state: self.add_filter('maya', state))

        act_show_others = QtWidgets.QWidgetAction(self._menu_show)
        checkbox_others = QtWidgets.QCheckBox('Others', self)
        act_show_others.setDefaultWidget(checkbox_others)
        checkbox_others.setChecked(self._proxy_model._show_others)
        checkbox_others.toggled.connect(lambda state: self.add_filter('others', state))

        self._menu_show.addAction(act_show_usd)
        self._menu_show.addAction(act_show_maya)
        self._menu_show.addAction(act_show_others)

        self.update_stage()

    def add_filter(self, filter_key, state):
        self._proxy_model.beginResetModel()
        self._proxy_model.set_filter(filter_key, state)
        self._proxy_model.endResetModel()
    
    @check_alusdmaya_plugin
    def update_stage(self):
        import AL.usdmaya
        #self.invalidate()
        proxies = pm.ls(type='AL_usdmaya_ProxyShape')

        if proxies:
            proxy = proxies[0]
            proxy_shape = AL.usdmaya.ProxyShape.getByName(proxy.name())
            self._stage = proxy_shape.getUsdStage()
            if self._stage:
                self._model.reset_model(self._stage)
                self._root_layer = self._stage.GetRootLayer()
                return

        # if no proxy or no stage
        self._model.invalidate()

    def version_changed(self, index):
        ref = self._proxy_model.ref_by_index(index)
        ref.update_current_version()
        ref.need_reload = True
        if not self._tree._is_modifing:
            self.update_context()

    def update_context(self, need_reload=True):
        # need import resolver python binding first, before querying it from stage
        import usd_wref_resolver
        config_pin = self._model.get_pin_config()
        config_dump = json.dumps(dict(pin=config_pin))
        global_context = wref_resolver.read_context_from_workspace()
        global_context.merge_with(config_dump, wref_resolver.MergeMask.VERSION, wref_resolver.MergePriorityMask.VERSION_LOCAL)
        pm.system.fileInfo["wref_resolver_config"] = config_dump
        ctx = self._stage.GetPathResolverContext()
        ctx.config = global_context.config
        wref_resolver.set_global_context(global_context)

        self._model.actualize_refs()

        if not need_reload:
            return
        reload_usd_paths, reload_maya_paths = self._model.get_paths_to_reload()
        layers_to_reload = []
        for layer in self._root_layer.GetLoadedLayers():
            if layer.identifier.replace('/?', '?') in reload_usd_paths:
                layers_to_reload.append(layer)
        with Ar.ResolverContextBinder(ctx):
            for layer in layers_to_reload:
                layer.UpdateAssetInfo()
            self._root_layer.ReloadLayers(layers_to_reload)

        for loaded_ref in pm.system.listReferences():
            if loaded_ref.unresolvedPath().replace('/?', '?') in reload_maya_paths:
                if loaded_ref.isLoaded():
                    loaded_ref.load(newFile=loaded_ref.unresolvedPath())
