import os
import wref_resolver
from pxr import Usd, UsdUtils


class WrefPath(object):
    path = ''
    resolved_path = ''
    pin = None
    current_version = None
    latest_version = None
    need_reload = False

    def __init__(self, path):
        asset_info = wref_resolver.AssetInfo()
        self.path = path
        self.resolved_path = wref_resolver.resolve(path, asset_info)
        self.current_version = asset_info.version

    def __hash__(self):
        return self.path

    def __eq__(self, other):
        return self.path == other.path

    def __str__(self):
        return self.path + ' ' + str(self.current_version) + ' ' + str(self.pin)

    def __repr__(self):
        return self.__str__()

    def is_latest(self):
        if not self.latest_version:
            self.latest_version = self.get_latest_version()
        return self.current_version == self.latest_version

    def get_latest_version(self):
        asset_info = wref_resolver.AssetInfo()
        wref_resolver.resolve_with_config(get_basepath(self.path), '', asset_info)
        return asset_info.version

    def update_current_version(self):
        asset_info = wref_resolver.AssetInfo()
        self.resolved_path = wref_resolver.resolve(os.path.expandvars(self.path), asset_info)
        self.current_version = asset_info.version

    def get_extention(self):
        return os.path.splitext(get_basepath(self.path))[-1]

    def is_usd_file(self):
        return True if self.get_extention() in ['.usd', '.usda', '.usdc'] else False

    def is_maya_file(self):
        return True if self.get_extention() in ['.mb', '.ma'] else False


def get_basepath(wref_path):
    return wref_path.split('&')[0]


def get_from_usd_refs(scene_path, refs):
    (allLayers, allReferences, allPayloads) = UsdUtils.ExtractExternalReferences(scene_path)
    asset_info = wref_resolver.AssetInfo()

    for path in allLayers + allReferences + allPayloads:
        if str(path).lower().startswith('wref') and asset_info.version:
            ref = WrefPath(path.replace('/?', '?'))
            # for case of duplication or recursion (no need to check path that already was added and checked)
            if ref in refs:
                continue
            refs.append(ref)
            path = ref.resolved_path

        get_from_usd_refs(path, refs)

    return refs


def get_from_usd_paths(scene_path):
    refs = []
    (allLayers, allReferences, allPayloads) = UsdUtils.ExtractExternalReferences(scene_path)

    for path in allLayers + allReferences + allPayloads:
        resolved = wref_resolver.resolve(path)
        if str(path).lower().startswith('wref'):
            refs.append(path.replace('/?', '?'))

        if resolved:
            children_refs = get_from_usd_paths(resolved)
            refs += children_refs

    return refs
