from __future__ import print_function
from __future__ import absolute_import
from __future__ import division
from builtins import range
from builtins import str
from past.builtins import basestring
from builtins import object
import os
import sys
import re
import pymel.util as _util
import pymel.internal.factories as _factories
import pymel.internal.startup as _startup
import pymel.internal as _internal
import pymel.versions as _versions
import maya.mel as _mm

from pymel.core import animation
from pymel.core import effects
from pymel.core import general
from pymel.core import modeling
from pymel.core import other
from pymel.core import rendering
from pymel.core import system
from pymel.core import windows
from future.utils import with_metaclass

if False:
    from typing import *
    import maya.cmds as cmds
else:
    import pymel.internal.pmcmds as cmds  # type: ignore[no-redef]

_f = _factories

_logger = _internal.getLogger(__name__)

# Dictionary mapping from maya node type names (ie, surfaceShape) to pymel
# class names, in this module - ie, SurfaceShape
mayaTypeNameToPymelTypeName = {}
pymelTypeNameToMayaTypeName = {}


def _addTypeNames():
    for name, obj in globals().items():
        if isinstance(obj, type) and issubclass(obj, PyUI):
            mayaTypeNameToPymelTypeName[obj.__melui__] = name
            pymelTypeNameToMayaTypeName[name] = obj.__melui__


def _resolveUIFunc(name):
    if isinstance(name, basestring):
        from pymel.core import windows
        try:
            return getattr(windows, name)
        except AttributeError:
            try:
                cls = globals()[name]
                return cls.__melcmd__()
            except (KeyError, AttributeError):
                pass
    else:
        import inspect
        if inspect.isfunction(name):
            return name
        elif inspect.isclass(name) and issubclass(name, PyUI):
            name.__melcmd__()

    raise ValueError("%r is not a known ui type" % name)


def toPyQtObject(mayaName):
    """
    Given the name of a Maya UI element of any type, return the corresponding QWidget or QAction.
    If the object does not exist, returns None

    When using this function you don't need to specify whether UI type is a control, layout,
    window, or menuItem, the first match -- in that order -- will be returned. If you have the full path to a UI object
    this should always be correct, however, if you only have the short name of the UI object,
    consider using one of the more specific variants: `toQtControl`, `toQtLayout`, `toQtWindow`, or `toQtMenuItem`.

    .. note:: Requires PyQt
    """
    import maya.OpenMayaUI as mui
    import sip
    import PyQt4.QtCore as qtcore
    import PyQt4.QtGui as qtgui
    ptr = mui.MQtUtil.findControl(mayaName)
    if ptr is None:
        ptr = mui.MQtUtil.findLayout(mayaName)
        if ptr is None:
            ptr = mui.MQtUtil.findMenuItem(mayaName)
    if ptr is not None:
        return sip.wrapinstance(int(ptr), qtcore.QObject)


def toPyQtControl(mayaName):
    """
    Given the name of a May UI control, return the corresponding QWidget.
    If the object does not exist, returns None

    .. note:: Requires PyQt
    """
    import maya.OpenMayaUI as mui
    import sip
    import PyQt4.QtCore as qtcore
    import PyQt4.QtGui as qtgui
    ptr = mui.MQtUtil.findControl(mayaName)
    if ptr is not None:
        return sip.wrapinstance(int(ptr), qtgui.QWidget)


def toPyQtLayout(mayaName):
    """
    Given the name of a May UI control, return the corresponding QWidget.
    If the object does not exist, returns None

    .. note:: Requires PyQt
    """
    import maya.OpenMayaUI as mui
    import sip
    import PyQt4.QtCore as qtcore
    import PyQt4.QtGui as qtgui
    ptr = mui.MQtUtil.findLayout(mayaName)
    if ptr is not None:
        return sip.wrapinstance(int(ptr), qtgui.QWidget)


def toPyQtWindow(mayaName):
    """
    Given the name of a May UI control, return the corresponding QWidget.
    If the object does not exist, returns None

    .. note:: Requires PyQt
    """
    import maya.OpenMayaUI as mui
    import sip
    import PyQt4.QtCore as qtcore
    import PyQt4.QtGui as qtgui
    ptr = mui.MQtUtil.findWindow(mayaName)
    if ptr is not None:
        return sip.wrapinstance(int(ptr), qtgui.QWidget)


def toPyQtMenuItem(mayaName):
    """
    Given the name of a May UI menuItem, return the corresponding QAction.
    If the object does not exist, returns None

    This only works for menu items. for Menus, use toQtControl or toQtObject

    .. note:: Requires PyQt
    """
    import maya.OpenMayaUI as mui
    import sip
    import PyQt4.QtCore as qtcore
    import PyQt4.QtGui as qtgui
    ptr = mui.MQtUtil.findMenuItem(mayaName)
    if ptr is not None:
        return sip.wrapinstance(int(ptr), qtgui.QAction)

# PYSIDE VERSIONS


def pysideWrapInstance(ptr, base=None):
    '''Utility to convert a point to a Qt Class and produce the same result
    as sip.wrapinstance using shiboken.wrapInstance.

    Note: This is modeled after nathanhorne.com/?p=486. The base arg isn't
    currently used, and defaults to QObject. The way that base arg was used
    seems like it would give a different result than the sip version. It would
    skip the checking for attribute and just use base as base, however the sip
    version will still return QMainWindow even if QObject is passed in.
    '''
    if ptr is None:
        return

    try:
        import PySide2.QtCore as qtcore
        import PySide2.QtGui as qtgui
        import PySide2.QtWidgets as qtwidgets
        from shiboken2 import wrapInstance
    except ImportError:
        import shiboken
        import PySide.QtCore as qtcore
        import PySide.QtGui as qtgui
        import PySide.QtGui as qtwidgets
        from shiboken import wrapInstance

    qObj = wrapInstance(int(ptr), qtcore.QObject)

    metaObj = qObj.metaObject()
    cls = metaObj.className()
    superCls = metaObj.superClass().className()
    if hasattr(qtgui, cls):
        base = getattr(qtgui, cls)
    elif hasattr(qtgui, superCls):
        base = getattr(qtgui, superCls)
    elif hasattr(qtwidgets, cls):
        base = getattr(qtwidgets, cls)
    elif hasattr(qtwidgets, superCls):
        base = getattr(qtwidgets, superCls)
    else:
        base = qtwidgets.QWidget
    return wrapInstance(int(ptr), base)


def toPySideObject(mayaName):
    """
    Given the name of a Maya UI element of any type, return the corresponding QWidget or QAction.
    If the object does not exist, returns None

    When using this function you don't need to specify whether UI type is a control, layout,
    window, or menuItem, the first match -- in that order -- will be returned. If you have the full path to a UI object
    this should always be correct, however, if you only have the short name of the UI object,
    consider using one of the more specific variants: `toQtControl`, `toQtLayout`, `toQtWindow`, or `toQtMenuItem`.

    .. note:: Requires PySide
    """
    import maya.OpenMayaUI as mui

    try:
        import PySide2.QtCore as qtcore
    except ImportError:
        import PySide.QtCore as qtcore

    ptr = mui.MQtUtil.findControl(mayaName)
    if ptr is None:
        ptr = mui.MQtUtil.findLayout(mayaName)
        if ptr is None:
            ptr = mui.MQtUtil.findMenuItem(mayaName)
    if ptr is not None:
        return pysideWrapInstance(int(ptr), qtcore.QObject)


def toPySideControl(mayaName):
    """
    Given the name of a May UI control, return the corresponding QWidget.
    If the object does not exist, returns None

    .. note:: Requires PySide
    """
    import maya.OpenMayaUI as mui

    try:
        import shiboken2
        import PySide2.QtCore as qtcore
        import PySide2.QtWidgets as qtwidgets
    except ImportError:
        import shiboken
        import PySide.QtCore as qtcore
        import PySide.QtGui as qtwidgets

    ptr = mui.MQtUtil.findControl(mayaName)
    if ptr is not None:
        return pysideWrapInstance(int(ptr), qtwidgets.QWidget)


def toPySideLayout(mayaName):
    """
    Given the name of a May UI control, return the corresponding QWidget.
    If the object does not exist, returns None

    .. note:: Requires PySide
    """
    import maya.OpenMayaUI as mui

    try:
        import shiboken2
        import PySide2.QtCore as qtcore
        import PySide2.QtWidgets as qtwidgets
    except ImportError:
        import shiboken
        import PySide.QtCore as qtcore
        import PySide.QtGui as qtwidgets

    ptr = mui.MQtUtil.findLayout(mayaName)
    if ptr is not None:
        return pysideWrapInstance(int(ptr), qtwidgets.QWidget)


def toPySideWindow(mayaName):
    """
    Given the name of a May UI control, return the corresponding QWidget.
    If the object does not exist, returns None

    .. note:: Requires PySide
    """
    import maya.OpenMayaUI as mui

    try:
        import shiboken2
        import PySide2.QtCore as qtcore
        import PySide2.QtWidgets as qtwidgets
    except ImportError:
        import shiboken
        import PySide.QtCore as qtcore
        import PySide.QtGui as qtwidgets
    ptr = mui.MQtUtil.findWindow(mayaName)
    if ptr is not None:
        return pysideWrapInstance(int(ptr), qtwidgets.QWidget)


def toPySideMenuItem(mayaName):
    """
    Given the name of a Maya UI menuItem, return the corresponding QAction.
    If the object does not exist, returns None

    This only works for menu items. for Menus, use toQtControl or toQtObject

    .. note:: Requires PySide
    """
    import maya.OpenMayaUI as mui
    try:
        import shiboken2
        import PySide2.QtCore as qtcore
        import PySide2.QtWidgets as qtwidgets
    except ImportError:
        import shiboken
        import PySide.QtCore as qtcore
        import PySide.QtGui as qtwidgets

    ptr = mui.MQtUtil.findMenuItem(mayaName)
    if ptr is not None:
        return pysideWrapInstance(int(ptr), qtwidgets.QAction)

# Assign functions to PyQt versions if PyQt is available, otherwise set to PySide versions
try:
    import sip
    import PyQt4
    pyQtAvailable = True
except ImportError:
    pyQtAvailable = False

try:
    import shiboken
    import PySide
    pySideAvailable = True
except ImportError:
    pySideAvailable = False
    try:
        import shiboken2
        import PySide2
        pySideAvailable = True
    except ImportError:
        pySideAvailable = False

if pyQtAvailable and not pySideAvailable:
    qtBinding = 'pyqt'
elif pySideAvailable and not pyQtAvailable:
    qtBinding = 'pyside'
else:
    qtBinding = _startup.pymel_options['preferred_python_qt_binding']

if qtBinding == 'pyqt':
    toQtObject = toPyQtObject
    toQtControl = toPyQtControl
    toQtLayout = toPyQtLayout
    toQtWindow = toPyQtWindow
    toQtMenuItem = toPyQtMenuItem
elif qtBinding == 'pyside':
    toQtObject = toPySideObject
    toQtControl = toPySideControl
    toQtLayout = toPySideLayout
    toQtWindow = toPySideWindow
    toQtMenuItem = toPySideMenuItem
else:
    raise ValueError('preferred_python_qt_binding must be set to either'
                     ' pyside or pyqt')


# really, this should be in core.windows; but, due to that fact that this module
# is "higher" in the import hierarchy than core.windows, and we need this function
# here, we're just defining it here
@_factories.addMelDocs('objectTypeUI')
def objectTypeUI(name, **kwargs):
    try:
        return cmds.objectTypeUI(name, **kwargs)
    except RuntimeError as topError:
        try:
            # some ui types (radioCollections) can only be identified with
            # their shortname
            return cmds.objectTypeUI(name.split('|')[-1], **kwargs)
        except RuntimeError:
            # we cannot query the type of rowGroupLayout children: check
            # common types for these
            uiType = None
            typesToCheck = 'checkBox floatField button floatSlider intSlider ' \
                'floatField textField intField optionMenu radioButton'.split()
            # 2012 SP2 introducted a bug where doing:
            # win = cmds.window(menuBar=True)
            # cmds.objectTypeUI(win)
            # would error...
            typesToCheck.append('window')
            for cmdName in typesToCheck:
                if getattr(cmds, cmdName)(name, ex=1, q=1):
                    uiType = cmdName
                    break
            if uiType:
                return uiType
            raise topError


class PyUI(str):
    __melui__ = None  # type: str

    def __new__(cls, name=None, create=False, **kwargs):
        """
        Provides the ability to create the PyUI Element when creating a class::

            import pymel.core as pm
            n = pm.Window("myWindow",create=True)
            n.__repr__()
            # Result: Window('myWindow')
        """
        if cls is PyUI:
            try:
                uiType = objectTypeUI(name)
            except RuntimeError:
                uiType = 'PyUI'
            uiType = _uiTypesToCommands.get(uiType, uiType)

            try:
                newcls = globals()[_util.capitalize(uiType)]
            except KeyError:
                newcls = PyUI
                # objectTypeUI for panels seems to return weird results -
                # ie, TmodelPane ... check for them this way.
                # Other types should be detected correctly by objectTypeUI,
                # but this just provides a failsafe...
                for testType in 'panel scriptedPanel window control layout menu'.split():
                    if getattr(cmds, testType)(name, ex=1, q=1):
                        newcls = globals().get(_util.capitalize(testType), PyUI)
                        if newcls != PyUI:
                            break
        else:
            newcls = cls

        if not newcls is PyUI:
            if cls._isBeingCreated(name, create, kwargs):
                name = newcls.__melcmd__(name, **kwargs)
                _logger.debug("PyUI: created... %s" % name)
            else:
                # find the long name
                if '|' not in name and not issubclass(newcls,
                                                      (Window,
                                                       Panel,
                                                       ScriptedPanel,
                                                       RadioCollection,
                                                       ToolCollection)):
                    from pymel.core import windows
                    try:
                        if issubclass(newcls, Layout):
                            parent = windows.layout(name, q=1, p=1)
                        elif issubclass(newcls, OptionMenu):
                            parent = windows.optionMenu(name, q=1, p=1)
                        elif issubclass(newcls, Menu):
                            parent = windows.menu(name, q=1, p=1)
                        else:
                            parent = windows.control(name, q=1, p=1)
                        if parent:
                            name = parent + '|' + name

                    except RuntimeError:
                        # editors don't have a long name, so we keep the short name
                        if name not in cmds.lsUI(long=True, editors=True):
                            raise

        # correct for optionMenu
        if newcls == PopupMenu and cmds.optionMenu(name, ex=1):
            newcls = OptionMenu
        return str.__new__(newcls, name)

    @staticmethod
    def _isBeingCreated(name, create, kwargs):
        """
        create a new node when any of these conditions occur:
           name is None
           create is True
           parent flag is set
        """
        return not name or create or ('q' not in kwargs and kwargs.get('parent', kwargs.get('p', None)))

    def __repr__(self):
        return u"ui.%s('%s')" % (self.__class__.__name__, self)

    def parent(self):
        # type: () -> Optional[PyUI]
        """
        Returns
        -------
        Optional[PyUI]
        """
        buf = str(self).split('|')[:-1]
        if not buf:
            return None
        return PyUI('|'.join(buf))

    getParent = parent

    def shortName(self):
        # type: () -> str
        """
        Returns
        -------
        str
        """
        return str(self).split('|')[-1]

    def name(self):
        # type: () -> str
        """
        Returns
        -------
        str
        """
        return str(self)

    def window(self):
        # type: () -> Window
        """
        Returns
        -------
        Window
        """
        return Window(self.name().split('|')[0])

    # delete = _factories.functionFactory('deleteUI', rename='delete')
    # rename = _factories.functionFactory('renameUI', rename='rename')
    delete = _factories.getCmdFunc('deleteUI')
    rename = _factories.getCmdFunc('renameUI')

    type = objectTypeUI

    @classmethod
    def exists(cls, name):
        return cls.__melcmd__(name, exists=True)

    asQtObject = toQtControl


class Panel(PyUI):

    """pymel panel class"""
# ------ Do not edit below this line --------
    __melcmd__ = staticmethod(windows.panel)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'panel'
    __melui__ = 'panel'
    __slots__ = ()

    @_f.addMelDocs('panel', 'copy')
    def copy(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.panel, kwargs, 'copy', val)

    @_f.addMelDocs('panel', 'createString')
    def createString(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.panel, kwargs, 'createString', val)

    @_f.addMelDocs('panel', 'editString')
    def editString(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.panel, kwargs, 'editString', val)

    @_f.addMelDocs('panel', 'control')
    def getControl(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.panel, kwargs, 'control')
        return res

    @_f.addMelDocs('panel', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.panel, kwargs, 'docTag')
        return res

    @_f.addMelDocs('panel', 'isUnique')
    def getIsUnique(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.panel, kwargs, 'isUnique')
        return res

    @_f.addMelDocs('panel', 'label')
    def getLabel(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.panel, kwargs, 'label')
        return res

    @_f.addMelDocs('panel', 'menuBarRepeatLast')
    def getMenuBarRepeatLast(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.panel, kwargs, 'menuBarRepeatLast')
        return res

    @_f.addMelDocs('panel', 'menuBarVisible')
    def getMenuBarVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.panel, kwargs, 'menuBarVisible')
        return res

    @_f.addMelDocs('panel', 'needsInit')
    def getNeedsInit(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.panel, kwargs, 'needsInit')
        return res

    @_f.addMelDocs('panel', 'popupMenuProcedure')
    def getPopupMenuProcedure(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.panel, kwargs, 'popupMenuProcedure')
        return res

    @_f.addMelDocs('panel', 'tearOff')
    def getTearOff(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.panel, kwargs, 'tearOff')
        return res

    @_f.addMelDocs('panel', 'init')
    def init(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.panel, kwargs, 'init', val)

    @_f.addMelDocs('panel', 'replacePanel')
    def replacePanel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.panel, kwargs, 'replacePanel', val)

    @_f.addMelDocs('panel', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.panel, kwargs, 'docTag', val)

    @_f.addMelDocs('panel', 'label')
    def setLabel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.panel, kwargs, 'label', val)

    @_f.addMelDocs('panel', 'menuBarRepeatLast')
    def setMenuBarRepeatLast(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.panel, kwargs, 'menuBarRepeatLast', val)

    @_f.addMelDocs('panel', 'menuBarVisible')
    def setMenuBarVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.panel, kwargs, 'menuBarVisible', val)

    @_f.addMelDocs('panel', 'needsInit')
    def setNeedsInit(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.panel, kwargs, 'needsInit', val)

    @_f.addMelDocs('panel', 'popupMenuProcedure')
    def setPopupMenuProcedure(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.panel, kwargs, 'popupMenuProcedure', val)

    @_f.addMelDocs('panel', 'tearOff')
    def setTearOff(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.panel, kwargs, 'tearOff', val)

    @_f.addMelDocs('panel', 'tearOffRestore')
    def tearOffRestore(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.panel, kwargs, 'tearOffRestore', val)

    @_f.addMelDocs('panel', 'unParent')
    def unParent(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.panel, kwargs, 'unParent', val)
# ------ Do not edit above this line --------


_withParentStack = []
_withParentMenuStack = []


class PyUIContainer(PyUI):

    def __enter__(self):
        global _withParentStack
        _withParentStack.append(self)
        self.makeDefault()
        return self

    def __exit__(self, type, value, traceback):
        global _withParentStack
        _withParentStack.pop()
        if _withParentStack:
            parent = _withParentStack[-1]
        else:
            parent = self.pop()
            while parent and objectTypeUI(parent) == u'rowGroupLayout':
                parent = parent.pop()
        cmds.setParent(parent)

    # TODO: add depth first and breadth first options
    def walkChildren(self):
        # type: () -> Iterator[PyUI]
        """
        recursively yield all children of this layout

        Returns
        -------
        Iterator[PyUI]
        """
        for child in self.children():
            yield child
            if hasattr(child, 'walkChildren'):
                for subChild in child.walkChildren():
                    yield subChild

    def findChild(self, shortName, recurse=False):
        # type: (str, bool) -> Optional[PyUI]
        """
        Parameters
        ----------
        shortName : str
        recurse : bool

        Returns
        -------
        Optional[PyUI]
        """
        if recurse:
            for child in self.walkChildren():
                if child.shortName() == shortName:
                    return child
        else:
            for child in self.children():
                if child.shortName() == shortName:
                    return child

    def addChild(self, uiType, name=None, **kwargs):
        if isinstance(uiType, basestring):
            uiType = globals()[uiType]
        assert hasattr(uiType, '__call__'), 'argument uiType must be the name of a known ui type, a UI subclass, or a callable object'
        args = []
        if name:
            args.append(name)
        if kwargs:
            if 'parent' in kwargs or 'p' in kwargs:
                _logger.warning('parent flag is set by addChild automatically. '
                                'passed value will be ignored')
                kwargs.pop('parent', None)
                kwargs.pop('p', None)
        kwargs['parent'] = self
        res = uiType(*args, **kwargs)
        if not isinstance(res, PyUI):
            res = PyUI(res)
        return res

    def makeDefault(self):
        """
        set this layout as the default parent
        """
        cmds.setParent(self)

    def pop(self):
        """
        set the default parent to the parent of this layout
        """
        p = self.parent()
        cmds.setParent(p)
        return p


class Layout(PyUIContainer):

    def children(self):
        # type: () -> List[PyUI]
        """
        Returns
        -------
        List[PyUI]
        """
        kids = cmds.layout(self, q=1, childArray=1)
        if kids:
            return [PyUI(self.name() + '|' + x) for x in kids]
        return []

    getChildren = children

    def clear(self):
        children = self.getChildArray()
        if children:
            for child in self.getChildArray():
                cmds.deleteUI(child)

    asQtObject = toQtLayout
# ------ Do not edit below this line --------
    __melcmd__ = staticmethod(windows.layout)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'layout'
    __melui__ = 'layout'
    __slots__ = ()

    @_f.addMelDocs('layout', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.layout, kwargs, 'dragCallback', val)

    @_f.addMelDocs('layout', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.layout, kwargs, 'dropCallback', val)

    @_f.addMelDocs('layout', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.layout, kwargs, 'annotation')
        return res

    @_f.addMelDocs('layout', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.layout, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('layout', 'childArray')
    def getChildArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.layout, kwargs, 'childArray')
        return res

    @_f.addMelDocs('layout', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.layout, kwargs, 'docTag')
        return res

    @_f.addMelDocs('layout', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.layout, kwargs, 'enable')
        return res

    @_f.addMelDocs('layout', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.layout, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('layout', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.layout, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('layout', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.layout, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('layout', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.layout, kwargs, 'height')
        return res

    @_f.addMelDocs('layout', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.layout, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('layout', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.layout, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('layout', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.layout, kwargs, 'manage')
        return res

    @_f.addMelDocs('layout', 'numberOfChildren')
    def getNumberOfChildren(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.layout, kwargs, 'numberOfChildren')
        return res

    @_f.addMelDocs('layout', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.layout, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('layout', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.layout, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('layout', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.layout, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('layout', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.layout, kwargs, 'visible')
        return res

    @_f.addMelDocs('layout', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.layout, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('layout', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.layout, kwargs, 'width')
        return res

    @_f.addMelDocs('layout', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.layout, kwargs, 'noBackground', val)

    @_f.addMelDocs('layout', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.layout, kwargs, 'annotation', val)

    @_f.addMelDocs('layout', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.layout, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('layout', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.layout, kwargs, 'docTag', val)

    @_f.addMelDocs('layout', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.layout, kwargs, 'enable', val)

    @_f.addMelDocs('layout', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.layout, kwargs, 'enableBackground', val)

    @_f.addMelDocs('layout', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.layout, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('layout', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.layout, kwargs, 'height', val)

    @_f.addMelDocs('layout', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.layout, kwargs, 'highlightColor', val)

    @_f.addMelDocs('layout', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.layout, kwargs, 'manage', val)

    @_f.addMelDocs('layout', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.layout, kwargs, 'preventOverride', val)

    @_f.addMelDocs('layout', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.layout, kwargs, 'visible', val)

    @_f.addMelDocs('layout', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.layout, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('layout', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.layout, kwargs, 'width', val)

    @_f.addMelDocs('layout', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.layout, kwargs, 'statusBarMessage', val)
# ------ Do not edit above this line --------


# customized ui classes
class Window(PyUIContainer):

    """pymel window class"""

#    if _versions.current() < _versions.v2011:
#        # don't set
#        def __enter__(self):
#            return self

    def __exit__(self, type, value, traceback):
        super(Window, self).__exit__(type, value, traceback)
        self.show()

    def show(self):
        cmds.showWindow(self)

    def delete(self):
        cmds.deleteUI(self, window=True)

    def layout(self):
        name = self.name()
        for layout in sorted(cmds.lsUI(long=True, controlLayouts=True)):
            # since we are sorted, shorter will be first, and the first layout we come across will be the base layout
            if layout.startswith(name):
                return PyUI(layout)

#            # create a child and then delete it to get the layout
#            res = self.addChild(cmds.columnLayout)
#            layout = res.parent()
#            res.delete()
#            return layout

    def children(self):
        res = self.layout()
        return [res] if res else []

    getChildren = children

    def window(self):
        return self

    def parent(self):
        return None
    getParent = parent

    asQtObject = toQtWindow
# ------ Do not edit below this line --------
    __melcmd__ = staticmethod(windows.window)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'window'
    __melui__ = 'window'
    __slots__ = ()

    @_f.addMelDocs('window', 'backgroundColor')
    def backgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.window, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('window', 'closeCommand')
    def closeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.window, kwargs, 'closeCommand', val)

    @_f.addMelDocs('window', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.window, kwargs, 'docTag')
        return res

    @_f.addMelDocs('window', 'dockingLayout')
    def getDockingLayout(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.window, kwargs, 'dockingLayout')
        return res

    @_f.addMelDocs('window', 'frontWindow')
    def getFrontWindow(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.window, kwargs, 'frontWindow')
        return res

    @_f.addMelDocs('window', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.window, kwargs, 'height')
        return res

    @_f.addMelDocs('window', 'iconName')
    def getIconName(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.window, kwargs, 'iconName')
        return res

    @_f.addMelDocs('window', 'iconify')
    def getIconify(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.window, kwargs, 'iconify')
        return res

    @_f.addMelDocs('window', 'leftEdge')
    def getLeftEdge(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.window, kwargs, 'leftEdge')
        return res

    @_f.addMelDocs('window', 'mainMenuBar')
    def getMainMenuBar(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.window, kwargs, 'mainMenuBar')
        return res

    @_f.addMelDocs('window', 'mainWindow')
    def getMainWindow(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.window, kwargs, 'mainWindow')
        return res

    @_f.addMelDocs('window', 'maximizeButton')
    def getMaximizeButton(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.window, kwargs, 'maximizeButton')
        return res

    @_f.addMelDocs('window', 'menuArray')
    def getMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.window, kwargs, 'menuArray')
        return res

    @_f.addMelDocs('window', 'menuBar')
    def getMenuBar(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.window, kwargs, 'menuBar')
        return res

    @_f.addMelDocs('window', 'menuBarCornerWidget')
    def getMenuBarCornerWidget(self, **kwargs):
        # type: (...) -> Tuple[str, str]
        res = _f.asQuery(self, windows.window, kwargs, 'menuBarCornerWidget')
        return res

    @_f.addMelDocs('window', 'menuBarVisible')
    def getMenuBarVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.window, kwargs, 'menuBarVisible')
        return res

    @_f.addMelDocs('window', 'minimizeButton')
    def getMinimizeButton(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.window, kwargs, 'minimizeButton')
        return res

    @_f.addMelDocs('window', 'numberOfMenus')
    def getNumberOfMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.window, kwargs, 'numberOfMenus')
        return res

    @_f.addMelDocs('window', 'resizeToFitChildren')
    def getResizeToFitChildren(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.window, kwargs, 'resizeToFitChildren')
        return res

    @_f.addMelDocs('window', 'sizeable')
    def getSizeable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.window, kwargs, 'sizeable')
        return res

    @_f.addMelDocs('window', 'state')
    def getState(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.window, kwargs, 'state')
        return res

    @_f.addMelDocs('window', 'title')
    def getTitle(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.window, kwargs, 'title')
        return res

    @_f.addMelDocs('window', 'titleBar')
    def getTitleBar(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.window, kwargs, 'titleBar')
        return res

    @_f.addMelDocs('window', 'titleBarMenu')
    def getTitleBarMenu(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.window, kwargs, 'titleBarMenu')
        return res

    @_f.addMelDocs('window', 'toolbox')
    def getToolbox(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.window, kwargs, 'toolbox')
        return res

    @_f.addMelDocs('window', 'topEdge')
    def getTopEdge(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.window, kwargs, 'topEdge')
        return res

    @_f.addMelDocs('window', 'topLeftCorner')
    def getTopLeftCorner(self, **kwargs):
        # type: (...) -> Tuple[int, int]
        res = _f.asQuery(self, windows.window, kwargs, 'topLeftCorner')
        return res

    @_f.addMelDocs('window', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.window, kwargs, 'visible')
        return res

    @_f.addMelDocs('window', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.window, kwargs, 'width')
        return res

    @_f.addMelDocs('window', 'widthHeight')
    def getWidthHeight(self, **kwargs):
        # type: (...) -> Tuple[int, int]
        res = _f.asQuery(self, windows.window, kwargs, 'widthHeight')
        return res

    @_f.addMelDocs('window', 'menuBarResize')
    def menuBarResize(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.window, kwargs, 'menuBarResize', val)

    @_f.addMelDocs('window', 'menuIndex')
    def menuIndex(self, val=True, **kwargs):
        # type: (Tuple[str, int], **Any) -> None
        return _f.asEdit(self, windows.window, kwargs, 'menuIndex', val)

    @_f.addMelDocs('window', 'minimizeCommand')
    def minimizeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.window, kwargs, 'minimizeCommand', val)

    @_f.addMelDocs('window', 'restoreCommand')
    def restoreCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.window, kwargs, 'restoreCommand', val)

    @_f.addMelDocs('window', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.window, kwargs, 'docTag', val)

    @_f.addMelDocs('window', 'dockingLayout')
    def setDockingLayout(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.window, kwargs, 'dockingLayout', val)

    @_f.addMelDocs('window', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.window, kwargs, 'height', val)

    @_f.addMelDocs('window', 'iconName')
    def setIconName(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.window, kwargs, 'iconName', val)

    @_f.addMelDocs('window', 'iconify')
    def setIconify(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.window, kwargs, 'iconify', val)

    @_f.addMelDocs('window', 'leftEdge')
    def setLeftEdge(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.window, kwargs, 'leftEdge', val)

    @_f.addMelDocs('window', 'mainMenuBar')
    def setMainMenuBar(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.window, kwargs, 'mainMenuBar', val)

    @_f.addMelDocs('window', 'mainWindow')
    def setMainWindow(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.window, kwargs, 'mainWindow', val)

    @_f.addMelDocs('window', 'maximizeButton')
    def setMaximizeButton(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.window, kwargs, 'maximizeButton', val)

    @_f.addMelDocs('window', 'menuBarCornerWidget')
    def setMenuBarCornerWidget(self, val=True, **kwargs):
        # type: (Tuple[str, str], **Any) -> None
        return _f.asEdit(self, windows.window, kwargs, 'menuBarCornerWidget', val)

    @_f.addMelDocs('window', 'menuBarVisible')
    def setMenuBarVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.window, kwargs, 'menuBarVisible', val)

    @_f.addMelDocs('window', 'minimizeButton')
    def setMinimizeButton(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.window, kwargs, 'minimizeButton', val)

    @_f.addMelDocs('window', 'resizeToFitChildren')
    def setResizeToFitChildren(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.window, kwargs, 'resizeToFitChildren', val)

    @_f.addMelDocs('window', 'sizeable')
    def setSizeable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.window, kwargs, 'sizeable', val)

    @_f.addMelDocs('window', 'state')
    def setState(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.window, kwargs, 'state', val)

    @_f.addMelDocs('window', 'title')
    def setTitle(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.window, kwargs, 'title', val)

    @_f.addMelDocs('window', 'titleBar')
    def setTitleBar(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.window, kwargs, 'titleBar', val)

    @_f.addMelDocs('window', 'titleBarMenu')
    def setTitleBarMenu(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.window, kwargs, 'titleBarMenu', val)

    @_f.addMelDocs('window', 'toolbox')
    def setToolbox(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.window, kwargs, 'toolbox', val)

    @_f.addMelDocs('window', 'topEdge')
    def setTopEdge(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.window, kwargs, 'topEdge', val)

    @_f.addMelDocs('window', 'topLeftCorner')
    def setTopLeftCorner(self, val=True, **kwargs):
        # type: (Tuple[int, int], **Any) -> None
        return _f.asEdit(self, windows.window, kwargs, 'topLeftCorner', val)

    @_f.addMelDocs('window', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.window, kwargs, 'visible', val)

    @_f.addMelDocs('window', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.window, kwargs, 'width', val)

    @_f.addMelDocs('window', 'widthHeight')
    def setWidthHeight(self, val=True, **kwargs):
        # type: (Tuple[int, int], **Any) -> None
        return _f.asEdit(self, windows.window, kwargs, 'widthHeight', val)
# ------ Do not edit above this line --------


class FormLayout(Layout):

    def __new__(cls, name=None, **kwargs):
        if kwargs:
            [kwargs.pop(k, None) for k in ['orientation', 'ratios', 'reversed', 'spacing']]

        self = Layout.__new__(cls, name, **kwargs)
        return self

    def __init__(self, name=None, orientation='vertical', spacing=2, reversed=False, ratios=None, **kwargs):
        """
        spacing - absolute space between controls
        orientation - the orientation of the layout [ AutoLayout.HORIZONTAL | AutoLayout.VERTICAL ]
        """
        Layout.__init__(self, **kwargs)
        self._spacing = spacing
        self._orientation = self.Orientation.getIndex(orientation)
        self._reversed = reversed
        self._ratios = ratios and list(ratios) or []

    def attachForm(self, *args):
        kwargs = {'edit': True}
        kwargs['attachForm'] = [args]
        cmds.formLayout(self, **kwargs)

    def attachControl(self, *args):
        kwargs = {'edit': True}
        kwargs['attachControl'] = [args]
        cmds.formLayout(self, **kwargs)

    def attachNone(self, *args):
        kwargs = {'edit': True}
        kwargs['attachNone'] = [args]
        cmds.formLayout(self, **kwargs)

    def attachPosition(self, *args):
        kwargs = {'edit': True}
        kwargs['attachPosition'] = [args]
        cmds.formLayout(self, **kwargs)

    HORIZONTAL = 0
    VERTICAL = 1
    Orientation = _util.enum.Enum('Orientation', ['horizontal', 'vertical'])

    def flip(self):
        """Flip the orientation of the layout """
        self._orientation = 1 - self._orientation
        self.redistribute(*self._ratios)

    def reverse(self):
        """Reverse the children order """
        self._reversed = not self._reversed
        self._ratios.reverse()
        self.redistribute(*self._ratios)

    def reset(self):
        self._ratios = []
        self._reversed = False
        self.redistribute()

    def redistribute(self, *ratios):
        """
        Redistribute the child controls based on the ratios.
        If not ratios are given (or not enough), 1 will be used
        """

        sides = [["top", "bottom"], ["left", "right"]]

        children = self.getChildArray()
        if not children:
            return
        if self._reversed:
            children.reverse()

        ratios = list(ratios) or self._ratios or []
        ratios += [1] * (len(children) - len(ratios))
        self._ratios = ratios
        total = sum(ratios)

        for i, child in enumerate(children):
            for side in sides[self._orientation]:
                self.attachForm(child, side, self._spacing)

            if i == 0:
                self.attachForm(child,
                                sides[1 - self._orientation][0],
                                self._spacing)
            else:
                self.attachControl(child,
                                   sides[1 - self._orientation][0],
                                   self._spacing,
                                   children[i - 1])

            if ratios[i]:
                self.attachPosition(children[i],
                                    sides[1 - self._orientation][1],
                                    self._spacing,
                                    float(sum(ratios[:i + 1])) / float(total) * 100)
            else:
                self.attachNone(children[i],
                                sides[1 - self._orientation][1])

    def vDistribute(self, *ratios):
        self._orientation = int(self.Orientation.vertical)
        self.redistribute(*ratios)

    def hDistribute(self, *ratios):
        self._orientation = int(self.Orientation.horizontal)
        self.redistribute(*ratios)
# ------ Do not edit below this line --------
    __melcmd__ = staticmethod(windows.formLayout)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'formLayout'
    __melui__ = 'formLayout'
    __slots__ = ()

    @_f.addMelDocs('formLayout', 'attachOppositeControl')
    def attachOppositeControl(self, val=True, **kwargs):
        # type: (Tuple[str, str, int, str] | List[Tuple[str, str, int, str]], **Any) -> None
        return _f.asEdit(self, windows.formLayout, kwargs, 'attachOppositeControl', val)

    @_f.addMelDocs('formLayout', 'attachOppositeForm')
    def attachOppositeForm(self, val=True, **kwargs):
        # type: (Tuple[str, str, int] | List[Tuple[str, str, int]], **Any) -> None
        return _f.asEdit(self, windows.formLayout, kwargs, 'attachOppositeForm', val)

    @_f.addMelDocs('formLayout', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.formLayout, kwargs, 'dragCallback', val)

    @_f.addMelDocs('formLayout', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.formLayout, kwargs, 'dropCallback', val)

    @_f.addMelDocs('formLayout', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.formLayout, kwargs, 'annotation')
        return res

    @_f.addMelDocs('formLayout', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.formLayout, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('formLayout', 'childArray')
    def getChildArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.formLayout, kwargs, 'childArray')
        return res

    @_f.addMelDocs('formLayout', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.formLayout, kwargs, 'docTag')
        return res

    @_f.addMelDocs('formLayout', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.formLayout, kwargs, 'enable')
        return res

    @_f.addMelDocs('formLayout', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.formLayout, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('formLayout', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.formLayout, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('formLayout', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.formLayout, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('formLayout', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.formLayout, kwargs, 'height')
        return res

    @_f.addMelDocs('formLayout', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.formLayout, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('formLayout', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.formLayout, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('formLayout', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.formLayout, kwargs, 'manage')
        return res

    @_f.addMelDocs('formLayout', 'numberOfChildren')
    def getNumberOfChildren(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.formLayout, kwargs, 'numberOfChildren')
        return res

    @_f.addMelDocs('formLayout', 'numberOfDivisions')
    def getNumberOfDivisions(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.formLayout, kwargs, 'numberOfDivisions')
        return res

    @_f.addMelDocs('formLayout', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.formLayout, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('formLayout', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.formLayout, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('formLayout', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.formLayout, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('formLayout', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.formLayout, kwargs, 'visible')
        return res

    @_f.addMelDocs('formLayout', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.formLayout, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('formLayout', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.formLayout, kwargs, 'width')
        return res

    @_f.addMelDocs('formLayout', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.formLayout, kwargs, 'noBackground', val)

    @_f.addMelDocs('formLayout', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.formLayout, kwargs, 'annotation', val)

    @_f.addMelDocs('formLayout', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.formLayout, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('formLayout', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.formLayout, kwargs, 'docTag', val)

    @_f.addMelDocs('formLayout', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.formLayout, kwargs, 'enable', val)

    @_f.addMelDocs('formLayout', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.formLayout, kwargs, 'enableBackground', val)

    @_f.addMelDocs('formLayout', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.formLayout, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('formLayout', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.formLayout, kwargs, 'height', val)

    @_f.addMelDocs('formLayout', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.formLayout, kwargs, 'highlightColor', val)

    @_f.addMelDocs('formLayout', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.formLayout, kwargs, 'manage', val)

    @_f.addMelDocs('formLayout', 'numberOfDivisions')
    def setNumberOfDivisions(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.formLayout, kwargs, 'numberOfDivisions', val)

    @_f.addMelDocs('formLayout', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.formLayout, kwargs, 'preventOverride', val)

    @_f.addMelDocs('formLayout', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.formLayout, kwargs, 'visible', val)

    @_f.addMelDocs('formLayout', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.formLayout, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('formLayout', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.formLayout, kwargs, 'width', val)

    @_f.addMelDocs('formLayout', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.formLayout, kwargs, 'statusBarMessage', val)
# ------ Do not edit above this line --------


class AutoLayout(FormLayout):

    """
    AutoLayout behaves exactly like `FormLayout`, but will call redistribute automatically
    at the end of a 'with' statement block
    """

    def __exit__(self, type, value, traceback):
        self.redistribute()
        super(AutoLayout, self).__exit__(type, value, traceback)


class RowLayout(Layout):
    pass
# ------ Do not edit below this line --------
    __melcmd__ = staticmethod(windows.rowLayout)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'rowLayout'
    __melui__ = 'rowLayout'
    __slots__ = ()

    @_f.addMelDocs('rowLayout', 'adjustableColumn')
    def adjustableColumn(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.rowLayout, kwargs, 'adjustableColumn', val)

    @_f.addMelDocs('rowLayout', 'columnAlign')
    def columnAlign(self, val=True, **kwargs):
        # type: (Tuple[int, str] | List[Tuple[int, str]], **Any) -> None
        return _f.asEdit(self, windows.rowLayout, kwargs, 'columnAlign', val)

    @_f.addMelDocs('rowLayout', 'columnAttach')
    def columnAttach(self, val=True, **kwargs):
        # type: (Tuple[int, str, int] | List[Tuple[int, str, int]], **Any) -> None
        return _f.asEdit(self, windows.rowLayout, kwargs, 'columnAttach', val)

    @_f.addMelDocs('rowLayout', 'columnWidth')
    def columnWidth(self, val=True, **kwargs):
        # type: (Tuple[int, int] | List[Tuple[int, int]], **Any) -> None
        return _f.asEdit(self, windows.rowLayout, kwargs, 'columnWidth', val)

    @_f.addMelDocs('rowLayout', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.rowLayout, kwargs, 'dragCallback', val)

    @_f.addMelDocs('rowLayout', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.rowLayout, kwargs, 'dropCallback', val)

    @_f.addMelDocs('rowLayout', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.rowLayout, kwargs, 'annotation')
        return res

    @_f.addMelDocs('rowLayout', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.rowLayout, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('rowLayout', 'childArray')
    def getChildArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.rowLayout, kwargs, 'childArray')
        return res

    @_f.addMelDocs('rowLayout', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.rowLayout, kwargs, 'docTag')
        return res

    @_f.addMelDocs('rowLayout', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.rowLayout, kwargs, 'enable')
        return res

    @_f.addMelDocs('rowLayout', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.rowLayout, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('rowLayout', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.rowLayout, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('rowLayout', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.rowLayout, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('rowLayout', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.rowLayout, kwargs, 'height')
        return res

    @_f.addMelDocs('rowLayout', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.rowLayout, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('rowLayout', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.rowLayout, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('rowLayout', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.rowLayout, kwargs, 'manage')
        return res

    @_f.addMelDocs('rowLayout', 'numberOfChildren')
    def getNumberOfChildren(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.rowLayout, kwargs, 'numberOfChildren')
        return res

    @_f.addMelDocs('rowLayout', 'numberOfColumns')
    def getNumberOfColumns(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.rowLayout, kwargs, 'numberOfColumns')
        return res

    @_f.addMelDocs('rowLayout', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.rowLayout, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('rowLayout', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.rowLayout, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('rowLayout', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.rowLayout, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('rowLayout', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.rowLayout, kwargs, 'visible')
        return res

    @_f.addMelDocs('rowLayout', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.rowLayout, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('rowLayout', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.rowLayout, kwargs, 'width')
        return res

    @_f.addMelDocs('rowLayout', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.rowLayout, kwargs, 'noBackground', val)

    @_f.addMelDocs('rowLayout', 'rowAttach')
    def rowAttach(self, val=True, **kwargs):
        # type: (Tuple[int, str, int] | List[Tuple[int, str, int]], **Any) -> None
        return _f.asEdit(self, windows.rowLayout, kwargs, 'rowAttach', val)

    @_f.addMelDocs('rowLayout', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.rowLayout, kwargs, 'annotation', val)

    @_f.addMelDocs('rowLayout', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.rowLayout, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('rowLayout', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.rowLayout, kwargs, 'docTag', val)

    @_f.addMelDocs('rowLayout', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.rowLayout, kwargs, 'enable', val)

    @_f.addMelDocs('rowLayout', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.rowLayout, kwargs, 'enableBackground', val)

    @_f.addMelDocs('rowLayout', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.rowLayout, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('rowLayout', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.rowLayout, kwargs, 'height', val)

    @_f.addMelDocs('rowLayout', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.rowLayout, kwargs, 'highlightColor', val)

    @_f.addMelDocs('rowLayout', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.rowLayout, kwargs, 'manage', val)

    @_f.addMelDocs('rowLayout', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.rowLayout, kwargs, 'preventOverride', val)

    @_f.addMelDocs('rowLayout', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.rowLayout, kwargs, 'visible', val)

    @_f.addMelDocs('rowLayout', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.rowLayout, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('rowLayout', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.rowLayout, kwargs, 'width', val)

    @_f.addMelDocs('rowLayout', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.rowLayout, kwargs, 'statusBarMessage', val)
# ------ Do not edit above this line --------


class TextScrollList(PyUI):

    def extend(self, appendList):
        """ append a list of strings"""

        for x in appendList:
            self.append(x)

    def selectIndexedItems(self, selectList):
        """select a list of indices"""
        for x in selectList:
            self.setSelectIndexedItem(x)

    def removeIndexedItems(self, removeList):
        """remove a list of indices"""
        for x in removeList:
            self.removeIndexedItem(x)

    def selectAll(self):
        """select all items"""
        numberOfItems = self.getNumberOfItems()
        self.selectIndexedItems(list(range(1, numberOfItems + 1)))
# ------ Do not edit below this line --------
    __melcmd__ = staticmethod(windows.textScrollList)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'textScrollList'
    __melui__ = 'textScrollList'
    __slots__ = ()

    @_f.addMelDocs('textScrollList', 'append')
    def append(self, val=True, **kwargs):
        # type: (str | List[str], **Any) -> None
        return _f.asEdit(self, windows.textScrollList, kwargs, 'append', val)

    @_f.addMelDocs('textScrollList', 'appendPosition')
    def appendPosition(self, val=True, **kwargs):
        # type: (Tuple[int, str] | List[Tuple[int, str]], **Any) -> None
        return _f.asEdit(self, windows.textScrollList, kwargs, 'appendPosition', val)

    @_f.addMelDocs('textScrollList', 'deleteKeyCommand')
    def deleteKeyCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.textScrollList, kwargs, 'deleteKeyCommand', val)

    @_f.addMelDocs('textScrollList', 'deselectAll')
    def deselectAll(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.textScrollList, kwargs, 'deselectAll', val)

    @_f.addMelDocs('textScrollList', 'deselectIndexedItem')
    def deselectIndexedItem(self, val=True, **kwargs):
        # type: (int | List[int], **Any) -> None
        return _f.asEdit(self, windows.textScrollList, kwargs, 'deselectIndexedItem', val)

    @_f.addMelDocs('textScrollList', 'deselectItem')
    def deselectItem(self, val=True, **kwargs):
        # type: (str | List[str], **Any) -> None
        return _f.asEdit(self, windows.textScrollList, kwargs, 'deselectItem', val)

    @_f.addMelDocs('textScrollList', 'doubleClickCommand')
    def doubleClickCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.textScrollList, kwargs, 'doubleClickCommand', val)

    @_f.addMelDocs('textScrollList', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.textScrollList, kwargs, 'dragCallback', val)

    @_f.addMelDocs('textScrollList', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.textScrollList, kwargs, 'dropCallback', val)

    @_f.addMelDocs('textScrollList', 'allItems')
    def getAllItems(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.textScrollList, kwargs, 'allItems')
        return res

    @_f.addMelDocs('textScrollList', 'allowAutomaticSelection')
    def getAllowAutomaticSelection(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.textScrollList, kwargs, 'allowAutomaticSelection')
        return res

    @_f.addMelDocs('textScrollList', 'allowMultiSelection')
    def getAllowMultiSelection(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.textScrollList, kwargs, 'allowMultiSelection')
        return res

    @_f.addMelDocs('textScrollList', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.textScrollList, kwargs, 'annotation')
        return res

    @_f.addMelDocs('textScrollList', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.textScrollList, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('textScrollList', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.textScrollList, kwargs, 'docTag')
        return res

    @_f.addMelDocs('textScrollList', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.textScrollList, kwargs, 'enable')
        return res

    @_f.addMelDocs('textScrollList', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.textScrollList, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('textScrollList', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.textScrollList, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('textScrollList', 'font')
    def getFont(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.textScrollList, kwargs, 'font')
        return res

    @_f.addMelDocs('textScrollList', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.textScrollList, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('textScrollList', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.textScrollList, kwargs, 'height')
        return res

    @_f.addMelDocs('textScrollList', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.textScrollList, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('textScrollList', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.textScrollList, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('textScrollList', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.textScrollList, kwargs, 'manage')
        return res

    @_f.addMelDocs('textScrollList', 'numberOfItems')
    def getNumberOfItems(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.textScrollList, kwargs, 'numberOfItems')
        return res

    @_f.addMelDocs('textScrollList', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.textScrollList, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('textScrollList', 'numberOfRows')
    def getNumberOfRows(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.textScrollList, kwargs, 'numberOfRows')
        return res

    @_f.addMelDocs('textScrollList', 'numberOfSelectedItems')
    def getNumberOfSelectedItems(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.textScrollList, kwargs, 'numberOfSelectedItems')
        return res

    @_f.addMelDocs('textScrollList', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.textScrollList, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('textScrollList', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.textScrollList, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('textScrollList', 'selectIndexedItem')
    def getSelectIndexedItem(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.textScrollList, kwargs, 'selectIndexedItem')
        return res

    @_f.addMelDocs('textScrollList', 'selectItem')
    def getSelectItem(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.textScrollList, kwargs, 'selectItem')
        return res

    @_f.addMelDocs('textScrollList', 'selectUniqueTagItem')
    def getSelectUniqueTagItem(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.textScrollList, kwargs, 'selectUniqueTagItem')
        return res

    @_f.addMelDocs('textScrollList', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.textScrollList, kwargs, 'visible')
        return res

    @_f.addMelDocs('textScrollList', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.textScrollList, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('textScrollList', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.textScrollList, kwargs, 'width')
        return res

    @_f.addMelDocs('textScrollList', 'lineFont')
    def lineFont(self, val=True, **kwargs):
        # type: (Tuple[int, str] | List[Tuple[int, str]], **Any) -> None
        return _f.asEdit(self, windows.textScrollList, kwargs, 'lineFont', val)

    @_f.addMelDocs('textScrollList', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.textScrollList, kwargs, 'noBackground', val)

    @_f.addMelDocs('textScrollList', 'removeAll')
    def removeAll(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.textScrollList, kwargs, 'removeAll', val)

    @_f.addMelDocs('textScrollList', 'removeIndexedItem')
    def removeIndexedItem(self, val=True, **kwargs):
        # type: (int | List[int], **Any) -> None
        return _f.asEdit(self, windows.textScrollList, kwargs, 'removeIndexedItem', val)

    @_f.addMelDocs('textScrollList', 'removeItem')
    def removeItem(self, val=True, **kwargs):
        # type: (str | List[str], **Any) -> None
        return _f.asEdit(self, windows.textScrollList, kwargs, 'removeItem', val)

    @_f.addMelDocs('textScrollList', 'selectCommand')
    def selectCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.textScrollList, kwargs, 'selectCommand', val)

    @_f.addMelDocs('textScrollList', 'allowAutomaticSelection')
    def setAllowAutomaticSelection(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.textScrollList, kwargs, 'allowAutomaticSelection', val)

    @_f.addMelDocs('textScrollList', 'allowMultiSelection')
    def setAllowMultiSelection(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.textScrollList, kwargs, 'allowMultiSelection', val)

    @_f.addMelDocs('textScrollList', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.textScrollList, kwargs, 'annotation', val)

    @_f.addMelDocs('textScrollList', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.textScrollList, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('textScrollList', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.textScrollList, kwargs, 'docTag', val)

    @_f.addMelDocs('textScrollList', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.textScrollList, kwargs, 'enable', val)

    @_f.addMelDocs('textScrollList', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.textScrollList, kwargs, 'enableBackground', val)

    @_f.addMelDocs('textScrollList', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.textScrollList, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('textScrollList', 'font')
    def setFont(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.textScrollList, kwargs, 'font', val)

    @_f.addMelDocs('textScrollList', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.textScrollList, kwargs, 'height', val)

    @_f.addMelDocs('textScrollList', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.textScrollList, kwargs, 'highlightColor', val)

    @_f.addMelDocs('textScrollList', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.textScrollList, kwargs, 'manage', val)

    @_f.addMelDocs('textScrollList', 'numberOfRows')
    def setNumberOfRows(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.textScrollList, kwargs, 'numberOfRows', val)

    @_f.addMelDocs('textScrollList', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.textScrollList, kwargs, 'preventOverride', val)

    @_f.addMelDocs('textScrollList', 'selectIndexedItem')
    def setSelectIndexedItem(self, val=True, **kwargs):
        # type: (int | List[int], **Any) -> None
        return _f.asEdit(self, windows.textScrollList, kwargs, 'selectIndexedItem', val)

    @_f.addMelDocs('textScrollList', 'selectItem')
    def setSelectItem(self, val=True, **kwargs):
        # type: (str | List[str], **Any) -> None
        return _f.asEdit(self, windows.textScrollList, kwargs, 'selectItem', val)

    @_f.addMelDocs('textScrollList', 'selectUniqueTagItem')
    def setSelectUniqueTagItem(self, val=True, **kwargs):
        # type: (str | List[str], **Any) -> None
        return _f.asEdit(self, windows.textScrollList, kwargs, 'selectUniqueTagItem', val)

    @_f.addMelDocs('textScrollList', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.textScrollList, kwargs, 'visible', val)

    @_f.addMelDocs('textScrollList', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.textScrollList, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('textScrollList', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.textScrollList, kwargs, 'width', val)

    @_f.addMelDocs('textScrollList', 'showIndexedItem')
    def showIndexedItem(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.textScrollList, kwargs, 'showIndexedItem', val)

    @_f.addMelDocs('textScrollList', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.textScrollList, kwargs, 'statusBarMessage', val)

    @_f.addMelDocs('textScrollList', 'uniqueTag')
    def uniqueTag(self, val=True, **kwargs):
        # type: (str | List[str], **Any) -> None
        return _f.asEdit(self, windows.textScrollList, kwargs, 'uniqueTag', val)
# ------ Do not edit above this line --------


class Menu(PyUI):

    def __enter__(self):
        global _withParentMenuStack
        _withParentMenuStack.append(self)
        self.makeDefault()
        return self

    def __exit__(self, type, value, traceback):
        global _withParentMenuStack
        _withParentMenuStack.pop()
        if _withParentMenuStack:
            cmds.setParent(_withParentMenuStack[-1], menu=True)
        else:
            parent = self
            while True:
                parent = parent.parent()
                # Maya 2012 Service Pack 2 (or SAP1, SP1) introduces a bug where
                # '' is returned, instead of None; problem being that doing
                # cmds.setParent(None, menu=True) is valid, but
                # cmds.setParent('', menu=True) is not
                if parent == '':
                    parent = None
                try:
                    cmds.setParent(parent, menu=True)
                except RuntimeError:
                    continue
                break

    def getItemArray(self):
        """
        Modified to return pymel instances """
        return [MenuItem(self + '|' + item)
                for item in cmds.menu(self, query=True, itemArray=True) or []]

    def makeDefault(self):
        """
        set this layout as the default parent
        """
        cmds.setParent(self, menu=True)
# ------ Do not edit below this line --------
    __melcmd__ = staticmethod(windows.menu)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'menu'
    __melui__ = 'menu'
    __slots__ = ()

    @_f.addMelDocs('menu', 'deleteAllItems')
    def deleteAllItems(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.menu, kwargs, 'deleteAllItems', val)

    @_f.addMelDocs('menu', 'allowOptionBoxes')
    def getAllowOptionBoxes(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.menu, kwargs, 'allowOptionBoxes')
        return res

    @_f.addMelDocs('menu', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.menu, kwargs, 'docTag')
        return res

    @_f.addMelDocs('menu', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.menu, kwargs, 'enable')
        return res

    @_f.addMelDocs('menu', 'familyImage')
    def getFamilyImage(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.menu, kwargs, 'familyImage')
        return res

    @_f.addMelDocs('menu', 'helpMenu')
    def getHelpMenu(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.menu, kwargs, 'helpMenu')
        return res

    @_f.addMelDocs('menu', 'label')
    def getLabel(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.menu, kwargs, 'label')
        return res

    @_f.addMelDocs('menu', 'ltVersion')
    def getLtVersion(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.menu, kwargs, 'ltVersion')
        return res

    @_f.addMelDocs('menu', 'mnemonic')
    def getMnemonic(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.menu, kwargs, 'mnemonic')
        return res

    @_f.addMelDocs('menu', 'numberOfItems')
    def getNumberOfItems(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.menu, kwargs, 'numberOfItems')
        return res

    @_f.addMelDocs('menu', 'postMenuCommandOnce')
    def getPostMenuCommandOnce(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.menu, kwargs, 'postMenuCommandOnce')
        return res

    @_f.addMelDocs('menu', 'scrollable')
    def getScrollable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.menu, kwargs, 'scrollable')
        return res

    @_f.addMelDocs('menu', 'version')
    def getVersion(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.menu, kwargs, 'version')
        return res

    @_f.addMelDocs('menu', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.menu, kwargs, 'visible')
        return res

    @_f.addMelDocs('menu', 'postMenuCommand')
    def postMenuCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.menu, kwargs, 'postMenuCommand', val)

    @_f.addMelDocs('menu', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.menu, kwargs, 'docTag', val)

    @_f.addMelDocs('menu', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.menu, kwargs, 'enable', val)

    @_f.addMelDocs('menu', 'familyImage')
    def setFamilyImage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.menu, kwargs, 'familyImage', val)

    @_f.addMelDocs('menu', 'helpMenu')
    def setHelpMenu(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.menu, kwargs, 'helpMenu', val)

    @_f.addMelDocs('menu', 'label')
    def setLabel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.menu, kwargs, 'label', val)

    @_f.addMelDocs('menu', 'ltVersion')
    def setLtVersion(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.menu, kwargs, 'ltVersion', val)

    @_f.addMelDocs('menu', 'mnemonic')
    def setMnemonic(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.menu, kwargs, 'mnemonic', val)

    @_f.addMelDocs('menu', 'postMenuCommandOnce')
    def setPostMenuCommandOnce(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.menu, kwargs, 'postMenuCommandOnce', val)

    @_f.addMelDocs('menu', 'scrollable')
    def setScrollable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.menu, kwargs, 'scrollable', val)

    @_f.addMelDocs('menu', 'version')
    def setVersion(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.menu, kwargs, 'version', val)

    @_f.addMelDocs('menu', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.menu, kwargs, 'visible', val)
# ------ Do not edit above this line --------


class PopupMenu(Menu):
    pass
# ------ Do not edit below this line --------
    __melcmd__ = staticmethod(windows.popupMenu)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'popupMenu'
    __melui__ = 'popupMenu'
    __slots__ = ()

    @_f.addMelDocs('popupMenu', 'deleteAllItems')
    def deleteAllItems(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.popupMenu, kwargs, 'deleteAllItems', val)

    @_f.addMelDocs('popupMenu', 'allowOptionBoxes')
    def getAllowOptionBoxes(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.popupMenu, kwargs, 'allowOptionBoxes')
        return res

    @_f.addMelDocs('popupMenu', 'altModifier')
    def getAltModifier(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.popupMenu, kwargs, 'altModifier')
        return res

    @_f.addMelDocs('popupMenu', 'button')
    def getButton(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.popupMenu, kwargs, 'button')
        return res

    @_f.addMelDocs('popupMenu', 'ctrlModifier')
    def getCtrlModifier(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.popupMenu, kwargs, 'ctrlModifier')
        return res

    @_f.addMelDocs('popupMenu', 'markingMenu')
    def getMarkingMenu(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.popupMenu, kwargs, 'markingMenu')
        return res

    @_f.addMelDocs('popupMenu', 'numberOfItems')
    def getNumberOfItems(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.popupMenu, kwargs, 'numberOfItems')
        return res

    @_f.addMelDocs('popupMenu', 'postMenuCommandOnce')
    def getPostMenuCommandOnce(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.popupMenu, kwargs, 'postMenuCommandOnce')
        return res

    @_f.addMelDocs('popupMenu', 'shiftModifier')
    def getShiftModifier(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.popupMenu, kwargs, 'shiftModifier')
        return res

    @_f.addMelDocs('popupMenu', 'postMenuCommand')
    def postMenuCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.popupMenu, kwargs, 'postMenuCommand', val)

    @_f.addMelDocs('popupMenu', 'altModifier')
    def setAltModifier(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.popupMenu, kwargs, 'altModifier', val)

    @_f.addMelDocs('popupMenu', 'button')
    def setButton(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.popupMenu, kwargs, 'button', val)

    @_f.addMelDocs('popupMenu', 'ctrlModifier')
    def setCtrlModifier(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.popupMenu, kwargs, 'ctrlModifier', val)

    @_f.addMelDocs('popupMenu', 'markingMenu')
    def setMarkingMenu(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.popupMenu, kwargs, 'markingMenu', val)

    @_f.addMelDocs('popupMenu', 'postMenuCommandOnce')
    def setPostMenuCommandOnce(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.popupMenu, kwargs, 'postMenuCommandOnce', val)

    @_f.addMelDocs('popupMenu', 'shiftModifier')
    def setShiftModifier(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.popupMenu, kwargs, 'shiftModifier', val)
# ------ Do not edit above this line --------


class OptionMenu(PopupMenu):

    def addMenuItems(self, items, title=None):
        """ Add the specified item list to the OptionMenu, with an optional 'title' item """
        if title:
            cmds.menuItem(l=title, en=0, parent=self)
        for item in items:
            cmds.menuItem(l=item, parent=self)

    def clear(self):
        """ Clear all menu items from this OptionMenu """
        for t in self.getItemListLong() or []:
            cmds.deleteUI(t)
    addItems = addMenuItems
# ------ Do not edit below this line --------
    __melcmd__ = staticmethod(windows.optionMenu)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'optionMenu'
    __melui__ = 'optionMenu'
    __slots__ = ()

    @_f.addMelDocs('optionMenu', 'beforeShowPopup')
    def beforeShowPopup(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.optionMenu, kwargs, 'beforeShowPopup', val)

    @_f.addMelDocs('optionMenu', 'changeCommand')
    def changeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.optionMenu, kwargs, 'changeCommand', val)

    @_f.addMelDocs('optionMenu', 'deleteAllItems')
    def deleteAllItems(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.optionMenu, kwargs, 'deleteAllItems', val)

    @_f.addMelDocs('optionMenu', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.optionMenu, kwargs, 'dragCallback', val)

    @_f.addMelDocs('optionMenu', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.optionMenu, kwargs, 'dropCallback', val)

    @_f.addMelDocs('optionMenu', 'alwaysCallChangeCommand')
    def getAlwaysCallChangeCommand(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.optionMenu, kwargs, 'alwaysCallChangeCommand')
        return res

    @_f.addMelDocs('optionMenu', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.optionMenu, kwargs, 'annotation')
        return res

    @_f.addMelDocs('optionMenu', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.optionMenu, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('optionMenu', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.optionMenu, kwargs, 'docTag')
        return res

    @_f.addMelDocs('optionMenu', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.optionMenu, kwargs, 'enable')
        return res

    @_f.addMelDocs('optionMenu', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.optionMenu, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('optionMenu', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.optionMenu, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('optionMenu', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.optionMenu, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('optionMenu', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.optionMenu, kwargs, 'height')
        return res

    @_f.addMelDocs('optionMenu', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.optionMenu, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('optionMenu', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.optionMenu, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('optionMenu', 'itemListLong')
    def getItemListLong(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.optionMenu, kwargs, 'itemListLong')
        return res

    @_f.addMelDocs('optionMenu', 'itemListShort')
    def getItemListShort(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.optionMenu, kwargs, 'itemListShort')
        return res

    @_f.addMelDocs('optionMenu', 'label')
    def getLabel(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.optionMenu, kwargs, 'label')
        return res

    @_f.addMelDocs('optionMenu', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.optionMenu, kwargs, 'manage')
        return res

    @_f.addMelDocs('optionMenu', 'maxVisibleItems')
    def getMaxVisibleItems(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.optionMenu, kwargs, 'maxVisibleItems')
        return res

    @_f.addMelDocs('optionMenu', 'numberOfItems')
    def getNumberOfItems(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.optionMenu, kwargs, 'numberOfItems')
        return res

    @_f.addMelDocs('optionMenu', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.optionMenu, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('optionMenu', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.optionMenu, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('optionMenu', 'postMenuCommandOnce')
    def getPostMenuCommandOnce(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.optionMenu, kwargs, 'postMenuCommandOnce')
        return res

    @_f.addMelDocs('optionMenu', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.optionMenu, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('optionMenu', 'select')
    def getSelect(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.optionMenu, kwargs, 'select')
        return res

    @_f.addMelDocs('optionMenu', 'value')
    def getValue(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.optionMenu, kwargs, 'value')
        return res

    @_f.addMelDocs('optionMenu', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.optionMenu, kwargs, 'visible')
        return res

    @_f.addMelDocs('optionMenu', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.optionMenu, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('optionMenu', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.optionMenu, kwargs, 'width')
        return res

    @_f.addMelDocs('optionMenu', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.optionMenu, kwargs, 'noBackground', val)

    @_f.addMelDocs('optionMenu', 'postMenuCommand')
    def postMenuCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.optionMenu, kwargs, 'postMenuCommand', val)

    @_f.addMelDocs('optionMenu', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.optionMenu, kwargs, 'annotation', val)

    @_f.addMelDocs('optionMenu', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.optionMenu, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('optionMenu', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.optionMenu, kwargs, 'docTag', val)

    @_f.addMelDocs('optionMenu', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.optionMenu, kwargs, 'enable', val)

    @_f.addMelDocs('optionMenu', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.optionMenu, kwargs, 'enableBackground', val)

    @_f.addMelDocs('optionMenu', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.optionMenu, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('optionMenu', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.optionMenu, kwargs, 'height', val)

    @_f.addMelDocs('optionMenu', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.optionMenu, kwargs, 'highlightColor', val)

    @_f.addMelDocs('optionMenu', 'label')
    def setLabel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.optionMenu, kwargs, 'label', val)

    @_f.addMelDocs('optionMenu', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.optionMenu, kwargs, 'manage', val)

    @_f.addMelDocs('optionMenu', 'maxVisibleItems')
    def setMaxVisibleItems(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.optionMenu, kwargs, 'maxVisibleItems', val)

    @_f.addMelDocs('optionMenu', 'postMenuCommandOnce')
    def setPostMenuCommandOnce(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.optionMenu, kwargs, 'postMenuCommandOnce', val)

    @_f.addMelDocs('optionMenu', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.optionMenu, kwargs, 'preventOverride', val)

    @_f.addMelDocs('optionMenu', 'select')
    def setSelect(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.optionMenu, kwargs, 'select', val)

    @_f.addMelDocs('optionMenu', 'value')
    def setValue(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.optionMenu, kwargs, 'value', val)

    @_f.addMelDocs('optionMenu', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.optionMenu, kwargs, 'visible', val)

    @_f.addMelDocs('optionMenu', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.optionMenu, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('optionMenu', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.optionMenu, kwargs, 'width', val)

    @_f.addMelDocs('optionMenu', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.optionMenu, kwargs, 'statusBarMessage', val)
# ------ Do not edit above this line --------


class OptionMenuGrp(RowLayout):

    def menu(self):
        for child in self.children():
            if isinstance(child, OptionMenu):
                return child

    # Want to set both the menu to the child |OptionMenu item, and the normal
    # parent to this...
    def __enter__(self):
        self.menu().__enter__()
        return super(OptionMenuGrp, self).__enter__()

    def __exit__(self, type, value, traceback):
        self.menu().__exit__(type, value, traceback)
        return super(OptionMenuGrp, self).__exit__(type, value, traceback)
# ------ Do not edit below this line --------
    __melcmd__ = staticmethod(windows.optionMenuGrp)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'optionMenuGrp'
    __melui__ = 'optionMenuGrp'
    __slots__ = ()

    @_f.addMelDocs('optionMenuGrp', 'adjustableColumn')
    def adjustableColumn(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'adjustableColumn', val)

    @_f.addMelDocs('optionMenuGrp', 'adjustableColumn2')
    def adjustableColumn2(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'adjustableColumn2', val)

    @_f.addMelDocs('optionMenuGrp', 'adjustableColumn3')
    def adjustableColumn3(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'adjustableColumn3', val)

    @_f.addMelDocs('optionMenuGrp', 'adjustableColumn4')
    def adjustableColumn4(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'adjustableColumn4', val)

    @_f.addMelDocs('optionMenuGrp', 'adjustableColumn5')
    def adjustableColumn5(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'adjustableColumn5', val)

    @_f.addMelDocs('optionMenuGrp', 'adjustableColumn6')
    def adjustableColumn6(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'adjustableColumn6', val)

    @_f.addMelDocs('optionMenuGrp', 'changeCommand')
    def changeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'changeCommand', val)

    @_f.addMelDocs('optionMenuGrp', 'columnAlign')
    def columnAlign(self, val=True, **kwargs):
        # type: (Tuple[int, str] | List[Tuple[int, str]], **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'columnAlign', val)

    @_f.addMelDocs('optionMenuGrp', 'columnAlign2')
    def columnAlign2(self, val=True, **kwargs):
        # type: (Tuple[str, str], **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'columnAlign2', val)

    @_f.addMelDocs('optionMenuGrp', 'columnAlign3')
    def columnAlign3(self, val=True, **kwargs):
        # type: (Tuple[str, str, str], **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'columnAlign3', val)

    @_f.addMelDocs('optionMenuGrp', 'columnAlign4')
    def columnAlign4(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'columnAlign4', val)

    @_f.addMelDocs('optionMenuGrp', 'columnAlign5')
    def columnAlign5(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'columnAlign5', val)

    @_f.addMelDocs('optionMenuGrp', 'columnAlign6')
    def columnAlign6(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'columnAlign6', val)

    @_f.addMelDocs('optionMenuGrp', 'columnAttach')
    def columnAttach(self, val=True, **kwargs):
        # type: (Tuple[int, str, int] | List[Tuple[int, str, int]], **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'columnAttach', val)

    @_f.addMelDocs('optionMenuGrp', 'columnAttach2')
    def columnAttach2(self, val=True, **kwargs):
        # type: (Tuple[str, str], **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'columnAttach2', val)

    @_f.addMelDocs('optionMenuGrp', 'columnAttach3')
    def columnAttach3(self, val=True, **kwargs):
        # type: (Tuple[str, str, str], **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'columnAttach3', val)

    @_f.addMelDocs('optionMenuGrp', 'columnAttach4')
    def columnAttach4(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'columnAttach4', val)

    @_f.addMelDocs('optionMenuGrp', 'columnAttach5')
    def columnAttach5(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'columnAttach5', val)

    @_f.addMelDocs('optionMenuGrp', 'columnAttach6')
    def columnAttach6(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'columnAttach6', val)

    @_f.addMelDocs('optionMenuGrp', 'columnOffset2')
    def columnOffset2(self, val=True, **kwargs):
        # type: (Tuple[int, int], **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'columnOffset2', val)

    @_f.addMelDocs('optionMenuGrp', 'columnOffset3')
    def columnOffset3(self, val=True, **kwargs):
        # type: (Tuple[int, int, int], **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'columnOffset3', val)

    @_f.addMelDocs('optionMenuGrp', 'columnOffset4')
    def columnOffset4(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'columnOffset4', val)

    @_f.addMelDocs('optionMenuGrp', 'columnOffset5')
    def columnOffset5(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'columnOffset5', val)

    @_f.addMelDocs('optionMenuGrp', 'columnOffset6')
    def columnOffset6(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'columnOffset6', val)

    @_f.addMelDocs('optionMenuGrp', 'columnWidth')
    def columnWidth(self, val=True, **kwargs):
        # type: (Tuple[int, int] | List[Tuple[int, int]], **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'columnWidth', val)

    @_f.addMelDocs('optionMenuGrp', 'columnWidth1')
    def columnWidth1(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'columnWidth1', val)

    @_f.addMelDocs('optionMenuGrp', 'columnWidth2')
    def columnWidth2(self, val=True, **kwargs):
        # type: (Tuple[int, int], **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'columnWidth2', val)

    @_f.addMelDocs('optionMenuGrp', 'columnWidth3')
    def columnWidth3(self, val=True, **kwargs):
        # type: (Tuple[int, int, int], **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'columnWidth3', val)

    @_f.addMelDocs('optionMenuGrp', 'columnWidth4')
    def columnWidth4(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'columnWidth4', val)

    @_f.addMelDocs('optionMenuGrp', 'columnWidth5')
    def columnWidth5(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'columnWidth5', val)

    @_f.addMelDocs('optionMenuGrp', 'columnWidth6')
    def columnWidth6(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'columnWidth6', val)

    @_f.addMelDocs('optionMenuGrp', 'deleteAllItems')
    def deleteAllItems(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'deleteAllItems', val)

    @_f.addMelDocs('optionMenuGrp', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'dragCallback', val)

    @_f.addMelDocs('optionMenuGrp', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'dropCallback', val)

    @_f.addMelDocs('optionMenuGrp', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.optionMenuGrp, kwargs, 'annotation')
        return res

    @_f.addMelDocs('optionMenuGrp', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.optionMenuGrp, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('optionMenuGrp', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.optionMenuGrp, kwargs, 'docTag')
        return res

    @_f.addMelDocs('optionMenuGrp', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.optionMenuGrp, kwargs, 'enable')
        return res

    @_f.addMelDocs('optionMenuGrp', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.optionMenuGrp, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('optionMenuGrp', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.optionMenuGrp, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('optionMenuGrp', 'extraLabel')
    def getExtraLabel(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.optionMenuGrp, kwargs, 'extraLabel')
        return res

    @_f.addMelDocs('optionMenuGrp', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.optionMenuGrp, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('optionMenuGrp', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.optionMenuGrp, kwargs, 'height')
        return res

    @_f.addMelDocs('optionMenuGrp', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.optionMenuGrp, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('optionMenuGrp', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.optionMenuGrp, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('optionMenuGrp', 'itemListLong')
    def getItemListLong(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.optionMenuGrp, kwargs, 'itemListLong')
        return res

    @_f.addMelDocs('optionMenuGrp', 'itemListShort')
    def getItemListShort(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.optionMenuGrp, kwargs, 'itemListShort')
        return res

    @_f.addMelDocs('optionMenuGrp', 'label')
    def getLabel(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.optionMenuGrp, kwargs, 'label')
        return res

    @_f.addMelDocs('optionMenuGrp', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.optionMenuGrp, kwargs, 'manage')
        return res

    @_f.addMelDocs('optionMenuGrp', 'numberOfItems')
    def getNumberOfItems(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.optionMenuGrp, kwargs, 'numberOfItems')
        return res

    @_f.addMelDocs('optionMenuGrp', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.optionMenuGrp, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('optionMenuGrp', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.optionMenuGrp, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('optionMenuGrp', 'postMenuCommandOnce')
    def getPostMenuCommandOnce(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.optionMenuGrp, kwargs, 'postMenuCommandOnce')
        return res

    @_f.addMelDocs('optionMenuGrp', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.optionMenuGrp, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('optionMenuGrp', 'select')
    def getSelect(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.optionMenuGrp, kwargs, 'select')
        return res

    @_f.addMelDocs('optionMenuGrp', 'value')
    def getValue(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.optionMenuGrp, kwargs, 'value')
        return res

    @_f.addMelDocs('optionMenuGrp', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.optionMenuGrp, kwargs, 'visible')
        return res

    @_f.addMelDocs('optionMenuGrp', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.optionMenuGrp, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('optionMenuGrp', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.optionMenuGrp, kwargs, 'width')
        return res

    @_f.addMelDocs('optionMenuGrp', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'noBackground', val)

    @_f.addMelDocs('optionMenuGrp', 'postMenuCommand')
    def postMenuCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'postMenuCommand', val)

    @_f.addMelDocs('optionMenuGrp', 'rowAttach')
    def rowAttach(self, val=True, **kwargs):
        # type: (Tuple[int, str, int] | List[Tuple[int, str, int]], **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'rowAttach', val)

    @_f.addMelDocs('optionMenuGrp', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'annotation', val)

    @_f.addMelDocs('optionMenuGrp', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('optionMenuGrp', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'docTag', val)

    @_f.addMelDocs('optionMenuGrp', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'enable', val)

    @_f.addMelDocs('optionMenuGrp', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'enableBackground', val)

    @_f.addMelDocs('optionMenuGrp', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('optionMenuGrp', 'extraLabel')
    def setExtraLabel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'extraLabel', val)

    @_f.addMelDocs('optionMenuGrp', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'height', val)

    @_f.addMelDocs('optionMenuGrp', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'highlightColor', val)

    @_f.addMelDocs('optionMenuGrp', 'label')
    def setLabel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'label', val)

    @_f.addMelDocs('optionMenuGrp', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'manage', val)

    @_f.addMelDocs('optionMenuGrp', 'postMenuCommandOnce')
    def setPostMenuCommandOnce(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'postMenuCommandOnce', val)

    @_f.addMelDocs('optionMenuGrp', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'preventOverride', val)

    @_f.addMelDocs('optionMenuGrp', 'select')
    def setSelect(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'select', val)

    @_f.addMelDocs('optionMenuGrp', 'value')
    def setValue(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'value', val)

    @_f.addMelDocs('optionMenuGrp', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'visible', val)

    @_f.addMelDocs('optionMenuGrp', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('optionMenuGrp', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'width', val)

    @_f.addMelDocs('optionMenuGrp', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.optionMenuGrp, kwargs, 'statusBarMessage', val)
# ------ Do not edit above this line --------


class SubMenuItem(Menu):

    def getBoldFont(self):
        return cmds.menuItem(self, query=True, boldFont=True)

    def getItalicized(self):
        return cmds.menuItem(self, query=True, italicized=True)

    asQtObject = toQtMenuItem


class CommandMenuItem(PyUI):
    __melui__ = 'menuItem'

    def __enter__(self):
        SubMenuItem(self).__enter__()
        return self

    def __exit__(self, type, value, traceback):
        return SubMenuItem(self).__exit__(type, value, traceback)
# ------ Do not edit below this line --------
    __melcmd__ = staticmethod(windows.menuItem)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'menuItem'
    __slots__ = ()

    @_f.addMelDocs('menuItem', 'allowOptionBoxes')
    def getAllowOptionBoxes(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.menuItem, kwargs, 'allowOptionBoxes')
        return res

    @_f.addMelDocs('menuItem', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.menuItem, kwargs, 'annotation')
        return res

    @_f.addMelDocs('menuItem', 'boldFont')
    def getBoldFont(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.menuItem, kwargs, 'boldFont')
        return res

    @_f.addMelDocs('menuItem', 'checkBox')
    def getCheckBox(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.menuItem, kwargs, 'checkBox')
        return res

    @_f.addMelDocs('menuItem', 'collection')
    def getCollection(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.menuItem, kwargs, 'collection')
        return res

    @_f.addMelDocs('menuItem', 'command')
    def getCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.menuItem, kwargs, 'command')
        return res

    @_f.addMelDocs('menuItem', 'data')
    def getData(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.menuItem, kwargs, 'data')
        return res

    @_f.addMelDocs('menuItem', 'divider')
    def getDivider(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.menuItem, kwargs, 'divider')
        return res

    @_f.addMelDocs('menuItem', 'dividerLabel')
    def getDividerLabel(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.menuItem, kwargs, 'dividerLabel')
        return res

    @_f.addMelDocs('menuItem', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.menuItem, kwargs, 'docTag')
        return res

    @_f.addMelDocs('menuItem', 'dragDoubleClickCommand')
    def getDragDoubleClickCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.menuItem, kwargs, 'dragDoubleClickCommand')
        return res

    @_f.addMelDocs('menuItem', 'dragMenuCommand')
    def getDragMenuCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.menuItem, kwargs, 'dragMenuCommand')
        return res

    @_f.addMelDocs('menuItem', 'echoCommand')
    def getEchoCommand(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.menuItem, kwargs, 'echoCommand')
        return res

    @_f.addMelDocs('menuItem', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.menuItem, kwargs, 'enable')
        return res

    @_f.addMelDocs('menuItem', 'enableCommandRepeat')
    def getEnableCommandRepeat(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.menuItem, kwargs, 'enableCommandRepeat')
        return res

    @_f.addMelDocs('menuItem', 'familyImage')
    def getFamilyImage(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.menuItem, kwargs, 'familyImage')
        return res

    @_f.addMelDocs('menuItem', 'image')
    def getImage(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.menuItem, kwargs, 'image')
        return res

    @_f.addMelDocs('menuItem', 'imageOverlayLabel')
    def getImageOverlayLabel(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.menuItem, kwargs, 'imageOverlayLabel')
        return res

    @_f.addMelDocs('menuItem', 'isCheckBox')
    def getIsCheckBox(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.menuItem, kwargs, 'isCheckBox')
        return res

    @_f.addMelDocs('menuItem', 'isOptionBox')
    def getIsOptionBox(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.menuItem, kwargs, 'isOptionBox')
        return res

    @_f.addMelDocs('menuItem', 'isRadioButton')
    def getIsRadioButton(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.menuItem, kwargs, 'isRadioButton')
        return res

    @_f.addMelDocs('menuItem', 'italicized')
    def getItalicized(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.menuItem, kwargs, 'italicized')
        return res

    @_f.addMelDocs('menuItem', 'label')
    def getLabel(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.menuItem, kwargs, 'label')
        return res

    @_f.addMelDocs('menuItem', 'longDivider')
    def getLongDivider(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.menuItem, kwargs, 'longDivider')
        return res

    @_f.addMelDocs('menuItem', 'ltVersion')
    def getLtVersion(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.menuItem, kwargs, 'ltVersion')
        return res

    @_f.addMelDocs('menuItem', 'optionBox')
    def getOptionBox(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.menuItem, kwargs, 'optionBox')
        return res

    @_f.addMelDocs('menuItem', 'optionBoxIcon')
    def getOptionBoxIcon(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.menuItem, kwargs, 'optionBoxIcon')
        return res

    @_f.addMelDocs('menuItem', 'postMenuCommand')
    def getPostMenuCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.menuItem, kwargs, 'postMenuCommand')
        return res

    @_f.addMelDocs('menuItem', 'postMenuCommandOnce')
    def getPostMenuCommandOnce(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.menuItem, kwargs, 'postMenuCommandOnce')
        return res

    @_f.addMelDocs('menuItem', 'radialPosition')
    def getRadialPosition(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.menuItem, kwargs, 'radialPosition')
        return res

    @_f.addMelDocs('menuItem', 'radioButton')
    def getRadioButton(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.menuItem, kwargs, 'radioButton')
        return res

    @_f.addMelDocs('menuItem', 'sourceType')
    def getSourceType(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.menuItem, kwargs, 'sourceType')
        return res

    @_f.addMelDocs('menuItem', 'subMenu')
    def getSubMenu(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.menuItem, kwargs, 'subMenu')
        return res

    @_f.addMelDocs('menuItem', 'tearOff')
    def getTearOff(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.menuItem, kwargs, 'tearOff')
        return res

    @_f.addMelDocs('menuItem', 'version')
    def getVersion(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.menuItem, kwargs, 'version')
        return res

    @_f.addMelDocs('menuItem', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.menuItem, kwargs, 'visible')
        return res

    @_f.addMelDocs('menuItem', 'runTimeCommand')
    def runTimeCommand(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.menuItem, kwargs, 'runTimeCommand', val)

    @_f.addMelDocs('menuItem', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.menuItem, kwargs, 'annotation', val)

    @_f.addMelDocs('menuItem', 'checkBox')
    def setCheckBox(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.menuItem, kwargs, 'checkBox', val)

    @_f.addMelDocs('menuItem', 'command')
    def setCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.menuItem, kwargs, 'command', val)

    @_f.addMelDocs('menuItem', 'data')
    def setData(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.menuItem, kwargs, 'data', val)

    @_f.addMelDocs('menuItem', 'dividerLabel')
    def setDividerLabel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.menuItem, kwargs, 'dividerLabel', val)

    @_f.addMelDocs('menuItem', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.menuItem, kwargs, 'docTag', val)

    @_f.addMelDocs('menuItem', 'dragDoubleClickCommand')
    def setDragDoubleClickCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.menuItem, kwargs, 'dragDoubleClickCommand', val)

    @_f.addMelDocs('menuItem', 'dragMenuCommand')
    def setDragMenuCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.menuItem, kwargs, 'dragMenuCommand', val)

    @_f.addMelDocs('menuItem', 'echoCommand')
    def setEchoCommand(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.menuItem, kwargs, 'echoCommand', val)

    @_f.addMelDocs('menuItem', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.menuItem, kwargs, 'enable', val)

    @_f.addMelDocs('menuItem', 'enableCommandRepeat')
    def setEnableCommandRepeat(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.menuItem, kwargs, 'enableCommandRepeat', val)

    @_f.addMelDocs('menuItem', 'image')
    def setImage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.menuItem, kwargs, 'image', val)

    @_f.addMelDocs('menuItem', 'imageOverlayLabel')
    def setImageOverlayLabel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.menuItem, kwargs, 'imageOverlayLabel', val)

    @_f.addMelDocs('menuItem', 'label')
    def setLabel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.menuItem, kwargs, 'label', val)

    @_f.addMelDocs('menuItem', 'longDivider')
    def setLongDivider(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.menuItem, kwargs, 'longDivider', val)

    @_f.addMelDocs('menuItem', 'ltVersion')
    def setLtVersion(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.menuItem, kwargs, 'ltVersion', val)

    @_f.addMelDocs('menuItem', 'optionBoxIcon')
    def setOptionBoxIcon(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.menuItem, kwargs, 'optionBoxIcon', val)

    @_f.addMelDocs('menuItem', 'postMenuCommand')
    def setPostMenuCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.menuItem, kwargs, 'postMenuCommand', val)

    @_f.addMelDocs('menuItem', 'postMenuCommandOnce')
    def setPostMenuCommandOnce(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.menuItem, kwargs, 'postMenuCommandOnce', val)

    @_f.addMelDocs('menuItem', 'radialPosition')
    def setRadialPosition(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.menuItem, kwargs, 'radialPosition', val)

    @_f.addMelDocs('menuItem', 'radioButton')
    def setRadioButton(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.menuItem, kwargs, 'radioButton', val)

    @_f.addMelDocs('menuItem', 'sourceType')
    def setSourceType(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.menuItem, kwargs, 'sourceType', val)

    @_f.addMelDocs('menuItem', 'version')
    def setVersion(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.menuItem, kwargs, 'version', val)

    @_f.addMelDocs('menuItem', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.menuItem, kwargs, 'visible', val)
# ------ Do not edit above this line --------


def MenuItem(name=None, create=False, **kwargs):
    if PyUI._isBeingCreated(name, create, kwargs):
        cls = CommandMenuItem
    else:
        try:
            uiType = objectTypeUI(name)
        except RuntimeError:
            cls = SubMenuItem
        else:
            if uiType == 'subMenuItem':
                cls = SubMenuItem
            else:
                cls = CommandMenuItem
    return cls(name, create, **kwargs)


class UITemplate(object):

    """
    from pymel.core import *

    # force deletes the template if it already exists
    template = ui.UITemplate( 'ExampleTemplate', force=True )

    template.define( button, width=100, height=40, align='left' )
    template.define( frameLayout, borderVisible=True, labelVisible=False )

    #    Create a window and apply the template.
    #
    with window():
        with template:
            with columnLayout( rowSpacing=5 ):
                with frameLayout():
                    with columnLayout():
                        button( label='One' )
                        button( label='Two' )
                        button( label='Three' )

                with frameLayout():
                    with columnLayout():
                        button( label='Red' )
                        button( label='Green' )
                        button( label='Blue' )
    """

    def __init__(self, name=None, force=False):
        if name and cmds.uiTemplate(name, exists=True):
            if force:
                cmds.deleteUI(name, uiTemplate=True)
            else:
                self._name = name
                return
        args = [name] if name else []
        self._name = cmds.uiTemplate(*args)

    def __repr__(self):
        return '%s(%r)' % (self.__class__.__name__, self._name)

    def __enter__(self):
        self.push()
        return self

    def __exit__(self, type, value, traceback):
        self.pop()

    def name(self):
        return self._name

    def push(self):
        cmds.setUITemplate(self._name, pushTemplate=True)

    def pop(self):
        cmds.setUITemplate(popTemplate=True)

    def define(self, uiType, **kwargs):
        """
        uiType can be:
            - a ui function or class
            - the name of a ui function or class
            - a list or tuple of the above
        """
        if isinstance(uiType, (list, tuple)):
            funcs = [_resolveUIFunc(x) for x in uiType]
        else:
            funcs = [_resolveUIFunc(uiType)]
        kwargs['defineTemplate'] = self._name
        for func in funcs:
            func(**kwargs)

    @staticmethod
    def exists(name):
        return cmds.uiTemplate(name, exists=True)


class AELoader(type):

    """
    Metaclass used by `AETemplate` class to create wrapping and loading
    mechanisms when an AETemplate instance is created
    """
    _loaded = []

    def __new__(cls, classname, bases, classdict):
        newcls = super(AELoader, cls).__new__(cls, classname, bases, classdict)
        try:
            nodeType = newcls.nodeType()
        except ValueError:
            _logger.debug("could not determine node type for " + classname)
        else:
            modname = classdict['__module__']
            if modname == '__builtin__':
                # since the module is __builtin__ our AE was probably included in the body of a scripted
                # plugin, which is called by maya in a strange way ( execfile? ).
                # give it a real home so we can load it later.
                mod = sys.modules['__builtin__']
                setattr(mod, classname, newcls)

            template = 'AE' + nodeType + 'Template'
            cls.makeAEProc(modname, classname, template)
            if template not in cls._loaded:
                cls._loaded.append(template)
        return newcls

    @staticmethod
    def makeAEProc(modname, classname, procname):
        _logger.debug("making AE loader procedure: %s" % procname)
        contents = '''global proc %(procname)s( string $nodeName ){
        python("import %(__name__)s;%(__name__)s.AELoader.load('%(modname)s','%(classname)s','" + $nodeName + "')");}'''
        d = locals().copy()
        d['__name__'] = __name__
        import maya.mel as mm
        mm.eval(contents % d)

    @staticmethod
    def load(modname, classname, nodename):
        mod = __import__(modname, globals(), locals(), [classname], -1)
        try:
            cls = getattr(mod, classname)
            cls(nodename)
        except Exception:
            print("failed to load python attribute editor template '%s.%s'" % (modname, classname))
            import traceback
            traceback.print_exc()

    @classmethod
    def loadedTemplates(cls):
        "Return the names of the loaded templates"
        return cls._loaded


class AETemplate(with_metaclass(AELoader, object)):

    """
    To create an Attribute Editor template using python, do the following:
         1. create a subclass of `uitypes.AETemplate`
        2. set its ``_nodeType`` class attribute to the name of the desired node type, or name the class using the
    convention ``AE<nodeType>Template``
        3. import the module

    AETemplates which do not meet one of the two requirements listed in step 2 will be ignored.  To ensure that your
    Template's node type is being detected correctly, use the ``AETemplate.nodeType()`` class method::

        import AETemplates
        AETemplates.AEmib_amb_occlusionTemplate.nodeType()

    As a convenience, when pymel is imported it will automatically import the module ``AETemplates``, if it exists,
    thereby causing any AETemplates within it or its sub-modules to be registered. Be sure to import pymel
    or modules containing your ``AETemplate`` classes before opening the Atrribute Editor for the node types in question.

    To check which python templates are loaded::

        from pymel.core.uitypes import AELoader
        print AELoader.loadedTemplates()
    """

    _nodeType = None

    def __init__(self, nodeName):
        self._nodeName = nodeName

    @property
    def nodeName(self):
        return self._nodeName

    @classmethod
    def nodeType(cls):
        if cls._nodeType:
            return cls._nodeType
        else:
            m = re.match('AE(.+)Template$', cls.__name__)
            if m:
                return m.groups()[0]
            else:
                raise ValueError("You must either name your AETemplate "
                                 "subclass of the form 'AE<nodeType>Template' "
                                 "or set the '_nodeType' class attribute")

    @classmethod
    def controlValue(cls, nodeName, control):
        return cmds.editorTemplate(queryControl=(nodeName, control))

    @classmethod
    def controlLabel(cls, nodeName, control):
        return cmds.editorTemplate(queryLabel=(nodeName, control))

    @classmethod
    def reload(cls):
        "Reload the template. Beware, this reloads the module in which the template exists!"
        nodeType = cls.nodeType()
        form = "AttrEd" + nodeType + "FormLayout"
        exists = cmds.control(form, exists=1) and cmds.formLayout(form, q=1, ca=1)

        if exists:
            sel = cmds.ls(sl=1)
            cmds.select(cl=True)
            cmds.deleteUI(form)
            if sel:
                cmds.select(sel)
        reload(sys.modules[cls.__module__])

    def addControl(self, control, label=None, changeCommand=None,
                   annotation=None, preventOverride=False, dynamic=False):
        args = [control]
        kwargs = {'preventOverride': preventOverride}
        if dynamic:
            kwargs['addDynamicControl'] = True
        else:
            kwargs['addControl'] = True
        if changeCommand:
            if hasattr(changeCommand, '__call__'):
                import pymel.tools.py2mel
                name = self.__class__.__name__ + '_callCustom_changeCommand_' + control
                changeCommand = pymel.tools.py2mel.py2melProc(
                    changeCommand, procName=name, argTypes=['string'])
            args.append(changeCommand)
        if label:
            kwargs['label'] = label
        if annotation:
            kwargs['annotation'] = annotation
        cmds.editorTemplate(*args, **kwargs)

    def callCustom(self, newFunc, replaceFunc, *attrs):
        # cmds.editorTemplate(callCustom=( (newFunc, replaceFunc) + attrs))
        import pymel.tools.py2mel
        if hasattr(newFunc, '__call__'):
            name = self.__class__.__name__ + '_callCustom_newFunc_' + '_'.join(attrs)
            newFunc = pymel.tools.py2mel.py2melProc(
                newFunc, procName=name, argTypes=['string'] * len(attrs))
        if hasattr(replaceFunc, '__call__'):
            name = self.__class__.__name__ + '_callCustom_replaceFunc_' + '_'.join(attrs)
            replaceFunc = pymel.tools.py2mel.py2melProc(
                replaceFunc, procName=name, argTypes=['string'] * len(attrs))
        args = (newFunc, replaceFunc) + attrs
        cmds.editorTemplate(callCustom=1, *args)

    def suppress(self, control):
        cmds.editorTemplate(suppress=control)

    def dimControl(self, nodeName, control, state):
        cmds.editorTemplate(dimControl=(nodeName, control, state))

    def beginLayout(self, name, collapse=True):
        cmds.editorTemplate(beginLayout=name, collapse=collapse)

    def endLayout(self):
        cmds.editorTemplate(endLayout=True)

    def beginScrollLayout(self):
        cmds.editorTemplate(beginScrollLayout=True)

    def endScrollLayout(self):
        cmds.editorTemplate(endScrollLayout=True)

    def beginNoOptimize(self):
        cmds.editorTemplate(beginNoOptimize=True)

    def endNoOptimize(self):
        cmds.editorTemplate(endNoOptimize=True)

    def interruptOptimize(self):
        cmds.editorTemplate(interruptOptimize=True)

    def addSeparator(self):
        cmds.editorTemplate(addSeparator=True)

    def addComponents(self):
        cmds.editorTemplate(addComponents=True)

    def addExtraControls(self, label=None):
        kwargs = {}
        if label:
            kwargs['extraControlsLabel'] = label
        cmds.editorTemplate(addExtraControls=True, **kwargs)

    # TODO: listExtraAttributes


class FloatFieldGrp(Layout):
    pass
# ------ Do not edit below this line --------
    __melcmd__ = staticmethod(windows.floatFieldGrp)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'floatFieldGrp'
    __melui__ = 'floatFieldGrp'
    __slots__ = ()

    @_f.addMelDocs('floatFieldGrp', 'adjustableColumn')
    def adjustableColumn(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'adjustableColumn', val)

    @_f.addMelDocs('floatFieldGrp', 'adjustableColumn2')
    def adjustableColumn2(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'adjustableColumn2', val)

    @_f.addMelDocs('floatFieldGrp', 'adjustableColumn3')
    def adjustableColumn3(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'adjustableColumn3', val)

    @_f.addMelDocs('floatFieldGrp', 'adjustableColumn4')
    def adjustableColumn4(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'adjustableColumn4', val)

    @_f.addMelDocs('floatFieldGrp', 'adjustableColumn5')
    def adjustableColumn5(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'adjustableColumn5', val)

    @_f.addMelDocs('floatFieldGrp', 'adjustableColumn6')
    def adjustableColumn6(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'adjustableColumn6', val)

    @_f.addMelDocs('floatFieldGrp', 'changeCommand')
    def changeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'changeCommand', val)

    @_f.addMelDocs('floatFieldGrp', 'columnAlign')
    def columnAlign(self, val=True, **kwargs):
        # type: (Tuple[int, str] | List[Tuple[int, str]], **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'columnAlign', val)

    @_f.addMelDocs('floatFieldGrp', 'columnAlign2')
    def columnAlign2(self, val=True, **kwargs):
        # type: (Tuple[str, str], **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'columnAlign2', val)

    @_f.addMelDocs('floatFieldGrp', 'columnAlign3')
    def columnAlign3(self, val=True, **kwargs):
        # type: (Tuple[str, str, str], **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'columnAlign3', val)

    @_f.addMelDocs('floatFieldGrp', 'columnAlign4')
    def columnAlign4(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'columnAlign4', val)

    @_f.addMelDocs('floatFieldGrp', 'columnAlign5')
    def columnAlign5(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'columnAlign5', val)

    @_f.addMelDocs('floatFieldGrp', 'columnAlign6')
    def columnAlign6(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'columnAlign6', val)

    @_f.addMelDocs('floatFieldGrp', 'columnAttach')
    def columnAttach(self, val=True, **kwargs):
        # type: (Tuple[int, str, int] | List[Tuple[int, str, int]], **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'columnAttach', val)

    @_f.addMelDocs('floatFieldGrp', 'columnAttach2')
    def columnAttach2(self, val=True, **kwargs):
        # type: (Tuple[str, str], **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'columnAttach2', val)

    @_f.addMelDocs('floatFieldGrp', 'columnAttach3')
    def columnAttach3(self, val=True, **kwargs):
        # type: (Tuple[str, str, str], **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'columnAttach3', val)

    @_f.addMelDocs('floatFieldGrp', 'columnAttach4')
    def columnAttach4(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'columnAttach4', val)

    @_f.addMelDocs('floatFieldGrp', 'columnAttach5')
    def columnAttach5(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'columnAttach5', val)

    @_f.addMelDocs('floatFieldGrp', 'columnAttach6')
    def columnAttach6(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'columnAttach6', val)

    @_f.addMelDocs('floatFieldGrp', 'columnOffset2')
    def columnOffset2(self, val=True, **kwargs):
        # type: (Tuple[int, int], **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'columnOffset2', val)

    @_f.addMelDocs('floatFieldGrp', 'columnOffset3')
    def columnOffset3(self, val=True, **kwargs):
        # type: (Tuple[int, int, int], **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'columnOffset3', val)

    @_f.addMelDocs('floatFieldGrp', 'columnOffset4')
    def columnOffset4(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'columnOffset4', val)

    @_f.addMelDocs('floatFieldGrp', 'columnOffset5')
    def columnOffset5(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'columnOffset5', val)

    @_f.addMelDocs('floatFieldGrp', 'columnOffset6')
    def columnOffset6(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'columnOffset6', val)

    @_f.addMelDocs('floatFieldGrp', 'columnWidth')
    def columnWidth(self, val=True, **kwargs):
        # type: (Tuple[int, int] | List[Tuple[int, int]], **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'columnWidth', val)

    @_f.addMelDocs('floatFieldGrp', 'columnWidth1')
    def columnWidth1(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'columnWidth1', val)

    @_f.addMelDocs('floatFieldGrp', 'columnWidth2')
    def columnWidth2(self, val=True, **kwargs):
        # type: (Tuple[int, int], **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'columnWidth2', val)

    @_f.addMelDocs('floatFieldGrp', 'columnWidth3')
    def columnWidth3(self, val=True, **kwargs):
        # type: (Tuple[int, int, int], **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'columnWidth3', val)

    @_f.addMelDocs('floatFieldGrp', 'columnWidth4')
    def columnWidth4(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'columnWidth4', val)

    @_f.addMelDocs('floatFieldGrp', 'columnWidth5')
    def columnWidth5(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'columnWidth5', val)

    @_f.addMelDocs('floatFieldGrp', 'columnWidth6')
    def columnWidth6(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'columnWidth6', val)

    @_f.addMelDocs('floatFieldGrp', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'dragCallback', val)

    @_f.addMelDocs('floatFieldGrp', 'dragCommand')
    def dragCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'dragCommand', val)

    @_f.addMelDocs('floatFieldGrp', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'dropCallback', val)

    @_f.addMelDocs('floatFieldGrp', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.floatFieldGrp, kwargs, 'annotation')
        return res

    @_f.addMelDocs('floatFieldGrp', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.floatFieldGrp, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('floatFieldGrp', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.floatFieldGrp, kwargs, 'docTag')
        return res

    @_f.addMelDocs('floatFieldGrp', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatFieldGrp, kwargs, 'enable')
        return res

    @_f.addMelDocs('floatFieldGrp', 'enable1')
    def getEnable1(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatFieldGrp, kwargs, 'enable1')
        return res

    @_f.addMelDocs('floatFieldGrp', 'enable2')
    def getEnable2(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatFieldGrp, kwargs, 'enable2')
        return res

    @_f.addMelDocs('floatFieldGrp', 'enable3')
    def getEnable3(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatFieldGrp, kwargs, 'enable3')
        return res

    @_f.addMelDocs('floatFieldGrp', 'enable4')
    def getEnable4(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatFieldGrp, kwargs, 'enable4')
        return res

    @_f.addMelDocs('floatFieldGrp', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatFieldGrp, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('floatFieldGrp', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatFieldGrp, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('floatFieldGrp', 'extraLabel')
    def getExtraLabel(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.floatFieldGrp, kwargs, 'extraLabel')
        return res

    @_f.addMelDocs('floatFieldGrp', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.floatFieldGrp, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('floatFieldGrp', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.floatFieldGrp, kwargs, 'height')
        return res

    @_f.addMelDocs('floatFieldGrp', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.floatFieldGrp, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('floatFieldGrp', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.floatFieldGrp, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('floatFieldGrp', 'label')
    def getLabel(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.floatFieldGrp, kwargs, 'label')
        return res

    @_f.addMelDocs('floatFieldGrp', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatFieldGrp, kwargs, 'manage')
        return res

    @_f.addMelDocs('floatFieldGrp', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.floatFieldGrp, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('floatFieldGrp', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.floatFieldGrp, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('floatFieldGrp', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatFieldGrp, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('floatFieldGrp', 'step')
    def getStep(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.floatFieldGrp, kwargs, 'step')
        return res

    @_f.addMelDocs('floatFieldGrp', 'value')
    def getValue(self, **kwargs):
        # type: (...) -> Tuple[float, float, float, float]
        res = _f.asQuery(self, windows.floatFieldGrp, kwargs, 'value')
        return res

    @_f.addMelDocs('floatFieldGrp', 'value1')
    def getValue1(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.floatFieldGrp, kwargs, 'value1')
        return res

    @_f.addMelDocs('floatFieldGrp', 'value2')
    def getValue2(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.floatFieldGrp, kwargs, 'value2')
        return res

    @_f.addMelDocs('floatFieldGrp', 'value3')
    def getValue3(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.floatFieldGrp, kwargs, 'value3')
        return res

    @_f.addMelDocs('floatFieldGrp', 'value4')
    def getValue4(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.floatFieldGrp, kwargs, 'value4')
        return res

    @_f.addMelDocs('floatFieldGrp', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatFieldGrp, kwargs, 'visible')
        return res

    @_f.addMelDocs('floatFieldGrp', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.floatFieldGrp, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('floatFieldGrp', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.floatFieldGrp, kwargs, 'width')
        return res

    @_f.addMelDocs('floatFieldGrp', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'noBackground', val)

    @_f.addMelDocs('floatFieldGrp', 'precision')
    def precision(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'precision', val)

    @_f.addMelDocs('floatFieldGrp', 'rowAttach')
    def rowAttach(self, val=True, **kwargs):
        # type: (Tuple[int, str, int] | List[Tuple[int, str, int]], **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'rowAttach', val)

    @_f.addMelDocs('floatFieldGrp', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'annotation', val)

    @_f.addMelDocs('floatFieldGrp', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('floatFieldGrp', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'docTag', val)

    @_f.addMelDocs('floatFieldGrp', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'enable', val)

    @_f.addMelDocs('floatFieldGrp', 'enable1')
    def setEnable1(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'enable1', val)

    @_f.addMelDocs('floatFieldGrp', 'enable2')
    def setEnable2(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'enable2', val)

    @_f.addMelDocs('floatFieldGrp', 'enable3')
    def setEnable3(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'enable3', val)

    @_f.addMelDocs('floatFieldGrp', 'enable4')
    def setEnable4(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'enable4', val)

    @_f.addMelDocs('floatFieldGrp', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'enableBackground', val)

    @_f.addMelDocs('floatFieldGrp', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('floatFieldGrp', 'extraLabel')
    def setExtraLabel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'extraLabel', val)

    @_f.addMelDocs('floatFieldGrp', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'height', val)

    @_f.addMelDocs('floatFieldGrp', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'highlightColor', val)

    @_f.addMelDocs('floatFieldGrp', 'label')
    def setLabel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'label', val)

    @_f.addMelDocs('floatFieldGrp', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'manage', val)

    @_f.addMelDocs('floatFieldGrp', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'preventOverride', val)

    @_f.addMelDocs('floatFieldGrp', 'step')
    def setStep(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'step', val)

    @_f.addMelDocs('floatFieldGrp', 'value')
    def setValue(self, val=True, **kwargs):
        # type: (Tuple[float, float, float, float], **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'value', val)

    @_f.addMelDocs('floatFieldGrp', 'value1')
    def setValue1(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'value1', val)

    @_f.addMelDocs('floatFieldGrp', 'value2')
    def setValue2(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'value2', val)

    @_f.addMelDocs('floatFieldGrp', 'value3')
    def setValue3(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'value3', val)

    @_f.addMelDocs('floatFieldGrp', 'value4')
    def setValue4(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'value4', val)

    @_f.addMelDocs('floatFieldGrp', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'visible', val)

    @_f.addMelDocs('floatFieldGrp', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('floatFieldGrp', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'width', val)

    @_f.addMelDocs('floatFieldGrp', 'showTrailingZeros')
    def showTrailingZeros(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'showTrailingZeros', val)

    @_f.addMelDocs('floatFieldGrp', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.floatFieldGrp, kwargs, 'statusBarMessage', val)
# ------ Do not edit above this line --------


class TextFieldButtonGrp(Layout):
    pass
# ------ Do not edit below this line --------
    __melcmd__ = staticmethod(windows.textFieldButtonGrp)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'textFieldButtonGrp'
    __melui__ = 'textFieldButtonGrp'
    __slots__ = ()

    @_f.addMelDocs('textFieldButtonGrp', 'adjustableColumn')
    def adjustableColumn(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'adjustableColumn', val)

    @_f.addMelDocs('textFieldButtonGrp', 'adjustableColumn2')
    def adjustableColumn2(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'adjustableColumn2', val)

    @_f.addMelDocs('textFieldButtonGrp', 'adjustableColumn3')
    def adjustableColumn3(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'adjustableColumn3', val)

    @_f.addMelDocs('textFieldButtonGrp', 'adjustableColumn4')
    def adjustableColumn4(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'adjustableColumn4', val)

    @_f.addMelDocs('textFieldButtonGrp', 'adjustableColumn5')
    def adjustableColumn5(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'adjustableColumn5', val)

    @_f.addMelDocs('textFieldButtonGrp', 'adjustableColumn6')
    def adjustableColumn6(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'adjustableColumn6', val)

    @_f.addMelDocs('textFieldButtonGrp', 'buttonCommand')
    def buttonCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'buttonCommand', val)

    @_f.addMelDocs('textFieldButtonGrp', 'changeCommand')
    def changeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'changeCommand', val)

    @_f.addMelDocs('textFieldButtonGrp', 'columnAlign')
    def columnAlign(self, val=True, **kwargs):
        # type: (Tuple[int, str] | List[Tuple[int, str]], **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'columnAlign', val)

    @_f.addMelDocs('textFieldButtonGrp', 'columnAlign2')
    def columnAlign2(self, val=True, **kwargs):
        # type: (Tuple[str, str], **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'columnAlign2', val)

    @_f.addMelDocs('textFieldButtonGrp', 'columnAlign3')
    def columnAlign3(self, val=True, **kwargs):
        # type: (Tuple[str, str, str], **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'columnAlign3', val)

    @_f.addMelDocs('textFieldButtonGrp', 'columnAlign4')
    def columnAlign4(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'columnAlign4', val)

    @_f.addMelDocs('textFieldButtonGrp', 'columnAlign5')
    def columnAlign5(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'columnAlign5', val)

    @_f.addMelDocs('textFieldButtonGrp', 'columnAlign6')
    def columnAlign6(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'columnAlign6', val)

    @_f.addMelDocs('textFieldButtonGrp', 'columnAttach')
    def columnAttach(self, val=True, **kwargs):
        # type: (Tuple[int, str, int] | List[Tuple[int, str, int]], **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'columnAttach', val)

    @_f.addMelDocs('textFieldButtonGrp', 'columnAttach2')
    def columnAttach2(self, val=True, **kwargs):
        # type: (Tuple[str, str], **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'columnAttach2', val)

    @_f.addMelDocs('textFieldButtonGrp', 'columnAttach3')
    def columnAttach3(self, val=True, **kwargs):
        # type: (Tuple[str, str, str], **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'columnAttach3', val)

    @_f.addMelDocs('textFieldButtonGrp', 'columnAttach4')
    def columnAttach4(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'columnAttach4', val)

    @_f.addMelDocs('textFieldButtonGrp', 'columnAttach5')
    def columnAttach5(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'columnAttach5', val)

    @_f.addMelDocs('textFieldButtonGrp', 'columnAttach6')
    def columnAttach6(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'columnAttach6', val)

    @_f.addMelDocs('textFieldButtonGrp', 'columnOffset2')
    def columnOffset2(self, val=True, **kwargs):
        # type: (Tuple[int, int], **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'columnOffset2', val)

    @_f.addMelDocs('textFieldButtonGrp', 'columnOffset3')
    def columnOffset3(self, val=True, **kwargs):
        # type: (Tuple[int, int, int], **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'columnOffset3', val)

    @_f.addMelDocs('textFieldButtonGrp', 'columnOffset4')
    def columnOffset4(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'columnOffset4', val)

    @_f.addMelDocs('textFieldButtonGrp', 'columnOffset5')
    def columnOffset5(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'columnOffset5', val)

    @_f.addMelDocs('textFieldButtonGrp', 'columnOffset6')
    def columnOffset6(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'columnOffset6', val)

    @_f.addMelDocs('textFieldButtonGrp', 'columnWidth')
    def columnWidth(self, val=True, **kwargs):
        # type: (Tuple[int, int] | List[Tuple[int, int]], **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'columnWidth', val)

    @_f.addMelDocs('textFieldButtonGrp', 'columnWidth1')
    def columnWidth1(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'columnWidth1', val)

    @_f.addMelDocs('textFieldButtonGrp', 'columnWidth2')
    def columnWidth2(self, val=True, **kwargs):
        # type: (Tuple[int, int], **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'columnWidth2', val)

    @_f.addMelDocs('textFieldButtonGrp', 'columnWidth3')
    def columnWidth3(self, val=True, **kwargs):
        # type: (Tuple[int, int, int], **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'columnWidth3', val)

    @_f.addMelDocs('textFieldButtonGrp', 'columnWidth4')
    def columnWidth4(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'columnWidth4', val)

    @_f.addMelDocs('textFieldButtonGrp', 'columnWidth5')
    def columnWidth5(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'columnWidth5', val)

    @_f.addMelDocs('textFieldButtonGrp', 'columnWidth6')
    def columnWidth6(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'columnWidth6', val)

    @_f.addMelDocs('textFieldButtonGrp', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'dragCallback', val)

    @_f.addMelDocs('textFieldButtonGrp', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'dropCallback', val)

    @_f.addMelDocs('textFieldButtonGrp', 'forceChangeCommand')
    def forceChangeCommand(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'forceChangeCommand', val)

    @_f.addMelDocs('textFieldButtonGrp', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.textFieldButtonGrp, kwargs, 'annotation')
        return res

    @_f.addMelDocs('textFieldButtonGrp', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.textFieldButtonGrp, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('textFieldButtonGrp', 'buttonLabel')
    def getButtonLabel(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.textFieldButtonGrp, kwargs, 'buttonLabel')
        return res

    @_f.addMelDocs('textFieldButtonGrp', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.textFieldButtonGrp, kwargs, 'docTag')
        return res

    @_f.addMelDocs('textFieldButtonGrp', 'editable')
    def getEditable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.textFieldButtonGrp, kwargs, 'editable')
        return res

    @_f.addMelDocs('textFieldButtonGrp', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.textFieldButtonGrp, kwargs, 'enable')
        return res

    @_f.addMelDocs('textFieldButtonGrp', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.textFieldButtonGrp, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('textFieldButtonGrp', 'enableButton')
    def getEnableButton(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.textFieldButtonGrp, kwargs, 'enableButton')
        return res

    @_f.addMelDocs('textFieldButtonGrp', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.textFieldButtonGrp, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('textFieldButtonGrp', 'fileName')
    def getFileName(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.textFieldButtonGrp, kwargs, 'fileName')
        return res

    @_f.addMelDocs('textFieldButtonGrp', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.textFieldButtonGrp, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('textFieldButtonGrp', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.textFieldButtonGrp, kwargs, 'height')
        return res

    @_f.addMelDocs('textFieldButtonGrp', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.textFieldButtonGrp, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('textFieldButtonGrp', 'insertionPosition')
    def getInsertionPosition(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.textFieldButtonGrp, kwargs, 'insertionPosition')
        return res

    @_f.addMelDocs('textFieldButtonGrp', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.textFieldButtonGrp, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('textFieldButtonGrp', 'label')
    def getLabel(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.textFieldButtonGrp, kwargs, 'label')
        return res

    @_f.addMelDocs('textFieldButtonGrp', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.textFieldButtonGrp, kwargs, 'manage')
        return res

    @_f.addMelDocs('textFieldButtonGrp', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.textFieldButtonGrp, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('textFieldButtonGrp', 'placeholderText')
    def getPlaceholderText(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.textFieldButtonGrp, kwargs, 'placeholderText')
        return res

    @_f.addMelDocs('textFieldButtonGrp', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.textFieldButtonGrp, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('textFieldButtonGrp', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.textFieldButtonGrp, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('textFieldButtonGrp', 'text')
    def getText(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.textFieldButtonGrp, kwargs, 'text')
        return res

    @_f.addMelDocs('textFieldButtonGrp', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.textFieldButtonGrp, kwargs, 'visible')
        return res

    @_f.addMelDocs('textFieldButtonGrp', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.textFieldButtonGrp, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('textFieldButtonGrp', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.textFieldButtonGrp, kwargs, 'width')
        return res

    @_f.addMelDocs('textFieldButtonGrp', 'insertText')
    def insertText(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'insertText', val)

    @_f.addMelDocs('textFieldButtonGrp', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'noBackground', val)

    @_f.addMelDocs('textFieldButtonGrp', 'rowAttach')
    def rowAttach(self, val=True, **kwargs):
        # type: (Tuple[int, str, int] | List[Tuple[int, str, int]], **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'rowAttach', val)

    @_f.addMelDocs('textFieldButtonGrp', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'annotation', val)

    @_f.addMelDocs('textFieldButtonGrp', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('textFieldButtonGrp', 'buttonLabel')
    def setButtonLabel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'buttonLabel', val)

    @_f.addMelDocs('textFieldButtonGrp', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'docTag', val)

    @_f.addMelDocs('textFieldButtonGrp', 'editable')
    def setEditable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'editable', val)

    @_f.addMelDocs('textFieldButtonGrp', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'enable', val)

    @_f.addMelDocs('textFieldButtonGrp', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'enableBackground', val)

    @_f.addMelDocs('textFieldButtonGrp', 'enableButton')
    def setEnableButton(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'enableButton', val)

    @_f.addMelDocs('textFieldButtonGrp', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('textFieldButtonGrp', 'fileName')
    def setFileName(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'fileName', val)

    @_f.addMelDocs('textFieldButtonGrp', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'height', val)

    @_f.addMelDocs('textFieldButtonGrp', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'highlightColor', val)

    @_f.addMelDocs('textFieldButtonGrp', 'insertionPosition')
    def setInsertionPosition(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'insertionPosition', val)

    @_f.addMelDocs('textFieldButtonGrp', 'label')
    def setLabel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'label', val)

    @_f.addMelDocs('textFieldButtonGrp', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'manage', val)

    @_f.addMelDocs('textFieldButtonGrp', 'placeholderText')
    def setPlaceholderText(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'placeholderText', val)

    @_f.addMelDocs('textFieldButtonGrp', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'preventOverride', val)

    @_f.addMelDocs('textFieldButtonGrp', 'text')
    def setText(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'text', val)

    @_f.addMelDocs('textFieldButtonGrp', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'visible', val)

    @_f.addMelDocs('textFieldButtonGrp', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('textFieldButtonGrp', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'width', val)

    @_f.addMelDocs('textFieldButtonGrp', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'statusBarMessage', val)

    @_f.addMelDocs('textFieldButtonGrp', 'textChangedCommand')
    def textChangedCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.textFieldButtonGrp, kwargs, 'textChangedCommand', val)
# ------ Do not edit above this line --------


class ProgressBar(PyUI):
    pass
# ------ Do not edit below this line --------
    __melcmd__ = staticmethod(windows.progressBar)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'progressBar'
    __melui__ = 'progressBar'
    __slots__ = ()

    @_f.addMelDocs('progressBar', 'beginProgress')
    def beginProgress(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.progressBar, kwargs, 'beginProgress', val)

    @_f.addMelDocs('progressBar', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.progressBar, kwargs, 'dragCallback', val)

    @_f.addMelDocs('progressBar', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.progressBar, kwargs, 'dropCallback', val)

    @_f.addMelDocs('progressBar', 'endProgress')
    def endProgress(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.progressBar, kwargs, 'endProgress', val)

    @_f.addMelDocs('progressBar', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.progressBar, kwargs, 'annotation')
        return res

    @_f.addMelDocs('progressBar', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.progressBar, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('progressBar', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.progressBar, kwargs, 'docTag')
        return res

    @_f.addMelDocs('progressBar', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.progressBar, kwargs, 'enable')
        return res

    @_f.addMelDocs('progressBar', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.progressBar, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('progressBar', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.progressBar, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('progressBar', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.progressBar, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('progressBar', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.progressBar, kwargs, 'height')
        return res

    @_f.addMelDocs('progressBar', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.progressBar, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('progressBar', 'isCancelled')
    def getIsCancelled(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.progressBar, kwargs, 'isCancelled')
        return res

    @_f.addMelDocs('progressBar', 'isInterruptable')
    def getIsInterruptable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.progressBar, kwargs, 'isInterruptable')
        return res

    @_f.addMelDocs('progressBar', 'isMainProgressBar')
    def getIsMainProgressBar(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.progressBar, kwargs, 'isMainProgressBar')
        return res

    @_f.addMelDocs('progressBar', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.progressBar, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('progressBar', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.progressBar, kwargs, 'manage')
        return res

    @_f.addMelDocs('progressBar', 'maxValue')
    def getMaxValue(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.progressBar, kwargs, 'maxValue')
        return res

    @_f.addMelDocs('progressBar', 'minValue')
    def getMinValue(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.progressBar, kwargs, 'minValue')
        return res

    @_f.addMelDocs('progressBar', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.progressBar, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('progressBar', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.progressBar, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('progressBar', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.progressBar, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('progressBar', 'progress')
    def getProgress(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.progressBar, kwargs, 'progress')
        return res

    @_f.addMelDocs('progressBar', 'status')
    def getStatus(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.progressBar, kwargs, 'status')
        return res

    @_f.addMelDocs('progressBar', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.progressBar, kwargs, 'visible')
        return res

    @_f.addMelDocs('progressBar', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.progressBar, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('progressBar', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.progressBar, kwargs, 'width')
        return res

    @_f.addMelDocs('progressBar', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.progressBar, kwargs, 'noBackground', val)

    @_f.addMelDocs('progressBar', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.progressBar, kwargs, 'annotation', val)

    @_f.addMelDocs('progressBar', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.progressBar, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('progressBar', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.progressBar, kwargs, 'docTag', val)

    @_f.addMelDocs('progressBar', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.progressBar, kwargs, 'enable', val)

    @_f.addMelDocs('progressBar', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.progressBar, kwargs, 'enableBackground', val)

    @_f.addMelDocs('progressBar', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.progressBar, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('progressBar', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.progressBar, kwargs, 'height', val)

    @_f.addMelDocs('progressBar', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.progressBar, kwargs, 'highlightColor', val)

    @_f.addMelDocs('progressBar', 'isInterruptable')
    def setIsInterruptable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.progressBar, kwargs, 'isInterruptable', val)

    @_f.addMelDocs('progressBar', 'isMainProgressBar')
    def setIsMainProgressBar(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.progressBar, kwargs, 'isMainProgressBar', val)

    @_f.addMelDocs('progressBar', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.progressBar, kwargs, 'manage', val)

    @_f.addMelDocs('progressBar', 'maxValue')
    def setMaxValue(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.progressBar, kwargs, 'maxValue', val)

    @_f.addMelDocs('progressBar', 'minValue')
    def setMinValue(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.progressBar, kwargs, 'minValue', val)

    @_f.addMelDocs('progressBar', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.progressBar, kwargs, 'preventOverride', val)

    @_f.addMelDocs('progressBar', 'progress')
    def setProgress(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.progressBar, kwargs, 'progress', val)

    @_f.addMelDocs('progressBar', 'status')
    def setStatus(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.progressBar, kwargs, 'status', val)

    @_f.addMelDocs('progressBar', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.progressBar, kwargs, 'visible', val)

    @_f.addMelDocs('progressBar', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.progressBar, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('progressBar', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.progressBar, kwargs, 'width', val)

    @_f.addMelDocs('progressBar', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.progressBar, kwargs, 'statusBarMessage', val)

    @_f.addMelDocs('progressBar', 'step')
    def step(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.progressBar, kwargs, 'step', val)
# ------ Do not edit above this line --------


class MainProgressBar(ProgressBar):

    '''Context manager for main progress bar

    If an exception occur after beginProgress() but before endProgress() maya
    gui becomes unresponsive. Use this class to escape this behavior.

    Here's an example::

        with MainProgressBar(0,20,True) as bar:
            bar.setStatus('Calculating...')
            for i in range(0,20):
                bar.setProgress(i)
                if bar.getIsCancelled():
                    break
    '''
    def __new__(cls, minValue=0, maxValue=100, interruptable=True):
        # type: (int, int, bool) -> MainProgressBar
        """

        Parameters
        ----------
        minValue : int
            Minimum or startingvalue of progress indicatior. If the progress
            value is less than the minValue, the progress value will be set
            to the minimum.  Default value is 0
        maxValue : int
            The maximum or endingvalue of the progress indicator. If the
            progress value is greater than the maxValue, the progress value
            will be set to the maximum. Default value is 100.
        interruptable : bool
            Set to True if the isCancelled flag should respond to attempts to
            cancel the operation. Setting this to true will put make the help
            line display message to the user indicating that they can cancel
            the operation.
        """
        from .language import melGlobals
        bar = ProgressBar.__new__(
            cls, melGlobals['gMainProgressBar'], create=False)
        bar.setMinValue(minValue)
        bar.setMaxValue(maxValue)
        bar.setIsInterruptable(interruptable)
        return bar

    def __enter__(self):
        self.beginProgress()
        return self

    def __exit__(self, *args):
        self.endProgress()


class VectorFieldGrp(FloatFieldGrp):

    def __new__(cls, name=None, create=False, *args, **kwargs):
        if create:
            kwargs.pop('nf', None)
            kwargs['numberOfFields'] = 3
            name = cmds.floatFieldGrp(name, *args, **kwargs)

        return FloatFieldGrp.__new__(cls, name, create=False, *args, **kwargs)

    def getVector(self):
        import pymel.core.datatypes as datatypes
        x = cmds.floatFieldGrp(self, q=1, v1=True)
        y = cmds.floatFieldGrp(self, q=1, v2=True)
        z = cmds.floatFieldGrp(self, q=1, v3=True)
        return datatypes.Vector([x, y, z])

    def setVector(self, vec):
        cmds.floatFieldGrp(self, e=1, v1=vec[0], v2=vec[1], v3=vec[2])


class PathButtonGrp(TextFieldButtonGrp):
    PROMPT_FUNCTION = 'promptForPath'

    def __new__(cls, name=None, create=False, *args, **kwargs):

        if create:
            from pymel.core import windows

            kwargs.pop('bl', None)
            kwargs['buttonLabel'] = 'Browse'
            kwargs.pop('bc', None)
            kwargs.pop('buttonCommand', None)

            name = cmds.textFieldButtonGrp(name, *args, **kwargs)

            promptFunction = getattr(windows, cls.PROMPT_FUNCTION)

            def setPathCB(name):
                f = promptFunction()
                if f:
                    cmds.textFieldButtonGrp(name, e=1, text=f, forceChangeCommand=True)

            from pymel.core import windows
            cb = windows.Callback(setPathCB, name)
            cmds.textFieldButtonGrp(name, e=1, buttonCommand=cb)

        return super(PathButtonGrp, cls).__new__(cls, name, create=False, *args, **kwargs)

    def setPath(self, path, **kwargs):
        kwargs['forceChangeCommand'] = kwargs.pop('fcc', kwargs.pop('forceChangeCommand', True))
        self.setText(path, **kwargs)

    def getPath(self):
        from pymel.core import system
        return system.Path(self.getText())


class FolderButtonGrp(PathButtonGrp):
    PROMPT_FUNCTION = 'promptForFolder'


# most of the keys here are names that are only used in certain circumstances
_uiTypesToCommands = {
    'radioCluster': 'radioCollection',
    'rowGroupLayout': 'rowLayout',
    'TcolorIndexSlider': 'rowLayout',
    'TcolorSlider': 'rowLayout',
    'floatingWindow': 'window',
    'field': 'textField',
    'staticText': 'text'
}

# dynModule._lazyModule_update()


# ------ Do not edit below this line --------
class AnimCurveEditor(PyUI):
    __melcmd__ = staticmethod(animation.animCurveEditor)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'animCurveEditor'
    __melui__ = 'animCurveEditor'
    __slots__ = ()

    @_f.addMelDocs('animCurveEditor', 'denormalizeCurvesCommand')
    def denormalizeCurvesCommand(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'denormalizeCurvesCommand', val)

    @_f.addMelDocs('animCurveEditor', 'displayActiveKeyTangents')
    def displayActiveKeyTangents(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'displayActiveKeyTangents', val)

    @_f.addMelDocs('animCurveEditor', 'displayActiveKeys')
    def displayActiveKeys(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'displayActiveKeys', val)

    @_f.addMelDocs('animCurveEditor', 'displayInfinities')
    def displayInfinities(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'displayInfinities', val)

    @_f.addMelDocs('animCurveEditor', 'displayKeys')
    def displayKeys(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'displayKeys', val)

    @_f.addMelDocs('animCurveEditor', 'displayTangents')
    def displayTangents(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'displayTangents', val)

    @_f.addMelDocs('animCurveEditor', 'displayValues')
    def displayValues(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'displayValues', val)

    @_f.addMelDocs('animCurveEditor', 'areCurvesSelected')
    def getAreCurvesSelected(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.animCurveEditor, kwargs, 'areCurvesSelected')
        return res

    @_f.addMelDocs('animCurveEditor', 'autoFit')
    def getAutoFit(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.animCurveEditor, kwargs, 'autoFit')
        return res

    @_f.addMelDocs('animCurveEditor', 'autoFitTime')
    def getAutoFitTime(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.animCurveEditor, kwargs, 'autoFitTime')
        return res

    @_f.addMelDocs('animCurveEditor', 'classicMode')
    def getClassicMode(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.animCurveEditor, kwargs, 'classicMode')
        return res

    @_f.addMelDocs('animCurveEditor', 'clipTime')
    def getClipTime(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.animCurveEditor, kwargs, 'clipTime')
        return res

    @_f.addMelDocs('animCurveEditor', 'constrainDrag')
    def getConstrainDrag(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, animation.animCurveEditor, kwargs, 'constrainDrag')
        return res

    @_f.addMelDocs('animCurveEditor', 'control')
    def getControl(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.animCurveEditor, kwargs, 'control')
        return res

    @_f.addMelDocs('animCurveEditor', 'curvesShown')
    def getCurvesShown(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.animCurveEditor, kwargs, 'curvesShown')
        return res

    @_f.addMelDocs('animCurveEditor', 'curvesShownForceUpdate')
    def getCurvesShownForceUpdate(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.animCurveEditor, kwargs, 'curvesShownForceUpdate')
        return res

    @_f.addMelDocs('animCurveEditor', 'displayNormalized')
    def getDisplayNormalized(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.animCurveEditor, kwargs, 'displayNormalized')
        return res

    @_f.addMelDocs('animCurveEditor', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.animCurveEditor, kwargs, 'docTag')
        return res

    @_f.addMelDocs('animCurveEditor', 'filter')
    def getFilter(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.animCurveEditor, kwargs, 'filter')
        return res

    @_f.addMelDocs('animCurveEditor', 'forceMainConnection')
    def getForceMainConnection(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.animCurveEditor, kwargs, 'forceMainConnection')
        return res

    @_f.addMelDocs('animCurveEditor', 'highlightAffectedCurves')
    def getHighlightAffectedCurves(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.animCurveEditor, kwargs, 'highlightAffectedCurves')
        return res

    @_f.addMelDocs('animCurveEditor', 'highlightConnection')
    def getHighlightConnection(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.animCurveEditor, kwargs, 'highlightConnection')
        return res

    @_f.addMelDocs('animCurveEditor', 'keyMinScale')
    def getKeyMinScale(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, animation.animCurveEditor, kwargs, 'keyMinScale')
        return res

    @_f.addMelDocs('animCurveEditor', 'keyScale')
    def getKeyScale(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, animation.animCurveEditor, kwargs, 'keyScale')
        return res

    @_f.addMelDocs('animCurveEditor', 'keyingTime')
    def getKeyingTime(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.animCurveEditor, kwargs, 'keyingTime')
        return res

    @_f.addMelDocs('animCurveEditor', 'lockPlayRangeShades')
    def getLockPlayRangeShades(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.animCurveEditor, kwargs, 'lockPlayRangeShades')
        return res

    @_f.addMelDocs('animCurveEditor', 'mainListConnection')
    def getMainListConnection(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.animCurveEditor, kwargs, 'mainListConnection')
        return res

    @_f.addMelDocs('animCurveEditor', 'outliner')
    def getOutliner(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.animCurveEditor, kwargs, 'outliner')
        return res

    @_f.addMelDocs('animCurveEditor', 'panel')
    def getPanel(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.animCurveEditor, kwargs, 'panel')
        return res

    @_f.addMelDocs('animCurveEditor', 'preSelectionHighlight')
    def getPreSelectionHighlight(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.animCurveEditor, kwargs, 'preSelectionHighlight')
        return res

    @_f.addMelDocs('animCurveEditor', 'resultSamples')
    def getResultSamples(self, **kwargs):
        # type: (...) -> int | float
        res = _f.asQuery(self, animation.animCurveEditor, kwargs, 'resultSamples')
        return res

    @_f.addMelDocs('animCurveEditor', 'resultScreenSamples')
    def getResultScreenSamples(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, animation.animCurveEditor, kwargs, 'resultScreenSamples')
        return res

    @_f.addMelDocs('animCurveEditor', 'resultUpdate')
    def getResultUpdate(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.animCurveEditor, kwargs, 'resultUpdate')
        return res

    @_f.addMelDocs('animCurveEditor', 'selectionConnection')
    def getSelectionConnection(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.animCurveEditor, kwargs, 'selectionConnection')
        return res

    @_f.addMelDocs('animCurveEditor', 'showActiveCurveNames')
    def getShowActiveCurveNames(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.animCurveEditor, kwargs, 'showActiveCurveNames')
        return res

    @_f.addMelDocs('animCurveEditor', 'showBufferCurves')
    def getShowBufferCurves(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.animCurveEditor, kwargs, 'showBufferCurves')
        return res

    @_f.addMelDocs('animCurveEditor', 'showCurveNames')
    def getShowCurveNames(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.animCurveEditor, kwargs, 'showCurveNames')
        return res

    @_f.addMelDocs('animCurveEditor', 'showPlayRangeShades')
    def getShowPlayRangeShades(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.animCurveEditor, kwargs, 'showPlayRangeShades')
        return res

    @_f.addMelDocs('animCurveEditor', 'showResults')
    def getShowResults(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.animCurveEditor, kwargs, 'showResults')
        return res

    @_f.addMelDocs('animCurveEditor', 'showUpstreamCurves')
    def getShowUpstreamCurves(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.animCurveEditor, kwargs, 'showUpstreamCurves')
        return res

    @_f.addMelDocs('animCurveEditor', 'simpleKeyView')
    def getSimpleKeyView(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.animCurveEditor, kwargs, 'simpleKeyView')
        return res

    @_f.addMelDocs('animCurveEditor', 'smoothness')
    def getSmoothness(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.animCurveEditor, kwargs, 'smoothness')
        return res

    @_f.addMelDocs('animCurveEditor', 'snapTime')
    def getSnapTime(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.animCurveEditor, kwargs, 'snapTime')
        return res

    @_f.addMelDocs('animCurveEditor', 'snapValue')
    def getSnapValue(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.animCurveEditor, kwargs, 'snapValue')
        return res

    @_f.addMelDocs('animCurveEditor', 'stackedCurves')
    def getStackedCurves(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.animCurveEditor, kwargs, 'stackedCurves')
        return res

    @_f.addMelDocs('animCurveEditor', 'stackedCurvesMax')
    def getStackedCurvesMax(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, animation.animCurveEditor, kwargs, 'stackedCurvesMax')
        return res

    @_f.addMelDocs('animCurveEditor', 'stackedCurvesMin')
    def getStackedCurvesMin(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, animation.animCurveEditor, kwargs, 'stackedCurvesMin')
        return res

    @_f.addMelDocs('animCurveEditor', 'stackedCurvesSpace')
    def getStackedCurvesSpace(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, animation.animCurveEditor, kwargs, 'stackedCurvesSpace')
        return res

    @_f.addMelDocs('animCurveEditor', 'stateString')
    def getStateString(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.animCurveEditor, kwargs, 'stateString')
        return res

    @_f.addMelDocs('animCurveEditor', 'timelinePositionTop')
    def getTimelinePositionTop(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.animCurveEditor, kwargs, 'timelinePositionTop')
        return res

    @_f.addMelDocs('animCurveEditor', 'lockMainConnection')
    def lockMainConnection(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'lockMainConnection', val)

    @_f.addMelDocs('animCurveEditor', 'lookAt')
    def lookAt(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'lookAt', val)

    @_f.addMelDocs('animCurveEditor', 'normalizeCurvesCommand')
    def normalizeCurvesCommand(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'normalizeCurvesCommand', val)

    @_f.addMelDocs('animCurveEditor', 'renormalizeCurves')
    def renormalizeCurves(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'renormalizeCurves', val)

    @_f.addMelDocs('animCurveEditor', 'autoFit')
    def setAutoFit(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'autoFit', val)

    @_f.addMelDocs('animCurveEditor', 'autoFitTime')
    def setAutoFitTime(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'autoFitTime', val)

    @_f.addMelDocs('animCurveEditor', 'classicMode')
    def setClassicMode(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'classicMode', val)

    @_f.addMelDocs('animCurveEditor', 'clipTime')
    def setClipTime(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'clipTime', val)

    @_f.addMelDocs('animCurveEditor', 'constrainDrag')
    def setConstrainDrag(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'constrainDrag', val)

    @_f.addMelDocs('animCurveEditor', 'displayNormalized')
    def setDisplayNormalized(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'displayNormalized', val)

    @_f.addMelDocs('animCurveEditor', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'docTag', val)

    @_f.addMelDocs('animCurveEditor', 'filter')
    def setFilter(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'filter', val)

    @_f.addMelDocs('animCurveEditor', 'forceMainConnection')
    def setForceMainConnection(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'forceMainConnection', val)

    @_f.addMelDocs('animCurveEditor', 'highlightAffectedCurves')
    def setHighlightAffectedCurves(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'highlightAffectedCurves', val)

    @_f.addMelDocs('animCurveEditor', 'highlightConnection')
    def setHighlightConnection(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'highlightConnection', val)

    @_f.addMelDocs('animCurveEditor', 'keyMinScale')
    def setKeyMinScale(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'keyMinScale', val)

    @_f.addMelDocs('animCurveEditor', 'keyScale')
    def setKeyScale(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'keyScale', val)

    @_f.addMelDocs('animCurveEditor', 'lockPlayRangeShades')
    def setLockPlayRangeShades(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'lockPlayRangeShades', val)

    @_f.addMelDocs('animCurveEditor', 'mainListConnection')
    def setMainListConnection(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'mainListConnection', val)

    @_f.addMelDocs('animCurveEditor', 'outliner')
    def setOutliner(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'outliner', val)

    @_f.addMelDocs('animCurveEditor', 'parent')
    def setParent(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'parent', val)

    @_f.addMelDocs('animCurveEditor', 'preSelectionHighlight')
    def setPreSelectionHighlight(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'preSelectionHighlight', val)

    @_f.addMelDocs('animCurveEditor', 'resultSamples')
    def setResultSamples(self, val=True, **kwargs):
        # type: (int | float, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'resultSamples', val)

    @_f.addMelDocs('animCurveEditor', 'resultScreenSamples')
    def setResultScreenSamples(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'resultScreenSamples', val)

    @_f.addMelDocs('animCurveEditor', 'resultUpdate')
    def setResultUpdate(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'resultUpdate', val)

    @_f.addMelDocs('animCurveEditor', 'selectionConnection')
    def setSelectionConnection(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'selectionConnection', val)

    @_f.addMelDocs('animCurveEditor', 'showActiveCurveNames')
    def setShowActiveCurveNames(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'showActiveCurveNames', val)

    @_f.addMelDocs('animCurveEditor', 'showBufferCurves')
    def setShowBufferCurves(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'showBufferCurves', val)

    @_f.addMelDocs('animCurveEditor', 'showCurveNames')
    def setShowCurveNames(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'showCurveNames', val)

    @_f.addMelDocs('animCurveEditor', 'showPlayRangeShades')
    def setShowPlayRangeShades(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'showPlayRangeShades', val)

    @_f.addMelDocs('animCurveEditor', 'showResults')
    def setShowResults(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'showResults', val)

    @_f.addMelDocs('animCurveEditor', 'showUpstreamCurves')
    def setShowUpstreamCurves(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'showUpstreamCurves', val)

    @_f.addMelDocs('animCurveEditor', 'simpleKeyView')
    def setSimpleKeyView(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'simpleKeyView', val)

    @_f.addMelDocs('animCurveEditor', 'smoothness')
    def setSmoothness(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'smoothness', val)

    @_f.addMelDocs('animCurveEditor', 'snapTime')
    def setSnapTime(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'snapTime', val)

    @_f.addMelDocs('animCurveEditor', 'snapValue')
    def setSnapValue(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'snapValue', val)

    @_f.addMelDocs('animCurveEditor', 'stackedCurves')
    def setStackedCurves(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'stackedCurves', val)

    @_f.addMelDocs('animCurveEditor', 'stackedCurvesMax')
    def setStackedCurvesMax(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'stackedCurvesMax', val)

    @_f.addMelDocs('animCurveEditor', 'stackedCurvesMin')
    def setStackedCurvesMin(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'stackedCurvesMin', val)

    @_f.addMelDocs('animCurveEditor', 'stackedCurvesSpace')
    def setStackedCurvesSpace(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'stackedCurvesSpace', val)

    @_f.addMelDocs('animCurveEditor', 'timelinePositionTop')
    def setTimelinePositionTop(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'timelinePositionTop', val)

    @_f.addMelDocs('animCurveEditor', 'unParent')
    def unParent(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'unParent', val)

    @_f.addMelDocs('animCurveEditor', 'unlockMainConnection')
    def unlockMainConnection(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'unlockMainConnection', val)

    @_f.addMelDocs('animCurveEditor', 'updateMainConnection')
    def updateMainConnection(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'updateMainConnection', val)

    @_f.addMelDocs('animCurveEditor', 'valueLinesToggle')
    def valueLinesToggle(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.animCurveEditor, kwargs, 'valueLinesToggle', val)


class AnimDisplay(PyUI):
    __melcmd__ = staticmethod(animation.animDisplay)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'animDisplay'
    __melui__ = 'animDisplay'
    __slots__ = ()

    @_f.addMelDocs('animDisplay', 'modelUpdate')
    def getModelUpdate(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.animDisplay, kwargs, 'modelUpdate')
        return res

    @_f.addMelDocs('animDisplay', 'refAnimCurvesEditable')
    def getRefAnimCurvesEditable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.animDisplay, kwargs, 'refAnimCurvesEditable')
        return res

    @_f.addMelDocs('animDisplay', 'timeCode')
    def getTimeCode(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.animDisplay, kwargs, 'timeCode')
        return res

    @_f.addMelDocs('animDisplay', 'timeCodeOffset')
    def getTimeCodeOffset(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.animDisplay, kwargs, 'timeCodeOffset')
        return res

    @_f.addMelDocs('animDisplay', 'modelUpdate')
    def setModelUpdate(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.animDisplay, kwargs, 'modelUpdate', val)

    @_f.addMelDocs('animDisplay', 'refAnimCurvesEditable')
    def setRefAnimCurvesEditable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.animDisplay, kwargs, 'refAnimCurvesEditable', val)

    @_f.addMelDocs('animDisplay', 'timeCode')
    def setTimeCode(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.animDisplay, kwargs, 'timeCode', val)

    @_f.addMelDocs('animDisplay', 'timeCodeOffset')
    def setTimeCodeOffset(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.animDisplay, kwargs, 'timeCodeOffset', val)


class AttrEnumOptionMenu(PyUI):
    __melcmd__ = staticmethod(windows.attrEnumOptionMenu)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'attrEnumOptionMenu'
    __melui__ = 'attrEnumOptionMenu'
    __slots__ = ()

    @_f.addMelDocs('attrEnumOptionMenu', 'attribute')
    def attribute(self, val=True, **kwargs):
        # type: (_general.PyNode | str, **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenu, kwargs, 'attribute', val)

    @_f.addMelDocs('attrEnumOptionMenu', 'changeCommand')
    def changeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenu, kwargs, 'changeCommand', val)

    @_f.addMelDocs('attrEnumOptionMenu', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenu, kwargs, 'dragCallback', val)

    @_f.addMelDocs('attrEnumOptionMenu', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenu, kwargs, 'dropCallback', val)

    @_f.addMelDocs('attrEnumOptionMenu', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.attrEnumOptionMenu, kwargs, 'annotation')
        return res

    @_f.addMelDocs('attrEnumOptionMenu', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.attrEnumOptionMenu, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('attrEnumOptionMenu', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.attrEnumOptionMenu, kwargs, 'docTag')
        return res

    @_f.addMelDocs('attrEnumOptionMenu', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.attrEnumOptionMenu, kwargs, 'enable')
        return res

    @_f.addMelDocs('attrEnumOptionMenu', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.attrEnumOptionMenu, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('attrEnumOptionMenu', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.attrEnumOptionMenu, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('attrEnumOptionMenu', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.attrEnumOptionMenu, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('attrEnumOptionMenu', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.attrEnumOptionMenu, kwargs, 'height')
        return res

    @_f.addMelDocs('attrEnumOptionMenu', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.attrEnumOptionMenu, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('attrEnumOptionMenu', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.attrEnumOptionMenu, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('attrEnumOptionMenu', 'label')
    def getLabel(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.attrEnumOptionMenu, kwargs, 'label')
        return res

    @_f.addMelDocs('attrEnumOptionMenu', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.attrEnumOptionMenu, kwargs, 'manage')
        return res

    @_f.addMelDocs('attrEnumOptionMenu', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.attrEnumOptionMenu, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('attrEnumOptionMenu', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.attrEnumOptionMenu, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('attrEnumOptionMenu', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.attrEnumOptionMenu, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('attrEnumOptionMenu', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.attrEnumOptionMenu, kwargs, 'visible')
        return res

    @_f.addMelDocs('attrEnumOptionMenu', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.attrEnumOptionMenu, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('attrEnumOptionMenu', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.attrEnumOptionMenu, kwargs, 'width')
        return res

    @_f.addMelDocs('attrEnumOptionMenu', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenu, kwargs, 'noBackground', val)

    @_f.addMelDocs('attrEnumOptionMenu', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenu, kwargs, 'annotation', val)

    @_f.addMelDocs('attrEnumOptionMenu', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenu, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('attrEnumOptionMenu', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenu, kwargs, 'docTag', val)

    @_f.addMelDocs('attrEnumOptionMenu', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenu, kwargs, 'enable', val)

    @_f.addMelDocs('attrEnumOptionMenu', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenu, kwargs, 'enableBackground', val)

    @_f.addMelDocs('attrEnumOptionMenu', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenu, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('attrEnumOptionMenu', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenu, kwargs, 'height', val)

    @_f.addMelDocs('attrEnumOptionMenu', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenu, kwargs, 'highlightColor', val)

    @_f.addMelDocs('attrEnumOptionMenu', 'label')
    def setLabel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenu, kwargs, 'label', val)

    @_f.addMelDocs('attrEnumOptionMenu', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenu, kwargs, 'manage', val)

    @_f.addMelDocs('attrEnumOptionMenu', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenu, kwargs, 'preventOverride', val)

    @_f.addMelDocs('attrEnumOptionMenu', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenu, kwargs, 'visible', val)

    @_f.addMelDocs('attrEnumOptionMenu', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenu, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('attrEnumOptionMenu', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenu, kwargs, 'width', val)

    @_f.addMelDocs('attrEnumOptionMenu', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenu, kwargs, 'statusBarMessage', val)


class AttributeMenu(PyUI):
    __melcmd__ = staticmethod(windows.attributeMenu)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'attributeMenu'
    __melui__ = 'attributeMenu'
    __slots__ = ()


class BlendShapeEditor(PyUI):
    __melcmd__ = staticmethod(animation.blendShapeEditor)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'blendShapeEditor'
    __melui__ = 'blendShapeEditor'
    __slots__ = ()

    @_f.addMelDocs('blendShapeEditor', 'control')
    def getControl(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.blendShapeEditor, kwargs, 'control')
        return res

    @_f.addMelDocs('blendShapeEditor', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.blendShapeEditor, kwargs, 'docTag')
        return res

    @_f.addMelDocs('blendShapeEditor', 'filter')
    def getFilter(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.blendShapeEditor, kwargs, 'filter')
        return res

    @_f.addMelDocs('blendShapeEditor', 'forceMainConnection')
    def getForceMainConnection(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.blendShapeEditor, kwargs, 'forceMainConnection')
        return res

    @_f.addMelDocs('blendShapeEditor', 'highlightConnection')
    def getHighlightConnection(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.blendShapeEditor, kwargs, 'highlightConnection')
        return res

    @_f.addMelDocs('blendShapeEditor', 'mainListConnection')
    def getMainListConnection(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.blendShapeEditor, kwargs, 'mainListConnection')
        return res

    @_f.addMelDocs('blendShapeEditor', 'panel')
    def getPanel(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.blendShapeEditor, kwargs, 'panel')
        return res

    @_f.addMelDocs('blendShapeEditor', 'selectionConnection')
    def getSelectionConnection(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.blendShapeEditor, kwargs, 'selectionConnection')
        return res

    @_f.addMelDocs('blendShapeEditor', 'stateString')
    def getStateString(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.blendShapeEditor, kwargs, 'stateString')
        return res

    @_f.addMelDocs('blendShapeEditor', 'targetControlList')
    def getTargetControlList(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.blendShapeEditor, kwargs, 'targetControlList')
        return res

    @_f.addMelDocs('blendShapeEditor', 'targetList')
    def getTargetList(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.blendShapeEditor, kwargs, 'targetList')
        return res

    @_f.addMelDocs('blendShapeEditor', 'verticalSliders')
    def getVerticalSliders(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.blendShapeEditor, kwargs, 'verticalSliders')
        return res

    @_f.addMelDocs('blendShapeEditor', 'lockMainConnection')
    def lockMainConnection(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.blendShapeEditor, kwargs, 'lockMainConnection', val)

    @_f.addMelDocs('blendShapeEditor', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.blendShapeEditor, kwargs, 'docTag', val)

    @_f.addMelDocs('blendShapeEditor', 'filter')
    def setFilter(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.blendShapeEditor, kwargs, 'filter', val)

    @_f.addMelDocs('blendShapeEditor', 'forceMainConnection')
    def setForceMainConnection(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.blendShapeEditor, kwargs, 'forceMainConnection', val)

    @_f.addMelDocs('blendShapeEditor', 'highlightConnection')
    def setHighlightConnection(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.blendShapeEditor, kwargs, 'highlightConnection', val)

    @_f.addMelDocs('blendShapeEditor', 'mainListConnection')
    def setMainListConnection(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.blendShapeEditor, kwargs, 'mainListConnection', val)

    @_f.addMelDocs('blendShapeEditor', 'parent')
    def setParent(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.blendShapeEditor, kwargs, 'parent', val)

    @_f.addMelDocs('blendShapeEditor', 'selectionConnection')
    def setSelectionConnection(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.blendShapeEditor, kwargs, 'selectionConnection', val)

    @_f.addMelDocs('blendShapeEditor', 'verticalSliders')
    def setVerticalSliders(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.blendShapeEditor, kwargs, 'verticalSliders', val)

    @_f.addMelDocs('blendShapeEditor', 'unParent')
    def unParent(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.blendShapeEditor, kwargs, 'unParent', val)

    @_f.addMelDocs('blendShapeEditor', 'unlockMainConnection')
    def unlockMainConnection(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.blendShapeEditor, kwargs, 'unlockMainConnection', val)

    @_f.addMelDocs('blendShapeEditor', 'updateMainConnection')
    def updateMainConnection(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.blendShapeEditor, kwargs, 'updateMainConnection', val)


class Button(PyUI):
    __melcmd__ = staticmethod(windows.button)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'button'
    __melui__ = 'button'
    __slots__ = ()

    @_f.addMelDocs('button', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.button, kwargs, 'dragCallback', val)

    @_f.addMelDocs('button', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.button, kwargs, 'dropCallback', val)

    @_f.addMelDocs('button', 'actOnPress')
    def getActOnPress(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.button, kwargs, 'actOnPress')
        return res

    @_f.addMelDocs('button', 'actionIsSubstitute')
    def getActionIsSubstitute(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.button, kwargs, 'actionIsSubstitute')
        return res

    @_f.addMelDocs('button', 'align')
    def getAlign(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.button, kwargs, 'align')
        return res

    @_f.addMelDocs('button', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.button, kwargs, 'annotation')
        return res

    @_f.addMelDocs('button', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.button, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('button', 'command')
    def getCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.button, kwargs, 'command')
        return res

    @_f.addMelDocs('button', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.button, kwargs, 'docTag')
        return res

    @_f.addMelDocs('button', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.button, kwargs, 'enable')
        return res

    @_f.addMelDocs('button', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.button, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('button', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.button, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('button', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.button, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('button', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.button, kwargs, 'height')
        return res

    @_f.addMelDocs('button', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.button, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('button', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.button, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('button', 'label')
    def getLabel(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.button, kwargs, 'label')
        return res

    @_f.addMelDocs('button', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.button, kwargs, 'manage')
        return res

    @_f.addMelDocs('button', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.button, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('button', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.button, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('button', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.button, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('button', 'recomputeSize')
    def getRecomputeSize(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.button, kwargs, 'recomputeSize')
        return res

    @_f.addMelDocs('button', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.button, kwargs, 'visible')
        return res

    @_f.addMelDocs('button', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.button, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('button', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.button, kwargs, 'width')
        return res

    @_f.addMelDocs('button', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.button, kwargs, 'noBackground', val)

    @_f.addMelDocs('button', 'actOnPress')
    def setActOnPress(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.button, kwargs, 'actOnPress', val)

    @_f.addMelDocs('button', 'actionIsSubstitute')
    def setActionIsSubstitute(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.button, kwargs, 'actionIsSubstitute', val)

    @_f.addMelDocs('button', 'align')
    def setAlign(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.button, kwargs, 'align', val)

    @_f.addMelDocs('button', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.button, kwargs, 'annotation', val)

    @_f.addMelDocs('button', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.button, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('button', 'command')
    def setCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.button, kwargs, 'command', val)

    @_f.addMelDocs('button', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.button, kwargs, 'docTag', val)

    @_f.addMelDocs('button', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.button, kwargs, 'enable', val)

    @_f.addMelDocs('button', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.button, kwargs, 'enableBackground', val)

    @_f.addMelDocs('button', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.button, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('button', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.button, kwargs, 'height', val)

    @_f.addMelDocs('button', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.button, kwargs, 'highlightColor', val)

    @_f.addMelDocs('button', 'label')
    def setLabel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.button, kwargs, 'label', val)

    @_f.addMelDocs('button', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.button, kwargs, 'manage', val)

    @_f.addMelDocs('button', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.button, kwargs, 'preventOverride', val)

    @_f.addMelDocs('button', 'recomputeSize')
    def setRecomputeSize(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.button, kwargs, 'recomputeSize', val)

    @_f.addMelDocs('button', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.button, kwargs, 'visible', val)

    @_f.addMelDocs('button', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.button, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('button', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.button, kwargs, 'width', val)

    @_f.addMelDocs('button', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.button, kwargs, 'statusBarMessage', val)


class Canvas(PyUI):
    __melcmd__ = staticmethod(windows.canvas)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'canvas'
    __melui__ = 'canvas'
    __slots__ = ()

    @_f.addMelDocs('canvas', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.canvas, kwargs, 'dragCallback', val)

    @_f.addMelDocs('canvas', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.canvas, kwargs, 'dropCallback', val)

    @_f.addMelDocs('canvas', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.canvas, kwargs, 'annotation')
        return res

    @_f.addMelDocs('canvas', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.canvas, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('canvas', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.canvas, kwargs, 'docTag')
        return res

    @_f.addMelDocs('canvas', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.canvas, kwargs, 'enable')
        return res

    @_f.addMelDocs('canvas', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.canvas, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('canvas', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.canvas, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('canvas', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.canvas, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('canvas', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.canvas, kwargs, 'height')
        return res

    @_f.addMelDocs('canvas', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.canvas, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('canvas', 'hsvValue')
    def getHsvValue(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.canvas, kwargs, 'hsvValue')
        return res

    @_f.addMelDocs('canvas', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.canvas, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('canvas', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.canvas, kwargs, 'manage')
        return res

    @_f.addMelDocs('canvas', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.canvas, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('canvas', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.canvas, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('canvas', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.canvas, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('canvas', 'rgbValue')
    def getRgbValue(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.canvas, kwargs, 'rgbValue')
        return res

    @_f.addMelDocs('canvas', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.canvas, kwargs, 'visible')
        return res

    @_f.addMelDocs('canvas', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.canvas, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('canvas', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.canvas, kwargs, 'width')
        return res

    @_f.addMelDocs('canvas', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.canvas, kwargs, 'noBackground', val)

    @_f.addMelDocs('canvas', 'pressCommand')
    def pressCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.canvas, kwargs, 'pressCommand', val)

    @_f.addMelDocs('canvas', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.canvas, kwargs, 'annotation', val)

    @_f.addMelDocs('canvas', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.canvas, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('canvas', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.canvas, kwargs, 'docTag', val)

    @_f.addMelDocs('canvas', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.canvas, kwargs, 'enable', val)

    @_f.addMelDocs('canvas', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.canvas, kwargs, 'enableBackground', val)

    @_f.addMelDocs('canvas', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.canvas, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('canvas', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.canvas, kwargs, 'height', val)

    @_f.addMelDocs('canvas', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.canvas, kwargs, 'highlightColor', val)

    @_f.addMelDocs('canvas', 'hsvValue')
    def setHsvValue(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.canvas, kwargs, 'hsvValue', val)

    @_f.addMelDocs('canvas', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.canvas, kwargs, 'manage', val)

    @_f.addMelDocs('canvas', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.canvas, kwargs, 'preventOverride', val)

    @_f.addMelDocs('canvas', 'rgbValue')
    def setRgbValue(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.canvas, kwargs, 'rgbValue', val)

    @_f.addMelDocs('canvas', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.canvas, kwargs, 'visible', val)

    @_f.addMelDocs('canvas', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.canvas, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('canvas', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.canvas, kwargs, 'width', val)

    @_f.addMelDocs('canvas', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.canvas, kwargs, 'statusBarMessage', val)


class ChannelBox(PyUI):
    __melcmd__ = staticmethod(windows.channelBox)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'channelBox'
    __melui__ = 'channelBox'
    __slots__ = ()

    @_f.addMelDocs('channelBox', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.channelBox, kwargs, 'dragCallback', val)

    @_f.addMelDocs('channelBox', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.channelBox, kwargs, 'dropCallback', val)

    @_f.addMelDocs('channelBox', 'execute')
    def execute(self, val=True, **kwargs):
        # type: (Tuple[str, bool | int], **Any) -> None
        return _f.asEdit(self, windows.channelBox, kwargs, 'execute', val)

    @_f.addMelDocs('channelBox', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.channelBox, kwargs, 'annotation')
        return res

    @_f.addMelDocs('channelBox', 'attrBgColor')
    def getAttrBgColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.channelBox, kwargs, 'attrBgColor')
        return res

    @_f.addMelDocs('channelBox', 'attrColor')
    def getAttrColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.channelBox, kwargs, 'attrColor')
        return res

    @_f.addMelDocs('channelBox', 'attrFilter')
    def getAttrFilter(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.channelBox, kwargs, 'attrFilter')
        return res

    @_f.addMelDocs('channelBox', 'attrRegex')
    def getAttrRegex(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.channelBox, kwargs, 'attrRegex')
        return res

    @_f.addMelDocs('channelBox', 'attributeEditorMode')
    def getAttributeEditorMode(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.channelBox, kwargs, 'attributeEditorMode')
        return res

    @_f.addMelDocs('channelBox', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.channelBox, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('channelBox', 'containerAtTop')
    def getContainerAtTop(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.channelBox, kwargs, 'containerAtTop')
        return res

    @_f.addMelDocs('channelBox', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.channelBox, kwargs, 'docTag')
        return res

    @_f.addMelDocs('channelBox', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.channelBox, kwargs, 'enable')
        return res

    @_f.addMelDocs('channelBox', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.channelBox, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('channelBox', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.channelBox, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('channelBox', 'enableLabelSelection')
    def getEnableLabelSelection(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.channelBox, kwargs, 'enableLabelSelection')
        return res

    @_f.addMelDocs('channelBox', 'fieldWidth')
    def getFieldWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.channelBox, kwargs, 'fieldWidth')
        return res

    @_f.addMelDocs('channelBox', 'fixedAttrList')
    def getFixedAttrList(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.channelBox, kwargs, 'fixedAttrList')
        return res

    @_f.addMelDocs('channelBox', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.channelBox, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('channelBox', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.channelBox, kwargs, 'height')
        return res

    @_f.addMelDocs('channelBox', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.channelBox, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('channelBox', 'historyObjectList')
    def getHistoryObjectList(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.channelBox, kwargs, 'historyObjectList')
        return res

    @_f.addMelDocs('channelBox', 'hyperbolic')
    def getHyperbolic(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.channelBox, kwargs, 'hyperbolic')
        return res

    @_f.addMelDocs('channelBox', 'inputs')
    def getInputs(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.channelBox, kwargs, 'inputs')
        return res

    @_f.addMelDocs('channelBox', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.channelBox, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('channelBox', 'labelWidth')
    def getLabelWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.channelBox, kwargs, 'labelWidth')
        return res

    @_f.addMelDocs('channelBox', 'longNames')
    def getLongNames(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.channelBox, kwargs, 'longNames')
        return res

    @_f.addMelDocs('channelBox', 'mainListConnection')
    def getMainListConnection(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.channelBox, kwargs, 'mainListConnection')
        return res

    @_f.addMelDocs('channelBox', 'mainObjectList')
    def getMainObjectList(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.channelBox, kwargs, 'mainObjectList')
        return res

    @_f.addMelDocs('channelBox', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.channelBox, kwargs, 'manage')
        return res

    @_f.addMelDocs('channelBox', 'maxHeight')
    def getMaxHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.channelBox, kwargs, 'maxHeight')
        return res

    @_f.addMelDocs('channelBox', 'maxWidth')
    def getMaxWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.channelBox, kwargs, 'maxWidth')
        return res

    @_f.addMelDocs('channelBox', 'niceNames')
    def getNiceNames(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.channelBox, kwargs, 'niceNames')
        return res

    @_f.addMelDocs('channelBox', 'nodeRegex')
    def getNodeRegex(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.channelBox, kwargs, 'nodeRegex')
        return res

    @_f.addMelDocs('channelBox', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.channelBox, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('channelBox', 'outputObjectList')
    def getOutputObjectList(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.channelBox, kwargs, 'outputObjectList')
        return res

    @_f.addMelDocs('channelBox', 'outputs')
    def getOutputs(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.channelBox, kwargs, 'outputs')
        return res

    @_f.addMelDocs('channelBox', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.channelBox, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('channelBox', 'precision')
    def getPrecision(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.channelBox, kwargs, 'precision')
        return res

    @_f.addMelDocs('channelBox', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.channelBox, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('channelBox', 'selectedHistoryAttributes')
    def getSelectedHistoryAttributes(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.channelBox, kwargs, 'selectedHistoryAttributes')
        return res

    @_f.addMelDocs('channelBox', 'selectedMainAttributes')
    def getSelectedMainAttributes(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.channelBox, kwargs, 'selectedMainAttributes')
        return res

    @_f.addMelDocs('channelBox', 'selectedOutputAttributes')
    def getSelectedOutputAttributes(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.channelBox, kwargs, 'selectedOutputAttributes')
        return res

    @_f.addMelDocs('channelBox', 'selectedShapeAttributes')
    def getSelectedShapeAttributes(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.channelBox, kwargs, 'selectedShapeAttributes')
        return res

    @_f.addMelDocs('channelBox', 'shapeObjectList')
    def getShapeObjectList(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.channelBox, kwargs, 'shapeObjectList')
        return res

    @_f.addMelDocs('channelBox', 'shapes')
    def getShapes(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.channelBox, kwargs, 'shapes')
        return res

    @_f.addMelDocs('channelBox', 'showNamespace')
    def getShowNamespace(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.channelBox, kwargs, 'showNamespace')
        return res

    @_f.addMelDocs('channelBox', 'showTransforms')
    def getShowTransforms(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.channelBox, kwargs, 'showTransforms')
        return res

    @_f.addMelDocs('channelBox', 'speed')
    def getSpeed(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.channelBox, kwargs, 'speed')
        return res

    @_f.addMelDocs('channelBox', 'useManips')
    def getUseManips(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.channelBox, kwargs, 'useManips')
        return res

    @_f.addMelDocs('channelBox', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.channelBox, kwargs, 'visible')
        return res

    @_f.addMelDocs('channelBox', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.channelBox, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('channelBox', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.channelBox, kwargs, 'width')
        return res

    @_f.addMelDocs('channelBox', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.channelBox, kwargs, 'noBackground', val)

    @_f.addMelDocs('channelBox', 'select')
    def select(self, val=True, **kwargs):
        # type: (str | List[str], **Any) -> None
        return _f.asEdit(self, windows.channelBox, kwargs, 'select', val)

    @_f.addMelDocs('channelBox', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.channelBox, kwargs, 'annotation', val)

    @_f.addMelDocs('channelBox', 'attrBgColor')
    def setAttrBgColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.channelBox, kwargs, 'attrBgColor', val)

    @_f.addMelDocs('channelBox', 'attrColor')
    def setAttrColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.channelBox, kwargs, 'attrColor', val)

    @_f.addMelDocs('channelBox', 'attrFilter')
    def setAttrFilter(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.channelBox, kwargs, 'attrFilter', val)

    @_f.addMelDocs('channelBox', 'attrRegex')
    def setAttrRegex(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.channelBox, kwargs, 'attrRegex', val)

    @_f.addMelDocs('channelBox', 'attributeEditorMode')
    def setAttributeEditorMode(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.channelBox, kwargs, 'attributeEditorMode', val)

    @_f.addMelDocs('channelBox', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.channelBox, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('channelBox', 'containerAtTop')
    def setContainerAtTop(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.channelBox, kwargs, 'containerAtTop', val)

    @_f.addMelDocs('channelBox', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.channelBox, kwargs, 'docTag', val)

    @_f.addMelDocs('channelBox', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.channelBox, kwargs, 'enable', val)

    @_f.addMelDocs('channelBox', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.channelBox, kwargs, 'enableBackground', val)

    @_f.addMelDocs('channelBox', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.channelBox, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('channelBox', 'enableLabelSelection')
    def setEnableLabelSelection(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.channelBox, kwargs, 'enableLabelSelection', val)

    @_f.addMelDocs('channelBox', 'fieldWidth')
    def setFieldWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.channelBox, kwargs, 'fieldWidth', val)

    @_f.addMelDocs('channelBox', 'fixedAttrList')
    def setFixedAttrList(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.channelBox, kwargs, 'fixedAttrList', val)

    @_f.addMelDocs('channelBox', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.channelBox, kwargs, 'height', val)

    @_f.addMelDocs('channelBox', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.channelBox, kwargs, 'highlightColor', val)

    @_f.addMelDocs('channelBox', 'hyperbolic')
    def setHyperbolic(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.channelBox, kwargs, 'hyperbolic', val)

    @_f.addMelDocs('channelBox', 'labelWidth')
    def setLabelWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.channelBox, kwargs, 'labelWidth', val)

    @_f.addMelDocs('channelBox', 'longNames')
    def setLongNames(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.channelBox, kwargs, 'longNames', val)

    @_f.addMelDocs('channelBox', 'mainListConnection')
    def setMainListConnection(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.channelBox, kwargs, 'mainListConnection', val)

    @_f.addMelDocs('channelBox', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.channelBox, kwargs, 'manage', val)

    @_f.addMelDocs('channelBox', 'maxHeight')
    def setMaxHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.channelBox, kwargs, 'maxHeight', val)

    @_f.addMelDocs('channelBox', 'maxWidth')
    def setMaxWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.channelBox, kwargs, 'maxWidth', val)

    @_f.addMelDocs('channelBox', 'niceNames')
    def setNiceNames(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.channelBox, kwargs, 'niceNames', val)

    @_f.addMelDocs('channelBox', 'nodeRegex')
    def setNodeRegex(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.channelBox, kwargs, 'nodeRegex', val)

    @_f.addMelDocs('channelBox', 'precision')
    def setPrecision(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.channelBox, kwargs, 'precision', val)

    @_f.addMelDocs('channelBox', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.channelBox, kwargs, 'preventOverride', val)

    @_f.addMelDocs('channelBox', 'showNamespace')
    def setShowNamespace(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.channelBox, kwargs, 'showNamespace', val)

    @_f.addMelDocs('channelBox', 'showTransforms')
    def setShowTransforms(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.channelBox, kwargs, 'showTransforms', val)

    @_f.addMelDocs('channelBox', 'speed')
    def setSpeed(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.channelBox, kwargs, 'speed', val)

    @_f.addMelDocs('channelBox', 'useManips')
    def setUseManips(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.channelBox, kwargs, 'useManips', val)

    @_f.addMelDocs('channelBox', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.channelBox, kwargs, 'visible', val)

    @_f.addMelDocs('channelBox', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.channelBox, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('channelBox', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.channelBox, kwargs, 'width', val)

    @_f.addMelDocs('channelBox', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.channelBox, kwargs, 'statusBarMessage', val)

    @_f.addMelDocs('channelBox', 'takeFocus')
    def takeFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.channelBox, kwargs, 'takeFocus', val)

    @_f.addMelDocs('channelBox', 'update')
    def update(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.channelBox, kwargs, 'update', val)


class CheckBox(PyUI):
    __melcmd__ = staticmethod(windows.checkBox)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'checkBox'
    __melui__ = 'checkBox'
    __slots__ = ()

    @_f.addMelDocs('checkBox', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.checkBox, kwargs, 'dragCallback', val)

    @_f.addMelDocs('checkBox', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.checkBox, kwargs, 'dropCallback', val)

    @_f.addMelDocs('checkBox', 'align')
    def getAlign(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.checkBox, kwargs, 'align')
        return res

    @_f.addMelDocs('checkBox', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.checkBox, kwargs, 'annotation')
        return res

    @_f.addMelDocs('checkBox', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.checkBox, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('checkBox', 'changeCommand')
    def getChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.checkBox, kwargs, 'changeCommand')
        return res

    @_f.addMelDocs('checkBox', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.checkBox, kwargs, 'docTag')
        return res

    @_f.addMelDocs('checkBox', 'editable')
    def getEditable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.checkBox, kwargs, 'editable')
        return res

    @_f.addMelDocs('checkBox', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.checkBox, kwargs, 'enable')
        return res

    @_f.addMelDocs('checkBox', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.checkBox, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('checkBox', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.checkBox, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('checkBox', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.checkBox, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('checkBox', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.checkBox, kwargs, 'height')
        return res

    @_f.addMelDocs('checkBox', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.checkBox, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('checkBox', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.checkBox, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('checkBox', 'label')
    def getLabel(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.checkBox, kwargs, 'label')
        return res

    @_f.addMelDocs('checkBox', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.checkBox, kwargs, 'manage')
        return res

    @_f.addMelDocs('checkBox', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.checkBox, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('checkBox', 'offCommand')
    def getOffCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.checkBox, kwargs, 'offCommand')
        return res

    @_f.addMelDocs('checkBox', 'onCommand')
    def getOnCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.checkBox, kwargs, 'onCommand')
        return res

    @_f.addMelDocs('checkBox', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.checkBox, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('checkBox', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.checkBox, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('checkBox', 'recomputeSize')
    def getRecomputeSize(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.checkBox, kwargs, 'recomputeSize')
        return res

    @_f.addMelDocs('checkBox', 'value')
    def getValue(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.checkBox, kwargs, 'value')
        return res

    @_f.addMelDocs('checkBox', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.checkBox, kwargs, 'visible')
        return res

    @_f.addMelDocs('checkBox', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.checkBox, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('checkBox', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.checkBox, kwargs, 'width')
        return res

    @_f.addMelDocs('checkBox', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.checkBox, kwargs, 'noBackground', val)

    @_f.addMelDocs('checkBox', 'align')
    def setAlign(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.checkBox, kwargs, 'align', val)

    @_f.addMelDocs('checkBox', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.checkBox, kwargs, 'annotation', val)

    @_f.addMelDocs('checkBox', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.checkBox, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('checkBox', 'changeCommand')
    def setChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.checkBox, kwargs, 'changeCommand', val)

    @_f.addMelDocs('checkBox', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.checkBox, kwargs, 'docTag', val)

    @_f.addMelDocs('checkBox', 'editable')
    def setEditable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.checkBox, kwargs, 'editable', val)

    @_f.addMelDocs('checkBox', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.checkBox, kwargs, 'enable', val)

    @_f.addMelDocs('checkBox', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.checkBox, kwargs, 'enableBackground', val)

    @_f.addMelDocs('checkBox', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.checkBox, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('checkBox', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.checkBox, kwargs, 'height', val)

    @_f.addMelDocs('checkBox', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.checkBox, kwargs, 'highlightColor', val)

    @_f.addMelDocs('checkBox', 'label')
    def setLabel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.checkBox, kwargs, 'label', val)

    @_f.addMelDocs('checkBox', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.checkBox, kwargs, 'manage', val)

    @_f.addMelDocs('checkBox', 'offCommand')
    def setOffCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.checkBox, kwargs, 'offCommand', val)

    @_f.addMelDocs('checkBox', 'onCommand')
    def setOnCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.checkBox, kwargs, 'onCommand', val)

    @_f.addMelDocs('checkBox', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.checkBox, kwargs, 'preventOverride', val)

    @_f.addMelDocs('checkBox', 'recomputeSize')
    def setRecomputeSize(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.checkBox, kwargs, 'recomputeSize', val)

    @_f.addMelDocs('checkBox', 'value')
    def setValue(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.checkBox, kwargs, 'value', val)

    @_f.addMelDocs('checkBox', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.checkBox, kwargs, 'visible', val)

    @_f.addMelDocs('checkBox', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.checkBox, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('checkBox', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.checkBox, kwargs, 'width', val)

    @_f.addMelDocs('checkBox', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.checkBox, kwargs, 'statusBarMessage', val)


class ClipEditor(PyUI):
    __melcmd__ = staticmethod(animation.clipEditor)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'clipEditor'
    __melui__ = 'clipEditor'
    __slots__ = ()

    @_f.addMelDocs('clipEditor', 'clipDropCmd')
    def clipDropCmd(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.clipEditor, kwargs, 'clipDropCmd', val)

    @_f.addMelDocs('clipEditor', 'deleteCmd')
    def deleteCmd(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.clipEditor, kwargs, 'deleteCmd', val)

    @_f.addMelDocs('clipEditor', 'deselectAll')
    def deselectAll(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.clipEditor, kwargs, 'deselectAll', val)

    @_f.addMelDocs('clipEditor', 'displayActiveKeyTangents')
    def displayActiveKeyTangents(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.clipEditor, kwargs, 'displayActiveKeyTangents', val)

    @_f.addMelDocs('clipEditor', 'displayActiveKeys')
    def displayActiveKeys(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.clipEditor, kwargs, 'displayActiveKeys', val)

    @_f.addMelDocs('clipEditor', 'displayInfinities')
    def displayInfinities(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.clipEditor, kwargs, 'displayInfinities', val)

    @_f.addMelDocs('clipEditor', 'displayKeys')
    def displayKeys(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.clipEditor, kwargs, 'displayKeys', val)

    @_f.addMelDocs('clipEditor', 'displayTangents')
    def displayTangents(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.clipEditor, kwargs, 'displayTangents', val)

    @_f.addMelDocs('clipEditor', 'displayValues')
    def displayValues(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.clipEditor, kwargs, 'displayValues', val)

    @_f.addMelDocs('clipEditor', 'frameAll')
    def frameAll(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.clipEditor, kwargs, 'frameAll', val)

    @_f.addMelDocs('clipEditor', 'autoFit')
    def getAutoFit(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.clipEditor, kwargs, 'autoFit')
        return res

    @_f.addMelDocs('clipEditor', 'autoFitTime')
    def getAutoFitTime(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.clipEditor, kwargs, 'autoFitTime')
        return res

    @_f.addMelDocs('clipEditor', 'clipStyle')
    def getClipStyle(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, animation.clipEditor, kwargs, 'clipStyle')
        return res

    @_f.addMelDocs('clipEditor', 'control')
    def getControl(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.clipEditor, kwargs, 'control')
        return res

    @_f.addMelDocs('clipEditor', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.clipEditor, kwargs, 'docTag')
        return res

    @_f.addMelDocs('clipEditor', 'filter')
    def getFilter(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.clipEditor, kwargs, 'filter')
        return res

    @_f.addMelDocs('clipEditor', 'forceMainConnection')
    def getForceMainConnection(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.clipEditor, kwargs, 'forceMainConnection')
        return res

    @_f.addMelDocs('clipEditor', 'frameRange')
    def getFrameRange(self, **kwargs):
        # type: (...) -> Tuple[float, float]
        res = _f.asQuery(self, animation.clipEditor, kwargs, 'frameRange')
        return res

    @_f.addMelDocs('clipEditor', 'highlightConnection')
    def getHighlightConnection(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.clipEditor, kwargs, 'highlightConnection')
        return res

    @_f.addMelDocs('clipEditor', 'highlightedBlend')
    def getHighlightedBlend(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.clipEditor, kwargs, 'highlightedBlend')
        return res

    @_f.addMelDocs('clipEditor', 'highlightedClip')
    def getHighlightedClip(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.clipEditor, kwargs, 'highlightedClip')
        return res

    @_f.addMelDocs('clipEditor', 'initialized')
    def getInitialized(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.clipEditor, kwargs, 'initialized')
        return res

    @_f.addMelDocs('clipEditor', 'mainListConnection')
    def getMainListConnection(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.clipEditor, kwargs, 'mainListConnection')
        return res

    @_f.addMelDocs('clipEditor', 'manageSequencer')
    def getManageSequencer(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.clipEditor, kwargs, 'manageSequencer')
        return res

    @_f.addMelDocs('clipEditor', 'menuContext')
    def getMenuContext(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.clipEditor, kwargs, 'menuContext')
        return res

    @_f.addMelDocs('clipEditor', 'panel')
    def getPanel(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.clipEditor, kwargs, 'panel')
        return res

    @_f.addMelDocs('clipEditor', 'selectBlend')
    def getSelectBlend(self, **kwargs):
        # type: (...) -> Tuple[str, str, str]
        res = _f.asQuery(self, animation.clipEditor, kwargs, 'selectBlend')
        return res

    @_f.addMelDocs('clipEditor', 'selectClip')
    def getSelectClip(self, **kwargs):
        # type: (...) -> Tuple[str, str]
        res = _f.asQuery(self, animation.clipEditor, kwargs, 'selectClip')
        return res

    @_f.addMelDocs('clipEditor', 'selectionConnection')
    def getSelectionConnection(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.clipEditor, kwargs, 'selectionConnection')
        return res

    @_f.addMelDocs('clipEditor', 'snapTime')
    def getSnapTime(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.clipEditor, kwargs, 'snapTime')
        return res

    @_f.addMelDocs('clipEditor', 'snapValue')
    def getSnapValue(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.clipEditor, kwargs, 'snapValue')
        return res

    @_f.addMelDocs('clipEditor', 'stateString')
    def getStateString(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.clipEditor, kwargs, 'stateString')
        return res

    @_f.addMelDocs('clipEditor', 'listAllCharacters')
    def listAllCharacters(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.clipEditor, kwargs, 'listAllCharacters', val)

    @_f.addMelDocs('clipEditor', 'listCurrentCharacters')
    def listCurrentCharacters(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.clipEditor, kwargs, 'listCurrentCharacters', val)

    @_f.addMelDocs('clipEditor', 'lockMainConnection')
    def lockMainConnection(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.clipEditor, kwargs, 'lockMainConnection', val)

    @_f.addMelDocs('clipEditor', 'lookAt')
    def lookAt(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.clipEditor, kwargs, 'lookAt', val)

    @_f.addMelDocs('clipEditor', 'autoFit')
    def setAutoFit(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.clipEditor, kwargs, 'autoFit', val)

    @_f.addMelDocs('clipEditor', 'autoFitTime')
    def setAutoFitTime(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.clipEditor, kwargs, 'autoFitTime', val)

    @_f.addMelDocs('clipEditor', 'clipStyle')
    def setClipStyle(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, animation.clipEditor, kwargs, 'clipStyle', val)

    @_f.addMelDocs('clipEditor', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.clipEditor, kwargs, 'docTag', val)

    @_f.addMelDocs('clipEditor', 'filter')
    def setFilter(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.clipEditor, kwargs, 'filter', val)

    @_f.addMelDocs('clipEditor', 'forceMainConnection')
    def setForceMainConnection(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.clipEditor, kwargs, 'forceMainConnection', val)

    @_f.addMelDocs('clipEditor', 'frameRange')
    def setFrameRange(self, val=True, **kwargs):
        # type: (Tuple[float, float], **Any) -> None
        return _f.asEdit(self, animation.clipEditor, kwargs, 'frameRange', val)

    @_f.addMelDocs('clipEditor', 'highlightConnection')
    def setHighlightConnection(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.clipEditor, kwargs, 'highlightConnection', val)

    @_f.addMelDocs('clipEditor', 'mainListConnection')
    def setMainListConnection(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.clipEditor, kwargs, 'mainListConnection', val)

    @_f.addMelDocs('clipEditor', 'manageSequencer')
    def setManageSequencer(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.clipEditor, kwargs, 'manageSequencer', val)

    @_f.addMelDocs('clipEditor', 'parent')
    def setParent(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.clipEditor, kwargs, 'parent', val)

    @_f.addMelDocs('clipEditor', 'selectBlend')
    def setSelectBlend(self, val=True, **kwargs):
        # type: (Tuple[str, str, str], **Any) -> None
        return _f.asEdit(self, animation.clipEditor, kwargs, 'selectBlend', val)

    @_f.addMelDocs('clipEditor', 'selectClip')
    def setSelectClip(self, val=True, **kwargs):
        # type: (Tuple[str, str], **Any) -> None
        return _f.asEdit(self, animation.clipEditor, kwargs, 'selectClip', val)

    @_f.addMelDocs('clipEditor', 'selectionConnection')
    def setSelectionConnection(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.clipEditor, kwargs, 'selectionConnection', val)

    @_f.addMelDocs('clipEditor', 'snapTime')
    def setSnapTime(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.clipEditor, kwargs, 'snapTime', val)

    @_f.addMelDocs('clipEditor', 'snapValue')
    def setSnapValue(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.clipEditor, kwargs, 'snapValue', val)

    @_f.addMelDocs('clipEditor', 'unParent')
    def unParent(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.clipEditor, kwargs, 'unParent', val)

    @_f.addMelDocs('clipEditor', 'unlockMainConnection')
    def unlockMainConnection(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.clipEditor, kwargs, 'unlockMainConnection', val)

    @_f.addMelDocs('clipEditor', 'updateMainConnection')
    def updateMainConnection(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.clipEditor, kwargs, 'updateMainConnection', val)


class ClipSchedulerOutliner(PyUI):
    __melcmd__ = staticmethod(animation.clipSchedulerOutliner)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'clipSchedulerOutliner'
    __melui__ = 'clipSchedulerOutliner'
    __slots__ = ()

    @_f.addMelDocs('clipSchedulerOutliner', 'clipScheduler')
    def clipScheduler(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.clipSchedulerOutliner, kwargs, 'clipScheduler', val)

    @_f.addMelDocs('clipSchedulerOutliner', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, animation.clipSchedulerOutliner, kwargs, 'dragCallback', val)

    @_f.addMelDocs('clipSchedulerOutliner', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, animation.clipSchedulerOutliner, kwargs, 'dropCallback', val)

    @_f.addMelDocs('clipSchedulerOutliner', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.clipSchedulerOutliner, kwargs, 'annotation')
        return res

    @_f.addMelDocs('clipSchedulerOutliner', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, animation.clipSchedulerOutliner, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('clipSchedulerOutliner', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.clipSchedulerOutliner, kwargs, 'docTag')
        return res

    @_f.addMelDocs('clipSchedulerOutliner', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.clipSchedulerOutliner, kwargs, 'enable')
        return res

    @_f.addMelDocs('clipSchedulerOutliner', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.clipSchedulerOutliner, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('clipSchedulerOutliner', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.clipSchedulerOutliner, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('clipSchedulerOutliner', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.clipSchedulerOutliner, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('clipSchedulerOutliner', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, animation.clipSchedulerOutliner, kwargs, 'height')
        return res

    @_f.addMelDocs('clipSchedulerOutliner', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, animation.clipSchedulerOutliner, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('clipSchedulerOutliner', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.clipSchedulerOutliner, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('clipSchedulerOutliner', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.clipSchedulerOutliner, kwargs, 'manage')
        return res

    @_f.addMelDocs('clipSchedulerOutliner', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.clipSchedulerOutliner, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('clipSchedulerOutliner', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.clipSchedulerOutliner, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('clipSchedulerOutliner', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.clipSchedulerOutliner, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('clipSchedulerOutliner', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.clipSchedulerOutliner, kwargs, 'visible')
        return res

    @_f.addMelDocs('clipSchedulerOutliner', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, animation.clipSchedulerOutliner, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('clipSchedulerOutliner', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, animation.clipSchedulerOutliner, kwargs, 'width')
        return res

    @_f.addMelDocs('clipSchedulerOutliner', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.clipSchedulerOutliner, kwargs, 'noBackground', val)

    @_f.addMelDocs('clipSchedulerOutliner', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.clipSchedulerOutliner, kwargs, 'annotation', val)

    @_f.addMelDocs('clipSchedulerOutliner', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, animation.clipSchedulerOutliner, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('clipSchedulerOutliner', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.clipSchedulerOutliner, kwargs, 'docTag', val)

    @_f.addMelDocs('clipSchedulerOutliner', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.clipSchedulerOutliner, kwargs, 'enable', val)

    @_f.addMelDocs('clipSchedulerOutliner', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.clipSchedulerOutliner, kwargs, 'enableBackground', val)

    @_f.addMelDocs('clipSchedulerOutliner', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.clipSchedulerOutliner, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('clipSchedulerOutliner', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, animation.clipSchedulerOutliner, kwargs, 'height', val)

    @_f.addMelDocs('clipSchedulerOutliner', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, animation.clipSchedulerOutliner, kwargs, 'highlightColor', val)

    @_f.addMelDocs('clipSchedulerOutliner', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.clipSchedulerOutliner, kwargs, 'manage', val)

    @_f.addMelDocs('clipSchedulerOutliner', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.clipSchedulerOutliner, kwargs, 'preventOverride', val)

    @_f.addMelDocs('clipSchedulerOutliner', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.clipSchedulerOutliner, kwargs, 'visible', val)

    @_f.addMelDocs('clipSchedulerOutliner', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, animation.clipSchedulerOutliner, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('clipSchedulerOutliner', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, animation.clipSchedulerOutliner, kwargs, 'width', val)

    @_f.addMelDocs('clipSchedulerOutliner', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.clipSchedulerOutliner, kwargs, 'statusBarMessage', val)


class CmdScrollFieldExecuter(PyUI):
    __melcmd__ = staticmethod(windows.cmdScrollFieldExecuter)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'cmdScrollFieldExecuter'
    __melui__ = 'cmdScrollFieldExecuter'
    __slots__ = ()

    @_f.addMelDocs('cmdScrollFieldExecuter', 'appendText')
    def appendText(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'appendText', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'clear')
    def clear(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'clear', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'copySelection')
    def copySelection(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'copySelection', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'cutSelection')
    def cutSelection(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'cutSelection', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'dragCallback', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'dropCallback', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'execute')
    def execute(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'execute', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'executeAll')
    def executeAll(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'executeAll', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'fileChangedCommand')
    def fileChangedCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'fileChangedCommand', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.cmdScrollFieldExecuter, kwargs, 'annotation')
        return res

    @_f.addMelDocs('cmdScrollFieldExecuter', 'autoCloseBraces')
    def getAutoCloseBraces(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.cmdScrollFieldExecuter, kwargs, 'autoCloseBraces')
        return res

    @_f.addMelDocs('cmdScrollFieldExecuter', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.cmdScrollFieldExecuter, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('cmdScrollFieldExecuter', 'commandCompletion')
    def getCommandCompletion(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.cmdScrollFieldExecuter, kwargs, 'commandCompletion')
        return res

    @_f.addMelDocs('cmdScrollFieldExecuter', 'currentLine')
    def getCurrentLine(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.cmdScrollFieldExecuter, kwargs, 'currentLine')
        return res

    @_f.addMelDocs('cmdScrollFieldExecuter', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.cmdScrollFieldExecuter, kwargs, 'docTag')
        return res

    @_f.addMelDocs('cmdScrollFieldExecuter', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.cmdScrollFieldExecuter, kwargs, 'enable')
        return res

    @_f.addMelDocs('cmdScrollFieldExecuter', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.cmdScrollFieldExecuter, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('cmdScrollFieldExecuter', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.cmdScrollFieldExecuter, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('cmdScrollFieldExecuter', 'filename')
    def getFilename(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.cmdScrollFieldExecuter, kwargs, 'filename')
        return res

    @_f.addMelDocs('cmdScrollFieldExecuter', 'filterKeyPress')
    def getFilterKeyPress(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.cmdScrollFieldExecuter, kwargs, 'filterKeyPress')
        return res

    @_f.addMelDocs('cmdScrollFieldExecuter', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.cmdScrollFieldExecuter, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('cmdScrollFieldExecuter', 'hasFocus')
    def getHasFocus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.cmdScrollFieldExecuter, kwargs, 'hasFocus')
        return res

    @_f.addMelDocs('cmdScrollFieldExecuter', 'hasSelection')
    def getHasSelection(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.cmdScrollFieldExecuter, kwargs, 'hasSelection')
        return res

    @_f.addMelDocs('cmdScrollFieldExecuter', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.cmdScrollFieldExecuter, kwargs, 'height')
        return res

    @_f.addMelDocs('cmdScrollFieldExecuter', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.cmdScrollFieldExecuter, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('cmdScrollFieldExecuter', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.cmdScrollFieldExecuter, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('cmdScrollFieldExecuter', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.cmdScrollFieldExecuter, kwargs, 'manage')
        return res

    @_f.addMelDocs('cmdScrollFieldExecuter', 'modified')
    def getModified(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.cmdScrollFieldExecuter, kwargs, 'modified')
        return res

    @_f.addMelDocs('cmdScrollFieldExecuter', 'numberOfLines')
    def getNumberOfLines(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.cmdScrollFieldExecuter, kwargs, 'numberOfLines')
        return res

    @_f.addMelDocs('cmdScrollFieldExecuter', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.cmdScrollFieldExecuter, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('cmdScrollFieldExecuter', 'objectPathCompletion')
    def getObjectPathCompletion(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.cmdScrollFieldExecuter, kwargs, 'objectPathCompletion')
        return res

    @_f.addMelDocs('cmdScrollFieldExecuter', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.cmdScrollFieldExecuter, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('cmdScrollFieldExecuter', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.cmdScrollFieldExecuter, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('cmdScrollFieldExecuter', 'searchAndSelect')
    def getSearchAndSelect(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.cmdScrollFieldExecuter, kwargs, 'searchAndSelect')
        return res

    @_f.addMelDocs('cmdScrollFieldExecuter', 'searchDown')
    def getSearchDown(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.cmdScrollFieldExecuter, kwargs, 'searchDown')
        return res

    @_f.addMelDocs('cmdScrollFieldExecuter', 'searchMatchCase')
    def getSearchMatchCase(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.cmdScrollFieldExecuter, kwargs, 'searchMatchCase')
        return res

    @_f.addMelDocs('cmdScrollFieldExecuter', 'searchString')
    def getSearchString(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.cmdScrollFieldExecuter, kwargs, 'searchString')
        return res

    @_f.addMelDocs('cmdScrollFieldExecuter', 'searchWraps')
    def getSearchWraps(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.cmdScrollFieldExecuter, kwargs, 'searchWraps')
        return res

    @_f.addMelDocs('cmdScrollFieldExecuter', 'selectedText')
    def getSelectedText(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.cmdScrollFieldExecuter, kwargs, 'selectedText')
        return res

    @_f.addMelDocs('cmdScrollFieldExecuter', 'showLineNumbers')
    def getShowLineNumbers(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.cmdScrollFieldExecuter, kwargs, 'showLineNumbers')
        return res

    @_f.addMelDocs('cmdScrollFieldExecuter', 'showTooltipHelp')
    def getShowTooltipHelp(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.cmdScrollFieldExecuter, kwargs, 'showTooltipHelp')
        return res

    @_f.addMelDocs('cmdScrollFieldExecuter', 'sourceType')
    def getSourceType(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.cmdScrollFieldExecuter, kwargs, 'sourceType')
        return res

    @_f.addMelDocs('cmdScrollFieldExecuter', 'spacesPerTab')
    def getSpacesPerTab(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.cmdScrollFieldExecuter, kwargs, 'spacesPerTab')
        return res

    @_f.addMelDocs('cmdScrollFieldExecuter', 'tabsForIndent')
    def getTabsForIndent(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.cmdScrollFieldExecuter, kwargs, 'tabsForIndent')
        return res

    @_f.addMelDocs('cmdScrollFieldExecuter', 'text')
    def getText(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.cmdScrollFieldExecuter, kwargs, 'text')
        return res

    @_f.addMelDocs('cmdScrollFieldExecuter', 'textLength')
    def getTextLength(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.cmdScrollFieldExecuter, kwargs, 'textLength')
        return res

    @_f.addMelDocs('cmdScrollFieldExecuter', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.cmdScrollFieldExecuter, kwargs, 'visible')
        return res

    @_f.addMelDocs('cmdScrollFieldExecuter', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.cmdScrollFieldExecuter, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('cmdScrollFieldExecuter', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.cmdScrollFieldExecuter, kwargs, 'width')
        return res

    @_f.addMelDocs('cmdScrollFieldExecuter', 'insertText')
    def insertText(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'insertText', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'load')
    def load(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'load', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'loadContents')
    def loadContents(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'loadContents', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'loadFile')
    def loadFile(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'loadFile', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'modificationChangedCommand')
    def modificationChangedCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'modificationChangedCommand', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'noBackground', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'pasteSelection')
    def pasteSelection(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'pasteSelection', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'redo')
    def redo(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'redo', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'removeStoredContents')
    def removeStoredContents(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'removeStoredContents', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'replaceAll')
    def replaceAll(self, val=True, **kwargs):
        # type: (Tuple[str, str], **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'replaceAll', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'saveFile')
    def saveFile(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'saveFile', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'saveSelection')
    def saveSelection(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'saveSelection', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'saveSelectionToShelf')
    def saveSelectionToShelf(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'saveSelectionToShelf', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'select')
    def select(self, val=True, **kwargs):
        # type: (Tuple[int, int], **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'select', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'selectAll')
    def selectAll(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'selectAll', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'annotation', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'autoCloseBraces')
    def setAutoCloseBraces(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'autoCloseBraces', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'commandCompletion')
    def setCommandCompletion(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'commandCompletion', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'currentLine')
    def setCurrentLine(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'currentLine', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'docTag', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'enable', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'enableBackground', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'filterKeyPress')
    def setFilterKeyPress(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'filterKeyPress', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'height', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'highlightColor', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'manage', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'modified')
    def setModified(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'modified', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'objectPathCompletion')
    def setObjectPathCompletion(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'objectPathCompletion', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'preventOverride', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'searchDown')
    def setSearchDown(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'searchDown', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'searchMatchCase')
    def setSearchMatchCase(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'searchMatchCase', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'searchString')
    def setSearchString(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'searchString', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'searchWraps')
    def setSearchWraps(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'searchWraps', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'showLineNumbers')
    def setShowLineNumbers(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'showLineNumbers', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'showTooltipHelp')
    def setShowTooltipHelp(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'showTooltipHelp', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'spacesPerTab')
    def setSpacesPerTab(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'spacesPerTab', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'tabsForIndent')
    def setTabsForIndent(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'tabsForIndent', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'text')
    def setText(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'text', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'visible', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'width', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'source')
    def source(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'source', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'statusBarMessage', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'storeContents')
    def storeContents(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'storeContents', val)

    @_f.addMelDocs('cmdScrollFieldExecuter', 'undo')
    def undo(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldExecuter, kwargs, 'undo', val)


class CmdScrollFieldReporter(PyUI):
    __melcmd__ = staticmethod(windows.cmdScrollFieldReporter)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'cmdScrollFieldReporter'
    __melui__ = 'cmdScrollFieldReporter'
    __slots__ = ()

    @_f.addMelDocs('cmdScrollFieldReporter', 'clear')
    def clear(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldReporter, kwargs, 'clear', val)

    @_f.addMelDocs('cmdScrollFieldReporter', 'copySelection')
    def copySelection(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldReporter, kwargs, 'copySelection', val)

    @_f.addMelDocs('cmdScrollFieldReporter', 'cutSelection')
    def cutSelection(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldReporter, kwargs, 'cutSelection', val)

    @_f.addMelDocs('cmdScrollFieldReporter', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldReporter, kwargs, 'dragCallback', val)

    @_f.addMelDocs('cmdScrollFieldReporter', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldReporter, kwargs, 'dropCallback', val)

    @_f.addMelDocs('cmdScrollFieldReporter', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.cmdScrollFieldReporter, kwargs, 'annotation')
        return res

    @_f.addMelDocs('cmdScrollFieldReporter', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.cmdScrollFieldReporter, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('cmdScrollFieldReporter', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.cmdScrollFieldReporter, kwargs, 'docTag')
        return res

    @_f.addMelDocs('cmdScrollFieldReporter', 'echoAllCommands')
    def getEchoAllCommands(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.cmdScrollFieldReporter, kwargs, 'echoAllCommands')
        return res

    @_f.addMelDocs('cmdScrollFieldReporter', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.cmdScrollFieldReporter, kwargs, 'enable')
        return res

    @_f.addMelDocs('cmdScrollFieldReporter', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.cmdScrollFieldReporter, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('cmdScrollFieldReporter', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.cmdScrollFieldReporter, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('cmdScrollFieldReporter', 'filterSourceType')
    def getFilterSourceType(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.cmdScrollFieldReporter, kwargs, 'filterSourceType')
        return res

    @_f.addMelDocs('cmdScrollFieldReporter', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.cmdScrollFieldReporter, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('cmdScrollFieldReporter', 'hasFocus')
    def getHasFocus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.cmdScrollFieldReporter, kwargs, 'hasFocus')
        return res

    @_f.addMelDocs('cmdScrollFieldReporter', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.cmdScrollFieldReporter, kwargs, 'height')
        return res

    @_f.addMelDocs('cmdScrollFieldReporter', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.cmdScrollFieldReporter, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('cmdScrollFieldReporter', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.cmdScrollFieldReporter, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('cmdScrollFieldReporter', 'lineNumbers')
    def getLineNumbers(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.cmdScrollFieldReporter, kwargs, 'lineNumbers')
        return res

    @_f.addMelDocs('cmdScrollFieldReporter', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.cmdScrollFieldReporter, kwargs, 'manage')
        return res

    @_f.addMelDocs('cmdScrollFieldReporter', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.cmdScrollFieldReporter, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('cmdScrollFieldReporter', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.cmdScrollFieldReporter, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('cmdScrollFieldReporter', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.cmdScrollFieldReporter, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('cmdScrollFieldReporter', 'stackTrace')
    def getStackTrace(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.cmdScrollFieldReporter, kwargs, 'stackTrace')
        return res

    @_f.addMelDocs('cmdScrollFieldReporter', 'suppressErrors')
    def getSuppressErrors(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.cmdScrollFieldReporter, kwargs, 'suppressErrors')
        return res

    @_f.addMelDocs('cmdScrollFieldReporter', 'suppressInfo')
    def getSuppressInfo(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.cmdScrollFieldReporter, kwargs, 'suppressInfo')
        return res

    @_f.addMelDocs('cmdScrollFieldReporter', 'suppressResults')
    def getSuppressResults(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.cmdScrollFieldReporter, kwargs, 'suppressResults')
        return res

    @_f.addMelDocs('cmdScrollFieldReporter', 'suppressStackTrace')
    def getSuppressStackTrace(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.cmdScrollFieldReporter, kwargs, 'suppressStackTrace')
        return res

    @_f.addMelDocs('cmdScrollFieldReporter', 'suppressWarnings')
    def getSuppressWarnings(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.cmdScrollFieldReporter, kwargs, 'suppressWarnings')
        return res

    @_f.addMelDocs('cmdScrollFieldReporter', 'text')
    def getText(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.cmdScrollFieldReporter, kwargs, 'text')
        return res

    @_f.addMelDocs('cmdScrollFieldReporter', 'textLength')
    def getTextLength(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.cmdScrollFieldReporter, kwargs, 'textLength')
        return res

    @_f.addMelDocs('cmdScrollFieldReporter', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.cmdScrollFieldReporter, kwargs, 'visible')
        return res

    @_f.addMelDocs('cmdScrollFieldReporter', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.cmdScrollFieldReporter, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('cmdScrollFieldReporter', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.cmdScrollFieldReporter, kwargs, 'width')
        return res

    @_f.addMelDocs('cmdScrollFieldReporter', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldReporter, kwargs, 'noBackground', val)

    @_f.addMelDocs('cmdScrollFieldReporter', 'pasteSelection')
    def pasteSelection(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldReporter, kwargs, 'pasteSelection', val)

    @_f.addMelDocs('cmdScrollFieldReporter', 'receiveFocusCommand')
    def receiveFocusCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldReporter, kwargs, 'receiveFocusCommand', val)

    @_f.addMelDocs('cmdScrollFieldReporter', 'saveSelection')
    def saveSelection(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldReporter, kwargs, 'saveSelection', val)

    @_f.addMelDocs('cmdScrollFieldReporter', 'saveSelectionToShelf')
    def saveSelectionToShelf(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldReporter, kwargs, 'saveSelectionToShelf', val)

    @_f.addMelDocs('cmdScrollFieldReporter', 'select')
    def select(self, val=True, **kwargs):
        # type: (Tuple[int, int], **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldReporter, kwargs, 'select', val)

    @_f.addMelDocs('cmdScrollFieldReporter', 'selectAll')
    def selectAll(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldReporter, kwargs, 'selectAll', val)

    @_f.addMelDocs('cmdScrollFieldReporter', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldReporter, kwargs, 'annotation', val)

    @_f.addMelDocs('cmdScrollFieldReporter', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldReporter, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('cmdScrollFieldReporter', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldReporter, kwargs, 'docTag', val)

    @_f.addMelDocs('cmdScrollFieldReporter', 'echoAllCommands')
    def setEchoAllCommands(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldReporter, kwargs, 'echoAllCommands', val)

    @_f.addMelDocs('cmdScrollFieldReporter', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldReporter, kwargs, 'enable', val)

    @_f.addMelDocs('cmdScrollFieldReporter', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldReporter, kwargs, 'enableBackground', val)

    @_f.addMelDocs('cmdScrollFieldReporter', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldReporter, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('cmdScrollFieldReporter', 'filterSourceType')
    def setFilterSourceType(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldReporter, kwargs, 'filterSourceType', val)

    @_f.addMelDocs('cmdScrollFieldReporter', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldReporter, kwargs, 'height', val)

    @_f.addMelDocs('cmdScrollFieldReporter', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldReporter, kwargs, 'highlightColor', val)

    @_f.addMelDocs('cmdScrollFieldReporter', 'lineNumbers')
    def setLineNumbers(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldReporter, kwargs, 'lineNumbers', val)

    @_f.addMelDocs('cmdScrollFieldReporter', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldReporter, kwargs, 'manage', val)

    @_f.addMelDocs('cmdScrollFieldReporter', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldReporter, kwargs, 'preventOverride', val)

    @_f.addMelDocs('cmdScrollFieldReporter', 'stackTrace')
    def setStackTrace(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldReporter, kwargs, 'stackTrace', val)

    @_f.addMelDocs('cmdScrollFieldReporter', 'suppressErrors')
    def setSuppressErrors(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldReporter, kwargs, 'suppressErrors', val)

    @_f.addMelDocs('cmdScrollFieldReporter', 'suppressInfo')
    def setSuppressInfo(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldReporter, kwargs, 'suppressInfo', val)

    @_f.addMelDocs('cmdScrollFieldReporter', 'suppressResults')
    def setSuppressResults(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldReporter, kwargs, 'suppressResults', val)

    @_f.addMelDocs('cmdScrollFieldReporter', 'suppressStackTrace')
    def setSuppressStackTrace(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldReporter, kwargs, 'suppressStackTrace', val)

    @_f.addMelDocs('cmdScrollFieldReporter', 'suppressWarnings')
    def setSuppressWarnings(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldReporter, kwargs, 'suppressWarnings', val)

    @_f.addMelDocs('cmdScrollFieldReporter', 'text')
    def setText(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldReporter, kwargs, 'text', val)

    @_f.addMelDocs('cmdScrollFieldReporter', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldReporter, kwargs, 'visible', val)

    @_f.addMelDocs('cmdScrollFieldReporter', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldReporter, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('cmdScrollFieldReporter', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldReporter, kwargs, 'width', val)

    @_f.addMelDocs('cmdScrollFieldReporter', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.cmdScrollFieldReporter, kwargs, 'statusBarMessage', val)


class CmdShell(PyUI):
    __melcmd__ = staticmethod(windows.cmdShell)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'cmdShell'
    __melui__ = 'cmdShell'
    __slots__ = ()

    @_f.addMelDocs('cmdShell', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.cmdShell, kwargs, 'dragCallback', val)

    @_f.addMelDocs('cmdShell', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.cmdShell, kwargs, 'dropCallback', val)

    @_f.addMelDocs('cmdShell', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.cmdShell, kwargs, 'annotation')
        return res

    @_f.addMelDocs('cmdShell', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.cmdShell, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('cmdShell', 'command')
    def getCommand(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.cmdShell, kwargs, 'command')
        return res

    @_f.addMelDocs('cmdShell', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.cmdShell, kwargs, 'docTag')
        return res

    @_f.addMelDocs('cmdShell', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.cmdShell, kwargs, 'enable')
        return res

    @_f.addMelDocs('cmdShell', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.cmdShell, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('cmdShell', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.cmdShell, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('cmdShell', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.cmdShell, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('cmdShell', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.cmdShell, kwargs, 'height')
        return res

    @_f.addMelDocs('cmdShell', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.cmdShell, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('cmdShell', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.cmdShell, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('cmdShell', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.cmdShell, kwargs, 'manage')
        return res

    @_f.addMelDocs('cmdShell', 'numberOfHistoryLines')
    def getNumberOfHistoryLines(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.cmdShell, kwargs, 'numberOfHistoryLines')
        return res

    @_f.addMelDocs('cmdShell', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.cmdShell, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('cmdShell', 'numberOfSavedLines')
    def getNumberOfSavedLines(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.cmdShell, kwargs, 'numberOfSavedLines')
        return res

    @_f.addMelDocs('cmdShell', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.cmdShell, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('cmdShell', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.cmdShell, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('cmdShell', 'prompt')
    def getPrompt(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.cmdShell, kwargs, 'prompt')
        return res

    @_f.addMelDocs('cmdShell', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.cmdShell, kwargs, 'visible')
        return res

    @_f.addMelDocs('cmdShell', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.cmdShell, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('cmdShell', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.cmdShell, kwargs, 'width')
        return res

    @_f.addMelDocs('cmdShell', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdShell, kwargs, 'noBackground', val)

    @_f.addMelDocs('cmdShell', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.cmdShell, kwargs, 'annotation', val)

    @_f.addMelDocs('cmdShell', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.cmdShell, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('cmdShell', 'command')
    def setCommand(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.cmdShell, kwargs, 'command', val)

    @_f.addMelDocs('cmdShell', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.cmdShell, kwargs, 'docTag', val)

    @_f.addMelDocs('cmdShell', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdShell, kwargs, 'enable', val)

    @_f.addMelDocs('cmdShell', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdShell, kwargs, 'enableBackground', val)

    @_f.addMelDocs('cmdShell', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdShell, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('cmdShell', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.cmdShell, kwargs, 'height', val)

    @_f.addMelDocs('cmdShell', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.cmdShell, kwargs, 'highlightColor', val)

    @_f.addMelDocs('cmdShell', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdShell, kwargs, 'manage', val)

    @_f.addMelDocs('cmdShell', 'numberOfHistoryLines')
    def setNumberOfHistoryLines(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.cmdShell, kwargs, 'numberOfHistoryLines', val)

    @_f.addMelDocs('cmdShell', 'numberOfSavedLines')
    def setNumberOfSavedLines(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.cmdShell, kwargs, 'numberOfSavedLines', val)

    @_f.addMelDocs('cmdShell', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdShell, kwargs, 'preventOverride', val)

    @_f.addMelDocs('cmdShell', 'prompt')
    def setPrompt(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.cmdShell, kwargs, 'prompt', val)

    @_f.addMelDocs('cmdShell', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.cmdShell, kwargs, 'visible', val)

    @_f.addMelDocs('cmdShell', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.cmdShell, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('cmdShell', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.cmdShell, kwargs, 'width', val)

    @_f.addMelDocs('cmdShell', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.cmdShell, kwargs, 'statusBarMessage', val)


class ColorEditor(PyUI):
    __melcmd__ = staticmethod(windows.colorEditor)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'colorEditor'
    __melui__ = 'colorEditor'
    __slots__ = ()

    @_f.addMelDocs('colorEditor', 'alpha')
    def getAlpha(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.colorEditor, kwargs, 'alpha')
        return res

    @_f.addMelDocs('colorEditor', 'hsvValue')
    def getHsvValue(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.colorEditor, kwargs, 'hsvValue')
        return res

    @_f.addMelDocs('colorEditor', 'result')
    def getResult(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.colorEditor, kwargs, 'result')
        return res

    @_f.addMelDocs('colorEditor', 'rgbValue')
    def getRgbValue(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.colorEditor, kwargs, 'rgbValue')
        return res


class CommandLine(PyUI):
    __melcmd__ = staticmethod(windows.commandLine)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'commandLine'
    __melui__ = 'commandLine'
    __slots__ = ()

    @_f.addMelDocs('commandLine', 'command')
    def command(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.commandLine, kwargs, 'command', val)

    @_f.addMelDocs('commandLine', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.commandLine, kwargs, 'dragCallback', val)

    @_f.addMelDocs('commandLine', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.commandLine, kwargs, 'dropCallback', val)

    @_f.addMelDocs('commandLine', 'enterCommand')
    def enterCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.commandLine, kwargs, 'enterCommand', val)

    @_f.addMelDocs('commandLine', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.commandLine, kwargs, 'annotation')
        return res

    @_f.addMelDocs('commandLine', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.commandLine, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('commandLine', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.commandLine, kwargs, 'docTag')
        return res

    @_f.addMelDocs('commandLine', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.commandLine, kwargs, 'enable')
        return res

    @_f.addMelDocs('commandLine', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.commandLine, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('commandLine', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.commandLine, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('commandLine', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.commandLine, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('commandLine', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.commandLine, kwargs, 'height')
        return res

    @_f.addMelDocs('commandLine', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.commandLine, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('commandLine', 'holdFocus')
    def getHoldFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.commandLine, kwargs, 'holdFocus')
        return res

    @_f.addMelDocs('commandLine', 'inputAnnotation')
    def getInputAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.commandLine, kwargs, 'inputAnnotation')
        return res

    @_f.addMelDocs('commandLine', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.commandLine, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('commandLine', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.commandLine, kwargs, 'manage')
        return res

    @_f.addMelDocs('commandLine', 'numberOfHistoryLines')
    def getNumberOfHistoryLines(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.commandLine, kwargs, 'numberOfHistoryLines')
        return res

    @_f.addMelDocs('commandLine', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.commandLine, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('commandLine', 'outputAnnotation')
    def getOutputAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.commandLine, kwargs, 'outputAnnotation')
        return res

    @_f.addMelDocs('commandLine', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.commandLine, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('commandLine', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.commandLine, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('commandLine', 'sourceType')
    def getSourceType(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.commandLine, kwargs, 'sourceType')
        return res

    @_f.addMelDocs('commandLine', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.commandLine, kwargs, 'visible')
        return res

    @_f.addMelDocs('commandLine', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.commandLine, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('commandLine', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.commandLine, kwargs, 'width')
        return res

    @_f.addMelDocs('commandLine', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.commandLine, kwargs, 'noBackground', val)

    @_f.addMelDocs('commandLine', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.commandLine, kwargs, 'annotation', val)

    @_f.addMelDocs('commandLine', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.commandLine, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('commandLine', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.commandLine, kwargs, 'docTag', val)

    @_f.addMelDocs('commandLine', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.commandLine, kwargs, 'enable', val)

    @_f.addMelDocs('commandLine', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.commandLine, kwargs, 'enableBackground', val)

    @_f.addMelDocs('commandLine', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.commandLine, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('commandLine', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.commandLine, kwargs, 'height', val)

    @_f.addMelDocs('commandLine', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.commandLine, kwargs, 'highlightColor', val)

    @_f.addMelDocs('commandLine', 'holdFocus')
    def setHoldFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.commandLine, kwargs, 'holdFocus', val)

    @_f.addMelDocs('commandLine', 'inputAnnotation')
    def setInputAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.commandLine, kwargs, 'inputAnnotation', val)

    @_f.addMelDocs('commandLine', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.commandLine, kwargs, 'manage', val)

    @_f.addMelDocs('commandLine', 'numberOfHistoryLines')
    def setNumberOfHistoryLines(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.commandLine, kwargs, 'numberOfHistoryLines', val)

    @_f.addMelDocs('commandLine', 'outputAnnotation')
    def setOutputAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.commandLine, kwargs, 'outputAnnotation', val)

    @_f.addMelDocs('commandLine', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.commandLine, kwargs, 'preventOverride', val)

    @_f.addMelDocs('commandLine', 'sourceType')
    def setSourceType(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.commandLine, kwargs, 'sourceType', val)

    @_f.addMelDocs('commandLine', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.commandLine, kwargs, 'visible', val)

    @_f.addMelDocs('commandLine', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.commandLine, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('commandLine', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.commandLine, kwargs, 'width', val)

    @_f.addMelDocs('commandLine', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.commandLine, kwargs, 'statusBarMessage', val)


class ConfirmDialog(PyUI):
    __melcmd__ = staticmethod(windows.confirmDialog)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'confirmDialog'
    __melui__ = 'confirmDialog'
    __slots__ = ()


class DefaultLightListCheckBox(PyUI):
    __melcmd__ = staticmethod(rendering.defaultLightListCheckBox)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'defaultLightListCheckBox'
    __melui__ = 'defaultLightListCheckBox'
    __slots__ = ()

    @_f.addMelDocs('defaultLightListCheckBox', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, rendering.defaultLightListCheckBox, kwargs, 'dragCallback', val)

    @_f.addMelDocs('defaultLightListCheckBox', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, rendering.defaultLightListCheckBox, kwargs, 'dropCallback', val)

    @_f.addMelDocs('defaultLightListCheckBox', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, rendering.defaultLightListCheckBox, kwargs, 'annotation')
        return res

    @_f.addMelDocs('defaultLightListCheckBox', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, rendering.defaultLightListCheckBox, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('defaultLightListCheckBox', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, rendering.defaultLightListCheckBox, kwargs, 'docTag')
        return res

    @_f.addMelDocs('defaultLightListCheckBox', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, rendering.defaultLightListCheckBox, kwargs, 'enable')
        return res

    @_f.addMelDocs('defaultLightListCheckBox', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, rendering.defaultLightListCheckBox, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('defaultLightListCheckBox', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, rendering.defaultLightListCheckBox, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('defaultLightListCheckBox', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, rendering.defaultLightListCheckBox, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('defaultLightListCheckBox', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, rendering.defaultLightListCheckBox, kwargs, 'height')
        return res

    @_f.addMelDocs('defaultLightListCheckBox', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, rendering.defaultLightListCheckBox, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('defaultLightListCheckBox', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, rendering.defaultLightListCheckBox, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('defaultLightListCheckBox', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, rendering.defaultLightListCheckBox, kwargs, 'manage')
        return res

    @_f.addMelDocs('defaultLightListCheckBox', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, rendering.defaultLightListCheckBox, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('defaultLightListCheckBox', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, rendering.defaultLightListCheckBox, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('defaultLightListCheckBox', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, rendering.defaultLightListCheckBox, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('defaultLightListCheckBox', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, rendering.defaultLightListCheckBox, kwargs, 'visible')
        return res

    @_f.addMelDocs('defaultLightListCheckBox', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, rendering.defaultLightListCheckBox, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('defaultLightListCheckBox', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, rendering.defaultLightListCheckBox, kwargs, 'width')
        return res

    @_f.addMelDocs('defaultLightListCheckBox', 'label')
    def label(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, rendering.defaultLightListCheckBox, kwargs, 'label', val)

    @_f.addMelDocs('defaultLightListCheckBox', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, rendering.defaultLightListCheckBox, kwargs, 'noBackground', val)

    @_f.addMelDocs('defaultLightListCheckBox', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, rendering.defaultLightListCheckBox, kwargs, 'annotation', val)

    @_f.addMelDocs('defaultLightListCheckBox', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, rendering.defaultLightListCheckBox, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('defaultLightListCheckBox', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, rendering.defaultLightListCheckBox, kwargs, 'docTag', val)

    @_f.addMelDocs('defaultLightListCheckBox', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, rendering.defaultLightListCheckBox, kwargs, 'enable', val)

    @_f.addMelDocs('defaultLightListCheckBox', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, rendering.defaultLightListCheckBox, kwargs, 'enableBackground', val)

    @_f.addMelDocs('defaultLightListCheckBox', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, rendering.defaultLightListCheckBox, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('defaultLightListCheckBox', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, rendering.defaultLightListCheckBox, kwargs, 'height', val)

    @_f.addMelDocs('defaultLightListCheckBox', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, rendering.defaultLightListCheckBox, kwargs, 'highlightColor', val)

    @_f.addMelDocs('defaultLightListCheckBox', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, rendering.defaultLightListCheckBox, kwargs, 'manage', val)

    @_f.addMelDocs('defaultLightListCheckBox', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, rendering.defaultLightListCheckBox, kwargs, 'preventOverride', val)

    @_f.addMelDocs('defaultLightListCheckBox', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, rendering.defaultLightListCheckBox, kwargs, 'visible', val)

    @_f.addMelDocs('defaultLightListCheckBox', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, rendering.defaultLightListCheckBox, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('defaultLightListCheckBox', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, rendering.defaultLightListCheckBox, kwargs, 'width', val)

    @_f.addMelDocs('defaultLightListCheckBox', 'shadingGroup')
    def shadingGroup(self, val=True, **kwargs):
        # type: (_general.PyNode | str, **Any) -> None
        return _f.asEdit(self, rendering.defaultLightListCheckBox, kwargs, 'shadingGroup', val)

    @_f.addMelDocs('defaultLightListCheckBox', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, rendering.defaultLightListCheckBox, kwargs, 'statusBarMessage', val)


class DeviceEditor(PyUI):
    __melcmd__ = staticmethod(system.deviceEditor)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'deviceEditor'
    __melui__ = 'deviceEditor'
    __slots__ = ()

    @_f.addMelDocs('deviceEditor', 'control')
    def getControl(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, system.deviceEditor, kwargs, 'control')
        return res

    @_f.addMelDocs('deviceEditor', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, system.deviceEditor, kwargs, 'docTag')
        return res

    @_f.addMelDocs('deviceEditor', 'filter')
    def getFilter(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, system.deviceEditor, kwargs, 'filter')
        return res

    @_f.addMelDocs('deviceEditor', 'forceMainConnection')
    def getForceMainConnection(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, system.deviceEditor, kwargs, 'forceMainConnection')
        return res

    @_f.addMelDocs('deviceEditor', 'highlightConnection')
    def getHighlightConnection(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, system.deviceEditor, kwargs, 'highlightConnection')
        return res

    @_f.addMelDocs('deviceEditor', 'mainListConnection')
    def getMainListConnection(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, system.deviceEditor, kwargs, 'mainListConnection')
        return res

    @_f.addMelDocs('deviceEditor', 'panel')
    def getPanel(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, system.deviceEditor, kwargs, 'panel')
        return res

    @_f.addMelDocs('deviceEditor', 'selectionConnection')
    def getSelectionConnection(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, system.deviceEditor, kwargs, 'selectionConnection')
        return res

    @_f.addMelDocs('deviceEditor', 'stateString')
    def getStateString(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, system.deviceEditor, kwargs, 'stateString')
        return res

    @_f.addMelDocs('deviceEditor', 'takePath')
    def getTakePath(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, system.deviceEditor, kwargs, 'takePath')
        return res

    @_f.addMelDocs('deviceEditor', 'lockMainConnection')
    def lockMainConnection(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, system.deviceEditor, kwargs, 'lockMainConnection', val)

    @_f.addMelDocs('deviceEditor', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, system.deviceEditor, kwargs, 'docTag', val)

    @_f.addMelDocs('deviceEditor', 'filter')
    def setFilter(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, system.deviceEditor, kwargs, 'filter', val)

    @_f.addMelDocs('deviceEditor', 'forceMainConnection')
    def setForceMainConnection(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, system.deviceEditor, kwargs, 'forceMainConnection', val)

    @_f.addMelDocs('deviceEditor', 'highlightConnection')
    def setHighlightConnection(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, system.deviceEditor, kwargs, 'highlightConnection', val)

    @_f.addMelDocs('deviceEditor', 'mainListConnection')
    def setMainListConnection(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, system.deviceEditor, kwargs, 'mainListConnection', val)

    @_f.addMelDocs('deviceEditor', 'parent')
    def setParent(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, system.deviceEditor, kwargs, 'parent', val)

    @_f.addMelDocs('deviceEditor', 'selectionConnection')
    def setSelectionConnection(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, system.deviceEditor, kwargs, 'selectionConnection', val)

    @_f.addMelDocs('deviceEditor', 'takePath')
    def setTakePath(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, system.deviceEditor, kwargs, 'takePath', val)

    @_f.addMelDocs('deviceEditor', 'unParent')
    def unParent(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, system.deviceEditor, kwargs, 'unParent', val)

    @_f.addMelDocs('deviceEditor', 'unlockMainConnection')
    def unlockMainConnection(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, system.deviceEditor, kwargs, 'unlockMainConnection', val)

    @_f.addMelDocs('deviceEditor', 'updateMainConnection')
    def updateMainConnection(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, system.deviceEditor, kwargs, 'updateMainConnection', val)


class DynPaintEditor(PyUI):
    __melcmd__ = staticmethod(effects.dynPaintEditor)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'dynPaintEditor'
    __melui__ = 'dynPaintEditor'
    __slots__ = ()

    @_f.addMelDocs('dynPaintEditor', 'canvasUndo')
    def canvasUndo(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'canvasUndo', val)

    @_f.addMelDocs('dynPaintEditor', 'clear')
    def clear(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'clear', val)

    @_f.addMelDocs('dynPaintEditor', 'activeOnly')
    def getActiveOnly(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, effects.dynPaintEditor, kwargs, 'activeOnly')
        return res

    @_f.addMelDocs('dynPaintEditor', 'autoSave')
    def getAutoSave(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, effects.dynPaintEditor, kwargs, 'autoSave')
        return res

    @_f.addMelDocs('dynPaintEditor', 'camera')
    def getCamera(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, effects.dynPaintEditor, kwargs, 'camera')
        return res

    @_f.addMelDocs('dynPaintEditor', 'canvasMode')
    def getCanvasMode(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, effects.dynPaintEditor, kwargs, 'canvasMode')
        return res

    @_f.addMelDocs('dynPaintEditor', 'changeCommand')
    def getChangeCommand(self, **kwargs):
        # type: (...) -> Tuple[str, str, str, str]
        res = _f.asQuery(self, effects.dynPaintEditor, kwargs, 'changeCommand')
        return res

    @_f.addMelDocs('dynPaintEditor', 'control')
    def getControl(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, effects.dynPaintEditor, kwargs, 'control')
        return res

    @_f.addMelDocs('dynPaintEditor', 'currentCanvasSize')
    def getCurrentCanvasSize(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, effects.dynPaintEditor, kwargs, 'currentCanvasSize')
        return res

    @_f.addMelDocs('dynPaintEditor', 'displayAppearance')
    def getDisplayAppearance(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, effects.dynPaintEditor, kwargs, 'displayAppearance')
        return res

    @_f.addMelDocs('dynPaintEditor', 'displayFog')
    def getDisplayFog(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, effects.dynPaintEditor, kwargs, 'displayFog')
        return res

    @_f.addMelDocs('dynPaintEditor', 'displayImage')
    def getDisplayImage(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, effects.dynPaintEditor, kwargs, 'displayImage')
        return res

    @_f.addMelDocs('dynPaintEditor', 'displayLights')
    def getDisplayLights(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, effects.dynPaintEditor, kwargs, 'displayLights')
        return res

    @_f.addMelDocs('dynPaintEditor', 'displayStyle')
    def getDisplayStyle(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, effects.dynPaintEditor, kwargs, 'displayStyle')
        return res

    @_f.addMelDocs('dynPaintEditor', 'displayTextures')
    def getDisplayTextures(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, effects.dynPaintEditor, kwargs, 'displayTextures')
        return res

    @_f.addMelDocs('dynPaintEditor', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, effects.dynPaintEditor, kwargs, 'docTag')
        return res

    @_f.addMelDocs('dynPaintEditor', 'doubleBuffer')
    def getDoubleBuffer(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, effects.dynPaintEditor, kwargs, 'doubleBuffer')
        return res

    @_f.addMelDocs('dynPaintEditor', 'drawAxis')
    def getDrawAxis(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, effects.dynPaintEditor, kwargs, 'drawAxis')
        return res

    @_f.addMelDocs('dynPaintEditor', 'drawContext')
    def getDrawContext(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, effects.dynPaintEditor, kwargs, 'drawContext')
        return res

    @_f.addMelDocs('dynPaintEditor', 'fileName')
    def getFileName(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, effects.dynPaintEditor, kwargs, 'fileName')
        return res

    @_f.addMelDocs('dynPaintEditor', 'filter')
    def getFilter(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, effects.dynPaintEditor, kwargs, 'filter')
        return res

    @_f.addMelDocs('dynPaintEditor', 'forceMainConnection')
    def getForceMainConnection(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, effects.dynPaintEditor, kwargs, 'forceMainConnection')
        return res

    @_f.addMelDocs('dynPaintEditor', 'highlightConnection')
    def getHighlightConnection(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, effects.dynPaintEditor, kwargs, 'highlightConnection')
        return res

    @_f.addMelDocs('dynPaintEditor', 'mainListConnection')
    def getMainListConnection(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, effects.dynPaintEditor, kwargs, 'mainListConnection')
        return res

    @_f.addMelDocs('dynPaintEditor', 'nbImages')
    def getNbImages(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, effects.dynPaintEditor, kwargs, 'nbImages')
        return res

    @_f.addMelDocs('dynPaintEditor', 'newImage')
    def getNewImage(self, **kwargs):
        # type: (...) -> Tuple[int, int, float, float, float]
        res = _f.asQuery(self, effects.dynPaintEditor, kwargs, 'newImage')
        return res

    @_f.addMelDocs('dynPaintEditor', 'panel')
    def getPanel(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, effects.dynPaintEditor, kwargs, 'panel')
        return res

    @_f.addMelDocs('dynPaintEditor', 'refreshMode')
    def getRefreshMode(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, effects.dynPaintEditor, kwargs, 'refreshMode')
        return res

    @_f.addMelDocs('dynPaintEditor', 'saveAlpha')
    def getSaveAlpha(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, effects.dynPaintEditor, kwargs, 'saveAlpha')
        return res

    @_f.addMelDocs('dynPaintEditor', 'saveBumpmap')
    def getSaveBumpmap(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, effects.dynPaintEditor, kwargs, 'saveBumpmap')
        return res

    @_f.addMelDocs('dynPaintEditor', 'scaleBlue')
    def getScaleBlue(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, effects.dynPaintEditor, kwargs, 'scaleBlue')
        return res

    @_f.addMelDocs('dynPaintEditor', 'scaleGreen')
    def getScaleGreen(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, effects.dynPaintEditor, kwargs, 'scaleGreen')
        return res

    @_f.addMelDocs('dynPaintEditor', 'scaleRed')
    def getScaleRed(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, effects.dynPaintEditor, kwargs, 'scaleRed')
        return res

    @_f.addMelDocs('dynPaintEditor', 'selectionConnection')
    def getSelectionConnection(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, effects.dynPaintEditor, kwargs, 'selectionConnection')
        return res

    @_f.addMelDocs('dynPaintEditor', 'singleBuffer')
    def getSingleBuffer(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, effects.dynPaintEditor, kwargs, 'singleBuffer')
        return res

    @_f.addMelDocs('dynPaintEditor', 'stateString')
    def getStateString(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, effects.dynPaintEditor, kwargs, 'stateString')
        return res

    @_f.addMelDocs('dynPaintEditor', 'wrap')
    def getWrap(self, **kwargs):
        # type: (...) -> Tuple[bool, bool]
        res = _f.asQuery(self, effects.dynPaintEditor, kwargs, 'wrap')
        return res

    @_f.addMelDocs('dynPaintEditor', 'zoom')
    def getZoom(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, effects.dynPaintEditor, kwargs, 'zoom')
        return res

    @_f.addMelDocs('dynPaintEditor', 'iconGrab')
    def iconGrab(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'iconGrab', val)

    @_f.addMelDocs('dynPaintEditor', 'loadImage')
    def loadImage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'loadImage', val)

    @_f.addMelDocs('dynPaintEditor', 'lockMainConnection')
    def lockMainConnection(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'lockMainConnection', val)

    @_f.addMelDocs('dynPaintEditor', 'paintAll')
    def paintAll(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'paintAll', val)

    @_f.addMelDocs('dynPaintEditor', 'redrawLast')
    def redrawLast(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'redrawLast', val)

    @_f.addMelDocs('dynPaintEditor', 'refresh')
    def refresh(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'refresh', val)

    @_f.addMelDocs('dynPaintEditor', 'removeAllImages')
    def removeAllImages(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'removeAllImages', val)

    @_f.addMelDocs('dynPaintEditor', 'removeImage')
    def removeImage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'removeImage', val)

    @_f.addMelDocs('dynPaintEditor', 'rollImage')
    def rollImage(self, val=True, **kwargs):
        # type: (Tuple[float, float], **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'rollImage', val)

    @_f.addMelDocs('dynPaintEditor', 'saveImage')
    def saveImage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'saveImage', val)

    @_f.addMelDocs('dynPaintEditor', 'activeOnly')
    def setActiveOnly(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'activeOnly', val)

    @_f.addMelDocs('dynPaintEditor', 'autoSave')
    def setAutoSave(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'autoSave', val)

    @_f.addMelDocs('dynPaintEditor', 'camera')
    def setCamera(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'camera', val)

    @_f.addMelDocs('dynPaintEditor', 'canvasMode')
    def setCanvasMode(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'canvasMode', val)

    @_f.addMelDocs('dynPaintEditor', 'changeCommand')
    def setChangeCommand(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str], **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'changeCommand', val)

    @_f.addMelDocs('dynPaintEditor', 'displayAppearance')
    def setDisplayAppearance(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'displayAppearance', val)

    @_f.addMelDocs('dynPaintEditor', 'displayFog')
    def setDisplayFog(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'displayFog', val)

    @_f.addMelDocs('dynPaintEditor', 'displayImage')
    def setDisplayImage(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'displayImage', val)

    @_f.addMelDocs('dynPaintEditor', 'displayLights')
    def setDisplayLights(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'displayLights', val)

    @_f.addMelDocs('dynPaintEditor', 'displayStyle')
    def setDisplayStyle(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'displayStyle', val)

    @_f.addMelDocs('dynPaintEditor', 'displayTextures')
    def setDisplayTextures(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'displayTextures', val)

    @_f.addMelDocs('dynPaintEditor', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'docTag', val)

    @_f.addMelDocs('dynPaintEditor', 'doubleBuffer')
    def setDoubleBuffer(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'doubleBuffer', val)

    @_f.addMelDocs('dynPaintEditor', 'drawAxis')
    def setDrawAxis(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'drawAxis', val)

    @_f.addMelDocs('dynPaintEditor', 'fileName')
    def setFileName(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'fileName', val)

    @_f.addMelDocs('dynPaintEditor', 'filter')
    def setFilter(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'filter', val)

    @_f.addMelDocs('dynPaintEditor', 'forceMainConnection')
    def setForceMainConnection(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'forceMainConnection', val)

    @_f.addMelDocs('dynPaintEditor', 'highlightConnection')
    def setHighlightConnection(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'highlightConnection', val)

    @_f.addMelDocs('dynPaintEditor', 'mainListConnection')
    def setMainListConnection(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'mainListConnection', val)

    @_f.addMelDocs('dynPaintEditor', 'newImage')
    def setNewImage(self, val=True, **kwargs):
        # type: (Tuple[int, int, float, float, float], **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'newImage', val)

    @_f.addMelDocs('dynPaintEditor', 'parent')
    def setParent(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'parent', val)

    @_f.addMelDocs('dynPaintEditor', 'refreshMode')
    def setRefreshMode(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'refreshMode', val)

    @_f.addMelDocs('dynPaintEditor', 'saveAlpha')
    def setSaveAlpha(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'saveAlpha', val)

    @_f.addMelDocs('dynPaintEditor', 'saveBumpmap')
    def setSaveBumpmap(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'saveBumpmap', val)

    @_f.addMelDocs('dynPaintEditor', 'scaleBlue')
    def setScaleBlue(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'scaleBlue', val)

    @_f.addMelDocs('dynPaintEditor', 'scaleGreen')
    def setScaleGreen(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'scaleGreen', val)

    @_f.addMelDocs('dynPaintEditor', 'scaleRed')
    def setScaleRed(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'scaleRed', val)

    @_f.addMelDocs('dynPaintEditor', 'selectionConnection')
    def setSelectionConnection(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'selectionConnection', val)

    @_f.addMelDocs('dynPaintEditor', 'singleBuffer')
    def setSingleBuffer(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'singleBuffer', val)

    @_f.addMelDocs('dynPaintEditor', 'wrap')
    def setWrap(self, val=True, **kwargs):
        # type: (Tuple[bool | int, bool | int], **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'wrap', val)

    @_f.addMelDocs('dynPaintEditor', 'zoom')
    def setZoom(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'zoom', val)

    @_f.addMelDocs('dynPaintEditor', 'snapShot')
    def snapShot(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'snapShot', val)

    @_f.addMelDocs('dynPaintEditor', 'tileSize')
    def tileSize(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'tileSize', val)

    @_f.addMelDocs('dynPaintEditor', 'unParent')
    def unParent(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'unParent', val)

    @_f.addMelDocs('dynPaintEditor', 'undoCache')
    def undoCache(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'undoCache', val)

    @_f.addMelDocs('dynPaintEditor', 'unlockMainConnection')
    def unlockMainConnection(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'unlockMainConnection', val)

    @_f.addMelDocs('dynPaintEditor', 'updateMainConnection')
    def updateMainConnection(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'updateMainConnection', val)

    @_f.addMelDocs('dynPaintEditor', 'writeImage')
    def writeImage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, effects.dynPaintEditor, kwargs, 'writeImage', val)


class DynRelEditor(PyUI):
    __melui__ = 'dynRelEditor'
    __slots__ = ()


class ExclusiveLightCheckBox(PyUI):
    __melcmd__ = staticmethod(rendering.exclusiveLightCheckBox)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'exclusiveLightCheckBox'
    __melui__ = 'exclusiveLightCheckBox'
    __slots__ = ()

    @_f.addMelDocs('exclusiveLightCheckBox', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, rendering.exclusiveLightCheckBox, kwargs, 'dragCallback', val)

    @_f.addMelDocs('exclusiveLightCheckBox', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, rendering.exclusiveLightCheckBox, kwargs, 'dropCallback', val)

    @_f.addMelDocs('exclusiveLightCheckBox', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, rendering.exclusiveLightCheckBox, kwargs, 'annotation')
        return res

    @_f.addMelDocs('exclusiveLightCheckBox', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, rendering.exclusiveLightCheckBox, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('exclusiveLightCheckBox', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, rendering.exclusiveLightCheckBox, kwargs, 'docTag')
        return res

    @_f.addMelDocs('exclusiveLightCheckBox', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, rendering.exclusiveLightCheckBox, kwargs, 'enable')
        return res

    @_f.addMelDocs('exclusiveLightCheckBox', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, rendering.exclusiveLightCheckBox, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('exclusiveLightCheckBox', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, rendering.exclusiveLightCheckBox, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('exclusiveLightCheckBox', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, rendering.exclusiveLightCheckBox, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('exclusiveLightCheckBox', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, rendering.exclusiveLightCheckBox, kwargs, 'height')
        return res

    @_f.addMelDocs('exclusiveLightCheckBox', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, rendering.exclusiveLightCheckBox, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('exclusiveLightCheckBox', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, rendering.exclusiveLightCheckBox, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('exclusiveLightCheckBox', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, rendering.exclusiveLightCheckBox, kwargs, 'manage')
        return res

    @_f.addMelDocs('exclusiveLightCheckBox', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, rendering.exclusiveLightCheckBox, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('exclusiveLightCheckBox', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, rendering.exclusiveLightCheckBox, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('exclusiveLightCheckBox', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, rendering.exclusiveLightCheckBox, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('exclusiveLightCheckBox', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, rendering.exclusiveLightCheckBox, kwargs, 'visible')
        return res

    @_f.addMelDocs('exclusiveLightCheckBox', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, rendering.exclusiveLightCheckBox, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('exclusiveLightCheckBox', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, rendering.exclusiveLightCheckBox, kwargs, 'width')
        return res

    @_f.addMelDocs('exclusiveLightCheckBox', 'label')
    def label(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, rendering.exclusiveLightCheckBox, kwargs, 'label', val)

    @_f.addMelDocs('exclusiveLightCheckBox', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, rendering.exclusiveLightCheckBox, kwargs, 'noBackground', val)

    @_f.addMelDocs('exclusiveLightCheckBox', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, rendering.exclusiveLightCheckBox, kwargs, 'annotation', val)

    @_f.addMelDocs('exclusiveLightCheckBox', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, rendering.exclusiveLightCheckBox, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('exclusiveLightCheckBox', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, rendering.exclusiveLightCheckBox, kwargs, 'docTag', val)

    @_f.addMelDocs('exclusiveLightCheckBox', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, rendering.exclusiveLightCheckBox, kwargs, 'enable', val)

    @_f.addMelDocs('exclusiveLightCheckBox', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, rendering.exclusiveLightCheckBox, kwargs, 'enableBackground', val)

    @_f.addMelDocs('exclusiveLightCheckBox', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, rendering.exclusiveLightCheckBox, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('exclusiveLightCheckBox', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, rendering.exclusiveLightCheckBox, kwargs, 'height', val)

    @_f.addMelDocs('exclusiveLightCheckBox', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, rendering.exclusiveLightCheckBox, kwargs, 'highlightColor', val)

    @_f.addMelDocs('exclusiveLightCheckBox', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, rendering.exclusiveLightCheckBox, kwargs, 'manage', val)

    @_f.addMelDocs('exclusiveLightCheckBox', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, rendering.exclusiveLightCheckBox, kwargs, 'preventOverride', val)

    @_f.addMelDocs('exclusiveLightCheckBox', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, rendering.exclusiveLightCheckBox, kwargs, 'visible', val)

    @_f.addMelDocs('exclusiveLightCheckBox', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, rendering.exclusiveLightCheckBox, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('exclusiveLightCheckBox', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, rendering.exclusiveLightCheckBox, kwargs, 'width', val)

    @_f.addMelDocs('exclusiveLightCheckBox', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, rendering.exclusiveLightCheckBox, kwargs, 'statusBarMessage', val)


class FloatField(PyUI):
    __melcmd__ = staticmethod(windows.floatField)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'floatField'
    __melui__ = 'floatField'
    __slots__ = ()

    @_f.addMelDocs('floatField', 'changeCommand')
    def changeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.floatField, kwargs, 'changeCommand', val)

    @_f.addMelDocs('floatField', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.floatField, kwargs, 'dragCallback', val)

    @_f.addMelDocs('floatField', 'dragCommand')
    def dragCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.floatField, kwargs, 'dragCommand', val)

    @_f.addMelDocs('floatField', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.floatField, kwargs, 'dropCallback', val)

    @_f.addMelDocs('floatField', 'enterCommand')
    def enterCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.floatField, kwargs, 'enterCommand', val)

    @_f.addMelDocs('floatField', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.floatField, kwargs, 'annotation')
        return res

    @_f.addMelDocs('floatField', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.floatField, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('floatField', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.floatField, kwargs, 'docTag')
        return res

    @_f.addMelDocs('floatField', 'editable')
    def getEditable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatField, kwargs, 'editable')
        return res

    @_f.addMelDocs('floatField', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatField, kwargs, 'enable')
        return res

    @_f.addMelDocs('floatField', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatField, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('floatField', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatField, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('floatField', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.floatField, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('floatField', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.floatField, kwargs, 'height')
        return res

    @_f.addMelDocs('floatField', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.floatField, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('floatField', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.floatField, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('floatField', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatField, kwargs, 'manage')
        return res

    @_f.addMelDocs('floatField', 'maxValue')
    def getMaxValue(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.floatField, kwargs, 'maxValue')
        return res

    @_f.addMelDocs('floatField', 'minValue')
    def getMinValue(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.floatField, kwargs, 'minValue')
        return res

    @_f.addMelDocs('floatField', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.floatField, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('floatField', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.floatField, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('floatField', 'precision')
    def getPrecision(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.floatField, kwargs, 'precision')
        return res

    @_f.addMelDocs('floatField', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatField, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('floatField', 'showTrailingZeros')
    def getShowTrailingZeros(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatField, kwargs, 'showTrailingZeros')
        return res

    @_f.addMelDocs('floatField', 'step')
    def getStep(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.floatField, kwargs, 'step')
        return res

    @_f.addMelDocs('floatField', 'value')
    def getValue(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.floatField, kwargs, 'value')
        return res

    @_f.addMelDocs('floatField', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatField, kwargs, 'visible')
        return res

    @_f.addMelDocs('floatField', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.floatField, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('floatField', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.floatField, kwargs, 'width')
        return res

    @_f.addMelDocs('floatField', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatField, kwargs, 'noBackground', val)

    @_f.addMelDocs('floatField', 'receiveFocusCommand')
    def receiveFocusCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.floatField, kwargs, 'receiveFocusCommand', val)

    @_f.addMelDocs('floatField', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.floatField, kwargs, 'annotation', val)

    @_f.addMelDocs('floatField', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.floatField, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('floatField', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.floatField, kwargs, 'docTag', val)

    @_f.addMelDocs('floatField', 'editable')
    def setEditable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatField, kwargs, 'editable', val)

    @_f.addMelDocs('floatField', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatField, kwargs, 'enable', val)

    @_f.addMelDocs('floatField', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatField, kwargs, 'enableBackground', val)

    @_f.addMelDocs('floatField', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatField, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('floatField', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.floatField, kwargs, 'height', val)

    @_f.addMelDocs('floatField', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.floatField, kwargs, 'highlightColor', val)

    @_f.addMelDocs('floatField', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatField, kwargs, 'manage', val)

    @_f.addMelDocs('floatField', 'maxValue')
    def setMaxValue(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.floatField, kwargs, 'maxValue', val)

    @_f.addMelDocs('floatField', 'minValue')
    def setMinValue(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.floatField, kwargs, 'minValue', val)

    @_f.addMelDocs('floatField', 'precision')
    def setPrecision(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.floatField, kwargs, 'precision', val)

    @_f.addMelDocs('floatField', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatField, kwargs, 'preventOverride', val)

    @_f.addMelDocs('floatField', 'showTrailingZeros')
    def setShowTrailingZeros(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatField, kwargs, 'showTrailingZeros', val)

    @_f.addMelDocs('floatField', 'step')
    def setStep(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.floatField, kwargs, 'step', val)

    @_f.addMelDocs('floatField', 'value')
    def setValue(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.floatField, kwargs, 'value', val)

    @_f.addMelDocs('floatField', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatField, kwargs, 'visible', val)

    @_f.addMelDocs('floatField', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.floatField, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('floatField', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.floatField, kwargs, 'width', val)

    @_f.addMelDocs('floatField', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.floatField, kwargs, 'statusBarMessage', val)


class FloatScrollBar(PyUI):
    __melcmd__ = staticmethod(windows.floatScrollBar)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'floatScrollBar'
    __melui__ = 'floatScrollBar'
    __slots__ = ()

    @_f.addMelDocs('floatScrollBar', 'changeCommand')
    def changeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.floatScrollBar, kwargs, 'changeCommand', val)

    @_f.addMelDocs('floatScrollBar', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.floatScrollBar, kwargs, 'dragCallback', val)

    @_f.addMelDocs('floatScrollBar', 'dragCommand')
    def dragCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.floatScrollBar, kwargs, 'dragCommand', val)

    @_f.addMelDocs('floatScrollBar', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.floatScrollBar, kwargs, 'dropCallback', val)

    @_f.addMelDocs('floatScrollBar', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.floatScrollBar, kwargs, 'annotation')
        return res

    @_f.addMelDocs('floatScrollBar', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.floatScrollBar, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('floatScrollBar', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.floatScrollBar, kwargs, 'docTag')
        return res

    @_f.addMelDocs('floatScrollBar', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatScrollBar, kwargs, 'enable')
        return res

    @_f.addMelDocs('floatScrollBar', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatScrollBar, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('floatScrollBar', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatScrollBar, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('floatScrollBar', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.floatScrollBar, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('floatScrollBar', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.floatScrollBar, kwargs, 'height')
        return res

    @_f.addMelDocs('floatScrollBar', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.floatScrollBar, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('floatScrollBar', 'horizontal')
    def getHorizontal(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.floatScrollBar, kwargs, 'horizontal')
        return res

    @_f.addMelDocs('floatScrollBar', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.floatScrollBar, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('floatScrollBar', 'largeStep')
    def getLargeStep(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.floatScrollBar, kwargs, 'largeStep')
        return res

    @_f.addMelDocs('floatScrollBar', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatScrollBar, kwargs, 'manage')
        return res

    @_f.addMelDocs('floatScrollBar', 'maxValue')
    def getMaxValue(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.floatScrollBar, kwargs, 'maxValue')
        return res

    @_f.addMelDocs('floatScrollBar', 'minValue')
    def getMinValue(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.floatScrollBar, kwargs, 'minValue')
        return res

    @_f.addMelDocs('floatScrollBar', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.floatScrollBar, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('floatScrollBar', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.floatScrollBar, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('floatScrollBar', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatScrollBar, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('floatScrollBar', 'step')
    def getStep(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.floatScrollBar, kwargs, 'step')
        return res

    @_f.addMelDocs('floatScrollBar', 'value')
    def getValue(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.floatScrollBar, kwargs, 'value')
        return res

    @_f.addMelDocs('floatScrollBar', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatScrollBar, kwargs, 'visible')
        return res

    @_f.addMelDocs('floatScrollBar', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.floatScrollBar, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('floatScrollBar', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.floatScrollBar, kwargs, 'width')
        return res

    @_f.addMelDocs('floatScrollBar', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatScrollBar, kwargs, 'noBackground', val)

    @_f.addMelDocs('floatScrollBar', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.floatScrollBar, kwargs, 'annotation', val)

    @_f.addMelDocs('floatScrollBar', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.floatScrollBar, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('floatScrollBar', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.floatScrollBar, kwargs, 'docTag', val)

    @_f.addMelDocs('floatScrollBar', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatScrollBar, kwargs, 'enable', val)

    @_f.addMelDocs('floatScrollBar', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatScrollBar, kwargs, 'enableBackground', val)

    @_f.addMelDocs('floatScrollBar', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatScrollBar, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('floatScrollBar', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.floatScrollBar, kwargs, 'height', val)

    @_f.addMelDocs('floatScrollBar', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.floatScrollBar, kwargs, 'highlightColor', val)

    @_f.addMelDocs('floatScrollBar', 'largeStep')
    def setLargeStep(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.floatScrollBar, kwargs, 'largeStep', val)

    @_f.addMelDocs('floatScrollBar', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatScrollBar, kwargs, 'manage', val)

    @_f.addMelDocs('floatScrollBar', 'maxValue')
    def setMaxValue(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.floatScrollBar, kwargs, 'maxValue', val)

    @_f.addMelDocs('floatScrollBar', 'minValue')
    def setMinValue(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.floatScrollBar, kwargs, 'minValue', val)

    @_f.addMelDocs('floatScrollBar', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatScrollBar, kwargs, 'preventOverride', val)

    @_f.addMelDocs('floatScrollBar', 'step')
    def setStep(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.floatScrollBar, kwargs, 'step', val)

    @_f.addMelDocs('floatScrollBar', 'value')
    def setValue(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.floatScrollBar, kwargs, 'value', val)

    @_f.addMelDocs('floatScrollBar', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatScrollBar, kwargs, 'visible', val)

    @_f.addMelDocs('floatScrollBar', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.floatScrollBar, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('floatScrollBar', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.floatScrollBar, kwargs, 'width', val)

    @_f.addMelDocs('floatScrollBar', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.floatScrollBar, kwargs, 'statusBarMessage', val)


class FloatSlider(PyUI):
    __melcmd__ = staticmethod(windows.floatSlider)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'floatSlider'
    __melui__ = 'floatSlider'
    __slots__ = ()

    @_f.addMelDocs('floatSlider', 'changeCommand')
    def changeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.floatSlider, kwargs, 'changeCommand', val)

    @_f.addMelDocs('floatSlider', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.floatSlider, kwargs, 'dragCallback', val)

    @_f.addMelDocs('floatSlider', 'dragCommand')
    def dragCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.floatSlider, kwargs, 'dragCommand', val)

    @_f.addMelDocs('floatSlider', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.floatSlider, kwargs, 'dropCallback', val)

    @_f.addMelDocs('floatSlider', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.floatSlider, kwargs, 'annotation')
        return res

    @_f.addMelDocs('floatSlider', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.floatSlider, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('floatSlider', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.floatSlider, kwargs, 'docTag')
        return res

    @_f.addMelDocs('floatSlider', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatSlider, kwargs, 'enable')
        return res

    @_f.addMelDocs('floatSlider', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatSlider, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('floatSlider', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatSlider, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('floatSlider', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.floatSlider, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('floatSlider', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.floatSlider, kwargs, 'height')
        return res

    @_f.addMelDocs('floatSlider', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.floatSlider, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('floatSlider', 'horizontal')
    def getHorizontal(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.floatSlider, kwargs, 'horizontal')
        return res

    @_f.addMelDocs('floatSlider', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.floatSlider, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('floatSlider', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatSlider, kwargs, 'manage')
        return res

    @_f.addMelDocs('floatSlider', 'maxValue')
    def getMaxValue(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.floatSlider, kwargs, 'maxValue')
        return res

    @_f.addMelDocs('floatSlider', 'minValue')
    def getMinValue(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.floatSlider, kwargs, 'minValue')
        return res

    @_f.addMelDocs('floatSlider', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.floatSlider, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('floatSlider', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.floatSlider, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('floatSlider', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatSlider, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('floatSlider', 'step')
    def getStep(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.floatSlider, kwargs, 'step')
        return res

    @_f.addMelDocs('floatSlider', 'value')
    def getValue(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.floatSlider, kwargs, 'value')
        return res

    @_f.addMelDocs('floatSlider', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatSlider, kwargs, 'visible')
        return res

    @_f.addMelDocs('floatSlider', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.floatSlider, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('floatSlider', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.floatSlider, kwargs, 'width')
        return res

    @_f.addMelDocs('floatSlider', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatSlider, kwargs, 'noBackground', val)

    @_f.addMelDocs('floatSlider', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.floatSlider, kwargs, 'annotation', val)

    @_f.addMelDocs('floatSlider', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.floatSlider, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('floatSlider', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.floatSlider, kwargs, 'docTag', val)

    @_f.addMelDocs('floatSlider', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatSlider, kwargs, 'enable', val)

    @_f.addMelDocs('floatSlider', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatSlider, kwargs, 'enableBackground', val)

    @_f.addMelDocs('floatSlider', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatSlider, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('floatSlider', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.floatSlider, kwargs, 'height', val)

    @_f.addMelDocs('floatSlider', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.floatSlider, kwargs, 'highlightColor', val)

    @_f.addMelDocs('floatSlider', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatSlider, kwargs, 'manage', val)

    @_f.addMelDocs('floatSlider', 'maxValue')
    def setMaxValue(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.floatSlider, kwargs, 'maxValue', val)

    @_f.addMelDocs('floatSlider', 'minValue')
    def setMinValue(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.floatSlider, kwargs, 'minValue', val)

    @_f.addMelDocs('floatSlider', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatSlider, kwargs, 'preventOverride', val)

    @_f.addMelDocs('floatSlider', 'step')
    def setStep(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.floatSlider, kwargs, 'step', val)

    @_f.addMelDocs('floatSlider', 'value')
    def setValue(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.floatSlider, kwargs, 'value', val)

    @_f.addMelDocs('floatSlider', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatSlider, kwargs, 'visible', val)

    @_f.addMelDocs('floatSlider', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.floatSlider, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('floatSlider', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.floatSlider, kwargs, 'width', val)

    @_f.addMelDocs('floatSlider', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.floatSlider, kwargs, 'statusBarMessage', val)


class FloatSlider2(PyUI):
    __melcmd__ = staticmethod(windows.floatSlider2)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'floatSlider2'
    __melui__ = 'floatSlider2'
    __slots__ = ()

    @_f.addMelDocs('floatSlider2', 'changeCommand1')
    def changeCommand1(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.floatSlider2, kwargs, 'changeCommand1', val)

    @_f.addMelDocs('floatSlider2', 'changeCommand2')
    def changeCommand2(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.floatSlider2, kwargs, 'changeCommand2', val)

    @_f.addMelDocs('floatSlider2', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.floatSlider2, kwargs, 'dragCallback', val)

    @_f.addMelDocs('floatSlider2', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.floatSlider2, kwargs, 'dropCallback', val)

    @_f.addMelDocs('floatSlider2', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.floatSlider2, kwargs, 'annotation')
        return res

    @_f.addMelDocs('floatSlider2', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.floatSlider2, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('floatSlider2', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.floatSlider2, kwargs, 'docTag')
        return res

    @_f.addMelDocs('floatSlider2', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatSlider2, kwargs, 'enable')
        return res

    @_f.addMelDocs('floatSlider2', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatSlider2, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('floatSlider2', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatSlider2, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('floatSlider2', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.floatSlider2, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('floatSlider2', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.floatSlider2, kwargs, 'height')
        return res

    @_f.addMelDocs('floatSlider2', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.floatSlider2, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('floatSlider2', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.floatSlider2, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('floatSlider2', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatSlider2, kwargs, 'manage')
        return res

    @_f.addMelDocs('floatSlider2', 'maximum')
    def getMaximum(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.floatSlider2, kwargs, 'maximum')
        return res

    @_f.addMelDocs('floatSlider2', 'minimum')
    def getMinimum(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.floatSlider2, kwargs, 'minimum')
        return res

    @_f.addMelDocs('floatSlider2', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.floatSlider2, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('floatSlider2', 'polarity')
    def getPolarity(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.floatSlider2, kwargs, 'polarity')
        return res

    @_f.addMelDocs('floatSlider2', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.floatSlider2, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('floatSlider2', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatSlider2, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('floatSlider2', 'value1')
    def getValue1(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.floatSlider2, kwargs, 'value1')
        return res

    @_f.addMelDocs('floatSlider2', 'value2')
    def getValue2(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.floatSlider2, kwargs, 'value2')
        return res

    @_f.addMelDocs('floatSlider2', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatSlider2, kwargs, 'visible')
        return res

    @_f.addMelDocs('floatSlider2', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.floatSlider2, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('floatSlider2', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.floatSlider2, kwargs, 'width')
        return res

    @_f.addMelDocs('floatSlider2', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatSlider2, kwargs, 'noBackground', val)

    @_f.addMelDocs('floatSlider2', 'positionControl1')
    def positionControl1(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.floatSlider2, kwargs, 'positionControl1', val)

    @_f.addMelDocs('floatSlider2', 'positionControl2')
    def positionControl2(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.floatSlider2, kwargs, 'positionControl2', val)

    @_f.addMelDocs('floatSlider2', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.floatSlider2, kwargs, 'annotation', val)

    @_f.addMelDocs('floatSlider2', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.floatSlider2, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('floatSlider2', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.floatSlider2, kwargs, 'docTag', val)

    @_f.addMelDocs('floatSlider2', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatSlider2, kwargs, 'enable', val)

    @_f.addMelDocs('floatSlider2', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatSlider2, kwargs, 'enableBackground', val)

    @_f.addMelDocs('floatSlider2', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatSlider2, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('floatSlider2', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.floatSlider2, kwargs, 'height', val)

    @_f.addMelDocs('floatSlider2', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.floatSlider2, kwargs, 'highlightColor', val)

    @_f.addMelDocs('floatSlider2', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatSlider2, kwargs, 'manage', val)

    @_f.addMelDocs('floatSlider2', 'maximum')
    def setMaximum(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.floatSlider2, kwargs, 'maximum', val)

    @_f.addMelDocs('floatSlider2', 'minimum')
    def setMinimum(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.floatSlider2, kwargs, 'minimum', val)

    @_f.addMelDocs('floatSlider2', 'polarity')
    def setPolarity(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.floatSlider2, kwargs, 'polarity', val)

    @_f.addMelDocs('floatSlider2', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatSlider2, kwargs, 'preventOverride', val)

    @_f.addMelDocs('floatSlider2', 'value1')
    def setValue1(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.floatSlider2, kwargs, 'value1', val)

    @_f.addMelDocs('floatSlider2', 'value2')
    def setValue2(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.floatSlider2, kwargs, 'value2', val)

    @_f.addMelDocs('floatSlider2', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatSlider2, kwargs, 'visible', val)

    @_f.addMelDocs('floatSlider2', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.floatSlider2, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('floatSlider2', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.floatSlider2, kwargs, 'width', val)

    @_f.addMelDocs('floatSlider2', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.floatSlider2, kwargs, 'statusBarMessage', val)

    @_f.addMelDocs('floatSlider2', 'values')
    def values(self, val=True, **kwargs):
        # type: (Tuple[float, float], **Any) -> None
        return _f.asEdit(self, windows.floatSlider2, kwargs, 'values', val)


class FontDialog(PyUI):
    __melcmd__ = staticmethod(windows.fontDialog)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'fontDialog'
    __melui__ = 'fontDialog'
    __slots__ = ()


class GlRenderEditor(PyUI):
    __melcmd__ = staticmethod(rendering.glRenderEditor)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'glRenderEditor'
    __melui__ = 'glRenderEditor'
    __slots__ = ()

    @_f.addMelDocs('glRenderEditor', 'control')
    def getControl(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, rendering.glRenderEditor, kwargs, 'control')
        return res

    @_f.addMelDocs('glRenderEditor', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, rendering.glRenderEditor, kwargs, 'docTag')
        return res

    @_f.addMelDocs('glRenderEditor', 'filter')
    def getFilter(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, rendering.glRenderEditor, kwargs, 'filter')
        return res

    @_f.addMelDocs('glRenderEditor', 'forceMainConnection')
    def getForceMainConnection(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, rendering.glRenderEditor, kwargs, 'forceMainConnection')
        return res

    @_f.addMelDocs('glRenderEditor', 'highlightConnection')
    def getHighlightConnection(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, rendering.glRenderEditor, kwargs, 'highlightConnection')
        return res

    @_f.addMelDocs('glRenderEditor', 'lookThru')
    def getLookThru(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, rendering.glRenderEditor, kwargs, 'lookThru')
        return res

    @_f.addMelDocs('glRenderEditor', 'mainListConnection')
    def getMainListConnection(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, rendering.glRenderEditor, kwargs, 'mainListConnection')
        return res

    @_f.addMelDocs('glRenderEditor', 'panel')
    def getPanel(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, rendering.glRenderEditor, kwargs, 'panel')
        return res

    @_f.addMelDocs('glRenderEditor', 'selectionConnection')
    def getSelectionConnection(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, rendering.glRenderEditor, kwargs, 'selectionConnection')
        return res

    @_f.addMelDocs('glRenderEditor', 'stateString')
    def getStateString(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, rendering.glRenderEditor, kwargs, 'stateString')
        return res

    @_f.addMelDocs('glRenderEditor', 'viewCameraName')
    def getViewCameraName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, rendering.glRenderEditor, kwargs, 'viewCameraName')
        return res

    @_f.addMelDocs('glRenderEditor', 'lockMainConnection')
    def lockMainConnection(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, rendering.glRenderEditor, kwargs, 'lockMainConnection', val)

    @_f.addMelDocs('glRenderEditor', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, rendering.glRenderEditor, kwargs, 'docTag', val)

    @_f.addMelDocs('glRenderEditor', 'filter')
    def setFilter(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, rendering.glRenderEditor, kwargs, 'filter', val)

    @_f.addMelDocs('glRenderEditor', 'forceMainConnection')
    def setForceMainConnection(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, rendering.glRenderEditor, kwargs, 'forceMainConnection', val)

    @_f.addMelDocs('glRenderEditor', 'highlightConnection')
    def setHighlightConnection(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, rendering.glRenderEditor, kwargs, 'highlightConnection', val)

    @_f.addMelDocs('glRenderEditor', 'lookThru')
    def setLookThru(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, rendering.glRenderEditor, kwargs, 'lookThru', val)

    @_f.addMelDocs('glRenderEditor', 'mainListConnection')
    def setMainListConnection(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, rendering.glRenderEditor, kwargs, 'mainListConnection', val)

    @_f.addMelDocs('glRenderEditor', 'parent')
    def setParent(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, rendering.glRenderEditor, kwargs, 'parent', val)

    @_f.addMelDocs('glRenderEditor', 'selectionConnection')
    def setSelectionConnection(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, rendering.glRenderEditor, kwargs, 'selectionConnection', val)

    @_f.addMelDocs('glRenderEditor', 'unParent')
    def unParent(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, rendering.glRenderEditor, kwargs, 'unParent', val)

    @_f.addMelDocs('glRenderEditor', 'unlockMainConnection')
    def unlockMainConnection(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, rendering.glRenderEditor, kwargs, 'unlockMainConnection', val)

    @_f.addMelDocs('glRenderEditor', 'updateMainConnection')
    def updateMainConnection(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, rendering.glRenderEditor, kwargs, 'updateMainConnection', val)


class GradientControl(PyUI):
    __melcmd__ = staticmethod(windows.gradientControl)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'gradientControl'
    __melui__ = 'gradientControl'
    __slots__ = ()

    @_f.addMelDocs('gradientControl', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.gradientControl, kwargs, 'dragCallback', val)

    @_f.addMelDocs('gradientControl', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.gradientControl, kwargs, 'dropCallback', val)

    @_f.addMelDocs('gradientControl', 'adaptiveScaling')
    def getAdaptiveScaling(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.gradientControl, kwargs, 'adaptiveScaling')
        return res

    @_f.addMelDocs('gradientControl', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.gradientControl, kwargs, 'annotation')
        return res

    @_f.addMelDocs('gradientControl', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.gradientControl, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('gradientControl', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.gradientControl, kwargs, 'docTag')
        return res

    @_f.addMelDocs('gradientControl', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.gradientControl, kwargs, 'enable')
        return res

    @_f.addMelDocs('gradientControl', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.gradientControl, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('gradientControl', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.gradientControl, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('gradientControl', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.gradientControl, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('gradientControl', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.gradientControl, kwargs, 'height')
        return res

    @_f.addMelDocs('gradientControl', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.gradientControl, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('gradientControl', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.gradientControl, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('gradientControl', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.gradientControl, kwargs, 'manage')
        return res

    @_f.addMelDocs('gradientControl', 'numberOfControls')
    def getNumberOfControls(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.gradientControl, kwargs, 'numberOfControls')
        return res

    @_f.addMelDocs('gradientControl', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.gradientControl, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('gradientControl', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.gradientControl, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('gradientControl', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.gradientControl, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('gradientControl', 'refreshOnRelease')
    def getRefreshOnRelease(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.gradientControl, kwargs, 'refreshOnRelease')
        return res

    @_f.addMelDocs('gradientControl', 'staticNumberOfControls')
    def getStaticNumberOfControls(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.gradientControl, kwargs, 'staticNumberOfControls')
        return res

    @_f.addMelDocs('gradientControl', 'staticPositions')
    def getStaticPositions(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.gradientControl, kwargs, 'staticPositions')
        return res

    @_f.addMelDocs('gradientControl', 'upperLimitControl')
    def getUpperLimitControl(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.gradientControl, kwargs, 'upperLimitControl')
        return res

    @_f.addMelDocs('gradientControl', 'verticalLayout')
    def getVerticalLayout(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.gradientControl, kwargs, 'verticalLayout')
        return res

    @_f.addMelDocs('gradientControl', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.gradientControl, kwargs, 'visible')
        return res

    @_f.addMelDocs('gradientControl', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.gradientControl, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('gradientControl', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.gradientControl, kwargs, 'width')
        return res

    @_f.addMelDocs('gradientControl', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.gradientControl, kwargs, 'noBackground', val)

    @_f.addMelDocs('gradientControl', 'selectedColorControl')
    def selectedColorControl(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.gradientControl, kwargs, 'selectedColorControl', val)

    @_f.addMelDocs('gradientControl', 'selectedInterpControl')
    def selectedInterpControl(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.gradientControl, kwargs, 'selectedInterpControl', val)

    @_f.addMelDocs('gradientControl', 'selectedPositionControl')
    def selectedPositionControl(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.gradientControl, kwargs, 'selectedPositionControl', val)

    @_f.addMelDocs('gradientControl', 'adaptiveScaling')
    def setAdaptiveScaling(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.gradientControl, kwargs, 'adaptiveScaling', val)

    @_f.addMelDocs('gradientControl', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.gradientControl, kwargs, 'annotation', val)

    @_f.addMelDocs('gradientControl', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.gradientControl, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('gradientControl', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.gradientControl, kwargs, 'docTag', val)

    @_f.addMelDocs('gradientControl', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.gradientControl, kwargs, 'enable', val)

    @_f.addMelDocs('gradientControl', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.gradientControl, kwargs, 'enableBackground', val)

    @_f.addMelDocs('gradientControl', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.gradientControl, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('gradientControl', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.gradientControl, kwargs, 'height', val)

    @_f.addMelDocs('gradientControl', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.gradientControl, kwargs, 'highlightColor', val)

    @_f.addMelDocs('gradientControl', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.gradientControl, kwargs, 'manage', val)

    @_f.addMelDocs('gradientControl', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.gradientControl, kwargs, 'preventOverride', val)

    @_f.addMelDocs('gradientControl', 'refreshOnRelease')
    def setRefreshOnRelease(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.gradientControl, kwargs, 'refreshOnRelease', val)

    @_f.addMelDocs('gradientControl', 'staticNumberOfControls')
    def setStaticNumberOfControls(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.gradientControl, kwargs, 'staticNumberOfControls', val)

    @_f.addMelDocs('gradientControl', 'staticPositions')
    def setStaticPositions(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.gradientControl, kwargs, 'staticPositions', val)

    @_f.addMelDocs('gradientControl', 'upperLimitControl')
    def setUpperLimitControl(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.gradientControl, kwargs, 'upperLimitControl', val)

    @_f.addMelDocs('gradientControl', 'verticalLayout')
    def setVerticalLayout(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.gradientControl, kwargs, 'verticalLayout', val)

    @_f.addMelDocs('gradientControl', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.gradientControl, kwargs, 'visible', val)

    @_f.addMelDocs('gradientControl', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.gradientControl, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('gradientControl', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.gradientControl, kwargs, 'width', val)

    @_f.addMelDocs('gradientControl', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.gradientControl, kwargs, 'statusBarMessage', val)


class GradientControlNoAttr(PyUI):
    __melcmd__ = staticmethod(windows.gradientControlNoAttr)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'gradientControlNoAttr'
    __melui__ = 'gradientControlNoAttr'
    __slots__ = ()

    @_f.addMelDocs('gradientControlNoAttr', 'changeCommand')
    def changeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.gradientControlNoAttr, kwargs, 'changeCommand', val)

    @_f.addMelDocs('gradientControlNoAttr', 'currentKeyChanged')
    def currentKeyChanged(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.gradientControlNoAttr, kwargs, 'currentKeyChanged', val)

    @_f.addMelDocs('gradientControlNoAttr', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.gradientControlNoAttr, kwargs, 'dragCallback', val)

    @_f.addMelDocs('gradientControlNoAttr', 'dragCommand')
    def dragCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.gradientControlNoAttr, kwargs, 'dragCommand', val)

    @_f.addMelDocs('gradientControlNoAttr', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.gradientControlNoAttr, kwargs, 'dropCallback', val)

    @_f.addMelDocs('gradientControlNoAttr', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.gradientControlNoAttr, kwargs, 'annotation')
        return res

    @_f.addMelDocs('gradientControlNoAttr', 'asString')
    def getAsString(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.gradientControlNoAttr, kwargs, 'asString')
        return res

    @_f.addMelDocs('gradientControlNoAttr', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.gradientControlNoAttr, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('gradientControlNoAttr', 'currentKey')
    def getCurrentKey(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.gradientControlNoAttr, kwargs, 'currentKey')
        return res

    @_f.addMelDocs('gradientControlNoAttr', 'currentKeyColorValue')
    def getCurrentKeyColorValue(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.gradientControlNoAttr, kwargs, 'currentKeyColorValue')
        return res

    @_f.addMelDocs('gradientControlNoAttr', 'currentKeyCurveValue')
    def getCurrentKeyCurveValue(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.gradientControlNoAttr, kwargs, 'currentKeyCurveValue')
        return res

    @_f.addMelDocs('gradientControlNoAttr', 'currentKeyInterpValue')
    def getCurrentKeyInterpValue(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.gradientControlNoAttr, kwargs, 'currentKeyInterpValue')
        return res

    @_f.addMelDocs('gradientControlNoAttr', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.gradientControlNoAttr, kwargs, 'docTag')
        return res

    @_f.addMelDocs('gradientControlNoAttr', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.gradientControlNoAttr, kwargs, 'enable')
        return res

    @_f.addMelDocs('gradientControlNoAttr', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.gradientControlNoAttr, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('gradientControlNoAttr', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.gradientControlNoAttr, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('gradientControlNoAttr', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.gradientControlNoAttr, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('gradientControlNoAttr', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.gradientControlNoAttr, kwargs, 'height')
        return res

    @_f.addMelDocs('gradientControlNoAttr', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.gradientControlNoAttr, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('gradientControlNoAttr', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.gradientControlNoAttr, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('gradientControlNoAttr', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.gradientControlNoAttr, kwargs, 'manage')
        return res

    @_f.addMelDocs('gradientControlNoAttr', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.gradientControlNoAttr, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('gradientControlNoAttr', 'optionVar')
    def getOptionVar(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.gradientControlNoAttr, kwargs, 'optionVar')
        return res

    @_f.addMelDocs('gradientControlNoAttr', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.gradientControlNoAttr, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('gradientControlNoAttr', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.gradientControlNoAttr, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('gradientControlNoAttr', 'rampAsColor')
    def getRampAsColor(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.gradientControlNoAttr, kwargs, 'rampAsColor')
        return res

    @_f.addMelDocs('gradientControlNoAttr', 'valueAtPoint')
    def getValueAtPoint(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.gradientControlNoAttr, kwargs, 'valueAtPoint')
        return res

    @_f.addMelDocs('gradientControlNoAttr', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.gradientControlNoAttr, kwargs, 'visible')
        return res

    @_f.addMelDocs('gradientControlNoAttr', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.gradientControlNoAttr, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('gradientControlNoAttr', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.gradientControlNoAttr, kwargs, 'width')
        return res

    @_f.addMelDocs('gradientControlNoAttr', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.gradientControlNoAttr, kwargs, 'noBackground', val)

    @_f.addMelDocs('gradientControlNoAttr', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.gradientControlNoAttr, kwargs, 'annotation', val)

    @_f.addMelDocs('gradientControlNoAttr', 'asString')
    def setAsString(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.gradientControlNoAttr, kwargs, 'asString', val)

    @_f.addMelDocs('gradientControlNoAttr', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.gradientControlNoAttr, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('gradientControlNoAttr', 'currentKey')
    def setCurrentKey(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.gradientControlNoAttr, kwargs, 'currentKey', val)

    @_f.addMelDocs('gradientControlNoAttr', 'currentKeyColorValue')
    def setCurrentKeyColorValue(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.gradientControlNoAttr, kwargs, 'currentKeyColorValue', val)

    @_f.addMelDocs('gradientControlNoAttr', 'currentKeyCurveValue')
    def setCurrentKeyCurveValue(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.gradientControlNoAttr, kwargs, 'currentKeyCurveValue', val)

    @_f.addMelDocs('gradientControlNoAttr', 'currentKeyInterpValue')
    def setCurrentKeyInterpValue(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.gradientControlNoAttr, kwargs, 'currentKeyInterpValue', val)

    @_f.addMelDocs('gradientControlNoAttr', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.gradientControlNoAttr, kwargs, 'docTag', val)

    @_f.addMelDocs('gradientControlNoAttr', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.gradientControlNoAttr, kwargs, 'enable', val)

    @_f.addMelDocs('gradientControlNoAttr', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.gradientControlNoAttr, kwargs, 'enableBackground', val)

    @_f.addMelDocs('gradientControlNoAttr', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.gradientControlNoAttr, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('gradientControlNoAttr', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.gradientControlNoAttr, kwargs, 'height', val)

    @_f.addMelDocs('gradientControlNoAttr', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.gradientControlNoAttr, kwargs, 'highlightColor', val)

    @_f.addMelDocs('gradientControlNoAttr', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.gradientControlNoAttr, kwargs, 'manage', val)

    @_f.addMelDocs('gradientControlNoAttr', 'optionVar')
    def setOptionVar(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.gradientControlNoAttr, kwargs, 'optionVar', val)

    @_f.addMelDocs('gradientControlNoAttr', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.gradientControlNoAttr, kwargs, 'preventOverride', val)

    @_f.addMelDocs('gradientControlNoAttr', 'rampAsColor')
    def setRampAsColor(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.gradientControlNoAttr, kwargs, 'rampAsColor', val)

    @_f.addMelDocs('gradientControlNoAttr', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.gradientControlNoAttr, kwargs, 'visible', val)

    @_f.addMelDocs('gradientControlNoAttr', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.gradientControlNoAttr, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('gradientControlNoAttr', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.gradientControlNoAttr, kwargs, 'width', val)

    @_f.addMelDocs('gradientControlNoAttr', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.gradientControlNoAttr, kwargs, 'statusBarMessage', val)


class HelpLine(PyUI):
    __melcmd__ = staticmethod(windows.helpLine)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'helpLine'
    __melui__ = 'helpLine'
    __slots__ = ()

    @_f.addMelDocs('helpLine', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.helpLine, kwargs, 'dragCallback', val)

    @_f.addMelDocs('helpLine', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.helpLine, kwargs, 'dropCallback', val)

    @_f.addMelDocs('helpLine', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.helpLine, kwargs, 'annotation')
        return res

    @_f.addMelDocs('helpLine', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.helpLine, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('helpLine', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.helpLine, kwargs, 'docTag')
        return res

    @_f.addMelDocs('helpLine', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.helpLine, kwargs, 'enable')
        return res

    @_f.addMelDocs('helpLine', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.helpLine, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('helpLine', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.helpLine, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('helpLine', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.helpLine, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('helpLine', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.helpLine, kwargs, 'height')
        return res

    @_f.addMelDocs('helpLine', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.helpLine, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('helpLine', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.helpLine, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('helpLine', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.helpLine, kwargs, 'manage')
        return res

    @_f.addMelDocs('helpLine', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.helpLine, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('helpLine', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.helpLine, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('helpLine', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.helpLine, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('helpLine', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.helpLine, kwargs, 'visible')
        return res

    @_f.addMelDocs('helpLine', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.helpLine, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('helpLine', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.helpLine, kwargs, 'width')
        return res

    @_f.addMelDocs('helpLine', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.helpLine, kwargs, 'noBackground', val)

    @_f.addMelDocs('helpLine', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.helpLine, kwargs, 'annotation', val)

    @_f.addMelDocs('helpLine', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.helpLine, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('helpLine', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.helpLine, kwargs, 'docTag', val)

    @_f.addMelDocs('helpLine', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.helpLine, kwargs, 'enable', val)

    @_f.addMelDocs('helpLine', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.helpLine, kwargs, 'enableBackground', val)

    @_f.addMelDocs('helpLine', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.helpLine, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('helpLine', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.helpLine, kwargs, 'height', val)

    @_f.addMelDocs('helpLine', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.helpLine, kwargs, 'highlightColor', val)

    @_f.addMelDocs('helpLine', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.helpLine, kwargs, 'manage', val)

    @_f.addMelDocs('helpLine', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.helpLine, kwargs, 'preventOverride', val)

    @_f.addMelDocs('helpLine', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.helpLine, kwargs, 'visible', val)

    @_f.addMelDocs('helpLine', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.helpLine, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('helpLine', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.helpLine, kwargs, 'width', val)

    @_f.addMelDocs('helpLine', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.helpLine, kwargs, 'statusBarMessage', val)


class HyperGraph(PyUI):
    __melcmd__ = staticmethod(windows.hyperGraph)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'hyperGraph'
    __melui__ = 'hyperGraph'
    __slots__ = ()

    @_f.addMelDocs('hyperGraph', 'addBookmark')
    def addBookmark(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'addBookmark', val)

    @_f.addMelDocs('hyperGraph', 'addDependGraph')
    def addDependGraph(self, val=True, **kwargs):
        # type: (_general.PyNode | str, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'addDependGraph', val)

    @_f.addMelDocs('hyperGraph', 'addDependNode')
    def addDependNode(self, val=True, **kwargs):
        # type: (_general.PyNode | str, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'addDependNode', val)

    @_f.addMelDocs('hyperGraph', 'attributeEditor')
    def attributeEditor(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'attributeEditor', val)

    @_f.addMelDocs('hyperGraph', 'backward')
    def backward(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'backward', val)

    @_f.addMelDocs('hyperGraph', 'clear')
    def clear(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'clear', val)

    @_f.addMelDocs('hyperGraph', 'collapseContainer')
    def collapseContainer(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'collapseContainer', val)

    @_f.addMelDocs('hyperGraph', 'connectionDrawStyle')
    def connectionDrawStyle(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'connectionDrawStyle', val)

    @_f.addMelDocs('hyperGraph', 'debug')
    def debug(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'debug', val)

    @_f.addMelDocs('hyperGraph', 'deleteBookmark')
    def deleteBookmark(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'deleteBookmark', val)

    @_f.addMelDocs('hyperGraph', 'dependGraph')
    def dependGraph(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'dependGraph', val)

    @_f.addMelDocs('hyperGraph', 'dependNode')
    def dependNode(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'dependNode', val)

    @_f.addMelDocs('hyperGraph', 'directoryPressCommand')
    def directoryPressCommand(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'directoryPressCommand', val)

    @_f.addMelDocs('hyperGraph', 'down')
    def down(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'down', val)

    @_f.addMelDocs('hyperGraph', 'downstream')
    def downstream(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'downstream', val)

    @_f.addMelDocs('hyperGraph', 'dragAndDropBehaviorCommand')
    def dragAndDropBehaviorCommand(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'dragAndDropBehaviorCommand', val)

    @_f.addMelDocs('hyperGraph', 'edgeDblClickCommand')
    def edgeDblClickCommand(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'edgeDblClickCommand', val)

    @_f.addMelDocs('hyperGraph', 'edgeDimmedDblClickCommand')
    def edgeDimmedDblClickCommand(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'edgeDimmedDblClickCommand', val)

    @_f.addMelDocs('hyperGraph', 'edgeDropCommand')
    def edgeDropCommand(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'edgeDropCommand', val)

    @_f.addMelDocs('hyperGraph', 'edgePressCommand')
    def edgePressCommand(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'edgePressCommand', val)

    @_f.addMelDocs('hyperGraph', 'edgeReleaseCommand')
    def edgeReleaseCommand(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'edgeReleaseCommand', val)

    @_f.addMelDocs('hyperGraph', 'enableAutomaticLayout')
    def enableAutomaticLayout(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'enableAutomaticLayout', val)

    @_f.addMelDocs('hyperGraph', 'expandContainer')
    def expandContainer(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'expandContainer', val)

    @_f.addMelDocs('hyperGraph', 'filterDetail')
    def filterDetail(self, val=True, **kwargs):
        # type: (Tuple[str, bool | int], **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'filterDetail', val)

    @_f.addMelDocs('hyperGraph', 'focusCommand')
    def focusCommand(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'focusCommand', val)

    @_f.addMelDocs('hyperGraph', 'fold')
    def fold(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'fold', val)

    @_f.addMelDocs('hyperGraph', 'forceRefresh')
    def forceRefresh(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'forceRefresh', val)

    @_f.addMelDocs('hyperGraph', 'forward')
    def forward(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'forward', val)

    @_f.addMelDocs('hyperGraph', 'frame')
    def frame(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'frame', val)

    @_f.addMelDocs('hyperGraph', 'frameBranch')
    def frameBranch(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'frameBranch', val)

    @_f.addMelDocs('hyperGraph', 'frameGraph')
    def frameGraph(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'frameGraph', val)

    @_f.addMelDocs('hyperGraph', 'frameGraphNoRebuild')
    def frameGraphNoRebuild(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'frameGraphNoRebuild', val)

    @_f.addMelDocs('hyperGraph', 'frameHierarchy')
    def frameHierarchy(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'frameHierarchy', val)

    @_f.addMelDocs('hyperGraph', 'animateTransition')
    def getAnimateTransition(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'animateTransition')
        return res

    @_f.addMelDocs('hyperGraph', 'bookmarkName')
    def getBookmarkName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'bookmarkName')
        return res

    @_f.addMelDocs('hyperGraph', 'breakConnectionCommand')
    def getBreakConnectionCommand(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'breakConnectionCommand')
        return res

    @_f.addMelDocs('hyperGraph', 'control')
    def getControl(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'control')
        return res

    @_f.addMelDocs('hyperGraph', 'currentEdge')
    def getCurrentEdge(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'currentEdge')
        return res

    @_f.addMelDocs('hyperGraph', 'currentNode')
    def getCurrentNode(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'currentNode')
        return res

    @_f.addMelDocs('hyperGraph', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'docTag')
        return res

    @_f.addMelDocs('hyperGraph', 'dropNode')
    def getDropNode(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'dropNode')
        return res

    @_f.addMelDocs('hyperGraph', 'dropTargetNode')
    def getDropTargetNode(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'dropTargetNode')
        return res

    @_f.addMelDocs('hyperGraph', 'feedbackGadget')
    def getFeedbackGadget(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'feedbackGadget')
        return res

    @_f.addMelDocs('hyperGraph', 'feedbackNode')
    def getFeedbackNode(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'feedbackNode')
        return res

    @_f.addMelDocs('hyperGraph', 'filter')
    def getFilter(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'filter')
        return res

    @_f.addMelDocs('hyperGraph', 'forceMainConnection')
    def getForceMainConnection(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'forceMainConnection')
        return res

    @_f.addMelDocs('hyperGraph', 'freeform')
    def getFreeform(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'freeform')
        return res

    @_f.addMelDocs('hyperGraph', 'fromAttr')
    def getFromAttr(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'fromAttr')
        return res

    @_f.addMelDocs('hyperGraph', 'fromNode')
    def getFromNode(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'fromNode')
        return res

    @_f.addMelDocs('hyperGraph', 'getNodeList')
    def getGetNodeList(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'getNodeList')
        return res

    @_f.addMelDocs('hyperGraph', 'getNodePosition')
    def getGetNodePosition(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'getNodePosition')
        return res

    @_f.addMelDocs('hyperGraph', 'graphLayoutStyle')
    def getGraphLayoutStyle(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'graphLayoutStyle')
        return res

    @_f.addMelDocs('hyperGraph', 'graphType')
    def getGraphType(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'graphType')
        return res

    @_f.addMelDocs('hyperGraph', 'heatMapDisplay')
    def getHeatMapDisplay(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'heatMapDisplay')
        return res

    @_f.addMelDocs('hyperGraph', 'highlightConnection')
    def getHighlightConnection(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'highlightConnection')
        return res

    @_f.addMelDocs('hyperGraph', 'iconSize')
    def getIconSize(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'iconSize')
        return res

    @_f.addMelDocs('hyperGraph', 'image')
    def getImage(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'image')
        return res

    @_f.addMelDocs('hyperGraph', 'imageEnabled')
    def getImageEnabled(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'imageEnabled')
        return res

    @_f.addMelDocs('hyperGraph', 'imageForContainer')
    def getImageForContainer(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'imageForContainer')
        return res

    @_f.addMelDocs('hyperGraph', 'imagePosition')
    def getImagePosition(self, **kwargs):
        # type: (...) -> Tuple[float, float]
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'imagePosition')
        return res

    @_f.addMelDocs('hyperGraph', 'imageScale')
    def getImageScale(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'imageScale')
        return res

    @_f.addMelDocs('hyperGraph', 'isHotkeyTarget')
    def getIsHotkeyTarget(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'isHotkeyTarget')
        return res

    @_f.addMelDocs('hyperGraph', 'mainListConnection')
    def getMainListConnection(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'mainListConnection')
        return res

    @_f.addMelDocs('hyperGraph', 'mergeConnections')
    def getMergeConnections(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'mergeConnections')
        return res

    @_f.addMelDocs('hyperGraph', 'opaqueContainers')
    def getOpaqueContainers(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'opaqueContainers')
        return res

    @_f.addMelDocs('hyperGraph', 'orientation')
    def getOrientation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'orientation')
        return res

    @_f.addMelDocs('hyperGraph', 'panel')
    def getPanel(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'panel')
        return res

    @_f.addMelDocs('hyperGraph', 'range')
    def getRange(self, **kwargs):
        # type: (...) -> Tuple[float, float]
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'range')
        return res

    @_f.addMelDocs('hyperGraph', 'selectionConnection')
    def getSelectionConnection(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'selectionConnection')
        return res

    @_f.addMelDocs('hyperGraph', 'showConnectionFromSelected')
    def getShowConnectionFromSelected(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'showConnectionFromSelected')
        return res

    @_f.addMelDocs('hyperGraph', 'showConnectionToSelected')
    def getShowConnectionToSelected(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'showConnectionToSelected')
        return res

    @_f.addMelDocs('hyperGraph', 'showConstraints')
    def getShowConstraints(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'showConstraints')
        return res

    @_f.addMelDocs('hyperGraph', 'showDeformers')
    def getShowDeformers(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'showDeformers')
        return res

    @_f.addMelDocs('hyperGraph', 'showExpressions')
    def getShowExpressions(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'showExpressions')
        return res

    @_f.addMelDocs('hyperGraph', 'showInvisible')
    def getShowInvisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'showInvisible')
        return res

    @_f.addMelDocs('hyperGraph', 'showRelationships')
    def getShowRelationships(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'showRelationships')
        return res

    @_f.addMelDocs('hyperGraph', 'showShapes')
    def getShowShapes(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'showShapes')
        return res

    @_f.addMelDocs('hyperGraph', 'showUnderworld')
    def getShowUnderworld(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'showUnderworld')
        return res

    @_f.addMelDocs('hyperGraph', 'stateString')
    def getStateString(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'stateString')
        return res

    @_f.addMelDocs('hyperGraph', 'toAttr')
    def getToAttr(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'toAttr')
        return res

    @_f.addMelDocs('hyperGraph', 'toNode')
    def getToNode(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'toNode')
        return res

    @_f.addMelDocs('hyperGraph', 'transitionFrames')
    def getTransitionFrames(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'transitionFrames')
        return res

    @_f.addMelDocs('hyperGraph', 'updateNodeAdded')
    def getUpdateNodeAdded(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'updateNodeAdded')
        return res

    @_f.addMelDocs('hyperGraph', 'updateSelection')
    def getUpdateSelection(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'updateSelection')
        return res

    @_f.addMelDocs('hyperGraph', 'useFeedbackList')
    def getUseFeedbackList(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'useFeedbackList')
        return res

    @_f.addMelDocs('hyperGraph', 'viewOption')
    def getViewOption(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.hyperGraph, kwargs, 'viewOption')
        return res

    @_f.addMelDocs('hyperGraph', 'graphDescription')
    def graphDescription(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'graphDescription', val)

    @_f.addMelDocs('hyperGraph', 'initializeScript')
    def initializeScript(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'initializeScript', val)

    @_f.addMelDocs('hyperGraph', 'layout')
    def layout(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'layout', val)

    @_f.addMelDocs('hyperGraph', 'layoutSelected')
    def layoutSelected(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'layoutSelected', val)

    @_f.addMelDocs('hyperGraph', 'limitGraphTraversal')
    def limitGraphTraversal(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'limitGraphTraversal', val)

    @_f.addMelDocs('hyperGraph', 'lockMainConnection')
    def lockMainConnection(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'lockMainConnection', val)

    @_f.addMelDocs('hyperGraph', 'look')
    def look(self, val=True, **kwargs):
        # type: (Tuple[float, float], **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'look', val)

    @_f.addMelDocs('hyperGraph', 'navigateHome')
    def navigateHome(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'navigateHome', val)

    @_f.addMelDocs('hyperGraph', 'navup')
    def navup(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'navup', val)

    @_f.addMelDocs('hyperGraph', 'newInputConnection')
    def newInputConnection(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'newInputConnection', val)

    @_f.addMelDocs('hyperGraph', 'newOutputConnection')
    def newOutputConnection(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'newOutputConnection', val)

    @_f.addMelDocs('hyperGraph', 'nextView')
    def nextView(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'nextView', val)

    @_f.addMelDocs('hyperGraph', 'nodeConnectCommand')
    def nodeConnectCommand(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'nodeConnectCommand', val)

    @_f.addMelDocs('hyperGraph', 'nodeDblClickCommand')
    def nodeDblClickCommand(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'nodeDblClickCommand', val)

    @_f.addMelDocs('hyperGraph', 'nodeDropCommand')
    def nodeDropCommand(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'nodeDropCommand', val)

    @_f.addMelDocs('hyperGraph', 'nodeMenuCommand')
    def nodeMenuCommand(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'nodeMenuCommand', val)

    @_f.addMelDocs('hyperGraph', 'nodePressCommand')
    def nodePressCommand(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'nodePressCommand', val)

    @_f.addMelDocs('hyperGraph', 'nodeReleaseCommand')
    def nodeReleaseCommand(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'nodeReleaseCommand', val)

    @_f.addMelDocs('hyperGraph', 'panView')
    def panView(self, val=True, **kwargs):
        # type: (Tuple[float, float], **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'panView', val)

    @_f.addMelDocs('hyperGraph', 'popupMenuScript')
    def popupMenuScript(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'popupMenuScript', val)

    @_f.addMelDocs('hyperGraph', 'previousView')
    def previousView(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'previousView', val)

    @_f.addMelDocs('hyperGraph', 'rebuild')
    def rebuild(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'rebuild', val)

    @_f.addMelDocs('hyperGraph', 'removeNode')
    def removeNode(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'removeNode', val)

    @_f.addMelDocs('hyperGraph', 'resetFreeform')
    def resetFreeform(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'resetFreeform', val)

    @_f.addMelDocs('hyperGraph', 'restoreBookmark')
    def restoreBookmark(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'restoreBookmark', val)

    @_f.addMelDocs('hyperGraph', 'scrollUpDownNoZoom')
    def scrollUpDownNoZoom(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'scrollUpDownNoZoom', val)

    @_f.addMelDocs('hyperGraph', 'animateTransition')
    def setAnimateTransition(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'animateTransition', val)

    @_f.addMelDocs('hyperGraph', 'breakConnectionCommand')
    def setBreakConnectionCommand(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'breakConnectionCommand', val)

    @_f.addMelDocs('hyperGraph', 'currentEdge')
    def setCurrentEdge(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'currentEdge', val)

    @_f.addMelDocs('hyperGraph', 'currentNode')
    def setCurrentNode(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'currentNode', val)

    @_f.addMelDocs('hyperGraph', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'docTag', val)

    @_f.addMelDocs('hyperGraph', 'filter')
    def setFilter(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'filter', val)

    @_f.addMelDocs('hyperGraph', 'forceMainConnection')
    def setForceMainConnection(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'forceMainConnection', val)

    @_f.addMelDocs('hyperGraph', 'freeform')
    def setFreeform(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'freeform', val)

    @_f.addMelDocs('hyperGraph', 'graphLayoutStyle')
    def setGraphLayoutStyle(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'graphLayoutStyle', val)

    @_f.addMelDocs('hyperGraph', 'heatMapDisplay')
    def setHeatMapDisplay(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'heatMapDisplay', val)

    @_f.addMelDocs('hyperGraph', 'highlightConnection')
    def setHighlightConnection(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'highlightConnection', val)

    @_f.addMelDocs('hyperGraph', 'iconSize')
    def setIconSize(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'iconSize', val)

    @_f.addMelDocs('hyperGraph', 'image')
    def setImage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'image', val)

    @_f.addMelDocs('hyperGraph', 'imageEnabled')
    def setImageEnabled(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'imageEnabled', val)

    @_f.addMelDocs('hyperGraph', 'imageForContainer')
    def setImageForContainer(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'imageForContainer', val)

    @_f.addMelDocs('hyperGraph', 'imagePosition')
    def setImagePosition(self, val=True, **kwargs):
        # type: (Tuple[float, float], **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'imagePosition', val)

    @_f.addMelDocs('hyperGraph', 'imageScale')
    def setImageScale(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'imageScale', val)

    @_f.addMelDocs('hyperGraph', 'mainListConnection')
    def setMainListConnection(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'mainListConnection', val)

    @_f.addMelDocs('hyperGraph', 'mergeConnections')
    def setMergeConnections(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'mergeConnections', val)

    @_f.addMelDocs('hyperGraph', 'setNodePosition')
    def setNodePosition(self, val=True, **kwargs):
        # type: (Tuple[str, float, float], **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'setNodePosition', val)

    @_f.addMelDocs('hyperGraph', 'opaqueContainers')
    def setOpaqueContainers(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'opaqueContainers', val)

    @_f.addMelDocs('hyperGraph', 'orientation')
    def setOrientation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'orientation', val)

    @_f.addMelDocs('hyperGraph', 'parent')
    def setParent(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'parent', val)

    @_f.addMelDocs('hyperGraph', 'range')
    def setRange(self, val=True, **kwargs):
        # type: (Tuple[float, float], **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'range', val)

    @_f.addMelDocs('hyperGraph', 'selectionConnection')
    def setSelectionConnection(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'selectionConnection', val)

    @_f.addMelDocs('hyperGraph', 'showConnectionFromSelected')
    def setShowConnectionFromSelected(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'showConnectionFromSelected', val)

    @_f.addMelDocs('hyperGraph', 'showConnectionToSelected')
    def setShowConnectionToSelected(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'showConnectionToSelected', val)

    @_f.addMelDocs('hyperGraph', 'showConstraints')
    def setShowConstraints(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'showConstraints', val)

    @_f.addMelDocs('hyperGraph', 'showDeformers')
    def setShowDeformers(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'showDeformers', val)

    @_f.addMelDocs('hyperGraph', 'showExpressions')
    def setShowExpressions(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'showExpressions', val)

    @_f.addMelDocs('hyperGraph', 'showInvisible')
    def setShowInvisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'showInvisible', val)

    @_f.addMelDocs('hyperGraph', 'showRelationships')
    def setShowRelationships(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'showRelationships', val)

    @_f.addMelDocs('hyperGraph', 'showShapes')
    def setShowShapes(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'showShapes', val)

    @_f.addMelDocs('hyperGraph', 'showUnderworld')
    def setShowUnderworld(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'showUnderworld', val)

    @_f.addMelDocs('hyperGraph', 'transitionFrames')
    def setTransitionFrames(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'transitionFrames', val)

    @_f.addMelDocs('hyperGraph', 'updateNodeAdded')
    def setUpdateNodeAdded(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'updateNodeAdded', val)

    @_f.addMelDocs('hyperGraph', 'updateSelection')
    def setUpdateSelection(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'updateSelection', val)

    @_f.addMelDocs('hyperGraph', 'useFeedbackList')
    def setUseFeedbackList(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'useFeedbackList', val)

    @_f.addMelDocs('hyperGraph', 'viewOption')
    def setViewOption(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'viewOption', val)

    @_f.addMelDocs('hyperGraph', 'showCachedConnections')
    def showCachedConnections(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'showCachedConnections', val)

    @_f.addMelDocs('hyperGraph', 'showConstraintLabels')
    def showConstraintLabels(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'showConstraintLabels', val)

    @_f.addMelDocs('hyperGraph', 'unParent')
    def unParent(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'unParent', val)

    @_f.addMelDocs('hyperGraph', 'unfold')
    def unfold(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'unfold', val)

    @_f.addMelDocs('hyperGraph', 'unfoldAll')
    def unfoldAll(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'unfoldAll', val)

    @_f.addMelDocs('hyperGraph', 'unfoldAllShapes')
    def unfoldAllShapes(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'unfoldAllShapes', val)

    @_f.addMelDocs('hyperGraph', 'unfoldHidden')
    def unfoldHidden(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'unfoldHidden', val)

    @_f.addMelDocs('hyperGraph', 'unlockMainConnection')
    def unlockMainConnection(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'unlockMainConnection', val)

    @_f.addMelDocs('hyperGraph', 'updateMainConnection')
    def updateMainConnection(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'updateMainConnection', val)

    @_f.addMelDocs('hyperGraph', 'upstream')
    def upstream(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'upstream', val)

    @_f.addMelDocs('hyperGraph', 'useDrawOverrideColor')
    def useDrawOverrideColor(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'useDrawOverrideColor', val)

    @_f.addMelDocs('hyperGraph', 'visibility')
    def visibility(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'visibility', val)

    @_f.addMelDocs('hyperGraph', 'zoom')
    def zoom(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.hyperGraph, kwargs, 'zoom', val)


class HyperShade(PyUI):
    __melcmd__ = staticmethod(windows.hyperShade)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'hyperShade'
    __melui__ = 'hyperShade'
    __slots__ = ()


class IconTextButton(PyUI):
    __melcmd__ = staticmethod(windows.iconTextButton)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'iconTextButton'
    __melui__ = 'iconTextButton'
    __slots__ = ()

    @_f.addMelDocs('iconTextButton', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'dragCallback', val)

    @_f.addMelDocs('iconTextButton', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'dropCallback', val)

    @_f.addMelDocs('iconTextButton', 'align')
    def getAlign(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'align')
        return res

    @_f.addMelDocs('iconTextButton', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'annotation')
        return res

    @_f.addMelDocs('iconTextButton', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('iconTextButton', 'command')
    def getCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'command')
        return res

    @_f.addMelDocs('iconTextButton', 'commandRepeatable')
    def getCommandRepeatable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'commandRepeatable')
        return res

    @_f.addMelDocs('iconTextButton', 'disabledImage')
    def getDisabledImage(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'disabledImage')
        return res

    @_f.addMelDocs('iconTextButton', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'docTag')
        return res

    @_f.addMelDocs('iconTextButton', 'doubleClickCommand')
    def getDoubleClickCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'doubleClickCommand')
        return res

    @_f.addMelDocs('iconTextButton', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'enable')
        return res

    @_f.addMelDocs('iconTextButton', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('iconTextButton', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('iconTextButton', 'flat')
    def getFlat(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'flat')
        return res

    @_f.addMelDocs('iconTextButton', 'flipX')
    def getFlipX(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'flipX')
        return res

    @_f.addMelDocs('iconTextButton', 'flipY')
    def getFlipY(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'flipY')
        return res

    @_f.addMelDocs('iconTextButton', 'font')
    def getFont(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'font')
        return res

    @_f.addMelDocs('iconTextButton', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('iconTextButton', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'height')
        return res

    @_f.addMelDocs('iconTextButton', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('iconTextButton', 'highlightImage')
    def getHighlightImage(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'highlightImage')
        return res

    @_f.addMelDocs('iconTextButton', 'image')
    def getImage(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'image')
        return res

    @_f.addMelDocs('iconTextButton', 'image1')
    def getImage1(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'image1')
        return res

    @_f.addMelDocs('iconTextButton', 'image2')
    def getImage2(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'image2')
        return res

    @_f.addMelDocs('iconTextButton', 'image3')
    def getImage3(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'image3')
        return res

    @_f.addMelDocs('iconTextButton', 'imageOverlayLabel')
    def getImageOverlayLabel(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'imageOverlayLabel')
        return res

    @_f.addMelDocs('iconTextButton', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('iconTextButton', 'label')
    def getLabel(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'label')
        return res

    @_f.addMelDocs('iconTextButton', 'labelOffset')
    def getLabelOffset(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'labelOffset')
        return res

    @_f.addMelDocs('iconTextButton', 'ltVersion')
    def getLtVersion(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'ltVersion')
        return res

    @_f.addMelDocs('iconTextButton', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'manage')
        return res

    @_f.addMelDocs('iconTextButton', 'marginHeight')
    def getMarginHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'marginHeight')
        return res

    @_f.addMelDocs('iconTextButton', 'marginWidth')
    def getMarginWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'marginWidth')
        return res

    @_f.addMelDocs('iconTextButton', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('iconTextButton', 'overlayLabelBackColor')
    def getOverlayLabelBackColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float, float]
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'overlayLabelBackColor')
        return res

    @_f.addMelDocs('iconTextButton', 'overlayLabelColor')
    def getOverlayLabelColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'overlayLabelColor')
        return res

    @_f.addMelDocs('iconTextButton', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('iconTextButton', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('iconTextButton', 'rotation')
    def getRotation(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'rotation')
        return res

    @_f.addMelDocs('iconTextButton', 'selectionImage')
    def getSelectionImage(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'selectionImage')
        return res

    @_f.addMelDocs('iconTextButton', 'sourceType')
    def getSourceType(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'sourceType')
        return res

    @_f.addMelDocs('iconTextButton', 'style')
    def getStyle(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'style')
        return res

    @_f.addMelDocs('iconTextButton', 'useAlpha')
    def getUseAlpha(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'useAlpha')
        return res

    @_f.addMelDocs('iconTextButton', 'version')
    def getVersion(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'version')
        return res

    @_f.addMelDocs('iconTextButton', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'visible')
        return res

    @_f.addMelDocs('iconTextButton', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('iconTextButton', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.iconTextButton, kwargs, 'width')
        return res

    @_f.addMelDocs('iconTextButton', 'handleNodeDropCallback')
    def handleNodeDropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'handleNodeDropCallback', val)

    @_f.addMelDocs('iconTextButton', 'labelEditingCallback')
    def labelEditingCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'labelEditingCallback', val)

    @_f.addMelDocs('iconTextButton', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'noBackground', val)

    @_f.addMelDocs('iconTextButton', 'scaleIcon')
    def scaleIcon(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'scaleIcon', val)

    @_f.addMelDocs('iconTextButton', 'align')
    def setAlign(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'align', val)

    @_f.addMelDocs('iconTextButton', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'annotation', val)

    @_f.addMelDocs('iconTextButton', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('iconTextButton', 'command')
    def setCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'command', val)

    @_f.addMelDocs('iconTextButton', 'commandRepeatable')
    def setCommandRepeatable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'commandRepeatable', val)

    @_f.addMelDocs('iconTextButton', 'disabledImage')
    def setDisabledImage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'disabledImage', val)

    @_f.addMelDocs('iconTextButton', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'docTag', val)

    @_f.addMelDocs('iconTextButton', 'doubleClickCommand')
    def setDoubleClickCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'doubleClickCommand', val)

    @_f.addMelDocs('iconTextButton', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'enable', val)

    @_f.addMelDocs('iconTextButton', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'enableBackground', val)

    @_f.addMelDocs('iconTextButton', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('iconTextButton', 'flat')
    def setFlat(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'flat', val)

    @_f.addMelDocs('iconTextButton', 'flipX')
    def setFlipX(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'flipX', val)

    @_f.addMelDocs('iconTextButton', 'flipY')
    def setFlipY(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'flipY', val)

    @_f.addMelDocs('iconTextButton', 'font')
    def setFont(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'font', val)

    @_f.addMelDocs('iconTextButton', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'height', val)

    @_f.addMelDocs('iconTextButton', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'highlightColor', val)

    @_f.addMelDocs('iconTextButton', 'highlightImage')
    def setHighlightImage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'highlightImage', val)

    @_f.addMelDocs('iconTextButton', 'image')
    def setImage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'image', val)

    @_f.addMelDocs('iconTextButton', 'image1')
    def setImage1(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'image1', val)

    @_f.addMelDocs('iconTextButton', 'image2')
    def setImage2(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'image2', val)

    @_f.addMelDocs('iconTextButton', 'image3')
    def setImage3(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'image3', val)

    @_f.addMelDocs('iconTextButton', 'imageOverlayLabel')
    def setImageOverlayLabel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'imageOverlayLabel', val)

    @_f.addMelDocs('iconTextButton', 'label')
    def setLabel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'label', val)

    @_f.addMelDocs('iconTextButton', 'labelOffset')
    def setLabelOffset(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'labelOffset', val)

    @_f.addMelDocs('iconTextButton', 'ltVersion')
    def setLtVersion(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'ltVersion', val)

    @_f.addMelDocs('iconTextButton', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'manage', val)

    @_f.addMelDocs('iconTextButton', 'marginHeight')
    def setMarginHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'marginHeight', val)

    @_f.addMelDocs('iconTextButton', 'marginWidth')
    def setMarginWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'marginWidth', val)

    @_f.addMelDocs('iconTextButton', 'overlayLabelBackColor')
    def setOverlayLabelBackColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float, float], **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'overlayLabelBackColor', val)

    @_f.addMelDocs('iconTextButton', 'overlayLabelColor')
    def setOverlayLabelColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'overlayLabelColor', val)

    @_f.addMelDocs('iconTextButton', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'preventOverride', val)

    @_f.addMelDocs('iconTextButton', 'rotation')
    def setRotation(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'rotation', val)

    @_f.addMelDocs('iconTextButton', 'selectionImage')
    def setSelectionImage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'selectionImage', val)

    @_f.addMelDocs('iconTextButton', 'sourceType')
    def setSourceType(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'sourceType', val)

    @_f.addMelDocs('iconTextButton', 'style')
    def setStyle(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'style', val)

    @_f.addMelDocs('iconTextButton', 'useAlpha')
    def setUseAlpha(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'useAlpha', val)

    @_f.addMelDocs('iconTextButton', 'version')
    def setVersion(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'version', val)

    @_f.addMelDocs('iconTextButton', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'visible', val)

    @_f.addMelDocs('iconTextButton', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('iconTextButton', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'width', val)

    @_f.addMelDocs('iconTextButton', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextButton, kwargs, 'statusBarMessage', val)


class IconTextCheckBox(PyUI):
    __melcmd__ = staticmethod(windows.iconTextCheckBox)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'iconTextCheckBox'
    __melui__ = 'iconTextCheckBox'
    __slots__ = ()

    @_f.addMelDocs('iconTextCheckBox', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'dragCallback', val)

    @_f.addMelDocs('iconTextCheckBox', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'dropCallback', val)

    @_f.addMelDocs('iconTextCheckBox', 'align')
    def getAlign(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'align')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'annotation')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'changeCommand')
    def getChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'changeCommand')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'disabledImage')
    def getDisabledImage(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'disabledImage')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'docTag')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'enable')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'flat')
    def getFlat(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'flat')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'flipX')
    def getFlipX(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'flipX')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'flipY')
    def getFlipY(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'flipY')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'font')
    def getFont(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'font')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'height')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'highlightImage')
    def getHighlightImage(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'highlightImage')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'image')
    def getImage(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'image')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'image1')
    def getImage1(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'image1')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'image2')
    def getImage2(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'image2')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'image3')
    def getImage3(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'image3')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'imageOverlayLabel')
    def getImageOverlayLabel(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'imageOverlayLabel')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'label')
    def getLabel(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'label')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'labelOffset')
    def getLabelOffset(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'labelOffset')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'ltVersion')
    def getLtVersion(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'ltVersion')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'manage')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'marginHeight')
    def getMarginHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'marginHeight')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'marginWidth')
    def getMarginWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'marginWidth')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'offCommand')
    def getOffCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'offCommand')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'onCommand')
    def getOnCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'onCommand')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'overlayLabelBackColor')
    def getOverlayLabelBackColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float, float]
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'overlayLabelBackColor')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'overlayLabelColor')
    def getOverlayLabelColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'overlayLabelColor')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'rotation')
    def getRotation(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'rotation')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'selectionHighlightImage')
    def getSelectionHighlightImage(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'selectionHighlightImage')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'selectionImage')
    def getSelectionImage(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'selectionImage')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'style')
    def getStyle(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'style')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'useAlpha')
    def getUseAlpha(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'useAlpha')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'value')
    def getValue(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'value')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'version')
    def getVersion(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'version')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'visible')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.iconTextCheckBox, kwargs, 'width')
        return res

    @_f.addMelDocs('iconTextCheckBox', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'noBackground', val)

    @_f.addMelDocs('iconTextCheckBox', 'align')
    def setAlign(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'align', val)

    @_f.addMelDocs('iconTextCheckBox', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'annotation', val)

    @_f.addMelDocs('iconTextCheckBox', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('iconTextCheckBox', 'changeCommand')
    def setChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'changeCommand', val)

    @_f.addMelDocs('iconTextCheckBox', 'disabledImage')
    def setDisabledImage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'disabledImage', val)

    @_f.addMelDocs('iconTextCheckBox', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'docTag', val)

    @_f.addMelDocs('iconTextCheckBox', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'enable', val)

    @_f.addMelDocs('iconTextCheckBox', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'enableBackground', val)

    @_f.addMelDocs('iconTextCheckBox', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('iconTextCheckBox', 'flat')
    def setFlat(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'flat', val)

    @_f.addMelDocs('iconTextCheckBox', 'flipX')
    def setFlipX(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'flipX', val)

    @_f.addMelDocs('iconTextCheckBox', 'flipY')
    def setFlipY(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'flipY', val)

    @_f.addMelDocs('iconTextCheckBox', 'font')
    def setFont(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'font', val)

    @_f.addMelDocs('iconTextCheckBox', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'height', val)

    @_f.addMelDocs('iconTextCheckBox', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'highlightColor', val)

    @_f.addMelDocs('iconTextCheckBox', 'highlightImage')
    def setHighlightImage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'highlightImage', val)

    @_f.addMelDocs('iconTextCheckBox', 'image')
    def setImage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'image', val)

    @_f.addMelDocs('iconTextCheckBox', 'image1')
    def setImage1(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'image1', val)

    @_f.addMelDocs('iconTextCheckBox', 'image2')
    def setImage2(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'image2', val)

    @_f.addMelDocs('iconTextCheckBox', 'image3')
    def setImage3(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'image3', val)

    @_f.addMelDocs('iconTextCheckBox', 'imageOverlayLabel')
    def setImageOverlayLabel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'imageOverlayLabel', val)

    @_f.addMelDocs('iconTextCheckBox', 'label')
    def setLabel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'label', val)

    @_f.addMelDocs('iconTextCheckBox', 'labelOffset')
    def setLabelOffset(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'labelOffset', val)

    @_f.addMelDocs('iconTextCheckBox', 'ltVersion')
    def setLtVersion(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'ltVersion', val)

    @_f.addMelDocs('iconTextCheckBox', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'manage', val)

    @_f.addMelDocs('iconTextCheckBox', 'marginHeight')
    def setMarginHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'marginHeight', val)

    @_f.addMelDocs('iconTextCheckBox', 'marginWidth')
    def setMarginWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'marginWidth', val)

    @_f.addMelDocs('iconTextCheckBox', 'offCommand')
    def setOffCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'offCommand', val)

    @_f.addMelDocs('iconTextCheckBox', 'onCommand')
    def setOnCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'onCommand', val)

    @_f.addMelDocs('iconTextCheckBox', 'overlayLabelBackColor')
    def setOverlayLabelBackColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float, float], **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'overlayLabelBackColor', val)

    @_f.addMelDocs('iconTextCheckBox', 'overlayLabelColor')
    def setOverlayLabelColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'overlayLabelColor', val)

    @_f.addMelDocs('iconTextCheckBox', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'preventOverride', val)

    @_f.addMelDocs('iconTextCheckBox', 'rotation')
    def setRotation(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'rotation', val)

    @_f.addMelDocs('iconTextCheckBox', 'selectionHighlightImage')
    def setSelectionHighlightImage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'selectionHighlightImage', val)

    @_f.addMelDocs('iconTextCheckBox', 'selectionImage')
    def setSelectionImage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'selectionImage', val)

    @_f.addMelDocs('iconTextCheckBox', 'style')
    def setStyle(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'style', val)

    @_f.addMelDocs('iconTextCheckBox', 'useAlpha')
    def setUseAlpha(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'useAlpha', val)

    @_f.addMelDocs('iconTextCheckBox', 'value')
    def setValue(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'value', val)

    @_f.addMelDocs('iconTextCheckBox', 'version')
    def setVersion(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'version', val)

    @_f.addMelDocs('iconTextCheckBox', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'visible', val)

    @_f.addMelDocs('iconTextCheckBox', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('iconTextCheckBox', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'width', val)

    @_f.addMelDocs('iconTextCheckBox', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextCheckBox, kwargs, 'statusBarMessage', val)


class IconTextRadioButton(PyUI):
    __melcmd__ = staticmethod(windows.iconTextRadioButton)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'iconTextRadioButton'
    __melui__ = 'iconTextRadioButton'
    __slots__ = ()

    @_f.addMelDocs('iconTextRadioButton', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'dragCallback', val)

    @_f.addMelDocs('iconTextRadioButton', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'dropCallback', val)

    @_f.addMelDocs('iconTextRadioButton', 'align')
    def getAlign(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'align')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'annotation')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'changeCommand')
    def getChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'changeCommand')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'disabledImage')
    def getDisabledImage(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'disabledImage')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'docTag')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'enable')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'flat')
    def getFlat(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'flat')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'flipX')
    def getFlipX(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'flipX')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'flipY')
    def getFlipY(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'flipY')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'font')
    def getFont(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'font')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'height')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'highlightImage')
    def getHighlightImage(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'highlightImage')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'image')
    def getImage(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'image')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'image1')
    def getImage1(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'image1')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'image2')
    def getImage2(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'image2')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'image3')
    def getImage3(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'image3')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'imageOverlayLabel')
    def getImageOverlayLabel(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'imageOverlayLabel')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'label')
    def getLabel(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'label')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'labelOffset')
    def getLabelOffset(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'labelOffset')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'ltVersion')
    def getLtVersion(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'ltVersion')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'manage')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'marginHeight')
    def getMarginHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'marginHeight')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'marginWidth')
    def getMarginWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'marginWidth')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'offCommand')
    def getOffCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'offCommand')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'onCommand')
    def getOnCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'onCommand')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'overlayLabelBackColor')
    def getOverlayLabelBackColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float, float]
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'overlayLabelBackColor')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'overlayLabelColor')
    def getOverlayLabelColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'overlayLabelColor')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'rotation')
    def getRotation(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'rotation')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'select')
    def getSelect(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'select')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'selectionHighlightImage')
    def getSelectionHighlightImage(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'selectionHighlightImage')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'selectionImage')
    def getSelectionImage(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'selectionImage')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'style')
    def getStyle(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'style')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'useAlpha')
    def getUseAlpha(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'useAlpha')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'version')
    def getVersion(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'version')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'visible')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.iconTextRadioButton, kwargs, 'width')
        return res

    @_f.addMelDocs('iconTextRadioButton', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'noBackground', val)

    @_f.addMelDocs('iconTextRadioButton', 'align')
    def setAlign(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'align', val)

    @_f.addMelDocs('iconTextRadioButton', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'annotation', val)

    @_f.addMelDocs('iconTextRadioButton', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('iconTextRadioButton', 'changeCommand')
    def setChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'changeCommand', val)

    @_f.addMelDocs('iconTextRadioButton', 'disabledImage')
    def setDisabledImage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'disabledImage', val)

    @_f.addMelDocs('iconTextRadioButton', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'docTag', val)

    @_f.addMelDocs('iconTextRadioButton', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'enable', val)

    @_f.addMelDocs('iconTextRadioButton', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'enableBackground', val)

    @_f.addMelDocs('iconTextRadioButton', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('iconTextRadioButton', 'flat')
    def setFlat(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'flat', val)

    @_f.addMelDocs('iconTextRadioButton', 'flipX')
    def setFlipX(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'flipX', val)

    @_f.addMelDocs('iconTextRadioButton', 'flipY')
    def setFlipY(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'flipY', val)

    @_f.addMelDocs('iconTextRadioButton', 'font')
    def setFont(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'font', val)

    @_f.addMelDocs('iconTextRadioButton', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'height', val)

    @_f.addMelDocs('iconTextRadioButton', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'highlightColor', val)

    @_f.addMelDocs('iconTextRadioButton', 'highlightImage')
    def setHighlightImage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'highlightImage', val)

    @_f.addMelDocs('iconTextRadioButton', 'image')
    def setImage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'image', val)

    @_f.addMelDocs('iconTextRadioButton', 'image1')
    def setImage1(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'image1', val)

    @_f.addMelDocs('iconTextRadioButton', 'image2')
    def setImage2(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'image2', val)

    @_f.addMelDocs('iconTextRadioButton', 'image3')
    def setImage3(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'image3', val)

    @_f.addMelDocs('iconTextRadioButton', 'imageOverlayLabel')
    def setImageOverlayLabel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'imageOverlayLabel', val)

    @_f.addMelDocs('iconTextRadioButton', 'label')
    def setLabel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'label', val)

    @_f.addMelDocs('iconTextRadioButton', 'labelOffset')
    def setLabelOffset(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'labelOffset', val)

    @_f.addMelDocs('iconTextRadioButton', 'ltVersion')
    def setLtVersion(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'ltVersion', val)

    @_f.addMelDocs('iconTextRadioButton', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'manage', val)

    @_f.addMelDocs('iconTextRadioButton', 'marginHeight')
    def setMarginHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'marginHeight', val)

    @_f.addMelDocs('iconTextRadioButton', 'marginWidth')
    def setMarginWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'marginWidth', val)

    @_f.addMelDocs('iconTextRadioButton', 'offCommand')
    def setOffCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'offCommand', val)

    @_f.addMelDocs('iconTextRadioButton', 'onCommand')
    def setOnCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'onCommand', val)

    @_f.addMelDocs('iconTextRadioButton', 'overlayLabelBackColor')
    def setOverlayLabelBackColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float, float], **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'overlayLabelBackColor', val)

    @_f.addMelDocs('iconTextRadioButton', 'overlayLabelColor')
    def setOverlayLabelColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'overlayLabelColor', val)

    @_f.addMelDocs('iconTextRadioButton', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'preventOverride', val)

    @_f.addMelDocs('iconTextRadioButton', 'rotation')
    def setRotation(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'rotation', val)

    @_f.addMelDocs('iconTextRadioButton', 'select')
    def setSelect(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'select', val)

    @_f.addMelDocs('iconTextRadioButton', 'selectionHighlightImage')
    def setSelectionHighlightImage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'selectionHighlightImage', val)

    @_f.addMelDocs('iconTextRadioButton', 'selectionImage')
    def setSelectionImage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'selectionImage', val)

    @_f.addMelDocs('iconTextRadioButton', 'style')
    def setStyle(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'style', val)

    @_f.addMelDocs('iconTextRadioButton', 'useAlpha')
    def setUseAlpha(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'useAlpha', val)

    @_f.addMelDocs('iconTextRadioButton', 'version')
    def setVersion(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'version', val)

    @_f.addMelDocs('iconTextRadioButton', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'visible', val)

    @_f.addMelDocs('iconTextRadioButton', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('iconTextRadioButton', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'width', val)

    @_f.addMelDocs('iconTextRadioButton', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioButton, kwargs, 'statusBarMessage', val)


class IconTextRadioCollection(PyUI):
    __melcmd__ = staticmethod(windows.iconTextRadioCollection)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'iconTextRadioCollection'
    __melui__ = 'iconTextRadioCollection'
    __slots__ = ()

    @_f.addMelDocs('iconTextRadioCollection', 'disableCommands')
    def disableCommands(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioCollection, kwargs, 'disableCommands', val)

    @_f.addMelDocs('iconTextRadioCollection', 'collectionItemArray')
    def getCollectionItemArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.iconTextRadioCollection, kwargs, 'collectionItemArray')
        return res

    @_f.addMelDocs('iconTextRadioCollection', 'gl')
    def getGl(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.iconTextRadioCollection, kwargs, 'gl')
        return res

    @_f.addMelDocs('iconTextRadioCollection', 'numberOfCollectionItems')
    def getNumberOfCollectionItems(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.iconTextRadioCollection, kwargs, 'numberOfCollectionItems')
        return res

    @_f.addMelDocs('iconTextRadioCollection', 'select')
    def getSelect(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextRadioCollection, kwargs, 'select')
        return res

    @_f.addMelDocs('iconTextRadioCollection', 'select')
    def setSelect(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextRadioCollection, kwargs, 'select', val)


class IconTextScrollList(PyUI):
    __melcmd__ = staticmethod(windows.iconTextScrollList)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'iconTextScrollList'
    __melui__ = 'iconTextScrollList'
    __slots__ = ()

    @_f.addMelDocs('iconTextScrollList', 'append')
    def append(self, val=True, **kwargs):
        # type: (str | List[str], **Any) -> None
        return _f.asEdit(self, windows.iconTextScrollList, kwargs, 'append', val)

    @_f.addMelDocs('iconTextScrollList', 'changeCommand')
    def changeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.iconTextScrollList, kwargs, 'changeCommand', val)

    @_f.addMelDocs('iconTextScrollList', 'deselectAll')
    def deselectAll(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextScrollList, kwargs, 'deselectAll', val)

    @_f.addMelDocs('iconTextScrollList', 'doubleClickCommand')
    def doubleClickCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.iconTextScrollList, kwargs, 'doubleClickCommand', val)

    @_f.addMelDocs('iconTextScrollList', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.iconTextScrollList, kwargs, 'dragCallback', val)

    @_f.addMelDocs('iconTextScrollList', 'dragFeedbackVisible')
    def dragFeedbackVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextScrollList, kwargs, 'dragFeedbackVisible', val)

    @_f.addMelDocs('iconTextScrollList', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.iconTextScrollList, kwargs, 'dropCallback', val)

    @_f.addMelDocs('iconTextScrollList', 'dropRectCallback')
    def dropRectCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.iconTextScrollList, kwargs, 'dropRectCallback', val)

    @_f.addMelDocs('iconTextScrollList', 'editIndexed')
    def editIndexed(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.iconTextScrollList, kwargs, 'editIndexed', val)

    @_f.addMelDocs('iconTextScrollList', 'editable')
    def editable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextScrollList, kwargs, 'editable', val)

    @_f.addMelDocs('iconTextScrollList', 'allowMultiSelection')
    def getAllowMultiSelection(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextScrollList, kwargs, 'allowMultiSelection')
        return res

    @_f.addMelDocs('iconTextScrollList', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextScrollList, kwargs, 'annotation')
        return res

    @_f.addMelDocs('iconTextScrollList', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.iconTextScrollList, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('iconTextScrollList', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextScrollList, kwargs, 'docTag')
        return res

    @_f.addMelDocs('iconTextScrollList', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextScrollList, kwargs, 'enable')
        return res

    @_f.addMelDocs('iconTextScrollList', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextScrollList, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('iconTextScrollList', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextScrollList, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('iconTextScrollList', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.iconTextScrollList, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('iconTextScrollList', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.iconTextScrollList, kwargs, 'height')
        return res

    @_f.addMelDocs('iconTextScrollList', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.iconTextScrollList, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('iconTextScrollList', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.iconTextScrollList, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('iconTextScrollList', 'itemAt')
    def getItemAt(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.iconTextScrollList, kwargs, 'itemAt')
        return res

    @_f.addMelDocs('iconTextScrollList', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextScrollList, kwargs, 'manage')
        return res

    @_f.addMelDocs('iconTextScrollList', 'numberOfIcons')
    def getNumberOfIcons(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.iconTextScrollList, kwargs, 'numberOfIcons')
        return res

    @_f.addMelDocs('iconTextScrollList', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.iconTextScrollList, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('iconTextScrollList', 'numberOfRows')
    def getNumberOfRows(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.iconTextScrollList, kwargs, 'numberOfRows')
        return res

    @_f.addMelDocs('iconTextScrollList', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.iconTextScrollList, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('iconTextScrollList', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextScrollList, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('iconTextScrollList', 'selectIndexedItem')
    def getSelectIndexedItem(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.iconTextScrollList, kwargs, 'selectIndexedItem')
        return res

    @_f.addMelDocs('iconTextScrollList', 'selectItem')
    def getSelectItem(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextScrollList, kwargs, 'selectItem')
        return res

    @_f.addMelDocs('iconTextScrollList', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextScrollList, kwargs, 'visible')
        return res

    @_f.addMelDocs('iconTextScrollList', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.iconTextScrollList, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('iconTextScrollList', 'visualRectAt')
    def getVisualRectAt(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.iconTextScrollList, kwargs, 'visualRectAt')
        return res

    @_f.addMelDocs('iconTextScrollList', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.iconTextScrollList, kwargs, 'width')
        return res

    @_f.addMelDocs('iconTextScrollList', 'itemTextColor')
    def itemTextColor(self, val=True, **kwargs):
        # type: (Tuple[int, float, float, float] | List[Tuple[int, float, float, float]], **Any) -> None
        return _f.asEdit(self, windows.iconTextScrollList, kwargs, 'itemTextColor', val)

    @_f.addMelDocs('iconTextScrollList', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextScrollList, kwargs, 'noBackground', val)

    @_f.addMelDocs('iconTextScrollList', 'removeAll')
    def removeAll(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextScrollList, kwargs, 'removeAll', val)

    @_f.addMelDocs('iconTextScrollList', 'selectCommand')
    def selectCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.iconTextScrollList, kwargs, 'selectCommand', val)

    @_f.addMelDocs('iconTextScrollList', 'allowMultiSelection')
    def setAllowMultiSelection(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextScrollList, kwargs, 'allowMultiSelection', val)

    @_f.addMelDocs('iconTextScrollList', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextScrollList, kwargs, 'annotation', val)

    @_f.addMelDocs('iconTextScrollList', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.iconTextScrollList, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('iconTextScrollList', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextScrollList, kwargs, 'docTag', val)

    @_f.addMelDocs('iconTextScrollList', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextScrollList, kwargs, 'enable', val)

    @_f.addMelDocs('iconTextScrollList', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextScrollList, kwargs, 'enableBackground', val)

    @_f.addMelDocs('iconTextScrollList', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextScrollList, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('iconTextScrollList', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.iconTextScrollList, kwargs, 'height', val)

    @_f.addMelDocs('iconTextScrollList', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.iconTextScrollList, kwargs, 'highlightColor', val)

    @_f.addMelDocs('iconTextScrollList', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextScrollList, kwargs, 'manage', val)

    @_f.addMelDocs('iconTextScrollList', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextScrollList, kwargs, 'preventOverride', val)

    @_f.addMelDocs('iconTextScrollList', 'selectIndexedItem')
    def setSelectIndexedItem(self, val=True, **kwargs):
        # type: (int | List[int], **Any) -> None
        return _f.asEdit(self, windows.iconTextScrollList, kwargs, 'selectIndexedItem', val)

    @_f.addMelDocs('iconTextScrollList', 'selectItem')
    def setSelectItem(self, val=True, **kwargs):
        # type: (str | List[str], **Any) -> None
        return _f.asEdit(self, windows.iconTextScrollList, kwargs, 'selectItem', val)

    @_f.addMelDocs('iconTextScrollList', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextScrollList, kwargs, 'visible', val)

    @_f.addMelDocs('iconTextScrollList', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.iconTextScrollList, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('iconTextScrollList', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.iconTextScrollList, kwargs, 'width', val)

    @_f.addMelDocs('iconTextScrollList', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextScrollList, kwargs, 'statusBarMessage', val)


class IconTextStaticLabel(PyUI):
    __melcmd__ = staticmethod(windows.iconTextStaticLabel)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'iconTextStaticLabel'
    __melui__ = 'iconTextStaticLabel'
    __slots__ = ()

    @_f.addMelDocs('iconTextStaticLabel', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.iconTextStaticLabel, kwargs, 'dragCallback', val)

    @_f.addMelDocs('iconTextStaticLabel', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.iconTextStaticLabel, kwargs, 'dropCallback', val)

    @_f.addMelDocs('iconTextStaticLabel', 'align')
    def getAlign(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextStaticLabel, kwargs, 'align')
        return res

    @_f.addMelDocs('iconTextStaticLabel', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextStaticLabel, kwargs, 'annotation')
        return res

    @_f.addMelDocs('iconTextStaticLabel', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.iconTextStaticLabel, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('iconTextStaticLabel', 'disabledImage')
    def getDisabledImage(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextStaticLabel, kwargs, 'disabledImage')
        return res

    @_f.addMelDocs('iconTextStaticLabel', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextStaticLabel, kwargs, 'docTag')
        return res

    @_f.addMelDocs('iconTextStaticLabel', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextStaticLabel, kwargs, 'enable')
        return res

    @_f.addMelDocs('iconTextStaticLabel', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextStaticLabel, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('iconTextStaticLabel', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextStaticLabel, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('iconTextStaticLabel', 'flipX')
    def getFlipX(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextStaticLabel, kwargs, 'flipX')
        return res

    @_f.addMelDocs('iconTextStaticLabel', 'flipY')
    def getFlipY(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextStaticLabel, kwargs, 'flipY')
        return res

    @_f.addMelDocs('iconTextStaticLabel', 'font')
    def getFont(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextStaticLabel, kwargs, 'font')
        return res

    @_f.addMelDocs('iconTextStaticLabel', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.iconTextStaticLabel, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('iconTextStaticLabel', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.iconTextStaticLabel, kwargs, 'height')
        return res

    @_f.addMelDocs('iconTextStaticLabel', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.iconTextStaticLabel, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('iconTextStaticLabel', 'image')
    def getImage(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextStaticLabel, kwargs, 'image')
        return res

    @_f.addMelDocs('iconTextStaticLabel', 'image1')
    def getImage1(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextStaticLabel, kwargs, 'image1')
        return res

    @_f.addMelDocs('iconTextStaticLabel', 'image2')
    def getImage2(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextStaticLabel, kwargs, 'image2')
        return res

    @_f.addMelDocs('iconTextStaticLabel', 'image3')
    def getImage3(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextStaticLabel, kwargs, 'image3')
        return res

    @_f.addMelDocs('iconTextStaticLabel', 'imageOverlayLabel')
    def getImageOverlayLabel(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextStaticLabel, kwargs, 'imageOverlayLabel')
        return res

    @_f.addMelDocs('iconTextStaticLabel', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.iconTextStaticLabel, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('iconTextStaticLabel', 'label')
    def getLabel(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextStaticLabel, kwargs, 'label')
        return res

    @_f.addMelDocs('iconTextStaticLabel', 'labelOffset')
    def getLabelOffset(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.iconTextStaticLabel, kwargs, 'labelOffset')
        return res

    @_f.addMelDocs('iconTextStaticLabel', 'ltVersion')
    def getLtVersion(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextStaticLabel, kwargs, 'ltVersion')
        return res

    @_f.addMelDocs('iconTextStaticLabel', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextStaticLabel, kwargs, 'manage')
        return res

    @_f.addMelDocs('iconTextStaticLabel', 'marginHeight')
    def getMarginHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.iconTextStaticLabel, kwargs, 'marginHeight')
        return res

    @_f.addMelDocs('iconTextStaticLabel', 'marginWidth')
    def getMarginWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.iconTextStaticLabel, kwargs, 'marginWidth')
        return res

    @_f.addMelDocs('iconTextStaticLabel', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.iconTextStaticLabel, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('iconTextStaticLabel', 'overlayLabelBackColor')
    def getOverlayLabelBackColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float, float]
        res = _f.asQuery(self, windows.iconTextStaticLabel, kwargs, 'overlayLabelBackColor')
        return res

    @_f.addMelDocs('iconTextStaticLabel', 'overlayLabelColor')
    def getOverlayLabelColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.iconTextStaticLabel, kwargs, 'overlayLabelColor')
        return res

    @_f.addMelDocs('iconTextStaticLabel', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.iconTextStaticLabel, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('iconTextStaticLabel', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextStaticLabel, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('iconTextStaticLabel', 'rotation')
    def getRotation(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.iconTextStaticLabel, kwargs, 'rotation')
        return res

    @_f.addMelDocs('iconTextStaticLabel', 'style')
    def getStyle(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextStaticLabel, kwargs, 'style')
        return res

    @_f.addMelDocs('iconTextStaticLabel', 'useAlpha')
    def getUseAlpha(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextStaticLabel, kwargs, 'useAlpha')
        return res

    @_f.addMelDocs('iconTextStaticLabel', 'version')
    def getVersion(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.iconTextStaticLabel, kwargs, 'version')
        return res

    @_f.addMelDocs('iconTextStaticLabel', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.iconTextStaticLabel, kwargs, 'visible')
        return res

    @_f.addMelDocs('iconTextStaticLabel', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.iconTextStaticLabel, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('iconTextStaticLabel', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.iconTextStaticLabel, kwargs, 'width')
        return res

    @_f.addMelDocs('iconTextStaticLabel', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextStaticLabel, kwargs, 'noBackground', val)

    @_f.addMelDocs('iconTextStaticLabel', 'align')
    def setAlign(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextStaticLabel, kwargs, 'align', val)

    @_f.addMelDocs('iconTextStaticLabel', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextStaticLabel, kwargs, 'annotation', val)

    @_f.addMelDocs('iconTextStaticLabel', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.iconTextStaticLabel, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('iconTextStaticLabel', 'disabledImage')
    def setDisabledImage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextStaticLabel, kwargs, 'disabledImage', val)

    @_f.addMelDocs('iconTextStaticLabel', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextStaticLabel, kwargs, 'docTag', val)

    @_f.addMelDocs('iconTextStaticLabel', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextStaticLabel, kwargs, 'enable', val)

    @_f.addMelDocs('iconTextStaticLabel', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextStaticLabel, kwargs, 'enableBackground', val)

    @_f.addMelDocs('iconTextStaticLabel', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextStaticLabel, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('iconTextStaticLabel', 'flipX')
    def setFlipX(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextStaticLabel, kwargs, 'flipX', val)

    @_f.addMelDocs('iconTextStaticLabel', 'flipY')
    def setFlipY(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextStaticLabel, kwargs, 'flipY', val)

    @_f.addMelDocs('iconTextStaticLabel', 'font')
    def setFont(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextStaticLabel, kwargs, 'font', val)

    @_f.addMelDocs('iconTextStaticLabel', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.iconTextStaticLabel, kwargs, 'height', val)

    @_f.addMelDocs('iconTextStaticLabel', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.iconTextStaticLabel, kwargs, 'highlightColor', val)

    @_f.addMelDocs('iconTextStaticLabel', 'image')
    def setImage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextStaticLabel, kwargs, 'image', val)

    @_f.addMelDocs('iconTextStaticLabel', 'image1')
    def setImage1(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextStaticLabel, kwargs, 'image1', val)

    @_f.addMelDocs('iconTextStaticLabel', 'image2')
    def setImage2(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextStaticLabel, kwargs, 'image2', val)

    @_f.addMelDocs('iconTextStaticLabel', 'image3')
    def setImage3(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextStaticLabel, kwargs, 'image3', val)

    @_f.addMelDocs('iconTextStaticLabel', 'imageOverlayLabel')
    def setImageOverlayLabel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextStaticLabel, kwargs, 'imageOverlayLabel', val)

    @_f.addMelDocs('iconTextStaticLabel', 'label')
    def setLabel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextStaticLabel, kwargs, 'label', val)

    @_f.addMelDocs('iconTextStaticLabel', 'labelOffset')
    def setLabelOffset(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.iconTextStaticLabel, kwargs, 'labelOffset', val)

    @_f.addMelDocs('iconTextStaticLabel', 'ltVersion')
    def setLtVersion(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextStaticLabel, kwargs, 'ltVersion', val)

    @_f.addMelDocs('iconTextStaticLabel', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextStaticLabel, kwargs, 'manage', val)

    @_f.addMelDocs('iconTextStaticLabel', 'marginHeight')
    def setMarginHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.iconTextStaticLabel, kwargs, 'marginHeight', val)

    @_f.addMelDocs('iconTextStaticLabel', 'marginWidth')
    def setMarginWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.iconTextStaticLabel, kwargs, 'marginWidth', val)

    @_f.addMelDocs('iconTextStaticLabel', 'overlayLabelBackColor')
    def setOverlayLabelBackColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float, float], **Any) -> None
        return _f.asEdit(self, windows.iconTextStaticLabel, kwargs, 'overlayLabelBackColor', val)

    @_f.addMelDocs('iconTextStaticLabel', 'overlayLabelColor')
    def setOverlayLabelColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.iconTextStaticLabel, kwargs, 'overlayLabelColor', val)

    @_f.addMelDocs('iconTextStaticLabel', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextStaticLabel, kwargs, 'preventOverride', val)

    @_f.addMelDocs('iconTextStaticLabel', 'rotation')
    def setRotation(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.iconTextStaticLabel, kwargs, 'rotation', val)

    @_f.addMelDocs('iconTextStaticLabel', 'style')
    def setStyle(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextStaticLabel, kwargs, 'style', val)

    @_f.addMelDocs('iconTextStaticLabel', 'useAlpha')
    def setUseAlpha(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextStaticLabel, kwargs, 'useAlpha', val)

    @_f.addMelDocs('iconTextStaticLabel', 'version')
    def setVersion(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextStaticLabel, kwargs, 'version', val)

    @_f.addMelDocs('iconTextStaticLabel', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.iconTextStaticLabel, kwargs, 'visible', val)

    @_f.addMelDocs('iconTextStaticLabel', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.iconTextStaticLabel, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('iconTextStaticLabel', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.iconTextStaticLabel, kwargs, 'width', val)

    @_f.addMelDocs('iconTextStaticLabel', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.iconTextStaticLabel, kwargs, 'statusBarMessage', val)


class Image(PyUI):
    __melcmd__ = staticmethod(windows.image)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'image'
    __melui__ = 'image'
    __slots__ = ()

    @_f.addMelDocs('image', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.image, kwargs, 'dragCallback', val)

    @_f.addMelDocs('image', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.image, kwargs, 'dropCallback', val)

    @_f.addMelDocs('image', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.image, kwargs, 'annotation')
        return res

    @_f.addMelDocs('image', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.image, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('image', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.image, kwargs, 'docTag')
        return res

    @_f.addMelDocs('image', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.image, kwargs, 'enable')
        return res

    @_f.addMelDocs('image', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.image, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('image', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.image, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('image', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.image, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('image', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.image, kwargs, 'height')
        return res

    @_f.addMelDocs('image', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.image, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('image', 'image')
    def getImage(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.image, kwargs, 'image')
        return res

    @_f.addMelDocs('image', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.image, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('image', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.image, kwargs, 'manage')
        return res

    @_f.addMelDocs('image', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.image, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('image', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.image, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('image', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.image, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('image', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.image, kwargs, 'visible')
        return res

    @_f.addMelDocs('image', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.image, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('image', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.image, kwargs, 'width')
        return res

    @_f.addMelDocs('image', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.image, kwargs, 'noBackground', val)

    @_f.addMelDocs('image', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.image, kwargs, 'annotation', val)

    @_f.addMelDocs('image', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.image, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('image', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.image, kwargs, 'docTag', val)

    @_f.addMelDocs('image', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.image, kwargs, 'enable', val)

    @_f.addMelDocs('image', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.image, kwargs, 'enableBackground', val)

    @_f.addMelDocs('image', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.image, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('image', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.image, kwargs, 'height', val)

    @_f.addMelDocs('image', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.image, kwargs, 'highlightColor', val)

    @_f.addMelDocs('image', 'image')
    def setImage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.image, kwargs, 'image', val)

    @_f.addMelDocs('image', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.image, kwargs, 'manage', val)

    @_f.addMelDocs('image', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.image, kwargs, 'preventOverride', val)

    @_f.addMelDocs('image', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.image, kwargs, 'visible', val)

    @_f.addMelDocs('image', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.image, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('image', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.image, kwargs, 'width', val)

    @_f.addMelDocs('image', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.image, kwargs, 'statusBarMessage', val)


class IntField(PyUI):
    __melcmd__ = staticmethod(windows.intField)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'intField'
    __melui__ = 'intField'
    __slots__ = ()

    @_f.addMelDocs('intField', 'changeCommand')
    def changeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.intField, kwargs, 'changeCommand', val)

    @_f.addMelDocs('intField', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.intField, kwargs, 'dragCallback', val)

    @_f.addMelDocs('intField', 'dragCommand')
    def dragCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.intField, kwargs, 'dragCommand', val)

    @_f.addMelDocs('intField', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.intField, kwargs, 'dropCallback', val)

    @_f.addMelDocs('intField', 'enterCommand')
    def enterCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.intField, kwargs, 'enterCommand', val)

    @_f.addMelDocs('intField', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.intField, kwargs, 'annotation')
        return res

    @_f.addMelDocs('intField', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.intField, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('intField', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.intField, kwargs, 'docTag')
        return res

    @_f.addMelDocs('intField', 'editable')
    def getEditable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.intField, kwargs, 'editable')
        return res

    @_f.addMelDocs('intField', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.intField, kwargs, 'enable')
        return res

    @_f.addMelDocs('intField', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.intField, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('intField', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.intField, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('intField', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.intField, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('intField', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.intField, kwargs, 'height')
        return res

    @_f.addMelDocs('intField', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.intField, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('intField', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.intField, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('intField', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.intField, kwargs, 'manage')
        return res

    @_f.addMelDocs('intField', 'maxValue')
    def getMaxValue(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.intField, kwargs, 'maxValue')
        return res

    @_f.addMelDocs('intField', 'minValue')
    def getMinValue(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.intField, kwargs, 'minValue')
        return res

    @_f.addMelDocs('intField', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.intField, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('intField', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.intField, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('intField', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.intField, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('intField', 'step')
    def getStep(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.intField, kwargs, 'step')
        return res

    @_f.addMelDocs('intField', 'value')
    def getValue(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.intField, kwargs, 'value')
        return res

    @_f.addMelDocs('intField', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.intField, kwargs, 'visible')
        return res

    @_f.addMelDocs('intField', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.intField, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('intField', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.intField, kwargs, 'width')
        return res

    @_f.addMelDocs('intField', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.intField, kwargs, 'noBackground', val)

    @_f.addMelDocs('intField', 'receiveFocusCommand')
    def receiveFocusCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.intField, kwargs, 'receiveFocusCommand', val)

    @_f.addMelDocs('intField', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.intField, kwargs, 'annotation', val)

    @_f.addMelDocs('intField', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.intField, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('intField', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.intField, kwargs, 'docTag', val)

    @_f.addMelDocs('intField', 'editable')
    def setEditable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.intField, kwargs, 'editable', val)

    @_f.addMelDocs('intField', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.intField, kwargs, 'enable', val)

    @_f.addMelDocs('intField', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.intField, kwargs, 'enableBackground', val)

    @_f.addMelDocs('intField', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.intField, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('intField', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.intField, kwargs, 'height', val)

    @_f.addMelDocs('intField', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.intField, kwargs, 'highlightColor', val)

    @_f.addMelDocs('intField', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.intField, kwargs, 'manage', val)

    @_f.addMelDocs('intField', 'maxValue')
    def setMaxValue(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.intField, kwargs, 'maxValue', val)

    @_f.addMelDocs('intField', 'minValue')
    def setMinValue(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.intField, kwargs, 'minValue', val)

    @_f.addMelDocs('intField', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.intField, kwargs, 'preventOverride', val)

    @_f.addMelDocs('intField', 'step')
    def setStep(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.intField, kwargs, 'step', val)

    @_f.addMelDocs('intField', 'value')
    def setValue(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.intField, kwargs, 'value', val)

    @_f.addMelDocs('intField', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.intField, kwargs, 'visible', val)

    @_f.addMelDocs('intField', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.intField, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('intField', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.intField, kwargs, 'width', val)

    @_f.addMelDocs('intField', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.intField, kwargs, 'statusBarMessage', val)


class IntScrollBar(PyUI):
    __melcmd__ = staticmethod(windows.intScrollBar)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'intScrollBar'
    __melui__ = 'intScrollBar'
    __slots__ = ()

    @_f.addMelDocs('intScrollBar', 'changeCommand')
    def changeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.intScrollBar, kwargs, 'changeCommand', val)

    @_f.addMelDocs('intScrollBar', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.intScrollBar, kwargs, 'dragCallback', val)

    @_f.addMelDocs('intScrollBar', 'dragCommand')
    def dragCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.intScrollBar, kwargs, 'dragCommand', val)

    @_f.addMelDocs('intScrollBar', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.intScrollBar, kwargs, 'dropCallback', val)

    @_f.addMelDocs('intScrollBar', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.intScrollBar, kwargs, 'annotation')
        return res

    @_f.addMelDocs('intScrollBar', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.intScrollBar, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('intScrollBar', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.intScrollBar, kwargs, 'docTag')
        return res

    @_f.addMelDocs('intScrollBar', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.intScrollBar, kwargs, 'enable')
        return res

    @_f.addMelDocs('intScrollBar', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.intScrollBar, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('intScrollBar', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.intScrollBar, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('intScrollBar', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.intScrollBar, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('intScrollBar', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.intScrollBar, kwargs, 'height')
        return res

    @_f.addMelDocs('intScrollBar', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.intScrollBar, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('intScrollBar', 'horizontal')
    def getHorizontal(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.intScrollBar, kwargs, 'horizontal')
        return res

    @_f.addMelDocs('intScrollBar', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.intScrollBar, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('intScrollBar', 'largeStep')
    def getLargeStep(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.intScrollBar, kwargs, 'largeStep')
        return res

    @_f.addMelDocs('intScrollBar', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.intScrollBar, kwargs, 'manage')
        return res

    @_f.addMelDocs('intScrollBar', 'maxValue')
    def getMaxValue(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.intScrollBar, kwargs, 'maxValue')
        return res

    @_f.addMelDocs('intScrollBar', 'minValue')
    def getMinValue(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.intScrollBar, kwargs, 'minValue')
        return res

    @_f.addMelDocs('intScrollBar', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.intScrollBar, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('intScrollBar', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.intScrollBar, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('intScrollBar', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.intScrollBar, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('intScrollBar', 'step')
    def getStep(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.intScrollBar, kwargs, 'step')
        return res

    @_f.addMelDocs('intScrollBar', 'value')
    def getValue(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.intScrollBar, kwargs, 'value')
        return res

    @_f.addMelDocs('intScrollBar', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.intScrollBar, kwargs, 'visible')
        return res

    @_f.addMelDocs('intScrollBar', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.intScrollBar, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('intScrollBar', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.intScrollBar, kwargs, 'width')
        return res

    @_f.addMelDocs('intScrollBar', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.intScrollBar, kwargs, 'noBackground', val)

    @_f.addMelDocs('intScrollBar', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.intScrollBar, kwargs, 'annotation', val)

    @_f.addMelDocs('intScrollBar', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.intScrollBar, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('intScrollBar', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.intScrollBar, kwargs, 'docTag', val)

    @_f.addMelDocs('intScrollBar', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.intScrollBar, kwargs, 'enable', val)

    @_f.addMelDocs('intScrollBar', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.intScrollBar, kwargs, 'enableBackground', val)

    @_f.addMelDocs('intScrollBar', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.intScrollBar, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('intScrollBar', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.intScrollBar, kwargs, 'height', val)

    @_f.addMelDocs('intScrollBar', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.intScrollBar, kwargs, 'highlightColor', val)

    @_f.addMelDocs('intScrollBar', 'largeStep')
    def setLargeStep(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.intScrollBar, kwargs, 'largeStep', val)

    @_f.addMelDocs('intScrollBar', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.intScrollBar, kwargs, 'manage', val)

    @_f.addMelDocs('intScrollBar', 'maxValue')
    def setMaxValue(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.intScrollBar, kwargs, 'maxValue', val)

    @_f.addMelDocs('intScrollBar', 'minValue')
    def setMinValue(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.intScrollBar, kwargs, 'minValue', val)

    @_f.addMelDocs('intScrollBar', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.intScrollBar, kwargs, 'preventOverride', val)

    @_f.addMelDocs('intScrollBar', 'step')
    def setStep(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.intScrollBar, kwargs, 'step', val)

    @_f.addMelDocs('intScrollBar', 'value')
    def setValue(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.intScrollBar, kwargs, 'value', val)

    @_f.addMelDocs('intScrollBar', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.intScrollBar, kwargs, 'visible', val)

    @_f.addMelDocs('intScrollBar', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.intScrollBar, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('intScrollBar', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.intScrollBar, kwargs, 'width', val)

    @_f.addMelDocs('intScrollBar', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.intScrollBar, kwargs, 'statusBarMessage', val)


class IntSlider(PyUI):
    __melcmd__ = staticmethod(windows.intSlider)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'intSlider'
    __melui__ = 'intSlider'
    __slots__ = ()

    @_f.addMelDocs('intSlider', 'changeCommand')
    def changeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.intSlider, kwargs, 'changeCommand', val)

    @_f.addMelDocs('intSlider', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.intSlider, kwargs, 'dragCallback', val)

    @_f.addMelDocs('intSlider', 'dragCommand')
    def dragCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.intSlider, kwargs, 'dragCommand', val)

    @_f.addMelDocs('intSlider', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.intSlider, kwargs, 'dropCallback', val)

    @_f.addMelDocs('intSlider', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.intSlider, kwargs, 'annotation')
        return res

    @_f.addMelDocs('intSlider', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.intSlider, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('intSlider', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.intSlider, kwargs, 'docTag')
        return res

    @_f.addMelDocs('intSlider', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.intSlider, kwargs, 'enable')
        return res

    @_f.addMelDocs('intSlider', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.intSlider, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('intSlider', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.intSlider, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('intSlider', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.intSlider, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('intSlider', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.intSlider, kwargs, 'height')
        return res

    @_f.addMelDocs('intSlider', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.intSlider, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('intSlider', 'horizontal')
    def getHorizontal(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.intSlider, kwargs, 'horizontal')
        return res

    @_f.addMelDocs('intSlider', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.intSlider, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('intSlider', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.intSlider, kwargs, 'manage')
        return res

    @_f.addMelDocs('intSlider', 'maxValue')
    def getMaxValue(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.intSlider, kwargs, 'maxValue')
        return res

    @_f.addMelDocs('intSlider', 'minValue')
    def getMinValue(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.intSlider, kwargs, 'minValue')
        return res

    @_f.addMelDocs('intSlider', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.intSlider, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('intSlider', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.intSlider, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('intSlider', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.intSlider, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('intSlider', 'step')
    def getStep(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.intSlider, kwargs, 'step')
        return res

    @_f.addMelDocs('intSlider', 'value')
    def getValue(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.intSlider, kwargs, 'value')
        return res

    @_f.addMelDocs('intSlider', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.intSlider, kwargs, 'visible')
        return res

    @_f.addMelDocs('intSlider', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.intSlider, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('intSlider', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.intSlider, kwargs, 'width')
        return res

    @_f.addMelDocs('intSlider', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.intSlider, kwargs, 'noBackground', val)

    @_f.addMelDocs('intSlider', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.intSlider, kwargs, 'annotation', val)

    @_f.addMelDocs('intSlider', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.intSlider, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('intSlider', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.intSlider, kwargs, 'docTag', val)

    @_f.addMelDocs('intSlider', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.intSlider, kwargs, 'enable', val)

    @_f.addMelDocs('intSlider', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.intSlider, kwargs, 'enableBackground', val)

    @_f.addMelDocs('intSlider', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.intSlider, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('intSlider', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.intSlider, kwargs, 'height', val)

    @_f.addMelDocs('intSlider', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.intSlider, kwargs, 'highlightColor', val)

    @_f.addMelDocs('intSlider', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.intSlider, kwargs, 'manage', val)

    @_f.addMelDocs('intSlider', 'maxValue')
    def setMaxValue(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.intSlider, kwargs, 'maxValue', val)

    @_f.addMelDocs('intSlider', 'minValue')
    def setMinValue(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.intSlider, kwargs, 'minValue', val)

    @_f.addMelDocs('intSlider', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.intSlider, kwargs, 'preventOverride', val)

    @_f.addMelDocs('intSlider', 'step')
    def setStep(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.intSlider, kwargs, 'step', val)

    @_f.addMelDocs('intSlider', 'value')
    def setValue(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.intSlider, kwargs, 'value', val)

    @_f.addMelDocs('intSlider', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.intSlider, kwargs, 'visible', val)

    @_f.addMelDocs('intSlider', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.intSlider, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('intSlider', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.intSlider, kwargs, 'width', val)

    @_f.addMelDocs('intSlider', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.intSlider, kwargs, 'statusBarMessage', val)


class KeyframeOutliner(PyUI):
    __melcmd__ = staticmethod(animation.keyframeOutliner)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'keyframeOutliner'
    __melui__ = 'keyframeOutliner'
    __slots__ = ()

    @_f.addMelDocs('keyframeOutliner', 'animCurve')
    def animCurve(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.keyframeOutliner, kwargs, 'animCurve', val)

    @_f.addMelDocs('keyframeOutliner', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, animation.keyframeOutliner, kwargs, 'dragCallback', val)

    @_f.addMelDocs('keyframeOutliner', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, animation.keyframeOutliner, kwargs, 'dropCallback', val)

    @_f.addMelDocs('keyframeOutliner', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.keyframeOutliner, kwargs, 'annotation')
        return res

    @_f.addMelDocs('keyframeOutliner', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, animation.keyframeOutliner, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('keyframeOutliner', 'display')
    def getDisplay(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.keyframeOutliner, kwargs, 'display')
        return res

    @_f.addMelDocs('keyframeOutliner', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.keyframeOutliner, kwargs, 'docTag')
        return res

    @_f.addMelDocs('keyframeOutliner', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.keyframeOutliner, kwargs, 'enable')
        return res

    @_f.addMelDocs('keyframeOutliner', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.keyframeOutliner, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('keyframeOutliner', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.keyframeOutliner, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('keyframeOutliner', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.keyframeOutliner, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('keyframeOutliner', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, animation.keyframeOutliner, kwargs, 'height')
        return res

    @_f.addMelDocs('keyframeOutliner', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, animation.keyframeOutliner, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('keyframeOutliner', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.keyframeOutliner, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('keyframeOutliner', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.keyframeOutliner, kwargs, 'manage')
        return res

    @_f.addMelDocs('keyframeOutliner', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.keyframeOutliner, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('keyframeOutliner', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.keyframeOutliner, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('keyframeOutliner', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.keyframeOutliner, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('keyframeOutliner', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.keyframeOutliner, kwargs, 'visible')
        return res

    @_f.addMelDocs('keyframeOutliner', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, animation.keyframeOutliner, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('keyframeOutliner', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, animation.keyframeOutliner, kwargs, 'width')
        return res

    @_f.addMelDocs('keyframeOutliner', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.keyframeOutliner, kwargs, 'noBackground', val)

    @_f.addMelDocs('keyframeOutliner', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.keyframeOutliner, kwargs, 'annotation', val)

    @_f.addMelDocs('keyframeOutliner', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, animation.keyframeOutliner, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('keyframeOutliner', 'display')
    def setDisplay(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.keyframeOutliner, kwargs, 'display', val)

    @_f.addMelDocs('keyframeOutliner', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.keyframeOutliner, kwargs, 'docTag', val)

    @_f.addMelDocs('keyframeOutliner', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.keyframeOutliner, kwargs, 'enable', val)

    @_f.addMelDocs('keyframeOutliner', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.keyframeOutliner, kwargs, 'enableBackground', val)

    @_f.addMelDocs('keyframeOutliner', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.keyframeOutliner, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('keyframeOutliner', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, animation.keyframeOutliner, kwargs, 'height', val)

    @_f.addMelDocs('keyframeOutliner', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, animation.keyframeOutliner, kwargs, 'highlightColor', val)

    @_f.addMelDocs('keyframeOutliner', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.keyframeOutliner, kwargs, 'manage', val)

    @_f.addMelDocs('keyframeOutliner', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.keyframeOutliner, kwargs, 'preventOverride', val)

    @_f.addMelDocs('keyframeOutliner', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.keyframeOutliner, kwargs, 'visible', val)

    @_f.addMelDocs('keyframeOutliner', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, animation.keyframeOutliner, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('keyframeOutliner', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, animation.keyframeOutliner, kwargs, 'width', val)

    @_f.addMelDocs('keyframeOutliner', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.keyframeOutliner, kwargs, 'statusBarMessage', val)


class KeyframeStats(PyUI):
    __melcmd__ = staticmethod(animation.keyframeStats)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'keyframeStats'
    __melui__ = 'keyframeStats'
    __slots__ = ()

    @_f.addMelDocs('keyframeStats', 'adjustableColumn')
    def adjustableColumn(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'adjustableColumn', val)

    @_f.addMelDocs('keyframeStats', 'adjustableColumn2')
    def adjustableColumn2(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'adjustableColumn2', val)

    @_f.addMelDocs('keyframeStats', 'adjustableColumn3')
    def adjustableColumn3(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'adjustableColumn3', val)

    @_f.addMelDocs('keyframeStats', 'adjustableColumn4')
    def adjustableColumn4(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'adjustableColumn4', val)

    @_f.addMelDocs('keyframeStats', 'adjustableColumn5')
    def adjustableColumn5(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'adjustableColumn5', val)

    @_f.addMelDocs('keyframeStats', 'adjustableColumn6')
    def adjustableColumn6(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'adjustableColumn6', val)

    @_f.addMelDocs('keyframeStats', 'columnAlign')
    def columnAlign(self, val=True, **kwargs):
        # type: (Tuple[int, str] | List[Tuple[int, str]], **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'columnAlign', val)

    @_f.addMelDocs('keyframeStats', 'columnAlign2')
    def columnAlign2(self, val=True, **kwargs):
        # type: (Tuple[str, str], **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'columnAlign2', val)

    @_f.addMelDocs('keyframeStats', 'columnAlign3')
    def columnAlign3(self, val=True, **kwargs):
        # type: (Tuple[str, str, str], **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'columnAlign3', val)

    @_f.addMelDocs('keyframeStats', 'columnAlign4')
    def columnAlign4(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str], **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'columnAlign4', val)

    @_f.addMelDocs('keyframeStats', 'columnAlign5')
    def columnAlign5(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'columnAlign5', val)

    @_f.addMelDocs('keyframeStats', 'columnAlign6')
    def columnAlign6(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'columnAlign6', val)

    @_f.addMelDocs('keyframeStats', 'columnAttach')
    def columnAttach(self, val=True, **kwargs):
        # type: (Tuple[int, str, int] | List[Tuple[int, str, int]], **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'columnAttach', val)

    @_f.addMelDocs('keyframeStats', 'columnAttach2')
    def columnAttach2(self, val=True, **kwargs):
        # type: (Tuple[str, str], **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'columnAttach2', val)

    @_f.addMelDocs('keyframeStats', 'columnAttach3')
    def columnAttach3(self, val=True, **kwargs):
        # type: (Tuple[str, str, str], **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'columnAttach3', val)

    @_f.addMelDocs('keyframeStats', 'columnAttach4')
    def columnAttach4(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str], **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'columnAttach4', val)

    @_f.addMelDocs('keyframeStats', 'columnAttach5')
    def columnAttach5(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'columnAttach5', val)

    @_f.addMelDocs('keyframeStats', 'columnAttach6')
    def columnAttach6(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'columnAttach6', val)

    @_f.addMelDocs('keyframeStats', 'columnOffset2')
    def columnOffset2(self, val=True, **kwargs):
        # type: (Tuple[int, int], **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'columnOffset2', val)

    @_f.addMelDocs('keyframeStats', 'columnOffset3')
    def columnOffset3(self, val=True, **kwargs):
        # type: (Tuple[int, int, int], **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'columnOffset3', val)

    @_f.addMelDocs('keyframeStats', 'columnOffset4')
    def columnOffset4(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int], **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'columnOffset4', val)

    @_f.addMelDocs('keyframeStats', 'columnOffset5')
    def columnOffset5(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'columnOffset5', val)

    @_f.addMelDocs('keyframeStats', 'columnOffset6')
    def columnOffset6(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'columnOffset6', val)

    @_f.addMelDocs('keyframeStats', 'columnWidth')
    def columnWidth(self, val=True, **kwargs):
        # type: (Tuple[int, int] | List[Tuple[int, int]], **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'columnWidth', val)

    @_f.addMelDocs('keyframeStats', 'columnWidth1')
    def columnWidth1(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'columnWidth1', val)

    @_f.addMelDocs('keyframeStats', 'columnWidth2')
    def columnWidth2(self, val=True, **kwargs):
        # type: (Tuple[int, int], **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'columnWidth2', val)

    @_f.addMelDocs('keyframeStats', 'columnWidth3')
    def columnWidth3(self, val=True, **kwargs):
        # type: (Tuple[int, int, int], **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'columnWidth3', val)

    @_f.addMelDocs('keyframeStats', 'columnWidth4')
    def columnWidth4(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int], **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'columnWidth4', val)

    @_f.addMelDocs('keyframeStats', 'columnWidth5')
    def columnWidth5(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'columnWidth5', val)

    @_f.addMelDocs('keyframeStats', 'columnWidth6')
    def columnWidth6(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'columnWidth6', val)

    @_f.addMelDocs('keyframeStats', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'dragCallback', val)

    @_f.addMelDocs('keyframeStats', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'dropCallback', val)

    @_f.addMelDocs('keyframeStats', 'animEditor')
    def getAnimEditor(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.keyframeStats, kwargs, 'animEditor')
        return res

    @_f.addMelDocs('keyframeStats', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.keyframeStats, kwargs, 'annotation')
        return res

    @_f.addMelDocs('keyframeStats', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, animation.keyframeStats, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('keyframeStats', 'classicMode')
    def getClassicMode(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.keyframeStats, kwargs, 'classicMode')
        return res

    @_f.addMelDocs('keyframeStats', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.keyframeStats, kwargs, 'docTag')
        return res

    @_f.addMelDocs('keyframeStats', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.keyframeStats, kwargs, 'enable')
        return res

    @_f.addMelDocs('keyframeStats', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.keyframeStats, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('keyframeStats', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.keyframeStats, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('keyframeStats', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.keyframeStats, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('keyframeStats', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, animation.keyframeStats, kwargs, 'height')
        return res

    @_f.addMelDocs('keyframeStats', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, animation.keyframeStats, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('keyframeStats', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.keyframeStats, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('keyframeStats', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.keyframeStats, kwargs, 'manage')
        return res

    @_f.addMelDocs('keyframeStats', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.keyframeStats, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('keyframeStats', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.keyframeStats, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('keyframeStats', 'precision')
    def getPrecision(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, animation.keyframeStats, kwargs, 'precision')
        return res

    @_f.addMelDocs('keyframeStats', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.keyframeStats, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('keyframeStats', 'timeAnnotation')
    def getTimeAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.keyframeStats, kwargs, 'timeAnnotation')
        return res

    @_f.addMelDocs('keyframeStats', 'valueAnnotation')
    def getValueAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.keyframeStats, kwargs, 'valueAnnotation')
        return res

    @_f.addMelDocs('keyframeStats', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.keyframeStats, kwargs, 'visible')
        return res

    @_f.addMelDocs('keyframeStats', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, animation.keyframeStats, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('keyframeStats', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, animation.keyframeStats, kwargs, 'width')
        return res

    @_f.addMelDocs('keyframeStats', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'noBackground', val)

    @_f.addMelDocs('keyframeStats', 'rowAttach')
    def rowAttach(self, val=True, **kwargs):
        # type: (Tuple[int, str, int] | List[Tuple[int, str, int]], **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'rowAttach', val)

    @_f.addMelDocs('keyframeStats', 'animEditor')
    def setAnimEditor(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'animEditor', val)

    @_f.addMelDocs('keyframeStats', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'annotation', val)

    @_f.addMelDocs('keyframeStats', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('keyframeStats', 'classicMode')
    def setClassicMode(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'classicMode', val)

    @_f.addMelDocs('keyframeStats', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'docTag', val)

    @_f.addMelDocs('keyframeStats', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'enable', val)

    @_f.addMelDocs('keyframeStats', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'enableBackground', val)

    @_f.addMelDocs('keyframeStats', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('keyframeStats', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'height', val)

    @_f.addMelDocs('keyframeStats', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'highlightColor', val)

    @_f.addMelDocs('keyframeStats', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'manage', val)

    @_f.addMelDocs('keyframeStats', 'precision')
    def setPrecision(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'precision', val)

    @_f.addMelDocs('keyframeStats', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'preventOverride', val)

    @_f.addMelDocs('keyframeStats', 'timeAnnotation')
    def setTimeAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'timeAnnotation', val)

    @_f.addMelDocs('keyframeStats', 'valueAnnotation')
    def setValueAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'valueAnnotation', val)

    @_f.addMelDocs('keyframeStats', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'visible', val)

    @_f.addMelDocs('keyframeStats', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('keyframeStats', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'width', val)

    @_f.addMelDocs('keyframeStats', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.keyframeStats, kwargs, 'statusBarMessage', val)


class LayerButton(PyUI):
    __melcmd__ = staticmethod(windows.layerButton)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'layerButton'
    __melui__ = 'layerButton'
    __slots__ = ()

    @_f.addMelDocs('layerButton', 'changeCommand')
    def changeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.layerButton, kwargs, 'changeCommand', val)

    @_f.addMelDocs('layerButton', 'color')
    def color(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.layerButton, kwargs, 'color', val)

    @_f.addMelDocs('layerButton', 'command')
    def command(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.layerButton, kwargs, 'command', val)

    @_f.addMelDocs('layerButton', 'current')
    def current(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.layerButton, kwargs, 'current', val)

    @_f.addMelDocs('layerButton', 'doubleClickCommand')
    def doubleClickCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.layerButton, kwargs, 'doubleClickCommand', val)

    @_f.addMelDocs('layerButton', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.layerButton, kwargs, 'dragCallback', val)

    @_f.addMelDocs('layerButton', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.layerButton, kwargs, 'dropCallback', val)

    @_f.addMelDocs('layerButton', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.layerButton, kwargs, 'annotation')
        return res

    @_f.addMelDocs('layerButton', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.layerButton, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('layerButton', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.layerButton, kwargs, 'docTag')
        return res

    @_f.addMelDocs('layerButton', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.layerButton, kwargs, 'enable')
        return res

    @_f.addMelDocs('layerButton', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.layerButton, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('layerButton', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.layerButton, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('layerButton', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.layerButton, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('layerButton', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.layerButton, kwargs, 'height')
        return res

    @_f.addMelDocs('layerButton', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.layerButton, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('layerButton', 'identification')
    def getIdentification(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.layerButton, kwargs, 'identification')
        return res

    @_f.addMelDocs('layerButton', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.layerButton, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('layerButton', 'label')
    def getLabel(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.layerButton, kwargs, 'label')
        return res

    @_f.addMelDocs('layerButton', 'labelWidth')
    def getLabelWidth(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.layerButton, kwargs, 'labelWidth')
        return res

    @_f.addMelDocs('layerButton', 'layerHideOnPlayback')
    def getLayerHideOnPlayback(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.layerButton, kwargs, 'layerHideOnPlayback')
        return res

    @_f.addMelDocs('layerButton', 'layerState')
    def getLayerState(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.layerButton, kwargs, 'layerState')
        return res

    @_f.addMelDocs('layerButton', 'layerVisible')
    def getLayerVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.layerButton, kwargs, 'layerVisible')
        return res

    @_f.addMelDocs('layerButton', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.layerButton, kwargs, 'manage')
        return res

    @_f.addMelDocs('layerButton', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.layerButton, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('layerButton', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.layerButton, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('layerButton', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.layerButton, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('layerButton', 'select')
    def getSelect(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.layerButton, kwargs, 'select')
        return res

    @_f.addMelDocs('layerButton', 'transparent')
    def getTransparent(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.layerButton, kwargs, 'transparent')
        return res

    @_f.addMelDocs('layerButton', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.layerButton, kwargs, 'visible')
        return res

    @_f.addMelDocs('layerButton', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.layerButton, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('layerButton', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.layerButton, kwargs, 'width')
        return res

    @_f.addMelDocs('layerButton', 'hideOnPlaybackCommand')
    def hideOnPlaybackCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.layerButton, kwargs, 'hideOnPlaybackCommand', val)

    @_f.addMelDocs('layerButton', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.layerButton, kwargs, 'noBackground', val)

    @_f.addMelDocs('layerButton', 'renameCommand')
    def renameCommand(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.layerButton, kwargs, 'renameCommand', val)

    @_f.addMelDocs('layerButton', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.layerButton, kwargs, 'annotation', val)

    @_f.addMelDocs('layerButton', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.layerButton, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('layerButton', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.layerButton, kwargs, 'docTag', val)

    @_f.addMelDocs('layerButton', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.layerButton, kwargs, 'enable', val)

    @_f.addMelDocs('layerButton', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.layerButton, kwargs, 'enableBackground', val)

    @_f.addMelDocs('layerButton', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.layerButton, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('layerButton', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.layerButton, kwargs, 'height', val)

    @_f.addMelDocs('layerButton', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.layerButton, kwargs, 'highlightColor', val)

    @_f.addMelDocs('layerButton', 'identification')
    def setIdentification(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.layerButton, kwargs, 'identification', val)

    @_f.addMelDocs('layerButton', 'label')
    def setLabel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.layerButton, kwargs, 'label', val)

    @_f.addMelDocs('layerButton', 'layerHideOnPlayback')
    def setLayerHideOnPlayback(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.layerButton, kwargs, 'layerHideOnPlayback', val)

    @_f.addMelDocs('layerButton', 'layerState')
    def setLayerState(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.layerButton, kwargs, 'layerState', val)

    @_f.addMelDocs('layerButton', 'layerVisible')
    def setLayerVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.layerButton, kwargs, 'layerVisible', val)

    @_f.addMelDocs('layerButton', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.layerButton, kwargs, 'manage', val)

    @_f.addMelDocs('layerButton', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.layerButton, kwargs, 'preventOverride', val)

    @_f.addMelDocs('layerButton', 'select')
    def setSelect(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.layerButton, kwargs, 'select', val)

    @_f.addMelDocs('layerButton', 'transparent')
    def setTransparent(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.layerButton, kwargs, 'transparent', val)

    @_f.addMelDocs('layerButton', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.layerButton, kwargs, 'visible', val)

    @_f.addMelDocs('layerButton', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.layerButton, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('layerButton', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.layerButton, kwargs, 'width', val)

    @_f.addMelDocs('layerButton', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.layerButton, kwargs, 'statusBarMessage', val)

    @_f.addMelDocs('layerButton', 'typeCommand')
    def typeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.layerButton, kwargs, 'typeCommand', val)

    @_f.addMelDocs('layerButton', 'visibleCommand')
    def visibleCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.layerButton, kwargs, 'visibleCommand', val)


class AttrColorSliderGrp(Layout):
    __melcmd__ = staticmethod(windows.attrColorSliderGrp)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'attrColorSliderGrp'
    __melui__ = 'attrColorSliderGrp'
    __slots__ = ()

    @_f.addMelDocs('attrColorSliderGrp', 'adjustableColumn')
    def adjustableColumn(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'adjustableColumn', val)

    @_f.addMelDocs('attrColorSliderGrp', 'adjustableColumn2')
    def adjustableColumn2(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'adjustableColumn2', val)

    @_f.addMelDocs('attrColorSliderGrp', 'adjustableColumn3')
    def adjustableColumn3(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'adjustableColumn3', val)

    @_f.addMelDocs('attrColorSliderGrp', 'adjustableColumn4')
    def adjustableColumn4(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'adjustableColumn4', val)

    @_f.addMelDocs('attrColorSliderGrp', 'adjustableColumn5')
    def adjustableColumn5(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'adjustableColumn5', val)

    @_f.addMelDocs('attrColorSliderGrp', 'adjustableColumn6')
    def adjustableColumn6(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'adjustableColumn6', val)

    @_f.addMelDocs('attrColorSliderGrp', 'columnAlign')
    def columnAlign(self, val=True, **kwargs):
        # type: (Tuple[int, str] | List[Tuple[int, str]], **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'columnAlign', val)

    @_f.addMelDocs('attrColorSliderGrp', 'columnAlign2')
    def columnAlign2(self, val=True, **kwargs):
        # type: (Tuple[str, str], **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'columnAlign2', val)

    @_f.addMelDocs('attrColorSliderGrp', 'columnAlign3')
    def columnAlign3(self, val=True, **kwargs):
        # type: (Tuple[str, str, str], **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'columnAlign3', val)

    @_f.addMelDocs('attrColorSliderGrp', 'columnAlign4')
    def columnAlign4(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'columnAlign4', val)

    @_f.addMelDocs('attrColorSliderGrp', 'columnAlign5')
    def columnAlign5(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'columnAlign5', val)

    @_f.addMelDocs('attrColorSliderGrp', 'columnAlign6')
    def columnAlign6(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'columnAlign6', val)

    @_f.addMelDocs('attrColorSliderGrp', 'columnAttach')
    def columnAttach(self, val=True, **kwargs):
        # type: (Tuple[int, str, int] | List[Tuple[int, str, int]], **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'columnAttach', val)

    @_f.addMelDocs('attrColorSliderGrp', 'columnAttach2')
    def columnAttach2(self, val=True, **kwargs):
        # type: (Tuple[str, str], **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'columnAttach2', val)

    @_f.addMelDocs('attrColorSliderGrp', 'columnAttach3')
    def columnAttach3(self, val=True, **kwargs):
        # type: (Tuple[str, str, str], **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'columnAttach3', val)

    @_f.addMelDocs('attrColorSliderGrp', 'columnAttach4')
    def columnAttach4(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'columnAttach4', val)

    @_f.addMelDocs('attrColorSliderGrp', 'columnAttach5')
    def columnAttach5(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'columnAttach5', val)

    @_f.addMelDocs('attrColorSliderGrp', 'columnAttach6')
    def columnAttach6(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'columnAttach6', val)

    @_f.addMelDocs('attrColorSliderGrp', 'columnOffset2')
    def columnOffset2(self, val=True, **kwargs):
        # type: (Tuple[int, int], **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'columnOffset2', val)

    @_f.addMelDocs('attrColorSliderGrp', 'columnOffset3')
    def columnOffset3(self, val=True, **kwargs):
        # type: (Tuple[int, int, int], **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'columnOffset3', val)

    @_f.addMelDocs('attrColorSliderGrp', 'columnOffset4')
    def columnOffset4(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'columnOffset4', val)

    @_f.addMelDocs('attrColorSliderGrp', 'columnOffset5')
    def columnOffset5(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'columnOffset5', val)

    @_f.addMelDocs('attrColorSliderGrp', 'columnOffset6')
    def columnOffset6(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'columnOffset6', val)

    @_f.addMelDocs('attrColorSliderGrp', 'columnWidth')
    def columnWidth(self, val=True, **kwargs):
        # type: (Tuple[int, int] | List[Tuple[int, int]], **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'columnWidth', val)

    @_f.addMelDocs('attrColorSliderGrp', 'columnWidth1')
    def columnWidth1(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'columnWidth1', val)

    @_f.addMelDocs('attrColorSliderGrp', 'columnWidth2')
    def columnWidth2(self, val=True, **kwargs):
        # type: (Tuple[int, int], **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'columnWidth2', val)

    @_f.addMelDocs('attrColorSliderGrp', 'columnWidth3')
    def columnWidth3(self, val=True, **kwargs):
        # type: (Tuple[int, int, int], **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'columnWidth3', val)

    @_f.addMelDocs('attrColorSliderGrp', 'columnWidth4')
    def columnWidth4(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'columnWidth4', val)

    @_f.addMelDocs('attrColorSliderGrp', 'columnWidth5')
    def columnWidth5(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'columnWidth5', val)

    @_f.addMelDocs('attrColorSliderGrp', 'columnWidth6')
    def columnWidth6(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'columnWidth6', val)

    @_f.addMelDocs('attrColorSliderGrp', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'dragCallback', val)

    @_f.addMelDocs('attrColorSliderGrp', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'dropCallback', val)

    @_f.addMelDocs('attrColorSliderGrp', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.attrColorSliderGrp, kwargs, 'annotation')
        return res

    @_f.addMelDocs('attrColorSliderGrp', 'attrNavDecision')
    def getAttrNavDecision(self, **kwargs):
        # type: (...) -> Tuple[_general.PyNode, str]
        res = _f.asQuery(self, windows.attrColorSliderGrp, kwargs, 'attrNavDecision')
        return res

    @_f.addMelDocs('attrColorSliderGrp', 'attribute')
    def getAttribute(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.attrColorSliderGrp, kwargs, 'attribute')
        return res

    @_f.addMelDocs('attrColorSliderGrp', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.attrColorSliderGrp, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('attrColorSliderGrp', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.attrColorSliderGrp, kwargs, 'docTag')
        return res

    @_f.addMelDocs('attrColorSliderGrp', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.attrColorSliderGrp, kwargs, 'enable')
        return res

    @_f.addMelDocs('attrColorSliderGrp', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.attrColorSliderGrp, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('attrColorSliderGrp', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.attrColorSliderGrp, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('attrColorSliderGrp', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.attrColorSliderGrp, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('attrColorSliderGrp', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.attrColorSliderGrp, kwargs, 'height')
        return res

    @_f.addMelDocs('attrColorSliderGrp', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.attrColorSliderGrp, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('attrColorSliderGrp', 'hsvValue')
    def getHsvValue(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.attrColorSliderGrp, kwargs, 'hsvValue')
        return res

    @_f.addMelDocs('attrColorSliderGrp', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.attrColorSliderGrp, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('attrColorSliderGrp', 'label')
    def getLabel(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.attrColorSliderGrp, kwargs, 'label')
        return res

    @_f.addMelDocs('attrColorSliderGrp', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.attrColorSliderGrp, kwargs, 'manage')
        return res

    @_f.addMelDocs('attrColorSliderGrp', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.attrColorSliderGrp, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('attrColorSliderGrp', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.attrColorSliderGrp, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('attrColorSliderGrp', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.attrColorSliderGrp, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('attrColorSliderGrp', 'rgbValue')
    def getRgbValue(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.attrColorSliderGrp, kwargs, 'rgbValue')
        return res

    @_f.addMelDocs('attrColorSliderGrp', 'showButton')
    def getShowButton(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.attrColorSliderGrp, kwargs, 'showButton')
        return res

    @_f.addMelDocs('attrColorSliderGrp', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.attrColorSliderGrp, kwargs, 'visible')
        return res

    @_f.addMelDocs('attrColorSliderGrp', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.attrColorSliderGrp, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('attrColorSliderGrp', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.attrColorSliderGrp, kwargs, 'width')
        return res

    @_f.addMelDocs('attrColorSliderGrp', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'noBackground', val)

    @_f.addMelDocs('attrColorSliderGrp', 'rowAttach')
    def rowAttach(self, val=True, **kwargs):
        # type: (Tuple[int, str, int] | List[Tuple[int, str, int]], **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'rowAttach', val)

    @_f.addMelDocs('attrColorSliderGrp', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'annotation', val)

    @_f.addMelDocs('attrColorSliderGrp', 'attrNavDecision')
    def setAttrNavDecision(self, val=True, **kwargs):
        # type: (Tuple[_general.PyNode | str, str], **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'attrNavDecision', val)

    @_f.addMelDocs('attrColorSliderGrp', 'attribute')
    def setAttribute(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'attribute', val)

    @_f.addMelDocs('attrColorSliderGrp', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('attrColorSliderGrp', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'docTag', val)

    @_f.addMelDocs('attrColorSliderGrp', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'enable', val)

    @_f.addMelDocs('attrColorSliderGrp', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'enableBackground', val)

    @_f.addMelDocs('attrColorSliderGrp', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('attrColorSliderGrp', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'height', val)

    @_f.addMelDocs('attrColorSliderGrp', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'highlightColor', val)

    @_f.addMelDocs('attrColorSliderGrp', 'hsvValue')
    def setHsvValue(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'hsvValue', val)

    @_f.addMelDocs('attrColorSliderGrp', 'label')
    def setLabel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'label', val)

    @_f.addMelDocs('attrColorSliderGrp', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'manage', val)

    @_f.addMelDocs('attrColorSliderGrp', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'preventOverride', val)

    @_f.addMelDocs('attrColorSliderGrp', 'rgbValue')
    def setRgbValue(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'rgbValue', val)

    @_f.addMelDocs('attrColorSliderGrp', 'showButton')
    def setShowButton(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'showButton', val)

    @_f.addMelDocs('attrColorSliderGrp', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'visible', val)

    @_f.addMelDocs('attrColorSliderGrp', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('attrColorSliderGrp', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'width', val)

    @_f.addMelDocs('attrColorSliderGrp', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.attrColorSliderGrp, kwargs, 'statusBarMessage', val)


class AttrControlGrp(Layout):
    __melcmd__ = staticmethod(windows.attrControlGrp)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'attrControlGrp'
    __melui__ = 'attrControlGrp'
    __slots__ = ()

    @_f.addMelDocs('attrControlGrp', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.attrControlGrp, kwargs, 'annotation')
        return res

    @_f.addMelDocs('attrControlGrp', 'attribute')
    def getAttribute(self, **kwargs):
        # type: (...) -> _general.PyNode
        res = _f.asQuery(self, windows.attrControlGrp, kwargs, 'attribute')
        return res

    @_f.addMelDocs('attrControlGrp', 'changeCommand')
    def getChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.attrControlGrp, kwargs, 'changeCommand')
        return res

    @_f.addMelDocs('attrControlGrp', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.attrControlGrp, kwargs, 'enable')
        return res

    @_f.addMelDocs('attrControlGrp', 'exists')
    def getExists(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.attrControlGrp, kwargs, 'exists')
        return res

    @_f.addMelDocs('attrControlGrp', 'handlesAttribute')
    def getHandlesAttribute(self, **kwargs):
        # type: (...) -> _general.PyNode
        res = _f.asQuery(self, windows.attrControlGrp, kwargs, 'handlesAttribute')
        return res

    @_f.addMelDocs('attrControlGrp', 'hideMapButton')
    def getHideMapButton(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.attrControlGrp, kwargs, 'hideMapButton')
        return res

    @_f.addMelDocs('attrControlGrp', 'label')
    def getLabel(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.attrControlGrp, kwargs, 'label')
        return res

    @_f.addMelDocs('attrControlGrp', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.attrControlGrp, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('attrControlGrp', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.attrControlGrp, kwargs, 'annotation', val)

    @_f.addMelDocs('attrControlGrp', 'attribute')
    def setAttribute(self, val=True, **kwargs):
        # type: (_general.PyNode | str, **Any) -> None
        return _f.asEdit(self, windows.attrControlGrp, kwargs, 'attribute', val)

    @_f.addMelDocs('attrControlGrp', 'changeCommand')
    def setChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.attrControlGrp, kwargs, 'changeCommand', val)

    @_f.addMelDocs('attrControlGrp', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrControlGrp, kwargs, 'enable', val)

    @_f.addMelDocs('attrControlGrp', 'exists')
    def setExists(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrControlGrp, kwargs, 'exists', val)

    @_f.addMelDocs('attrControlGrp', 'handlesAttribute')
    def setHandlesAttribute(self, val=True, **kwargs):
        # type: (_general.PyNode | str, **Any) -> None
        return _f.asEdit(self, windows.attrControlGrp, kwargs, 'handlesAttribute', val)

    @_f.addMelDocs('attrControlGrp', 'hideMapButton')
    def setHideMapButton(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrControlGrp, kwargs, 'hideMapButton', val)

    @_f.addMelDocs('attrControlGrp', 'label')
    def setLabel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.attrControlGrp, kwargs, 'label', val)

    @_f.addMelDocs('attrControlGrp', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrControlGrp, kwargs, 'preventOverride', val)


class AttrEnumOptionMenuGrp(Layout):
    __melcmd__ = staticmethod(windows.attrEnumOptionMenuGrp)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'attrEnumOptionMenuGrp'
    __melui__ = 'attrEnumOptionMenuGrp'
    __slots__ = ()

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'adjustableColumn')
    def adjustableColumn(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'adjustableColumn', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'adjustableColumn2')
    def adjustableColumn2(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'adjustableColumn2', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'adjustableColumn3')
    def adjustableColumn3(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'adjustableColumn3', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'adjustableColumn4')
    def adjustableColumn4(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'adjustableColumn4', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'adjustableColumn5')
    def adjustableColumn5(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'adjustableColumn5', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'adjustableColumn6')
    def adjustableColumn6(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'adjustableColumn6', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'attribute')
    def attribute(self, val=True, **kwargs):
        # type: (_general.PyNode | str, **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'attribute', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'columnAlign')
    def columnAlign(self, val=True, **kwargs):
        # type: (Tuple[int, str] | List[Tuple[int, str]], **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'columnAlign', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'columnAlign2')
    def columnAlign2(self, val=True, **kwargs):
        # type: (Tuple[str, str], **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'columnAlign2', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'columnAlign3')
    def columnAlign3(self, val=True, **kwargs):
        # type: (Tuple[str, str, str], **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'columnAlign3', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'columnAlign4')
    def columnAlign4(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'columnAlign4', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'columnAlign5')
    def columnAlign5(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'columnAlign5', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'columnAlign6')
    def columnAlign6(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'columnAlign6', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'columnAttach')
    def columnAttach(self, val=True, **kwargs):
        # type: (Tuple[int, str, int] | List[Tuple[int, str, int]], **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'columnAttach', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'columnAttach2')
    def columnAttach2(self, val=True, **kwargs):
        # type: (Tuple[str, str], **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'columnAttach2', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'columnAttach3')
    def columnAttach3(self, val=True, **kwargs):
        # type: (Tuple[str, str, str], **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'columnAttach3', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'columnAttach4')
    def columnAttach4(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'columnAttach4', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'columnAttach5')
    def columnAttach5(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'columnAttach5', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'columnAttach6')
    def columnAttach6(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'columnAttach6', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'columnOffset2')
    def columnOffset2(self, val=True, **kwargs):
        # type: (Tuple[int, int], **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'columnOffset2', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'columnOffset3')
    def columnOffset3(self, val=True, **kwargs):
        # type: (Tuple[int, int, int], **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'columnOffset3', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'columnOffset4')
    def columnOffset4(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'columnOffset4', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'columnOffset5')
    def columnOffset5(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'columnOffset5', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'columnOffset6')
    def columnOffset6(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'columnOffset6', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'columnWidth')
    def columnWidth(self, val=True, **kwargs):
        # type: (Tuple[int, int] | List[Tuple[int, int]], **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'columnWidth', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'columnWidth1')
    def columnWidth1(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'columnWidth1', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'columnWidth2')
    def columnWidth2(self, val=True, **kwargs):
        # type: (Tuple[int, int], **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'columnWidth2', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'columnWidth3')
    def columnWidth3(self, val=True, **kwargs):
        # type: (Tuple[int, int, int], **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'columnWidth3', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'columnWidth4')
    def columnWidth4(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'columnWidth4', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'columnWidth5')
    def columnWidth5(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'columnWidth5', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'columnWidth6')
    def columnWidth6(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'columnWidth6', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'dragCallback', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'dropCallback', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.attrEnumOptionMenuGrp, kwargs, 'annotation')
        return res

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.attrEnumOptionMenuGrp, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.attrEnumOptionMenuGrp, kwargs, 'docTag')
        return res

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.attrEnumOptionMenuGrp, kwargs, 'enable')
        return res

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.attrEnumOptionMenuGrp, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.attrEnumOptionMenuGrp, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.attrEnumOptionMenuGrp, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.attrEnumOptionMenuGrp, kwargs, 'height')
        return res

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.attrEnumOptionMenuGrp, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.attrEnumOptionMenuGrp, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'label')
    def getLabel(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.attrEnumOptionMenuGrp, kwargs, 'label')
        return res

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.attrEnumOptionMenuGrp, kwargs, 'manage')
        return res

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.attrEnumOptionMenuGrp, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.attrEnumOptionMenuGrp, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.attrEnumOptionMenuGrp, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.attrEnumOptionMenuGrp, kwargs, 'visible')
        return res

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.attrEnumOptionMenuGrp, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.attrEnumOptionMenuGrp, kwargs, 'width')
        return res

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'noBackground', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'rowAttach')
    def rowAttach(self, val=True, **kwargs):
        # type: (Tuple[int, str, int] | List[Tuple[int, str, int]], **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'rowAttach', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'annotation', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'docTag', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'enable', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'enableBackground', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'height', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'highlightColor', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'label')
    def setLabel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'label', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'manage', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'preventOverride', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'visible', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'width', val)

    @_f.addMelDocs('attrEnumOptionMenuGrp', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.attrEnumOptionMenuGrp, kwargs, 'statusBarMessage', val)


class AttrFieldGrp(Layout):
    __melcmd__ = staticmethod(windows.attrFieldGrp)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'attrFieldGrp'
    __melui__ = 'attrFieldGrp'
    __slots__ = ()

    @_f.addMelDocs('attrFieldGrp', 'adjustableColumn')
    def adjustableColumn(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'adjustableColumn', val)

    @_f.addMelDocs('attrFieldGrp', 'adjustableColumn2')
    def adjustableColumn2(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'adjustableColumn2', val)

    @_f.addMelDocs('attrFieldGrp', 'adjustableColumn3')
    def adjustableColumn3(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'adjustableColumn3', val)

    @_f.addMelDocs('attrFieldGrp', 'adjustableColumn4')
    def adjustableColumn4(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'adjustableColumn4', val)

    @_f.addMelDocs('attrFieldGrp', 'adjustableColumn5')
    def adjustableColumn5(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'adjustableColumn5', val)

    @_f.addMelDocs('attrFieldGrp', 'adjustableColumn6')
    def adjustableColumn6(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'adjustableColumn6', val)

    @_f.addMelDocs('attrFieldGrp', 'changeCommand')
    def changeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'changeCommand', val)

    @_f.addMelDocs('attrFieldGrp', 'columnAlign')
    def columnAlign(self, val=True, **kwargs):
        # type: (Tuple[int, str] | List[Tuple[int, str]], **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'columnAlign', val)

    @_f.addMelDocs('attrFieldGrp', 'columnAlign2')
    def columnAlign2(self, val=True, **kwargs):
        # type: (Tuple[str, str], **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'columnAlign2', val)

    @_f.addMelDocs('attrFieldGrp', 'columnAlign3')
    def columnAlign3(self, val=True, **kwargs):
        # type: (Tuple[str, str, str], **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'columnAlign3', val)

    @_f.addMelDocs('attrFieldGrp', 'columnAlign4')
    def columnAlign4(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'columnAlign4', val)

    @_f.addMelDocs('attrFieldGrp', 'columnAlign5')
    def columnAlign5(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'columnAlign5', val)

    @_f.addMelDocs('attrFieldGrp', 'columnAlign6')
    def columnAlign6(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'columnAlign6', val)

    @_f.addMelDocs('attrFieldGrp', 'columnAttach')
    def columnAttach(self, val=True, **kwargs):
        # type: (Tuple[int, str, int] | List[Tuple[int, str, int]], **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'columnAttach', val)

    @_f.addMelDocs('attrFieldGrp', 'columnAttach2')
    def columnAttach2(self, val=True, **kwargs):
        # type: (Tuple[str, str], **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'columnAttach2', val)

    @_f.addMelDocs('attrFieldGrp', 'columnAttach3')
    def columnAttach3(self, val=True, **kwargs):
        # type: (Tuple[str, str, str], **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'columnAttach3', val)

    @_f.addMelDocs('attrFieldGrp', 'columnAttach4')
    def columnAttach4(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'columnAttach4', val)

    @_f.addMelDocs('attrFieldGrp', 'columnAttach5')
    def columnAttach5(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'columnAttach5', val)

    @_f.addMelDocs('attrFieldGrp', 'columnAttach6')
    def columnAttach6(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'columnAttach6', val)

    @_f.addMelDocs('attrFieldGrp', 'columnOffset2')
    def columnOffset2(self, val=True, **kwargs):
        # type: (Tuple[int, int], **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'columnOffset2', val)

    @_f.addMelDocs('attrFieldGrp', 'columnOffset3')
    def columnOffset3(self, val=True, **kwargs):
        # type: (Tuple[int, int, int], **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'columnOffset3', val)

    @_f.addMelDocs('attrFieldGrp', 'columnOffset4')
    def columnOffset4(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'columnOffset4', val)

    @_f.addMelDocs('attrFieldGrp', 'columnOffset5')
    def columnOffset5(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'columnOffset5', val)

    @_f.addMelDocs('attrFieldGrp', 'columnOffset6')
    def columnOffset6(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'columnOffset6', val)

    @_f.addMelDocs('attrFieldGrp', 'columnWidth')
    def columnWidth(self, val=True, **kwargs):
        # type: (Tuple[int, int] | List[Tuple[int, int]], **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'columnWidth', val)

    @_f.addMelDocs('attrFieldGrp', 'columnWidth1')
    def columnWidth1(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'columnWidth1', val)

    @_f.addMelDocs('attrFieldGrp', 'columnWidth2')
    def columnWidth2(self, val=True, **kwargs):
        # type: (Tuple[int, int], **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'columnWidth2', val)

    @_f.addMelDocs('attrFieldGrp', 'columnWidth3')
    def columnWidth3(self, val=True, **kwargs):
        # type: (Tuple[int, int, int], **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'columnWidth3', val)

    @_f.addMelDocs('attrFieldGrp', 'columnWidth4')
    def columnWidth4(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'columnWidth4', val)

    @_f.addMelDocs('attrFieldGrp', 'columnWidth5')
    def columnWidth5(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'columnWidth5', val)

    @_f.addMelDocs('attrFieldGrp', 'columnWidth6')
    def columnWidth6(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'columnWidth6', val)

    @_f.addMelDocs('attrFieldGrp', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'dragCallback', val)

    @_f.addMelDocs('attrFieldGrp', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'dropCallback', val)

    @_f.addMelDocs('attrFieldGrp', 'extraButtonCommand')
    def extraButtonCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'extraButtonCommand', val)

    @_f.addMelDocs('attrFieldGrp', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.attrFieldGrp, kwargs, 'annotation')
        return res

    @_f.addMelDocs('attrFieldGrp', 'attribute')
    def getAttribute(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.attrFieldGrp, kwargs, 'attribute')
        return res

    @_f.addMelDocs('attrFieldGrp', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.attrFieldGrp, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('attrFieldGrp', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.attrFieldGrp, kwargs, 'docTag')
        return res

    @_f.addMelDocs('attrFieldGrp', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.attrFieldGrp, kwargs, 'enable')
        return res

    @_f.addMelDocs('attrFieldGrp', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.attrFieldGrp, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('attrFieldGrp', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.attrFieldGrp, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('attrFieldGrp', 'extraButtonIcon')
    def getExtraButtonIcon(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.attrFieldGrp, kwargs, 'extraButtonIcon')
        return res

    @_f.addMelDocs('attrFieldGrp', 'extraLabel')
    def getExtraLabel(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.attrFieldGrp, kwargs, 'extraLabel')
        return res

    @_f.addMelDocs('attrFieldGrp', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.attrFieldGrp, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('attrFieldGrp', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.attrFieldGrp, kwargs, 'height')
        return res

    @_f.addMelDocs('attrFieldGrp', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.attrFieldGrp, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('attrFieldGrp', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.attrFieldGrp, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('attrFieldGrp', 'label')
    def getLabel(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.attrFieldGrp, kwargs, 'label')
        return res

    @_f.addMelDocs('attrFieldGrp', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.attrFieldGrp, kwargs, 'manage')
        return res

    @_f.addMelDocs('attrFieldGrp', 'maxValue')
    def getMaxValue(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.attrFieldGrp, kwargs, 'maxValue')
        return res

    @_f.addMelDocs('attrFieldGrp', 'minValue')
    def getMinValue(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.attrFieldGrp, kwargs, 'minValue')
        return res

    @_f.addMelDocs('attrFieldGrp', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.attrFieldGrp, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('attrFieldGrp', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.attrFieldGrp, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('attrFieldGrp', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.attrFieldGrp, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('attrFieldGrp', 'step')
    def getStep(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.attrFieldGrp, kwargs, 'step')
        return res

    @_f.addMelDocs('attrFieldGrp', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.attrFieldGrp, kwargs, 'visible')
        return res

    @_f.addMelDocs('attrFieldGrp', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.attrFieldGrp, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('attrFieldGrp', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.attrFieldGrp, kwargs, 'width')
        return res

    @_f.addMelDocs('attrFieldGrp', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'noBackground', val)

    @_f.addMelDocs('attrFieldGrp', 'precision')
    def precision(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'precision', val)

    @_f.addMelDocs('attrFieldGrp', 'rowAttach')
    def rowAttach(self, val=True, **kwargs):
        # type: (Tuple[int, str, int] | List[Tuple[int, str, int]], **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'rowAttach', val)

    @_f.addMelDocs('attrFieldGrp', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'annotation', val)

    @_f.addMelDocs('attrFieldGrp', 'attribute')
    def setAttribute(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'attribute', val)

    @_f.addMelDocs('attrFieldGrp', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('attrFieldGrp', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'docTag', val)

    @_f.addMelDocs('attrFieldGrp', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'enable', val)

    @_f.addMelDocs('attrFieldGrp', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'enableBackground', val)

    @_f.addMelDocs('attrFieldGrp', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('attrFieldGrp', 'extraButtonIcon')
    def setExtraButtonIcon(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'extraButtonIcon', val)

    @_f.addMelDocs('attrFieldGrp', 'extraLabel')
    def setExtraLabel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'extraLabel', val)

    @_f.addMelDocs('attrFieldGrp', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'height', val)

    @_f.addMelDocs('attrFieldGrp', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'highlightColor', val)

    @_f.addMelDocs('attrFieldGrp', 'label')
    def setLabel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'label', val)

    @_f.addMelDocs('attrFieldGrp', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'manage', val)

    @_f.addMelDocs('attrFieldGrp', 'maxValue')
    def setMaxValue(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'maxValue', val)

    @_f.addMelDocs('attrFieldGrp', 'minValue')
    def setMinValue(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'minValue', val)

    @_f.addMelDocs('attrFieldGrp', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'preventOverride', val)

    @_f.addMelDocs('attrFieldGrp', 'step')
    def setStep(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'step', val)

    @_f.addMelDocs('attrFieldGrp', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'visible', val)

    @_f.addMelDocs('attrFieldGrp', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('attrFieldGrp', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'width', val)

    @_f.addMelDocs('attrFieldGrp', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.attrFieldGrp, kwargs, 'statusBarMessage', val)


class AttrFieldSliderGrp(Layout):
    __melcmd__ = staticmethod(windows.attrFieldSliderGrp)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'attrFieldSliderGrp'
    __melui__ = 'attrFieldSliderGrp'
    __slots__ = ()

    @_f.addMelDocs('attrFieldSliderGrp', 'adjustableColumn')
    def adjustableColumn(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'adjustableColumn', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'adjustableColumn2')
    def adjustableColumn2(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'adjustableColumn2', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'adjustableColumn3')
    def adjustableColumn3(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'adjustableColumn3', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'adjustableColumn4')
    def adjustableColumn4(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'adjustableColumn4', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'adjustableColumn5')
    def adjustableColumn5(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'adjustableColumn5', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'adjustableColumn6')
    def adjustableColumn6(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'adjustableColumn6', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'changeCommand')
    def changeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'changeCommand', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'columnAlign')
    def columnAlign(self, val=True, **kwargs):
        # type: (Tuple[int, str] | List[Tuple[int, str]], **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'columnAlign', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'columnAlign2')
    def columnAlign2(self, val=True, **kwargs):
        # type: (Tuple[str, str], **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'columnAlign2', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'columnAlign3')
    def columnAlign3(self, val=True, **kwargs):
        # type: (Tuple[str, str, str], **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'columnAlign3', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'columnAlign4')
    def columnAlign4(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'columnAlign4', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'columnAlign5')
    def columnAlign5(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'columnAlign5', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'columnAlign6')
    def columnAlign6(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'columnAlign6', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'columnAttach')
    def columnAttach(self, val=True, **kwargs):
        # type: (Tuple[int, str, int] | List[Tuple[int, str, int]], **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'columnAttach', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'columnAttach2')
    def columnAttach2(self, val=True, **kwargs):
        # type: (Tuple[str, str], **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'columnAttach2', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'columnAttach3')
    def columnAttach3(self, val=True, **kwargs):
        # type: (Tuple[str, str, str], **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'columnAttach3', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'columnAttach4')
    def columnAttach4(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'columnAttach4', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'columnAttach5')
    def columnAttach5(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'columnAttach5', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'columnAttach6')
    def columnAttach6(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'columnAttach6', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'columnOffset2')
    def columnOffset2(self, val=True, **kwargs):
        # type: (Tuple[int, int], **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'columnOffset2', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'columnOffset3')
    def columnOffset3(self, val=True, **kwargs):
        # type: (Tuple[int, int, int], **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'columnOffset3', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'columnOffset4')
    def columnOffset4(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'columnOffset4', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'columnOffset5')
    def columnOffset5(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'columnOffset5', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'columnOffset6')
    def columnOffset6(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'columnOffset6', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'columnWidth')
    def columnWidth(self, val=True, **kwargs):
        # type: (Tuple[int, int] | List[Tuple[int, int]], **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'columnWidth', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'columnWidth1')
    def columnWidth1(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'columnWidth1', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'columnWidth2')
    def columnWidth2(self, val=True, **kwargs):
        # type: (Tuple[int, int], **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'columnWidth2', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'columnWidth3')
    def columnWidth3(self, val=True, **kwargs):
        # type: (Tuple[int, int, int], **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'columnWidth3', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'columnWidth4')
    def columnWidth4(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'columnWidth4', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'columnWidth5')
    def columnWidth5(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'columnWidth5', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'columnWidth6')
    def columnWidth6(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'columnWidth6', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'dragCallback', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'dropCallback', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'extraButtonCommand')
    def extraButtonCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'extraButtonCommand', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.attrFieldSliderGrp, kwargs, 'annotation')
        return res

    @_f.addMelDocs('attrFieldSliderGrp', 'attribute')
    def getAttribute(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.attrFieldSliderGrp, kwargs, 'attribute')
        return res

    @_f.addMelDocs('attrFieldSliderGrp', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.attrFieldSliderGrp, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('attrFieldSliderGrp', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.attrFieldSliderGrp, kwargs, 'docTag')
        return res

    @_f.addMelDocs('attrFieldSliderGrp', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.attrFieldSliderGrp, kwargs, 'enable')
        return res

    @_f.addMelDocs('attrFieldSliderGrp', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.attrFieldSliderGrp, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('attrFieldSliderGrp', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.attrFieldSliderGrp, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('attrFieldSliderGrp', 'extraButtonIcon')
    def getExtraButtonIcon(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.attrFieldSliderGrp, kwargs, 'extraButtonIcon')
        return res

    @_f.addMelDocs('attrFieldSliderGrp', 'fieldMaxValue')
    def getFieldMaxValue(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.attrFieldSliderGrp, kwargs, 'fieldMaxValue')
        return res

    @_f.addMelDocs('attrFieldSliderGrp', 'fieldMinValue')
    def getFieldMinValue(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.attrFieldSliderGrp, kwargs, 'fieldMinValue')
        return res

    @_f.addMelDocs('attrFieldSliderGrp', 'fieldStep')
    def getFieldStep(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.attrFieldSliderGrp, kwargs, 'fieldStep')
        return res

    @_f.addMelDocs('attrFieldSliderGrp', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.attrFieldSliderGrp, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('attrFieldSliderGrp', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.attrFieldSliderGrp, kwargs, 'height')
        return res

    @_f.addMelDocs('attrFieldSliderGrp', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.attrFieldSliderGrp, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('attrFieldSliderGrp', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.attrFieldSliderGrp, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('attrFieldSliderGrp', 'label')
    def getLabel(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.attrFieldSliderGrp, kwargs, 'label')
        return res

    @_f.addMelDocs('attrFieldSliderGrp', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.attrFieldSliderGrp, kwargs, 'manage')
        return res

    @_f.addMelDocs('attrFieldSliderGrp', 'maxValue')
    def getMaxValue(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.attrFieldSliderGrp, kwargs, 'maxValue')
        return res

    @_f.addMelDocs('attrFieldSliderGrp', 'minValue')
    def getMinValue(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.attrFieldSliderGrp, kwargs, 'minValue')
        return res

    @_f.addMelDocs('attrFieldSliderGrp', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.attrFieldSliderGrp, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('attrFieldSliderGrp', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.attrFieldSliderGrp, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('attrFieldSliderGrp', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.attrFieldSliderGrp, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('attrFieldSliderGrp', 'sliderMaxValue')
    def getSliderMaxValue(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.attrFieldSliderGrp, kwargs, 'sliderMaxValue')
        return res

    @_f.addMelDocs('attrFieldSliderGrp', 'sliderMinValue')
    def getSliderMinValue(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.attrFieldSliderGrp, kwargs, 'sliderMinValue')
        return res

    @_f.addMelDocs('attrFieldSliderGrp', 'sliderStep')
    def getSliderStep(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.attrFieldSliderGrp, kwargs, 'sliderStep')
        return res

    @_f.addMelDocs('attrFieldSliderGrp', 'step')
    def getStep(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.attrFieldSliderGrp, kwargs, 'step')
        return res

    @_f.addMelDocs('attrFieldSliderGrp', 'vertical')
    def getVertical(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.attrFieldSliderGrp, kwargs, 'vertical')
        return res

    @_f.addMelDocs('attrFieldSliderGrp', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.attrFieldSliderGrp, kwargs, 'visible')
        return res

    @_f.addMelDocs('attrFieldSliderGrp', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.attrFieldSliderGrp, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('attrFieldSliderGrp', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.attrFieldSliderGrp, kwargs, 'width')
        return res

    @_f.addMelDocs('attrFieldSliderGrp', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'noBackground', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'precision')
    def precision(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'precision', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'rowAttach')
    def rowAttach(self, val=True, **kwargs):
        # type: (Tuple[int, str, int] | List[Tuple[int, str, int]], **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'rowAttach', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'annotation', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'attribute')
    def setAttribute(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'attribute', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'docTag', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'enable', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'enableBackground', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'extraButtonIcon')
    def setExtraButtonIcon(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'extraButtonIcon', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'fieldMaxValue')
    def setFieldMaxValue(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'fieldMaxValue', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'fieldMinValue')
    def setFieldMinValue(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'fieldMinValue', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'fieldStep')
    def setFieldStep(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'fieldStep', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'height', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'highlightColor', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'label')
    def setLabel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'label', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'manage', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'maxValue')
    def setMaxValue(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'maxValue', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'minValue')
    def setMinValue(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'minValue', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'preventOverride', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'sliderMaxValue')
    def setSliderMaxValue(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'sliderMaxValue', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'sliderMinValue')
    def setSliderMinValue(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'sliderMinValue', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'sliderStep')
    def setSliderStep(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'sliderStep', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'step')
    def setStep(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'step', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'visible', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'width', val)

    @_f.addMelDocs('attrFieldSliderGrp', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.attrFieldSliderGrp, kwargs, 'statusBarMessage', val)


class AttrNavigationControlGrp(Layout):
    __melcmd__ = staticmethod(windows.attrNavigationControlGrp)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'attrNavigationControlGrp'
    __melui__ = 'attrNavigationControlGrp'
    __slots__ = ()

    @_f.addMelDocs('attrNavigationControlGrp', 'adjustableColumn')
    def adjustableColumn(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'adjustableColumn', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'adjustableColumn2')
    def adjustableColumn2(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'adjustableColumn2', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'adjustableColumn3')
    def adjustableColumn3(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'adjustableColumn3', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'adjustableColumn4')
    def adjustableColumn4(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'adjustableColumn4', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'adjustableColumn5')
    def adjustableColumn5(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'adjustableColumn5', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'adjustableColumn6')
    def adjustableColumn6(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'adjustableColumn6', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'columnAlign')
    def columnAlign(self, val=True, **kwargs):
        # type: (Tuple[int, str] | List[Tuple[int, str]], **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'columnAlign', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'columnAlign2')
    def columnAlign2(self, val=True, **kwargs):
        # type: (Tuple[str, str], **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'columnAlign2', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'columnAlign3')
    def columnAlign3(self, val=True, **kwargs):
        # type: (Tuple[str, str, str], **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'columnAlign3', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'columnAlign4')
    def columnAlign4(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'columnAlign4', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'columnAlign5')
    def columnAlign5(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'columnAlign5', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'columnAlign6')
    def columnAlign6(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'columnAlign6', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'columnAttach')
    def columnAttach(self, val=True, **kwargs):
        # type: (Tuple[int, str, int] | List[Tuple[int, str, int]], **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'columnAttach', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'columnAttach2')
    def columnAttach2(self, val=True, **kwargs):
        # type: (Tuple[str, str], **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'columnAttach2', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'columnAttach3')
    def columnAttach3(self, val=True, **kwargs):
        # type: (Tuple[str, str, str], **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'columnAttach3', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'columnAttach4')
    def columnAttach4(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'columnAttach4', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'columnAttach5')
    def columnAttach5(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'columnAttach5', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'columnAttach6')
    def columnAttach6(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'columnAttach6', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'columnOffset2')
    def columnOffset2(self, val=True, **kwargs):
        # type: (Tuple[int, int], **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'columnOffset2', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'columnOffset3')
    def columnOffset3(self, val=True, **kwargs):
        # type: (Tuple[int, int, int], **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'columnOffset3', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'columnOffset4')
    def columnOffset4(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'columnOffset4', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'columnOffset5')
    def columnOffset5(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'columnOffset5', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'columnOffset6')
    def columnOffset6(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'columnOffset6', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'columnWidth')
    def columnWidth(self, val=True, **kwargs):
        # type: (Tuple[int, int] | List[Tuple[int, int]], **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'columnWidth', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'columnWidth1')
    def columnWidth1(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'columnWidth1', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'columnWidth2')
    def columnWidth2(self, val=True, **kwargs):
        # type: (Tuple[int, int], **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'columnWidth2', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'columnWidth3')
    def columnWidth3(self, val=True, **kwargs):
        # type: (Tuple[int, int, int], **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'columnWidth3', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'columnWidth4')
    def columnWidth4(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'columnWidth4', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'columnWidth5')
    def columnWidth5(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'columnWidth5', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'columnWidth6')
    def columnWidth6(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'columnWidth6', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'dragCallback', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'dropCallback', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'extraButtonCommand')
    def extraButtonCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'extraButtonCommand', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.attrNavigationControlGrp, kwargs, 'annotation')
        return res

    @_f.addMelDocs('attrNavigationControlGrp', 'attrNavDecision')
    def getAttrNavDecision(self, **kwargs):
        # type: (...) -> Tuple[_general.PyNode, str]
        res = _f.asQuery(self, windows.attrNavigationControlGrp, kwargs, 'attrNavDecision')
        return res

    @_f.addMelDocs('attrNavigationControlGrp', 'attribute')
    def getAttribute(self, **kwargs):
        # type: (...) -> _general.PyNode
        res = _f.asQuery(self, windows.attrNavigationControlGrp, kwargs, 'attribute')
        return res

    @_f.addMelDocs('attrNavigationControlGrp', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.attrNavigationControlGrp, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('attrNavigationControlGrp', 'connectAttrToDropped')
    def getConnectAttrToDropped(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.attrNavigationControlGrp, kwargs, 'connectAttrToDropped')
        return res

    @_f.addMelDocs('attrNavigationControlGrp', 'connectNodeToDropped')
    def getConnectNodeToDropped(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.attrNavigationControlGrp, kwargs, 'connectNodeToDropped')
        return res

    @_f.addMelDocs('attrNavigationControlGrp', 'connectToExisting')
    def getConnectToExisting(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.attrNavigationControlGrp, kwargs, 'connectToExisting')
        return res

    @_f.addMelDocs('attrNavigationControlGrp', 'createNew')
    def getCreateNew(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.attrNavigationControlGrp, kwargs, 'createNew')
        return res

    @_f.addMelDocs('attrNavigationControlGrp', 'defaultTraversal')
    def getDefaultTraversal(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.attrNavigationControlGrp, kwargs, 'defaultTraversal')
        return res

    @_f.addMelDocs('attrNavigationControlGrp', 'delete')
    def getDelete(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.attrNavigationControlGrp, kwargs, 'delete')
        return res

    @_f.addMelDocs('attrNavigationControlGrp', 'disconnect')
    def getDisconnect(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.attrNavigationControlGrp, kwargs, 'disconnect')
        return res

    @_f.addMelDocs('attrNavigationControlGrp', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.attrNavigationControlGrp, kwargs, 'docTag')
        return res

    @_f.addMelDocs('attrNavigationControlGrp', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.attrNavigationControlGrp, kwargs, 'enable')
        return res

    @_f.addMelDocs('attrNavigationControlGrp', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.attrNavigationControlGrp, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('attrNavigationControlGrp', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.attrNavigationControlGrp, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('attrNavigationControlGrp', 'extraButtonIcon')
    def getExtraButtonIcon(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.attrNavigationControlGrp, kwargs, 'extraButtonIcon')
        return res

    @_f.addMelDocs('attrNavigationControlGrp', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.attrNavigationControlGrp, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('attrNavigationControlGrp', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.attrNavigationControlGrp, kwargs, 'height')
        return res

    @_f.addMelDocs('attrNavigationControlGrp', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.attrNavigationControlGrp, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('attrNavigationControlGrp', 'ignore')
    def getIgnore(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.attrNavigationControlGrp, kwargs, 'ignore')
        return res

    @_f.addMelDocs('attrNavigationControlGrp', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.attrNavigationControlGrp, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('attrNavigationControlGrp', 'label')
    def getLabel(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.attrNavigationControlGrp, kwargs, 'label')
        return res

    @_f.addMelDocs('attrNavigationControlGrp', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.attrNavigationControlGrp, kwargs, 'manage')
        return res

    @_f.addMelDocs('attrNavigationControlGrp', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.attrNavigationControlGrp, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('attrNavigationControlGrp', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.attrNavigationControlGrp, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('attrNavigationControlGrp', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.attrNavigationControlGrp, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('attrNavigationControlGrp', 'relatedNodes')
    def getRelatedNodes(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.attrNavigationControlGrp, kwargs, 'relatedNodes')
        return res

    @_f.addMelDocs('attrNavigationControlGrp', 'unignore')
    def getUnignore(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.attrNavigationControlGrp, kwargs, 'unignore')
        return res

    @_f.addMelDocs('attrNavigationControlGrp', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.attrNavigationControlGrp, kwargs, 'visible')
        return res

    @_f.addMelDocs('attrNavigationControlGrp', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.attrNavigationControlGrp, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('attrNavigationControlGrp', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.attrNavigationControlGrp, kwargs, 'width')
        return res

    @_f.addMelDocs('attrNavigationControlGrp', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'noBackground', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'noIgnorableMenu')
    def noIgnorableMenu(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'noIgnorableMenu', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'noKeyableMenu')
    def noKeyableMenu(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'noKeyableMenu', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'rowAttach')
    def rowAttach(self, val=True, **kwargs):
        # type: (Tuple[int, str, int] | List[Tuple[int, str, int]], **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'rowAttach', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'annotation', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'attrNavDecision')
    def setAttrNavDecision(self, val=True, **kwargs):
        # type: (Tuple[_general.PyNode | str, str], **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'attrNavDecision', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'attribute')
    def setAttribute(self, val=True, **kwargs):
        # type: (_general.PyNode | str, **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'attribute', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'connectAttrToDropped')
    def setConnectAttrToDropped(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'connectAttrToDropped', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'connectNodeToDropped')
    def setConnectNodeToDropped(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'connectNodeToDropped', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'connectToExisting')
    def setConnectToExisting(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'connectToExisting', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'createNew')
    def setCreateNew(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'createNew', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'defaultTraversal')
    def setDefaultTraversal(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'defaultTraversal', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'delete')
    def setDelete(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'delete', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'disconnect')
    def setDisconnect(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'disconnect', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'docTag', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'enable', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'enableBackground', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'extraButtonIcon')
    def setExtraButtonIcon(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'extraButtonIcon', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'height', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'highlightColor', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'ignore')
    def setIgnore(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'ignore', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'label')
    def setLabel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'label', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'manage', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'preventOverride', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'relatedNodes')
    def setRelatedNodes(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'relatedNodes', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'unignore')
    def setUnignore(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'unignore', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'visible', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'width', val)

    @_f.addMelDocs('attrNavigationControlGrp', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.attrNavigationControlGrp, kwargs, 'statusBarMessage', val)


class CheckBoxGrp(Layout):
    __melcmd__ = staticmethod(windows.checkBoxGrp)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'checkBoxGrp'
    __melui__ = 'checkBoxGrp'
    __slots__ = ()

    @_f.addMelDocs('checkBoxGrp', 'adjustableColumn')
    def adjustableColumn(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'adjustableColumn', val)

    @_f.addMelDocs('checkBoxGrp', 'adjustableColumn2')
    def adjustableColumn2(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'adjustableColumn2', val)

    @_f.addMelDocs('checkBoxGrp', 'adjustableColumn3')
    def adjustableColumn3(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'adjustableColumn3', val)

    @_f.addMelDocs('checkBoxGrp', 'adjustableColumn4')
    def adjustableColumn4(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'adjustableColumn4', val)

    @_f.addMelDocs('checkBoxGrp', 'adjustableColumn5')
    def adjustableColumn5(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'adjustableColumn5', val)

    @_f.addMelDocs('checkBoxGrp', 'adjustableColumn6')
    def adjustableColumn6(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'adjustableColumn6', val)

    @_f.addMelDocs('checkBoxGrp', 'changeCommand')
    def changeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'changeCommand', val)

    @_f.addMelDocs('checkBoxGrp', 'changeCommand1')
    def changeCommand1(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'changeCommand1', val)

    @_f.addMelDocs('checkBoxGrp', 'changeCommand2')
    def changeCommand2(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'changeCommand2', val)

    @_f.addMelDocs('checkBoxGrp', 'changeCommand3')
    def changeCommand3(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'changeCommand3', val)

    @_f.addMelDocs('checkBoxGrp', 'changeCommand4')
    def changeCommand4(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'changeCommand4', val)

    @_f.addMelDocs('checkBoxGrp', 'columnAlign')
    def columnAlign(self, val=True, **kwargs):
        # type: (Tuple[int, str] | List[Tuple[int, str]], **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'columnAlign', val)

    @_f.addMelDocs('checkBoxGrp', 'columnAlign2')
    def columnAlign2(self, val=True, **kwargs):
        # type: (Tuple[str, str], **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'columnAlign2', val)

    @_f.addMelDocs('checkBoxGrp', 'columnAlign3')
    def columnAlign3(self, val=True, **kwargs):
        # type: (Tuple[str, str, str], **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'columnAlign3', val)

    @_f.addMelDocs('checkBoxGrp', 'columnAlign4')
    def columnAlign4(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'columnAlign4', val)

    @_f.addMelDocs('checkBoxGrp', 'columnAlign5')
    def columnAlign5(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'columnAlign5', val)

    @_f.addMelDocs('checkBoxGrp', 'columnAlign6')
    def columnAlign6(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'columnAlign6', val)

    @_f.addMelDocs('checkBoxGrp', 'columnAttach')
    def columnAttach(self, val=True, **kwargs):
        # type: (Tuple[int, str, int] | List[Tuple[int, str, int]], **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'columnAttach', val)

    @_f.addMelDocs('checkBoxGrp', 'columnAttach2')
    def columnAttach2(self, val=True, **kwargs):
        # type: (Tuple[str, str], **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'columnAttach2', val)

    @_f.addMelDocs('checkBoxGrp', 'columnAttach3')
    def columnAttach3(self, val=True, **kwargs):
        # type: (Tuple[str, str, str], **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'columnAttach3', val)

    @_f.addMelDocs('checkBoxGrp', 'columnAttach4')
    def columnAttach4(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'columnAttach4', val)

    @_f.addMelDocs('checkBoxGrp', 'columnAttach5')
    def columnAttach5(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'columnAttach5', val)

    @_f.addMelDocs('checkBoxGrp', 'columnAttach6')
    def columnAttach6(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'columnAttach6', val)

    @_f.addMelDocs('checkBoxGrp', 'columnOffset2')
    def columnOffset2(self, val=True, **kwargs):
        # type: (Tuple[int, int], **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'columnOffset2', val)

    @_f.addMelDocs('checkBoxGrp', 'columnOffset3')
    def columnOffset3(self, val=True, **kwargs):
        # type: (Tuple[int, int, int], **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'columnOffset3', val)

    @_f.addMelDocs('checkBoxGrp', 'columnOffset4')
    def columnOffset4(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'columnOffset4', val)

    @_f.addMelDocs('checkBoxGrp', 'columnOffset5')
    def columnOffset5(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'columnOffset5', val)

    @_f.addMelDocs('checkBoxGrp', 'columnOffset6')
    def columnOffset6(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'columnOffset6', val)

    @_f.addMelDocs('checkBoxGrp', 'columnWidth')
    def columnWidth(self, val=True, **kwargs):
        # type: (Tuple[int, int] | List[Tuple[int, int]], **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'columnWidth', val)

    @_f.addMelDocs('checkBoxGrp', 'columnWidth1')
    def columnWidth1(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'columnWidth1', val)

    @_f.addMelDocs('checkBoxGrp', 'columnWidth2')
    def columnWidth2(self, val=True, **kwargs):
        # type: (Tuple[int, int], **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'columnWidth2', val)

    @_f.addMelDocs('checkBoxGrp', 'columnWidth3')
    def columnWidth3(self, val=True, **kwargs):
        # type: (Tuple[int, int, int], **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'columnWidth3', val)

    @_f.addMelDocs('checkBoxGrp', 'columnWidth4')
    def columnWidth4(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'columnWidth4', val)

    @_f.addMelDocs('checkBoxGrp', 'columnWidth5')
    def columnWidth5(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'columnWidth5', val)

    @_f.addMelDocs('checkBoxGrp', 'columnWidth6')
    def columnWidth6(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'columnWidth6', val)

    @_f.addMelDocs('checkBoxGrp', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'dragCallback', val)

    @_f.addMelDocs('checkBoxGrp', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'dropCallback', val)

    @_f.addMelDocs('checkBoxGrp', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.checkBoxGrp, kwargs, 'annotation')
        return res

    @_f.addMelDocs('checkBoxGrp', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.checkBoxGrp, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('checkBoxGrp', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.checkBoxGrp, kwargs, 'docTag')
        return res

    @_f.addMelDocs('checkBoxGrp', 'editable')
    def getEditable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.checkBoxGrp, kwargs, 'editable')
        return res

    @_f.addMelDocs('checkBoxGrp', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.checkBoxGrp, kwargs, 'enable')
        return res

    @_f.addMelDocs('checkBoxGrp', 'enable1')
    def getEnable1(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.checkBoxGrp, kwargs, 'enable1')
        return res

    @_f.addMelDocs('checkBoxGrp', 'enable2')
    def getEnable2(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.checkBoxGrp, kwargs, 'enable2')
        return res

    @_f.addMelDocs('checkBoxGrp', 'enable3')
    def getEnable3(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.checkBoxGrp, kwargs, 'enable3')
        return res

    @_f.addMelDocs('checkBoxGrp', 'enable4')
    def getEnable4(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.checkBoxGrp, kwargs, 'enable4')
        return res

    @_f.addMelDocs('checkBoxGrp', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.checkBoxGrp, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('checkBoxGrp', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.checkBoxGrp, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('checkBoxGrp', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.checkBoxGrp, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('checkBoxGrp', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.checkBoxGrp, kwargs, 'height')
        return res

    @_f.addMelDocs('checkBoxGrp', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.checkBoxGrp, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('checkBoxGrp', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.checkBoxGrp, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('checkBoxGrp', 'label')
    def getLabel(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.checkBoxGrp, kwargs, 'label')
        return res

    @_f.addMelDocs('checkBoxGrp', 'label1')
    def getLabel1(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.checkBoxGrp, kwargs, 'label1')
        return res

    @_f.addMelDocs('checkBoxGrp', 'label2')
    def getLabel2(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.checkBoxGrp, kwargs, 'label2')
        return res

    @_f.addMelDocs('checkBoxGrp', 'label3')
    def getLabel3(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.checkBoxGrp, kwargs, 'label3')
        return res

    @_f.addMelDocs('checkBoxGrp', 'label4')
    def getLabel4(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.checkBoxGrp, kwargs, 'label4')
        return res

    @_f.addMelDocs('checkBoxGrp', 'labelArray2')
    def getLabelArray2(self, **kwargs):
        # type: (...) -> Tuple[str, str]
        res = _f.asQuery(self, windows.checkBoxGrp, kwargs, 'labelArray2')
        return res

    @_f.addMelDocs('checkBoxGrp', 'labelArray3')
    def getLabelArray3(self, **kwargs):
        # type: (...) -> Tuple[str, str, str]
        res = _f.asQuery(self, windows.checkBoxGrp, kwargs, 'labelArray3')
        return res

    @_f.addMelDocs('checkBoxGrp', 'labelArray4')
    def getLabelArray4(self, **kwargs):
        # type: (...) -> Tuple[str, str, str, str]
        res = _f.asQuery(self, windows.checkBoxGrp, kwargs, 'labelArray4')
        return res

    @_f.addMelDocs('checkBoxGrp', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.checkBoxGrp, kwargs, 'manage')
        return res

    @_f.addMelDocs('checkBoxGrp', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.checkBoxGrp, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('checkBoxGrp', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.checkBoxGrp, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('checkBoxGrp', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.checkBoxGrp, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('checkBoxGrp', 'value1')
    def getValue1(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.checkBoxGrp, kwargs, 'value1')
        return res

    @_f.addMelDocs('checkBoxGrp', 'value2')
    def getValue2(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.checkBoxGrp, kwargs, 'value2')
        return res

    @_f.addMelDocs('checkBoxGrp', 'value3')
    def getValue3(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.checkBoxGrp, kwargs, 'value3')
        return res

    @_f.addMelDocs('checkBoxGrp', 'value4')
    def getValue4(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.checkBoxGrp, kwargs, 'value4')
        return res

    @_f.addMelDocs('checkBoxGrp', 'valueArray2')
    def getValueArray2(self, **kwargs):
        # type: (...) -> Tuple[bool, bool]
        res = _f.asQuery(self, windows.checkBoxGrp, kwargs, 'valueArray2')
        return res

    @_f.addMelDocs('checkBoxGrp', 'valueArray3')
    def getValueArray3(self, **kwargs):
        # type: (...) -> Tuple[bool, bool, bool]
        res = _f.asQuery(self, windows.checkBoxGrp, kwargs, 'valueArray3')
        return res

    @_f.addMelDocs('checkBoxGrp', 'valueArray4')
    def getValueArray4(self, **kwargs):
        # type: (...) -> Tuple[bool, bool, bool, bool]
        res = _f.asQuery(self, windows.checkBoxGrp, kwargs, 'valueArray4')
        return res

    @_f.addMelDocs('checkBoxGrp', 'vertical')
    def getVertical(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.checkBoxGrp, kwargs, 'vertical')
        return res

    @_f.addMelDocs('checkBoxGrp', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.checkBoxGrp, kwargs, 'visible')
        return res

    @_f.addMelDocs('checkBoxGrp', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.checkBoxGrp, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('checkBoxGrp', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.checkBoxGrp, kwargs, 'width')
        return res

    @_f.addMelDocs('checkBoxGrp', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'noBackground', val)

    @_f.addMelDocs('checkBoxGrp', 'offCommand')
    def offCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'offCommand', val)

    @_f.addMelDocs('checkBoxGrp', 'offCommand1')
    def offCommand1(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'offCommand1', val)

    @_f.addMelDocs('checkBoxGrp', 'offCommand2')
    def offCommand2(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'offCommand2', val)

    @_f.addMelDocs('checkBoxGrp', 'offCommand3')
    def offCommand3(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'offCommand3', val)

    @_f.addMelDocs('checkBoxGrp', 'offCommand4')
    def offCommand4(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'offCommand4', val)

    @_f.addMelDocs('checkBoxGrp', 'onCommand')
    def onCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'onCommand', val)

    @_f.addMelDocs('checkBoxGrp', 'onCommand1')
    def onCommand1(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'onCommand1', val)

    @_f.addMelDocs('checkBoxGrp', 'onCommand2')
    def onCommand2(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'onCommand2', val)

    @_f.addMelDocs('checkBoxGrp', 'onCommand3')
    def onCommand3(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'onCommand3', val)

    @_f.addMelDocs('checkBoxGrp', 'onCommand4')
    def onCommand4(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'onCommand4', val)

    @_f.addMelDocs('checkBoxGrp', 'rowAttach')
    def rowAttach(self, val=True, **kwargs):
        # type: (Tuple[int, str, int] | List[Tuple[int, str, int]], **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'rowAttach', val)

    @_f.addMelDocs('checkBoxGrp', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'annotation', val)

    @_f.addMelDocs('checkBoxGrp', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('checkBoxGrp', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'docTag', val)

    @_f.addMelDocs('checkBoxGrp', 'editable')
    def setEditable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'editable', val)

    @_f.addMelDocs('checkBoxGrp', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'enable', val)

    @_f.addMelDocs('checkBoxGrp', 'enable1')
    def setEnable1(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'enable1', val)

    @_f.addMelDocs('checkBoxGrp', 'enable2')
    def setEnable2(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'enable2', val)

    @_f.addMelDocs('checkBoxGrp', 'enable3')
    def setEnable3(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'enable3', val)

    @_f.addMelDocs('checkBoxGrp', 'enable4')
    def setEnable4(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'enable4', val)

    @_f.addMelDocs('checkBoxGrp', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'enableBackground', val)

    @_f.addMelDocs('checkBoxGrp', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('checkBoxGrp', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'height', val)

    @_f.addMelDocs('checkBoxGrp', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'highlightColor', val)

    @_f.addMelDocs('checkBoxGrp', 'label')
    def setLabel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'label', val)

    @_f.addMelDocs('checkBoxGrp', 'label1')
    def setLabel1(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'label1', val)

    @_f.addMelDocs('checkBoxGrp', 'label2')
    def setLabel2(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'label2', val)

    @_f.addMelDocs('checkBoxGrp', 'label3')
    def setLabel3(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'label3', val)

    @_f.addMelDocs('checkBoxGrp', 'label4')
    def setLabel4(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'label4', val)

    @_f.addMelDocs('checkBoxGrp', 'labelArray2')
    def setLabelArray2(self, val=True, **kwargs):
        # type: (Tuple[str, str], **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'labelArray2', val)

    @_f.addMelDocs('checkBoxGrp', 'labelArray3')
    def setLabelArray3(self, val=True, **kwargs):
        # type: (Tuple[str, str, str], **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'labelArray3', val)

    @_f.addMelDocs('checkBoxGrp', 'labelArray4')
    def setLabelArray4(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'labelArray4', val)

    @_f.addMelDocs('checkBoxGrp', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'manage', val)

    @_f.addMelDocs('checkBoxGrp', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'preventOverride', val)

    @_f.addMelDocs('checkBoxGrp', 'value1')
    def setValue1(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'value1', val)

    @_f.addMelDocs('checkBoxGrp', 'value2')
    def setValue2(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'value2', val)

    @_f.addMelDocs('checkBoxGrp', 'value3')
    def setValue3(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'value3', val)

    @_f.addMelDocs('checkBoxGrp', 'value4')
    def setValue4(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'value4', val)

    @_f.addMelDocs('checkBoxGrp', 'valueArray2')
    def setValueArray2(self, val=True, **kwargs):
        # type: (Tuple[bool | int, bool | int], **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'valueArray2', val)

    @_f.addMelDocs('checkBoxGrp', 'valueArray3')
    def setValueArray3(self, val=True, **kwargs):
        # type: (Tuple[bool | int, bool | int, bool | int], **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'valueArray3', val)

    @_f.addMelDocs('checkBoxGrp', 'valueArray4')
    def setValueArray4(self, val=True, **kwargs):
        # type: (Tuple[bool | int, bool | int, bool | int, bool | int], **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'valueArray4', val)

    @_f.addMelDocs('checkBoxGrp', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'visible', val)

    @_f.addMelDocs('checkBoxGrp', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('checkBoxGrp', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'width', val)

    @_f.addMelDocs('checkBoxGrp', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.checkBoxGrp, kwargs, 'statusBarMessage', val)


class ColorIndexSliderGrp(Layout):
    __melcmd__ = staticmethod(windows.colorIndexSliderGrp)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'colorIndexSliderGrp'
    __melui__ = 'colorIndexSliderGrp'
    __slots__ = ()

    @_f.addMelDocs('colorIndexSliderGrp', 'adjustableColumn')
    def adjustableColumn(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'adjustableColumn', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'adjustableColumn2')
    def adjustableColumn2(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'adjustableColumn2', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'adjustableColumn3')
    def adjustableColumn3(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'adjustableColumn3', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'adjustableColumn4')
    def adjustableColumn4(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'adjustableColumn4', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'adjustableColumn5')
    def adjustableColumn5(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'adjustableColumn5', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'adjustableColumn6')
    def adjustableColumn6(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'adjustableColumn6', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'changeCommand')
    def changeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'changeCommand', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'columnAlign')
    def columnAlign(self, val=True, **kwargs):
        # type: (Tuple[int, str] | List[Tuple[int, str]], **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'columnAlign', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'columnAlign2')
    def columnAlign2(self, val=True, **kwargs):
        # type: (Tuple[str, str], **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'columnAlign2', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'columnAlign3')
    def columnAlign3(self, val=True, **kwargs):
        # type: (Tuple[str, str, str], **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'columnAlign3', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'columnAlign4')
    def columnAlign4(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'columnAlign4', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'columnAlign5')
    def columnAlign5(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'columnAlign5', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'columnAlign6')
    def columnAlign6(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'columnAlign6', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'columnAttach')
    def columnAttach(self, val=True, **kwargs):
        # type: (Tuple[int, str, int] | List[Tuple[int, str, int]], **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'columnAttach', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'columnAttach2')
    def columnAttach2(self, val=True, **kwargs):
        # type: (Tuple[str, str], **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'columnAttach2', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'columnAttach3')
    def columnAttach3(self, val=True, **kwargs):
        # type: (Tuple[str, str, str], **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'columnAttach3', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'columnAttach4')
    def columnAttach4(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'columnAttach4', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'columnAttach5')
    def columnAttach5(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'columnAttach5', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'columnAttach6')
    def columnAttach6(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'columnAttach6', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'columnOffset2')
    def columnOffset2(self, val=True, **kwargs):
        # type: (Tuple[int, int], **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'columnOffset2', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'columnOffset3')
    def columnOffset3(self, val=True, **kwargs):
        # type: (Tuple[int, int, int], **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'columnOffset3', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'columnOffset4')
    def columnOffset4(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'columnOffset4', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'columnOffset5')
    def columnOffset5(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'columnOffset5', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'columnOffset6')
    def columnOffset6(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'columnOffset6', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'columnWidth')
    def columnWidth(self, val=True, **kwargs):
        # type: (Tuple[int, int] | List[Tuple[int, int]], **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'columnWidth', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'columnWidth1')
    def columnWidth1(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'columnWidth1', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'columnWidth2')
    def columnWidth2(self, val=True, **kwargs):
        # type: (Tuple[int, int], **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'columnWidth2', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'columnWidth3')
    def columnWidth3(self, val=True, **kwargs):
        # type: (Tuple[int, int, int], **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'columnWidth3', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'columnWidth4')
    def columnWidth4(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'columnWidth4', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'columnWidth5')
    def columnWidth5(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'columnWidth5', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'columnWidth6')
    def columnWidth6(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'columnWidth6', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'dragCallback', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'dragCommand')
    def dragCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'dragCommand', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'dropCallback', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.colorIndexSliderGrp, kwargs, 'annotation')
        return res

    @_f.addMelDocs('colorIndexSliderGrp', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.colorIndexSliderGrp, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('colorIndexSliderGrp', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.colorIndexSliderGrp, kwargs, 'docTag')
        return res

    @_f.addMelDocs('colorIndexSliderGrp', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.colorIndexSliderGrp, kwargs, 'enable')
        return res

    @_f.addMelDocs('colorIndexSliderGrp', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.colorIndexSliderGrp, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('colorIndexSliderGrp', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.colorIndexSliderGrp, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('colorIndexSliderGrp', 'extraLabel')
    def getExtraLabel(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.colorIndexSliderGrp, kwargs, 'extraLabel')
        return res

    @_f.addMelDocs('colorIndexSliderGrp', 'forceDragRefresh')
    def getForceDragRefresh(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.colorIndexSliderGrp, kwargs, 'forceDragRefresh')
        return res

    @_f.addMelDocs('colorIndexSliderGrp', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.colorIndexSliderGrp, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('colorIndexSliderGrp', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.colorIndexSliderGrp, kwargs, 'height')
        return res

    @_f.addMelDocs('colorIndexSliderGrp', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.colorIndexSliderGrp, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('colorIndexSliderGrp', 'invisible')
    def getInvisible(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.colorIndexSliderGrp, kwargs, 'invisible')
        return res

    @_f.addMelDocs('colorIndexSliderGrp', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.colorIndexSliderGrp, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('colorIndexSliderGrp', 'label')
    def getLabel(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.colorIndexSliderGrp, kwargs, 'label')
        return res

    @_f.addMelDocs('colorIndexSliderGrp', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.colorIndexSliderGrp, kwargs, 'manage')
        return res

    @_f.addMelDocs('colorIndexSliderGrp', 'maxValue')
    def getMaxValue(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.colorIndexSliderGrp, kwargs, 'maxValue')
        return res

    @_f.addMelDocs('colorIndexSliderGrp', 'minValue')
    def getMinValue(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.colorIndexSliderGrp, kwargs, 'minValue')
        return res

    @_f.addMelDocs('colorIndexSliderGrp', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.colorIndexSliderGrp, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('colorIndexSliderGrp', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.colorIndexSliderGrp, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('colorIndexSliderGrp', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.colorIndexSliderGrp, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('colorIndexSliderGrp', 'value')
    def getValue(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.colorIndexSliderGrp, kwargs, 'value')
        return res

    @_f.addMelDocs('colorIndexSliderGrp', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.colorIndexSliderGrp, kwargs, 'visible')
        return res

    @_f.addMelDocs('colorIndexSliderGrp', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.colorIndexSliderGrp, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('colorIndexSliderGrp', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.colorIndexSliderGrp, kwargs, 'width')
        return res

    @_f.addMelDocs('colorIndexSliderGrp', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'noBackground', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'rowAttach')
    def rowAttach(self, val=True, **kwargs):
        # type: (Tuple[int, str, int] | List[Tuple[int, str, int]], **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'rowAttach', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'annotation', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'docTag', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'enable', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'enableBackground', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'extraLabel')
    def setExtraLabel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'extraLabel', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'forceDragRefresh')
    def setForceDragRefresh(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'forceDragRefresh', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'height', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'highlightColor', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'invisible')
    def setInvisible(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'invisible', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'label')
    def setLabel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'label', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'manage', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'maxValue')
    def setMaxValue(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'maxValue', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'minValue')
    def setMinValue(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'minValue', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'preventOverride', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'value')
    def setValue(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'value', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'visible', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'width', val)

    @_f.addMelDocs('colorIndexSliderGrp', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.colorIndexSliderGrp, kwargs, 'statusBarMessage', val)


class ColorSliderButtonGrp(Layout):
    __melcmd__ = staticmethod(windows.colorSliderButtonGrp)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'colorSliderButtonGrp'
    __melui__ = 'colorSliderButtonGrp'
    __slots__ = ()

    @_f.addMelDocs('colorSliderButtonGrp', 'adjustableColumn')
    def adjustableColumn(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'adjustableColumn', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'adjustableColumn2')
    def adjustableColumn2(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'adjustableColumn2', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'adjustableColumn3')
    def adjustableColumn3(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'adjustableColumn3', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'adjustableColumn4')
    def adjustableColumn4(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'adjustableColumn4', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'adjustableColumn5')
    def adjustableColumn5(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'adjustableColumn5', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'adjustableColumn6')
    def adjustableColumn6(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'adjustableColumn6', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'buttonCommand')
    def buttonCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'buttonCommand', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'changeCommand')
    def changeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'changeCommand', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'columnAlign')
    def columnAlign(self, val=True, **kwargs):
        # type: (Tuple[int, str] | List[Tuple[int, str]], **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'columnAlign', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'columnAlign2')
    def columnAlign2(self, val=True, **kwargs):
        # type: (Tuple[str, str], **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'columnAlign2', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'columnAlign3')
    def columnAlign3(self, val=True, **kwargs):
        # type: (Tuple[str, str, str], **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'columnAlign3', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'columnAlign4')
    def columnAlign4(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'columnAlign4', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'columnAlign5')
    def columnAlign5(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'columnAlign5', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'columnAlign6')
    def columnAlign6(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'columnAlign6', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'columnAttach')
    def columnAttach(self, val=True, **kwargs):
        # type: (Tuple[int, str, int] | List[Tuple[int, str, int]], **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'columnAttach', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'columnAttach2')
    def columnAttach2(self, val=True, **kwargs):
        # type: (Tuple[str, str], **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'columnAttach2', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'columnAttach3')
    def columnAttach3(self, val=True, **kwargs):
        # type: (Tuple[str, str, str], **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'columnAttach3', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'columnAttach4')
    def columnAttach4(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'columnAttach4', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'columnAttach5')
    def columnAttach5(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'columnAttach5', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'columnAttach6')
    def columnAttach6(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'columnAttach6', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'columnOffset2')
    def columnOffset2(self, val=True, **kwargs):
        # type: (Tuple[int, int], **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'columnOffset2', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'columnOffset3')
    def columnOffset3(self, val=True, **kwargs):
        # type: (Tuple[int, int, int], **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'columnOffset3', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'columnOffset4')
    def columnOffset4(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'columnOffset4', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'columnOffset5')
    def columnOffset5(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'columnOffset5', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'columnOffset6')
    def columnOffset6(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'columnOffset6', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'columnWidth')
    def columnWidth(self, val=True, **kwargs):
        # type: (Tuple[int, int] | List[Tuple[int, int]], **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'columnWidth', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'columnWidth1')
    def columnWidth1(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'columnWidth1', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'columnWidth2')
    def columnWidth2(self, val=True, **kwargs):
        # type: (Tuple[int, int], **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'columnWidth2', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'columnWidth3')
    def columnWidth3(self, val=True, **kwargs):
        # type: (Tuple[int, int, int], **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'columnWidth3', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'columnWidth4')
    def columnWidth4(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'columnWidth4', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'columnWidth5')
    def columnWidth5(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'columnWidth5', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'columnWidth6')
    def columnWidth6(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'columnWidth6', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'dragCallback', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'dragCommand')
    def dragCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'dragCommand', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'dropCallback', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.colorSliderButtonGrp, kwargs, 'annotation')
        return res

    @_f.addMelDocs('colorSliderButtonGrp', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.colorSliderButtonGrp, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('colorSliderButtonGrp', 'buttonLabel')
    def getButtonLabel(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.colorSliderButtonGrp, kwargs, 'buttonLabel')
        return res

    @_f.addMelDocs('colorSliderButtonGrp', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.colorSliderButtonGrp, kwargs, 'docTag')
        return res

    @_f.addMelDocs('colorSliderButtonGrp', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.colorSliderButtonGrp, kwargs, 'enable')
        return res

    @_f.addMelDocs('colorSliderButtonGrp', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.colorSliderButtonGrp, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('colorSliderButtonGrp', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.colorSliderButtonGrp, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('colorSliderButtonGrp', 'forceDragRefresh')
    def getForceDragRefresh(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.colorSliderButtonGrp, kwargs, 'forceDragRefresh')
        return res

    @_f.addMelDocs('colorSliderButtonGrp', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.colorSliderButtonGrp, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('colorSliderButtonGrp', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.colorSliderButtonGrp, kwargs, 'height')
        return res

    @_f.addMelDocs('colorSliderButtonGrp', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.colorSliderButtonGrp, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('colorSliderButtonGrp', 'hsvValue')
    def getHsvValue(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.colorSliderButtonGrp, kwargs, 'hsvValue')
        return res

    @_f.addMelDocs('colorSliderButtonGrp', 'image')
    def getImage(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.colorSliderButtonGrp, kwargs, 'image')
        return res

    @_f.addMelDocs('colorSliderButtonGrp', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.colorSliderButtonGrp, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('colorSliderButtonGrp', 'label')
    def getLabel(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.colorSliderButtonGrp, kwargs, 'label')
        return res

    @_f.addMelDocs('colorSliderButtonGrp', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.colorSliderButtonGrp, kwargs, 'manage')
        return res

    @_f.addMelDocs('colorSliderButtonGrp', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.colorSliderButtonGrp, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('colorSliderButtonGrp', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.colorSliderButtonGrp, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('colorSliderButtonGrp', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.colorSliderButtonGrp, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('colorSliderButtonGrp', 'rgbValue')
    def getRgbValue(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.colorSliderButtonGrp, kwargs, 'rgbValue')
        return res

    @_f.addMelDocs('colorSliderButtonGrp', 'symbolButtonDisplay')
    def getSymbolButtonDisplay(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.colorSliderButtonGrp, kwargs, 'symbolButtonDisplay')
        return res

    @_f.addMelDocs('colorSliderButtonGrp', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.colorSliderButtonGrp, kwargs, 'visible')
        return res

    @_f.addMelDocs('colorSliderButtonGrp', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.colorSliderButtonGrp, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('colorSliderButtonGrp', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.colorSliderButtonGrp, kwargs, 'width')
        return res

    @_f.addMelDocs('colorSliderButtonGrp', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'noBackground', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'rowAttach')
    def rowAttach(self, val=True, **kwargs):
        # type: (Tuple[int, str, int] | List[Tuple[int, str, int]], **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'rowAttach', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'annotation', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'buttonLabel')
    def setButtonLabel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'buttonLabel', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'docTag', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'enable', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'enableBackground', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'forceDragRefresh')
    def setForceDragRefresh(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'forceDragRefresh', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'height', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'highlightColor', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'hsvValue')
    def setHsvValue(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'hsvValue', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'image')
    def setImage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'image', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'label')
    def setLabel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'label', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'manage', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'preventOverride', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'rgbValue')
    def setRgbValue(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'rgbValue', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'symbolButtonDisplay')
    def setSymbolButtonDisplay(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'symbolButtonDisplay', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'visible', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'width', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'statusBarMessage', val)

    @_f.addMelDocs('colorSliderButtonGrp', 'symbolButtonCommand')
    def symbolButtonCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.colorSliderButtonGrp, kwargs, 'symbolButtonCommand', val)


class ColorSliderGrp(Layout):
    __melcmd__ = staticmethod(windows.colorSliderGrp)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'colorSliderGrp'
    __melui__ = 'colorSliderGrp'
    __slots__ = ()

    @_f.addMelDocs('colorSliderGrp', 'adjustableColumn')
    def adjustableColumn(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'adjustableColumn', val)

    @_f.addMelDocs('colorSliderGrp', 'adjustableColumn2')
    def adjustableColumn2(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'adjustableColumn2', val)

    @_f.addMelDocs('colorSliderGrp', 'adjustableColumn3')
    def adjustableColumn3(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'adjustableColumn3', val)

    @_f.addMelDocs('colorSliderGrp', 'adjustableColumn4')
    def adjustableColumn4(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'adjustableColumn4', val)

    @_f.addMelDocs('colorSliderGrp', 'adjustableColumn5')
    def adjustableColumn5(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'adjustableColumn5', val)

    @_f.addMelDocs('colorSliderGrp', 'adjustableColumn6')
    def adjustableColumn6(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'adjustableColumn6', val)

    @_f.addMelDocs('colorSliderGrp', 'changeCommand')
    def changeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'changeCommand', val)

    @_f.addMelDocs('colorSliderGrp', 'columnAlign')
    def columnAlign(self, val=True, **kwargs):
        # type: (Tuple[int, str] | List[Tuple[int, str]], **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'columnAlign', val)

    @_f.addMelDocs('colorSliderGrp', 'columnAlign2')
    def columnAlign2(self, val=True, **kwargs):
        # type: (Tuple[str, str], **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'columnAlign2', val)

    @_f.addMelDocs('colorSliderGrp', 'columnAlign3')
    def columnAlign3(self, val=True, **kwargs):
        # type: (Tuple[str, str, str], **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'columnAlign3', val)

    @_f.addMelDocs('colorSliderGrp', 'columnAlign4')
    def columnAlign4(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'columnAlign4', val)

    @_f.addMelDocs('colorSliderGrp', 'columnAlign5')
    def columnAlign5(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'columnAlign5', val)

    @_f.addMelDocs('colorSliderGrp', 'columnAlign6')
    def columnAlign6(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'columnAlign6', val)

    @_f.addMelDocs('colorSliderGrp', 'columnAttach')
    def columnAttach(self, val=True, **kwargs):
        # type: (Tuple[int, str, int] | List[Tuple[int, str, int]], **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'columnAttach', val)

    @_f.addMelDocs('colorSliderGrp', 'columnAttach2')
    def columnAttach2(self, val=True, **kwargs):
        # type: (Tuple[str, str], **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'columnAttach2', val)

    @_f.addMelDocs('colorSliderGrp', 'columnAttach3')
    def columnAttach3(self, val=True, **kwargs):
        # type: (Tuple[str, str, str], **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'columnAttach3', val)

    @_f.addMelDocs('colorSliderGrp', 'columnAttach4')
    def columnAttach4(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'columnAttach4', val)

    @_f.addMelDocs('colorSliderGrp', 'columnAttach5')
    def columnAttach5(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'columnAttach5', val)

    @_f.addMelDocs('colorSliderGrp', 'columnAttach6')
    def columnAttach6(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'columnAttach6', val)

    @_f.addMelDocs('colorSliderGrp', 'columnOffset2')
    def columnOffset2(self, val=True, **kwargs):
        # type: (Tuple[int, int], **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'columnOffset2', val)

    @_f.addMelDocs('colorSliderGrp', 'columnOffset3')
    def columnOffset3(self, val=True, **kwargs):
        # type: (Tuple[int, int, int], **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'columnOffset3', val)

    @_f.addMelDocs('colorSliderGrp', 'columnOffset4')
    def columnOffset4(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'columnOffset4', val)

    @_f.addMelDocs('colorSliderGrp', 'columnOffset5')
    def columnOffset5(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'columnOffset5', val)

    @_f.addMelDocs('colorSliderGrp', 'columnOffset6')
    def columnOffset6(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'columnOffset6', val)

    @_f.addMelDocs('colorSliderGrp', 'columnWidth')
    def columnWidth(self, val=True, **kwargs):
        # type: (Tuple[int, int] | List[Tuple[int, int]], **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'columnWidth', val)

    @_f.addMelDocs('colorSliderGrp', 'columnWidth1')
    def columnWidth1(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'columnWidth1', val)

    @_f.addMelDocs('colorSliderGrp', 'columnWidth2')
    def columnWidth2(self, val=True, **kwargs):
        # type: (Tuple[int, int], **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'columnWidth2', val)

    @_f.addMelDocs('colorSliderGrp', 'columnWidth3')
    def columnWidth3(self, val=True, **kwargs):
        # type: (Tuple[int, int, int], **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'columnWidth3', val)

    @_f.addMelDocs('colorSliderGrp', 'columnWidth4')
    def columnWidth4(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'columnWidth4', val)

    @_f.addMelDocs('colorSliderGrp', 'columnWidth5')
    def columnWidth5(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'columnWidth5', val)

    @_f.addMelDocs('colorSliderGrp', 'columnWidth6')
    def columnWidth6(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'columnWidth6', val)

    @_f.addMelDocs('colorSliderGrp', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'dragCallback', val)

    @_f.addMelDocs('colorSliderGrp', 'dragCommand')
    def dragCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'dragCommand', val)

    @_f.addMelDocs('colorSliderGrp', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'dropCallback', val)

    @_f.addMelDocs('colorSliderGrp', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.colorSliderGrp, kwargs, 'annotation')
        return res

    @_f.addMelDocs('colorSliderGrp', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.colorSliderGrp, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('colorSliderGrp', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.colorSliderGrp, kwargs, 'docTag')
        return res

    @_f.addMelDocs('colorSliderGrp', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.colorSliderGrp, kwargs, 'enable')
        return res

    @_f.addMelDocs('colorSliderGrp', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.colorSliderGrp, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('colorSliderGrp', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.colorSliderGrp, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('colorSliderGrp', 'forceDragRefresh')
    def getForceDragRefresh(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.colorSliderGrp, kwargs, 'forceDragRefresh')
        return res

    @_f.addMelDocs('colorSliderGrp', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.colorSliderGrp, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('colorSliderGrp', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.colorSliderGrp, kwargs, 'height')
        return res

    @_f.addMelDocs('colorSliderGrp', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.colorSliderGrp, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('colorSliderGrp', 'hsvValue')
    def getHsvValue(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.colorSliderGrp, kwargs, 'hsvValue')
        return res

    @_f.addMelDocs('colorSliderGrp', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.colorSliderGrp, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('colorSliderGrp', 'label')
    def getLabel(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.colorSliderGrp, kwargs, 'label')
        return res

    @_f.addMelDocs('colorSliderGrp', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.colorSliderGrp, kwargs, 'manage')
        return res

    @_f.addMelDocs('colorSliderGrp', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.colorSliderGrp, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('colorSliderGrp', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.colorSliderGrp, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('colorSliderGrp', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.colorSliderGrp, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('colorSliderGrp', 'rgbValue')
    def getRgbValue(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.colorSliderGrp, kwargs, 'rgbValue')
        return res

    @_f.addMelDocs('colorSliderGrp', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.colorSliderGrp, kwargs, 'visible')
        return res

    @_f.addMelDocs('colorSliderGrp', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.colorSliderGrp, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('colorSliderGrp', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.colorSliderGrp, kwargs, 'width')
        return res

    @_f.addMelDocs('colorSliderGrp', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'noBackground', val)

    @_f.addMelDocs('colorSliderGrp', 'rowAttach')
    def rowAttach(self, val=True, **kwargs):
        # type: (Tuple[int, str, int] | List[Tuple[int, str, int]], **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'rowAttach', val)

    @_f.addMelDocs('colorSliderGrp', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'annotation', val)

    @_f.addMelDocs('colorSliderGrp', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('colorSliderGrp', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'docTag', val)

    @_f.addMelDocs('colorSliderGrp', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'enable', val)

    @_f.addMelDocs('colorSliderGrp', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'enableBackground', val)

    @_f.addMelDocs('colorSliderGrp', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('colorSliderGrp', 'forceDragRefresh')
    def setForceDragRefresh(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'forceDragRefresh', val)

    @_f.addMelDocs('colorSliderGrp', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'height', val)

    @_f.addMelDocs('colorSliderGrp', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'highlightColor', val)

    @_f.addMelDocs('colorSliderGrp', 'hsvValue')
    def setHsvValue(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'hsvValue', val)

    @_f.addMelDocs('colorSliderGrp', 'label')
    def setLabel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'label', val)

    @_f.addMelDocs('colorSliderGrp', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'manage', val)

    @_f.addMelDocs('colorSliderGrp', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'preventOverride', val)

    @_f.addMelDocs('colorSliderGrp', 'rgbValue')
    def setRgbValue(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'rgbValue', val)

    @_f.addMelDocs('colorSliderGrp', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'visible', val)

    @_f.addMelDocs('colorSliderGrp', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('colorSliderGrp', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'width', val)

    @_f.addMelDocs('colorSliderGrp', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.colorSliderGrp, kwargs, 'statusBarMessage', val)


class ColumnLayout(Layout):
    __melcmd__ = staticmethod(windows.columnLayout)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'columnLayout'
    __melui__ = 'columnLayout'
    __slots__ = ()

    @_f.addMelDocs('columnLayout', 'adjustableColumn')
    def adjustableColumn(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.columnLayout, kwargs, 'adjustableColumn', val)

    @_f.addMelDocs('columnLayout', 'columnAlign')
    def columnAlign(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.columnLayout, kwargs, 'columnAlign', val)

    @_f.addMelDocs('columnLayout', 'columnAttach')
    def columnAttach(self, val=True, **kwargs):
        # type: (Tuple[str, int], **Any) -> None
        return _f.asEdit(self, windows.columnLayout, kwargs, 'columnAttach', val)

    @_f.addMelDocs('columnLayout', 'columnOffset')
    def columnOffset(self, val=True, **kwargs):
        # type: (Tuple[str, int], **Any) -> None
        return _f.asEdit(self, windows.columnLayout, kwargs, 'columnOffset', val)

    @_f.addMelDocs('columnLayout', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.columnLayout, kwargs, 'dragCallback', val)

    @_f.addMelDocs('columnLayout', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.columnLayout, kwargs, 'dropCallback', val)

    @_f.addMelDocs('columnLayout', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.columnLayout, kwargs, 'annotation')
        return res

    @_f.addMelDocs('columnLayout', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.columnLayout, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('columnLayout', 'childArray')
    def getChildArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.columnLayout, kwargs, 'childArray')
        return res

    @_f.addMelDocs('columnLayout', 'columnWidth')
    def getColumnWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.columnLayout, kwargs, 'columnWidth')
        return res

    @_f.addMelDocs('columnLayout', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.columnLayout, kwargs, 'docTag')
        return res

    @_f.addMelDocs('columnLayout', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.columnLayout, kwargs, 'enable')
        return res

    @_f.addMelDocs('columnLayout', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.columnLayout, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('columnLayout', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.columnLayout, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('columnLayout', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.columnLayout, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('columnLayout', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.columnLayout, kwargs, 'height')
        return res

    @_f.addMelDocs('columnLayout', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.columnLayout, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('columnLayout', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.columnLayout, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('columnLayout', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.columnLayout, kwargs, 'manage')
        return res

    @_f.addMelDocs('columnLayout', 'numberOfChildren')
    def getNumberOfChildren(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.columnLayout, kwargs, 'numberOfChildren')
        return res

    @_f.addMelDocs('columnLayout', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.columnLayout, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('columnLayout', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.columnLayout, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('columnLayout', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.columnLayout, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('columnLayout', 'rowSpacing')
    def getRowSpacing(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.columnLayout, kwargs, 'rowSpacing')
        return res

    @_f.addMelDocs('columnLayout', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.columnLayout, kwargs, 'visible')
        return res

    @_f.addMelDocs('columnLayout', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.columnLayout, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('columnLayout', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.columnLayout, kwargs, 'width')
        return res

    @_f.addMelDocs('columnLayout', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.columnLayout, kwargs, 'noBackground', val)

    @_f.addMelDocs('columnLayout', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.columnLayout, kwargs, 'annotation', val)

    @_f.addMelDocs('columnLayout', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.columnLayout, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('columnLayout', 'columnWidth')
    def setColumnWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.columnLayout, kwargs, 'columnWidth', val)

    @_f.addMelDocs('columnLayout', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.columnLayout, kwargs, 'docTag', val)

    @_f.addMelDocs('columnLayout', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.columnLayout, kwargs, 'enable', val)

    @_f.addMelDocs('columnLayout', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.columnLayout, kwargs, 'enableBackground', val)

    @_f.addMelDocs('columnLayout', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.columnLayout, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('columnLayout', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.columnLayout, kwargs, 'height', val)

    @_f.addMelDocs('columnLayout', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.columnLayout, kwargs, 'highlightColor', val)

    @_f.addMelDocs('columnLayout', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.columnLayout, kwargs, 'manage', val)

    @_f.addMelDocs('columnLayout', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.columnLayout, kwargs, 'preventOverride', val)

    @_f.addMelDocs('columnLayout', 'rowSpacing')
    def setRowSpacing(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.columnLayout, kwargs, 'rowSpacing', val)

    @_f.addMelDocs('columnLayout', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.columnLayout, kwargs, 'visible', val)

    @_f.addMelDocs('columnLayout', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.columnLayout, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('columnLayout', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.columnLayout, kwargs, 'width', val)

    @_f.addMelDocs('columnLayout', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.columnLayout, kwargs, 'statusBarMessage', val)


class FloatSliderButtonGrp(Layout):
    __melcmd__ = staticmethod(windows.floatSliderButtonGrp)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'floatSliderButtonGrp'
    __melui__ = 'floatSliderButtonGrp'
    __slots__ = ()

    @_f.addMelDocs('floatSliderButtonGrp', 'adjustableColumn')
    def adjustableColumn(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'adjustableColumn', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'adjustableColumn2')
    def adjustableColumn2(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'adjustableColumn2', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'adjustableColumn3')
    def adjustableColumn3(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'adjustableColumn3', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'adjustableColumn4')
    def adjustableColumn4(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'adjustableColumn4', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'adjustableColumn5')
    def adjustableColumn5(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'adjustableColumn5', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'adjustableColumn6')
    def adjustableColumn6(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'adjustableColumn6', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'buttonCommand')
    def buttonCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'buttonCommand', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'changeCommand')
    def changeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'changeCommand', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'columnAlign')
    def columnAlign(self, val=True, **kwargs):
        # type: (Tuple[int, str] | List[Tuple[int, str]], **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'columnAlign', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'columnAlign2')
    def columnAlign2(self, val=True, **kwargs):
        # type: (Tuple[str, str], **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'columnAlign2', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'columnAlign3')
    def columnAlign3(self, val=True, **kwargs):
        # type: (Tuple[str, str, str], **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'columnAlign3', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'columnAlign4')
    def columnAlign4(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'columnAlign4', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'columnAlign5')
    def columnAlign5(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'columnAlign5', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'columnAlign6')
    def columnAlign6(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'columnAlign6', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'columnAttach')
    def columnAttach(self, val=True, **kwargs):
        # type: (Tuple[int, str, int] | List[Tuple[int, str, int]], **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'columnAttach', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'columnAttach2')
    def columnAttach2(self, val=True, **kwargs):
        # type: (Tuple[str, str], **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'columnAttach2', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'columnAttach3')
    def columnAttach3(self, val=True, **kwargs):
        # type: (Tuple[str, str, str], **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'columnAttach3', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'columnAttach4')
    def columnAttach4(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'columnAttach4', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'columnAttach5')
    def columnAttach5(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'columnAttach5', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'columnAttach6')
    def columnAttach6(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'columnAttach6', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'columnOffset2')
    def columnOffset2(self, val=True, **kwargs):
        # type: (Tuple[int, int], **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'columnOffset2', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'columnOffset3')
    def columnOffset3(self, val=True, **kwargs):
        # type: (Tuple[int, int, int], **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'columnOffset3', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'columnOffset4')
    def columnOffset4(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'columnOffset4', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'columnOffset5')
    def columnOffset5(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'columnOffset5', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'columnOffset6')
    def columnOffset6(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'columnOffset6', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'columnWidth')
    def columnWidth(self, val=True, **kwargs):
        # type: (Tuple[int, int] | List[Tuple[int, int]], **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'columnWidth', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'columnWidth1')
    def columnWidth1(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'columnWidth1', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'columnWidth2')
    def columnWidth2(self, val=True, **kwargs):
        # type: (Tuple[int, int], **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'columnWidth2', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'columnWidth3')
    def columnWidth3(self, val=True, **kwargs):
        # type: (Tuple[int, int, int], **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'columnWidth3', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'columnWidth4')
    def columnWidth4(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'columnWidth4', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'columnWidth5')
    def columnWidth5(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'columnWidth5', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'columnWidth6')
    def columnWidth6(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'columnWidth6', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'dragCallback', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'dragCommand')
    def dragCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'dragCommand', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'dropCallback', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.floatSliderButtonGrp, kwargs, 'annotation')
        return res

    @_f.addMelDocs('floatSliderButtonGrp', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.floatSliderButtonGrp, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('floatSliderButtonGrp', 'buttonLabel')
    def getButtonLabel(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.floatSliderButtonGrp, kwargs, 'buttonLabel')
        return res

    @_f.addMelDocs('floatSliderButtonGrp', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.floatSliderButtonGrp, kwargs, 'docTag')
        return res

    @_f.addMelDocs('floatSliderButtonGrp', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatSliderButtonGrp, kwargs, 'enable')
        return res

    @_f.addMelDocs('floatSliderButtonGrp', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatSliderButtonGrp, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('floatSliderButtonGrp', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatSliderButtonGrp, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('floatSliderButtonGrp', 'extraLabel')
    def getExtraLabel(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.floatSliderButtonGrp, kwargs, 'extraLabel')
        return res

    @_f.addMelDocs('floatSliderButtonGrp', 'fieldMaxValue')
    def getFieldMaxValue(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.floatSliderButtonGrp, kwargs, 'fieldMaxValue')
        return res

    @_f.addMelDocs('floatSliderButtonGrp', 'fieldMinValue')
    def getFieldMinValue(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.floatSliderButtonGrp, kwargs, 'fieldMinValue')
        return res

    @_f.addMelDocs('floatSliderButtonGrp', 'fieldStep')
    def getFieldStep(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.floatSliderButtonGrp, kwargs, 'fieldStep')
        return res

    @_f.addMelDocs('floatSliderButtonGrp', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.floatSliderButtonGrp, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('floatSliderButtonGrp', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.floatSliderButtonGrp, kwargs, 'height')
        return res

    @_f.addMelDocs('floatSliderButtonGrp', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.floatSliderButtonGrp, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('floatSliderButtonGrp', 'image')
    def getImage(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.floatSliderButtonGrp, kwargs, 'image')
        return res

    @_f.addMelDocs('floatSliderButtonGrp', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.floatSliderButtonGrp, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('floatSliderButtonGrp', 'label')
    def getLabel(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.floatSliderButtonGrp, kwargs, 'label')
        return res

    @_f.addMelDocs('floatSliderButtonGrp', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatSliderButtonGrp, kwargs, 'manage')
        return res

    @_f.addMelDocs('floatSliderButtonGrp', 'maxValue')
    def getMaxValue(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.floatSliderButtonGrp, kwargs, 'maxValue')
        return res

    @_f.addMelDocs('floatSliderButtonGrp', 'minValue')
    def getMinValue(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.floatSliderButtonGrp, kwargs, 'minValue')
        return res

    @_f.addMelDocs('floatSliderButtonGrp', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.floatSliderButtonGrp, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('floatSliderButtonGrp', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.floatSliderButtonGrp, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('floatSliderButtonGrp', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatSliderButtonGrp, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('floatSliderButtonGrp', 'sliderStep')
    def getSliderStep(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.floatSliderButtonGrp, kwargs, 'sliderStep')
        return res

    @_f.addMelDocs('floatSliderButtonGrp', 'step')
    def getStep(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.floatSliderButtonGrp, kwargs, 'step')
        return res

    @_f.addMelDocs('floatSliderButtonGrp', 'symbolButtonDisplay')
    def getSymbolButtonDisplay(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatSliderButtonGrp, kwargs, 'symbolButtonDisplay')
        return res

    @_f.addMelDocs('floatSliderButtonGrp', 'value')
    def getValue(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.floatSliderButtonGrp, kwargs, 'value')
        return res

    @_f.addMelDocs('floatSliderButtonGrp', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatSliderButtonGrp, kwargs, 'visible')
        return res

    @_f.addMelDocs('floatSliderButtonGrp', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.floatSliderButtonGrp, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('floatSliderButtonGrp', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.floatSliderButtonGrp, kwargs, 'width')
        return res

    @_f.addMelDocs('floatSliderButtonGrp', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'noBackground', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'precision')
    def precision(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'precision', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'rowAttach')
    def rowAttach(self, val=True, **kwargs):
        # type: (Tuple[int, str, int] | List[Tuple[int, str, int]], **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'rowAttach', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'annotation', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'buttonLabel')
    def setButtonLabel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'buttonLabel', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'docTag', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'enable', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'enableBackground', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'extraLabel')
    def setExtraLabel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'extraLabel', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'fieldMaxValue')
    def setFieldMaxValue(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'fieldMaxValue', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'fieldMinValue')
    def setFieldMinValue(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'fieldMinValue', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'fieldStep')
    def setFieldStep(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'fieldStep', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'height', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'highlightColor', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'image')
    def setImage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'image', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'label')
    def setLabel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'label', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'manage', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'maxValue')
    def setMaxValue(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'maxValue', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'minValue')
    def setMinValue(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'minValue', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'preventOverride', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'sliderStep')
    def setSliderStep(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'sliderStep', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'step')
    def setStep(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'step', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'symbolButtonDisplay')
    def setSymbolButtonDisplay(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'symbolButtonDisplay', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'value')
    def setValue(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'value', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'visible', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'width', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'statusBarMessage', val)

    @_f.addMelDocs('floatSliderButtonGrp', 'symbolButtonCommand')
    def symbolButtonCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.floatSliderButtonGrp, kwargs, 'symbolButtonCommand', val)


class FloatSliderGrp(Layout):
    __melcmd__ = staticmethod(windows.floatSliderGrp)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'floatSliderGrp'
    __melui__ = 'floatSliderGrp'
    __slots__ = ()

    @_f.addMelDocs('floatSliderGrp', 'adjustableColumn')
    def adjustableColumn(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'adjustableColumn', val)

    @_f.addMelDocs('floatSliderGrp', 'adjustableColumn2')
    def adjustableColumn2(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'adjustableColumn2', val)

    @_f.addMelDocs('floatSliderGrp', 'adjustableColumn3')
    def adjustableColumn3(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'adjustableColumn3', val)

    @_f.addMelDocs('floatSliderGrp', 'adjustableColumn4')
    def adjustableColumn4(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'adjustableColumn4', val)

    @_f.addMelDocs('floatSliderGrp', 'adjustableColumn5')
    def adjustableColumn5(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'adjustableColumn5', val)

    @_f.addMelDocs('floatSliderGrp', 'adjustableColumn6')
    def adjustableColumn6(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'adjustableColumn6', val)

    @_f.addMelDocs('floatSliderGrp', 'changeCommand')
    def changeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'changeCommand', val)

    @_f.addMelDocs('floatSliderGrp', 'columnAlign')
    def columnAlign(self, val=True, **kwargs):
        # type: (Tuple[int, str] | List[Tuple[int, str]], **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'columnAlign', val)

    @_f.addMelDocs('floatSliderGrp', 'columnAlign2')
    def columnAlign2(self, val=True, **kwargs):
        # type: (Tuple[str, str], **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'columnAlign2', val)

    @_f.addMelDocs('floatSliderGrp', 'columnAlign3')
    def columnAlign3(self, val=True, **kwargs):
        # type: (Tuple[str, str, str], **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'columnAlign3', val)

    @_f.addMelDocs('floatSliderGrp', 'columnAlign4')
    def columnAlign4(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'columnAlign4', val)

    @_f.addMelDocs('floatSliderGrp', 'columnAlign5')
    def columnAlign5(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'columnAlign5', val)

    @_f.addMelDocs('floatSliderGrp', 'columnAlign6')
    def columnAlign6(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'columnAlign6', val)

    @_f.addMelDocs('floatSliderGrp', 'columnAttach')
    def columnAttach(self, val=True, **kwargs):
        # type: (Tuple[int, str, int] | List[Tuple[int, str, int]], **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'columnAttach', val)

    @_f.addMelDocs('floatSliderGrp', 'columnAttach2')
    def columnAttach2(self, val=True, **kwargs):
        # type: (Tuple[str, str], **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'columnAttach2', val)

    @_f.addMelDocs('floatSliderGrp', 'columnAttach3')
    def columnAttach3(self, val=True, **kwargs):
        # type: (Tuple[str, str, str], **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'columnAttach3', val)

    @_f.addMelDocs('floatSliderGrp', 'columnAttach4')
    def columnAttach4(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'columnAttach4', val)

    @_f.addMelDocs('floatSliderGrp', 'columnAttach5')
    def columnAttach5(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'columnAttach5', val)

    @_f.addMelDocs('floatSliderGrp', 'columnAttach6')
    def columnAttach6(self, val=True, **kwargs):
        # type: (Tuple[str, str, str, str, str, str], **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'columnAttach6', val)

    @_f.addMelDocs('floatSliderGrp', 'columnOffset2')
    def columnOffset2(self, val=True, **kwargs):
        # type: (Tuple[int, int], **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'columnOffset2', val)

    @_f.addMelDocs('floatSliderGrp', 'columnOffset3')
    def columnOffset3(self, val=True, **kwargs):
        # type: (Tuple[int, int, int], **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'columnOffset3', val)

    @_f.addMelDocs('floatSliderGrp', 'columnOffset4')
    def columnOffset4(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'columnOffset4', val)

    @_f.addMelDocs('floatSliderGrp', 'columnOffset5')
    def columnOffset5(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'columnOffset5', val)

    @_f.addMelDocs('floatSliderGrp', 'columnOffset6')
    def columnOffset6(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'columnOffset6', val)

    @_f.addMelDocs('floatSliderGrp', 'columnWidth')
    def columnWidth(self, val=True, **kwargs):
        # type: (Tuple[int, int] | List[Tuple[int, int]], **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'columnWidth', val)

    @_f.addMelDocs('floatSliderGrp', 'columnWidth1')
    def columnWidth1(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'columnWidth1', val)

    @_f.addMelDocs('floatSliderGrp', 'columnWidth2')
    def columnWidth2(self, val=True, **kwargs):
        # type: (Tuple[int, int], **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'columnWidth2', val)

    @_f.addMelDocs('floatSliderGrp', 'columnWidth3')
    def columnWidth3(self, val=True, **kwargs):
        # type: (Tuple[int, int, int], **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'columnWidth3', val)

    @_f.addMelDocs('floatSliderGrp', 'columnWidth4')
    def columnWidth4(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'columnWidth4', val)

    @_f.addMelDocs('floatSliderGrp', 'columnWidth5')
    def columnWidth5(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'columnWidth5', val)

    @_f.addMelDocs('floatSliderGrp', 'columnWidth6')
    def columnWidth6(self, val=True, **kwargs):
        # type: (Tuple[int, int, int, int, int, int], **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'columnWidth6', val)

    @_f.addMelDocs('floatSliderGrp', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'dragCallback', val)

    @_f.addMelDocs('floatSliderGrp', 'dragCommand')
    def dragCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'dragCommand', val)

    @_f.addMelDocs('floatSliderGrp', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'dropCallback', val)

    @_f.addMelDocs('floatSliderGrp', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.floatSliderGrp, kwargs, 'annotation')
        return res

    @_f.addMelDocs('floatSliderGrp', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.floatSliderGrp, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('floatSliderGrp', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.floatSliderGrp, kwargs, 'docTag')
        return res

    @_f.addMelDocs('floatSliderGrp', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatSliderGrp, kwargs, 'enable')
        return res

    @_f.addMelDocs('floatSliderGrp', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatSliderGrp, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('floatSliderGrp', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatSliderGrp, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('floatSliderGrp', 'extraLabel')
    def getExtraLabel(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.floatSliderGrp, kwargs, 'extraLabel')
        return res

    @_f.addMelDocs('floatSliderGrp', 'fieldMaxValue')
    def getFieldMaxValue(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.floatSliderGrp, kwargs, 'fieldMaxValue')
        return res

    @_f.addMelDocs('floatSliderGrp', 'fieldMinValue')
    def getFieldMinValue(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.floatSliderGrp, kwargs, 'fieldMinValue')
        return res

    @_f.addMelDocs('floatSliderGrp', 'fieldStep')
    def getFieldStep(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.floatSliderGrp, kwargs, 'fieldStep')
        return res

    @_f.addMelDocs('floatSliderGrp', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.floatSliderGrp, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('floatSliderGrp', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.floatSliderGrp, kwargs, 'height')
        return res

    @_f.addMelDocs('floatSliderGrp', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.floatSliderGrp, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('floatSliderGrp', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.floatSliderGrp, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('floatSliderGrp', 'label')
    def getLabel(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.floatSliderGrp, kwargs, 'label')
        return res

    @_f.addMelDocs('floatSliderGrp', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatSliderGrp, kwargs, 'manage')
        return res

    @_f.addMelDocs('floatSliderGrp', 'maxValue')
    def getMaxValue(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.floatSliderGrp, kwargs, 'maxValue')
        return res

    @_f.addMelDocs('floatSliderGrp', 'minValue')
    def getMinValue(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.floatSliderGrp, kwargs, 'minValue')
        return res

    @_f.addMelDocs('floatSliderGrp', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.floatSliderGrp, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('floatSliderGrp', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.floatSliderGrp, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('floatSliderGrp', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatSliderGrp, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('floatSliderGrp', 'sliderStep')
    def getSliderStep(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.floatSliderGrp, kwargs, 'sliderStep')
        return res

    @_f.addMelDocs('floatSliderGrp', 'step')
    def getStep(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.floatSliderGrp, kwargs, 'step')
        return res

    @_f.addMelDocs('floatSliderGrp', 'value')
    def getValue(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, windows.floatSliderGrp, kwargs, 'value')
        return res

    @_f.addMelDocs('floatSliderGrp', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.floatSliderGrp, kwargs, 'visible')
        return res

    @_f.addMelDocs('floatSliderGrp', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.floatSliderGrp, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('floatSliderGrp', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.floatSliderGrp, kwargs, 'width')
        return res

    @_f.addMelDocs('floatSliderGrp', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'noBackground', val)

    @_f.addMelDocs('floatSliderGrp', 'precision')
    def precision(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'precision', val)

    @_f.addMelDocs('floatSliderGrp', 'rowAttach')
    def rowAttach(self, val=True, **kwargs):
        # type: (Tuple[int, str, int] | List[Tuple[int, str, int]], **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'rowAttach', val)

    @_f.addMelDocs('floatSliderGrp', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'annotation', val)

    @_f.addMelDocs('floatSliderGrp', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('floatSliderGrp', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'docTag', val)

    @_f.addMelDocs('floatSliderGrp', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'enable', val)

    @_f.addMelDocs('floatSliderGrp', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'enableBackground', val)

    @_f.addMelDocs('floatSliderGrp', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('floatSliderGrp', 'extraLabel')
    def setExtraLabel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'extraLabel', val)

    @_f.addMelDocs('floatSliderGrp', 'fieldMaxValue')
    def setFieldMaxValue(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'fieldMaxValue', val)

    @_f.addMelDocs('floatSliderGrp', 'fieldMinValue')
    def setFieldMinValue(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'fieldMinValue', val)

    @_f.addMelDocs('floatSliderGrp', 'fieldStep')
    def setFieldStep(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'fieldStep', val)

    @_f.addMelDocs('floatSliderGrp', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'height', val)

    @_f.addMelDocs('floatSliderGrp', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'highlightColor', val)

    @_f.addMelDocs('floatSliderGrp', 'label')
    def setLabel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'label', val)

    @_f.addMelDocs('floatSliderGrp', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'manage', val)

    @_f.addMelDocs('floatSliderGrp', 'maxValue')
    def setMaxValue(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'maxValue', val)

    @_f.addMelDocs('floatSliderGrp', 'minValue')
    def setMinValue(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'minValue', val)

    @_f.addMelDocs('floatSliderGrp', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'preventOverride', val)

    @_f.addMelDocs('floatSliderGrp', 'sliderStep')
    def setSliderStep(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'sliderStep', val)

    @_f.addMelDocs('floatSliderGrp', 'step')
    def setStep(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'step', val)

    @_f.addMelDocs('floatSliderGrp', 'value')
    def setValue(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'value', val)

    @_f.addMelDocs('floatSliderGrp', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'visible', val)

    @_f.addMelDocs('floatSliderGrp', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('floatSliderGrp', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'width', val)

    @_f.addMelDocs('floatSliderGrp', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.floatSliderGrp, kwargs, 'statusBarMessage', val)


class FrameLayout(Layout):
    __melcmd__ = staticmethod(windows.frameLayout)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'frameLayout'
    __melui__ = 'frameLayout'
    __slots__ = ()

    @_f.addMelDocs('frameLayout', 'collapseCommand')
    def collapseCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.frameLayout, kwargs, 'collapseCommand', val)

    @_f.addMelDocs('frameLayout', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.frameLayout, kwargs, 'dragCallback', val)

    @_f.addMelDocs('frameLayout', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.frameLayout, kwargs, 'dropCallback', val)

    @_f.addMelDocs('frameLayout', 'expandCommand')
    def expandCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.frameLayout, kwargs, 'expandCommand', val)

    @_f.addMelDocs('frameLayout', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.frameLayout, kwargs, 'annotation')
        return res

    @_f.addMelDocs('frameLayout', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.frameLayout, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('frameLayout', 'backgroundShade')
    def getBackgroundShade(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.frameLayout, kwargs, 'backgroundShade')
        return res

    @_f.addMelDocs('frameLayout', 'borderStyle')
    def getBorderStyle(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.frameLayout, kwargs, 'borderStyle')
        return res

    @_f.addMelDocs('frameLayout', 'borderVisible')
    def getBorderVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.frameLayout, kwargs, 'borderVisible')
        return res

    @_f.addMelDocs('frameLayout', 'childArray')
    def getChildArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.frameLayout, kwargs, 'childArray')
        return res

    @_f.addMelDocs('frameLayout', 'collapsable')
    def getCollapsable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.frameLayout, kwargs, 'collapsable')
        return res

    @_f.addMelDocs('frameLayout', 'collapse')
    def getCollapse(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.frameLayout, kwargs, 'collapse')
        return res

    @_f.addMelDocs('frameLayout', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.frameLayout, kwargs, 'docTag')
        return res

    @_f.addMelDocs('frameLayout', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.frameLayout, kwargs, 'enable')
        return res

    @_f.addMelDocs('frameLayout', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.frameLayout, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('frameLayout', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.frameLayout, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('frameLayout', 'font')
    def getFont(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.frameLayout, kwargs, 'font')
        return res

    @_f.addMelDocs('frameLayout', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.frameLayout, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('frameLayout', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.frameLayout, kwargs, 'height')
        return res

    @_f.addMelDocs('frameLayout', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.frameLayout, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('frameLayout', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.frameLayout, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('frameLayout', 'label')
    def getLabel(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.frameLayout, kwargs, 'label')
        return res

    @_f.addMelDocs('frameLayout', 'labelAlign')
    def getLabelAlign(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.frameLayout, kwargs, 'labelAlign')
        return res

    @_f.addMelDocs('frameLayout', 'labelIndent')
    def getLabelIndent(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.frameLayout, kwargs, 'labelIndent')
        return res

    @_f.addMelDocs('frameLayout', 'labelVisible')
    def getLabelVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.frameLayout, kwargs, 'labelVisible')
        return res

    @_f.addMelDocs('frameLayout', 'labelWidth')
    def getLabelWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.frameLayout, kwargs, 'labelWidth')
        return res

    @_f.addMelDocs('frameLayout', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.frameLayout, kwargs, 'manage')
        return res

    @_f.addMelDocs('frameLayout', 'marginHeight')
    def getMarginHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.frameLayout, kwargs, 'marginHeight')
        return res

    @_f.addMelDocs('frameLayout', 'marginWidth')
    def getMarginWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.frameLayout, kwargs, 'marginWidth')
        return res

    @_f.addMelDocs('frameLayout', 'numberOfChildren')
    def getNumberOfChildren(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.frameLayout, kwargs, 'numberOfChildren')
        return res

    @_f.addMelDocs('frameLayout', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.frameLayout, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('frameLayout', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.frameLayout, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('frameLayout', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.frameLayout, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('frameLayout', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.frameLayout, kwargs, 'visible')
        return res

    @_f.addMelDocs('frameLayout', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.frameLayout, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('frameLayout', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.frameLayout, kwargs, 'width')
        return res

    @_f.addMelDocs('frameLayout', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.frameLayout, kwargs, 'noBackground', val)

    @_f.addMelDocs('frameLayout', 'preCollapseCommand')
    def preCollapseCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.frameLayout, kwargs, 'preCollapseCommand', val)

    @_f.addMelDocs('frameLayout', 'preExpandCommand')
    def preExpandCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.frameLayout, kwargs, 'preExpandCommand', val)

    @_f.addMelDocs('frameLayout', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.frameLayout, kwargs, 'annotation', val)

    @_f.addMelDocs('frameLayout', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.frameLayout, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('frameLayout', 'backgroundShade')
    def setBackgroundShade(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.frameLayout, kwargs, 'backgroundShade', val)

    @_f.addMelDocs('frameLayout', 'borderStyle')
    def setBorderStyle(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.frameLayout, kwargs, 'borderStyle', val)

    @_f.addMelDocs('frameLayout', 'borderVisible')
    def setBorderVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.frameLayout, kwargs, 'borderVisible', val)

    @_f.addMelDocs('frameLayout', 'collapsable')
    def setCollapsable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.frameLayout, kwargs, 'collapsable', val)

    @_f.addMelDocs('frameLayout', 'collapse')
    def setCollapse(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.frameLayout, kwargs, 'collapse', val)

    @_f.addMelDocs('frameLayout', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.frameLayout, kwargs, 'docTag', val)

    @_f.addMelDocs('frameLayout', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.frameLayout, kwargs, 'enable', val)

    @_f.addMelDocs('frameLayout', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.frameLayout, kwargs, 'enableBackground', val)

    @_f.addMelDocs('frameLayout', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.frameLayout, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('frameLayout', 'font')
    def setFont(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.frameLayout, kwargs, 'font', val)

    @_f.addMelDocs('frameLayout', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.frameLayout, kwargs, 'height', val)

    @_f.addMelDocs('frameLayout', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.frameLayout, kwargs, 'highlightColor', val)

    @_f.addMelDocs('frameLayout', 'label')
    def setLabel(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.frameLayout, kwargs, 'label', val)

    @_f.addMelDocs('frameLayout', 'labelAlign')
    def setLabelAlign(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.frameLayout, kwargs, 'labelAlign', val)

    @_f.addMelDocs('frameLayout', 'labelIndent')
    def setLabelIndent(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.frameLayout, kwargs, 'labelIndent', val)

    @_f.addMelDocs('frameLayout', 'labelVisible')
    def setLabelVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.frameLayout, kwargs, 'labelVisible', val)

    @_f.addMelDocs('frameLayout', 'labelWidth')
    def setLabelWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.frameLayout, kwargs, 'labelWidth', val)

    @_f.addMelDocs('frameLayout', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.frameLayout, kwargs, 'manage', val)

    @_f.addMelDocs('frameLayout', 'marginHeight')
    def setMarginHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.frameLayout, kwargs, 'marginHeight', val)

    @_f.addMelDocs('frameLayout', 'marginWidth')
    def setMarginWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.frameLayout, kwargs, 'marginWidth', val)

    @_f.addMelDocs('frameLayout', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.frameLayout, kwargs, 'preventOverride', val)

    @_f.addMelDocs('frameLayout', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.frameLayout, kwargs, 'visible', val)

    @_f.addMelDocs('frameLayout', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.frameLayout, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('frameLayout', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.frameLayout, kwargs, 'width', val)

    @_f.addMelDocs('frameLayout', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.frameLayout, kwargs, 'statusBarMessage', val)


class GridLayout(Layout):
    __melcmd__ = staticmethod(windows.gridLayout)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'gridLayout'
    __melui__ = 'gridLayout'
    __slots__ = ()

    @_f.addMelDocs('gridLayout', 'cellWidthHeight')
    def cellWidthHeight(self, val=True, **kwargs):
        # type: (Tuple[int, int], **Any) -> None
        return _f.asEdit(self, windows.gridLayout, kwargs, 'cellWidthHeight', val)

    @_f.addMelDocs('gridLayout', 'dragCallback')
    def dragCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.gridLayout, kwargs, 'dragCallback', val)

    @_f.addMelDocs('gridLayout', 'dropCallback')
    def dropCallback(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.gridLayout, kwargs, 'dropCallback', val)

    @_f.addMelDocs('gridLayout', 'allowEmptyCells')
    def getAllowEmptyCells(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.gridLayout, kwargs, 'allowEmptyCells')
        return res

    @_f.addMelDocs('gridLayout', 'annotation')
    def getAnnotation(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.gridLayout, kwargs, 'annotation')
        return res

    @_f.addMelDocs('gridLayout', 'autoGrow')
    def getAutoGrow(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.gridLayout, kwargs, 'autoGrow')
        return res

    @_f.addMelDocs('gridLayout', 'backgroundColor')
    def getBackgroundColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.gridLayout, kwargs, 'backgroundColor')
        return res

    @_f.addMelDocs('gridLayout', 'cellHeight')
    def getCellHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.gridLayout, kwargs, 'cellHeight')
        return res

    @_f.addMelDocs('gridLayout', 'cellWidth')
    def getCellWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.gridLayout, kwargs, 'cellWidth')
        return res

    @_f.addMelDocs('gridLayout', 'childArray')
    def getChildArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.gridLayout, kwargs, 'childArray')
        return res

    @_f.addMelDocs('gridLayout', 'columnsResizable')
    def getColumnsResizable(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.gridLayout, kwargs, 'columnsResizable')
        return res

    @_f.addMelDocs('gridLayout', 'docTag')
    def getDocTag(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, windows.gridLayout, kwargs, 'docTag')
        return res

    @_f.addMelDocs('gridLayout', 'enable')
    def getEnable(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.gridLayout, kwargs, 'enable')
        return res

    @_f.addMelDocs('gridLayout', 'enableBackground')
    def getEnableBackground(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.gridLayout, kwargs, 'enableBackground')
        return res

    @_f.addMelDocs('gridLayout', 'enableKeyboardFocus')
    def getEnableKeyboardFocus(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.gridLayout, kwargs, 'enableKeyboardFocus')
        return res

    @_f.addMelDocs('gridLayout', 'fullPathName')
    def getFullPathName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.gridLayout, kwargs, 'fullPathName')
        return res

    @_f.addMelDocs('gridLayout', 'gridOrder')
    def getGridOrder(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.gridLayout, kwargs, 'gridOrder')
        return res

    @_f.addMelDocs('gridLayout', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.gridLayout, kwargs, 'height')
        return res

    @_f.addMelDocs('gridLayout', 'highlightColor')
    def getHighlightColor(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, windows.gridLayout, kwargs, 'highlightColor')
        return res

    @_f.addMelDocs('gridLayout', 'isObscured')
    def getIsObscured(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.gridLayout, kwargs, 'isObscured')
        return res

    @_f.addMelDocs('gridLayout', 'manage')
    def getManage(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.gridLayout, kwargs, 'manage')
        return res

    @_f.addMelDocs('gridLayout', 'numberOfChildren')
    def getNumberOfChildren(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.gridLayout, kwargs, 'numberOfChildren')
        return res

    @_f.addMelDocs('gridLayout', 'numberOfColumns')
    def getNumberOfColumns(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.gridLayout, kwargs, 'numberOfColumns')
        return res

    @_f.addMelDocs('gridLayout', 'numberOfPopupMenus')
    def getNumberOfPopupMenus(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.gridLayout, kwargs, 'numberOfPopupMenus')
        return res

    @_f.addMelDocs('gridLayout', 'numberOfRows')
    def getNumberOfRows(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.gridLayout, kwargs, 'numberOfRows')
        return res

    @_f.addMelDocs('gridLayout', 'popupMenuArray')
    def getPopupMenuArray(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, windows.gridLayout, kwargs, 'popupMenuArray')
        return res

    @_f.addMelDocs('gridLayout', 'preventOverride')
    def getPreventOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.gridLayout, kwargs, 'preventOverride')
        return res

    @_f.addMelDocs('gridLayout', 'visible')
    def getVisible(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, windows.gridLayout, kwargs, 'visible')
        return res

    @_f.addMelDocs('gridLayout', 'visibleChangeCommand')
    def getVisibleChangeCommand(self, **kwargs):
        # type: (...) -> str | Callable
        res = _f.asQuery(self, windows.gridLayout, kwargs, 'visibleChangeCommand')
        return res

    @_f.addMelDocs('gridLayout', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, windows.gridLayout, kwargs, 'width')
        return res

    @_f.addMelDocs('gridLayout', 'noBackground')
    def noBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.gridLayout, kwargs, 'noBackground', val)

    @_f.addMelDocs('gridLayout', 'numberOfRowsColumns')
    def numberOfRowsColumns(self, val=True, **kwargs):
        # type: (Tuple[int, int], **Any) -> None
        return _f.asEdit(self, windows.gridLayout, kwargs, 'numberOfRowsColumns', val)

    @_f.addMelDocs('gridLayout', 'position')
    def position(self, val=True, **kwargs):
        # type: (Tuple[str, int] | List[Tuple[str, int]], **Any) -> None
        return _f.asEdit(self, windows.gridLayout, kwargs, 'position', val)

    @_f.addMelDocs('gridLayout', 'annotation')
    def setAnnotation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.gridLayout, kwargs, 'annotation', val)

    @_f.addMelDocs('gridLayout', 'backgroundColor')
    def setBackgroundColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.gridLayout, kwargs, 'backgroundColor', val)

    @_f.addMelDocs('gridLayout', 'cellHeight')
    def setCellHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.gridLayout, kwargs, 'cellHeight', val)

    @_f.addMelDocs('gridLayout', 'cellWidth')
    def setCellWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.gridLayout, kwargs, 'cellWidth', val)

    @_f.addMelDocs('gridLayout', 'docTag')
    def setDocTag(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.gridLayout, kwargs, 'docTag', val)

    @_f.addMelDocs('gridLayout', 'enable')
    def setEnable(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.gridLayout, kwargs, 'enable', val)

    @_f.addMelDocs('gridLayout', 'enableBackground')
    def setEnableBackground(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.gridLayout, kwargs, 'enableBackground', val)

    @_f.addMelDocs('gridLayout', 'enableKeyboardFocus')
    def setEnableKeyboardFocus(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.gridLayout, kwargs, 'enableKeyboardFocus', val)

    @_f.addMelDocs('gridLayout', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.gridLayout, kwargs, 'height', val)

    @_f.addMelDocs('gridLayout', 'highlightColor')
    def setHighlightColor(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, windows.gridLayout, kwargs, 'highlightColor', val)

    @_f.addMelDocs('gridLayout', 'manage')
    def setManage(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.gridLayout, kwargs, 'manage', val)

    @_f.addMelDocs('gridLayout', 'numberOfColumns')
    def setNumberOfColumns(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.gridLayout, kwargs, 'numberOfColumns', val)

    @_f.addMelDocs('gridLayout', 'numberOfRows')
    def setNumberOfRows(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.gridLayout, kwargs, 'numberOfRows', val)

    @_f.addMelDocs('gridLayout', 'preventOverride')
    def setPreventOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.gridLayout, kwargs, 'preventOverride', val)

    @_f.addMelDocs('gridLayout', 'visible')
    def setVisible(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, windows.gridLayout, kwargs, 'visible', val)

    @_f.addMelDocs('gridLayout', 'visibleChangeCommand')
    def setVisibleChangeCommand(self, val=True, **kwargs):
        # type: (str | Callable, **Any) -> None
        return _f.asEdit(self, windows.gridLayout, kwargs, 'visibleChangeCommand', val)

    @_f.addMelDocs('gridLayout', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, windows.gridLayout, kwargs, 'width', val)

    @_f.addMelDocs('gridLayout', 'statusBarMessage')
    def statusBarMessage(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, windows.gridLayout, kwargs, 'statusBarMessage', val)


cla