"""
Contains classes corresponding to the Maya type hierarchy, including `DependNode`, `Transform`, `Mesh`, and `Camera`.
"""
from __future__ import print_function
from __future__ import division
from __future__ import absolute_import
from builtins import range
from builtins import str
from past.builtins import basestring
import sys
import os
import re
import inspect
import itertools
import math

import pymel.util as _util

import pymel.internal.factories as _factories
import pymel.internal.apicache as _apicache
import pymel.internal.pwarnings as _warnings
from pymel.internal import getLogger as _getLogger
from pymel.internal.startup import pymel_options as _pymel_options
import pymel.core.datatypes as datatypes

# to make sure Maya is up
import pymel.internal as internal
import pymel.versions as versions

from maya.cmds import about as _about
import maya.mel as mm

from pymel.core import animation
from pymel.core import effects
from pymel.core import general
from pymel.core import modeling
from pymel.core import other
from pymel.core import rendering
from pymel.core import system
from pymel.core import windows

from pymel.core.animation import listAnimatable as _listAnimatable
from pymel.core.system import namespaceInfo as _namespaceInfo, FileReference as _FileReference
from pymel.util.enum import Enum
from future.utils import PY2, with_metaclass

TYPE_CHECKING = False
if TYPE_CHECKING:
    # do not do a wild import of typing here, because typing.Container overrides
    # Container in the stubs
    from typing import Any, Iterable, List, Optional, Sequence, Set, Type, Tuple, Union, overload
    from typing_extensions import Literal
    from maya import cmds
    import maya.OpenMaya as _api

    DependNodeT = TypeVar('DependNodeT', bound=DependNode)
    DagNodeT = TypeVar('DagNodeT', bound=DagNode)
    ShapeT = TypeVar('ShapeT', bound=Shape)

else:
    import pymel.api as _api
    import pymel.internal.pmcmds as cmds  # type: ignore[no-redef]
    from pymel.util.common import overload

_logger = _getLogger(__name__)
_thisModule = sys.modules[__name__]

_f = _factories

#__all__ = ['Component', 'MeshEdge', 'MeshVertex', 'MeshFace', 'Attribute', 'DependNode' ]

# Mesh Components

# If we're reloading, clear the pynode types out
_factories.clearPyNodeTypes()

# for backwards compatibility, need these in this namespace
mayaTypeNameToPymelTypeName = _factories.mayaTypeNameToPymelTypeName
pymelTypeNameToMayaTypeName = _factories.pymelTypeNameToMayaTypeName


class NodetypesLazyLoadModule(_util.LazyLoadModule):
    '''
    Like a standard lazy load  module, but with dynamic PyNode class creation
    '''
    _checkedForNewReservedTypes = False

    @classmethod
    def _unwrappedNodeTypes(cls):
        # get node types, but avoid checking inheritance for all nodes for
        # speed. Note that this means we're potentially missing some abstract
        # edge cases - TadskAssetInstanceNode_TdependNode type stuff, that only
        # shows up in inheritance hierarchies - but I can't see people wanting
        # to access those directly from nodetypes anyway, so I'm judging that
        # an acceptable risk
        allNodes = _apicache._getAllMayaTypes(addAncestors=False,
                                              noManips='fast')
        unknownNodes = allNodes - set(mayaTypeNameToPymelTypeName)
        if unknownNodes:
            # first, check for any new abstract node types - this can happen
            # if, for instance, we have an "extension" release of maya,
            # which introduces new nodes (and abstract nodes), but the caches
            # were built with the "base" release
            # we do these first because they can't be queried by creating nodes,
            # and for derived nodes, we may be able to get by just using the
            # type for the abstract node...
            if not cls._checkedForNewReservedTypes:
                cls._checkedForNewReservedTypes = True
                # this should build mayaTypesToApiTypes and mayaTypesToApiEnums
                # for all reserved types...
                cache = _apicache.ApiCache()
                cache._buildMayaToApiInfo(reservedOnly=True)
                # and update the cache in use with these results...
                # ...now update with any that were missing...
                for mayaType, apiType in cache.mayaTypesToApiTypes.items():
                    if mayaType not in _factories._apiCacheInst.mayaTypesToApiTypes:
                        _factories._apiCacheInst.mayaTypesToApiTypes[mayaType] = apiType
                        _factories._apiCacheInst.mayaTypesToApiEnums[mayaType] = cache.mayaTypesToApiEnums[mayaType]

        return unknownNodes

    def __getattr__(self, name):
        '''Check to see if the name corresponds to a PyNode that hasn't been
        added yet'''
        # In the normal course of operation, this __getattr__ shouldn't be
        # needed - PyNodes corresponding to maya node types should be created
        # when pymel starts up, or when a plugin loads.
        # However, there are some unusual sitations that can arise where new
        # node types are missed... because a plugin can actually register new
        # nodes at any time, not just during it's initialization!
        #
        # This happened with mtoa - if you have an mtoa extension, which adds
        # some maya nodes, but requires another maya plugin... those nodes
        # will not be added until that other maya plugin is loaded... but
        # the nodes will be added to mtoa, NOT the plugin that triggered
        # the plugin-loaded callback. Also, the node adding happens within
        # ANOTHER plugin-loaded callback, which generally runs AFTER pymel's
        # plugin-loaded callback!
        uncapName = _util.uncapitalize(name)
        if uncapName in self._unwrappedNodeTypes():
            # it's a maya node we haven't wrapped yet! Wrap it and return!
            import pymel.core

            mayaType = uncapName

            # See if it's a plugin node...
            nodeClass = _api.MNodeClass(mayaType)
            try:
                pluginPath = nodeClass.pluginName()
                plugin = cmds.pluginInfo(pluginPath, q=1, name=1)
            except RuntimeError:
                # if we can't find a plugin
                pyNodeName =_factories.addCustomPyNode(self, mayaType)
            else:
                pyNodeName = pymel.core._addPluginNode(plugin, mayaType)
            if pyNodeName != name:
                _logger.raiseLog(_logger.WARNING,
                                 "dynamically added node when %r requested, but"
                                 " returned PyNode had name %r" % (
                                     name, pyNodeName))
            return self.__dict__[pyNodeName]
        raise AttributeError(name)


def _addTypeNames():
    # type: () -> NodetypesLazyLoadModule
    for name, obj in globals().items():
        if isinstance(obj, type) and issubclass(obj, DependNode):
            # only want to add this entry if it's defined on THIS class, not
            # if it's inherited!
            if '__melnode__' in obj.__dict__:
                mayaTypeNameToPymelTypeName[obj.__melnode__] = name
                pymelTypeNameToMayaTypeName[name] = obj.__melnode__
    # turn the module into a NodetypesLazyLoadModule, so it can dynamically
    # catch nodes created outside of a plugin-loaded-initialization, using it's
    # __getattr__
    return NodetypesLazyLoadModule(__name__, globals())


class DependNode(with_metaclass(_factories.MetaMayaTypeRegistry, general.PyNode)):
    __apicls__ = _api.MFnDependencyNode

    # ------------------------------
    #    Name Info and Manipulation
    # ------------------------------
#    def __new__(cls,name,create=False):
#        """
#        Provides the ability to create the object when creating a class
#
#            >>> n = pm.Transform("persp",create=True)
#            >>> n.__repr__()
#            # Result: nt.Transform(u'persp1')
#        """
#        if create:
#            ntype = cls.__melnode__
#            name = createNode(ntype,n=name,ss=1)
#        return general.PyNode.__new__(cls,name)

#    def __init__(self, *args, **kwargs ):
#        self.apicls.__init__(self, self._apiobject.object() )

    @_util.universalmethod
    def __melobject__(self):
        # type: () -> str
        """Special method for returning a mel-friendly representation."""
        if isinstance(self, DependNode):
            # For instance, return the node's name...
            return self.name()
        else:
            # For the class itself, return the mel node name
            return self.__melnode__

    def __repr__(self):
        # type: () -> str
        """
        Returns
        -------
        str
        """
        return u"nt.%s(%r)" % (self.__class__.__name__, self.name())

    def _updateName(self):
        # type: () -> None
        # test validity
        self.__apimobject__()
        self._name = self.__apimfn__().name()
        return self._name

    # TODO: unify handling of name parsing (perhaps around the name parser
    # classes?
    def name(self, update=True, stripNamespace=False, levels=0, long=False,
             stripUnderWorld=False):
        # type: (bool, bool, int, bool, bool) -> str
        """The name of the node

        Parameters
        ----------
        update : bool
            if True, will always query to underlying maya object to get it's
            current name (and will therefore detect renames, re-parenting, etc);
            if False, it will use a cached value if available (which is slightly
            faster, but may be out of date)
        stripNamespace : bool
            if True, all nodes will have their namespaces stipped off of them
            (or a certain number of them, if levels is also used)
        levels : int
            if stripNamespace is True, then this number will determine the how
            many namespaces will be removed; if 0 (the default), then all
            leading namespaces will be removed; otherwise, this value gives the
            number of left-most levels to strip
        long : bool
            ignored; included simply to unify the interface between DependNode
            and DagNode, to make it easier to loop over lists of them
        stripUnderWorld : bool
            ignored; included simply to unify the interface between DependNode
            and DagNode, to make it easier to loop over lists of them

        Returns
        -------
        str

        Examples
        --------
        >>> import pymel.core as pm
        >>> pm.newFile(f=1)
        ''
        >>> node = pm.createNode('blinn')

        >>> pm.namespace(add='foo')
        'foo'
        >>> pm.namespace(add='bar', parent='foo')
        'foo:bar'
        >>> pm.namespace(add='stuff', parent='foo:bar')
        'foo:bar:stuff'

        >>> node.rename(':foo:bar:stuff:blinn1')
        nt.Blinn('foo:bar:stuff:blinn1')

        >>> node.name()
        'foo:bar:stuff:blinn1'
        >>> node.name(stripNamespace=True)
        'blinn1'
        >>> node.name(stripNamespace=True, levels=1)
        'bar:stuff:blinn1'
        >>> node.name(stripNamespace=True, levels=2)
        'stuff:blinn1'
        """
        if update or self._name is None:
            try:
                self._updateName()
            except general.MayaObjectError:
                general.DeletedMayaNodeError.handle(self)
        name = self._name
        if stripNamespace:
            if levels:
                spaceSplit = name.split(':')
                name = ':'.join(spaceSplit[min(len(spaceSplit) - 1, levels):])
            else:
                name = name.rsplit(':', 1)[-1]
        return name

    def namespace(self, root=False):
        # type: (bool) -> str
        """Returns the namespace of the object with trailing colon included.

        See `DependNode.parentNamespace` for a variant which does not include
        the trailing colon.

        Parameters
        ----------
        root : bool
            By default, if the object is in the root namespace, an empty string
            is returned; if root is True, ':' is returned in this case.

        Returns
        -------
        str
        """
        ns = self.parentNamespace()
        if ns or root:
            ns += ':'
        return ns

    def shortName(self, update=True, stripNamespace=False, levels=0, long=False,
                 stripUnderWorld=False):
        # type: (bool, bool, int, bool, bool) -> str
        """
        This produces the same results as `DependNode.name` and is included to
        simplify looping over lists of nodes that include both Dag and Depend
        nodes.
        """
        return self.name(update=update, stripNamespace=stripNamespace,
                         levels=levels, long=long,
                         stripUnderWorld=stripUnderWorld)

    def longName(self, update=True, stripNamespace=False, levels=0, long=False,
                 stripUnderWorld=False):
        # type: (bool, bool, int, bool, bool) -> str
        """
        This produces the same results as `DependNode.name` and is included to
        simplify looping over lists of nodes that include both Dag and Depend
        nodes.
        """
        return self.name(update=update, stripNamespace=stripNamespace,
                         levels=levels, long=long,
                         stripUnderWorld=stripUnderWorld)

    def nodeName(self, update=True, stripNamespace=False, levels=0, long=False,
                 stripUnderWorld=False):
        # type: (bool, bool, int, bool, bool) -> str
        """
        This produces the same results as `DependNode.name` and is included to
        simplify looping over lists of nodes that include both Dag and Depend
        nodes.
        """
        return self.name(update=update, stripNamespace=stripNamespace,
                         levels=levels, long=long,
                         stripUnderWorld=stripUnderWorld)

    @overload
    @_factories.addMelDocs('rename')
    def rename(self, name, preserveNamespace=Ellipsis, **kwargs):
        # type: (Any, bool, **Any) -> DependNode
        pass
    
    @_factories.addMelDocs('rename')
    def rename(self, name, **kwargs):
        # type: (Any, **Any) -> DependNode
        """
        Returns
        -------
        DependNode
        """
        # self.setName( name ) # no undo support

        # check for preserveNamespace a pymel unique flag
        if kwargs.pop('preserveNamespace', False):
            name = self.namespace(root=True) + name

        # ensure shortname
        if '|' in name:
            name = name.split('|')[-1]

        return general.rename(self, name, **kwargs)

    def __apiobject__(self):
        # type: () -> Any
        "get the default API object (MObject) for this node if it is valid"
        return self.__apimobject__()

    def __apimobject__(self):
        # type: () -> _api.MObject
        "get the ``maya.OpenMaya.MObject`` for this node if it is valid"
        handle = self.__apihandle__()
        if _api.isValidMObjectHandle(handle):
            return handle.object()
        raise general.MayaNodeError(self._name)

    def __apihandle__(self):
        # type: () -> _api.MObjectHandle
        "get the ``maya.OpenMaya.MObjectHandle`` for this node if it is valid"
        return self.__apiobjects__['MObjectHandle']

    def __str__(self):
        # type: () -> str
        return "%s" % self.name()

    if PY2:
        def __unicode__(self):
            return u"%s" % self.name()

    def __hash__(self):
        # type: () -> int
        return self.__apihandle__().hashCode()

    def node(self):
        # type: () -> DependNode
        """for compatibility with Attribute class

        Returns
        -------
        DependNode
        """
        return self

    # -------------------------
    #    Modification
    # -------------------------

    def lock(self):
        # type: () -> None
        'lockNode -lock 1'
        #kwargs['lock'] = True
        # kwargs.pop('l',None)
        # return cmds.lockNode( self, **kwargs)
        return self.setLocked(True)

    def unlock(self):
        # type: () -> None
        'lockNode -lock 0'
        #kwargs['lock'] = False
        # kwargs.pop('l',None)
        # return cmds.lockNode( self, **kwargs)
        return self.setLocked(False)

    @_factories.addMelDocs('nodeCast')
    def cast(self, swapNode, **kwargs):
        """nodeCast"""
        return cmds.nodeCast(self, swapNode, *kwargs)

    duplicate = general.duplicate

# -------------------------
# xxx{    Presets
# -------------------------

    def savePreset(self, presetName, custom=None, attributes=()):
        # type: (str, Optional[str], Iterable[str]) -> None
        kwargs = {'save': True}
        if attributes:
            kwargs['attributes'] = ' '.join(attributes)
        if custom:
            kwargs['custom'] = custom

        return cmds.nodePreset(presetName, **kwargs)

    def loadPreset(self, presetName):
        kwargs = {'load': True}
        return cmds.nodePreset(presetName, **kwargs)

    def deletePreset(self, presetName):
        kwargs = {'delete': True}
        return cmds.nodePreset(presetName, **kwargs)

    def listPresets(self):
        kwargs = {'list': True}
        return cmds.nodePreset(**kwargs)
#}

# -------------------------
# xxx{    Info
# -------------------------
    type = general.nodeType

    @classmethod
    def typeName(cls):
        # type: () -> str
        """Similar to type / nodeType, but a classmethod that does no dynamic
        querying"""
        return cls.__melnode__

    @overload
    def referenceFile(self, allowNone):
        # type: (Literal[False]) -> _FileReference
        pass

    @overload
    def referenceFile(self):
        # type: () -> Optional[_FileReference]
        pass

    def referenceFile(self, allowNone=True):
        # type: (bool) -> Optional[_FileReference]
        """referenceQuery -file
        Return the reference file to which this object belongs.

        If object is not referenced, returns None, or raises RuntimeError if
        allowNone is False

        Returns
        -------
        Optional[_FileReference]
        """
        try:
            return _FileReference(cmds.referenceQuery(self, f=1))
        except RuntimeError:
            if allowNone:
                return None
            else:
                raise

    def referencePath(self):
        # type: () -> system.Path
        """
        Return the path to the reference that this object belongs to.

        If object is not referenced, raises RuntimeError.
        """
        return self.referenceFile(allowNone=False).path

    def classification(self, **kwargs):
        # type: (**Any) -> str
        'getClassification'
        return general.getClassification(self.type(), **kwargs)
        # return self.__apimfn__().classification( self.type() )

#}
# -------------------------
# xxx{   Connections
# -------------------------

    # connections=True, plugs=True
    @overload
    @_factories.addMelDocs('listConnections', excludeFlags=['source', 'destination'])
    def inputs(self, connections=True, plugs=True, sourceFirst=Ellipsis, type=Ellipsis, **kwargs):
        # type: (Literal[True], Literal[True], bool, general.TypeArgTypes, **Any) -> List[Tuple[general.Attribute, general.Attribute]]
        pass

    # connections=True, type=Type
    @overload
    @_factories.addMelDocs('listConnections', excludeFlags=['source', 'destination'])
    def inputs(self, connections=True, type=None, plugs=Ellipsis, **kwargs):
        # type: (Literal[True], Type[DependNodeT], Literal[False], **Any) -> List[Tuple[general.Attribute, DependNodeT]]
        pass

    # connections=True
    @overload
    @_factories.addMelDocs('listConnections', excludeFlags=['source', 'destination'])
    def inputs(self, connections=True, plugs=Ellipsis, type=Ellipsis, **kwargs):
        # type: (Literal[True], Literal[False], general.LimitedTypeArgTypes, **Any) -> List[Tuple[general.Attribute, DependNode]]
        pass

    # plugs=True
    @overload
    @_factories.addMelDocs('listConnections', excludeFlags=['source', 'destination'])
    def inputs(self, plugs=True, connections=Ellipsis, type=Ellipsis, **kwargs):
        # type: (Literal[True], Literal[False], general.TypeArgTypes, *Any) -> List[general.Attribute]
        pass

    # type=Type
    @overload
    @_factories.addMelDocs('listConnections', excludeFlags=['source', 'destination'])
    def inputs(self, type=None, plugs=Ellipsis, connections=Ellipsis, **kwargs):
        # type: (Type[DependNodeT], Literal[False], Literal[False], *Any) -> List[DependNodeT]
        pass

    # uninspectable type
    @overload
    @_factories.addMelDocs('listConnections', excludeFlags=['source', 'destination'])
    def inputs(self, type=Ellipsis, **kwargs):
        # type: (general.LimitedTypeArgTypes, **Any) -> List[DependNode]
        pass

    @_factories.addMelDocs('listConnections', excludeFlags=['source', 'destination'])
    def inputs(self, **kwargs):
        # type: (**Any) -> List[general.PyNode]
        """listConnections -source 1 -destination 0

        Returns
        -------
        List[general.PyNode]
        """
        kwargs['source'] = True
        kwargs.pop('s', None)
        kwargs['destination'] = False
        kwargs.pop('d', None)
        return general.listConnections(self, **kwargs)

    # connections=True, plugs=True
    @overload
    @_factories.addMelDocs('listConnections', excludeFlags=['source', 'destination'])
    def outputs(self, connections=True, plugs=True, sourceFirst=Ellipsis, type=Ellipsis, **kwargs):
        # type: (Literal[True], Literal[True], bool, general.TypeArgTypes, **Any) -> List[Tuple[general.Attribute, general.Attribute]]
        pass

    # connections=True, type=Type
    @overload
    @_factories.addMelDocs('listConnections', excludeFlags=['source', 'destination'])
    def outputs(self, connections=True, type=None, plugs=Ellipsis, **kwargs):
        # type: (Literal[True], Type[DependNodeT], Literal[False], **Any) -> List[Tuple[general.Attribute, DependNodeT]]
        pass

    # connections=True
    @overload
    @_factories.addMelDocs('listConnections', excludeFlags=['source', 'destination'])
    def outputs(self, connections=True, plugs=Ellipsis, type=Ellipsis, **kwargs):
        # type: (Literal[True], Literal[False], general.LimitedTypeArgTypes, **Any) -> List[Tuple[general.Attribute, DependNode]]
        pass

    # plugs=True
    @overload
    @_factories.addMelDocs('listConnections', excludeFlags=['source', 'destination'])
    def outputs(self, plugs=True, connections=Ellipsis, type=Ellipsis, **kwargs):
        # type: (Literal[True], Literal[False], general.TypeArgTypes, *Any) -> List[general.Attribute]
        pass

    # type=Type
    @overload
    @_factories.addMelDocs('listConnections', excludeFlags=['source', 'destination'])
    def outputs(self, type=None, plugs=Ellipsis, connections=Ellipsis, **kwargs):
        # type: (Type[DependNodeT], Literal[False], Literal[False], *Any) -> List[DependNodeT]
        pass

    # uninspectable type
    @overload
    @_factories.addMelDocs('listConnections', excludeFlags=['source', 'destination'])
    def outputs(self, type=Ellipsis, **kwargs):
        # type: (general.LimitedTypeArgTypes, **Any) -> List[DependNode]
        pass

    @_factories.addMelDocs('listConnections', excludeFlags=['source', 'destination'])
    def outputs(self, **kwargs):
        # type: (**Any) -> List[general.PyNode]
        """listConnections -source 0 -destination 1

        Returns
        -------
        List[general.PyNode]
        """
        kwargs['source'] = False
        kwargs.pop('s', None)
        kwargs['destination'] = True
        kwargs.pop('d', None)
        return general.listConnections(self, **kwargs)

    sources = inputs

    # @overload
    # @_factories.addMelDocs('listConnections', excludeFlags=['source', 'destination'])
    # def sources(self, connections=True, plugs=True, sourceFirst=Ellipsis, **kwargs):
    #     # type: (Literal[True], Literal[True], bool, **Any) -> List[Tuple[general.Attribute, general.Attribute]]
    #     pass
    #
    # @overload
    # @_factories.addMelDocs('listConnections', excludeFlags=['source', 'destination'])
    # def sources(self, connections=True, **kwargs):
    #     # type: (Literal[True], **Any) -> List[Tuple[general.Attribute, DependNode]]
    #     pass
    #
    # @overload
    # # exclude plugs, because if plugs=True is provided, we want the first overload
    # @_factories.addMelDocs('listConnections', excludeFlags=['source', 'destination', 'plugs'])
    # def sources(self, connections=True, type=None, **kwargs):
    #     # type: (Literal[True], Type[DependNodeT], **Any) -> List[Tuple[general.Attribute, DependNodeT]]
    #     pass
    #
    # @overload
    # @_factories.addMelDocs('listConnections', excludeFlags=['source', 'destination'])
    # def sources(self, plugs=True, **kwargs):
    #     # type: (Literal[True], *Any) -> List[general.Attribute]
    #     pass
    #
    # @overload
    # # exclude plugs, because if plugs=True is provided, we want the previous overload
    # @_factories.addMelDocs('listConnections', excludeFlags=['source', 'destination', 'plugs'])
    # def sources(self, type=None, **kwargs):
    #     # type: (Type[DependNodeT], *Any) -> List[DependNodeT]
    #     pass
    #
    # @overload
    # @_factories.addMelDocs('listConnections', excludeFlags=['source', 'destination'])
    # def sources(self, type=Ellipsis, **kwargs):
    #     # type: (Union[str, Iterable[str], Iterable[Type[DependNode]]], **Any) -> List[Union[DependNode, general.Attribute]]
    #     pass
    #
    # @_factories.addMelDocs('listConnections', excludeFlags=['source', 'destination'])
    # def sources(self, **kwargs):
    #     # type: (**Any) -> List[general.PyNode]
    #     """listConnections -source 1 -destination 0
    #
    #     Returns
    #     -------
    #     List[general.PyNode]
    #     """
    #     kwargs['source'] = True
    #     kwargs.pop('s', None)
    #     kwargs['destination'] = False
    #     kwargs.pop('d', None)
    #     return general.listConnections(self, **kwargs)

    # # connections=True, plugs=True
    # @overload
    # @_factories.addMelDocs('listConnections', excludeFlags=['source', 'destination'])
    # def destinations(self, connections=True, plugs=True, sourceFirst=Ellipsis, type=Ellipsis, **kwargs):
    #     # type: (Literal[True], Literal[True], bool, Union[str, Type[DependNode]], **Any) -> List[Tuple[general.Attribute, general.Attribute]]
    #     pass
    #
    # # connections=True
    # @overload
    # @_factories.addMelDocs('listConnections', excludeFlags=['source', 'destination', 'type'])
    # def destinations(self, connections=True, plugs=Ellipsis, **kwargs):
    #     # type: (Literal[True], Literal[False], **Any) -> List[Tuple[general.Attribute, DependNode]]
    #     pass
    #
    # # connections=True, type=Type
    # @overload
    # @_factories.addMelDocs('listConnections', excludeFlags=['source', 'destination'])
    # def destinations(self, connections=True, type=None, plugs=Ellipsis, **kwargs):
    #     # type: (Literal[True], Type[DependNodeT], Literal[False], **Any) -> List[Tuple[general.Attribute, DependNodeT]]
    #     pass
    #
    # # plugs=True
    # @overload
    # @_factories.addMelDocs('listConnections', excludeFlags=['source', 'destination', 'type'])
    # def destinations(self, plugs=True, connections=Ellipsis, **kwargs):
    #     # type: (Literal[True], Literal[False], *Any) -> List[general.Attribute]
    #     pass
    #
    # # type=Type
    # @overload
    # @_factories.addMelDocs('listConnections', excludeFlags=['source', 'destination'])
    # def destinations(self, type=None, plugs=Ellipsis, connections=Ellipsis, **kwargs):
    #     # type: (Type[DependNodeT], Literal[False], Literal[False], *Any) -> List[DependNodeT]
    #     pass
    #
    # # uninspectable type
    # @overload
    # @_factories.addMelDocs('listConnections', excludeFlags=['source', 'destination'])
    # def destinations(self, type=Ellipsis, **kwargs):
    #     # type: (Union[str, Iterable[str], Iterable[Type[DependNode]]], **Any) -> List[DependNode]
    #     pass
    #
    # @_factories.addMelDocs('listConnections', excludeFlags=['source', 'destination'])
    # def destinations(self, **kwargs):
    #     # type: (**Any) -> List[general.PyNode]
    #     """listConnections -source 0 -destination 1
    #
    #     Returns
    #     -------
    #     List[general.PyNode]
    #     """
    #     kwargs['source'] = False
    #     kwargs.pop('s', None)
    #     kwargs['destination'] = True
    #     kwargs.pop('d', None)
    #     return general.listConnections(self, **kwargs)

    destinations = outputs

    def shadingGroups(self):
        # type: () -> List[ShadingEngine]
        """list any shading groups in the future of this object - works for
        shading nodes, transforms, and shapes

        Also see listSets(type=1) - which returns which 'rendering sets' the
        object is a member of (and 'rendering sets' seem to consist only of
        shading groups), whereas this method searches the object's future for
        any nodes of type 'shadingEngine'.

        Returns
        -------
        List[ShadingEngine]
        """
        return self.future(type='shadingEngine')

#}
# -------------------------
# xxx{    Attributes
# -------------------------
    def __getattr__(self, attr):
        # type: (str) -> general.Attribute
        try:
            return getattr(super(general.PyNode, self), attr)
        except AttributeError:
            try:
                return DependNode.attr(self, attr)
            except general.MayaAttributeError as e:
                # since we're being called via __getattr__ we don't know whether the user was intending
                # to get a class method or a maya attribute, so we raise a more generic AttributeError
                raise AttributeError("%r has no attribute or method named '%s'" % (self, attr))

    @_util.universalmethod
    def attrSpec(obj, attr):  # @NoSelf
        # type: (Any, Any) -> general.AttributeSpec
        """
        Access to an attribute specifications of a node.  This does not require
        an instance:

            >>> nt.Transform.attrSpec('tx').isKeyable()
            True

        but it can use one if needed ( for example, for dynamically created attributes )

            >>> nt.Transform(u'persp').attrSpec('tx').isKeyable()

        Note: this is still experimental.
        """
        # if it's a MPlug, MObject, Attribute, or AttributeSpec, we don't
        # even need either cls/self
        if isinstance(attr, (_api.MPlug, _api.MObject,
                             general.Attribute, general.AttributeSpec)):
            return general.AttributeSpec(attr)

        if inspect.isclass(obj):
            self = None
            cls = obj  # keep things familiar
        else:
            self = obj  # keep things familiar
            cls = type(obj)

        attributes = cls.__apiobjects__.setdefault('MFnAttributes', {})
        attrObj = attributes.get(attr, None)
        if _api.isValidMObject(attrObj):
            # we don't store the AttributeDefault object in the cash, in case
            # users store extra information on them, ie:
            #    attrDef = myNode.attrSpec('foobar')
            #    attrDef.customInfo = 'awesome'
            return general.AttributeSpec(attrObj)

        def toAttrObj(apiObj):
            try:
                attrObj = apiObj.attribute(attr)
                if attrObj.isNull():
                    raise RuntimeError
            except RuntimeError:
                # just try it first, then check if it has the attribute if
                # we errored (as opposed to always check first if the node
                # has the attribute), on the assumption that this will be
                # "faster" for most cases, where the node actually DOES have
                # the attribute...
                if not apiObj.hasAttribute(attr):
                    raise general.MayaAttributeError('%s.%s' % (cls.__melnode__, attr))
                else:
                    # don't know why we got this error, so just reraise
                    raise
            return attrObj

        if self is None:
            # Yay, we have MNodeClass, use it!
            nodeCls = _api.MNodeClass(cls.__melnode__)
            attrObj = toAttrObj(nodeCls)
        else:
            nodeMfn = self.__apimfn__()
            attrObj = toAttrObj(nodeMfn)

        attrSpec = general.AttributeSpec(attrObj)
        if not attrSpec.isDynamic() and not attrSpec.isExtension():
            attributes[attr] = attrObj
        return attrSpec

    # keep attrDefaults name for backwards compatibility
    attrDefaults = attrSpec

    def attr(self, attr):
        # type: (str) -> general.Attribute
        """
        access to attribute plug of a node. returns an instance of the Attribute class for the
        given attribute name.

        Returns
        -------
        general.Attribute
        """
        return self._attr(attr, False)

    # Just have this alias because it will sometimes return attributes for an
    # underlying shape, which we may want for DagNode.attr, but don't want for
    # DependNode.attr (and using the on-shape result, instead of throwing it
    # away and then finding it again on the shape, saves time for the DagNode
    # case)
    def _attr(self, attr, allowOtherNode):
        if not isinstance(attr, str):
            attr = str(attr)
        # return Attribute( '%s.%s' % (self, attr) )
        try:
            if '.' in attr or '[' in attr:
                # Compound or Multi Attribute
                # there are a couple of different ways we can proceed:
                # Option 1: back out to _api.toApiObject (via general.PyNode)
                # return Attribute( self.__apiobject__(), self.name() + '.' + attr )

                # Option 2: nameparse.
                # this avoids calling self.name(), which can be slow
                import pymel.util.nameparse as nameparse
                nameTokens = nameparse.getBasicPartList('dummy.' + attr)
                result = self.__apiobject__()
                for token in nameTokens[1:]:  # skip the first, bc it's the node, which we already have
                    if isinstance(token, nameparse.MayaName):
                        if isinstance(result, _api.MPlug):
                            # you can't get a child plug from a multi/array plug.
                            # if result is currently 'defaultLightList1.lightDataArray' (an array)
                            # and we're trying to get the next plug, 'lightDirection', then we need a dummy index.
                            # the following line will reuslt in 'defaultLightList1.lightDataArray[-1].lightDirection'
                            if result.isArray():
                                result = self.__apimfn__().findPlug(str(token))
                            else:
                                result = result.child(self.__apimfn__().attribute(str(token)))
                        else:  # Node
                            result = self.__apimfn__().findPlug(str(token))
#                                # search children for the attribute to simulate  cam.focalLength --> perspShape.focalLength
#                                except TypeError:
#                                    for i in range(fn.childCount()):
#                                        try: result = _api.MFnDagNode( fn.child(i) ).findPlug( unicode(token) )
#                                        except TypeError: pass
#                                        else:break
                    if isinstance(token, nameparse.NameIndex):
                        if token.value != -1:
                            result = result.elementByLogicalIndex(token.value)
                plug = result
            else:
                try:
                    plug = self.__apimfn__().findPlug(attr, False)
                except RuntimeError:
                    # Don't use .findAlias, as it always returns the 'base'
                    # attribute - ie, if the alias is to foo[0].bar, it will
                    # just point to foo
                    # aliases
                    #obj = _api.MObject()
                    #self.__apimfn__().findAlias( attr, obj )
                    #plug = self.__apimfn__().findPlug( obj, False )

                    # the following technique gets aliased attributes as well.
                    # turning dagPlugs to off saves time because we already
                    # know the dagNode. however, certain attributes, such as
                    # rotatePivot, are detected as components,
                    # despite the fact that findPlug finds them as MPlugs.
                    # need to look into this
                    # TODO: test speed versus above method
                    try:
                        plug = _api.toApiObject(self.name() + '.' + attr,
                                                dagPlugs=False, comps=False)
                    except RuntimeError:
                        raise
                    if not isinstance(plug, _api.MPlug):
                        raise RuntimeError

                if not (allowOtherNode or plug.node() == self.__apimobject__()):
                    # we could have gotten an attribute on a shape object,
                    # which we don't want
                    raise RuntimeError
            return general.Attribute(self.__apiobject__(), plug)

        except RuntimeError:
            # raise our own MayaAttributeError, which subclasses AttributeError and MayaObjectError
            raise general.MayaAttributeError('%s.%s' % (self, attr))

    hasAttr = general.hasAttr

    @_factories.addMelDocs('setAttr')
    def setAttr(self, attr, *args, **kwargs):
        # type: (str, *Any, **Any) -> Any
        # for now, using strings is better, because there is no MPlug support
        return general.setAttr("%s.%s" % (self, attr), *args, **kwargs)

    @_factories.addMelDocs('setAttr')
    def setDynamicAttr(self, attr, *args, **kwargs):
        # type: (str, *Any, **Any) -> Any
        """
        same as `DependNode.setAttr` with the force flag set to True.  This causes
        the attribute to be created based on the passed input value.
        """

        # for now, using strings is better, because there is no MPlug support
        kwargs['force'] = True
        return general.setAttr("%s.%s" % (self, attr), *args, **kwargs)

    @_factories.addMelDocs('getAttr')
    def getAttr(self, attr, *args, **kwargs):
        # type: (str, *Any, **Any) -> Any
        # for now, using strings is better, because there is no MPlug support
        return general.getAttr("%s.%s" % (self, attr), *args, **kwargs)

    @overload
    @_factories.addMelDocs('addAttr')
    def addAttr(self, attr, type=Ellipsis, childSuffixes=Ellipsis, enumName=Ellipsis, **kwargs):
        # type: (str, Union[str, Type], Sequence[str], general.EnumArgTypes, **Any) -> Any
        pass

    @_factories.addMelDocs('addAttr')
    def addAttr(self, attr, **kwargs):
        # type: (str, *Any, **Any) -> Any
        # for now, using strings is better, because there is no MPlug support
        assert 'longName' not in kwargs and 'ln' not in kwargs
        kwargs['longName'] = attr
        return general.addAttr(str(self), **kwargs)

    @_factories.addMelDocs('deleteAttr')
    def deleteAttr(self, attr, *args, **kwargs):
        # type: (str, *Any, **Any) -> None
        # for now, using strings is better, because there is no MPlug support
        if isinstance(attr, general.Attribute):
            if not attr.node() == self:
                raise general.MayaAttributeError(
                    "Cannot delete an attribute from another node - thisNode:"
                    " {} - attr: {}".format(self, attr))
            return general.deleteAttr(attr.name(), *args, **kwargs)
        return general.deleteAttr("%s.%s" % (self, attr), *args, **kwargs)
    # a former bug caused DependNode.removeAttribute to be wrapped on many
    # nodes... made it an alias for deleteAttr method to preserve backward
    # compatibility
    removeAttribute = deleteAttr

    @_factories.addMelDocs('connectAttr')
    def connectAttr(self, attr, destination, **kwargs):
        # type: (str, Union[str, general.Attribute], **Any) -> None
        # for now, using strings is better, because there is no MPlug support
        return general.connectAttr("%s.%s" % (self, attr), destination, **kwargs)

    @_factories.addMelDocs('disconnectAttr')
    def disconnectAttr(self, attr, destination=None, **kwargs):
        # type: (str, Union[str, general.Attribute, None], **Any) -> None
        # for now, using strings is better, because there is no MPlug support
        return general.disconnectAttr("%s.%s" % (self, attr), destination, **kwargs)

    listAnimatable = _listAnimatable

    @_factories.addMelDocs('listAttr')
    def listAttr(self, topLevel=False, descendants=False, alias=False, **kwargs):
        # type: (bool, bool, bool, **Any) -> List[general.Attribute]
        """
        listAttr

        Modifications:
          - returns an empty list when the result is None
          - added 'alias' keyword to list attributes that have aliases
          - added 'topLevel' keyword to only return attributes that are not
            compound children; may not be used in combination with
            'descendants'
          - added 'descendants' keyword to return all top-level attributes
            and all their descendants; note that the standard call may return
            some attributes that 'descendants' will not, if there are compound
            multi attributes with no existing indices; ie, the standard call
            might return "node.parentAttr[-1].childAttr", but the 'descendants'
            version would only return childAttr if an index exists for
            parentAttr, ie "node.parentAttr[0].childAttr"; may not be used in
            combination with 'topLevel'

        Returns
        -------
        List[general.Attribute]
        """
        if descendants:
            if topLevel:
                raise ValueError("may not specify both topLevel and descendants")
            # get the topLevel ones, then aggregate all the descendants...
            topChildren = self.listAttr(topLevel=True, **kwargs)
            res = list(topChildren)
            for child in topChildren:
                res.extend(child.iterDescendants())
            return res

        # stringify fix
        res = [self.attr(x) for x in _util.listForNone(cmds.listAttr(self.name(), **kwargs))]
        if alias:
            # need to make sure that our alias wasn't filtered out by one of
            # the other kwargs (keyable, etc)...
            # HOWEVER, we can't just do a straight up check to see if the
            # results of listAlias() are in res - because the attributes in
            # res are index-less (ie, ,myAttr[-1]), while the results returned
            # by listAliases() have indices (ie, .myAttr[25])... so instead we
            # just do a comparison of the names (which are easily hashable)
            res = set(x.attrName() for x in res)
            res = [x[1] for x in self.listAliases() if x[1].attrName() in res]
        if topLevel:
            res = [x for x in res if x.getParent() is None]
        return res

    def listAliases(self):
        # type: () -> List[Tuple[str, general.Attribute]]
        """
        aliasAttr

        Modifications:
          - returns an empty list when the result is None
          - when queried, returns a list of (alias, `Attribute`) pairs.

        Returns
        -------
        List[Tuple[str, general.Attribute]]
        """

        #tmp = _util.listForNone(cmds.aliasAttr(self.name(),query=True))
        tmp = []
        self.__apimfn__().getAliasList(tmp)
        res = []
        for i in range(0, len(tmp), 2):
            res.append((tmp[i], general.Attribute(self.node() + '.' + tmp[i + 1])))
        return res

    @_factories.addMelDocs('attributeInfo')
    def attrInfo(self, **kwargs):
        # type: (**Any) -> List[general.Attribute]
        """attributeInfo

        Returns
        -------
        List[general.Attribute]
        """
        # stringify fix
        return [self.attr(x) for x
                in _util.listForNone(cmds.attributeInfo(self.name(), **kwargs))]


#}
# ----------------------------------------
# xxx{ Name Info and Manipulation
# ----------------------------------------

# Now just wraps NameParser functions

    def stripNum(self):
        # type: () -> str
        """
        Return the name of the node with trailing numbers stripped off.

        If no trailing numbers are found the name will be returned unchanged.

        >>> from pymel.core import *
        >>> SCENE.lambert1.stripNum()
        'lambert'

        Returns
        -------
        str
        """
        return other.NameParser(self).stripNum()

    def extractNum(self):
        # type: () -> str
        """
        Return the trailing numbers of the node name.

        If no trailing numbers are found an error will be raised.

        >>> from pymel.core import *
        >>> SCENE.lambert1.extractNum()
        '1'

        Returns
        -------
        str
        """
        return other.NameParser(self).extractNum()

    def nextUniqueName(self):
        # type: () -> other.DependNodeName
        """
        Increment the trailing number of the object until a unique name is
        found.

        If there is no trailing number, appends '1' to the name.

        Returns
        -------
        str
        """
        return other.NameParser(self).nextUniqueName()

    def nextName(self):
        # type: () -> other.DependNodeName
        """
        Increment the trailing number of the object by 1

        Raises an error if the name has no trailing number.

        >>> from pymel.core import *
        >>> SCENE.lambert1.nextName()
        DependNodeName('lambert2')

        Returns
        -------
        str
        """
        return other.NameParser(self).nextName()

    def prevName(self):
        # type: () -> other.DependNodeName
        """
        Decrement the trailing number of the object by 1

        Raises an error if the name has no trailing number.

        Returns
        -------
        str
        """
        return other.NameParser(self).prevName()

    @classmethod
    def registerVirtualSubClass(cls, nameRequired=False):
        """
        Deprecated
        """
        _factories.registerVirtualClass(cls, nameRequired)
# ------ Do not edit below this line --------
    __melnode__ = 'dependNode'
    __slots__ = ()
    MAttrClass = Enum('MAttrClass', [('localDynamicAttr', 1), ('kLocalDynamicAttr', 1), ('normalAttr', 2), ('kNormalAttr', 2), ('extensionAttr', 3), ('kExtensionAttr', 3), ('invalidAttr', 4), ('kInvalidAttr', 4)], multiKeys=True)
    MdgTimerMetric = Enum('MdgTimerMetric', [('metric_callback', 0), ('kTimerMetric_callback', 0), ('metric_compute', 1), ('kTimerMetric_compute', 1), ('metric_dirty', 2), ('kTimerMetric_dirty', 2), ('metric_draw', 3), ('kTimerMetric_draw', 3), ('metric_fetch', 4), ('kTimerMetric_fetch', 4), ('metric_callbackViaAPI', 5), ('kTimerMetric_callbackViaAPI', 5), ('metric_callbackNotViaAPI', 6), ('kTimerMetric_callbackNotViaAPI', 6), ('metric_computeDuringCallback', 7), ('kTimerMetric_computeDuringCallback', 7), ('metric_computeNotDuringCallback', 8), ('kTimerMetric_computeNotDuringCallback', 8), ('metrics', 9), ('kTimerMetrics', 9)], multiKeys=True)
    MdgTimerState = Enum('MdgTimerState', [('off', 0), ('kTimerOff', 0), ('on', 1), ('kTimerOn', 1), ('uninitialized', 2), ('kTimerUninitialized', 2), ('invalidState', 3), ('kTimerInvalidState', 3)], multiKeys=True)
    MdgTimerType = Enum('MdgTimerType', [('type_self', 0), ('kTimerType_self', 0), ('type_inclusive', 1), ('kTimerType_inclusive', 1), ('type_count', 2), ('kTimerType_count', 2), ('types', 3), ('kTimerTypes', 3)], multiKeys=True)

    @_f.addApiDocs(_api.MFnDependencyNode, 'absoluteName')
    def absoluteName(self):
        # type: () -> str
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'absoluteName')
        return _f.ApiArgUtil._castResult(self, res, 'MString', None)

    @classmethod
    @_f.addApiDocs(_api.MFnDependencyNode, 'allocateFlag')
    def allocateFlag(self, pluginName):
        # type: (str) -> int
        do, final_do, outTypes = _f.getDoArgs([pluginName], [('pluginName', 'MString', 'in', None)])
        res = _api.MFnDependencyNode.allocateFlag(*final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'uint', None)
        return res

    @_f.addApiDocs(_api.MFnDependencyNode, 'attributeClass')
    def attributeClass(self, attribute):
        # type: (str | DependNode) -> DependencyNode.MAttrClass
        do, final_do, outTypes = _f.getDoArgs([attribute], [('attribute', 'MObject', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'attributeClass', final_do)
        res = _f.ApiArgUtil._castResult(self, res, ('MFnDependencyNode', 'MAttrClass'), None)
        return res

    @_f.addApiDocs(_api.MFnDependencyNode, 'attributeCount')
    def attributeCount(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'attributeCount')
        return _f.ApiArgUtil._castResult(self, res, 'uint', None)

    @_f.addApiDocs(_api.MFnDependencyNode, 'canBeWritten')
    def canBeWritten(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'canBeWritten')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @classmethod
    @_f.addApiDocs(_api.MFnDependencyNode, 'deallocateAllFlags')
    def deallocateAllFlags(self, pluginName):
        # type: (str) -> None
        do, final_do, outTypes = _f.getDoArgs([pluginName], [('pluginName', 'MString', 'in', None)])
        res = _api.MFnDependencyNode.deallocateAllFlags(*final_do)
        return res

    @classmethod
    @_f.addApiDocs(_api.MFnDependencyNode, 'deallocateFlag')
    def deallocateFlag(self, pluginName, flag):
        # type: (str, int) -> None
        do, final_do, outTypes = _f.getDoArgs([pluginName, flag], [('pluginName', 'MString', 'in', None), ('flag', 'uint', 'in', None)])
        res = _api.MFnDependencyNode.deallocateFlag(*final_do)
        return res

    @_f.addApiDocs(_api.MFnDependencyNode, 'dgCallbacks')
    def dgCallbacks(self, timerType, callbackName, value):
        # type: (DependencyNode.MdgTimerType, List[str], List[float]) -> None
        do, final_do, outTypes = _f.getDoArgs([timerType, callbackName, value], [('timerType', ('MFnDependencyNode', 'MdgTimerType'), 'in', None), ('callbackName', 'MStringArray', 'in', None), ('value', 'MDoubleArray', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'dgCallbacks', final_do)
        return res

    @_f.addApiDocs(_api.MFnDependencyNode, 'dgTimer')
    def dgTimer(self, timerMetric, timerType):
        # type: (DependencyNode.MdgTimerMetric, DependencyNode.MdgTimerType) -> float
        do, final_do, outTypes = _f.getDoArgs([timerMetric, timerType], [('timerMetric', ('MFnDependencyNode', 'MdgTimerMetric'), 'in', None), ('timerType', ('MFnDependencyNode', 'MdgTimerType'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'dgTimer', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'double', None)
        return res

    @_f.addApiDocs(_api.MFnDependencyNode, 'dgTimerOff')
    def dgTimerOff(self):
        # type: () -> None
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'dgTimerOff')
        return res

    @_f.addApiDocs(_api.MFnDependencyNode, 'dgTimerOn')
    def dgTimerOn(self):
        # type: () -> None
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'dgTimerOn')
        return res

    @_f.addApiDocs(_api.MFnDependencyNode, 'dgTimerQueryState')
    def dgTimerQueryState(self):
        # type: () -> DependencyNode.MdgTimerState
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'dgTimerQueryState')
        return _f.ApiArgUtil._castResult(self, res, ('MFnDependencyNode', 'MdgTimerState'), None)

    @_f.addApiDocs(_api.MFnDependencyNode, 'dgTimerReset')
    def dgTimerReset(self):
        # type: () -> None
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'dgTimerReset')
        return res

    @classmethod
    @_f.addApiDocs(_api.MFnDependencyNode, 'enableDGTiming')
    def enableDGTiming(self, enable):
        # type: (bool) -> None
        do, final_do, outTypes = _f.getDoArgs([enable], [('enable', 'bool', 'in', None)])
        res = _api.MFnDependencyNode.enableDGTiming(*final_do)
        return res

    @_f.addApiDocs(_api.MFnDependencyNode, 'affectsAnimation')
    def getAffectsAnimation(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'affectsAnimation')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnDependencyNode, 'getAliasAttr')
    def getAliasAttr(self, force):
        # type: (bool) -> DependNode
        do, final_do, outTypes = _f.getDoArgs([force], [('force', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAliasAttr', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MObject', None)
        return res

    @_f.addApiDocs(_api.MFnDependencyNode, 'icon')
    def getIcon(self):
        # type: () -> str
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'icon')
        return _f.ApiArgUtil._castResult(self, res, 'MString', None)

    @_f.addApiDocs(_api.MFnDependencyNode, 'name')
    def getName(self):
        # type: () -> str
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'name')
        return _f.ApiArgUtil._castResult(self, res, 'MString', None)

    @_f.addApiDocs(_api.MFnDependencyNode, 'hasAttribute')
    def hasAttribute(self, attrName):
        # type: (str) -> bool
        do, final_do, outTypes = _f.getDoArgs([attrName], [('attrName', 'MString', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'hasAttribute', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res

    @_f.addApiDocs(_api.MFnDependencyNode, 'hasUniqueName')
    def hasUniqueName(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'hasUniqueName')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)
    isUniquelyNamed = hasUniqueName

    @_f.addApiDocs(_api.MFnDependencyNode, 'isDefaultNode')
    def isDefaultNode(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'isDefaultNode')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnDependencyNode, 'isFlagSet')
    def isFlagSet(self, flag):
        # type: (int) -> bool
        do, final_do, outTypes = _f.getDoArgs([flag], [('flag', 'uint', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'isFlagSet', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res

    @_f.addApiDocs(_api.MFnDependencyNode, 'isFromReferencedFile')
    def isFromReferencedFile(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'isFromReferencedFile')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)
    isReadOnly = isFromReferencedFile
    isReferenced = isFromReferencedFile

    @_f.addApiDocs(_api.MFnDependencyNode, 'isLocked')
    def isLocked(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'isLocked')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnDependencyNode, 'isNewAttribute')
    def isNewAttribute(self, attr):
        # type: (str | DependNode) -> bool
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'isNewAttribute', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res

    @_f.addApiDocs(_api.MFnDependencyNode, 'isShared')
    def isShared(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'isShared')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnDependencyNode, 'isTrackingEdits')
    def isTrackingEdits(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'isTrackingEdits')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnDependencyNode, 'parentNamespace')
    def parentNamespace(self):
        # type: () -> str
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'parentNamespace')
        return _f.ApiArgUtil._castResult(self, res, 'MString', None)

    @_f.addApiDocs(_api.MFnDependencyNode, 'pluginName')
    def pluginName(self):
        # type: () -> str
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'pluginName')
        return _f.ApiArgUtil._castResult(self, res, 'MString', None)

    @_f.addApiDocs(_api.MFnDependencyNode, 'setAffectsAnimation')
    def setAffectsAnimation(self):
        # type: () -> None
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'setAffectsAnimation')
        return res

    @_f.addApiDocs(_api.MFnDependencyNode, 'setDoNotWrite')
    def setDoNotWrite(self, flag):
        # type: (bool) -> None
        do, final_do, outTypes = _f.getDoArgs([flag], [('flag', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'setDoNotWrite', final_do)
        return res

    @_f.addApiDocs(_api.MFnDependencyNode, 'setFlag')
    def setFlag(self, flag, state):
        # type: (int, bool) -> None
        do, final_do, outTypes = _f.getDoArgs([flag, state], [('flag', 'uint', 'in', None), ('state', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'setFlag', final_do)
        return res

    @_f.addApiDocs(_api.MFnDependencyNode, 'setIcon')
    def setIcon(self, filename):
        # type: (str) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([filename], [('filename', 'MString', 'in', None)], self.getIcon, self.setIcon, [])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'setIcon', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnDependencyNode, 'setLocked')
    def setLocked(self, lock):
        # type: (bool) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([lock], [('lock', 'bool', 'in', None)], self.isLocked, self.setLocked, [])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'setLocked', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnDependencyNode, 'setName')
    def setName(self, name, createNamespace=False):
        # type: (str, bool) -> str
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([name, createNamespace], [('name', 'MString', 'in', None), ('createNamespace', 'bool', 'in', None)], self.getName, self.setName, [])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'setName', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        res = _f.ApiArgUtil._castResult(self, res, 'MString', None)
        return res

    @_f.addApiDocs(_api.MFnDependencyNode, 'uniqueName')
    def uniqueName(self):
        # type: () -> str
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'uniqueName')
        return _f.ApiArgUtil._castResult(self, res, 'MString', None)
# ------ Do not edit above this line --------


class ContainerBase(DependNode):
    pass
# ------ Do not edit below this line --------
    __melnode__ = 'containerBase'
    __slots__ = ()
# ------ Do not edit above this line --------


class Entity(ContainerBase):
    pass
# ------ Do not edit below this line --------
    __melnode__ = 'entity'
    __slots__ = ()
# ------ Do not edit above this line --------


class DagNode(Entity):

    #:group Path Info and Modification: ``*parent*``, ``*Parent*``, ``*child*``, ``*Child*``
    """
    """

    __apicls__ = _api.MFnDagNode

    _componentAttributes = {}  # type: Dict[str, Union[tuple, type]]

    def comp(self, compName):
        # type: (str) -> Optional[general.Component]
        """
        Will retrieve a Component object for this node; similar to
        DependNode.attr(), but for components.
        """
        if compName in self._componentAttributes:
            compClass = self._componentAttributes[compName]
            if isinstance(compClass, tuple):
                # We have something like:
                # 'uIsoparm'    : (NurbsSurfaceIsoparm, 'u')
                # need to specify what 'flavor' of the basic
                # component we need...
                return compClass[0](self, {compClass[1]: general.ComponentIndex(label=compClass[1])})
            else:
                return compClass(self)
        # if we do self.getShape(), and this is a shape node, we will
        # enter a recursive loop if compName isn't actually a comp:
        # since shape doesn't have 'getShape', it will call __getattr__
        # for 'getShape', which in turn call comp to check if it's a comp,
        # which will call __getattr__, etc
        # ..soo... check if we have a 'getShape'!
        # ...also, don't use 'hasattr', as this will also call __getattr__!
        try:
            object.__getattribute__(self, 'getShape')
        except AttributeError:
            raise general.MayaComponentError('%s.%s' % (self, compName))
        else:
            shape = self.getShape()
            if shape:
                return shape.comp(compName)

    def listComp(self, names=False):
        # type: (bool) -> List[str]
        """Will return a list of all component objects for this object

        Is to .comp() what .listAttr() is to .attr(); will NOT check the shape
        node.

        Parameters
        ----------
        names : bool
            By default, will return a list of actual usabale pymel Component
            objects; if you just want a list of string names which would
            be compatible with .comp(), set names to True

        Returns
        -------
        List[str]
        """
        keys = sorted(self._componentAttributes.keys())
        if names:
            return keys

        compTypes = set()
        comps = []
        # use the sorted keys, so the order matches that returned by names,
        # minus duplicate entries for aliases
        for name in keys:
            compType = self._componentAttributes[name]
            if compType not in compTypes:
                compTypes.add(compType)
                comps.append(self.comp(name))
        return comps

    def _updateName(self, long=False):
        # if _api.isValidMObjectHandle(self._apiobject) :
            #obj = self._apiobject.object()
            #dagFn = _api.MFnDagNode(obj)
            #dagPath = _api.MDagPath()
            # dagFn.getPath(dagPath)
        dag = self.__apimdagpath__()
        if dag:
            name = dag.partialPathName()
            if not name:
                raise general.MayaNodeError

            self._name = name
            if long:
                return dag.fullPathName()

        return self._name

    # TODO: unify handling of name parsing (perhaps around the name parser
    # classes?
    # TODO: support for underworld nodes
    def name(self, update=True, long=False, stripNamespace=False, levels=0,
             stripUnderWorld=False):
        # type: (bool, Optional[bool], bool, int, bool) -> str
        '''The name of the node

        Parameters
        ----------
        update : bool
            if True, will always query to underlying maya object to get it's
            current name (and will therefore detect renames, re-parenting, etc);
            if False, it will use a cached value if available (which is slightly
            faster, but may be out of date)
        long : Optional[bool]
            if True, will include always include the full dag path, starting
            from the world root, including leading pipe ( | ); if False, will
            return the shortest-unique path; if None, node names will always be
            returned without any parents, if if they are not unique
        stripNamespace : bool
            if True, all nodes will have their namespaces stipped off of them
            (or a certain number of them, if levels is also used)
        levels : int
            if stripNamespace is True, then this number will determine the how
            many namespaces will be removed; if 0 (the default), then all
            leading namespaces will be removed; otherwise, this value gives the
            number of left-most levels to strip
        stripUnderWorld : bool
            if stripUnderWorld is True, and the name has underworld components
            (ie, topNode|topNodeShape->underWorld|underWorldShape), then only
            the portion in the "deepest underworld" is returned (ie,
            underWorld|underWorldShape)


        Returns
        -------
        str


        Examples
        --------
        >>> import pymel.core as pm
        >>> pm.newFile(f=1)
        ''
        >>> cube1 = pm.polyCube()[0]
        >>> cube2 = pm.polyCube()[0]
        >>> cube3 = pm.polyCube()[0]
        >>> cube3Shape = cube3.getShape()

        >>> cube2.setParent(cube1)
        nt.Transform('pCube2')
        >>> cube3.setParent(cube2)
        nt.Transform('pCube3')

        >>> pm.namespace(add='foo')
        'foo'
        >>> pm.namespace(add='bar', parent='foo')
        'foo:bar'
        >>> pm.namespace(add='stuff', parent='foo:bar')
        'foo:bar:stuff'

        >>> cube2.rename(':foo:pCube2')
        nt.Transform('foo:pCube2')
        >>> cube3.rename(':foo:bar:pCube3')
        nt.Transform('foo:bar:pCube3')
        >>> cube3Shape.rename(':foo:bar:stuff:pCubeShape3')
        nt.Mesh('foo:bar:stuff:pCubeShape3')

        >>> cube3Shape.name()
        'foo:bar:stuff:pCubeShape3'
        >>> cube3Shape.name(stripNamespace=True)
        'pCubeShape3'
        >>> cube3Shape.name(long=True)
        '|pCube1|foo:pCube2|foo:bar:pCube3|foo:bar:stuff:pCubeShape3'
        >>> cube3Shape.name(long=True, stripNamespace=True)
        '|pCube1|pCube2|pCube3|pCubeShape3'
        >>> cube3Shape.name(long=True, stripNamespace=True, levels=1)
        '|pCube1|pCube2|bar:pCube3|bar:stuff:pCubeShape3'
        >>> cube3Shape.name(long=True, stripNamespace=True, levels=2)
        '|pCube1|pCube2|pCube3|stuff:pCubeShape3'

        >>> cam = pm.camera()[0]
        >>> cam.setParent(cube2)
        nt.Transform('camera1')
        >>> imagePlane = pm.imagePlane(camera=cam.getShape())[1]
        >>> imagePlane.rename('foo:bar:stuff:imagePlaneShape1')
        nt.ImagePlane('cameraShape1->foo:bar:stuff:imagePlaneShape1')

        >>> imagePlane.name()
        'cameraShape1->foo:bar:stuff:imagePlaneShape1'
        >>> imagePlane.name(stripUnderWorld=True)
        'foo:bar:stuff:imagePlaneShape1'
        >>> imagePlane.name(stripNamespace=True, levels=1)
        'cameraShape1->bar:stuff:imagePlaneShape1'
        >>> imagePlane.name(stripUnderWorld=True, long=True)
        '|imagePlane1|foo:bar:stuff:imagePlaneShape1'
        >>> imagePlane.name(stripUnderWorld=True, stripNamespace=True, long=True)
        '|imagePlane1|imagePlaneShape1'
        '''
        if update or long or self._name is None:
            exists = True
            if _pymel_options['deleted_pynode_name_access'] != 'ignore':
                exists = self.exists()
            try:
                name = self._updateName(long)
                # _updateName for dag nodes won't actually check if the object
                # still exists... so we do that check ourselves. Also, even
                # though we may already know the object doesn't exist, we still
                # try to update the name... because having an updated name may
                # be nice for any potential error / warning messages
                if not exists:
                    raise general.MayaObjectError
            except general.MayaObjectError:
                # if we have an error, but we're only looking for the nodeName,
                # use the non-dag version
                if long is None and exists:
                    # don't use DependNode._updateName, as that can still
                    # raise MayaInstanceError - want this to work, so people
                    # have a way to get the correct instance, assuming they know
                    # what the parent should be
                    name = _api.MFnDependencyNode(self.__apimobject__()).name()
                else:
                    if not exists:
                        general.DeletedMayaNodeError.handle(self)
                    name = self._name
                    if name is None:
                        # if we've never gotten a name, but we're set to ignore
                        # deleted node errors, then just reraise the original
                        # error
                        raise
        else:
            name = self._name

        if stripNamespace or stripUnderWorld or long is None:
            worlds = []
            underworldSplit = name.split('->')
            if stripUnderWorld:
                underworldSplit = [underworldSplit[-1]]

            for worldName in underworldSplit:
                if stripNamespace or long is None:
                    parentSplit = worldName.split('|')
                    if long is None:
                        parentSplit = [parentSplit[-1]]
                    if stripNamespace:
                        nodes = []
                        for node in parentSplit:
                            # not sure what dag node has a "." in it's name, but keeping
                            # this code here just in case...
                            dotSplit = node.split('.')
                            spaceSplit = dotSplit[0].split(':')
                            if levels:
                                dotSplit[0] = ':'.join(spaceSplit[min(len(spaceSplit) - 1,
                                                                      levels):])
                            else:
                                dotSplit[0] = spaceSplit[-1]
                            nodes.append('.'.join(dotSplit))
                    else:
                        nodes = parentSplit
                    worldName = '|'.join(nodes)
                worlds.append(worldName)
            name = '->'.join(worlds)
        return name

    def longName(self, **kwargs):
        # type: (**Any) -> str
        """
        The full dag path to the object, including leading pipe ( | )

        Returns
        -------
        str

        Examples
        --------
        >>> import pymel.core as pm
        >>> pm.newFile(f=1)
        ''
        >>> cube1 = pm.polyCube()[0]
        >>> cube2 = pm.polyCube()[0]
        >>> cube3 = pm.polyCube()[0]
        >>> cube3Shape = cube3.getShape()

        >>> cube2.setParent(cube1)
        nt.Transform('pCube2')
        >>> cube3.setParent(cube2)
        nt.Transform('pCube3')

        >>> pm.namespace(add='foo')
        'foo'
        >>> pm.namespace(add='bar', parent='foo')
        'foo:bar'
        >>> pm.namespace(add='stuff', parent='foo:bar')
        'foo:bar:stuff'

        >>> cube2.rename(':foo:pCube2')
        nt.Transform('foo:pCube2')
        >>> cube3.rename(':foo:bar:pCube3')
        nt.Transform('foo:bar:pCube3')
        >>> cube3Shape.rename(':foo:bar:stuff:pCubeShape3')
        nt.Mesh('foo:bar:stuff:pCubeShape3')

        >>> cube3Shape.longName()
        '|pCube1|foo:pCube2|foo:bar:pCube3|foo:bar:stuff:pCubeShape3'
        >>> cube3Shape.longName(stripNamespace=True)
        '|pCube1|pCube2|pCube3|pCubeShape3'
        >>> cube3Shape.longName(stripNamespace=True, levels=1)
        '|pCube1|pCube2|bar:pCube3|bar:stuff:pCubeShape3'
        >>> cube3Shape.longName(stripNamespace=True, levels=2)
        '|pCube1|pCube2|pCube3|stuff:pCubeShape3'

        >>> cam = pm.camera()[0]
        >>> cam.setParent(cube2)
        nt.Transform('camera1')
        >>> imagePlane = pm.imagePlane(camera=cam.getShape())[1]
        >>> imagePlane.rename('foo:bar:stuff:imagePlaneShape1')
        nt.ImagePlane('cameraShape1->foo:bar:stuff:imagePlaneShape1')

        >>> imagePlane.longName()
        '|pCube1|foo:pCube2|camera1|cameraShape1->|imagePlane1|foo:bar:stuff:imagePlaneShape1'
        >>> imagePlane.longName(stripUnderWorld=True)
        '|imagePlane1|foo:bar:stuff:imagePlaneShape1'
        >>> imagePlane.longName(stripNamespace=True, levels=1)
        '|pCube1|pCube2|camera1|cameraShape1->|imagePlane1|bar:stuff:imagePlaneShape1'
        >>> imagePlane.longName(stripUnderWorld=True, stripNamespace=True)
        '|imagePlane1|imagePlaneShape1'
        """
        return self.name(long=True, **kwargs)
    fullPath = longName

    def shortName(self, **kwargs):
        # type: (**Any) -> str
        """
        The shortest unique name.

        Returns
        -------
        str

        Examples
        --------
        >>> import pymel.core as pm
        >>> pm.newFile(f=1)
        ''
        >>> cube1 = pm.polyCube()[0]
        >>> cube2 = pm.polyCube()[0]
        >>> cube3 = pm.polyCube()[0]
        >>> cube3Shape = cube3.getShape()

        >>> cube2.setParent(cube1)
        nt.Transform('pCube2')
        >>> cube3.setParent(cube2)
        nt.Transform('pCube3')

        >>> pm.namespace(add='foo')
        'foo'
        >>> pm.namespace(add='bar', parent='foo')
        'foo:bar'
        >>> pm.namespace(add='stuff', parent='foo:bar')
        'foo:bar:stuff'

        >>> cube2.rename(':foo:pCube2')
        nt.Transform('foo:pCube2')
        >>> cube3.rename(':foo:bar:pCube3')
        nt.Transform('foo:bar:pCube3')
        >>> cube3Shape.rename(':foo:bar:stuff:pCubeShape3')
        nt.Mesh('foo:bar:stuff:pCubeShape3')

        >>> cube3Shape.shortName()
        'foo:bar:stuff:pCubeShape3'
        >>> cube3Shape.shortName(stripNamespace=True)
        'pCubeShape3'
        >>> cube3Shape.shortName(stripNamespace=True, levels=1)
        'bar:stuff:pCubeShape3'
        >>> cube3Shape.shortName(stripNamespace=True, levels=2)
        'stuff:pCubeShape3'

        >>> cam = pm.camera()[0]
        >>> cam.setParent(cube2)
        nt.Transform('camera1')
        >>> imagePlane = pm.imagePlane(camera=cam.getShape())[1]
        >>> imagePlane.rename('foo:bar:stuff:imagePlaneShape1')
        nt.ImagePlane('cameraShape1->foo:bar:stuff:imagePlaneShape1')

        >>> imagePlane.shortName()
        'cameraShape1->foo:bar:stuff:imagePlaneShape1'
        >>> imagePlane.shortName(stripUnderWorld=True)
        'foo:bar:stuff:imagePlaneShape1'
        >>> imagePlane.shortName(stripNamespace=True, levels=1)
        'cameraShape1->bar:stuff:imagePlaneShape1'
        >>> imagePlane.shortName(stripUnderWorld=True)
        'foo:bar:stuff:imagePlaneShape1'
        >>> imagePlane.shortName(stripUnderWorld=True, stripNamespace=True)
        'imagePlaneShape1'
        """
        return self.name(long=False, **kwargs)

    # TODO: better support for underworld nodes (ie, in conjunction with
    # stripNamespace)
    def nodeName(self, stripUnderWorld=True, **kwargs):
        # type: (Any, **Any) -> str
        """
        Just the name of the node, without any dag path

        Returns
        -------
        str

        Examples
        --------
        >>> import pymel.core as pm
        >>> pm.newFile(f=1)
        ''
        >>> cube1 = pm.polyCube()[0]
        >>> cube2 = pm.polyCube()[0]
        >>> cube3 = pm.polyCube()[0]
        >>> cube3Shape = cube3.getShape()

        >>> cube2.setParent(cube1)
        nt.Transform('pCube2')
        >>> cube3.setParent(cube2)
        nt.Transform('pCube3')

        >>> pm.namespace(add='foo')
        'foo'
        >>> pm.namespace(add='bar', parent='foo')
        'foo:bar'
        >>> pm.namespace(add='stuff', parent='foo:bar')
        'foo:bar:stuff'

        >>> cube2.rename(':foo:pCube2')
        nt.Transform('foo:pCube2')
        >>> cube3.rename(':foo:bar:pCube3')
        nt.Transform('foo:bar:pCube3')
        >>> cube3Shape.rename(':foo:bar:stuff:pCubeShape3')
        nt.Mesh('foo:bar:stuff:pCubeShape3')

        >>> # create an object with the same name as pCube3 / pCube4
        >>> cube3Twin = pm.polyCube()[0]
        >>> cube3Twin.rename('foo:bar:pCube3')
        nt.Transform('|foo:bar:pCube3')
        >>> cube3ShapeTwin = cube3Twin.getShape()
        >>> cube3ShapeTwin.rename('foo:bar:stuff:pCubeShape3')
        nt.Mesh('|foo:bar:pCube3|foo:bar:stuff:pCubeShape3')

        >>> cube3Shape.shortName()
        'foo:pCube2|foo:bar:pCube3|foo:bar:stuff:pCubeShape3'
        >>> cube3Shape.nodeName()
        'foo:bar:stuff:pCubeShape3'
        >>> cube3Shape.nodeName(stripNamespace=True)
        'pCubeShape3'
        >>> cube3Shape.nodeName(stripNamespace=True, levels=1)
        'bar:stuff:pCubeShape3'
        >>> cube3Shape.nodeName(stripNamespace=True, levels=2)
        'stuff:pCubeShape3'

        >>> cam = pm.camera()[0]
        >>> cam.setParent(cube2)
        nt.Transform('camera1')
        >>> imagePlaneTrans, imagePlane = pm.imagePlane(camera=cam.getShape())
        >>> imagePlane.rename('foo:bar:stuff:imagePlaneShape1')
        nt.ImagePlane('cameraShape1->foo:bar:stuff:imagePlaneShape1')

        >>> # create an object with the same name as cam
        >>> pm.camera()[0].setParent(cube3Twin).rename('camera1')
        nt.Transform('|foo:bar:pCube3|camera1')

        >>> # create an object with the same name as imagePlane
        >>> imagePlaneTwinTrans, imagePlaneTwin = pm.imagePlane(camera=cam.getShape())
        >>> imagePlaneTwin.rename('foo:bar:stuff:imagePlaneShape1')
        nt.ImagePlane('foo:pCube2|camera1|cameraShape1->imagePlane2|foo:bar:stuff:imagePlaneShape1')

        >>> imagePlane.shortName()
        'foo:pCube2|camera1|cameraShape1->imagePlane1|foo:bar:stuff:imagePlaneShape1'
        >>> imagePlane.nodeName()
        'foo:bar:stuff:imagePlaneShape1'
        >>> imagePlane.nodeName(stripUnderWorld=False)
        'cameraShape1->foo:bar:stuff:imagePlaneShape1'
        >>> imagePlane.nodeName(stripNamespace=True)
        'imagePlaneShape1'
        >>> imagePlane.nodeName(stripNamespace=True, levels=1)
        'bar:stuff:imagePlaneShape1'
        """
        return self.name(long=None, stripUnderWorld=stripUnderWorld, **kwargs)

    def __apiobject__(self):
        "get the ``maya.OpenMaya.MDagPath`` for this object if it is valid"
        return self.__apimdagpath__()

    def __apimdagpath__(self):
        "get the ``maya.OpenMaya.MDagPath`` for this object if it is valid"

        try:
            dag = self.__apiobjects__['MDagPath']
            # If we have a cached mobject, test for validity: if the object is
            # not valid an error will be raised
            # Check if MObjectHandle in self.__apiobjects__ to avoid recursive
            # loop...
            if 'MObjectHandle' in self.__apiobjects__:
                self.__apimobject__()
            if not dag.isValid():
                # Usually this only happens if the object was reparented, with
                # instancing.
                #
                # Most of the time, this makes sense - there's no way to know
                # which of the new instances we "want".  Hoever, occasionally,
                # when the object was reparented, there were multiple instances,
                # and the MDagPath was invalidated; however, subsequently, other
                # instances were removed, so it's no longer instanced. Check for
                # this...

                # in some cases, doing dag.node() will crash maya if the dag
                # isn't valid... so try using MObjectHandle
                handle = self.__apiobjects__.get('MObjectHandle')
                if handle is not None and handle.isValid():
                    mfnDag = _api.MFnDagNode(handle.object())
                    if not mfnDag.isInstanced():
                        # throw a KeyError, this will cause to regen from
                        # first MDagPath
                        raise KeyError
                    raise general.MayaInstanceError(mfnDag.name())
                else:
                    name = getattr(self, '_name', '<unknown>')
                    raise general.MayaInstanceError(name)
            return dag
        except KeyError:
            # class was instantiated from an MObject, but we can still retrieve the first MDagPath

            #assert argObj.hasFn( _api.MFn.kDagNode )
            dag = _api.MDagPath()
            # we can't use self.__apimfn__() becaue the mfn is instantiated from an MDagPath
            # which we are in the process of finding out
            mfn = _api.MFnDagNode(self.__apimobject__())
            mfn.getPath(dag)
            self.__apiobjects__['MDagPath'] = dag
            return dag
#            if dag.isValid():
#                #argObj = dag
#                if dag.fullPathName():
#                    argObj = dag
#                else:
#                    print 'produced valid MDagPath with no name: %s(%s)' % ( argObj.apiTypeStr(), _api.MFnDependencyNode(argObj).name() )

    def __apihandle__(self):
        "get the ``maya.OpenMaya.MObjectHandle`` for this node if it is valid"
        try:
            handle = self.__apiobjects__['MObjectHandle']
        except KeyError:
            try:
                handle = _api.MObjectHandle(self.__apimdagpath__().node())
            except general.MayaInstanceError:
                if 'MDagPath' in self.__apiobjects__:
                    handle = _api.MObjectHandle(self.__apiobjects__['MDagPath'].node())
                else:
                    raise general.MayaNodeError(self._name)
            except RuntimeError:
                raise general.MayaNodeError(self._name)
            self.__apiobjects__['MObjectHandle'] = handle
        return handle

#    def __apimfn__(self):
#        if self._apimfn:
#            return self._apimfn
#        elif self.__apicls__:
#            obj = self._apiobject
#            if _api.isValidMDagPath(obj):
#                try:
#                    self._apimfn = self.__apicls__(obj)
#                    return self._apimfn
#                except KeyError:
#                    pass

#    def __init__(self, *args, **kwargs):
#        if self._apiobject:
#            if isinstance(self._apiobject, _api.MObjectHandle):
#                dagPath = _api.MDagPath()
#                _api.MDagPath.getAPathTo( self._apiobject.object(), dagPath )
#                self._apiobject = dagPath
#
#            assert _api.isValidMDagPath( self._apiobject )


#    def __init__(self, *args, **kwargs) :
#        if args :
#            arg = args[0]
#            if len(args) > 1 :
#                comp = args[1]
#            if isinstance(arg, DagNode) :
#                self._name = unicode(arg.name())
#                self._apiobject = _api.MObjectHandle(arg.object())
#            elif _api.isValidMObject(arg) or _api.isValidMObjectHandle(arg) :
#                objHandle = _api.MObjectHandle(arg)
#                obj = objHandle.object()
#                if _api.isValidMDagNode(obj) :
#                    self._apiobject = objHandle
#                    self._updateName()
#                else :
#                    raise TypeError, "%r might be a dependencyNode, but not a dagNode" % arg
#            elif isinstance(arg, basestring) :
#                obj = _api.toMObject (arg)
#                if obj :
#                    # creation for existing object
#                    if _api.isValidMDagNode (obj):
#                        self._apiobject = _api.MObjectHandle(obj)
#                        self._updateName()
#                    else :
#                        raise TypeError, "%r might be a dependencyNode, but not a dagNode" % arg
#                else :
#                    # creation for inexistent object
#                    self._name = arg
#            else :
#                raise TypeError, "don't know how to make a DagNode out of a %s : %r" % (type(arg), arg)


# -------------------------------
# xxx{  Path Info and Modification
# -------------------------------

    def root(self):
        # type: () -> DagNode
        """rootOf

        Returns
        -------
        DagNode
        """
        return DagNode('|' + self.longName()[1:].split('|')[0])

    def hasParent(self, parent):
        # type: (Union[str, DagNode]) -> bool
        '''
        Modifications:
        - handles underworld nodes correctly
        '''
        toMObj = _factories.ApiTypeRegister.inCast['MObject']
        # This will error if parent is not a dag, or not a node, like default
        # wrapped implementation
        parentMObj = toMObj(parent)
        thisMFn = self.__apimfn__()
        if thisMFn.hasParent(parentMObj):
            return True

        # quick out... MFnDagNode handles things right if all instances aren't
        # underworld nodes

        if not thisMFn.isInstanced() and not thisMFn.inUnderWorld():
            return False

        # See if it's underworld parent is the given parent...
        # Note that MFnDagPath implementation considers all instances, so we
        # should too...
        allPaths = _api.MDagPathArray()
        thisMFn.getAllPaths(allPaths)
        for i in range(allPaths.length()):
            path = allPaths[i]
            pathCount = path.pathCount()
            if pathCount <= 1:
                continue
            # if there's an underworld, there should be at least 3 nodes -
            # the top parent, the underworld root, and the node in the
            # underworld
            assert path.length() >= 3
            # if they are in the same underworld, MFnDagNode.hasParent would
            # work - only care about the case where the "immediate" parent is
            # outside of this node's underworld
            # Unfortunately, MDagPath.pop() has some strange behavior   - ie, if
            #     path = |camera1|cameraShape1->|imagePlane1
            # Then popping it once gives:
            #     path = |camera1|cameraShape1->|
            # ...and again gives:
            #     path = |camera1|cameraShape1
            # So, we check that popping once has the same pathCount, but twice
            # has a different path count
            path.pop()
            if path.pathCount() != pathCount:
                continue
            path.pop()
            if path.pathCount() != pathCount - 1:
                continue
            if path.node() == parentMObj:
                return True
        return False

    def hasChild(self, child):
        # type: (Union[str, DagNode]) -> bool
        '''
        Modifications:
        - handles underworld nodes correctly
        '''
        toMObj = _factories.ApiTypeRegister.inCast['MObject']
        # This will error if parent is not a dag, or not a node, like default
        # wrapped implementation
        childMObj = toMObj(child)
        thisMFn = self.__apimfn__()
        if self.__apimfn__().hasChild(childMObj):
            return True

        # because hasChild / hasParent consider all instances,
        # self.hasChild(child) is equivalent to child.hasParent(self)...
        if not isinstance(child, general.PyNode):
            child = DagNode(childMObj)
        return child.hasParent(self)

    def isChildOf(self, parent):
        # type: (Union[str, DagNode]) -> bool
        '''
        Modifications:
        - handles underworld nodes correctly
        '''
        toMObj = _factories.ApiTypeRegister.inCast['MObject']
        # This will error if parent is not a dag, or not a node, like default
        # wrapped implementation
        parentMObj = toMObj(parent)
        thisMFn = self.__apimfn__()
        if thisMFn.isChildOf(parentMObj):
            return True

        # quick out... MFnDagNode handles things right if all instances aren't
        # underworld nodes
        if not thisMFn.isInstanced() and not thisMFn.inUnderWorld():
            return False

        # For each instance path, if it's in the underworld, check to see
        # if the parent at the same "underworld" level as the parent is the
        # parent, or a child of it
        dagArray = _api.MDagPathArray()
        _api.MDagPath.getAllPathsTo(parentMObj, dagArray)
        allParentLevels = set()
        for i in range(dagArray.length()):
            parentMDag = dagArray[i]
            allParentLevels.add(parentMDag.pathCount())
        # we only get one parentMFn, but this should be fine as (aside from
        # not dealing with underworld correctly), MFnDagNode.isParentOf works
        # correctly for all instances...
        parentMFn = _api.MFnDagNode(parentMObj)

        dagArray.clear()
        thisMFn.getAllPaths(dagArray)
        for i in range(dagArray.length()):
            childMDag = dagArray[i]
            childPathLevels = childMDag.pathCount()
            if childPathLevels <= 1:
                continue
            for parentUnderworldLevel in allParentLevels:
                if childMDag.pathCount() <= parentUnderworldLevel:
                    # standard MFnDagNode.isChildOf would have handled this...
                    continue
                sameLevelMDag = _api.MDagPath()
                childMDag.getPath(sameLevelMDag, parentUnderworldLevel - 1)
                sameLevelMObj = sameLevelMDag.node()
                if sameLevelMObj == parentMObj:
                    return True
                if parentMFn.isParentOf(sameLevelMObj):
                    return True
        return False

    def isParentOf(self, child):
        # type: (Union[str, DagNode]) -> bool
        '''
        Modifications:
        - handles underworld nodes correctly
        '''
        toMObj = _factories.ApiTypeRegister.inCast['MObject']
        # This will error if parent is not a dag, or not a node, like default
        # wrapped implementation
        childMObj = toMObj(child)
        thisMFn = self.__apimfn__()
        if thisMFn.isParentOf(childMObj):
            return True

        # because isChildOf / isParentOf consider all instances,
        # self.isParentOf(child) is equivalent to child.isChildOf(self)...
        if not isinstance(child, general.PyNode):
            child = DagNode(childMObj)
        return child.isChildOf(self)

    def isInstanceOf(self, other):
        # type: (Union[str, DagNode]) -> bool
        """
        Returns
        -------
        bool
        """
        if isinstance(other, general.PyNode):
            return self.__apimobject__() == other.__apimobject__()
        else:
            try:
                return self.__apimobject__() == general.PyNode(other).__apimobject__()
            except:
                return False

    def instanceNumber(self):
        # type: () -> int
        """
        returns the instance number that this path represents in the DAG. The instance number can be used to determine which
        element of the world space array attributes of a DAG node to connect to get information regarding this instance.

        Returns
        -------
        int
        """
        return self.__apimdagpath__().instanceNumber()

    def getInstances(self, includeSelf=True):
        # type: (bool) -> List[DagNode]
        """
        Examples
        --------
        >>> from pymel.core import *
        >>> f=newFile(f=1) #start clean
        >>>
        >>> s = polyPlane()[0]
        >>> instance(s)
        [nt.Transform('pPlane2')]
        >>> instance(s)
        [nt.Transform('pPlane3')]
        >>> s.getShape().getInstances()
        [nt.Mesh('pPlane1|pPlaneShape1'), nt.Mesh('pPlane2|pPlaneShape1'), nt.Mesh('pPlane3|pPlaneShape1')]
        >>> s.getShape().getInstances(includeSelf=False)
        [nt.Mesh('pPlane2|pPlaneShape1'), nt.Mesh('pPlane3|pPlaneShape1')]

        """
        d = _api.MDagPathArray()
        self.__apimfn__().getAllPaths(d)
        thisDagPath = self.__apimdagpath__()
        result = [general.PyNode(_api.MDagPath(d[i])) for i in range(d.length()) if includeSelf or not d[i] == thisDagPath]

        return result

    def getOtherInstances(self):
        # type: () -> List[DagNode]
        """
        same as `DagNode.getInstances` with includeSelf=False.

        Returns
        -------
        List[DagNode]
        """
        return self.getInstances(includeSelf=False)

    def firstParent(self):
        # type: () -> DagNode
        """firstParentOf

        Returns
        -------
        DagNode
        """
        try:
            return DagNode('|'.join(self.longName().split('|')[:-1]))
        except TypeError:
            return DagNode('|'.join(self.split('|')[:-1]))

#    def numChildren(self):
#        """
#        see also `childCount`
#
#        :rtype: `int`
#        """
#        return self.__apimdagpath__().childCount()

#    def getParent(self, **kwargs):
#        # TODO : print warning regarding removal of kwargs, test speed difference
#        parent = _api.MDagPath( self.__apiobject__() )
#        try:
#            parent.pop()
#            return general.PyNode(parent)
#        except RuntimeError:
#            pass
#
#    def getChildren(self, **kwargs):
#        # TODO : print warning regarding removal of kwargs
#        children = []
#        thisDag = self.__apiobject__()
#        for i in range( thisDag.childCount() ):
#            child = _api.MDagPath( thisDag )
#            child.push( thisDag.child(i) )
#            children.append( general.PyNode(child) )
#        return children

    @_factories.addMelDocs('listRelatives', excludeFlags=['parent', 'fullPath'])
    def firstParent2(self, **kwargs):
        # type: (**Any) -> Optional[DagNode]
        """unlike the firstParent command which determines the parent via string formatting, this
        command uses the listRelatives command
        """

        kwargs['parent'] = True
        kwargs.pop('p', None)
        # if longNames:
        kwargs['fullPath'] = True
        kwargs.pop('f', None)

        try:
            res = cmds.listRelatives(self, **kwargs)[0]
        except TypeError:
            return None

        res = general.PyNode(res)
        return res

    @staticmethod
    def _getDagParent(dag):
        if dag.length() <= 1:
            return None
        # Need a copy as we'll be modifying it...
        parentDag = _api.MDagPath(dag)
        parentDag.pop()

        # do a check for underworld paths - if we have a path:
        # |rootTrans|rootShape -> |underwoldTrans|underworldShape
        # then two parents up, we will get:
        # |rootTrans|rootShape ->
        # ...whose .node() will be unusable. check for this scenario, and if
        # we get it, skip this dag path, so we go straight to:
        # |rootTrans|rootShape

        pathCount = parentDag.pathCount()
        if pathCount > 1:
            # get just the last "path piece" - if it is "empty", do an extra
            # pop, to get out of the underworld
            underworldPath = _api.MDagPath()
            parentDag.getPath(underworldPath, pathCount - 1)
            if underworldPath.length() == 0:
                parentDag.pop()

        return parentDag

    @overload
    def getParent(self, generations):
        # type: (Union[Literal[-1], Literal[-2]]) -> Transform
        pass

    @overload
    def getParent(self, generations=1):
        # type: (int) -> Optional[Transform]
        pass

    @overload
    def getParent(self, generations):
        # type: (None) -> List[Transform]
        pass

    def getParent(self, generations=1):
        # type: (Optional[int]) -> Optional[Transform]
        """
        Modifications:
            - added optional generations flag, which gives the number of levels up that you wish to go for the parent;
              ie:
                  >>> from pymel.core import *
                  >>> select(cl=1)
                  >>> bottom = group(n='bottom')
                  >>> group(n='almostThere')
                  nt.Transform('almostThere')
                  >>> group(n='nextLevel')
                  nt.Transform('nextLevel')
                  >>> group(n='topLevel')
                  nt.Transform('topLevel')
                  >>> bottom.longName()
                  '|topLevel|nextLevel|almostThere|bottom'
                  >>> bottom.getParent(2)
                  nt.Transform('nextLevel')

              Negative values will traverse from the top:

                  >>> bottom.getParent(generations=-3)
                  nt.Transform('almostThere')

              A value of 0 will return the same node.
              The default value is 1.

              If generations is None, it will be interpreted as 'return all
              parents', and a list will be returned.

              Since the original command returned None if there is no parent, to sync with this behavior, None will
              be returned if generations is out of bounds (no IndexError will be thrown).

        Returns
        -------
        Optional[DagNode]
        """

        # Get the parent through the api - listRelatives doesn't handle instances correctly,
        # and string processing seems unreliable...

        res = general._getParent(self._getDagParent, self.__apimdagpath__(), generations)

        if generations is None:
            if res is None:
                return []
            return [general.PyNode(x) for x in res]
        elif res is not None:
            return general.PyNode(res)

    def getAllParents(self):
        # type: () -> List[DagNode]
        """
        Return a list of all parents above this.

        Starts from the parent immediately above, going up.

        Returns
        -------
        List[DagNode]
        """
        return self.getParent(generations=None)

    # keep these overloads in sync with listRelatives
    @overload
    @_factories.addMelDocs('listRelatives', excludeFlags=['children'])
    def getChildren(self, type=None, **kwargs):
        # type: (Type[general.DagNodeT], **Any) -> List[general.DagNodeT]
        pass

    @overload
    @_factories.addMelDocs('listRelatives', excludeFlags=['children'])
    def getChildren(self, shapes=True, **kwargs):
        # type: (Literal[True], **Any) -> List[Shape]
        pass

    @overload
    @_factories.addMelDocs('listRelatives', excludeFlags=['children'])
    def getChildren(self, type=Ellipsis, **kwargs):
        # type: (Union[str, Iterable[Union[str, Type[DagNode]]]], **Any) -> List[DagNode]
        pass

    @_factories.addMelDocs('listRelatives', excludeFlags=['children'])
    def getChildren(self, **kwargs):
        # type: (**Any) -> List[DagNode]
        """
        see also `childAtIndex`

        for flags, see ``pymel.core.general.listRelatives``

        Returns
        -------
        List[DagNode]
        """
        kwargs['children'] = True
        kwargs.pop('c', None)

        return general.listRelatives(self, **kwargs)

    # FIXME: this needs the same overrides as listRelatives
    @_factories.addMelDocs('listRelatives', excludeFlags=['children'])
    def getSiblings(self, **kwargs):
        # type: (**Any) -> List[DagNode]
        """
        for flags, see ``pymel.core.general.listRelatives``

        Returns
        -------
        List[DagNode]
        """
        # pass
        try:
            return [x for x in self.getParent().getChildren(**kwargs) if x != self]
        except:
            return []

    @overload
    @_factories.addMelDocs('listRelatives')
    def listRelatives(self, type=None, **kwargs):
        # type: (Type[DagNodeT], **Any) -> List[DagNodeT]
        pass

    @overload
    @_factories.addMelDocs('listRelatives')
    def listRelatives(self, shapes=True, **kwargs):
        # type: (Literal[True], **Any) -> List[Shape]
        pass

    @overload
    @_factories.addMelDocs('listRelatives')
    def listRelatives(self, type=Ellipsis, **kwargs):
        # type: (Union[str, Iterable[Union[str, Type[DagNode]]]], **Any) -> List[DagNode]
        pass

    @_factories.addMelDocs('listRelatives')
    def listRelatives(self, **kwargs):
        # type: (**Any) -> List[DagNode]
        """
        for flags, see ``pymel.core.general.listRelatives``

        Returns
        -------
        List[DagNode]
        """
        return general.listRelatives(self, **kwargs)

    @_factories.addMelDocs('parent')
    def setParent(self, *args, **kwargs):
        """
        parent

        Modifications:
            - if parent is 'None', world=True is automatically set
            - if the given parent is the current parent, don't error

        """
        result = general.parent(self, *args, **kwargs)
        if result:
            result = result[0]
        return result

    @_factories.addMelDocs('parent')
    def addChild(self, child, **kwargs):
        # type: (Union[str, DagNode], **Any) -> DagNode
        """parent (reversed)
        """
        cmds.parent(child, self, **kwargs)
        if not isinstance(child, general.PyNode):
            child = general.PyNode(child)
        return child

    def __or__(self, child, **kwargs):
        # type: (DagNode, **Any) -> DagNode
        """
        operator for `addChild`. Use to easily daisy-chain together parenting operations.
        The operation order visually mimics the resulting dag path:

            >>> from pymel.core import *
            >>> s = polySphere(name='sphere')[0]
            >>> c = polyCube(name='cube')[0]
            >>> t = polyTorus(name='torus')[0]
            >>> s | c | t
            nt.Transform('torus')
            >>> print(t.fullPath())
            |sphere|cube|torus

        Returns
        -------
        DagNode
        """
        return self.addChild(child, **kwargs)

#}
    #instance = instance

    # -------------------------
    #    Shading
    # -------------------------

    def isDisplaced(self):
        # type: () -> bool
        """
        Returns whether any of this object's shading groups have a
        displacement shader input

        Returns
        -------
        bool
        """
        for sg in self.shadingGroups():
            if len(sg.attr('displacementShader').inputs()):
                return True
        return False

    def hide(self):
        # type: () -> None
        self.visibility.set(0)

    def show(self):
        # type: () -> None
        self.visibility.set(1)

    def isVisible(self, checkOverride=True):
        # type: (bool) -> bool
        if not self.attr('visibility').get():
            return False
        if (checkOverride and self.attr('overrideEnabled').get()
                and not self.attr('overrideVisibility').get()):
            return False
        parent = self.getParent()
        if not parent:
            return True
        else:
            return parent.isVisible(checkOverride=checkOverride)

    def setObjectColor(self, color=None):
        """
        This command sets the dormant wireframe color of the specified objects
        to an integer representing one of the user defined colors, or, if set
        to None, to the default class color
        """

        kwargs = {}
        if color:
            kwargs['userDefined'] = color
        cmds.color(self, **kwargs)

    def makeLive(self, state=True):
        # type: (bool) -> None
        if not state:
            cmds.makeLive(none=True)
        else:
            cmds.makeLive(self)
# ------ Do not edit below this line --------
    __melnode__ = 'dagNode'
    __slots__ = ()
    MObjectColorType = Enum('MObjectColorType', [('defaultColor', 0), ('kUseDefaultColor', 0), ('indexColor', 1), ('kUseIndexColor', 1), ('RGBColor', 2), ('kUseRGBColor', 2)], multiKeys=True)

    @_f.addApiDocs(_api.MFnDagNode, 'activeColor')
    def activeColor(self):
        # type: () -> datatypes.Color
        res = _f.getProxyResult(self, _api.MFnDagNode, 'activeColor')
        return _f.ApiArgUtil._castResult(self, res, 'MColor', None)

    @_f.addApiDocs(_api.MFnDagNode, 'boundingBox')
    def boundingBox(self):
        # type: () -> datatypes.BoundingBox
        res = _f.getProxyResult(self, _api.MFnDagNode, 'boundingBox')
        return _f.ApiArgUtil._castResult(self, res, 'MBoundingBox', None)

    @_f.addApiDocs(_api.MFnDagNode, 'child')
    def child(self, i):
        # type: (int) -> DependNode
        do, final_do, outTypes = _f.getDoArgs([i], [('i', 'uint', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDagNode, 'child', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MObject', None)
        return res
    childAtIndex = child

    @_f.addApiDocs(_api.MFnDagNode, 'childCount')
    def childCount(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnDagNode, 'childCount')
        return _f.ApiArgUtil._castResult(self, res, 'uint', None)
    numChildren = childCount

    @_f.addApiDocs(_api.MFnDagNode, 'dagRoot')
    def dagRoot(self):
        # type: () -> DependNode
        res = _f.getProxyResult(self, _api.MFnDagNode, 'dagRoot')
        return _f.ApiArgUtil._castResult(self, res, 'MObject', None)
    root = dagRoot

    @_f.addApiDocs(_api.MFnDagNode, 'dormantColor')
    def dormantColor(self):
        # type: () -> datatypes.Color
        res = _f.getProxyResult(self, _api.MFnDagNode, 'dormantColor')
        return _f.ApiArgUtil._castResult(self, res, 'MColor', None)

    @_f.addApiDocs(_api.MFnDagNode, 'drawOverrideColor')
    def drawOverrideColor(self):
        # type: () -> Tuple[bool, datatypes.Color]
        do, final_do, outTypes = _f.getDoArgs([], [('color', 'MColor', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDagNode, 'drawOverrideColor', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnDagNode, 'drawOverrideEnabled')
    def drawOverrideEnabled(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnDagNode, 'drawOverrideEnabled')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnDagNode, 'drawOverrideIsReference')
    def drawOverrideIsReference(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnDagNode, 'drawOverrideIsReference')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnDagNode, 'drawOverrideIsTemplate')
    def drawOverrideIsTemplate(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnDagNode, 'drawOverrideIsTemplate')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnDagNode, 'fullPathName')
    def fullPathName(self):
        # type: () -> str
        res = _f.getProxyResult(self, _api.MFnDagNode, 'fullPathName')
        return _f.ApiArgUtil._castResult(self, res, 'MString', None)

    @_f.deprecated('Use getInstances instead.')
    def getAllPaths(self):
        # type: () -> List[DagNode]
        do, final_do, outTypes = _f.getDoArgs([], [('paths', 'MDagPathArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDagNode, 'getAllPaths', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnDagNode, 'getConnectedSetsAndMembers')
    def getConnectedSetsAndMembers(self, instanceNumber, renderableSetsOnly):
        # type: (int, bool) -> Tuple[List[DependNode], List[DependNode]]
        do, final_do, outTypes = _f.getDoArgs([instanceNumber, renderableSetsOnly], [('instanceNumber', 'uint', 'in', None), ('sets', 'MObjectArray', 'out', None), ('comps', 'MObjectArray', 'out', None), ('renderableSetsOnly', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDagNode, 'getConnectedSetsAndMembers', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.maya_deprecated
    def getObjectColor(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnDagNode, 'objectColor')
        return _f.ApiArgUtil._castResult(self, res, 'int', None)

    @_f.addApiDocs(_api.MFnDagNode, 'objectColorType')
    def getObjectColorType(self):
        # type: () -> DagNode.MObjectColorType
        res = _f.getProxyResult(self, _api.MFnDagNode, 'objectColorType')
        return _f.ApiArgUtil._castResult(self, res, ('MFnDagNode', 'MObjectColorType'), None)

    @_f.addApiDocs(_api.MFnDagNode, 'hiliteColor')
    def hiliteColor(self):
        # type: () -> datatypes.Color
        res = _f.getProxyResult(self, _api.MFnDagNode, 'hiliteColor')
        return _f.ApiArgUtil._castResult(self, res, 'MColor', None)

    @_f.addApiDocs(_api.MFnDagNode, 'inModel')
    def inModel(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnDagNode, 'inModel')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnDagNode, 'inUnderWorld')
    def inUnderWorld(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnDagNode, 'inUnderWorld')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnDagNode, 'instanceCount')
    def instanceCount(self, total=True):
        # type: (bool) -> int
        do, final_do, outTypes = _f.getDoArgs([total], [('total', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDagNode, 'instanceCount', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'uint', None)
        return res

    @_f.addApiDocs(_api.MFnDagNode, 'isInstanceable')
    def isInstanceable(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnDagNode, 'isInstanceable')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnDagNode, 'isInstanced')
    def isInstanced(self, indirect=True):
        # type: (bool) -> bool
        do, final_do, outTypes = _f.getDoArgs([indirect], [('indirect', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDagNode, 'isInstanced', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res

    @_f.addApiDocs(_api.MFnDagNode, 'isInstancedAttribute')
    def isInstancedAttribute(self, attribute):
        # type: (str | DependNode) -> bool
        do, final_do, outTypes = _f.getDoArgs([attribute], [('attribute', 'MObject', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDagNode, 'isInstancedAttribute', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res

    @_f.addApiDocs(_api.MFnDagNode, 'isIntermediateObject')
    def isIntermediateObject(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnDagNode, 'isIntermediateObject')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)
    isIntermediate = isIntermediateObject

    @_f.addApiDocs(_api.MFnDagNode, 'objectColorIndex')
    def objectColorIndex(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnDagNode, 'objectColorIndex')
        return _f.ApiArgUtil._castResult(self, res, 'int', None)

    @_f.addApiDocs(_api.MFnDagNode, 'objectColorRGB')
    def objectColorRGB(self):
        # type: () -> datatypes.Color
        res = _f.getProxyResult(self, _api.MFnDagNode, 'objectColorRGB')
        return _f.ApiArgUtil._castResult(self, res, 'MColor', None)

    @_f.addApiDocs(_api.MFnDagNode, 'objectGroupComponent')
    def objectGroupComponent(self, groupId, compList):
        # type: (int, List[str]) -> None
        do, final_do, outTypes = _f.getDoArgs([groupId, compList], [('groupId', 'uint', 'in', None), ('compList', 'MStringArray', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDagNode, 'objectGroupComponent', final_do)
        return res

    @_f.addApiDocs(_api.MFnDagNode, 'parent')
    def parent(self, i):
        # type: (int) -> DependNode
        do, final_do, outTypes = _f.getDoArgs([i], [('i', 'uint', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDagNode, 'parent', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MObject', None)
        return res
    parentAtIndex = parent

    @_f.addApiDocs(_api.MFnDagNode, 'parentCount')
    def parentCount(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnDagNode, 'parentCount')
        return _f.ApiArgUtil._castResult(self, res, 'uint', None)

    @_f.addApiDocs(_api.MFnDagNode, 'partialPathName')
    def partialPathName(self):
        # type: () -> str
        res = _f.getProxyResult(self, _api.MFnDagNode, 'partialPathName')
        return _f.ApiArgUtil._castResult(self, res, 'MString', None)

    @_f.addApiDocs(_api.MFnDagNode, 'setInstanceable')
    def setInstanceable(self, how):
        # type: (bool) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([how], [('how', 'bool', 'in', None)], self.isInstanceable, self.setInstanceable, [])
        res = _f.getProxyResult(self, _api.MFnDagNode, 'setInstanceable', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnDagNode, 'setIntermediateObject')
    def setIntermediateObject(self, isIntermediate):
        # type: (bool) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([isIntermediate], [('isIntermediate', 'bool', 'in', None)], self.isIntermediateObject, self.setIntermediateObject, [])
        res = _f.getProxyResult(self, _api.MFnDagNode, 'setIntermediateObject', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res
    setIntermediate = setIntermediateObject

    @_f.addApiDocs(_api.MFnDagNode, 'setObjectColorType')
    def setObjectColorType(self, type):
        # type: (DagNode.MObjectColorType) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([type], [('type', ('MFnDagNode', 'MObjectColorType'), 'in', None)], self.getObjectColorType, self.setObjectColorType, [])
        res = _f.getProxyResult(self, _api.MFnDagNode, 'setObjectColorType', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.maya_deprecated
    def setUseObjectColor(self, useObjectColor):
        # type: (bool) -> None
        do, final_do, outTypes = _f.getDoArgs([useObjectColor], [('useObjectColor', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDagNode, 'setUseObjectColor', final_do)
        return res

    @_f.addApiDocs(_api.MFnDagNode, 'transformationMatrix')
    def transformationMatrix(self):
        # type: () -> datatypes.Matrix
        res = _f.getProxyResult(self, _api.MFnDagNode, 'transformationMatrix')
        return _f.ApiArgUtil._castResult(self, res, 'MMatrix', None)

    @_f.addApiDocs(_api.MFnDagNode, 'usingHiliteColor')
    def usingHiliteColor(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnDagNode, 'usingHiliteColor')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.maya_deprecated
    def usingObjectColor(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnDagNode, 'usingObjectColor')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)
    isUsingObjectColor = usingObjectColor
# ------ Do not edit above this line --------


class Shape(DagNode):

    if TYPE_CHECKING:
        @overload
        def getParent(self):
            # type: () -> Transform
            pass

        @overload
        def getParent(self, generations):
            # type: (Optional[int]) -> Optional[Transform]
            pass

    def getTransform(self):
        return self.getParent(generations=1)

    def setParent(self, *args, **kwargs):
        if 'shape' not in kwargs and 's' not in kwargs:
            kwargs['s'] = True
        super(Shape, self).setParent(*args, **kwargs)
# ------ Do not edit below this line --------
    __melnode__ = 'shape'
    __slots__ = ()
# ------ Do not edit above this line --------


class Camera(Shape):
    # FIXME: the functionFactory is causing these methods to have their docs doubled-up,  in both pymel.track, and pymel.Camera.track
    if not _factories.building:
        orbit = rendering.orbit
        track = rendering.track
        tumble = rendering.tumble

    def applyBookmark(self, bookmark):
        kwargs = {}
        kwargs['camera'] = self
        kwargs['edit'] = True
        kwargs['setCamera'] = True

        cmds.cameraView(bookmark, **kwargs)

    def addBookmark(self, bookmark=None):
        kwargs = {}
        kwargs['camera'] = self
        kwargs['addBookmark'] = True
        if bookmark:
            kwargs['name'] = bookmark

        cmds.cameraView(**kwargs)

    def removeBookmark(self, bookmark):
        kwargs = {}
        kwargs['camera'] = self
        kwargs['removeBookmark'] = True
        kwargs['name'] = bookmark

        cmds.cameraView(**kwargs)

    def updateBookmark(self, bookmark):
        kwargs = {}
        kwargs['camera'] = self
        kwargs['edit'] = True
        kwargs['setView'] = True

        cmds.cameraView(bookmark, **kwargs)

    def listBookmarks(self):
        return self.bookmarks.inputs()

    @_factories.addMelDocs('dolly')
    def dolly(self, distance, relative=True):
        kwargs = {}
        kwargs['distance'] = distance
        if relative:
            kwargs['relative'] = True
        else:
            kwargs['absolute'] = True
        cmds.dolly(self, **kwargs)

    @_factories.addMelDocs('roll')
    def roll(self, degree, relative=True):
        kwargs = {}
        kwargs['degree'] = degree
        if relative:
            kwargs['relative'] = True
        else:
            kwargs['absolute'] = True
        cmds.roll(self, **kwargs)

    @_f.addApiDocs(_api.MFnCamera, 'setNearFarClippingPlanes')
    def setNearFarClippingPlanes(self, dNear, dFar):
        redoArgs = (dNear, dFar)
        undoArgs = (self.getNearClippingPlane(), self.getFarClippingPlane())
        self._setNearFarClippingPlanes(dNear, dFar)
        undoItem = _factories.ApiUndoItem(self._setNearFarClippingPlanes,
                                          redoArgs, undoArgs)
        _factories.apiUndo.append(undoItem)
    # add an alias - don't technically need one, but since we now have
    # aliases for other variants, makes sense to make this one too
    setNearFarClipPlanes = setNearFarClippingPlanes
# ------ Do not edit below this line --------
    __apicls__ = _api.MFnCamera
    __melcmd__ = staticmethod(rendering.camera)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'camera'
    __melnode__ = 'camera'
    __slots__ = ()
    FilmFit = Enum('FilmFit', [('fillFilmFit', 0), ('kFillFilmFit', 0), ('horizontalFilmFit', 1), ('kHorizontalFilmFit', 1), ('verticalFilmFit', 2), ('kVerticalFilmFit', 2), ('overscanFilmFit', 3), ('kOverscanFilmFit', 3), ('invalid', 4), ('kInvalid', 4)], multiKeys=True)
    RollOrder = Enum('RollOrder', [('rotateTranslate', 0), ('kRotateTranslate', 0), ('translateRotate', 1), ('kTranslateRotate', 1)], multiKeys=True)

    @_f.addApiDocs(_api.MFnCamera, 'setNearFarClippingPlanes')
    def _setNearFarClippingPlanes(self, dNear, dFar):
        # type: (float, float) -> None
        do, final_do, outTypes = _f.getDoArgs([dNear, dFar], [('dNear', 'double', 'in', 'linear'), ('dFar', 'double', 'in', 'linear')])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setNearFarClippingPlanes', final_do)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'computeDepthOfField')
    def computeDepthOfField(self):
        # type: () -> None
        res = _f.getProxyResult(self, _api.MFnCamera, 'computeDepthOfField')
        return res

    @_f.addApiDocs(_api.MFnCamera, 'copyViewFrom')
    def copyViewFrom(self, otherCamera):
        # type: (DagNode) -> None
        do, final_do, outTypes = _f.getDoArgs([otherCamera], [('otherCamera', 'MDagPath', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnCamera, 'copyViewFrom', final_do)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'aspectRatio')
    def getAspectRatio(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnCamera, 'aspectRatio')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addApiDocs(_api.MFnCamera, 'getAspectRatioLimits')
    def getAspectRatioLimits(self):
        # type: () -> Tuple[float, float]
        do, final_do, outTypes = _f.getDoArgs([], [('min', 'double', 'out', None), ('max', 'double', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnCamera, 'getAspectRatioLimits', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnCamera, 'cameraScale')
    def getCameraScale(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnCamera, 'cameraScale')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addApiDocs(_api.MFnCamera, 'centerOfInterest')
    def getCenterOfInterest(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnCamera, 'centerOfInterest')
        return _f.ApiArgUtil._castResult(self, res, 'double', 'linear')

    @_f.addApiDocs(_api.MFnCamera, 'centerOfInterestPoint')
    def getCenterOfInterestPoint(self, space='preTransform'):
        # type: (datatypes.Space.Space) -> datatypes.Point
        do, final_do, outTypes = _f.getDoArgs([space], [('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnCamera, 'centerOfInterestPoint', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MPoint', None)
        return res

    @_f.addMelDocs('camera', 'displayFieldChart')
    def getDisplayFieldChart(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, rendering.camera, kwargs, 'displayFieldChart')
        return res

    @_f.addMelDocs('camera', 'displayFilmOrigin')
    def getDisplayFilmOrigin(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, rendering.camera, kwargs, 'displayFilmOrigin')
        return res

    @_f.addMelDocs('camera', 'displayFilmPivot')
    def getDisplayFilmPivot(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, rendering.camera, kwargs, 'displayFilmPivot')
        return res

    @_f.addMelDocs('camera', 'displayGateMask')
    def getDisplayGateMask(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, rendering.camera, kwargs, 'displayGateMask')
        return res

    @_f.addMelDocs('camera', 'displayResolution')
    def getDisplayResolution(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, rendering.camera, kwargs, 'displayResolution')
        return res

    @_f.addMelDocs('camera', 'displaySafeAction')
    def getDisplaySafeAction(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, rendering.camera, kwargs, 'displaySafeAction')
        return res

    @_f.addMelDocs('camera', 'displaySafeTitle')
    def getDisplaySafeTitle(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, rendering.camera, kwargs, 'displaySafeTitle')
        return res

    @_f.addApiDocs(_api.MFnCamera, 'eyePoint')
    def getEyePoint(self, space='preTransform'):
        # type: (datatypes.Space.Space) -> datatypes.Point
        do, final_do, outTypes = _f.getDoArgs([space], [('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnCamera, 'eyePoint', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MPoint', None)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'fStop')
    def getFStop(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnCamera, 'fStop')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addApiDocs(_api.MFnCamera, 'farClippingPlane')
    def getFarClippingPlane(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnCamera, 'farClippingPlane')
        return _f.ApiArgUtil._castResult(self, res, 'double', 'linear')
    getFarClipPlane = getFarClippingPlane

    @_f.addApiDocs(_api.MFnCamera, 'farFocusDistance')
    def getFarFocusDistance(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnCamera, 'farFocusDistance')
        return _f.ApiArgUtil._castResult(self, res, 'double', 'linear')

    @_f.addApiDocs(_api.MFnCamera, 'getFilmApertureLimits')
    def getFilmApertureLimits(self):
        # type: () -> Tuple[float, float]
        do, final_do, outTypes = _f.getDoArgs([], [('min', 'double', 'out', None), ('max', 'double', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnCamera, 'getFilmApertureLimits', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnCamera, 'filmFit')
    def getFilmFit(self):
        # type: () -> Camera.FilmFit
        res = _f.getProxyResult(self, _api.MFnCamera, 'filmFit')
        return _f.ApiArgUtil._castResult(self, res, ('MFnCamera', 'FilmFit'), None)

    @_f.addApiDocs(_api.MFnCamera, 'filmFitOffset')
    def getFilmFitOffset(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnCamera, 'filmFitOffset')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addApiDocs(_api.MFnCamera, 'getFilmFrustum')
    def getFilmFrustum(self, distance, applyPanZoom=False):
        # type: (float, bool) -> List[datatypes.Point]
        do, final_do, outTypes = _f.getDoArgs([distance, applyPanZoom], [('distance', 'double', 'in', 'linear'), ('clipPlanes', 'MPointArray', 'out', None), ('applyPanZoom', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnCamera, 'getFilmFrustum', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnCamera, 'filmRollOrder')
    def getFilmRollOrder(self):
        # type: () -> Camera.RollOrder
        res = _f.getProxyResult(self, _api.MFnCamera, 'filmRollOrder')
        return _f.ApiArgUtil._castResult(self, res, ('MFnCamera', 'RollOrder'), None)

    @_f.addApiDocs(_api.MFnCamera, 'filmRollValue')
    def getFilmRollValue(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnCamera, 'filmRollValue')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addApiDocs(_api.MFnCamera, 'filmTranslateH')
    def getFilmTranslateH(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnCamera, 'filmTranslateH')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addApiDocs(_api.MFnCamera, 'filmTranslateV')
    def getFilmTranslateV(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnCamera, 'filmTranslateV')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addApiDocs(_api.MFnCamera, 'focalLength')
    def getFocalLength(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnCamera, 'focalLength')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addApiDocs(_api.MFnCamera, 'getFocalLengthLimits')
    def getFocalLengthLimits(self):
        # type: () -> Tuple[float, float]
        do, final_do, outTypes = _f.getDoArgs([], [('min', 'double', 'out', None), ('max', 'double', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnCamera, 'getFocalLengthLimits', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnCamera, 'focusDistance')
    def getFocusDistance(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnCamera, 'focusDistance')
        return _f.ApiArgUtil._castResult(self, res, 'double', 'linear')

    @_f.addMelDocs('camera', 'homeCommand')
    def getHomeCommand(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, rendering.camera, kwargs, 'homeCommand')
        return res

    @_f.addApiDocs(_api.MFnCamera, 'horizontalFieldOfView')
    def getHorizontalFieldOfView(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnCamera, 'horizontalFieldOfView')
        return _f.ApiArgUtil._castResult(self, res, 'double', 'angular')

    @_f.addApiDocs(_api.MFnCamera, 'horizontalFilmAperture')
    def getHorizontalFilmAperture(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnCamera, 'horizontalFilmAperture')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addApiDocs(_api.MFnCamera, 'horizontalFilmOffset')
    def getHorizontalFilmOffset(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnCamera, 'horizontalFilmOffset')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addApiDocs(_api.MFnCamera, 'horizontalPan')
    def getHorizontalPan(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnCamera, 'horizontalPan')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addApiDocs(_api.MFnCamera, 'horizontalRollPivot')
    def getHorizontalRollPivot(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnCamera, 'horizontalRollPivot')
        return _f.ApiArgUtil._castResult(self, res, 'double', 'linear')

    @_f.addApiDocs(_api.MFnCamera, 'horizontalShake')
    def getHorizontalShake(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnCamera, 'horizontalShake')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addMelDocs('camera', 'journalCommand')
    def getJournalCommand(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, rendering.camera, kwargs, 'journalCommand')
        return res

    @_f.addApiDocs(_api.MFnCamera, 'lensSqueezeRatio')
    def getLensSqueezeRatio(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnCamera, 'lensSqueezeRatio')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addMelDocs('camera', 'lockTransform')
    def getLockTransform(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, rendering.camera, kwargs, 'lockTransform')
        return res

    @_f.addApiDocs(_api.MFnCamera, 'nearClippingPlane')
    def getNearClippingPlane(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnCamera, 'nearClippingPlane')
        return _f.ApiArgUtil._castResult(self, res, 'double', 'linear')
    getNearClipPlane = getNearClippingPlane

    @_f.addApiDocs(_api.MFnCamera, 'nearFocusDistance')
    def getNearFocusDistance(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnCamera, 'nearFocusDistance')
        return _f.ApiArgUtil._castResult(self, res, 'double', 'linear')

    @_f.addApiDocs(_api.MFnCamera, 'orthoWidth')
    def getOrthoWidth(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnCamera, 'orthoWidth')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addApiDocs(_api.MFnCamera, 'overscan')
    def getOverscan(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnCamera, 'overscan')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addApiDocs(_api.MFnCamera, 'panZoomEnabled')
    def getPanZoomEnabled(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnCamera, 'panZoomEnabled')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnCamera, 'getPortFieldOfView')
    def getPortFieldOfView(self, width, height):
        # type: (int, int) -> Tuple[float, float]
        do, final_do, outTypes = _f.getDoArgs([width, height], [('width', 'int', 'in', None), ('height', 'int', 'in', None), ('horizontalFOV', 'double', 'out', 'angular'), ('verticalFOV', 'double', 'out', 'angular')])
        res = _f.getProxyResult(self, _api.MFnCamera, 'getPortFieldOfView', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnCamera, 'postScale')
    def getPostScale(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnCamera, 'postScale')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addApiDocs(_api.MFnCamera, 'preScale')
    def getPreScale(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnCamera, 'preScale')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addApiDocs(_api.MFnCamera, 'renderPanZoom')
    def getRenderPanZoom(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnCamera, 'renderPanZoom')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnCamera, 'getRenderingFrustum')
    def getRenderingFrustum(self, windowAspect):
        # type: (float) -> Tuple[float, float, float, float]
        do, final_do, outTypes = _f.getDoArgs([windowAspect], [('windowAspect', 'double', 'in', None), ('left', 'double', 'out', None), ('right', 'double', 'out', None), ('bottom', 'double', 'out', None), ('top', 'double', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnCamera, 'getRenderingFrustum', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnCamera, 'shakeEnabled')
    def getShakeEnabled(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnCamera, 'shakeEnabled')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnCamera, 'shakeOverscan')
    def getShakeOverscan(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnCamera, 'shakeOverscan')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addApiDocs(_api.MFnCamera, 'shakeOverscanEnabled')
    def getShakeOverscanEnabled(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnCamera, 'shakeOverscanEnabled')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnCamera, 'shutterAngle')
    def getShutterAngle(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnCamera, 'shutterAngle')
        return _f.ApiArgUtil._castResult(self, res, 'double', 'angular')

    @_f.addMelDocs('camera', 'startupCamera')
    def getStartupCamera(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, rendering.camera, kwargs, 'startupCamera')
        return res

    @_f.addApiDocs(_api.MFnCamera, 'stereoHIT')
    def getStereoHIT(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnCamera, 'stereoHIT')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addApiDocs(_api.MFnCamera, 'stereoHITEnabled')
    def getStereoHITEnabled(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnCamera, 'stereoHITEnabled')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnCamera, 'tumblePivot')
    def getTumblePivot(self):
        # type: () -> datatypes.Point
        res = _f.getProxyResult(self, _api.MFnCamera, 'tumblePivot')
        return _f.ApiArgUtil._castResult(self, res, 'MPoint', None)

    @_f.addApiDocs(_api.MFnCamera, 'usePivotAsLocalSpace')
    def getUsePivotAsLocalSpace(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnCamera, 'usePivotAsLocalSpace')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnCamera, 'verticalFieldOfView')
    def getVerticalFieldOfView(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnCamera, 'verticalFieldOfView')
        return _f.ApiArgUtil._castResult(self, res, 'double', 'angular')

    @_f.addApiDocs(_api.MFnCamera, 'verticalFilmAperture')
    def getVerticalFilmAperture(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnCamera, 'verticalFilmAperture')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addApiDocs(_api.MFnCamera, 'verticalFilmOffset')
    def getVerticalFilmOffset(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnCamera, 'verticalFilmOffset')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addApiDocs(_api.MFnCamera, 'verticalPan')
    def getVerticalPan(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnCamera, 'verticalPan')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addApiDocs(_api.MFnCamera, 'verticalRollPivot')
    def getVerticalRollPivot(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnCamera, 'verticalRollPivot')
        return _f.ApiArgUtil._castResult(self, res, 'double', 'linear')

    @_f.addApiDocs(_api.MFnCamera, 'verticalShake')
    def getVerticalShake(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnCamera, 'verticalShake')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addApiDocs(_api.MFnCamera, 'getViewParameters')
    def getViewParameters(self, windowAspect, applyOverscan=False, applySqueeze=False, applyPanZoom=False):
        # type: (float, bool, bool, bool) -> Tuple[float, float, float, float]
        do, final_do, outTypes = _f.getDoArgs([windowAspect, applyOverscan, applySqueeze, applyPanZoom], [('windowAspect', 'double', 'in', None), ('apertureX', 'double', 'out', None), ('apertureY', 'double', 'out', None), ('offsetX', 'double', 'out', None), ('offsetY', 'double', 'out', None), ('applyOverscan', 'bool', 'in', None), ('applySqueeze', 'bool', 'in', None), ('applyPanZoom', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnCamera, 'getViewParameters', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnCamera, 'getViewingFrustum')
    def getViewingFrustum(self, windowAspect, applyOverscan=False, applySqueeze=False, applyPanZoom=False):
        # type: (float, bool, bool, bool) -> Tuple[float, float, float, float]
        do, final_do, outTypes = _f.getDoArgs([windowAspect, applyOverscan, applySqueeze, applyPanZoom], [('windowAspect', 'double', 'in', None), ('left', 'double', 'out', None), ('right', 'double', 'out', None), ('bottom', 'double', 'out', None), ('top', 'double', 'out', None), ('applyOverscan', 'bool', 'in', None), ('applySqueeze', 'bool', 'in', None), ('applyPanZoom', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnCamera, 'getViewingFrustum', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addMelDocs('camera', 'worldCenterOfInterest')
    def getWorldCenterOfInterest(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, rendering.camera, kwargs, 'worldCenterOfInterest')
        return res

    @_f.addMelDocs('camera', 'worldUp')
    def getWorldUp(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, rendering.camera, kwargs, 'worldUp')
        return res

    @_f.addApiDocs(_api.MFnCamera, 'zoom')
    def getZoom(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnCamera, 'zoom')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addApiDocs(_api.MFnCamera, 'hasSamePerspective')
    def hasSamePerspective(self, otherCamera):
        # type: (DagNode) -> bool
        do, final_do, outTypes = _f.getDoArgs([otherCamera], [('otherCamera', 'MDagPath', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnCamera, 'hasSamePerspective', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'isClippingPlanes')
    def isClippingPlanes(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnCamera, 'isClippingPlanes')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnCamera, 'isDepthOfField')
    def isDepthOfField(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnCamera, 'isDepthOfField')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnCamera, 'isDisplayFilmGate')
    def isDisplayFilmGate(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnCamera, 'isDisplayFilmGate')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnCamera, 'isDisplayGateMask')
    def isDisplayGateMask(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnCamera, 'isDisplayGateMask')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnCamera, 'isMotionBlur')
    def isMotionBlur(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnCamera, 'isMotionBlur')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)
    isMotionBlurred = isMotionBlur

    @_f.addApiDocs(_api.MFnCamera, 'isOrtho')
    def isOrtho(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnCamera, 'isOrtho')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnCamera, 'isVerticalLock')
    def isVerticalLock(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnCamera, 'isVerticalLock')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnCamera, 'postProjectionMatrix')
    def postProjectionMatrix(self):
        # type: () -> datatypes.FloatMatrix
        res = _f.getProxyResult(self, _api.MFnCamera, 'postProjectionMatrix')
        return _f.ApiArgUtil._castResult(self, res, 'MFloatMatrix', None)

    @_f.addApiDocs(_api.MFnCamera, 'projectionMatrix')
    def projectionMatrix(self):
        # type: () -> datatypes.FloatMatrix
        res = _f.getProxyResult(self, _api.MFnCamera, 'projectionMatrix')
        return _f.ApiArgUtil._castResult(self, res, 'MFloatMatrix', None)

    @_f.addApiDocs(_api.MFnCamera, 'rightDirection')
    def rightDirection(self, space='preTransform'):
        # type: (datatypes.Space.Space) -> datatypes.Vector
        do, final_do, outTypes = _f.getDoArgs([space], [('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnCamera, 'rightDirection', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MVector', None)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'set')
    def set(self, wsEyeLocation, wsViewDirection, wsUpDirection, horizFieldOfView, aspectRatio):
        # type: (datatypes.Point, datatypes.Vector, datatypes.Vector, float, float) -> None
        do, final_do, outTypes = _f.getDoArgs([wsEyeLocation, wsViewDirection, wsUpDirection, horizFieldOfView, aspectRatio], [('wsEyeLocation', 'MPoint', 'in', None), ('wsViewDirection', 'MVector', 'in', None), ('wsUpDirection', 'MVector', 'in', None), ('horizFieldOfView', 'double', 'in', None), ('aspectRatio', 'double', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnCamera, 'set', final_do)
        return res
    setupCamera = set

    @_f.addApiDocs(_api.MFnCamera, 'setAspectRatio')
    def setAspectRatio(self, aspectRatio):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([aspectRatio], [('aspectRatio', 'double', 'in', None)], self.getAspectRatio, self.setAspectRatio, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setAspectRatio', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'setCameraScale')
    def setCameraScale(self, scale):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([scale], [('scale', 'double', 'in', None)], self.getCameraScale, self.setCameraScale, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setCameraScale', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'setCenterOfInterest')
    def setCenterOfInterest(self, dist):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([dist], [('dist', 'double', 'in', 'linear')], self.getCenterOfInterest, self.setCenterOfInterest, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setCenterOfInterest', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'setCenterOfInterestPoint')
    def setCenterOfInterestPoint(self, centerOfInterest, space='preTransform'):
        # type: (datatypes.Point, datatypes.Space.Space) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([centerOfInterest, space], [('centerOfInterest', 'MPoint', 'in', None), ('space', ('MSpace', 'Space'), 'in', None)], self.getCenterOfInterestPoint, self.setCenterOfInterestPoint, ['space'])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setCenterOfInterestPoint', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'setClippingPlanes')
    def setClippingPlanes(self, ClippingPlanes):
        # type: (bool) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([ClippingPlanes], [('ClippingPlanes', 'bool', 'in', None)], self.isClippingPlanes, self.setClippingPlanes, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setClippingPlanes', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'setDepthOfField')
    def setDepthOfField(self, depthOfField):
        # type: (bool) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([depthOfField], [('depthOfField', 'bool', 'in', None)], self.isDepthOfField, self.setDepthOfField, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setDepthOfField', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addMelDocs('camera', 'displayFieldChart')
    def setDisplayFieldChart(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, rendering.camera, kwargs, 'displayFieldChart', val)

    @_f.addApiDocs(_api.MFnCamera, 'setDisplayFilmGate')
    def setDisplayFilmGate(self, displayFilmGate):
        # type: (bool) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([displayFilmGate], [('displayFilmGate', 'bool', 'in', None)], self.isDisplayFilmGate, self.setDisplayFilmGate, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setDisplayFilmGate', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addMelDocs('camera', 'displayFilmOrigin')
    def setDisplayFilmOrigin(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, rendering.camera, kwargs, 'displayFilmOrigin', val)

    @_f.addMelDocs('camera', 'displayFilmPivot')
    def setDisplayFilmPivot(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, rendering.camera, kwargs, 'displayFilmPivot', val)

    @_f.addApiDocs(_api.MFnCamera, 'setDisplayGateMask')
    def setDisplayGateMask(self, displayGateMask):
        # type: (bool) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([displayGateMask], [('displayGateMask', 'bool', 'in', None)], self.isDisplayGateMask, self.setDisplayGateMask, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setDisplayGateMask', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addMelDocs('camera', 'displayResolution')
    def setDisplayResolution(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, rendering.camera, kwargs, 'displayResolution', val)

    @_f.addMelDocs('camera', 'displaySafeAction')
    def setDisplaySafeAction(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, rendering.camera, kwargs, 'displaySafeAction', val)

    @_f.addMelDocs('camera', 'displaySafeTitle')
    def setDisplaySafeTitle(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, rendering.camera, kwargs, 'displaySafeTitle', val)

    @_f.addApiDocs(_api.MFnCamera, 'setEyePoint')
    def setEyePoint(self, eyeLocation, space='preTransform'):
        # type: (datatypes.Point, datatypes.Space.Space) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([eyeLocation, space], [('eyeLocation', 'MPoint', 'in', None), ('space', ('MSpace', 'Space'), 'in', None)], self.getEyePoint, self.setEyePoint, ['space'])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setEyePoint', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'setFStop')
    def setFStop(self, fStop):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([fStop], [('fStop', 'double', 'in', None)], self.getFStop, self.setFStop, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setFStop', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'setFarClippingPlane')
    def setFarClippingPlane(self, dFar):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([dFar], [('dFar', 'double', 'in', 'linear')], self.getFarClippingPlane, self.setFarClippingPlane, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setFarClippingPlane', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res
    setFarClipPlane = setFarClippingPlane

    @_f.addApiDocs(_api.MFnCamera, 'setFarFocusDistance')
    def setFarFocusDistance(self, farFocusDistance):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([farFocusDistance], [('farFocusDistance', 'double', 'in', 'linear')], self.getFarFocusDistance, self.setFarFocusDistance, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setFarFocusDistance', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'setFilmFit')
    def setFilmFit(self, filmFit):
        # type: (Camera.FilmFit) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([filmFit], [('filmFit', ('MFnCamera', 'FilmFit'), 'in', None)], self.getFilmFit, self.setFilmFit, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setFilmFit', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'setFilmFitOffset')
    def setFilmFitOffset(self, filmFitOffset):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([filmFitOffset], [('filmFitOffset', 'double', 'in', None)], self.getFilmFitOffset, self.setFilmFitOffset, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setFilmFitOffset', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'setFilmRollOrder')
    def setFilmRollOrder(self, filmRollOrder):
        # type: (Camera.RollOrder) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([filmRollOrder], [('filmRollOrder', ('MFnCamera', 'RollOrder'), 'in', None)], self.getFilmRollOrder, self.setFilmRollOrder, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setFilmRollOrder', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'setFilmRollValue')
    def setFilmRollValue(self, filmRollValue):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([filmRollValue], [('filmRollValue', 'double', 'in', None)], self.getFilmRollValue, self.setFilmRollValue, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setFilmRollValue', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'setFilmTranslateH')
    def setFilmTranslateH(self, translate):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([translate], [('translate', 'double', 'in', None)], self.getFilmTranslateH, self.setFilmTranslateH, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setFilmTranslateH', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'setFilmTranslateV')
    def setFilmTranslateV(self, translate):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([translate], [('translate', 'double', 'in', None)], self.getFilmTranslateV, self.setFilmTranslateV, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setFilmTranslateV', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'setFocalLength')
    def setFocalLength(self, focalLength):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([focalLength], [('focalLength', 'double', 'in', None)], self.getFocalLength, self.setFocalLength, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setFocalLength', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'setFocusDistance')
    def setFocusDistance(self, distance):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([distance], [('distance', 'double', 'in', 'linear')], self.getFocusDistance, self.setFocusDistance, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setFocusDistance', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addMelDocs('camera', 'homeCommand')
    def setHomeCommand(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, rendering.camera, kwargs, 'homeCommand', val)

    @_f.addApiDocs(_api.MFnCamera, 'setHorizontalFieldOfView')
    def setHorizontalFieldOfView(self, fov):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([fov], [('fov', 'double', 'in', 'angular')], self.getHorizontalFieldOfView, self.setHorizontalFieldOfView, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setHorizontalFieldOfView', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'setHorizontalFilmAperture')
    def setHorizontalFilmAperture(self, hFilmAperture):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([hFilmAperture], [('hFilmAperture', 'double', 'in', None)], self.getHorizontalFilmAperture, self.setHorizontalFilmAperture, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setHorizontalFilmAperture', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'setHorizontalFilmOffset')
    def setHorizontalFilmOffset(self, hOffset):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([hOffset], [('hOffset', 'double', 'in', None)], self.getHorizontalFilmOffset, self.setHorizontalFilmOffset, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setHorizontalFilmOffset', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'setHorizontalPan')
    def setHorizontalPan(self, hPan):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([hPan], [('hPan', 'double', 'in', None)], self.getHorizontalPan, self.setHorizontalPan, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setHorizontalPan', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'setHorizontalRollPivot')
    def setHorizontalRollPivot(self, horizontalRollPivot):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([horizontalRollPivot], [('horizontalRollPivot', 'double', 'in', 'linear')], self.getHorizontalRollPivot, self.setHorizontalRollPivot, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setHorizontalRollPivot', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'setHorizontalShake')
    def setHorizontalShake(self, hOffset):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([hOffset], [('hOffset', 'double', 'in', None)], self.getHorizontalShake, self.setHorizontalShake, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setHorizontalShake', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addMelDocs('camera', 'journalCommand')
    def setJournalCommand(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, rendering.camera, kwargs, 'journalCommand', val)

    @_f.addApiDocs(_api.MFnCamera, 'setLensSqueezeRatio')
    def setLensSqueezeRatio(self, aspectRatio):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([aspectRatio], [('aspectRatio', 'double', 'in', None)], self.getLensSqueezeRatio, self.setLensSqueezeRatio, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setLensSqueezeRatio', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addMelDocs('camera', 'lockTransform')
    def setLockTransform(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, rendering.camera, kwargs, 'lockTransform', val)

    @_f.addApiDocs(_api.MFnCamera, 'setMotionBlur')
    def setMotionBlur(self, motionBlur):
        # type: (bool) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([motionBlur], [('motionBlur', 'bool', 'in', None)], self.isMotionBlur, self.setMotionBlur, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setMotionBlur', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res
    setMotionBlurred = setMotionBlur

    @_f.addApiDocs(_api.MFnCamera, 'setNearClippingPlane')
    def setNearClippingPlane(self, dNear):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([dNear], [('dNear', 'double', 'in', 'linear')], self.getNearClippingPlane, self.setNearClippingPlane, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setNearClippingPlane', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res
    setNearClipPlane = setNearClippingPlane

    @_f.addApiDocs(_api.MFnCamera, 'setNearFocusDistance')
    def setNearFocusDistance(self, nearFocusDistance):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([nearFocusDistance], [('nearFocusDistance', 'double', 'in', 'linear')], self.getNearFocusDistance, self.setNearFocusDistance, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setNearFocusDistance', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'setIsOrtho')
    def setOrtho(self, orthoState):
        # type: (bool) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([orthoState], [('orthoState', 'bool', 'in', None)], self.isOrtho, self.setOrtho, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setIsOrtho', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'setOrthoWidth')
    def setOrthoWidth(self, orthoWidth):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([orthoWidth], [('orthoWidth', 'double', 'in', None)], self.getOrthoWidth, self.setOrthoWidth, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setOrthoWidth', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'setOverscan')
    def setOverscan(self, overscan):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([overscan], [('overscan', 'double', 'in', None)], self.getOverscan, self.setOverscan, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setOverscan', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'setPanZoomEnabled')
    def setPanZoomEnabled(self, enabled):
        # type: (bool) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([enabled], [('enabled', 'bool', 'in', None)], self.getPanZoomEnabled, self.setPanZoomEnabled, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setPanZoomEnabled', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'setPostScale')
    def setPostScale(self, sf):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([sf], [('sf', 'double', 'in', None)], self.getPostScale, self.setPostScale, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setPostScale', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'setPreScale')
    def setPreScale(self, sf):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([sf], [('sf', 'double', 'in', None)], self.getPreScale, self.setPreScale, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setPreScale', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'setRenderPanZoom')
    def setRenderPanZoom(self, render):
        # type: (bool) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([render], [('render', 'bool', 'in', None)], self.getRenderPanZoom, self.setRenderPanZoom, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setRenderPanZoom', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'setShakeEnabled')
    def setShakeEnabled(self, enabled):
        # type: (bool) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([enabled], [('enabled', 'bool', 'in', None)], self.getShakeEnabled, self.setShakeEnabled, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setShakeEnabled', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'setShakeOverscan')
    def setShakeOverscan(self, overscan):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([overscan], [('overscan', 'double', 'in', None)], self.getShakeOverscan, self.setShakeOverscan, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setShakeOverscan', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'setShakeOverscanEnabled')
    def setShakeOverscanEnabled(self, enabled):
        # type: (bool) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([enabled], [('enabled', 'bool', 'in', None)], self.getShakeOverscanEnabled, self.setShakeOverscanEnabled, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setShakeOverscanEnabled', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'setShutterAngle')
    def setShutterAngle(self, shutterAngle):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([shutterAngle], [('shutterAngle', 'double', 'in', 'angular')], self.getShutterAngle, self.setShutterAngle, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setShutterAngle', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addMelDocs('camera', 'startupCamera')
    def setStartupCamera(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, rendering.camera, kwargs, 'startupCamera', val)

    @_f.addApiDocs(_api.MFnCamera, 'setStereoHIT')
    def setStereoHIT(self, hit):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([hit], [('hit', 'double', 'in', None)], self.getStereoHIT, self.setStereoHIT, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setStereoHIT', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'setStereoHITEnabled')
    def setStereoHITEnabled(self, enabled):
        # type: (bool) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([enabled], [('enabled', 'bool', 'in', None)], self.getStereoHITEnabled, self.setStereoHITEnabled, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setStereoHITEnabled', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'setTumblePivot')
    def setTumblePivot(self, point):
        # type: (datatypes.Point) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([point], [('point', 'MPoint', 'in', None)], self.getTumblePivot, self.setTumblePivot, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setTumblePivot', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'setUsePivotAsLocalSpace')
    def setUsePivotAsLocalSpace(self, how):
        # type: (bool) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([how], [('how', 'bool', 'in', None)], self.getUsePivotAsLocalSpace, self.setUsePivotAsLocalSpace, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setUsePivotAsLocalSpace', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'setVerticalFieldOfView')
    def setVerticalFieldOfView(self, fov):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([fov], [('fov', 'double', 'in', 'angular')], self.getVerticalFieldOfView, self.setVerticalFieldOfView, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setVerticalFieldOfView', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'setVerticalFilmAperture')
    def setVerticalFilmAperture(self, vFilmAperture):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([vFilmAperture], [('vFilmAperture', 'double', 'in', None)], self.getVerticalFilmAperture, self.setVerticalFilmAperture, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setVerticalFilmAperture', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'setVerticalFilmOffset')
    def setVerticalFilmOffset(self, vOffset):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([vOffset], [('vOffset', 'double', 'in', None)], self.getVerticalFilmOffset, self.setVerticalFilmOffset, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setVerticalFilmOffset', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'setVerticalLock')
    def setVerticalLock(self, lockFlag):
        # type: (bool) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([lockFlag], [('lockFlag', 'bool', 'in', None)], self.isVerticalLock, self.setVerticalLock, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setVerticalLock', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'setVerticalPan')
    def setVerticalPan(self, vPan):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([vPan], [('vPan', 'double', 'in', None)], self.getVerticalPan, self.setVerticalPan, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setVerticalPan', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'setVerticalRollPivot')
    def setVerticalRollPivot(self, verticalRollPivot):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([verticalRollPivot], [('verticalRollPivot', 'double', 'in', 'linear')], self.getVerticalRollPivot, self.setVerticalRollPivot, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setVerticalRollPivot', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'setVerticalShake')
    def setVerticalShake(self, vOffset):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([vOffset], [('vOffset', 'double', 'in', None)], self.getVerticalShake, self.setVerticalShake, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setVerticalShake', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addMelDocs('camera', 'worldCenterOfInterest')
    def setWorldCenterOfInterest(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, rendering.camera, kwargs, 'worldCenterOfInterest', val)

    @_f.addMelDocs('camera', 'worldUp')
    def setWorldUp(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, rendering.camera, kwargs, 'worldUp', val)

    @_f.addApiDocs(_api.MFnCamera, 'setZoom')
    def setZoom(self, zoom):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([zoom], [('zoom', 'double', 'in', None)], self.getZoom, self.setZoom, [])
        res = _f.getProxyResult(self, _api.MFnCamera, 'setZoom', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'unnormalizedFarClippingPlane')
    def unnormalizedFarClippingPlane(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnCamera, 'unnormalizedFarClippingPlane')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addApiDocs(_api.MFnCamera, 'unnormalizedNearClippingPlane')
    def unnormalizedNearClippingPlane(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnCamera, 'unnormalizedNearClippingPlane')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addApiDocs(_api.MFnCamera, 'upDirection')
    def upDirection(self, space='preTransform'):
        # type: (datatypes.Space.Space) -> datatypes.Vector
        do, final_do, outTypes = _f.getDoArgs([space], [('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnCamera, 'upDirection', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MVector', None)
        return res

    @_f.addApiDocs(_api.MFnCamera, 'viewDirection')
    def viewDirection(self, space='preTransform'):
        # type: (datatypes.Space.Space) -> datatypes.Vector
        do, final_do, outTypes = _f.getDoArgs([space], [('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnCamera, 'viewDirection', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MVector', None)
        return res
# ------ Do not edit above this line --------


class Transform(DagNode):
    _componentAttributes = {
        'rotatePivot': (general.Pivot, 'rotatePivot'),
        'scalePivot': (general.Pivot, 'scalePivot'),
    }
    if TYPE_CHECKING:
        # FIXME: move to build
        rotatePivot = None  # type: general.Pivot
        scalePivot = None  # type: general.Pivot

#    def __getattr__(self, attr):
#        try :
#            return super(general.PyNode, self).__getattr__(attr)
#        except AttributeError, msg:
#            try:
#                return self.getShape().attr(attr)
#            except AttributeError:
#                pass
#
#            # it doesn't exist on the class
#            try:
#                return self.attr(attr)
#            except MayaAttributeError, msg:
#                # try the shape
#                try: return self.getShape().attr(attr)
#                except AttributeError: pass
#                # since we're being called via __getattr__ we don't know whether the user was trying
#                # to get a class method or a maya attribute, so we raise a more generic AttributeError
#                raise AttributeError, msg

    def __getattr__(self, attr):
        """
        Checks in the following order:
            1. Functions on this node class
            2. Attributes on this node class
            3. Functions on this node class's shape
            4. Attributes on this node class's shape
        """
        try:
            # print "Transform.__getattr__(%r)" % attr
            # Functions through normal inheritance
            res = DependNode.__getattr__(self, attr)
        except AttributeError as e:
            # Functions via shape inheritance , and then, implicitly, Attributes
            for shape in self.getShapes():
                try:
                    return getattr(shape, attr)
                except AttributeError:
                    pass
            raise e
        return res

    def __setattr__(self, attr, val):
        # type: (str, Any) -> None
        """
        Checks in the following order:
            1. Functions on this node class
            2. Attributes on this node class
            3. Functions on this node class's shape
            4. Attributes on this node class's shape
        """
        try:
            # print "Transform.__setattr__", attr, val
            # Functions through normal inheritance
            return DependNode.__setattr__(self, attr, val)
        except AttributeError as e:
            # Functions via shape inheritance , and then, implicitly, Attributes
            # print "Trying shape"
            shape = self.getShape()
            if shape:
                try:
                    return setattr(shape, attr, val)
                except AttributeError:
                    pass
            raise e

    def attr(self, attr, checkShape=True):
        # type: (str, bool) -> general.Attribute
        """
        when checkShape is enabled, if the attribute does not exist the
        transform but does on the shape, then the shape's attribute will
        be returned.

        Parameters
        ----------
        attr : str
        checkShape : bool

        Returns
        -------
        general.Attribute
        """
        # print "ATTR: Transform"
        try:
            res = self._attr(attr, checkShape)
        except general.MayaAttributeError as e:
            if checkShape:
                try:
                    res = self.getShape().attr(attr)
                except AttributeError:
                    raise e
            raise e
        return res

#    def __getattr__(self, attr):
#        if attr.startswith('__') and attr.endswith('__'):
#            return super(general.PyNode, self).__getattr__(attr)
#
#        at = Attribute( '%s.%s' % (self, attr) )
#
#        # if the attribute does not exist on this node try the shape node
#        if not at.exists():
#            try:
#                childAttr = getattr( self.getShape(), attr)
#                try:
#                    if childAttr.exists():
#                        return childAttr
#                except AttributeError:
#                    return childAttr
#            except (AttributeError,TypeError):
#                pass
#
#        return at
#
#    def __setattr__(self, attr,val):
#        if attr.startswith('_'):
#            attr = attr[1:]
#
#        at = Attribute( '%s.%s' % (self, attr) )
#
#        # if the attribute does not exist on this node try the shape node
#        if not at.exists():
#            try:
#                childAttr = getattr( self.getShape(), attr )
#                try:
#                    if childAttr.exists():
#                        return childAttr.set(val)
#                except AttributeError:
#                    return childAttr.set(val)
#            except (AttributeError,TypeError):
#                pass
#
#        return at.set(val)

    """
    def move( self, *args, **kwargs ):
        return move( self, *args, **kwargs )
    def scale( self, *args, **kwargs ):
        return scale( self, *args, **kwargs )
    def rotate( self, *args, **kwargs ):
        return rotate( self, *args, **kwargs )
    def align( self, *args, **kwargs):
        args = (self,) + args
        cmds.align(self, *args, **kwargs)
    """
    # NOTE : removed this via proxyClass
#    # workaround for conflict with translate method on basestring
#    def _getTranslate(self):
#        return self.__getattr__("translate")
#    def _setTranslate(self, val):
#        return self.__setattr__("translate", val)
#    translate = property( _getTranslate , _setTranslate )

    def getShape(self, **kwargs):
        # type: (**Any) -> Shape
        """
        Returns
        -------
        Shape
        """
        kwargs['shapes'] = True
        try:
            return self.getChildren(**kwargs)[0]
        except IndexError:
            pass

    @overload
    @_factories.addMelDocs('listRelatives', excludeFlags=['shapes', 'children'])
    def getShapes(self, type=None, **kwargs):
        # type: (Type[general.DagNodeT], **Any) -> List[general.ShapeT]
        pass

    @overload
    @_factories.addMelDocs('listRelatives', excludeFlags=['shapes', 'children'])
    def getShapes(self, type=Ellipsis, **kwargs):
        # type: (Union[str, Iterable[Union[str, Type[Shape]]]], **Any) -> List[Shape]
        pass

    def getShapes(self, **kwargs):
        # type: (**Any) -> List[Shape]
        """
        Returns
        -------
        List[Shape]
        """
        kwargs['shapes'] = True
        return self.getChildren(**kwargs)

    def ungroup(self, **kwargs):
        return cmds.ungroup(self, **kwargs)


#    @_factories.editflag('xform','scale')
#    def setScale( self, val, **kwargs ):
#        cmds.xform( self, **kwargs )

#    @_factories.editflag('xform','rotation')
#    def setRotationOld( self, val, **kwargs ):
#        cmds.xform( self, **kwargs )
#
#    @_factories.editflag('xform','translation')
#    def setTranslationOld( self, val, **kwargs ):
#        cmds.xform( self, **kwargs )
#
#    @_factories.editflag('xform','scalePivot')
#    def setScalePivotOld( self, val, **kwargs ):
#        cmds.xform( self, **kwargs )
#
#    @_factories.editflag('xform','rotatePivot')
#    def setRotatePivotOld( self, val, **kwargs ):
#        cmds.xform( self, **kwargs )

#    @_factories.editflag('xform','pivots')
#    def setPivots( self, val, **kwargs ):
#        cmds.xform( self, **kwargs )

#    @_factories.editflag('xform','rotateAxis')
#    def setRotateAxisOld( self, val, **kwargs ):
#        cmds.xform( self, **kwargs )
#
#    @_factories.editflag('xform','shear')
#    def setShearingOld( self, val, **kwargs ):
#        cmds.xform( self, **kwargs )

    @_factories.addMelDocs('xform', 'rotateAxis')
    def setMatrix(self, val, **kwargs):
        """xform -scale"""
        kwargs['matrix'] = val
        cmds.xform(self, **kwargs)

#    @_factories.queryflag('xform','scale')
#    def getScaleOld( self, **kwargs ):
#        return datatypes.Vector( cmds.xform( self, **kwargs ) )

    def _getSpaceArg(self, space, kwargs):
        """for internal use only"""
        if kwargs.pop('worldSpace', kwargs.pop('ws', False)):
            space = 'world'
        elif kwargs.pop('objectSpace', kwargs.pop('os', False)):
            space = 'object'
        return space

    def _isRelativeArg(self, kwargs):

        isRelative = kwargs.pop('relative', kwargs.pop('r', None))
        if isRelative is None:
            isRelative = not kwargs.pop('absolute', kwargs.pop('a', True))
        return isRelative

#    @_factories.queryflag('xform','translation')
#    def getTranslationOld( self, **kwargs ):
#        return datatypes.Vector( cmds.xform( self, **kwargs ) )

    @_factories.addApiDocs(_api.MFnTransform, 'setTranslation')
    def setTranslation(self, vector, space='object', **kwargs):
        if self._isRelativeArg(kwargs):
            return self.translateBy(vector, space, **kwargs)
        space = self._getSpaceArg(space, kwargs)
        return self._setTranslation(vector, space=space)

    @_factories.addApiDocs(_api.MFnTransform, 'getTranslation')
    def getTranslation(self, space='object', **kwargs):
        space = self._getSpaceArg(space, kwargs)
        return self._getTranslation(space=space)

    @_factories.addApiDocs(_api.MFnTransform, 'translateBy')
    def translateBy(self, vector, space='object', **kwargs):
        space = self._getSpaceArg(space, kwargs)
        curr = self._getTranslation(space)
        self._translateBy(vector, space)
        new = self._getTranslation(space)
        undoItem = _factories.ApiUndoItem(Transform.setTranslation, (self, new, space), (self, curr, space))
        _factories.apiUndo.append(undoItem)

    @_factories.addApiDocs(_api.MFnTransform, 'setScale')
    def setScale(self, scale, **kwargs):
        if self._isRelativeArg(kwargs):
            return self.scaleBy(scale, **kwargs)
        return self._setScale(scale)

    @_factories.addApiDocs(_api.MFnTransform, 'scaleBy')
    def scaleBy(self, scale, **kwargs):
        curr = self.getScale()
        self._scaleBy(scale)
        new = self.getScale()
        undoItem = _factories.ApiUndoItem(Transform.setScale, (self, new), (self, curr))
        _factories.apiUndo.append(undoItem)

    @_factories.addApiDocs(_api.MFnTransform, 'setShear')
    def setShear(self, shear, **kwargs):
        if self._isRelativeArg(kwargs):
            return self.shearBy(shear, **kwargs)
        return self._setShear(shear)

    @_factories.addApiDocs(_api.MFnTransform, 'shearBy')
    def shearBy(self, shear, **kwargs):
        curr = self.getShear()
        self._shearBy(shear)
        new = self.getShear()
        undoItem = _factories.ApiUndoItem(Transform.setShear, (self, new), (self, curr))
        _factories.apiUndo.append(undoItem)


#    @_factories.queryflag('xform','rotatePivot')
#    def getRotatePivotOld( self, **kwargs ):
#        return datatypes.Vector( cmds.xform( self, **kwargs ) )

    @_factories.addApiDocs(_api.MFnTransform, 'setRotatePivot')
    def setRotatePivot(self, point, space='object', balance=True, **kwargs):
        space = self._getSpaceArg(space, kwargs)
        return self._setRotatePivot(point, space=space, balance=balance)

    @_factories.addApiDocs(_api.MFnTransform, 'rotatePivot')
    def getRotatePivot(self, space='object', **kwargs):
        space = self._getSpaceArg(space, kwargs)
        return self._getRotatePivot(space=space)

    @_factories.addApiDocs(_api.MFnTransform, 'setRotatePivotTranslation')
    def setRotatePivotTranslation(self, vector, space='object', **kwargs):
        space = self._getSpaceArg(space, kwargs)
        return self._setRotatePivotTranslation(vector, space=space)

    @_factories.addApiDocs(_api.MFnTransform, 'rotatePivotTranslation')
    def getRotatePivotTranslation(self, space='object', **kwargs):
        space = self._getSpaceArg(space, kwargs)
        return self._getRotatePivotTranslation(space=space)


#    @_factories.queryflag('xform','rotation')
#    def getRotationOld( self, **kwargs ):
#        return datatypes.Vector( cmds.xform( self, **kwargs ) )

    @_factories.addApiDocs(_api.MFnTransform, 'setRotation')
    def setRotation(self, rotation, space='object', **kwargs):
        '''
    Modifications:
      - rotation may be given as an EulerRotation, Quaternion, or iterable of 3
        or 4 components (to specify an euler/quaternion, respectively)
        '''
        # quaternions are the only method that support a space parameter
        if self._isRelativeArg(kwargs):
            return self.rotateBy(rotation, space, **kwargs)
        spaceIndex = datatypes.Spaces.getIndex(self._getSpaceArg(space, kwargs))

        if not isinstance(rotation, (_api.MQuaternion, _api.MEulerRotation)):
            rotation = list(rotation)
            if len(rotation) == 3:
                # using datatypes.Angle(x) means current angle-unit should be
                # respected
                rotation = [datatypes.Angle(x).asRadians() for x in rotation]
                rotation = _api.MEulerRotation(*rotation)
            elif len(rotation) == 4:
                rotation = _api.MQuaternion(*rotation)
            else:
                raise ValueError(
                    "rotation given to setRotation must have either 3 or 4 "
                    "elements (for euler or quaternion, respectively)")
        if isinstance(rotation, _api.MEulerRotation):
            # MFnTransform.setRotation doesn't have a (non-deprecated) override
            # which takes euler angles AND a transform space... this sort of
            # makes sense, since the "unique" information that euler angles can
            # potentially carry - ie, rotation > 360 degress - only really makes
            # sense within the "transform" space. So, only use EulerRotation if
            # we're using transform space...
            if datatypes.equivalentSpace(spaceIndex, _api.MSpace.kTransform,
                                         rotationOnly=True):
                self.__apimfn__().setRotation(rotation)
                return
            else:
                rotation = rotation.asQuaternion()
        self.__apimfn__().setRotation(rotation, spaceIndex)

#    @_factories.addApiDocs( _api.MFnTransform, 'getRotation' )
#    def getRotation(self, space='object', **kwargs):
#        # quaternions are the only method that support a space parameter
#        space = self._getSpaceArg(space, kwargs )
#        quat = _api.MQuaternion()
#        _api.MFnTransform(self.__apimfn__()).getRotation(quat, datatypes.Spaces.getIndex(space) )
#        return datatypes.EulerRotation( quat.asEulerRotation() )

    @_factories.addApiDocs(_api.MFnTransform, 'getRotation', overloadIndex=1)
    def getRotation(self, space='object', quaternion=False, **kwargs):
        '''
    Modifications:
      - added 'quaternion' keyword arg, to specify whether the result
        be returned as a Quaternion object, as opposed to the default
        EulerRotation object
      - added 'space' keyword arg, which defaults to 'object'
        '''
        # quaternions are the only method that support a space parameter
        space = self._getSpaceArg(space, kwargs)
        if space.lower() in ('object', 'pretransform', 'transform') and not quaternion:
            # In this case, we can just go straight to the EulerRotation,
            # without having to go through Quaternion - this means we will
            # get information like angles > 360 degrees
            euler = _api.MEulerRotation()
            self.__apimfn__().getRotation(euler)
            rot = datatypes.EulerRotation(euler)
        else:
            rot = self._getRotation(space=space)
            if not quaternion:
                rot = rot.asEulerRotation()
        if isinstance(rot, datatypes.EulerRotation):
            rot.setDisplayUnit(datatypes.Angle.getUIUnit())
        return rot

    @_factories.addApiDocs(_api.MFnTransform, 'rotateBy')
    def rotateBy(self, rotation, space='object', **kwargs):
        '''
        Modifications:
          - rotation may be a Quaternion or MQuaternion
        '''
        if isinstance(rotation, (_api.MQuaternion, datatypes.Quaternion)):
            self.rotateByQuaternion(rotation, space=space)
        space = self._getSpaceArg(space, kwargs)
        curr = self.getRotation(space)
        self._rotateBy(rotation, space)
        new = self.getRotation(space)
        undoItem = _factories.ApiUndoItem(Transform.setRotation,
                                          (self, new, space),
                                          (self, curr, space))
        _factories.apiUndo.append(undoItem)

    @_factories.addApiDocs(_api.MFnTransform, 'rotateByQuaternion')
    def rotateByQuaternion(self, xOrQuaternion, y=None, z=None, w=None, space='transform'):
        # type: (Union[float, Quaternion, List[float], Tuple[float]], float, float, float, datatypes.Space.Space) -> None
        '''
        Modifications:
          - may feed in a single Quaternion object/list/tuple instead of 4 separate floats
        '''
        if isinstance(xOrQuaternion, (int, float)):
            x = xOrQuaternion
        else:
            if y is not None or z is not None or w is not None:
                raise ValueError("if xOrQuaternion is a Quaternion or iterable, you may not specify y/z/w")
            x, y, z, w = xOrQuaternion
        return self._rotateByQuaternion(x, y, z, w, space=space)

    #    @_factories.queryflag('xform','scalePivot')
#    def getScalePivotOld( self, **kwargs ):
#        return datatypes.Vector( cmds.xform( self, **kwargs ) )

    @_factories.addApiDocs(_api.MFnTransform, 'setScalePivot')
    def setScalePivot(self, point, space='object', balance=True, **kwargs):
        space = self._getSpaceArg(space, kwargs)
        return self._setScalePivot(point, space=space, balance=balance)

    @_factories.addApiDocs(_api.MFnTransform, 'scalePivot')
    def getScalePivot(self, space='object', **kwargs):
        space = self._getSpaceArg(space, kwargs)
        return self._getScalePivot(space=space)

    @_factories.addApiDocs(_api.MFnTransform, 'setScalePivotTranslation')
    def setScalePivotTranslation(self, vector, space='object', **kwargs):
        space = self._getSpaceArg(space, kwargs)
        return self._setScalePivotTranslation(vector, space=space)

    @_factories.addApiDocs(_api.MFnTransform, 'scalePivotTranslation')
    def getScalePivotTranslation(self, space='object', **kwargs):
        space = self._getSpaceArg(space, kwargs)
        return self._getScalePivotTranslation(space=space)

    @_factories.queryflag('xform', 'pivots')
    def getPivots(self, **kwargs):
        res = cmds.xform(self, **kwargs)
        return (datatypes.Vector(res[:3]), datatypes.Vector(res[3:]))

    @_factories.queryflag('xform', 'rotateAxis')
    def getRotateAxis(self, **kwargs):
        return datatypes.Vector(cmds.xform(self, **kwargs))

    # getRotateAxis / setRotateAxis wrap the mel commands, and get/set
    # euler rotations.  get/setRotateOrientation wrap api, and get/set a
    # quaternion - which is what is actually stored under the hood (even though
    # the external "interface" is an euler rotation attribute

    @_factories.addApiDocs(_api.MFnTransform, 'rotateOrientation')
    def getRotateOrientation(self, space='transform'):
        # space is currently ignored, and transform is always used, according to
        # docs; however, allow the kwarg for backwards compat
        return self._getRotateOrientation(space=space)

    @_factories.addApiDocs(_api.MFnTransform, 'setRotateOrientation')
    def setRotateOrientation(self, quat, space='transform', balance=True):
        # space is currently ignored, and transform is always used, according to
        # docs; however, allow the kwarg for backwards compat
        return self._setRotateOrientation(quat, space=space,
            balance=balance)

#    @_factories.queryflag('xform','shear')
#    def getShearOld( self, **kwargs ):
#        return datatypes.Vector( cmds.xform( self, **kwargs ) )

    @_factories.queryflag('xform', 'matrix')
    def getMatrix(self, **kwargs):
        return datatypes.Matrix(cmds.xform(self, **kwargs))

    # TODO: create API equivalent of `xform -boundingBoxInvisible` so we can replace this with _api.
    def getBoundingBox(self, invisible=False, space='object'):
        # type: (Any, Any) -> datatypes.BoundingBox
        """xform -boundingBox and xform -boundingBoxInvisible

        Returns
        -------
        datatypes.BoundingBox
        """
        kwargs = {'query': True}
        if invisible:
            kwargs['boundingBoxInvisible'] = True
        else:
            kwargs['boundingBox'] = True
        if space == 'object':
            kwargs['objectSpace'] = True
        elif space == 'world':
            kwargs['worldSpace'] = True
        else:
            raise ValueError('unknown space %r' % space)

        res = cmds.xform(self, **kwargs)
        # return ( datatypes.Vector(res[:3]), datatypes.Vector(res[3:]) )
        return datatypes.BoundingBox(res[:3], res[3:])

    def getBoundingBoxMin(self, invisible=False, space='object'):
        # type: (Any, Any) -> datatypes.Vector
        """
        Returns
        -------
        datatypes.Vector
        """
        return self.getBoundingBox(invisible, space)[0]
        # return self.getBoundingBox(invisible).min()

    def getBoundingBoxMax(self, invisible=False, space='object'):
        # type: (Any, Any) -> datatypes.Vector
        """
        Returns
        -------
        datatypes.Vector
        """
        return self.getBoundingBox(invisible, space)[1]
# ------ Do not edit below this line --------
    __apicls__ = _api.MFnTransform
    __melcmd__ = staticmethod(general.xform)
    __melcmd_isinfo__ = True
    __melcmdname__ = 'xform'
    __melnode__ = 'transform'
    __slots__ = ()
    LimitType = Enum('LimitType', [('scaleMinX', 0), ('kScaleMinX', 0), ('scaleMaxX', 1), ('kScaleMaxX', 1), ('scaleMinY', 2), ('kScaleMinY', 2), ('scaleMaxY', 3), ('kScaleMaxY', 3), ('scaleMinZ', 4), ('kScaleMinZ', 4), ('scaleMaxZ', 5), ('kScaleMaxZ', 5), ('shearMinXY', 6), ('kShearMinXY', 6), ('shearMaxXY', 7), ('kShearMaxXY', 7), ('shearMinXZ', 8), ('kShearMinXZ', 8), ('shearMaxXZ', 9), ('kShearMaxXZ', 9), ('shearMinYZ', 10), ('kShearMinYZ', 10), ('shearMaxYZ', 11), ('kShearMaxYZ', 11), ('rotateMinX', 12), ('kRotateMinX', 12), ('rotateMaxX', 13), ('kRotateMaxX', 13), ('rotateMinY', 14), ('kRotateMinY', 14), ('rotateMaxY', 15), ('kRotateMaxY', 15), ('rotateMinZ', 16), ('kRotateMinZ', 16), ('rotateMaxZ', 17), ('kRotateMaxZ', 17), ('translateMinX', 18), ('kTranslateMinX', 18), ('translateMaxX', 19), ('kTranslateMaxX', 19), ('translateMinY', 20), ('kTranslateMinY', 20), ('translateMaxY', 21), ('kTranslateMaxY', 21), ('translateMinZ', 22), ('kTranslateMinZ', 22), ('translateMaxZ', 23), ('kTranslateMaxZ', 23)], multiKeys=True)

    @_f.addApiDocs(_api.MFnTransform, 'rotateOrientation')
    def _getRotateOrientation(self, space):
        # type: (datatypes.Space.Space) -> datatypes.Quaternion
        do, final_do, outTypes = _f.getDoArgs([space], [('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnTransform, 'rotateOrientation', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MQuaternion', None)
        return res

    @_f.addApiDocs(_api.MFnTransform, 'rotatePivot')
    def _getRotatePivot(self, space):
        # type: (datatypes.Space.Space) -> datatypes.Point
        do, final_do, outTypes = _f.getDoArgs([space], [('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnTransform, 'rotatePivot', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MPoint', None)
        return res

    @_f.addApiDocs(_api.MFnTransform, 'rotatePivotTranslation')
    def _getRotatePivotTranslation(self, space):
        # type: (datatypes.Space.Space) -> datatypes.Vector
        do, final_do, outTypes = _f.getDoArgs([space], [('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnTransform, 'rotatePivotTranslation', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MVector', 'linear')
        return res

    @_f.addApiDocs(_api.MFnTransform, 'getRotation')
    def _getRotation(self, space='transform'):
        # type: (datatypes.Space.Space) -> datatypes.Quaternion
        do, final_do, outTypes = _f.getDoArgs([space], [('quaternion', 'MQuaternion', 'out', None), ('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnTransform, 'getRotation', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnTransform, 'scalePivot')
    def _getScalePivot(self, space):
        # type: (datatypes.Space.Space) -> datatypes.Point
        do, final_do, outTypes = _f.getDoArgs([space], [('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnTransform, 'scalePivot', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MPoint', None)
        return res

    @_f.addApiDocs(_api.MFnTransform, 'scalePivotTranslation')
    def _getScalePivotTranslation(self, space):
        # type: (datatypes.Space.Space) -> datatypes.Vector
        do, final_do, outTypes = _f.getDoArgs([space], [('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnTransform, 'scalePivotTranslation', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MVector', 'linear')
        return res

    @_f.addApiDocs(_api.MFnTransform, 'getTranslation')
    def _getTranslation(self, space):
        # type: (datatypes.Space.Space) -> datatypes.Vector
        do, final_do, outTypes = _f.getDoArgs([space], [('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnTransform, 'getTranslation', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MVector', 'linear')
        return res

    @_f.addApiDocs(_api.MFnTransform, 'rotateBy')
    def _rotateBy(self, rotation, space='transform'):
        # type: (datatypes.EulerRotation, datatypes.Space.Space) -> None
        do, final_do, outTypes = _f.getDoArgs([rotation, space], [('rotation', 'MEulerRotation', 'in', None), ('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnTransform, 'rotateBy', final_do)
        return res

    @_f.addApiDocs(_api.MFnTransform, 'rotateByQuaternion')
    def _rotateByQuaternion(self, x, y, z, w, space='transform'):
        # type: (float, float, float, float, datatypes.Space.Space) -> None
        do, final_do, outTypes = _f.getDoArgs([x, y, z, w, space], [('x', 'double', 'in', None), ('y', 'double', 'in', None), ('z', 'double', 'in', None), ('w', 'double', 'in', None), ('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnTransform, 'rotateByQuaternion', final_do)
        return res

    @_f.addApiDocs(_api.MFnTransform, 'scaleBy')
    def _scaleBy(self, scale):
        # type: (Tuple[float, float, float]) -> None
        do, final_do, outTypes = _f.getDoArgs([scale], [('scale', 'double__array3', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnTransform, 'scaleBy', final_do)
        return res

    @_f.addApiDocs(_api.MFnTransform, 'setRotateOrientation')
    def _setRotateOrientation(self, quat, space, balance=True):
        # type: (datatypes.Quaternion, datatypes.Space.Space, bool) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([quat, space, balance], [('quat', 'MQuaternion', 'in', None), ('space', ('MSpace', 'Space'), 'in', None), ('balance', 'bool', 'in', None)], self._getRotateOrientation, self._setRotateOrientation, ['space'])
        res = _f.getProxyResult(self, _api.MFnTransform, 'setRotateOrientation', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnTransform, 'setRotatePivot')
    def _setRotatePivot(self, point, space, balance=True):
        # type: (datatypes.Point, datatypes.Space.Space, bool) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([point, space, balance], [('point', 'MPoint', 'in', None), ('space', ('MSpace', 'Space'), 'in', None), ('balance', 'bool', 'in', None)], self._getRotatePivot, self._setRotatePivot, ['space'])
        res = _f.getProxyResult(self, _api.MFnTransform, 'setRotatePivot', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnTransform, 'setRotatePivotTranslation')
    def _setRotatePivotTranslation(self, vector, space):
        # type: (datatypes.Vector, datatypes.Space.Space) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([vector, space], [('vector', 'MVector', 'in', 'linear'), ('space', ('MSpace', 'Space'), 'in', None)], self._getRotatePivotTranslation, self._setRotatePivotTranslation, ['space'])
        res = _f.getProxyResult(self, _api.MFnTransform, 'setRotatePivotTranslation', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnTransform, 'setRotation')
    def _setRotation(self, quaternion, space='transform'):
        # type: (datatypes.Quaternion, datatypes.Space.Space) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([quaternion, space], [('quaternion', 'MQuaternion', 'in', None), ('space', ('MSpace', 'Space'), 'in', None)], self._getRotation, self._setRotation, ['space'])
        res = _f.getProxyResult(self, _api.MFnTransform, 'setRotation', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnTransform, 'setScale')
    def _setScale(self, scale):
        # type: (Tuple[float, float, float]) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([scale], [('scale', 'double__array3', 'in', None)], self.getScale, self._setScale, [])
        res = _f.getProxyResult(self, _api.MFnTransform, 'setScale', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnTransform, 'setScalePivot')
    def _setScalePivot(self, point, space, balance=True):
        # type: (datatypes.Point, datatypes.Space.Space, bool) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([point, space, balance], [('point', 'MPoint', 'in', None), ('space', ('MSpace', 'Space'), 'in', None), ('balance', 'bool', 'in', None)], self._getScalePivot, self._setScalePivot, ['space'])
        res = _f.getProxyResult(self, _api.MFnTransform, 'setScalePivot', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnTransform, 'setScalePivotTranslation')
    def _setScalePivotTranslation(self, vector, space):
        # type: (datatypes.Vector, datatypes.Space.Space) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([vector, space], [('vector', 'MVector', 'in', 'linear'), ('space', ('MSpace', 'Space'), 'in', None)], self._getScalePivotTranslation, self._setScalePivotTranslation, ['space'])
        res = _f.getProxyResult(self, _api.MFnTransform, 'setScalePivotTranslation', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnTransform, 'setShear')
    def _setShear(self, shear):
        # type: (Tuple[float, float, float]) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([shear], [('shear', 'double__array3', 'in', None)], self.getShear, self._setShear, [])
        res = _f.getProxyResult(self, _api.MFnTransform, 'setShear', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnTransform, 'setTranslation')
    def _setTranslation(self, vector, space):
        # type: (datatypes.Vector, datatypes.Space.Space) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([vector, space], [('vector', 'MVector', 'in', 'linear'), ('space', ('MSpace', 'Space'), 'in', None)], self._getTranslation, self._setTranslation, ['space'])
        res = _f.getProxyResult(self, _api.MFnTransform, 'setTranslation', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnTransform, 'shearBy')
    def _shearBy(self, shear):
        # type: (Tuple[float, float, float]) -> None
        do, final_do, outTypes = _f.getDoArgs([shear], [('shear', 'double__array3', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnTransform, 'shearBy', final_do)
        return res

    @_f.addApiDocs(_api.MFnTransform, 'translateBy')
    def _translateBy(self, vector, space):
        # type: (datatypes.Vector, datatypes.Space.Space) -> None
        do, final_do, outTypes = _f.getDoArgs([vector, space], [('vector', 'MVector', 'in', 'linear'), ('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnTransform, 'translateBy', final_do)
        return res

    @classmethod
    @_f.addApiDocs(_api.MFnTransform, 'balanceTransformation')
    def balanceTransformation(self, mtx, rotateAxis='MQuaternion::identity', jointOrient='MQuaternion::identity', segmentScaleCompensate=True, inverseScale='MVector::one', rotateOrder='XYZ'):
        # type: (datatypes.TransformationMatrix, datatypes.Quaternion, datatypes.Quaternion, bool, datatypes.Vector, datatypes.EulerRotation.RotationOrder) -> Tuple[datatypes.Vector, datatypes.EulerRotation, datatypes.Vector]
        do, final_do, outTypes = _f.getDoArgs([mtx, rotateAxis, jointOrient, segmentScaleCompensate, inverseScale, rotateOrder], [('localTranslate', 'MVector', 'out', None), ('localRotate', 'MEulerRotation', 'out', None), ('localScale', 'MVector', 'out', None), ('mtx', 'MTransformationMatrix', 'in', None), ('rotateAxis', 'MQuaternion', 'in', None), ('jointOrient', 'MQuaternion', 'in', None), ('segmentScaleCompensate', 'bool', 'in', None), ('inverseScale', 'MVector', 'in', None), ('rotateOrder', ('MEulerRotation', 'RotationOrder'), 'in', None)])
        res = _api.MFnTransform.balanceTransformation(*final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addMelDocs('xform', 'centerPivots')
    def centerPivots(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, general.xform, kwargs, 'centerPivots', val)

    @_f.addMelDocs('xform', 'centerPivotsOnComponents')
    def centerPivotsOnComponents(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, general.xform, kwargs, 'centerPivotsOnComponents', val)

    @_f.addApiDocs(_api.MFnTransform, 'clearRestPosition')
    def clearRestPosition(self):
        # type: () -> None
        res = _f.getProxyResult(self, _api.MFnTransform, 'clearRestPosition')
        return res

    @_f.addMelDocs('xform', 'deletePriorHistory')
    def deletePriorHistory(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, general.xform, kwargs, 'deletePriorHistory', val)

    @_f.addApiDocs(_api.MFnTransform, 'enableLimit')
    def enableLimit(self, type, flag):
        # type: (Transform.LimitType, bool) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([type, flag], [('type', ('MFnTransform', 'LimitType'), 'in', None), ('flag', 'bool', 'in', None)], self.isLimited, self.enableLimit, ['type'])
        res = _f.getProxyResult(self, _api.MFnTransform, 'enableLimit', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res
    setLimited = enableLimit

    @_f.addMelDocs('xform', 'boundingBoxInvisible')
    def getBoundingBoxInvisible(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, general.xform, kwargs, 'boundingBoxInvisible')
        return res

    @_f.addApiDocs(_api.MFnTransform, 'restPosition')
    def getRestPosition(self):
        # type: () -> datatypes.TransformationMatrix
        res = _f.getProxyResult(self, _api.MFnTransform, 'restPosition')
        return _f.ApiArgUtil._castResult(self, res, 'MTransformationMatrix', None)

    @_f.addApiDocs(_api.MFnTransform, 'rotationOrder')
    def getRotationOrder(self):
        # type: () -> datatypes.TransformationMatrix.RotationOrder
        res = _f.getProxyResult(self, _api.MFnTransform, 'rotationOrder')
        return _f.ApiArgUtil._castResult(self, res, ('MTransformationMatrix', 'RotationOrder'), None)

    @_f.addApiDocs(_api.MFnTransform, 'getScale')
    def getScale(self):
        # type: () -> Tuple[float, float, float]
        do, final_do, outTypes = _f.getDoArgs([], [('scale', 'double__array3', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnTransform, 'getScale', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnTransform, 'getShear')
    def getShear(self):
        # type: () -> Tuple[float, float, float]
        do, final_do, outTypes = _f.getDoArgs([], [('shear', 'double__array3', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnTransform, 'getShear', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnTransform, 'isLimited')
    def isLimited(self, type):
        # type: (Transform.LimitType) -> bool
        do, final_do, outTypes = _f.getDoArgs([type], [('type', ('MFnTransform', 'LimitType'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnTransform, 'isLimited', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res

    @_f.addApiDocs(_api.MFnTransform, 'limitValue')
    def limitValue(self, type):
        # type: (Transform.LimitType) -> float
        do, final_do, outTypes = _f.getDoArgs([type], [('type', ('MFnTransform', 'LimitType'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnTransform, 'limitValue', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'double', None)
        return res
    getLimit = limitValue

    @_f.addMelDocs('xform', 'preserveUV')
    def preserveUV(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, general.xform, kwargs, 'preserveUV', val)

    @_f.addMelDocs('xform', 'reflection')
    def reflection(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, general.xform, kwargs, 'reflection', val)

    @_f.addMelDocs('xform', 'reflectionAboutBBox')
    def reflectionAboutBBox(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, general.xform, kwargs, 'reflectionAboutBBox', val)

    @_f.addMelDocs('xform', 'reflectionAboutOrigin')
    def reflectionAboutOrigin(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, general.xform, kwargs, 'reflectionAboutOrigin', val)

    @_f.addMelDocs('xform', 'reflectionAboutX')
    def reflectionAboutX(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, general.xform, kwargs, 'reflectionAboutX', val)

    @_f.addMelDocs('xform', 'reflectionAboutY')
    def reflectionAboutY(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, general.xform, kwargs, 'reflectionAboutY', val)

    @_f.addMelDocs('xform', 'reflectionAboutZ')
    def reflectionAboutZ(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, general.xform, kwargs, 'reflectionAboutZ', val)

    @_f.addMelDocs('xform', 'reflectionTolerance')
    def reflectionTolerance(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, general.xform, kwargs, 'reflectionTolerance', val)

    @_f.addApiDocs(_api.MFnTransform, 'resetFromRestPosition')
    def resetFromRestPosition(self):
        # type: () -> None
        res = _f.getProxyResult(self, _api.MFnTransform, 'resetFromRestPosition')
        return res

    @_f.addApiDocs(_api.MFnTransform, 'resetTransformation')
    def resetTransformation(self, transform):
        # type: (datatypes.Matrix) -> None
        do, final_do, outTypes = _f.getDoArgs([transform], [('transform', 'MMatrix', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnTransform, 'resetTransformation', final_do)
        return res

    @_f.addApiDocs(_api.MFnTransform, 'set')
    def set(self, transform):
        # type: (datatypes.TransformationMatrix) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([transform], [('transform', 'MTransformationMatrix', 'in', None)], self.transformation, self.set, [])
        res = _f.getProxyResult(self, _api.MFnTransform, 'set', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res
    setTransformation = set

    @_f.addApiDocs(_api.MFnTransform, 'setLimit')
    def setLimit(self, type, value):
        # type: (Transform.LimitType, float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([type, value], [('type', ('MFnTransform', 'LimitType'), 'in', None), ('value', 'double', 'in', 'linear')], self.limitValue, self.setLimit, ['type'])
        res = _f.getProxyResult(self, _api.MFnTransform, 'setLimit', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addMelDocs('xform', 'pivots')
    def setPivots(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, general.xform, kwargs, 'pivots', val)

    @_f.addApiDocs(_api.MFnTransform, 'setRestPosition')
    def setRestPosition(self, matrix):
        # type: (datatypes.TransformationMatrix) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([matrix], [('matrix', 'MTransformationMatrix', 'in', None)], self.getRestPosition, self.setRestPosition, [])
        res = _f.getProxyResult(self, _api.MFnTransform, 'setRestPosition', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addMelDocs('xform', 'rotateAxis')
    def setRotateAxis(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, general.xform, kwargs, 'rotateAxis', val)

    @_f.addApiDocs(_api.MFnTransform, 'setRotationOrder')
    def setRotationOrder(self, order, reorder):
        # type: (datatypes.TransformationMatrix.RotationOrder, bool) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([order, reorder], [('order', ('MTransformationMatrix', 'RotationOrder'), 'in', None), ('reorder', 'bool', 'in', None)], self.getRotationOrder, self.setRotationOrder, [])
        res = _f.getProxyResult(self, _api.MFnTransform, 'setRotationOrder', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnTransform, 'transformation')
    def transformation(self):
        # type: () -> datatypes.TransformationMatrix
        res = _f.getProxyResult(self, _api.MFnTransform, 'transformation')
        return _f.ApiArgUtil._castResult(self, res, 'MTransformationMatrix', None)
    getTransformation = transformation

    @_f.addMelDocs('xform', 'zeroTransformPivots')
    def zeroTransformPivots(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, general.xform, kwargs, 'zeroTransformPivots', val)
# ------ Do not edit above this line --------
#    def centerPivots(self, **kwargs):
#        """xform -centerPivots"""
#        kwargs['centerPivots'] = True
#        cmds.xform( self, **kwargs )
#
#    def zeroTransformPivots(self, **kwargs):
#        """xform -zeroTransformPivots"""
#        kwargs['zeroTransformPivots'] = True
#        cmds.xform( self, **kwargs )


class Joint(Transform):
    if not _factories.building:
        connect = animation.connectJoint
        disconnect = animation.disconnectJoint
        insert = animation.insertJoint
# ------ Do not edit below this line --------
    __apicls__ = _api.MFnIkJoint
    __melcmd__ = staticmethod(animation.joint)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'joint'
    __melnode__ = 'joint'
    __slots__ = ()
    Axis = Enum('Axis', [('XAxis', 0), ('kXAxis', 0), ('YAxis', 1), ('kYAxis', 1), ('ZAxis', 2), ('kZAxis', 2), ('none', 3), ('kNone', 3)], multiKeys=True)

    @_f.addMelDocs('joint', 'assumePreferredAngles')
    def assumePreferredAngles(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.joint, kwargs, 'assumePreferredAngles', val)

    @_f.addMelDocs('joint', 'children')
    def children(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.joint, kwargs, 'children', val)

    @_f.addMelDocs('joint', 'component')
    def component(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.joint, kwargs, 'component', val)

    @_f.addMelDocs('joint', 'angleX')
    def getAngleX(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, animation.joint, kwargs, 'angleX')
        return res

    @_f.addMelDocs('joint', 'angleY')
    def getAngleY(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, animation.joint, kwargs, 'angleY')
        return res

    @_f.addMelDocs('joint', 'angleZ')
    def getAngleZ(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, animation.joint, kwargs, 'angleZ')
        return res

    @_f.addMelDocs('joint', 'degreeOfFreedom')
    def getDegreeOfFreedom(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.joint, kwargs, 'degreeOfFreedom')
        return res

    @_f.addApiDocs(_api.MFnIkJoint, 'getDegreesOfFreedom')
    def getDegreesOfFreedom(self):
        # type: () -> Tuple[bool, bool, bool]
        do, final_do, outTypes = _f.getDoArgs([], [('freeInX', 'bool', 'out', None), ('freeInY', 'bool', 'out', None), ('freeInZ', 'bool', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnIkJoint, 'getDegreesOfFreedom', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addMelDocs('joint', 'exists')
    def getExists(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.joint, kwargs, 'exists')
        return res

    @_f.addMelDocs('joint', 'limitSwitchX')
    def getLimitSwitchX(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.joint, kwargs, 'limitSwitchX')
        return res

    @_f.addMelDocs('joint', 'limitSwitchY')
    def getLimitSwitchY(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.joint, kwargs, 'limitSwitchY')
        return res

    @_f.addMelDocs('joint', 'limitSwitchZ')
    def getLimitSwitchZ(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.joint, kwargs, 'limitSwitchZ')
        return res

    @_f.addMelDocs('joint', 'limitX')
    def getLimitX(self, **kwargs):
        # type: (...) -> Tuple[float, float]
        res = _f.asQuery(self, animation.joint, kwargs, 'limitX')
        return res

    @_f.addMelDocs('joint', 'limitY')
    def getLimitY(self, **kwargs):
        # type: (...) -> Tuple[float, float]
        res = _f.asQuery(self, animation.joint, kwargs, 'limitY')
        return res

    @_f.addMelDocs('joint', 'limitZ')
    def getLimitZ(self, **kwargs):
        # type: (...) -> Tuple[float, float]
        res = _f.asQuery(self, animation.joint, kwargs, 'limitZ')
        return res

    @_f.addApiDocs(_api.MFnIkJoint, 'maxRotateDampXRange')
    def getMaxRotateDampXRange(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnIkJoint, 'maxRotateDampXRange')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addApiDocs(_api.MFnIkJoint, 'maxRotateDampXStrength')
    def getMaxRotateDampXStrength(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnIkJoint, 'maxRotateDampXStrength')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addApiDocs(_api.MFnIkJoint, 'maxRotateDampYRange')
    def getMaxRotateDampYRange(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnIkJoint, 'maxRotateDampYRange')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addApiDocs(_api.MFnIkJoint, 'maxRotateDampYStrength')
    def getMaxRotateDampYStrength(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnIkJoint, 'maxRotateDampYStrength')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addApiDocs(_api.MFnIkJoint, 'maxRotateDampZRange')
    def getMaxRotateDampZRange(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnIkJoint, 'maxRotateDampZRange')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addApiDocs(_api.MFnIkJoint, 'maxRotateDampZStrength')
    def getMaxRotateDampZStrength(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnIkJoint, 'maxRotateDampZStrength')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addApiDocs(_api.MFnIkJoint, 'minRotateDampXRange')
    def getMinRotateDampXRange(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnIkJoint, 'minRotateDampXRange')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addApiDocs(_api.MFnIkJoint, 'minRotateDampXStrength')
    def getMinRotateDampXStrength(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnIkJoint, 'minRotateDampXStrength')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addApiDocs(_api.MFnIkJoint, 'minRotateDampYRange')
    def getMinRotateDampYRange(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnIkJoint, 'minRotateDampYRange')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addApiDocs(_api.MFnIkJoint, 'minRotateDampYStrength')
    def getMinRotateDampYStrength(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnIkJoint, 'minRotateDampYStrength')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addApiDocs(_api.MFnIkJoint, 'minRotateDampZRange')
    def getMinRotateDampZRange(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnIkJoint, 'minRotateDampZRange')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addApiDocs(_api.MFnIkJoint, 'minRotateDampZStrength')
    def getMinRotateDampZStrength(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnIkJoint, 'minRotateDampZStrength')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addApiDocs(_api.MFnIkJoint, 'getOrientation')
    def getOrientation(self):
        # type: () -> datatypes.Quaternion
        do, final_do, outTypes = _f.getDoArgs([], [('quaternion', 'MQuaternion', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnIkJoint, 'getOrientation', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.maya_deprecated
    def getPreferedAngle(self):
        # type: () -> Tuple[float, float, float]
        do, final_do, outTypes = _f.getDoArgs([], [('rotation', 'double__array3', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnIkJoint, 'getPreferedAngle', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnIkJoint, 'getPreferredAngle')
    def getPreferredAngle(self):
        # type: () -> Tuple[float, float, float]
        do, final_do, outTypes = _f.getDoArgs([], [('rotation', 'double__array3', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnIkJoint, 'getPreferredAngle', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addMelDocs('joint', 'radius')
    def getRadius(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, animation.joint, kwargs, 'radius')
        return res

    @_f.addMelDocs('joint', 'scaleCompensate')
    def getScaleCompensate(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.joint, kwargs, 'scaleCompensate')
        return res

    @_f.addApiDocs(_api.MFnIkJoint, 'getScaleOrientation')
    def getScaleOrientation(self):
        # type: () -> datatypes.Quaternion
        do, final_do, outTypes = _f.getDoArgs([], [('quaternion', 'MQuaternion', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnIkJoint, 'getScaleOrientation', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnIkJoint, 'getSegmentScale')
    def getSegmentScale(self):
        # type: () -> Tuple[float, float, float]
        do, final_do, outTypes = _f.getDoArgs([], [('scale', 'double__array3', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnIkJoint, 'getSegmentScale', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnIkJoint, 'getStiffness')
    def getStiffness(self):
        # type: () -> Tuple[float, float, float]
        do, final_do, outTypes = _f.getDoArgs([], [('stiffness', 'double__array3', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnIkJoint, 'getStiffness', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addMelDocs('joint', 'stiffnessX')
    def getStiffnessX(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, animation.joint, kwargs, 'stiffnessX')
        return res

    @_f.addMelDocs('joint', 'stiffnessY')
    def getStiffnessY(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, animation.joint, kwargs, 'stiffnessY')
        return res

    @_f.addMelDocs('joint', 'stiffnessZ')
    def getStiffnessZ(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, animation.joint, kwargs, 'stiffnessZ')
        return res

    @_f.addApiDocs(_api.MFnIkJoint, 'hikJointName')
    def hikJointName(self):
        # type: () -> str
        res = _f.getProxyResult(self, _api.MFnIkJoint, 'hikJointName')
        return _f.ApiArgUtil._castResult(self, res, 'MString', None)

    @_f.addMelDocs('joint', 'orientJoint')
    def orientJoint(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.joint, kwargs, 'orientJoint', val)

    @_f.addMelDocs('joint', 'secondaryAxisOrient')
    def secondaryAxisOrient(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.joint, kwargs, 'secondaryAxisOrient', val)

    @_f.addMelDocs('joint', 'angleX')
    def setAngleX(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, animation.joint, kwargs, 'angleX', val)

    @_f.addMelDocs('joint', 'angleY')
    def setAngleY(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, animation.joint, kwargs, 'angleY', val)

    @_f.addMelDocs('joint', 'angleZ')
    def setAngleZ(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, animation.joint, kwargs, 'angleZ', val)

    @_f.addMelDocs('joint', 'degreeOfFreedom')
    def setDegreeOfFreedom(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.joint, kwargs, 'degreeOfFreedom', val)

    @_f.addApiDocs(_api.MFnIkJoint, 'setDegreesOfFreedom')
    def setDegreesOfFreedom(self, freeInX, freeInY, freeInZ):
        # type: (bool, bool, bool) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([freeInX, freeInY, freeInZ], [('freeInX', 'bool', 'in', None), ('freeInY', 'bool', 'in', None), ('freeInZ', 'bool', 'in', None)], self.getDegreesOfFreedom, self.setDegreesOfFreedom, [])
        res = _f.getProxyResult(self, _api.MFnIkJoint, 'setDegreesOfFreedom', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addMelDocs('joint', 'limitSwitchX')
    def setLimitSwitchX(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.joint, kwargs, 'limitSwitchX', val)

    @_f.addMelDocs('joint', 'limitSwitchY')
    def setLimitSwitchY(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.joint, kwargs, 'limitSwitchY', val)

    @_f.addMelDocs('joint', 'limitSwitchZ')
    def setLimitSwitchZ(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.joint, kwargs, 'limitSwitchZ', val)

    @_f.addMelDocs('joint', 'limitX')
    def setLimitX(self, val=True, **kwargs):
        # type: (Tuple[float, float], **Any) -> None
        return _f.asEdit(self, animation.joint, kwargs, 'limitX', val)

    @_f.addMelDocs('joint', 'limitY')
    def setLimitY(self, val=True, **kwargs):
        # type: (Tuple[float, float], **Any) -> None
        return _f.asEdit(self, animation.joint, kwargs, 'limitY', val)

    @_f.addMelDocs('joint', 'limitZ')
    def setLimitZ(self, val=True, **kwargs):
        # type: (Tuple[float, float], **Any) -> None
        return _f.asEdit(self, animation.joint, kwargs, 'limitZ', val)

    @_f.addApiDocs(_api.MFnIkJoint, 'setMaxRotateDampXRange')
    def setMaxRotateDampXRange(self, angle):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([angle], [('angle', 'double', 'in', None)], self.getMaxRotateDampXRange, self.setMaxRotateDampXRange, [])
        res = _f.getProxyResult(self, _api.MFnIkJoint, 'setMaxRotateDampXRange', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnIkJoint, 'setMaxRotateDampXStrength')
    def setMaxRotateDampXStrength(self, angle):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([angle], [('angle', 'double', 'in', None)], self.getMaxRotateDampXStrength, self.setMaxRotateDampXStrength, [])
        res = _f.getProxyResult(self, _api.MFnIkJoint, 'setMaxRotateDampXStrength', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnIkJoint, 'setMaxRotateDampYRange')
    def setMaxRotateDampYRange(self, angle):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([angle], [('angle', 'double', 'in', None)], self.getMaxRotateDampYRange, self.setMaxRotateDampYRange, [])
        res = _f.getProxyResult(self, _api.MFnIkJoint, 'setMaxRotateDampYRange', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnIkJoint, 'setMaxRotateDampYStrength')
    def setMaxRotateDampYStrength(self, angle):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([angle], [('angle', 'double', 'in', None)], self.getMaxRotateDampYStrength, self.setMaxRotateDampYStrength, [])
        res = _f.getProxyResult(self, _api.MFnIkJoint, 'setMaxRotateDampYStrength', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnIkJoint, 'setMaxRotateDampZRange')
    def setMaxRotateDampZRange(self, angle):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([angle], [('angle', 'double', 'in', None)], self.getMaxRotateDampZRange, self.setMaxRotateDampZRange, [])
        res = _f.getProxyResult(self, _api.MFnIkJoint, 'setMaxRotateDampZRange', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnIkJoint, 'setMaxRotateDampZStrength')
    def setMaxRotateDampZStrength(self, angle):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([angle], [('angle', 'double', 'in', None)], self.getMaxRotateDampZStrength, self.setMaxRotateDampZStrength, [])
        res = _f.getProxyResult(self, _api.MFnIkJoint, 'setMaxRotateDampZStrength', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnIkJoint, 'setMinRotateDampXRange')
    def setMinRotateDampXRange(self, angle):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([angle], [('angle', 'double', 'in', None)], self.getMinRotateDampXRange, self.setMinRotateDampXRange, [])
        res = _f.getProxyResult(self, _api.MFnIkJoint, 'setMinRotateDampXRange', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnIkJoint, 'setMinRotateDampXStrength')
    def setMinRotateDampXStrength(self, angle):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([angle], [('angle', 'double', 'in', None)], self.getMinRotateDampXStrength, self.setMinRotateDampXStrength, [])
        res = _f.getProxyResult(self, _api.MFnIkJoint, 'setMinRotateDampXStrength', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnIkJoint, 'setMinRotateDampYRange')
    def setMinRotateDampYRange(self, angle):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([angle], [('angle', 'double', 'in', None)], self.getMinRotateDampYRange, self.setMinRotateDampYRange, [])
        res = _f.getProxyResult(self, _api.MFnIkJoint, 'setMinRotateDampYRange', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnIkJoint, 'setMinRotateDampYStrength')
    def setMinRotateDampYStrength(self, angle):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([angle], [('angle', 'double', 'in', None)], self.getMinRotateDampYStrength, self.setMinRotateDampYStrength, [])
        res = _f.getProxyResult(self, _api.MFnIkJoint, 'setMinRotateDampYStrength', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnIkJoint, 'setMinRotateDampZRange')
    def setMinRotateDampZRange(self, angle):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([angle], [('angle', 'double', 'in', None)], self.getMinRotateDampZRange, self.setMinRotateDampZRange, [])
        res = _f.getProxyResult(self, _api.MFnIkJoint, 'setMinRotateDampZRange', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnIkJoint, 'setMinRotateDampZStrength')
    def setMinRotateDampZStrength(self, angle):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([angle], [('angle', 'double', 'in', None)], self.getMinRotateDampZStrength, self.setMinRotateDampZStrength, [])
        res = _f.getProxyResult(self, _api.MFnIkJoint, 'setMinRotateDampZStrength', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnIkJoint, 'setOrientation')
    def setOrientation(self, quaternion):
        # type: (datatypes.Quaternion) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([quaternion], [('quaternion', 'MQuaternion', 'in', None)], self.getOrientation, self.setOrientation, [])
        res = _f.getProxyResult(self, _api.MFnIkJoint, 'setOrientation', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.maya_deprecated
    def setPreferedAngle(self, rotation):
        # type: (Tuple[float, float, float]) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([rotation], [('rotation', 'double__array3', 'in', None)], self.getPreferedAngle, self.setPreferedAngle, [])
        res = _f.getProxyResult(self, _api.MFnIkJoint, 'setPreferedAngle', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnIkJoint, 'setPreferredAngle')
    def setPreferredAngle(self, rotation):
        # type: (Tuple[float, float, float]) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([rotation], [('rotation', 'double__array3', 'in', None)], self.getPreferredAngle, self.setPreferredAngle, [])
        res = _f.getProxyResult(self, _api.MFnIkJoint, 'setPreferredAngle', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addMelDocs('joint', 'setPreferredAngles')
    def setPreferredAngles(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.joint, kwargs, 'setPreferredAngles', val)

    @_f.addMelDocs('joint', 'radius')
    def setRadius(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, animation.joint, kwargs, 'radius', val)

    @_f.addMelDocs('joint', 'scaleCompensate')
    def setScaleCompensate(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.joint, kwargs, 'scaleCompensate', val)

    @_f.addApiDocs(_api.MFnIkJoint, 'setScaleOrientation')
    def setScaleOrientation(self, quaternion):
        # type: (datatypes.Quaternion) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([quaternion], [('quaternion', 'MQuaternion', 'in', None)], self.getScaleOrientation, self.setScaleOrientation, [])
        res = _f.getProxyResult(self, _api.MFnIkJoint, 'setScaleOrientation', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnIkJoint, 'setSegmentScale')
    def setSegmentScale(self, scale):
        # type: (Tuple[float, float, float]) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([scale], [('scale', 'double__array3', 'in', None)], self.getSegmentScale, self.setSegmentScale, [])
        res = _f.getProxyResult(self, _api.MFnIkJoint, 'setSegmentScale', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnIkJoint, 'setStiffness')
    def setStiffness(self, stiffness):
        # type: (Tuple[float, float, float]) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([stiffness], [('stiffness', 'double__array3', 'in', None)], self.getStiffness, self.setStiffness, [])
        res = _f.getProxyResult(self, _api.MFnIkJoint, 'setStiffness', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addMelDocs('joint', 'stiffnessX')
    def setStiffnessX(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, animation.joint, kwargs, 'stiffnessX', val)

    @_f.addMelDocs('joint', 'stiffnessY')
    def setStiffnessY(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, animation.joint, kwargs, 'stiffnessY', val)

    @_f.addMelDocs('joint', 'stiffnessZ')
    def setStiffnessZ(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, animation.joint, kwargs, 'stiffnessZ', val)

    @_f.addMelDocs('joint', 'symmetry')
    def symmetry(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.joint, kwargs, 'symmetry', val)

    @_f.addMelDocs('joint', 'symmetryAxis')
    def symmetryAxis(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.joint, kwargs, 'symmetryAxis', val)

    @_f.addMelDocs('joint', 'zeroScaleOrient')
    def zeroScaleOrient(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.joint, kwargs, 'zeroScaleOrient', val)
# ------ Do not edit above this line --------


class DynBase(Transform):
    pass
# ------ Do not edit below this line --------
    __melnode__ = 'dynBase'
    __slots__ = ()

    @_f.deprecated('Use getRotation(quaternion=True) instead.')
    def getRotationQuaternion(self, space='transform'):
        # type: (datatypes.Space.Space) -> Tuple[float, float, float, float]
        do, final_do, outTypes = _f.getDoArgs([space], [('x', 'double', 'out', None), ('y', 'double', 'out', None), ('z', 'double', 'out', None), ('w', 'double', 'out', None), ('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnTransform, 'getRotationQuaternion', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use setRotation, and pass in a Quaternion object, instead.')
    def setRotationQuaternion(self, x, y, z, w, space='transform'):
        # type: (float, float, float, float, datatypes.Space.Space) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([x, y, z, w, space], [('x', 'double', 'in', None), ('y', 'double', 'in', None), ('z', 'double', 'in', None), ('w', 'double', 'in', None), ('space', ('MSpace', 'Space'), 'in', None)], self.getRotationQuaternion, self.setRotationQuaternion, ['space'])
        res = _f.getProxyResult(self, _api.MFnTransform, 'setRotationQuaternion', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res
# ------ Do not edit above this line --------


class PointEmitter(DynBase):
    pass
# ------ Do not edit below this line --------
    __melnode__ = 'pointEmitter'
    __slots__ = ()
# ------ Do not edit above this line --------


class FluidEmitter(PointEmitter):
    if not _factories.building:
        fluidVoxelInfo = effects.fluidVoxelInfo
        loadFluid = effects.loadFluid
        resampleFluid = effects.resampleFluid
        saveFluid = effects.saveFluid
        setFluidAttr = effects.setFluidAttr
        getFluidAttr = effects.getFluidAttr
# ------ Do not edit below this line --------
    __melcmd__ = staticmethod(effects.fluidEmitter)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'fluidEmitter'
    __melnode__ = 'fluidEmitter'
    __slots__ = ()

    @_f.addMelDocs('fluidEmitter', 'cycleEmission')
    def getCycleEmission(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, effects.fluidEmitter, kwargs, 'cycleEmission')
        return res

    @_f.addMelDocs('fluidEmitter', 'cycleInterval')
    def getCycleInterval(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, effects.fluidEmitter, kwargs, 'cycleInterval')
        return res

    @_f.addMelDocs('fluidEmitter', 'densityEmissionRate')
    def getDensityEmissionRate(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, effects.fluidEmitter, kwargs, 'densityEmissionRate')
        return res

    @_f.addMelDocs('fluidEmitter', 'fluidDropoff')
    def getFluidDropoff(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, effects.fluidEmitter, kwargs, 'fluidDropoff')
        return res

    @_f.addMelDocs('fluidEmitter', 'fuelEmissionRate')
    def getFuelEmissionRate(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, effects.fluidEmitter, kwargs, 'fuelEmissionRate')
        return res

    @_f.addMelDocs('fluidEmitter', 'heatEmissionRate')
    def getHeatEmissionRate(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, effects.fluidEmitter, kwargs, 'heatEmissionRate')
        return res

    @_f.addMelDocs('fluidEmitter', 'maxDistance')
    def getMaxDistance(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, effects.fluidEmitter, kwargs, 'maxDistance')
        return res

    @_f.addMelDocs('fluidEmitter', 'minDistance')
    def getMinDistance(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, effects.fluidEmitter, kwargs, 'minDistance')
        return res

    @_f.addMelDocs('fluidEmitter', 'position')
    def getPosition(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, effects.fluidEmitter, kwargs, 'position')
        return res

    @_f.addMelDocs('fluidEmitter', 'rate')
    def getRate(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, effects.fluidEmitter, kwargs, 'rate')
        return res

    @_f.addMelDocs('fluidEmitter', 'torusSectionRadius')
    def getTorusSectionRadius(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, effects.fluidEmitter, kwargs, 'torusSectionRadius')
        return res

    @_f.addMelDocs('fluidEmitter', 'type')
    def getType(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, effects.fluidEmitter, kwargs, 'type')
        return res

    @_f.addMelDocs('fluidEmitter', 'volumeOffset')
    def getVolumeOffset(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, effects.fluidEmitter, kwargs, 'volumeOffset')
        return res

    @_f.addMelDocs('fluidEmitter', 'volumeShape')
    def getVolumeShape(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, effects.fluidEmitter, kwargs, 'volumeShape')
        return res

    @_f.addMelDocs('fluidEmitter', 'volumeSweep')
    def getVolumeSweep(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, effects.fluidEmitter, kwargs, 'volumeSweep')
        return res

    @_f.addMelDocs('fluidEmitter', 'cycleEmission')
    def setCycleEmission(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, effects.fluidEmitter, kwargs, 'cycleEmission', val)

    @_f.addMelDocs('fluidEmitter', 'cycleInterval')
    def setCycleInterval(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, effects.fluidEmitter, kwargs, 'cycleInterval', val)

    @_f.addMelDocs('fluidEmitter', 'densityEmissionRate')
    def setDensityEmissionRate(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, effects.fluidEmitter, kwargs, 'densityEmissionRate', val)

    @_f.addMelDocs('fluidEmitter', 'fluidDropoff')
    def setFluidDropoff(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, effects.fluidEmitter, kwargs, 'fluidDropoff', val)

    @_f.addMelDocs('fluidEmitter', 'fuelEmissionRate')
    def setFuelEmissionRate(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, effects.fluidEmitter, kwargs, 'fuelEmissionRate', val)

    @_f.addMelDocs('fluidEmitter', 'heatEmissionRate')
    def setHeatEmissionRate(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, effects.fluidEmitter, kwargs, 'heatEmissionRate', val)

    @_f.addMelDocs('fluidEmitter', 'maxDistance')
    def setMaxDistance(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, effects.fluidEmitter, kwargs, 'maxDistance', val)

    @_f.addMelDocs('fluidEmitter', 'minDistance')
    def setMinDistance(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, effects.fluidEmitter, kwargs, 'minDistance', val)

    @_f.addMelDocs('fluidEmitter', 'position')
    def setPosition(self, val=True, **kwargs):
        # type: (Tuple[float, float, float] | List[Tuple[float, float, float]], **Any) -> None
        return _f.asEdit(self, effects.fluidEmitter, kwargs, 'position', val)

    @_f.addMelDocs('fluidEmitter', 'rate')
    def setRate(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, effects.fluidEmitter, kwargs, 'rate', val)

    @_f.addMelDocs('fluidEmitter', 'torusSectionRadius')
    def setTorusSectionRadius(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, effects.fluidEmitter, kwargs, 'torusSectionRadius', val)

    @_f.addMelDocs('fluidEmitter', 'type')
    def setType(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, effects.fluidEmitter, kwargs, 'type', val)

    @_f.addMelDocs('fluidEmitter', 'volumeOffset')
    def setVolumeOffset(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, effects.fluidEmitter, kwargs, 'volumeOffset', val)

    @_f.addMelDocs('fluidEmitter', 'volumeShape')
    def setVolumeShape(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, effects.fluidEmitter, kwargs, 'volumeShape', val)

    @_f.addMelDocs('fluidEmitter', 'volumeSweep')
    def setVolumeSweep(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, effects.fluidEmitter, kwargs, 'volumeSweep', val)
# ------ Do not edit above this line --------


class RenderLayer(DependNode):
    # this way, when we create it it's automatically hooked up to the
    # renderLayerManager
    __melcmdname__ = 'createRenderLayer'

    def __new__(cls, *args, **kwargs):
        if not args and 'empty' not in kwargs and 'e' not in kwargs:
            # want to default to empty=False, to match former creation behavior
            kwargs['e'] = True
        return super(RenderLayer, cls).__new__(cls, *args, **kwargs)

    def listMembers(self, fullNames=True):
        if fullNames:
            return [general.PyNode(x) for x in _util.listForNone(
                cmds.editRenderLayerMembers(self, q=1, fullNames=True))]
        else:
            return _util.listForNone(cmds.editRenderLayerMembers(self, q=1, fullNames=False))

    def addMembers(self, members, noRecurse=True):
        cmds.editRenderLayerMembers(self, members, noRecurse=noRecurse)

    def removeMembers(self, members):
        cmds.editRenderLayerMembers(self, members, remove=True)

    def listAdjustments(self):
        return [general.PyNode(x) for x in _util.listForNone(
            cmds.editRenderLayerAdjustment(self, layer=1, q=1))]

    def addAdjustments(self, members):
        return cmds.editRenderLayerAdjustment(members, layer=self)

    def removeAdjustments(self, members):
        return cmds.editRenderLayerAdjustment(members, layer=self, remove=True)

    def setCurrent(self):
        cmds.editRenderLayerGlobals(currentRenderLayer=self)
# ------ Do not edit below this line --------
    __apicls__ = _api.MFnRenderLayer
    __melcmd__ = staticmethod(rendering.createRenderLayer)
    __melcmd_isinfo__ = False
    __melnode__ = 'renderLayer'
    __slots__ = ()

    @_f.addApiDocs(_api.MFnRenderLayer, 'adjustmentPlug')
    def adjustmentPlug(self, scenePlug):
        # type: (str | general.Attribute) -> general.Attribute
        do, final_do, outTypes = _f.getDoArgs([scenePlug], [('scenePlug', 'MPlug', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnRenderLayer, 'adjustmentPlug', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MPlug', None)
        return res

    @classmethod
    @_f.addApiDocs(_api.MFnRenderLayer, 'currentLayer')
    def currentLayer(self):
        # type: () -> DependNode
        res = _api.MFnRenderLayer.currentLayer()
        return _f.ApiArgUtil._castResult(self, res, 'MObject', None)

    @classmethod
    @_f.addApiDocs(_api.MFnRenderLayer, 'defaultRenderLayer')
    def defaultRenderLayer(self):
        # type: () -> DependNode
        res = _api.MFnRenderLayer.defaultRenderLayer()
        return _f.ApiArgUtil._castResult(self, res, 'MObject', None)

    @_f.addApiDocs(_api.MFnRenderLayer, 'externalRenderPasses')
    def externalRenderPasses(self):
        # type: () -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([], [('renderPassArray', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnRenderLayer, 'externalRenderPasses', final_do)
        return _f.processApiResult(res, outTypes, do)

    @classmethod
    @_f.addApiDocs(_api.MFnRenderLayer, 'findLayerByName')
    def findLayerByName(self, renderLayer):
        # type: (str) -> DependNode
        do, final_do, outTypes = _f.getDoArgs([renderLayer], [('renderLayer', 'MString', 'in', None)])
        res = _api.MFnRenderLayer.findLayerByName(*final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MObject', None)
        return res

    @_f.addApiDocs(_api.MFnRenderLayer, 'inCurrentRenderLayer')
    def inCurrentRenderLayer(self, objectPath):
        # type: (DagNode) -> bool
        do, final_do, outTypes = _f.getDoArgs([objectPath], [('objectPath', 'MDagPath', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnRenderLayer, 'inCurrentRenderLayer', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res

    @_f.addApiDocs(_api.MFnRenderLayer, 'inLayer')
    def inLayer(self, transform):
        # type: (str | DependNode) -> bool
        do, final_do, outTypes = _f.getDoArgs([transform], [('transform', 'MObject', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnRenderLayer, 'inLayer', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res

    @_f.addApiDocs(_api.MFnRenderLayer, 'isPlugAdjusted')
    def isPlugAdjusted(self, scenePlug):
        # type: (str | general.Attribute) -> bool
        do, final_do, outTypes = _f.getDoArgs([scenePlug], [('scenePlug', 'MPlug', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnRenderLayer, 'isPlugAdjusted', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res

    @_f.addApiDocs(_api.MFnRenderLayer, 'layerChildren')
    def layerChildren(self, recurse=False):
        # type: (bool) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([recurse], [('array', 'MObjectArray', 'out', None), ('recurse', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnRenderLayer, 'layerChildren', final_do)
        return _f.processApiResult(res, outTypes, do)

    @classmethod
    @_f.addApiDocs(_api.MFnRenderLayer, 'listAllRenderLayers')
    def listAllRenderLayers(self):
        # type: () -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([], [('array', 'MObjectArray', 'out', None)])
        res = _api.MFnRenderLayer.listAllRenderLayers(*final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnRenderLayer, 'passHasLight')
    def passHasLight(self, renderPass, light):
        # type: (str | DependNode, str | DependNode) -> bool
        do, final_do, outTypes = _f.getDoArgs([renderPass, light], [('renderPass', 'MObject', 'in', None), ('light', 'MObject', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnRenderLayer, 'passHasLight', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res

    @_f.addApiDocs(_api.MFnRenderLayer, 'passHasObject')
    def passHasObject(self, renderPass, objectInstance):
        # type: (str | DependNode, DagNode) -> bool
        do, final_do, outTypes = _f.getDoArgs([renderPass, objectInstance], [('renderPass', 'MObject', 'in', None), ('objectInstance', 'MDagPath', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnRenderLayer, 'passHasObject', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res
# ------ Do not edit above this line --------


class DisplayLayer(DependNode):
    # this way, when we create it it's automatically hooked up to the
    # displayLayerManager
    __melcmdname__ = 'createDisplayLayer'

    def __new__(cls, *args, **kwargs):
        if not args and 'empty' not in kwargs and 'e' not in kwargs:
            # want to default to empty=False, to match former creation behavior
            kwargs['e'] = True
        return super(DisplayLayer, cls).__new__(cls, *args, **kwargs)

    def listMembers(self, fullNames=True):
        if fullNames:
            return [general.PyNode(x) for x in _util.listForNone(
                cmds.editDisplayLayerMembers(self, q=1, fullNames=True))]
        else:
            return _util.listForNone(cmds.editDisplayLayerMembers(self, q=1, fullNames=False))

    def addMembers(self, members, noRecurse=True):
        cmds.editDisplayLayerMembers(self, members, noRecurse=noRecurse)

    def removeMembers(self, members, noRecurse=True):
        cmds.editDisplayLayerMembers('defaultLayer', members, noRecurse=noRecurse)

    def setCurrent(self):
        cmds.editDisplayLayerMembers(currentDisplayLayer=self)
# ------ Do not edit below this line --------
    __melcmd__ = staticmethod(general.createDisplayLayer)
    __melcmd_isinfo__ = False
    __melnode__ = 'displayLayer'
    __slots__ = ()

    @_f.deprecated
    def findAlias(self, alias):
        # type: (str) -> Tuple[bool, DependNode]
        do, final_do, outTypes = _f.getDoArgs([alias], [('alias', 'MString', 'in', None), ('attrObj', 'MObject', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'findAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affected instead.')
    def getAffectedAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affects instead.')
    def getAffectedByAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedByAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedByAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use listAliases instead.')
    def getAliasList(self):
        # type: () -> Tuple[bool, List[str]]
        do, final_do, outTypes = _f.getDoArgs([], [('strArray', 'MStringArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAliasList', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use inputs/outputs, or general.listConnections instead.')
    def getConnections(self):
        # type: () -> List[general.Attribute]
        do, final_do, outTypes = _f.getDoArgs([], [('array', 'MPlugArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getConnections', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.getAlias instead.')
    def plugsAlias(self, plug):
        # type: (str | general.Attribute) -> str
        do, final_do, outTypes = _f.getDoArgs([plug], [('plug', 'MPlug', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'plugsAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MString', None)
        return res

    @_f.deprecated('Use Attribute.setAlias instead.')
    def setAlias(self, alias, name, plug, add=True):
        # type: (str, str, str | general.Attribute, bool) -> bool
        do, final_do, outTypes = _f.getDoArgs([alias, name, plug, add], [('alias', 'MString', 'in', None), ('name', 'MString', 'in', None), ('plug', 'MPlug', 'in', None), ('add', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'setAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res
# ------ Do not edit above this line --------


class Constraint(Transform):

    def setWeight(self, weight, *targetObjects):
        inFunc = getattr(cmds, self.type())
        if not targetObjects:
            targetObjects = self.getTargetList()

        constraintObj = self.constraintParentInverseMatrix.inputs()[0]
        args = list(targetObjects) + [constraintObj]
        return inFunc(*args, **{'edit': True, 'weight': weight})

    def getWeight(self, *targetObjects):
        inFunc = getattr(cmds, self.type())
        if not targetObjects:
            targetObjects = self.getTargetList()

        constraintObj = self.constraintParentInverseMatrix.inputs()[0]
        args = list(targetObjects) + [constraintObj]
        return inFunc(*args, **{'query': True, 'weight': True})
# ------ Do not edit below this line --------
    __melnode__ = 'constraint'
    __slots__ = ()

    @_f.deprecated('Use getRotation(quaternion=True) instead.')
    def getRotationQuaternion(self, space='transform'):
        # type: (datatypes.Space.Space) -> Tuple[float, float, float, float]
        do, final_do, outTypes = _f.getDoArgs([space], [('x', 'double', 'out', None), ('y', 'double', 'out', None), ('z', 'double', 'out', None), ('w', 'double', 'out', None), ('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnTransform, 'getRotationQuaternion', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use setRotation, and pass in a Quaternion object, instead.')
    def setRotationQuaternion(self, x, y, z, w, space='transform'):
        # type: (float, float, float, float, datatypes.Space.Space) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([x, y, z, w, space], [('x', 'double', 'in', None), ('y', 'double', 'in', None), ('z', 'double', 'in', None), ('w', 'double', 'in', None), ('space', ('MSpace', 'Space'), 'in', None)], self.getRotationQuaternion, self.setRotationQuaternion, ['space'])
        res = _f.getProxyResult(self, _api.MFnTransform, 'setRotationQuaternion', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res
# ------ Do not edit above this line --------


class GeometryShape(Shape):

    def __getattr__(self, attr):
        # print "Mesh.__getattr__", attr
        try:
            return self.comp(attr)
        except general.MayaComponentError:
            # print "getting super", attr
            return super(GeometryShape, self).__getattr__(attr)
# ------ Do not edit below this line --------
    __melnode__ = 'geometryShape'
    __slots__ = ()
# ------ Do not edit above this line --------


class DeformableShape(GeometryShape):

    @classmethod
    def _numCVsFunc_generator(cls, formFunc, spansPlusDegreeFunc, spansFunc,
                              name=None, doc=None):
        """
        Intended to be used by NurbsCurve / NurbsSurface to generate
        functions which give the 'true' number of editable CVs,
        as opposed to just numSpans + degree.
        (The two values will differ if we have a periodic curve).

        Note that this will usually need to be called outside/after the
        class definition, as formFunc/spansFunc/etc will not be defined
        until then, as they are added by the metaclass.
        """

        def _numCvs_generatedFunc(self, editableOnly=True):
            if editableOnly and formFunc(self) == self.Form.periodic:
                return spansFunc(self)
            else:
                return spansPlusDegreeFunc(self)
        if name:
            _numCvs_generatedFunc.__name__ = name
        if doc:
            _numCvs_generatedFunc.__doc__ = doc
        return _numCvs_generatedFunc

    @classmethod
    def _numEPsFunc_generator(cls, formFunc, spansFunc,
                              name=None, doc=None):
        """
        Intended to be used by NurbsCurve / NurbsSurface to generate
        functions which give the 'true' number of editable EPs,
        as opposed to just numSpans.
        (The two values will differ if we have a periodic curve).

        Note that this will usually need to be called outside/after the
        class definition, as formFunc/spansFunc will not be defined
        until then, as they are added by the metaclass.
        """

        def _numEPs_generatedFunc(self, editableOnly=True):
            if editableOnly and formFunc(self) == self.Form.periodic:
                return spansFunc(self)
            else:
                return spansFunc(self) + 1
        if name:
            _numEPs_generatedFunc.__name__ = name
        if doc:
            _numEPs_generatedFunc.__doc__ = doc
        return _numEPs_generatedFunc
# ------ Do not edit below this line --------
    __melcmd__ = staticmethod(animation.deformableShape)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'deformableShape'
    __melnode__ = 'deformableShape'
    __slots__ = ()
# ------ Do not edit above this line --------


class ControlPoint(DeformableShape):
    pass
# ------ Do not edit below this line --------
    __melnode__ = 'controlPoint'
    __slots__ = ()
# ------ Do not edit above this line --------


class CurveShape(DeformableShape):
    pass
# ------ Do not edit below this line --------
    __melnode__ = 'curveShape'
    __slots__ = ()
# ------ Do not edit above this line --------


class NurbsCurve(CurveShape):
    _componentAttributes = {
        'u': general.NurbsCurveParameter,
        'cv': general.NurbsCurveCV,
        'controlVerts': general.NurbsCurveCV,
        'ep': general.NurbsCurveEP,
        'editPoints': general.NurbsCurveEP,
        'knot': general.NurbsCurveKnot,
        'knots': general.NurbsCurveKnot,
    }
    if TYPE_CHECKING:
        # FIXME: move to build
        u = None  # type: general.NurbsCurveParameter
        cv = None  # type: general.NurbsCurveCV
        controlVerts = None  # type: general.NurbsCurveCV
        ep = None  # type: general.NurbsCurveEP
        editPoints = None  # type: general.NurbsCurveEP
        knot = None  # type: general.NurbsCurveKnot
        knots = None  # type: general.NurbsCurveKnot
# ------ Do not edit below this line --------
    __apicls__ = _api.MFnNurbsCurve
    __melcmd__ = staticmethod(modeling.curve)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'curve'
    __melnode__ = 'nurbsCurve'
    __slots__ = ()
    Form = Enum('Form', [('invalid', 0), ('kInvalid', 0), ('open', 1), ('kOpen', 1), ('closed', 2), ('kClosed', 2), ('periodic', 3), ('kPeriodic', 3), ('last', 4), ('kLast', 4)], multiKeys=True)

    @_f.addApiDocs(_api.MFnNurbsCurve, 'numCVs')
    def _numCVsApi(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnNurbsCurve, 'numCVs')
        return _f.ApiArgUtil._castResult(self, res, 'int', None)

    @_f.addApiDocs(_api.MFnNurbsCurve, 'area')
    def area(self, tolerance=0.001):
        # type: (float) -> float
        do, final_do, outTypes = _f.getDoArgs([tolerance], [('tolerance', 'double', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsCurve, 'area', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'double', None)
        return res

    @_f.addApiDocs(_api.MFnNurbsCurve, 'closestPoint')
    def closestPoint(self, toThisPoint, param=None, tolerance=0.001, space='preTransform'):
        # type: (datatypes.Point, float, float, datatypes.Space.Space) -> datatypes.Point
        do, final_do, outTypes = _f.getDoArgs([toThisPoint, param, tolerance, space], [('toThisPoint', 'MPoint', 'in', None), ('param', 'double', 'in', None), ('tolerance', 'double', 'in', None), ('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsCurve, 'closestPoint', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MPoint', None)
        return res

    @_f.addApiDocs(_api.MFnNurbsCurve, 'copy')
    def copy(self, source, parentOrOwner='MObject::kNullObj'):
        # type: (str | DependNode, str | DependNode) -> DependNode
        do, final_do, outTypes = _f.getDoArgs([source, parentOrOwner], [('source', 'MObject', 'in', None), ('parentOrOwner', 'MObject', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsCurve, 'copy', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MObject', None)
        return res

    @_f.addApiDocs(_api.MFnNurbsCurve, 'createWithEditPoints')
    def createWithEditPoints(self, editPoints, degree, form, create2D, createRational, uniformParam, parentOrOwner='MObject::kNullObj'):
        # type: (List[datatypes.Point], int, NurbsCurve.Form, bool, bool, bool, str | DependNode) -> DependNode
        do, final_do, outTypes = _f.getDoArgs([editPoints, degree, form, create2D, createRational, uniformParam, parentOrOwner], [('editPoints', 'MPointArray', 'in', None), ('degree', 'uint', 'in', None), ('form', ('MFnNurbsCurve', 'Form'), 'in', None), ('create2D', 'bool', 'in', None), ('createRational', 'bool', 'in', None), ('uniformParam', 'bool', 'in', None), ('parentOrOwner', 'MObject', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsCurve, 'createWithEditPoints', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MObject', None)
        return res

    @_f.addApiDocs(_api.MFnNurbsCurve, 'cvs')
    def cvs(self, startIndex, endIndex):
        # type: (int, int) -> DependNode
        do, final_do, outTypes = _f.getDoArgs([startIndex, endIndex], [('startIndex', 'uint', 'in', None), ('endIndex', 'uint', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsCurve, 'cvs', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MObject', None)
        return res

    @_f.addApiDocs(_api.MFnNurbsCurve, 'degree')
    def degree(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnNurbsCurve, 'degree')
        return _f.ApiArgUtil._castResult(self, res, 'int', None)

    @_f.addApiDocs(_api.MFnNurbsCurve, 'distanceToPoint')
    def distanceToPoint(self, pt, space='preTransform'):
        # type: (datatypes.Point, datatypes.Space.Space) -> float
        do, final_do, outTypes = _f.getDoArgs([pt, space], [('pt', 'MPoint', 'in', None), ('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsCurve, 'distanceToPoint', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'double', None)
        return res

    @_f.addApiDocs(_api.MFnNurbsCurve, 'findLengthFromParam')
    def findLengthFromParam(self, param):
        # type: (float) -> float
        do, final_do, outTypes = _f.getDoArgs([param], [('param', 'double', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsCurve, 'findLengthFromParam', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'double', None)
        return res

    @_f.addApiDocs(_api.MFnNurbsCurve, 'findParamFromLength')
    def findParamFromLength(self, partLength):
        # type: (float) -> float
        do, final_do, outTypes = _f.getDoArgs([partLength], [('partLength', 'double', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsCurve, 'findParamFromLength', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'double', None)
        return res

    @_f.addApiDocs(_api.MFnNurbsCurve, 'form')
    def form(self):
        # type: () -> NurbsCurve.Form
        res = _f.getProxyResult(self, _api.MFnNurbsCurve, 'form')
        return _f.ApiArgUtil._castResult(self, res, ('MFnNurbsCurve', 'Form'), None)

    @_f.addApiDocs(_api.MFnNurbsCurve, 'getCV')
    def getCV(self, index, space='preTransform'):
        # type: (int, datatypes.Space.Space) -> datatypes.Point
        do, final_do, outTypes = _f.getDoArgs([index, space], [('index', 'uint', 'in', None), ('pt', 'MPoint', 'out', None), ('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsCurve, 'getCV', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnNurbsCurve, 'getCVs')
    def getCVs(self, space='preTransform'):
        # type: (datatypes.Space.Space) -> List[datatypes.Point]
        do, final_do, outTypes = _f.getDoArgs([space], [('array', 'MPointArray', 'out', None), ('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsCurve, 'getCVs', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnNurbsCurve, 'getDerivativesAtParm')
    def getDerivativesAtParm(self, param, space):
        # type: (float, datatypes.Space.Space) -> Tuple[datatypes.Point, datatypes.Vector, datatypes.Vector]
        do, final_do, outTypes = _f.getDoArgs([param, space], [('param', 'double', 'in', None), ('pos', 'MPoint', 'out', None), ('dU', 'MVector', 'out', None), ('space', ('MSpace', 'Space'), 'in', None), ('dUU', 'MVector', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsCurve, 'getDerivativesAtParm', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnNurbsCurve, 'knot')
    def getKnot(self, index):
        # type: (int) -> float
        do, final_do, outTypes = _f.getDoArgs([index], [('index', 'uint', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsCurve, 'knot', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'double', None)
        return res

    @_f.addApiDocs(_api.MFnNurbsCurve, 'getKnotDomain')
    def getKnotDomain(self):
        # type: () -> Tuple[float, float]
        do, final_do, outTypes = _f.getDoArgs([], [('start', 'double', 'out', None), ('end', 'double', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsCurve, 'getKnotDomain', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnNurbsCurve, 'getKnots')
    def getKnots(self):
        # type: () -> List[float]
        do, final_do, outTypes = _f.getDoArgs([], [('array', 'MDoubleArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsCurve, 'getKnots', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnNurbsCurve, 'getParamAtPoint')
    def getParamAtPoint(self, atThisPoint, space='preTransform'):
        # type: (datatypes.Point, datatypes.Space.Space) -> float
        do, final_do, outTypes = _f.getDoArgs([atThisPoint, space], [('atThisPoint', 'MPoint', 'in', None), ('param', 'double', 'out', None), ('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsCurve, 'getParamAtPoint', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnNurbsCurve, 'getPointAtParam')
    def getPointAtParam(self, param, space='preTransform'):
        # type: (float, datatypes.Space.Space) -> datatypes.Point
        do, final_do, outTypes = _f.getDoArgs([param, space], [('param', 'double', 'in', None), ('point', 'MPoint', 'out', None), ('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsCurve, 'getPointAtParam', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnNurbsCurve, 'hasHistoryOnCreate')
    def hasHistoryOnCreate(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnNurbsCurve, 'hasHistoryOnCreate')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnNurbsCurve, 'isParamOnCurve')
    def isParamOnCurve(self, param):
        # type: (float) -> bool
        do, final_do, outTypes = _f.getDoArgs([param], [('param', 'double', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsCurve, 'isParamOnCurve', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res

    @_f.addApiDocs(_api.MFnNurbsCurve, 'isPlanar')
    def isPlanar(self, planeNormal=None):
        # type: (datatypes.Vector) -> bool
        do, final_do, outTypes = _f.getDoArgs([planeNormal], [('planeNormal', 'MVector', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsCurve, 'isPlanar', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res

    @_f.addApiDocs(_api.MFnNurbsCurve, 'isPointOnCurve')
    def isPointOnCurve(self, point, tolerance=0.001, space='preTransform'):
        # type: (datatypes.Point, float, datatypes.Space.Space) -> bool
        do, final_do, outTypes = _f.getDoArgs([point, tolerance, space], [('point', 'MPoint', 'in', None), ('tolerance', 'double', 'in', None), ('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsCurve, 'isPointOnCurve', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res

    @_f.addApiDocs(_api.MFnNurbsCurve, 'length')
    def length(self, tolerance=0.001):
        # type: (float) -> float
        do, final_do, outTypes = _f.getDoArgs([tolerance], [('tolerance', 'double', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsCurve, 'length', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'double', None)
        return res

    @_f.addApiDocs(_api.MFnNurbsCurve, 'makeMultipleEndKnots')
    def makeMultipleEndKnots(self):
        # type: () -> None
        res = _f.getProxyResult(self, _api.MFnNurbsCurve, 'makeMultipleEndKnots')
        return res

    @_f.addApiDocs(_api.MFnNurbsCurve, 'normal')
    def normal(self, param, space='preTransform'):
        # type: (float, datatypes.Space.Space) -> datatypes.Vector
        do, final_do, outTypes = _f.getDoArgs([param, space], [('param', 'double', 'in', None), ('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsCurve, 'normal', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MVector', None)
        return res

    @_f.addApiDocs(_api.MFnNurbsCurve, 'numKnots')
    def numKnots(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnNurbsCurve, 'numKnots')
        return _f.ApiArgUtil._castResult(self, res, 'int', None)

    @_f.addApiDocs(_api.MFnNurbsCurve, 'numSpans')
    def numSpans(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnNurbsCurve, 'numSpans')
        return _f.ApiArgUtil._castResult(self, res, 'int', None)

    @_f.addApiDocs(_api.MFnNurbsCurve, 'rebuild')
    def rebuild(self, endKnots=0, keepEndPoints=True, keepTangents=True, keepControlPoints=False, tolerance=0.01):
        # type: (int, bool, bool, bool, float) -> Tuple[DependNode, int, int, int]
        do, final_do, outTypes = _f.getDoArgs([endKnots, keepEndPoints, keepTangents, keepControlPoints, tolerance], [('spans', 'uint', 'out', None), ('degree', 'uint', 'out', None), ('keepRange', 'uint', 'out', None), ('endKnots', 'uint', 'in', None), ('keepEndPoints', 'bool', 'in', None), ('keepTangents', 'bool', 'in', None), ('keepControlPoints', 'bool', 'in', None), ('tolerance', 'float', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsCurve, 'rebuild', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MObject', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnNurbsCurve, 'removeKnot')
    def removeKnot(self, atThisParam, removeAll=False):
        # type: (float, bool) -> None
        do, final_do, outTypes = _f.getDoArgs([atThisParam, removeAll], [('atThisParam', 'double', 'in', None), ('removeAll', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsCurve, 'removeKnot', final_do)
        return res

    @_f.addApiDocs(_api.MFnNurbsCurve, 'reverse')
    def reverse(self, constructionHistory=False):
        # type: (bool) -> None
        do, final_do, outTypes = _f.getDoArgs([constructionHistory], [('constructionHistory', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsCurve, 'reverse', final_do)
        return res

    @_f.addApiDocs(_api.MFnNurbsCurve, 'setCV')
    def setCV(self, index, pt, space='preTransform'):
        # type: (int, datatypes.Point, datatypes.Space.Space) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([index, pt, space], [('index', 'uint', 'in', None), ('pt', 'MPoint', 'in', None), ('space', ('MSpace', 'Space'), 'in', None)], self.getCV, self.setCV, ['index', 'space'])
        res = _f.getProxyResult(self, _api.MFnNurbsCurve, 'setCV', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnNurbsCurve, 'setCVs')
    def setCVs(self, array, space='preTransform'):
        # type: (List[datatypes.Point], datatypes.Space.Space) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([array, space], [('array', 'MPointArray', 'in', None), ('space', ('MSpace', 'Space'), 'in', None)], self.getCVs, self.setCVs, ['space'])
        res = _f.getProxyResult(self, _api.MFnNurbsCurve, 'setCVs', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnNurbsCurve, 'setKnot')
    def setKnot(self, index, param):
        # type: (int, float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([index, param], [('index', 'uint', 'in', None), ('param', 'double', 'in', None)], self.getKnot, self.setKnot, ['index'])
        res = _f.getProxyResult(self, _api.MFnNurbsCurve, 'setKnot', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnNurbsCurve, 'setKnots')
    def setKnots(self, array, startIndex, endIndex):
        # type: (List[float], int, int) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([array, startIndex, endIndex], [('array', 'MDoubleArray', 'in', None), ('startIndex', 'uint', 'in', None), ('endIndex', 'uint', 'in', None)], self.getKnots, self.setKnots, [])
        res = _f.getProxyResult(self, _api.MFnNurbsCurve, 'setKnots', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnNurbsCurve, 'tangent')
    def tangent(self, param, space='preTransform'):
        # type: (float, datatypes.Space.Space) -> datatypes.Vector
        do, final_do, outTypes = _f.getDoArgs([param, space], [('param', 'double', 'in', None), ('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsCurve, 'tangent', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MVector', None)
        return res

    @_f.addApiDocs(_api.MFnNurbsCurve, 'updateCurve')
    def updateCurve(self):
        # type: () -> None
        res = _f.getProxyResult(self, _api.MFnNurbsCurve, 'updateCurve')
        return res
# ------ Do not edit above this line --------



if not _factories.building:
    # keep this safe to load if the templates have not been built yet

    # apiToMelBridge maps MFnNurbsCurve.numCVs => NurbsCurve._numCVsApi
    NurbsCurve.numCVs = \
        NurbsCurve._numCVsFunc_generator(NurbsCurve.form,
                                         NurbsCurve._numCVsApi,
                                         NurbsCurve.numSpans,
                                         name='numCVs',
                                         doc="""
            Returns the number of CVs.
    
            :Parameters:
            editableOnly : `bool`
                If editableOnly evaluates to True (default), then this will return
                the number of cvs that can be actually edited (and also the highest
                index that may be used for cv's - ie, if
                    myCurve.numCVs(editableOnly=True) == 4
                then allowable cv indices go from
                    myCurve.cv[0] to mySurf.cv[3]
    
                If editablyOnly is False, then this will return the underlying
                number of cvs used to define the mathematical curve -
                degree + numSpans.
    
                These will only differ if the form is 'periodic', in which
                case the editable number will be numSpans (as the last 'degree'
                cv's are 'locked' to be the same as the first 'degree' cvs).
                In all other cases, the number of cvs will be degree + numSpans.
    
            :Examples:
                >>> from pymel.core import *
                >>> # a periodic curve
                >>> myCurve = curve(name='periodicCurve1', d=3, periodic=True, k=(0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12), pw=[(4, -4, 0, 1), (5.5, 0, 0, 1), (4, 4, 0, 1), (0, 5.5, 0, 1), (-4, 4, 0, 1), (-5.5, 0, 0, 1), (-4, -4, 0, 1), (0, -5.5, 0, 1), (4, -4, 0, 1), (5.5, 0, 0, 1), (4, 4, 0, 1)] )
                >>> myCurve.cv
                NurbsCurveCV('periodicCurveShape1.cv[0:7]')
                >>> myCurve.numCVs()
                8
                >>> myCurve.numCVs(editableOnly=False)
                11
                >>>
                >>> # an open curve
                >>> myCurve = curve(name='openCurve1', d=3, periodic=False, k=(0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12), pw=[(4, -4, 0, 1), (5.5, 0, 0, 1), (4, 4, 0, 1), (0, 5.5, 0, 1), (-4, 4, 0, 1), (-5.5, 0, 0, 1), (-4, -4, 0, 1), (0, -5.5, 0, 1), (4, -4, 0, 1), (5.5, 0, 0, 1), (4, 4, 0, 1)] )
                >>> myCurve.cv
                NurbsCurveCV('openCurveShape1.cv[0:10]')
                >>> myCurve.numCVs()
                11
                >>> myCurve.numCVs(editableOnly=False)
                11
    
            :rtype: `int`
            """)

    NurbsCurve.numEPs = \
        NurbsCurve._numEPsFunc_generator(NurbsCurve.form,
                                         NurbsCurve.numSpans,
                                         name='numEPs',
                                         doc="""
            Returns the number of EPs.
    
            :Examples:
                >>> from pymel.core import *
                >>> # a periodic curve
                >>> myCurve = curve(name='periodicCurve2', d=3, periodic=True, k=(0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12), pw=[(4, -4, 0, 1), (5.5, 0, 0, 1), (4, 4, 0, 1), (0, 5.5, 0, 1), (-4, 4, 0, 1), (-5.5, 0, 0, 1), (-4, -4, 0, 1), (0, -5.5, 0, 1), (4, -4, 0, 1), (5.5, 0, 0, 1), (4, 4, 0, 1)] )
                >>> myCurve.ep
                NurbsCurveEP('periodicCurveShape2.ep[0:7]')
                >>> myCurve.numEPs()
                8
                >>>
                >>> # an open curve
                >>> myCurve = curve(name='openCurve2', d=3, periodic=False, k=(0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12), pw=[(4, -4, 0, 1), (5.5, 0, 0, 1), (4, 4, 0, 1), (0, 5.5, 0, 1), (-4, 4, 0, 1), (-5.5, 0, 0, 1), (-4, -4, 0, 1), (0, -5.5, 0, 1), (4, -4, 0, 1), (5.5, 0, 0, 1), (4, 4, 0, 1)] )
                >>> myCurve.ep
                NurbsCurveEP('openCurveShape2.ep[0:8]')
                >>> myCurve.numEPs()
                9
    
            Returns
            -------
            int
            """)


class SurfaceShape(ControlPoint):
    pass
# ------ Do not edit below this line --------
    __melnode__ = 'surfaceShape'
    __slots__ = ()
# ------ Do not edit above this line --------


class NurbsSurface(SurfaceShape):
    _componentAttributes = {
        'u': (general.NurbsSurfaceRange, 'u'),
        'uIsoparm': (general.NurbsSurfaceRange, 'u'),
        'v': (general.NurbsSurfaceRange, 'v'),
        'vIsoparm': (general.NurbsSurfaceRange, 'v'),
        'uv': (general.NurbsSurfaceRange, 'uv'),
        'cv': general.NurbsSurfaceCV,
        'controlVerts': general.NurbsSurfaceCV,
        'ep': general.NurbsSurfaceEP,
        'editPoints': general.NurbsSurfaceEP,
        'knot': general.NurbsSurfaceKnot,
        'knots': general.NurbsSurfaceKnot,
        'sf': general.NurbsSurfaceFace,
        'faces': general.NurbsSurfaceFace,
    }

    if TYPE_CHECKING:
        # FIXME: move to build
        u = None  # type: general.NurbsSurfaceRange
        uIsoparm = None  # type: general.NurbsSurfaceRange
        v = None  # type: general.NurbsSurfaceRange
        vIsoparm = None  # type: general.NurbsSurfaceRange
        uv = None  # type: general.NurbsSurfaceRange
        cv = None  # type: general.NurbsSurfaceCV
        controlVerts = None  # type: general.NurbsSurfaceCV
        ep = None  # type: general.NurbsSurfaceEP
        editPoints = None  # type: general.NurbsSurfaceEP
        knot = None  # type: general.NurbsSurfaceKnot
        knots = None  # type: general.NurbsSurfaceKnot
        sf = None  # type: general.NurbsSurfaceFace
        faces = None  # type: general.NurbsSurfaceFace
# ------ Do not edit below this line --------
    __apicls__ = _api.MFnNurbsSurface
    __melcmd__ = staticmethod(modeling.surface)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'surface'
    __melnode__ = 'nurbsSurface'
    __slots__ = ()
    BoundaryType = Enum('BoundaryType', [('invalidBoundary', 0), ('kInvalidBoundary', 0), ('outer', 1), ('kOuter', 1), ('inner', 2), ('kInner', 2), ('segment', 3), ('kSegment', 3), ('closedSegment', 4), ('kClosedSegment', 4)], multiKeys=True)
    Form = Enum('Form', [('invalid', 0), ('kInvalid', 0), ('open', 1), ('kOpen', 1), ('closed', 2), ('kClosed', 2), ('periodic', 3), ('kPeriodic', 3), ('last', 4), ('kLast', 4)], multiKeys=True)

    @_f.addApiDocs(_api.MFnNurbsSurface, 'numCVsInU')
    def _numCVsInUApi(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'numCVsInU')
        return _f.ApiArgUtil._castResult(self, res, 'int', None)

    @_f.addApiDocs(_api.MFnNurbsSurface, 'numCVsInV')
    def _numCVsInVApi(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'numCVsInV')
        return _f.ApiArgUtil._castResult(self, res, 'int', None)

    @_f.addApiDocs(_api.MFnNurbsSurface, 'area')
    def area(self, tolerance=0.001):
        # type: (float) -> float
        do, final_do, outTypes = _f.getDoArgs([tolerance], [('tolerance', 'double', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'area', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'double', None)
        return res

    @_f.addApiDocs(_api.MFnNurbsSurface, 'assignUV')
    def assignUV(self, patchId, cornerIndex, uvId):
        # type: (int, int, int) -> None
        do, final_do, outTypes = _f.getDoArgs([patchId, cornerIndex, uvId], [('patchId', 'int', 'in', None), ('cornerIndex', 'int', 'in', None), ('uvId', 'int', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'assignUV', final_do)
        return res

    @_f.addApiDocs(_api.MFnNurbsSurface, 'assignUVs')
    def assignUVs(self, uvCounts, uvIds):
        # type: (List[int], List[int]) -> None
        do, final_do, outTypes = _f.getDoArgs([uvCounts, uvIds], [('uvCounts', 'MIntArray', 'in', None), ('uvIds', 'MIntArray', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'assignUVs', final_do)
        return res

    @_f.addApiDocs(_api.MFnNurbsSurface, 'boundaryType')
    def boundaryType(self, region, boundary):
        # type: (int, int) -> NurbsSurface.BoundaryType
        do, final_do, outTypes = _f.getDoArgs([region, boundary], [('region', 'uint', 'in', None), ('boundary', 'uint', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'boundaryType', final_do)
        res = _f.ApiArgUtil._castResult(self, res, ('MFnNurbsSurface', 'BoundaryType'), None)
        return res

    @_f.addApiDocs(_api.MFnNurbsSurface, 'clearUVs')
    def clearUVs(self):
        # type: () -> None
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'clearUVs')
        return res

    @_f.addApiDocs(_api.MFnNurbsSurface, 'closestPoint')
    def closestPoint(self, toThisPoint, ignoreTrimBoundaries=False, tolerance=0.001, space='preTransform'):
        # type: (datatypes.Point, bool, float, datatypes.Space.Space) -> Tuple[datatypes.Point, float, float]
        do, final_do, outTypes = _f.getDoArgs([toThisPoint, ignoreTrimBoundaries, tolerance, space], [('toThisPoint', 'MPoint', 'in', None), ('paramU', 'double', 'out', None), ('paramV', 'double', 'out', None), ('ignoreTrimBoundaries', 'bool', 'in', None), ('tolerance', 'double', 'in', None), ('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'closestPoint', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MPoint', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnNurbsSurface, 'copy')
    def copy(self, source, parentOrOwner='MObject::kNullObj'):
        # type: (str | DependNode, str | DependNode) -> DependNode
        do, final_do, outTypes = _f.getDoArgs([source, parentOrOwner], [('source', 'MObject', 'in', None), ('parentOrOwner', 'MObject', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'copy', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MObject', None)
        return res

    @_f.addApiDocs(_api.MFnNurbsSurface, 'cvsInU')
    def cvsInU(self, startIndex, endIndex, rowIndex):
        # type: (int, int, int) -> DependNode
        do, final_do, outTypes = _f.getDoArgs([startIndex, endIndex, rowIndex], [('startIndex', 'uint', 'in', None), ('endIndex', 'uint', 'in', None), ('rowIndex', 'uint', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'cvsInU', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MObject', None)
        return res

    @_f.addApiDocs(_api.MFnNurbsSurface, 'cvsInV')
    def cvsInV(self, startIndex, endIndex, rowIndex):
        # type: (int, int, int) -> DependNode
        do, final_do, outTypes = _f.getDoArgs([startIndex, endIndex, rowIndex], [('startIndex', 'uint', 'in', None), ('endIndex', 'uint', 'in', None), ('rowIndex', 'uint', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'cvsInV', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MObject', None)
        return res

    @_f.addApiDocs(_api.MFnNurbsSurface, 'degreeU')
    def degreeU(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'degreeU')
        return _f.ApiArgUtil._castResult(self, res, 'int', None)

    @_f.addApiDocs(_api.MFnNurbsSurface, 'degreeV')
    def degreeV(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'degreeV')
        return _f.ApiArgUtil._castResult(self, res, 'int', None)

    @_f.addApiDocs(_api.MFnNurbsSurface, 'distanceToPoint')
    def distanceToPoint(self, pt, space='preTransform'):
        # type: (datatypes.Point, datatypes.Space.Space) -> float
        do, final_do, outTypes = _f.getDoArgs([pt, space], [('pt', 'MPoint', 'in', None), ('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'distanceToPoint', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'double', None)
        return res

    @_f.addApiDocs(_api.MFnNurbsSurface, 'edge')
    def edge(self, region, boundary, edge, paramEdge=False):
        # type: (int, int, int, bool) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([region, boundary, edge, paramEdge], [('region', 'uint', 'in', None), ('boundary', 'uint', 'in', None), ('edge', 'uint', 'in', None), ('paramEdge', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'edge', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MObjectArray', None)
        return res

    @_f.addApiDocs(_api.MFnNurbsSurface, 'formInU')
    def formInU(self):
        # type: () -> NurbsSurface.Form
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'formInU')
        return _f.ApiArgUtil._castResult(self, res, ('MFnNurbsSurface', 'Form'), None)

    @_f.addApiDocs(_api.MFnNurbsSurface, 'formInV')
    def formInV(self):
        # type: () -> NurbsSurface.Form
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'formInV')
        return _f.ApiArgUtil._castResult(self, res, ('MFnNurbsSurface', 'Form'), None)

    @_f.addApiDocs(_api.MFnNurbsSurface, 'getAssignedUVs')
    def getAssignedUVs(self):
        # type: () -> Tuple[List[int], List[int]]
        do, final_do, outTypes = _f.getDoArgs([], [('uvCounts', 'MIntArray', 'out', None), ('uvIds', 'MIntArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'getAssignedUVs', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnNurbsSurface, 'getCV')
    def getCV(self, indexU, indexV, space='preTransform'):
        # type: (int, int, datatypes.Space.Space) -> datatypes.Point
        do, final_do, outTypes = _f.getDoArgs([indexU, indexV, space], [('indexU', 'uint', 'in', None), ('indexV', 'uint', 'in', None), ('pnt', 'MPoint', 'out', None), ('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'getCV', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnNurbsSurface, 'getCVs')
    def getCVs(self, space='preTransform'):
        # type: (datatypes.Space.Space) -> List[datatypes.Point]
        do, final_do, outTypes = _f.getDoArgs([space], [('array', 'MPointArray', 'out', None), ('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'getCVs', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnNurbsSurface, 'getConnectedShaders')
    def getConnectedShaders(self, instanceNumber):
        # type: (int) -> Tuple[List[DependNode], List[int]]
        do, final_do, outTypes = _f.getDoArgs([instanceNumber], [('instanceNumber', 'uint', 'in', None), ('shaders', 'MObjectArray', 'out', None), ('indices', 'MIntArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'getConnectedShaders', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnNurbsSurface, 'getDataObject')
    def getDataObject(self):
        # type: () -> DependNode
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'getDataObject')
        return _f.ApiArgUtil._castResult(self, res, 'MObject', None)

    @_f.addApiDocs(_api.MFnNurbsSurface, 'getDerivativesAtParm')
    def getDerivativesAtParm(self, paramInU, paramInV, space):
        # type: (float, float, datatypes.Space.Space) -> Tuple[datatypes.Point, datatypes.Vector, datatypes.Vector, datatypes.Vector, datatypes.Vector, datatypes.Vector]
        do, final_do, outTypes = _f.getDoArgs([paramInU, paramInV, space], [('paramInU', 'double', 'in', None), ('paramInV', 'double', 'in', None), ('pos', 'MPoint', 'out', None), ('dU', 'MVector', 'out', None), ('dV', 'MVector', 'out', None), ('space', ('MSpace', 'Space'), 'in', None), ('dUU', 'MVector', 'out', None), ('dVV', 'MVector', 'out', None), ('dUV', 'MVector', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'getDerivativesAtParm', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnNurbsSurface, 'getKnotDomain')
    def getKnotDomain(self):
        # type: () -> Tuple[float, float, float, float]
        do, final_do, outTypes = _f.getDoArgs([], [('startU', 'double', 'out', None), ('endU', 'double', 'out', None), ('startV', 'double', 'out', None), ('endV', 'double', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'getKnotDomain', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnNurbsSurface, 'knotInU')
    def getKnotInU(self, index):
        # type: (int) -> float
        do, final_do, outTypes = _f.getDoArgs([index], [('index', 'uint', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'knotInU', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'double', None)
        return res

    @_f.addApiDocs(_api.MFnNurbsSurface, 'knotInV')
    def getKnotInV(self, index):
        # type: (int) -> float
        do, final_do, outTypes = _f.getDoArgs([index], [('index', 'uint', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'knotInV', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'double', None)
        return res

    @_f.addApiDocs(_api.MFnNurbsSurface, 'getKnotsInU')
    def getKnotsInU(self):
        # type: () -> List[float]
        do, final_do, outTypes = _f.getDoArgs([], [('array', 'MDoubleArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'getKnotsInU', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnNurbsSurface, 'getKnotsInV')
    def getKnotsInV(self):
        # type: () -> List[float]
        do, final_do, outTypes = _f.getDoArgs([], [('array', 'MDoubleArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'getKnotsInV', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.maya_deprecated
    def getParamAtPoint(self, atThisPoint, space='preTransform'):
        # type: (datatypes.Point, datatypes.Space.Space) -> Tuple[float, float]
        do, final_do, outTypes = _f.getDoArgs([atThisPoint, space], [('atThisPoint', 'MPoint', 'in', None), ('paramU', 'double', 'out', None), ('paramV', 'double', 'out', None), ('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'getParamAtPoint', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnNurbsSurface, 'getPatchUV')
    def getPatchUV(self, patchId, cornerIndex):
        # type: (int, int) -> Tuple[float, float]
        do, final_do, outTypes = _f.getDoArgs([patchId, cornerIndex], [('patchId', 'int', 'in', None), ('cornerIndex', 'int', 'in', None), ('u', 'float', 'out', None), ('v', 'float', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'getPatchUV', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnNurbsSurface, 'getPatchUVid')
    def getPatchUVid(self, patchId, cornerIndex):
        # type: (int, int) -> int
        do, final_do, outTypes = _f.getDoArgs([patchId, cornerIndex], [('patchId', 'int', 'in', None), ('cornerIndex', 'int', 'in', None), ('uvId', 'int', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'getPatchUVid', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnNurbsSurface, 'getPatchUVs')
    def getPatchUVs(self, patchId):
        # type: (int) -> Tuple[List[float], List[float]]
        do, final_do, outTypes = _f.getDoArgs([patchId], [('patchId', 'int', 'in', None), ('uArray', 'MFloatArray', 'out', None), ('vArray', 'MFloatArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'getPatchUVs', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnNurbsSurface, 'getPointAtParam')
    def getPointAtParam(self, paramU, paramV, space='preTransform'):
        # type: (float, float, datatypes.Space.Space) -> datatypes.Point
        do, final_do, outTypes = _f.getDoArgs([paramU, paramV, space], [('paramU', 'double', 'in', None), ('paramV', 'double', 'in', None), ('point', 'MPoint', 'out', None), ('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'getPointAtParam', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnNurbsSurface, 'getTangents')
    def getTangents(self, paramInU, paramInV, space='preTransform'):
        # type: (float, float, datatypes.Space.Space) -> Tuple[datatypes.Vector, datatypes.Vector]
        do, final_do, outTypes = _f.getDoArgs([paramInU, paramInV, space], [('paramInU', 'double', 'in', None), ('paramInV', 'double', 'in', None), ('vectorInU', 'MVector', 'out', None), ('vectorInV', 'MVector', 'out', None), ('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'getTangents', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnNurbsSurface, 'hasHistoryOnCreate')
    def hasHistoryOnCreate(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'hasHistoryOnCreate')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnNurbsSurface, 'intersect')
    def intersect(self, rayStartingPoint, alongThisDirection, tolerance=0.001, space='preTransform', calculateDistance=False, calculateExactHit=False):
        # type: (datatypes.Point, datatypes.Vector, float, datatypes.Space.Space, bool, bool) -> Tuple[bool, float, float, datatypes.Point, float, bool]
        do, final_do, outTypes = _f.getDoArgs([rayStartingPoint, alongThisDirection, tolerance, space, calculateDistance, calculateExactHit], [('rayStartingPoint', 'MPoint', 'in', None), ('alongThisDirection', 'MVector', 'in', None), ('u', 'double', 'out', None), ('v', 'double', 'out', None), ('pntOfIntersection', 'MPoint', 'out', None), ('tolerance', 'double', 'in', None), ('space', ('MSpace', 'Space'), 'in', None), ('calculateDistance', 'bool', 'in', None), ('distance', 'double', 'out', None), ('calculateExactHit', 'bool', 'in', None), ('wasExactHit', 'bool', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'intersect', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnNurbsSurface, 'isBezier')
    def isBezier(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'isBezier')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnNurbsSurface, 'isFlipNorm')
    def isFlipNorm(self, region):
        # type: (int) -> bool
        do, final_do, outTypes = _f.getDoArgs([region], [('region', 'uint', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'isFlipNorm', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res

    @_f.addApiDocs(_api.MFnNurbsSurface, 'isFoldedOnBispan')
    def isFoldedOnBispan(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'isFoldedOnBispan')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnNurbsSurface, 'isKnotU')
    def isKnotU(self, param):
        # type: (float) -> bool
        do, final_do, outTypes = _f.getDoArgs([param], [('param', 'double', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'isKnotU', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res

    @_f.addApiDocs(_api.MFnNurbsSurface, 'isKnotV')
    def isKnotV(self, param):
        # type: (float) -> bool
        do, final_do, outTypes = _f.getDoArgs([param], [('param', 'double', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'isKnotV', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res

    @_f.addApiDocs(_api.MFnNurbsSurface, 'isParamOnSurface')
    def isParamOnSurface(self, paramU, paramV):
        # type: (float, float) -> bool
        do, final_do, outTypes = _f.getDoArgs([paramU, paramV], [('paramU', 'double', 'in', None), ('paramV', 'double', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'isParamOnSurface', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res

    @_f.addApiDocs(_api.MFnNurbsSurface, 'isPointInTrimmedRegion')
    def isPointInTrimmedRegion(self, u, v):
        # type: (float, float) -> bool
        do, final_do, outTypes = _f.getDoArgs([u, v], [('u', 'double', 'in', None), ('v', 'double', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'isPointInTrimmedRegion', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res

    @_f.addApiDocs(_api.MFnNurbsSurface, 'isPointOnSurface')
    def isPointOnSurface(self, point, tolerance=0.001, space='preTransform'):
        # type: (datatypes.Point, float, datatypes.Space.Space) -> bool
        do, final_do, outTypes = _f.getDoArgs([point, tolerance, space], [('point', 'MPoint', 'in', None), ('tolerance', 'double', 'in', None), ('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'isPointOnSurface', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res

    @_f.addApiDocs(_api.MFnNurbsSurface, 'isTrimmedSurface')
    def isTrimmedSurface(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'isTrimmedSurface')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnNurbsSurface, 'isUniform')
    def isUniform(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'isUniform')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnNurbsSurface, 'normal')
    def normal(self, paramInU, paramInV, space='preTransform'):
        # type: (float, float, datatypes.Space.Space) -> datatypes.Vector
        do, final_do, outTypes = _f.getDoArgs([paramInU, paramInV, space], [('paramInU', 'double', 'in', None), ('paramInV', 'double', 'in', None), ('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'normal', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MVector', None)
        return res

    @_f.addApiDocs(_api.MFnNurbsSurface, 'numBoundaries')
    def numBoundaries(self, region):
        # type: (int) -> int
        do, final_do, outTypes = _f.getDoArgs([region], [('region', 'uint', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'numBoundaries', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'uint', None)
        return res

    @_f.addApiDocs(_api.MFnNurbsSurface, 'numEdges')
    def numEdges(self, region, boundary):
        # type: (int, int) -> int
        do, final_do, outTypes = _f.getDoArgs([region, boundary], [('region', 'uint', 'in', None), ('boundary', 'uint', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'numEdges', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'uint', None)
        return res

    @_f.addApiDocs(_api.MFnNurbsSurface, 'numKnotsInU')
    def numKnotsInU(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'numKnotsInU')
        return _f.ApiArgUtil._castResult(self, res, 'int', None)

    @_f.addApiDocs(_api.MFnNurbsSurface, 'numKnotsInV')
    def numKnotsInV(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'numKnotsInV')
        return _f.ApiArgUtil._castResult(self, res, 'int', None)

    @_f.addApiDocs(_api.MFnNurbsSurface, 'numNonZeroSpansInU')
    def numNonZeroSpansInU(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'numNonZeroSpansInU')
        return _f.ApiArgUtil._castResult(self, res, 'int', None)

    @_f.addApiDocs(_api.MFnNurbsSurface, 'numNonZeroSpansInV')
    def numNonZeroSpansInV(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'numNonZeroSpansInV')
        return _f.ApiArgUtil._castResult(self, res, 'int', None)

    @_f.addApiDocs(_api.MFnNurbsSurface, 'numPatches')
    def numPatches(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'numPatches')
        return _f.ApiArgUtil._castResult(self, res, 'uint', None)

    @_f.addApiDocs(_api.MFnNurbsSurface, 'numPatchesInU')
    def numPatchesInU(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'numPatchesInU')
        return _f.ApiArgUtil._castResult(self, res, 'uint', None)

    @_f.addApiDocs(_api.MFnNurbsSurface, 'numPatchesInV')
    def numPatchesInV(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'numPatchesInV')
        return _f.ApiArgUtil._castResult(self, res, 'uint', None)

    @_f.addApiDocs(_api.MFnNurbsSurface, 'numRegions')
    def numRegions(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'numRegions')
        return _f.ApiArgUtil._castResult(self, res, 'uint', None)

    @_f.addApiDocs(_api.MFnNurbsSurface, 'numSpansInU')
    def numSpansInU(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'numSpansInU')
        return _f.ApiArgUtil._castResult(self, res, 'int', None)

    @_f.addApiDocs(_api.MFnNurbsSurface, 'numSpansInV')
    def numSpansInV(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'numSpansInV')
        return _f.ApiArgUtil._castResult(self, res, 'int', None)

    @_f.addApiDocs(_api.MFnNurbsSurface, 'numUVs')
    def numUVs(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'numUVs')
        return _f.ApiArgUtil._castResult(self, res, 'int', None)

    @_f.addApiDocs(_api.MFnNurbsSurface, 'projectCurve')
    def projectCurve(self, curve, direction=None, constructionHistory=False):
        # type: (DagNode, datatypes.Vector, bool) -> None
        do, final_do, outTypes = _f.getDoArgs([curve, direction, constructionHistory], [('curve', 'MDagPath', 'in', None), ('direction', 'MVector', 'in', None), ('constructionHistory', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'projectCurve', final_do)
        return res

    @_f.addApiDocs(_api.MFnNurbsSurface, 'removeKnotInU')
    def removeKnotInU(self, atThisParam, removeAll=False):
        # type: (float, bool) -> None
        do, final_do, outTypes = _f.getDoArgs([atThisParam, removeAll], [('atThisParam', 'double', 'in', None), ('removeAll', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'removeKnotInU', final_do)
        return res

    @_f.addApiDocs(_api.MFnNurbsSurface, 'removeKnotInV')
    def removeKnotInV(self, atThisParam, removeAll=False):
        # type: (float, bool) -> None
        do, final_do, outTypes = _f.getDoArgs([atThisParam, removeAll], [('atThisParam', 'double', 'in', None), ('removeAll', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'removeKnotInV', final_do)
        return res

    @_f.addApiDocs(_api.MFnNurbsSurface, 'removeOneKnotInU')
    def removeOneKnotInU(self, atThisParam):
        # type: (float) -> None
        do, final_do, outTypes = _f.getDoArgs([atThisParam], [('atThisParam', 'double', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'removeOneKnotInU', final_do)
        return res

    @_f.addApiDocs(_api.MFnNurbsSurface, 'removeOneKnotInV')
    def removeOneKnotInV(self, atThisParam):
        # type: (float) -> None
        do, final_do, outTypes = _f.getDoArgs([atThisParam], [('atThisParam', 'double', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'removeOneKnotInV', final_do)
        return res

    @_f.addApiDocs(_api.MFnNurbsSurface, 'setCV')
    def setCV(self, indexU, indexV, pt, space='preTransform'):
        # type: (int, int, datatypes.Point, datatypes.Space.Space) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([indexU, indexV, pt, space], [('indexU', 'uint', 'in', None), ('indexV', 'uint', 'in', None), ('pt', 'MPoint', 'in', None), ('space', ('MSpace', 'Space'), 'in', None)], self.getCV, self.setCV, ['indexU', 'indexV', 'space'])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'setCV', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnNurbsSurface, 'setCVs')
    def setCVs(self, array, space='preTransform'):
        # type: (List[datatypes.Point], datatypes.Space.Space) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([array, space], [('array', 'MPointArray', 'in', None), ('space', ('MSpace', 'Space'), 'in', None)], self.getCVs, self.setCVs, ['space'])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'setCVs', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnNurbsSurface, 'setKnotInU')
    def setKnotInU(self, index, param):
        # type: (int, float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([index, param], [('index', 'int', 'in', None), ('param', 'double', 'in', None)], self.getKnotInU, self.setKnotInU, ['index'])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'setKnotInU', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnNurbsSurface, 'setKnotInV')
    def setKnotInV(self, index, param):
        # type: (int, float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([index, param], [('index', 'uint', 'in', None), ('param', 'double', 'in', None)], self.getKnotInV, self.setKnotInV, ['index'])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'setKnotInV', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnNurbsSurface, 'setKnotsInU')
    def setKnotsInU(self, array, startIndex, endIndex):
        # type: (List[float], int, int) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([array, startIndex, endIndex], [('array', 'MDoubleArray', 'in', None), ('startIndex', 'uint', 'in', None), ('endIndex', 'uint', 'in', None)], self.getKnotsInU, self.setKnotsInU, [])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'setKnotsInU', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnNurbsSurface, 'setKnotsInV')
    def setKnotsInV(self, array, startIndex, endIndex):
        # type: (List[float], int, int) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([array, startIndex, endIndex], [('array', 'MDoubleArray', 'in', None), ('startIndex', 'uint', 'in', None), ('endIndex', 'uint', 'in', None)], self.getKnotsInV, self.setKnotsInV, [])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'setKnotsInV', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnNurbsSurface, 'tesselate')
    def tesselate(self, parms='MTesselationParams::fsDefaultTesselationParams', parentOrOwner='MObject::kNullObj'):
        # type: (datatypes.TesselationParams, str | DependNode) -> DependNode
        do, final_do, outTypes = _f.getDoArgs([parms, parentOrOwner], [('parms', 'MTesselationParams', 'in', None), ('parentOrOwner', 'MObject', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'tesselate', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MObject', None)
        return res

    @_f.addApiDocs(_api.MFnNurbsSurface, 'trim')
    def trim(self, locatorU, locatorV, constructionHistory=False):
        # type: (List[float], List[float], bool) -> None
        do, final_do, outTypes = _f.getDoArgs([locatorU, locatorV, constructionHistory], [('locatorU', 'MDoubleArray', 'in', None), ('locatorV', 'MDoubleArray', 'in', None), ('constructionHistory', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'trim', final_do)
        return res

    @_f.addApiDocs(_api.MFnNurbsSurface, 'updateSurface')
    def updateSurface(self):
        # type: () -> None
        res = _f.getProxyResult(self, _api.MFnNurbsSurface, 'updateSurface')
        return res
# ------ Do not edit above this line --------



if not _factories.building:
    # keep this safe to load if the templates have not been built yet

    # apiToMelBridge maps MFnNurbsCurve._numCVsInU => NurbsCurve._numCVsInUApi
    NurbsSurface.numCVsInU = \
        NurbsSurface._numCVsFunc_generator(NurbsSurface.formInU,
                                           NurbsSurface._numCVsInUApi,
                                           NurbsSurface.numSpansInU,
                                           name='numCVsInU',
                                           doc="""
            Returns the number of CVs in the U direction.
    
            :Parameters:
            editableOnly : `bool`
                If editableOnly evaluates to True (default), then this will return
                the number of cvs that can be actually edited (and also the highest
                index that may be used for u - ie, if
                    mySurf.numCVsInU(editableOnly=True) == 4
                then allowable u indices go from
                    mySurf.cv[0][*] to mySurf.cv[3][*]
    
                If editablyOnly is False, then this will return the underlying
                number of cvs used to define the mathematical curve in u -
                degreeU + numSpansInU.
    
                These will only differ if the form in u is 'periodic', in which
                case the editable number will be numSpansInU (as the last 'degree'
                cv's are 'locked' to be the same as the first 'degree' cvs).
                In all other cases, the number of cvs will be degreeU + numSpansInU.
    
            :Examples:
                >>> from pymel.core import *
                >>> # a periodic surface
                >>> mySurf = surface(name='periodicSurf1', du=3, dv=1, fu='periodic', fv='open', ku=(0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12), kv=(0, 1), pw=[(4, -4, 0, 1), (4, -4, -2.5, 1), (5.5, 0, 0, 1), (5.5, 0, -2.5, 1), (4, 4, 0, 1), (4, 4, -2.5, 1), (0, 5.5, 0, 1), (0, 5.5, -2.5, 1), (-4, 4, 0, 1), (-4, 4, -2.5, 1), (-5.5, 0, 0, 1), (-5.5, 0, -2.5, 1), (-4, -4, 0, 1), (-4, -4, -2.5, 1), (0, -5.5, 0, 1), (0, -5.5, -2.5, 1), (4, -4, 0, 1), (4, -4, -2.5, 1), (5.5, 0, 0, 1), (5.5, 0, -2.5, 1), (4, 4, 0, 1), (4, 4, -2.5, 1)] )
                >>> sorted(mySurf.cv[:][0].indices())        # doctest: +ELLIPSIS
                [ComponentIndex((0, 0), ... ComponentIndex((7, 0), label=None)]
                >>> mySurf.numCVsInU()
                8
                >>> mySurf.numCVsInU(editableOnly=False)
                11
                >>>
                >>> # an open surface
                >>> mySurf = surface(name='openSurf1', du=3, dv=1, fu='open', fv='open', ku=(0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12), kv=(0, 1), pw=((4, -4, 0, 1), (4, -4, -2.5, 1), (5.5, 0, 0, 1), (5.5, 0, -2.5, 1), (4, 4, 0, 1), (4, 4, -2.5, 1), (0, 5.5, 0, 1), (0, 5.5, -2.5, 1), (-4, 4, 0, 1), (-4, 4, -2.5, 1), (-5.5, 0, 0, 1), (-5.5, 0, -2.5, 1), (-4, -4, 0, 1), (-4, -4, -2.5, 1), (0, -5.5, 0, 1), (0, -5.5, -2.5, 1), (4, -4, 0, 1), (4, -4, -2.5, 1), (5.5, 0, 0, 1), (5.5, 0, -2.5, 1), (4, 4, 0, 1), (4, 4, -2.5, 1)) )
                >>> sorted(mySurf.cv[:][0].indices())        # doctest: +ELLIPSIS
                [ComponentIndex((0, 0), ... ComponentIndex((10, 0), label=None)]
                >>> mySurf.numCVsInU()
                11
                >>> mySurf.numCVsInU(editableOnly=False)
                11
    
            :rtype: `int`
            """)

    # apiToMelBridge maps MFnNurbsCurve._numCVsInV => NurbsCurve._numCVsInVApi
    NurbsSurface.numCVsInV = \
        NurbsSurface._numCVsFunc_generator(NurbsSurface.formInV,
                                           NurbsSurface._numCVsInVApi,
                                           NurbsSurface.numSpansInV,
                                           name='numCVsInV',
                                           doc="""
            Returns the number of CVs in the V direction.
    
            :Parameters:
            editableOnly : `bool`
                If editableOnly evaluates to True (default), then this will return
                the number of cvs that can be actually edited (and also the highest
                index that may be used for v - ie, if
                    mySurf.numCVsInV(editableOnly=True) == 4
                then allowable v indices go from
                    mySurf.cv[*][0] to mySurf.cv[*][3]
    
                If editablyOnly is False, then this will return the underlying
                number of cvs used to define the mathematical curve in v -
                degreeV + numSpansInV.
    
                These will only differ if the form in v is 'periodic', in which
                case the editable number will be numSpansInV (as the last 'degree'
                cv's are 'locked' to be the same as the first 'degree' cvs).
                In all other cases, the number of cvs will be degreeV + numSpansInV.
    
            :Examples:
                >>> from pymel.core import *
                >>> # a periodic surface
                >>> mySurf = surface(name='periodicSurf2', du=1, dv=3, fu='open', fv='periodic', ku=(0, 1), kv=(0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12), pw=[(4, -4, 0, 1), (5.5, 0, 0, 1), (4, 4, 0, 1), (0, 5.5, 0, 1), (-4, 4, 0, 1), (-5.5, 0, 0, 1), (-4, -4, 0, 1), (0, -5.5, 0, 1), (4, -4, 0, 1), (5.5, 0, 0, 1), (4, 4, 0, 1), (4, -4, -2.5, 1), (5.5, 0, -2.5, 1), (4, 4, -2.5, 1), (0, 5.5, -2.5, 1), (-4, 4, -2.5, 1), (-5.5, 0, -2.5, 1), (-4, -4, -2.5, 1), (0, -5.5, -2.5, 1), (4, -4, -2.5, 1), (5.5, 0, -2.5, 1), (4, 4, -2.5, 1)] )
                >>> sorted(mySurf.cv[0].indices())         # doctest: +ELLIPSIS
                [ComponentIndex((0, 0), ... ComponentIndex((0, 7), label='cv')]
                >>> mySurf.numCVsInV()
                8
                >>> mySurf.numCVsInV(editableOnly=False)
                11
                >>>
                >>> # an open surface
                >>> mySurf = surface(name='openSurf2', du=1, dv=3, fu='open', fv='open', ku=(0, 1), kv=(0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12), pw=[(4, -4, 0, 1), (5.5, 0, 0, 1), (4, 4, 0, 1), (0, 5.5, 0, 1), (-4, 4, 0, 1), (-5.5, 0, 0, 1), (-4, -4, 0, 1), (0, -5.5, 0, 1), (4, -4, 0, 1), (5.5, 0, 0, 1), (4, 4, 0, 1), (4, -4, -2.5, 1), (5.5, 0, -2.5, 1), (4, 4, -2.5, 1), (0, 5.5, -2.5, 1), (-4, 4, -2.5, 1), (-5.5, 0, -2.5, 1), (-4, -4, -2.5, 1), (0, -5.5, -2.5, 1), (4, -4, -2.5, 1), (5.5, 0, -2.5, 1), (4, 4, -2.5, 1)] )
                >>> sorted(mySurf.cv[0].indices())          # doctest: +ELLIPSIS
                [ComponentIndex((0, 0), ... ComponentIndex((0, 10), label='cv')]
                >>> mySurf.numCVsInV()
                11
                >>> mySurf.numCVsInV(editableOnly=False)
                11
    
            :rtype: `int`
            """)

    NurbsSurface.numEPsInU = \
        NurbsSurface._numEPsFunc_generator(NurbsSurface.formInU,
                                           NurbsSurface.numSpansInU,
                                           name='numEPsInU',
                                           doc="""
            Returns the number of EPs in the U direction.
    
            :Examples:
                >>> from pymel.core import *
                >>> # a periodic surface
                >>> mySurf = surface(name='periodicSurf3', du=3, dv=1, fu='periodic', fv='open', ku=(0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12), kv=(0, 1), pw=[(4, -4, 0, 1), (4, -4, -2.5, 1), (5.5, 0, 0, 1), (5.5, 0, -2.5, 1), (4, 4, 0, 1), (4, 4, -2.5, 1), (0, 5.5, 0, 1), (0, 5.5, -2.5, 1), (-4, 4, 0, 1), (-4, 4, -2.5, 1), (-5.5, 0, 0, 1), (-5.5, 0, -2.5, 1), (-4, -4, 0, 1), (-4, -4, -2.5, 1), (0, -5.5, 0, 1), (0, -5.5, -2.5, 1), (4, -4, 0, 1), (4, -4, -2.5, 1), (5.5, 0, 0, 1), (5.5, 0, -2.5, 1), (4, 4, 0, 1), (4, 4, -2.5, 1)] )
                >>> sorted(mySurf.ep[:][0].indices())      # doctest: +ELLIPSIS
                [ComponentIndex((0, 0), ... ComponentIndex((7, 0), label=None)]
                >>> mySurf.numEPsInU()
                8
                >>>
                >>> # an open surface
                >>> mySurf = surface(name='openSurf3', du=3, dv=1, fu='open', fv='open', ku=(0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12), kv=(0, 1), pw=[(4, -4, 0, 1), (4, -4, -2.5, 1), (5.5, 0, 0, 1), (5.5, 0, -2.5, 1), (4, 4, 0, 1), (4, 4, -2.5, 1), (0, 5.5, 0, 1), (0, 5.5, -2.5, 1), (-4, 4, 0, 1), (-4, 4, -2.5, 1), (-5.5, 0, 0, 1), (-5.5, 0, -2.5, 1), (-4, -4, 0, 1), (-4, -4, -2.5, 1), (0, -5.5, 0, 1), (0, -5.5, -2.5, 1), (4, -4, 0, 1), (4, -4, -2.5, 1), (5.5, 0, 0, 1), (5.5, 0, -2.5, 1), (4, 4, 0, 1), (4, 4, -2.5, 1)] )
                >>> sorted(mySurf.ep[:][0].indices())      # doctest: +ELLIPSIS
                [ComponentIndex((0, 0), ... ComponentIndex((8, 0), label=None)]
                >>> mySurf.numEPsInU()
                9
    
            :rtype: `int`
            """)

    NurbsSurface.numEPsInV = \
        NurbsSurface._numEPsFunc_generator(NurbsSurface.formInV,
                                           NurbsSurface.numSpansInV,
                                           name='numEPsInV',
                                           doc="""
            Returns the number of EPs in the V direction.
    
            :Examples:
                >>> from pymel.core import *
                >>> # a periodic surface
                >>> mySurf = surface(name='periodicSurf4', du=1, dv=3, fu='open', fv='periodic', ku=(0, 1), kv=(0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12), pw=[(4, -4, 0, 1), (5.5, 0, 0, 1), (4, 4, 0, 1), (0, 5.5, 0, 1), (-4, 4, 0, 1), (-5.5, 0, 0, 1), (-4, -4, 0, 1), (0, -5.5, 0, 1), (4, -4, 0, 1), (5.5, 0, 0, 1), (4, 4, 0, 1), (4, -4, -2.5, 1), (5.5, 0, -2.5, 1), (4, 4, -2.5, 1), (0, 5.5, -2.5, 1), (-4, 4, -2.5, 1), (-5.5, 0, -2.5, 1), (-4, -4, -2.5, 1), (0, -5.5, -2.5, 1), (4, -4, -2.5, 1), (5.5, 0, -2.5, 1), (4, 4, -2.5, 1)] )
                >>> sorted(mySurf.ep[0][:].indices())      # doctest: +ELLIPSIS
                [ComponentIndex((0, 0), ... ComponentIndex((0, 7), label=None)]
                >>> mySurf.numEPsInV()
                8
                >>>
                >>> # an open surface
                >>> mySurf = surface(name='openSurf4', du=1, dv=3, fu='open', fv='open', ku=(0, 1), kv=(0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12), pw=[(4, -4, 0, 1), (5.5, 0, 0, 1), (4, 4, 0, 1), (0, 5.5, 0, 1), (-4, 4, 0, 1), (-5.5, 0, 0, 1), (-4, -4, 0, 1), (0, -5.5, 0, 1), (4, -4, 0, 1), (5.5, 0, 0, 1), (4, 4, 0, 1), (4, -4, -2.5, 1), (5.5, 0, -2.5, 1), (4, 4, -2.5, 1), (0, 5.5, -2.5, 1), (-4, 4, -2.5, 1), (-5.5, 0, -2.5, 1), (-4, -4, -2.5, 1), (0, -5.5, -2.5, 1), (4, -4, -2.5, 1), (5.5, 0, -2.5, 1), (4, 4, -2.5, 1)] )
                >>> sorted(mySurf.ep[0][:].indices())      # doctest: +ELLIPSIS
                [ComponentIndex((0, 0), ... ComponentIndex((0, 8), label=None)]
                >>> mySurf.numEPsInV()
                9
    
            :rtype: `int`
            """)


class Mesh(SurfaceShape):

    """
    The Mesh class provides wrapped access to many API methods for querying and modifying meshes.  Be aware that
    modifying meshes using API commands outside of the context of a plugin is still somewhat uncharted territory,
    so proceed at our own risk.


    The component types can be accessed from the `Mesh` type (or it's transform) using the names you are
    familiar with from MEL:

        >>> from pymel.core import *
        >>> p = polySphere( name='theMoon', sa=7, sh=7 )[0]
        >>> p.vtx
        MeshVertex('theMoonShape.vtx[0:43]')
        >>> p.e
        MeshEdge('theMoonShape.e[0:90]')
        >>> p.f
        MeshFace('theMoonShape.f[0:48]')

    They are also accessible from their more descriptive alternatives:

        >>> p.verts
        MeshVertex('theMoonShape.vtx[0:43]')
        >>> p.edges
        MeshEdge('theMoonShape.e[0:90]')
        >>> p.faces
        MeshFace('theMoonShape.f[0:48]')

    As you'd expect, these components are all indexible:

        >>> p.vtx[0]
        MeshVertex('theMoonShape.vtx[0]')

    The classes themselves contain methods for getting information about the component.

        >>> p.vtx[0].connectedEdges()
        MeshEdge('theMoonShape.e[0,6,42,77]')

    This class provides support for python's extended slice notation. Typical maya ranges express a start and stop value separated
    by a colon.  Extended slices add a step parameter and can also represent multiple ranges separated by commas.
    Thus, a single component object can represent any collection of indices.

    This includes start, stop, and step values.

        >>> # do every other edge between 0 and 10
        >>> for edge in p.e[0:10:2]:
        ...     print(edge)
        ...
        theMoonShape.e[0]
        theMoonShape.e[2]
        theMoonShape.e[4]
        theMoonShape.e[6]
        theMoonShape.e[8]
        theMoonShape.e[10]

    Negative indices can be used for getting indices relative to the end:

        >>> p.edges  # the full range
        MeshEdge('theMoonShape.e[0:90]')
        >>> p.edges[5:-10]  # index 5 through to 10 from the last
        MeshEdge('theMoonShape.e[5:80]')

    Just like with python ranges, you can leave an index out, and the logical result will follow:

        >>> p.edges[:-10]  # from the beginning
        MeshEdge('theMoonShape.e[0:80]')
        >>> p.edges[20:]
        MeshEdge('theMoonShape.e[20:90]')

    Or maybe you want the position of every tenth vert:

        >>> for x in p.vtx[::10]:
        ...     print(x, x.getPosition())
        ...
        theMoonShape.vtx[0] [0.2705221176..., -0.9009688496..., -0.3392239511...]
        theMoonShape.vtx[10] [-0.7044058442..., -0.6234897971..., 0.3392239511...]
        theMoonShape.vtx[20] [0.9749279022..., -0.2225208580..., 0.0]
        theMoonShape.vtx[30] [-0.7044057846..., 0.6234897971..., -0.3392240107...]
        theMoonShape.vtx[40] [0.2705220878..., 0.9009688496..., 0.3392239809...]


    To be compatible with Maya's range notation, these slices are inclusive of the stop index.

        >>> # face at index 8 will be included in the sequence
        >>> for f in p.f[4:8]: print(f)
        ...
        theMoonShape.f[4]
        theMoonShape.f[5]
        theMoonShape.f[6]
        theMoonShape.f[7]
        theMoonShape.f[8]

    >>> from pymel.core import *
    >>> obj = polyTorus()[0]
    >>> colors = []
    >>> for i, vtx in enumerate(obj.vtx):   # doctest: +SKIP
    ...     edgs=vtx.toEdges()              # doctest: +SKIP
    ...     totalLen=0                      # doctest: +SKIP
    ...     edgCnt=0                        # doctest: +SKIP
    ...     for edg in edgs:                # doctest: +SKIP
    ...         edgCnt += 1                 # doctest: +SKIP
    ...         l = edg.getLength()         # doctest: +SKIP
    ...         totalLen += l               # doctest: +SKIP
    ...     avgLen=totalLen / edgCnt        # doctest: +SKIP
    ...     #print avgLen                   # doctest: +SKIP
    ...     currColor = vtx.getColor(0)     # doctest: +SKIP
    ...     color = datatypes.Color.black   # doctest: +SKIP
    ...     # only set blue if it has not been set before
    ...     if currColor.b<=0.0:            # doctest: +SKIP
    ...         color.b = avgLen            # doctest: +SKIP
    ...     color.r = avgLen                # doctest: +SKIP
    ...     colors.append(color)            # doctest: +SKIP


    """
#    def __init__(self, *args, **kwargs ):
#        SurfaceShape.__init__(self, self._apiobject )
#        self.vtx = MeshEdge(self.__apimobject__() )
    _componentAttributes = {
        'vtx': general.MeshVertex,
        'verts': general.MeshVertex,
        'e': general.MeshEdge,
        'edges': general.MeshEdge,
        'f': general.MeshFace,
        'faces': general.MeshFace,
        'map': general.MeshUV,
        'uvs': general.MeshUV,
        'vtxFace': general.MeshVertexFace,
        'faceVerts': general.MeshVertexFace,
    }

    if TYPE_CHECKING:
        # FIXME: move to build
        vtx = None  # type: general.MeshVertex
        verts = None  # type: general.MeshVertex
        e = None  # type: general.MeshEdge
        edges = None  # type: general.MeshEdge
        f = None  # type: general.MeshFace
        faces = None  # type: general.MeshFace
        map = None  # type: general.MeshUV
        uvs = None  # type: general.MeshUV
        vtxFace = None  # type: general.MeshVertexFace
        faceVerts = None  # type: general.MeshVertexFace

    # Unfortunately, objects that don't yet have any mesh data - ie, if you do
    # createNode('mesh') - can't be fed into MFnMesh (even though it is a mesh
    # node).  This means that all the methods wrapped from MFnMesh won't be
    # usable in this case.  While it might make sense for some methods - ie,
    # editing methods like collapseEdges - to fail in this situation, some
    # basic methods like numVertices should still be usable.  Therefore,
    # we override some of these with the mel versions (which still work...)
    numVertices = _factories.makeCreateFlagMethod(cmds.polyEvaluate, 'vertex', 'numVertices')
    numEdges = _factories.makeCreateFlagMethod(cmds.polyEvaluate, 'edge', 'numEdges')
    numFaces = _factories.makeCreateFlagMethod(cmds.polyEvaluate, 'face', 'numFaces')

    numTriangles = _factories.makeCreateFlagMethod(cmds.polyEvaluate, 'triangle', 'numTriangles')
    numSelectedTriangles = _factories.makeCreateFlagMethod(cmds.polyEvaluate, 'triangleComponent', 'numSelectedTriangles')
    numSelectedFaces = _factories.makeCreateFlagMethod(cmds.polyEvaluate, 'faceComponent', 'numSelectedFaces')
    numSelectedEdges = _factories.makeCreateFlagMethod(cmds.polyEvaluate, 'edgeComponent', 'numSelectedEdges')
    numSelectedVertices = _factories.makeCreateFlagMethod(cmds.polyEvaluate, 'vertexComponent', 'numSelectedVertices')

    area = _factories.makeCreateFlagMethod(cmds.polyEvaluate, 'area')
    worldArea = _factories.makeCreateFlagMethod(cmds.polyEvaluate, 'worldArea')

    @_factories.addApiDocs(_api.MFnMesh, 'getUVAtPoint')
    def getUVAtPoint(self, uvPoint, space=_api.MSpace.kObject, uvSet=None, returnClosestPolygon=False):
        result = self._getUVAtPoint(uvPoint, space, uvSet)
        if returnClosestPolygon:
            return result
        return result[0]

    @_factories.addApiDocs(_api.MFnMesh, 'currentUVSetName')
    def getCurrentUVSetName(self):
        return self.__apimfn__().currentUVSetName(self.instanceNumber())

    @_factories.addApiDocs(_api.MFnMesh, 'currentColorSetName')
    def getCurrentColorSetName(self):
        return self.__apimfn__().currentColorSetName(self.instanceNumber())

    @_factories.addApiDocs(_api.MFnMesh, 'currentColorSetName')
    def getVertexColors(self, colorSet=None, defaultUnsetColor=None):
        '''
        Modifications:
        - raises more informative error if colorSet is missing (or are none)
        '''
        try:
            return self._getVertexColors(colorSet=colorSet,
                                         defaultUnsetColor=defaultUnsetColor)
        except RuntimeError:
            if colorSet is None:
                if self.numColorSets() == 0:
                    raise RuntimeError('mesh {} had no color sets'
                                       .format(self))
            elif colorSet not in self.getColorSetNames():
                raise RuntimeError('mesh {} had no color set named {}'.
                                   format(self, colorSet))
            raise

    @_factories.addApiDocs(_api.MFnMesh, 'numColors')
    def numColors(self, colorSet=None):
        mfn = self.__apimfn__()
        # If we have an empty mesh, we will get an MFnDagNode...
        if not isinstance(mfn, _api.MFnMesh):
            return 0
        args = []
        if colorSet:
            args.append(colorSet)
        return mfn.numColors(*args)
# ------ Do not edit below this line --------
    __apicls__ = _api.MFnMesh
    __melnode__ = 'mesh'
    __slots__ = ()
    if versions.current() >= 20240000:
        BoolClassification = Enum('BoolClassification', [('edgeClassification', 1), ('kEdgeClassification', 1), ('normalClassification', 2), ('kNormalClassification', 2)], multiKeys=True)
    BoolOperation = Enum('BoolOperation', [('union', 1), ('kUnion', 1), ('difference', 2), ('kDifference', 2), ('intersection', 3), ('kIntersection', 3)], multiKeys=True)
    if versions.current() >= versions.v2023:
        BorderInfo = Enum('BorderInfo', [('geomBorder', -2), ('kGeomBorder', -2), ('UVBorder', -1), ('kUVBorder', -1), ('sharedUV', 0), ('kSharedUV', 0), ('unsharedUV', 1), ('kUnsharedUV', 1)], multiKeys=True)
    MColorRepresentation = Enum('MColorRepresentation', [('alpha', 1), ('kAlpha', 1), ('RGB', 3), ('kRGB', 3), ('RGBA', 4), ('kRGBA', 4)], multiKeys=True)
    SplitPlacement = Enum('SplitPlacement', [('onEdge', 0), ('kOnEdge', 0), ('internalPoint', 1), ('kInternalPoint', 1), ('invalid', 2), ('kInvalid', 2)], multiKeys=True)

    @_f.addApiDocs(_api.MFnMesh, 'getUVAtPoint')
    def _getUVAtPoint(self, pt, space='preTransform', uvSet=None):
        # type: (datatypes.Point, datatypes.Space.Space, str) -> Tuple[Tuple[float, float], int]
        do, final_do, outTypes = _f.getDoArgs([pt, space, uvSet], [('pt', 'MPoint', 'in', None), ('uvPoint', 'float2', 'out', None), ('space', ('MSpace', 'Space'), 'in', None), ('uvSet', 'MString', 'in', None), ('closestPolygon', 'int', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getUVAtPoint', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getVertexColors')
    def _getVertexColors(self, colorSet=None, defaultUnsetColor=None):
        # type: (str, datatypes.Color) -> List[datatypes.Color]
        do, final_do, outTypes = _f.getDoArgs([colorSet, defaultUnsetColor], [('colors', 'MColorArray', 'out', None), ('colorSet', 'MString', 'in', None), ('defaultUnsetColor', 'MColor', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getVertexColors', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'numColorSets')
    def _numColorSets(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnMesh, 'numColorSets')
        return _f.ApiArgUtil._castResult(self, res, 'int', None)

    @_f.addApiDocs(_api.MFnMesh, 'numColors')
    def _numColors(self, colorSet):
        # type: (str) -> int
        do, final_do, outTypes = _f.getDoArgs([colorSet], [('colorSet', 'MString', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'numColors', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'int', None)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'numEdges')
    def _numEdges(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnMesh, 'numEdges')
        return _f.ApiArgUtil._castResult(self, res, 'int', None)

    @_f.addApiDocs(_api.MFnMesh, 'numFaceVertices')
    def _numFaceVertices(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnMesh, 'numFaceVertices')
        return _f.ApiArgUtil._castResult(self, res, 'int', None)

    @_f.addApiDocs(_api.MFnMesh, 'numNormals')
    def _numNormals(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnMesh, 'numNormals')
        return _f.ApiArgUtil._castResult(self, res, 'int', None)

    @_f.addApiDocs(_api.MFnMesh, 'numUVSets')
    def _numUVSets(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnMesh, 'numUVSets')
        return _f.ApiArgUtil._castResult(self, res, 'int', None)

    @_f.addApiDocs(_api.MFnMesh, 'numUVs')
    def _numUVs(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnMesh, 'numUVs')
        return _f.ApiArgUtil._castResult(self, res, 'int', None)

    @_f.addApiDocs(_api.MFnMesh, 'numVertices')
    def _numVertices(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnMesh, 'numVertices')
        return _f.ApiArgUtil._castResult(self, res, 'int', None)

    @_f.addApiDocs(_api.MFnMesh, 'addHoles')
    def addHoles(self, faceIndex, vertexArray, loopCounts, mergeVertices=True, pointTolerance=1e-10):
        # type: (int, List[datatypes.Point], List[int], bool, float) -> None
        do, final_do, outTypes = _f.getDoArgs([faceIndex, vertexArray, loopCounts, mergeVertices, pointTolerance], [('faceIndex', 'int', 'in', None), ('vertexArray', 'MPointArray', 'in', None), ('loopCounts', 'MIntArray', 'in', None), ('mergeVertices', 'bool', 'in', None), ('pointTolerance', 'double', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'addHoles', final_do)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'assignColor')
    def assignColor(self, polygonId, vertexIndex, colorId, colorSet=None):
        # type: (int, int, int, str) -> None
        do, final_do, outTypes = _f.getDoArgs([polygonId, vertexIndex, colorId, colorSet], [('polygonId', 'int', 'in', None), ('vertexIndex', 'int', 'in', None), ('colorId', 'int', 'in', None), ('colorSet', 'MString', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'assignColor', final_do)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'assignColors')
    def assignColors(self, colorIds, colorSet=None):
        # type: (List[int], str) -> None
        do, final_do, outTypes = _f.getDoArgs([colorIds, colorSet], [('colorIds', 'MIntArray', 'in', None), ('colorSet', 'MString', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'assignColors', final_do)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'assignUV')
    def assignUV(self, polygonId, vertexIndex, uvId, uvSet=None):
        # type: (int, int, int, str) -> None
        do, final_do, outTypes = _f.getDoArgs([polygonId, vertexIndex, uvId, uvSet], [('polygonId', 'int', 'in', None), ('vertexIndex', 'int', 'in', None), ('uvId', 'int', 'in', None), ('uvSet', 'MString', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'assignUV', final_do)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'assignUVs')
    def assignUVs(self, uvCounts, uvIds, uvSet=None):
        # type: (List[int], List[int], str) -> None
        do, final_do, outTypes = _f.getDoArgs([uvCounts, uvIds, uvSet], [('uvCounts', 'MIntArray', 'in', None), ('uvIds', 'MIntArray', 'in', None), ('uvSet', 'MString', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'assignUVs', final_do)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'booleanOps')
    def booleanOps(self, op, meshes, classification='normalClassification'):
        # type: (Mesh.BoolOperation, List[str | DependNode], Mesh.BoolClassification) -> bool
        do, final_do, outTypes = _f.getDoArgs([op, meshes, classification], [('op', ('MFnMesh', 'BoolOperation'), 'in', None), ('meshes', 'MObjectArray', 'in', None), ('useLegacy', 'bool', 'out', None), ('classification', ('MFnMesh', 'BoolClassification'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'booleanOps', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'cleanupEdgeSmoothing')
    def cleanupEdgeSmoothing(self):
        # type: () -> None
        res = _f.getProxyResult(self, _api.MFnMesh, 'cleanupEdgeSmoothing')
        return res

    @_f.addApiDocs(_api.MFnMesh, 'clearColors')
    def clearColors(self, colorSet=None):
        # type: (str) -> None
        do, final_do, outTypes = _f.getDoArgs([colorSet], [('colorSet', 'MString', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'clearColors', final_do)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'clearUVs')
    def clearUVs(self, uvSet=None):
        # type: (str) -> None
        do, final_do, outTypes = _f.getDoArgs([uvSet], [('uvSet', 'MString', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'clearUVs', final_do)
        return res

    @classmethod
    @_f.addApiDocs(_api.MFnMesh, 'componentTypeFromName')
    def componentTypeFromName(self, assocName):
        # type: (str) -> datatypes.Fn.Type
        do, final_do, outTypes = _f.getDoArgs([assocName], [('assocName', 'MString', 'in', None)])
        res = _api.MFnMesh.componentTypeFromName(*final_do)
        res = _f.ApiArgUtil._castResult(self, res, ('MFn', 'Type'), None)
        return res

    @classmethod
    @_f.addApiDocs(_api.MFnMesh, 'componentTypeName')
    def componentTypeName(self, compType):
        # type: (datatypes.Fn.Type) -> str
        do, final_do, outTypes = _f.getDoArgs([compType], [('compType', ('MFn', 'Type'), 'in', None)])
        res = _api.MFnMesh.componentTypeName(*final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MString', None)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'createColorSetDataMesh')
    def createColorSetDataMesh(self, colorSetName, clamped=False, rep='RGBA'):
        # type: (str, bool, Mesh.MColorRepresentation) -> None
        do, final_do, outTypes = _f.getDoArgs([colorSetName, clamped, rep], [('colorSetName', 'MString', 'in', None), ('clamped', 'bool', 'in', None), ('rep', ('MFnMesh', 'MColorRepresentation'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'createColorSetDataMesh', final_do)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'createColorSetWithName')
    def createColorSetWithName(self, colorSetName, modifier=None, instances=None):
        # type: (str, datatypes.DGModifier, datatypes.UintArray) -> str
        do, final_do, outTypes = _f.getDoArgs([colorSetName, modifier, instances], [('colorSetName', 'MString', 'in', None), ('modifier', 'MDGModifier', 'in', None), ('instances', 'MUintArray', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'createColorSetWithName', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MString', None)
        return res
    createColorSet = createColorSetWithName

    @_f.addApiDocs(_api.MFnMesh, 'createColorSetWithNameDataMesh')
    def createColorSetWithNameDataMesh(self, colorSetName, clamped, rep):
        # type: (str, bool, Mesh.MColorRepresentation) -> str
        do, final_do, outTypes = _f.getDoArgs([colorSetName, clamped, rep], [('colorSetName', 'MString', 'in', None), ('clamped', 'bool', 'in', None), ('rep', ('MFnMesh', 'MColorRepresentation'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'createColorSetWithNameDataMesh', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MString', None)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'createUVSetDataMeshWithName')
    def createUVSetDataMeshWithName(self, uvSetName):
        # type: (str) -> str
        do, final_do, outTypes = _f.getDoArgs([uvSetName], [('uvSetName', 'MString', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'createUVSetDataMeshWithName', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MString', None)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'createUVSetWithName')
    def createUVSetWithName(self, uvSetName, modifier=None, instances=None):
        # type: (str, datatypes.DGModifier, datatypes.UintArray) -> str
        do, final_do, outTypes = _f.getDoArgs([uvSetName, modifier, instances], [('uvSetName', 'MString', 'in', None), ('modifier', 'MDGModifier', 'in', None), ('instances', 'MUintArray', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'createUVSetWithName', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MString', None)
        return res
    createUVSet = createUVSetWithName

    @_f.addApiDocs(_api.MFnMesh, 'deleteColorSet')
    def deleteColorSet(self, colorSetName, modifier=None, currentSelection=None):
        # type: (str, datatypes.DGModifier, SelectionSet) -> None
        do, final_do, outTypes = _f.getDoArgs([colorSetName, modifier, currentSelection], [('colorSetName', 'MString', 'in', None), ('modifier', 'MDGModifier', 'in', None), ('currentSelection', 'MSelectionList', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'deleteColorSet', final_do)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'deleteUVSet')
    def deleteUVSet(self, setName, modifier=None, currentSelection=None):
        # type: (str, datatypes.DGModifier, SelectionSet) -> None
        do, final_do, outTypes = _f.getDoArgs([setName, modifier, currentSelection], [('setName', 'MString', 'in', None), ('modifier', 'MDGModifier', 'in', None), ('currentSelection', 'MSelectionList', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'deleteUVSet', final_do)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'edgeBorderInfo')
    def edgeBorderInfo(self, edgeId):
        # type: (int) -> Tuple[Mesh.BorderInfo, int]
        do, final_do, outTypes = _f.getDoArgs([edgeId], [('edgeId', 'int', 'in', None), ('ReturnStatus', 'uint', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'edgeBorderInfo', final_do)
        res = _f.ApiArgUtil._castResult(self, res, ('MFnMesh', 'BorderInfo'), None)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getAssignedUVs')
    def getAssignedUVs(self, uvSet=None):
        # type: (str) -> Tuple[List[int], List[int]]
        do, final_do, outTypes = _f.getDoArgs([uvSet], [('uvCounts', 'MIntArray', 'out', None), ('uvIds', 'MIntArray', 'out', None), ('uvSet', 'MString', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getAssignedUVs', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getAssociatedColorSetInstances')
    def getAssociatedColorSetInstances(self, colorSetName):
        # type: (str) -> List[int]
        do, final_do, outTypes = _f.getDoArgs([colorSetName], [('colorSetName', 'MString', 'in', None), ('instances', 'MIntArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getAssociatedColorSetInstances', final_do)
        return _f.processApiResult(res, outTypes, do)
    _getAssociatedColorSetInstances = getAssociatedColorSetInstances

    @_f.addApiDocs(_api.MFnMesh, 'getAssociatedUVSetInstances')
    def getAssociatedUVSetInstances(self, uvSetName):
        # type: (str) -> List[int]
        do, final_do, outTypes = _f.getDoArgs([uvSetName], [('uvSetName', 'MString', 'in', None), ('instances', 'MIntArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getAssociatedUVSetInstances', final_do)
        return _f.processApiResult(res, outTypes, do)
    _getAssociatedUVSetInstances = getAssociatedUVSetInstances

    @_f.addApiDocs(_api.MFnMesh, 'getAssociatedUVSetTextures')
    def getAssociatedUVSetTextures(self, uvSetName):
        # type: (str) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([uvSetName], [('uvSetName', 'MString', 'in', None), ('textures', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getAssociatedUVSetTextures', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getAxisAtPoint')
    def getAxisAtPoint(self, pt, space='preTransform', uvSet=None):
        # type: (datatypes.Point, datatypes.Space.Space, str) -> Tuple[datatypes.Vector, datatypes.Vector, datatypes.Vector, int]
        do, final_do, outTypes = _f.getDoArgs([pt, space, uvSet], [('pt', 'MPoint', 'in', None), ('normal', 'MVector', 'out', None), ('uTangent', 'MVector', 'out', None), ('vTangent', 'MVector', 'out', None), ('space', ('MSpace', 'Space'), 'in', None), ('uvSet', 'MString', 'in', None), ('closestPolygon', 'int', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getAxisAtPoint', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getBinormals')
    def getBinormals(self, space='preTransform', uvSet=None):
        # type: (datatypes.Space.Space, str) -> List[datatypes.FloatVector]
        do, final_do, outTypes = _f.getDoArgs([space, uvSet], [('binormals', 'MFloatVectorArray', 'out', None), ('space', ('MSpace', 'Space'), 'in', None), ('uvSet', 'MString', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getBinormals', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getCheckSamePointTwice')
    def getCheckSamePointTwice(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnMesh, 'getCheckSamePointTwice')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnMesh, 'getClosestNormal')
    def getClosestNormal(self, toThisPoint, space='preTransform', accelParams=None):
        # type: (datatypes.Point, datatypes.Space.Space, datatypes.MeshIsectAccelParams) -> Tuple[datatypes.Vector, int]
        do, final_do, outTypes = _f.getDoArgs([toThisPoint, space, accelParams], [('toThisPoint', 'MPoint', 'in', None), ('theNormal', 'MVector', 'out', None), ('space', ('MSpace', 'Space'), 'in', None), ('closestPolygon', 'int', 'out', None), ('accelParams', 'MMeshIsectAccelParams', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getClosestNormal', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getClosestPoint')
    def getClosestPoint(self, toThisPoint, space='preTransform', accelParams=None):
        # type: (datatypes.Point, datatypes.Space.Space, datatypes.MeshIsectAccelParams) -> Tuple[datatypes.Point, int]
        do, final_do, outTypes = _f.getDoArgs([toThisPoint, space, accelParams], [('toThisPoint', 'MPoint', 'in', None), ('theClosestPoint', 'MPoint', 'out', None), ('space', ('MSpace', 'Space'), 'in', None), ('closestPolygon', 'int', 'out', None), ('accelParams', 'MMeshIsectAccelParams', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getClosestPoint', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getClosestPointAndNormal')
    def getClosestPointAndNormal(self, toThisPoint, space='preTransform', accelParams=None):
        # type: (datatypes.Point, datatypes.Space.Space, datatypes.MeshIsectAccelParams) -> Tuple[datatypes.Point, datatypes.Vector, int]
        do, final_do, outTypes = _f.getDoArgs([toThisPoint, space, accelParams], [('toThisPoint', 'MPoint', 'in', None), ('theClosestPoint', 'MPoint', 'out', None), ('theNormal', 'MVector', 'out', None), ('space', ('MSpace', 'Space'), 'in', None), ('closestPolygon', 'int', 'out', None), ('accelParams', 'MMeshIsectAccelParams', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getClosestPointAndNormal', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getClosestUVs')
    def getClosestUVs(self, toThisUVPoint, uvSet=None):
        # type: (Tuple[float, float], str) -> List[int]
        do, final_do, outTypes = _f.getDoArgs([toThisUVPoint, uvSet], [('toThisUVPoint', 'float2', 'in', None), ('uvIds', 'MIntArray', 'out', None), ('uvSet', 'MString', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getClosestUVs', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getColor')
    def getColor(self, colorId, colorSet=None, defaultUnsetColor=None):
        # type: (int, str, datatypes.Color) -> datatypes.Color
        do, final_do, outTypes = _f.getDoArgs([colorId, colorSet, defaultUnsetColor], [('colorId', 'int', 'in', None), ('color', 'MColor', 'out', None), ('colorSet', 'MString', 'in', None), ('defaultUnsetColor', 'MColor', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getColor', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getColorRepresentation')
    def getColorRepresentation(self, colorSet):
        # type: (str) -> Mesh.MColorRepresentation
        do, final_do, outTypes = _f.getDoArgs([colorSet], [('colorSet', 'MString', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getColorRepresentation', final_do)
        res = _f.ApiArgUtil._castResult(self, res, ('MFnMesh', 'MColorRepresentation'), None)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'getColorSetFamilyNames')
    def getColorSetFamilyNames(self):
        # type: () -> List[str]
        do, final_do, outTypes = _f.getDoArgs([], [('familyNames', 'MStringArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getColorSetFamilyNames', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getColorSetNames')
    def getColorSetNames(self):
        # type: () -> List[str]
        do, final_do, outTypes = _f.getDoArgs([], [('setNames', 'MStringArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getColorSetNames', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getColors')
    def getColors(self, colorSet=None, defaultUnsetColor=None):
        # type: (str, datatypes.Color) -> List[datatypes.Color]
        do, final_do, outTypes = _f.getDoArgs([colorSet, defaultUnsetColor], [('colorArray', 'MColorArray', 'out', None), ('colorSet', 'MString', 'in', None), ('defaultUnsetColor', 'MColor', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getColors', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'displayColors')
    def getDisplayColors(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnMesh, 'displayColors')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnMesh, 'getEdgeVertices')
    def getEdgeVertices(self, edgeId):
        # type: (int) -> Tuple[int, int]
        do, final_do, outTypes = _f.getDoArgs([edgeId], [('edgeId', 'int', 'in', None), ('vertexList', 'int2', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getEdgeVertices', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getFaceNormalIds')
    def getFaceNormalIds(self, faceIndex):
        # type: (int) -> List[int]
        do, final_do, outTypes = _f.getDoArgs([faceIndex], [('faceIndex', 'int', 'in', None), ('normals', 'MIntArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getFaceNormalIds', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getFaceUVSetNames')
    def getFaceUVSetNames(self, polygonId):
        # type: (int) -> List[str]
        do, final_do, outTypes = _f.getDoArgs([polygonId], [('polygonId', 'int', 'in', None), ('setNames', 'MStringArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getFaceUVSetNames', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getFaceVertexBinormal')
    def getFaceVertexBinormal(self, faceIndex, vertexIndex, space='preTransform', uvSet=None):
        # type: (int, int, datatypes.Space.Space, str) -> datatypes.Vector
        do, final_do, outTypes = _f.getDoArgs([faceIndex, vertexIndex, space, uvSet], [('faceIndex', 'int', 'in', None), ('vertexIndex', 'int', 'in', None), ('binormal', 'MVector', 'out', None), ('space', ('MSpace', 'Space'), 'in', None), ('uvSet', 'MString', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getFaceVertexBinormal', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getFaceVertexBinormals')
    def getFaceVertexBinormals(self, faceIndex, space='preTransform', uvSet=None):
        # type: (int, datatypes.Space.Space, str) -> List[datatypes.FloatVector]
        do, final_do, outTypes = _f.getDoArgs([faceIndex, space, uvSet], [('faceIndex', 'int', 'in', None), ('binormals', 'MFloatVectorArray', 'out', None), ('space', ('MSpace', 'Space'), 'in', None), ('uvSet', 'MString', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getFaceVertexBinormals', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getFaceVertexColorIndex')
    def getFaceVertexColorIndex(self, faceIndex, localVertexIndex, colorSet=None):
        # type: (int, int, str) -> int
        do, final_do, outTypes = _f.getDoArgs([faceIndex, localVertexIndex, colorSet], [('faceIndex', 'int', 'in', None), ('localVertexIndex', 'int', 'in', None), ('colorIndex', 'int', 'out', None), ('colorSet', 'MString', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getFaceVertexColorIndex', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getFaceVertexColors')
    def getFaceVertexColors(self, colorSet=None, defaultUnsetColor=None):
        # type: (str, datatypes.Color) -> List[datatypes.Color]
        do, final_do, outTypes = _f.getDoArgs([colorSet, defaultUnsetColor], [('colors', 'MColorArray', 'out', None), ('colorSet', 'MString', 'in', None), ('defaultUnsetColor', 'MColor', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getFaceVertexColors', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getFaceVertexNormal')
    def getFaceVertexNormal(self, faceIndex, vertexIndex, space='preTransform'):
        # type: (int, int, datatypes.Space.Space) -> datatypes.Vector
        do, final_do, outTypes = _f.getDoArgs([faceIndex, vertexIndex, space], [('faceIndex', 'int', 'in', None), ('vertexIndex', 'int', 'in', None), ('normal', 'MVector', 'out', None), ('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getFaceVertexNormal', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getFaceVertexNormals')
    def getFaceVertexNormals(self, faceIndex, space='preTransform'):
        # type: (int, datatypes.Space.Space) -> List[datatypes.FloatVector]
        do, final_do, outTypes = _f.getDoArgs([faceIndex, space], [('faceIndex', 'int', 'in', None), ('normals', 'MFloatVectorArray', 'out', None), ('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getFaceVertexNormals', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getFaceVertexTangent')
    def getFaceVertexTangent(self, faceIndex, vertexIndex, space='preTransform', uvSet=None):
        # type: (int, int, datatypes.Space.Space, str) -> datatypes.Vector
        do, final_do, outTypes = _f.getDoArgs([faceIndex, vertexIndex, space, uvSet], [('faceIndex', 'int', 'in', None), ('vertexIndex', 'int', 'in', None), ('tangent', 'MVector', 'out', None), ('space', ('MSpace', 'Space'), 'in', None), ('uvSet', 'MString', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getFaceVertexTangent', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getFaceVertexTangents')
    def getFaceVertexTangents(self, faceIndex, space='preTransform', uvSet=None):
        # type: (int, datatypes.Space.Space, str) -> List[datatypes.FloatVector]
        do, final_do, outTypes = _f.getDoArgs([faceIndex, space, uvSet], [('faceIndex', 'int', 'in', None), ('tangents', 'MFloatVectorArray', 'out', None), ('space', ('MSpace', 'Space'), 'in', None), ('uvSet', 'MString', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getFaceVertexTangents', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getHoles')
    def getHoles(self):
        # type: () -> Tuple[int, List[int], List[int]]
        do, final_do, outTypes = _f.getDoArgs([], [('holeInfoArray', 'MIntArray', 'out', None), ('holeVertexArray', 'MIntArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getHoles', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'int', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getMeshShellsIds')
    def getMeshShellsIds(self, compType):
        # type: (datatypes.Fn.Type) -> Tuple[List[int], int]
        do, final_do, outTypes = _f.getDoArgs([compType], [('compType', ('MFn', 'Type'), 'in', None), ('meshShellIds', 'MIntArray', 'out', None), ('nbMeshShells', 'uint', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getMeshShellsIds', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getNormalIds')
    def getNormalIds(self):
        # type: () -> Tuple[List[int], List[int]]
        do, final_do, outTypes = _f.getDoArgs([], [('normalCounts', 'MIntArray', 'out', None), ('normals', 'MIntArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getNormalIds', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getNormals')
    def getNormals(self, space='preTransform'):
        # type: (datatypes.Space.Space) -> List[datatypes.FloatVector]
        do, final_do, outTypes = _f.getDoArgs([space], [('normals', 'MFloatVectorArray', 'out', None), ('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getNormals', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getPoint')
    def getPoint(self, vertexId, space='preTransform'):
        # type: (int, datatypes.Space.Space) -> datatypes.Point
        do, final_do, outTypes = _f.getDoArgs([vertexId, space], [('vertexId', 'int', 'in', None), ('pos', 'MPoint', 'out', None), ('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getPoint', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getPointAtUV')
    def getPointAtUV(self, polygonId, uvPoint, space='preTransform', uvSet=None, tolerance=0.0):
        # type: (int, Tuple[float, float], datatypes.Space.Space, str, float) -> datatypes.Point
        do, final_do, outTypes = _f.getDoArgs([polygonId, uvPoint, space, uvSet, tolerance], [('polygonId', 'int', 'in', None), ('toThisPoint', 'MPoint', 'out', None), ('uvPoint', 'float2', 'in', None), ('space', ('MSpace', 'Space'), 'in', None), ('uvSet', 'MString', 'in', None), ('tolerance', 'float', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getPointAtUV', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getPoints')
    def getPoints(self, space='preTransform'):
        # type: (datatypes.Space.Space) -> List[datatypes.Point]
        do, final_do, outTypes = _f.getDoArgs([space], [('vertexArray', 'MPointArray', 'out', None), ('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getPoints', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getPointsAtUV')
    def getPointsAtUV(self, uvPoint, space='preTransform', uvSet=None, tolerance=0.001):
        # type: (Tuple[float, float], datatypes.Space.Space, str, float) -> Tuple[List[int], List[datatypes.Point]]
        do, final_do, outTypes = _f.getDoArgs([uvPoint, space, uvSet, tolerance], [('polygonIds', 'MIntArray', 'out', None), ('points', 'MPointArray', 'out', None), ('uvPoint', 'float2', 'in', None), ('space', ('MSpace', 'Space'), 'in', None), ('uvSet', 'MString', 'in', None), ('tolerance', 'float', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getPointsAtUV', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getPolygonNormal')
    def getPolygonNormal(self, polygonId, space='preTransform'):
        # type: (int, datatypes.Space.Space) -> datatypes.Vector
        do, final_do, outTypes = _f.getDoArgs([polygonId, space], [('polygonId', 'int', 'in', None), ('normal', 'MVector', 'out', None), ('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getPolygonNormal', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getPolygonTriangleVertices')
    def getPolygonTriangleVertices(self, polygonId, triangleId):
        # type: (int, int) -> Tuple[int, int, int]
        do, final_do, outTypes = _f.getDoArgs([polygonId, triangleId], [('polygonId', 'int', 'in', None), ('triangleId', 'int', 'in', None), ('vertexList', 'int__array3', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getPolygonTriangleVertices', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getPolygonUV')
    def getPolygonUV(self, polygonId, vertexIndex, uvSet=None):
        # type: (int, int, str) -> Tuple[float, float]
        do, final_do, outTypes = _f.getDoArgs([polygonId, vertexIndex, uvSet], [('polygonId', 'int', 'in', None), ('vertexIndex', 'int', 'in', None), ('u', 'float', 'out', None), ('v', 'float', 'out', None), ('uvSet', 'MString', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getPolygonUV', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getPolygonUVid')
    def getPolygonUVid(self, polygonId, vertexIndex, uvSet=None):
        # type: (int, int, str) -> int
        do, final_do, outTypes = _f.getDoArgs([polygonId, vertexIndex, uvSet], [('polygonId', 'int', 'in', None), ('vertexIndex', 'int', 'in', None), ('uvId', 'int', 'out', None), ('uvSet', 'MString', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getPolygonUVid', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getPolygonVertices')
    def getPolygonVertices(self, polygonId):
        # type: (int) -> List[int]
        do, final_do, outTypes = _f.getDoArgs([polygonId], [('polygonId', 'int', 'in', None), ('vertexList', 'MIntArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getPolygonVertices', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getRawUVs')
    def getRawUVs(self, uvSet=None):
        # type: (str) -> float
        do, final_do, outTypes = _f.getDoArgs([uvSet], [('uvSet', 'MString', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getRawUVs', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'float', None)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'getTangentId')
    def getTangentId(self, faceIndex, vertexIndex):
        # type: (int, int) -> int
        do, final_do, outTypes = _f.getDoArgs([faceIndex, vertexIndex], [('faceIndex', 'int', 'in', None), ('vertexIndex', 'int', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getTangentId', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'int', None)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'getTangents')
    def getTangents(self, space='preTransform', uvSet=None):
        # type: (datatypes.Space.Space, str) -> List[datatypes.FloatVector]
        do, final_do, outTypes = _f.getDoArgs([space, uvSet], [('tangents', 'MFloatVectorArray', 'out', None), ('space', ('MSpace', 'Space'), 'in', None), ('uvSet', 'MString', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getTangents', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getTriangleOffsets')
    def getTriangleOffsets(self):
        # type: () -> Tuple[List[int], List[int]]
        do, final_do, outTypes = _f.getDoArgs([], [('triangleCounts', 'MIntArray', 'out', None), ('triangleIndices', 'MIntArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getTriangleOffsets', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getTriangles')
    def getTriangles(self):
        # type: () -> Tuple[List[int], List[int]]
        do, final_do, outTypes = _f.getDoArgs([], [('triangleCounts', 'MIntArray', 'out', None), ('triangleVertices', 'MIntArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getTriangles', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getUV')
    def getUV(self, uvId, uvSet=None):
        # type: (int, str) -> Tuple[float, float]
        do, final_do, outTypes = _f.getDoArgs([uvId, uvSet], [('uvId', 'int', 'in', None), ('u', 'float', 'out', None), ('v', 'float', 'out', None), ('uvSet', 'MString', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getUV', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getUVBorderEdges')
    def getUVBorderEdges(self, setId):
        # type: (int) -> List[int]
        do, final_do, outTypes = _f.getDoArgs([setId], [('setId', 'uint', 'in', None), ('edgeList', 'MIntArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getUVBorderEdges', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getUVSetFamilyNames')
    def getUVSetFamilyNames(self):
        # type: () -> List[str]
        do, final_do, outTypes = _f.getDoArgs([], [('familyNames', 'MStringArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getUVSetFamilyNames', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getUVSetNames')
    def getUVSetNames(self):
        # type: () -> List[str]
        do, final_do, outTypes = _f.getDoArgs([], [('setNames', 'MStringArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getUVSetNames', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getUVSetsInFamily')
    def getUVSetsInFamily(self, familyName):
        # type: (str) -> List[str]
        do, final_do, outTypes = _f.getDoArgs([familyName], [('familyName', 'MString', 'in', None), ('setNames', 'MStringArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getUVSetsInFamily', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getUVs')
    def getUVs(self, uvSet=None):
        # type: (str) -> Tuple[List[float], List[float]]
        do, final_do, outTypes = _f.getDoArgs([uvSet], [('uArray', 'MFloatArray', 'out', None), ('vArray', 'MFloatArray', 'out', None), ('uvSet', 'MString', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getUVs', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getUvShellsIds')
    def getUvShellsIds(self, uvSet=None):
        # type: (str) -> Tuple[List[int], int]
        do, final_do, outTypes = _f.getDoArgs([uvSet], [('uvShellIds', 'MIntArray', 'out', None), ('nbUvShells', 'uint', 'out', None), ('uvSet', 'MString', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getUvShellsIds', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getVertexNormal')
    def getVertexNormal(self, vertexId, angleWeighted, space='preTransform'):
        # type: (int, bool, datatypes.Space.Space) -> datatypes.Vector
        do, final_do, outTypes = _f.getDoArgs([vertexId, angleWeighted, space], [('vertexId', 'int', 'in', None), ('angleWeighted', 'bool', 'in', None), ('normal', 'MVector', 'out', None), ('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getVertexNormal', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getVertexNormals')
    def getVertexNormals(self, angleWeighted, space='preTransform'):
        # type: (bool, datatypes.Space.Space) -> List[datatypes.FloatVector]
        do, final_do, outTypes = _f.getDoArgs([angleWeighted, space], [('angleWeighted', 'bool', 'in', None), ('normals', 'MFloatVectorArray', 'out', None), ('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getVertexNormals', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'getVertices')
    def getVertices(self):
        # type: () -> Tuple[List[int], List[int]]
        do, final_do, outTypes = _f.getDoArgs([], [('vertexCount', 'MIntArray', 'out', None), ('vertexList', 'MIntArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'getVertices', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'hasAlphaChannels')
    def hasAlphaChannels(self, colorSet):
        # type: (str) -> bool
        do, final_do, outTypes = _f.getDoArgs([colorSet], [('colorSet', 'MString', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'hasAlphaChannels', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'hasColorChannels')
    def hasColorChannels(self, colorSet):
        # type: (str) -> bool
        do, final_do, outTypes = _f.getDoArgs([colorSet], [('colorSet', 'MString', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'hasColorChannels', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'intersect')
    def intersect(self, raySource, rayDirection, tolerance=1e-10, space='preTransform'):
        # type: (datatypes.Point, datatypes.Vector, float, datatypes.Space.Space) -> Tuple[bool, List[datatypes.Point], List[int]]
        do, final_do, outTypes = _f.getDoArgs([raySource, rayDirection, tolerance, space], [('raySource', 'MPoint', 'in', None), ('rayDirection', 'MVector', 'in', None), ('points', 'MPointArray', 'out', None), ('tolerance', 'double', 'in', None), ('space', ('MSpace', 'Space'), 'in', None), ('polygonIds', 'MIntArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'intersect', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'intersectFaceAtUV')
    def intersectFaceAtUV(self, toThisUVPoint, uvSet=None):
        # type: (Tuple[float, float], str) -> int
        do, final_do, outTypes = _f.getDoArgs([toThisUVPoint, uvSet], [('toThisUVPoint', 'float2', 'in', None), ('faceId', 'int', 'out', None), ('uvSet', 'MString', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'intersectFaceAtUV', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'isColorClamped')
    def isColorClamped(self, colorSet):
        # type: (str) -> bool
        do, final_do, outTypes = _f.getDoArgs([colorSet], [('colorSet', 'MString', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'isColorClamped', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'isColorSetPerInstance')
    def isColorSetPerInstance(self, name):
        # type: (str) -> bool
        do, final_do, outTypes = _f.getDoArgs([name], [('name', 'MString', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'isColorSetPerInstance', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'isEdgeSmooth')
    def isEdgeSmooth(self, edgeId):
        # type: (int) -> bool
        do, final_do, outTypes = _f.getDoArgs([edgeId], [('edgeId', 'int', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'isEdgeSmooth', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'isNormalLocked')
    def isNormalLocked(self, normalId):
        # type: (int) -> bool
        do, final_do, outTypes = _f.getDoArgs([normalId], [('normalId', 'int', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'isNormalLocked', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'isPolygonConvex')
    def isPolygonConvex(self, faceIndex):
        # type: (int) -> bool
        do, final_do, outTypes = _f.getDoArgs([faceIndex], [('faceIndex', 'int', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'isPolygonConvex', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'isPolygonUVReversed')
    def isPolygonUVReversed(self, polygonId, uvSet=None):
        # type: (int, str) -> bool
        do, final_do, outTypes = _f.getDoArgs([polygonId, uvSet], [('polygonId', 'int', 'in', None), ('uvSet', 'MString', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'isPolygonUVReversed', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'isRightHandedTangent')
    def isRightHandedTangent(self, tangentId, uvSetName=None):
        # type: (int, str) -> bool
        do, final_do, outTypes = _f.getDoArgs([tangentId, uvSetName], [('tangentId', 'int', 'in', None), ('uvSetName', 'MString', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'isRightHandedTangent', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'isUVSetPerInstance')
    def isUVSetPerInstance(self, name):
        # type: (str) -> bool
        do, final_do, outTypes = _f.getDoArgs([name], [('name', 'MString', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'isUVSetPerInstance', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'lockFaceVertexNormals')
    def lockFaceVertexNormals(self, faceList, vertexList):
        # type: (List[int], List[int]) -> None
        do, final_do, outTypes = _f.getDoArgs([faceList, vertexList], [('faceList', 'MIntArray', 'in', None), ('vertexList', 'MIntArray', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'lockFaceVertexNormals', final_do)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'lockVertexNormals')
    def lockVertexNormals(self, vertexList):
        # type: (List[int]) -> None
        do, final_do, outTypes = _f.getDoArgs([vertexList], [('vertexList', 'MIntArray', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'lockVertexNormals', final_do)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'numPolygons')
    def numPolygons(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnMesh, 'numPolygons')
        return _f.ApiArgUtil._castResult(self, res, 'int', None)
    _numPolygons = numPolygons

    @_f.addApiDocs(_api.MFnMesh, 'onBoundary')
    def onBoundary(self, polygonId):
        # type: (int) -> bool
        do, final_do, outTypes = _f.getDoArgs([polygonId], [('polygonId', 'int', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'onBoundary', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res

    @classmethod
    @_f.addApiDocs(_api.MFnMesh, 'polyTriangulate')
    def polyTriangulate(self, pointsAndHoles, pointsCount, outerPointsCount, holesCount, considerNormal, normals):
        # type: (float, int, int, int, bool, float) -> Tuple[int, int]
        do, final_do, outTypes = _f.getDoArgs([pointsAndHoles, pointsCount, outerPointsCount, holesCount, considerNormal, normals], [('pointsAndHoles', 'float', 'in', None), ('pointsCount', 'uint', 'in', None), ('outerPointsCount', 'uint', 'in', None), ('holesCount', 'uint', 'in', None), ('considerNormal', 'bool', 'in', None), ('normals', 'float', 'in', None), ('triangles', 'short', 'out', None), ('trianglesCount', 'int', 'out', None)])
        res = _api.MFnMesh.polyTriangulate(*final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnMesh, 'polygonVertexCount')
    def polygonVertexCount(self, polygonId):
        # type: (int) -> int
        do, final_do, outTypes = _f.getDoArgs([polygonId], [('polygonId', 'int', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'polygonVertexCount', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'int', None)
        return res
    numPolygonVertices = polygonVertexCount

    @_f.addApiDocs(_api.MFnMesh, 'removeFaceColors')
    def removeFaceColors(self, faceList):
        # type: (List[int]) -> None
        do, final_do, outTypes = _f.getDoArgs([faceList], [('faceList', 'MIntArray', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'removeFaceColors', final_do)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'removeFaceVertexColors')
    def removeFaceVertexColors(self, faceList, vertexList):
        # type: (List[int], List[int]) -> None
        do, final_do, outTypes = _f.getDoArgs([faceList, vertexList], [('faceList', 'MIntArray', 'in', None), ('vertexList', 'MIntArray', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'removeFaceVertexColors', final_do)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'removeVertexColors')
    def removeVertexColors(self, vertexList):
        # type: (List[int]) -> None
        do, final_do, outTypes = _f.getDoArgs([vertexList], [('vertexList', 'MIntArray', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'removeVertexColors', final_do)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'renameUVSet')
    def renameUVSet(self, origName, newName, modifier=None):
        # type: (str, str, datatypes.DGModifier) -> None
        do, final_do, outTypes = _f.getDoArgs([origName, newName, modifier], [('origName', 'MString', 'in', None), ('newName', 'MString', 'in', None), ('modifier', 'MDGModifier', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'renameUVSet', final_do)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'setCheckSamePointTwice')
    def setCheckSamePointTwice(self, check=True):
        # type: (bool) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([check], [('check', 'bool', 'in', None)], self.getCheckSamePointTwice, self.setCheckSamePointTwice, [])
        res = _f.getProxyResult(self, _api.MFnMesh, 'setCheckSamePointTwice', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'setColor')
    def setColor(self, colorId, color, colorSet=None):
        # type: (int, datatypes.Color, str) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([colorId, color, colorSet], [('colorId', 'int', 'in', None), ('color', 'MColor', 'in', None), ('colorSet', 'MString', 'in', None)], self.getColor, self.setColor, ['colorId', 'colorSet', 'defaultUnsetColor'])
        res = _f.getProxyResult(self, _api.MFnMesh, 'setColor', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'setIsColorClamped')
    def setColorClamped(self, colorSet, clamped):
        # type: (str, bool) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([colorSet, clamped], [('colorSet', 'MString', 'in', None), ('clamped', 'bool', 'in', None)], self.isColorClamped, self.setColorClamped, ['colorSet'])
        res = _f.getProxyResult(self, _api.MFnMesh, 'setIsColorClamped', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'setColors')
    def setColors(self, colorArray, colorSet=None):
        # type: (List[datatypes.Color], str) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([colorArray, colorSet], [('colorArray', 'MColorArray', 'in', None), ('colorSet', 'MString', 'in', None)], self.getColors, self.setColors, ['colorSet', 'defaultUnsetColor'])
        res = _f.getProxyResult(self, _api.MFnMesh, 'setColors', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'setCurrentColorSetName')
    def setCurrentColorSetName(self, setName, modifier=None, currentSelection=None):
        # type: (str, datatypes.DGModifier, SelectionSet) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([setName, modifier, currentSelection], [('setName', 'MString', 'in', None), ('modifier', 'MDGModifier', 'in', None), ('currentSelection', 'MSelectionList', 'in', None)], self.getCurrentColorSetName, self.setCurrentColorSetName, ['instance'])
        res = _f.getProxyResult(self, _api.MFnMesh, 'setCurrentColorSetName', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'setCurrentUVSetName')
    def setCurrentUVSetName(self, setName, modifier=None, currentSelection=None):
        # type: (str, datatypes.DGModifier, SelectionSet) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([setName, modifier, currentSelection], [('setName', 'MString', 'in', None), ('modifier', 'MDGModifier', 'in', None), ('currentSelection', 'MSelectionList', 'in', None)], self.getCurrentUVSetName, self.setCurrentUVSetName, ['instance'])
        res = _f.getProxyResult(self, _api.MFnMesh, 'setCurrentUVSetName', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'setDisplayColors')
    def setDisplayColors(self, enable):
        # type: (bool) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([enable], [('enable', 'bool', 'in', None)], self.getDisplayColors, self.setDisplayColors, [])
        res = _f.getProxyResult(self, _api.MFnMesh, 'setDisplayColors', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'setEdgeSmoothing')
    def setEdgeSmoothing(self, edgeId, smooth=True):
        # type: (int, bool) -> None
        do, final_do, outTypes = _f.getDoArgs([edgeId, smooth], [('edgeId', 'int', 'in', None), ('smooth', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'setEdgeSmoothing', final_do)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'setEdgeSmoothings')
    def setEdgeSmoothings(self, edgeIds, smooths):
        # type: (List[int], List[int]) -> None
        do, final_do, outTypes = _f.getDoArgs([edgeIds, smooths], [('edgeIds', 'MIntArray', 'in', None), ('smooths', 'MIntArray', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'setEdgeSmoothings', final_do)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'setFaceColor')
    def setFaceColor(self, color, faceIndex):
        # type: (datatypes.Color, int) -> None
        do, final_do, outTypes = _f.getDoArgs([color, faceIndex], [('color', 'MColor', 'in', None), ('faceIndex', 'int', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'setFaceColor', final_do)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'setFaceColors')
    def setFaceColors(self, colors, faceList):
        # type: (List[datatypes.Color], List[int]) -> None
        do, final_do, outTypes = _f.getDoArgs([colors, faceList], [('colors', 'MColorArray', 'in', None), ('faceList', 'MIntArray', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'setFaceColors', final_do)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'setFaceVertexColor')
    def setFaceVertexColor(self, color, faceIndex, vertexIndex, modifier=None):
        # type: (datatypes.Color, int, int, datatypes.DGModifier) -> None
        do, final_do, outTypes = _f.getDoArgs([color, faceIndex, vertexIndex, modifier], [('color', 'MColor', 'in', None), ('faceIndex', 'int', 'in', None), ('vertexIndex', 'int', 'in', None), ('modifier', 'MDGModifier', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'setFaceVertexColor', final_do)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'setFaceVertexNormal')
    def setFaceVertexNormal(self, normalIn, faceId, vertexId, space='preTransform', modifier=None):
        # type: (datatypes.Vector, int, int, datatypes.Space.Space, datatypes.DGModifier) -> None
        do, final_do, outTypes = _f.getDoArgs([normalIn, faceId, vertexId, space, modifier], [('normalIn', 'MVector', 'in', None), ('faceId', 'int', 'in', None), ('vertexId', 'int', 'in', None), ('space', ('MSpace', 'Space'), 'in', None), ('modifier', 'MDGModifier', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'setFaceVertexNormal', final_do)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'setNormals')
    def setNormals(self, normals, space='preTransform'):
        # type: (List[datatypes.FloatVector], datatypes.Space.Space) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([normals, space], [('normals', 'MFloatVectorArray', 'in', None), ('space', ('MSpace', 'Space'), 'in', None)], self.getNormals, self.setNormals, ['space'])
        res = _f.getProxyResult(self, _api.MFnMesh, 'setNormals', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'setPoint')
    def setPoint(self, vertexId, pos, space='preTransform'):
        # type: (int, datatypes.Point, datatypes.Space.Space) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([vertexId, pos, space], [('vertexId', 'int', 'in', None), ('pos', 'MPoint', 'in', None), ('space', ('MSpace', 'Space'), 'in', None)], self.getPoint, self.setPoint, ['vertexId', 'space'])
        res = _f.getProxyResult(self, _api.MFnMesh, 'setPoint', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'setPoints')
    def setPoints(self, vertexArray, space='preTransform'):
        # type: (List[datatypes.Point], datatypes.Space.Space) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([vertexArray, space], [('vertexArray', 'MPointArray', 'in', None), ('space', ('MSpace', 'Space'), 'in', None)], self.getPoints, self.setPoints, ['space'])
        res = _f.getProxyResult(self, _api.MFnMesh, 'setPoints', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'setSomeColors')
    def setSomeColors(self, colorIds, colorArray, colorSet=None):
        # type: (List[int], List[datatypes.Color], str) -> None
        do, final_do, outTypes = _f.getDoArgs([colorIds, colorArray, colorSet], [('colorIds', 'MIntArray', 'in', None), ('colorArray', 'MColorArray', 'in', None), ('colorSet', 'MString', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'setSomeColors', final_do)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'setSomeUVs')
    def setSomeUVs(self, uvIds, uArray, vArray, uvSet=None):
        # type: (List[int], List[float], List[float], str) -> None
        do, final_do, outTypes = _f.getDoArgs([uvIds, uArray, vArray, uvSet], [('uvIds', 'MIntArray', 'in', None), ('uArray', 'MFloatArray', 'in', None), ('vArray', 'MFloatArray', 'in', None), ('uvSet', 'MString', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'setSomeUVs', final_do)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'setUV')
    def setUV(self, uvId, u, v, uvSet=None):
        # type: (int, float, float, str) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([uvId, u, v, uvSet], [('uvId', 'int', 'in', None), ('u', 'float', 'in', None), ('v', 'float', 'in', None), ('uvSet', 'MString', 'in', None)], self.getUV, self.setUV, ['uvId', 'uvSet'])
        res = _f.getProxyResult(self, _api.MFnMesh, 'setUV', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'setUVs')
    def setUVs(self, uArray, vArray, uvSet=None):
        # type: (List[float], List[float], str) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([uArray, vArray, uvSet], [('uArray', 'MFloatArray', 'in', None), ('vArray', 'MFloatArray', 'in', None), ('uvSet', 'MString', 'in', None)], self.getUVs, self.setUVs, ['uvSet'])
        res = _f.getProxyResult(self, _api.MFnMesh, 'setUVs', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'setVertexColor')
    def setVertexColor(self, color, vertexIndex, modifier=None):
        # type: (datatypes.Color, int, datatypes.DGModifier) -> None
        do, final_do, outTypes = _f.getDoArgs([color, vertexIndex, modifier], [('color', 'MColor', 'in', None), ('vertexIndex', 'int', 'in', None), ('modifier', 'MDGModifier', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'setVertexColor', final_do)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'setVertexNormal')
    def setVertexNormal(self, normalIn, vertexId, space='preTransform', modifier=None):
        # type: (datatypes.Vector, int, datatypes.Space.Space, datatypes.DGModifier) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([normalIn, vertexId, space, modifier], [('normalIn', 'MVector', 'in', None), ('vertexId', 'int', 'in', None), ('space', ('MSpace', 'Space'), 'in', None), ('modifier', 'MDGModifier', 'in', None)], self.getVertexNormal, self.setVertexNormal, ['vertexId', 'space'])
        res = _f.getProxyResult(self, _api.MFnMesh, 'setVertexNormal', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'syncObject')
    def syncObject(self):
        # type: () -> None
        res = _f.getProxyResult(self, _api.MFnMesh, 'syncObject')
        return res

    @_f.addApiDocs(_api.MFnMesh, 'unlockFaceVertexNormals')
    def unlockFaceVertexNormals(self, faceList, vertexList):
        # type: (List[int], List[int]) -> None
        do, final_do, outTypes = _f.getDoArgs([faceList, vertexList], [('faceList', 'MIntArray', 'in', None), ('vertexList', 'MIntArray', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'unlockFaceVertexNormals', final_do)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'unlockVertexNormals')
    def unlockVertexNormals(self, vertexList):
        # type: (List[int]) -> None
        do, final_do, outTypes = _f.getDoArgs([vertexList], [('vertexList', 'MIntArray', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnMesh, 'unlockVertexNormals', final_do)
        return res

    @_f.addApiDocs(_api.MFnMesh, 'updateSurface')
    def updateSurface(self):
        # type: () -> None
        res = _f.getProxyResult(self, _api.MFnMesh, 'updateSurface')
        return res
# ------ Do not edit above this line --------


# Unfortunately, objects that don't yet have any mesh data - ie, if you do
# createNode('mesh') - can't be fed into MFnMesh (even though it is a mesh
# node).  This means that all the methods wrapped from MFnMesh won't be
# usable in this case.  While it might make sense for some methods - ie,
# editing methods like collapseEdges - to fail in this situation, some
# basic methods like numVertices should still be usable.  Therefore,
# we override some of these with the mel versions (which still work...)

def _makeApiMethodWrapForEmptyMesh(apiMethodName, baseMethodName=None,
                                   resultName=None, defaultVal=0):
    if baseMethodName is None:
        baseMethodName = '_' + apiMethodName
    if resultName is None:
        resultName = apiMethodName

    baseMethod = getattr(Mesh, baseMethodName)

    @_factories.addApiDocs(_api.MFnMesh, apiMethodName)
    def methodWrapForEmptyMesh(self, *args, **kwargs):
        # If we have an empty mesh, we will get an MFnDagNode...
        mfn = self.__apimfn__()
        if not isinstance(mfn, _api.MFnMesh):
            return defaultVal
        return baseMethod(self, *args, **kwargs)
    methodWrapForEmptyMesh.__name__ = resultName
    return methodWrapForEmptyMesh


if not _factories.building:
    # keep this safe to load if the templates have not been built yet
    for _apiMethodName in '''numColorSets
                        numFaceVertices
                        numNormals
                        numUVSets
                        numUVs'''.split():
        _wrappedFunc = _makeApiMethodWrapForEmptyMesh(_apiMethodName)
        setattr(Mesh, _wrappedFunc.__name__, _wrappedFunc)


class Subdiv(SurfaceShape):

    _componentAttributes = {
        'smp': general.SubdVertex,
        'verts': general.SubdVertex,
        'sme': general.SubdEdge,
        'edges': general.SubdEdge,
        'smf': general.SubdFace,
        'faces': general.SubdFace,
        'smm': general.SubdUV,
        'uvs': general.SubdUV,
    }

    if TYPE_CHECKING:
        # FIXME: move to build
        smp = None  # type: general.SubdVertex
        verts = None  # type: general.SubdVertex
        sme = None  # type: general.SubdEdge
        edges = None  # type: general.SubdEdge
        smf = None  # type: general.SubdFace
        faces = None  # type: general.SubdFace
        smm = None  # type: general.SubdUV
        uvs = None  # type: general.SubdUV

    def getTweakedVerts(self, **kwargs):
        return cmds.querySubdiv(action=1, **kwargs)

    def getSharpenedVerts(self, **kwargs):
        return cmds.querySubdiv(action=2, **kwargs)

    def getSharpenedEdges(self, **kwargs):
        return cmds.querySubdiv(action=3, **kwargs)

    def getEdges(self, **kwargs):
        return cmds.querySubdiv(action=4, **kwargs)

    def cleanTopology(self):
        cmds.subdCleanTopology(self)
# ------ Do not edit below this line --------
    __apicls__ = _api.MFnSubd
    __melcmd__ = staticmethod(modeling.subdiv)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'subdiv'
    __melnode__ = 'subdiv'
    __slots__ = ()

    @_f.addApiDocs(_api.MFnSubd, 'collapse')
    def collapse(self, level, makeNewNodeInvisible):
        # type: (int, bool) -> DependNode
        do, final_do, outTypes = _f.getDoArgs([level, makeNewNodeInvisible], [('level', 'int', 'in', None), ('makeNewNodeInvisible', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnSubd, 'collapse', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MObject', None)
        return res

    @_f.addApiDocs(_api.MFnSubd, 'convertToNurbs')
    def convertToNurbs(self):
        # type: () -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([], [('newNurbsSurfaces', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnSubd, 'convertToNurbs', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnSubd, 'copy')
    def copy(self, source, parentOrOwner='MObject::kNullObj'):
        # type: (str | DependNode, str | DependNode) -> DependNode
        do, final_do, outTypes = _f.getDoArgs([source, parentOrOwner], [('source', 'MObject', 'in', None), ('parentOrOwner', 'MObject', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnSubd, 'copy', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MObject', None)
        return res

    @_f.addApiDocs(_api.MFnSubd, 'creasesClearAll')
    def creasesClearAll(self):
        # type: () -> None
        res = _f.getProxyResult(self, _api.MFnSubd, 'creasesClearAll')
        return res

    @_f.addApiDocs(_api.MFnSubd, 'createBaseMesh')
    def createBaseMesh(self, reverseNormal, numVertices, numPolygons, vertexArray, polygonCounts, polygonConnects, parentOrOwner='MObject::kNullObj'):
        # type: (bool, int, int, List[datatypes.Point], List[int], List[int], str | DependNode) -> DependNode
        do, final_do, outTypes = _f.getDoArgs([reverseNormal, numVertices, numPolygons, vertexArray, polygonCounts, polygonConnects, parentOrOwner], [('reverseNormal', 'bool', 'in', None), ('numVertices', 'int', 'in', None), ('numPolygons', 'int', 'in', None), ('vertexArray', 'MPointArray', 'in', None), ('polygonCounts', 'MIntArray', 'in', None), ('polygonConnects', 'MIntArray', 'in', None), ('parentOrOwner', 'MObject', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnSubd, 'createBaseMesh', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MObject', None)
        return res

    @_f.addApiDocs(_api.MFnSubd, 'editsPending')
    def editsPending(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnSubd, 'editsPending')
        return _f.ApiArgUtil._castResult(self, res, 'int', None)

    @_f.addApiDocs(_api.MFnSubd, 'editsUpdateAll')
    def editsUpdateAll(self):
        # type: () -> None
        res = _f.getProxyResult(self, _api.MFnSubd, 'editsUpdateAll')
        return res

    @_f.addMelDocs('subdiv', 'currentLevel')
    def getCurrentLevel(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, modeling.subdiv, kwargs, 'currentLevel')
        return res

    @_f.addMelDocs('subdiv', 'currentSubdLevel')
    def getCurrentSubdLevel(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, modeling.subdiv, kwargs, 'currentSubdLevel')
        return res

    @_f.addMelDocs('subdiv', 'deepestLevel')
    def getDeepestLevel(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, modeling.subdiv, kwargs, 'deepestLevel')
        return res

    @_f.addMelDocs('subdiv', 'displayLoad')
    def getDisplayLoad(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, modeling.subdiv, kwargs, 'displayLoad')
        return res

    @_f.addMelDocs('subdiv', 'edgeStats')
    def getEdgeStats(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, modeling.subdiv, kwargs, 'edgeStats')
        return res

    @_f.addMelDocs('subdiv', 'faceStats')
    def getFaceStats(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, modeling.subdiv, kwargs, 'faceStats')
        return res

    @_f.addMelDocs('subdiv', 'maxPossibleLevel')
    def getMaxPossibleLevel(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, modeling.subdiv, kwargs, 'maxPossibleLevel')
        return res

    @_f.addMelDocs('subdiv', 'proxyMode')
    def getProxyMode(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, modeling.subdiv, kwargs, 'proxyMode')
        return res

    @_f.addMelDocs('subdiv', 'smallOffsets')
    def getSmallOffsets(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, modeling.subdiv, kwargs, 'smallOffsets')
        return res

    @_f.addApiDocs(_api.MFnSubd, 'levelFullySubdivideTo')
    def levelFullySubdivideTo(self, level):
        # type: (int) -> None
        do, final_do, outTypes = _f.getDoArgs([level], [('level', 'int', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnSubd, 'levelFullySubdivideTo', final_do)
        return res

    @_f.addApiDocs(_api.MFnSubd, 'levelMaxAllowed')
    def levelMaxAllowed(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnSubd, 'levelMaxAllowed')
        return _f.ApiArgUtil._castResult(self, res, 'uint', None)

    @_f.addApiDocs(_api.MFnSubd, 'levelMaxCurrent')
    def levelMaxCurrent(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnSubd, 'levelMaxCurrent')
        return _f.ApiArgUtil._castResult(self, res, 'uint', None)

    @_f.addApiDocs(_api.MFnSubd, 'edgeCount')
    def numEdges(self, level=0):
        # type: (int) -> int
        do, final_do, outTypes = _f.getDoArgs([level], [('level', 'int', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnSubd, 'edgeCount', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'uint', None)
        return res

    @_f.addApiDocs(_api.MFnSubd, 'polygonCount')
    def numFaces(self, level=0):
        # type: (int) -> int
        do, final_do, outTypes = _f.getDoArgs([level], [('level', 'int', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnSubd, 'polygonCount', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'uint', None)
        return res

    @_f.addApiDocs(_api.MFnSubd, 'vertexCount')
    def numVertices(self, level=0):
        # type: (int) -> int
        do, final_do, outTypes = _f.getDoArgs([level], [('level', 'int', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnSubd, 'vertexCount', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'uint', None)
        return res

    @_f.addApiDocs(_api.MFnSubd, 'polygonCountMaxWithGivenBaseMesh')
    def polygonCountMaxWithGivenBaseMesh(self, level=1):
        # type: (int) -> int
        do, final_do, outTypes = _f.getDoArgs([level], [('level', 'int', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnSubd, 'polygonCountMaxWithGivenBaseMesh', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'int', None)
        return res

    @_f.addApiDocs(_api.MFnSubd, 'tesselate')
    def tesselate(self, uniform, depth, sample, parentOrOwner='MObject::kNullObj'):
        # type: (bool, int, int, str | DependNode) -> DependNode
        do, final_do, outTypes = _f.getDoArgs([uniform, depth, sample, parentOrOwner], [('uniform', 'bool', 'in', None), ('depth', 'int', 'in', None), ('sample', 'int', 'in', None), ('parentOrOwner', 'MObject', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnSubd, 'tesselate', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MObject', None)
        return res

    @_f.addApiDocs(_api.MFnSubd, 'tessellateViaNurbs')
    def tessellateViaNurbs(self, parentOrOwner='MObject::kNullObj'):
        # type: (str | DependNode) -> DependNode
        do, final_do, outTypes = _f.getDoArgs([parentOrOwner], [('parentOrOwner', 'MObject', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnSubd, 'tessellateViaNurbs', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MObject', None)
        return res

    @_f.addApiDocs(_api.MFnSubd, 'updateAllEditsAndCreases')
    def updateAllEditsAndCreases(self):
        # type: () -> None
        res = _f.getProxyResult(self, _api.MFnSubd, 'updateAllEditsAndCreases')
        return res

    @_f.addApiDocs(_api.MFnSubd, 'updateSubdSurface')
    def updateSubdSurface(self):
        # type: () -> None
        res = _f.getProxyResult(self, _api.MFnSubd, 'updateSubdSurface')
        return res

    @_f.addApiDocs(_api.MFnSubd, 'vertexBaseMeshAdd')
    def vertexBaseMeshAdd(self, x, y, z):
        # type: (float, float, float) -> int
        do, final_do, outTypes = _f.getDoArgs([x, y, z], [('x', 'double', 'in', None), ('y', 'double', 'in', None), ('z', 'double', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnSubd, 'vertexBaseMeshAdd', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'int', None)
        return res

    @_f.addApiDocs(_api.MFnSubd, 'vertexBaseMeshAddWithIndex')
    def vertexBaseMeshAddWithIndex(self, x, y, z, index):
        # type: (float, float, float, int) -> None
        do, final_do, outTypes = _f.getDoArgs([x, y, z, index], [('x', 'double', 'in', None), ('y', 'double', 'in', None), ('z', 'double', 'in', None), ('index', 'int', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnSubd, 'vertexBaseMeshAddWithIndex', final_do)
        return res

    @_f.addApiDocs(_api.MFnSubd, 'vertexBaseMeshGet')
    def vertexBaseMeshGet(self, space='preTransform'):
        # type: (datatypes.Space.Space) -> List[datatypes.Point]
        do, final_do, outTypes = _f.getDoArgs([space], [('positions', 'MPointArray', 'out', None), ('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnSubd, 'vertexBaseMeshGet', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnSubd, 'vertexBaseMeshSet')
    def vertexBaseMeshSet(self, space='preTransform'):
        # type: (datatypes.Space.Space) -> List[datatypes.Point]
        do, final_do, outTypes = _f.getDoArgs([space], [('positions', 'MPointArray', 'out', None), ('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnSubd, 'vertexBaseMeshSet', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnSubd, 'vertexEditsClearAllNonBase')
    def vertexEditsClearAllNonBase(self):
        # type: () -> None
        res = _f.getProxyResult(self, _api.MFnSubd, 'vertexEditsClearAllNonBase')
        return res
# ------ Do not edit above this line --------


class Lattice(ControlPoint):
    _componentAttributes = {
        'pt': general.LatticePoint,
        'points': general.LatticePoint,
    }
    if TYPE_CHECKING:
        # FIXME: move to build
        pt = None  # type: general.LatticePoint
        points = None  # type: general.LatticePoint
# ------ Do not edit below this line --------
    __apicls__ = _api.MFnLattice
    __melcmd__ = staticmethod(animation.lattice)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'lattice'
    __melnode__ = 'lattice'
    __slots__ = ()

    @_f.addMelDocs('lattice', 'after')
    def after(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.lattice, kwargs, 'after', val)

    @_f.addMelDocs('lattice', 'afterReference')
    def afterReference(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.lattice, kwargs, 'afterReference', val)

    @_f.addMelDocs('lattice', 'before')
    def before(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.lattice, kwargs, 'before', val)

    @_f.addMelDocs('lattice', 'frontOfChain')
    def frontOfChain(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.lattice, kwargs, 'frontOfChain', val)

    @_f.addMelDocs('lattice', 'deformerTools')
    def getDeformerTools(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.lattice, kwargs, 'deformerTools')
        return res

    @_f.addApiDocs(_api.MFnLattice, 'getDivisions')
    def getDivisions(self):
        # type: () -> Tuple[int, int, int]
        do, final_do, outTypes = _f.getDoArgs([], [('s', 'uint', 'out', None), ('t', 'uint', 'out', None), ('u', 'uint', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnLattice, 'getDivisions', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addMelDocs('lattice', 'exclusive')
    def getExclusive(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.lattice, kwargs, 'exclusive')
        return res

    @_f.addMelDocs('lattice', 'freezeMapping')
    def getFreezeMapping(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.lattice, kwargs, 'freezeMapping')
        return res

    @_f.addMelDocs('lattice', 'geometry')
    def getGeometry(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.lattice, kwargs, 'geometry')
        return res

    @_f.addMelDocs('lattice', 'geometryIndices')
    def getGeometryIndices(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.lattice, kwargs, 'geometryIndices')
        return res

    @_f.addMelDocs('lattice', 'ldivisions')
    def getLdivisions(self, **kwargs):
        # type: (...) -> Tuple[int, int, int]
        res = _f.asQuery(self, animation.lattice, kwargs, 'ldivisions')
        return res

    @_f.addMelDocs('lattice', 'useComponentTags')
    def getUseComponentTags(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.lattice, kwargs, 'useComponentTags')
        return res

    @_f.addMelDocs('lattice', 'latticeReset')
    def latticeReset(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.lattice, kwargs, 'latticeReset', val)

    @_f.addMelDocs('lattice', 'parallel')
    def parallel(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.lattice, kwargs, 'parallel', val)

    @_f.addApiDocs(_api.MFnLattice, 'point')
    def point(self, s, t, u):
        # type: (int, int, int) -> datatypes.Point
        do, final_do, outTypes = _f.getDoArgs([s, t, u], [('s', 'uint', 'in', None), ('t', 'uint', 'in', None), ('u', 'uint', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnLattice, 'point', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MPoint', None)
        return res

    @_f.addMelDocs('lattice', 'prune')
    def prune(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.lattice, kwargs, 'prune', val)

    @_f.addMelDocs('lattice', 'remove')
    def remove(self, val=True, **kwargs):
        # type: (bool | int | List[bool | int], **Any) -> None
        return _f.asEdit(self, animation.lattice, kwargs, 'remove', val)

    @_f.addMelDocs('lattice', 'removeTweaks')
    def removeTweaks(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.lattice, kwargs, 'removeTweaks', val)

    @_f.addApiDocs(_api.MFnLattice, 'reset')
    def reset(self, sSize=1.0, tSize=1.0, uSize=1.0):
        # type: (float, float, float) -> None
        do, final_do, outTypes = _f.getDoArgs([sSize, tSize, uSize], [('sSize', 'double', 'in', None), ('tSize', 'double', 'in', None), ('uSize', 'double', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnLattice, 'reset', final_do)
        return res

    @_f.addApiDocs(_api.MFnLattice, 'setDivisions')
    def setDivisions(self, s, t, u):
        # type: (int, int, int) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([s, t, u], [('s', 'uint', 'in', None), ('t', 'uint', 'in', None), ('u', 'uint', 'in', None)], self.getDivisions, self.setDivisions, [])
        res = _f.getProxyResult(self, _api.MFnLattice, 'setDivisions', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addMelDocs('lattice', 'freezeMapping')
    def setFreezeMapping(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.lattice, kwargs, 'freezeMapping', val)

    @_f.addMelDocs('lattice', 'geometry')
    def setGeometry(self, val=True, **kwargs):
        # type: (str | List[str], **Any) -> None
        return _f.asEdit(self, animation.lattice, kwargs, 'geometry', val)

    @_f.addMelDocs('lattice', 'ldivisions')
    def setLdivisions(self, val=True, **kwargs):
        # type: (Tuple[int, int, int], **Any) -> None
        return _f.asEdit(self, animation.lattice, kwargs, 'ldivisions', val)
# ------ Do not edit above this line --------



class Particle(DeformableShape):
    __apicls__ = _api.MFnParticleSystem
    _componentAttributes = {
        'pt': general.ParticleComponent,
        'points': general.ParticleComponent,
    }
    if TYPE_CHECKING:
        # FIXME: move to build
        pt = None  # type: general.ParticleComponent
        points = None  # type: general.ParticleComponent

    # for backwards compatibility
    Point = general.ParticleComponent

    # for backwards compatibility, keep these two, even though the api wrap
    # will also provide 'count'
    def pointCount(self):
        return cmds.particle(self, q=1, count=1)
    num = pointCount
# ------ Do not edit below this line --------
    __melcmd__ = staticmethod(effects.particle)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'particle'
    __melnode__ = 'particle'
    __slots__ = ()
    RenderType = Enum('RenderType', [('cloud', 0), ('kCloud', 0), ('tube', 1), ('kTube', 1), ('blobby', 2), ('kBlobby', 2), ('multiPoint', 3), ('kMultiPoint', 3), ('multiStreak', 4), ('kMultiStreak', 4), ('numeric', 5), ('kNumeric', 5), ('points', 6), ('kPoints', 6), ('spheres', 7), ('kSpheres', 7), ('sprites', 8), ('kSprites', 8), ('streak', 9), ('kStreak', 9)], multiKeys=True)

    @_f.addApiDocs(_api.MFnParticleSystem, 'acceleration')
    def acceleration(self):
        # type: () -> List[datatypes.Vector]
        do, final_do, outTypes = _f.getDoArgs([], [('array', 'MVectorArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'acceleration', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnParticleSystem, 'age')
    def age(self, ages):
        # type: (List[float]) -> None
        do, final_do, outTypes = _f.getDoArgs([ages], [('ages', 'MDoubleArray', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'age', final_do)
        return res

    @_f.addApiDocs(_api.MFnParticleSystem, 'betterIllum')
    def betterIllum(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'betterIllum')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnParticleSystem, 'castsShadows')
    def castsShadows(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'castsShadows')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnParticleSystem, 'deformedParticleShape')
    def deformedParticleShape(self):
        # type: () -> DependNode
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'deformedParticleShape')
        return _f.ApiArgUtil._castResult(self, res, 'MObject', None)

    @_f.addApiDocs(_api.MFnParticleSystem, 'disableCloudAxis')
    def disableCloudAxis(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'disableCloudAxis')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnParticleSystem, 'emission')
    def emission(self, emissions):
        # type: (List[datatypes.Vector]) -> None
        do, final_do, outTypes = _f.getDoArgs([emissions], [('emissions', 'MVectorArray', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'emission', final_do)
        return res

    @_f.addApiDocs(_api.MFnParticleSystem, 'emit')
    def emit(self, v):
        # type: (datatypes.Point) -> None
        do, final_do, outTypes = _f.getDoArgs([v], [('v', 'MPoint', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'emit', final_do)
        return res

    @_f.addApiDocs(_api.MFnParticleSystem, 'evaluateDynamics')
    def evaluateDynamics(self, to, runupFromStart):
        # type: (datatypes.Time, bool) -> None
        do, final_do, outTypes = _f.getDoArgs([to, runupFromStart], [('to', 'MTime', 'in', None), ('runupFromStart', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'evaluateDynamics', final_do)
        return res

    @_f.addApiDocs(_api.MFnParticleSystem, 'flatShaded')
    def flatShaded(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'flatShaded')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addMelDocs('particle', 'floatValue')
    def floatValue(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, effects.particle, kwargs, 'floatValue', val)

    @_f.addMelDocs('particle', 'attribute')
    def getAttribute(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, effects.particle, kwargs, 'attribute')
        return res

    @_f.addMelDocs('particle', 'cache')
    def getCache(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, effects.particle, kwargs, 'cache')
        return res

    @_f.addMelDocs('particle', 'conserve')
    def getConserve(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, effects.particle, kwargs, 'conserve')
        return res

    @_f.addApiDocs(_api.MFnParticleSystem, 'count')
    def getCount(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'count')
        return _f.ApiArgUtil._castResult(self, res, 'uint', None)

    @_f.addMelDocs('particle', 'dynamicAttrList')
    def getDynamicAttrList(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, effects.particle, kwargs, 'dynamicAttrList')
        return res

    @_f.addMelDocs('particle', 'gridSpacing')
    def getGridSpacing(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, effects.particle, kwargs, 'gridSpacing')
        return res

    @_f.addMelDocs('particle', 'inherit')
    def getInherit(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, effects.particle, kwargs, 'inherit')
        return res

    @_f.addMelDocs('particle', 'jitterBasePoint')
    def getJitterBasePoint(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, effects.particle, kwargs, 'jitterBasePoint')
        return res

    @_f.addMelDocs('particle', 'jitterRadius')
    def getJitterRadius(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, effects.particle, kwargs, 'jitterRadius')
        return res

    @_f.addMelDocs('particle', 'lowerLeft')
    def getLowerLeft(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, effects.particle, kwargs, 'lowerLeft')
        return res

    @_f.addMelDocs('particle', 'numJitters')
    def getNumJitters(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, effects.particle, kwargs, 'numJitters')
        return res

    @_f.addMelDocs('particle', 'order')
    def getOrder(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, effects.particle, kwargs, 'order')
        return res

    @_f.addMelDocs('particle', 'particleId')
    def getParticleId(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, effects.particle, kwargs, 'particleId')
        return res

    @_f.addApiDocs(_api.MFnParticleSystem, 'getPerParticleAttribute')
    def getPerParticleAttribute(self, attrName):
        # type: (str) -> Tuple[int, List[int]]
        do, final_do, outTypes = _f.getDoArgs([attrName], [('attrName', 'MString', 'in', None), ('array', 'MIntArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'getPerParticleAttribute', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'uint', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.addMelDocs('particle', 'perParticleDouble')
    def getPerParticleDouble(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, effects.particle, kwargs, 'perParticleDouble')
        return res

    @_f.addMelDocs('particle', 'perParticleVector')
    def getPerParticleVector(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, effects.particle, kwargs, 'perParticleVector')
        return res

    @_f.addMelDocs('particle', 'shapeName')
    def getShapeName(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, effects.particle, kwargs, 'shapeName')
        return res

    @_f.addMelDocs('particle', 'upperRight')
    def getUpperRight(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, effects.particle, kwargs, 'upperRight')
        return res

    @_f.addApiDocs(_api.MFnParticleSystem, 'hasEmission')
    def hasEmission(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'hasEmission')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnParticleSystem, 'hasLifespan')
    def hasLifespan(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'hasLifespan')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnParticleSystem, 'hasOpacity')
    def hasOpacity(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'hasOpacity')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnParticleSystem, 'hasRgb')
    def hasRgb(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'hasRgb')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnParticleSystem, 'isDeformedParticleShape')
    def isDeformedParticleShape(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'isDeformedParticleShape')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnParticleSystem, 'isPerParticleDoubleAttribute')
    def isPerParticleDoubleAttribute(self, attrName):
        # type: (str) -> bool
        do, final_do, outTypes = _f.getDoArgs([attrName], [('attrName', 'MString', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'isPerParticleDoubleAttribute', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res

    @_f.addApiDocs(_api.MFnParticleSystem, 'isPerParticleIntAttribute')
    def isPerParticleIntAttribute(self, attrName):
        # type: (str) -> bool
        do, final_do, outTypes = _f.getDoArgs([attrName], [('attrName', 'MString', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'isPerParticleIntAttribute', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res

    @_f.addApiDocs(_api.MFnParticleSystem, 'isPerParticleVectorAttribute')
    def isPerParticleVectorAttribute(self, attrName):
        # type: (str) -> bool
        do, final_do, outTypes = _f.getDoArgs([attrName], [('attrName', 'MString', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'isPerParticleVectorAttribute', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res

    @_f.addApiDocs(_api.MFnParticleSystem, 'isValid')
    def isValid(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'isValid')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnParticleSystem, 'lifespan')
    def lifespan(self, lifespans):
        # type: (List[float]) -> None
        do, final_do, outTypes = _f.getDoArgs([lifespans], [('lifespans', 'MDoubleArray', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'lifespan', final_do)
        return res

    @_f.addApiDocs(_api.MFnParticleSystem, 'mass')
    def mass(self):
        # type: () -> List[float]
        do, final_do, outTypes = _f.getDoArgs([], [('array', 'MDoubleArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'mass', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnParticleSystem, 'opacity')
    def opacity(self, opacities):
        # type: (List[float]) -> None
        do, final_do, outTypes = _f.getDoArgs([opacities], [('opacities', 'MDoubleArray', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'opacity', final_do)
        return res

    @_f.addApiDocs(_api.MFnParticleSystem, 'originalParticleShape')
    def originalParticleShape(self):
        # type: () -> DependNode
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'originalParticleShape')
        return _f.ApiArgUtil._castResult(self, res, 'MObject', None)

    @_f.addApiDocs(_api.MFnParticleSystem, 'particleIds')
    def particleIds(self):
        # type: () -> List[int]
        do, final_do, outTypes = _f.getDoArgs([], [('ids', 'MIntArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'particleIds', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnParticleSystem, 'particleName')
    def particleName(self):
        # type: () -> str
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'particleName')
        return _f.ApiArgUtil._castResult(self, res, 'MString', None)

    @_f.addApiDocs(_api.MFnParticleSystem, 'position')
    def position(self, positions):
        # type: (List[datatypes.Vector]) -> None
        do, final_do, outTypes = _f.getDoArgs([positions], [('positions', 'MVectorArray', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'position', final_do)
        return res

    @_f.addApiDocs(_api.MFnParticleSystem, 'position0')
    def position0(self, positions):
        # type: (List[datatypes.Vector]) -> None
        do, final_do, outTypes = _f.getDoArgs([positions], [('positions', 'MVectorArray', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'position0', final_do)
        return res

    @_f.addApiDocs(_api.MFnParticleSystem, 'position1')
    def position1(self, positions):
        # type: (List[datatypes.Vector]) -> None
        do, final_do, outTypes = _f.getDoArgs([positions], [('positions', 'MVectorArray', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'position1', final_do)
        return res

    @_f.addApiDocs(_api.MFnParticleSystem, 'primaryVisibility')
    def primaryVisibility(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'primaryVisibility')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnParticleSystem, 'radius')
    def radius(self, radii):
        # type: (List[float]) -> None
        do, final_do, outTypes = _f.getDoArgs([radii], [('radii', 'MDoubleArray', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'radius', final_do)
        return res

    @_f.addApiDocs(_api.MFnParticleSystem, 'radius0')
    def radius0(self, radii):
        # type: (List[float]) -> None
        do, final_do, outTypes = _f.getDoArgs([radii], [('radii', 'MDoubleArray', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'radius0', final_do)
        return res

    @_f.addApiDocs(_api.MFnParticleSystem, 'radius1')
    def radius1(self, radii):
        # type: (List[float]) -> None
        do, final_do, outTypes = _f.getDoArgs([radii], [('radii', 'MDoubleArray', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'radius1', final_do)
        return res

    @_f.addApiDocs(_api.MFnParticleSystem, 'receiveShadows')
    def receiveShadows(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'receiveShadows')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnParticleSystem, 'renderType')
    def renderType(self):
        # type: () -> ParticleSystem.RenderType
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'renderType')
        return _f.ApiArgUtil._castResult(self, res, ('MFnParticleSystem', 'RenderType'), None)

    @_f.addApiDocs(_api.MFnParticleSystem, 'rgb')
    def rgb(self, colors):
        # type: (List[datatypes.Vector]) -> None
        do, final_do, outTypes = _f.getDoArgs([colors], [('colors', 'MVectorArray', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'rgb', final_do)
        return res

    @_f.addApiDocs(_api.MFnParticleSystem, 'saveInitialState')
    def saveInitialState(self):
        # type: () -> None
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'saveInitialState')
        return res

    @_f.addMelDocs('particle', 'attribute')
    def setAttribute(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, effects.particle, kwargs, 'attribute', val)

    @_f.addMelDocs('particle', 'cache')
    def setCache(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, effects.particle, kwargs, 'cache', val)

    @_f.addMelDocs('particle', 'conserve')
    def setConserve(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, effects.particle, kwargs, 'conserve', val)

    @_f.addApiDocs(_api.MFnParticleSystem, 'setCount')
    def setCount(self, count):
        # type: (int) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([count], [('count', 'uint', 'in', None)], self.getCount, self.setCount, [])
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'setCount', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addMelDocs('particle', 'inherit')
    def setInherit(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, effects.particle, kwargs, 'inherit', val)

    @_f.addMelDocs('particle', 'order')
    def setOrder(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, effects.particle, kwargs, 'order', val)

    @_f.addMelDocs('particle', 'particleId')
    def setParticleId(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, effects.particle, kwargs, 'particleId', val)

    @_f.addApiDocs(_api.MFnParticleSystem, 'setPerParticleAttribute')
    def setPerParticleAttribute(self, attrName, array):
        # type: (str, List[datatypes.Vector]) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([attrName, array], [('attrName', 'MString', 'in', None), ('array', 'MVectorArray', 'in', None)], self.getPerParticleAttribute, self.setPerParticleAttribute, ['attrName'])
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'setPerParticleAttribute', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addMelDocs('particle', 'shapeName')
    def setShapeName(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, effects.particle, kwargs, 'shapeName', val)

    @_f.addApiDocs(_api.MFnParticleSystem, 'surfaceShading')
    def surfaceShading(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'surfaceShading')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addApiDocs(_api.MFnParticleSystem, 'tailSize')
    def tailSize(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'tailSize')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addApiDocs(_api.MFnParticleSystem, 'threshold')
    def threshold(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'threshold')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addMelDocs('particle', 'vectorValue')
    def vectorValue(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, effects.particle, kwargs, 'vectorValue', val)

    @_f.addApiDocs(_api.MFnParticleSystem, 'velocity')
    def velocity(self):
        # type: () -> List[datatypes.Vector]
        do, final_do, outTypes = _f.getDoArgs([], [('array', 'MVectorArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'velocity', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnParticleSystem, 'visibleInReflections')
    def visibleInReflections(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'visibleInReflections')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnParticleSystem, 'visibleInRefractions')
    def visibleInRefractions(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnParticleSystem, 'visibleInRefractions')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)
# ------ Do not edit above this line --------


class SelectionSet(with_metaclass(_factories.MetaMayaTypeRegistry, _api.MSelectionList)):
    apicls = _api.MSelectionList

    def __init__(self, objs):
        # type: (Union[_api.MSelectionList, SelectionSet, ObjectSet]) -> None
        """
        can be initialized from a list of objects, another SelectionSet, an
        MSelectionList, or an ObjectSet
        """
        if isinstance(objs, _api.MSelectionList):
            _api.MSelectionList.__init__(self, objs)

        elif isinstance(objs, ObjectSet):
            _api.MSelectionList.__init__(self, objs.asSelectionSet())

        else:
            _api.MSelectionList.__init__(self)
            for obj in objs:
                if isinstance(obj, (DependNode, DagNode)):
                    self.apicls.add(self, obj.__apiobject__())
                elif isinstance(obj, general.Attribute):
                    self.apicls.add(self, obj.__apiobject__(), True)
    #            elif isinstance(obj, Component):
    #                sel.add( obj.__apiobject__(), True )
                elif isinstance(obj, basestring):
                    self.apicls.add(self, obj)
                else:
                    raise TypeError

    def __melobject__(self):
        # If the list contains components, THEIR __melobject__ is a list -
        # so need to iterate through, and flatten if needed
        melList = []
        for selItem in self:
            selItem = selItem.__melobject__()
            if _util.isIterable(selItem):
                melList.extend(selItem)
            else:
                melList.append(selItem)
        return melList

    def __len__(self):
        # type: () -> int
        return self.apicls.length(self)

    def __contains__(self, item):
        # type: (Any) -> bool
        if isinstance(item, (DependNode, DagNode, general.Attribute)):
            return self.apicls.hasItem(self, item.__apiobject__())
        elif isinstance(item, general.Component):
            raise NotImplementedError('Components not yet supported')
        else:
            return self.apicls.hasItem(self, general.PyNode(item).__apiobject__())

    def __repr__(self):
        # type: () -> str
        names = []
        self.apicls.getSelectionStrings(self, names)
        return 'nt.%s(%s)' % (self.__class__.__name__, names)

    def __getitem__(self, index):
        # type: (Any) -> general.SelectableTypes
        if index >= len(self):
            raise IndexError("index out of range")

        plug = _api.MPlug()
        obj = _api.MObject()
        dag = _api.MDagPath()
        comp = _api.MObject()

        # Go from most specific to least - plug, dagPath, dependNode
        try:
            self.apicls.getPlug(self, index, plug)
            assert not plug.isNull()
        except (RuntimeError, AssertionError):
            try:
                self.apicls.getDagPath(self, index, dag, comp)
            except RuntimeError:
                try:
                    self.apicls.getDependNode(self, index, obj)
                    return general.PyNode(obj)
                except:
                    pass
            else:
                if comp.isNull():
                    return general.PyNode(dag)
                else:
                    return general.PyNode(dag, comp)
        else:
            return general.PyNode(plug)

    def __setitem__(self, index, item):

        if isinstance(item, (DependNode, DagNode, general.Attribute)):
            return self.apicls.replace(self, index, item.__apiobject__())
        elif isinstance(item, general.Component):
            raise NotImplementedError('Components not yet supported')
        else:
            return self.apicls.replace(self, general.PyNode(item).__apiobject__())

    def __and__(self, s):
        "operator for `SelectionSet.getIntersection`"
        return self.getIntersection(s)

    def __iand__(self, s):
        "operator for `SelectionSet.intersection`"
        return self.intersection(s)

    def __or__(self, s):
        "operator for `SelectionSet.getUnion`"
        return self.getUnion(s)

    def __ior__(self, s):
        "operator for `SelectionSet.union`"
        return self.union(s)

    def __lt__(self, s):
        "operator for `SelectionSet.isSubSet`"
        return self.isSubSet(s)

    def __gt__(self, s):
        "operator for `SelectionSet.isSuperSet`"
        return self.isSuperSet(s)

    def __sub__(self, s):
        "operator for `SelectionSet.getDifference`"
        return self.getDifference(s)

    def __isub__(self, s):
        "operator for `SelectionSet.difference`"
        return self.difference(s)

    def __xor__(self, s):
        "operator for `SelectionSet.symmetricDifference`"
        return self.getSymmetricDifference(s)

    def __ixor__(self, s):
        "operator for `SelectionSet.symmetricDifference`"
        return self.symmetricDifference(s)

    def add(self, item):
        if isinstance(item, (DependNode, DagNode, general.Attribute)):
            return self.apicls.add(self, item.__apiobject__())
        elif isinstance(item, general.Component):
            raise NotImplementedError('Components not yet supported')
        else:
            return self.apicls.add(self, general.PyNode(item).__apiobject__())

    def pop(self, index):
        # type: (int) -> general.SelectableTypes
        if index >= len(self):
            raise IndexError("index out of range")
        return self.apicls.remove(self, index)

    def isSubSet(self, other):
        # type: (Any) -> bool
        """
        Returns
        -------
        bool
        """
        if isinstance(other, ObjectSet):
            other = other.asSelectionSet()
        return set(self).issubset(other)

    def isSuperSet(self, other, flatten=True):
        # type: (Any, Any) -> bool
        """
        Returns
        -------
        bool
        """
        if isinstance(other, ObjectSet):
            other = other.asSelectionSet()
        return set(self).issuperset(other)

    def getIntersection(self, other):
        # type: (Any) -> SelectionSet
        """
        Returns
        -------
        SelectionSet
        """
        # diff = self-other
        # intersect = self-diff
        diff = self.getDifference(other)
        return self.getDifference(diff)

    def intersection(self, other):
        # type: (Any) -> None
        diff = self.getDifference(other)
        self.difference(diff)

    def getDifference(self, other):
        # type: (Any) -> SelectionSet
        """
        Returns
        -------
        SelectionSet
        """
        # create a new SelectionSet so that we don't modify our current one
        newSet = SelectionSet(self)
        newSet.difference(other)
        return newSet

    def difference(self, other):
        # type: (Any) -> None
        if not isinstance(other, _api.MSelectionList):
            other = SelectionSet(other)
        self.apicls.merge(self, other, _api.MSelectionList.kRemoveFromList)

    def getUnion(self, other):
        # type: (Any) -> SelectionSet
        """
        Returns
        -------
        SelectionSet
        """
        newSet = SelectionSet(self)
        newSet.union(other)
        return newSet

    def union(self, other):
        # type: (Any) -> None
        if not isinstance(other, _api.MSelectionList):
            other = SelectionSet(other)
        self.apicls.merge(self, other, _api.MSelectionList.kMergeNormal)

    def getSymmetricDifference(self, other):
        # type: (Any) -> SelectionSet
        """
        Also known as XOR

        Returns
        -------
        SelectionSet
        """
        # create a new SelectionSet so that we don't modify our current one
        newSet = SelectionSet(self)
        newSet.symmetricDifference(other)
        return newSet

    def symmetricDifference(self, other):
        # type: (Any) -> None
        if not isinstance(other, _api.MSelectionList):
            other = SelectionSet(other)
        # FIXME: does kXOR exist?  completion says only kXORWithList exists
        self.apicls.merge(self, other, _api.MSelectionList.kXOR)

    def asObjectSet(self):
        return general.sets(self)
# ------ Do not edit below this line --------
    __slots__ = ()
    MergeStrategy = Enum('MergeStrategy', [('mergeNormal', 0), ('kMergeNormal', 0), ('XORWithList', 1), ('kXORWithList', 1), ('removeFromList', 2), ('kRemoveFromList', 2)], multiKeys=True)

    @_f.addApiDocs(_api.MSelectionList, 'clear')
    def clear(self):
        # type: () -> None
        res = _api.MSelectionList.clear(self)
        return res

    @_f.addApiDocs(_api.MSelectionList, 'getSelectionStrings')
    def getSelectionStrings(self):
        # type: () -> List[str]
        do, final_do, outTypes = _f.getDoArgs([], [('array', 'MStringArray', 'out', None)])
        res = _api.MSelectionList.getSelectionStrings(self, *final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MSelectionList, 'intersect')
    def intersect(self, other, expandToLeaves=False):
        # type: (SelectionSet, bool) -> None
        do, final_do, outTypes = _f.getDoArgs([other, expandToLeaves], [('other', 'MSelectionList', 'in', None), ('expandToLeaves', 'bool', 'in', None)])
        res = _api.MSelectionList.intersect(self, *final_do)
        return res

    @_f.addApiDocs(_api.MSelectionList, 'isEmpty')
    def isEmpty(self):
        # type: () -> bool
        res = _api.MSelectionList.isEmpty(self)
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MSelectionList, 'remove')
    def removeAtIndex(self, index):
        # type: (int) -> None
        do, final_do, outTypes = _f.getDoArgs([index], [('index', 'uint', 'in', None)])
        res = _api.MSelectionList.remove(self, *final_do)
        return res
# ------ Do not edit above this line --------
#    def intersect(self, other):
#        self.apicls.merge( other, _api.MSelectionList.kXORWithList )


class ObjectSet(Entity):

    """
    The ObjectSet class and `SelectionSet` class work together.  Both classes have a very similar interface,
    the primary difference is that the ObjectSet class represents connections to an objectSet node, while the
    `SelectionSet` class is a generic set, akin to pythons built-in `set`.


    create some sets:

        >>> from pymel.core import *
        >>> f=newFile(f=1) #start clean
        >>>
        >>> s = sets()  # create an empty set
        >>> s.union( ls( type=nt.Camera) )  # add some cameras to it
        >>> s.members()  # doctest: +SKIP
        [nt.Camera('sideShape'), nt.Camera('frontShape'), nt.Camera('topShape'), nt.Camera('perspShape')]
        >>> sel = s.asSelectionSet() # or as a SelectionSet
        >>> sel # doctest: +SKIP
        nt.SelectionSet(['sideShape', 'frontShape', 'topShape', 'perspShape'])
        >>> sorted(sel) # as a sorted list
        [nt.Camera('frontShape'), nt.Camera('perspShape'), nt.Camera('sideShape'), nt.Camera('topShape')]

    Operations between sets result in `SelectionSet` objects:

        >>> t = sets()  # create another set
        >>> t.add( 'perspShape' )  # add the persp camera shape to it
        >>> s.getIntersection(t)
        nt.SelectionSet(['perspShape'])
        >>> diff = s.getDifference(t)
        >>> diff #doctest: +SKIP
        nt.SelectionSet(['sideShape', 'frontShape', 'topShape'])
        >>> sorted(diff)
        [nt.Camera('frontShape'), nt.Camera('sideShape'), nt.Camera('topShape')]
        >>> s.isSuperSet(t)
        True
    """


#        >>> u = sets( s&t ) # intersection
#        >>> print u.elements(), s.elements()
#        >>> if u < s: print "%s is a sub-set of %s" % (u, s)
#
#    place a set inside another, take1
#
#        >>> # like python's built-in set, the add command expects a single element
#        >>> s.add( t )
#
#    place a set inside another, take2
#
#        >>> # like python's built-in set, the update command expects a set or a list
#        >>> t.update([u])
#
#        >>> # put the sets back where they were
#        >>> s.remove(t)
#        >>> t.remove(u)
#
#    now put the **contents** of a set into another set
#
#        >>> t.update(u)
#
#    mixed operation between pymel.core.ObjectSet and built-in set
#
#        >>> v = set(['polyCube3', 'pSphere3'])
#        >>> print s.intersection(v)
#        >>> print v.intersection(s)  # not supported yet
#        >>> u.clear()
#
#        >>> delete( s )
#        >>> delete( t )
#        >>> delete( u )
#
#
#    these will return the results of the operation as python sets containing lists of pymel node classes::
#
#        s&t     # s.intersection(t)
#        s|t     # s.union(t)
#        s^t     # s.symmetric_difference(t)
#        s-t     # s.difference(t)
#
#    the following will alter the contents of the maya set::
#
#        s&=t    # s.intersection_update(t)
#        s|=t    # s.update(t)
#        s^=t    # s.symmetric_difference_update(t)
#        s-=t    # s.difference_update(t)
#
#    def _elements(self):
#        """ used internally to get a list of elements without casting to node classes"""
#        return sets( self, q=True)
#    # ----------------------
#    # Maya Methods
#    # ----------------------

    # ----------------------
    # Python ObjectSet Methods
    # ----------------------

    @classmethod
    def _getApiObjs(cls, item, tryCast=True):
        """
        Returns a tuple of api objects suitable (after unpacking) for
        feeding to most of the MFnSet methods (ie, remove, isMember, etc)
        """
        if isinstance(item, DagNode):
            return (item.__apimdagpath__(), _api.MObject())
        elif isinstance(item, (DependNode, general.Attribute)):
            return (item.__apiobject__(), )
        elif isinstance(item, general.Component):
            return (item.__apimdagpath__(), item.__apimobject__())
        elif tryCast:
            return cls._getApiObjs(general.PyNode(item), tryCast=False)
        else:
            raise TypeError(item)

    def __contains__(self, item):
        # type: (Any) -> bool
        """
        Returns
        -------
        bool
        """
        return self.__apimfn__().isMember(*self._getApiObjs(item))

    def __getitem__(self, index):
        return self.asSelectionSet()[index]

    def __len__(self):
        # type: () -> int
        """
        Returns
        -------
        int
        """
        return cmds.sets(self, q=1, size=1)

    # def __eq__(self, s):
    #    return s == self._elements()

    # def __ne__(self, s):
    #    return s != self._elements()

    def __and__(self, s):
        # type: (Union[str, ObjectSet]) -> SelectionSet
        "operator for `ObjectSet.getIntersection`"
        return self.getIntersection(s)

    def __iand__(self, s):
        # type: (Union[str, ObjectSet]) -> SelectionSet
        "operator for `ObjectSet.intersection`"
        return self.intersection(s)

    def __or__(self, s):
        # type: (Union[str, ObjectSet]) -> SelectionSet
        "operator for `ObjectSet.getUnion`"
        return self.getUnion(s)

    def __ior__(self, s):
        # type: (Union[str, ObjectSet]) -> SelectionSet
        "operator for `ObjectSet.union`"
        return self.union(s)

#    def __lt__(self, s):
#        "operator for `ObjectSet.isSubSet`"
#        return self.isSubSet(s)
#
#    def __gt__(self, s):
#        "operator for `ObjectSet.isSuperSet`"
#        return self.isSuperSet(s)

    def __sub__(self, s):
        "operator for `ObjectSet.getDifference`"
        return self.getDifference(s)

    def __isub__(self, s):
        "operator for `ObjectSet.difference`"
        return self.difference(s)

    def __xor__(self, s):
        "operator for `ObjectSet.symmetricDifference`"
        return self.getSymmetricDifference(s)

    def __ixor__(self, s):
        "operator for `ObjectSet.symmetricDifference`"
        return self.symmetricDifference(s)

#
#    def subtract(self, set2):
#        return sets( self, subtract=set2 )
#
#    def add(self, element):
#        return sets( self, add=[element] )
#
#    def clear(self):
#        return sets( self, clear=True )
#
#    def copy(self ):
#        return sets( self, copy=True )
#
#    def difference(self, elements):
#        if isinstance(elements,basestring):
#            elements = cmds.sets( elements, q=True)
#        return list(set(self.elements()).difference(elements))
#
#        '''
#        if isinstance(s, ObjectSet) or isinstance(s, str):
#            return sets( s, subtract=self )
#
#        s = sets( s )
#        res = sets( s, subtract=self )
#        cmds.delete(s)
#        return res'''
#
#    def difference_update(self, elements ):
#        return sets( self, remove=elements)
#
#    def discard( self, element ):
#        try:
#            return self.remove(element)
#        except TypeError:
#            pass
#
#    def intersection(self, elements):
#        if isinstance(elements,basestring):
#            elements = cmds.sets( elements, q=True)
#        return set(self.elements()).intersection(elements)
#
#    def intersection_update(self, elements):
#        self.clear()
#        sets( self, add=self.intersections(elements) )
#
#
#    def remove( self, element ):
#        return sets( self, remove=[element])
#
#    def symmetric_difference(self, elements):
#        if isinstance(elements,basestring):
#            elements = cmds.sets( elements, q=True)
#        return set(self.elements()).symmetric_difference(elements)
#
#    def union( self, elements ):
#        if isinstance(elements,basestring):
#            elements = cmds.sets( elements, q=True)
#        return set(self.elements()).union(elements)
#
#    def update( self, set2 ):
#        sets( self, forceElement=set2 )

    def forceElement(self, member):
        """
        Forces addition of the items to the set. If the items are in
        another set which is in the same partition as the given set,
        the items will be removed from the other set in order to keep the
        sets in the partition mutually exclusive with respect to membership."""
        cmds.sets(member, forceElement=self)

    def members(self, flatten=False):
        # type: (Any) -> List[general.SelectableTypes]
        """
        return members as a list

        Returns
        -------
        List[general.PyNode]
        """
        return list(self.asSelectionSet(flatten))

    @_warnings.deprecated('Use ObjectSet.members instead', 'ObjectSet')
    def elements(self, flatten=False):
        # type: (bool) -> List[general.SelectableTypes]
        """
        return members as a list
        """
        return list(self.asSelectionSet(flatten))

    def flattened(self):
        # type: () -> List[general.SelectableTypes]
        """
        return a flattened list of members.

        equivalent to `ObjectSet.members(flatten=True)`
        """
        return self.members(flatten=True)

    def resetTo(self, newContents):
        """
        clear and set the members to the passed list/set
        """
        self.clear()
        self.addMembers(newContents)

    def add(self, item):
        return self.__apimfn__().addMember(*self._getApiObjs(item))

    # alias for backwards compatibility
    addMember = add

    def remove(self, item):
        try:
            return self.__apimfn__().removeMember(*self._getApiObjs(item))
        except RuntimeError:
            # Provide a more informative error if object is not in set
            if item not in self:
                try:
                    itemStr = repr(item)
                except Exception:
                    itemStr = 'item'
                raise ValueError("%s not in set %r" % (itemStr, self))
            else:
                raise

    # alias for backwards compatibility
    removeMember = remove

    def isSubSet(self, other):
        # type: (Union[str, ObjectSet]) -> bool
        """
        Parameters
        ----------
        other : Union[str, ObjectSet]

        Returns
        -------
        bool
        """
        return self.asSelectionSet().isSubSet(other)

    def isSuperSet(self, other):
        # type: (Union[str, ObjectSet]) -> bool
        """
        Parameters
        ----------
        other : Union[str, ObjectSet]

        Returns
        -------
        bool
        """
        return self.asSelectionSet().isSuperSet(other)

    def isEqual(self, other):
        # type: (Union[str, ObjectSet]) -> bool
        """
        do not use __eq__ to test equality of set contents. __eq__ will only tell you if
        the passed object is the same node, not if this set and the passed set
        have the same contents.

        Parameters
        ----------
        other : Union[str, ObjectSet]

        Returns
        -------
        bool
        """
        return self.asSelectionSet() == SelectionSet(other)

    def getDifference(self, other):
        # type: (Union[str, ObjectSet]) -> SelectionSet
        """
        Parameters
        ----------
        other : Union[str, ObjectSet]

        Returns
        -------
        SelectionSet
        """
        sel = self.asSelectionSet()
        sel.difference(other)
        return sel

    def difference(self, other):
        # type: (Union[str, ObjectSet]) -> None
        sel = self.getDifference(other)
        self.resetTo(sel)

    def getSymmetricDifference(self, other):
        # type: (Union[str, ObjectSet]) -> SelectionSet
        """
        also known as XOR

        Parameters
        ----------
        other : Union[str, ObjectSet]

        Returns
        -------
        SelectionSet
        """
        sel = self.getSymmetricDifference()
        sel.difference(other)
        return sel

    def symmetricDifference(self, other):
        sel = self.symmetricDifference(other)
        self.resetTo(sel)

    def getIntersection(self, other):
        # type: (Union[str, ObjectSet]) -> SelectionSet
        """
        Parameters
        ----------
        other : Union[str, ObjectSet]

        Returns
        -------
        SelectionSet
        """
        if isinstance(other, ObjectSet):
            return self._getIntersection(other)

        # elif isinstance(other, SelectionSet) or hasattr(other, '__iter__'):
        selSet = self.asSelectionSet()
        selSet.intersection(other)
        return selSet

        #raise TypeError, 'Cannot perform intersection with non-iterable type %s' % type(other)

    def intersection(self, other):
        # type: (Union[str, ObjectSet]) -> None
        sel = self.getIntersection(other)
        self.resetTo(sel)

    def getUnion(self, other):
        # type: (Union[str, ObjectSet]) -> SelectionSet
        """
        Parameters
        ----------
        other : Union[str, ObjectSet]

        Returns
        -------
        SelectionSet
        """
        if isinstance(other, ObjectSet):
            return self._getUnion(other)

        selSet = self.asSelectionSet()
        selSet.union(other)
        return selSet

    def union(self, other):
        # type: (Union[str, ObjectSet]) -> None
        self.addMembers(other)

    def isRenderable(self):
        '''Mimics cmds.sets(self, q=True, renderable=True).

        Alternatively you can use isinstance(someset, pm.nt.ShadingEngine)
        since shadingEngine is the only renderable set in maya now
        '''
        return bool(cmds.sets(self, q=True, r=True))
# ------ Do not edit below this line --------
    __apicls__ = _api.MFnSet
    __melnode__ = 'objectSet'
    __slots__ = ()
    Restriction = Enum('Restriction', [('none', 0), ('kNone', 0), ('verticesOnly', 1), ('kVerticesOnly', 1), ('edgesOnly', 2), ('kEdgesOnly', 2), ('facetsOnly', 3), ('kFacetsOnly', 3), ('editPointsOnly', 4), ('kEditPointsOnly', 4), ('renderableOnly', 5), ('kRenderableOnly', 5)], multiKeys=True)

    @_f.addApiDocs(_api.MFnSet, 'getIntersection')
    def _getIntersection(self, withSet):
        # type: (str | DependNode) -> SelectionSet
        do, final_do, outTypes = _f.getDoArgs([withSet], [('withSet', 'MObject', 'in', None), ('result', 'MSelectionList', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnSet, 'getIntersection', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnSet, 'getUnion')
    def _getUnion(self, withSet):
        # type: (str | DependNode) -> SelectionSet
        do, final_do, outTypes = _f.getDoArgs([withSet], [('withSet', 'MObject', 'in', None), ('result', 'MSelectionList', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnSet, 'getUnion', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnSet, 'addMembers')
    def addMembers(self, list):
        # type: (SelectionSet) -> None
        do, final_do, outTypes = _f.getDoArgs([list], [('list', 'MSelectionList', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnSet, 'addMembers', final_do)
        return res

    @_f.addApiDocs(_api.MFnSet, 'getMembers')
    def asSelectionSet(self, flatten=False):
        # type: (bool) -> SelectionSet
        do, final_do, outTypes = _f.getDoArgs([flatten], [('members', 'MSelectionList', 'out', None), ('flatten', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnSet, 'getMembers', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnSet, 'clear')
    def clear(self):
        # type: () -> None
        res = _f.getProxyResult(self, _api.MFnSet, 'clear')
        return res

    @_f.addApiDocs(_api.MFnSet, 'annotation')
    def getAnnotation(self):
        # type: () -> str
        res = _f.getProxyResult(self, _api.MFnSet, 'annotation')
        return _f.ApiArgUtil._castResult(self, res, 'MString', None)

    @_f.addApiDocs(_api.MFnSet, 'getMemberPaths')
    def getMemberPaths(self, shading=False):
        # type: (bool) -> List[DagNode]
        do, final_do, outTypes = _f.getDoArgs([shading], [('members', 'MDagPathArray', 'out', None), ('shading', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnSet, 'getMemberPaths', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnSet, 'hasRestrictions')
    def hasRestrictions(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnSet, 'hasRestrictions')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnSet, 'intersectsWith')
    def intersectsWith(self, otherSet):
        # type: (str | DependNode) -> bool
        do, final_do, outTypes = _f.getDoArgs([otherSet], [('otherSet', 'MObject', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnSet, 'intersectsWith', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res

    @_f.addApiDocs(_api.MFnSet, 'isMember')
    def isMember(self, object):
        # type: (str | DependNode) -> bool
        do, final_do, outTypes = _f.getDoArgs([object], [('object', 'MObject', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnSet, 'isMember', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res

    @_f.addApiDocs(_api.MFnSet, 'removeMembers')
    def removeMembers(self, list):
        # type: (SelectionSet) -> None
        do, final_do, outTypes = _f.getDoArgs([list], [('list', 'MSelectionList', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnSet, 'removeMembers', final_do)
        return res

    @_f.addApiDocs(_api.MFnSet, 'restriction')
    def restriction(self):
        # type: () -> Set.Restriction
        res = _f.getProxyResult(self, _api.MFnSet, 'restriction')
        return _f.ApiArgUtil._castResult(self, res, ('MFnSet', 'Restriction'), None)

    @_f.addApiDocs(_api.MFnSet, 'setAnnotation')
    def setAnnotation(self, annotation):
        # type: (str) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([annotation], [('annotation', 'MString', 'in', None)], self.getAnnotation, self.setAnnotation, [])
        res = _f.getProxyResult(self, _api.MFnSet, 'setAnnotation', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res
# ------ Do not edit above this line --------


class ShadingEngine(ObjectSet):

    @classmethod
    def _getApiObjs(cls, item, tryCast=True):
        # Since shading groups can't contain transforms, as a convenience,
        # use getShape on any transforms
        if isinstance(item, Transform):
            shape = item.getShape()
            if shape:
                return cls._getApiObjs(shape)
            else:
                try:
                    itemStr = repr(item)
                except Exception:
                    itemStr = 'item'
                raise TypeError("%s has no shape, and %s objects cannot contain Transforms" % (itemStr, cls.__name__))
        else:
            return super(ShadingEngine, cls)._getApiObjs(item, tryCast=tryCast)
# ------ Do not edit below this line --------
    __melnode__ = 'shadingEngine'
    __slots__ = ()
# ------ Do not edit above this line --------


class AnimLayer(ObjectSet):

    def getAttribute(self):
        '''
        Retrieve the attributes animated on this AnimLayer
        '''
        # Unfortunately, cmds.animLayer('MyAnimLayer', q=1, attribute=1)
        # returns none unique attribute names, ie,
        #   MyNode.myAttr
        # even if there are foo|MyNode and bar|MyNode in the scene, and there
        # doesn't seem to be a flag to tell it to give unique / full paths.
        # Therefore, query it ourselves, by gettin inputs to dagSetMembers.
        # Testing has shown that animLayers only use dagSetMembers, and never
        # dnSetMembers - if you add a non-dag node to an animLayer, it makes
        # a connection to dagSetMembers; and even if you manually make a connection
        # to dnSetMembers, those connections don't seem to show up in
        # animLayer(q=1, attribute=1)
        return self.attr('dagSetMembers').inputs(plugs=1)

    getAttributes = getAttribute
# ------ Do not edit below this line --------
    __melcmd__ = staticmethod(animation.animLayer)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'animLayer'
    __melnode__ = 'animLayer'
    __slots__ = ()

    @_f.addMelDocs('animLayer', 'copy')
    def copy(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.animLayer, kwargs, 'copy', val)

    @_f.addMelDocs('animLayer', 'copyAnimation')
    def copyAnimation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.animLayer, kwargs, 'copyAnimation', val)

    @_f.addMelDocs('animLayer', 'copyNoAnimation')
    def copyNoAnimation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.animLayer, kwargs, 'copyNoAnimation', val)

    @_f.addMelDocs('animLayer', 'extractAnimation')
    def extractAnimation(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.animLayer, kwargs, 'extractAnimation', val)

    @_f.addMelDocs('animLayer', 'addRelatedKG')
    def getAddRelatedKG(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.animLayer, kwargs, 'addRelatedKG')
        return res

    @_f.addMelDocs('animLayer', 'addSelectedObjects')
    def getAddSelectedObjects(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.animLayer, kwargs, 'addSelectedObjects')
        return res

    @_f.addMelDocs('animLayer', 'affectedLayers')
    def getAffectedLayers(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.animLayer, kwargs, 'affectedLayers')
        return res

    @_f.addMelDocs('animLayer', 'animCurves')
    def getAnimCurves(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.animLayer, kwargs, 'animCurves')
        return res

    @_f.addMelDocs('animLayer', 'baseAnimCurves')
    def getBaseAnimCurves(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.animLayer, kwargs, 'baseAnimCurves')
        return res

    @_f.addMelDocs('animLayer', 'bestAnimLayer')
    def getBestAnimLayer(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.animLayer, kwargs, 'bestAnimLayer')
        return res

    @_f.addMelDocs('animLayer', 'bestLayer')
    def getBestLayer(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.animLayer, kwargs, 'bestLayer')
        return res

    @_f.addMelDocs('animLayer', 'blendNodes')
    def getBlendNodes(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.animLayer, kwargs, 'blendNodes')
        return res

    @_f.addMelDocs('animLayer', 'children')
    def getChildren(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.animLayer, kwargs, 'children')
        return res

    @_f.addMelDocs('animLayer', 'collapse')
    def getCollapse(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.animLayer, kwargs, 'collapse')
        return res

    @_f.addMelDocs('animLayer', 'excludeBoolean')
    def getExcludeBoolean(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.animLayer, kwargs, 'excludeBoolean')
        return res

    @_f.addMelDocs('animLayer', 'excludeDynamic')
    def getExcludeDynamic(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.animLayer, kwargs, 'excludeDynamic')
        return res

    @_f.addMelDocs('animLayer', 'excludeEnum')
    def getExcludeEnum(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.animLayer, kwargs, 'excludeEnum')
        return res

    @_f.addMelDocs('animLayer', 'excludeRotate')
    def getExcludeRotate(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.animLayer, kwargs, 'excludeRotate')
        return res

    @_f.addMelDocs('animLayer', 'excludeScale')
    def getExcludeScale(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.animLayer, kwargs, 'excludeScale')
        return res

    @_f.addMelDocs('animLayer', 'excludeTranslate')
    def getExcludeTranslate(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.animLayer, kwargs, 'excludeTranslate')
        return res

    @_f.addMelDocs('animLayer', 'excludeVisibility')
    def getExcludeVisibility(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.animLayer, kwargs, 'excludeVisibility')
        return res

    @_f.addMelDocs('animLayer', 'exists')
    def getExists(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.animLayer, kwargs, 'exists')
        return res

    @_f.addMelDocs('animLayer', 'findCurveForPlug')
    def getFindCurveForPlug(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.animLayer, kwargs, 'findCurveForPlug')
        return res

    @_f.addMelDocs('animLayer', 'layeredPlug')
    def getLayeredPlug(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.animLayer, kwargs, 'layeredPlug')
        return res

    @_f.addMelDocs('animLayer', 'lock')
    def getLock(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.animLayer, kwargs, 'lock')
        return res

    @_f.addMelDocs('animLayer', 'maxLayers')
    def getMaxLayers(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.animLayer, kwargs, 'maxLayers')
        return res

    @_f.addMelDocs('animLayer', 'mute')
    def getMute(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.animLayer, kwargs, 'mute')
        return res

    @_f.addMelDocs('animLayer', 'override')
    def getOverride(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.animLayer, kwargs, 'override')
        return res

    @_f.addMelDocs('animLayer', 'parent')
    def getParent(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.animLayer, kwargs, 'parent')
        return res

    @_f.addMelDocs('animLayer', 'passthrough')
    def getPassthrough(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.animLayer, kwargs, 'passthrough')
        return res

    @_f.addMelDocs('animLayer', 'preferred')
    def getPreferred(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.animLayer, kwargs, 'preferred')
        return res

    @_f.addMelDocs('animLayer', 'root')
    def getRoot(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.animLayer, kwargs, 'root')
        return res

    @_f.addMelDocs('animLayer', 'selected')
    def getSelected(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.animLayer, kwargs, 'selected')
        return res

    @_f.addMelDocs('animLayer', 'solo')
    def getSolo(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.animLayer, kwargs, 'solo')
        return res

    @_f.addMelDocs('animLayer', 'weight')
    def getWeight(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, animation.animLayer, kwargs, 'weight')
        return res

    @_f.addMelDocs('animLayer', 'moveLayerAfter')
    def moveLayerAfter(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.animLayer, kwargs, 'moveLayerAfter', val)

    @_f.addMelDocs('animLayer', 'moveLayerBefore')
    def moveLayerBefore(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.animLayer, kwargs, 'moveLayerBefore', val)

    @_f.addMelDocs('animLayer', 'removeAllAttributes')
    def removeAllAttributes(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.animLayer, kwargs, 'removeAllAttributes', val)

    @_f.addMelDocs('animLayer', 'addRelatedKG')
    def setAddRelatedKG(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.animLayer, kwargs, 'addRelatedKG', val)

    @_f.addMelDocs('animLayer', 'addSelectedObjects')
    def setAddSelectedObjects(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.animLayer, kwargs, 'addSelectedObjects', val)

    @_f.addMelDocs('animLayer', 'animCurves')
    def setAnimCurves(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.animLayer, kwargs, 'animCurves', val)

    @_f.addMelDocs('animLayer', 'attribute')
    def setAttribute(self, val=True, **kwargs):
        # type: (str | List[str], **Any) -> None
        return _f.asEdit(self, animation.animLayer, kwargs, 'attribute', val)

    @_f.addMelDocs('animLayer', 'baseAnimCurves')
    def setBaseAnimCurves(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.animLayer, kwargs, 'baseAnimCurves', val)

    @_f.addMelDocs('animLayer', 'bestAnimLayer')
    def setBestAnimLayer(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.animLayer, kwargs, 'bestAnimLayer', val)

    @_f.addMelDocs('animLayer', 'blendNodes')
    def setBlendNodes(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.animLayer, kwargs, 'blendNodes', val)

    @_f.addMelDocs('animLayer', 'collapse')
    def setCollapse(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.animLayer, kwargs, 'collapse', val)

    @_f.addMelDocs('animLayer', 'excludeBoolean')
    def setExcludeBoolean(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.animLayer, kwargs, 'excludeBoolean', val)

    @_f.addMelDocs('animLayer', 'excludeDynamic')
    def setExcludeDynamic(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.animLayer, kwargs, 'excludeDynamic', val)

    @_f.addMelDocs('animLayer', 'excludeEnum')
    def setExcludeEnum(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.animLayer, kwargs, 'excludeEnum', val)

    @_f.addMelDocs('animLayer', 'excludeRotate')
    def setExcludeRotate(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.animLayer, kwargs, 'excludeRotate', val)

    @_f.addMelDocs('animLayer', 'excludeScale')
    def setExcludeScale(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.animLayer, kwargs, 'excludeScale', val)

    @_f.addMelDocs('animLayer', 'excludeTranslate')
    def setExcludeTranslate(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.animLayer, kwargs, 'excludeTranslate', val)

    @_f.addMelDocs('animLayer', 'excludeVisibility')
    def setExcludeVisibility(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.animLayer, kwargs, 'excludeVisibility', val)

    @_f.addMelDocs('animLayer', 'findCurveForPlug')
    def setFindCurveForPlug(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.animLayer, kwargs, 'findCurveForPlug', val)

    @_f.addMelDocs('animLayer', 'lock')
    def setLock(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.animLayer, kwargs, 'lock', val)

    @_f.addMelDocs('animLayer', 'mute')
    def setMute(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.animLayer, kwargs, 'mute', val)

    @_f.addMelDocs('animLayer', 'override')
    def setOverride(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.animLayer, kwargs, 'override', val)

    @_f.addMelDocs('animLayer', 'parent')
    def setParent(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.animLayer, kwargs, 'parent', val)

    @_f.addMelDocs('animLayer', 'passthrough')
    def setPassthrough(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.animLayer, kwargs, 'passthrough', val)

    @_f.addMelDocs('animLayer', 'preferred')
    def setPreferred(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.animLayer, kwargs, 'preferred', val)

    @_f.addMelDocs('animLayer', 'selected')
    def setSelected(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.animLayer, kwargs, 'selected', val)

    @_f.addMelDocs('animLayer', 'solo')
    def setSolo(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.animLayer, kwargs, 'solo', val)

    @_f.addMelDocs('animLayer', 'weight')
    def setWeight(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, animation.animLayer, kwargs, 'weight', val)

    @_f.addMelDocs('animLayer', 'writeBlendnodeDestinations')
    def writeBlendnodeDestinations(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.animLayer, kwargs, 'writeBlendnodeDestinations', val)
# ------ Do not edit above this line --------


class AnimCurve(DependNode):

    def addKeys(self, time, values, tangentInType='linear',
                tangentOutType='linear', unit=None, keepExistingKeys=False):
        if not unit:
            unit = _api.MTime.uiUnit()
        times = _api.MTimeArray()
        for frame in time:
            times.append(_api.MTime(frame, unit))
        keys = _api.MDoubleArray()
        for value in values:
            keys.append(value)
        tangentInType = self.TangentType.getIndex(tangentInType)
        tangentOutType = self.TangentType.getIndex(tangentOutType)
        change = _api.MAnimCurveChange()
        result = self.__apimfn__().addKeys(
            times, keys, tangentInType, tangentOutType, keepExistingKeys,
            change)
        undoItem = _factories.MAnimCurveChangeUndoItem(change)
        _factories.apiUndo.append(undoItem, undoName="addKeys")
        return result

    @_f.addApiDocs(_api.MFnAnimCurve, 'addKey', 0)
    def addKeyTU(self, time, value, tangentInType='global_', tangentOutType='global_'):
        # type: (Union[datatypes.Time, float], float, AnimCurve.TangentType, AnimCurve.TangentType) -> int
        time = datatypes.Time._inCast(time)
        value = float(value)
        tangentInType = self.TangentType.getIndex(tangentInType)
        tangentOutType = self.TangentType.getIndex(tangentOutType)
        change = _api.MAnimCurveChange()
        res = self.__apimfn__().addKey(time, value, tangentInType, tangentOutType, change)
        undoItem = _factories.MAnimCurveChangeUndoItem(change)
        _factories.apiUndo.append(undoItem, undoName="addKeyTU")
        return res

    @_f.addApiDocs(_api.MFnAnimCurve, 'addKey', 1)
    def addKeyTT(self, timeInput, timeValue, tangentInType='global_', tangentOutType='global_'):
        # type: (Union[datatypes.Time, float], Union[datatypes.Time, float], AnimCurve.TangentType, AnimCurve.TangentType) -> int
        timeInput = datatypes.Time._inCast(timeInput)
        timeValue = datatypes.Time._inCast(timeValue)
        tangentInType = self.TangentType.getIndex(tangentInType)
        tangentOutType = self.TangentType.getIndex(tangentOutType)
        change = _api.MAnimCurveChange()
        res = self.__apimfn__().addKey(timeInput, timeValue, tangentInType, tangentOutType, change)
        undoItem = _factories.MAnimCurveChangeUndoItem(change)
        _factories.apiUndo.append(undoItem, undoName="addKeyTT")
        return res

    @_f.addApiDocs(_api.MFnAnimCurve, 'addKey', 2)
    def addKeyUU(self, unitlessInput, value, tangentInType='global_', tangentOutType='global_'):
        # type: (float, float, AnimCurve.TangentType, AnimCurve.TangentType) -> int
        unitlessInput = float(unitlessInput)
        value = float(value)
        tangentInType = self.TangentType.getIndex(tangentInType)
        tangentOutType = self.TangentType.getIndex(tangentOutType)
        change = _api.MAnimCurveChange()
        res = self.__apimfn__().addKey(unitlessInput, value, tangentInType, tangentOutType, change)
        undoItem = _factories.MAnimCurveChangeUndoItem(change)
        _factories.apiUndo.append(undoItem, undoName="addKeyUU")
        return res

    @_f.addApiDocs(_api.MFnAnimCurve, 'addKey', 3)
    def addKeyUT(self, unitlessInput, timeValue, tangentInType='global_', tangentOutType='global_'):
        # type: (float, Union[datatypes.Time, float], AnimCurve.TangentType, AnimCurve.TangentType) -> int
        unitlessInput = float(unitlessInput)
        timeValue = datatypes.Time._inCast(timeValue)
        tangentInType = self.TangentType.getIndex(tangentInType)
        tangentOutType = self.TangentType.getIndex(tangentOutType)
        change = _api.MAnimCurveChange()
        res = self.__apimfn__().addKey(unitlessInput, timeValue, tangentInType, tangentOutType, change)
        undoItem = _factories.MAnimCurveChangeUndoItem(change)
        _factories.apiUndo.append(undoItem, undoName="addKeyUT")
        return res

    # because addKeyframe only works with TA/TL/TU, make it an alias for addKey
    def addKeyframe(self, *args, **kwargs):
        return self.addKey(*args, **kwargs)

    @_f.addApiDocs(_api.MFnAnimCurve, 'evaluate', 0)
    def evaluateTU(self, atTime):
        # type: (datatypes.Time) -> float
        atTime = datatypes.Time._inCast(atTime)
        return self.__apimfn__().evaluate(atTime)

    @_f.addApiDocs(_api.MFnAnimCurve, 'evaluate', 2)
    def evaluateTT(self, atTime):
        # type: (datatypes.Time) -> datatypes.Time
        atTime = datatypes.Time._inCast(atTime)
        timeValue = _api.MTime()
        self.__apimfn__().evaluate(atTime, timeValue)
        uiUnit = _api.MTime.uiUnit()
        return datatypes.Time(timeValue.asUnits(uiUnit), uiUnit)

    @_f.addApiDocs(_api.MFnAnimCurve, 'evaluate', 3)
    def evaluateUU(self, atUnitlessInput):
        # type: (float) -> float
        atUnitlessInput = float(atUnitlessInput)
        outValuePtr = _api.SafeApiPtr('double')
        self.__apimfn__().evaluate(atUnitlessInput, outValuePtr())
        return outValuePtr.get()

    @_f.addApiDocs(_api.MFnAnimCurve, 'evaluate', 4)
    def evaluateUT(self, atUnitlessInput):
        # type: (float) -> datatypes.Time
        atUnitlessInput = float(atUnitlessInput)
        timeValue = _api.MTime()
        self.__apimfn__().evaluate(atUnitlessInput, timeValue)
        uiUnit = _api.MTime.uiUnit()
        return datatypes.Time(timeValue.asUnits(uiUnit), uiUnit)

    @_f.addApiDocs(_api.MFnAnimCurve, 'findClosest', 0)
    def findClosestT(self, time):
        # type: (datatypes.Time) -> int
        time = datatypes.Time._inCast(time)
        return self.__apimfn__().findClosest(time)

    @_f.addApiDocs(_api.MFnAnimCurve, 'findClosest', 1)
    def findClosestU(self, unitlessInput):
        # type: (float) -> int
        unitlessInput = float(unitlessInput)
        return self.__apimfn__().findClosest(unitlessInput)

    def numKeyframes(self):
        # just because MFnAnimCurve.numKeyframes is deprecated...
        return self.numKeys()

    # originally wasn't intending to wrap these two, but they were due to a bug,
    # and we decided not to break backwards compatibility. However, we're
    # turning them into classmethods, as they make more sense that way
    @classmethod
    @_f.addApiDocs(_api.MFnAnimCurve, 'timedAnimCurveTypeForPlug')
    def timedAnimCurveTypeForPlug(cls, plug):
        # type: (general.Attribute) -> AnimCurve.AnimCurveType
        do, final_do, outTypes = _f.getDoArgs([plug], [('plug', 'MPlug', 'in', None)])
        res = cls.__apicls__().timedAnimCurveTypeForPlug(*final_do)
        return cls.AnimCurveType[res]

    @classmethod
    @_f.addApiDocs(_api.MFnAnimCurve, 'unitlessAnimCurveTypeForPlug')
    def unitlessAnimCurveTypeForPlug(cls, plug):
        # type: (general.Attribute) -> AnimCurve.AnimCurveType
        do, final_do, outTypes = _f.getDoArgs([plug], [('plug', 'MPlug', 'in', None)])
        res = cls.__apicls__().unitlessAnimCurveTypeForPlug(*final_do)
        return cls.AnimCurveType[res]
# ------ Do not edit below this line --------
    __apicls__ = _api.MFnAnimCurve
    __melnode__ = 'animCurve'
    __slots__ = ()
    AnimCurveType = Enum('AnimCurveType', [('TA', 0), ('kAnimCurveTA', 0), ('TL', 1), ('kAnimCurveTL', 1), ('TT', 2), ('kAnimCurveTT', 2), ('TU', 3), ('kAnimCurveTU', 3), ('UA', 4), ('kAnimCurveUA', 4), ('UL', 5), ('kAnimCurveUL', 5), ('UT', 6), ('kAnimCurveUT', 6), ('UU', 7), ('kAnimCurveUU', 7), ('unknown', 8), ('kAnimCurveUnknown', 8)], multiKeys=True)
    InfinityType = Enum('InfinityType', [('constant', 0), ('kConstant', 0), ('linear', 1), ('kLinear', 1), ('cycle', 3), ('kCycle', 3), ('cycleRelative', 4), ('kCycleRelative', 4), ('oscillate', 5), ('kOscillate', 5)], multiKeys=True)
    if versions.current() >= versions.v2022:
        TangentType = Enum('TangentType', [('global_', 0), ('kTangentGlobal', 0), ('fixed', 1), ('kTangentFixed', 1), ('linear', 2), ('kTangentLinear', 2), ('flat', 3), ('kTangentFlat', 3), ('smooth', 4), ('kTangentSmooth', 4), ('step', 5), ('kTangentStep', 5), ('slow', 6), ('kTangentSlow', 6), ('fast', 7), ('kTangentFast', 7), ('clamped', 8), ('kTangentClamped', 8), ('plateau', 9), ('kTangentPlateau', 9), ('stepNext', 10), ('kTangentStepNext', 10), ('auto', 11), ('kTangentAuto', 11), ('shared1', 19), ('kTangentShared1', 19), ('shared2', 20), ('kTangentShared2', 20), ('shared3', 21), ('kTangentShared3', 21), ('shared4', 22), ('kTangentShared4', 22), ('shared5', 23), ('kTangentShared5', 23), ('shared6', 24), ('kTangentShared6', 24), ('shared7', 25), ('kTangentShared7', 25), ('shared8', 26), ('kTangentShared8', 26), ('autoMix', 27), ('kTangentAutoMix', 27), ('autoEase', 28), ('kTangentAutoEase', 28), ('autoCustom', 29), ('kTangentAutoCustom', 29), ('customStart', 64), ('kTangentCustomStart', 64), ('customEnd', 32767), ('kTangentCustomEnd', 32767), ('typeCount', 32768), ('kTangentTypeCount', 32768)], multiKeys=True)
    else:
        TangentType = Enum('TangentType', [('global_', 0), ('kTangentGlobal', 0), ('fixed', 1), ('kTangentFixed', 1), ('linear', 2), ('kTangentLinear', 2), ('flat', 3), ('kTangentFlat', 3), ('smooth', 4), ('kTangentSmooth', 4), ('step', 5), ('kTangentStep', 5), ('slow', 6), ('kTangentSlow', 6), ('fast', 7), ('kTangentFast', 7), ('clamped', 8), ('kTangentClamped', 8), ('plateau', 9), ('kTangentPlateau', 9), ('stepNext', 10), ('kTangentStepNext', 10), ('auto', 11), ('kTangentAuto', 11), ('shared1', 19), ('kTangentShared1', 19), ('shared2', 20), ('kTangentShared2', 20), ('shared3', 21), ('kTangentShared3', 21), ('shared4', 22), ('kTangentShared4', 22), ('shared5', 23), ('kTangentShared5', 23), ('shared6', 24), ('kTangentShared6', 24), ('shared7', 25), ('kTangentShared7', 25), ('shared8', 26), ('kTangentShared8', 26), ('customStart', 64), ('kTangentCustomStart', 64), ('customEnd', 32767), ('kTangentCustomEnd', 32767), ('typeCount', 32768), ('kTangentTypeCount', 32768)], multiKeys=True)

    @_f.addApiDocs(_api.MFnAnimCurve, 'addKeysWithTangents')
    def addKeysWithTangents(self, timeArray, valueArray, tangentInType='global_', tangentOutType='global_', tangentInTypeArray=None, tangentOutTypeArray=None, tangentInXArray=None, tangentInYArray=None, tangentOutXArray=None, tangentOutYArray=None, tangentsLockedArray=None, weightsLockedArray=None, convertUnits=True, keepExistingKeys=False):
        # type: (List[datatypes.Time], List[float], AnimCurve.TangentType, AnimCurve.TangentType, List[int], List[int], List[float], List[float], List[float], List[float], List[int], List[int], bool, bool) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsAnimCurveUndo([timeArray, valueArray, tangentInType, tangentOutType, tangentInTypeArray, tangentOutTypeArray, tangentInXArray, tangentInYArray, tangentOutXArray, tangentOutYArray, tangentsLockedArray, weightsLockedArray, convertUnits, keepExistingKeys], [('timeArray', 'MTimeArray', 'in', None), ('valueArray', 'MDoubleArray', 'in', None), ('tangentInType', ('MFnAnimCurve', 'TangentType'), 'in', None), ('tangentOutType', ('MFnAnimCurve', 'TangentType'), 'in', None), ('tangentInTypeArray', 'MIntArray', 'in', None), ('tangentOutTypeArray', 'MIntArray', 'in', None), ('tangentInXArray', 'MDoubleArray', 'in', None), ('tangentInYArray', 'MDoubleArray', 'in', None), ('tangentOutXArray', 'MDoubleArray', 'in', None), ('tangentOutYArray', 'MDoubleArray', 'in', None), ('tangentsLockedArray', 'MIntArray', 'in', None), ('weightsLockedArray', 'MIntArray', 'in', None), ('convertUnits', 'bool', 'in', None), ('keepExistingKeys', 'bool', 'in', None), ('change', 'MAnimCurveChange', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnAnimCurve, 'addKeysWithTangents', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnAnimCurve, 'animCurveType')
    def animCurveType(self):
        # type: () -> AnimCurve.AnimCurveType
        res = _f.getProxyResult(self, _api.MFnAnimCurve, 'animCurveType')
        return _f.ApiArgUtil._castResult(self, res, ('MFnAnimCurve', 'AnimCurveType'), None)

    @_f.addApiDocs(_api.MFnAnimCurve, 'inTangentType')
    def getInTangentType(self, index):
        # type: (int) -> AnimCurve.TangentType
        do, final_do, outTypes = _f.getDoArgs([index], [('index', 'uint', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnAnimCurve, 'inTangentType', final_do)
        res = _f.ApiArgUtil._castResult(self, res, ('MFnAnimCurve', 'TangentType'), None)
        return res

    @_f.addApiDocs(_api.MFnAnimCurve, 'outTangentType')
    def getOutTangentType(self, index):
        # type: (int) -> AnimCurve.TangentType
        do, final_do, outTypes = _f.getDoArgs([index], [('index', 'uint', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnAnimCurve, 'outTangentType', final_do)
        res = _f.ApiArgUtil._castResult(self, res, ('MFnAnimCurve', 'TangentType'), None)
        return res

    @_f.addApiDocs(_api.MFnAnimCurve, 'postInfinityType')
    def getPostInfinityType(self):
        # type: () -> AnimCurve.InfinityType
        res = _f.getProxyResult(self, _api.MFnAnimCurve, 'postInfinityType')
        return _f.ApiArgUtil._castResult(self, res, ('MFnAnimCurve', 'InfinityType'), None)

    @_f.addApiDocs(_api.MFnAnimCurve, 'preInfinityType')
    def getPreInfinityType(self):
        # type: () -> AnimCurve.InfinityType
        res = _f.getProxyResult(self, _api.MFnAnimCurve, 'preInfinityType')
        return _f.ApiArgUtil._castResult(self, res, ('MFnAnimCurve', 'InfinityType'), None)

    @_f.addApiDocs(_api.MFnAnimCurve, 'quaternionW')
    def getQuaternionW(self, index):
        # type: (int) -> float
        do, final_do, outTypes = _f.getDoArgs([index], [('index', 'uint', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnAnimCurve, 'quaternionW', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'double', None)
        return res

    @_f.addApiDocs(_api.MFnAnimCurve, 'tangentsLocked')
    def getTangentsLocked(self, index):
        # type: (int) -> bool
        do, final_do, outTypes = _f.getDoArgs([index], [('index', 'uint', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnAnimCurve, 'tangentsLocked', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res

    @_f.addApiDocs(_api.MFnAnimCurve, 'time')
    def getTime(self, index):
        # type: (int) -> datatypes.Time
        do, final_do, outTypes = _f.getDoArgs([index], [('index', 'uint', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnAnimCurve, 'time', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MTime', None)
        return res

    @_f.addApiDocs(_api.MFnAnimCurve, 'unitlessInput')
    def getUnitlessInput(self, index):
        # type: (int) -> float
        do, final_do, outTypes = _f.getDoArgs([index], [('index', 'uint', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnAnimCurve, 'unitlessInput', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'double', None)
        return res

    @_f.addApiDocs(_api.MFnAnimCurve, 'value')
    def getValue(self, index):
        # type: (int) -> float
        do, final_do, outTypes = _f.getDoArgs([index], [('index', 'uint', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnAnimCurve, 'value', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'double', None)
        return res

    @_f.addApiDocs(_api.MFnAnimCurve, 'weightsLocked')
    def getWeightsLocked(self, index):
        # type: (int) -> bool
        do, final_do, outTypes = _f.getDoArgs([index], [('index', 'uint', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnAnimCurve, 'weightsLocked', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res

    @_f.addApiDocs(_api.MFnAnimCurve, 'insertKey')
    def insertKey(self, time, breakdown=False):
        # type: (datatypes.Time, bool) -> int
        do, final_do, outTypes, undoItem = _f.getDoArgsAnimCurveUndo([time, breakdown], [('time', 'MTime', 'in', None), ('breakdown', 'bool', 'in', None), ('change', 'MAnimCurveChange', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnAnimCurve, 'insertKey', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        res = _f.ApiArgUtil._castResult(self, res, 'uint', None)
        return res

    @_f.addApiDocs(_api.MFnAnimCurve, 'isBreakdown')
    def isBreakdown(self, index):
        # type: (int) -> bool
        do, final_do, outTypes = _f.getDoArgs([index], [('index', 'uint', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnAnimCurve, 'isBreakdown', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res

    @_f.addApiDocs(_api.MFnAnimCurve, 'isStatic')
    def isStatic(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnAnimCurve, 'isStatic')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnAnimCurve, 'isTimeInput')
    def isTimeInput(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnAnimCurve, 'isTimeInput')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnAnimCurve, 'isUnitlessInput')
    def isUnitlessInput(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnAnimCurve, 'isUnitlessInput')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnAnimCurve, 'isWeighted')
    def isWeighted(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnAnimCurve, 'isWeighted')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnAnimCurve, 'numKeys')
    def numKeys(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnAnimCurve, 'numKeys')
        return _f.ApiArgUtil._castResult(self, res, 'uint', None)

    @_f.addApiDocs(_api.MFnAnimCurve, 'remove')
    def remove(self, index):
        # type: (int) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsAnimCurveUndo([index], [('index', 'uint', 'in', None), ('change', 'MAnimCurveChange', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnAnimCurve, 'remove', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnAnimCurve, 'setAngle')
    def setAngle(self, index, angle, inTangent):
        # type: (int, datatypes.Angle, bool) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsAnimCurveUndo([index, angle, inTangent], [('index', 'uint', 'in', None), ('angle', 'MAngle', 'in', None), ('inTangent', 'bool', 'in', None), ('change', 'MAnimCurveChange', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnAnimCurve, 'setAngle', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnAnimCurve, 'setIsBreakdown')
    def setBreakdown(self, index, isBreakdown):
        # type: (int, bool) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsAnimCurveUndo([index, isBreakdown], [('index', 'uint', 'in', None), ('isBreakdown', 'bool', 'in', None), ('change', 'MAnimCurveChange', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnAnimCurve, 'setIsBreakdown', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnAnimCurve, 'setInTangentType')
    def setInTangentType(self, index, tangentType):
        # type: (int, AnimCurve.TangentType) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsAnimCurveUndo([index, tangentType], [('index', 'uint', 'in', None), ('tangentType', ('MFnAnimCurve', 'TangentType'), 'in', None), ('change', 'MAnimCurveChange', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnAnimCurve, 'setInTangentType', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnAnimCurve, 'setOutTangentType')
    def setOutTangentType(self, index, tangentType):
        # type: (int, AnimCurve.TangentType) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsAnimCurveUndo([index, tangentType], [('index', 'uint', 'in', None), ('tangentType', ('MFnAnimCurve', 'TangentType'), 'in', None), ('change', 'MAnimCurveChange', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnAnimCurve, 'setOutTangentType', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnAnimCurve, 'setPostInfinityType')
    def setPostInfinityType(self, infinityType):
        # type: (AnimCurve.InfinityType) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsAnimCurveUndo([infinityType], [('infinityType', ('MFnAnimCurve', 'InfinityType'), 'in', None), ('change', 'MAnimCurveChange', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnAnimCurve, 'setPostInfinityType', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnAnimCurve, 'setPreInfinityType')
    def setPreInfinityType(self, infinityType):
        # type: (AnimCurve.InfinityType) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsAnimCurveUndo([infinityType], [('infinityType', ('MFnAnimCurve', 'InfinityType'), 'in', None), ('change', 'MAnimCurveChange', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnAnimCurve, 'setPreInfinityType', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnAnimCurve, 'setQuaternionW')
    def setQuaternionW(self, index, quaternionW):
        # type: (int, float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsAnimCurveUndo([index, quaternionW], [('index', 'uint', 'in', None), ('quaternionW', 'double', 'in', None), ('change', 'MAnimCurveChange', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnAnimCurve, 'setQuaternionW', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnAnimCurve, 'setTangentTypes')
    def setTangentTypes(self, indexArray, inTangentType='global_', outTangentType='global_'):
        # type: (List[int], AnimCurve.TangentType, AnimCurve.TangentType) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsAnimCurveUndo([indexArray, inTangentType, outTangentType], [('indexArray', 'MIntArray', 'in', None), ('inTangentType', ('MFnAnimCurve', 'TangentType'), 'in', None), ('outTangentType', ('MFnAnimCurve', 'TangentType'), 'in', None), ('change', 'MAnimCurveChange', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnAnimCurve, 'setTangentTypes', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnAnimCurve, 'setTangentsLocked')
    def setTangentsLocked(self, index, locked):
        # type: (int, bool) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsAnimCurveUndo([index, locked], [('index', 'uint', 'in', None), ('locked', 'bool', 'in', None), ('change', 'MAnimCurveChange', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnAnimCurve, 'setTangentsLocked', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnAnimCurve, 'setTime')
    def setTime(self, index, time):
        # type: (int, datatypes.Time) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsAnimCurveUndo([index, time], [('index', 'uint', 'in', None), ('time', 'MTime', 'in', None), ('change', 'MAnimCurveChange', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnAnimCurve, 'setTime', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnAnimCurve, 'setUnitlessInput')
    def setUnitlessInput(self, index, unitlessInput):
        # type: (int, float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsAnimCurveUndo([index, unitlessInput], [('index', 'uint', 'in', None), ('unitlessInput', 'double', 'in', None), ('change', 'MAnimCurveChange', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnAnimCurve, 'setUnitlessInput', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnAnimCurve, 'setValue')
    def setValue(self, index, value):
        # type: (int, float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsAnimCurveUndo([index, value], [('index', 'uint', 'in', None), ('value', 'double', 'in', None), ('change', 'MAnimCurveChange', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnAnimCurve, 'setValue', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnAnimCurve, 'setWeight')
    def setWeight(self, index, weight, inTangent):
        # type: (int, float, bool) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsAnimCurveUndo([index, weight, inTangent], [('index', 'uint', 'in', None), ('weight', 'double', 'in', None), ('inTangent', 'bool', 'in', None), ('change', 'MAnimCurveChange', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnAnimCurve, 'setWeight', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnAnimCurve, 'setIsWeighted')
    def setWeighted(self, isWeighted):
        # type: (bool) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsAnimCurveUndo([isWeighted], [('isWeighted', 'bool', 'in', None), ('change', 'MAnimCurveChange', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnAnimCurve, 'setIsWeighted', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnAnimCurve, 'setWeightsLocked')
    def setWeightsLocked(self, index, locked):
        # type: (int, bool) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsAnimCurveUndo([index, locked], [('index', 'uint', 'in', None), ('locked', 'bool', 'in', None), ('change', 'MAnimCurveChange', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnAnimCurve, 'setWeightsLocked', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res
# ------ Do not edit above this line --------


class AnimCurveTA(AnimCurve):
    addKey = AnimCurve.addKeyTU
    evaluate = AnimCurve.evaluateTU
    findClosest = AnimCurve.findClosestT
# ------ Do not edit below this line --------
    __melnode__ = 'animCurveTA'
    __slots__ = ()
# ------ Do not edit above this line --------


class AnimCurveTL(AnimCurve):
    addKey = AnimCurve.addKeyTU
    evaluate = AnimCurve.evaluateTU
    findClosest = AnimCurve.findClosestT
# ------ Do not edit below this line --------
    __melnode__ = 'animCurveTL'
    __slots__ = ()
# ------ Do not edit above this line --------


class AnimCurveTT(AnimCurve):
    addKey = AnimCurve.addKeyTT
    evaluate = AnimCurve.evaluateTT
    findClosest = AnimCurve.findClosestT
# ------ Do not edit below this line --------
    __melnode__ = 'animCurveTT'
    __slots__ = ()
# ------ Do not edit above this line --------


class AnimCurveTU(AnimCurve):
    addKey = AnimCurve.addKeyTU
    evaluate = AnimCurve.evaluateTU
    findClosest = AnimCurve.findClosestT
# ------ Do not edit below this line --------
    __melnode__ = 'animCurveTU'
    __slots__ = ()
# ------ Do not edit above this line --------


class AnimCurveUA(AnimCurve):
    addKey = AnimCurve.addKeyUU
    evaluate = AnimCurve.evaluateUU
    findClosest = AnimCurve.findClosestU
# ------ Do not edit below this line --------
    __melnode__ = 'animCurveUA'
    __slots__ = ()
# ------ Do not edit above this line --------


class AnimCurveUL(AnimCurve):
    addKey = AnimCurve.addKeyUU
    evaluate = AnimCurve.evaluateUU
    findClosest = AnimCurve.findClosestU
# ------ Do not edit below this line --------
    __melnode__ = 'animCurveUL'
    __slots__ = ()
# ------ Do not edit above this line --------


class AnimCurveUT(AnimCurve):
    addKey = AnimCurve.addKeyUT
    evaluate = AnimCurve.evaluateUT
    findClosest = AnimCurve.findClosestU
# ------ Do not edit below this line --------
    __melnode__ = 'animCurveUT'
    __slots__ = ()
# ------ Do not edit above this line --------


class AnimCurveUU(AnimCurve):
    addKey = AnimCurve.addKeyUU
    evaluate = AnimCurve.evaluateUU
    findClosest = AnimCurve.findClosestU
# ------ Do not edit below this line --------
    __melnode__ = 'animCurveUU'
    __slots__ = ()
# ------ Do not edit above this line --------



class ResultCurve(AnimCurve):
    pass
# ------ Do not edit below this line --------
    __melnode__ = 'resultCurve'
    __slots__ = ()
# ------ Do not edit above this line --------


class ResultCurveTimeToAngular(ResultCurve):
    addKey = AnimCurve.addKeyTU
    evaluate = AnimCurve.evaluateTU
    findClosest = AnimCurve.findClosestT
# ------ Do not edit below this line --------
    __melnode__ = 'resultCurveTimeToAngular'
    __slots__ = ()
# ------ Do not edit above this line --------


class ResultCurveTimeToLinear(ResultCurve):
    addKey = AnimCurve.addKeyTU
    evaluate = AnimCurve.evaluateTU
    findClosest = AnimCurve.findClosestT
# ------ Do not edit below this line --------
    __melnode__ = 'resultCurveTimeToLinear'
    __slots__ = ()
# ------ Do not edit above this line --------


class ResultCurveTimeToTime(ResultCurve):
    addKey = AnimCurve.addKeyTT
    evaluate = AnimCurve.evaluateTT
    findClosest = AnimCurve.findClosestT
# ------ Do not edit below this line --------
    __melnode__ = 'resultCurveTimeToTime'
    __slots__ = ()
# ------ Do not edit above this line --------


class ResultCurveTimeToUnitless(ResultCurve):
    addKey = AnimCurve.addKeyTU
    evaluate = AnimCurve.evaluateTU
    findClosest = AnimCurve.findClosestT
# ------ Do not edit below this line --------
    __melnode__ = 'resultCurveTimeToUnitless'
    __slots__ = ()
# ------ Do not edit above this line --------


class GeometryFilter(DependNode):
    pass
# ------ Do not edit below this line --------
    __apicls__ = _api.MFnGeometryFilter
    __melnode__ = 'geometryFilter'
    __slots__ = ()

    @_f.addApiDocs(_api.MFnGeometryFilter, 'deformerSet')
    def deformerSet(self):
        # type: () -> DependNode
        res = _f.getProxyResult(self, _api.MFnGeometryFilter, 'deformerSet')
        return _f.ApiArgUtil._castResult(self, res, 'MObject', None)

    @_f.addApiDocs(_api.MFnGeometryFilter, 'getComponentAtIndex')
    def getComponentAtIndex(self, index):
        # type: (int) -> DependNode
        do, final_do, outTypes = _f.getDoArgs([index], [('index', 'uint', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnGeometryFilter, 'getComponentAtIndex', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MObject', None)
        return res

    @_f.addApiDocs(_api.MFnGeometryFilter, 'envelope')
    def getEnvelope(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnGeometryFilter, 'envelope')
        return _f.ApiArgUtil._castResult(self, res, 'float', None)

    @_f.addApiDocs(_api.MFnGeometryFilter, 'getInputGeometry')
    def getInputGeometry(self):
        # type: () -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([], [('geomList', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnGeometryFilter, 'getInputGeometry', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnGeometryFilter, 'getOutputGeometry')
    def getOutputGeometry(self):
        # type: () -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([], [('geomList', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnGeometryFilter, 'getOutputGeometry', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnGeometryFilter, 'getPathAtIndex')
    def getPathAtIndex(self, index):
        # type: (int) -> DagNode
        do, final_do, outTypes = _f.getDoArgs([index], [('index', 'uint', 'in', None), ('dagPath', 'MDagPath', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnGeometryFilter, 'getPathAtIndex', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnGeometryFilter, 'groupIdAtIndex')
    def groupIdAtIndex(self, index):
        # type: (int) -> int
        do, final_do, outTypes = _f.getDoArgs([index], [('index', 'uint', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnGeometryFilter, 'groupIdAtIndex', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'uint', None)
        return res

    @_f.addApiDocs(_api.MFnGeometryFilter, 'indexForGroupId')
    def indexForGroupId(self, groupId):
        # type: (int) -> int
        do, final_do, outTypes = _f.getDoArgs([groupId], [('groupId', 'uint', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnGeometryFilter, 'indexForGroupId', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'uint', None)
        return res

    @_f.addApiDocs(_api.MFnGeometryFilter, 'indexForOutputConnection')
    def indexForOutputConnection(self, connectionIndex):
        # type: (int) -> int
        do, final_do, outTypes = _f.getDoArgs([connectionIndex], [('connectionIndex', 'uint', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnGeometryFilter, 'indexForOutputConnection', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'uint', None)
        return res

    @_f.addApiDocs(_api.MFnGeometryFilter, 'indexForOutputShape')
    def indexForOutputShape(self, shape):
        # type: (str | DependNode) -> int
        do, final_do, outTypes = _f.getDoArgs([shape], [('shape', 'MObject', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnGeometryFilter, 'indexForOutputShape', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'uint', None)
        return res

    @_f.addApiDocs(_api.MFnGeometryFilter, 'inputShapeAtIndex')
    def inputShapeAtIndex(self, index):
        # type: (int) -> DependNode
        do, final_do, outTypes = _f.getDoArgs([index], [('index', 'uint', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnGeometryFilter, 'inputShapeAtIndex', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MObject', None)
        return res

    @_f.addApiDocs(_api.MFnGeometryFilter, 'numOutputConnections')
    def numOutputConnections(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnGeometryFilter, 'numOutputConnections')
        return _f.ApiArgUtil._castResult(self, res, 'uint', None)

    @_f.addApiDocs(_api.MFnGeometryFilter, 'outputShapeAtIndex')
    def outputShapeAtIndex(self, index):
        # type: (int) -> DependNode
        do, final_do, outTypes = _f.getDoArgs([index], [('index', 'uint', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnGeometryFilter, 'outputShapeAtIndex', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MObject', None)
        return res

    @_f.addApiDocs(_api.MFnGeometryFilter, 'setEnvelope')
    def setEnvelope(self, envelope):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([envelope], [('envelope', 'float', 'in', None)], self.getEnvelope, self.setEnvelope, [])
        res = _f.getProxyResult(self, _api.MFnGeometryFilter, 'setEnvelope', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res
# ------ Do not edit above this line --------


class SkinCluster(GeometryFilter):

    def getWeights(self, geometry, influenceIndex=None):
        if not isinstance(geometry, general.PyNode):
            geometry = general.PyNode(geometry)

        if isinstance(geometry, Transform):
            try:
                geometry = geometry.getShape()
            except:
                raise TypeError("%s is a transform with no shape" % geometry)

        if isinstance(geometry, GeometryShape):
            components = _api.toComponentMObject(geometry.__apimdagpath__())
        elif isinstance(geometry, general.Component):
            components = geometry.__apiobject__()

        else:
            raise TypeError

        if influenceIndex is not None:
            weights = _api.MDoubleArray()
            self.__apimfn__().getWeights(geometry.__apimdagpath__(), components, influenceIndex, weights)
            return iter(weights)
        else:
            weights = _api.MDoubleArray()
            index = _api.SafeApiPtr('uint')
            self.__apimfn__().getWeights(geometry.__apimdagpath__(), components, weights, index())
            index = index.get()
            args = [iter(weights)] * index
            return zip(*args)

    def setWeights(self, geometry, influnces, weights, normalize=True):
        if not isinstance(geometry, general.PyNode):
            geometry = general.PyNode(geometry)

        if isinstance(geometry, Transform):
            try:
                geometry = geometry.getShape()
            except:
                raise TypeError("%s is a transform with no shape" % geometry)

        if isinstance(geometry, GeometryShape):
            components = _api.toComponentMObject(geometry.__apimdagpath__())
        elif isinstance(geometry, general.Component):
            components = geometry.__apiobject__()

        else:
            raise TypeError

        if not isinstance(influnces, _api.MIntArray):
            api_influnces = _api.MIntArray()
            for influnce in influnces:
                api_influnces.append(influnce)
            influnces = api_influnces

        if not isinstance(weights, _api.MDoubleArray):
            api_weights = _api.MDoubleArray()
            for weight in weights:
                api_weights.append(weight)
            weights = api_weights

        old_weights = _api.MDoubleArray()
        su = _api.MScriptUtil()
        su.createFromInt(0)
        index = su.asUintPtr()
        self.__apimfn__().getWeights(geometry.__apimdagpath__(), components, old_weights, index)
        return self.__apimfn__().setWeights(geometry.__apimdagpath__(), components, influnces, weights, normalize, old_weights)

    @_factories.addApiDocs(_api.MFnSkinCluster, 'influenceObjects')
    def influenceObjects(self):
        return self._influenceObjects()[1]

    def numInfluenceObjects(self):
        return self._influenceObjects()[0]
# ------ Do not edit below this line --------
    __apicls__ = _api.MFnSkinCluster
    __melcmd__ = staticmethod(animation.skinCluster)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'skinCluster'
    __melnode__ = 'skinCluster'
    __slots__ = ()

    @_f.addApiDocs(_api.MFnSkinCluster, 'influenceObjects')
    def _influenceObjects(self):
        # type: () -> Tuple[int, List[DagNode]]
        do, final_do, outTypes = _f.getDoArgs([], [('paths', 'MDagPathArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnSkinCluster, 'influenceObjects', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'uint', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.addMelDocs('skinCluster', 'addInfluence')
    def addInfluence(self, val=True, **kwargs):
        # type: (str | List[str], **Any) -> None
        return _f.asEdit(self, animation.skinCluster, kwargs, 'addInfluence', val)

    @_f.addMelDocs('skinCluster', 'addToSelection')
    def addToSelection(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.skinCluster, kwargs, 'addToSelection', val)

    @_f.addMelDocs('skinCluster', 'after')
    def after(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.skinCluster, kwargs, 'after', val)

    @_f.addMelDocs('skinCluster', 'afterReference')
    def afterReference(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.skinCluster, kwargs, 'afterReference', val)

    @_f.addMelDocs('skinCluster', 'baseShape')
    def baseShape(self, val=True, **kwargs):
        # type: (str | List[str], **Any) -> None
        return _f.asEdit(self, animation.skinCluster, kwargs, 'baseShape', val)

    @_f.addMelDocs('skinCluster', 'before')
    def before(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.skinCluster, kwargs, 'before', val)

    @_f.addMelDocs('skinCluster', 'forceNormalizeWeights')
    def forceNormalizeWeights(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.skinCluster, kwargs, 'forceNormalizeWeights', val)

    @_f.addMelDocs('skinCluster', 'frontOfChain')
    def frontOfChain(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.skinCluster, kwargs, 'frontOfChain', val)

    @_f.addMelDocs('skinCluster', 'bindMethod')
    def getBindMethod(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.skinCluster, kwargs, 'bindMethod')
        return res

    @_f.addApiDocs(_api.MFnSkinCluster, 'getBlendWeights')
    def getBlendWeights(self, path, components):
        # type: (DagNode, str | DependNode) -> List[float]
        do, final_do, outTypes = _f.getDoArgs([path, components], [('path', 'MDagPath', 'in', None), ('components', 'MObject', 'in', None), ('weights', 'MDoubleArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnSkinCluster, 'getBlendWeights', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addMelDocs('skinCluster', 'deformerTools')
    def getDeformerTools(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.skinCluster, kwargs, 'deformerTools')
        return res

    @_f.addMelDocs('skinCluster', 'dropoffRate')
    def getDropoffRate(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, animation.skinCluster, kwargs, 'dropoffRate')
        return res

    @_f.addMelDocs('skinCluster', 'exclusive')
    def getExclusive(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.skinCluster, kwargs, 'exclusive')
        return res

    @_f.addMelDocs('skinCluster', 'geometry')
    def getGeometry(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.skinCluster, kwargs, 'geometry')
        return res

    @_f.addMelDocs('skinCluster', 'geometryIndices')
    def getGeometryIndices(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.skinCluster, kwargs, 'geometryIndices')
        return res

    @_f.addMelDocs('skinCluster', 'ignoreHierarchy')
    def getIgnoreHierarchy(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.skinCluster, kwargs, 'ignoreHierarchy')
        return res

    @_f.addMelDocs('skinCluster', 'influence')
    def getInfluence(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.skinCluster, kwargs, 'influence')
        return res

    @_f.addMelDocs('skinCluster', 'lockWeights')
    def getLockWeights(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.skinCluster, kwargs, 'lockWeights')
        return res

    @_f.addMelDocs('skinCluster', 'maximumInfluences')
    def getMaximumInfluences(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, animation.skinCluster, kwargs, 'maximumInfluences')
        return res

    @_f.addMelDocs('skinCluster', 'normalizeWeights')
    def getNormalizeWeights(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, animation.skinCluster, kwargs, 'normalizeWeights')
        return res

    @_f.addMelDocs('skinCluster', 'obeyMaxInfluences')
    def getObeyMaxInfluences(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, animation.skinCluster, kwargs, 'obeyMaxInfluences')
        return res

    @_f.addApiDocs(_api.MFnSkinCluster, 'getPointsAffectedByInfluence')
    def getPointsAffectedByInfluence(self, path):
        # type: (DagNode) -> Tuple[SelectionSet, List[float]]
        do, final_do, outTypes = _f.getDoArgs([path], [('path', 'MDagPath', 'in', None), ('result', 'MSelectionList', 'out', None), ('weights', 'MDoubleArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnSkinCluster, 'getPointsAffectedByInfluence', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addMelDocs('skinCluster', 'skinMethod')
    def getSkinMethod(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, animation.skinCluster, kwargs, 'skinMethod')
        return res

    @_f.addMelDocs('skinCluster', 'useComponentTags')
    def getUseComponentTags(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.skinCluster, kwargs, 'useComponentTags')
        return res

    @_f.addMelDocs('skinCluster', 'weightDistribution')
    def getWeightDistribution(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, animation.skinCluster, kwargs, 'weightDistribution')
        return res

    @_f.addMelDocs('skinCluster', 'weightedInfluence')
    def getWeightedInfluence(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.skinCluster, kwargs, 'weightedInfluence')
        return res

    @_f.addMelDocs('skinCluster', 'ignoreBindPose')
    def ignoreBindPose(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.skinCluster, kwargs, 'ignoreBindPose', val)

    @_f.addApiDocs(_api.MFnSkinCluster, 'indexForInfluenceObject')
    def indexForInfluenceObject(self, mpath):
        # type: (DagNode) -> int
        do, final_do, outTypes = _f.getDoArgs([mpath], [('mpath', 'MDagPath', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnSkinCluster, 'indexForInfluenceObject', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'uint', None)
        return res

    @_f.addMelDocs('skinCluster', 'moveJointsMode')
    def moveJointsMode(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.skinCluster, kwargs, 'moveJointsMode', val)

    @_f.addMelDocs('skinCluster', 'nurbsSamples')
    def nurbsSamples(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, animation.skinCluster, kwargs, 'nurbsSamples', val)

    @_f.addMelDocs('skinCluster', 'parallel')
    def parallel(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.skinCluster, kwargs, 'parallel', val)

    @_f.addMelDocs('skinCluster', 'polySmoothness')
    def polySmoothness(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, animation.skinCluster, kwargs, 'polySmoothness', val)

    @_f.addMelDocs('skinCluster', 'prune')
    def prune(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.skinCluster, kwargs, 'prune', val)

    @_f.addMelDocs('skinCluster', 'recacheBindMatrices')
    def recacheBindMatrices(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.skinCluster, kwargs, 'recacheBindMatrices', val)

    @_f.addMelDocs('skinCluster', 'remove')
    def remove(self, val=True, **kwargs):
        # type: (bool | int | List[bool | int], **Any) -> None
        return _f.asEdit(self, animation.skinCluster, kwargs, 'remove', val)

    @_f.addMelDocs('skinCluster', 'removeFromSelection')
    def removeFromSelection(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.skinCluster, kwargs, 'removeFromSelection', val)

    @_f.addMelDocs('skinCluster', 'removeInfluence')
    def removeInfluence(self, val=True, **kwargs):
        # type: (str | List[str], **Any) -> None
        return _f.asEdit(self, animation.skinCluster, kwargs, 'removeInfluence', val)

    @_f.addMelDocs('skinCluster', 'selectInfluenceVerts')
    def selectInfluenceVerts(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.skinCluster, kwargs, 'selectInfluenceVerts', val)

    @_f.addApiDocs(_api.MFnSkinCluster, 'setBlendWeights')
    def setBlendWeights(self, path, components, weights):
        # type: (DagNode, str | DependNode, List[float]) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([path, components, weights], [('path', 'MDagPath', 'in', None), ('components', 'MObject', 'in', None), ('weights', 'MDoubleArray', 'in', None)], self.getBlendWeights, self.setBlendWeights, ['path', 'components'])
        res = _f.getProxyResult(self, _api.MFnSkinCluster, 'setBlendWeights', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addMelDocs('skinCluster', 'dropoffRate')
    def setDropoffRate(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, animation.skinCluster, kwargs, 'dropoffRate', val)

    @_f.addMelDocs('skinCluster', 'geometry')
    def setGeometry(self, val=True, **kwargs):
        # type: (str | List[str], **Any) -> None
        return _f.asEdit(self, animation.skinCluster, kwargs, 'geometry', val)

    @_f.addMelDocs('skinCluster', 'influence')
    def setInfluence(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.skinCluster, kwargs, 'influence', val)

    @_f.addMelDocs('skinCluster', 'lockWeights')
    def setLockWeights(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.skinCluster, kwargs, 'lockWeights', val)

    @_f.addMelDocs('skinCluster', 'maximumInfluences')
    def setMaximumInfluences(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, animation.skinCluster, kwargs, 'maximumInfluences', val)

    @_f.addMelDocs('skinCluster', 'normalizeWeights')
    def setNormalizeWeights(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, animation.skinCluster, kwargs, 'normalizeWeights', val)

    @_f.addMelDocs('skinCluster', 'obeyMaxInfluences')
    def setObeyMaxInfluences(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.skinCluster, kwargs, 'obeyMaxInfluences', val)

    @_f.addMelDocs('skinCluster', 'skinMethod')
    def setSkinMethod(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, animation.skinCluster, kwargs, 'skinMethod', val)

    @_f.addMelDocs('skinCluster', 'weightDistribution')
    def setWeightDistribution(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, animation.skinCluster, kwargs, 'weightDistribution', val)

    @_f.addMelDocs('skinCluster', 'smoothWeights')
    def smoothWeights(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, animation.skinCluster, kwargs, 'smoothWeights', val)

    @_f.addMelDocs('skinCluster', 'smoothWeightsMaxIterations')
    def smoothWeightsMaxIterations(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, animation.skinCluster, kwargs, 'smoothWeightsMaxIterations', val)

    @_f.addMelDocs('skinCluster', 'unbind')
    def unbind(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.skinCluster, kwargs, 'unbind', val)

    @_f.addMelDocs('skinCluster', 'unbindKeepHistory')
    def unbindKeepHistory(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.skinCluster, kwargs, 'unbindKeepHistory', val)

    @_f.addMelDocs('skinCluster', 'useGeometry')
    def useGeometry(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.skinCluster, kwargs, 'useGeometry', val)

    @_f.addMelDocs('skinCluster', 'weight')
    def weight(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, animation.skinCluster, kwargs, 'weight', val)
# ------ Do not edit above this line --------


# TODO: if hikHandle bug ever fixed:
#   - remove entry in apiCache.ApiCache.API_TO_MFN_OVERRIDES
#   - remove hard-code setting of HikHandle's parent to Transform
class HikHandle(Transform):
    pass
# ------ Do not edit below this line --------
    __melnode__ = 'hikHandle'
    __slots__ = ()

    @_f.deprecated('Use getRotation(quaternion=True) instead.')
    def getRotationQuaternion(self, space='transform'):
        # type: (datatypes.Space.Space) -> Tuple[float, float, float, float]
        do, final_do, outTypes = _f.getDoArgs([space], [('x', 'double', 'out', None), ('y', 'double', 'out', None), ('z', 'double', 'out', None), ('w', 'double', 'out', None), ('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnTransform, 'getRotationQuaternion', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use setRotation, and pass in a Quaternion object, instead.')
    def setRotationQuaternion(self, x, y, z, w, space='transform'):
        # type: (float, float, float, float, datatypes.Space.Space) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([x, y, z, w, space], [('x', 'double', 'in', None), ('y', 'double', 'in', None), ('z', 'double', 'in', None), ('w', 'double', 'in', None), ('space', ('MSpace', 'Space'), 'in', None)], self.getRotationQuaternion, self.setRotationQuaternion, ['space'])
        res = _f.getProxyResult(self, _api.MFnTransform, 'setRotationQuaternion', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res
# ------ Do not edit above this line --------

# TODO: if jointFfd bug ever fixed:
#   - remove entry in apiCache.ApiCache.API_TO_MFN_OVERRIDES
#   - remove hard-code setting of JointFfd's parent to DependNode
class JointFfd(DependNode):
    pass
# ------ Do not edit below this line --------
    __melnode__ = 'jointFfd'
    __slots__ = ()

    @_f.deprecated
    def findAlias(self, alias):
        # type: (str) -> Tuple[bool, DependNode]
        do, final_do, outTypes = _f.getDoArgs([alias], [('alias', 'MString', 'in', None), ('attrObj', 'MObject', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'findAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affected instead.')
    def getAffectedAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affects instead.')
    def getAffectedByAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedByAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedByAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use listAliases instead.')
    def getAliasList(self):
        # type: () -> Tuple[bool, List[str]]
        do, final_do, outTypes = _f.getDoArgs([], [('strArray', 'MStringArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAliasList', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use inputs/outputs, or general.listConnections instead.')
    def getConnections(self):
        # type: () -> List[general.Attribute]
        do, final_do, outTypes = _f.getDoArgs([], [('array', 'MPlugArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getConnections', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.getAlias instead.')
    def plugsAlias(self, plug):
        # type: (str | general.Attribute) -> str
        do, final_do, outTypes = _f.getDoArgs([plug], [('plug', 'MPlug', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'plugsAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MString', None)
        return res

    @_f.deprecated('Use Attribute.setAlias instead.')
    def setAlias(self, alias, name, plug, add=True):
        # type: (str, str, str | general.Attribute, bool) -> bool
        do, final_do, outTypes = _f.getDoArgs([alias, name, plug, add], [('alias', 'MString', 'in', None), ('name', 'MString', 'in', None), ('plug', 'MPlug', 'in', None), ('add', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'setAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res
# ------ Do not edit above this line --------

class AngleBetween(DependNode):
    @staticmethod
    def __melcmd__(*args, **kwargs):
        # if you don't turn on construction history, it won't make a node
        kwargs['ch'] = 1
        return cmds.angleBetween(*args, **kwargs)
# ------ Do not edit below this line --------
    __melcmd_isinfo__ = False
    __melcmdname__ = 'angleBetween'
    __melnode__ = 'angleBetween'
    __slots__ = ()

    @_f.deprecated
    def findAlias(self, alias):
        # type: (str) -> Tuple[bool, DependNode]
        do, final_do, outTypes = _f.getDoArgs([alias], [('alias', 'MString', 'in', None), ('attrObj', 'MObject', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'findAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affected instead.')
    def getAffectedAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affects instead.')
    def getAffectedByAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedByAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedByAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use listAliases instead.')
    def getAliasList(self):
        # type: () -> Tuple[bool, List[str]]
        do, final_do, outTypes = _f.getDoArgs([], [('strArray', 'MStringArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAliasList', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use inputs/outputs, or general.listConnections instead.')
    def getConnections(self):
        # type: () -> List[general.Attribute]
        do, final_do, outTypes = _f.getDoArgs([], [('array', 'MPlugArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getConnections', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.getAlias instead.')
    def plugsAlias(self, plug):
        # type: (str | general.Attribute) -> str
        do, final_do, outTypes = _f.getDoArgs([plug], [('plug', 'MPlug', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'plugsAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MString', None)
        return res

    @_f.deprecated('Use Attribute.setAlias instead.')
    def setAlias(self, alias, name, plug, add=True):
        # type: (str, str, str | general.Attribute, bool) -> bool
        do, final_do, outTypes = _f.getDoArgs([alias, name, plug, add], [('alias', 'MString', 'in', None), ('name', 'MString', 'in', None), ('plug', 'MPlug', 'in', None), ('add', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'setAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res
# ------ Do not edit above this line --------

class UfeProxyCameraShape(Camera):
    # hardcoding to MFnCamera for now, due to a bug of the mobj reporting it's
    # apiType as kBase
    __apicls__ = _api.MFnCamera
    __melnode__ = 'ufeProxyCameraShape'
# ------ Do not edit below this line --------
    __slots__ = ()
# ------ Do not edit above this line --------

_factories.ApiTypeRegister.register('MSelectionList', SelectionSet)

# def listToMSelection( objs ):
#    sel = _api.MSelectionList()
#    for obj in objs:
#        if isinstance(obj, DependNode):
#            sel.add( obj.__apiobject__() )
#        elif isinstance(obj, Attribute):
#            sel.add( obj.__apiobject__(), True )
#        elif isinstance(obj, Component):
#            pass
#            #sel.add( obj.__apiobject__(), True )
#        else:
#            raise TypeError

# ------ Do not edit below this line --------
class THdependNode(DependNode):
    __melnode__ = 'THdependNode'
    __slots__ = ()

    @_f.deprecated
    def findAlias(self, alias):
        # type: (str) -> Tuple[bool, DependNode]
        do, final_do, outTypes = _f.getDoArgs([alias], [('alias', 'MString', 'in', None), ('attrObj', 'MObject', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'findAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affected instead.')
    def getAffectedAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affects instead.')
    def getAffectedByAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedByAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedByAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use listAliases instead.')
    def getAliasList(self):
        # type: () -> Tuple[bool, List[str]]
        do, final_do, outTypes = _f.getDoArgs([], [('strArray', 'MStringArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAliasList', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use inputs/outputs, or general.listConnections instead.')
    def getConnections(self):
        # type: () -> List[general.Attribute]
        do, final_do, outTypes = _f.getDoArgs([], [('array', 'MPlugArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getConnections', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.getAlias instead.')
    def plugsAlias(self, plug):
        # type: (str | general.Attribute) -> str
        do, final_do, outTypes = _f.getDoArgs([plug], [('plug', 'MPlug', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'plugsAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MString', None)
        return res

    @_f.deprecated('Use Attribute.setAlias instead.')
    def setAlias(self, alias, name, plug, add=True):
        # type: (str, str, str | general.Attribute, bool) -> bool
        do, final_do, outTypes = _f.getDoArgs([alias, name, plug, add], [('alias', 'MString', 'in', None), ('name', 'MString', 'in', None), ('plug', 'MPlug', 'in', None), ('add', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'setAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res


class TadskAssetInstanceNode_TdependNode(DependNode):
    __melnode__ = 'TadskAssetInstanceNode_TdependNode'
    __slots__ = ()


class AdskMaterial(TadskAssetInstanceNode_TdependNode):
    __melnode__ = 'adskMaterial'
    __slots__ = ()


class AboutToSetValueTestNode(DependNode):
    __melnode__ = 'aboutToSetValueTestNode'
    __slots__ = ()


class AbstractBaseCreate(DependNode):
    __melnode__ = 'abstractBaseCreate'
    __slots__ = ()

    @_f.deprecated
    def findAlias(self, alias):
        # type: (str) -> Tuple[bool, DependNode]
        do, final_do, outTypes = _f.getDoArgs([alias], [('alias', 'MString', 'in', None), ('attrObj', 'MObject', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'findAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affected instead.')
    def getAffectedAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affects instead.')
    def getAffectedByAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedByAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedByAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use listAliases instead.')
    def getAliasList(self):
        # type: () -> Tuple[bool, List[str]]
        do, final_do, outTypes = _f.getDoArgs([], [('strArray', 'MStringArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAliasList', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use inputs/outputs, or general.listConnections instead.')
    def getConnections(self):
        # type: () -> List[general.Attribute]
        do, final_do, outTypes = _f.getDoArgs([], [('array', 'MPlugArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getConnections', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.getAlias instead.')
    def plugsAlias(self, plug):
        # type: (str | general.Attribute) -> str
        do, final_do, outTypes = _f.getDoArgs([plug], [('plug', 'MPlug', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'plugsAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MString', None)
        return res

    @_f.deprecated('Use Attribute.setAlias instead.')
    def setAlias(self, alias, name, plug, add=True):
        # type: (str, str, str | general.Attribute, bool) -> bool
        do, final_do, outTypes = _f.getDoArgs([alias, name, plug, add], [('alias', 'MString', 'in', None), ('name', 'MString', 'in', None), ('plug', 'MPlug', 'in', None), ('add', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'setAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res


class AbstractBaseNurbsConversion(AbstractBaseCreate):
    __melnode__ = 'abstractBaseNurbsConversion'
    __slots__ = ()


class NurbsToSubdiv(AbstractBaseNurbsConversion):
    __melcmd__ = staticmethod(modeling.nurbsToSubdiv)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'nurbsToSubdiv'
    __melnode__ = 'nurbsToSubdiv'
    __slots__ = ()

    @_f.addMelDocs('nurbsToSubdiv', 'addUnderTransform')
    def getAddUnderTransform(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.nurbsToSubdiv, kwargs, 'addUnderTransform')
        return res

    @_f.addMelDocs('nurbsToSubdiv', 'caching')
    def getCaching(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.nurbsToSubdiv, kwargs, 'caching')
        return res

    @_f.addMelDocs('nurbsToSubdiv', 'collapsePoles')
    def getCollapsePoles(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.nurbsToSubdiv, kwargs, 'collapsePoles')
        return res

    @_f.addMelDocs('nurbsToSubdiv', 'matchPeriodic')
    def getMatchPeriodic(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.nurbsToSubdiv, kwargs, 'matchPeriodic')
        return res

    @_f.addMelDocs('nurbsToSubdiv', 'maxPolyCount')
    def getMaxPolyCount(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.nurbsToSubdiv, kwargs, 'maxPolyCount')
        return res

    @_f.addMelDocs('nurbsToSubdiv', 'nodeState')
    def getNodeState(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.nurbsToSubdiv, kwargs, 'nodeState')
        return res

    @_f.addMelDocs('nurbsToSubdiv', 'reverseNormal')
    def getReverseNormal(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.nurbsToSubdiv, kwargs, 'reverseNormal')
        return res

    @_f.addMelDocs('nurbsToSubdiv', 'addUnderTransform')
    def setAddUnderTransform(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.nurbsToSubdiv, kwargs, 'addUnderTransform', val)

    @_f.addMelDocs('nurbsToSubdiv', 'caching')
    def setCaching(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.nurbsToSubdiv, kwargs, 'caching', val)

    @_f.addMelDocs('nurbsToSubdiv', 'collapsePoles')
    def setCollapsePoles(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.nurbsToSubdiv, kwargs, 'collapsePoles', val)

    @_f.addMelDocs('nurbsToSubdiv', 'matchPeriodic')
    def setMatchPeriodic(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.nurbsToSubdiv, kwargs, 'matchPeriodic', val)

    @_f.addMelDocs('nurbsToSubdiv', 'maxPolyCount')
    def setMaxPolyCount(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.nurbsToSubdiv, kwargs, 'maxPolyCount', val)

    @_f.addMelDocs('nurbsToSubdiv', 'nodeState')
    def setNodeState(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.nurbsToSubdiv, kwargs, 'nodeState', val)

    @_f.addMelDocs('nurbsToSubdiv', 'reverseNormal')
    def setReverseNormal(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.nurbsToSubdiv, kwargs, 'reverseNormal', val)


class NurbsToSubdivProc(NurbsToSubdiv):
    __melnode__ = 'nurbsToSubdivProc'
    __slots__ = ()


class ParentTessellate(AbstractBaseNurbsConversion):
    __melnode__ = 'parentTessellate'
    __slots__ = ()


class NurbsTessellate(ParentTessellate):
    __melnode__ = 'nurbsTessellate'
    __slots__ = ()


class ShellTessellate(ParentTessellate):
    __melnode__ = 'shellTessellate'
    __slots__ = ()


class RebuildSurface(AbstractBaseNurbsConversion):
    __melcmd__ = staticmethod(modeling.rebuildSurface)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'rebuildSurface'
    __melnode__ = 'rebuildSurface'
    __slots__ = ()

    @_f.addMelDocs('rebuildSurface', 'caching')
    def getCaching(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.rebuildSurface, kwargs, 'caching')
        return res

    @_f.addMelDocs('rebuildSurface', 'degreeU')
    def getDegreeU(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.rebuildSurface, kwargs, 'degreeU')
        return res

    @_f.addMelDocs('rebuildSurface', 'degreeV')
    def getDegreeV(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.rebuildSurface, kwargs, 'degreeV')
        return res

    @_f.addMelDocs('rebuildSurface', 'direction')
    def getDirection(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.rebuildSurface, kwargs, 'direction')
        return res

    @_f.addMelDocs('rebuildSurface', 'endKnots')
    def getEndKnots(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.rebuildSurface, kwargs, 'endKnots')
        return res

    @_f.addMelDocs('rebuildSurface', 'fitRebuild')
    def getFitRebuild(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.rebuildSurface, kwargs, 'fitRebuild')
        return res

    @_f.addMelDocs('rebuildSurface', 'keepControlPoints')
    def getKeepControlPoints(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.rebuildSurface, kwargs, 'keepControlPoints')
        return res

    @_f.addMelDocs('rebuildSurface', 'keepCorners')
    def getKeepCorners(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.rebuildSurface, kwargs, 'keepCorners')
        return res

    @_f.addMelDocs('rebuildSurface', 'keepRange')
    def getKeepRange(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.rebuildSurface, kwargs, 'keepRange')
        return res

    @_f.addMelDocs('rebuildSurface', 'nodeState')
    def getNodeState(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.rebuildSurface, kwargs, 'nodeState')
        return res

    @_f.addMelDocs('rebuildSurface', 'rebuildType')
    def getRebuildType(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.rebuildSurface, kwargs, 'rebuildType')
        return res

    @_f.addMelDocs('rebuildSurface', 'spansU')
    def getSpansU(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.rebuildSurface, kwargs, 'spansU')
        return res

    @_f.addMelDocs('rebuildSurface', 'spansV')
    def getSpansV(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.rebuildSurface, kwargs, 'spansV')
        return res

    @_f.addMelDocs('rebuildSurface', 'tolerance')
    def getTolerance(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.rebuildSurface, kwargs, 'tolerance')
        return res

    @_f.addMelDocs('rebuildSurface', 'caching')
    def setCaching(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.rebuildSurface, kwargs, 'caching', val)

    @_f.addMelDocs('rebuildSurface', 'degreeU')
    def setDegreeU(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.rebuildSurface, kwargs, 'degreeU', val)

    @_f.addMelDocs('rebuildSurface', 'degreeV')
    def setDegreeV(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.rebuildSurface, kwargs, 'degreeV', val)

    @_f.addMelDocs('rebuildSurface', 'direction')
    def setDirection(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.rebuildSurface, kwargs, 'direction', val)

    @_f.addMelDocs('rebuildSurface', 'endKnots')
    def setEndKnots(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.rebuildSurface, kwargs, 'endKnots', val)

    @_f.addMelDocs('rebuildSurface', 'fitRebuild')
    def setFitRebuild(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.rebuildSurface, kwargs, 'fitRebuild', val)

    @_f.addMelDocs('rebuildSurface', 'keepControlPoints')
    def setKeepControlPoints(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.rebuildSurface, kwargs, 'keepControlPoints', val)

    @_f.addMelDocs('rebuildSurface', 'keepCorners')
    def setKeepCorners(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.rebuildSurface, kwargs, 'keepCorners', val)

    @_f.addMelDocs('rebuildSurface', 'keepRange')
    def setKeepRange(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.rebuildSurface, kwargs, 'keepRange', val)

    @_f.addMelDocs('rebuildSurface', 'nodeState')
    def setNodeState(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.rebuildSurface, kwargs, 'nodeState', val)

    @_f.addMelDocs('rebuildSurface', 'rebuildType')
    def setRebuildType(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.rebuildSurface, kwargs, 'rebuildType', val)

    @_f.addMelDocs('rebuildSurface', 'spansU')
    def setSpansU(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.rebuildSurface, kwargs, 'spansU', val)

    @_f.addMelDocs('rebuildSurface', 'spansV')
    def setSpansV(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.rebuildSurface, kwargs, 'spansV', val)

    @_f.addMelDocs('rebuildSurface', 'tolerance')
    def setTolerance(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.rebuildSurface, kwargs, 'tolerance', val)


class AlignCurve(AbstractBaseCreate):
    __melcmd__ = staticmethod(modeling.alignCurve)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'alignCurve'
    __melnode__ = 'alignCurve'
    __slots__ = ()

    @_f.addMelDocs('alignCurve', 'caching')
    def getCaching(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.alignCurve, kwargs, 'caching')
        return res

    @_f.addMelDocs('alignCurve', 'curvatureContinuity')
    def getCurvatureContinuity(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.alignCurve, kwargs, 'curvatureContinuity')
        return res

    @_f.addMelDocs('alignCurve', 'curvatureScale1')
    def getCurvatureScale1(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.alignCurve, kwargs, 'curvatureScale1')
        return res

    @_f.addMelDocs('alignCurve', 'curvatureScale2')
    def getCurvatureScale2(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.alignCurve, kwargs, 'curvatureScale2')
        return res

    @_f.addMelDocs('alignCurve', 'joinParameter')
    def getJoinParameter(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.alignCurve, kwargs, 'joinParameter')
        return res

    @_f.addMelDocs('alignCurve', 'nodeState')
    def getNodeState(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.alignCurve, kwargs, 'nodeState')
        return res

    @_f.addMelDocs('alignCurve', 'positionalContinuity')
    def getPositionalContinuity(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.alignCurve, kwargs, 'positionalContinuity')
        return res

    @_f.addMelDocs('alignCurve', 'positionalContinuityType')
    def getPositionalContinuityType(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.alignCurve, kwargs, 'positionalContinuityType')
        return res

    @_f.addMelDocs('alignCurve', 'reverse1')
    def getReverse1(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.alignCurve, kwargs, 'reverse1')
        return res

    @_f.addMelDocs('alignCurve', 'reverse2')
    def getReverse2(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.alignCurve, kwargs, 'reverse2')
        return res

    @_f.addMelDocs('alignCurve', 'tangentContinuity')
    def getTangentContinuity(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.alignCurve, kwargs, 'tangentContinuity')
        return res

    @_f.addMelDocs('alignCurve', 'tangentContinuityType')
    def getTangentContinuityType(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.alignCurve, kwargs, 'tangentContinuityType')
        return res

    @_f.addMelDocs('alignCurve', 'tangentScale1')
    def getTangentScale1(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.alignCurve, kwargs, 'tangentScale1')
        return res

    @_f.addMelDocs('alignCurve', 'tangentScale2')
    def getTangentScale2(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.alignCurve, kwargs, 'tangentScale2')
        return res

    @_f.addMelDocs('alignCurve', 'caching')
    def setCaching(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.alignCurve, kwargs, 'caching', val)

    @_f.addMelDocs('alignCurve', 'curvatureContinuity')
    def setCurvatureContinuity(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.alignCurve, kwargs, 'curvatureContinuity', val)

    @_f.addMelDocs('alignCurve', 'curvatureScale1')
    def setCurvatureScale1(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.alignCurve, kwargs, 'curvatureScale1', val)

    @_f.addMelDocs('alignCurve', 'curvatureScale2')
    def setCurvatureScale2(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.alignCurve, kwargs, 'curvatureScale2', val)

    @_f.addMelDocs('alignCurve', 'joinParameter')
    def setJoinParameter(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.alignCurve, kwargs, 'joinParameter', val)

    @_f.addMelDocs('alignCurve', 'nodeState')
    def setNodeState(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.alignCurve, kwargs, 'nodeState', val)

    @_f.addMelDocs('alignCurve', 'positionalContinuity')
    def setPositionalContinuity(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.alignCurve, kwargs, 'positionalContinuity', val)

    @_f.addMelDocs('alignCurve', 'positionalContinuityType')
    def setPositionalContinuityType(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.alignCurve, kwargs, 'positionalContinuityType', val)

    @_f.addMelDocs('alignCurve', 'reverse1')
    def setReverse1(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.alignCurve, kwargs, 'reverse1', val)

    @_f.addMelDocs('alignCurve', 'reverse2')
    def setReverse2(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.alignCurve, kwargs, 'reverse2', val)

    @_f.addMelDocs('alignCurve', 'tangentContinuity')
    def setTangentContinuity(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.alignCurve, kwargs, 'tangentContinuity', val)

    @_f.addMelDocs('alignCurve', 'tangentContinuityType')
    def setTangentContinuityType(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.alignCurve, kwargs, 'tangentContinuityType', val)

    @_f.addMelDocs('alignCurve', 'tangentScale1')
    def setTangentScale1(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.alignCurve, kwargs, 'tangentScale1', val)

    @_f.addMelDocs('alignCurve', 'tangentScale2')
    def setTangentScale2(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.alignCurve, kwargs, 'tangentScale2', val)


class AlignSurface(AbstractBaseCreate):
    __melcmd__ = staticmethod(modeling.alignSurface)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'alignSurface'
    __melnode__ = 'alignSurface'
    __slots__ = ()

    @_f.addMelDocs('alignSurface', 'caching')
    def getCaching(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.alignSurface, kwargs, 'caching')
        return res

    @_f.addMelDocs('alignSurface', 'curvatureContinuity')
    def getCurvatureContinuity(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.alignSurface, kwargs, 'curvatureContinuity')
        return res

    @_f.addMelDocs('alignSurface', 'curvatureScale1')
    def getCurvatureScale1(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.alignSurface, kwargs, 'curvatureScale1')
        return res

    @_f.addMelDocs('alignSurface', 'curvatureScale2')
    def getCurvatureScale2(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.alignSurface, kwargs, 'curvatureScale2')
        return res

    @_f.addMelDocs('alignSurface', 'directionU')
    def getDirectionU(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.alignSurface, kwargs, 'directionU')
        return res

    @_f.addMelDocs('alignSurface', 'joinParameter')
    def getJoinParameter(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.alignSurface, kwargs, 'joinParameter')
        return res

    @_f.addMelDocs('alignSurface', 'nodeState')
    def getNodeState(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.alignSurface, kwargs, 'nodeState')
        return res

    @_f.addMelDocs('alignSurface', 'positionalContinuity')
    def getPositionalContinuity(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.alignSurface, kwargs, 'positionalContinuity')
        return res

    @_f.addMelDocs('alignSurface', 'positionalContinuityType')
    def getPositionalContinuityType(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.alignSurface, kwargs, 'positionalContinuityType')
        return res

    @_f.addMelDocs('alignSurface', 'reverse1')
    def getReverse1(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.alignSurface, kwargs, 'reverse1')
        return res

    @_f.addMelDocs('alignSurface', 'reverse2')
    def getReverse2(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.alignSurface, kwargs, 'reverse2')
        return res

    @_f.addMelDocs('alignSurface', 'swap1')
    def getSwap1(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.alignSurface, kwargs, 'swap1')
        return res

    @_f.addMelDocs('alignSurface', 'swap2')
    def getSwap2(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.alignSurface, kwargs, 'swap2')
        return res

    @_f.addMelDocs('alignSurface', 'tangentContinuity')
    def getTangentContinuity(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.alignSurface, kwargs, 'tangentContinuity')
        return res

    @_f.addMelDocs('alignSurface', 'tangentContinuityType')
    def getTangentContinuityType(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.alignSurface, kwargs, 'tangentContinuityType')
        return res

    @_f.addMelDocs('alignSurface', 'tangentScale1')
    def getTangentScale1(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.alignSurface, kwargs, 'tangentScale1')
        return res

    @_f.addMelDocs('alignSurface', 'tangentScale2')
    def getTangentScale2(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.alignSurface, kwargs, 'tangentScale2')
        return res

    @_f.addMelDocs('alignSurface', 'twist')
    def getTwist(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.alignSurface, kwargs, 'twist')
        return res

    @_f.addMelDocs('alignSurface', 'caching')
    def setCaching(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.alignSurface, kwargs, 'caching', val)

    @_f.addMelDocs('alignSurface', 'curvatureContinuity')
    def setCurvatureContinuity(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.alignSurface, kwargs, 'curvatureContinuity', val)

    @_f.addMelDocs('alignSurface', 'curvatureScale1')
    def setCurvatureScale1(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.alignSurface, kwargs, 'curvatureScale1', val)

    @_f.addMelDocs('alignSurface', 'curvatureScale2')
    def setCurvatureScale2(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.alignSurface, kwargs, 'curvatureScale2', val)

    @_f.addMelDocs('alignSurface', 'directionU')
    def setDirectionU(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.alignSurface, kwargs, 'directionU', val)

    @_f.addMelDocs('alignSurface', 'joinParameter')
    def setJoinParameter(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.alignSurface, kwargs, 'joinParameter', val)

    @_f.addMelDocs('alignSurface', 'nodeState')
    def setNodeState(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.alignSurface, kwargs, 'nodeState', val)

    @_f.addMelDocs('alignSurface', 'positionalContinuity')
    def setPositionalContinuity(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.alignSurface, kwargs, 'positionalContinuity', val)

    @_f.addMelDocs('alignSurface', 'positionalContinuityType')
    def setPositionalContinuityType(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.alignSurface, kwargs, 'positionalContinuityType', val)

    @_f.addMelDocs('alignSurface', 'reverse1')
    def setReverse1(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.alignSurface, kwargs, 'reverse1', val)

    @_f.addMelDocs('alignSurface', 'reverse2')
    def setReverse2(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.alignSurface, kwargs, 'reverse2', val)

    @_f.addMelDocs('alignSurface', 'swap1')
    def setSwap1(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.alignSurface, kwargs, 'swap1', val)

    @_f.addMelDocs('alignSurface', 'swap2')
    def setSwap2(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.alignSurface, kwargs, 'swap2', val)

    @_f.addMelDocs('alignSurface', 'tangentContinuity')
    def setTangentContinuity(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.alignSurface, kwargs, 'tangentContinuity', val)

    @_f.addMelDocs('alignSurface', 'tangentContinuityType')
    def setTangentContinuityType(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.alignSurface, kwargs, 'tangentContinuityType', val)

    @_f.addMelDocs('alignSurface', 'tangentScale1')
    def setTangentScale1(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.alignSurface, kwargs, 'tangentScale1', val)

    @_f.addMelDocs('alignSurface', 'tangentScale2')
    def setTangentScale2(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.alignSurface, kwargs, 'tangentScale2', val)

    @_f.addMelDocs('alignSurface', 'twist')
    def setTwist(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.alignSurface, kwargs, 'twist', val)


class ArubaTessellate(AbstractBaseCreate):
    __melnode__ = 'arubaTessellate'
    __slots__ = ()


class AttachCurve(AbstractBaseCreate):
    __melcmd__ = staticmethod(modeling.attachCurve)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'attachCurve'
    __melnode__ = 'attachCurve'
    __slots__ = ()

    @_f.addMelDocs('attachCurve', 'blendBias')
    def getBlendBias(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.attachCurve, kwargs, 'blendBias')
        return res

    @_f.addMelDocs('attachCurve', 'blendKnotInsertion')
    def getBlendKnotInsertion(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.attachCurve, kwargs, 'blendKnotInsertion')
        return res

    @_f.addMelDocs('attachCurve', 'caching')
    def getCaching(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.attachCurve, kwargs, 'caching')
        return res

    @_f.addMelDocs('attachCurve', 'keepMultipleKnots')
    def getKeepMultipleKnots(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.attachCurve, kwargs, 'keepMultipleKnots')
        return res

    @_f.addMelDocs('attachCurve', 'method')
    def getMethod(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.attachCurve, kwargs, 'method')
        return res

    @_f.addMelDocs('attachCurve', 'nodeState')
    def getNodeState(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.attachCurve, kwargs, 'nodeState')
        return res

    @_f.addMelDocs('attachCurve', 'parameter')
    def getParameter(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.attachCurve, kwargs, 'parameter')
        return res

    @_f.addMelDocs('attachCurve', 'reverse1')
    def getReverse1(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.attachCurve, kwargs, 'reverse1')
        return res

    @_f.addMelDocs('attachCurve', 'reverse2')
    def getReverse2(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.attachCurve, kwargs, 'reverse2')
        return res

    @_f.addMelDocs('attachCurve', 'blendBias')
    def setBlendBias(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.attachCurve, kwargs, 'blendBias', val)

    @_f.addMelDocs('attachCurve', 'blendKnotInsertion')
    def setBlendKnotInsertion(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.attachCurve, kwargs, 'blendKnotInsertion', val)

    @_f.addMelDocs('attachCurve', 'caching')
    def setCaching(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.attachCurve, kwargs, 'caching', val)

    @_f.addMelDocs('attachCurve', 'keepMultipleKnots')
    def setKeepMultipleKnots(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.attachCurve, kwargs, 'keepMultipleKnots', val)

    @_f.addMelDocs('attachCurve', 'method')
    def setMethod(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.attachCurve, kwargs, 'method', val)

    @_f.addMelDocs('attachCurve', 'nodeState')
    def setNodeState(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.attachCurve, kwargs, 'nodeState', val)

    @_f.addMelDocs('attachCurve', 'parameter')
    def setParameter(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.attachCurve, kwargs, 'parameter', val)

    @_f.addMelDocs('attachCurve', 'reverse1')
    def setReverse1(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.attachCurve, kwargs, 'reverse1', val)

    @_f.addMelDocs('attachCurve', 'reverse2')
    def setReverse2(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.attachCurve, kwargs, 'reverse2', val)


class AttachSurface(AbstractBaseCreate):
    __melcmd__ = staticmethod(modeling.attachSurface)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'attachSurface'
    __melnode__ = 'attachSurface'
    __slots__ = ()

    @_f.addMelDocs('attachSurface', 'blendBias')
    def getBlendBias(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.attachSurface, kwargs, 'blendBias')
        return res

    @_f.addMelDocs('attachSurface', 'blendKnotInsertion')
    def getBlendKnotInsertion(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.attachSurface, kwargs, 'blendKnotInsertion')
        return res

    @_f.addMelDocs('attachSurface', 'caching')
    def getCaching(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.attachSurface, kwargs, 'caching')
        return res

    @_f.addMelDocs('attachSurface', 'directionU')
    def getDirectionU(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.attachSurface, kwargs, 'directionU')
        return res

    @_f.addMelDocs('attachSurface', 'keepMultipleKnots')
    def getKeepMultipleKnots(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.attachSurface, kwargs, 'keepMultipleKnots')
        return res

    @_f.addMelDocs('attachSurface', 'method')
    def getMethod(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.attachSurface, kwargs, 'method')
        return res

    @_f.addMelDocs('attachSurface', 'nodeState')
    def getNodeState(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.attachSurface, kwargs, 'nodeState')
        return res

    @_f.addMelDocs('attachSurface', 'parameter')
    def getParameter(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.attachSurface, kwargs, 'parameter')
        return res

    @_f.addMelDocs('attachSurface', 'reverse1')
    def getReverse1(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.attachSurface, kwargs, 'reverse1')
        return res

    @_f.addMelDocs('attachSurface', 'reverse2')
    def getReverse2(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.attachSurface, kwargs, 'reverse2')
        return res

    @_f.addMelDocs('attachSurface', 'swap1')
    def getSwap1(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.attachSurface, kwargs, 'swap1')
        return res

    @_f.addMelDocs('attachSurface', 'swap2')
    def getSwap2(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.attachSurface, kwargs, 'swap2')
        return res

    @_f.addMelDocs('attachSurface', 'twist')
    def getTwist(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.attachSurface, kwargs, 'twist')
        return res

    @_f.addMelDocs('attachSurface', 'blendBias')
    def setBlendBias(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.attachSurface, kwargs, 'blendBias', val)

    @_f.addMelDocs('attachSurface', 'blendKnotInsertion')
    def setBlendKnotInsertion(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.attachSurface, kwargs, 'blendKnotInsertion', val)

    @_f.addMelDocs('attachSurface', 'caching')
    def setCaching(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.attachSurface, kwargs, 'caching', val)

    @_f.addMelDocs('attachSurface', 'directionU')
    def setDirectionU(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.attachSurface, kwargs, 'directionU', val)

    @_f.addMelDocs('attachSurface', 'keepMultipleKnots')
    def setKeepMultipleKnots(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.attachSurface, kwargs, 'keepMultipleKnots', val)

    @_f.addMelDocs('attachSurface', 'method')
    def setMethod(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.attachSurface, kwargs, 'method', val)

    @_f.addMelDocs('attachSurface', 'nodeState')
    def setNodeState(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.attachSurface, kwargs, 'nodeState', val)

    @_f.addMelDocs('attachSurface', 'parameter')
    def setParameter(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.attachSurface, kwargs, 'parameter', val)

    @_f.addMelDocs('attachSurface', 'reverse1')
    def setReverse1(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.attachSurface, kwargs, 'reverse1', val)

    @_f.addMelDocs('attachSurface', 'reverse2')
    def setReverse2(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.attachSurface, kwargs, 'reverse2', val)

    @_f.addMelDocs('attachSurface', 'swap1')
    def setSwap1(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.attachSurface, kwargs, 'swap1', val)

    @_f.addMelDocs('attachSurface', 'swap2')
    def setSwap2(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.attachSurface, kwargs, 'swap2', val)

    @_f.addMelDocs('attachSurface', 'twist')
    def setTwist(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.attachSurface, kwargs, 'twist', val)


class AvgCurves(AbstractBaseCreate):
    __melnode__ = 'avgCurves'
    __slots__ = ()


class AvgNurbsSurfacePoints(AbstractBaseCreate):
    __melnode__ = 'avgNurbsSurfacePoints'
    __slots__ = ()


class AvgSurfacePoints(AbstractBaseCreate):
    __melnode__ = 'avgSurfacePoints'
    __slots__ = ()


class Bevel(AbstractBaseCreate):
    __melcmd__ = staticmethod(modeling.bevel)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'bevel'
    __melnode__ = 'bevel'
    __slots__ = ()

    @_f.addMelDocs('bevel', 'bevelShapeType')
    def getBevelShapeType(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.bevel, kwargs, 'bevelShapeType')
        return res

    @_f.addMelDocs('bevel', 'caching')
    def getCaching(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.bevel, kwargs, 'caching')
        return res

    @_f.addMelDocs('bevel', 'cornerType')
    def getCornerType(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.bevel, kwargs, 'cornerType')
        return res

    @_f.addMelDocs('bevel', 'depth')
    def getDepth(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.bevel, kwargs, 'depth')
        return res

    @_f.addMelDocs('bevel', 'extrudeDepth')
    def getExtrudeDepth(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.bevel, kwargs, 'extrudeDepth')
        return res

    @_f.addMelDocs('bevel', 'joinSurfaces')
    def getJoinSurfaces(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.bevel, kwargs, 'joinSurfaces')
        return res

    @_f.addMelDocs('bevel', 'nodeState')
    def getNodeState(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.bevel, kwargs, 'nodeState')
        return res

    @_f.addMelDocs('bevel', 'numberOfSides')
    def getNumberOfSides(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.bevel, kwargs, 'numberOfSides')
        return res

    @_f.addMelDocs('bevel', 'tolerance')
    def getTolerance(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.bevel, kwargs, 'tolerance')
        return res

    @_f.addMelDocs('bevel', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.bevel, kwargs, 'width')
        return res

    @_f.addMelDocs('bevel', 'bevelShapeType')
    def setBevelShapeType(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.bevel, kwargs, 'bevelShapeType', val)

    @_f.addMelDocs('bevel', 'caching')
    def setCaching(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.bevel, kwargs, 'caching', val)

    @_f.addMelDocs('bevel', 'cornerType')
    def setCornerType(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.bevel, kwargs, 'cornerType', val)

    @_f.addMelDocs('bevel', 'depth')
    def setDepth(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.bevel, kwargs, 'depth', val)

    @_f.addMelDocs('bevel', 'extrudeDepth')
    def setExtrudeDepth(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.bevel, kwargs, 'extrudeDepth', val)

    @_f.addMelDocs('bevel', 'joinSurfaces')
    def setJoinSurfaces(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.bevel, kwargs, 'joinSurfaces', val)

    @_f.addMelDocs('bevel', 'nodeState')
    def setNodeState(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.bevel, kwargs, 'nodeState', val)

    @_f.addMelDocs('bevel', 'numberOfSides')
    def setNumberOfSides(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.bevel, kwargs, 'numberOfSides', val)

    @_f.addMelDocs('bevel', 'tolerance')
    def setTolerance(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.bevel, kwargs, 'tolerance', val)

    @_f.addMelDocs('bevel', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.bevel, kwargs, 'width', val)


class BevelPlus(AbstractBaseCreate):
    __melcmd__ = staticmethod(modeling.bevelPlus)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'bevelPlus'
    __melnode__ = 'bevelPlus'
    __slots__ = ()

    @_f.addMelDocs('bevelPlus', 'bevelInside')
    def getBevelInside(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.bevelPlus, kwargs, 'bevelInside')
        return res

    @_f.addMelDocs('bevelPlus', 'capSides')
    def getCapSides(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, modeling.bevelPlus, kwargs, 'capSides')
        return res

    @_f.addMelDocs('bevelPlus', 'innerStyle')
    def getInnerStyle(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.bevelPlus, kwargs, 'innerStyle')
        return res

    @_f.addMelDocs('bevelPlus', 'joinSurfaces')
    def getJoinSurfaces(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.bevelPlus, kwargs, 'joinSurfaces')
        return res

    @_f.addMelDocs('bevelPlus', 'normalsOutwards')
    def getNormalsOutwards(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.bevelPlus, kwargs, 'normalsOutwards')
        return res

    @_f.addMelDocs('bevelPlus', 'numberOfSides')
    def getNumberOfSides(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.bevelPlus, kwargs, 'numberOfSides')
        return res

    @_f.addMelDocs('bevelPlus', 'outerStyle')
    def getOuterStyle(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.bevelPlus, kwargs, 'outerStyle')
        return res

    @_f.addMelDocs('bevelPlus', 'bevelInside')
    def setBevelInside(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.bevelPlus, kwargs, 'bevelInside', val)

    @_f.addMelDocs('bevelPlus', 'innerStyle')
    def setInnerStyle(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.bevelPlus, kwargs, 'innerStyle', val)

    @_f.addMelDocs('bevelPlus', 'joinSurfaces')
    def setJoinSurfaces(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.bevelPlus, kwargs, 'joinSurfaces', val)

    @_f.addMelDocs('bevelPlus', 'normalsOutwards')
    def setNormalsOutwards(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.bevelPlus, kwargs, 'normalsOutwards', val)

    @_f.addMelDocs('bevelPlus', 'numberOfSides')
    def setNumberOfSides(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.bevelPlus, kwargs, 'numberOfSides', val)

    @_f.addMelDocs('bevelPlus', 'outerStyle')
    def setOuterStyle(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.bevelPlus, kwargs, 'outerStyle', val)


class BezierCurveToNurbs(AbstractBaseCreate):
    __melcmd__ = staticmethod(modeling.bezierCurveToNurbs)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'bezierCurveToNurbs'
    __melnode__ = 'bezierCurveToNurbs'
    __slots__ = ()


class BirailSrf(AbstractBaseCreate):
    __melnode__ = 'birailSrf'
    __slots__ = ()


class DpBirailSrf(BirailSrf):
    __melnode__ = 'dpBirailSrf'
    __slots__ = ()


class MpBirailSrf(BirailSrf):
    __melnode__ = 'mpBirailSrf'
    __slots__ = ()


class SpBirailSrf(BirailSrf):
    __melnode__ = 'spBirailSrf'
    __slots__ = ()


class Boolean(AbstractBaseCreate):
    __melnode__ = 'boolean'
    __slots__ = ()


class BoundaryBase(AbstractBaseCreate):
    __melnode__ = 'boundaryBase'
    __slots__ = ()


class Boundary(BoundaryBase):
    __melcmd__ = staticmethod(modeling.boundary)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'boundary'
    __melnode__ = 'boundary'
    __slots__ = ()

    @_f.addMelDocs('boundary', 'caching')
    def getCaching(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.boundary, kwargs, 'caching')
        return res

    @_f.addMelDocs('boundary', 'endPoint')
    def getEndPoint(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.boundary, kwargs, 'endPoint')
        return res

    @_f.addMelDocs('boundary', 'endPointTolerance')
    def getEndPointTolerance(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.boundary, kwargs, 'endPointTolerance')
        return res

    @_f.addMelDocs('boundary', 'nodeState')
    def getNodeState(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.boundary, kwargs, 'nodeState')
        return res

    @_f.addMelDocs('boundary', 'order')
    def getOrder(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.boundary, kwargs, 'order')
        return res

    @_f.addMelDocs('boundary', 'caching')
    def setCaching(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.boundary, kwargs, 'caching', val)

    @_f.addMelDocs('boundary', 'endPoint')
    def setEndPoint(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.boundary, kwargs, 'endPoint', val)

    @_f.addMelDocs('boundary', 'endPointTolerance')
    def setEndPointTolerance(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.boundary, kwargs, 'endPointTolerance', val)

    @_f.addMelDocs('boundary', 'nodeState')
    def setNodeState(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.boundary, kwargs, 'nodeState', val)

    @_f.addMelDocs('boundary', 'order')
    def setOrder(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.boundary, kwargs, 'order', val)


class SquareSrf(BoundaryBase):
    __melnode__ = 'squareSrf'
    __slots__ = ()


class CloseCurve(AbstractBaseCreate):
    __melcmd__ = staticmethod(modeling.closeCurve)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'closeCurve'
    __melnode__ = 'closeCurve'
    __slots__ = ()

    @_f.addMelDocs('closeCurve', 'blendBias')
    def getBlendBias(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.closeCurve, kwargs, 'blendBias')
        return res

    @_f.addMelDocs('closeCurve', 'blendKnotInsertion')
    def getBlendKnotInsertion(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.closeCurve, kwargs, 'blendKnotInsertion')
        return res

    @_f.addMelDocs('closeCurve', 'caching')
    def getCaching(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.closeCurve, kwargs, 'caching')
        return res

    @_f.addMelDocs('closeCurve', 'nodeState')
    def getNodeState(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.closeCurve, kwargs, 'nodeState')
        return res

    @_f.addMelDocs('closeCurve', 'parameter')
    def getParameter(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.closeCurve, kwargs, 'parameter')
        return res

    @_f.addMelDocs('closeCurve', 'preserveShape')
    def getPreserveShape(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.closeCurve, kwargs, 'preserveShape')
        return res

    @_f.addMelDocs('closeCurve', 'blendBias')
    def setBlendBias(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.closeCurve, kwargs, 'blendBias', val)

    @_f.addMelDocs('closeCurve', 'blendKnotInsertion')
    def setBlendKnotInsertion(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.closeCurve, kwargs, 'blendKnotInsertion', val)

    @_f.addMelDocs('closeCurve', 'caching')
    def setCaching(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.closeCurve, kwargs, 'caching', val)

    @_f.addMelDocs('closeCurve', 'nodeState')
    def setNodeState(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.closeCurve, kwargs, 'nodeState', val)

    @_f.addMelDocs('closeCurve', 'parameter')
    def setParameter(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.closeCurve, kwargs, 'parameter', val)

    @_f.addMelDocs('closeCurve', 'preserveShape')
    def setPreserveShape(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.closeCurve, kwargs, 'preserveShape', val)


class CloseSurface(AbstractBaseCreate):
    __melcmd__ = staticmethod(modeling.closeSurface)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'closeSurface'
    __melnode__ = 'closeSurface'
    __slots__ = ()

    @_f.addMelDocs('closeSurface', 'blendBias')
    def getBlendBias(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.closeSurface, kwargs, 'blendBias')
        return res

    @_f.addMelDocs('closeSurface', 'blendKnotInsertion')
    def getBlendKnotInsertion(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.closeSurface, kwargs, 'blendKnotInsertion')
        return res

    @_f.addMelDocs('closeSurface', 'caching')
    def getCaching(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.closeSurface, kwargs, 'caching')
        return res

    @_f.addMelDocs('closeSurface', 'direction')
    def getDirection(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.closeSurface, kwargs, 'direction')
        return res

    @_f.addMelDocs('closeSurface', 'nodeState')
    def getNodeState(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.closeSurface, kwargs, 'nodeState')
        return res

    @_f.addMelDocs('closeSurface', 'parameter')
    def getParameter(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.closeSurface, kwargs, 'parameter')
        return res

    @_f.addMelDocs('closeSurface', 'preserveShape')
    def getPreserveShape(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.closeSurface, kwargs, 'preserveShape')
        return res

    @_f.addMelDocs('closeSurface', 'blendBias')
    def setBlendBias(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.closeSurface, kwargs, 'blendBias', val)

    @_f.addMelDocs('closeSurface', 'blendKnotInsertion')
    def setBlendKnotInsertion(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.closeSurface, kwargs, 'blendKnotInsertion', val)

    @_f.addMelDocs('closeSurface', 'caching')
    def setCaching(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.closeSurface, kwargs, 'caching', val)

    @_f.addMelDocs('closeSurface', 'direction')
    def setDirection(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.closeSurface, kwargs, 'direction', val)

    @_f.addMelDocs('closeSurface', 'nodeState')
    def setNodeState(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.closeSurface, kwargs, 'nodeState', val)

    @_f.addMelDocs('closeSurface', 'parameter')
    def setParameter(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.closeSurface, kwargs, 'parameter', val)

    @_f.addMelDocs('closeSurface', 'preserveShape')
    def setPreserveShape(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.closeSurface, kwargs, 'preserveShape', val)


class ClosestPointOnSurface(AbstractBaseCreate):
    __melnode__ = 'closestPointOnSurface'
    __slots__ = ()


class CurveInfo(AbstractBaseCreate):
    __melnode__ = 'curveInfo'
    __slots__ = ()


class CurveIntersect(AbstractBaseCreate):
    __melcmd__ = staticmethod(modeling.curveIntersect)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'curveIntersect'
    __melnode__ = 'curveIntersect'
    __slots__ = ()

    @_f.addMelDocs('curveIntersect', 'caching')
    def getCaching(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.curveIntersect, kwargs, 'caching')
        return res

    @_f.addMelDocs('curveIntersect', 'direction')
    def getDirection(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, modeling.curveIntersect, kwargs, 'direction')
        return res

    @_f.addMelDocs('curveIntersect', 'directionX')
    def getDirectionX(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.curveIntersect, kwargs, 'directionX')
        return res

    @_f.addMelDocs('curveIntersect', 'directionY')
    def getDirectionY(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.curveIntersect, kwargs, 'directionY')
        return res

    @_f.addMelDocs('curveIntersect', 'directionZ')
    def getDirectionZ(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.curveIntersect, kwargs, 'directionZ')
        return res

    @_f.addMelDocs('curveIntersect', 'nodeState')
    def getNodeState(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.curveIntersect, kwargs, 'nodeState')
        return res

    @_f.addMelDocs('curveIntersect', 'tolerance')
    def getTolerance(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.curveIntersect, kwargs, 'tolerance')
        return res

    @_f.addMelDocs('curveIntersect', 'useDirection')
    def getUseDirection(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.curveIntersect, kwargs, 'useDirection')
        return res

    @_f.addMelDocs('curveIntersect', 'caching')
    def setCaching(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.curveIntersect, kwargs, 'caching', val)

    @_f.addMelDocs('curveIntersect', 'direction')
    def setDirection(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, modeling.curveIntersect, kwargs, 'direction', val)

    @_f.addMelDocs('curveIntersect', 'directionX')
    def setDirectionX(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.curveIntersect, kwargs, 'directionX', val)

    @_f.addMelDocs('curveIntersect', 'directionY')
    def setDirectionY(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.curveIntersect, kwargs, 'directionY', val)

    @_f.addMelDocs('curveIntersect', 'directionZ')
    def setDirectionZ(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.curveIntersect, kwargs, 'directionZ', val)

    @_f.addMelDocs('curveIntersect', 'nodeState')
    def setNodeState(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.curveIntersect, kwargs, 'nodeState', val)

    @_f.addMelDocs('curveIntersect', 'tolerance')
    def setTolerance(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.curveIntersect, kwargs, 'tolerance', val)

    @_f.addMelDocs('curveIntersect', 'useDirection')
    def setUseDirection(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.curveIntersect, kwargs, 'useDirection', val)


class CurveRange(AbstractBaseCreate):
    __melnode__ = 'curveRange'
    __slots__ = ()


class CurveFromSubdiv(CurveRange):
    __melnode__ = 'curveFromSubdiv'
    __slots__ = ()


class CurveFromSubdivEdge(CurveFromSubdiv):
    __melnode__ = 'curveFromSubdivEdge'
    __slots__ = ()


class CurveFromSubdivFace(CurveFromSubdiv):
    __melnode__ = 'curveFromSubdivFace'
    __slots__ = ()


class CurveFromSurface(CurveRange):
    __melnode__ = 'curveFromSurface'
    __slots__ = ()


class CurveFromSurfaceBnd(CurveFromSurface):
    __melnode__ = 'curveFromSurfaceBnd'
    __slots__ = ()


class CurveFromSurfaceCoS(CurveFromSurface):
    __melnode__ = 'curveFromSurfaceCoS'
    __slots__ = ()


class CurveFromSurfaceIso(CurveFromSurface):
    __melnode__ = 'curveFromSurfaceIso'
    __slots__ = ()


class SubCurve(CurveRange):
    __melnode__ = 'subCurve'
    __slots__ = ()


class DetachCurve(AbstractBaseCreate):
    __melcmd__ = staticmethod(modeling.detachCurve)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'detachCurve'
    __melnode__ = 'detachCurve'
    __slots__ = ()

    @_f.addMelDocs('detachCurve', 'caching')
    def getCaching(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.detachCurve, kwargs, 'caching')
        return res

    @_f.addMelDocs('detachCurve', 'keep')
    def getKeep(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.detachCurve, kwargs, 'keep')
        return res

    @_f.addMelDocs('detachCurve', 'nodeState')
    def getNodeState(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.detachCurve, kwargs, 'nodeState')
        return res

    @_f.addMelDocs('detachCurve', 'parameter')
    def getParameter(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.detachCurve, kwargs, 'parameter')
        return res

    @_f.addMelDocs('detachCurve', 'caching')
    def setCaching(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.detachCurve, kwargs, 'caching', val)

    @_f.addMelDocs('detachCurve', 'keep')
    def setKeep(self, val=True, **kwargs):
        # type: (bool | int | List[bool | int], **Any) -> None
        return _f.asEdit(self, modeling.detachCurve, kwargs, 'keep', val)

    @_f.addMelDocs('detachCurve', 'nodeState')
    def setNodeState(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.detachCurve, kwargs, 'nodeState', val)

    @_f.addMelDocs('detachCurve', 'parameter')
    def setParameter(self, val=True, **kwargs):
        # type: (float | List[float], **Any) -> None
        return _f.asEdit(self, modeling.detachCurve, kwargs, 'parameter', val)


class DetachSurface(AbstractBaseCreate):
    __melcmd__ = staticmethod(modeling.detachSurface)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'detachSurface'
    __melnode__ = 'detachSurface'
    __slots__ = ()

    @_f.addMelDocs('detachSurface', 'caching')
    def getCaching(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.detachSurface, kwargs, 'caching')
        return res

    @_f.addMelDocs('detachSurface', 'direction')
    def getDirection(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.detachSurface, kwargs, 'direction')
        return res

    @_f.addMelDocs('detachSurface', 'keep')
    def getKeep(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.detachSurface, kwargs, 'keep')
        return res

    @_f.addMelDocs('detachSurface', 'nodeState')
    def getNodeState(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.detachSurface, kwargs, 'nodeState')
        return res

    @_f.addMelDocs('detachSurface', 'parameter')
    def getParameter(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.detachSurface, kwargs, 'parameter')
        return res

    @_f.addMelDocs('detachSurface', 'caching')
    def setCaching(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.detachSurface, kwargs, 'caching', val)

    @_f.addMelDocs('detachSurface', 'direction')
    def setDirection(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.detachSurface, kwargs, 'direction', val)

    @_f.addMelDocs('detachSurface', 'keep')
    def setKeep(self, val=True, **kwargs):
        # type: (bool | int | List[bool | int], **Any) -> None
        return _f.asEdit(self, modeling.detachSurface, kwargs, 'keep', val)

    @_f.addMelDocs('detachSurface', 'nodeState')
    def setNodeState(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.detachSurface, kwargs, 'nodeState', val)

    @_f.addMelDocs('detachSurface', 'parameter')
    def setParameter(self, val=True, **kwargs):
        # type: (float | List[float], **Any) -> None
        return _f.asEdit(self, modeling.detachSurface, kwargs, 'parameter', val)


class ExplodeNurbsShell(AbstractBaseCreate):
    __melnode__ = 'explodeNurbsShell'
    __slots__ = ()


class ExtendCurve(AbstractBaseCreate):
    __melcmd__ = staticmethod(modeling.extendCurve)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'extendCurve'
    __melnode__ = 'extendCurve'
    __slots__ = ()

    @_f.addMelDocs('extendCurve', 'caching')
    def getCaching(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.extendCurve, kwargs, 'caching')
        return res

    @_f.addMelDocs('extendCurve', 'distance')
    def getDistance(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.extendCurve, kwargs, 'distance')
        return res

    @_f.addMelDocs('extendCurve', 'extendMethod')
    def getExtendMethod(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.extendCurve, kwargs, 'extendMethod')
        return res

    @_f.addMelDocs('extendCurve', 'extensionType')
    def getExtensionType(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.extendCurve, kwargs, 'extensionType')
        return res

    @_f.addMelDocs('extendCurve', 'inputPoint')
    def getInputPoint(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, modeling.extendCurve, kwargs, 'inputPoint')
        return res

    @_f.addMelDocs('extendCurve', 'join')
    def getJoin(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.extendCurve, kwargs, 'join')
        return res

    @_f.addMelDocs('extendCurve', 'noChanges')
    def getNoChanges(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.extendCurve, kwargs, 'noChanges')
        return res

    @_f.addMelDocs('extendCurve', 'nodeState')
    def getNodeState(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.extendCurve, kwargs, 'nodeState')
        return res

    @_f.addMelDocs('extendCurve', 'pointX')
    def getPointX(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.extendCurve, kwargs, 'pointX')
        return res

    @_f.addMelDocs('extendCurve', 'pointY')
    def getPointY(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.extendCurve, kwargs, 'pointY')
        return res

    @_f.addMelDocs('extendCurve', 'pointZ')
    def getPointZ(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.extendCurve, kwargs, 'pointZ')
        return res

    @_f.addMelDocs('extendCurve', 'removeMultipleKnots')
    def getRemoveMultipleKnots(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.extendCurve, kwargs, 'removeMultipleKnots')
        return res

    @_f.addMelDocs('extendCurve', 'start')
    def getStart(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.extendCurve, kwargs, 'start')
        return res

    @_f.addMelDocs('extendCurve', 'caching')
    def setCaching(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.extendCurve, kwargs, 'caching', val)

    @_f.addMelDocs('extendCurve', 'distance')
    def setDistance(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.extendCurve, kwargs, 'distance', val)

    @_f.addMelDocs('extendCurve', 'extendMethod')
    def setExtendMethod(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.extendCurve, kwargs, 'extendMethod', val)

    @_f.addMelDocs('extendCurve', 'extensionType')
    def setExtensionType(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.extendCurve, kwargs, 'extensionType', val)

    @_f.addMelDocs('extendCurve', 'inputPoint')
    def setInputPoint(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, modeling.extendCurve, kwargs, 'inputPoint', val)

    @_f.addMelDocs('extendCurve', 'join')
    def setJoin(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.extendCurve, kwargs, 'join', val)

    @_f.addMelDocs('extendCurve', 'noChanges')
    def setNoChanges(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.extendCurve, kwargs, 'noChanges', val)

    @_f.addMelDocs('extendCurve', 'nodeState')
    def setNodeState(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.extendCurve, kwargs, 'nodeState', val)

    @_f.addMelDocs('extendCurve', 'pointX')
    def setPointX(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.extendCurve, kwargs, 'pointX', val)

    @_f.addMelDocs('extendCurve', 'pointY')
    def setPointY(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.extendCurve, kwargs, 'pointY', val)

    @_f.addMelDocs('extendCurve', 'pointZ')
    def setPointZ(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.extendCurve, kwargs, 'pointZ', val)

    @_f.addMelDocs('extendCurve', 'removeMultipleKnots')
    def setRemoveMultipleKnots(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.extendCurve, kwargs, 'removeMultipleKnots', val)

    @_f.addMelDocs('extendCurve', 'start')
    def setStart(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.extendCurve, kwargs, 'start', val)


class ExtendSurface(AbstractBaseCreate):
    __melcmd__ = staticmethod(modeling.extendSurface)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'extendSurface'
    __melnode__ = 'extendSurface'
    __slots__ = ()

    @_f.addMelDocs('extendSurface', 'caching')
    def getCaching(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.extendSurface, kwargs, 'caching')
        return res

    @_f.addMelDocs('extendSurface', 'distance')
    def getDistance(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.extendSurface, kwargs, 'distance')
        return res

    @_f.addMelDocs('extendSurface', 'extendDirection')
    def getExtendDirection(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.extendSurface, kwargs, 'extendDirection')
        return res

    @_f.addMelDocs('extendSurface', 'extendMethod')
    def getExtendMethod(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.extendSurface, kwargs, 'extendMethod')
        return res

    @_f.addMelDocs('extendSurface', 'extendSide')
    def getExtendSide(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.extendSurface, kwargs, 'extendSide')
        return res

    @_f.addMelDocs('extendSurface', 'extensionType')
    def getExtensionType(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.extendSurface, kwargs, 'extensionType')
        return res

    @_f.addMelDocs('extendSurface', 'join')
    def getJoin(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.extendSurface, kwargs, 'join')
        return res

    @_f.addMelDocs('extendSurface', 'nodeState')
    def getNodeState(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.extendSurface, kwargs, 'nodeState')
        return res

    @_f.addMelDocs('extendSurface', 'caching')
    def setCaching(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.extendSurface, kwargs, 'caching', val)

    @_f.addMelDocs('extendSurface', 'distance')
    def setDistance(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.extendSurface, kwargs, 'distance', val)

    @_f.addMelDocs('extendSurface', 'extendDirection')
    def setExtendDirection(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.extendSurface, kwargs, 'extendDirection', val)

    @_f.addMelDocs('extendSurface', 'extendMethod')
    def setExtendMethod(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.extendSurface, kwargs, 'extendMethod', val)

    @_f.addMelDocs('extendSurface', 'extendSide')
    def setExtendSide(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.extendSurface, kwargs, 'extendSide', val)

    @_f.addMelDocs('extendSurface', 'extensionType')
    def setExtensionType(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.extendSurface, kwargs, 'extensionType', val)

    @_f.addMelDocs('extendSurface', 'join')
    def setJoin(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.extendSurface, kwargs, 'join', val)

    @_f.addMelDocs('extendSurface', 'nodeState')
    def setNodeState(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.extendSurface, kwargs, 'nodeState', val)


class Extrude(AbstractBaseCreate):
    __melcmd__ = staticmethod(modeling.extrude)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'extrude'
    __melnode__ = 'extrude'
    __slots__ = ()

    @_f.addMelDocs('extrude', 'caching')
    def getCaching(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.extrude, kwargs, 'caching')
        return res

    @_f.addMelDocs('extrude', 'degreeAlongLength')
    def getDegreeAlongLength(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.extrude, kwargs, 'degreeAlongLength')
        return res

    @_f.addMelDocs('extrude', 'direction')
    def getDirection(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, modeling.extrude, kwargs, 'direction')
        return res

    @_f.addMelDocs('extrude', 'directionX')
    def getDirectionX(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.extrude, kwargs, 'directionX')
        return res

    @_f.addMelDocs('extrude', 'directionY')
    def getDirectionY(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.extrude, kwargs, 'directionY')
        return res

    @_f.addMelDocs('extrude', 'directionZ')
    def getDirectionZ(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.extrude, kwargs, 'directionZ')
        return res

    @_f.addMelDocs('extrude', 'extrudeType')
    def getExtrudeType(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.extrude, kwargs, 'extrudeType')
        return res

    @_f.addMelDocs('extrude', 'fixedPath')
    def getFixedPath(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.extrude, kwargs, 'fixedPath')
        return res

    @_f.addMelDocs('extrude', 'length')
    def getLength(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.extrude, kwargs, 'length')
        return res

    @_f.addMelDocs('extrude', 'nodeState')
    def getNodeState(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.extrude, kwargs, 'nodeState')
        return res

    @_f.addMelDocs('extrude', 'pivot')
    def getPivot(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, modeling.extrude, kwargs, 'pivot')
        return res

    @_f.addMelDocs('extrude', 'reverseSurfaceIfPathReversed')
    def getReverseSurfaceIfPathReversed(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.extrude, kwargs, 'reverseSurfaceIfPathReversed')
        return res

    @_f.addMelDocs('extrude', 'rotation')
    def getRotation(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.extrude, kwargs, 'rotation')
        return res

    @_f.addMelDocs('extrude', 'scale')
    def getScale(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.extrude, kwargs, 'scale')
        return res

    @_f.addMelDocs('extrude', 'subCurveSubSurface')
    def getSubCurveSubSurface(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.extrude, kwargs, 'subCurveSubSurface')
        return res

    @_f.addMelDocs('extrude', 'useComponentPivot')
    def getUseComponentPivot(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.extrude, kwargs, 'useComponentPivot')
        return res

    @_f.addMelDocs('extrude', 'useProfileNormal')
    def getUseProfileNormal(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.extrude, kwargs, 'useProfileNormal')
        return res

    @_f.addMelDocs('extrude', 'caching')
    def setCaching(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.extrude, kwargs, 'caching', val)

    @_f.addMelDocs('extrude', 'degreeAlongLength')
    def setDegreeAlongLength(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.extrude, kwargs, 'degreeAlongLength', val)

    @_f.addMelDocs('extrude', 'direction')
    def setDirection(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, modeling.extrude, kwargs, 'direction', val)

    @_f.addMelDocs('extrude', 'directionX')
    def setDirectionX(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.extrude, kwargs, 'directionX', val)

    @_f.addMelDocs('extrude', 'directionY')
    def setDirectionY(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.extrude, kwargs, 'directionY', val)

    @_f.addMelDocs('extrude', 'directionZ')
    def setDirectionZ(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.extrude, kwargs, 'directionZ', val)

    @_f.addMelDocs('extrude', 'extrudeType')
    def setExtrudeType(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.extrude, kwargs, 'extrudeType', val)

    @_f.addMelDocs('extrude', 'fixedPath')
    def setFixedPath(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.extrude, kwargs, 'fixedPath', val)

    @_f.addMelDocs('extrude', 'length')
    def setLength(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.extrude, kwargs, 'length', val)

    @_f.addMelDocs('extrude', 'nodeState')
    def setNodeState(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.extrude, kwargs, 'nodeState', val)

    @_f.addMelDocs('extrude', 'pivot')
    def setPivot(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, modeling.extrude, kwargs, 'pivot', val)

    @_f.addMelDocs('extrude', 'reverseSurfaceIfPathReversed')
    def setReverseSurfaceIfPathReversed(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.extrude, kwargs, 'reverseSurfaceIfPathReversed', val)

    @_f.addMelDocs('extrude', 'rotation')
    def setRotation(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.extrude, kwargs, 'rotation', val)

    @_f.addMelDocs('extrude', 'scale')
    def setScale(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.extrude, kwargs, 'scale', val)

    @_f.addMelDocs('extrude', 'subCurveSubSurface')
    def setSubCurveSubSurface(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.extrude, kwargs, 'subCurveSubSurface', val)

    @_f.addMelDocs('extrude', 'useComponentPivot')
    def setUseComponentPivot(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.extrude, kwargs, 'useComponentPivot', val)

    @_f.addMelDocs('extrude', 'useProfileNormal')
    def setUseProfileNormal(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.extrude, kwargs, 'useProfileNormal', val)


class FfBlendSrf(AbstractBaseCreate):
    __melnode__ = 'ffBlendSrf'
    __slots__ = ()


class FfBlendSrfObsolete(AbstractBaseCreate):
    __melnode__ = 'ffBlendSrfObsolete'
    __slots__ = ()


class FfFilletSrf(AbstractBaseCreate):
    __melnode__ = 'ffFilletSrf'
    __slots__ = ()


class FilletCurve(AbstractBaseCreate):
    __melcmd__ = staticmethod(modeling.filletCurve)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'filletCurve'
    __melnode__ = 'filletCurve'
    __slots__ = ()

    @_f.addMelDocs('filletCurve', 'bias')
    def getBias(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.filletCurve, kwargs, 'bias')
        return res

    @_f.addMelDocs('filletCurve', 'blendControl')
    def getBlendControl(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.filletCurve, kwargs, 'blendControl')
        return res

    @_f.addMelDocs('filletCurve', 'caching')
    def getCaching(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.filletCurve, kwargs, 'caching')
        return res

    @_f.addMelDocs('filletCurve', 'circular')
    def getCircular(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.filletCurve, kwargs, 'circular')
        return res

    @_f.addMelDocs('filletCurve', 'curveParameter1')
    def getCurveParameter1(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.filletCurve, kwargs, 'curveParameter1')
        return res

    @_f.addMelDocs('filletCurve', 'curveParameter2')
    def getCurveParameter2(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.filletCurve, kwargs, 'curveParameter2')
        return res

    @_f.addMelDocs('filletCurve', 'depth')
    def getDepth(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.filletCurve, kwargs, 'depth')
        return res

    @_f.addMelDocs('filletCurve', 'freeformBlend')
    def getFreeformBlend(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.filletCurve, kwargs, 'freeformBlend')
        return res

    @_f.addMelDocs('filletCurve', 'nodeState')
    def getNodeState(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.filletCurve, kwargs, 'nodeState')
        return res

    @_f.addMelDocs('filletCurve', 'radius')
    def getRadius(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.filletCurve, kwargs, 'radius')
        return res

    @_f.addMelDocs('filletCurve', 'bias')
    def setBias(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.filletCurve, kwargs, 'bias', val)

    @_f.addMelDocs('filletCurve', 'blendControl')
    def setBlendControl(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.filletCurve, kwargs, 'blendControl', val)

    @_f.addMelDocs('filletCurve', 'caching')
    def setCaching(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.filletCurve, kwargs, 'caching', val)

    @_f.addMelDocs('filletCurve', 'circular')
    def setCircular(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.filletCurve, kwargs, 'circular', val)

    @_f.addMelDocs('filletCurve', 'curveParameter1')
    def setCurveParameter1(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.filletCurve, kwargs, 'curveParameter1', val)

    @_f.addMelDocs('filletCurve', 'curveParameter2')
    def setCurveParameter2(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.filletCurve, kwargs, 'curveParameter2', val)

    @_f.addMelDocs('filletCurve', 'depth')
    def setDepth(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.filletCurve, kwargs, 'depth', val)

    @_f.addMelDocs('filletCurve', 'freeformBlend')
    def setFreeformBlend(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.filletCurve, kwargs, 'freeformBlend', val)

    @_f.addMelDocs('filletCurve', 'nodeState')
    def setNodeState(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.filletCurve, kwargs, 'nodeState', val)

    @_f.addMelDocs('filletCurve', 'radius')
    def setRadius(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.filletCurve, kwargs, 'radius', val)


class FitBspline(AbstractBaseCreate):
    __melcmd__ = staticmethod(modeling.fitBspline)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'fitBspline'
    __melnode__ = 'fitBspline'
    __slots__ = ()

    @_f.addMelDocs('fitBspline', 'caching')
    def getCaching(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.fitBspline, kwargs, 'caching')
        return res

    @_f.addMelDocs('fitBspline', 'nodeState')
    def getNodeState(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.fitBspline, kwargs, 'nodeState')
        return res

    @_f.addMelDocs('fitBspline', 'tolerance')
    def getTolerance(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.fitBspline, kwargs, 'tolerance')
        return res

    @_f.addMelDocs('fitBspline', 'caching')
    def setCaching(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.fitBspline, kwargs, 'caching', val)

    @_f.addMelDocs('fitBspline', 'nodeState')
    def setNodeState(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.fitBspline, kwargs, 'nodeState', val)

    @_f.addMelDocs('fitBspline', 'tolerance')
    def setTolerance(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.fitBspline, kwargs, 'tolerance', val)


class GlobalStitch(AbstractBaseCreate):
    __melcmd__ = staticmethod(modeling.globalStitch)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'globalStitch'
    __melnode__ = 'globalStitch'
    __slots__ = ()

    @_f.addMelDocs('globalStitch', 'caching')
    def getCaching(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.globalStitch, kwargs, 'caching')
        return res

    @_f.addMelDocs('globalStitch', 'lockSurface')
    def getLockSurface(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.globalStitch, kwargs, 'lockSurface')
        return res

    @_f.addMelDocs('globalStitch', 'maxSeparation')
    def getMaxSeparation(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.globalStitch, kwargs, 'maxSeparation')
        return res

    @_f.addMelDocs('globalStitch', 'modificationResistance')
    def getModificationResistance(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.globalStitch, kwargs, 'modificationResistance')
        return res

    @_f.addMelDocs('globalStitch', 'nodeState')
    def getNodeState(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.globalStitch, kwargs, 'nodeState')
        return res

    @_f.addMelDocs('globalStitch', 'sampling')
    def getSampling(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.globalStitch, kwargs, 'sampling')
        return res

    @_f.addMelDocs('globalStitch', 'stitchCorners')
    def getStitchCorners(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.globalStitch, kwargs, 'stitchCorners')
        return res

    @_f.addMelDocs('globalStitch', 'stitchEdges')
    def getStitchEdges(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.globalStitch, kwargs, 'stitchEdges')
        return res

    @_f.addMelDocs('globalStitch', 'stitchPartialEdges')
    def getStitchPartialEdges(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.globalStitch, kwargs, 'stitchPartialEdges')
        return res

    @_f.addMelDocs('globalStitch', 'stitchSmoothness')
    def getStitchSmoothness(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.globalStitch, kwargs, 'stitchSmoothness')
        return res

    @_f.addMelDocs('globalStitch', 'caching')
    def setCaching(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.globalStitch, kwargs, 'caching', val)

    @_f.addMelDocs('globalStitch', 'lockSurface')
    def setLockSurface(self, val=True, **kwargs):
        # type: (bool | int | List[bool | int], **Any) -> None
        return _f.asEdit(self, modeling.globalStitch, kwargs, 'lockSurface', val)

    @_f.addMelDocs('globalStitch', 'maxSeparation')
    def setMaxSeparation(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.globalStitch, kwargs, 'maxSeparation', val)

    @_f.addMelDocs('globalStitch', 'modificationResistance')
    def setModificationResistance(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.globalStitch, kwargs, 'modificationResistance', val)

    @_f.addMelDocs('globalStitch', 'nodeState')
    def setNodeState(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.globalStitch, kwargs, 'nodeState', val)

    @_f.addMelDocs('globalStitch', 'sampling')
    def setSampling(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.globalStitch, kwargs, 'sampling', val)

    @_f.addMelDocs('globalStitch', 'stitchCorners')
    def setStitchCorners(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.globalStitch, kwargs, 'stitchCorners', val)

    @_f.addMelDocs('globalStitch', 'stitchEdges')
    def setStitchEdges(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.globalStitch, kwargs, 'stitchEdges', val)

    @_f.addMelDocs('globalStitch', 'stitchPartialEdges')
    def setStitchPartialEdges(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.globalStitch, kwargs, 'stitchPartialEdges', val)

    @_f.addMelDocs('globalStitch', 'stitchSmoothness')
    def setStitchSmoothness(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.globalStitch, kwargs, 'stitchSmoothness', val)


class HardenPoint(AbstractBaseCreate):
    __melnode__ = 'hardenPoint'
    __slots__ = ()


class InsertKnotCurve(AbstractBaseCreate):
    __melcmd__ = staticmethod(modeling.insertKnotCurve)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'insertKnotCurve'
    __melnode__ = 'insertKnotCurve'
    __slots__ = ()

    @_f.addMelDocs('insertKnotCurve', 'addKnots')
    def getAddKnots(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.insertKnotCurve, kwargs, 'addKnots')
        return res

    @_f.addMelDocs('insertKnotCurve', 'caching')
    def getCaching(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.insertKnotCurve, kwargs, 'caching')
        return res

    @_f.addMelDocs('insertKnotCurve', 'insertBetween')
    def getInsertBetween(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.insertKnotCurve, kwargs, 'insertBetween')
        return res

    @_f.addMelDocs('insertKnotCurve', 'nodeState')
    def getNodeState(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.insertKnotCurve, kwargs, 'nodeState')
        return res

    @_f.addMelDocs('insertKnotCurve', 'numberOfKnots')
    def getNumberOfKnots(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.insertKnotCurve, kwargs, 'numberOfKnots')
        return res

    @_f.addMelDocs('insertKnotCurve', 'parameter')
    def getParameter(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.insertKnotCurve, kwargs, 'parameter')
        return res

    @_f.addMelDocs('insertKnotCurve', 'addKnots')
    def setAddKnots(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.insertKnotCurve, kwargs, 'addKnots', val)

    @_f.addMelDocs('insertKnotCurve', 'caching')
    def setCaching(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.insertKnotCurve, kwargs, 'caching', val)

    @_f.addMelDocs('insertKnotCurve', 'insertBetween')
    def setInsertBetween(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.insertKnotCurve, kwargs, 'insertBetween', val)

    @_f.addMelDocs('insertKnotCurve', 'nodeState')
    def setNodeState(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.insertKnotCurve, kwargs, 'nodeState', val)

    @_f.addMelDocs('insertKnotCurve', 'numberOfKnots')
    def setNumberOfKnots(self, val=True, **kwargs):
        # type: (int | List[int], **Any) -> None
        return _f.asEdit(self, modeling.insertKnotCurve, kwargs, 'numberOfKnots', val)

    @_f.addMelDocs('insertKnotCurve', 'parameter')
    def setParameter(self, val=True, **kwargs):
        # type: (float | List[float], **Any) -> None
        return _f.asEdit(self, modeling.insertKnotCurve, kwargs, 'parameter', val)


class InsertKnotSurface(AbstractBaseCreate):
    __melcmd__ = staticmethod(modeling.insertKnotSurface)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'insertKnotSurface'
    __melnode__ = 'insertKnotSurface'
    __slots__ = ()

    @_f.addMelDocs('insertKnotSurface', 'addKnots')
    def getAddKnots(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.insertKnotSurface, kwargs, 'addKnots')
        return res

    @_f.addMelDocs('insertKnotSurface', 'caching')
    def getCaching(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.insertKnotSurface, kwargs, 'caching')
        return res

    @_f.addMelDocs('insertKnotSurface', 'direction')
    def getDirection(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.insertKnotSurface, kwargs, 'direction')
        return res

    @_f.addMelDocs('insertKnotSurface', 'insertBetween')
    def getInsertBetween(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.insertKnotSurface, kwargs, 'insertBetween')
        return res

    @_f.addMelDocs('insertKnotSurface', 'nodeState')
    def getNodeState(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.insertKnotSurface, kwargs, 'nodeState')
        return res

    @_f.addMelDocs('insertKnotSurface', 'numberOfKnots')
    def getNumberOfKnots(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.insertKnotSurface, kwargs, 'numberOfKnots')
        return res

    @_f.addMelDocs('insertKnotSurface', 'parameter')
    def getParameter(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.insertKnotSurface, kwargs, 'parameter')
        return res

    @_f.addMelDocs('insertKnotSurface', 'addKnots')
    def setAddKnots(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.insertKnotSurface, kwargs, 'addKnots', val)

    @_f.addMelDocs('insertKnotSurface', 'caching')
    def setCaching(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.insertKnotSurface, kwargs, 'caching', val)

    @_f.addMelDocs('insertKnotSurface', 'direction')
    def setDirection(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.insertKnotSurface, kwargs, 'direction', val)

    @_f.addMelDocs('insertKnotSurface', 'insertBetween')
    def setInsertBetween(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.insertKnotSurface, kwargs, 'insertBetween', val)

    @_f.addMelDocs('insertKnotSurface', 'nodeState')
    def setNodeState(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.insertKnotSurface, kwargs, 'nodeState', val)

    @_f.addMelDocs('insertKnotSurface', 'numberOfKnots')
    def setNumberOfKnots(self, val=True, **kwargs):
        # type: (int | List[int], **Any) -> None
        return _f.asEdit(self, modeling.insertKnotSurface, kwargs, 'numberOfKnots', val)

    @_f.addMelDocs('insertKnotSurface', 'parameter')
    def setParameter(self, val=True, **kwargs):
        # type: (float | List[float], **Any) -> None
        return _f.asEdit(self, modeling.insertKnotSurface, kwargs, 'parameter', val)


class IntersectSurface(AbstractBaseCreate):
    __melnode__ = 'intersectSurface'
    __slots__ = ()


class LeastSquaresModifier(AbstractBaseCreate):
    __melnode__ = 'leastSquaresModifier'
    __slots__ = ()


class Loft(AbstractBaseCreate):
    __melcmd__ = staticmethod(modeling.loft)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'loft'
    __melnode__ = 'loft'
    __slots__ = ()

    @_f.addMelDocs('loft', 'autoReverse')
    def getAutoReverse(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.loft, kwargs, 'autoReverse')
        return res

    @_f.addMelDocs('loft', 'caching')
    def getCaching(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.loft, kwargs, 'caching')
        return res

    @_f.addMelDocs('loft', 'close')
    def getClose(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.loft, kwargs, 'close')
        return res

    @_f.addMelDocs('loft', 'createCusp')
    def getCreateCusp(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.loft, kwargs, 'createCusp')
        return res

    @_f.addMelDocs('loft', 'degree')
    def getDegree(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.loft, kwargs, 'degree')
        return res

    @_f.addMelDocs('loft', 'nodeState')
    def getNodeState(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.loft, kwargs, 'nodeState')
        return res

    @_f.addMelDocs('loft', 'reverse')
    def getReverse(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.loft, kwargs, 'reverse')
        return res

    @_f.addMelDocs('loft', 'reverseSurfaceNormals')
    def getReverseSurfaceNormals(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.loft, kwargs, 'reverseSurfaceNormals')
        return res

    @_f.addMelDocs('loft', 'sectionSpans')
    def getSectionSpans(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.loft, kwargs, 'sectionSpans')
        return res

    @_f.addMelDocs('loft', 'uniform')
    def getUniform(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.loft, kwargs, 'uniform')
        return res

    @_f.addMelDocs('loft', 'autoReverse')
    def setAutoReverse(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.loft, kwargs, 'autoReverse', val)

    @_f.addMelDocs('loft', 'caching')
    def setCaching(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.loft, kwargs, 'caching', val)

    @_f.addMelDocs('loft', 'close')
    def setClose(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.loft, kwargs, 'close', val)

    @_f.addMelDocs('loft', 'createCusp')
    def setCreateCusp(self, val=True, **kwargs):
        # type: (bool | int | List[bool | int], **Any) -> None
        return _f.asEdit(self, modeling.loft, kwargs, 'createCusp', val)

    @_f.addMelDocs('loft', 'degree')
    def setDegree(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.loft, kwargs, 'degree', val)

    @_f.addMelDocs('loft', 'nodeState')
    def setNodeState(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.loft, kwargs, 'nodeState', val)

    @_f.addMelDocs('loft', 'reverse')
    def setReverse(self, val=True, **kwargs):
        # type: (bool | int | List[bool | int], **Any) -> None
        return _f.asEdit(self, modeling.loft, kwargs, 'reverse', val)

    @_f.addMelDocs('loft', 'reverseSurfaceNormals')
    def setReverseSurfaceNormals(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.loft, kwargs, 'reverseSurfaceNormals', val)

    @_f.addMelDocs('loft', 'sectionSpans')
    def setSectionSpans(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.loft, kwargs, 'sectionSpans', val)

    @_f.addMelDocs('loft', 'uniform')
    def setUniform(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.loft, kwargs, 'uniform', val)


class MakeCircularArc(AbstractBaseCreate):
    __melnode__ = 'makeCircularArc'
    __slots__ = ()


class MakeThreePointCircularArc(MakeCircularArc):
    __melnode__ = 'makeThreePointCircularArc'
    __slots__ = ()


class MakeTwoPointCircularArc(MakeCircularArc):
    __melnode__ = 'makeTwoPointCircularArc'
    __slots__ = ()


class MakeIllustratorCurves(AbstractBaseCreate):
    __melnode__ = 'makeIllustratorCurves'
    __slots__ = ()


class MakeNurbCircle(AbstractBaseCreate):
    __melcmd__ = staticmethod(modeling.circle)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'circle'
    __melnode__ = 'makeNurbCircle'
    __slots__ = ()

    @_f.addMelDocs('circle', 'caching')
    def getCaching(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.circle, kwargs, 'caching')
        res = bool(res)
        return res

    @_f.addMelDocs('circle', 'center')
    def getCenter(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, modeling.circle, kwargs, 'center')
        return res

    @_f.addMelDocs('circle', 'centerX')
    def getCenterX(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.circle, kwargs, 'centerX')
        return res

    @_f.addMelDocs('circle', 'centerY')
    def getCenterY(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.circle, kwargs, 'centerY')
        return res

    @_f.addMelDocs('circle', 'centerZ')
    def getCenterZ(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.circle, kwargs, 'centerZ')
        return res

    @_f.addMelDocs('circle', 'degree')
    def getDegree(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.circle, kwargs, 'degree')
        res = int(res)
        return res

    @_f.addMelDocs('circle', 'first')
    def getFirst(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, modeling.circle, kwargs, 'first')
        return res

    @_f.addMelDocs('circle', 'firstPointX')
    def getFirstPointX(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.circle, kwargs, 'firstPointX')
        return res

    @_f.addMelDocs('circle', 'firstPointY')
    def getFirstPointY(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.circle, kwargs, 'firstPointY')
        return res

    @_f.addMelDocs('circle', 'firstPointZ')
    def getFirstPointZ(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.circle, kwargs, 'firstPointZ')
        return res

    @_f.addMelDocs('circle', 'fixCenter')
    def getFixCenter(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.circle, kwargs, 'fixCenter')
        res = bool(res)
        return res

    @_f.addMelDocs('circle', 'frozen')
    def getFrozen(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.circle, kwargs, 'frozen')
        return res

    @_f.addMelDocs('circle', 'nodeState')
    def getNodeState(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.circle, kwargs, 'nodeState')
        res = int(res)
        return res

    @_f.addMelDocs('circle', 'normal')
    def getNormal(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, modeling.circle, kwargs, 'normal')
        return res

    @_f.addMelDocs('circle', 'normalX')
    def getNormalX(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.circle, kwargs, 'normalX')
        return res

    @_f.addMelDocs('circle', 'normalY')
    def getNormalY(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.circle, kwargs, 'normalY')
        return res

    @_f.addMelDocs('circle', 'normalZ')
    def getNormalZ(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.circle, kwargs, 'normalZ')
        return res

    @_f.addMelDocs('circle', 'radius')
    def getRadius(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.circle, kwargs, 'radius')
        return res

    @_f.addMelDocs('circle', 'sections')
    def getSections(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.circle, kwargs, 'sections')
        res = int(res)
        return res

    @_f.addMelDocs('circle', 'sweep')
    def getSweep(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.circle, kwargs, 'sweep')
        return res

    @_f.addMelDocs('circle', 'tolerance')
    def getTolerance(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.circle, kwargs, 'tolerance')
        return res

    @_f.addMelDocs('circle', 'useTolerance')
    def getUseTolerance(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.circle, kwargs, 'useTolerance')
        res = bool(res)
        return res

    @_f.addMelDocs('circle', 'caching')
    def setCaching(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.circle, kwargs, 'caching', val)

    @_f.addMelDocs('circle', 'center')
    def setCenter(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, modeling.circle, kwargs, 'center', val)

    @_f.addMelDocs('circle', 'centerX')
    def setCenterX(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.circle, kwargs, 'centerX', val)

    @_f.addMelDocs('circle', 'centerY')
    def setCenterY(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.circle, kwargs, 'centerY', val)

    @_f.addMelDocs('circle', 'centerZ')
    def setCenterZ(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.circle, kwargs, 'centerZ', val)

    @_f.addMelDocs('circle', 'degree')
    def setDegree(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.circle, kwargs, 'degree', val)

    @_f.addMelDocs('circle', 'first')
    def setFirst(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, modeling.circle, kwargs, 'first', val)

    @_f.addMelDocs('circle', 'firstPointX')
    def setFirstPointX(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.circle, kwargs, 'firstPointX', val)

    @_f.addMelDocs('circle', 'firstPointY')
    def setFirstPointY(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.circle, kwargs, 'firstPointY', val)

    @_f.addMelDocs('circle', 'firstPointZ')
    def setFirstPointZ(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.circle, kwargs, 'firstPointZ', val)

    @_f.addMelDocs('circle', 'fixCenter')
    def setFixCenter(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.circle, kwargs, 'fixCenter', val)

    @_f.addMelDocs('circle', 'frozen')
    def setFrozen(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.circle, kwargs, 'frozen', val)

    @_f.addMelDocs('circle', 'nodeState')
    def setNodeState(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.circle, kwargs, 'nodeState', val)

    @_f.addMelDocs('circle', 'normal')
    def setNormal(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, modeling.circle, kwargs, 'normal', val)

    @_f.addMelDocs('circle', 'normalX')
    def setNormalX(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.circle, kwargs, 'normalX', val)

    @_f.addMelDocs('circle', 'normalY')
    def setNormalY(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.circle, kwargs, 'normalY', val)

    @_f.addMelDocs('circle', 'normalZ')
    def setNormalZ(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.circle, kwargs, 'normalZ', val)

    @_f.addMelDocs('circle', 'radius')
    def setRadius(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.circle, kwargs, 'radius', val)

    @_f.addMelDocs('circle', 'sections')
    def setSections(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.circle, kwargs, 'sections', val)

    @_f.addMelDocs('circle', 'sweep')
    def setSweep(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.circle, kwargs, 'sweep', val)

    @_f.addMelDocs('circle', 'tolerance')
    def setTolerance(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.circle, kwargs, 'tolerance', val)

    @_f.addMelDocs('circle', 'useTolerance')
    def setUseTolerance(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.circle, kwargs, 'useTolerance', val)


class MakeNurbsSquare(AbstractBaseCreate):
    __melcmd__ = staticmethod(modeling.nurbsSquare)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'nurbsSquare'
    __melnode__ = 'makeNurbsSquare'
    __slots__ = ()

    @_f.addMelDocs('nurbsSquare', 'caching')
    def getCaching(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.nurbsSquare, kwargs, 'caching')
        res = bool(res)
        return res

    @_f.addMelDocs('nurbsSquare', 'center')
    def getCenter(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, modeling.nurbsSquare, kwargs, 'center')
        return res

    @_f.addMelDocs('nurbsSquare', 'centerX')
    def getCenterX(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.nurbsSquare, kwargs, 'centerX')
        return res

    @_f.addMelDocs('nurbsSquare', 'centerY')
    def getCenterY(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.nurbsSquare, kwargs, 'centerY')
        return res

    @_f.addMelDocs('nurbsSquare', 'centerZ')
    def getCenterZ(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.nurbsSquare, kwargs, 'centerZ')
        return res

    @_f.addMelDocs('nurbsSquare', 'degree')
    def getDegree(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.nurbsSquare, kwargs, 'degree')
        res = int(res)
        return res

    @_f.addMelDocs('nurbsSquare', 'frozen')
    def getFrozen(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.nurbsSquare, kwargs, 'frozen')
        return res

    @_f.addMelDocs('nurbsSquare', 'nodeState')
    def getNodeState(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.nurbsSquare, kwargs, 'nodeState')
        res = int(res)
        return res

    @_f.addMelDocs('nurbsSquare', 'normal')
    def getNormal(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, modeling.nurbsSquare, kwargs, 'normal')
        return res

    @_f.addMelDocs('nurbsSquare', 'normalX')
    def getNormalX(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.nurbsSquare, kwargs, 'normalX')
        return res

    @_f.addMelDocs('nurbsSquare', 'normalY')
    def getNormalY(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.nurbsSquare, kwargs, 'normalY')
        return res

    @_f.addMelDocs('nurbsSquare', 'normalZ')
    def getNormalZ(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.nurbsSquare, kwargs, 'normalZ')
        return res

    @_f.addMelDocs('nurbsSquare', 'sideLength1')
    def getSideLength1(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.nurbsSquare, kwargs, 'sideLength1')
        return res

    @_f.addMelDocs('nurbsSquare', 'sideLength2')
    def getSideLength2(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.nurbsSquare, kwargs, 'sideLength2')
        return res

    @_f.addMelDocs('nurbsSquare', 'spansPerSide')
    def getSpansPerSide(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.nurbsSquare, kwargs, 'spansPerSide')
        res = int(res)
        return res

    @_f.addMelDocs('nurbsSquare', 'caching')
    def setCaching(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.nurbsSquare, kwargs, 'caching', val)

    @_f.addMelDocs('nurbsSquare', 'center')
    def setCenter(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, modeling.nurbsSquare, kwargs, 'center', val)

    @_f.addMelDocs('nurbsSquare', 'centerX')
    def setCenterX(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.nurbsSquare, kwargs, 'centerX', val)

    @_f.addMelDocs('nurbsSquare', 'centerY')
    def setCenterY(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.nurbsSquare, kwargs, 'centerY', val)

    @_f.addMelDocs('nurbsSquare', 'centerZ')
    def setCenterZ(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.nurbsSquare, kwargs, 'centerZ', val)

    @_f.addMelDocs('nurbsSquare', 'degree')
    def setDegree(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.nurbsSquare, kwargs, 'degree', val)

    @_f.addMelDocs('nurbsSquare', 'frozen')
    def setFrozen(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.nurbsSquare, kwargs, 'frozen', val)

    @_f.addMelDocs('nurbsSquare', 'nodeState')
    def setNodeState(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.nurbsSquare, kwargs, 'nodeState', val)

    @_f.addMelDocs('nurbsSquare', 'normal')
    def setNormal(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, modeling.nurbsSquare, kwargs, 'normal', val)

    @_f.addMelDocs('nurbsSquare', 'normalX')
    def setNormalX(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.nurbsSquare, kwargs, 'normalX', val)

    @_f.addMelDocs('nurbsSquare', 'normalY')
    def setNormalY(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.nurbsSquare, kwargs, 'normalY', val)

    @_f.addMelDocs('nurbsSquare', 'normalZ')
    def setNormalZ(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.nurbsSquare, kwargs, 'normalZ', val)

    @_f.addMelDocs('nurbsSquare', 'sideLength1')
    def setSideLength1(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.nurbsSquare, kwargs, 'sideLength1', val)

    @_f.addMelDocs('nurbsSquare', 'sideLength2')
    def setSideLength2(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.nurbsSquare, kwargs, 'sideLength2', val)

    @_f.addMelDocs('nurbsSquare', 'spansPerSide')
    def setSpansPerSide(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.nurbsSquare, kwargs, 'spansPerSide', val)


class MakeTextCurves(AbstractBaseCreate):
    __melnode__ = 'makeTextCurves'
    __slots__ = ()


class NearestPointOnCurve(AbstractBaseCreate):
    __melnode__ = 'nearestPointOnCurve'
    __slots__ = ()


class NurbsCurveToBezier(AbstractBaseCreate):
    __melcmd__ = staticmethod(modeling.nurbsCurveToBezier)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'nurbsCurveToBezier'
    __melnode__ = 'nurbsCurveToBezier'
    __slots__ = ()


class OffsetCos(AbstractBaseCreate):
    __melnode__ = 'offsetCos'
    __slots__ = ()


class OffsetCurve(AbstractBaseCreate):
    __melcmd__ = staticmethod(modeling.offsetCurve)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'offsetCurve'
    __melnode__ = 'offsetCurve'
    __slots__ = ()

    @_f.addMelDocs('offsetCurve', 'caching')
    def getCaching(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.offsetCurve, kwargs, 'caching')
        return res

    @_f.addMelDocs('offsetCurve', 'connectBreaks')
    def getConnectBreaks(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.offsetCurve, kwargs, 'connectBreaks')
        return res

    @_f.addMelDocs('offsetCurve', 'cutLoop')
    def getCutLoop(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.offsetCurve, kwargs, 'cutLoop')
        return res

    @_f.addMelDocs('offsetCurve', 'cutRadius')
    def getCutRadius(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.offsetCurve, kwargs, 'cutRadius')
        return res

    @_f.addMelDocs('offsetCurve', 'distance')
    def getDistance(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.offsetCurve, kwargs, 'distance')
        return res

    @_f.addMelDocs('offsetCurve', 'nodeState')
    def getNodeState(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.offsetCurve, kwargs, 'nodeState')
        return res

    @_f.addMelDocs('offsetCurve', 'normal')
    def getNormal(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, modeling.offsetCurve, kwargs, 'normal')
        return res

    @_f.addMelDocs('offsetCurve', 'reparameterize')
    def getReparameterize(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.offsetCurve, kwargs, 'reparameterize')
        return res

    @_f.addMelDocs('offsetCurve', 'stitch')
    def getStitch(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.offsetCurve, kwargs, 'stitch')
        return res

    @_f.addMelDocs('offsetCurve', 'subdivisionDensity')
    def getSubdivisionDensity(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.offsetCurve, kwargs, 'subdivisionDensity')
        return res

    @_f.addMelDocs('offsetCurve', 'tolerance')
    def getTolerance(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.offsetCurve, kwargs, 'tolerance')
        return res

    @_f.addMelDocs('offsetCurve', 'useGivenNormal')
    def getUseGivenNormal(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.offsetCurve, kwargs, 'useGivenNormal')
        return res

    @_f.addMelDocs('offsetCurve', 'caching')
    def setCaching(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.offsetCurve, kwargs, 'caching', val)

    @_f.addMelDocs('offsetCurve', 'connectBreaks')
    def setConnectBreaks(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.offsetCurve, kwargs, 'connectBreaks', val)

    @_f.addMelDocs('offsetCurve', 'cutLoop')
    def setCutLoop(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.offsetCurve, kwargs, 'cutLoop', val)

    @_f.addMelDocs('offsetCurve', 'cutRadius')
    def setCutRadius(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.offsetCurve, kwargs, 'cutRadius', val)

    @_f.addMelDocs('offsetCurve', 'distance')
    def setDistance(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.offsetCurve, kwargs, 'distance', val)

    @_f.addMelDocs('offsetCurve', 'nodeState')
    def setNodeState(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.offsetCurve, kwargs, 'nodeState', val)

    @_f.addMelDocs('offsetCurve', 'normal')
    def setNormal(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, modeling.offsetCurve, kwargs, 'normal', val)

    @_f.addMelDocs('offsetCurve', 'reparameterize')
    def setReparameterize(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.offsetCurve, kwargs, 'reparameterize', val)

    @_f.addMelDocs('offsetCurve', 'stitch')
    def setStitch(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.offsetCurve, kwargs, 'stitch', val)

    @_f.addMelDocs('offsetCurve', 'subdivisionDensity')
    def setSubdivisionDensity(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.offsetCurve, kwargs, 'subdivisionDensity', val)

    @_f.addMelDocs('offsetCurve', 'tolerance')
    def setTolerance(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.offsetCurve, kwargs, 'tolerance', val)

    @_f.addMelDocs('offsetCurve', 'useGivenNormal')
    def setUseGivenNormal(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.offsetCurve, kwargs, 'useGivenNormal', val)


class OffsetSurface(AbstractBaseCreate):
    __melcmd__ = staticmethod(modeling.offsetSurface)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'offsetSurface'
    __melnode__ = 'offsetSurface'
    __slots__ = ()

    @_f.addMelDocs('offsetSurface', 'caching')
    def getCaching(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.offsetSurface, kwargs, 'caching')
        return res

    @_f.addMelDocs('offsetSurface', 'distance')
    def getDistance(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.offsetSurface, kwargs, 'distance')
        return res

    @_f.addMelDocs('offsetSurface', 'method')
    def getMethod(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.offsetSurface, kwargs, 'method')
        return res

    @_f.addMelDocs('offsetSurface', 'nodeState')
    def getNodeState(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.offsetSurface, kwargs, 'nodeState')
        return res

    @_f.addMelDocs('offsetSurface', 'caching')
    def setCaching(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.offsetSurface, kwargs, 'caching', val)

    @_f.addMelDocs('offsetSurface', 'distance')
    def setDistance(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.offsetSurface, kwargs, 'distance', val)

    @_f.addMelDocs('offsetSurface', 'method')
    def setMethod(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.offsetSurface, kwargs, 'method', val)

    @_f.addMelDocs('offsetSurface', 'nodeState')
    def setNodeState(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.offsetSurface, kwargs, 'nodeState', val)


class PlanarTrimSurface(AbstractBaseCreate):
    __melnode__ = 'planarTrimSurface'
    __slots__ = ()


class PointOnCurveInfo(AbstractBaseCreate):
    __melnode__ = 'pointOnCurveInfo'
    __slots__ = ()


class PointOnSurfaceInfo(AbstractBaseCreate):
    __melnode__ = 'pointOnSurfaceInfo'
    __slots__ = ()


class PolyEdgeToCurve(AbstractBaseCreate):
    __melnode__ = 'polyEdgeToCurve'
    __slots__ = ()


class PolyProjectCurve(AbstractBaseCreate):
    __melcmd__ = staticmethod(modeling.polyProjectCurve)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'polyProjectCurve'
    __melnode__ = 'polyProjectCurve'
    __slots__ = ()

    @_f.addMelDocs('polyProjectCurve', 'caching')
    def getCaching(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.polyProjectCurve, kwargs, 'caching')
        return res

    @_f.addMelDocs('polyProjectCurve', 'direction')
    def getDirection(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, modeling.polyProjectCurve, kwargs, 'direction')
        return res

    @_f.addMelDocs('polyProjectCurve', 'directionX')
    def getDirectionX(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.polyProjectCurve, kwargs, 'directionX')
        return res

    @_f.addMelDocs('polyProjectCurve', 'directionY')
    def getDirectionY(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.polyProjectCurve, kwargs, 'directionY')
        return res

    @_f.addMelDocs('polyProjectCurve', 'directionZ')
    def getDirectionZ(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.polyProjectCurve, kwargs, 'directionZ')
        return res

    @_f.addMelDocs('polyProjectCurve', 'nodeState')
    def getNodeState(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.polyProjectCurve, kwargs, 'nodeState')
        return res

    @_f.addMelDocs('polyProjectCurve', 'tolerance')
    def getTolerance(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.polyProjectCurve, kwargs, 'tolerance')
        return res

    @_f.addMelDocs('polyProjectCurve', 'caching')
    def setCaching(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.polyProjectCurve, kwargs, 'caching', val)

    @_f.addMelDocs('polyProjectCurve', 'direction')
    def setDirection(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, modeling.polyProjectCurve, kwargs, 'direction', val)

    @_f.addMelDocs('polyProjectCurve', 'directionX')
    def setDirectionX(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.polyProjectCurve, kwargs, 'directionX', val)

    @_f.addMelDocs('polyProjectCurve', 'directionY')
    def setDirectionY(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.polyProjectCurve, kwargs, 'directionY', val)

    @_f.addMelDocs('polyProjectCurve', 'directionZ')
    def setDirectionZ(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.polyProjectCurve, kwargs, 'directionZ', val)

    @_f.addMelDocs('polyProjectCurve', 'nodeState')
    def setNodeState(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.polyProjectCurve, kwargs, 'nodeState', val)

    @_f.addMelDocs('polyProjectCurve', 'tolerance')
    def setTolerance(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.polyProjectCurve, kwargs, 'tolerance', val)


class Primitive(AbstractBaseCreate):
    __melnode__ = 'primitive'
    __slots__ = ()


class MakeNurbCube(Primitive):
    __melcmd__ = staticmethod(modeling.nurbsCube)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'nurbsCube'
    __melnode__ = 'makeNurbCube'
    __slots__ = ()

    @_f.addMelDocs('nurbsCube', 'axis')
    def getAxis(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, modeling.nurbsCube, kwargs, 'axis')
        return res

    @_f.addMelDocs('nurbsCube', 'caching')
    def getCaching(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.nurbsCube, kwargs, 'caching')
        res = bool(res)
        return res

    @_f.addMelDocs('nurbsCube', 'degree')
    def getDegree(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.nurbsCube, kwargs, 'degree')
        res = int(res)
        return res

    @_f.addMelDocs('nurbsCube', 'frozen')
    def getFrozen(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.nurbsCube, kwargs, 'frozen')
        return res

    @_f.addMelDocs('nurbsCube', 'heightRatio')
    def getHeightRatio(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.nurbsCube, kwargs, 'heightRatio')
        return res

    @_f.addMelDocs('nurbsCube', 'lengthRatio')
    def getLengthRatio(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.nurbsCube, kwargs, 'lengthRatio')
        return res

    @_f.addMelDocs('nurbsCube', 'nodeState')
    def getNodeState(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.nurbsCube, kwargs, 'nodeState')
        res = int(res)
        return res

    @_f.addMelDocs('nurbsCube', 'patchesU')
    def getPatchesU(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.nurbsCube, kwargs, 'patchesU')
        res = int(res)
        return res

    @_f.addMelDocs('nurbsCube', 'patchesV')
    def getPatchesV(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.nurbsCube, kwargs, 'patchesV')
        res = int(res)
        return res

    @_f.addMelDocs('nurbsCube', 'pivot')
    def getPivot(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, modeling.nurbsCube, kwargs, 'pivot')
        return res

    @_f.addMelDocs('nurbsCube', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.nurbsCube, kwargs, 'width')
        return res

    @_f.addMelDocs('nurbsCube', 'axis')
    def setAxis(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, modeling.nurbsCube, kwargs, 'axis', val)

    @_f.addMelDocs('nurbsCube', 'caching')
    def setCaching(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.nurbsCube, kwargs, 'caching', val)

    @_f.addMelDocs('nurbsCube', 'degree')
    def setDegree(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.nurbsCube, kwargs, 'degree', val)

    @_f.addMelDocs('nurbsCube', 'frozen')
    def setFrozen(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.nurbsCube, kwargs, 'frozen', val)

    @_f.addMelDocs('nurbsCube', 'heightRatio')
    def setHeightRatio(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.nurbsCube, kwargs, 'heightRatio', val)

    @_f.addMelDocs('nurbsCube', 'lengthRatio')
    def setLengthRatio(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.nurbsCube, kwargs, 'lengthRatio', val)

    @_f.addMelDocs('nurbsCube', 'nodeState')
    def setNodeState(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.nurbsCube, kwargs, 'nodeState', val)

    @_f.addMelDocs('nurbsCube', 'patchesU')
    def setPatchesU(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.nurbsCube, kwargs, 'patchesU', val)

    @_f.addMelDocs('nurbsCube', 'patchesV')
    def setPatchesV(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.nurbsCube, kwargs, 'patchesV', val)

    @_f.addMelDocs('nurbsCube', 'pivot')
    def setPivot(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, modeling.nurbsCube, kwargs, 'pivot', val)

    @_f.addMelDocs('nurbsCube', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.nurbsCube, kwargs, 'width', val)


class MakeNurbPlane(Primitive):
    __melcmd__ = staticmethod(modeling.nurbsPlane)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'nurbsPlane'
    __melnode__ = 'makeNurbPlane'
    __slots__ = ()

    @_f.addMelDocs('nurbsPlane', 'axis')
    def getAxis(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, modeling.nurbsPlane, kwargs, 'axis')
        return res

    @_f.addMelDocs('nurbsPlane', 'caching')
    def getCaching(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.nurbsPlane, kwargs, 'caching')
        res = bool(res)
        return res

    @_f.addMelDocs('nurbsPlane', 'degree')
    def getDegree(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.nurbsPlane, kwargs, 'degree')
        res = int(res)
        return res

    @_f.addMelDocs('nurbsPlane', 'frozen')
    def getFrozen(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.nurbsPlane, kwargs, 'frozen')
        return res

    @_f.addMelDocs('nurbsPlane', 'lengthRatio')
    def getLengthRatio(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.nurbsPlane, kwargs, 'lengthRatio')
        return res

    @_f.addMelDocs('nurbsPlane', 'nodeState')
    def getNodeState(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.nurbsPlane, kwargs, 'nodeState')
        res = int(res)
        return res

    @_f.addMelDocs('nurbsPlane', 'patchesU')
    def getPatchesU(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.nurbsPlane, kwargs, 'patchesU')
        res = int(res)
        return res

    @_f.addMelDocs('nurbsPlane', 'patchesV')
    def getPatchesV(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.nurbsPlane, kwargs, 'patchesV')
        res = int(res)
        return res

    @_f.addMelDocs('nurbsPlane', 'pivot')
    def getPivot(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, modeling.nurbsPlane, kwargs, 'pivot')
        return res

    @_f.addMelDocs('nurbsPlane', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.nurbsPlane, kwargs, 'width')
        return res

    @_f.addMelDocs('nurbsPlane', 'axis')
    def setAxis(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, modeling.nurbsPlane, kwargs, 'axis', val)

    @_f.addMelDocs('nurbsPlane', 'caching')
    def setCaching(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.nurbsPlane, kwargs, 'caching', val)

    @_f.addMelDocs('nurbsPlane', 'degree')
    def setDegree(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.nurbsPlane, kwargs, 'degree', val)

    @_f.addMelDocs('nurbsPlane', 'frozen')
    def setFrozen(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.nurbsPlane, kwargs, 'frozen', val)

    @_f.addMelDocs('nurbsPlane', 'lengthRatio')
    def setLengthRatio(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.nurbsPlane, kwargs, 'lengthRatio', val)

    @_f.addMelDocs('nurbsPlane', 'nodeState')
    def setNodeState(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.nurbsPlane, kwargs, 'nodeState', val)

    @_f.addMelDocs('nurbsPlane', 'patchesU')
    def setPatchesU(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.nurbsPlane, kwargs, 'patchesU', val)

    @_f.addMelDocs('nurbsPlane', 'patchesV')
    def setPatchesV(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.nurbsPlane, kwargs, 'patchesV', val)

    @_f.addMelDocs('nurbsPlane', 'pivot')
    def setPivot(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, modeling.nurbsPlane, kwargs, 'pivot', val)

    @_f.addMelDocs('nurbsPlane', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.nurbsPlane, kwargs, 'width', val)


class RevolvedPrimitive(Primitive):
    __melnode__ = 'revolvedPrimitive'
    __slots__ = ()


class MakeNurbCone(RevolvedPrimitive):
    __melcmd__ = staticmethod(modeling.cone)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'cone'
    __melnode__ = 'makeNurbCone'
    __slots__ = ()

    @_f.addMelDocs('cone', 'axis')
    def getAxis(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, modeling.cone, kwargs, 'axis')
        return res

    @_f.addMelDocs('cone', 'caching')
    def getCaching(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.cone, kwargs, 'caching')
        res = bool(res)
        return res

    @_f.addMelDocs('cone', 'degree')
    def getDegree(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.cone, kwargs, 'degree')
        res = int(res)
        return res

    @_f.addMelDocs('cone', 'endSweep')
    def getEndSweep(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.cone, kwargs, 'endSweep')
        return res

    @_f.addMelDocs('cone', 'frozen')
    def getFrozen(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.cone, kwargs, 'frozen')
        return res

    @_f.addMelDocs('cone', 'heightRatio')
    def getHeightRatio(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.cone, kwargs, 'heightRatio')
        return res

    @_f.addMelDocs('cone', 'nodeState')
    def getNodeState(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.cone, kwargs, 'nodeState')
        res = int(res)
        return res

    @_f.addMelDocs('cone', 'pivot')
    def getPivot(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, modeling.cone, kwargs, 'pivot')
        return res

    @_f.addMelDocs('cone', 'radius')
    def getRadius(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.cone, kwargs, 'radius')
        return res

    @_f.addMelDocs('cone', 'sections')
    def getSections(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.cone, kwargs, 'sections')
        res = int(res)
        return res

    @_f.addMelDocs('cone', 'spans')
    def getSpans(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.cone, kwargs, 'spans')
        res = int(res)
        return res

    @_f.addMelDocs('cone', 'startSweep')
    def getStartSweep(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.cone, kwargs, 'startSweep')
        return res

    @_f.addMelDocs('cone', 'tolerance')
    def getTolerance(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.cone, kwargs, 'tolerance')
        return res

    @_f.addMelDocs('cone', 'useOldInitBehaviour')
    def getUseOldInitBehaviour(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.cone, kwargs, 'useOldInitBehaviour')
        res = bool(res)
        return res

    @_f.addMelDocs('cone', 'useTolerance')
    def getUseTolerance(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.cone, kwargs, 'useTolerance')
        res = bool(res)
        return res

    @_f.addMelDocs('cone', 'axis')
    def setAxis(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, modeling.cone, kwargs, 'axis', val)

    @_f.addMelDocs('cone', 'caching')
    def setCaching(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.cone, kwargs, 'caching', val)

    @_f.addMelDocs('cone', 'degree')
    def setDegree(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.cone, kwargs, 'degree', val)

    @_f.addMelDocs('cone', 'endSweep')
    def setEndSweep(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.cone, kwargs, 'endSweep', val)

    @_f.addMelDocs('cone', 'frozen')
    def setFrozen(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.cone, kwargs, 'frozen', val)

    @_f.addMelDocs('cone', 'heightRatio')
    def setHeightRatio(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.cone, kwargs, 'heightRatio', val)

    @_f.addMelDocs('cone', 'nodeState')
    def setNodeState(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.cone, kwargs, 'nodeState', val)

    @_f.addMelDocs('cone', 'pivot')
    def setPivot(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, modeling.cone, kwargs, 'pivot', val)

    @_f.addMelDocs('cone', 'radius')
    def setRadius(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.cone, kwargs, 'radius', val)

    @_f.addMelDocs('cone', 'sections')
    def setSections(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.cone, kwargs, 'sections', val)

    @_f.addMelDocs('cone', 'spans')
    def setSpans(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.cone, kwargs, 'spans', val)

    @_f.addMelDocs('cone', 'startSweep')
    def setStartSweep(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.cone, kwargs, 'startSweep', val)

    @_f.addMelDocs('cone', 'tolerance')
    def setTolerance(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.cone, kwargs, 'tolerance', val)

    @_f.addMelDocs('cone', 'useOldInitBehaviour')
    def setUseOldInitBehaviour(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.cone, kwargs, 'useOldInitBehaviour', val)

    @_f.addMelDocs('cone', 'useTolerance')
    def setUseTolerance(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.cone, kwargs, 'useTolerance', val)


class MakeNurbCylinder(RevolvedPrimitive):
    __melcmd__ = staticmethod(modeling.cylinder)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'cylinder'
    __melnode__ = 'makeNurbCylinder'
    __slots__ = ()

    @_f.addMelDocs('cylinder', 'axis')
    def getAxis(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, modeling.cylinder, kwargs, 'axis')
        return res

    @_f.addMelDocs('cylinder', 'caching')
    def getCaching(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.cylinder, kwargs, 'caching')
        res = bool(res)
        return res

    @_f.addMelDocs('cylinder', 'degree')
    def getDegree(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.cylinder, kwargs, 'degree')
        res = int(res)
        return res

    @_f.addMelDocs('cylinder', 'endSweep')
    def getEndSweep(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.cylinder, kwargs, 'endSweep')
        return res

    @_f.addMelDocs('cylinder', 'frozen')
    def getFrozen(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.cylinder, kwargs, 'frozen')
        return res

    @_f.addMelDocs('cylinder', 'heightRatio')
    def getHeightRatio(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.cylinder, kwargs, 'heightRatio')
        return res

    @_f.addMelDocs('cylinder', 'nodeState')
    def getNodeState(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.cylinder, kwargs, 'nodeState')
        res = int(res)
        return res

    @_f.addMelDocs('cylinder', 'pivot')
    def getPivot(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, modeling.cylinder, kwargs, 'pivot')
        return res

    @_f.addMelDocs('cylinder', 'radius')
    def getRadius(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.cylinder, kwargs, 'radius')
        return res

    @_f.addMelDocs('cylinder', 'sections')
    def getSections(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.cylinder, kwargs, 'sections')
        res = int(res)
        return res

    @_f.addMelDocs('cylinder', 'spans')
    def getSpans(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.cylinder, kwargs, 'spans')
        res = int(res)
        return res

    @_f.addMelDocs('cylinder', 'startSweep')
    def getStartSweep(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.cylinder, kwargs, 'startSweep')
        return res

    @_f.addMelDocs('cylinder', 'tolerance')
    def getTolerance(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.cylinder, kwargs, 'tolerance')
        return res

    @_f.addMelDocs('cylinder', 'useTolerance')
    def getUseTolerance(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.cylinder, kwargs, 'useTolerance')
        res = bool(res)
        return res

    @_f.addMelDocs('cylinder', 'axis')
    def setAxis(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, modeling.cylinder, kwargs, 'axis', val)

    @_f.addMelDocs('cylinder', 'caching')
    def setCaching(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.cylinder, kwargs, 'caching', val)

    @_f.addMelDocs('cylinder', 'degree')
    def setDegree(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.cylinder, kwargs, 'degree', val)

    @_f.addMelDocs('cylinder', 'endSweep')
    def setEndSweep(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.cylinder, kwargs, 'endSweep', val)

    @_f.addMelDocs('cylinder', 'frozen')
    def setFrozen(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.cylinder, kwargs, 'frozen', val)

    @_f.addMelDocs('cylinder', 'heightRatio')
    def setHeightRatio(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.cylinder, kwargs, 'heightRatio', val)

    @_f.addMelDocs('cylinder', 'nodeState')
    def setNodeState(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.cylinder, kwargs, 'nodeState', val)

    @_f.addMelDocs('cylinder', 'pivot')
    def setPivot(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, modeling.cylinder, kwargs, 'pivot', val)

    @_f.addMelDocs('cylinder', 'radius')
    def setRadius(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.cylinder, kwargs, 'radius', val)

    @_f.addMelDocs('cylinder', 'sections')
    def setSections(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.cylinder, kwargs, 'sections', val)

    @_f.addMelDocs('cylinder', 'spans')
    def setSpans(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.cylinder, kwargs, 'spans', val)

    @_f.addMelDocs('cylinder', 'startSweep')
    def setStartSweep(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.cylinder, kwargs, 'startSweep', val)

    @_f.addMelDocs('cylinder', 'tolerance')
    def setTolerance(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.cylinder, kwargs, 'tolerance', val)

    @_f.addMelDocs('cylinder', 'useTolerance')
    def setUseTolerance(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.cylinder, kwargs, 'useTolerance', val)


class MakeNurbSphere(RevolvedPrimitive):
    __melcmd__ = staticmethod(modeling.sphere)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'sphere'
    __melnode__ = 'makeNurbSphere'
    __slots__ = ()

    @_f.addMelDocs('sphere', 'axis')
    def getAxis(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, modeling.sphere, kwargs, 'axis')
        return res

    @_f.addMelDocs('sphere', 'caching')
    def getCaching(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.sphere, kwargs, 'caching')
        res = bool(res)
        return res

    @_f.addMelDocs('sphere', 'degree')
    def getDegree(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.sphere, kwargs, 'degree')
        res = int(res)
        return res

    @_f.addMelDocs('sphere', 'endSweep')
    def getEndSweep(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.sphere, kwargs, 'endSweep')
        return res

    @_f.addMelDocs('sphere', 'frozen')
    def getFrozen(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.sphere, kwargs, 'frozen')
        return res

    @_f.addMelDocs('sphere', 'heightRatio')
    def getHeightRatio(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.sphere, kwargs, 'heightRatio')
        return res

    @_f.addMelDocs('sphere', 'nodeState')
    def getNodeState(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.sphere, kwargs, 'nodeState')
        res = int(res)
        return res

    @_f.addMelDocs('sphere', 'pivot')
    def getPivot(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, modeling.sphere, kwargs, 'pivot')
        return res

    @_f.addMelDocs('sphere', 'radius')
    def getRadius(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.sphere, kwargs, 'radius')
        return res

    @_f.addMelDocs('sphere', 'sections')
    def getSections(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.sphere, kwargs, 'sections')
        res = int(res)
        return res

    @_f.addMelDocs('sphere', 'spans')
    def getSpans(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.sphere, kwargs, 'spans')
        res = int(res)
        return res

    @_f.addMelDocs('sphere', 'startSweep')
    def getStartSweep(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.sphere, kwargs, 'startSweep')
        return res

    @_f.addMelDocs('sphere', 'tolerance')
    def getTolerance(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.sphere, kwargs, 'tolerance')
        return res

    @_f.addMelDocs('sphere', 'useTolerance')
    def getUseTolerance(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.sphere, kwargs, 'useTolerance')
        res = bool(res)
        return res

    @_f.addMelDocs('sphere', 'axis')
    def setAxis(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, modeling.sphere, kwargs, 'axis', val)

    @_f.addMelDocs('sphere', 'caching')
    def setCaching(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.sphere, kwargs, 'caching', val)

    @_f.addMelDocs('sphere', 'degree')
    def setDegree(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.sphere, kwargs, 'degree', val)

    @_f.addMelDocs('sphere', 'endSweep')
    def setEndSweep(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.sphere, kwargs, 'endSweep', val)

    @_f.addMelDocs('sphere', 'frozen')
    def setFrozen(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.sphere, kwargs, 'frozen', val)

    @_f.addMelDocs('sphere', 'heightRatio')
    def setHeightRatio(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.sphere, kwargs, 'heightRatio', val)

    @_f.addMelDocs('sphere', 'nodeState')
    def setNodeState(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.sphere, kwargs, 'nodeState', val)

    @_f.addMelDocs('sphere', 'pivot')
    def setPivot(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, modeling.sphere, kwargs, 'pivot', val)

    @_f.addMelDocs('sphere', 'radius')
    def setRadius(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.sphere, kwargs, 'radius', val)

    @_f.addMelDocs('sphere', 'sections')
    def setSections(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.sphere, kwargs, 'sections', val)

    @_f.addMelDocs('sphere', 'spans')
    def setSpans(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.sphere, kwargs, 'spans', val)

    @_f.addMelDocs('sphere', 'startSweep')
    def setStartSweep(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.sphere, kwargs, 'startSweep', val)

    @_f.addMelDocs('sphere', 'tolerance')
    def setTolerance(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.sphere, kwargs, 'tolerance', val)

    @_f.addMelDocs('sphere', 'useTolerance')
    def setUseTolerance(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.sphere, kwargs, 'useTolerance', val)


class MakeNurbTorus(RevolvedPrimitive):
    __melcmd__ = staticmethod(modeling.torus)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'torus'
    __melnode__ = 'makeNurbTorus'
    __slots__ = ()

    @_f.addMelDocs('torus', 'axis')
    def getAxis(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, modeling.torus, kwargs, 'axis')
        return res

    @_f.addMelDocs('torus', 'caching')
    def getCaching(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.torus, kwargs, 'caching')
        res = bool(res)
        return res

    @_f.addMelDocs('torus', 'degree')
    def getDegree(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.torus, kwargs, 'degree')
        res = int(res)
        return res

    @_f.addMelDocs('torus', 'endSweep')
    def getEndSweep(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.torus, kwargs, 'endSweep')
        return res

    @_f.addMelDocs('torus', 'frozen')
    def getFrozen(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.torus, kwargs, 'frozen')
        return res

    @_f.addMelDocs('torus', 'heightRatio')
    def getHeightRatio(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.torus, kwargs, 'heightRatio')
        return res

    @_f.addMelDocs('torus', 'minorSweep')
    def getMinorSweep(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.torus, kwargs, 'minorSweep')
        return res

    @_f.addMelDocs('torus', 'nodeState')
    def getNodeState(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.torus, kwargs, 'nodeState')
        res = int(res)
        return res

    @_f.addMelDocs('torus', 'pivot')
    def getPivot(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, modeling.torus, kwargs, 'pivot')
        return res

    @_f.addMelDocs('torus', 'radius')
    def getRadius(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.torus, kwargs, 'radius')
        return res

    @_f.addMelDocs('torus', 'sections')
    def getSections(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.torus, kwargs, 'sections')
        res = int(res)
        return res

    @_f.addMelDocs('torus', 'spans')
    def getSpans(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.torus, kwargs, 'spans')
        res = int(res)
        return res

    @_f.addMelDocs('torus', 'startSweep')
    def getStartSweep(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.torus, kwargs, 'startSweep')
        return res

    @_f.addMelDocs('torus', 'tolerance')
    def getTolerance(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.torus, kwargs, 'tolerance')
        return res

    @_f.addMelDocs('torus', 'useTolerance')
    def getUseTolerance(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.torus, kwargs, 'useTolerance')
        res = bool(res)
        return res

    @_f.addMelDocs('torus', 'axis')
    def setAxis(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, modeling.torus, kwargs, 'axis', val)

    @_f.addMelDocs('torus', 'caching')
    def setCaching(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.torus, kwargs, 'caching', val)

    @_f.addMelDocs('torus', 'degree')
    def setDegree(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.torus, kwargs, 'degree', val)

    @_f.addMelDocs('torus', 'endSweep')
    def setEndSweep(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.torus, kwargs, 'endSweep', val)

    @_f.addMelDocs('torus', 'frozen')
    def setFrozen(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.torus, kwargs, 'frozen', val)

    @_f.addMelDocs('torus', 'heightRatio')
    def setHeightRatio(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.torus, kwargs, 'heightRatio', val)

    @_f.addMelDocs('torus', 'minorSweep')
    def setMinorSweep(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.torus, kwargs, 'minorSweep', val)

    @_f.addMelDocs('torus', 'nodeState')
    def setNodeState(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.torus, kwargs, 'nodeState', val)

    @_f.addMelDocs('torus', 'pivot')
    def setPivot(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, modeling.torus, kwargs, 'pivot', val)

    @_f.addMelDocs('torus', 'radius')
    def setRadius(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.torus, kwargs, 'radius', val)

    @_f.addMelDocs('torus', 'sections')
    def setSections(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.torus, kwargs, 'sections', val)

    @_f.addMelDocs('torus', 'spans')
    def setSpans(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.torus, kwargs, 'spans', val)

    @_f.addMelDocs('torus', 'startSweep')
    def setStartSweep(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.torus, kwargs, 'startSweep', val)

    @_f.addMelDocs('torus', 'tolerance')
    def setTolerance(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.torus, kwargs, 'tolerance', val)

    @_f.addMelDocs('torus', 'useTolerance')
    def setUseTolerance(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.torus, kwargs, 'useTolerance', val)


class ProjectCurve(AbstractBaseCreate):
    __melcmd__ = staticmethod(modeling.projectCurve)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'projectCurve'
    __melnode__ = 'projectCurve'
    __slots__ = ()

    @_f.addMelDocs('projectCurve', 'caching')
    def getCaching(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.projectCurve, kwargs, 'caching')
        return res

    @_f.addMelDocs('projectCurve', 'direction')
    def getDirection(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, modeling.projectCurve, kwargs, 'direction')
        return res

    @_f.addMelDocs('projectCurve', 'directionX')
    def getDirectionX(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.projectCurve, kwargs, 'directionX')
        return res

    @_f.addMelDocs('projectCurve', 'directionY')
    def getDirectionY(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.projectCurve, kwargs, 'directionY')
        return res

    @_f.addMelDocs('projectCurve', 'directionZ')
    def getDirectionZ(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.projectCurve, kwargs, 'directionZ')
        return res

    @_f.addMelDocs('projectCurve', 'nodeState')
    def getNodeState(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.projectCurve, kwargs, 'nodeState')
        return res

    @_f.addMelDocs('projectCurve', 'tolerance')
    def getTolerance(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.projectCurve, kwargs, 'tolerance')
        return res

    @_f.addMelDocs('projectCurve', 'useNormal')
    def getUseNormal(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.projectCurve, kwargs, 'useNormal')
        return res

    @_f.addMelDocs('projectCurve', 'caching')
    def setCaching(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.projectCurve, kwargs, 'caching', val)

    @_f.addMelDocs('projectCurve', 'direction')
    def setDirection(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, modeling.projectCurve, kwargs, 'direction', val)

    @_f.addMelDocs('projectCurve', 'directionX')
    def setDirectionX(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.projectCurve, kwargs, 'directionX', val)

    @_f.addMelDocs('projectCurve', 'directionY')
    def setDirectionY(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.projectCurve, kwargs, 'directionY', val)

    @_f.addMelDocs('projectCurve', 'directionZ')
    def setDirectionZ(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.projectCurve, kwargs, 'directionZ', val)

    @_f.addMelDocs('projectCurve', 'nodeState')
    def setNodeState(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.projectCurve, kwargs, 'nodeState', val)

    @_f.addMelDocs('projectCurve', 'tolerance')
    def setTolerance(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.projectCurve, kwargs, 'tolerance', val)

    @_f.addMelDocs('projectCurve', 'useNormal')
    def setUseNormal(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.projectCurve, kwargs, 'useNormal', val)


class ProjectTangent(AbstractBaseCreate):
    __melcmd__ = staticmethod(modeling.projectTangent)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'projectTangent'
    __melnode__ = 'projectTangent'
    __slots__ = ()

    @_f.addMelDocs('projectTangent', 'caching')
    def getCaching(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.projectTangent, kwargs, 'caching')
        return res

    @_f.addMelDocs('projectTangent', 'curvature')
    def getCurvature(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.projectTangent, kwargs, 'curvature')
        return res

    @_f.addMelDocs('projectTangent', 'curvatureScale')
    def getCurvatureScale(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.projectTangent, kwargs, 'curvatureScale')
        return res

    @_f.addMelDocs('projectTangent', 'ignoreEdges')
    def getIgnoreEdges(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.projectTangent, kwargs, 'ignoreEdges')
        return res

    @_f.addMelDocs('projectTangent', 'nodeState')
    def getNodeState(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.projectTangent, kwargs, 'nodeState')
        return res

    @_f.addMelDocs('projectTangent', 'reverseTangent')
    def getReverseTangent(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.projectTangent, kwargs, 'reverseTangent')
        return res

    @_f.addMelDocs('projectTangent', 'rotate')
    def getRotate(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.projectTangent, kwargs, 'rotate')
        return res

    @_f.addMelDocs('projectTangent', 'tangentDirection')
    def getTangentDirection(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.projectTangent, kwargs, 'tangentDirection')
        return res

    @_f.addMelDocs('projectTangent', 'tangentScale')
    def getTangentScale(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.projectTangent, kwargs, 'tangentScale')
        return res

    @_f.addMelDocs('projectTangent', 'caching')
    def setCaching(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.projectTangent, kwargs, 'caching', val)

    @_f.addMelDocs('projectTangent', 'curvature')
    def setCurvature(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.projectTangent, kwargs, 'curvature', val)

    @_f.addMelDocs('projectTangent', 'curvatureScale')
    def setCurvatureScale(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.projectTangent, kwargs, 'curvatureScale', val)

    @_f.addMelDocs('projectTangent', 'ignoreEdges')
    def setIgnoreEdges(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.projectTangent, kwargs, 'ignoreEdges', val)

    @_f.addMelDocs('projectTangent', 'nodeState')
    def setNodeState(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.projectTangent, kwargs, 'nodeState', val)

    @_f.addMelDocs('projectTangent', 'reverseTangent')
    def setReverseTangent(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.projectTangent, kwargs, 'reverseTangent', val)

    @_f.addMelDocs('projectTangent', 'rotate')
    def setRotate(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.projectTangent, kwargs, 'rotate', val)

    @_f.addMelDocs('projectTangent', 'tangentDirection')
    def setTangentDirection(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.projectTangent, kwargs, 'tangentDirection', val)

    @_f.addMelDocs('projectTangent', 'tangentScale')
    def setTangentScale(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.projectTangent, kwargs, 'tangentScale', val)


class RbfSrf(AbstractBaseCreate):
    __melnode__ = 'rbfSrf'
    __slots__ = ()


class RebuildCurve(AbstractBaseCreate):
    __melcmd__ = staticmethod(modeling.rebuildCurve)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'rebuildCurve'
    __melnode__ = 'rebuildCurve'
    __slots__ = ()

    @_f.addMelDocs('rebuildCurve', 'caching')
    def getCaching(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.rebuildCurve, kwargs, 'caching')
        return res

    @_f.addMelDocs('rebuildCurve', 'degree')
    def getDegree(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.rebuildCurve, kwargs, 'degree')
        return res

    @_f.addMelDocs('rebuildCurve', 'endKnots')
    def getEndKnots(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.rebuildCurve, kwargs, 'endKnots')
        return res

    @_f.addMelDocs('rebuildCurve', 'fitRebuild')
    def getFitRebuild(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.rebuildCurve, kwargs, 'fitRebuild')
        return res

    @_f.addMelDocs('rebuildCurve', 'keepControlPoints')
    def getKeepControlPoints(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.rebuildCurve, kwargs, 'keepControlPoints')
        return res

    @_f.addMelDocs('rebuildCurve', 'keepEndPoints')
    def getKeepEndPoints(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.rebuildCurve, kwargs, 'keepEndPoints')
        return res

    @_f.addMelDocs('rebuildCurve', 'keepRange')
    def getKeepRange(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.rebuildCurve, kwargs, 'keepRange')
        return res

    @_f.addMelDocs('rebuildCurve', 'keepTangents')
    def getKeepTangents(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.rebuildCurve, kwargs, 'keepTangents')
        return res

    @_f.addMelDocs('rebuildCurve', 'nodeState')
    def getNodeState(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.rebuildCurve, kwargs, 'nodeState')
        return res

    @_f.addMelDocs('rebuildCurve', 'rebuildType')
    def getRebuildType(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.rebuildCurve, kwargs, 'rebuildType')
        return res

    @_f.addMelDocs('rebuildCurve', 'smartSurfaceCurveRebuild')
    def getSmartSurfaceCurveRebuild(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.rebuildCurve, kwargs, 'smartSurfaceCurveRebuild')
        return res

    @_f.addMelDocs('rebuildCurve', 'spans')
    def getSpans(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.rebuildCurve, kwargs, 'spans')
        return res

    @_f.addMelDocs('rebuildCurve', 'tolerance')
    def getTolerance(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.rebuildCurve, kwargs, 'tolerance')
        return res

    @_f.addMelDocs('rebuildCurve', 'caching')
    def setCaching(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.rebuildCurve, kwargs, 'caching', val)

    @_f.addMelDocs('rebuildCurve', 'degree')
    def setDegree(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.rebuildCurve, kwargs, 'degree', val)

    @_f.addMelDocs('rebuildCurve', 'endKnots')
    def setEndKnots(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.rebuildCurve, kwargs, 'endKnots', val)

    @_f.addMelDocs('rebuildCurve', 'fitRebuild')
    def setFitRebuild(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.rebuildCurve, kwargs, 'fitRebuild', val)

    @_f.addMelDocs('rebuildCurve', 'keepControlPoints')
    def setKeepControlPoints(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.rebuildCurve, kwargs, 'keepControlPoints', val)

    @_f.addMelDocs('rebuildCurve', 'keepEndPoints')
    def setKeepEndPoints(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.rebuildCurve, kwargs, 'keepEndPoints', val)

    @_f.addMelDocs('rebuildCurve', 'keepRange')
    def setKeepRange(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.rebuildCurve, kwargs, 'keepRange', val)

    @_f.addMelDocs('rebuildCurve', 'keepTangents')
    def setKeepTangents(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.rebuildCurve, kwargs, 'keepTangents', val)

    @_f.addMelDocs('rebuildCurve', 'nodeState')
    def setNodeState(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.rebuildCurve, kwargs, 'nodeState', val)

    @_f.addMelDocs('rebuildCurve', 'rebuildType')
    def setRebuildType(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.rebuildCurve, kwargs, 'rebuildType', val)

    @_f.addMelDocs('rebuildCurve', 'smartSurfaceCurveRebuild')
    def setSmartSurfaceCurveRebuild(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.rebuildCurve, kwargs, 'smartSurfaceCurveRebuild', val)

    @_f.addMelDocs('rebuildCurve', 'spans')
    def setSpans(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.rebuildCurve, kwargs, 'spans', val)

    @_f.addMelDocs('rebuildCurve', 'tolerance')
    def setTolerance(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.rebuildCurve, kwargs, 'tolerance', val)


class ReverseCurve(AbstractBaseCreate):
    __melcmd__ = staticmethod(modeling.reverseCurve)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'reverseCurve'
    __melnode__ = 'reverseCurve'
    __slots__ = ()

    @_f.addMelDocs('reverseCurve', 'caching')
    def getCaching(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.reverseCurve, kwargs, 'caching')
        return res

    @_f.addMelDocs('reverseCurve', 'nodeState')
    def getNodeState(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.reverseCurve, kwargs, 'nodeState')
        return res

    @_f.addMelDocs('reverseCurve', 'caching')
    def setCaching(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.reverseCurve, kwargs, 'caching', val)

    @_f.addMelDocs('reverseCurve', 'nodeState')
    def setNodeState(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.reverseCurve, kwargs, 'nodeState', val)


class ReverseSurface(AbstractBaseCreate):
    __melcmd__ = staticmethod(modeling.reverseSurface)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'reverseSurface'
    __melnode__ = 'reverseSurface'
    __slots__ = ()

    @_f.addMelDocs('reverseSurface', 'caching')
    def getCaching(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.reverseSurface, kwargs, 'caching')
        return res

    @_f.addMelDocs('reverseSurface', 'direction')
    def getDirection(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.reverseSurface, kwargs, 'direction')
        return res

    @_f.addMelDocs('reverseSurface', 'nodeState')
    def getNodeState(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.reverseSurface, kwargs, 'nodeState')
        return res

    @_f.addMelDocs('reverseSurface', 'caching')
    def setCaching(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.reverseSurface, kwargs, 'caching', val)

    @_f.addMelDocs('reverseSurface', 'direction')
    def setDirection(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.reverseSurface, kwargs, 'direction', val)

    @_f.addMelDocs('reverseSurface', 'nodeState')
    def setNodeState(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.reverseSurface, kwargs, 'nodeState', val)


class Revolve(AbstractBaseCreate):
    __melcmd__ = staticmethod(modeling.revolve)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'revolve'
    __melnode__ = 'revolve'
    __slots__ = ()

    @_f.addMelDocs('revolve', 'autoCorrectNormal')
    def getAutoCorrectNormal(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.revolve, kwargs, 'autoCorrectNormal')
        return res

    @_f.addMelDocs('revolve', 'axis')
    def getAxis(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, modeling.revolve, kwargs, 'axis')
        return res

    @_f.addMelDocs('revolve', 'axisChoice')
    def getAxisChoice(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.revolve, kwargs, 'axisChoice')
        return res

    @_f.addMelDocs('revolve', 'axisX')
    def getAxisX(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.revolve, kwargs, 'axisX')
        return res

    @_f.addMelDocs('revolve', 'axisY')
    def getAxisY(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.revolve, kwargs, 'axisY')
        return res

    @_f.addMelDocs('revolve', 'axisZ')
    def getAxisZ(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.revolve, kwargs, 'axisZ')
        return res

    @_f.addMelDocs('revolve', 'bridge')
    def getBridge(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.revolve, kwargs, 'bridge')
        return res

    @_f.addMelDocs('revolve', 'caching')
    def getCaching(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.revolve, kwargs, 'caching')
        return res

    @_f.addMelDocs('revolve', 'computePivotAndAxis')
    def getComputePivotAndAxis(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.revolve, kwargs, 'computePivotAndAxis')
        return res

    @_f.addMelDocs('revolve', 'degree')
    def getDegree(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.revolve, kwargs, 'degree')
        return res

    @_f.addMelDocs('revolve', 'endSweep')
    def getEndSweep(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.revolve, kwargs, 'endSweep')
        return res

    @_f.addMelDocs('revolve', 'nodeState')
    def getNodeState(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.revolve, kwargs, 'nodeState')
        return res

    @_f.addMelDocs('revolve', 'pivot')
    def getPivot(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, modeling.revolve, kwargs, 'pivot')
        return res

    @_f.addMelDocs('revolve', 'pivotX')
    def getPivotX(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.revolve, kwargs, 'pivotX')
        return res

    @_f.addMelDocs('revolve', 'pivotY')
    def getPivotY(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.revolve, kwargs, 'pivotY')
        return res

    @_f.addMelDocs('revolve', 'pivotZ')
    def getPivotZ(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.revolve, kwargs, 'pivotZ')
        return res

    @_f.addMelDocs('revolve', 'radius')
    def getRadius(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.revolve, kwargs, 'radius')
        return res

    @_f.addMelDocs('revolve', 'radiusAnchor')
    def getRadiusAnchor(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.revolve, kwargs, 'radiusAnchor')
        return res

    @_f.addMelDocs('revolve', 'sections')
    def getSections(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.revolve, kwargs, 'sections')
        return res

    @_f.addMelDocs('revolve', 'startSweep')
    def getStartSweep(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.revolve, kwargs, 'startSweep')
        return res

    @_f.addMelDocs('revolve', 'tolerance')
    def getTolerance(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.revolve, kwargs, 'tolerance')
        return res

    @_f.addMelDocs('revolve', 'useLocalPivot')
    def getUseLocalPivot(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.revolve, kwargs, 'useLocalPivot')
        return res

    @_f.addMelDocs('revolve', 'useTolerance')
    def getUseTolerance(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.revolve, kwargs, 'useTolerance')
        return res

    @_f.addMelDocs('revolve', 'autoCorrectNormal')
    def setAutoCorrectNormal(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.revolve, kwargs, 'autoCorrectNormal', val)

    @_f.addMelDocs('revolve', 'axis')
    def setAxis(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, modeling.revolve, kwargs, 'axis', val)

    @_f.addMelDocs('revolve', 'axisChoice')
    def setAxisChoice(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.revolve, kwargs, 'axisChoice', val)

    @_f.addMelDocs('revolve', 'axisX')
    def setAxisX(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.revolve, kwargs, 'axisX', val)

    @_f.addMelDocs('revolve', 'axisY')
    def setAxisY(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.revolve, kwargs, 'axisY', val)

    @_f.addMelDocs('revolve', 'axisZ')
    def setAxisZ(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.revolve, kwargs, 'axisZ', val)

    @_f.addMelDocs('revolve', 'bridge')
    def setBridge(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.revolve, kwargs, 'bridge', val)

    @_f.addMelDocs('revolve', 'caching')
    def setCaching(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.revolve, kwargs, 'caching', val)

    @_f.addMelDocs('revolve', 'computePivotAndAxis')
    def setComputePivotAndAxis(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.revolve, kwargs, 'computePivotAndAxis', val)

    @_f.addMelDocs('revolve', 'degree')
    def setDegree(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.revolve, kwargs, 'degree', val)

    @_f.addMelDocs('revolve', 'endSweep')
    def setEndSweep(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.revolve, kwargs, 'endSweep', val)

    @_f.addMelDocs('revolve', 'nodeState')
    def setNodeState(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.revolve, kwargs, 'nodeState', val)

    @_f.addMelDocs('revolve', 'pivot')
    def setPivot(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, modeling.revolve, kwargs, 'pivot', val)

    @_f.addMelDocs('revolve', 'pivotX')
    def setPivotX(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.revolve, kwargs, 'pivotX', val)

    @_f.addMelDocs('revolve', 'pivotY')
    def setPivotY(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.revolve, kwargs, 'pivotY', val)

    @_f.addMelDocs('revolve', 'pivotZ')
    def setPivotZ(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.revolve, kwargs, 'pivotZ', val)

    @_f.addMelDocs('revolve', 'radius')
    def setRadius(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.revolve, kwargs, 'radius', val)

    @_f.addMelDocs('revolve', 'radiusAnchor')
    def setRadiusAnchor(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.revolve, kwargs, 'radiusAnchor', val)

    @_f.addMelDocs('revolve', 'sections')
    def setSections(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.revolve, kwargs, 'sections', val)

    @_f.addMelDocs('revolve', 'startSweep')
    def setStartSweep(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.revolve, kwargs, 'startSweep', val)

    @_f.addMelDocs('revolve', 'tolerance')
    def setTolerance(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.revolve, kwargs, 'tolerance', val)

    @_f.addMelDocs('revolve', 'useLocalPivot')
    def setUseLocalPivot(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.revolve, kwargs, 'useLocalPivot', val)

    @_f.addMelDocs('revolve', 'useTolerance')
    def setUseTolerance(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.revolve, kwargs, 'useTolerance', val)


class RoundConstantRadius(AbstractBaseCreate):
    __melcmd__ = staticmethod(modeling.roundConstantRadius)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'roundConstantRadius'
    __melnode__ = 'roundConstantRadius'
    __slots__ = ()


class SmoothCurve(AbstractBaseCreate):
    __melcmd__ = staticmethod(modeling.smoothCurve)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'smoothCurve'
    __melnode__ = 'smoothCurve'
    __slots__ = ()

    @_f.addMelDocs('smoothCurve', 'caching')
    def getCaching(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.smoothCurve, kwargs, 'caching')
        return res

    @_f.addMelDocs('smoothCurve', 'nodeState')
    def getNodeState(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.smoothCurve, kwargs, 'nodeState')
        return res

    @_f.addMelDocs('smoothCurve', 'smoothness')
    def getSmoothness(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.smoothCurve, kwargs, 'smoothness')
        return res

    @_f.addMelDocs('smoothCurve', 'caching')
    def setCaching(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.smoothCurve, kwargs, 'caching', val)

    @_f.addMelDocs('smoothCurve', 'nodeState')
    def setNodeState(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.smoothCurve, kwargs, 'nodeState', val)

    @_f.addMelDocs('smoothCurve', 'smoothness')
    def setSmoothness(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.smoothCurve, kwargs, 'smoothness', val)


class SmoothTangentSrf(AbstractBaseCreate):
    __melnode__ = 'smoothTangentSrf'
    __slots__ = ()


class StitchAsNurbsShell(AbstractBaseCreate):
    __melnode__ = 'stitchAsNurbsShell'
    __slots__ = ()


class StitchSrf(AbstractBaseCreate):
    __melnode__ = 'stitchSrf'
    __slots__ = ()


class StyleCurve(AbstractBaseCreate):
    __melnode__ = 'styleCurve'
    __slots__ = ()


class SubSurface(AbstractBaseCreate):
    __melnode__ = 'subSurface'
    __slots__ = ()


class SubdivToNurbs(AbstractBaseCreate):
    __melnode__ = 'subdivToNurbs'
    __slots__ = ()


class SurfaceInfo(AbstractBaseCreate):
    __melnode__ = 'surfaceInfo'
    __slots__ = ()


class TransformGeometry(AbstractBaseCreate):
    __melnode__ = 'transformGeometry'
    __slots__ = ()


class Trim(AbstractBaseCreate):
    __melcmd__ = staticmethod(modeling.trim)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'trim'
    __melnode__ = 'trim'
    __slots__ = ()

    @_f.addMelDocs('trim', 'caching')
    def getCaching(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.trim, kwargs, 'caching')
        return res

    @_f.addMelDocs('trim', 'locatorU')
    def getLocatorU(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.trim, kwargs, 'locatorU')
        return res

    @_f.addMelDocs('trim', 'locatorV')
    def getLocatorV(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.trim, kwargs, 'locatorV')
        return res

    @_f.addMelDocs('trim', 'nodeState')
    def getNodeState(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.trim, kwargs, 'nodeState')
        return res

    @_f.addMelDocs('trim', 'selected')
    def getSelected(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.trim, kwargs, 'selected')
        return res

    @_f.addMelDocs('trim', 'shrink')
    def getShrink(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.trim, kwargs, 'shrink')
        return res

    @_f.addMelDocs('trim', 'tolerance')
    def getTolerance(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, modeling.trim, kwargs, 'tolerance')
        return res

    @_f.addMelDocs('trim', 'caching')
    def setCaching(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.trim, kwargs, 'caching', val)

    @_f.addMelDocs('trim', 'locatorU')
    def setLocatorU(self, val=True, **kwargs):
        # type: (float | List[float], **Any) -> None
        return _f.asEdit(self, modeling.trim, kwargs, 'locatorU', val)

    @_f.addMelDocs('trim', 'locatorV')
    def setLocatorV(self, val=True, **kwargs):
        # type: (float | List[float], **Any) -> None
        return _f.asEdit(self, modeling.trim, kwargs, 'locatorV', val)

    @_f.addMelDocs('trim', 'nodeState')
    def setNodeState(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.trim, kwargs, 'nodeState', val)

    @_f.addMelDocs('trim', 'selected')
    def setSelected(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.trim, kwargs, 'selected', val)

    @_f.addMelDocs('trim', 'shrink')
    def setShrink(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.trim, kwargs, 'shrink', val)

    @_f.addMelDocs('trim', 'tolerance')
    def setTolerance(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, modeling.trim, kwargs, 'tolerance', val)


class TrimWithBoundaries(AbstractBaseCreate):
    __melnode__ = 'trimWithBoundaries'
    __slots__ = ()


class Untrim(AbstractBaseCreate):
    __melcmd__ = staticmethod(modeling.untrim)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'untrim'
    __melnode__ = 'untrim'
    __slots__ = ()

    @_f.addMelDocs('untrim', 'caching')
    def getCaching(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.untrim, kwargs, 'caching')
        return res

    @_f.addMelDocs('untrim', 'noChanges')
    def getNoChanges(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.untrim, kwargs, 'noChanges')
        return res

    @_f.addMelDocs('untrim', 'nodeState')
    def getNodeState(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, modeling.untrim, kwargs, 'nodeState')
        return res

    @_f.addMelDocs('untrim', 'untrimAll')
    def getUntrimAll(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, modeling.untrim, kwargs, 'untrimAll')
        return res

    @_f.addMelDocs('untrim', 'caching')
    def setCaching(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.untrim, kwargs, 'caching', val)

    @_f.addMelDocs('untrim', 'noChanges')
    def setNoChanges(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.untrim, kwargs, 'noChanges', val)

    @_f.addMelDocs('untrim', 'nodeState')
    def setNodeState(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, modeling.untrim, kwargs, 'nodeState', val)

    @_f.addMelDocs('untrim', 'untrimAll')
    def setUntrimAll(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, modeling.untrim, kwargs, 'untrimAll', val)


class AddDoubleLinear(DependNode):
    __melnode__ = 'addDoubleLinear'
    __slots__ = ()

    @_f.deprecated
    def findAlias(self, alias):
        # type: (str) -> Tuple[bool, DependNode]
        do, final_do, outTypes = _f.getDoArgs([alias], [('alias', 'MString', 'in', None), ('attrObj', 'MObject', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'findAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affected instead.')
    def getAffectedAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affects instead.')
    def getAffectedByAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedByAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedByAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use listAliases instead.')
    def getAliasList(self):
        # type: () -> Tuple[bool, List[str]]
        do, final_do, outTypes = _f.getDoArgs([], [('strArray', 'MStringArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAliasList', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use inputs/outputs, or general.listConnections instead.')
    def getConnections(self):
        # type: () -> List[general.Attribute]
        do, final_do, outTypes = _f.getDoArgs([], [('array', 'MPlugArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getConnections', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.getAlias instead.')
    def plugsAlias(self, plug):
        # type: (str | general.Attribute) -> str
        do, final_do, outTypes = _f.getDoArgs([plug], [('plug', 'MPlug', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'plugsAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MString', None)
        return res

    @_f.deprecated('Use Attribute.setAlias instead.')
    def setAlias(self, alias, name, plug, add=True):
        # type: (str, str, str | general.Attribute, bool) -> bool
        do, final_do, outTypes = _f.getDoArgs([alias, name, plug, add], [('alias', 'MString', 'in', None), ('name', 'MString', 'in', None), ('plug', 'MPlug', 'in', None), ('add', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'setAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res


class AddMatrix(DependNode):
    __melnode__ = 'addMatrix'
    __slots__ = ()

    @_f.deprecated
    def findAlias(self, alias):
        # type: (str) -> Tuple[bool, DependNode]
        do, final_do, outTypes = _f.getDoArgs([alias], [('alias', 'MString', 'in', None), ('attrObj', 'MObject', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'findAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affected instead.')
    def getAffectedAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affects instead.')
    def getAffectedByAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedByAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedByAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use listAliases instead.')
    def getAliasList(self):
        # type: () -> Tuple[bool, List[str]]
        do, final_do, outTypes = _f.getDoArgs([], [('strArray', 'MStringArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAliasList', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use inputs/outputs, or general.listConnections instead.')
    def getConnections(self):
        # type: () -> List[general.Attribute]
        do, final_do, outTypes = _f.getDoArgs([], [('array', 'MPlugArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getConnections', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.getAlias instead.')
    def plugsAlias(self, plug):
        # type: (str | general.Attribute) -> str
        do, final_do, outTypes = _f.getDoArgs([plug], [('plug', 'MPlug', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'plugsAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MString', None)
        return res

    @_f.deprecated('Use Attribute.setAlias instead.')
    def setAlias(self, alias, name, plug, add=True):
        # type: (str, str, str | general.Attribute, bool) -> bool
        do, final_do, outTypes = _f.getDoArgs([alias, name, plug, add], [('alias', 'MString', 'in', None), ('name', 'MString', 'in', None), ('plug', 'MPlug', 'in', None), ('add', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'setAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res


class AimMatrix(DependNode):
    __melnode__ = 'aimMatrix'
    __slots__ = ()


class AnimBlend(DependNode):
    __melnode__ = 'animBlend'
    __slots__ = ()

    @_f.deprecated
    def findAlias(self, alias):
        # type: (str) -> Tuple[bool, DependNode]
        do, final_do, outTypes = _f.getDoArgs([alias], [('alias', 'MString', 'in', None), ('attrObj', 'MObject', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'findAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affected instead.')
    def getAffectedAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affects instead.')
    def getAffectedByAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedByAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedByAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use listAliases instead.')
    def getAliasList(self):
        # type: () -> Tuple[bool, List[str]]
        do, final_do, outTypes = _f.getDoArgs([], [('strArray', 'MStringArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAliasList', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use inputs/outputs, or general.listConnections instead.')
    def getConnections(self):
        # type: () -> List[general.Attribute]
        do, final_do, outTypes = _f.getDoArgs([], [('array', 'MPlugArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getConnections', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.getAlias instead.')
    def plugsAlias(self, plug):
        # type: (str | general.Attribute) -> str
        do, final_do, outTypes = _f.getDoArgs([plug], [('plug', 'MPlug', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'plugsAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MString', None)
        return res

    @_f.deprecated('Use Attribute.setAlias instead.')
    def setAlias(self, alias, name, plug, add=True):
        # type: (str, str, str | general.Attribute, bool) -> bool
        do, final_do, outTypes = _f.getDoArgs([alias, name, plug, add], [('alias', 'MString', 'in', None), ('name', 'MString', 'in', None), ('plug', 'MPlug', 'in', None), ('add', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'setAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res


class AnimBlendInOut(AnimBlend):
    __melnode__ = 'animBlendInOut'
    __slots__ = ()


class AnimBlendNodeBase(DependNode):
    __melnode__ = 'animBlendNodeBase'
    __slots__ = ()

    @_f.deprecated
    def findAlias(self, alias):
        # type: (str) -> Tuple[bool, DependNode]
        do, final_do, outTypes = _f.getDoArgs([alias], [('alias', 'MString', 'in', None), ('attrObj', 'MObject', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'findAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affected instead.')
    def getAffectedAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affects instead.')
    def getAffectedByAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedByAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedByAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use listAliases instead.')
    def getAliasList(self):
        # type: () -> Tuple[bool, List[str]]
        do, final_do, outTypes = _f.getDoArgs([], [('strArray', 'MStringArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAliasList', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use inputs/outputs, or general.listConnections instead.')
    def getConnections(self):
        # type: () -> List[general.Attribute]
        do, final_do, outTypes = _f.getDoArgs([], [('array', 'MPlugArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getConnections', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.getAlias instead.')
    def plugsAlias(self, plug):
        # type: (str | general.Attribute) -> str
        do, final_do, outTypes = _f.getDoArgs([plug], [('plug', 'MPlug', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'plugsAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MString', None)
        return res

    @_f.deprecated('Use Attribute.setAlias instead.')
    def setAlias(self, alias, name, plug, add=True):
        # type: (str, str, str | general.Attribute, bool) -> bool
        do, final_do, outTypes = _f.getDoArgs([alias, name, plug, add], [('alias', 'MString', 'in', None), ('name', 'MString', 'in', None), ('plug', 'MPlug', 'in', None), ('add', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'setAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res


class AnimBlendNodeAdditive(AnimBlendNodeBase):
    __melnode__ = 'animBlendNodeAdditive'
    __slots__ = ()


class AnimBlendNodeAdditiveDA(AnimBlendNodeBase):
    __melnode__ = 'animBlendNodeAdditiveDA'
    __slots__ = ()


class AnimBlendNodeAdditiveDL(AnimBlendNodeBase):
    __melnode__ = 'animBlendNodeAdditiveDL'
    __slots__ = ()


class AnimBlendNodeAdditiveF(AnimBlendNodeBase):
    __melnode__ = 'animBlendNodeAdditiveF'
    __slots__ = ()


class AnimBlendNodeAdditiveFA(AnimBlendNodeBase):
    __melnode__ = 'animBlendNodeAdditiveFA'
    __slots__ = ()


class AnimBlendNodeAdditiveFL(AnimBlendNodeBase):
    __melnode__ = 'animBlendNodeAdditiveFL'
    __slots__ = ()


class AnimBlendNodeAdditiveI16(AnimBlendNodeBase):
    __melnode__ = 'animBlendNodeAdditiveI16'
    __slots__ = ()


class AnimBlendNodeAdditiveI32(AnimBlendNodeBase):
    __melnode__ = 'animBlendNodeAdditiveI32'
    __slots__ = ()


class AnimBlendNodeAdditiveRotation(AnimBlendNodeBase):
    __melnode__ = 'animBlendNodeAdditiveRotation'
    __slots__ = ()


class AnimBlendNodeAdditiveScale(AnimBlendNodeBase):
    __melnode__ = 'animBlendNodeAdditiveScale'
    __slots__ = ()


class AnimBlendNodeBoolean(AnimBlendNodeBase):
    __melnode__ = 'animBlendNodeBoolean'
    __slots__ = ()


class AnimBlendNodeEnum(AnimBlendNodeBase):
    __melnode__ = 'animBlendNodeEnum'
    __slots__ = ()


class AnimBlendNodeTime(AnimBlendNodeBase):
    __melnode__ = 'animBlendNodeTime'
    __slots__ = ()


class AnimClip(DependNode):
    __apicls__ = _api.MFnClip
    __melnode__ = 'animClip'
    __slots__ = ()

    @_f.addApiDocs(_api.MFnClip, 'getAbsoluteChannelSettings')
    def getAbsoluteChannelSettings(self):
        # type: () -> List[int]
        do, final_do, outTypes = _f.getDoArgs([], [('absoluteChannels', 'MIntArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnClip, 'getAbsoluteChannelSettings', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnClip, 'getEnabled')
    def getEnabled(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnClip, 'getEnabled')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnClip, 'getMemberAnimCurves')
    def getMemberAnimCurves(self):
        # type: () -> Tuple[List[DependNode], List[general.Attribute]]
        do, final_do, outTypes = _f.getDoArgs([], [('curves', 'MObjectArray', 'out', None), ('associatedAttrs', 'MPlugArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnClip, 'getMemberAnimCurves', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnClip, 'getPostCycle')
    def getPostCycle(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnClip, 'getPostCycle')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addApiDocs(_api.MFnClip, 'getPreCycle')
    def getPreCycle(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnClip, 'getPreCycle')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addApiDocs(_api.MFnClip, 'getScale')
    def getScale(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnClip, 'getScale')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addApiDocs(_api.MFnClip, 'getSourceDuration')
    def getSourceDuration(self):
        # type: () -> datatypes.Time
        res = _f.getProxyResult(self, _api.MFnClip, 'getSourceDuration')
        return _f.ApiArgUtil._castResult(self, res, 'MTime', None)

    @_f.addApiDocs(_api.MFnClip, 'getSourceStart')
    def getSourceStart(self):
        # type: () -> datatypes.Time
        res = _f.getProxyResult(self, _api.MFnClip, 'getSourceStart')
        return _f.ApiArgUtil._castResult(self, res, 'MTime', None)

    @_f.addApiDocs(_api.MFnClip, 'getStartFrame')
    def getStartFrame(self):
        # type: () -> datatypes.Time
        res = _f.getProxyResult(self, _api.MFnClip, 'getStartFrame')
        return _f.ApiArgUtil._castResult(self, res, 'MTime', None)

    @_f.addApiDocs(_api.MFnClip, 'getWeight')
    def getWeight(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnClip, 'getWeight')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addApiDocs(_api.MFnClip, 'isInstancedClip')
    def isInstancedClip(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnClip, 'isInstancedClip')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnClip, 'isPose')
    def isPose(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnClip, 'isPose')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnClip, 'setAbsoluteChannelSettings')
    def setAbsoluteChannelSettings(self, absoluteChannels, mod=None):
        # type: (List[int], datatypes.DGModifier) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([absoluteChannels, mod], [('absoluteChannels', 'MIntArray', 'in', None), ('mod', 'MDGModifier', 'in', None)], self.getAbsoluteChannelSettings, self.setAbsoluteChannelSettings, [])
        res = _f.getProxyResult(self, _api.MFnClip, 'setAbsoluteChannelSettings', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnClip, 'setEnabled')
    def setEnabled(self, val, mod=None):
        # type: (bool, datatypes.DGModifier) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([val, mod], [('val', 'bool', 'in', None), ('mod', 'MDGModifier', 'in', None)], self.getEnabled, self.setEnabled, [])
        res = _f.getProxyResult(self, _api.MFnClip, 'setEnabled', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnClip, 'setPoseClip')
    def setPoseClip(self, state, mod=None):
        # type: (bool, datatypes.DGModifier) -> None
        do, final_do, outTypes = _f.getDoArgs([state, mod], [('state', 'bool', 'in', None), ('mod', 'MDGModifier', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnClip, 'setPoseClip', final_do)
        return res

    @_f.addApiDocs(_api.MFnClip, 'setPostCycle')
    def setPostCycle(self, cycle, mod=None):
        # type: (float, datatypes.DGModifier) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([cycle, mod], [('cycle', 'double', 'in', None), ('mod', 'MDGModifier', 'in', None)], self.getPostCycle, self.setPostCycle, [])
        res = _f.getProxyResult(self, _api.MFnClip, 'setPostCycle', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnClip, 'setPreCycle')
    def setPreCycle(self, cycle, mod=None):
        # type: (float, datatypes.DGModifier) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([cycle, mod], [('cycle', 'double', 'in', None), ('mod', 'MDGModifier', 'in', None)], self.getPreCycle, self.setPreCycle, [])
        res = _f.getProxyResult(self, _api.MFnClip, 'setPreCycle', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnClip, 'setScale')
    def setScale(self, scale, mod=None):
        # type: (float, datatypes.DGModifier) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([scale, mod], [('scale', 'double', 'in', None), ('mod', 'MDGModifier', 'in', None)], self.getScale, self.setScale, [])
        res = _f.getProxyResult(self, _api.MFnClip, 'setScale', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnClip, 'setSourceData')
    def setSourceData(self, start, duration, mod=None):
        # type: (datatypes.Time, datatypes.Time, datatypes.DGModifier) -> None
        do, final_do, outTypes = _f.getDoArgs([start, duration, mod], [('start', 'MTime', 'in', None), ('duration', 'MTime', 'in', None), ('mod', 'MDGModifier', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnClip, 'setSourceData', final_do)
        return res

    @_f.addApiDocs(_api.MFnClip, 'setStartFrame')
    def setStartFrame(self, start, mod=None):
        # type: (datatypes.Time, datatypes.DGModifier) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([start, mod], [('start', 'MTime', 'in', None), ('mod', 'MDGModifier', 'in', None)], self.getStartFrame, self.setStartFrame, [])
        res = _f.getProxyResult(self, _api.MFnClip, 'setStartFrame', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnClip, 'setWeight')
    def setWeight(self, wt, mod=None):
        # type: (float, datatypes.DGModifier) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([wt, mod], [('wt', 'double', 'in', None), ('mod', 'MDGModifier', 'in', None)], self.getWeight, self.setWeight, [])
        res = _f.getProxyResult(self, _api.MFnClip, 'setWeight', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnClip, 'sourceClip')
    def sourceClip(self):
        # type: () -> DependNode
        res = _f.getProxyResult(self, _api.MFnClip, 'sourceClip')
        return _f.ApiArgUtil._castResult(self, res, 'MObject', None)


class ArrayMapper(DependNode):
    __melcmd__ = staticmethod(effects.arrayMapper)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'arrayMapper'
    __melnode__ = 'arrayMapper'
    __slots__ = ()

    @_f.deprecated
    def findAlias(self, alias):
        # type: (str) -> Tuple[bool, DependNode]
        do, final_do, outTypes = _f.getDoArgs([alias], [('alias', 'MString', 'in', None), ('attrObj', 'MObject', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'findAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affected instead.')
    def getAffectedAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affects instead.')
    def getAffectedByAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedByAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedByAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use listAliases instead.')
    def getAliasList(self):
        # type: () -> Tuple[bool, List[str]]
        do, final_do, outTypes = _f.getDoArgs([], [('strArray', 'MStringArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAliasList', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use inputs/outputs, or general.listConnections instead.')
    def getConnections(self):
        # type: () -> List[general.Attribute]
        do, final_do, outTypes = _f.getDoArgs([], [('array', 'MPlugArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getConnections', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.getAlias instead.')
    def plugsAlias(self, plug):
        # type: (str | general.Attribute) -> str
        do, final_do, outTypes = _f.getDoArgs([plug], [('plug', 'MPlug', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'plugsAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MString', None)
        return res

    @_f.deprecated('Use Attribute.setAlias instead.')
    def setAlias(self, alias, name, plug, add=True):
        # type: (str, str, str | general.Attribute, bool) -> bool
        do, final_do, outTypes = _f.getDoArgs([alias, name, plug, add], [('alias', 'MString', 'in', None), ('name', 'MString', 'in', None), ('plug', 'MPlug', 'in', None), ('add', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'setAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res


class THarrayMapper(ArrayMapper):
    __melnode__ = 'THarrayMapper'
    __slots__ = ()


class AttrHierarchyTest(DependNode):
    __melnode__ = 'attrHierarchyTest'
    __slots__ = ()

    @_f.deprecated
    def findAlias(self, alias):
        # type: (str) -> Tuple[bool, DependNode]
        do, final_do, outTypes = _f.getDoArgs([alias], [('alias', 'MString', 'in', None), ('attrObj', 'MObject', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'findAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affected instead.')
    def getAffectedAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affects instead.')
    def getAffectedByAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedByAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedByAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use listAliases instead.')
    def getAliasList(self):
        # type: () -> Tuple[bool, List[str]]
        do, final_do, outTypes = _f.getDoArgs([], [('strArray', 'MStringArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAliasList', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use inputs/outputs, or general.listConnections instead.')
    def getConnections(self):
        # type: () -> List[general.Attribute]
        do, final_do, outTypes = _f.getDoArgs([], [('array', 'MPlugArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getConnections', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.getAlias instead.')
    def plugsAlias(self, plug):
        # type: (str | general.Attribute) -> str
        do, final_do, outTypes = _f.getDoArgs([plug], [('plug', 'MPlug', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'plugsAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MString', None)
        return res

    @_f.deprecated('Use Attribute.setAlias instead.')
    def setAlias(self, alias, name, plug, add=True):
        # type: (str, str, str | general.Attribute, bool) -> bool
        do, final_do, outTypes = _f.getDoArgs([alias, name, plug, add], [('alias', 'MString', 'in', None), ('name', 'MString', 'in', None), ('plug', 'MPlug', 'in', None), ('add', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'setAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res


class Audio(DependNode):
    __melnode__ = 'audio'
    __slots__ = ()

    @_f.deprecated
    def findAlias(self, alias):
        # type: (str) -> Tuple[bool, DependNode]
        do, final_do, outTypes = _f.getDoArgs([alias], [('alias', 'MString', 'in', None), ('attrObj', 'MObject', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'findAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affected instead.')
    def getAffectedAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affects instead.')
    def getAffectedByAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedByAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedByAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use listAliases instead.')
    def getAliasList(self):
        # type: () -> Tuple[bool, List[str]]
        do, final_do, outTypes = _f.getDoArgs([], [('strArray', 'MStringArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAliasList', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use inputs/outputs, or general.listConnections instead.')
    def getConnections(self):
        # type: () -> List[general.Attribute]
        do, final_do, outTypes = _f.getDoArgs([], [('array', 'MPlugArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getConnections', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.getAlias instead.')
    def plugsAlias(self, plug):
        # type: (str | general.Attribute) -> str
        do, final_do, outTypes = _f.getDoArgs([plug], [('plug', 'MPlug', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'plugsAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MString', None)
        return res

    @_f.deprecated('Use Attribute.setAlias instead.')
    def setAlias(self, alias, name, plug, add=True):
        # type: (str, str, str | general.Attribute, bool) -> bool
        do, final_do, outTypes = _f.getDoArgs([alias, name, plug, add], [('alias', 'MString', 'in', None), ('name', 'MString', 'in', None), ('plug', 'MPlug', 'in', None), ('add', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'setAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res


class Blend(DependNode):
    __melcmd__ = staticmethod(other.blend)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'blend'
    __melnode__ = 'blend'
    __slots__ = ()

    @_f.deprecated
    def findAlias(self, alias):
        # type: (str) -> Tuple[bool, DependNode]
        do, final_do, outTypes = _f.getDoArgs([alias], [('alias', 'MString', 'in', None), ('attrObj', 'MObject', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'findAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affected instead.')
    def getAffectedAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affects instead.')
    def getAffectedByAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedByAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedByAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use listAliases instead.')
    def getAliasList(self):
        # type: () -> Tuple[bool, List[str]]
        do, final_do, outTypes = _f.getDoArgs([], [('strArray', 'MStringArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAliasList', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use inputs/outputs, or general.listConnections instead.')
    def getConnections(self):
        # type: () -> List[general.Attribute]
        do, final_do, outTypes = _f.getDoArgs([], [('array', 'MPlugArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getConnections', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.getAlias instead.')
    def plugsAlias(self, plug):
        # type: (str | general.Attribute) -> str
        do, final_do, outTypes = _f.getDoArgs([plug], [('plug', 'MPlug', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'plugsAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MString', None)
        return res

    @_f.deprecated('Use Attribute.setAlias instead.')
    def setAlias(self, alias, name, plug, add=True):
        # type: (str, str, str | general.Attribute, bool) -> bool
        do, final_do, outTypes = _f.getDoArgs([alias, name, plug, add], [('alias', 'MString', 'in', None), ('name', 'MString', 'in', None), ('plug', 'MPlug', 'in', None), ('add', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'setAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res


class BlendDevice(Blend):
    __melnode__ = 'blendDevice'
    __slots__ = ()


class BlendTwoAttr(Blend):
    __melcmd__ = staticmethod(animation.blendTwoAttr)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'blendTwoAttr'
    __melnode__ = 'blendTwoAttr'
    __slots__ = ()

    @_f.addMelDocs('blendTwoAttr', 'attribute0')
    def getAttribute0(self, **kwargs):
        # type: (...) -> general.PyNode
        res = _f.asQuery(self, animation.blendTwoAttr, kwargs, 'attribute0')
        return res

    @_f.addMelDocs('blendTwoAttr', 'attribute1')
    def getAttribute1(self, **kwargs):
        # type: (...) -> general.PyNode
        res = _f.asQuery(self, animation.blendTwoAttr, kwargs, 'attribute1')
        return res

    @_f.addMelDocs('blendTwoAttr', 'blender')
    def getBlender(self, **kwargs):
        # type: (...) -> general.PyNode
        res = _f.asQuery(self, animation.blendTwoAttr, kwargs, 'blender')
        return res

    @_f.addMelDocs('blendTwoAttr', 'driver')
    def getDriver(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, animation.blendTwoAttr, kwargs, 'driver')
        return res

    @_f.addMelDocs('blendTwoAttr', 'attribute0')
    def setAttribute0(self, val=True, **kwargs):
        # type: (general.PyNode | str, **Any) -> None
        return _f.asEdit(self, animation.blendTwoAttr, kwargs, 'attribute0', val)

    @_f.addMelDocs('blendTwoAttr', 'attribute1')
    def setAttribute1(self, val=True, **kwargs):
        # type: (general.PyNode | str, **Any) -> None
        return _f.asEdit(self, animation.blendTwoAttr, kwargs, 'attribute1', val)

    @_f.addMelDocs('blendTwoAttr', 'blender')
    def setBlender(self, val=True, **kwargs):
        # type: (general.PyNode | str, **Any) -> None
        return _f.asEdit(self, animation.blendTwoAttr, kwargs, 'blender', val)

    @_f.addMelDocs('blendTwoAttr', 'driver')
    def setDriver(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, animation.blendTwoAttr, kwargs, 'driver', val)


class BlendWeighted(Blend):
    __melnode__ = 'blendWeighted'
    __slots__ = ()


class BlendColors(DependNode):
    __melnode__ = 'blendColors'
    __slots__ = ()

    @_f.deprecated
    def findAlias(self, alias):
        # type: (str) -> Tuple[bool, DependNode]
        do, final_do, outTypes = _f.getDoArgs([alias], [('alias', 'MString', 'in', None), ('attrObj', 'MObject', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'findAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affected instead.')
    def getAffectedAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affects instead.')
    def getAffectedByAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedByAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedByAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use listAliases instead.')
    def getAliasList(self):
        # type: () -> Tuple[bool, List[str]]
        do, final_do, outTypes = _f.getDoArgs([], [('strArray', 'MStringArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAliasList', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use inputs/outputs, or general.listConnections instead.')
    def getConnections(self):
        # type: () -> List[general.Attribute]
        do, final_do, outTypes = _f.getDoArgs([], [('array', 'MPlugArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getConnections', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.getAlias instead.')
    def plugsAlias(self, plug):
        # type: (str | general.Attribute) -> str
        do, final_do, outTypes = _f.getDoArgs([plug], [('plug', 'MPlug', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'plugsAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MString', None)
        return res

    @_f.deprecated('Use Attribute.setAlias instead.')
    def setAlias(self, alias, name, plug, add=True):
        # type: (str, str, str | general.Attribute, bool) -> bool
        do, final_do, outTypes = _f.getDoArgs([alias, name, plug, add], [('alias', 'MString', 'in', None), ('name', 'MString', 'in', None), ('plug', 'MPlug', 'in', None), ('add', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'setAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res


class BlendFalloff(DependNode):
    __melnode__ = 'blendFalloff'
    __slots__ = ()


class BlendMatrix(DependNode):
    __melnode__ = 'blendMatrix'
    __slots__ = ()


class BlindDataTemplate(DependNode):
    __melnode__ = 'blindDataTemplate'
    __slots__ = ()

    @_f.deprecated
    def findAlias(self, alias):
        # type: (str) -> Tuple[bool, DependNode]
        do, final_do, outTypes = _f.getDoArgs([alias], [('alias', 'MString', 'in', None), ('attrObj', 'MObject', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'findAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affected instead.')
    def getAffectedAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affects instead.')
    def getAffectedByAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedByAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedByAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use listAliases instead.')
    def getAliasList(self):
        # type: () -> Tuple[bool, List[str]]
        do, final_do, outTypes = _f.getDoArgs([], [('strArray', 'MStringArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAliasList', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use inputs/outputs, or general.listConnections instead.')
    def getConnections(self):
        # type: () -> List[general.Attribute]
        do, final_do, outTypes = _f.getDoArgs([], [('array', 'MPlugArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getConnections', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.getAlias instead.')
    def plugsAlias(self, plug):
        # type: (str | general.Attribute) -> str
        do, final_do, outTypes = _f.getDoArgs([plug], [('plug', 'MPlug', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'plugsAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MString', None)
        return res

    @_f.deprecated('Use Attribute.setAlias instead.')
    def setAlias(self, alias, name, plug, add=True):
        # type: (str, str, str | general.Attribute, bool) -> bool
        do, final_do, outTypes = _f.getDoArgs([alias, name, plug, add], [('alias', 'MString', 'in', None), ('name', 'MString', 'in', None), ('plug', 'MPlug', 'in', None), ('add', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'setAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res


class SubdHierBlind(BlindDataTemplate):
    __melnode__ = 'subdHierBlind'
    __slots__ = ()


class Brush(DependNode):
    __melnode__ = 'brush'
    __slots__ = ()

    @_f.deprecated
    def findAlias(self, alias):
        # type: (str) -> Tuple[bool, DependNode]
        do, final_do, outTypes = _f.getDoArgs([alias], [('alias', 'MString', 'in', None), ('attrObj', 'MObject', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'findAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affected instead.')
    def getAffectedAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affects instead.')
    def getAffectedByAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedByAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedByAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use listAliases instead.')
    def getAliasList(self):
        # type: () -> Tuple[bool, List[str]]
        do, final_do, outTypes = _f.getDoArgs([], [('strArray', 'MStringArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAliasList', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use inputs/outputs, or general.listConnections instead.')
    def getConnections(self):
        # type: () -> List[general.Attribute]
        do, final_do, outTypes = _f.getDoArgs([], [('array', 'MPlugArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getConnections', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.getAlias instead.')
    def plugsAlias(self, plug):
        # type: (str | general.Attribute) -> str
        do, final_do, outTypes = _f.getDoArgs([plug], [('plug', 'MPlug', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'plugsAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MString', None)
        return res

    @_f.deprecated('Use Attribute.setAlias instead.')
    def setAlias(self, alias, name, plug, add=True):
        # type: (str, str, str | general.Attribute, bool) -> bool
        do, final_do, outTypes = _f.getDoArgs([alias, name, plug, add], [('alias', 'MString', 'in', None), ('name', 'MString', 'in', None), ('plug', 'MPlug', 'in', None), ('add', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'setAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res


class Bump3d(DependNode):
    __melnode__ = 'bump3d'
    __slots__ = ()

    @_f.deprecated
    def findAlias(self, alias):
        # type: (str) -> Tuple[bool, DependNode]
        do, final_do, outTypes = _f.getDoArgs([alias], [('alias', 'MString', 'in', None), ('attrObj', 'MObject', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'findAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affected instead.')
    def getAffectedAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affects instead.')
    def getAffectedByAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedByAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedByAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use listAliases instead.')
    def getAliasList(self):
        # type: () -> Tuple[bool, List[str]]
        do, final_do, outTypes = _f.getDoArgs([], [('strArray', 'MStringArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAliasList', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use inputs/outputs, or general.listConnections instead.')
    def getConnections(self):
        # type: () -> List[general.Attribute]
        do, final_do, outTypes = _f.getDoArgs([], [('array', 'MPlugArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getConnections', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.getAlias instead.')
    def plugsAlias(self, plug):
        # type: (str | general.Attribute) -> str
        do, final_do, outTypes = _f.getDoArgs([plug], [('plug', 'MPlug', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'plugsAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MString', None)
        return res

    @_f.deprecated('Use Attribute.setAlias instead.')
    def setAlias(self, alias, name, plug, add=True):
        # type: (str, str, str | general.Attribute, bool) -> bool
        do, final_do, outTypes = _f.getDoArgs([alias, name, plug, add], [('alias', 'MString', 'in', None), ('name', 'MString', 'in', None), ('plug', 'MPlug', 'in', None), ('add', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'setAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res


class CacheBase(DependNode):
    __melnode__ = 'cacheBase'
    __slots__ = ()

    @_f.deprecated
    def findAlias(self, alias):
        # type: (str) -> Tuple[bool, DependNode]
        do, final_do, outTypes = _f.getDoArgs([alias], [('alias', 'MString', 'in', None), ('attrObj', 'MObject', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'findAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affected instead.')
    def getAffectedAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affects instead.')
    def getAffectedByAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedByAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedByAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use listAliases instead.')
    def getAliasList(self):
        # type: () -> Tuple[bool, List[str]]
        do, final_do, outTypes = _f.getDoArgs([], [('strArray', 'MStringArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAliasList', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use inputs/outputs, or general.listConnections instead.')
    def getConnections(self):
        # type: () -> List[general.Attribute]
        do, final_do, outTypes = _f.getDoArgs([], [('array', 'MPlugArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getConnections', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.getAlias instead.')
    def plugsAlias(self, plug):
        # type: (str | general.Attribute) -> str
        do, final_do, outTypes = _f.getDoArgs([plug], [('plug', 'MPlug', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'plugsAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MString', None)
        return res

    @_f.deprecated('Use Attribute.setAlias instead.')
    def setAlias(self, alias, name, plug, add=True):
        # type: (str, str, str | general.Attribute, bool) -> bool
        do, final_do, outTypes = _f.getDoArgs([alias, name, plug, add], [('alias', 'MString', 'in', None), ('name', 'MString', 'in', None), ('plug', 'MPlug', 'in', None), ('add', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'setAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res


class CacheBlend(CacheBase):
    __melnode__ = 'cacheBlend'
    __slots__ = ()


class CacheFile(CacheBase):
    __melcmd__ = staticmethod(system.cacheFile)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'cacheFile'
    __melnode__ = 'cacheFile'
    __slots__ = ()

    @_f.addMelDocs('cacheFile', 'cacheFormat')
    def getCacheFormat(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, system.cacheFile, kwargs, 'cacheFormat')
        return res

    @_f.addMelDocs('cacheFile', 'cacheInfo')
    def getCacheInfo(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, system.cacheFile, kwargs, 'cacheInfo')
        return res

    @_f.addMelDocs('cacheFile', 'cacheableAttrs')
    def getCacheableAttrs(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, system.cacheFile, kwargs, 'cacheableAttrs')
        return res

    @_f.addMelDocs('cacheFile', 'channelIndex')
    def getChannelIndex(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, system.cacheFile, kwargs, 'channelIndex')
        return res

    @_f.addMelDocs('cacheFile', 'channelName')
    def getChannelName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, system.cacheFile, kwargs, 'channelName')
        return res

    @_f.addMelDocs('cacheFile', 'dataSize')
    def getDataSize(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, system.cacheFile, kwargs, 'dataSize')
        return res

    @_f.addMelDocs('cacheFile', 'descriptionFileName')
    def getDescriptionFileName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, system.cacheFile, kwargs, 'descriptionFileName')
        return res

    @_f.addMelDocs('cacheFile', 'directory')
    def getDirectory(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, system.cacheFile, kwargs, 'directory')
        return res

    @_f.addMelDocs('cacheFile', 'fileName')
    def getFileName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, system.cacheFile, kwargs, 'fileName')
        return res

    @_f.addMelDocs('cacheFile', 'geometry')
    def getGeometry(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, system.cacheFile, kwargs, 'geometry')
        return res

    @_f.addMelDocs('cacheFile', 'pointCount')
    def getPointCount(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, system.cacheFile, kwargs, 'pointCount')
        return res

    @_f.addMelDocs('cacheFile', 'runupFrames')
    def getRunupFrames(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, system.cacheFile, kwargs, 'runupFrames')
        return res

    @_f.addMelDocs('cacheFile', 'sampleMultiplier')
    def getSampleMultiplier(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, system.cacheFile, kwargs, 'sampleMultiplier')
        return res

    @_f.addMelDocs('cacheFile', 'simulationRate')
    def getSimulationRate(self, **kwargs):
        # type: (...) -> int | float
        res = _f.asQuery(self, system.cacheFile, kwargs, 'simulationRate')
        return res

    @_f.addMelDocs('cacheFile', 'staticCache')
    def getStaticCache(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, system.cacheFile, kwargs, 'staticCache')
        return res

    @_f.addMelDocs('cacheFile', 'replaceWithoutSimulating')
    def replaceWithoutSimulating(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, system.cacheFile, kwargs, 'replaceWithoutSimulating', val)

    @_f.addMelDocs('cacheFile', 'runupFrames')
    def setRunupFrames(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, system.cacheFile, kwargs, 'runupFrames', val)

    @_f.addMelDocs('cacheFile', 'sampleMultiplier')
    def setSampleMultiplier(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, system.cacheFile, kwargs, 'sampleMultiplier', val)

    @_f.addMelDocs('cacheFile', 'simulationRate')
    def setSimulationRate(self, val=True, **kwargs):
        # type: (int | float, **Any) -> None
        return _f.asEdit(self, system.cacheFile, kwargs, 'simulationRate', val)


class CameraSet(DependNode):
    __apicls__ = _api.MFnCameraSet
    __melcmd__ = staticmethod(rendering.cameraSet)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'cameraSet'
    __melnode__ = 'cameraSet'
    __slots__ = ()

    @_f.addApiDocs(_api.MFnCameraSet, 'appendLayer')
    def appendLayer(self, camera, set):
        # type: (DagNode, str | DependNode) -> None
        do, final_do, outTypes = _f.getDoArgs([camera, set], [('camera', 'MDagPath', 'in', None), ('set', 'MObject', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnCameraSet, 'appendLayer', final_do)
        return res

    @_f.addMelDocs('cameraSet', 'appendTo')
    def appendTo(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, rendering.cameraSet, kwargs, 'appendTo', val)

    @_f.addApiDocs(_api.MFnCameraSet, 'clear')
    def clear(self):
        # type: () -> None
        res = _f.getProxyResult(self, _api.MFnCameraSet, 'clear')
        return res

    @_f.addMelDocs('cameraSet', 'deleteAll')
    def deleteAll(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, rendering.cameraSet, kwargs, 'deleteAll', val)

    @_f.addApiDocs(_api.MFnCameraSet, 'deleteLayer')
    def deleteLayer(self, index):
        # type: (int) -> None
        do, final_do, outTypes = _f.getDoArgs([index], [('index', 'uint', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnCameraSet, 'deleteLayer', final_do)
        return res

    @_f.addMelDocs('cameraSet', 'active')
    def getActive(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, rendering.cameraSet, kwargs, 'active')
        return res

    @_f.addMelDocs('cameraSet', 'camera')
    def getCamera(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, rendering.cameraSet, kwargs, 'camera')
        return res

    @_f.addMelDocs('cameraSet', 'clearDepth')
    def getClearDepth(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, rendering.cameraSet, kwargs, 'clearDepth')
        return res

    @_f.addMelDocs('cameraSet', 'layer')
    def getLayer(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, rendering.cameraSet, kwargs, 'layer')
        return res

    @_f.addApiDocs(_api.MFnCameraSet, 'getLayerCamera')
    def getLayerCamera(self, index):
        # type: (int) -> DagNode
        do, final_do, outTypes = _f.getDoArgs([index], [('index', 'uint', 'in', None), ('camera', 'MDagPath', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnCameraSet, 'getLayerCamera', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnCameraSet, 'getLayerClearDepthValue')
    def getLayerClearDepthValue(self, index):
        # type: (int) -> bool
        do, final_do, outTypes = _f.getDoArgs([index], [('index', 'uint', 'in', None), ('depth', 'bool', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnCameraSet, 'getLayerClearDepthValue', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnCameraSet, 'getLayerOrder')
    def getLayerOrder(self, index):
        # type: (int) -> int
        do, final_do, outTypes = _f.getDoArgs([index], [('index', 'uint', 'in', None), ('order', 'int', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnCameraSet, 'getLayerOrder', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnCameraSet, 'getLayerSceneData')
    def getLayerSceneData(self, index):
        # type: (int) -> DependNode
        do, final_do, outTypes = _f.getDoArgs([index], [('index', 'uint', 'in', None), ('set', 'MObject', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnCameraSet, 'getLayerSceneData', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnCameraSet, 'getNumLayers')
    def getNumLayers(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnCameraSet, 'getNumLayers')
        return _f.ApiArgUtil._castResult(self, res, 'uint', None)

    @_f.addMelDocs('cameraSet', 'objectSet')
    def getObjectSet(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, rendering.cameraSet, kwargs, 'objectSet')
        return res

    @_f.addMelDocs('cameraSet', 'order')
    def getOrder(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, rendering.cameraSet, kwargs, 'order')
        return res

    @_f.addMelDocs('cameraSet', 'insertAt')
    def insertAt(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, rendering.cameraSet, kwargs, 'insertAt', val)

    @_f.addApiDocs(_api.MFnCameraSet, 'insertLayer')
    def insertLayer(self, index, camera, set):
        # type: (int, DagNode, str | DependNode) -> None
        do, final_do, outTypes = _f.getDoArgs([index, camera, set], [('index', 'uint', 'in', None), ('camera', 'MDagPath', 'in', None), ('set', 'MObject', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnCameraSet, 'insertLayer', final_do)
        return res

    @_f.addApiDocs(_api.MFnCameraSet, 'isLayerActive')
    def isLayerActive(self, index):
        # type: (int) -> bool
        do, final_do, outTypes = _f.getDoArgs([index], [('index', 'uint', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnCameraSet, 'isLayerActive', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res

    @_f.addMelDocs('cameraSet', 'active')
    def setActive(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, rendering.cameraSet, kwargs, 'active', val)

    @_f.addMelDocs('cameraSet', 'camera')
    def setCamera(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, rendering.cameraSet, kwargs, 'camera', val)

    @_f.addMelDocs('cameraSet', 'clearDepth')
    def setClearDepth(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, rendering.cameraSet, kwargs, 'clearDepth', val)

    @_f.addMelDocs('cameraSet', 'layer')
    def setLayer(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, rendering.cameraSet, kwargs, 'layer', val)

    @_f.addApiDocs(_api.MFnCameraSet, 'setLayerActive')
    def setLayerActive(self, index, isActive):
        # type: (int, bool) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([index, isActive], [('index', 'uint', 'in', None), ('isActive', 'bool', 'in', None)], self.isLayerActive, self.setLayerActive, ['index'])
        res = _f.getProxyResult(self, _api.MFnCameraSet, 'setLayerActive', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCameraSet, 'setLayerCamera')
    def setLayerCamera(self, index, camera):
        # type: (int, DagNode) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([index, camera], [('index', 'uint', 'in', None), ('camera', 'MDagPath', 'in', None)], self.getLayerCamera, self.setLayerCamera, ['index'])
        res = _f.getProxyResult(self, _api.MFnCameraSet, 'setLayerCamera', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCameraSet, 'setLayerClearDepthValue')
    def setLayerClearDepthValue(self, index, depth):
        # type: (int, bool) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([index, depth], [('index', 'uint', 'in', None), ('depth', 'bool', 'in', None)], self.getLayerClearDepthValue, self.setLayerClearDepthValue, ['index'])
        res = _f.getProxyResult(self, _api.MFnCameraSet, 'setLayerClearDepthValue', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCameraSet, 'setLayerOrder')
    def setLayerOrder(self, index, order):
        # type: (int, int) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([index, order], [('index', 'uint', 'in', None), ('order', 'int', 'in', None)], self.getLayerOrder, self.setLayerOrder, ['index'])
        res = _f.getProxyResult(self, _api.MFnCameraSet, 'setLayerOrder', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnCameraSet, 'setLayerSceneData')
    def setLayerSceneData(self, index, set):
        # type: (int, str | DependNode) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([index, set], [('index', 'uint', 'in', None), ('set', 'MObject', 'in', None)], self.getLayerSceneData, self.setLayerSceneData, ['index'])
        res = _f.getProxyResult(self, _api.MFnCameraSet, 'setLayerSceneData', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addMelDocs('cameraSet', 'objectSet')
    def setObjectSet(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, rendering.cameraSet, kwargs, 'objectSet', val)

    @_f.addMelDocs('cameraSet', 'order')
    def setOrder(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, rendering.cameraSet, kwargs, 'order', val)


class THcameraSet(CameraSet):
    __melnode__ = 'THcameraSet'
    __slots__ = ()


class CameraView(DependNode):
    __melcmd__ = staticmethod(rendering.cameraView)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'cameraView'
    __melnode__ = 'cameraView'
    __slots__ = ()

    @_f.addMelDocs('cameraView', 'addBookmark')
    def addBookmark(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, rendering.cameraView, kwargs, 'addBookmark', val)

    @_f.addMelDocs('cameraView', 'animate')
    def animate(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, rendering.cameraView, kwargs, 'animate', val)

    @_f.addMelDocs('cameraView', 'camera')
    def camera(self, val=True, **kwargs):
        # type: (general.PyNode | str, **Any) -> None
        return _f.asEdit(self, rendering.cameraView, kwargs, 'camera', val)

    @_f.deprecated
    def findAlias(self, alias):
        # type: (str) -> Tuple[bool, DependNode]
        do, final_do, outTypes = _f.getDoArgs([alias], [('alias', 'MString', 'in', None), ('attrObj', 'MObject', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'findAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affected instead.')
    def getAffectedAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affects instead.')
    def getAffectedByAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedByAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedByAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use listAliases instead.')
    def getAliasList(self):
        # type: () -> Tuple[bool, List[str]]
        do, final_do, outTypes = _f.getDoArgs([], [('strArray', 'MStringArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAliasList', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use inputs/outputs, or general.listConnections instead.')
    def getConnections(self):
        # type: () -> List[general.Attribute]
        do, final_do, outTypes = _f.getDoArgs([], [('array', 'MPlugArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getConnections', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.getAlias instead.')
    def plugsAlias(self, plug):
        # type: (str | general.Attribute) -> str
        do, final_do, outTypes = _f.getDoArgs([plug], [('plug', 'MPlug', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'plugsAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MString', None)
        return res

    @_f.addMelDocs('cameraView', 'removeBookmark')
    def removeBookmark(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, rendering.cameraView, kwargs, 'removeBookmark', val)

    @_f.deprecated('Use Attribute.setAlias instead.')
    def setAlias(self, alias, name, plug, add=True):
        # type: (str, str, str | general.Attribute, bool) -> bool
        do, final_do, outTypes = _f.getDoArgs([alias, name, plug, add], [('alias', 'MString', 'in', None), ('name', 'MString', 'in', None), ('plug', 'MPlug', 'in', None), ('add', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'setAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res

    @_f.addMelDocs('cameraView', 'setCamera')
    def setCamera(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, rendering.cameraView, kwargs, 'setCamera', val)

    @_f.addMelDocs('cameraView', 'setView')
    def setView(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, rendering.cameraView, kwargs, 'setView', val)


class CharacterMap(DependNode):
    __melcmd__ = staticmethod(animation.characterMap)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'characterMap'
    __melnode__ = 'characterMap'
    __slots__ = ()

    @_f.deprecated
    def findAlias(self, alias):
        # type: (str) -> Tuple[bool, DependNode]
        do, final_do, outTypes = _f.getDoArgs([alias], [('alias', 'MString', 'in', None), ('attrObj', 'MObject', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'findAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affected instead.')
    def getAffectedAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affects instead.')
    def getAffectedByAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedByAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedByAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use listAliases instead.')
    def getAliasList(self):
        # type: () -> Tuple[bool, List[str]]
        do, final_do, outTypes = _f.getDoArgs([], [('strArray', 'MStringArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAliasList', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use inputs/outputs, or general.listConnections instead.')
    def getConnections(self):
        # type: () -> List[general.Attribute]
        do, final_do, outTypes = _f.getDoArgs([], [('array', 'MPlugArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getConnections', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addMelDocs('characterMap', 'mapAttr')
    def getMapAttr(self, **kwargs):
        # type: (...) -> Tuple[str, str]
        res = _f.asQuery(self, animation.characterMap, kwargs, 'mapAttr')
        return res

    @_f.addMelDocs('characterMap', 'mapNode')
    def getMapNode(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.characterMap, kwargs, 'mapNode')
        return res

    @_f.addMelDocs('characterMap', 'mapping')
    def getMapping(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.characterMap, kwargs, 'mapping')
        return res

    @_f.addMelDocs('characterMap', 'proposedMapping')
    def getProposedMapping(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.characterMap, kwargs, 'proposedMapping')
        return res

    @_f.deprecated('Use Attribute.getAlias instead.')
    def plugsAlias(self, plug):
        # type: (str | general.Attribute) -> str
        do, final_do, outTypes = _f.getDoArgs([plug], [('plug', 'MPlug', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'plugsAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MString', None)
        return res

    @_f.deprecated('Use Attribute.setAlias instead.')
    def setAlias(self, alias, name, plug, add=True):
        # type: (str, str, str | general.Attribute, bool) -> bool
        do, final_do, outTypes = _f.getDoArgs([alias, name, plug, add], [('alias', 'MString', 'in', None), ('name', 'MString', 'in', None), ('plug', 'MPlug', 'in', None), ('add', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'setAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res

    @_f.addMelDocs('characterMap', 'mapAttr')
    def setMapAttr(self, val=True, **kwargs):
        # type: (Tuple[str, str], **Any) -> None
        return _f.asEdit(self, animation.characterMap, kwargs, 'mapAttr', val)

    @_f.addMelDocs('characterMap', 'unmapAttr')
    def unmapAttr(self, val=True, **kwargs):
        # type: (Tuple[str, str], **Any) -> None
        return _f.asEdit(self, animation.characterMap, kwargs, 'unmapAttr', val)


class CharacterOffset(DependNode):
    __melnode__ = 'characterOffset'
    __slots__ = ()

    @_f.deprecated
    def findAlias(self, alias):
        # type: (str) -> Tuple[bool, DependNode]
        do, final_do, outTypes = _f.getDoArgs([alias], [('alias', 'MString', 'in', None), ('attrObj', 'MObject', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'findAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affected instead.')
    def getAffectedAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affects instead.')
    def getAffectedByAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedByAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedByAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use listAliases instead.')
    def getAliasList(self):
        # type: () -> Tuple[bool, List[str]]
        do, final_do, outTypes = _f.getDoArgs([], [('strArray', 'MStringArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAliasList', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use inputs/outputs, or general.listConnections instead.')
    def getConnections(self):
        # type: () -> List[general.Attribute]
        do, final_do, outTypes = _f.getDoArgs([], [('array', 'MPlugArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getConnections', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.getAlias instead.')
    def plugsAlias(self, plug):
        # type: (str | general.Attribute) -> str
        do, final_do, outTypes = _f.getDoArgs([plug], [('plug', 'MPlug', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'plugsAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MString', None)
        return res

    @_f.deprecated('Use Attribute.setAlias instead.')
    def setAlias(self, alias, name, plug, add=True):
        # type: (str, str, str | general.Attribute, bool) -> bool
        do, final_do, outTypes = _f.getDoArgs([alias, name, plug, add], [('alias', 'MString', 'in', None), ('name', 'MString', 'in', None), ('plug', 'MPlug', 'in', None), ('add', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'setAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res


class Choice(DependNode):
    __melcmd__ = staticmethod(animation.choice)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'choice'
    __melnode__ = 'choice'
    __slots__ = ()

    @_f.deprecated
    def findAlias(self, alias):
        # type: (str) -> Tuple[bool, DependNode]
        do, final_do, outTypes = _f.getDoArgs([alias], [('alias', 'MString', 'in', None), ('attrObj', 'MObject', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'findAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affected instead.')
    def getAffectedAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affects instead.')
    def getAffectedByAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedByAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedByAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use listAliases instead.')
    def getAliasList(self):
        # type: () -> Tuple[bool, List[str]]
        do, final_do, outTypes = _f.getDoArgs([], [('strArray', 'MStringArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAliasList', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use inputs/outputs, or general.listConnections instead.')
    def getConnections(self):
        # type: () -> List[general.Attribute]
        do, final_do, outTypes = _f.getDoArgs([], [('array', 'MPlugArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getConnections', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addMelDocs('choice', 'index')
    def getIndex(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.choice, kwargs, 'index')
        return res

    @_f.addMelDocs('choice', 'selector')
    def getSelector(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.choice, kwargs, 'selector')
        return res

    @_f.deprecated('Use Attribute.getAlias instead.')
    def plugsAlias(self, plug):
        # type: (str | general.Attribute) -> str
        do, final_do, outTypes = _f.getDoArgs([plug], [('plug', 'MPlug', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'plugsAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MString', None)
        return res

    @_f.deprecated('Use Attribute.setAlias instead.')
    def setAlias(self, alias, name, plug, add=True):
        # type: (str, str, str | general.Attribute, bool) -> bool
        do, final_do, outTypes = _f.getDoArgs([alias, name, plug, add], [('alias', 'MString', 'in', None), ('name', 'MString', 'in', None), ('plug', 'MPlug', 'in', None), ('add', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'setAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res


class Chooser(DependNode):
    __melnode__ = 'chooser'
    __slots__ = ()

    @_f.deprecated
    def findAlias(self, alias):
        # type: (str) -> Tuple[bool, DependNode]
        do, final_do, outTypes = _f.getDoArgs([alias], [('alias', 'MString', 'in', None), ('attrObj', 'MObject', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'findAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affected instead.')
    def getAffectedAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affects instead.')
    def getAffectedByAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedByAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedByAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use listAliases instead.')
    def getAliasList(self):
        # type: () -> Tuple[bool, List[str]]
        do, final_do, outTypes = _f.getDoArgs([], [('strArray', 'MStringArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAliasList', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use inputs/outputs, or general.listConnections instead.')
    def getConnections(self):
        # type: () -> List[general.Attribute]
        do, final_do, outTypes = _f.getDoArgs([], [('array', 'MPlugArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getConnections', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.getAlias instead.')
    def plugsAlias(self, plug):
        # type: (str | general.Attribute) -> str
        do, final_do, outTypes = _f.getDoArgs([plug], [('plug', 'MPlug', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'plugsAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MString', None)
        return res

    @_f.deprecated('Use Attribute.setAlias instead.')
    def setAlias(self, alias, name, plug, add=True):
        # type: (str, str, str | general.Attribute, bool) -> bool
        do, final_do, outTypes = _f.getDoArgs([alias, name, plug, add], [('alias', 'MString', 'in', None), ('name', 'MString', 'in', None), ('plug', 'MPlug', 'in', None), ('add', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'setAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res


class Clamp(DependNode):
    __melnode__ = 'clamp'
    __slots__ = ()

    @_f.deprecated
    def findAlias(self, alias):
        # type: (str) -> Tuple[bool, DependNode]
        do, final_do, outTypes = _f.getDoArgs([alias], [('alias', 'MString', 'in', None), ('attrObj', 'MObject', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'findAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affected instead.')
    def getAffectedAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affects instead.')
    def getAffectedByAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedByAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedByAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use listAliases instead.')
    def getAliasList(self):
        # type: () -> Tuple[bool, List[str]]
        do, final_do, outTypes = _f.getDoArgs([], [('strArray', 'MStringArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAliasList', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use inputs/outputs, or general.listConnections instead.')
    def getConnections(self):
        # type: () -> List[general.Attribute]
        do, final_do, outTypes = _f.getDoArgs([], [('array', 'MPlugArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getConnections', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.getAlias instead.')
    def plugsAlias(self, plug):
        # type: (str | general.Attribute) -> str
        do, final_do, outTypes = _f.getDoArgs([plug], [('plug', 'MPlug', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'plugsAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MString', None)
        return res

    @_f.deprecated('Use Attribute.setAlias instead.')
    def setAlias(self, alias, name, plug, add=True):
        # type: (str, str, str | general.Attribute, bool) -> bool
        do, final_do, outTypes = _f.getDoArgs([alias, name, plug, add], [('alias', 'MString', 'in', None), ('name', 'MString', 'in', None), ('plug', 'MPlug', 'in', None), ('add', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'setAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res


class ClipLibrary(DependNode):
    __melnode__ = 'clipLibrary'
    __slots__ = ()

    @_f.deprecated
    def findAlias(self, alias):
        # type: (str) -> Tuple[bool, DependNode]
        do, final_do, outTypes = _f.getDoArgs([alias], [('alias', 'MString', 'in', None), ('attrObj', 'MObject', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'findAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affected instead.')
    def getAffectedAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affects instead.')
    def getAffectedByAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedByAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedByAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use listAliases instead.')
    def getAliasList(self):
        # type: () -> Tuple[bool, List[str]]
        do, final_do, outTypes = _f.getDoArgs([], [('strArray', 'MStringArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAliasList', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use inputs/outputs, or general.listConnections instead.')
    def getConnections(self):
        # type: () -> List[general.Attribute]
        do, final_do, outTypes = _f.getDoArgs([], [('array', 'MPlugArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getConnections', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.getAlias instead.')
    def plugsAlias(self, plug):
        # type: (str | general.Attribute) -> str
        do, final_do, outTypes = _f.getDoArgs([plug], [('plug', 'MPlug', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'plugsAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MString', None)
        return res

    @_f.deprecated('Use Attribute.setAlias instead.')
    def setAlias(self, alias, name, plug, add=True):
        # type: (str, str, str | general.Attribute, bool) -> bool
        do, final_do, outTypes = _f.getDoArgs([alias, name, plug, add], [('alias', 'MString', 'in', None), ('name', 'MString', 'in', None), ('plug', 'MPlug', 'in', None), ('add', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'setAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res


class ClipScheduler(DependNode):
    __melnode__ = 'clipScheduler'
    __slots__ = ()

    @_f.deprecated
    def findAlias(self, alias):
        # type: (str) -> Tuple[bool, DependNode]
        do, final_do, outTypes = _f.getDoArgs([alias], [('alias', 'MString', 'in', None), ('attrObj', 'MObject', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'findAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affected instead.')
    def getAffectedAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affects instead.')
    def getAffectedByAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedByAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedByAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use listAliases instead.')
    def getAliasList(self):
        # type: () -> Tuple[bool, List[str]]
        do, final_do, outTypes = _f.getDoArgs([], [('strArray', 'MStringArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAliasList', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use inputs/outputs, or general.listConnections instead.')
    def getConnections(self):
        # type: () -> List[general.Attribute]
        do, final_do, outTypes = _f.getDoArgs([], [('array', 'MPlugArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getConnections', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.getAlias instead.')
    def plugsAlias(self, plug):
        # type: (str | general.Attribute) -> str
        do, final_do, outTypes = _f.getDoArgs([plug], [('plug', 'MPlug', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'plugsAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MString', None)
        return res

    @_f.deprecated('Use Attribute.setAlias instead.')
    def setAlias(self, alias, name, plug, add=True):
        # type: (str, str, str | general.Attribute, bool) -> bool
        do, final_do, outTypes = _f.getDoArgs([alias, name, plug, add], [('alias', 'MString', 'in', None), ('name', 'MString', 'in', None), ('plug', 'MPlug', 'in', None), ('add', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'setAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res


class ClipToGhostData(DependNode):
    __melnode__ = 'clipToGhostData'
    __slots__ = ()


class ClosestPointOnMesh(DependNode):
    __melnode__ = 'closestPointOnMesh'
    __slots__ = ()

    @_f.deprecated
    def findAlias(self, alias):
        # type: (str) -> Tuple[bool, DependNode]
        do, final_do, outTypes = _f.getDoArgs([alias], [('alias', 'MString', 'in', None), ('attrObj', 'MObject', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'findAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affected instead.')
    def getAffectedAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affects instead.')
    def getAffectedByAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedByAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedByAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use listAliases instead.')
    def getAliasList(self):
        # type: () -> Tuple[bool, List[str]]
        do, final_do, outTypes = _f.getDoArgs([], [('strArray', 'MStringArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAliasList', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use inputs/outputs, or general.listConnections instead.')
    def getConnections(self):
        # type: () -> List[general.Attribute]
        do, final_do, outTypes = _f.getDoArgs([], [('array', 'MPlugArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getConnections', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.getAlias instead.')
    def plugsAlias(self, plug):
        # type: (str | general.Attribute) -> str
        do, final_do, outTypes = _f.getDoArgs([plug], [('plug', 'MPlug', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'plugsAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MString', None)
        return res

    @_f.deprecated('Use Attribute.setAlias instead.')
    def setAlias(self, alias, name, plug, add=True):
        # type: (str, str, str | general.Attribute, bool) -> bool
        do, final_do, outTypes = _f.getDoArgs([alias, name, plug, add], [('alias', 'MString', 'in', None), ('name', 'MString', 'in', None), ('plug', 'MPlug', 'in', None), ('add', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'setAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res


class ColorManagementGlobals(DependNode):
    __melnode__ = 'colorManagementGlobals'
    __slots__ = ()


class ColorProfile(DependNode):
    __melnode__ = 'colorProfile'
    __slots__ = ()

    @_f.deprecated
    def findAlias(self, alias):
        # type: (str) -> Tuple[bool, DependNode]
        do, final_do, outTypes = _f.getDoArgs([alias], [('alias', 'MString', 'in', None), ('attrObj', 'MObject', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'findAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affected instead.')
    def getAffectedAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affects instead.')
    def getAffectedByAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedByAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedByAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use listAliases instead.')
    def getAliasList(self):
        # type: () -> Tuple[bool, List[str]]
        do, final_do, outTypes = _f.getDoArgs([], [('strArray', 'MStringArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAliasList', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use inputs/outputs, or general.listConnections instead.')
    def getConnections(self):
        # type: () -> List[general.Attribute]
        do, final_do, outTypes = _f.getDoArgs([], [('array', 'MPlugArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getConnections', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.getAlias instead.')
    def plugsAlias(self, plug):
        # type: (str | general.Attribute) -> str
        do, final_do, outTypes = _f.getDoArgs([plug], [('plug', 'MPlug', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'plugsAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MString', None)
        return res

    @_f.deprecated('Use Attribute.setAlias instead.')
    def setAlias(self, alias, name, plug, add=True):
        # type: (str, str, str | general.Attribute, bool) -> bool
        do, final_do, outTypes = _f.getDoArgs([alias, name, plug, add], [('alias', 'MString', 'in', None), ('name', 'MString', 'in', None), ('plug', 'MPlug', 'in', None), ('add', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'setAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res


class CombinationShape(DependNode):
    __melcmd__ = staticmethod(animation.combinationShape)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'combinationShape'
    __melnode__ = 'combinationShape'
    __slots__ = ()

    @_f.addMelDocs('combinationShape', 'allDrivers')
    def getAllDrivers(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.combinationShape, kwargs, 'allDrivers')
        return res

    @_f.addMelDocs('combinationShape', 'combineMethod')
    def getCombineMethod(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, animation.combinationShape, kwargs, 'combineMethod')
        return res

    @_f.addMelDocs('combinationShape', 'exist')
    def getExist(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.combinationShape, kwargs, 'exist')
        return res

    @_f.addMelDocs('combinationShape', 'combineMethod')
    def setCombineMethod(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, animation.combinationShape, kwargs, 'combineMethod', val)


class CompactPlugArrayTest(DependNode):
    __melnode__ = 'compactPlugArrayTest'
    __slots__ = ()


class ComponentFalloff(DependNode):
    __melnode__ = 'componentFalloff'
    __slots__ = ()


class ComponentMatch(DependNode):
    __melnode__ = 'componentMatch'
    __slots__ = ()


class ComposeMatrix(DependNode):
    __melnode__ = 'composeMatrix'
    __slots__ = ()


class Condition(DependNode):
    __melnode__ = 'condition'
    __slots__ = ()

    @_f.deprecated
    def findAlias(self, alias):
        # type: (str) -> Tuple[bool, DependNode]
        do, final_do, outTypes = _f.getDoArgs([alias], [('alias', 'MString', 'in', None), ('attrObj', 'MObject', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'findAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affected instead.')
    def getAffectedAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.affects instead.')
    def getAffectedByAttributes(self, attr):
        # type: (str | DependNode) -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([attr], [('attr', 'MObject', 'in', None), ('affectedByAttributes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAffectedByAttributes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use listAliases instead.')
    def getAliasList(self):
        # type: () -> Tuple[bool, List[str]]
        do, final_do, outTypes = _f.getDoArgs([], [('strArray', 'MStringArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getAliasList', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use inputs/outputs, or general.listConnections instead.')
    def getConnections(self):
        # type: () -> List[general.Attribute]
        do, final_do, outTypes = _f.getDoArgs([], [('array', 'MPlugArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'getConnections', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use Attribute.getAlias instead.')
    def plugsAlias(self, plug):
        # type: (str | general.Attribute) -> str
        do, final_do, outTypes = _f.getDoArgs([plug], [('plug', 'MPlug', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'plugsAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MString', None)
        return res

    @_f.deprecated('Use Attribute.setAlias instead.')
    def setAlias(self, alias, name, plug, add=True):
        # type: (str, str, str | general.Attribute, bool) -> bool
        do, final_do, outTypes = _f.getDoArgs([alias, name, plug, add], [('alias', 'MString', 'in', None), ('name', 'MString', 'in', None), ('plug', 'MPlug', 'in', None), ('add', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnDependencyNode, 'setAlias', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return res


class Container(ContainerBase):
    __apicls__ = _api.MFnContainerNode
    __melcmd__ = staticmethod(general.container)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'container'
    __melnode__ = 'container'
    __slots__ = ()
    MPublishNodeType = Enum('MPublishNodeType', [('parentAnchor', 0), ('kParentAnchor', 0), ('childAnchor', 1), ('kChildAnchor', 1), ('generic', 2), ('kGeneric', 2)], multiKeys=True)

    @_f.addMelDocs('container', 'addNode')
    def addNode(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, general.container, kwargs, 'addNode', val)

    @_f.addApiDocs(_api.MFnContainerNode, 'clear')
    def clear(self):
        # type: () -> None
        res = _f.getProxyResult(self, _api.MFnContainerNode, 'clear')
        return res

    @_f.addMelDocs('container', 'force')
    def force(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, general.container, kwargs, 'force', val)

    @_f.addMelDocs('container', 'asset')
    def getAsset(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, general.container, kwargs, 'asset')
        return res

    @_f.addMelDocs('container', 'assetMember')
    def getAssetMember(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, general.container, kwargs, 'assetMember')
        return res

    @_f.addMelDocs('container', 'bindAttr')
    def getBindAttr(self, **kwargs):
        # type: (...) -> Tuple[str, str]
        res = _f.asQuery(self, general.container, kwargs, 'bindAttr')
        return res

    @_f.addMelDocs('container', 'connectionList')
    def getConnectionList(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, general.container, kwargs, 'connectionList')
        return res

    @classmethod
    @_f.addApiDocs(_api.MFnContainerNode, 'getCurrentAsMObject')
    def getCurrent(self):
        # type: () -> DependNode
        res = _api.MFnContainerNode.getCurrentAsMObject()
        return _f.ApiArgUtil._castResult(self, res, 'MObject', None)

    @_f.addMelDocs('container', 'fileName')
    def getFileName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, general.container, kwargs, 'fileName')
        return res

    @_f.addMelDocs('container', 'findContainer')
    def getFindContainer(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, general.container, kwargs, 'findContainer')
        return res

    @_f.addMelDocs('container', 'isContainer')
    def getIsContainer(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, general.container, kwargs, 'isContainer')
        return res

    @_f.addApiDocs(_api.MFnContainerNode, 'getMembers')
    def getMembers(self):
        # type: () -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([], [('members', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnContainerNode, 'getMembers', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addMelDocs('container', 'nodeList')
    def getNodeList(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, general.container, kwargs, 'nodeList')
        return res

    @_f.addApiDocs(_api.MFnContainerNode, 'getParentContainer')
    def getParentContainer(self):
        # type: () -> DependNode
        do, final_do, outTypes = _f.getDoArgs([], [('parent', 'MObject', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnContainerNode, 'getParentContainer', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addMelDocs('container', 'publishAsChild')
    def getPublishAsChild(self, **kwargs):
        # type: (...) -> Tuple[str, str]
        res = _f.asQuery(self, general.container, kwargs, 'publishAsChild')
        return res

    @_f.addMelDocs('container', 'publishAsParent')
    def getPublishAsParent(self, **kwargs):
        # type: (...) -> Tuple[str, str]
        res = _f.asQuery(self, general.container, kwargs, 'publishAsParent')
        return res

    @_f.addMelDocs('container', 'publishAsRoot')
    def getPublishAsRoot(self, **kwargs):
        # type: (...) -> Tuple[str, bool]
        res = _f.asQuery(self, general.container, kwargs, 'publishAsRoot')
        return res

    @_f.addMelDocs('container', 'publishAttr')
    def getPublishAttr(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, general.container, kwargs, 'publishAttr')
        return res

    @_f.addMelDocs('container', 'publishName')
    def getPublishName(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, general.container, kwargs, 'publishName')
        return res

    @_f.addApiDocs(_api.MFnContainerNode, 'getPublishedNames')
    def getPublishedNames(self, unboundOnly):
        # type: (bool) -> List[str]
        do, final_do, outTypes = _f.getDoArgs([unboundOnly], [('publishedNames', 'MStringArray', 'out', None), ('unboundOnly', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnContainerNode, 'getPublishedNames', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnContainerNode, 'getPublishedNodes')
    def getPublishedNodes(self, type):
        # type: (ContainerNode.MPublishNodeType) -> Tuple[List[str], List[DependNode]]
        do, final_do, outTypes = _f.getDoArgs([type], [('type', ('MFnContainerNode', 'MPublishNodeType'), 'in', None), ('publishedNames', 'MStringArray', 'out', None), ('nodes', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnContainerNode, 'getPublishedNodes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnContainerNode, 'getPublishedPlugs')
    def getPublishedPlugs(self):
        # type: () -> Tuple[List[general.Attribute], List[str]]
        do, final_do, outTypes = _f.getDoArgs([], [('publishedPlugs', 'MPlugArray', 'out', None), ('publishedNames', 'MStringArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnContainerNode, 'getPublishedPlugs', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnContainerNode, 'getRootTransform')
    def getRootTransform(self):
        # type: () -> DependNode
        do, final_do, outTypes = _f.getDoArgs([], [('root', 'MObject', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnContainerNode, 'getRootTransform', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnContainerNode, 'getSubcontainers')
    def getSubcontainers(self):
        # type: () -> List[DependNode]
        do, final_do, outTypes = _f.getDoArgs([], [('subcontainers', 'MObjectArray', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnContainerNode, 'getSubcontainers', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addMelDocs('container', 'type')
    def getType(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, general.container, kwargs, 'type')
        return res

    @_f.addMelDocs('container', 'unbindAttr')
    def getUnbindAttr(self, **kwargs):
        # type: (...) -> Tuple[str, str]
        res = _f.asQuery(self, general.container, kwargs, 'unbindAttr')
        return res

    @_f.addMelDocs('container', 'unsortedOrder')
    def getUnsortedOrder(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, general.container, kwargs, 'unsortedOrder')
        return res

    @_f.addMelDocs('container', 'includeHierarchyAbove')
    def includeHierarchyAbove(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, general.container, kwargs, 'includeHierarchyAbove', val)

    @_f.addMelDocs('container', 'includeHierarchyBelow')
    def includeHierarchyBelow(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, general.container, kwargs, 'includeHierarchyBelow', val)

    @_f.addMelDocs('container', 'includeNetwork')
    def includeNetwork(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, general.container, kwargs, 'includeNetwork', val)

    @_f.addMelDocs('container', 'includeNetworkDetails')
    def includeNetworkDetails(self, val=True, **kwargs):
        # type: (str | List[str], **Any) -> None
        return _f.asEdit(self, general.container, kwargs, 'includeNetworkDetails', val)

    @_f.addMelDocs('container', 'includeShaders')
    def includeShaders(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, general.container, kwargs, 'includeShaders', val)

    @_f.addMelDocs('container', 'includeShapes')
    def includeShapes(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, general.container, kwargs, 'includeShapes', val)

    @_f.addMelDocs('container', 'includeTransform')
    def includeTransform(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, general.container, kwargs, 'includeTransform', val)

    @_f.addApiDocs(_api.MFnContainerNode, 'isCurrent')
    def isCurrent(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnContainerNode, 'isCurrent')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnContainerNode, 'makeCurrent')
    def makeCurrent(self, isCurrent=True):
        # type: (bool) -> None
        do, final_do, outTypes = _f.getDoArgs([isCurrent], [('isCurrent', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnContainerNode, 'makeCurrent', final_do)
        return res

    @_f.addMelDocs('container', 'nodeNamePrefix')
    def nodeNamePrefix(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, general.container, kwargs, 'nodeNamePrefix', val)

    @_f.addMelDocs('container', 'publishAndBind')
    def publishAndBind(self, val=True, **kwargs):
        # type: (Tuple[str, str], **Any) -> None
        return _f.asEdit(self, general.container, kwargs, 'publishAndBind', val)

    @_f.addMelDocs('container', 'publishConnections')
    def publishConnections(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, general.container, kwargs, 'publishConnections', val)

    @_f.addMelDocs('container', 'removeContainer')
    def removeContainer(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, general.container, kwargs, 'removeContainer', val)

    @_f.addMelDocs('container', 'removeNode')
    def removeNode(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, general.container, kwargs, 'removeNode', val)

    @_f.addMelDocs('container', 'bindAttr')
    def setBindAttr(self, val=True, **kwargs):
        # type: (Tuple[str, str], **Any) -> None
        return _f.asEdit(self, general.container, kwargs, 'bindAttr', val)

    @_f.addMelDocs('container', 'current')
    def setCurrent(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, general.container, kwargs, 'current', val)

    @_f.addMelDocs('container', 'publishAsChild')
    def setPublishAsChild(self, val=True, **kwargs):
        # type: (Tuple[str, str], **Any) -> None
        return _f.asEdit(self, general.container, kwargs, 'publishAsChild', val)

    @_f.addMelDocs('container', 'publishAsParent')
    def setPublishAsParent(self, val=True, **kwargs):
        # type: (Tuple[str, str], **Any) -> None
        return _f.asEdit(self, general.container, kwargs, 'publishAsParent', val)

    @_f.addMelDocs('container', 'publishAsRoot')
    def setPublishAsRoot(self, val=True, **kwargs):
        # type: (Tuple[str, bool | int], **Any) -> None
        return _f.asEdit(self, general.container, kwargs, 'publishAsRoot', val)

    @_f.addMelDocs('container', 'publishName')
    def setPublishName(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, general.container, kwargs, 'publishName', val)

    @_f.addMelDocs('container', 'unbindAttr')
    def setUnbindAttr(self, val=True, **kwargs):
        # type: (Tuple[str, str], **Any) -> None
        return _f.asEdit(self, general.container, kwargs, 'unbindAttr', val)

    @_f.addMelDocs('container', 'unbindAndUnpublish')
    def unbindAndUnpublish(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, general.container, kwargs, 'unbindAndUnpublish', val)

    @_f.addMelDocs('container', 'unbindChild')
    def unbindChild(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, general.container, kwargs, 'unbindChild', val)

    @_f.addMelDocs('container', 'unbindParent')
    def unbindParent(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, general.container, kwargs, 'unbindParent', val)

    @_f.addMelDocs('container', 'unpublishChild')
    def unpublishChild(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, general.container, kwargs, 'unpublishChild', val)

    @_f.addMelDocs('container', 'unpublishName')
    def unpublishName(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, general.container, kwargs, 'unpublishName', val)

    @_f.addMelDocs('container', 'unpublishParent')
    def unpublishParent(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, general.container, kwargs, 'unpublishParent', val)


class BaseLattice(Shape):
    __melnode__ = 'baseLattice'
    __slots__ = ()


class StereoRigCamera(Camera):
    __melnode__ = 'stereoRigCamera'
    __slots__ = ()


class ClusterHandle(Shape):
    __melnode__ = 'clusterHandle'
    __slots__ = ()


class DeformFunc(Shape):
    __melnode__ = 'deformFunc'
    __slots__ = ()


class DeformBend(DeformFunc):
    __melnode__ = 'deformBend'
    __slots__ = ()


class DeformFlare(DeformFunc):
    __melnode__ = 'deformFlare'
    __slots__ = ()


class DeformSine(DeformFunc):
    __melnode__ = 'deformSine'
    __slots__ = ()


class DeformSquash(DeformFunc):
    __melnode__ = 'deformSquash'
    __slots__ = ()


class DeformTwist(DeformFunc):
    __melnode__ = 'deformTwist'
    __slots__ = ()


class DeformWave(DeformFunc):
    __melnode__ = 'deformWave'
    __slots__ = ()


class DimensionShape(Shape):
    __melnode__ = 'dimensionShape'
    __slots__ = ()


class AngleDimension(DimensionShape):
    __melnode__ = 'angleDimension'
    __slots__ = ()


class AnnotationShape(DimensionShape):
    __melcmd__ = staticmethod(windows.annotate)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'annotate'
    __melnode__ = 'annotationShape'
    __slots__ = ()


class DistanceDimShape(DimensionShape):
    __melnode__ = 'distanceDimShape'
    __slots__ = ()


class NurbsDimShape(DimensionShape):
    __melnode__ = 'nurbsDimShape'
    __slots__ = ()


class ArcLengthDimension(NurbsDimShape):
    __melcmd__ = staticmethod(modeling.arcLengthDimension)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'arcLengthDimension'
    __melnode__ = 'arcLengthDimension'
    __slots__ = ()


class ParamDimension(NurbsDimShape):
    __melcmd__ = staticmethod(general.paramDimension)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'paramDimension'
    __melnode__ = 'paramDimension'
    __slots__ = ()


class DynHolder(Shape):
    __melnode__ = 'dynHolder'
    __slots__ = ()


class DynamicConstraint(Shape):
    __melnode__ = 'dynamicConstraint'
    __slots__ = ()


class FlexorShape(Shape):
    __melnode__ = 'flexorShape'
    __slots__ = ()


class ClusterFlexorShape(FlexorShape):
    __melnode__ = 'clusterFlexorShape'
    __slots__ = ()


class Follicle(Shape):
    __melnode__ = 'follicle'
    __slots__ = ()


class GeoConnectable(Shape):
    __melnode__ = 'geoConnectable'
    __slots__ = ()


class BezierCurve(NurbsCurve):
    __melnode__ = 'bezierCurve'
    __slots__ = ()


class SnapshotShape(ControlPoint):
    __melnode__ = 'snapshotShape'
    __slots__ = ()


class MotionTrailShape(SnapshotShape):
    __melnode__ = 'motionTrailShape'
    __slots__ = ()


class THsurfaceShape(SurfaceShape):
    __melnode__ = 'THsurfaceShape'
    __slots__ = ()


class FluidShape(SurfaceShape):
    __apicls__ = _api.MFnFluid
    __melnode__ = 'fluidShape'
    __slots__ = ()
    ColorMethod = Enum('ColorMethod', [('useShadingColor', 0), ('kUseShadingColor', 0), ('staticColorGrid', 1), ('kStaticColorGrid', 1), ('dynamicColorGrid', 2), ('kDynamicColorGrid', 2)], multiKeys=True)
    CoordinateMethod = Enum('CoordinateMethod', [('fixed', 0), ('kFixed', 0), ('grid', 1), ('kGrid', 1)], multiKeys=True)
    FalloffMethod = Enum('FalloffMethod', [('noFalloffGrid', 0), ('kNoFalloffGrid', 0), ('staticFalloffGrid', 1), ('kStaticFalloffGrid', 1)], multiKeys=True)
    FluidGradient = Enum('FluidGradient', [('constant', 0), ('kConstant', 0), ('XGradient', 1), ('kXGradient', 1), ('YGradient', 2), ('kYGradient', 2), ('ZGradient', 3), ('kZGradient', 3), ('negXGradient', 4), ('kNegXGradient', 4), ('negYGradient', 5), ('kNegYGradient', 5), ('negZGradient', 6), ('kNegZGradient', 6), ('centerGradient', 7), ('kCenterGradient', 7)], multiKeys=True)
    FluidMethod = Enum('FluidMethod', [('zero', 0), ('kZero', 0), ('staticGrid', 1), ('kStaticGrid', 1), ('dynamicGrid', 2), ('kDynamicGrid', 2), ('gradient', 3), ('kGradient', 3)], multiKeys=True)

    @_f.addApiDocs(_api.MFnFluid, 'create2D')
    def create2D(self, Xres, Yres, Xdim, Ydim, parent='MObject::kNullObj'):
        # type: (int, int, float, float, str | DependNode) -> DependNode
        do, final_do, outTypes = _f.getDoArgs([Xres, Yres, Xdim, Ydim, parent], [('Xres', 'uint', 'in', None), ('Yres', 'uint', 'in', None), ('Xdim', 'double', 'in', None), ('Ydim', 'double', 'in', None), ('parent', 'MObject', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnFluid, 'create2D', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MObject', None)
        return res

    @_f.addApiDocs(_api.MFnFluid, 'create3D')
    def create3D(self, Xres, Yres, Zres, Xdim, Ydim, Zdim, parent='MObject::kNullObj'):
        # type: (int, int, int, float, float, float, str | DependNode) -> DependNode
        do, final_do, outTypes = _f.getDoArgs([Xres, Yres, Zres, Xdim, Ydim, Zdim, parent], [('Xres', 'uint', 'in', None), ('Yres', 'uint', 'in', None), ('Zres', 'uint', 'in', None), ('Xdim', 'double', 'in', None), ('Ydim', 'double', 'in', None), ('Zdim', 'double', 'in', None), ('parent', 'MObject', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnFluid, 'create3D', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MObject', None)
        return res

    @_f.addApiDocs(_api.MFnFluid, 'density')
    def density(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnFluid, 'density')
        return _f.ApiArgUtil._castResult(self, res, 'float', None)

    @_f.addApiDocs(_api.MFnFluid, 'emitIntoArrays')
    def emitIntoArrays(self, val, x, y, z, density, heat, fuel, doColor, emitColor):
        # type: (float, int, int, int, float, float, float, bool, datatypes.Color) -> None
        do, final_do, outTypes = _f.getDoArgs([val, x, y, z, density, heat, fuel, doColor, emitColor], [('val', 'float', 'in', None), ('x', 'int', 'in', None), ('y', 'int', 'in', None), ('z', 'int', 'in', None), ('density', 'float', 'in', None), ('heat', 'float', 'in', None), ('fuel', 'float', 'in', None), ('doColor', 'bool', 'in', None), ('emitColor', 'MColor', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnFluid, 'emitIntoArrays', final_do)
        return res

    @_f.addApiDocs(_api.MFnFluid, 'expandToInclude')
    def expandToInclude(self, min, max):
        # type: (datatypes.Point, datatypes.Point) -> None
        do, final_do, outTypes = _f.getDoArgs([min, max], [('min', 'MPoint', 'in', None), ('max', 'MPoint', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnFluid, 'expandToInclude', final_do)
        return res

    @_f.addApiDocs(_api.MFnFluid, 'falloff')
    def falloff(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnFluid, 'falloff')
        return _f.ApiArgUtil._castResult(self, res, 'float', None)

    @_f.addApiDocs(_api.MFnFluid, 'fuel')
    def fuel(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnFluid, 'fuel')
        return _f.ApiArgUtil._castResult(self, res, 'float', None)

    @_f.addApiDocs(_api.MFnFluid, 'getColorMode')
    def getColorMode(self):
        # type: () -> Fluid.ColorMethod
        do, final_do, outTypes = _f.getDoArgs([], [('method', ('MFnFluid', 'ColorMethod'), 'out', None)])
        res = _f.getProxyResult(self, _api.MFnFluid, 'getColorMode', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnFluid, 'getColors')
    def getColors(self):
        # type: () -> Tuple[float, float, float]
        do, final_do, outTypes = _f.getDoArgs([], [('r', 'float', 'out', None), ('g', 'float', 'out', None), ('b', 'float', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnFluid, 'getColors', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnFluid, 'getCoordinateMode')
    def getCoordinateMode(self):
        # type: () -> Fluid.CoordinateMethod
        do, final_do, outTypes = _f.getDoArgs([], [('method', ('MFnFluid', 'CoordinateMethod'), 'out', None)])
        res = _f.getProxyResult(self, _api.MFnFluid, 'getCoordinateMode', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnFluid, 'getCoordinates')
    def getCoordinates(self):
        # type: () -> Tuple[float, float, float]
        do, final_do, outTypes = _f.getDoArgs([], [('u', 'float', 'out', None), ('v', 'float', 'out', None), ('w', 'float', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnFluid, 'getCoordinates', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnFluid, 'getDensityMode')
    def getDensityMode(self):
        # type: () -> Tuple[Fluid.FluidMethod, Fluid.FluidGradient]
        do, final_do, outTypes = _f.getDoArgs([], [('method', ('MFnFluid', 'FluidMethod'), 'out', None), ('gradient', ('MFnFluid', 'FluidGradient'), 'out', None)])
        res = _f.getProxyResult(self, _api.MFnFluid, 'getDensityMode', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnFluid, 'getDimensions')
    def getDimensions(self):
        # type: () -> Tuple[float, float, float]
        do, final_do, outTypes = _f.getDoArgs([], [('Xdim', 'double', 'out', None), ('Ydim', 'double', 'out', None), ('Zdim', 'double', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnFluid, 'getDimensions', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnFluid, 'getFalloffMode')
    def getFalloffMode(self):
        # type: () -> Fluid.FalloffMethod
        do, final_do, outTypes = _f.getDoArgs([], [('method', ('MFnFluid', 'FalloffMethod'), 'out', None)])
        res = _f.getProxyResult(self, _api.MFnFluid, 'getFalloffMode', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnFluid, 'getForceAtPoint')
    def getForceAtPoint(self, point, velocity, mass, deltaTime=0.041666666666666664):
        # type: (List[datatypes.Point], List[datatypes.Vector], List[float], float) -> List[datatypes.Vector]
        do, final_do, outTypes = _f.getDoArgs([point, velocity, mass, deltaTime], [('point', 'MPointArray', 'in', None), ('velocity', 'MVectorArray', 'in', None), ('mass', 'MDoubleArray', 'in', None), ('force', 'MVectorArray', 'out', None), ('deltaTime', 'double', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnFluid, 'getForceAtPoint', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnFluid, 'getFuelMode')
    def getFuelMode(self):
        # type: () -> Tuple[Fluid.FluidMethod, Fluid.FluidGradient]
        do, final_do, outTypes = _f.getDoArgs([], [('method', ('MFnFluid', 'FluidMethod'), 'out', None), ('gradient', ('MFnFluid', 'FluidGradient'), 'out', None)])
        res = _f.getProxyResult(self, _api.MFnFluid, 'getFuelMode', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnFluid, 'getResolution')
    def getResolution(self):
        # type: () -> Tuple[int, int, int]
        do, final_do, outTypes = _f.getDoArgs([], [('Xres', 'uint', 'out', None), ('Yres', 'uint', 'out', None), ('Zres', 'uint', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnFluid, 'getResolution', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnFluid, 'getTemperatureMode')
    def getTemperatureMode(self):
        # type: () -> Tuple[Fluid.FluidMethod, Fluid.FluidGradient]
        do, final_do, outTypes = _f.getDoArgs([], [('method', ('MFnFluid', 'FluidMethod'), 'out', None), ('gradient', ('MFnFluid', 'FluidGradient'), 'out', None)])
        res = _f.getProxyResult(self, _api.MFnFluid, 'getTemperatureMode', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnFluid, 'getVelocity')
    def getVelocity(self):
        # type: () -> Tuple[float, float, float]
        do, final_do, outTypes = _f.getDoArgs([], [('Xvel', 'float', 'out', None), ('Yvel', 'float', 'out', None), ('Zvel', 'float', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnFluid, 'getVelocity', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnFluid, 'getVelocityMode')
    def getVelocityMode(self):
        # type: () -> Tuple[Fluid.FluidMethod, Fluid.FluidGradient]
        do, final_do, outTypes = _f.getDoArgs([], [('method', ('MFnFluid', 'FluidMethod'), 'out', None), ('gradient', ('MFnFluid', 'FluidGradient'), 'out', None)])
        res = _f.getProxyResult(self, _api.MFnFluid, 'getVelocityMode', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnFluid, 'gridSize')
    def gridSize(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnFluid, 'gridSize')
        return _f.ApiArgUtil._castResult(self, res, 'uint', None)

    @_f.addApiDocs(_api.MFnFluid, 'isAutoResize')
    def isAutoResize(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnFluid, 'isAutoResize')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnFluid, 'isResizeToEmitter')
    def isResizeToEmitter(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnFluid, 'isResizeToEmitter')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnFluid, 'pressure')
    def pressure(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnFluid, 'pressure')
        return _f.ApiArgUtil._castResult(self, res, 'float', None)

    @_f.addApiDocs(_api.MFnFluid, 'setColorMode')
    def setColorMode(self, method):
        # type: (Fluid.ColorMethod) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([method], [('method', ('MFnFluid', 'ColorMethod'), 'in', None)], self.getColorMode, self.setColorMode, [])
        res = _f.getProxyResult(self, _api.MFnFluid, 'setColorMode', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnFluid, 'setCoordinateMode')
    def setCoordinateMode(self, method):
        # type: (Fluid.CoordinateMethod) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([method], [('method', ('MFnFluid', 'CoordinateMethod'), 'in', None)], self.getCoordinateMode, self.setCoordinateMode, [])
        res = _f.getProxyResult(self, _api.MFnFluid, 'setCoordinateMode', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnFluid, 'setDensityMode')
    def setDensityMode(self, method, gradient):
        # type: (Fluid.FluidMethod, Fluid.FluidGradient) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([method, gradient], [('method', ('MFnFluid', 'FluidMethod'), 'in', None), ('gradient', ('MFnFluid', 'FluidGradient'), 'in', None)], self.getDensityMode, self.setDensityMode, [])
        res = _f.getProxyResult(self, _api.MFnFluid, 'setDensityMode', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnFluid, 'setFalloffMode')
    def setFalloffMode(self, method):
        # type: (Fluid.FalloffMethod) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([method], [('method', ('MFnFluid', 'FalloffMethod'), 'in', None)], self.getFalloffMode, self.setFalloffMode, [])
        res = _f.getProxyResult(self, _api.MFnFluid, 'setFalloffMode', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnFluid, 'setFuelMode')
    def setFuelMode(self, method, gradient):
        # type: (Fluid.FluidMethod, Fluid.FluidGradient) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([method, gradient], [('method', ('MFnFluid', 'FluidMethod'), 'in', None), ('gradient', ('MFnFluid', 'FluidGradient'), 'in', None)], self.getFuelMode, self.setFuelMode, [])
        res = _f.getProxyResult(self, _api.MFnFluid, 'setFuelMode', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnFluid, 'setSize')
    def setSize(self, Xres, Yres, Zres, Xdim, Ydim, Zdim, resample):
        # type: (int, int, int, float, float, float, bool) -> None
        do, final_do, outTypes = _f.getDoArgs([Xres, Yres, Zres, Xdim, Ydim, Zdim, resample], [('Xres', 'uint', 'in', None), ('Yres', 'uint', 'in', None), ('Zres', 'uint', 'in', None), ('Xdim', 'double', 'in', None), ('Ydim', 'double', 'in', None), ('Zdim', 'double', 'in', None), ('resample', 'bool', 'in', None)])
        res = _f.getProxyResult(self, _api.MFnFluid, 'setSize', final_do)
        return res

    @_f.addApiDocs(_api.MFnFluid, 'setTemperatureMode')
    def setTemperatureMode(self, method, gradient):
        # type: (Fluid.FluidMethod, Fluid.FluidGradient) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([method, gradient], [('method', ('MFnFluid', 'FluidMethod'), 'in', None), ('gradient', ('MFnFluid', 'FluidGradient'), 'in', None)], self.getTemperatureMode, self.setTemperatureMode, [])
        res = _f.getProxyResult(self, _api.MFnFluid, 'setTemperatureMode', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnFluid, 'setVelocityMode')
    def setVelocityMode(self, method, gradient):
        # type: (Fluid.FluidMethod, Fluid.FluidGradient) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([method, gradient], [('method', ('MFnFluid', 'FluidMethod'), 'in', None), ('gradient', ('MFnFluid', 'FluidGradient'), 'in', None)], self.getVelocityMode, self.setVelocityMode, [])
        res = _f.getProxyResult(self, _api.MFnFluid, 'setVelocityMode', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnFluid, 'temperature')
    def temperature(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnFluid, 'temperature')
        return _f.ApiArgUtil._castResult(self, res, 'float', None)

    @_f.addApiDocs(_api.MFnFluid, 'toGridIndex')
    def toGridIndex(self, objectSpacePoint):
        # type: (datatypes.Point) -> Tuple[bool, Tuple[int, int, int]]
        do, final_do, outTypes = _f.getDoArgs([objectSpacePoint], [('objectSpacePoint', 'MPoint', 'in', None), ('gridCoords', 'int3', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnFluid, 'toGridIndex', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'bool', None)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnFluid, 'updateGrid')
    def updateGrid(self):
        # type: () -> None
        res = _f.getProxyResult(self, _api.MFnFluid, 'updateGrid')
        return res

    @_f.addApiDocs(_api.MFnFluid, 'velocityGridSizes')
    def velocityGridSizes(self):
        # type: () -> Tuple[int, int, int]
        do, final_do, outTypes = _f.getDoArgs([], [('xsize', 'int', 'out', None), ('ysize', 'int', 'out', None), ('zsize', 'int', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnFluid, 'velocityGridSizes', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.addApiDocs(_api.MFnFluid, 'voxelCenterPosition')
    def voxelCenterPosition(self, xi, yi, zi):
        # type: (int, int, int) -> datatypes.Point
        do, final_do, outTypes = _f.getDoArgs([xi, yi, zi], [('xi', 'int', 'in', None), ('yi', 'int', 'in', None), ('zi', 'int', 'in', None), ('objectSpacePoint', 'MPoint', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnFluid, 'voxelCenterPosition', final_do)
        return _f.processApiResult(res, outTypes, do)


class FluidTexture2D(FluidShape):
    __melnode__ = 'fluidTexture2D'
    __slots__ = ()


class FluidTexture3D(FluidShape):
    __melnode__ = 'fluidTexture3D'
    __slots__ = ()


class HeightField(SurfaceShape):
    __melnode__ = 'heightField'
    __slots__ = ()


class GreasePlaneRenderShape(Mesh):
    __melnode__ = 'greasePlaneRenderShape'
    __slots__ = ()


class NBase(Particle):
    __melcmd__ = staticmethod(effects.nBase)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'nBase'
    __melnode__ = 'nBase'
    __slots__ = ()

    @_f.addMelDocs('nBase', 'clearCachedTextureMap')
    def clearCachedTextureMap(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, effects.nBase, kwargs, 'clearCachedTextureMap', val)

    @_f.addMelDocs('nBase', 'clearStart')
    def clearStart(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, effects.nBase, kwargs, 'clearStart', val)

    @_f.addMelDocs('nBase', 'stuffStart')
    def stuffStart(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, effects.nBase, kwargs, 'stuffStart', val)

    @_f.addMelDocs('nBase', 'textureToVertex')
    def textureToVertex(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, effects.nBase, kwargs, 'textureToVertex', val)


class NCloth(NBase):
    __melnode__ = 'nCloth'
    __slots__ = ()


class NParticle(NBase):
    __melcmd__ = staticmethod(effects.nParticle)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'nParticle'
    __melnode__ = 'nParticle'
    __slots__ = ()

    @_f.addMelDocs('nParticle', 'floatValue')
    def floatValue(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, effects.nParticle, kwargs, 'floatValue', val)

    @_f.addMelDocs('nParticle', 'attribute')
    def getAttribute(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, effects.nParticle, kwargs, 'attribute')
        return res

    @_f.addMelDocs('nParticle', 'cache')
    def getCache(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, effects.nParticle, kwargs, 'cache')
        return res

    @_f.addMelDocs('nParticle', 'conserve')
    def getConserve(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, effects.nParticle, kwargs, 'conserve')
        return res

    @_f.addMelDocs('nParticle', 'dynamicAttrList')
    def getDynamicAttrList(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, effects.nParticle, kwargs, 'dynamicAttrList')
        return res

    @_f.addMelDocs('nParticle', 'gridSpacing')
    def getGridSpacing(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, effects.nParticle, kwargs, 'gridSpacing')
        return res

    @_f.addMelDocs('nParticle', 'inherit')
    def getInherit(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, effects.nParticle, kwargs, 'inherit')
        return res

    @_f.addMelDocs('nParticle', 'jitterBasePoint')
    def getJitterBasePoint(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, effects.nParticle, kwargs, 'jitterBasePoint')
        return res

    @_f.addMelDocs('nParticle', 'jitterRadius')
    def getJitterRadius(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, effects.nParticle, kwargs, 'jitterRadius')
        return res

    @_f.addMelDocs('nParticle', 'lowerLeft')
    def getLowerLeft(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, effects.nParticle, kwargs, 'lowerLeft')
        return res

    @_f.addMelDocs('nParticle', 'numJitters')
    def getNumJitters(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, effects.nParticle, kwargs, 'numJitters')
        return res

    @_f.addMelDocs('nParticle', 'order')
    def getOrder(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, effects.nParticle, kwargs, 'order')
        return res

    @_f.addMelDocs('nParticle', 'particleId')
    def getParticleId(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, effects.nParticle, kwargs, 'particleId')
        return res

    @_f.addMelDocs('nParticle', 'perParticleDouble')
    def getPerParticleDouble(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, effects.nParticle, kwargs, 'perParticleDouble')
        return res

    @_f.addMelDocs('nParticle', 'perParticleVector')
    def getPerParticleVector(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, effects.nParticle, kwargs, 'perParticleVector')
        return res

    @_f.addMelDocs('nParticle', 'shapeName')
    def getShapeName(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, effects.nParticle, kwargs, 'shapeName')
        return res

    @_f.addMelDocs('nParticle', 'upperRight')
    def getUpperRight(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, effects.nParticle, kwargs, 'upperRight')
        return res

    @_f.addMelDocs('nParticle', 'attribute')
    def setAttribute(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, effects.nParticle, kwargs, 'attribute', val)

    @_f.addMelDocs('nParticle', 'cache')
    def setCache(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, effects.nParticle, kwargs, 'cache', val)

    @_f.addMelDocs('nParticle', 'conserve')
    def setConserve(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, effects.nParticle, kwargs, 'conserve', val)

    @_f.addMelDocs('nParticle', 'inherit')
    def setInherit(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, effects.nParticle, kwargs, 'inherit', val)

    @_f.addMelDocs('nParticle', 'order')
    def setOrder(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, effects.nParticle, kwargs, 'order', val)

    @_f.addMelDocs('nParticle', 'particleId')
    def setParticleId(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, effects.nParticle, kwargs, 'particleId', val)

    @_f.addMelDocs('nParticle', 'shapeName')
    def setShapeName(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, effects.nParticle, kwargs, 'shapeName', val)

    @_f.addMelDocs('nParticle', 'vectorValue')
    def vectorValue(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, effects.nParticle, kwargs, 'vectorValue', val)


class NRigid(NBase):
    __melnode__ = 'nRigid'
    __slots__ = ()


class DirectedDisc(GeometryShape):
    __melnode__ = 'directedDisc'
    __slots__ = ()


class EnvironmentFog(GeometryShape):
    __melnode__ = 'environmentFog'
    __slots__ = ()


class ImplicitBox(GeometryShape):
    __melnode__ = 'implicitBox'
    __slots__ = ()


class RenderBox(ImplicitBox):
    __melnode__ = 'renderBox'
    __slots__ = ()


class ImplicitCone(GeometryShape):
    __melnode__ = 'implicitCone'
    __slots__ = ()


class RenderCone(ImplicitCone):
    __melnode__ = 'renderCone'
    __slots__ = ()


class ImplicitSphere(GeometryShape):
    __melnode__ = 'implicitSphere'
    __slots__ = ()


class RenderSphere(ImplicitSphere):
    __melnode__ = 'renderSphere'
    __slots__ = ()


class Locator(GeometryShape):
    __melcmd__ = staticmethod(general.spaceLocator)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'spaceLocator'
    __melnode__ = 'locator'
    __slots__ = ()

    @_f.addMelDocs('spaceLocator', 'absolute')
    def absolute(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, general.spaceLocator, kwargs, 'absolute', val)

    @_f.addMelDocs('spaceLocator', 'position')
    def getPosition(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, general.spaceLocator, kwargs, 'position')
        return res

    @_f.addMelDocs('spaceLocator', 'relative')
    def relative(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, general.spaceLocator, kwargs, 'relative', val)

    @_f.addMelDocs('spaceLocator', 'position')
    def setPosition(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, general.spaceLocator, kwargs, 'position', val)


class THlocatorShape(Locator):
    __melnode__ = 'THlocatorShape'
    __slots__ = ()


class DropoffLocator(Locator):
    __melcmd__ = staticmethod(animation.dropoffLocator)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'dropoffLocator'
    __melnode__ = 'dropoffLocator'
    __slots__ = ()


class HikFloorContactMarker(Locator):
    __melnode__ = 'hikFloorContactMarker'
    __slots__ = ()


class PositionMarker(Locator):
    __melnode__ = 'positionMarker'
    __slots__ = ()


class OrientationMarker(PositionMarker):
    __melnode__ = 'orientationMarker'
    __slots__ = ()


class Plane(GeometryShape):
    __melcmd__ = staticmethod(modeling.plane)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'plane'
    __melnode__ = 'plane'
    __slots__ = ()


class SketchPlane(Plane):
    __melnode__ = 'sketchPlane'
    __slots__ = ()


class GroundPlane(SketchPlane):
    __melnode__ = 'groundPlane'
    __slots__ = ()


class OrthoGrid(SketchPlane):
    __melnode__ = 'orthoGrid'
    __slots__ = ()


class RenderRect(GeometryShape):
    __melnode__ = 'renderRect'
    __slots__ = ()


class HairConstraint(Shape):
    __melnode__ = 'hairConstraint'
    __slots__ = ()


class HairSystem(Shape):
    __melnode__ = 'hairSystem'
    __slots__ = ()


class ImagePlane(Shape):
    __melcmd__ = staticmethod(rendering.imagePlane)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'imagePlane'
    __melnode__ = 'imagePlane'
    __slots__ = ()

    @_f.addMelDocs('imagePlane', 'detach')
    def detach(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, rendering.imagePlane, kwargs, 'detach', val)

    @_f.addMelDocs('imagePlane', 'fileName')
    def fileName(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, rendering.imagePlane, kwargs, 'fileName', val)

    @_f.addMelDocs('imagePlane', 'camera')
    def getCamera(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, rendering.imagePlane, kwargs, 'camera')
        return res

    @_f.addMelDocs('imagePlane', 'dropFrame')
    def getDropFrame(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, rendering.imagePlane, kwargs, 'dropFrame')
        return res

    @_f.addMelDocs('imagePlane', 'frameDuration')
    def getFrameDuration(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, rendering.imagePlane, kwargs, 'frameDuration')
        return res

    @_f.addMelDocs('imagePlane', 'height')
    def getHeight(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, rendering.imagePlane, kwargs, 'height')
        return res

    @_f.addMelDocs('imagePlane', 'imageSize')
    def getImageSize(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, rendering.imagePlane, kwargs, 'imageSize')
        return res

    @_f.addMelDocs('imagePlane', 'lookThrough')
    def getLookThrough(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, rendering.imagePlane, kwargs, 'lookThrough')
        return res

    @_f.addMelDocs('imagePlane', 'maintainRatio')
    def getMaintainRatio(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, rendering.imagePlane, kwargs, 'maintainRatio')
        return res

    @_f.addMelDocs('imagePlane', 'negTimesOK')
    def getNegTimesOK(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, rendering.imagePlane, kwargs, 'negTimesOK')
        return res

    @_f.addMelDocs('imagePlane', 'numFrames')
    def getNumFrames(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, rendering.imagePlane, kwargs, 'numFrames')
        return res

    @_f.addMelDocs('imagePlane', 'quickTime')
    def getQuickTime(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, rendering.imagePlane, kwargs, 'quickTime')
        return res

    @_f.addMelDocs('imagePlane', 'showInAllViews')
    def getShowInAllViews(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, rendering.imagePlane, kwargs, 'showInAllViews')
        return res

    @_f.addMelDocs('imagePlane', 'timeCode')
    def getTimeCode(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, rendering.imagePlane, kwargs, 'timeCode')
        return res

    @_f.addMelDocs('imagePlane', 'timeCodeTrack')
    def getTimeCodeTrack(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, rendering.imagePlane, kwargs, 'timeCodeTrack')
        return res

    @_f.addMelDocs('imagePlane', 'timeScale')
    def getTimeScale(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, rendering.imagePlane, kwargs, 'timeScale')
        return res

    @_f.addMelDocs('imagePlane', 'twentyFourHourMax')
    def getTwentyFourHourMax(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, rendering.imagePlane, kwargs, 'twentyFourHourMax')
        return res

    @_f.addMelDocs('imagePlane', 'width')
    def getWidth(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, rendering.imagePlane, kwargs, 'width')
        return res

    @_f.addMelDocs('imagePlane', 'camera')
    def setCamera(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, rendering.imagePlane, kwargs, 'camera', val)

    @_f.addMelDocs('imagePlane', 'height')
    def setHeight(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, rendering.imagePlane, kwargs, 'height', val)

    @_f.addMelDocs('imagePlane', 'lookThrough')
    def setLookThrough(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, rendering.imagePlane, kwargs, 'lookThrough', val)

    @_f.addMelDocs('imagePlane', 'maintainRatio')
    def setMaintainRatio(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, rendering.imagePlane, kwargs, 'maintainRatio', val)

    @_f.addMelDocs('imagePlane', 'showInAllViews')
    def setShowInAllViews(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, rendering.imagePlane, kwargs, 'showInAllViews', val)

    @_f.addMelDocs('imagePlane', 'width')
    def setWidth(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, rendering.imagePlane, kwargs, 'width', val)


class THimagePlane(ImagePlane):
    __melnode__ = 'THimagePlane'
    __slots__ = ()


class GreasePlane(ImagePlane):
    __melnode__ = 'greasePlane'
    __slots__ = ()


class Light(Shape):
    __apicls__ = _api.MFnLight
    __melnode__ = 'light'
    __slots__ = ()

    @_f.addApiDocs(_api.MFnLight, 'centerOfIllumination')
    def getCenterOfIllumination(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnLight, 'centerOfIllumination')
        return _f.ApiArgUtil._castResult(self, res, 'double', 'linear')

    @_f.addApiDocs(_api.MFnLight, 'color')
    def getColor(self):
        # type: () -> datatypes.Color
        res = _f.getProxyResult(self, _api.MFnLight, 'color')
        return _f.ApiArgUtil._castResult(self, res, 'MColor', None)

    @_f.addApiDocs(_api.MFnLight, 'intensity')
    def getIntensity(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnLight, 'intensity')
        return _f.ApiArgUtil._castResult(self, res, 'float', None)

    @_f.addApiDocs(_api.MFnLight, 'numShadowSamples')
    def getNumShadowSamples(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnLight, 'numShadowSamples')
        return _f.ApiArgUtil._castResult(self, res, 'short', None)

    @_f.addApiDocs(_api.MFnLight, 'opticalFXvisibility')
    def getOpticalFXvisibility(self):
        # type: () -> datatypes.Color
        res = _f.getProxyResult(self, _api.MFnLight, 'opticalFXvisibility')
        return _f.ApiArgUtil._castResult(self, res, 'MColor', None)

    @_f.addApiDocs(_api.MFnLight, 'rayDepthLimit')
    def getRayDepthLimit(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnLight, 'rayDepthLimit')
        return _f.ApiArgUtil._castResult(self, res, 'short', None)

    @_f.addApiDocs(_api.MFnLight, 'shadowColor')
    def getShadowColor(self):
        # type: () -> datatypes.Color
        res = _f.getProxyResult(self, _api.MFnLight, 'shadowColor')
        return _f.ApiArgUtil._castResult(self, res, 'MColor', None)

    @_f.addApiDocs(_api.MFnLight, 'useRayTraceShadows')
    def getUseRayTraceShadows(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnLight, 'useRayTraceShadows')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnLight, 'lightAmbient')
    def lightAmbient(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnLight, 'lightAmbient')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnLight, 'lightDiffuse')
    def lightDiffuse(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnLight, 'lightDiffuse')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnLight, 'lightDirection')
    def lightDirection(self, instance, space='world'):
        # type: (int, datatypes.Space.Space) -> datatypes.FloatVector
        do, final_do, outTypes = _f.getDoArgs([instance, space], [('instance', 'int', 'in', None), ('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnLight, 'lightDirection', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'MFloatVector', None)
        return res

    @_f.addApiDocs(_api.MFnLight, 'lightIntensity')
    def lightIntensity(self):
        # type: () -> datatypes.Color
        res = _f.getProxyResult(self, _api.MFnLight, 'lightIntensity')
        return _f.ApiArgUtil._castResult(self, res, 'MColor', None)

    @_f.addApiDocs(_api.MFnLight, 'lightSpecular')
    def lightSpecular(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnLight, 'lightSpecular')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnLight, 'setCenterOfIllumination')
    def setCenterOfIllumination(self, dist):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([dist], [('dist', 'double', 'in', 'linear')], self.getCenterOfIllumination, self.setCenterOfIllumination, [])
        res = _f.getProxyResult(self, _api.MFnLight, 'setCenterOfIllumination', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnLight, 'setColor')
    def setColor(self, col):
        # type: (datatypes.Color) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([col], [('col', 'MColor', 'in', None)], self.getColor, self.setColor, [])
        res = _f.getProxyResult(self, _api.MFnLight, 'setColor', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnLight, 'setIntensity')
    def setIntensity(self, intens):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([intens], [('intens', 'float', 'in', None)], self.getIntensity, self.setIntensity, [])
        res = _f.getProxyResult(self, _api.MFnLight, 'setIntensity', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnLight, 'setNumShadowSamples')
    def setNumShadowSamples(self, num_shadow_samples):
        # type: (int) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([num_shadow_samples], [('num_shadow_samples', 'short', 'in', None)], self.getNumShadowSamples, self.setNumShadowSamples, [])
        res = _f.getProxyResult(self, _api.MFnLight, 'setNumShadowSamples', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnLight, 'setOpticalFXvisibility')
    def setOpticalFXvisibility(self, visibility):
        # type: (datatypes.Color) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([visibility], [('visibility', 'MColor', 'in', None)], self.getOpticalFXvisibility, self.setOpticalFXvisibility, [])
        res = _f.getProxyResult(self, _api.MFnLight, 'setOpticalFXvisibility', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnLight, 'setRayDepthLimit')
    def setRayDepthLimit(self, ray_depth_limit):
        # type: (int) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([ray_depth_limit], [('ray_depth_limit', 'short', 'in', None)], self.getRayDepthLimit, self.setRayDepthLimit, [])
        res = _f.getProxyResult(self, _api.MFnLight, 'setRayDepthLimit', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnLight, 'setShadowColor')
    def setShadowColor(self, shadow_color):
        # type: (datatypes.Color) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([shadow_color], [('shadow_color', 'MColor', 'in', None)], self.getShadowColor, self.setShadowColor, [])
        res = _f.getProxyResult(self, _api.MFnLight, 'setShadowColor', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnLight, 'setUseRayTraceShadows')
    def setUseRayTraceShadows(self, useRayTraceShadows):
        # type: (bool) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([useRayTraceShadows], [('useRayTraceShadows', 'bool', 'in', None)], self.getUseRayTraceShadows, self.setUseRayTraceShadows, [])
        res = _f.getProxyResult(self, _api.MFnLight, 'setUseRayTraceShadows', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res


class TadskAssetInstanceNode_TlightShape(Light):
    __melnode__ = 'TadskAssetInstanceNode_TlightShape'
    __slots__ = ()


class RenderLight(Light):
    __melnode__ = 'renderLight'
    __slots__ = ()


class AmbientLight(RenderLight):
    __apicls__ = _api.MFnAmbientLight
    __melcmd__ = staticmethod(rendering.ambientLight)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'ambientLight'
    __melnode__ = 'ambientLight'
    __slots__ = ()

    @_f.addApiDocs(_api.MFnAmbientLight, 'ambientShade')
    def getAmbientShade(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnAmbientLight, 'ambientShade')
        return _f.ApiArgUtil._castResult(self, res, 'float', None)

    @_f.addApiDocs(_api.MFnAmbientLight, 'castSoftShadows')
    def getCastSoftShadows(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnAmbientLight, 'castSoftShadows')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addMelDocs('ambientLight', 'discRadius')
    def getDiscRadius(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, rendering.ambientLight, kwargs, 'discRadius')
        return res

    @_f.addMelDocs('ambientLight', 'exclusive')
    def getExclusive(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, rendering.ambientLight, kwargs, 'exclusive')
        return res

    @_f.addMelDocs('ambientLight', 'position')
    def getPosition(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, rendering.ambientLight, kwargs, 'position')
        return res

    @_f.addMelDocs('ambientLight', 'rgb')
    def getRgb(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, rendering.ambientLight, kwargs, 'rgb')
        return res

    @_f.addMelDocs('ambientLight', 'rotation')
    def getRotation(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, rendering.ambientLight, kwargs, 'rotation')
        return res

    @_f.addMelDocs('ambientLight', 'shadowDither')
    def getShadowDither(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, rendering.ambientLight, kwargs, 'shadowDither')
        return res

    @_f.addApiDocs(_api.MFnAmbientLight, 'shadowRadius')
    def getShadowRadius(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnAmbientLight, 'shadowRadius')
        return _f.ApiArgUtil._castResult(self, res, 'float', None)

    @_f.addMelDocs('ambientLight', 'shadowSamples')
    def getShadowSamples(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, rendering.ambientLight, kwargs, 'shadowSamples')
        return res

    @_f.addMelDocs('ambientLight', 'softShadow')
    def getSoftShadow(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, rendering.ambientLight, kwargs, 'softShadow')
        return res

    @_f.addApiDocs(_api.MFnAmbientLight, 'setAmbientShade')
    def setAmbientShade(self, ambient_shade):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([ambient_shade], [('ambient_shade', 'float', 'in', None)], self.getAmbientShade, self.setAmbientShade, [])
        res = _f.getProxyResult(self, _api.MFnAmbientLight, 'setAmbientShade', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnAmbientLight, 'setCastSoftShadows')
    def setCastSoftShadows(self, cast_soft_shadows):
        # type: (bool) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([cast_soft_shadows], [('cast_soft_shadows', 'bool', 'in', None)], self.getCastSoftShadows, self.setCastSoftShadows, [])
        res = _f.getProxyResult(self, _api.MFnAmbientLight, 'setCastSoftShadows', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addMelDocs('ambientLight', 'discRadius')
    def setDiscRadius(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, rendering.ambientLight, kwargs, 'discRadius', val)

    @_f.addMelDocs('ambientLight', 'shadowDither')
    def setShadowDither(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, rendering.ambientLight, kwargs, 'shadowDither', val)

    @_f.addApiDocs(_api.MFnAmbientLight, 'setShadowRadius')
    def setShadowRadius(self, shadow_radius):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([shadow_radius], [('shadow_radius', 'float', 'in', None)], self.getShadowRadius, self.setShadowRadius, [])
        res = _f.getProxyResult(self, _api.MFnAmbientLight, 'setShadowRadius', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addMelDocs('ambientLight', 'shadowSamples')
    def setShadowSamples(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, rendering.ambientLight, kwargs, 'shadowSamples', val)

    @_f.addMelDocs('ambientLight', 'softShadow')
    def setSoftShadow(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, rendering.ambientLight, kwargs, 'softShadow', val)


class NonAmbientLightShapeNode(RenderLight):
    __apicls__ = _api.MFnNonAmbientLight
    __melnode__ = 'nonAmbientLightShapeNode'
    __slots__ = ()

    @_f.addApiDocs(_api.MFnNonAmbientLight, 'decayRate')
    def getDecayRate(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnNonAmbientLight, 'decayRate')
        return _f.ApiArgUtil._castResult(self, res, 'short', None)

    @_f.addApiDocs(_api.MFnNonAmbientLight, 'setDecayRate')
    def setDecayRate(self, decay_rate):
        # type: (int) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([decay_rate], [('decay_rate', 'short', 'in', None)], self.getDecayRate, self.setDecayRate, [])
        res = _f.getProxyResult(self, _api.MFnNonAmbientLight, 'setDecayRate', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res


class NonExtendedLightShapeNode(NonAmbientLightShapeNode):
    __apicls__ = _api.MFnNonExtendedLight
    __melnode__ = 'nonExtendedLightShapeNode'
    __slots__ = ()

    @_f.addApiDocs(_api.MFnNonExtendedLight, 'castSoftShadows')
    def getCastSoftShadows(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnNonExtendedLight, 'castSoftShadows')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnNonExtendedLight, 'depthMapBias')
    def getDepthMapBias(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnNonExtendedLight, 'depthMapBias')
        return _f.ApiArgUtil._castResult(self, res, 'float', None)

    @_f.addApiDocs(_api.MFnNonExtendedLight, 'depthMapFilterSize')
    def getDepthMapFilterSize(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnNonExtendedLight, 'depthMapFilterSize')
        return _f.ApiArgUtil._castResult(self, res, 'short', None)

    @_f.addApiDocs(_api.MFnNonExtendedLight, 'depthMapFocus')
    def getDepthMapFocus(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnNonExtendedLight, 'depthMapFocus')
        return _f.ApiArgUtil._castResult(self, res, 'float', None)

    @_f.addApiDocs(_api.MFnNonExtendedLight, 'depthMapResolution')
    def getDepthMapResolution(self):
        # type: () -> int
        res = _f.getProxyResult(self, _api.MFnNonExtendedLight, 'depthMapResolution')
        return _f.ApiArgUtil._castResult(self, res, 'short', None)

    @_f.addApiDocs(_api.MFnNonExtendedLight, 'depthMapWidthFocus')
    def getDepthMapWidthFocus(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnNonExtendedLight, 'depthMapWidthFocus')
        return _f.ApiArgUtil._castResult(self, res, 'float', None)

    @_f.addApiDocs(_api.MFnNonExtendedLight, 'shadowRadius')
    def getShadowRadius(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnNonExtendedLight, 'shadowRadius')
        return _f.ApiArgUtil._castResult(self, res, 'float', None)

    @_f.addApiDocs(_api.MFnNonExtendedLight, 'useDepthMapAutoFocus')
    def getUseDepthMapAutoFocus(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnNonExtendedLight, 'useDepthMapAutoFocus')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnNonExtendedLight, 'useDepthMapShadows')
    def getUseDepthMapShadows(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnNonExtendedLight, 'useDepthMapShadows')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnNonExtendedLight, 'setCastSoftShadows')
    def setCastSoftShadows(self, cast_soft_shadows):
        # type: (bool) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([cast_soft_shadows], [('cast_soft_shadows', 'bool', 'in', None)], self.getCastSoftShadows, self.setCastSoftShadows, [])
        res = _f.getProxyResult(self, _api.MFnNonExtendedLight, 'setCastSoftShadows', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnNonExtendedLight, 'setDepthMapBias')
    def setDepthMapBias(self, depth_map_bias):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([depth_map_bias], [('depth_map_bias', 'float', 'in', None)], self.getDepthMapBias, self.setDepthMapBias, [])
        res = _f.getProxyResult(self, _api.MFnNonExtendedLight, 'setDepthMapBias', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnNonExtendedLight, 'setDepthMapFilterSize')
    def setDepthMapFilterSize(self, depth_map_filter_size):
        # type: (int) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([depth_map_filter_size], [('depth_map_filter_size', 'short', 'in', None)], self.getDepthMapFilterSize, self.setDepthMapFilterSize, [])
        res = _f.getProxyResult(self, _api.MFnNonExtendedLight, 'setDepthMapFilterSize', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnNonExtendedLight, 'setDepthMapFocus')
    def setDepthMapFocus(self, depth_map_focus):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([depth_map_focus], [('depth_map_focus', 'float', 'in', None)], self.getDepthMapFocus, self.setDepthMapFocus, [])
        res = _f.getProxyResult(self, _api.MFnNonExtendedLight, 'setDepthMapFocus', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnNonExtendedLight, 'setDepthMapResolution')
    def setDepthMapResolution(self, depth_map_resolution):
        # type: (int) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([depth_map_resolution], [('depth_map_resolution', 'short', 'in', None)], self.getDepthMapResolution, self.setDepthMapResolution, [])
        res = _f.getProxyResult(self, _api.MFnNonExtendedLight, 'setDepthMapResolution', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnNonExtendedLight, 'setDepthMapWidthFocus')
    def setDepthMapWidthFocus(self, depth_map_width_focus):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([depth_map_width_focus], [('depth_map_width_focus', 'float', 'in', None)], self.getDepthMapWidthFocus, self.setDepthMapWidthFocus, [])
        res = _f.getProxyResult(self, _api.MFnNonExtendedLight, 'setDepthMapWidthFocus', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnNonExtendedLight, 'setShadowRadius')
    def setShadowRadius(self, shadow_radius):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([shadow_radius], [('shadow_radius', 'float', 'in', None)], self.getShadowRadius, self.setShadowRadius, [])
        res = _f.getProxyResult(self, _api.MFnNonExtendedLight, 'setShadowRadius', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnNonExtendedLight, 'setUseDepthMapAutoFocus')
    def setUseDepthMapAutoFocus(self, use_depth_map_auto_focus):
        # type: (bool) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([use_depth_map_auto_focus], [('use_depth_map_auto_focus', 'bool', 'in', None)], self.getUseDepthMapAutoFocus, self.setUseDepthMapAutoFocus, [])
        res = _f.getProxyResult(self, _api.MFnNonExtendedLight, 'setUseDepthMapAutoFocus', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnNonExtendedLight, 'setUseDepthMapShadows')
    def setUseDepthMapShadows(self, use_depth_map):
        # type: (bool) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([use_depth_map], [('use_depth_map', 'bool', 'in', None)], self.getUseDepthMapShadows, self.setUseDepthMapShadows, [])
        res = _f.getProxyResult(self, _api.MFnNonExtendedLight, 'setUseDepthMapShadows', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res


class AreaLight(NonExtendedLightShapeNode):
    __apicls__ = _api.MFnAreaLight
    __melnode__ = 'areaLight'
    __slots__ = ()


class DirectionalLight(NonExtendedLightShapeNode):
    __apicls__ = _api.MFnDirectionalLight
    __melcmd__ = staticmethod(rendering.directionalLight)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'directionalLight'
    __melnode__ = 'directionalLight'
    __slots__ = ()

    @_f.addMelDocs('directionalLight', 'discRadius')
    def getDiscRadius(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, rendering.directionalLight, kwargs, 'discRadius')
        return res

    @_f.addMelDocs('directionalLight', 'exclusive')
    def getExclusive(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, rendering.directionalLight, kwargs, 'exclusive')
        return res

    @_f.addMelDocs('directionalLight', 'rgb')
    def getRgb(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, rendering.directionalLight, kwargs, 'rgb')
        return res

    @_f.addApiDocs(_api.MFnDirectionalLight, 'shadowAngle')
    def getShadowAngle(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnDirectionalLight, 'shadowAngle')
        return _f.ApiArgUtil._castResult(self, res, 'float', None)

    @_f.addMelDocs('directionalLight', 'shadowDither')
    def getShadowDither(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, rendering.directionalLight, kwargs, 'shadowDither')
        return res

    @_f.addMelDocs('directionalLight', 'shadowSamples')
    def getShadowSamples(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, rendering.directionalLight, kwargs, 'shadowSamples')
        return res

    @_f.addMelDocs('directionalLight', 'softShadow')
    def getSoftShadow(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, rendering.directionalLight, kwargs, 'softShadow')
        return res

    @_f.addApiDocs(_api.MFnDirectionalLight, 'useLightPosition')
    def getUseLightPosition(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnDirectionalLight, 'useLightPosition')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnDirectionalLight, 'setShadowAngle')
    def setShadowAngle(self, shadow_angle):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([shadow_angle], [('shadow_angle', 'float', 'in', None)], self.getShadowAngle, self.setShadowAngle, [])
        res = _f.getProxyResult(self, _api.MFnDirectionalLight, 'setShadowAngle', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnDirectionalLight, 'setUseLightPosition')
    def setUseLightPosition(self, use_light_position):
        # type: (bool) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([use_light_position], [('use_light_position', 'bool', 'in', None)], self.getUseLightPosition, self.setUseLightPosition, [])
        res = _f.getProxyResult(self, _api.MFnDirectionalLight, 'setUseLightPosition', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res


class PointLight(NonExtendedLightShapeNode):
    __apicls__ = _api.MFnPointLight
    __melcmd__ = staticmethod(rendering.pointLight)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'pointLight'
    __melnode__ = 'pointLight'
    __slots__ = ()

    @_f.addMelDocs('pointLight', 'discRadius')
    def getDiscRadius(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, rendering.pointLight, kwargs, 'discRadius')
        return res

    @_f.addMelDocs('pointLight', 'exclusive')
    def getExclusive(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, rendering.pointLight, kwargs, 'exclusive')
        return res

    @_f.addMelDocs('pointLight', 'rgb')
    def getRgb(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, rendering.pointLight, kwargs, 'rgb')
        return res

    @_f.addMelDocs('pointLight', 'shadowDither')
    def getShadowDither(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, rendering.pointLight, kwargs, 'shadowDither')
        return res

    @_f.addMelDocs('pointLight', 'shadowSamples')
    def getShadowSamples(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, rendering.pointLight, kwargs, 'shadowSamples')
        return res

    @_f.addMelDocs('pointLight', 'softShadow')
    def getSoftShadow(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, rendering.pointLight, kwargs, 'softShadow')
        return res

    @_f.addMelDocs('pointLight', 'discRadius')
    def setDiscRadius(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, rendering.pointLight, kwargs, 'discRadius', val)

    @_f.addMelDocs('pointLight', 'exclusive')
    def setExclusive(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, rendering.pointLight, kwargs, 'exclusive', val)

    @_f.addMelDocs('pointLight', 'rgb')
    def setRgb(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, rendering.pointLight, kwargs, 'rgb', val)

    @_f.addMelDocs('pointLight', 'shadowDither')
    def setShadowDither(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, rendering.pointLight, kwargs, 'shadowDither', val)

    @_f.addMelDocs('pointLight', 'shadowSamples')
    def setShadowSamples(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, rendering.pointLight, kwargs, 'shadowSamples', val)

    @_f.addMelDocs('pointLight', 'softShadow')
    def setSoftShadow(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, rendering.pointLight, kwargs, 'softShadow', val)


class VolumeLight(PointLight):
    __apicls__ = _api.MFnVolumeLight
    __melnode__ = 'volumeLight'
    __slots__ = ()
    MLightDirection = Enum('MLightDirection', [('outward', 0), ('kOutward', 0), ('inward', 1), ('kInward', 1), ('downAxis', 2), ('kDownAxis', 2)], multiKeys=True)
    MLightShape = Enum('MLightShape', [('boxVolume', 0), ('kBoxVolume', 0), ('sphereVolume', 1), ('kSphereVolume', 1), ('cylinderVolume', 2), ('kCylinderVolume', 2), ('coneVolume', 3), ('kConeVolume', 3)], multiKeys=True)

    @_f.addApiDocs(_api.MFnVolumeLight, 'arc')
    def getArc(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnVolumeLight, 'arc')
        return _f.ApiArgUtil._castResult(self, res, 'float', None)

    @_f.addApiDocs(_api.MFnVolumeLight, 'coneEndRadius')
    def getConeEndRadius(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnVolumeLight, 'coneEndRadius')
        return _f.ApiArgUtil._castResult(self, res, 'float', None)

    @_f.addApiDocs(_api.MFnVolumeLight, 'emitAmbient')
    def getEmitAmbient(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnVolumeLight, 'emitAmbient')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnVolumeLight, 'lightShape')
    def getLightShape(self):
        # type: () -> VolumeLight.MLightShape
        res = _f.getProxyResult(self, _api.MFnVolumeLight, 'lightShape')
        return _f.ApiArgUtil._castResult(self, res, ('MFnVolumeLight', 'MLightShape'), None)

    @_f.addApiDocs(_api.MFnVolumeLight, 'shadowAngle')
    def getShadowAngle(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnVolumeLight, 'shadowAngle')
        return _f.ApiArgUtil._castResult(self, res, 'float', None)

    @_f.addApiDocs(_api.MFnVolumeLight, 'volumeLightDirection')
    def getVolumeLightDirection(self):
        # type: () -> VolumeLight.MLightDirection
        res = _f.getProxyResult(self, _api.MFnVolumeLight, 'volumeLightDirection')
        return _f.ApiArgUtil._castResult(self, res, ('MFnVolumeLight', 'MLightDirection'), None)

    @_f.addApiDocs(_api.MFnVolumeLight, 'setArc')
    def setArc(self, arc):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([arc], [('arc', 'float', 'in', None)], self.getArc, self.setArc, [])
        res = _f.getProxyResult(self, _api.MFnVolumeLight, 'setArc', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnVolumeLight, 'setConeEndRadius')
    def setConeEndRadius(self, cone_end_radius):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([cone_end_radius], [('cone_end_radius', 'float', 'in', None)], self.getConeEndRadius, self.setConeEndRadius, [])
        res = _f.getProxyResult(self, _api.MFnVolumeLight, 'setConeEndRadius', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnVolumeLight, 'setEmitAmbient')
    def setEmitAmbient(self, emit_ambient):
        # type: (bool) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([emit_ambient], [('emit_ambient', 'bool', 'in', None)], self.getEmitAmbient, self.setEmitAmbient, [])
        res = _f.getProxyResult(self, _api.MFnVolumeLight, 'setEmitAmbient', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnVolumeLight, 'setLightShape')
    def setLightShape(self, light_shape):
        # type: (VolumeLight.MLightShape) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([light_shape], [('light_shape', ('MFnVolumeLight', 'MLightShape'), 'in', None)], self.getLightShape, self.setLightShape, [])
        res = _f.getProxyResult(self, _api.MFnVolumeLight, 'setLightShape', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnVolumeLight, 'setShadowAngle')
    def setShadowAngle(self, shadow_angle):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([shadow_angle], [('shadow_angle', 'float', 'in', None)], self.getShadowAngle, self.setShadowAngle, [])
        res = _f.getProxyResult(self, _api.MFnVolumeLight, 'setShadowAngle', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnVolumeLight, 'setVolumeLightDirection')
    def setVolumeLightDirection(self, volume_light_direction):
        # type: (VolumeLight.MLightDirection) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([volume_light_direction], [('volume_light_direction', ('MFnVolumeLight', 'MLightDirection'), 'in', None)], self.getVolumeLightDirection, self.setVolumeLightDirection, [])
        res = _f.getProxyResult(self, _api.MFnVolumeLight, 'setVolumeLightDirection', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res


class SpotLight(NonExtendedLightShapeNode):
    __apicls__ = _api.MFnSpotLight
    __melcmd__ = staticmethod(rendering.spotLight)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'spotLight'
    __melnode__ = 'spotLight'
    __slots__ = ()
    MBarnDoor = Enum('MBarnDoor', [('left', 0), ('kLeft', 0), ('right', 1), ('kRight', 1), ('top', 2), ('kTop', 2), ('bottom', 3), ('kBottom', 3)], multiKeys=True)
    MDecayRegion = Enum('MDecayRegion', [('first', 0), ('kFirst', 0), ('second', 1), ('kSecond', 1), ('third', 2), ('kThird', 2)], multiKeys=True)

    @_f.addApiDocs(_api.MFnSpotLight, 'barnDoorAngle')
    def getBarnDoorAngle(self, which_door):
        # type: (SpotLight.MBarnDoor) -> float
        do, final_do, outTypes = _f.getDoArgs([which_door], [('which_door', ('MFnSpotLight', 'MBarnDoor'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnSpotLight, 'barnDoorAngle', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'double', 'angular')
        return res

    @_f.addApiDocs(_api.MFnSpotLight, 'barnDoors')
    def getBarnDoors(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnSpotLight, 'barnDoors')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addMelDocs('spotLight', 'bottomBarnDoorAngle')
    def getBottomBarnDoorAngle(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, rendering.spotLight, kwargs, 'bottomBarnDoorAngle')
        return res

    @_f.addApiDocs(_api.MFnSpotLight, 'coneAngle')
    def getConeAngle(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnSpotLight, 'coneAngle')
        return _f.ApiArgUtil._castResult(self, res, 'double', 'angular')

    @_f.addMelDocs('spotLight', 'discRadius')
    def getDiscRadius(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, rendering.spotLight, kwargs, 'discRadius')
        return res

    @_f.addApiDocs(_api.MFnSpotLight, 'dropOff')
    def getDropOff(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnSpotLight, 'dropOff')
        return _f.ApiArgUtil._castResult(self, res, 'double', None)

    @_f.addApiDocs(_api.MFnSpotLight, 'endDistance')
    def getEndDistance(self, which_region):
        # type: (SpotLight.MDecayRegion) -> float
        do, final_do, outTypes = _f.getDoArgs([which_region], [('which_region', ('MFnSpotLight', 'MDecayRegion'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnSpotLight, 'endDistance', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'float', None)
        return res

    @_f.addMelDocs('spotLight', 'exclusive')
    def getExclusive(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, rendering.spotLight, kwargs, 'exclusive')
        return res

    @_f.addMelDocs('spotLight', 'leftBarnDoorAngle')
    def getLeftBarnDoorAngle(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, rendering.spotLight, kwargs, 'leftBarnDoorAngle')
        return res

    @_f.addApiDocs(_api.MFnSpotLight, 'penumbraAngle')
    def getPenumbra(self):
        # type: () -> float
        res = _f.getProxyResult(self, _api.MFnSpotLight, 'penumbraAngle')
        return _f.ApiArgUtil._castResult(self, res, 'double', 'angular')

    @_f.addMelDocs('spotLight', 'rgb')
    def getRgb(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, rendering.spotLight, kwargs, 'rgb')
        return res

    @_f.addMelDocs('spotLight', 'rightBarnDoorAngle')
    def getRightBarnDoorAngle(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, rendering.spotLight, kwargs, 'rightBarnDoorAngle')
        return res

    @_f.addMelDocs('spotLight', 'shadowDither')
    def getShadowDither(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, rendering.spotLight, kwargs, 'shadowDither')
        return res

    @_f.addMelDocs('spotLight', 'shadowSamples')
    def getShadowSamples(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, rendering.spotLight, kwargs, 'shadowSamples')
        return res

    @_f.addMelDocs('spotLight', 'softShadow')
    def getSoftShadow(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, rendering.spotLight, kwargs, 'softShadow')
        return res

    @_f.addApiDocs(_api.MFnSpotLight, 'startDistance')
    def getStartDistance(self, which_region):
        # type: (SpotLight.MDecayRegion) -> float
        do, final_do, outTypes = _f.getDoArgs([which_region], [('which_region', ('MFnSpotLight', 'MDecayRegion'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnSpotLight, 'startDistance', final_do)
        res = _f.ApiArgUtil._castResult(self, res, 'float', None)
        return res

    @_f.addMelDocs('spotLight', 'topBarnDoorAngle')
    def getTopBarnDoorAngle(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, rendering.spotLight, kwargs, 'topBarnDoorAngle')
        return res

    @_f.addApiDocs(_api.MFnSpotLight, 'useDecayRegions')
    def getUseDecayRegions(self):
        # type: () -> bool
        res = _f.getProxyResult(self, _api.MFnSpotLight, 'useDecayRegions')
        return _f.ApiArgUtil._castResult(self, res, 'bool', None)

    @_f.addApiDocs(_api.MFnSpotLight, 'setBarnDoorAngle')
    def setBarnDoorAngle(self, which_door, barn_door_angle):
        # type: (SpotLight.MBarnDoor, float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([which_door, barn_door_angle], [('which_door', ('MFnSpotLight', 'MBarnDoor'), 'in', None), ('barn_door_angle', 'double', 'in', 'angular')], self.getBarnDoorAngle, self.setBarnDoorAngle, ['which_door'])
        res = _f.getProxyResult(self, _api.MFnSpotLight, 'setBarnDoorAngle', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnSpotLight, 'setBarnDoors')
    def setBarnDoors(self, barn_doors):
        # type: (bool) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([barn_doors], [('barn_doors', 'bool', 'in', None)], self.getBarnDoors, self.setBarnDoors, [])
        res = _f.getProxyResult(self, _api.MFnSpotLight, 'setBarnDoors', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addMelDocs('spotLight', 'bottomBarnDoorAngle')
    def setBottomBarnDoorAngle(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, rendering.spotLight, kwargs, 'bottomBarnDoorAngle', val)

    @_f.addApiDocs(_api.MFnSpotLight, 'setConeAngle')
    def setConeAngle(self, cone_angle):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([cone_angle], [('cone_angle', 'double', 'in', 'angular')], self.getConeAngle, self.setConeAngle, [])
        res = _f.getProxyResult(self, _api.MFnSpotLight, 'setConeAngle', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnSpotLight, 'setDropOff')
    def setDropOff(self, drop_off):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([drop_off], [('drop_off', 'double', 'in', None)], self.getDropOff, self.setDropOff, [])
        res = _f.getProxyResult(self, _api.MFnSpotLight, 'setDropOff', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addApiDocs(_api.MFnSpotLight, 'setEndDistance')
    def setEndDistance(self, which_region, end_distance):
        # type: (SpotLight.MDecayRegion, float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([which_region, end_distance], [('which_region', ('MFnSpotLight', 'MDecayRegion'), 'in', None), ('end_distance', 'float', 'in', None)], self.getEndDistance, self.setEndDistance, ['which_region'])
        res = _f.getProxyResult(self, _api.MFnSpotLight, 'setEndDistance', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addMelDocs('spotLight', 'leftBarnDoorAngle')
    def setLeftBarnDoorAngle(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, rendering.spotLight, kwargs, 'leftBarnDoorAngle', val)

    @_f.addApiDocs(_api.MFnSpotLight, 'setPenumbraAngle')
    def setPenumbra(self, penumbra_angle):
        # type: (float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([penumbra_angle], [('penumbra_angle', 'double', 'in', 'angular')], self.getPenumbra, self.setPenumbra, [])
        res = _f.getProxyResult(self, _api.MFnSpotLight, 'setPenumbraAngle', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addMelDocs('spotLight', 'rightBarnDoorAngle')
    def setRightBarnDoorAngle(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, rendering.spotLight, kwargs, 'rightBarnDoorAngle', val)

    @_f.addApiDocs(_api.MFnSpotLight, 'setStartDistance')
    def setStartDistance(self, which_region, start_distance):
        # type: (SpotLight.MDecayRegion, float) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([which_region, start_distance], [('which_region', ('MFnSpotLight', 'MDecayRegion'), 'in', None), ('start_distance', 'float', 'in', None)], self.getStartDistance, self.setStartDistance, ['which_region'])
        res = _f.getProxyResult(self, _api.MFnSpotLight, 'setStartDistance', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res

    @_f.addMelDocs('spotLight', 'topBarnDoorAngle')
    def setTopBarnDoorAngle(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, rendering.spotLight, kwargs, 'topBarnDoorAngle', val)

    @_f.addApiDocs(_api.MFnSpotLight, 'setUseDecayRegions')
    def setUseDecayRegions(self, use_decay_regions):
        # type: (bool) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([use_decay_regions], [('use_decay_regions', 'bool', 'in', None)], self.getUseDecayRegions, self.setUseDecayRegions, [])
        res = _f.getProxyResult(self, _api.MFnSpotLight, 'setUseDecayRegions', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res


class LineModifier(Shape):
    __melnode__ = 'lineModifier'
    __slots__ = ()


class PfxGeometry(Shape):
    __apicls__ = _api.MFnPfxGeometry
    __melnode__ = 'pfxGeometry'
    __slots__ = ()

    @_f.addApiDocs(_api.MFnPfxGeometry, 'getBoundingBox')
    def getBoundingBox(self):
        # type: () -> Tuple[float, float]
        do, final_do, outTypes = _f.getDoArgs([], [('min', 'double', 'out', None), ('max', 'double', 'out', None)])
        res = _f.getProxyResult(self, _api.MFnPfxGeometry, 'getBoundingBox', final_do)
        return _f.processApiResult(res, outTypes, do)


class PfxHair(PfxGeometry):
    __melnode__ = 'pfxHair'
    __slots__ = ()


class PfxToon(PfxGeometry):
    __melnode__ = 'pfxToon'
    __slots__ = ()


class Stroke(PfxGeometry):
    __melcmd__ = staticmethod(effects.stroke)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'stroke'
    __melnode__ = 'stroke'
    __slots__ = ()


class RigidBody(Shape):
    __melcmd__ = staticmethod(effects.rigidBody)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'rigidBody'
    __melnode__ = 'rigidBody'
    __slots__ = ()

    @_f.addMelDocs('rigidBody', 'deleteCache')
    def deleteCache(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, effects.rigidBody, kwargs, 'deleteCache', val)

    @_f.addMelDocs('rigidBody', 'active')
    def getActive(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, effects.rigidBody, kwargs, 'active')
        return res

    @_f.addMelDocs('rigidBody', 'angularVelocity')
    def getAngularVelocity(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, effects.rigidBody, kwargs, 'angularVelocity')
        return res

    @_f.addMelDocs('rigidBody', 'applyForceAt')
    def getApplyForceAt(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, effects.rigidBody, kwargs, 'applyForceAt')
        return res

    @_f.addMelDocs('rigidBody', 'bounciness')
    def getBounciness(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, effects.rigidBody, kwargs, 'bounciness')
        return res

    @_f.addMelDocs('rigidBody', 'cache')
    def getCache(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, effects.rigidBody, kwargs, 'cache')
        return res

    @_f.addMelDocs('rigidBody', 'centerOfMass')
    def getCenterOfMass(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, effects.rigidBody, kwargs, 'centerOfMass')
        return res

    @_f.addMelDocs('rigidBody', 'collisions')
    def getCollisions(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, effects.rigidBody, kwargs, 'collisions')
        return res

    @_f.addMelDocs('rigidBody', 'contactCount')
    def getContactCount(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, effects.rigidBody, kwargs, 'contactCount')
        return res

    @_f.addMelDocs('rigidBody', 'contactName')
    def getContactName(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, effects.rigidBody, kwargs, 'contactName')
        return res

    @_f.addMelDocs('rigidBody', 'contactPosition')
    def getContactPosition(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, effects.rigidBody, kwargs, 'contactPosition')
        return res

    @_f.addMelDocs('rigidBody', 'damping')
    def getDamping(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, effects.rigidBody, kwargs, 'damping')
        return res

    @_f.addMelDocs('rigidBody', 'dynamicFriction')
    def getDynamicFriction(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, effects.rigidBody, kwargs, 'dynamicFriction')
        return res

    @_f.addMelDocs('rigidBody', 'force')
    def getForce(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, effects.rigidBody, kwargs, 'force')
        return res

    @_f.addMelDocs('rigidBody', 'ignore')
    def getIgnore(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, effects.rigidBody, kwargs, 'ignore')
        return res

    @_f.addMelDocs('rigidBody', 'initialAngularVelocity')
    def getInitialAngularVelocity(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, effects.rigidBody, kwargs, 'initialAngularVelocity')
        return res

    @_f.addMelDocs('rigidBody', 'initialVelocity')
    def getInitialVelocity(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, effects.rigidBody, kwargs, 'initialVelocity')
        return res

    @_f.addMelDocs('rigidBody', 'layer')
    def getLayer(self, **kwargs):
        # type: (...) -> int
        res = _f.asQuery(self, effects.rigidBody, kwargs, 'layer')
        return res

    @_f.addMelDocs('rigidBody', 'lockCenterOfMass')
    def getLockCenterOfMass(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, effects.rigidBody, kwargs, 'lockCenterOfMass')
        return res

    @_f.addMelDocs('rigidBody', 'mass')
    def getMass(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, effects.rigidBody, kwargs, 'mass')
        return res

    @_f.addMelDocs('rigidBody', 'orientation')
    def getOrientation(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, effects.rigidBody, kwargs, 'orientation')
        return res

    @_f.addMelDocs('rigidBody', 'particleCollision')
    def getParticleCollision(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, effects.rigidBody, kwargs, 'particleCollision')
        return res

    @_f.addMelDocs('rigidBody', 'passive')
    def getPassive(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, effects.rigidBody, kwargs, 'passive')
        return res

    @_f.addMelDocs('rigidBody', 'position')
    def getPosition(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, effects.rigidBody, kwargs, 'position')
        return res

    @_f.addMelDocs('rigidBody', 'removeShape')
    def getRemoveShape(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, effects.rigidBody, kwargs, 'removeShape')
        return res

    @_f.addMelDocs('rigidBody', 'solver')
    def getSolver(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, effects.rigidBody, kwargs, 'solver')
        return res

    @_f.addMelDocs('rigidBody', 'standInObject')
    def getStandInObject(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, effects.rigidBody, kwargs, 'standInObject')
        return res

    @_f.addMelDocs('rigidBody', 'staticFriction')
    def getStaticFriction(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, effects.rigidBody, kwargs, 'staticFriction')
        return res

    @_f.addMelDocs('rigidBody', 'tesselationFactor')
    def getTesselationFactor(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, effects.rigidBody, kwargs, 'tesselationFactor')
        return res

    @_f.addMelDocs('rigidBody', 'velocity')
    def getVelocity(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, effects.rigidBody, kwargs, 'velocity')
        return res

    @_f.addMelDocs('rigidBody', 'impulse')
    def impulse(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, effects.rigidBody, kwargs, 'impulse', val)

    @_f.addMelDocs('rigidBody', 'impulsePosition')
    def impulsePosition(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, effects.rigidBody, kwargs, 'impulsePosition', val)

    @_f.addMelDocs('rigidBody', 'active')
    def setActive(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, effects.rigidBody, kwargs, 'active', val)

    @_f.addMelDocs('rigidBody', 'applyForceAt')
    def setApplyForceAt(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, effects.rigidBody, kwargs, 'applyForceAt', val)

    @_f.addMelDocs('rigidBody', 'bounciness')
    def setBounciness(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, effects.rigidBody, kwargs, 'bounciness', val)

    @_f.addMelDocs('rigidBody', 'cache')
    def setCache(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, effects.rigidBody, kwargs, 'cache', val)

    @_f.addMelDocs('rigidBody', 'centerOfMass')
    def setCenterOfMass(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, effects.rigidBody, kwargs, 'centerOfMass', val)

    @_f.addMelDocs('rigidBody', 'collisions')
    def setCollisions(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, effects.rigidBody, kwargs, 'collisions', val)

    @_f.addMelDocs('rigidBody', 'damping')
    def setDamping(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, effects.rigidBody, kwargs, 'damping', val)

    @_f.addMelDocs('rigidBody', 'dynamicFriction')
    def setDynamicFriction(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, effects.rigidBody, kwargs, 'dynamicFriction', val)

    @_f.addMelDocs('rigidBody', 'ignore')
    def setIgnore(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, effects.rigidBody, kwargs, 'ignore', val)

    @_f.addMelDocs('rigidBody', 'initialAngularVelocity')
    def setInitialAngularVelocity(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, effects.rigidBody, kwargs, 'initialAngularVelocity', val)

    @_f.addMelDocs('rigidBody', 'initialVelocity')
    def setInitialVelocity(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, effects.rigidBody, kwargs, 'initialVelocity', val)

    @_f.addMelDocs('rigidBody', 'layer')
    def setLayer(self, val=True, **kwargs):
        # type: (int, **Any) -> None
        return _f.asEdit(self, effects.rigidBody, kwargs, 'layer', val)

    @_f.addMelDocs('rigidBody', 'lockCenterOfMass')
    def setLockCenterOfMass(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, effects.rigidBody, kwargs, 'lockCenterOfMass', val)

    @_f.addMelDocs('rigidBody', 'mass')
    def setMass(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, effects.rigidBody, kwargs, 'mass', val)

    @_f.addMelDocs('rigidBody', 'orientation')
    def setOrientation(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, effects.rigidBody, kwargs, 'orientation', val)

    @_f.addMelDocs('rigidBody', 'particleCollision')
    def setParticleCollision(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, effects.rigidBody, kwargs, 'particleCollision', val)

    @_f.addMelDocs('rigidBody', 'passive')
    def setPassive(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, effects.rigidBody, kwargs, 'passive', val)

    @_f.addMelDocs('rigidBody', 'position')
    def setPosition(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, effects.rigidBody, kwargs, 'position', val)

    @_f.addMelDocs('rigidBody', 'removeShape')
    def setRemoveShape(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, effects.rigidBody, kwargs, 'removeShape', val)

    @_f.addMelDocs('rigidBody', 'solver')
    def setSolver(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, effects.rigidBody, kwargs, 'solver', val)

    @_f.addMelDocs('rigidBody', 'standInObject')
    def setStandInObject(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, effects.rigidBody, kwargs, 'standInObject', val)

    @_f.addMelDocs('rigidBody', 'staticFriction')
    def setStaticFriction(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, effects.rigidBody, kwargs, 'staticFriction', val)

    @_f.addMelDocs('rigidBody', 'spinImpulse')
    def spinImpulse(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, effects.rigidBody, kwargs, 'spinImpulse', val)


class SoftModHandle(Shape):
    __melnode__ = 'softModHandle'
    __slots__ = ()


class Spring(Shape):
    __melcmd__ = staticmethod(effects.spring)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'spring'
    __melnode__ = 'spring'
    __slots__ = ()

    @_f.addMelDocs('spring', 'allPoints')
    def allPoints(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, effects.spring, kwargs, 'allPoints', val)

    @_f.addMelDocs('spring', 'count')
    def getCount(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, effects.spring, kwargs, 'count')
        return res

    @_f.addMelDocs('spring', 'damping')
    def getDamping(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, effects.spring, kwargs, 'damping')
        return res

    @_f.addMelDocs('spring', 'dampingPS')
    def getDampingPS(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, effects.spring, kwargs, 'dampingPS')
        return res

    @_f.addMelDocs('spring', 'endForceWeight')
    def getEndForceWeight(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, effects.spring, kwargs, 'endForceWeight')
        return res

    @_f.addMelDocs('spring', 'length')
    def getLength(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, effects.spring, kwargs, 'length')
        return res

    @_f.addMelDocs('spring', 'restLength')
    def getRestLength(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, effects.spring, kwargs, 'restLength')
        return res

    @_f.addMelDocs('spring', 'restLengthPS')
    def getRestLengthPS(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, effects.spring, kwargs, 'restLengthPS')
        return res

    @_f.addMelDocs('spring', 'startForceWeight')
    def getStartForceWeight(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, effects.spring, kwargs, 'startForceWeight')
        return res

    @_f.addMelDocs('spring', 'stiffness')
    def getStiffness(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, effects.spring, kwargs, 'stiffness')
        return res

    @_f.addMelDocs('spring', 'stiffnessPS')
    def getStiffnessPS(self, **kwargs):
        # type: (...) -> float
        res = _f.asQuery(self, effects.spring, kwargs, 'stiffnessPS')
        return res

    @_f.addMelDocs('spring', 'useDampingPS')
    def getUseDampingPS(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, effects.spring, kwargs, 'useDampingPS')
        return res

    @_f.addMelDocs('spring', 'useRestLengthPS')
    def getUseRestLengthPS(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, effects.spring, kwargs, 'useRestLengthPS')
        return res

    @_f.addMelDocs('spring', 'useStiffnessPS')
    def getUseStiffnessPS(self, **kwargs):
        # type: (...) -> bool
        res = _f.asQuery(self, effects.spring, kwargs, 'useStiffnessPS')
        return res

    @_f.addMelDocs('spring', 'maxDistance')
    def maxDistance(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, effects.spring, kwargs, 'maxDistance', val)

    @_f.addMelDocs('spring', 'damping')
    def setDamping(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, effects.spring, kwargs, 'damping', val)

    @_f.addMelDocs('spring', 'dampingPS')
    def setDampingPS(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, effects.spring, kwargs, 'dampingPS', val)

    @_f.addMelDocs('spring', 'endForceWeight')
    def setEndForceWeight(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, effects.spring, kwargs, 'endForceWeight', val)

    @_f.addMelDocs('spring', 'length')
    def setLength(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, effects.spring, kwargs, 'length', val)

    @_f.addMelDocs('spring', 'restLength')
    def setRestLength(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, effects.spring, kwargs, 'restLength', val)

    @_f.addMelDocs('spring', 'restLengthPS')
    def setRestLengthPS(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, effects.spring, kwargs, 'restLengthPS', val)

    @_f.addMelDocs('spring', 'startForceWeight')
    def setStartForceWeight(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, effects.spring, kwargs, 'startForceWeight', val)

    @_f.addMelDocs('spring', 'stiffness')
    def setStiffness(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, effects.spring, kwargs, 'stiffness', val)

    @_f.addMelDocs('spring', 'stiffnessPS')
    def setStiffnessPS(self, val=True, **kwargs):
        # type: (float, **Any) -> None
        return _f.asEdit(self, effects.spring, kwargs, 'stiffnessPS', val)

    @_f.addMelDocs('spring', 'useDampingPS')
    def setUseDampingPS(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, effects.spring, kwargs, 'useDampingPS', val)

    @_f.addMelDocs('spring', 'useRestLengthPS')
    def setUseRestLengthPS(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, effects.spring, kwargs, 'useRestLengthPS', val)

    @_f.addMelDocs('spring', 'useStiffnessPS')
    def setUseStiffnessPS(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, effects.spring, kwargs, 'useStiffnessPS', val)


class THdynSpring(Spring):
    __melnode__ = 'THdynSpring'
    __slots__ = ()


class THcustomTransform(Transform):
    __melnode__ = 'THcustomTransform'
    __slots__ = ()


class BaseGeometryVarGroup(Transform):
    __melnode__ = 'baseGeometryVarGroup'
    __slots__ = ()

    @_f.deprecated('Use getRotation(quaternion=True) instead.')
    def getRotationQuaternion(self, space='transform'):
        # type: (datatypes.Space.Space) -> Tuple[float, float, float, float]
        do, final_do, outTypes = _f.getDoArgs([space], [('x', 'double', 'out', None), ('y', 'double', 'out', None), ('z', 'double', 'out', None), ('w', 'double', 'out', None), ('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnTransform, 'getRotationQuaternion', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use setRotation, and pass in a Quaternion object, instead.')
    def setRotationQuaternion(self, x, y, z, w, space='transform'):
        # type: (float, float, float, float, datatypes.Space.Space) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([x, y, z, w, space], [('x', 'double', 'in', None), ('y', 'double', 'in', None), ('z', 'double', 'in', None), ('w', 'double', 'in', None), ('space', ('MSpace', 'Space'), 'in', None)], self.getRotationQuaternion, self.setRotationQuaternion, ['space'])
        res = _f.getProxyResult(self, _api.MFnTransform, 'setRotationQuaternion', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res


class CurveVarGroup(BaseGeometryVarGroup):
    __melnode__ = 'curveVarGroup'
    __slots__ = ()


class GeometryVarGroup(BaseGeometryVarGroup):
    __melnode__ = 'geometryVarGroup'
    __slots__ = ()


class MeshVarGroup(BaseGeometryVarGroup):
    __melnode__ = 'meshVarGroup'
    __slots__ = ()


class SubdivSurfaceVarGroup(BaseGeometryVarGroup):
    __melnode__ = 'subdivSurfaceVarGroup'
    __slots__ = ()


class SurfaceVarGroup(BaseGeometryVarGroup):
    __melnode__ = 'surfaceVarGroup'
    __slots__ = ()


class ClipGhostShape(Transform):
    __melnode__ = 'clipGhostShape'
    __slots__ = ()


class THconstraint(Constraint):
    __melnode__ = 'THconstraint'
    __slots__ = ()


class AimConstraint(Constraint):
    __melcmd__ = staticmethod(animation.aimConstraint)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'aimConstraint'
    __melnode__ = 'aimConstraint'
    __slots__ = ()

    @_f.addMelDocs('aimConstraint', 'aimVector')
    def getAimVector(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, animation.aimConstraint, kwargs, 'aimVector')
        return res

    @_f.addMelDocs('aimConstraint', 'offset')
    def getOffset(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, animation.aimConstraint, kwargs, 'offset')
        return res

    @_f.addMelDocs('aimConstraint', 'targetList')
    def getTargetList(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.aimConstraint, kwargs, 'targetList')
        return res

    @_f.addMelDocs('aimConstraint', 'upVector')
    def getUpVector(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, animation.aimConstraint, kwargs, 'upVector')
        return res

    @_f.addMelDocs('aimConstraint', 'weightAliasList')
    def getWeightAliasList(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.aimConstraint, kwargs, 'weightAliasList')
        return res

    @_f.addMelDocs('aimConstraint', 'worldUpObject')
    def getWorldUpObject(self, **kwargs):
        # type: (...) -> general.PyNode
        res = _f.asQuery(self, animation.aimConstraint, kwargs, 'worldUpObject')
        return res

    @_f.addMelDocs('aimConstraint', 'worldUpType')
    def getWorldUpType(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.aimConstraint, kwargs, 'worldUpType')
        return res

    @_f.addMelDocs('aimConstraint', 'worldUpVector')
    def getWorldUpVector(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, animation.aimConstraint, kwargs, 'worldUpVector')
        return res

    @_f.addMelDocs('aimConstraint', 'layer')
    def layer(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.aimConstraint, kwargs, 'layer', val)

    @_f.addMelDocs('aimConstraint', 'remove')
    def remove(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.aimConstraint, kwargs, 'remove', val)

    @_f.addMelDocs('aimConstraint', 'aimVector')
    def setAimVector(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, animation.aimConstraint, kwargs, 'aimVector', val)

    @_f.addMelDocs('aimConstraint', 'offset')
    def setOffset(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, animation.aimConstraint, kwargs, 'offset', val)

    @_f.addMelDocs('aimConstraint', 'upVector')
    def setUpVector(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, animation.aimConstraint, kwargs, 'upVector', val)

    @_f.addMelDocs('aimConstraint', 'worldUpObject')
    def setWorldUpObject(self, val=True, **kwargs):
        # type: (general.PyNode | str, **Any) -> None
        return _f.asEdit(self, animation.aimConstraint, kwargs, 'worldUpObject', val)

    @_f.addMelDocs('aimConstraint', 'worldUpType')
    def setWorldUpType(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.aimConstraint, kwargs, 'worldUpType', val)

    @_f.addMelDocs('aimConstraint', 'worldUpVector')
    def setWorldUpVector(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, animation.aimConstraint, kwargs, 'worldUpVector', val)

    @_f.addMelDocs('aimConstraint', 'skip')
    def skip(self, val=True, **kwargs):
        # type: (str | List[str], **Any) -> None
        return _f.asEdit(self, animation.aimConstraint, kwargs, 'skip', val)


class LookAt(AimConstraint):
    __melnode__ = 'lookAt'
    __slots__ = ()


class GeometryConstraint(Constraint):
    __melcmd__ = staticmethod(animation.geometryConstraint)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'geometryConstraint'
    __melnode__ = 'geometryConstraint'
    __slots__ = ()

    @_f.addMelDocs('geometryConstraint', 'targetList')
    def getTargetList(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.geometryConstraint, kwargs, 'targetList')
        return res

    @_f.addMelDocs('geometryConstraint', 'weightAliasList')
    def getWeightAliasList(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.geometryConstraint, kwargs, 'weightAliasList')
        return res

    @_f.addMelDocs('geometryConstraint', 'layer')
    def layer(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.geometryConstraint, kwargs, 'layer', val)

    @_f.addMelDocs('geometryConstraint', 'remove')
    def remove(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.geometryConstraint, kwargs, 'remove', val)


class NormalConstraint(Constraint):
    __melcmd__ = staticmethod(animation.normalConstraint)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'normalConstraint'
    __melnode__ = 'normalConstraint'
    __slots__ = ()

    @_f.addMelDocs('normalConstraint', 'aimVector')
    def getAimVector(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, animation.normalConstraint, kwargs, 'aimVector')
        return res

    @_f.addMelDocs('normalConstraint', 'targetList')
    def getTargetList(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.normalConstraint, kwargs, 'targetList')
        return res

    @_f.addMelDocs('normalConstraint', 'upVector')
    def getUpVector(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, animation.normalConstraint, kwargs, 'upVector')
        return res

    @_f.addMelDocs('normalConstraint', 'weightAliasList')
    def getWeightAliasList(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.normalConstraint, kwargs, 'weightAliasList')
        return res

    @_f.addMelDocs('normalConstraint', 'worldUpObject')
    def getWorldUpObject(self, **kwargs):
        # type: (...) -> general.PyNode
        res = _f.asQuery(self, animation.normalConstraint, kwargs, 'worldUpObject')
        return res

    @_f.addMelDocs('normalConstraint', 'worldUpType')
    def getWorldUpType(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.normalConstraint, kwargs, 'worldUpType')
        return res

    @_f.addMelDocs('normalConstraint', 'worldUpVector')
    def getWorldUpVector(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, animation.normalConstraint, kwargs, 'worldUpVector')
        return res

    @_f.addMelDocs('normalConstraint', 'layer')
    def layer(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.normalConstraint, kwargs, 'layer', val)

    @_f.addMelDocs('normalConstraint', 'remove')
    def remove(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.normalConstraint, kwargs, 'remove', val)

    @_f.addMelDocs('normalConstraint', 'aimVector')
    def setAimVector(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, animation.normalConstraint, kwargs, 'aimVector', val)

    @_f.addMelDocs('normalConstraint', 'upVector')
    def setUpVector(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, animation.normalConstraint, kwargs, 'upVector', val)

    @_f.addMelDocs('normalConstraint', 'worldUpObject')
    def setWorldUpObject(self, val=True, **kwargs):
        # type: (general.PyNode | str, **Any) -> None
        return _f.asEdit(self, animation.normalConstraint, kwargs, 'worldUpObject', val)

    @_f.addMelDocs('normalConstraint', 'worldUpType')
    def setWorldUpType(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.normalConstraint, kwargs, 'worldUpType', val)

    @_f.addMelDocs('normalConstraint', 'worldUpVector')
    def setWorldUpVector(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, animation.normalConstraint, kwargs, 'worldUpVector', val)


class OldNormalConstraint(Constraint):
    __melnode__ = 'oldNormalConstraint'
    __slots__ = ()


class OldTangentConstraint(Constraint):
    __melnode__ = 'oldTangentConstraint'
    __slots__ = ()


class OrientConstraint(Constraint):
    __melcmd__ = staticmethod(animation.orientConstraint)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'orientConstraint'
    __melnode__ = 'orientConstraint'
    __slots__ = ()

    @_f.addMelDocs('orientConstraint', 'createCache')
    def createCache(self, val=True, **kwargs):
        # type: (Tuple[float, float], **Any) -> None
        return _f.asEdit(self, animation.orientConstraint, kwargs, 'createCache', val)

    @_f.addMelDocs('orientConstraint', 'deleteCache')
    def deleteCache(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.orientConstraint, kwargs, 'deleteCache', val)

    @_f.addMelDocs('orientConstraint', 'offset')
    def getOffset(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, animation.orientConstraint, kwargs, 'offset')
        return res

    @_f.addMelDocs('orientConstraint', 'targetList')
    def getTargetList(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.orientConstraint, kwargs, 'targetList')
        return res

    @_f.addMelDocs('orientConstraint', 'weightAliasList')
    def getWeightAliasList(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.orientConstraint, kwargs, 'weightAliasList')
        return res

    @_f.addMelDocs('orientConstraint', 'layer')
    def layer(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.orientConstraint, kwargs, 'layer', val)

    @_f.addMelDocs('orientConstraint', 'remove')
    def remove(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.orientConstraint, kwargs, 'remove', val)

    @_f.addMelDocs('orientConstraint', 'offset')
    def setOffset(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, animation.orientConstraint, kwargs, 'offset', val)

    @_f.addMelDocs('orientConstraint', 'skip')
    def skip(self, val=True, **kwargs):
        # type: (str | List[str], **Any) -> None
        return _f.asEdit(self, animation.orientConstraint, kwargs, 'skip', val)


class ParentConstraint(Constraint):
    __melcmd__ = staticmethod(animation.parentConstraint)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'parentConstraint'
    __melnode__ = 'parentConstraint'
    __slots__ = ()

    @_f.addMelDocs('parentConstraint', 'createCache')
    def createCache(self, val=True, **kwargs):
        # type: (Tuple[float, float], **Any) -> None
        return _f.asEdit(self, animation.parentConstraint, kwargs, 'createCache', val)

    @_f.addMelDocs('parentConstraint', 'deleteCache')
    def deleteCache(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.parentConstraint, kwargs, 'deleteCache', val)

    @_f.addMelDocs('parentConstraint', 'targetList')
    def getTargetList(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.parentConstraint, kwargs, 'targetList')
        return res

    @_f.addMelDocs('parentConstraint', 'weightAliasList')
    def getWeightAliasList(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.parentConstraint, kwargs, 'weightAliasList')
        return res

    @_f.addMelDocs('parentConstraint', 'layer')
    def layer(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.parentConstraint, kwargs, 'layer', val)

    @_f.addMelDocs('parentConstraint', 'remove')
    def remove(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.parentConstraint, kwargs, 'remove', val)


class PointConstraint(Constraint):
    __melcmd__ = staticmethod(animation.pointConstraint)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'pointConstraint'
    __melnode__ = 'pointConstraint'
    __slots__ = ()

    @_f.addMelDocs('pointConstraint', 'offset')
    def getOffset(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, animation.pointConstraint, kwargs, 'offset')
        return res

    @_f.addMelDocs('pointConstraint', 'targetList')
    def getTargetList(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.pointConstraint, kwargs, 'targetList')
        return res

    @_f.addMelDocs('pointConstraint', 'weightAliasList')
    def getWeightAliasList(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.pointConstraint, kwargs, 'weightAliasList')
        return res

    @_f.addMelDocs('pointConstraint', 'layer')
    def layer(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.pointConstraint, kwargs, 'layer', val)

    @_f.addMelDocs('pointConstraint', 'remove')
    def remove(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.pointConstraint, kwargs, 'remove', val)

    @_f.addMelDocs('pointConstraint', 'offset')
    def setOffset(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, animation.pointConstraint, kwargs, 'offset', val)

    @_f.addMelDocs('pointConstraint', 'skip')
    def skip(self, val=True, **kwargs):
        # type: (str | List[str], **Any) -> None
        return _f.asEdit(self, animation.pointConstraint, kwargs, 'skip', val)


class PoleVectorConstraint(PointConstraint):
    __melcmd__ = staticmethod(animation.poleVectorConstraint)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'poleVectorConstraint'
    __melnode__ = 'poleVectorConstraint'
    __slots__ = ()

    @_f.addMelDocs('poleVectorConstraint', 'targetList')
    def getTargetList(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.poleVectorConstraint, kwargs, 'targetList')
        return res

    @_f.addMelDocs('poleVectorConstraint', 'weightAliasList')
    def getWeightAliasList(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.poleVectorConstraint, kwargs, 'weightAliasList')
        return res

    @_f.addMelDocs('poleVectorConstraint', 'layer')
    def layer(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.poleVectorConstraint, kwargs, 'layer', val)

    @_f.addMelDocs('poleVectorConstraint', 'remove')
    def remove(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.poleVectorConstraint, kwargs, 'remove', val)


class PointOnPolyConstraint(Constraint):
    __melcmd__ = staticmethod(animation.pointOnPolyConstraint)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'pointOnPolyConstraint'
    __melnode__ = 'pointOnPolyConstraint'
    __slots__ = ()

    @_f.addMelDocs('pointOnPolyConstraint', 'offset')
    def getOffset(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, animation.pointOnPolyConstraint, kwargs, 'offset')
        return res

    @_f.addMelDocs('pointOnPolyConstraint', 'targetList')
    def getTargetList(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.pointOnPolyConstraint, kwargs, 'targetList')
        return res

    @_f.addMelDocs('pointOnPolyConstraint', 'weightAliasList')
    def getWeightAliasList(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.pointOnPolyConstraint, kwargs, 'weightAliasList')
        return res

    @_f.addMelDocs('pointOnPolyConstraint', 'layer')
    def layer(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.pointOnPolyConstraint, kwargs, 'layer', val)

    @_f.addMelDocs('pointOnPolyConstraint', 'remove')
    def remove(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.pointOnPolyConstraint, kwargs, 'remove', val)

    @_f.addMelDocs('pointOnPolyConstraint', 'offset')
    def setOffset(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, animation.pointOnPolyConstraint, kwargs, 'offset', val)

    @_f.addMelDocs('pointOnPolyConstraint', 'skip')
    def skip(self, val=True, **kwargs):
        # type: (str | List[str], **Any) -> None
        return _f.asEdit(self, animation.pointOnPolyConstraint, kwargs, 'skip', val)


class ScaleConstraint(Constraint):
    __melcmd__ = staticmethod(animation.scaleConstraint)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'scaleConstraint'
    __melnode__ = 'scaleConstraint'
    __slots__ = ()

    @_f.addMelDocs('scaleConstraint', 'offset')
    def getOffset(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, animation.scaleConstraint, kwargs, 'offset')
        return res

    @_f.addMelDocs('scaleConstraint', 'targetList')
    def getTargetList(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.scaleConstraint, kwargs, 'targetList')
        return res

    @_f.addMelDocs('scaleConstraint', 'weightAliasList')
    def getWeightAliasList(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.scaleConstraint, kwargs, 'weightAliasList')
        return res

    @_f.addMelDocs('scaleConstraint', 'layer')
    def layer(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.scaleConstraint, kwargs, 'layer', val)

    @_f.addMelDocs('scaleConstraint', 'remove')
    def remove(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.scaleConstraint, kwargs, 'remove', val)

    @_f.addMelDocs('scaleConstraint', 'scaleCompensate')
    def scaleCompensate(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.scaleConstraint, kwargs, 'scaleCompensate', val)

    @_f.addMelDocs('scaleConstraint', 'offset')
    def setOffset(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, animation.scaleConstraint, kwargs, 'offset', val)

    @_f.addMelDocs('scaleConstraint', 'skip')
    def skip(self, val=True, **kwargs):
        # type: (str | List[str], **Any) -> None
        return _f.asEdit(self, animation.scaleConstraint, kwargs, 'skip', val)


class SymmetryConstraint(Constraint):
    __melnode__ = 'symmetryConstraint'
    __slots__ = ()


class TangentConstraint(Constraint):
    __melcmd__ = staticmethod(animation.tangentConstraint)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'tangentConstraint'
    __melnode__ = 'tangentConstraint'
    __slots__ = ()

    @_f.addMelDocs('tangentConstraint', 'aimVector')
    def getAimVector(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, animation.tangentConstraint, kwargs, 'aimVector')
        return res

    @_f.addMelDocs('tangentConstraint', 'targetList')
    def getTargetList(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.tangentConstraint, kwargs, 'targetList')
        return res

    @_f.addMelDocs('tangentConstraint', 'upVector')
    def getUpVector(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, animation.tangentConstraint, kwargs, 'upVector')
        return res

    @_f.addMelDocs('tangentConstraint', 'weightAliasList')
    def getWeightAliasList(self, **kwargs):
        # type: (...) -> Any
        res = _f.asQuery(self, animation.tangentConstraint, kwargs, 'weightAliasList')
        return res

    @_f.addMelDocs('tangentConstraint', 'worldUpObject')
    def getWorldUpObject(self, **kwargs):
        # type: (...) -> general.PyNode
        res = _f.asQuery(self, animation.tangentConstraint, kwargs, 'worldUpObject')
        return res

    @_f.addMelDocs('tangentConstraint', 'worldUpType')
    def getWorldUpType(self, **kwargs):
        # type: (...) -> str
        res = _f.asQuery(self, animation.tangentConstraint, kwargs, 'worldUpType')
        return res

    @_f.addMelDocs('tangentConstraint', 'worldUpVector')
    def getWorldUpVector(self, **kwargs):
        # type: (...) -> Tuple[float, float, float]
        res = _f.asQuery(self, animation.tangentConstraint, kwargs, 'worldUpVector')
        return res

    @_f.addMelDocs('tangentConstraint', 'layer')
    def layer(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.tangentConstraint, kwargs, 'layer', val)

    @_f.addMelDocs('tangentConstraint', 'remove')
    def remove(self, val=True, **kwargs):
        # type: (bool | int, **Any) -> None
        return _f.asEdit(self, animation.tangentConstraint, kwargs, 'remove', val)

    @_f.addMelDocs('tangentConstraint', 'aimVector')
    def setAimVector(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, animation.tangentConstraint, kwargs, 'aimVector', val)

    @_f.addMelDocs('tangentConstraint', 'upVector')
    def setUpVector(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, animation.tangentConstraint, kwargs, 'upVector', val)

    @_f.addMelDocs('tangentConstraint', 'worldUpObject')
    def setWorldUpObject(self, val=True, **kwargs):
        # type: (general.PyNode | str, **Any) -> None
        return _f.asEdit(self, animation.tangentConstraint, kwargs, 'worldUpObject', val)

    @_f.addMelDocs('tangentConstraint', 'worldUpType')
    def setWorldUpType(self, val=True, **kwargs):
        # type: (_util.ProxyUnicode | str, **Any) -> None
        return _f.asEdit(self, animation.tangentConstraint, kwargs, 'worldUpType', val)

    @_f.addMelDocs('tangentConstraint', 'worldUpVector')
    def setWorldUpVector(self, val=True, **kwargs):
        # type: (Tuple[float, float, float], **Any) -> None
        return _f.asEdit(self, animation.tangentConstraint, kwargs, 'worldUpVector', val)


class DagContainer(Transform):
    __melnode__ = 'dagContainer'
    __slots__ = ()

    @_f.deprecated('Use getRotation(quaternion=True) instead.')
    def getRotationQuaternion(self, space='transform'):
        # type: (datatypes.Space.Space) -> Tuple[float, float, float, float]
        do, final_do, outTypes = _f.getDoArgs([space], [('x', 'double', 'out', None), ('y', 'double', 'out', None), ('z', 'double', 'out', None), ('w', 'double', 'out', None), ('space', ('MSpace', 'Space'), 'in', None)])
        res = _f.getProxyResult(self, _api.MFnTransform, 'getRotationQuaternion', final_do)
        return _f.processApiResult(res, outTypes, do)

    @_f.deprecated('Use setRotation, and pass in a Quaternion object, instead.')
    def setRotationQuaternion(self, x, y, z, w, space='transform'):
        # type: (float, float, float, float, datatypes.Space.Space) -> None
        do, final_do, outTypes, undoItem = _f.getDoArgsGetterUndo([x, y, z, w, space], [('x', 'double', 'in', None), ('y', 'double', 'in', None), ('z', 'double', 'in', None), ('w', 'double', 'in', None), ('space', ('MSpace', 'Space'), 'in', None)], self.getRotationQuaternion, self.setRotationQuaternion, ['space'])
        res = _f.getProxyResult(self, _api.MFnTransform, 'setRotationQuaternion', final_do)
        if undoItem is not None: _f.apiUndo.append(undoItem)
        return res


class Assembly(DagContainer):
    __apicls__ = _api.MFnAssembly
    __melcmd__ = staticmethod(general.assembly)
    __melcmd_isinfo__ = False
    __melcmdname__ = 'assembly'
    __melnode__ = 'assembly'
    __slots__ = ()

    @_f.addApiDocs(_api.MFnAssembly, 'activate')
    def activate(self, representation):
        # type: (str) -> None
        do, final_do, outTypes = _f.getDoArgs([representation], [('representation', 'MString', 'in', None)])
       