# -*- coding: utf-8 -*-

import os
import re
import time
import xml.etree.ElementTree as ET
import datetime
import glob
import subprocess
import string
import difflib
import json
import random
from xml.sax.saxutils import escape, unescape

EvaluatorServerPath = os.getenv("EV_SERVER_PATH")
EvaluatorLocalPath = os.getenv("EV_LOCAL_PATH")

evaluatorTemplate =\
'''
#include <evtypes.hpp>
#include <draw.hpp>
#include <gpu.hpp>
#include <soa/soa.h>

#include "patch_ispc.h"

using namespace std;

@INCLUDES

@DEFINES

class MyNode : public EvNode
{
  public:
    void getAttributes(GetAttributes &);
    EvStatus run(RunAttributes &, DeformAttributes &);

    void draw(MHWRender::MUIDrawManager &);

    virtual EvStatus gpuRun(RunAttributes &, GPUDeformerData &);

    void gpuTerminate();

    @PRIVATE

private:
    bool STARTUP = true;
    bool GPU_STARTUP = true;
};

DLL_EXPORT EvNode *create() { return new MyNode; }

@COMMANDS

BEGIN_COMMAND(help)
SET_RESULT(R"(Commands: @HELP_COMMAND)");
END_COMMAND

void MyNode::draw(MHWRender::MUIDrawManager &DRAWER)
{
   {@DRAW}
}

void MyNode::getAttributes(GetAttributes &attributes)
{
    EvAttribute attr;

    @GET_ATTRIBUTES
}

EvStatus MyNode::run(RunAttributes &ATTRIBUTES, DeformAttributes &DEFORM_ATTRIBUTES)
{
    EvStatus STATUS = kNormal;

    GET_INPUT_ATTRIBUTE(ATTRIBUTES, EvInt, DEBUG_MODE);
    GET_INPUT_ATTRIBUTE(ATTRIBUTES, EvBoolean, CACHE_ATTRIBUTES);
    GET_INPUT_ATTRIBUTE(ATTRIBUTES, EvFloat, ENVELOPE);
    GET_INPUT_ATTRIBUTE(ATTRIBUTES, EvBoolean, SOA_ENABLED);

    auto *SOA_POINTS_PTR = (SoA<double, double, double>*)shared_in["SOA_POINTS"];
    auto &SOA_POINTS = *SOA_POINTS_PTR;
    constexpr int XS = 0, YS = 1, ZS = 2;

    const auto IS_DEFORMER = DEFORM_ATTRIBUTES.ok;
    const auto &LOCAL2WORLD = DEFORM_ATTRIBUTES.local2world;
    const auto LOCAL2WORLD_INVERSE = LOCAL2WORLD.inverse();
    const auto &DATA = DEFORM_ATTRIBUTES.data;
    const auto &GEOM_INDEX = DEFORM_ATTRIBUTES.geomIndex;
    const int NUM_ELEMENTS = SOA_ENABLED ? SOA_POINTS.size() : DEFORM_ATTRIBUTES.points.size();

    auto &POINTS = DEFORM_ATTRIBUTES.points;
    const auto &WEIGHTS = DEFORM_ATTRIBUTES.weights;
    const auto &INDICES = DEFORM_ATTRIBUTES.indices;

    if (ENVELOPE < EPSILON)
        return kNormal;

    @ATTRIBUTES

    @RUN_TEMPLATE

    STARTUP = false;

    return STATUS;
}

EvStatus MyNode::gpuRun(RunAttributes &ATTRIBUTES, GPUDeformerData &GPU_DATA)
{
    @GPU_TEMPLATE
}

void MyNode::gpuTerminate()
{
    @GPU_TERMINATE

    GPU_STARTUP = true;
}

@GPU_CL_PROGRAMS
'''

evaluatorRunTemplate =\
'''
// Run
{@RUN}

if (IS_DEFORMER)
{
    const bool GPU_INIT = false;

    // Deform Init
    {@DEFORM_INIT}

    // Deform
    {@DEFORM}

    // Deform parallel per point
    if (!SOA_ENABLED)
    {
        @DEFORM_PARALLEL_TEMPLATE
    }
}
'''

evaluatorDeformParallelTemplate =\
'''
BEGIN_PARALLEL_FOR (I, POINTS.size())
{
   if (WEIGHTS[I] < EPSILON)
     continue;

   V3d &POINT = POINTS[I];

   // Deform per point
   {@DEFORM_PARALLEL}
}
END_PARALLEL_FOR
'''

evaluatorGpuTemplate =\
'''
EvStatus STATUS = kNormal;

GET_INPUT_ATTRIBUTE(ATTRIBUTES, EvInt, DEBUG_MODE);
GET_INPUT_ATTRIBUTE(ATTRIBUTES, EvBoolean, CACHE_ATTRIBUTES);
GET_INPUT_ATTRIBUTE(ATTRIBUTES, EvFloat, ENVELOPE);

MAutoCLMem &INPUT_BUFFER = GPU_DATA.inputBuffer;
MAutoCLMem &OUTPUT_BUFFER = GPU_DATA.outputBuffer;
MAutoCLEvent &INPUT_EVENT = GPU_DATA.inputEvent;
MAutoCLEvent &OUTPUT_EVENT = GPU_DATA.outputEvent;
MAutoCLMem &INTERMEDIATE_GEOM_BUFFER = GPU_DATA.intermediateGeomBuffer;
const MAutoCLMem &LOCAL2WORLD_BUFFER = GPU_DATA.local2worldBuffer;
const MAutoCLMem &LOCAL2WORLD_INVERSE_BUFFER = GPU_DATA.local2worldInverseBuffer;
const MAutoCLMem &WEIGHTS_BUFFER = GPU_DATA.weightsBuffer;
const int &NUM_ELEMENTS = GPU_DATA.numElements;

const bool STARTUP = GPU_STARTUP;
const bool GPU_INIT = true;

@ATTRIBUTES

// Deform Init
{@DEFORM_INIT}

bool CL_firstKernel = true; // when false, copy OUTPUT_BUFFER to INTERMEDIATE_GEOM_BUFFER

@GPU_RUN_KERNEL

GPU_STARTUP = false;
return kNormal;
'''

def isFileLocked(path):
    try:
        with open(path, "r+"):
            pass
    except Exception as err:
        if err.errno == 13: # permissions denied
            return True

    return False

def formatCpp(text):
    startupinfo = subprocess.STARTUPINFO()
    startupinfo.dwFlags |= subprocess.STARTF_USESHOWWINDOW

    cwd = os.getcwd()
    os.chdir(EvaluatorLocalPath) # set root directory for .clang-format file

    process = subprocess.Popen([EvaluatorLocalPath+"/clang/bin/clang-format.exe"],
                               stdin=subprocess.PIPE,
                               stdout=subprocess.PIPE,
                               stderr=subprocess.STDOUT,
                               startupinfo=startupinfo)

    data, err = process.communicate(text.encode("utf-8"))

    os.chdir(cwd)

    return data.decode("utf-8") if not err else text

def cpp2line(path):
    lines = []
    with open(path, "r") as f:
        for line in f.read().split("\n"):
            line = line.strip()
            if not line:
                continue

            lines.append(line.split("//")[0]) # skip comments

    return "".join(lines)

def clangCLCheck(fileName):
    startupinfo = subprocess.STARTUPINFO()
    startupinfo.dwFlags |= subprocess.STARTF_USESHOWWINDOW
    process = subprocess.Popen([EvaluatorServerPath+"/nodes/clang_cl_check.bat", fileName],
                               stdin=subprocess.PIPE,
                               stdout=subprocess.PIPE,
                               stderr=subprocess.STDOUT,
                               startupinfo=startupinfo)

    data, err = process.communicate()
    return process.returncode, data

def clangBuild(fileName, codeType, mayaVersion):
    startupinfo = subprocess.STARTUPINFO()
    startupinfo.dwFlags |= subprocess.STARTF_USESHOWWINDOW
    process = subprocess.Popen([EvaluatorServerPath+"/nodes/clang_build.bat", fileName, codeType,  str(mayaVersion)],
                               stdin=subprocess.PIPE,
                               stdout=subprocess.PIPE,
                               stderr=subprocess.STDOUT,
                               startupinfo=startupinfo)

    data, err = process.communicate()
    return process.returncode, data

def ispcBuild(fileName):
    startupinfo = subprocess.STARTUPINFO()
    startupinfo.dwFlags |= subprocess.STARTF_USESHOWWINDOW
    process = subprocess.Popen([EvaluatorServerPath+"/ispc/ispc_build.bat", fileName],
                               stdin=subprocess.PIPE,
                               stdout=subprocess.PIPE,
                               stderr=subprocess.STDOUT,
                               startupinfo=startupinfo)

    data, err = process.communicate()
    return process.returncode, data    

def clangAutoCompletion(rootFile, currentFile, line, column, CL=False):
    startupinfo = subprocess.STARTUPINFO()
    startupinfo.dwFlags |= subprocess.STARTF_USESHOWWINDOW

    command = "clang_completion" if not CL else "clang_cl_completion"
    process = subprocess.Popen([EvaluatorServerPath+"/nodes/"+command+".bat", "%s:%s:%s"%(currentFile, line, column), rootFile],
                               stdout=subprocess.PIPE,
                               stderr=subprocess.PIPE,
                               startupinfo=startupinfo)

    MaxItems = 300

    items = []
    for i, line in enumerate(process.stdout):
        if i > MaxItems:
            break

        if not line:
            continue

        r = re.search("^COMPLETION: ([^:]+) : \\[([^\\]]+)\\][^\\(]+\\(([^\\)]*)\\)", line) # function expression
        if r:
            name, ret, args = r.groups("")
            args = ", ".join(re.findall("<#(?:([^(#>)]*)#>)", args))
            items.append("%s [%s] (%s)"%(name, ret.replace("#",""), args))
        else:
            r = re.search("^COMPLETION: ([^:]+) : \\[([^\\]]+)\\]+", line) # variable expression
            if r:
                name, args = r.groups("")
                items.append("%s %s"%(name, args.replace("#","")))
            else:
                r = re.search("^COMPLETION: ([^:]+) :", line) # types expression
                if r:
                    name = r.groups("")[0]
                    items.append(name)

    return items

def diffText(text1, text2):
    d = difflib.Differ()
    return list(d.compare(text1.splitlines(), text2.splitlines()))

def codeTypeToLabel(codeType):
    codeTypeItems = [c.lower() for c in re.findall(r'[A-Z]?[a-z]+|[A-Z]+(?=[A-Z]|$)', codeType)]
    return " ".join([c.capitalize() for c in codeTypeItems[:-1]])

class Attribute(object):
    CL_none = 0
    CL_Global = 1
    CL_Constant = 2
    CL_GlobalAlloc = 3
    CL_LocalAlloc = 4

    CppTypesToCL = {"cl_int": "int",
                    "cl_float": "float",
                    "cl_float4": "float4",
                    "cl_float16": "float16",
                    "bool": "int",
                    "int": "int",
                    "float": "float",
                    "double": "float",
                    "V3f": "float3",
                    "V3d": "float3",
                    "Quatf": "float4",
                    "Quatd": "float4",
                    "M44f": "float16",
                    "M44d": "float16",
                    "vector<cl_int>": "int*",
                    "vector<cl_float>": "float*",
                    "vector<cl_float3>": "float3*",
                    "vector<cl_float4>": "float4*",
                    "vector<cl_float16>": "float4*",
                    "vector<bool>": "int*",
                    "vector<int>": "int*",
                    "vector<float>": "float*",
                    "vector<double>": "float*",
                    "vector<V3f>": "float3*",
                    "vector<V3d>": "float3*",
                    "vector<Quatf>": "float4*",
                    "vector<Quatd>": "float4*",
                    "vector<M44f>": "float16*",
                    "vector<M44d>": "float16*",
                    "EvBoolean": "int",
                    "EvInt": "int",
                    "EvFloat": "float",
                    "EvDouble": "float",
                    "EvVector": "float3",
                    "EvMatrix": "float16",
                    "EvEnum": "int"}

    EvTypes = ["EvBoolean",
               "EvInt",
               "EvFloat",
               "EvDouble",
               "EvVector",
               "EvMatrix",
               "EvMesh",
               "EvNurbsCurve",
               "EvNurbsSurface",
               "EvString",
               "EvCompound",
               "EvEnum",
               "EvMessage"]

    EvTypesToCpp = {"EvBoolean": "bool",
                    "EvInt": "int",
                    "EvFloat": "float",
                    "EvDouble": "double",
                    "EvVector": "V3d",
                    "EvMatrix": "M44d",
                    "EvMesh": "MObject",
                    "EvNurbsCurve": "NurbsCurve",
                    "EvNurbsSurface": "MObject",
                    "EvString": "string",
                    "EvEnum": "int",
                    "EvMessage":"MPlug",
                    "bool":"EvBoolean",
                    "int":"EvInt",
                    "float":"EvFloat",
                    "double":"EvDouble",
                    "V3d":"EvVector",
                    "M44d":"EvMatrix",
                    "string":"EvString",
                    "MPlug": "EvMessage"}

    TypeRefExp = "((?:\\/|\\$)*\\w*)::(\\w+)"

    def __init__(self,
                 name,
                 type,
                 output=False,
                 defaultValue="",
                 keyable=True,
                 hidden=False,
                 minValue="",
                 maxValue="",
                 cached=False,
                 array=False,
                 cl=CL_none,
                 connectedOnly=False,
                 description="",
                 child=False,
                 items="",
                 muted=False):

        self.name = name
        self.type = type

        self.output = output

        self.defaultValue = defaultValue

        self.keyable = keyable
        self.hidden = hidden

        self.minValue = minValue
        self.maxValue = maxValue

        self.cached = cached
        self.array = array
        self.cl = array
        self.connectedOnly = connectedOnly

        self.description = description
        self.child = child
        self.items = items

        self.muted = muted

    def __eq__(self, other):
        ok = True
        ok = ok and self.name == other.name
        ok = ok and self.type == other.type
        ok = ok and self.output == other.output
        ok = ok and self.defaultValue == other.defaultValue
        ok = ok and self.keyable == other.keyable
        ok = ok and self.hidden == other.hidden
        ok = ok and self.array == other.array
        ok = ok and self.cached == other.cached
        ok = ok and self.minValue == other.minValue
        ok = ok and self.maxValue == other.maxValue
        ok = ok and self.cl == other.cl
        ok = ok and self.connectedOnly == other.connectedOnly
        ok = ok and self.description == other.description
        ok = ok and self.child == other.child
        ok = ok and self.items == other.items
        ok = ok and self.muted == other.muted

        return ok

    def copy(self):
        a = Attribute(self.name, self.type)

        a.output = self.output

        a.defaultValue = self.defaultValue

        a.keyable = self.keyable
        a.hidden = self.hidden

        a.minValue = self.minValue
        a.maxValue = self.maxValue

        a.cached = self.cached
        a.array = self.array
        a.cl = self.cl
        a.connectedOnly = self.connectedOnly

        a.description = self.description
        a.child = self.child
        a.items = self.items

        a.muted = self.muted

        return a

    def toXml(self):
        def toStr(v):
            return v if v!=None else ""

        template = "<attr>\n"
        template += "<name>%s</name>\n"%self.name.strip()
        template += "<type><![CDATA[%s]]></type>\n"%self.type
        template += "<output>%d</output>\n"%self.output
        template += "<defaultValue><![CDATA[%s]]></defaultValue>\n"%toStr(self.defaultValue)
        template += "<keyable>%d</keyable>\n"%self.keyable
        template += "<hidden>%d</hidden>\n"%self.hidden
        template += "<minValue>%s</minValue>\n"%toStr(self.minValue)
        template += "<maxValue>%s</maxValue>\n"%toStr(self.maxValue)
        template += "<cached>%d</cached>\n"%self.cached
        template += "<array>%d</array>\n"%self.array
        template += "<cl>%d</cl>\n"%self.cl
        template += "<connectedOnly>%d</connectedOnly>\n"%self.connectedOnly
        template += "<description><![CDATA[%s]]></description>"%self.description.strip()
        template += "<child>%d</child>"%self.child
        template += "<items><![CDATA[%s]]></items>"%self.items
        template += "<muted>%d</muted>"%toStr(self.muted)
        template += "</attr>"

        return template

    @staticmethod
    def fromXml(attrElement):
        a = Attribute("", type="")

        a.name = attrElement.findtext("name").strip()
        a.type = attrElement.findtext("type").strip()
        a.output = int(attrElement.findtext("output").strip())
        a.defaultValue = attrElement.findtext("defaultValue").strip()
        a.keyable = int(attrElement.findtext("keyable").strip())
        a.hidden = int(attrElement.findtext("hidden").strip())
        a.minValue = attrElement.findtext("minValue").strip()
        a.maxValue = attrElement.findtext("maxValue").strip()
        a.cached = int(attrElement.findtext("cached").strip())
        a.array = int(attrElement.findtext("array").strip())
        a.cl = int(attrElement.findtext("cl") or 0)
        a.connectedOnly = int(attrElement.findtext("connectedOnly").strip())
        a.description = attrElement.findtext("description").strip()
        a.child = int(attrElement.findtext("child") or False)
        a.items = (attrElement.findtext("items") or "").strip()
        a.muted = int(attrElement.findtext("muted") or 0)

        return a

    def isEmpty(self):
        return not self.type

    def isReference(self):
        return re.match("^\\$|/", self.defaultValue) or self.type.endswith("&")

    def isMaya(self):
        return self.type in Attribute.EvTypes

    def isCpp(self):
       return not self.isMaya()

    def isKeyable(self):
        return self.keyable and not self.output and not self.hidden and self.type in ["EvBoolean", "EvInt", "EvFloat", "EvDouble", "EvVector", "EvEnum"]

    def toCLType(self):
        if self.cl == Attribute.CL_none:
            return ""

        typ = Attribute.CppTypesToCL.get(self.type)
        if typ:
            if self.isMaya() and self.array:
                typ += "*"
        else: # custom structs
            typ = re.sub(Attribute.TypeRefExp, "\\2", self.type) # remove links
            if typ.startswith("vector<"):
                typ = re.sub("vector<(\\w+)>", "\\1", typ) + "*"

            typ = "struct " + typ

        return typ

    def needCLBuffer(self):
        return self.toCLType().endswith("*") or self.type.startswith("vector<") # only pointers needs MAutoCLMem buffer

    def toCLKernelArg(self):
        if self.isEmpty() or self.cl == Attribute.CL_none or self.muted:
            return ""

        CL_memtypes = {Attribute.CL_Global: "global const",
                       Attribute.CL_Constant: "constant",
                       Attribute.CL_GlobalAlloc: "global",
                       Attribute.CL_LocalAlloc: "local"}

        allocator = self.cl in [Attribute.CL_GlobalAlloc, Attribute.CL_LocalAlloc]

        memtype = CL_memtypes[self.cl] if self.needCLBuffer() or allocator else "const"

        return "{memtype} {type} @{name}".format(memtype=memtype,
                                                 type=self.toCLType(),
                                                 name=self.name)

    def toGetAttributesCpp(self, namespace="", path=""):
        def toCppBool(v):
            return "true" if v else "false"

        if self.isEmpty() or not self.isMaya() or self.muted:
            return ""

        pathName = path.replace("/", "__")

        attrVar = namespace+self.name+"_attr" if self.child else "attr"
        varTemplate = "attr = EvAttribute" if not self.child else "auto %s = EvAttribute"%attrVar

        argumentsTemplate = "(\"{name}\", \"{localName}\", EvAttributeType::{type}, {output});".format(
            name=namespace + self.name,
            localName=pathName + self.name,
            type=self.type.replace("Ev", "k"),
            output=toCppBool(self.output))

        templates = ["// %s attribute"%(path+self.name)]
        templates.append(varTemplate+argumentsTemplate)

        if self.type not in ["EvCompound", "EvMesh", "EvNurbsSurface", "EvNurbsCurve", "EvMatrix"] and not self.output:
            if self.defaultValue:
                if self.type == "EvVector":
                    templates.append("%s.default_vector = V3d(%s);"%(attrVar, self.defaultValue))
                elif self.type == "EvString":
                    templates.append("%s.default_string = string(%s);"%(attrVar, self.defaultValue))
                else:
                    templates.append("%s.default_value = %s;"%(attrVar, self.defaultValue))

            if self.minValue:
                if self.type == "EvVector":
                    templates.append("{attr}.has_min = true; {attr}.min_vector = V3d({value});".format(attr=attrVar, value=self.minValue))
                else:
                    templates.append("{attr}.has_min = true; {attr}.min = {value};".format(attr=attrVar, value=self.minValue))

            if self.maxValue:
                if self.type == "EvVector":
                    templates.append("{attr}.has_max = true; {attr}.max_vector = V3d({value});".format(attr=attrVar, value=self.maxValue))
                else:
                    templates.append("{attr}.has_max = true; {attr}.max = {value};".format(attr=attrVar, value=self.maxValue))

        templates.append("%s.keyable = %s;"%(attrVar, toCppBool(self.isKeyable())))
        templates.append("%s.cached = %s;"%(attrVar, toCppBool(self.cached and not self.output)))
        templates.append("%s.hidden = %s;"%(attrVar, toCppBool(self.hidden)))
        templates.append("%s.array = %s;"%(attrVar, toCppBool(self.array)))
        templates.append("%s.connected_only = %s;"%(attrVar, toCppBool(self.connectedOnly and self.array)))

        if self.items:
            items = re.split("[ ,]*", self.items.strip())
            if self.type == "EvCompound":
                children = [namespace+it+"_attr" for it in items]
                templates.append("%s.children = {%s};"%(attrVar, ",".join(children)))
            elif self.type == "EvEnum":
                items = ["\"%s\""%it for it in items]
                templates.append("%s.items = {%s};"%(attrVar, ",".join(items)))

        if not self.child:
            templates.append("ADD_ATTRIBUTE(%s);"%attrVar)

        return "\n".join(templates)

    def toRunCpp(self, namespace="", path="", patches=[]):
        if self.isEmpty() or self.muted:
            return ""

        pathName = path.replace("/","__")

        templates = []
        if self.isMaya(): # for maya attributes
            cmd_in = "GET_INPUT_ATTRIBUTE" if not self.array else "GET_INPUT_ARRAY_ATTRIBUTE"
            cmd_out = "GET_OUTPUT_ATTRIBUTE" if not self.array else "GET_OUTPUT_ARRAY_ATTRIBUTE"

            get = "{command}(ATTRIBUTES, {type}, {var});"
            get = get.format(command=cmd_out if self.output or self.type == "EvCompound" else cmd_in,
                             type=self.type if self.type != "EvEnum" else "EvInt",
                             name=namespace + self.name,
                             var=pathName + self.name)

            templates.append(get)

            if self.type == "EvCompound" and not self.array:
                for c in self.items.split(","): # get compound children
                    attr = patches[-1].findAttribute(c.strip())
                    if attr:
                        attr = attr[0]
                        attrName = pathName + attr.name

                        attrCpp = attr.toRunCpp(namespace, path, patches).replace("(ATTRIBUTES, ", "(%s,"%(pathName+self.name))

                        templates.append(attrCpp)
                    else:
                        raise Exception("toRunCpp: cannot find compound child '%s.%s'"%(pathName+self.name, c))

        else: # c++ attributes
            isRef = self.isReference()
            value = self.defaultValue

            if value and isRef:
                resolvedValue = ""
                if value.startswith("$"):
                    exprPatchName = value.split("/")[0][1:] # $biped/leg/aux => biped
                    exprPatchData = patches[0].findPatchByName(exprPatchName)
                    if exprPatchData:
                        _, exprPatchPath = exprPatchData[0] # use first found
                        resolvedValue = value.replace("$"+exprPatchName, exprPatchPath)
                    else:
                        raise Exception("toRunCpp: cannot find '%s'"%exprPatchName)

                elif value.startswith("/"):
                    parentPath = "/".join(path.split("/")[:-2])
                    resolvedValue = parentPath + value

                if resolvedValue:
                    value = resolvedValue.replace("/", "__")

            if value:
                value = value.replace("@", pathName)
                value = " = %s"%value if isRef or self.cached else "(%s)"%value

            varType = self.type[:-1] if self.type.endswith("&") else self.type

            r = re.search(Attribute.TypeRefExp, self.type)
            if r:
                link, typ = r.groups("")

                parentNamespace = ""
                if not link:
                    parentNamespace = pathName[:-2]
                else:
                    rootPatch = None
                    if link.startswith("/") and len(patches) > 1:
                        rootPatch = patches[-2]
                    elif link.startswith("$"):
                        rootPatch = Patch()
                        rootPatch.children.append(patches[0])

                    if rootPatch:
                        parentPatch = rootPatch.resolveLink(link, attribute=False)
                        parentNamespace = patches[0].getNamespaceFor(parentPatch) if parentPatch else ""
                    else:
                        parentNamespace = ""

                varType = re.sub(Attribute.TypeRefExp, parentNamespace + "_namespace::" + typ, self.type)

            attrCpp = "{type} {ref}{var}{rest};".format(type="auto" if isRef else varType,
                                                        ref="&" if isRef else "",
                                                        var=pathName + self.name,
                                                        rest=value)
            templates.append(attrCpp)

        return "\n".join(templates)

class Patch(object):
    StateNormal = 0
    StateMuted = 1
    StateMutedAll = 2

    EmbedCode = True
    ParallelEnabled = True

    VersionPattern = "v(\\d{3})"

    def __init__(self,
                 name="patch",
                 type="",
                 attributes=[],
                 children=[],
                 parallelRun=False,
                 parallelDeformInit=False,
                 parallelDeform=False,
                 version=0,
                 state=False,
                 hasNamespace=False,
                 customNamespace="",
                 debugOpenCL=False,
                 block=False,
                 description="",
                 definesCode="",
                 commandsCode="",
                 drawCode="",
                 deformInitCode="",
                 deformCode="",
                 deformParallelCode="",
                 runCode="",
                 clCode="",
                 ispcCode="",
                 codeBlocks=True):

        self.name = name.strip()
        self.type = type.strip()

        self.attributes = list(attributes)
        self.children = list(children)

        self.parallelRun = parallelRun
        self.parallelDeform = parallelDeform
        self.parallelDeformInit = parallelDeformInit

        self.version = version

        self.state = state
        self.hasNamespace = hasNamespace
        self.customNamespace = customNamespace
        self.debugOpenCL = debugOpenCL
        self.block = block

        self.description = description

        self.definesCode = definesCode
        self.commandsCode = commandsCode
        self.deformInitCode = deformInitCode
        self.deformCode = deformCode
        self.deformParallelCode = deformParallelCode
        self.runCode = runCode
        self.drawCode = drawCode
        self.clCode = clCode
        self.ispcCode = ispcCode

        self.codeBlocks = codeBlocks

        self.embedCode = False

        self.userData = {}

        self.loadedFrom = ""

    def __eq__(self, other):
        ok = True
        ok = ok and self.name == other.name
        ok = ok and self.type == other.type

        ok = ok and len(self.attributes) == len(other.attributes) and all([a==b for a,b in zip(self.attributes, other.attributes)])
        ok = ok and len(self.children) == len(other.children) and all([a==b for a,b in zip(self.children, other.children)])

        ok = ok and self.parallelRun == other.parallelRun
        ok = ok and self.parallelDeform == other.parallelDeform
        ok = ok and self.parallelDeformInit == other.parallelDeformInit

        ok = ok and self.version == other.version
        ok = ok and self.state == other.state
        ok = ok and self.hasNamespace == other.hasNamespace
        ok = ok and self.customNamespace == other.customNamespace
        ok = ok and self.debugOpenCL == other.debugOpenCL
        ok = ok and self.block == other.block

        ok = ok and self.description == other.description

        ok = ok and self.definesCode == other.definesCode
        ok = ok and self.commandsCode == other.commandsCode
        ok = ok and self.deformInitCode == other.deformInitCode
        ok = ok and self.deformCode == other.deformCode
        ok = ok and self.deformParallelCode == other.deformParallelCode
        ok = ok and self.runCode == other.runCode
        ok = ok and self.drawCode == other.drawCode
        ok = ok and self.clCode == other.clCode
        ok = ok and self.ispcCode == other.ispcCode

        ok = ok and self.codeBlocks == other.codeBlocks
        ok = ok and self.userData == other.userData

        return ok

    def copy(self):
        p = Patch()
        p.name = self.name
        p.type = self.type

        p.attributes = [a.copy() for a in self.attributes]
        p.children = [c.copy() for c in self.children]

        p.parallelRun = self.parallelRun
        p.parallelDeform = self.parallelDeform
        p.parallelDeformInit = self.parallelDeformInit

        p.version = self.version
        p.state = self.state
        p.codeBlocks = self.codeBlocks
        p.hasNamespace = self.hasNamespace
        p.customNamespace = self.customNamespace
        p.debugOpenCL = self.debugOpenCL
        p.block = self.block

        p.description = self.description

        p.definesCode = self.definesCode
        p.commandsCode = self.commandsCode
        p.drawCode = self.drawCode
        p.deformInitCode = self.deformInitCode
        p.deformCode = self.deformCode
        p.deformParallelCode = self.deformParallelCode
        p.runCode = self.runCode
        p.clCode = self.clCode
        p.ispcCode = self.ispcCode

        p.userData = dict(self.userData)

        return p

    @staticmethod
    def getPatchTags(patchFile, tags):
        with open(patchFile, "r") as f:
            fileData = f.read().decode("utf-8")
            data = {}
            for tg in tags:
                exp = re.search("<{tag}>([\\S\\s]*?)</{tag}>".format(tag=tg), fileData)
                if exp:
                    item = exp.groups("")[0].replace("<![CDATA[","").replace("]]>","").strip()
                    data[tg] = item

        return data

    def findAttribute(self, name):
        return [a for a in self.attributes if a.name == name]

    def findChildren(self, name):
        return [c for c in self.children if c.name == name]

    def removeAttribute(self, name):
        self.attributes = [a for a in self.attributes if a.name != name]

    def removeChild(self, name):
        self.children = [c for c in self.children if c.name != name]

    def removeChildAtIndex(self, index):
        self.children = [c for i, c in enumerate(self.children) if i != index]

    def toXml(self):
        template = "<patch>\n"
        template += "<name><![CDATA[%s]]></name>\n"%self.name.strip()
        template += "<type><![CDATA[%s]]></type>\n"%self.type.strip()
        template += "<version>%d</version>"%self.version
        template += "<state>%d</state>\n"%self.state

        template += "<parallelRun>%d</parallelRun>\n"%self.parallelRun
        template += "<parallelDeform>%d</parallelDeform>\n"%self.parallelDeform
        template += "<parallelDeformInit>%d</parallelDeformInit>\n"%self.parallelDeformInit

        template += "<hasNamespace>%d</hasNamespace>\n"%self.hasNamespace
        template += "<customNamespace>%s</customNamespace>\n"%self.customNamespace
        template += "<codeBlocks>%d</codeBlocks>\n"%self.codeBlocks
        template += "<debugOpenCL>%d</debugOpenCL>\n"%self.debugOpenCL
        template += "<block>%d</block>\n"%self.block

        template += "<description>\n<![CDATA[%s]]>\n</description>\n"%self.description.strip()
        template += "<definesCode>\n<![CDATA[%s]]>\n</definesCode>\n"%self.definesCode.strip()
        template += "<commandsCode>\n<![CDATA[%s]]>\n</commandsCode>\n"%self.commandsCode.strip()
        template += "<drawCode>\n<![CDATA[%s]]>\n</drawCode>\n"%self.drawCode.strip()
        template += "<deformInitCode>\n<![CDATA[%s]]>\n</deformInitCode>\n"%self.deformInitCode.strip()
        template += "<deformCode>\n<![CDATA[%s]]>\n</deformCode>\n"%self.deformCode.strip()
        template += "<deformParallelCode>\n<![CDATA[%s]]>\n</deformParallelCode>\n"%self.deformParallelCode.strip()
        template += "<runCode>\n<![CDATA[%s]]>\n</runCode>\n"%self.runCode.strip()
        template += "<clCode>\n<![CDATA[%s]]>\n</clCode>\n"%self.clCode.strip()
        template += "<ispcCode>\n<![CDATA[%s]]>\n</ispcCode>\n"%self.ispcCode.strip()
        template += "<userData>\n<![CDATA[%s]]>\n</userData>\n"%json.dumps(self.userData)
        template += "<attributes>\n%s\n</attributes>\n"%("\n".join([a.toXml() for a in self.attributes]))
        template += "<children>\n%s\n</children>\n"%("\n".join([c.toXml() for c in self.children]))
        template += "</patch>"

        return template

    @staticmethod
    def fromXml(templateElement):
        p = Patch()
        p.name = templateElement.findtext("name").strip()
        p.type = templateElement.findtext("type").strip()

        p.version = int(templateElement.findtext("version") or 0) # 0 - no version
        p.state = int(templateElement.findtext("state"))

        p.parallelRun = int(templateElement.findtext("parallelRun") or False)
        p.parallelDeform = int(templateElement.findtext("parallelDeform") or False)
        p.parallelDeformInit = int(templateElement.findtext("parallelDeformInit") or False)

        p.hasNamespace = int(templateElement.findtext("hasNamespace"))
        p.customNamespace = templateElement.findtext("customNamespace") or ""
        p.codeBlocks = int(templateElement.findtext("codeBlocks") or True)
        p.debugOpenCL = int(templateElement.findtext("debugOpenCL") or 0)
        p.block = int(templateElement.findtext("block") or 0)

        p.description = templateElement.findtext("description").strip()

        p.definesCode = (templateElement.findtext("definesCode") or "").strip()
        p.commandsCode = (templateElement.findtext("commandsCode") or "").strip()
        p.deformInitCode = (templateElement.findtext("deformInitCode") or "").strip()
        p.deformCode = (templateElement.findtext("deformCode") or "").strip()
        p.deformParallelCode = (templateElement.findtext("deformParallelCode") or "").strip()
        p.runCode = (templateElement.findtext("runCode") or "").strip()
        p.drawCode = (templateElement.findtext("drawCode") or "").strip()
        p.clCode = (templateElement.findtext("clCode") or "").strip()
        p.ispcCode = (templateElement.findtext("ispcCode") or "").strip()

        p.userData = json.loads((templateElement.findtext("userData") or "{}").strip())

        p.attributes = [Attribute.fromXml(a) for a in templateElement.find("attributes").getchildren()]
        p.children = [Patch.fromXml(c) for c in templateElement.find("children").getchildren()]

        return p

    @staticmethod
    def getVersionByFileName(fileName):
        basename = os.path.splitext(os.path.basename(fileName))[0]
        r = re.search(Patch.VersionPattern, basename)
        return int(r.groups()[0] if r else 0)

    @staticmethod
    def getLatestVersionByFileName(fileName):
        ver = Patch.getVersionByFileName(fileName)
        if not ver:
            return 0

        fn = fileName
        while os.path.exists(fn):
            ver += 1
            fn = re.sub(Patch.VersionPattern, "v%.3d"%ver, fn)

        return ver-1

    @staticmethod
    def getLatestDllByFileName(fileName, mayaVersion):
        ver = Patch.getVersionByFileName(fileName)
        DllPath = EvaluatorServerPath + "/nodes"
        baseName = os.path.splitext(os.path.basename(fileName))[0]
        baseName += "_%d.dll"%mayaVersion

        while ver > 0:
            fname = re.sub(Patch.VersionPattern, "v%.3d"%ver, baseName)
            if os.path.exists(DllPath + "/" + fname):
                return DllPath + "/" + fname

            ver -= 1

        noVerFileName = re.sub(Patch.VersionPattern+"_", "", baseName)
        if os.path.exists(DllPath + "/" + noVerFileName):
            return DllPath + "/" + noVerFileName

    def findLatestFile(self):
        if not self.version:
            return

        folder = ("/" + self.type) if self.type else ""
        files = sorted(glob.glob(EvaluatorLocalPath+"/patch/patches" + folder + "/*.xml"), key=Patch.getVersionByFileName)

        if files:
            return files[-1]

    def isActualVersion(self):
        latestFile = self.findLatestFile()
        if not latestFile:
            return True

        return Patch.getVersionByFileName(latestFile) == self.version

    def getFileName(self, ext=True):
        if self.type and self.version:
            version = "v%.3d"%self.version
            fileName = "{path}/patch/patches/{type}/{type}_{version}.xml".format(path=EvaluatorLocalPath, type=self.type, version=version)
        else:
            fileName = "{dir}/patch/patches/{name}.xml".format(dir=EvaluatorLocalPath, name=self.name)

        realpath = os.path.realpath(fileName)
        return realpath if ext else os.path.splitext(realpath)[0]

    def isCodeActual(self):
        fileName = self.getFileName()
        if not fileName or not os.path.exists(fileName):
            return {}

        codeTypes = ["definesCode",
                     "commandsCode",
                     "runCode",
                     "deformInitCode",
                     "deformCode",
                     "deformParallelCode",
                     "clCode",
                     "ispcCode",
                     "drawCode"]

        codes = Patch.getPatchTags(fileName, codeTypes)

        result = {}

        for ct in codeTypes:
            result[ct] = self.__getattribute__(ct) == codes.get(ct,"")

        return result

    def findPatchByName(self, name, path=""):
        if self.name == name:
            return [(self, path + self.name)]

        result = []
        for ch in self.children:
            found = ch.findPatchByName(name, path + self.name + "/")
            if found:
                result += found

        return result

    def getPatchByPath(self, path, currentPath=""):
        if path == currentPath + self.name:
            return self

        for ch in self.children:
            found = ch.getPatchByPath(path, currentPath + self.name + "/")
            if found:
                return found

    def findCodeLinks(self, patch, code):
        root = Patch("", "")
        root.children = [patch]

        results = {}
        for link in re.findall("\\$\\w+/[/\\w]+", code):            
            p, attrs = root.resolveLink(link)
            if p and attrs:
                ns = root.getNamespaceFor(p)[2:] # skip first __
                results[link] = ns+"__"+attrs[0].name # first found attribute
        return results
    
    def resolveLink(self, link, attribute=True):
        if link.startswith("/"): # parent relative link
            if attribute:
                path = "/".join(link[1:].split("/")[:-1]) # /child/attr => child
                a = link[1:].split("/")[-1].split(".")[0].split("[")[0] # attr, skip struct members and [] if they are
            else:
                path = link[1:] # skip first /
                a = ""

            p = self.getPatchByPath(self.name + "/"+ path) if path else self
            return (p, p.findAttribute(a)) if p and attribute else p

        elif link.startswith("$"): # absolute link
            n = link.split("/")[0][1:] # $biped/leg/input => biped
            a = link.split("/")[-1].split(".")[0] # input

            found = self.findPatchByName(n)
            if found:
                p, path = found[0]
                fullPath = link.replace("$"+n, path)
                if attribute:
                    patchPath = "/".join(fullPath.split("/")[:-1]) # skip last attribute
                else:
                    patchPath = "/".join(fullPath.split("/"))

                p = self.getPatchByPath(patchPath)
                return (p, p.findAttribute(a)) if p and attribute else p

    def renameAbsoluteLinks(self, old, new):
        for ch in self.children:
            for a in ch.attributes:
                if a.defaultValue.startswith("$"):
                    a.defaultValue = re.sub("\\b"+old+"\\b", new, a.defaultValue)

            ch.renameAbsoluteLinks(old, new)

    def renameRelativeLinks(self, old, new):
        for ch in self.children:
            for a in ch.attributes:
                if a.defaultValue.startswith("/"):
                    a.defaultValue = re.sub("\\b"+old+"\\b", new, a.defaultValue)

    def updateFromPatch(self, src, attributes=False, children=True, keepDefaultValue=True):
        p = src.copy()

        self.type = p.type
        self.version = p.version

        self.parallelRun = p.parallelRun
        self.parallelDeform = p.parallelDeform
        self.parallelDeformInit = p.parallelDeformInit

        self.codeBlocks = p.codeBlocks
        # self.hasNamespace = p.hasNamespace
        # self.customNamespace = p.customNamespace
        # self.debugOpenCL = p.debugOpenCL
        self.block = p.block

        self.description = p.description

        newAttributes = []
        if attributes: # replace
            for attr in p.attributes:
                a = self.findAttribute(attr.name)

                if a and keepDefaultValue:
                    attr.defaultValue = a[0].defaultValue

                newAttributes.append(attr)

        else: # just add
            for attr in p.attributes:
                a = self.findAttribute(attr.name)
                newAttributes.append(a[0] if a else attr)

        self.attributes = newAttributes

        if children:
            self.children = p.children

        self.definesCode = p.definesCode
        self.commandsCode = p.commandsCode
        self.deformInitCode = p.deformInitCode
        self.deformCode = p.deformCode
        self.deformParallelCode = p.deformParallelCode
        self.runCode = p.runCode
        self.clCode = p.clCode
        self.ispcCode = p.ispcCode
        self.drawCode = p.drawCode

        self.userData = p.userData

    def backup(self):
        name = os.path.basename(self.getFileName())
        backupFile = "{dir}/patch/patches/__backup__/{name}".format(dir=EvaluatorLocalPath, name=name)
        data = self.toXml()

        with open(backupFile, "w") as f:
            f.write(data.encode("utf-8"))

        return backupFile

    def update(self, revert=False, attributes=False, children=True):
        fileName = self.getFileName() if revert else self.findLatestFile()

        if not fileName or not os.path.exists(fileName):
            return

        p = Patch.loadFromFile(fileName)
        self.updateFromPatch(p, attributes=attributes, children=children)
        return True

    @staticmethod
    def listPatches(path, mask=""):
        files = []
        for f in sorted(glob.iglob(path+"/*"), key=Patch.getVersionByFileName):
            if os.path.isdir(f):
                files += Patch.listPatches(f, mask)
            else:
                if f.endswith(".xml"):
                    files.insert(0, f)

        return files

    @staticmethod
    def loadFromFile(fileName):
        root = ET.parse(fileName).getroot()
        p = Patch.fromXml(root)
        p.loadedFrom = os.path.realpath(fileName)
        return p

    def publish(self, saveOnly=False):
        fileName = ""

        if not self.type:
            self.type = self.name

        self.version = self.getLatestVersionByFileName(self.getFileName()) or 1
        
        if not saveOnly: # publish new version
            self.version += 1

            fileName = self.getFileName()
            folder = os.path.dirname(fileName)

            path = os.path.realpath(folder)
            if not os.path.exists(path):
                os.makedirs(path)
        else: # just save current version
            fileName = self.getFileName()
            folder = os.path.dirname(fileName)
            if not os.path.exists(folder):
                os.makedirs(folder)

        xml = self.toXml()
        with open(fileName, "w") as f:
            f.write(xml.encode("utf-8"))

        self.loadedFrom = os.path.realpath(fileName)

    def findTextRecursively(self, pattern, flags=0, namespace=[]):
        code = []

        name = " - ".join(namespace + [self.name])
        for ct in ["definesCode",
                   "commandsCode",
                   "runCode",
                   "deformInitCode",
                   "deformCode",
                   "deformParallelCode",
                   "drawCode"]:
            matches = ""
            for n, line in enumerate(self.__getattribute__(ct).split("\n")):
                if re.search(pattern, line, flags):
                    matches += "\t%-5d\t\t%s\n"%(n+1, line.strip())

            if matches:
                code.append("".join(["%s [%s]\n"%(name, codeTypeToLabel(ct)),
                                    matches,
                                    "\n"]))

        for c in self.children:
            code.append(c.findTextRecursively(pattern, flags, namespace=namespace+[self.name]))

        return "".join(code)

    def parseSetCommand(self):
        if self.state == Patch.StateMutedAll:
            return ""

        template = []
        for attr in self.getMayaAttributes():
            cmd = attr["attribute"].parseSetCommand(attr["mayaName"])
            if cmd:
                template.append(cmd)

        return "".join(["import pymel.core as pm\npy = pm.PyNode\n",
                        "\n".join(template)])

    def isAttributeChildOfArrayCompound(self, name):
        for a in self.attributes:
            if re.search("\\b%s\\b"%name, a.items) and a.array and a.type == "EvCompound":
                return True

        return False

    def getNamespaceFor(self, patch, namespace=""):
        if self == patch:
            return namespace + self.name

        for ch in self.children:
            ns = ch.getNamespaceFor(patch, namespace+self.name+"__")
            if ns:
                return ns

    def getMayaNamespace(self, namespace=""):
        return namespace+(self.customNamespace or self.name)+"__" if self.hasNamespace else namespace

    def getMayaAttributes(self, namespace=""):
        if self.state == Patch.StateMutedAll:
            return ""

        mayaNamespace = self.getMayaNamespace(namespace)

        attributes = []
        for a in self.attributes:
            if not a.muted and a.isMaya():
                attrData = {"attribute": a, "mayaName": mayaNamespace+a.name}
                attributes.append(attrData)

        for ch in self.children:
            attributes += ch.getMayaAttributes(mayaNamespace)

        return attributes

    def toGetAttributesCpp(self, namespace="", path=""):
        if self.state == Patch.StateMutedAll:
            return ""

        mayaNamespace = self.getMayaNamespace(namespace)
        pathName = path.replace("/","__")

        templates = ["// {name} patch".format(name=path+self.name)]
        for block in [self.attributes, self.children]:
            for element in block:
                cpp = element.toGetAttributesCpp(mayaNamespace, path+self.name+"/")
                if cpp:
                    templates.append(cpp)

        code = "\n\n".join(templates)

        if code and not (self.embedCode and Patch.EmbedCode):
            codeFile = "{dir}/patch/temp/{name}__getAttributes.cpp".format(dir=EvaluatorLocalPath, name=pathName+self.name)
            with open(codeFile, "w") as f:
                f.write(code.encode("utf-8"))

            code = "\n#include \"%s\"\n"%os.path.basename(codeFile)

        return code

    def toPrivateCpp(self, namespace="", path="", patches=[]):
        if self.state in [Patch.StateMuted, Patch.StateMutedAll]:
            return ""

        mayaNamespace = self.getMayaNamespace(namespace)
        pathName = path.replace("/","__") + self.name+"__"

        template = []

        if self.clCode:
            template.append(
                '''
                MAutoCLKernel @CL_kernel;
                static string @CL_program;
                '''.replace("@", pathName))

            if self.debugOpenCL:
                template.append(
                '''
                vector<float> @CL_debugValues; // for OpenCL debug. You can visualize these values
                vector<V3d> @CL_debugPoints;
                MAutoCLMem @CL_debugBuffer;
                '''.replace("@", pathName))

        for a in self.attributes:
            if a.muted:
                continue

            # c++ cached variable without references
            if not a.isMaya() and a.cached and not a.isReference():
                template.append(a.toRunCpp(pathName, path+self.name+"/", patches + [self]))

            if a.needCLBuffer() or a.cl == Attribute.CL_GlobalAlloc:
                template.append("MAutoCLMem %s_buffer;"%(pathName+a.name))

        for ch in self.children:
            template.append(ch.toPrivateCpp(mayaNamespace, path+self.name+"/", patches+[self]))

        return "".join(["\n",
                        "\n".join(template),
                        "\n"])

    def parseDefines(self):
        includes, defines, sharedFunctions = [], [], []
        lastDiez = False
        lastContinuedLine = False

        for line in self.definesCode.split("\n"):
            if line.startswith("#"):
                lastDiez = True
                includes.append(line)
            elif lastDiez:
                if line.strip().endswith("\\"):
                    includes.append(line)
                    lastContinuedLine = True
                else:
                    if lastContinuedLine:
                        includes.append(line)
                    else:
                        defines.append(line)

                    lastContinuedLine = False
                    lastDiez = False
            else:
                defines.append(line)

        return ("\n".join(includes), "\n".join(defines))

    def toHelpCommandCpp(self, path=""):
        if self.state in [Patch.StateMuted, Patch.StateMutedAll]:
            return ""

        template = re.findall("BEGIN_COMMAND\\((\\w+)\\)", self.commandsCode)

        for ch in self.children:
            res = ch.toHelpCommandCpp(path+self.name+"/")
            if res:
                template.append(res)

        return ", ".join(template)

    def toCommandsCpp(self, namespace="", path="", patches=[]):
        if self.state in [Patch.StateMuted, Patch.StateMutedAll]:
            return ""

        mayaNamespace = self.getMayaNamespace(namespace)
        pathName = path.replace("/","__")

        template = ["\n\n// {name} commands\n".format(name=path+self.name)]

        commands = self.commandsCode.replace("$@", mayaNamespace).replace("@", pathName+self.name+"__")

        root = patches[0] if patches else self
        
        links = self.findCodeLinks(root, commands)
        for f, r in links.items():
            commands = commands.replace(f, "NODE."+r)
        
        if commands and not (self.embedCode and Patch.EmbedCode):
            commandsFile = "{dir}/patch/temp/{name}__commands.cpp".format(dir=EvaluatorLocalPath, name=pathName+self.name)
            with open(commandsFile, "w") as f:
                f.write(commands.encode("utf-8"))

            commands = "\n#include \"%s\"\n"%os.path.basename(commandsFile)

        commands = "\n".join(["namespace %s_namespace{"%(pathName+self.name),
                              commands,
                              "};"])

        template.append(commands)

        for ch in self.children:
            code = ch.toCommandsCpp(mayaNamespace, path+self.name+"/", patches + [self])
            if code:
                template.append(code)

        return "\n".join(template)

    def toIncludesCpp(self, namespace="", path=""):
        if self.state in [Patch.StateMuted, Patch.StateMutedAll]:
            return ""

        mayaNamespace = self.getMayaNamespace(namespace)
        pathName = path.replace("/","__")

        template = []

        includes, _ = self.parseDefines()

        template.append(includes.replace("@", pathName+self.name+"__"))
        for ch in self.children:
            code = ch.toIncludesCpp(mayaNamespace, path+self.name+"/")
            if code:
                template.append(code)

        return "\n".join(set(template))

    def toDefinesCpp(self, namespace="", path=""):
        if self.state in [Patch.StateMuted, Patch.StateMutedAll]:
            return ""

        mayaNamespace = self.getMayaNamespace(namespace)
        pathName = path.replace("/","__")

        template = ["\n\n// {name} patch defines\n".format(name=path+self.name)]

        _, defines = self.parseDefines()

        if defines and not (self.embedCode and Patch.EmbedCode):
            definesFile = "{dir}/patch/temp/{name}__defines.cpp".format(dir=EvaluatorLocalPath, name=pathName+self.name)
            with open(definesFile, "w") as f:
                f.write(defines.encode("utf-8"))

            defines = "\n#include \"%s\"\n"%os.path.basename(definesFile)

        defines = "".join(["namespace %s_namespace {"%(pathName+self.name),
                           defines,
                           "\n}"])

        template.append(defines)

        for ch in self.children:
            code = ch.toDefinesCpp(mayaNamespace, path+self.name+"/")
            if code:
                template.append(code)

        return "\n".join(template)

    def toAttributesCpp(self, namespace="", path="", patches=[], fileName="attributes", outputs=True):
        if self.state in [Patch.StateMuted, Patch.StateMutedAll]:
            return ""

        mayaNamespace = self.getMayaNamespace(namespace)
        pathName = path.replace("/","__")

        attributesBlocks = []
        for attr in self.attributes: # local attributes
            if attr.child:
                continue

            if not outputs and attr.output:
                continue

            if attr.cached and attr.isCpp() and not attr.isReference(): # skip c++ cached attributes
                continue

            cpp = attr.toRunCpp(mayaNamespace, path+self.name+"/", patches+[self])
            if cpp:
                attributesBlocks.append(cpp)

        for ch in self.children: # all the children attributes
            cpp = ch.toAttributesCpp(mayaNamespace, path+self.name+"/", patches+[self], fileName, outputs)
            if cpp:
                attributesBlocks.append(cpp)

        code = "\n".join(attributesBlocks)

        if not (self.embedCode and Patch.EmbedCode):
            file = "{dir}/patch/temp/{name}__{fileName}.cpp".format(dir=EvaluatorLocalPath, name=pathName+self.name, fileName=fileName)
            with open(file, "w") as f:
                f.write(code.encode("utf-8"))

            code = "#include \"%s\"\n"%os.path.basename(file)

        return code

    def toIsActualCacheCpp(self, namespace="", path=""):
        if self.state in [Patch.StateMuted, Patch.StateMutedAll]:
            return ""

        mayaNamespace = self.getMayaNamespace(namespace)
        pathName = path.replace("/","__")

        attributesBlocks = []
        for attr in self.attributes: # local attributes
            if attr.isMaya() and attr.cached and attr.type != "EvCompound":
                cpp = "if (name == \"{name}\") return true;".format(name=mayaNamespace+attr.name)
                attributesBlocks.append(cpp)

        for ch in self.children: # all the children attributes
            cpp = ch.toIsActualCacheCpp(mayaNamespace, path+self.name+"/")
            if cpp:
                attributesBlocks.append(cpp)

        code = "\n".join(attributesBlocks)

        if not path and not (self.embedCode and Patch.EmbedCode):
            codeFile = "{dir}/patch/temp/{name}__isActualCache.cpp".format(dir=EvaluatorLocalPath, name=pathName+self.name)
            with open(codeFile, "w") as f:
                f.write(code.encode("utf-8"))

            code = "\n#include \"%s\"\n"%os.path.basename(codeFile)

        return code

    def checkCL(self):
        program = self.getCLProgram()

        outputs = []

        if program:
            programFile = EvaluatorLocalPath + "/patch/temp/"+self.name+".cl"
            with open(programFile, "w") as f:
                f.write(program)

            err, output = clangCLCheck(programFile)
            if err:
                return err, output

            outputs.append(output)

        for ch in self.children:
            err, output = ch.checkCL()

            if output:
                outputs.append(output)

            if err:
                return err, output

        return 0, "\n".join(outputs)

    def getKernelArguments(self):
        argList = ["global const float* INPUT_POSITIONS /*float3 POINT*/",
                   "global float* OUTPUT_POSITIONS",
                   "const int NUM_ELEMENTS",
                   "global const float* WEIGHTS",
                   "const float ENVELOPE"]

        if self.debugOpenCL:
            argList.append("global float* _DEBUG")

        for attr in self.attributes:
            arg = attr.toCLKernelArg()
            if arg:
                argList.append(arg)

        return argList

    def getCLProgram(self):
        if not self.clCode:
            return ""

        includePath = EvaluatorServerPath + "/include/clh"

        code = self.clCode.replace("@", "attr_")
        code = re.sub("\\b(\\d+\\.\\d+)\\b(?!f)","\\1f", code) # convert float numbers to 1.0f format

        kernelDeclCode = "".join(["__kernel void run(",
                                  ",".join(self.getKernelArguments()).replace("@", "attr_"),
                                  "){",
                                  "".join(["const int I = get_global_id(0);",
                                           "float3 POINT = vload3(I, INPUT_POSITIONS);"])])

        lines = []
        for line in code.split("\n"):
            line = line.rstrip()

            r = re.match("#include\\s*(\\w+)", line)
            if r: # include parse
                fname = r.group(1)
                fpath = includePath + "/" + fname + ".cl"

                if not os.path.exists(fpath):
                    raise Exception("getCLProgram: cannot find '%s'"%fpath)

                lines.append(cpp2line(fpath))
            else:
                if line in ["__kernel", "kernel"]:
                    lines.append(kernelDeclCode)
                else:
                    lines.append(line)

        lines.append("vstore3(POINT, I, OUTPUT_POSITIONS);}")

        return "\n".join(lines)

    def toGpuStaticPrograms(self, path=""):
        if self.state in [Patch.StateMuted, Patch.StateMutedAll]:
            return ""

        templates = []
        if self.clCode:
            code = self.getCLProgram()

            pathName = path.replace("/", "__")
            templates.append("".join(["string MyNode::%sCL_program = R\"("%(pathName + self.name + "__"),
                                      code,
                                      ")\";"]))

        for ch in self.children: # all the children attributes
            cpp = ch.toGpuStaticPrograms(path+self.name+"/")
            if cpp:
                templates.append(cpp)

        return "\n\n".join(templates)

    def toGpuTerminateCpp(self, path=""):
        if self.state in [Patch.StateMuted, Patch.StateMutedAll]:
            return ""

        pathName = path.replace("/", "__") + self.name + "__"

        templates = []
        if self.clCode:
            templates.append("".join(["MOpenCLInfo::releaseOpenCLKernel(@CL_kernel);",
                                      "@CL_kernel.reset();",
                                      "@CL_debugBuffer.reset();" if self.debugOpenCL else ""])
                             .replace("@", pathName))

            for attr in self.attributes: # local attributes
                if attr.needCLBuffer() and not attr.muted:
                    templates.append("%s_buffer.reset();"%(pathName + attr.name))

        for ch in self.children: # all the children attributes
            cpp = ch.toGpuTerminateCpp(path+self.name+"/")
            if cpp:
                templates.append(cpp)

        code = "\n".join(templates)

        if not path and not (self.embedCode and Patch.EmbedCode):
            codeFile = "{dir}/patch/temp/{name}__gpuTerminate.cpp".format(dir=EvaluatorLocalPath, name=pathName[:-2])
            with open(codeFile, "w") as f:
                f.write(code.encode("utf-8"))

            code = "\n#include \"%s\"\n"%os.path.basename(codeFile)

        return code

    def toRunKernelCpp(self, path=""):
        if self.state in [Patch.StateMuted, Patch.StateMutedAll]:
            return ""

        pathName = path.replace("/", "__")

        templates = []
        if self.clCode:
            programId = str(random.randint(1, 0xFFFF))

            templates.append(
                '''
                { // $PATH OpenCL kernel preparation
                const auto CL_initKernelTime = getMeasureTime();

                if (@CL_kernel.isNull())
                {
                    @CL_kernel = MOpenCLInfo::getOpenCLKernelFromString(toMString(@CL_program), "@CL_program_$ID", "run");
                    if (@CL_kernel.isNull()) {error("Cannot compile OpenCL program: $PATH. See Output Window"); return kError;}
                }

                if (!CL_firstKernel)
                {
                    INPUT_EVENT.swap(OUTPUT_EVENT);

                    clEnqueueCopyBuffer(
                        MOpenCLInfo::getMayaDefaultOpenCLCommandQueue(),
                        OUTPUT_BUFFER.get(),
                        INTERMEDIATE_GEOM_BUFFER.get(),
                        0, 0,
                        3 * NUM_ELEMENTS * sizeof(float),
                        1,
                        INPUT_EVENT.getReadOnlyRef(),
                        OUTPUT_EVENT.getReferenceForAssignment());

                    INPUT_EVENT.swap(OUTPUT_EVENT);
                    INPUT_BUFFER = INTERMEDIATE_GEOM_BUFFER;
                }
                '''.replace("$ID", programId).replace("$PATH", path + self.name).replace("@", pathName + self.name + "__"))

            templates.append(
                '''
                int parameterId = 0;
                CL_CHECK_ERROR( setKernelArg(@CL_kernel, parameterId++, INPUT_BUFFER) );
                CL_CHECK_ERROR( setKernelArg(@CL_kernel, parameterId++, OUTPUT_BUFFER) );
                CL_CHECK_ERROR( setKernelArg(@CL_kernel, parameterId++, NUM_ELEMENTS) );
                CL_CHECK_ERROR( setKernelArg(@CL_kernel, parameterId++, WEIGHTS_BUFFER) );
                CL_CHECK_ERROR( setKernelArg(@CL_kernel, parameterId++, ENVELOPE) );
                '''.replace("@", pathName + self.name + "__"))

            if self.debugOpenCL:
                templates.append(
                    '''
                    // debug only, add const float *_DEBUG variable as a kernel argument.
                    if (@CL_debugBuffer.isNull())
                    {
                        @CL_debugPoints.resize(NUM_ELEMENTS);
                        @CL_debugValues.resize(NUM_ELEMENTS);

                        CL_CHECK_ERROR( enqueueBuffer(@CL_debugBuffer, NUM_ELEMENTS * sizeof(float), NULL, CL_MEM_READ_WRITE) );
                    }
                    CL_CHECK_ERROR( setKernelArg(@CL_kernel, parameterId++, @CL_debugBuffer) );
                    '''.replace("@", pathName + self.name + "__"))

            for attr in self.attributes: # local attributes
                if attr.cl == Attribute.CL_none or attr.muted:
                    continue

                if attr.cl == Attribute.CL_LocalAlloc: # alloc GPU local memory
                    templates.append(
                        '''
                        if (@attr.size() <= 0) {error("$PATH: $NAME.size() (local memory size) must be > 0"); return kError;}
                        CL_CHECK_ERROR( clSetKernelArg(@CL_kernel.get(), parameterId++, vectorSizeof(@attr), NULL) );
                        '''
                        .replace("$PATH", path + self.name)\
                        .replace("$NAME", attr.name)\
                        .replace("@attr", pathName + self.name + "__"+ attr.name)\
                        .replace("@", pathName + self.name + "__"))

                elif attr.cl == Attribute.CL_GlobalAlloc: # alloc GPU global memory
                    templates.append(
                        '''
                        if (@attr_buffer.isNull())
                        {
                            if (@attr.size() == 0) {error("$PATH: $NAME.size() (global memory size) must be > 0"); return kError;}
                            CL_CHECK_ERROR( enqueueBuffer(@attr_buffer, vectorSizeof(@attr), &@attr[0], CL_MEM_COPY_HOST_PTR | CL_MEM_READ_WRITE) );
                        }
                        CL_CHECK_ERROR( setKernelArg(@CL_kernel, parameterId++, @attr_buffer) )
                        '''
                        .replace("$PATH", path + self.name)\
                        .replace("$NAME", attr.name)\
                        .replace("@attr", pathName + self.name + "__"+ attr.name)\
                        .replace("@", pathName + self.name + "__"))

                elif attr.cl in [Attribute.CL_Global, Attribute.CL_Constant] and attr.needCLBuffer():
                    if attr.cached:
                        templates.append("if (GPU_STARTUP || !CACHE_ATTRIBUTES)")
                    templates.append("if (@attr.size() > 0) CL_CHECK_ERROR( writeBuffer(@attr_buffer, @attr) );".replace("@attr", pathName + self.name + "__"+ attr.name))

                    templates.append(
                        '''
                        CL_CHECK_ERROR( setKernelArg(@CL_kernel, parameterId++, @attr_buffer) );
                        '''
                        .replace("$PATH", path + self.name)\
                        .replace("$NAME", attr.name)\
                        .replace("@attr", pathName + self.name + "__"+ attr.name)\
                        .replace("@", pathName + self.name + "__")\
                        .replace("$ATTR", "@"+attr.name))

                else:
                    templates.append(
                        '''
                        CL_CHECK_ERROR( setKernelArg(@CL_kernel, parameterId++, @attr) );
                        '''.replace("@attr", pathName + self.name + "__"+ attr.name)\
                           .replace("@", pathName + self.name + "__"))

            templates.append(
                '''
                if (DEBUG_MODE == DebugMode::DebugGPUPerfomance)
                    measureTime("$PATH: init kernel time", CL_initKernelTime);

                size_t globalWorkSize, localWorkSize;
                CL_CHECK_ERROR( getGroupWorkSize(MOpenCLInfo::getOpenCLDeviceId(), @CL_kernel.get(), NUM_ELEMENTS, globalWorkSize, localWorkSize) );
                CL_CHECK_ERROR( runKernel(@CL_kernel, globalWorkSize, localWorkSize, INPUT_EVENT, OUTPUT_EVENT) );
                CL_firstKernel = false;
                '''
                .replace("$PATH", path + self.name)\
                .replace("@", pathName + self.name + "__"))

            if self.debugOpenCL:
                templates.append(
                '''
                // debug only, read _DEBUG and OUTPUT_POSITIONS and make them available as vectors.
                // visualize them inside draw function.
                {
                float *valuesData = new float[NUM_ELEMENTS];
                clEnqueueReadBuffer(
                    MOpenCLInfo::getMayaDefaultOpenCLCommandQueue(),
                    @CL_debugBuffer.get(),
                    true, 0, NUM_ELEMENTS * sizeof(float),
                    valuesData, 1, OUTPUT_EVENT.getReadOnlyRef(), NULL);

                float *pointsData = new float[3 * NUM_ELEMENTS];
                clEnqueueReadBuffer(
                    MOpenCLInfo::getMayaDefaultOpenCLCommandQueue(),
                    OUTPUT_BUFFER.get(),
                    true, 0, 3 * NUM_ELEMENTS * sizeof(float),
                    pointsData, 1, OUTPUT_EVENT.getReadOnlyRef(), NULL);

                BEGIN_PARALLEL_FOR(i, NUM_ELEMENTS)
                @CL_debugPoints[i].x = pointsData[i*3+0];
                @CL_debugPoints[i].y = pointsData[i*3+1];
                @CL_debugPoints[i].z = pointsData[i*3+2];
                @CL_debugValues[i] = valuesData[i];
                END_PARALLEL_FOR;

                delete[] pointsData;
                delete[] valuesData;
                }
                '''.replace("@", pathName + self.name + "__")\
                   .replace("$PATH", path + self.name))

            templates.append("} // end local block")

        if templates:
            comment = "// %s run kernel\n"%(path + self.name)
            templates.insert(0, comment)

        for ch in self.children: # all the children attributes
            cpp = ch.toRunKernelCpp(path+self.name+"/")
            if cpp:
                templates.append(cpp)

        if not templates:
            return ""

        code = "\n".join(templates)

        if self.embedCode and Patch.EmbedCode: # top level only
            codeFile = "{dir}/patch/temp/{name}__runKernel.cpp".format(dir=EvaluatorLocalPath, name=pathName + self.name)
            with open(codeFile, "w") as f:
                f.write(code.encode("utf-8"))

            code = "\n#include \"%s\"\n"%os.path.basename(codeFile)

        return code

    def toIspc(self, path=""):
        if self.state in [Patch.StateMuted, Patch.StateMutedAll]:
            return

        pathName = path.replace("/", "__")

        templates = []

        if self.ispcCode:
            code = self.ispcCode.replace("@", pathName + self.name + "__")
            templates.append(code)

        for ch in self.children: # all the children attributes
            cpp = ch.toIspc(path+self.name+"/")
            if cpp:
                templates.append(cpp)

        if not path:
            templates.append("export void helloIspc() { print(\"Hello from ispc\"); }")

        code = "\n".join(templates)

        if not (self.embedCode and Patch.EmbedCode):
            codeFile = "{dir}/patch/temp/{name}__ispc.ispc".format(dir=EvaluatorLocalPath, name=pathName + self.name)
            with open(codeFile, "w") as f:
                f.write(code.encode("utf-8"))

            code = "\n#include \"%s\"\n"%os.path.basename(codeFile)

        return code

    def generateIspcFile(self):
        code = self.toIspc()
        codeFile = "{dir}/patch/temp/patch_ispc.ispc".format(dir=EvaluatorLocalPath)
        with open(codeFile, "w") as f:
            f.write(code.encode("utf-8"))

        return codeFile

    def toRunCpp(self, codeType, namespace="", path="", children=True, inBlock=False):
        if self.state in [Patch.StateMuted, Patch.StateMutedAll]:
            return ""

        if codeType in ["run", "deformInit", "deform", "deformParallel"] and inBlock:
            return ""

        mayaNamespace = self.getMayaNamespace(namespace)
        pathName = path.replace("/", "__")

        code = ""
        parallel = False
        if codeType == "draw":
            code = "\n".join([self.drawCode,
                              "\n".join(["DRAWER.setPointSize(5);",
                                         "DRAWER.setFontSize(10);",
                                         "drawPoints(DRAWER, @CL_debugPoints, &@CL_debugValues);"])
                              if self.clCode and self.debugOpenCL else ""])

            parallel = False

        elif codeType == "run":
            code = self.runCode
            parallel = self.parallelRun

        elif codeType == "deformInit":
            code = self.deformInitCode
            parallel = self.parallelDeformInit

        elif codeType == "deform":
            code = self.deformCode
            parallel = self.parallelDeform

        elif codeType == "deformParallel":
            code = self.deformParallelCode
            parallel = False

        parallel = parallel and Patch.ParallelEnabled

        if code:
            code = code.replace("$@", mayaNamespace).replace("@", pathName+self.name+"__")

            if not (self.embedCode and Patch.EmbedCode):
                file = "{dir}/patch/temp/{name}__{type}.cpp".format(dir=EvaluatorLocalPath, name=pathName+self.name, type=codeType)
                with open(file, "w") as f:
                    f.write(code.encode("utf-8"))

                code = "#include \"%s\"\n"%os.path.basename(file)

        childrenBlocks = []
        parallelChildren = []
        if children:
            notMutedChildren = [c for c in self.children if c.state == Patch.StateNormal]

            for i, ch in enumerate(notMutedChildren): # process children
                cpp = ch.toRunCpp(codeType, mayaNamespace, path+self.name+"/", children, self.block)
                if cpp:
                    if parallel and len(notMutedChildren)>1:
                        lambdaName = pathName + self.name + "__" + ch.name + "_lambda"
                        cpp = "\n".join(["function<EvStatus()> %s = [&]() -> EvStatus {"%lambdaName,
                                         "EvStatus &STATUS = __STATUS_LIST__[%d];"%i,
                                         cpp,
                                         "STATUS = kNormal;",
                                         "return STATUS;",
                                         "};\n"])

                        parallelChildren.append(lambdaName)

                    if not childrenBlocks and parallel and len(notMutedChildren)>1:
                        childrenBlocks.append("vector<EvStatus> __STATUS_LIST__(%d, kNotImplemented);"%len(notMutedChildren))

                    childrenBlocks.append(cpp)

            if codeType == "run" and self.block:
                self.block = False                
                deformParallelCode = self.toRunCpp("deformParallel", mayaNamespace, path, children, False)
                if deformParallelCode:
                    deformParallelCode = evaluatorDeformParallelTemplate.replace("@DEFORM_PARALLEL", deformParallelCode)

                deformInit = self.toRunCpp("deformInit", mayaNamespace, path, children, False)
                deform = self.toRunCpp("deform", mayaNamespace, path, children, False)
                run = self.toRunCpp("run", mayaNamespace, path, children, False)
                runTemplate = evaluatorRunTemplate.replace("@RUN", run)\
                                                  .replace("@DEFORM_INIT", deformInit)\
                                                  .replace("@DEFORM_PARALLEL_TEMPLATE", deformParallelCode)\
                                                  .replace("@DEFORM", deform)

                childrenBlocks.append("// %s run fullBlock"%(path+self.name))
                childrenBlocks.append(runTemplate)

                self.block = True

        childrenCode = "\n".join(childrenBlocks)

        parallelInvokeCode = ""
        if childrenCode and parallel and len(parallelChildren) > 1:
            parallelInvokeCode = "".join(["tbb::parallel_invoke(",
                                          ", ".join(parallelChildren),
                                          ");\n",
                                          "for (const auto &stat : __STATUS_LIST__) if (stat != kNormal) {STATUS = stat; return STATUS;}"])

        if not code and not childrenCode:
            return ""

        code = "\n".join(["{" if self.codeBlocks else "",
                          "using namespace %s_namespace;"%(pathName+self.name), # skin last __ in namespace
                          code,
                          "};" if self.codeBlocks else ""])

        if childrenCode:
            childrenCode = "\n".join(["{",
                                      childrenCode,
                                      parallelInvokeCode,
                                      "}"])

        comment = "// %s %s"%(path+self.name, codeType)

        if not path:
            links = self.findCodeLinks(self, code)
            for f, r in links.items():
                code = code.replace(f, r)

            links = self.findCodeLinks(self, childrenCode)
            for f, r in links.items():
                childrenCode = childrenCode.replace(f, r)

        return "\n".join([comment, code, childrenCode])

    def buildCpp(self):
        deformParallelCode = self.toRunCpp("deformParallel")
        if deformParallelCode:
            deformParallelCode = evaluatorDeformParallelTemplate.replace("@DEFORM_PARALLEL", deformParallelCode)

        deformInit = self.toRunCpp("deformInit")
        runTemplate = evaluatorRunTemplate.replace("@RUN", self.toRunCpp("run"))\
                                          .replace("@DEFORM_INIT", deformInit)\
                                          .replace("@DEFORM_PARALLEL_TEMPLATE", deformParallelCode)\
                                          .replace("@DEFORM", self.toRunCpp("deform"))

        gpuTemplate = evaluatorGpuTemplate.replace("@ATTRIBUTES", self.toAttributesCpp(fileName="gpuAttributes", outputs=False))\
                                          .replace("@DEFORM_INIT", deformInit)\
                                          .replace("@GPU_RUN_KERNEL", self.toRunKernelCpp())

        code = evaluatorTemplate.replace("@RUN_TEMPLATE", runTemplate)\
                                .replace("@INCLUDES",self.toIncludesCpp())\
                                .replace("@COMMANDS", self.toCommandsCpp())\
                                .replace("@HELP_COMMAND", self.toHelpCommandCpp())\
                                .replace("@PRIVATE", self.toPrivateCpp())\
                                .replace("@DEFINES", self.toDefinesCpp())\
                                .replace("@IS_ACTUAL_CACHE", self.toIsActualCacheCpp())\
                                .replace("@DRAW", self.toRunCpp("draw"))\
                                .replace("@GET_ATTRIBUTES", self.toGetAttributesCpp())\
                                .replace("@ATTRIBUTES", self.toAttributesCpp())\
                                .replace("@GPU_CL_PROGRAMS", self.toGpuStaticPrograms())\
                                .replace("@GPU_TERMINATE", self.toGpuTerminateCpp())\
                                .replace("@GPU_TEMPLATE", gpuTemplate)

        buildFile = "{dir}/patch/temp/{name}.cpp".format(dir=EvaluatorLocalPath, name=self.name)

        with open(buildFile, "w") as f:
            f.write(formatCpp(code))

        return buildFile

    @staticmethod
    def cleanupBuild():
        for f in glob.glob(EvaluatorLocalPath+"/patch/temp/*.*"):
            os.remove(f)

def diffPatch(p1, p2, indent=""):
    pname = lambda f: os.path.splitext(os.path.basename(f))[0]
    result = [indent+"diff %s -> %s"%(pname(p1.getFileName()), pname(p2.getFileName()))]

    result.append(indent+"[ attributes ]")

    localIndent = "  "

    for a2 in p2.attributes: # added attributes
        a1 = p1.findAttribute(a2.name)
        if not a1:
            result.append(indent + localIndent + "+ " + a2.name)

    for a1 in p1.attributes: # removed attributes
        a2 = p2.findAttribute(a1.name)
        if not a2:
            result.append(indent + localIndent + "- "+a1.name)

    result.append(indent+"[ children ]")
    for ch2 in p2.children:
        ch1 = p1.findChildren(ch2.name)
        if not ch1:
            result.append(indent + localIndent + "+ "+ch2.name)
        else:
            result += diffPatch(ch1[0], ch2, indent+"\t")

    for ch1 in p1.children:
        ch2 = p2.findChildren(ch1.name)
        if not ch2:
            result.append(indent + localIndent + "- "+ch1.name)

    return result

# fileName = r"D:\Scripts\evaluator\patch\patches\bezierRig\bezierRig_v004.xml"
# p = Patch.loadFromFile(fileName)
# print Patch.getLatestDllByFileName(fileName, 2018)
