# -*- coding: utf-8 -*-

import sys
import re
import time
import string
import shutil
import json

try:
    from PySide2.QtCore import *
    from PySide2.QtGui import *
    from PySide2.QtWidgets import *

    import pymel.core as core
    MayaCurrentVersion = int(core.about(v=True))

except:
    from PyQt4.QtCore import *
    from PyQt4.QtGui import *

    MayaCurrentVersion = 2018

from patch_classes import *

EvaluatorBuildParams = {"version": MayaCurrentVersion, "type": "release", "fromServer": True}

PatchMainWindow = None
EvaluatorDocPath = "//azagoruyko/evaluator_docs/html/index.html"

def clamp(mn, mx, val):
    if val < mn:
        return mn
    elif val > mx:
        return mx
    else:
        return val

class CppHighlighter(QSyntaxHighlighter):
    def __init__(self, parent=None):
        super(CppHighlighter, self).__init__(parent)

        self.highlightingRules = []

        classFormat = QTextCharFormat()
        classFormat.setForeground(QColor(150, 140, 230))
        self.highlightingRules.append((QRegExp("\\b[A-Z]\\w*\\b"), classFormat))

        constFormat = QTextCharFormat()
        constFormat.setForeground(QColor(150, 200, 150))
        self.highlightingRules.append((QRegExp("\\b[A-Z0-9_]+\\b"), constFormat))

        numFormat = QTextCharFormat()
        numFormat.setForeground(QColor(150, 200, 150))
        self.highlightingRules.append((QRegExp("\\b(0x[0-9]+)\\b|\\b[0-9\\.]+f*\\b"), numFormat))

        functionFormat = QTextCharFormat()
        functionFormat.setForeground(QColor(130, 160, 220))
        self.highlightingRules.append((QRegExp("\\b\\w+(?=\\s*\\()"), functionFormat))

        templateFunctionFormat = QTextCharFormat()
        templateFunctionFormat.setForeground(QColor(130, 160, 220))
        self.highlightingRules.append((QRegExp("\\w+\\s*(?=(<[^>]+>)\\()"), templateFunctionFormat))

        keywordFormat = QTextCharFormat()
        keywordFormat.setForeground(QColor(180, 100, 140))

        keywords = ["\\b%s\\b"%k for k in ["alignas", "alignof", "and", "and_eq", "asm", "auto", "bitand", "bitor",
                                           "bool", "break", "case", "catch", "char", "char16_t", "char32_t", "class",
                                           "compl", "const", "constexpr", "const_cast", "continue", "decltype",
                                           "default", "delete", "do", "double", "dynamic_cast", "else", "enum",
                                           "explicit", "export", "extern", "false", "float", "for", "friend", "goto",
                                           "if", "inline", "int", "long", "mutable", "namespace", "new", "noexcept",
                                           "not", "not_eq", "nullptr", "operator", "or", "or_eq", "private",
                                           "protected", "public", "register", "reinterpret_cast", "return", "short",
                                           "signed", "sizeof", "size_t", "static", "static_assert", "static_cast",
                                           "struct", "switch", "template", "this", "thread_local", "throw", "true",
                                           "try", "typedef", "typeid", "typename", "union", "unsigned", "using",
                                           "virtual", "void", "volatile", "wchar_t", "while", "xor", "xor_eq",
                                           "__global", "global", "local", "__local", "__kernel", "kernel", "__constant",
                                           "constant", "__private", "private", "uchar", "char2", "char3", "char4",
                                           "char8", "char16", "uchar2", "uchar3", "uchar4", "uchar8", "uchar16",
                                           "short2", "short3", "short4", "short8", "short16", "ushort2", "ushort3",
                                           "ushort4", "ushort8", "ushort16", "int2", "int3", "int4", "int8", "int16",
                                           "uint", "uint2", "uint3", "uint4", "uint8", "uint16", "long2", "long3",
                                           "long4", "long8", "long16", "ulong2", "ulong3", "ulong4", "ulong8",
                                           "ulong16", "float2", "float3", "float4", "float8", "float16", "half2",
                                           "half3", "half4", "half8", "half16", "double2", "double3", "double4",
                                           "double8", "double16", "uniform", "varying"]]

        self.highlightingRules += [(QRegExp(pattern), keywordFormat) for pattern in keywords]

        attrFormat = QTextCharFormat()
        attrFormat.setForeground(QColor(100, 180, 180))
        self.highlightingRules.append((QRegExp("@\\b\\w+\\b"), attrFormat))

        connFormat = QTextCharFormat()
        connFormat.setForeground(QColor(100, 230, 120))
        self.highlightingRules.append((QRegExp("\\$\\w+/[/\\w]+"), connFormat))

        diezFormat = QTextCharFormat()
        diezFormat.setForeground(QColor(100, 100, 150))
        self.highlightingRules.append((QRegExp("#\\b\\w+\\b"), diezFormat))        
        
        self.quotationFormat = QTextCharFormat()
        self.quotationFormat.setForeground(QColor(200, 200, 100))
        self.highlightingRules.append((QRegExp("\"(\\\\\"|[^\"])*\""), self.quotationFormat))

        singleLineCommentFormat = QTextCharFormat()
        singleLineCommentFormat.setForeground(QColor(90, 90, 90))
        self.highlightingRules.append((QRegExp("//[^\n]*"), singleLineCommentFormat))

        self.multiLineCommentFormat = QTextCharFormat()
        self.multiLineCommentFormat.setForeground(QColor(90, 90, 90))

        self.commentStartExpression = QRegExp("/\\*")
        self.commentEndExpression = QRegExp("\\*/")

        self.highlightedWordFormat = QTextCharFormat()
        self.highlightedWordFormat.setForeground(QColor(200, 200, 200))
        self.highlightedWordFormat.setBackground(QBrush(QColor(100, 55, 170)))
        self.highlightedWordRegexp = None

    def highlightBlock(self, text):
        for pattern, format in self.highlightingRules:
            if not pattern:
                continue

            expression = QRegExp(pattern)
            index = expression.indexIn(text)
            while index >= 0:
                length = expression.matchedLength()
                self.setFormat(index, length, format)
                index = expression.indexIn(text, index + length)

        self.setCurrentBlockState(0)

        for i, (startExpr, endExpr, fmt) in enumerate([(QRegExp("R\"\\("), QRegExp("\\f)\""), self.quotationFormat),
                                                       (self.commentStartExpression, self.commentEndExpression, self.multiLineCommentFormat)]):

            idx = i + 1
            startIndex = 0
            if self.previousBlockState() != idx:
                startIndex = startExpr.indexIn(text)

            while startIndex >= 0:
                endIndex = endExpr.indexIn(text, startIndex)

                if endIndex == -1:
                    self.setCurrentBlockState(idx)
                    length = len(text) - startIndex
                else:
                    length = endIndex - startIndex + endExpr.matchedLength()

                self.setFormat(startIndex, length, fmt)
                startIndex = startExpr.indexIn(text, startIndex + length)

        if self.highlightedWordRegexp:
            expression = QRegExp(self.highlightedWordRegexp)
            index = expression.indexIn(text)
            while index >= 0:
                length = expression.matchedLength()
                self.setFormat(index, length, self.highlightedWordFormat)
                index = expression.indexIn(text, index + length)

class CustomThread(QThread):
    def __init__(self, runFunc, callbackFunc=None, **kwargs):
        super(CustomThread, self).__init__(**kwargs)

        self.runFunc = runFunc
        self.callbackFunc = callbackFunc

        self.finished.connect(self.threadFinished)

    def threadFinished(self):
        if self.callbackFunc:
            self.callbackFunc()

    def run(self):
        self.runFunc()

class CompletionThread(QThread):
    def __init__(self, editor, finishedCallback=None, textCursor=None, **kwargs):
        super(CompletionThread, self).__init__(**kwargs)

        self.editor = editor
        self.textCursor = textCursor if textCursor else editor.textCursor()
        self.finishedCallback = finishedCallback
        self.finished.connect(self.threadFinished)

        self.completions = []
        self.error = None

    def threadFinished(self):
        if self.error:
            logWidget = PatchMainWindow.browserWidget.rightBrowserWidget.logWidget
            logWidget.update(self.error, False)

        if self.finishedCallback and self.completions:
            self.finishedCallback()

    def run(self):
        self.error = None

        tw = PatchMainWindow.browserWidget.leftBrowserWidget.treeWidget
        tw_item = tw.currentItem()

        CL = False
        if self.editor.codeType == "clCode":
            program = self.editor.patch.getCLProgram()

            programFile = EvaluatorLocalPath + "/patch/temp/"+self.editor.patch.name+".cl"
            with open(programFile, "w") as f:
                f.write(program)

            cursor = self.editor.textCursor() if not self.textCursor else self.textCursor
            line = cursor.blockNumber()+1
            column = cursor.columnNumber()+1
            lineText = unicode(cursor.block().text())
            refcount = string.count(lineText[:column], "@")
            offset = refcount * len("attr_") - refcount
            column += offset

            rootFileName = programFile
            currentFileName = programFile
            CL = True

        else:
            topLevelItem = tw_item
            codePath = [tw_item.patch.name]
            while topLevelItem.parent():
                topLevelItem = topLevelItem.parent()
                codePath.append(topLevelItem.patch.name)

            codePath = "__".join(codePath[::-1])

            Patch.EmbedCode = False
            ispcBuild(topLevelItem.patch.generateIspcFile())

            try:
                topLevelItem.patch.buildCpp()
            except Exception as e:
                self.error = str(e)
                return

            Patch.EmbedCode = True
            rootFileName = os.path.realpath(EvaluatorLocalPath+"/patch/temp/%s.cpp"%topLevelItem.patch.name)
            currentFileName = os.path.realpath(EvaluatorLocalPath+"/patch/temp/%s__%s.cpp"%(codePath, self.editor.codeType.replace("Code", "")))

            cursor = self.editor.textCursor() if not self.textCursor else self.textCursor

            line = cursor.blockNumber()+1
            column = cursor.columnNumber()+1

            if codePath:
                namespace = codePath + "__"
                lineText = unicode(cursor.block().text())
                refcount = string.count(lineText[:column], "@")
                offset = refcount * len(namespace) - refcount
                column += offset

        self.completions = clangAutoCompletion(rootFileName, currentFileName, line, column, CL)

def highlightLine(widget, line=-1, clear=False):
    if line == -1:
        block = widget.textCursor().block()
    else:
        block = widget.document().findBlockByLineNumber(line)

        if not block.isValid():
            return

    fmt = QTextCharFormat()
    if not clear:
        fmt.setBackground(QColor(50, 80, 100))

    blockPos = block.position()

    cursor = widget.textCursor()
    cursor.setPosition(blockPos)
    cursor.select(QTextCursor.LineUnderCursor)
    cursor.setCharFormat(fmt)
    cursor.clearSelection()
    cursor.movePosition(QTextCursor.StartOfLine)

    widget.setTextCursor(cursor)

class SwoopHighligher(QSyntaxHighlighter):
    def __init__(self, parent=None):
        super(SwoopHighligher, self).__init__(parent)

        self.highlightingRules = []

        linumFormat = QTextCharFormat()
        linumFormat.setForeground(QColor(180, 100, 120))
        self.highlightingRules.append((QRegExp("^\\s*\\d+\\s+"), linumFormat))

        headerFormat = QTextCharFormat()
        headerFormat.setForeground(QColor(120, 100, 180))
        headerFormat.setFontWeight(QFont.Bold)
        self.highlightingRules.append((QRegExp("^[a-zA-Z][\\w -]*"), headerFormat))

        subHeaderFormat = QTextCharFormat()
        subHeaderFormat.setForeground(QColor(120, 180, 120))
        self.highlightingRules.append((QRegExp("\\[[\\w ]+\\]$"), subHeaderFormat))

        commentFormat = QTextCharFormat()
        commentFormat.setForeground(QColor(90, 90, 90))
        self.highlightingRules.append((QRegExp("//.*$"), commentFormat))

        highlightedWordsFormat = QTextCharFormat()
        highlightedWordsFormat.setForeground(QColor(200, 200, 200))
        highlightedWordsFormat.setBackground(QBrush(QColor(100, 55, 170)))
        self.highlightingRules.append((None, highlightedWordsFormat))

    def highlightBlock(self, text):
        for pattern, format in self.highlightingRules:
            if not pattern:
                continue

            expression = QRegExp(pattern)
            index = expression.indexIn(text)
            while index >= 0:
                length = expression.matchedLength()
                self.setFormat(index, length, format)
                index = expression.indexIn(text, index + length)

        self.setCurrentBlockState(0)

class SwoopSearchDialog(QDialog):
    def __init__(self, edit, **kwargs):
        super(SwoopSearchDialog, self).__init__(**kwargs)

        self.edit = edit

        self.setWindowFlags(Qt.FramelessWindowHint)
        self.setWindowTitle("Swoop")

        layout = QVBoxLayout()
        self.setLayout(layout)

        self.filterWidget = QLineEdit()
        self.filterWidget.setToolTip("Ctrl-C - case sensitive<br>Ctrl-W - word boundary<br>Ctrl-B - find inside brackets<br>Ctrl-D - down only<br>Ctrl-R - replace mode")
        self.filterWidget.textChanged.connect(self.filterTextChanged)
        self.filterWidget.keyPressEvent = self.filterKeyPressEvent

        self.resultsWidget = QTextEdit()
        self.resultsWidget.setReadOnly(True)
        self.resultsWidget.setWordWrapMode(QTextOption.NoWrap)
        self.resultsWidget.syntax = SwoopHighligher(self.resultsWidget.document())
        self.resultsWidget.mousePressEvent = self.resultsMousePressEvent
        self.resultsWidget.keyPressEvent = self.filterWidget.keyPressEvent

        self.statusWidget = QLabel()
        self.statusWidget.hide()

        layout.addWidget(self.filterWidget)
        layout.addWidget(self.resultsWidget)
        layout.addWidget(self.statusWidget)
        self.rejected.connect(self.whenRejected)

    def initialize(self):
        self.useWordBoundary = False
        self.findInsideBrackets = False
        self.caseSensitive = True
        self.downOnly = False

        self.replaceMode = False
        self.numberSeparator = "  "

        self.previousPattern = None
        self.previousLines = []

        self.savedSettings = {}

        self.text = unicode(self.edit.toPlainText())
        lines = self.text.split("\n")
        cursor = self.edit.textCursor()

        self.updateSavedCursor()

        self.savedSettings["lines"] = lines

        findText = unicode(cursor.selectedText())
        if not findText:
            findText = wordAtCursor(cursor)[0]

        self.filterWidget.setText(findText)
        self.filterWidget.setStyleSheet("")

    def updateSavedCursor(self):
        cursor = self.edit.textCursor()
        brackets = findBracketSpans(self.text, cursor.position(), brackets="{")
        self.savedSettings["cursor"] = cursor
        self.savedSettings["scroll"] = self.edit.verticalScrollBar().value()
        self.savedSettings["brackets"] = brackets

        self.findInsideBrackets = self.findInsideBrackets and brackets[0]

    def showEvent(self, event):
        self.updateSavedCursor()
        self.reposition()
        self.filterWidget.setFocus()

    def update(self):
        self.initialize()
        self.updateStatus()
        self.filterTextChanged()

    def resultsMousePressEvent(self, event):
        cursor = self.resultsWidget.cursorForPosition(event.pos())

        highlightLine(self.resultsWidget, clear=True)
        highlightLine(self.resultsWidget, cursor.block().blockNumber())
        self.resultsLineChanged()

    def reposition(self):
        rect = self.edit.cursorRect()
        c = self.edit.mapToGlobal(rect.topLeft())

        w = self.resultsWidget.document().idealWidth() + 30
        h = self.resultsWidget.document().blockCount()*self.resultsWidget.cursorRect().height() + 110

        self.setGeometry(c.x(), c.y() + 22, clamp(0, 500, w), clamp(0, 400, h))

    def resultsLineChanged(self):
        if self.replaceMode:
            return

        cursor = self.resultsWidget.textCursor()
        cursor.select(QTextCursor.LineUnderCursor)
        line = unicode(cursor.selectedText())

        if not line:
            return

        lineNumber, text = re.search("^([0-9]+)\\s-*(.*)$", line).groups("")
        self.edit.gotoLine(int(lineNumber))

        currentFilter = self.getFilterPattern()

        r = re.search(currentFilter, text, re.IGNORECASE if not self.caseSensitive else 0)

        if r:
            cursor = self.edit.textCursor()
            pos = cursor.block().position() + r.start() - 1
            if pos >- 0:
                cursor.setPosition(pos)
                self.edit.setTextCursor(cursor)

            cursorY = self.edit.cursorRect().top()
            scrollBar = self.edit.verticalScrollBar()
            scrollBar.setValue(scrollBar.value() + cursorY - self.edit.geometry().height()/2)

        self.reposition()

    def updateStatus(self):
        items = []

        if self.useWordBoundary:
            items.append("[word]")

        if self.caseSensitive:
            items.append("[case]")

        if self.findInsideBrackets:
            items.append("[brackets]")

        if self.downOnly:
            items.append("[down]")

        if self.replaceMode:
            items.append("[REPLACE '%s']"%self.previousPattern)

        if items:
            self.statusWidget.setText(" ".join(items))
            self.statusWidget.show()
        else:
            self.statusWidget.hide()

    def filterKeyPressEvent(self, event):
        shift = event.modifiers() & Qt.ShiftModifier
        ctrl = event.modifiers() & Qt.ControlModifier
        alt = event.modifiers() & Qt.AltModifier

        rw = self.resultsWidget
        line = rw.textCursor().block().blockNumber()
        lineCount = rw.document().blockCount()-1

        if event.key() in [Qt.Key_Down, Qt.Key_Up, Qt.Key_PageDown, Qt.Key_PageUp]:
            if event.key() == Qt.Key_Down:
                highlightLine(rw, clamp(0, lineCount, line), clear=True)
                highlightLine(rw, clamp(0, lineCount, line+1))
            elif event.key() == Qt.Key_Up:
                highlightLine(rw, clamp(0, lineCount, line), clear=True)
                highlightLine(rw, clamp(0, lineCount, line-1))

            elif event.key() == Qt.Key_PageDown:
                highlightLine(rw, clamp(0, lineCount, line), clear=True)
                highlightLine(rw, clamp(0, lineCount, line+5))
            elif event.key() == Qt.Key_PageUp:
                highlightLine(rw, clamp(0, lineCount, line), clear=True)
                highlightLine(rw, clamp(0, lineCount, line-5))

            self.resultsLineChanged()

        elif ctrl and event.key() == Qt.Key_W: # use word boundary
            if not self.replaceMode:
                self.useWordBoundary = not self.useWordBoundary
                self.updateStatus()
                self.filterTextChanged()

        elif ctrl and event.key() == Qt.Key_B: # find inside brackets
            if not self.replaceMode:
                self.findInsideBrackets = not self.findInsideBrackets
                self.updateSavedCursor()
                self.updateStatus()
                self.filterTextChanged()

        elif ctrl and event.key() == Qt.Key_D: # down only
            if not self.replaceMode:
                self.downOnly = not self.downOnly

                self.updateSavedCursor()

                self.updateStatus()
                self.filterTextChanged()

        elif ctrl and event.key() == Qt.Key_C: # case sensitive
            if self.filterWidget.selectedText():
                self.filterWidget.copy()
            else:
                if not self.replaceMode:
                    self.caseSensitive = not self.caseSensitive
                    self.updateStatus()
                    self.filterTextChanged()

        elif ctrl and event.key() == Qt.Key_R: # replace mode
            self.replaceMode = not self.replaceMode
            if self.replaceMode:
                self.filterWidget.setStyleSheet("background-color: #433567")
                self.previousPattern = self.getFilterPattern()
            else:
                self.filterWidget.setStyleSheet("")
                self.filterTextChanged()

            self.updateStatus()

        elif event.key() == Qt.Key_F3:
            self.accept()

        elif event.key() == Qt.Key_Return: # accept
            if self.replaceMode:
                cursor = self.edit.textCursor()

                savedBlock = self.savedSettings["cursor"].block()
                savedColumn = self.savedSettings["cursor"].positionInBlock()

                doc = self.edit.document()

                cursor.beginEditBlock()
                lines = unicode(self.resultsWidget.toPlainText()).split("\n")
                for line in lines:
                    if not line.strip():
                        continue

                    lineNumber, text = re.search("^([0-9]+)%s(.*)$"%self.numberSeparator, line).groups("")
                    lineNumber = int(lineNumber)

                    blockPos = doc.findBlockByLineNumber(lineNumber-1).position()
                    cursor.setPosition(blockPos)
                    cursor.select(QTextCursor.LineUnderCursor)
                    cursor.removeSelectedText()
                    cursor.insertText(text)

                cursor.endEditBlock()

                cursor.setPosition(savedBlock.position() + savedColumn)
                self.edit.setTextCursor(cursor)
                self.edit.verticalScrollBar().setValue(self.savedSettings["scroll"])

            self.accept()

        else:
            QLineEdit.keyPressEvent(self.filterWidget, event)

    def whenRejected(self):
        self.edit.setTextCursor(self.savedSettings["cursor"])
        self.edit.verticalScrollBar().setValue(self.savedSettings["scroll"])
        self.edit.setFocus()

    def getFilterPattern(self):
        currentFilter = re.escape(unicode(self.filterWidget.text()))
        if not currentFilter:
            return ""

        if self.useWordBoundary:
            currentFilter = "\\b" + currentFilter + "\\b"

        return currentFilter

    def filterTextChanged(self):
        self.resultsWidget.clear()
        self.resultsWidget.setCurrentCharFormat(QTextCharFormat())

        if self.replaceMode: # replace mode
            subStr = unicode(self.filterWidget.text()).replace("\\", "\\\\")

            pattern = self.getFilterPattern()

            lines = []
            for line in self.previousLines:
                n, text = re.search("^([0-9]+)%s(.*)$"%self.numberSeparator, line).groups("")

                text = re.sub(self.previousPattern, subStr, text, 0, re.IGNORECASE if not self.caseSensitive else 0)
                newLine = "%s%s%s"%(n, self.numberSeparator, text)
                lines.append(newLine)

            self.resultsWidget.setText("\n".join(lines))
            self.resultsWidget.syntax.highlightingRules[-1] = (pattern, self.resultsWidget.syntax.highlightingRules[-1][1])
            self.resultsWidget.syntax.rehighlight()

        else: # search mode
            startBlock, endBlock = 0, 0

            if self.findInsideBrackets:
                cursor = QTextCursor(self.savedSettings["cursor"])
                cursor.setPosition(self.savedSettings["brackets"][1])
                startBlock = cursor.block().blockNumber()
                cursor.setPosition(self.savedSettings["brackets"][2])
                endBlock = cursor.block().blockNumber()

            if self.downOnly:
                cursor = QTextCursor(self.savedSettings["cursor"])
                startBlock = cursor.block().blockNumber()

            currentFilter = self.getFilterPattern()

            currentBlock = self.edit.textCursor().block().blockNumber()

            self.previousLines = []

            currentFilterText = unicode(self.filterWidget.text()).replace("\\", "\\\\")
            counter = 0
            currentIndex = 0

            for i, line in enumerate(self.savedSettings["lines"]):
                if not line.strip():
                    continue

                if self.findInsideBrackets and (i < startBlock or i > endBlock):
                    continue

                if self.downOnly and i < startBlock:
                    continue

                if i == currentBlock:
                    currentIndex = counter

                r = re.search(currentFilter, line, re.IGNORECASE if not self.caseSensitive else 0)
                if r:
                    item = "%s%s%s"%(i+1, self.numberSeparator, line)
                    self.previousLines.append(item)
                    counter += 1

            self.resultsWidget.setText("\n".join(self.previousLines))

            self.resultsWidget.syntax.highlightingRules[-1] = (currentFilter, self.resultsWidget.syntax.highlightingRules[-1][1])
            self.resultsWidget.syntax.rehighlight()

            highlightLine(self.resultsWidget, currentIndex)
            self.resultsLineChanged()

class SearchEverywhereWidget(QFrame):
    def __init__(self, **kwargs):
        super(SearchEverywhereWidget, self).__init__(**kwargs)

        self.lastCursorData = None

        self.useWordBoundary = False
        self.useCaseSensitive = False

        self.previousHighlightLine = None

        self.setWindowTitle("Search Everywhere")
        self.setGeometry(600, 300, 800, 500)

        layout = QVBoxLayout()
        self.setLayout(layout)

        hlayout = QHBoxLayout()

        self.filterWidget = QLineEdit()
        self.filterWidget.keyPressEvent = self.filterKeyPressEvent
        self.filterWidget.textChanged.connect(self.filterTextChanged)

        self.filterStatusWidget = QLabel()
        hlayout.addWidget(self.filterWidget)
        hlayout.addWidget(self.filterStatusWidget)

        layout.addLayout(hlayout)

        self.resultsWidget = QTextEdit()
        self.resultsWidget.syntax = SwoopHighligher(self.resultsWidget.document())
        self.resultsWidget.setReadOnly(True)
        self.resultsWidget.setWordWrapMode(QTextOption.NoWrap)
        self.resultsWidget.setTabStopWidth(16)
        self.resultsWidget.mousePressEvent = self.resultsMousePressEvent
        self.resultsWidget.keyPressEvent = self.resultsKeyPressEvent

        self.goBackBtn = QPushButton("Go back")
        self.goBackBtn.clicked.connect(self.goBackClicked)

        layout.addWidget(self.resultsWidget)
        layout.addWidget(self.goBackBtn)

    def showEvent(self, event):
        tw = PatchMainWindow.browserWidget.leftBrowserWidget.treeWidget
        tabWidget = PatchMainWindow.browserWidget.rightBrowserWidget.patchCppEditorWidget.tabWidget
        ed = tabWidget.widget(tabWidget.currentIndex()).editorWidget
        self.lastCursorData = (tw.currentItem(), tabWidget.currentIndex(), ed.textCursor().position(), ed.verticalScrollBar().value())

    def update(self):
        tabWidget = PatchMainWindow.browserWidget.rightBrowserWidget.patchCppEditorWidget.tabWidget
        ed = tabWidget.widget(tabWidget.currentIndex()).editorWidget

        word, _, _ = wordAtCursor(ed.textCursor())
        self.filterWidget.setText(word)
        self.filterTextChanged()

        self.filterWidget.setFocus()

    def resultsKeyPressEvent(self, event):
        if event.key() == Qt.Key_Down:
            self.selectLine(self.resultsWidget.textCursor().blockNumber()+1)

        elif event.key() == Qt.Key_Up:
            self.selectLine(self.resultsWidget.textCursor().blockNumber()-1)

        elif event.key() == Qt.Key_Return:
            self.followLine(self.resultsWidget.textCursor())

        else:
            QTextEdit.keyPressEvent(self.resultsWidget, event)

    def filterKeyPressEvent(self, event):
        shift = event.modifiers() & Qt.ShiftModifier
        ctrl = event.modifiers() & Qt.ControlModifier
        alt = event.modifiers() & Qt.AltModifier

        if event.key() == Qt.Key_Escape:
            self.close()

        elif event.key() == Qt.Key_Return:
            self.filterTextChanged()

        elif ctrl and event.key() == Qt.Key_W:
            self.useWordBoundary = not self.useWordBoundary
            self.updateFilterStatus()

        elif ctrl and event.key() == Qt.Key_C:
            if self.filterWidget.selectedText():
                self.filterWidget.copy()
            else:
                self.useCaseSensitive = not self.useCaseSensitive
                self.updateFilterStatus()

        else:
            QLineEdit.keyPressEvent(self.filterWidget, event)

    def selectLine(self, line):
        if line < 1 or line > self.resultsWidget.document().blockCount()-3:
            return

        if self.previousHighlightLine and self.previousHighlightLine != line:
            highlightLine(self.resultsWidget, self.previousHighlightLine, True)

        highlightLine(self.resultsWidget, line, False)

        self.previousHighlightLine = line

    def resultsMousePressEvent(self, event):
        if event.button() == Qt.LeftButton:
            cursor = self.resultsWidget.cursorForPosition(event.pos())
            if not re.search("^\\s*(\\d+)\\b", unicode(cursor.block().text())):
                return

            self.selectLine(cursor.block().blockNumber())
            self.followLine(cursor)

    def updateFilterStatus(self):
        items = []

        if self.useWordBoundary:
            items.append("[word]")

        if self.useCaseSensitive:
            items.append("[case]")

        self.filterStatusWidget.setText(" ".join(items))
        self.filterTextChanged()

    def goBackClicked(self):
        item, tabIndex, pos, scroll = self.lastCursorData
        if not item:
            return

        tw = PatchMainWindow.browserWidget.leftBrowserWidget.treeWidget
        tw.setCurrentItem(item)
        tw.treeItemClicked(item)

        patchEditor = PatchMainWindow.browserWidget.rightBrowserWidget.patchCppEditorWidget
        tabWidget = patchEditor.tabWidget
        tabWidget.setCurrentIndex(tabIndex)

        ed = tabWidget.widget(tabIndex).editorWidget
        cursor = ed.textCursor()
        cursor.setPosition(pos)
        ed.setTextCursor(cursor)
        ed.verticalScrollBar().setValue(scroll)

    def followLine(self, cursor):
        line = unicode(cursor.block().text())
        r = re.search("^\\s*(\\d+)\\b", line)
        if not r:
            return

        linum = int(r.group(1))
        block = cursor.block()
        while block.isValid() and not re.match("[^ \t]", unicode(block.text())):
            block = block.previous()

        path, ct = re.search("^([^\\[]+)\\[([\\w ]+)\\]$", unicode(block.text())).groups()
        codeType = "".join([(w.lower() if i == 0 else w) for i, w in enumerate(ct.split())])+"Code"
        path = [p.strip() for p in path.split(" - ")]

        tw = PatchMainWindow.browserWidget.leftBrowserWidget.treeWidget

        root = None
        for p in path:
            item = tw.findItemByName(p, root)
            root = item

        tw.setCurrentItem(item)
        PatchMainWindow.browserWidget.leftBrowserWidget.update(item.patch)
        PatchMainWindow.browserWidget.rightBrowserWidget.update(item.patch)

        patchEditor = PatchMainWindow.browserWidget.rightBrowserWidget.patchCppEditorWidget
        tabWidget = patchEditor.tabWidget
        idx = patchEditor.codeTypes.index(codeType)
        tabWidget.setCurrentIndex(idx)

        ed = tabWidget.widget(tabWidget.currentIndex()).editorWidget

        ed.gotoLine(linum)
        ed.centerLine()

    def filterTextChanged(self):
        pattern = re.escape(unicode(self.filterWidget.text()))
        if self.useWordBoundary:
            pattern = "\\b%s\\b"%pattern

        if len(pattern) < 3:
            self.resultsWidget.clear()
            return

        tw = PatchMainWindow.browserWidget.leftBrowserWidget.treeWidget
        rootItem = tw.invisibleRootItem()

        result = []
        for i in range(rootItem.childCount()):
            found = rootItem.child(i).patch.findTextRecursively(pattern, re.IGNORECASE if not self.useCaseSensitive else 0)
            if found:
                result.append(found)

        self.resultsWidget.setCurrentCharFormat(QTextCharFormat())
        self.resultsWidget.setText("\n".join(result))
        # self.resultsWidget.syntax.highlightingRules[-1] = (pattern, self.resultsWidget.syntax.highlightingRules[-1][1])
        # self.resultsWidget.syntax.rehighlight()

class CppEditorWidget(QTextEdit):
    def __init__(self, patch=None, codeType="", **kwargs):
        super(CppEditorWidget, self).__init__(**kwargs)

        self.patch = patch
        self.codeType = codeType
        self.lastSearch = ""
        self.lastReplace = ""

        self.editorsState = {}

        self.thread = None
        self.canShowCompletions = True

        self.currentFontPointSize = 16

        self.words = []
        self.clangCompletion = []
        self.currentWord = ("", 0, 0)

        self.fuzzySearch = False
        self.cursorColor = QColor(150, 50, 50)

        self.searchStartWord = ("", 0, 0)
        self.prevCursorPosition = 0

        self.swoopSearchDialog = SwoopSearchDialog(self)

        words = ["STARTUP",
                 "ASSERT",
                 "ENVELOPE",
                 "CACHE_ATTRIBUTES",
                 "ATTRIBUTES",
                 "BEGIN_PARALLEL_FOR",
                 "END_PARALLEL_FOR",
                 "GET_INPUT_ATTRIBUTE",
                 "GET_OUTPUT_ATTRIBUTE",
                 "GET_INPUT_ARRAY_ATTRIBUTE",
                 "GET_OUTPUT_ARRAY_ATTRIBUTE",
                 "GET_SHARED", "SET_SHARED",
                 "DEFORM_PARALLEL_ENABLED",
                 "EXIT", "ERROR", "NOT_IMPLEMENTED"] + Attribute.EvTypes

        deformerWords = ["POINTS", "LOCAL2WORLD", "LOCAL2WORLD_INVERSE", "WEIGHTS", "INDICES", "DATA", "GEOM_INDEX", "SOA_POINTS", "SOA_ENABLED"]

        self.wordsPerCodeType = {
            "definesCode": set(["BEGIN_PARALLEL_FOR", "END_PARALLEL_FOR", "EV_NODE", "EV_NODE_VAR"]),
            "commandsCode": set(["BEGIN_COMMAND", "END_COMMAND", "SET_RESULT", "GET_ATTRIBUTES", "RUN_ATTRIBUTES", "DEFORM_ATTRIBUTES"] + words),
            "runCode": set(words + ["IS_DEFORMER"]),
            "deformInitCode": set(words + ["NUM_ELEMENTS", "__GPU_INIT__"]),
            "deformCode": set(words + deformerWords),
            "deformParallelCode": set(words + deformerWords + ["POINT"]),
            "drawCode": set(["DRAWER"]),
            "clCode": set(["__kernel",
                           "ENVELOPE",
                           "NUM_ELEMENTS",
                           "INPUT_POSITIONS",
                           "OUTPUT_POSITIONS",
                           "POINT",
                           "LOCAL2WORLD",
                           "LOCAL2WORLD_INVERSE",
                           "WEIGHTS",
                           "_DEBUG"]),
            "ispcCode": set(["uniform", "varying", "export"])}

        self.setContextMenuPolicy(Qt.DefaultContextMenu)

        self.completionWidget = CompletionWidget([], parent=self)
        self.completionWidget.hide()

        if self.patch:
            self.setWindowTitle("C++ Editor - %s.%s"%(self.patch.name, self.codeType))

        self.syntax = CppHighlighter(self.document())
        self.setTabStopWidth(16)
        self.setAcceptRichText(False)
        self.setCursorWidth(8)
        self.setWordWrapMode(QTextOption.NoWrap)

        self.cursorPositionChanged.connect(self.editorCursorPositionChanged)
        self.verticalScrollBar().valueChanged.connect(lambda _: self.saveState(cursor=False, scroll=True, bookmarks=False))
        self.textChanged.connect(self.editorTextChanged)

    def setBookmark(self, line=-1):
        if line == -1:
            block = self.textCursor().block()
        else:
            block = self.document().findBlockByNumber(line)

        blockData = block.userData()

        if not blockData:
            blockData = TextBlockData()
            blockData.hasBookmark = True
        else:
            blockData.hasBookmark = not blockData.hasBookmark

        if isinstance(self.parent(), CppEditorWithNumbersWidget):
            self.parent().numberBarWidget.update()

        block.setUserData(blockData)
        self.saveState(cursor=False, scroll=False, bookmarks=True)

    def gotoNextBookmark(self, start=-1):
        doc = self.document()

        if start == -1:
            start =  self.textCursor().block().blockNumber()+1

        for i in range(start, doc.blockCount()):
            b = doc.findBlockByNumber(i)

            blockData = b.userData()
            if blockData and blockData.hasBookmark:
                self.setTextCursor(QTextCursor(b))
                self.centerLine()
                break

    def loadState(self, cursor=True, scroll=True, bookmarks=True):
        scrollBar = self.verticalScrollBar()

        self.blockSignals(True)
        scrollBar.blockSignals(True)

        if not self.patch or not self.editorsState.get(self.patch) or not self.editorsState[self.patch].get(self.codeType):
            c = self.textCursor()
            c.setPosition(0)
            self.setTextCursor(c)
            scrollBar.setValue(0)

        else:
            state = self.editorsState[self.patch][self.codeType]
            if cursor:
                c = self.textCursor()
                c.setPosition(state["cursor"])
                self.setTextCursor(c)

            if scroll:
                scrollBar = self.verticalScrollBar()
                scrollBar.setValue(state["scroll"])

            if bookmarks:
                doc = self.document()
                for i in state.get("bookmarks", []):
                    b = doc.findBlockByNumber(i)
                    self.setBookmark(i)

        self.blockSignals(False)
        scrollBar.blockSignals(False)

    def saveState(self, cursor=True, scroll=True, bookmarks=False):
        if not self.patch:
            return

        if not self.editorsState.get(self.patch):
            self.editorsState[self.patch] = {}

        if not self.editorsState[self.patch].get(self.codeType):
            self.editorsState[self.patch][self.codeType] = {"cursor": 0, "scroll": 0, "bookmarks": []}

        state = self.editorsState[self.patch][self.codeType]

        if cursor:
            state["cursor"] = self.textCursor().position()
        if scroll:
            state["scroll"] = self.verticalScrollBar().value()
        if bookmarks:
            doc = self.document()

            state["bookmarks"] = []
            for i in range(doc.blockCount()):
                b = doc.findBlockByNumber(i)
                data = b.userData()
                if data and data.hasBookmark:
                    state["bookmarks"].append(i)

    def contextMenuEvent(self, event):
        menu = QMenu(self)

        swoopAction = QAction("Swoop search\tF3", self)
        swoopAction.triggered.connect(lambda: self.swoopSearch(True))
        menu.addAction(swoopAction)

        searchEverywhereAction = QAction("Search everywhere\tF5", self)
        searchEverywhereAction.triggered.connect(lambda: self.searchEverywhere(True))
        menu.addAction(searchEverywhereAction)

        gotoLineAction = QAction("Goto line\tCtrl-G", self)
        gotoLineAction.triggered.connect(self.gotoLine)
        menu.addAction(gotoLineAction)

        selectAllAction = QAction("Select All", self)
        selectAllAction.triggered.connect(self.selectAll)
        menu.addAction(selectAllAction)

        menu.addSeparator()

        formatAction = QAction("Format\tAlt-Shift-F", self)
        formatAction.triggered.connect(lambda: self.setTextSafe(formatCpp(unicode(self.toPlainText()))))
        menu.addAction(formatAction)

        checkAction = QAction("Check code\tCtrl-C", self)
        checkAction.triggered.connect(lambda: PatchMainWindow.browserWidget.leftBrowserWidget.checkBtnClicked())
        menu.addAction(checkAction)

        saveAction = QAction("Save top-level\tCtrl-S", self)
        saveAction.triggered.connect(lambda: PatchMainWindow.browserWidget.leftBrowserWidget.treeWidget.saveItemPatch(topLevel=True))
        menu.addAction(saveAction)

        diffAction = QAction("Diff\tCtrl-Alt-D", self)
        diffAction.triggered.connect(self.diff)
        diffAction.setEnabled(True if self.patch and os.path.exists(self.patch.getFileName()) else False)
        menu.addAction(diffAction)

        menu.addSeparator()

        helpAction = QAction("Help\tF1", self)
        helpAction.triggered.connect(self.showHelp)
        menu.addAction(helpAction)

        menu.popup(event.globalPos())

    def paintEvent(self, event):
        if not isinstance(event, QPaintEvent): # sometimes event is QHideEvent (!)
            rect = QRect(0, 0, self.width(), self.height())
            event = QPaintEvent(rect)

        super(CppEditorWidget, self).paintEvent(event)

        if self.completionWidget.isVisible():
            self.showCompletions([])

        if not self.isReadOnly() and self.isEnabled():
            cursor = self.textCursor()
            rect = self.cursorRect(cursor)
            painter = QPainter(self.viewport())
            rect.setWidth(self.cursorWidth())
            painter.fillRect(rect, self.cursorColor)

            if not cursor.atEnd() and not cursor.selectedText():
                cursor.movePosition(QTextCursor.Right, QTextCursor.KeepAnchor)
                letter = cursor.selectedText()
                pen = QPen(QColor(0, 0, 0))
                painter.setPen(pen)
                painter.drawText(QRectF(rect), letter)

    def wheelEvent(self, event):
        shift = event.modifiers() & Qt.ShiftModifier
        ctrl = event.modifiers() & Qt.ControlModifier
        alt = event.modifiers() & Qt.AltModifier

        if ctrl:
            d = event.delta() / abs(event.delta())
            self.currentFontPointSize = clamp(8, 20, self.currentFontPointSize + d)

            self.setStyleSheet("font-size: %dpx;"%self.currentFontPointSize)

        else:
            QTextEdit.wheelEvent(self, event)

    def setTextSafe(self, text, withUndo=True):
        scrollBar = self.verticalScrollBar()
        self.blockSignals(True)
        scrollBar.blockSignals(True)

        scroll = scrollBar.value()
        cursor = self.textCursor()
        pos = cursor.position()

        if withUndo:
            cursor.select(QTextCursor.Document)
            cursor.beginEditBlock()
            cursor.removeSelectedText()
            cursor.insertText(text)
            cursor.endEditBlock()
        else:
            self.setText(text)

        self.patch.__setattr__(self.codeType, text)

        if pos < len(text):
            cursor.setPosition(pos)
            self.setTextCursor(cursor)

        scrollBar.setValue(scroll)

        self.blockSignals(False)
        scrollBar.blockSignals(False)

    def completionFinishedCallback(self):
        self.clangCompletion = self.thread.completions
        self.currentWord = wordAtCursor(self.textCursor()) # update current word

        if not self.canShowCompletions:
            return

        if self.currentWord[0]:
            currentWord = self.currentWord[0]
            items = [w for w in self.clangCompletion if re.match(currentWord, w)]

            if items:
                self.showCompletions(items)
        else:
            self.showCompletions(self.clangCompletion)

    def keyPressEvent(self, event):
        shift = event.modifiers() & Qt.ShiftModifier
        ctrl = event.modifiers() & Qt.ControlModifier
        alt = event.modifiers() & Qt.AltModifier
        key = event.key()

        if alt and shift and key == Qt.Key_F: # format cpp
            self.setTextSafe(formatCpp(unicode(self.toPlainText())))

        elif ctrl and key == Qt.Key_B: # rebuild
            PatchMainWindow.browserWidget.leftBrowserWidget.buildBtnClicked()

        elif ctrl and key == Qt.Key_S: # save
            PatchMainWindow.browserWidget.leftBrowserWidget.treeWidget.saveItemPatch(topLevel=True)

        elif ctrl and key == Qt.Key_P: # publish
            PatchMainWindow.browserWidget.leftBrowserWidget.treeWidget.publishItem()

        elif ctrl and key == Qt.Key_C: # check
            cursor = self.textCursor()
            if cursor.selectionStart() == cursor.selectionEnd():
                PatchMainWindow.browserWidget.leftBrowserWidget.checkBtnClicked()
            else:
                self.copy()

        elif key == Qt.Key_F1: # help
            self.showHelp()

        elif ctrl and key == Qt.Key_G: # cancel (emacs)
            self.completionWidget.hide()
            cursor = self.textCursor()
            cursor.clearSelection()
            self.setTextCursor(cursor)

        elif alt and key == Qt.Key_M: # back to indentation
            cursor = self.textCursor()
            linePos = cursor.block().position()
            cursor.select(QTextCursor.LineUnderCursor)
            text = cursor.selectedText()
            cursor.clearSelection()

            found = re.findall("^\s*", unicode(text))
            offset = len(found[0]) if found else 0

            cursor.setPosition(linePos + offset)

            self.setTextCursor(cursor)

        elif ctrl and key == Qt.Key_H: # highlight selected
            self.highlightSelected()

        elif ctrl and key == Qt.Key_F: # search
            self.setFuzzySearch()

        elif ctrl and alt and key == Qt.Key_Space:
            cursor = self.textCursor()
            pos = cursor.position()
            _, start, end = findBracketSpans(unicode(self.toPlainText()), pos)
            if start != end:
                cursor.setPosition(start+1)
                cursor.setPosition(end, QTextCursor.KeepAnchor)
                self.setTextCursor(cursor)

        elif key == Qt.Key_F12: # full screen editor mode
            PatchMainWindow.browserWidget.leftBrowserWidget.setVisible(not PatchMainWindow.browserWidget.leftBrowserWidget.isVisible())
            PatchMainWindow.browserWidget.rightBrowserWidget.attributesWidget.setVisible(not PatchMainWindow.browserWidget.rightBrowserWidget.attributesWidget.isVisible())

        elif alt and key == Qt.Key_F2: # set bookmark
            self.setBookmark()

        elif key == Qt.Key_F2: # next bookmark
            n = self.textCursor().block().blockNumber()
            self.gotoNextBookmark()
            if self.textCursor().block().blockNumber() == n:
                self.gotoNextBookmark(0)

        elif key == Qt.Key_F5: # search everywhere
            self.searchEverywhere(not ctrl)

        elif key == Qt.Key_F3: # emacs swoop
            self.swoopSearch(not ctrl)

        elif alt and key == Qt.Key_G: # goto line
            self.gotoLine()

        elif key == Qt.Key_Escape:
            self.completionWidget.hide()
            PatchMainWindow.browserWidget.rightBrowserWidget.logWidget.makeVisible(False)
            self.fuzzySearch = False

        elif ctrl and key == Qt.Key_Space:
            self.clangAutoCompleteAtPoint(finishedCallback=self.completionFinishedCallback)

        elif key == Qt.Key_Period: # dot
            QTextEdit.keyPressEvent(self, event)
            #self.clangAutoCompleteAtPoint(finishedCallback=self.completionFinishedCallback)

        elif key == Qt.Key_Return:
            if self.completionWidget.isVisible():
                self.replaceWithAutoCompletion()

                self.completionWidget.hide()
            else:
                cursor = self.textCursor()
                block = unicode(cursor.block().text())
                spc = re.search("^(\\s*)", block).groups("")[0]

                QTextEdit.keyPressEvent(self, event)

                if spc:
                    cursor.insertText(spc)
                    self.setTextCursor(cursor)

        elif alt and key == Qt.Key_Up: # move line up
            self.moveLineUp()

        elif alt and key == Qt.Key_Down: # move line down
            self.moveLineDown()

        elif key in [Qt.Key_Up, Qt.Key_Down, Qt.Key_PageDown, Qt.Key_PageUp]:
            if self.completionWidget.isVisible():
                if key == Qt.Key_Down:
                    d = 1
                elif key == Qt.Key_Up:
                    d = -1
                elif key == Qt.Key_PageDown:
                    d = 10
                elif key == Qt.Key_PageUp:
                    d = -10

                line = self.completionWidget.currentLine()
                highlightLine(self.completionWidget, line, clear=True)
                highlightLine(self.completionWidget, clamp(0, self.completionWidget.lineCount()-1, line+d))
            else:
                QTextEdit.keyPressEvent(self, event)

        elif alt and ctrl and key == Qt.Key_D: # diff
            self.diff()

        elif ctrl and key == Qt.Key_L: # center line
            self.centerLine()

        elif ctrl and key == Qt.Key_K: # kill line
            self.killLine()

        elif ctrl and key == Qt.Key_O: # remove redundant lines
            cursor = self.textCursor()

            cursor.beginEditBlock()
            if not unicode(cursor.block().text()).strip():
                cursor.movePosition(QTextCursor.StartOfBlock)
                cursor.movePosition(QTextCursor.NextBlock, QTextCursor.KeepAnchor)
                cursor.removeSelectedText()
                cursor.movePosition(QTextCursor.Up)

            while not unicode(cursor.block().text()).strip() and not cursor.atStart(): # remove empty lines but last one
                if unicode(cursor.block().previous().text()):
                    break

                cursor.movePosition(QTextCursor.StartOfBlock)
                cursor.movePosition(QTextCursor.NextBlock, QTextCursor.KeepAnchor)
                cursor.removeSelectedText()
                cursor.movePosition(QTextCursor.Up)

            cursor.endEditBlock()
            self.setTextCursor(cursor)

        elif ctrl and key in [Qt.Key_BracketLeft, Qt.Key_BracketRight]:
            cursor = self.textCursor()
            pos = cursor.position()
            _, start, end = findBracketSpans(unicode(self.toPlainText()), pos)
            if start != end:
                cursor.setPosition(start if key == Qt.Key_BracketLeft else end)
                self.setTextCursor(cursor)

        elif ctrl and key == Qt.Key_I: # import include
            if self.codeType == "clCode":
                cursor = self.textCursor()
                line = cursor.block().text()
                r = re.search("^\\#include\\s+(\\w+)\\s*", line)
                if r:
                    includePath = EvaluatorServerPath + "/include/CL"
                    fname = r.groups("")[0]
                    fpath = includePath + "/" + fname + ".cl"
                    if not os.path.exists(fpath):
                        QMessageBox.critical(self, "Patch", "Cannot find '/include/CL/%s.cl'"%fname)
                    else:
                        with open(fpath, "r") as f:
                            text = f.read()

                            cursor.beginEditBlock()
                            cursor.movePosition(QTextCursor.StartOfBlock)
                            cursor.movePosition(QTextCursor.EndOfBlock, QTextCursor.KeepAnchor)
                            cursor.removeSelectedText()

                            cursor.insertText(text)
                            cursor.endEditBlock()
                            self.setTextCursor(cursor)

        elif ctrl and key == Qt.Key_D: # duplicate line
            cursor = self.textCursor()
            line = cursor.block().text()
            cursor.movePosition(QTextCursor.EndOfBlock)
            cursor.beginEditBlock()
            cursor.insertBlock()
            cursor.insertText(line)
            cursor.endEditBlock()
            self.setTextCursor(cursor)

        elif ctrl and key == Qt.Key_Semicolon: # comment
            cursor = self.textCursor()

            if cursor.selectedText():
                self.toggleCommentBlock()
            else:
                self.toggleCommentLine()
        else:
            QTextEdit.keyPressEvent(self, event)

    def swoopSearch(self, update=True):
        if update:
            self.swoopSearchDialog.update()

        self.swoopSearchDialog.exec_()

    def searchEverywhere(self, update=True):
        w = PatchMainWindow.browserWidget.leftBrowserWidget.treeWidget.searchEverywhereWidget
        if update:
            w.update()

        w.show()
        w.activateWindow()

    def moveLineUp(self):
        cursor = self.textCursor()
        if not cursor.block().previous().isValid() or cursor.selectedText():
            return

        text = cursor.block().text()
        pos = cursor.positionInBlock()

        cursor.beginEditBlock()
        cursor.movePosition(QTextCursor.StartOfBlock)
        cursor.movePosition(QTextCursor.EndOfBlock, QTextCursor.KeepAnchor)
        cursor.removeSelectedText()
        cursor.deletePreviousChar()
        cursor.movePosition(QTextCursor.StartOfBlock)
        cursor.insertText(text)
        cursor.insertBlock()
        cursor.endEditBlock()

        cursor.movePosition(QTextCursor.Up)
        cursor.movePosition(QTextCursor.StartOfBlock)
        cursor.movePosition(QTextCursor.Right, n=pos)

        self.setTextCursor(cursor)

    def moveLineDown(self):
        cursor = self.textCursor()
        if not cursor.block().next().isValid() or cursor.selectedText():
            return

        text = cursor.block().text()
        pos = cursor.positionInBlock()

        cursor.beginEditBlock()
        cursor.movePosition(QTextCursor.StartOfBlock)
        cursor.movePosition(QTextCursor.EndOfBlock, QTextCursor.KeepAnchor)
        cursor.removeSelectedText()
        cursor.deleteChar()
        cursor.movePosition(QTextCursor.EndOfBlock)
        cursor.insertBlock()
        cursor.insertText(text)
        cursor.endEditBlock()

        cursor.movePosition(QTextCursor.StartOfBlock)
        cursor.movePosition(QTextCursor.Right, n=pos)

        self.setTextCursor(cursor)

    def centerLine(self):
        cursorY = self.cursorRect().top()
        scrollBar = self.verticalScrollBar()
        scrollBar.setValue(scrollBar.value() + cursorY - self.geometry().height()/2)

    def killLine(self):
        cursor = self.textCursor()

        if not cursor.block().text():
            cursor.movePosition(QTextCursor.StartOfBlock)
            cursor.movePosition(QTextCursor.NextBlock, QTextCursor.KeepAnchor)
        else:
            cursor.movePosition(QTextCursor.EndOfBlock, QTextCursor.KeepAnchor)

        cursor.removeSelectedText()
        self.setTextCursor(cursor)

    def setFuzzySearch(self, state=None):
        self.fuzzySearch = not self.fuzzySearch if state == None else state

        self.cursorColor = QColor(150, 50, 50) if not self.fuzzySearch else QColor(150, 150, 50)
        self.update()

    def showHelp(self):
		os.system(EvaluatorDocPath)

    def toggleCommentBlock(self):
        cursor = self.textCursor()
        selText = cursor.selectedText()
        if not selText:
            return

        cursor.beginEditBlock()

        if re.match("^/\\*",selText):
            selText = selText.replace("/*","").replace("*/","")
        else:
            selText = "/*%s*/"%selText

        cursor.removeSelectedText()
        cursor.insertText(selText)
        cursor.endEditBlock()

        self.setTextCursor(cursor)

    def toggleCommentLine(self):
        comment = "// "
        commentSize = len(comment)

        cursor = self.textCursor()

        pos = cursor.position()
        linePos = cursor.block().position()
        cursor.select(QTextCursor.LineUnderCursor)
        lineText = cursor.selectedText()
        cursor.clearSelection()

        found = re.findall("^\s*", unicode(lineText))
        offset = len(found[0]) if found else 0

        cursor.setPosition(linePos + offset)

        newPos = pos + commentSize

        cursor.beginEditBlock()
        if not re.match("^\s*%s"%comment, lineText):
            cursor.insertText(comment)
        else:
            for i in range(len(comment)):
                cursor.deleteChar()
            newPos = pos - commentSize

        cursor.endEditBlock()

        cursor.setPosition(newPos)

        self.setTextCursor(cursor)

    def diff(self):
        if not self.patch:
            return

        p = self.patch

        fileName = p.getFileName()

        if fileName and os.path.exists(fileName):
            codes = Patch.getPatchTags(fileName, [self.codeType])

            diffData = diffText(codes[self.codeType], unicode(self.toPlainText()))

            coloredText = ""
            for line in diffData:
                if line.startswith("?"):
                    continue

                newLine = escape(line).replace(" ", "&nbsp;")

                if line.startswith("+"):
                    coloredText += "<font color=#55aa55>%s</font><br>"%newLine
                elif line.startswith("-"):
                    coloredText += "<font color=#aa5555>%s</font><br>"%newLine
                else:
                    coloredText += newLine + "<br>"

            self._textDialog = TextDialog(coloredText)
            self._textDialog.setWindowTitle("Diff with %s"%os.path.basename(fileName))
            self._textDialog.show()

    def gotoLine(self, line=-1):
        if line == -1:
            cursor = self.textCursor()
            currentLine = cursor.blockNumber()+1
            maxLine = self.document().lineCount()
            line, ok = QInputDialog.getInt(self, "Patch", "Goto line number", currentLine, 1, maxLine)
            if not ok:
                return

        self.setTextCursor(QTextCursor(self.document().findBlockByLineNumber(line-1)))

    def clangAutoCompleteAtPoint(self, finishedCallback=None, textCursor=None):
        if not self.thread or not self.thread.isRunning():
            self.thread = CompletionThread(self, finishedCallback, textCursor, parent=self)
            self.thread.start()
            self.canShowCompletions = True

    def replaceWithAutoCompletion(self):
        if self.completionWidget.lineCount() == 0:
            return

        modifiers = QApplication.queryKeyboardModifiers()
        shift = modifiers & Qt.ShiftModifier
        ctrl = modifiers & Qt.ControlModifier
        alt = modifiers & Qt.AltModifier

        block = self.completionWidget.textCursor().block()
        row = block.blockNumber() if block.isValid() else 0

        if ctrl:
            word = unicode(block.text())
        else:
            word = re.split("\\s*", unicode(block.text()))[0]

        cursor = self.textCursor()
        cursor.setPosition(self.currentWord[1])
        cursor.setPosition(self.currentWord[2], QTextCursor.KeepAnchor)
        cursor.removeSelectedText()
        cursor.insertText(word)
        self.setTextCursor(cursor)
        self.canShowCompletions = False

    def highlightSelected(self):
        cursor = self.textCursor()
        sel = cursor.selectedText()

        reg = None
        if sel:
            reg = QRegExp("%s"%QRegExp.escape(sel))
        else:
            word, _,_ = wordAtCursor(cursor)
            if word:
                if word.startswith("@"):
                    reg = QRegExp("@\\b%s\\b"%QRegExp.escape(word[1:]))
                else:
                    reg = QRegExp("\\b%s\\b"%QRegExp.escape(word))

        self.syntax.highlightedWordRegexp = reg

        self.blockSignals(True)
        self.syntax.rehighlight()
        self.blockSignals(False)

    def editorCursorPositionChanged(self):
        cursor = self.textCursor()
        pos = cursor.position()

        if abs(pos - self.prevCursorPosition) > 1:
            self.completionWidget.hide()

        if cursor.selectedText():
            self.setExtraSelections([])
            return

        self.saveState(cursor=True, scroll=False, bookmarks=False)

        self.prevCursorPosition = pos

        text, start, end = findBracketSpans(unicode(self.toPlainText()), pos)

        extra = []

        if start != end:
            for pos in [start, end]:
                cursor = self.textCursor()
                cursor.setPosition(pos)
                cursor.setPosition(pos+1, QTextCursor.KeepAnchor)
                es = QTextEdit.ExtraSelection()
                es.cursor = cursor
                es.format.setForeground(QColor(0, 0, 0))
                es.format.setBackground(QBrush(QColor(70, 130, 140)))
                extra.append(es)

        self.setExtraSelections(extra)

    def editorTextChanged(self):
        if not self.completionWidget.isVisible():
            self.setFuzzySearch(False)

        if not self.patch:
            return

        text = unicode(self.toPlainText())

        self.patch.__setattr__(self.codeType, text)

        cursor = self.textCursor()
        pos = cursor.position()

        self.currentWord = wordAtCursor(cursor)
        currentWord, start, end = self.currentWord

        if start == 0 and end - start <= 1:
            return

        self.words = set(self.clangCompletion)
        self.words |= set(re.split("[^\\w@]+", text))

        self.words |= self.wordsPerCodeType[self.codeType]

        # parse attributes
        if self.codeType == "clCode":
            self.words |= set(["@"+a.name for a in self.patch.attributes if a.cl != Attribute.CL_none])

        elif self.codeType != "drawCode":
            self.words |= set(["@"+a.name for a in self.patch.attributes])

        self.words -= set([currentWord])

        if currentWord:
            if self.fuzzySearch:
                ssw = self.searchStartWord[0]
                items = [w for w in self.words if re.search("^%s.*%s"%(ssw, currentWord.replace(ssw,"")), w, re.IGNORECASE)]
            else:
                self.searchStartWord = self.currentWord
                items = [w for w in self.words if re.match(currentWord, w, re.IGNORECASE)]

            if items and cursor.position() == end:
                self.showCompletions(items)
            else:
                self.fuzzySearch = False
                self.completionWidget.hide()

        else:
            self.fuzzySearch = False
            self.completionWidget.hide()

    def showCompletions(self, items):
        if self.fuzzySearch:
            self.completionWidget.setStyleSheet("QListView::item:selected { background: #7d7d28; color: #FFFFFF; }")
        else:
            self.completionWidget.setStyleSheet("")

        rect = self.cursorRect()
        c = rect.center()

        self.completionWidget.setGeometry(c.x(), c.y()+10, 200, 200)
        if items:
            self.completionWidget.update(items)

        self.completionWidget.show()

def findBracketSpans(text, pos, brackets="([{"):
        if not text:
            return ("", 0, 0)

        textLen = len(text)

        # when no spaces at the current line then do nothing
        start = pos-1
        while start > 0 and text[start] != "\n":
            start -= 1

        if not re.search("^\\s+|[{\(\[]+", text[start+1:pos]):
            return ("", 0, 0)

        start = pos-1
        end = pos

        bracketDict = {"(":0, "[": 0, "{": 0}

        bracketChar = ""
        ok = False
        while True:
            if (bracketDict["("] < 0 and "(" in brackets) or\
               (bracketDict["["] < 0 and "[" in brackets) or\
               (bracketDict["{"] < 0 and "{" in brackets):
                ok = True
                break

            if start < 0:
                break

            ch = text[start]
            if ch in ["(", ")", "{", "}", "[", "]"]:
                bracketChar = str(ch)
                if ch == ")": bracketDict["("] += 1
                elif ch == "(": bracketDict["("] -= 1

                elif ch == "]": bracketDict["["] += 1
                elif ch == "[": bracketDict["["] -= 1

                elif ch == "}": bracketDict["{"] += 1
                elif ch == "{": bracketDict["{"] -= 1

            start -= 1

        start += 1

        if ok:
            bracketDict = {"(":0, "[": 0, "{": 0}
            ok = False
            while True:
                if bracketDict[bracketChar] < 0:
                    ok = True
                    break

                if end >= textLen:
                    break

                ch = text[end]

                if ch in ["(", ")", "{", "}", "[", "]"]:
                    if ch == "(": bracketDict["("] += 1
                    elif ch == ")": bracketDict["("] -= 1

                    if ch == "[": bracketDict["["] += 1
                    elif ch == "]": bracketDict["["] -= 1

                    if ch == "{": bracketDict["{"] += 1
                    elif ch == "}": bracketDict["{"] -= 1

                end += 1

            end -= 1

            if ok:
                return (text[start:end], start, end)

        return ("", 0, 0)

def wordAtCursor(cursor):
    cursor = QTextCursor(cursor)
    pos = cursor.position()

    lpart = ""
    start = pos-1
    ch = unicode(cursor.document().characterAt(start))
    while ch and re.match("[@\\w]", ch):
        lpart += ch
        start -= 1

        if ch == "@": # @ can be the first character only
            break

        ch = unicode(cursor.document().characterAt(start))

    rpart = ""
    end = pos
    ch = unicode(cursor.document().characterAt(end))
    while ch and re.match("[\\w]", ch):
        rpart += ch
        end += 1
        ch = unicode(cursor.document().characterAt(end))

    return (lpart[::-1]+rpart, start+1, end)

class CompletionWidget(QTextEdit):
    def __init__(self, items, **kwargs):
        super(CompletionWidget, self).__init__(**kwargs)

        self.setWindowFlags(Qt.FramelessWindowHint)
        self.setAttribute(Qt.WA_ShowWithoutActivating)

        self.setReadOnly(True)
        self.setWordWrapMode(QTextOption.NoWrap)
        self.syntax = CppHighlighter(self)

        self.update([])

    def lineCount(self):
        return self.document().blockCount()

    def currentLine(self):
        return self.textCursor().block().blockNumber()

    def mousePressEvent(self, event):
        self.parent().setFocus()
        event.accept()

    def keyPressEvent(self, event):
        shift = event.modifiers() & Qt.ShiftModifier
        ctrl = event.modifiers() & Qt.ControlModifier
        alt = event.modifiers() & Qt.AltModifier

        line = self.textCursor().block().blockNumber()
        lineCount = self.document().blockCount()-1

        if event.key() == Qt.Key_Down:
            highlightLine(self, clamp(0, lineCount, line), clear=True)
            highlightLine(self, clamp(0, lineCount, line+1))
        elif event.key() == Qt.Key_Up:
            highlightLine(self, clamp(0, lineCount, line), clear=True)
            highlightLine(self, clamp(0, lineCount, line-1))

        elif event.key() == Qt.Key_PageDown:
            highlightLine(self, clamp(0, lineCount, line), clear=True)
            highlightLine(self, clamp(0, lineCount, line+5))
        elif event.key() == Qt.Key_PageUp:
            highlightLine(self, clamp(0, lineCount, line), clear=True)
            highlightLine(self, clamp(0, lineCount, line-5))

        elif event.key() == Qt.Key_Return: # accept
            pass

        else:
            QTextEdit.keyPressEvent(self, event)

    def update(self, items):
        if not items:
            return

        self.clear()
        self.setCurrentCharFormat(QTextCharFormat())

        lines = []
        for line in items:
            lines.append(line)

        self.setText("\n".join(lines))
        highlightLine(self, 0)

        self.autoResize()

    def autoResize(self):
        w = self.document().idealWidth() + 10
        h = self.document().blockCount()*self.cursorRect().height() + 30

        maxHeight = clamp(0, 400, self.parent().height() - self.parent().cursorRect().top() - 30)

        self.setFixedSize(clamp(0, 500, w), clamp(0, maxHeight, h))

    def showEvent(self, event):
        self.autoResize()

class NumberBarWidget(QWidget):
    def __init__(self, edit, *kwargs):
        super(NumberBarWidget, self).__init__(*kwargs)
        self.edit = edit
        self.highest_line = 0

    def update(self, *args):
        self.setStyleSheet(self.edit.styleSheet())

        width = self.fontMetrics().width(str(self.highest_line)) + 19
        self.setFixedWidth(width)

        QWidget.update(self, *args)

    def paintEvent(self, event):
        contents_y = self.edit.verticalScrollBar().value()
        page_bottom = contents_y + self.edit.viewport().height()
        font_metrics = self.fontMetrics()
        current_block = self.edit.document().findBlock(self.edit.textCursor().position())

        painter = QPainter(self)

        line_count = 0
        # Iterate over all text blocks in the document.
        block = self.edit.document().begin()
        while block.isValid():
            line_count += 1

            # The top left position of the block in the document
            position = self.edit.document().documentLayout().blockBoundingRect(block).topLeft()

            # Check if the position of the block is out side of the visible
            # area.
            if position.y() > page_bottom:
                break

            # Draw the line number right justified at the y position of the
            # line. 3 is a magic padding number. drawText(x, y, text).
            painter.drawText(self.width() - font_metrics.width(str(line_count)) - 3, round(position.y()) - contents_y + font_metrics.ascent(), str(line_count))
            data = block.userData()
            if data and data.hasBookmark:
                painter.drawText(3, round(position.y()) - contents_y + font_metrics.ascent(), u"►")

            block = block.next()

        self.highest_line = self.edit.document().blockCount()
        painter.end()

        QWidget.paintEvent(self, event)

class TextBlockData(QTextBlockUserData):
    def __init__(self):
        super(TextBlockData, self).__init__()
        self.hasBookmark = False

class CppEditorWithNumbersWidget(QWidget):
    def __init__(self, patch, codeType, **kwargs):
        super(CppEditorWithNumbersWidget, self).__init__(**kwargs)

        self.editorWidget = CppEditorWidget(patch, codeType)

        self.numberBarWidget = NumberBarWidget(self.editorWidget)
        self.editorWidget.document().blockCountChanged.connect(lambda _: self.numberBarWidget.update())
        self.editorWidget.document().documentLayoutChanged.connect(self.numberBarWidget.update)
        self.editorWidget.verticalScrollBar().valueChanged.connect(lambda _: self.numberBarWidget.update())

        hlayout = QHBoxLayout()
        hlayout.addWidget(self.numberBarWidget)
        hlayout.addWidget(self.editorWidget)

        self.setLayout(hlayout)

class TabBar(QTabBar):
    def __init__(self, parent):
        QTabBar.__init__(self, parent)
        self.tabColors = {}

    def paintEvent(self, event):
        painter = QPainter(self)
        for index in range(self.count()):
            option = QStyleOptionTab()
            self.initStyleOption(option, index)

            color = self.tabColors.get(index, QColor(100,100,100))
            text = option.text
            rect = option.rect

            if index == self.currentIndex():
                color = QColor(250, 250, 250)
                text = "[" + option.text[1:-1] + "]"
                option.text = text

            painter.setPen(color)
            painter.drawText(rect, 1, text)

class PatchCppEditorWidget(QWidget):
    def __init__(self, patch, **kwargs):
        super(PatchCppEditorWidget, self).__init__(**kwargs)

        self.patch = patch
        self.codeTypes = ["definesCode",
                          "runCode",
                          "deformInitCode",
                          "deformCode",
                          "deformParallelCode",
                          "ispcCode",
                          "clCode",
                          "drawCode",
                          "commandsCode"]

        self.tabByPatch = {}

        layout = QVBoxLayout()
        layout.setMargin(7)
        self.setLayout(layout)

        self.tabWidget = QTabWidget()
        self.tabWidget.setTabBar(TabBar(self.tabWidget))

        for ct in self.codeTypes:
            self.tabWidget.addTab(CppEditorWithNumbersWidget(self.patch, ct), " %s "%codeTypeToLabel(ct))

        layout.addWidget(self.tabWidget)

        self.update(self.patch)

    def update(self, newPatch):
        if not newPatch:
            self.tabWidget.setEnabled(False)
            return

        self.tabWidget.setEnabled(True)

        currentIndex = self.tabWidget.currentIndex()
        self.tabByPatch[self.patch] = currentIndex
        idx = self.tabByPatch.get(newPatch, currentIndex)
        self.tabWidget.setCurrentIndex(idx)

        self.patch = newPatch
        
        codeActual = self.patch.isCodeActual()
        for i, ct in enumerate(self.codeTypes):
            mod = "*" if not codeActual.get(ct, True) else ""
            self.tabWidget.setTabText(i, " %s "%(codeTypeToLabel(ct)+mod))

            ed = self.tabWidget.widget(i).editorWidget
            ed.patch = self.patch
            text = self.patch.__getattribute__(ct)
            self.tabWidget.tabBar().tabColors[i] = QColor(90, 90, 90) if not text else QColor(190, 190, 190)
            ed.setTextSafe(text)
            ed.document().clearUndoRedoStacks()

            ed.loadState()

class TextDialog(QFrame):
    def __init__(self, text, **kwargs):
        super(TextDialog, self).__init__(**kwargs)

        self.setWindowTitle("Text Viewer")

        # self.setWindowFlags(self.windowFlags() | Qt.WindowSystemMenuHint | Qt.WindowMinMaxButtonsHint)

        layout = QVBoxLayout()
        self.setLayout(layout)

        self.setGeometry(400, 300, 600, 300)

        self.textWidget = QTextEdit()
        self.textWidget.setTabStopWidth(16)
        self.textWidget.setAcceptRichText(False)
        self.textWidget.setWordWrapMode(QTextOption.NoWrap)
        self.textWidget.setText(text)
        self.textWidget.setReadOnly(True)

        font = self.textWidget.font()
        self.textWidget.setFont(font)

        layout.addWidget(self.textWidget)

class PatchListDialog(QDialog):
    def __init__(self, **kwargs):
        super(PatchListDialog, self).__init__(**kwargs)

        self.selectedFileName = ""
        self.patchList = []

        self.setWindowTitle("Patch Selector")

        layout = QVBoxLayout()
        self.setLayout(layout)

        self.directoryWidget = QComboBox()
        self.directoryWidget.addItem(EvaluatorServerPath)

        if EvaluatorServerPath != EvaluatorLocalPath:
            self.directoryWidget.addItem(EvaluatorLocalPath)

        self.directoryWidget.currentIndexChanged.connect(self.update)

        self.maskWidget = QLineEdit()
        self.maskWidget.textChanged.connect(self.update)

        self.treeWidget = QTreeWidget()
        self.treeWidget.setHeaderLabels(["Name", "Version", "Modification Time","Description"])
        self.treeWidget.keyPressEvent = self.treeKeyPressEvent
        self.treeWidget.itemActivated.connect(self.treeItemActivated)
        self.treeWidget.header().resizeSection(0, 300)
        self.treeWidget.header().resizeSection(1, 200)
        self.treeWidget.setSortingEnabled(True)
        self.treeWidget.sortItems(2, Qt.DescendingOrder) # mod time

        self.summaryWidget = SummaryWidget()
        self.summaryWidget.setReadOnly(True)

        layout.addWidget(self.directoryWidget)
        layout.addWidget(self.maskWidget)
        layout.addWidget(self.treeWidget)

        self.maskWidget.setFocus()

    def updatePatchList(self):
        directory = unicode(self.directoryWidget.currentText())
        self.patchList = Patch.listPatches(directory+"/patch/patches")

    def showEvent(self, event):
        self.updatePatchList()

        pos = QCursor.pos()
        self.setGeometry(pos.x(), pos.y(), 900, 400)

        self.selectedFileName = ""
        self.update()

    def treeItemActivated(self, item):
        if item.childCount() == 0:
            directory = unicode(self.directoryWidget.currentText())
            parentDir = item.parent().text(0) if item.parent() else ""

            fileName = unicode(item.text(0))
            self.selectedFileName = "{dir}/patch/patches/{parent}/{fileName}.xml".format(dir=directory, parent=parentDir,fileName=fileName)
            self.done(0)

    def treeKeyPressEvent(self, event):
        if not self.treeWidget.currentItem():
            super(QTreeWidget, self.treeWidget).keyPressEvent(event)
            return

        item = self.treeWidget.currentItem()

        directory = unicode(self.directoryWidget.currentText())
        directoryIndex = self.directoryWidget.currentIndex()

        localName = unicode(item.text(0))
        localFolder = item.parent().text(0) if item.parent() else ""
        fileName = "{dir}/patch/patches/{localFolder}/{localName}".format(dir=directory, localFolder=localFolder,localName=localName)

        if event.key() == Qt.Key_O: # open file
            file = fileName + (".xml" if item.childCount() == 0 else "")
            os.system("explorer /select,%s"%os.path.realpath(file))

        elif event.key() == Qt.Key_Delete: # remove file
            if directoryIndex == 0 and self.directoryWidget.count() > 1: # you cannot remove server files
                return

            ok = False
            if item.childCount() == 0:
                ok = QMessageBox.question(self, "Patch", "Really remove '%s' patch?"%localName, QMessageBox.Yes and QMessageBox.No, QMessageBox.Yes) == QMessageBox.Yes
                if ok:
                    os.remove(fileName + ".xml")
            else:
                if localName in ["__backup__"]:
                    return

                ok = QMessageBox.question(self, "Patch", "Really remove '%s' type?"%localName, QMessageBox.Yes and QMessageBox.No, QMessageBox.Yes) == QMessageBox.Yes
                if ok:
                    for f in glob.glob(fileName+"/*.xml"):
                        os.remove(f)

                    os.rmdir(fileName)

            if ok:
                self.updatePatchList()
                self.update()
                self.treeWidget.setFocus()

        elif event.key() == Qt.Key_R: # rename
            if directoryIndex == 0 and self.directoryWidget.count() > 1: # you cannot rename server files
                return

            if item.childCount() == 0:
                localNameNoExt = os.path.basename(fileName).replace(".xml","")
                newName, ok = QInputDialog.getText(self, "Rename file", "New name", QLineEdit.Normal, localNameNoExt)
                if ok:
                    newName = unicode(newName)

                    if not re.match("[a-zA-Z0-9_]+", newName):
                        QMessageBox.critical(self, "Patch", "Invalid name")
                    else:
                        newName = "{dir}/patch/patches/{name}.xml".format(dir=directory, name=newName)
                        if not os.path.exists(newName):
                            os.rename(fileName+".xml", newName)
                            self.patchList = Patch.patchList(directory+"/patch/patches")
                            self.update()
                        else:
                            QMessageBox.critical(self, "Patch", "This name is busy")

            else: # group renaming
                newType, ok = QInputDialog.getText(self, "Rename patch type", "New type", QLineEdit.Normal, localName)
                if ok and newType!=localName:
                    newType = unicode(newType)

                    path = os.path.realpath(directory + "/patch/patches/" + localName)
                    for f in glob.glob(path+"/*.xml"):
                        newLocalName = os.path.basename(f).replace(localName, newType)

                        p = Patch.loadFromFile(f)
                        p.type = newType
                        p.publish(saveOnly=True)

                    shutil.rmtree(path) # remove old files

                    self.patchList = Patch.patchList(directory+"/patch/patches")
                    self.update()

        elif event.key() == Qt.Key_Space: # expand
            item.setExpanded(not item.isExpanded())
        else:
            super(QTreeWidget, self.treeWidget).keyPressEvent(event)

    def update(self):
        directory = os.path.realpath(self.directoryWidget.currentText() + "/patch/patches")
        mask = re.escape(unicode(self.maskWidget.text()))

        self.updatePatchList()

        tw = self.treeWidget
        tw.clear()

        lastFolder = ""
        currentDirItem = ""
        for f in self.patchList:
            if not os.path.exists(f):
                continue

            f = os.path.realpath(f)
            localName = os.path.basename(f).replace(".xml", "")
            localPath = f.replace(os.path.commonprefix([directory, f])+"\\","")

            if re.search(mask, localName, re.IGNORECASE):
                tagsData = Patch.getPatchTags(f, ["description"])
                description = re.sub("[\n\r]*","", tagsData.get("description",""))
                description = description if len(description) < 30 else (description[:30]+"...")

                modtime = time.strftime("%Y/%m/%d %H:%M", time.localtime(os.path.getmtime(f)))

                version = Patch.getVersionByFileName(f)
                item = QTreeWidgetItem([localName, str(version or ""), modtime, description])

                font = item.font(2)
                font.setItalic(True)
                item.setFont(2, font)
                item.setToolTip(3, tagsData.get("description",""))

                folder = os.path.dirname(f)
                if "\\" in localPath:
                    if folder == lastFolder:
                        if os.path.basename(folder) not in ["__backup__"] and Patch.getVersionByFileName(f) != Patch.getLatestVersionByFileName(f):
                            for i in range(4):
                                item.setTextColor(i, QColor(100, 100, 100))

                        currentDirItem.addChild(item)
                    else:
                        localFolder = "\\".join(localPath.split("\\")[:-1])
                        currentDirItem = QTreeWidgetItem([localFolder,  "", "", ""])
                        currentDirItem.setForeground(0, QColor(130, 130, 230))
                        currentDirItem.addChild(item)

                        tw.addTopLevelItem(currentDirItem)

                        currentDirItem.setExpanded(True if mask else False)
                else:
                    tw.addTopLevelItem(item)

                lastFolder = folder

class TreeWidget(QTreeWidget):
    def __init__(self, **kwargs):
        super(TreeWidget, self).__init__(**kwargs)

        self.clipboard = []
        self.itemUnderMouse = None

        self.undoQueue = []

        self.patchListDialog = PatchListDialog(parent=PatchMainWindow)
        self.searchEverywhereWidget = SearchEverywhereWidget(parent=PatchMainWindow)

        self.setHeaderLabels(["Patch", "Type", "Version"])

        if "setSectionResizeMode" in dir(self.header()):
            self.header().setSectionResizeMode(QHeaderView.ResizeToContents) # Qt5
        else:
            self.header().setResizeMode(QHeaderView.ResizeToContents) # Qt4

        self.setSelectionMode(QAbstractItemView.ExtendedSelection) # ExtendedSelection

        self.setDragEnabled(True)
        self.setDragDropMode(QAbstractItemView.InternalMove)
        self.setDropIndicatorShown(True)

        self.setContextMenuPolicy(Qt.DefaultContextMenu)

        self.itemClicked.connect(self.treeItemClicked)
        self.setIndentation(30)

        self.setMouseTracking(True)

        self.entered.connect(self.itemEntered)

    def contextMenuEvent(self, event):
        selectedItems = self.selectedItems()
        item = selectedItems[0] if selectedItems else None

        menu = QMenu(self)

        addAction = QAction("Add\tINS", self)
        addAction.triggered.connect(self.insertItem)
        menu.addAction(addAction)

        renameAction = QAction("Set name\tENTER", self)
        renameAction.triggered.connect(self.renameItem)
        renameAction.setEnabled(True if item else False)
        menu.addAction(renameAction)

        renameTypeAction = QAction("Set type\tCtrl-ENTER", self)
        renameTypeAction.triggered.connect(self.renameItemType)
        renameTypeAction.setEnabled(True if item else False)
        menu.addAction(renameTypeAction)

        muteMenu = QMenu("Mute", self)

        muteAction = QAction("Mute\tM", self)
        muteAction.triggered.connect(self.muteItem)
        muteAction.setEnabled(True if item else False)
        muteMenu.addAction(muteAction)

        muteOthersAction = QAction("Mute others\tShift-M", self)
        muteOthersAction.triggered.connect(self.toggleMuteOthers)
        muteOthersAction.setEnabled(True if item else False)
        muteMenu.addAction(muteOthersAction)

        muteAllAction = QAction("Mute all\tCtrl-M", self)
        muteAllAction.triggered.connect(self.muteAll)
        muteAllAction.setEnabled(True if item else False)
        muteMenu.addAction(muteAllAction)

        menu.addMenu(muteMenu)

        menu.addSeparator()

        copyAction = QAction("Copy\tCtrl-C", self)
        copyAction.triggered.connect(self.copyClipboard)
        copyAction.setEnabled(True if item else False)
        menu.addAction(copyAction)

        cutAction = QAction("Cut\tCtrl-X", self)
        cutAction.triggered.connect(self.cutClipboard)
        cutAction.setEnabled(True if item else False)
        menu.addAction(cutAction)

        pasteAction = QAction("Paste\tCtrl-V", self)
        pasteAction.triggered.connect(lambda: self.pasteClipboard(pasteAsChild=True))
        pasteAction.setEnabled(True if self.clipboard else False)
        menu.addAction(pasteAction)

        duplicateAction = QAction("Duplicate\tCtrl-D", self)
        duplicateAction.triggered.connect(lambda: (self.copyClipboard(), self.pasteClipboard(pasteAsChild=False)))
        duplicateAction.setEnabled(True if item else False)
        menu.addAction(duplicateAction)

        menu.addSeparator()

        saveAction = QAction("Save\tCtrl-S", self)
        saveAction.triggered.connect(self.saveItemPatch)
        saveAction.setEnabled(True if item else False)
        menu.addAction(saveAction)

        publishAction = QAction("Publish\tCtrl-P", self)
        publishAction.triggered.connect(self.publishItem)
        publishAction.setEnabled(True if item and item.patch.type else False)
        menu.addAction(publishAction)

        updateMenu = QMenu("Update", self)

        updateAction = QAction("Update\tU", self)
        updateAction.triggered.connect(self.updateItem)
        updateAction.setEnabled(True if item else False)
        updateMenu.addAction(updateAction)

        updateFullAction = QAction("Update (full)\tShift-U", self)
        updateFullAction.triggered.connect(lambda: self.updateItem(attributes=True))
        updateFullAction.setEnabled(True if item else False)
        updateMenu.addAction(updateFullAction)

        updateFromAction = QAction("Update from\tCtrl-U", self)
        updateFromAction.triggered.connect(self.updateItemFromClipboard)
        updateFromAction.setEnabled(True if item and self.clipboard else False)
        updateMenu.addAction(updateFromAction)

        updateFromFullAction = QAction("Update from (full)\tCtrl-Shift-U", self)
        updateFromFullAction.triggered.connect(lambda: self.updateItemFromClipboard(attributes=True))
        updateFromFullAction.setEnabled(True if item and self.clipboard else False)
        updateMenu.addAction(updateFromFullAction)

        revertAction = QAction("Revert\tR", self)
        revertAction.triggered.connect(self.revertItem)
        revertAction.setEnabled(True if item else False)
        updateMenu.addAction(revertAction)

        menu.addMenu(updateMenu)

        menu.addSeparator()

        removeAction = QAction("Remove\tDEL", self)
        removeAction.triggered.connect(self.removeItem)
        removeAction.setEnabled(True if item else False)
        menu.addAction(removeAction)

        removeAllAction = QAction("Remove all\tShift-DEL", self)
        removeAllAction.triggered.connect(self.removeAll)
        menu.addAction(removeAllAction)

        undoAction = QAction("Undo", self)
        undoAction.triggered.connect(self.undoChange)
        undoAction.setEnabled(True if self.undoQueue else False)
        menu.addAction(undoAction)

        menu.addSeparator()

        searchAction = QAction("Search everywhere\tF5", self)
        searchAction.triggered.connect(self.searchEverywhereWidget.show)
        menu.addAction(searchAction)

        menu.popup(event.globalPos())

    def itemEntered(self, modelIdx):
        item = self.itemFromIndex(modelIdx)
        self.itemUnderMouse = item
        self.repaint()

    def leaveEvent(self, event):
        self.itemUnderMouse = None
        self.repaint()

    def dragEnterEvent(self, event):
        if event.mouseButtons() == Qt.MiddleButton:
            QTreeWidget.dragEnterEvent(self, event)
            self.dragItems = self.selectedItems()
            self.dragItemsParent = [it.parent() for it in self.dragItems]

    def dragMoveEvent(self, event):
        QTreeWidget.dragMoveEvent(self, event)

    def dropEvent(self, event):
        self.trackHistory()

        QTreeWidget.dropEvent(self, event)

        insertIdx = self.indexFromItem(self.dragItems[0]).row()

        dropIndex = self.indexAt(event.pos())
        for oldParent, item in zip(self.dragItemsParent, self.dragItems):
            if oldParent:
                id = oldParent.patch.children.index(item.patch)
                oldParent.patch.removeChildAtIndex(id)

            newParent = item.parent()
            if newParent:
                newId = newParent.indexOfChild(item)
                newParent.removeChild(item)
                newParent.insertChild(insertIdx, item)

                newParent.patch.children.insert(newId, item.patch)
            else:
                self.invisibleRootItem().removeChild(item)
                self.insertTopLevelItem(insertIdx, item)

    def drawRow(self, painter, options, modelIdx):
        painter.save()

        rect = self.visualRect(modelIdx)
        item = self.itemFromIndex(modelIdx)

        indent = self.indentation()

        if rect.width() < 0:
            return

        painter.setPen(QPen(QBrush(QColor(60, 60, 60)), 1, Qt.SolidLine))
        numberBranch = rect.x() / indent
        if numberBranch > 1:
            for i in range(1, numberBranch):
                plusInt = i * indent + 10
                x = rect.x() - plusInt
                painter.drawLine(x, rect.y(), x, rect.y() + rect.height())

        parentIndex = modelIdx.parent()
        parentItem = self.itemFromIndex(parentIndex)

        while parentIndex.isValid():
            parentItem = self.itemFromIndex(parentIndex)
            parentRect = self.visualRect(parentIndex)

            if parentItem and parentItem.patch.parallelRun:
                painter.setPen(QColor(200, 90, 90))
                painter.drawLine(parentRect.x()-15, rect.y(), parentRect.x()-15, rect.y() + rect.height())

            if parentItem and parentItem.patch.parallelDeform:
                painter.setPen(QColor(90, 200, 90))
                painter.drawLine(parentRect.x()-10, rect.y(), parentRect.x()-10, rect.y() + rect.height())

            if parentItem and parentItem.patch.parallelDeformInit:
                painter.setPen(QColor(70, 120, 205))
                painter.drawLine(parentRect.x()-5, rect.y(), parentRect.x()-5, rect.y() + rect.height())

            if parentItem and parentItem.patch.block:
                painter.setPen(QColor(100, 130, 100))
                painter.drawLine(parentRect.x()-indent+7, rect.y(), parentRect.x()-indent+7, rect.top() + rect.height())

            parentIndex = parentIndex.parent()

        if item.childCount() and rect.x() + rect.width() > rect.x():
            painter.setPen(QPen(QBrush(QColor(100, 100, 100)), 1, Qt.SolidLine))
            painter.fillRect(QRect(rect.x() - 16, rect.y() + 4, 12, 12), QColor(45, 45, 45))
            painter.drawRect(rect.x() - 16, rect.y() + 4, 12, 12)
            painter.setPen(QPen(QBrush(QColor(120, 120, 120)), 1, Qt.SolidLine))
            if item.isExpanded():
                painter.drawLine(rect.x() - 7, rect.y() + 10, rect.x() - 13, rect.y() + 10)
            else:
                painter.drawLine(rect.x() - 10, rect.y() + 7, rect.x() - 10, rect.y() + 14)
                painter.drawLine(rect.x() - 7, rect.y() + 10, rect.x() - 13, rect.y() + 10)

        nameIdx = modelIdx.sibling(modelIdx.row(), 0)
        nameRect = self.visualRect(nameIdx)

        typeIdx = modelIdx.sibling(modelIdx.row(), 1)
        typeRect = self.visualRect(typeIdx)

        versionIdx = modelIdx.sibling(modelIdx.row(), 2)
        versionRect = self.visualRect(versionIdx)

        painter.setPen(QColor(100, 100, 100))

        if not re.match("^[a-zA-Z]\\w*$", unicode(item.patch.name)):
            painter.fillRect(nameRect, QBrush(QColor(170, 50, 50)))

        itemParent = item.parent()
        if itemParent and len(itemParent.patch.findChildren(item.patch.name)) > 1:
            painter.fillRect(nameRect, QBrush(QColor(170, 50, 50)))

        if item.patch.block:
            painter.setPen(QColor(100, 130, 100))
            painter.drawText(nameRect.x()-27, nameRect.center().y()+5, u"♦")

        # set selected style
        if modelIdx in self.selectedIndexes():
            painter.fillRect(rect.x()-1, rect.y(), rect.width(), rect.height()-1, QColor(80, 96, 154, 60))
            painter.setPen(QColor(73, 146, 158))
            painter.drawRect(rect.x()-1, rect.y(), rect.width(), rect.height()-1)

        if item.patch.state in [Patch.StateMuted, Patch.StateMutedAll]:
            painter.setPen(QColor(90, 90, 90))
        else:
            painter.setPen(QColor(210, 210, 210))

        if item.patch.state == Patch.StateMutedAll:
            font = painter.font()
            font.setStrikeOut(True)
            painter.setFont(font)

        if self.itemUnderMouse is item:
            painter.setPen(QColor(255, 255, 255))

        painter.drawText(nameRect, Qt.AlignLeft | Qt.AlignVCenter, item.patch.name)

        if item.patch.type:
            if not re.match("^([a-zA-Z_]\\w*)*$", unicode(item.patch.type)):
                painter.setPen(QColor(200, 200, 200))
                painter.fillRect(typeRect, QBrush(QColor(170, 50, 50)))

            painter.setPen(QColor(110, 110, 110))
            painter.drawText(typeRect, Qt.AlignLeft | Qt.AlignVCenter, item.patch.type)

        if item.patch.version:
            if item.patch.isActualVersion():
                painter.setPen(QColor(150, 250, 150))
            else:
                painter.setPen(QColor(250, 150, 150))

            painter.drawText(versionRect, Qt.AlignLeft | Qt.AlignVCenter, str(item.patch.version))

        painter.restore()

    def undoChange(self):
        if not self.undoQueue:
            return

        self.clear()

        patches = self.undoQueue.pop()

        for p in patches:
            item = self.makeTreeItemFromPatch(p)
            self.addTopLevelItem(item)

    def trackHistory(self):
        self.undoQueue.append([self.invisibleRootItem().child(i).patch.copy() for i in range(self.invisibleRootItem().childCount())])

    def renameItem(self, item=None):
        if not item:
            item = self.currentItem()
            if not item:
                return

        topLevelItem = item
        while topLevelItem.parent():
            topLevelItem = topLevelItem.parent()

        oldName = item.patch.name
        newName, ok = QInputDialog.getText(self, "Patch", "Name", QLineEdit.Normal, oldName)
        if ok:
            self.trackHistory()

            topLevelItem.patch.renameAbsoluteLinks(oldName, newName)
            parentPatch = item.parent() and item.parent().patch

            if parentPatch and len(parentPatch.findChildren(oldName)) == 1:
                parentPatch.renameRelativeLinks(oldName, newName) # this resolves links like this /child/attr (i.e parent relative)

            item.patch.name = unicode(newName)
            item.setText(0, item.patch.name)

    def renameItemType(self, item=None):
        items = []

        if not item:
            items = self.selectedItems()
        else:
            items = [item]

        if not items:
            return

        newType, ok = QInputDialog.getText(self, "Patch", "Type", QLineEdit.Normal, items[0].patch.type)
        if ok:
            self.trackHistory()

            for item in items:
                item.patch.type = unicode(newType)
                item.setText(1, item.patch.type)

                if not item.patch.type:
                    item.patch.version = 0

    def mouseDoubleClickEvent(self, event):
        index = self.indexAt(event.pos())
        if not index.isValid():
            return

        item = self.itemFromIndex(index)

        if index.column() == 0:
            self.renameItem(item)
        elif index.column() == 1:
            self.renameItemType(item)

    def event(self, event):
        if event.type() == QEvent.KeyPress:
            if event.key() == Qt.Key_Tab:
                self.patchListDialog.exec_()

                if self.patchListDialog.selectedFileName:
                    self.trackHistory()

                    p = Patch.loadFromFile(self.patchListDialog.selectedFileName)
                    item = self.makeTreeItemFromPatch(p)
                    self.addTopLevelItem(item)

                event.accept()
                return True

        return QTreeWidget.event(self, event)

    def reset(self):
        self.clearSelection()
        PatchMainWindow.browserWidget.rightBrowserWidget.logWidget.makeVisible(False)

        browser = PatchMainWindow.browserWidget
        browser.rightBrowserWidget.patch = None
        browser.rightBrowserWidget.update()

        browser.leftBrowserWidget.update(None)

    def findItemByName(self, name, root=None, walk=True):
        if not root:
            root = self.invisibleRootItem()
        elif root.patch.name == name:
            return root

        for i in range(root.childCount()):
            ch = root.child(i)
            if ch.patch.name == name:
                return ch
            elif walk:
                res = self.findItemByName(name, ch)
                if res:
                    return res

    def copyClipboard(self):
        selectedItems = self.selectedItems()
        if not selectedItems:
            return

        self.clipboard = [item.patch.copy() for item in selectedItems]

    def pasteClipboard(self, pasteAsChild=True, position=0): # position: 0 - to the end, -1 - below, 1 - above
        if not self.clipboard:
            return

        self.trackHistory()

        for p in self.clipboard:
            item = self.makeTreeItemFromPatch(p.copy())

            selectedItems = self.selectedItems()
            if selectedItems:
                currentItem = selectedItems[0]

                if pasteAsChild:
                    currentItem.addChild(item)
                    currentItem.patch.children.append(item.patch)
                else:
                    parent = (currentItem.parent() or self.invisibleRootItem())

                    index = parent.indexOfChild(currentItem) + position
                    parent.insertChild(index, item)

                    if currentItem.parent():
                        parent.patch.children.insert(index, item.patch)
            else:
                self.addTopLevelItem(item)

        self.clearSelection()
        self.setCurrentItem(item)

    def cutClipboard(self):
        selectedItems = self.selectedItems()
        if not selectedItems:
            return

        self.trackHistory()
        self.copyClipboard()

        for item in selectedItems:
            parent = (item.parent() or self.invisibleRootItem())
            index = parent.indexOfChild(item)

            if item.parent():
                parent.patch.removeChildAtIndex(index)

            parent.removeChild(item)

        self.setCurrentItem(parent)

    def saveItemPatch(self, topLevel=False):
        names = ", ".join([item.patch.name for item in self.selectedItems()])
        msg = "Save top level patch?" if topLevel else "Save selected patches?\n%s"%names

        ok = QMessageBox.question(self, "Patch", msg, QMessageBox.Yes and QMessageBox.No, QMessageBox.Yes) == QMessageBox.Yes
        if ok:
            for item in self.selectedItems():
                if topLevel:
                    while item.parent():
                        item = item.parent()

                p = item.patch
                if not p.description:
                    QMessageBox.critical(self, "Patch", "Description is empty for '%s'. Cannot save!"%p.name)
                else:
                    name = p.type if p.type else p.name
                    p.publish(saveOnly=True)
                    item.setText(1, p.type)
                    item.setText(2, str(p.version))

    def keyPressEvent(self, event):
        shift = event.modifiers() & Qt.ShiftModifier
        ctrl = event.modifiers() & Qt.ControlModifier
        alt = event.modifiers() & Qt.AltModifier

        key = event.key()

        if key == Qt.Key_Delete and shift: # remove all
            self.removeAll()

        elif ctrl and key == Qt.Key_M: # mute all
            self.muteAll()

        elif ctrl and key == Qt.Key_U: # update from clipboard patch
            self.updateItemFromClipboard(attributes=shift)

        elif ctrl and key == Qt.Key_C: # copy
            self.copyClipboard()

        elif ctrl and key == Qt.Key_V: # paste
            self.pasteClipboard(pasteAsChild=not shift)

        elif ctrl and key == Qt.Key_X: # cut
            self.cutClipboard()

        elif ctrl and key == Qt.Key_D: # duplicate
            self.copyClipboard()
            self.pasteClipboard(pasteAsChild=False)

        elif ctrl and key == Qt.Key_S: # save
            self.saveItemPatch()

        elif ctrl and key == Qt.Key_P: # publish
            self.publishItem()

        elif ctrl and key == Qt.Key_B: # build
            PatchMainWindow.browserWidget.leftBrowserWidget.buildBtnClicked()

        elif key == Qt.Key_Up and shift: # move up
            self.moveItem(up=True)

        elif key == Qt.Key_Down and shift: # move down
            self.moveItem(up=False)

        elif key in [Qt.Key_Home, Qt.Key_End]:
            parentItem = (self.currentItem().parent() or self.invisibleRootItem())
            if parentItem.childCount() > 0:
                childItem = parentItem.child(0) if key == Qt.Key_Home else parentItem.child(parentItem.childCount()-1)
                self.setCurrentItem(childItem, self.currentColumn())
                self.treeItemClicked(childItem)

        elif key == Qt.Key_Left: # go up to parent
            parentItem = self.currentItem().parent()
            if parentItem:
                self.setCurrentItem(parentItem, self.currentColumn())
                self.treeItemClicked(parentItem)

        elif key in [Qt.Key_Right]: # go to the first child
            item = self.currentItem()
            if item.childCount() > 0:
                self.setCurrentItem(item.child(0), self.currentColumn())
                self.treeItemClicked(item.child(0))

        elif key == Qt.Key_Up:
            nextItem = self.itemAbove(self.currentItem())
            if nextItem:
                self.setCurrentItem(nextItem, self.currentColumn())
                self.treeItemClicked(nextItem)

        elif key == Qt.Key_Down:
            nextItem = self.itemBelow(self.currentItem())
            if nextItem:
                self.setCurrentItem(nextItem, self.currentColumn())
                self.treeItemClicked(nextItem)

        elif key == Qt.Key_Delete:
            self.removeItem()

        elif key == Qt.Key_Escape:
            self.reset()

        elif key == Qt.Key_F5:
            self.searchEverywhereWidget.show()

        elif ctrl and key == Qt.Key_Space:
            def expandRecursive(item, v):
                for i in range(item.childCount()):
                    ch = item.child(i)
                    ch.setExpanded(v)
                    expandRecursive(ch, v)

            for item in self.selectedItems():
                v = not item.isExpanded()
                item.setExpanded(v)
                expandRecursive(item, v)

        elif key == Qt.Key_Space:
            for item in self.selectedItems():
                item.setExpanded(not item.isExpanded())

        elif ctrl and key == Qt.Key_Return:
            self.renameItemType()

        elif key == Qt.Key_Return:
            self.renameItem()

        elif shift and key == Qt.Key_M:
            self.toggleMuteOthers()

        elif key == Qt.Key_M: # mute
            self.muteItem()

        elif key == Qt.Key_Insert:
            self.insertItem(asChild=not shift)

        elif ctrl and key == Qt.Key_G: # group
            items = self.selectedItems()
            parent = self.insertItem(asChild=False)
            for it in items:
                it.parent().removeChild(it)
                parent.addChild(it)
            
        elif key == Qt.Key_U: # update to the latest version
            self.updateItem(attributes=shift)

        elif key == Qt.Key_R: # revert current version
            self.revertItem()

    def moveItem(self, up=False):
        selectedItems = sorted(self.selectedItems(), key=lambda item:self.indexFromItem(item))
        if not up:
            selectedItems.reverse()

        self.trackHistory()

        for item in selectedItems:
            parent = (item.parent() or self.invisibleRootItem())
            index = parent.indexOfChild(item)

            if (up  and index > 0) or (not up and index+1 < parent.childCount()):
                d = -1 if up else 1

                parent.removeChild(item)
                parent.insertChild(index+d, item)

                if parent is not self.invisibleRootItem():
                    parent.patch.removeChildAtIndex(index)
                    parent.patch.children.insert(index+d, item.patch)

            self.setItemSelected(item, True)

    def muteAll(self):
        for item in self.selectedItems():
            self.muteItem(item, Patch.StateMutedAll)

    def toggleMuteOthers(self):
        for item in self.selectedItems():
            parent = item.parent()
            if parent:
                index = parent.indexOfChild(item)
                for i in range(parent.childCount()):
                    ch = parent.child(i)
                    if i != index:
                        self.muteItem(ch)

    def updateItemFromClipboard(self, attributes=False):
        if not self.clipboard:
            return

        patchFromClipboard = self.clipboard[-1]

        for item in self.selectedItems():
            p = item.patch

            ok = QMessageBox.question(self,
                                      "Patch",
                                      "Update '%s' patch%s with clipboard ('%s')?"%(p.name, " (+ attributes)" if attributes else "", patchFromClipboard.name),
                                      QMessageBox.Yes and QMessageBox.No,
                                      QMessageBox.Yes) == QMessageBox.Yes
            if ok:
                p.updateFromPatch(patchFromClipboard.copy(), attributes=attributes)
                self.updateItemPatch(item)

    def removeItem(self):
        names = ", ".join([item.patch.name for item in self.selectedItems()])
        ok = QMessageBox.question(self, "Patch", "Really remove selected patches?\n%s"%names, QMessageBox.Yes and QMessageBox.No, QMessageBox.Yes) == QMessageBox.Yes
        if ok:
            self.cutClipboard()
            self.reset()

    def revertItem(self):
        names = ", ".join([item.patch.name for item in self.selectedItems()])
        ok = QMessageBox.question(self, "Patch", "Revert selected patches?\n%s"%names, QMessageBox.Yes and QMessageBox.No, QMessageBox.Yes) == QMessageBox.Yes
        if ok:
            for item in self.selectedItems():
                p = item.patch

                if p.update(revert=True, attributes=True):
                    self.updateItemPatch(item)
                else:
                    QMessageBox.critical(self, "Patch", "Cannot revert '%s'! Missing files"%p.name)

    def removeAll(self):
        if self.invisibleRootItem().childCount() > 0:
            ok = QMessageBox.question(self, "Patch", "Remove all?", QMessageBox.Yes and QMessageBox.No, QMessageBox.Yes) == QMessageBox.Yes
            if ok:
                self.trackHistory()
                self.clear()
                self.reset()

    def publishItem(self):
        names = ", ".join([item.patch.name for item in self.selectedItems()])
        ok = QMessageBox.question(self,
                                  "Patch",
                                  "Publish selected patches?\n%s"%names,
                                  QMessageBox.Yes and QMessageBox.No,
                                  QMessageBox.Yes) == QMessageBox.Yes
        if ok:
            for item in self.selectedItems():
                p = item.patch
                if not p.type:
                    QMessageBox.critical(self, "Patch", "Cannot publish '%s'. Specify type!"%p.name)
                else:
                    p.publish()
                    item.setText(1, p.type) 
                    item.setText(2, str(p.version)) # update version

    def updateItem(self, attributes=False):
        names = ", ".join([item.patch.name for item in self.selectedItems()])

        ok = QMessageBox.question(self,
                                  "Patch",
                                  "Update selected patches%s?\n%s"%(" (+attributes)" if attributes else "", names),
                                  QMessageBox.Yes and QMessageBox.No,
                                  QMessageBox.Yes) == QMessageBox.Yes
        if ok:
            for item in self.selectedItems():
                p = item.patch

                if not p.isActualVersion():
                    if p.update(revert=False, attributes=attributes):
                        self.updateItemPatch(item)
                    else:
                        QMessageBox.critical(self, "Patch", "Cannot update '%s'! Missing files"%p.name)

    def insertItem(self, asChild=True):
        item = self.makeTreeItemFromPatch(Patch())

        self.trackHistory()

        selectedItems = self.selectedItems()
        if selectedItems:
            currentItem = selectedItems[0]

            if not asChild and currentItem.parent():
                currentItem.parent().patch.children.append(item.patch)
                currentItem.parent().addChild(item)
            else:
                currentItem.patch.children.append(item.patch)
                currentItem.addChild(item)
        else:
            self.addTopLevelItem(item)

        return item

    def muteItem(self, item=None, state=None):
        items = []
        if not item:
            items = self.selectedItems()
        else:
            items = [item]

        self.trackHistory()

        for item in items:
            item.patch.state = state if state!=None else (Patch.StateNormal if item.patch.state in [Patch.StateMuted, Patch.StateMutedAll] else Patch.StateMuted)

            for i in range(item.childCount()):
                self.muteItem(item.child(i), state)

            expanded = item.isExpanded() # updation
            item.setExpanded(True)
            item.setExpanded(expanded)

    def updateItemPatch(self, item):
        self.trackHistory()

        p = item.patch

        parent = item.parent() or self.invisibleRootItem()
        index = parent.indexOfChild(item)
        expanded = item.isExpanded()

        newItem = self.makeTreeItemFromPatch(p)
        if item.parent():
            parent.insertChild(index, newItem)
        else:
            self.insertTopLevelItem(index, newItem)

        parent.removeChild(item)

        newItem.setExpanded(expanded)
        self.setCurrentItem(newItem)
        self.treeItemClicked(newItem)

        return newItem

    def treeItemClicked(self, item):
        browser = PatchMainWindow.browserWidget
        rightBrowserWidget = browser.rightBrowserWidget

        rightBrowserWidget.patch = item.patch
        rightBrowserWidget.update()

        browser.leftBrowserWidget.update(item.patch)

    def makeTreeItemFromPatch(self, patch):
        item = QTreeWidgetItem([patch.name+" ", patch.type+" ", str(patch.version if patch.version else "")])
        item.setFlags(Qt.ItemIsSelectable | Qt.ItemIsEditable | Qt.ItemIsEnabled | Qt.ItemIsDragEnabled | Qt.ItemIsDropEnabled)

        item.patch = patch

        for c in patch.children:
            item.addChild(self.makeTreeItemFromPatch(c))

        return item

class SummaryWidget(QTextEdit):
    def __init__(self, patch=None, **kwargs):
        super(SummaryWidget, self).__init__(**kwargs)

        self.setReadOnly(True)
        self.setWordWrapMode(QTextOption.NoWrap)

        self.update(patch)

    def update(self, patch):
        self.patch = patch

        if not self.patch:
            self.clear()
            return

        text = ""

        if os.path.exists(self.patch.loadedFrom):
            modtime = time.strftime("%Y/%m/%d %H:%M", time.localtime(os.path.getmtime(self.patch.loadedFrom)))
            text += "<b>File: </b>%s<br>"%self.patch.loadedFrom
            text += "<b>File modification time: </b>%s<br><br>"%modtime

        text += "<b>Name: </b>%s<br>"%self.patch.name

        if self.patch.type:
            text += "<b>Type: </b>%s<br>"%self.patch.type

        if self.patch.version:
            if self.patch.isActualVersion():
                actual = " <span style=\"background-color: #338833; color: #ffffff\">(actual)</span>"
            else:
                actual = ""

            text += "<b>Version: </b>v%.3d %s<br>"%(self.patch.version, actual)

        text += "<b>Maya attributes:</b><br>"
        for attrData in self.patch.getMayaAttributes():
            a = "%s %s"%(attrData["mayaName"], attrData["attribute"].type)

            name, typ = a.replace("<", "&lt;").replace(">", "&gt;").split()

            text += "{name} (<span style=\"color: #968ce6\">{type}</span>)<br>".format(name=name, type=typ)

        if self.patch.clCode:
            text += "<br><b>OpenCL kernel arguments:</b><br>"
            for arg in self.patch.getKernelArguments():
                text += "%s<br>"%arg

        self.setHtml(text)

def setEvaluatorLock(ev, state):
    if not core.objExists(ev):
        return

    ev = core.PyNode(ev)

    if state == False:
        ev._module.unlock()
        ev._userData.unlock()
    else:
        ev._module.lock()
        ev._userData.lock()

class LeftBrowserWidget(QWidget):
    def __init__(self, **kwargs):
        super(LeftBrowserWidget, self).__init__(**kwargs)

        layout = QVBoxLayout()
        self.setLayout(layout)
        layout.setMargin(0)

        self.buildBtn = QPushButton()
        self.setBuildParams()
        self.buildBtn.setContextMenuPolicy(Qt.DefaultContextMenu)
        self.buildBtn.clicked.connect(self.buildBtnClicked)
        self.buildBtn.contextMenuEvent = self.buildBtnContextMenuEvent

        self.checkBtn = QPushButton("Check")
        self.checkBtn.clicked.connect(self.checkBtnClicked)

        self.viewCodeBtn = QPushButton("View")
        self.viewCodeBtn.clicked.connect(self.viewCodeBtnClicked)

        self.nodeLabel = QLabel("(not attached)")
        self.nodeLabel.setAlignment(Qt.AlignHCenter)
        self.nodeLabel.setContextMenuPolicy(Qt.DefaultContextMenu)
        self.nodeLabel.contextMenuEvent = self.nodeLabelContextMenuEvent
        self.nodeLabel.mouseDoubleClickEvent = self.nodeLabelDoubleClicked

        btnlayout = QHBoxLayout()
        btnlayout.addWidget(self.buildBtn)
        btnlayout.addWidget(self.checkBtn)
        btnlayout.addWidget(self.viewCodeBtn)

        self.treeWidget = TreeWidget()

        self.summaryWidget = SummaryWidget()

        self.patchSettingsWidget = PatchSettingsWidget(None)

        splitter = QSplitter(Qt.Vertical)
        splitter.addWidget(self.treeWidget)
        splitter.addWidget(self.patchSettingsWidget)
        splitter.addWidget(self.summaryWidget)

        splitter.setSizes([400, 200, 500])

        layout.addLayout(btnlayout)
        layout.addWidget(self.nodeLabel)
        layout.addWidget(splitter)

    def nodeLabelDoubleClicked(self, event):
        oldName = self.nodeLabel.text()
        if oldName == "(not attached)":
            return

        newName, ok = QInputDialog.getText(self, "Patch", "Rename evaluator", QLineEdit.Normal, oldName)
        if ok:
            self.nodeLabel.setText(newName)
            core.rename(oldName, newName)

    def update(self, patch):
        self.summaryWidget.update(patch)
        self.patchSettingsWidget.update(patch)

    def attachLocatorClicked(self):
        node = unicode(self.nodeLabel.text())
        if core.objExists(node):
            node = core.PyNode(node)

            if not node.message.listConnections(s=False, d=True, type="evaluatorLocator"):
                loc = core.createNode("evaluatorLocator")
                locParent = loc.getParent()
                locParent.t.setLocked(True)
                locParent.r.setLocked(True)
                locParent.s.setLocked(True)
                node.message >> loc.evaluatorLink

    def loadClicked(self):
        self.attachClicked()

        node = unicode(self.nodeLabel.text())
        if core.objExists(node):
            node = core.PyNode(node)

            try:
                userData = json.loads(node._userData.get())
            except:
                QMessageBox.critical(self, "Patch", "UserData is invalid. No patch is attached")
                return

            patchFile = userData.get("patch", "")
            if os.path.exists(patchFile):
                tw = PatchMainWindow.browserWidget.leftBrowserWidget.treeWidget
                item = tw.makeTreeItemFromPatch(Patch.loadFromFile(patchFile))
                tw.addTopLevelItem(item)
                tw.setCurrentItem(item)
                tw.treeItemClicked(item)

    def nodeLabelContextMenuEvent(self, event):
        menu = QMenu(self)

        if "pymel.core" in sys.modules: # when Maya
            evaluatorsMenu = QMenu("Evaluators", self)
            for ev in core.ls(type="evaluator"):
                action = QAction(ev.name(), self)
                action.triggered.connect(lambda n=ev.name(): (core.select(n), self.attachClicked()))
                evaluatorsMenu.addAction(action)

            menu.addMenu(evaluatorsMenu)

        loadAction = QAction("Attach and load from selected", self)
        loadAction.triggered.connect(self.loadClicked)
        menu.addAction(loadAction)

        attachAction = QAction("Attach", self)
        attachAction.triggered.connect(self.attachClicked)
        menu.addAction(attachAction)

        attachLocatorAction = QAction("Attach draw locator", self)
        attachLocatorAction.triggered.connect(self.attachLocatorClicked)
        menu.addAction(attachLocatorAction)

        menu.addSeparator()

        newAction = QAction("New", self)
        newAction.triggered.connect(lambda: self.newClicked(False))
        menu.addAction(newAction)

        newDeformerAction = QAction("New deformer", self)
        newDeformerAction.triggered.connect(lambda: self.newClicked(True))
        menu.addAction(newDeformerAction)

        menu.addSeparator()

        lockMenu = QMenu("Lock", self)
        lockAction = QAction("Lock", self)
        lockAction.triggered.connect(lambda: self.lockClicked(True))
        lockMenu.addAction(lockAction)

        unlockAction = QAction("Unlock", self)
        unlockAction.triggered.connect(lambda: self.lockClicked(False))
        lockMenu.addAction(unlockAction)

        menu.addMenu(lockMenu)

        menu.addSeparator()

        removeAction = QAction("Remove", self)
        removeAction.triggered.connect(self.removeClicked)
        menu.addAction(removeAction)

        menu.popup(event.globalPos())

    def lockClicked(self, state):
        node = self.nodeLabel.text()
        setEvaluatorLock(node, state)

    def removeClicked(self):
        node = self.nodeLabel.text()
        if core.objExists(node):
            core.delete(node)
            self.nodeLabel.setText("(not attached)")

    def newClicked(self, asDeformer=False):
        if asDeformer:
            ev = core.deformer(type="evaluator")[0]
        else:
            ev = core.createNode("evaluator")

        self.nodeLabel.setText(ev.name())

    def attachClicked(self):
        ls = core.ls(sl=True, type="evaluator")
        if not ls:
            QMessageBox.information(self, "Patch", "Select an evaluator node in Maya")
            return

        self.nodeLabel.setText(ls[0].name())

    def buildBtnContextMenuEvent(self, event):
        menu = QMenu(self)

        fromServerAction = QAction("From server", self)
        fromServerAction.triggered.connect(lambda _=None: self.setBuildParams(fromServer=True))
        menu.addAction(fromServerAction)

        menu.addSeparator()

        for v in [2018, 2019]:
            action = QAction("[ Maya %d ]"%v, self)
            font = self.font()
            font.setWeight(QFont.Bold)
            action.setFont(font)
            menu.addAction(action)

            for t in ["debug", "release"]:
                action = QAction(t.capitalize(), self)
                action.triggered.connect(lambda _=None, v=v, t=t: self.setBuildParams(v, t, fromServer=False))
                menu.addAction(action)

            menu.addSeparator()

        menu.popup(event.globalPos())

    def setBuildParams(self, version=EvaluatorBuildParams["version"], type=EvaluatorBuildParams["type"], fromServer=EvaluatorBuildParams["fromServer"]):
        global EvaluatorBuildParams

        EvaluatorBuildParams["version"] = version
        EvaluatorBuildParams["type"] = type
        EvaluatorBuildParams["fromServer"] = fromServer

        if fromServer:
            self.buildBtn.setText("Build (from server)")
        else:
            self.buildBtn.setText("Build (%s, %s)"%(version, type))

    def checkBtnClicked(self):
        if not self.treeWidget.selectedItems():
            QMessageBox.critical(self, "Patch", "Nothing is selected to check")
            return

        item = self.treeWidget.currentItem()

        while item.parent():
            item = item.parent()

        Patch.cleanupBuild()
        p = item.patch        

        outputs = []

        err, output = ispcBuild(item.patch.generateIspcFile())
        outputs.append(output)

        if not err: # check OpenCL
            err, output = item.patch.checkCL()
            outputs.append(output)

            if not err:            
                cppFileName = p.buildCpp()            
                err, output = clangBuild(cppFileName, "syntax", EvaluatorBuildParams["version"]) # make dll using clang_build.bat
                outputs.append(output)

        logWidget = PatchMainWindow.browserWidget.rightBrowserWidget.logWidget
        outputsStr = "\n".join(outputs)
        logWidget.update(outputsStr, not outputsStr)

    def buildBtnClicked(self):
        if not self.treeWidget.selectedItems():
            QMessageBox.critical(self, "Patch", "Nothing is selected to build")
            return

        item = self.treeWidget.currentItem()
        while item.parent():
            item = item.parent()

        embedToEvaluator = True

        node = unicode(self.nodeLabel.text())
        if node.startswith("(") or not core.objExists(node):
            embedToEvaluator = False
            self.nodeLabel.setText("(not attached)")

        ev = core.PyNode(node) if embedToEvaluator else None

        Patch.cleanupBuild()

        p = item.patch       

        patchBaseName = os.path.basename(p.getFileName(ext=False))

        # build
        if not EvaluatorBuildParams["fromServer"]: # build local module
            logWidget = PatchMainWindow.browserWidget.rightBrowserWidget.logWidget

            outputs = []

            err, output = p.checkCL()
            if err:
                logWidget.update(output, False)
                return

            outputs.append(output)

            err, output = ispcBuild(p.generateIspcFile())
            if err:
                logWidget.update(output, False)
                return

            outputs.append(output)

            try:
                Patch.ParallelEnabled = EvaluatorBuildParams["type"] == "release"
                cppFileName = p.buildCpp()
            except Exception as e:
                logWidget.update(str(e), False)
                return

            dllFileName = "{dir}/nodes/{name}_{version}.dll".format(dir=EvaluatorLocalPath,
                                                                    name=patchBaseName,
                                                                    version=EvaluatorBuildParams["version"])

            if not embedToEvaluator and type != "syntax": # debug or release
                if isFileLocked(dllFileName):
                    QMessageBox.critical(self, "Patch", "DLL file is busy: %s"%dllFileName)
                    return

            if embedToEvaluator:
                module = os.path.realpath(EvaluatorLocalPath + "/nodes/" + patchBaseName)
                core.evaluatorInfo(un=True) # unload all modules

            if not err:            
                err, output = clangBuild(cppFileName, EvaluatorBuildParams["type"], EvaluatorBuildParams["version"]) # make dll using clang_build.bat
                outputs.append(output)

                outputsStr = "\n".join(outputs)

                logWidget = PatchMainWindow.browserWidget.rightBrowserWidget.logWidget
                logWidget.update(outputsStr, not outputsStr)

                if not err:
                    p.backup()

                    if os.path.exists(dllFileName): # remove old dll
                        os.remove(dllFileName)

                    tempDllFile = "{dir}/patch/temp/{name}_{version}.dll".format(dir=EvaluatorLocalPath,
                                                                                 name=p.name,
                                                                                 version=EvaluatorBuildParams["version"])
                    if os.path.exists(tempDllFile):
                        os.rename(tempDllFile, dllFileName) # move new dll file

            module = "$EV_LOCAL_PATH/nodes/%s"%patchBaseName
            if embedToEvaluator:
                userData = {"patch": p.loadedFrom.replace("\\", "/"),
                            "creator": os.path.expandvars("%USERNAME%"),
                            "datetime": time.strftime("%Y/%m/%d %H:%M", time.localtime(time.time())),
                            "compile": EvaluatorBuildParams["type"]}

                userData = json.dumps(userData)

                setEvaluatorLock(ev, False) # unlock
                ev._module.set(module)
                if not err:
                    ev._userData.set(userData)
                setEvaluatorLock(ev, True) # lock

                core.evaluatorInfo(u=True) # update all evaluators

        else: # load dll from server
            dllfile = Patch.getLatestDllByFileName(p.getFileName(), EvaluatorBuildParams["version"])

            if not dllfile:
                QMessageBox.critical(self, "Patch", "Cannot find '%s' module on the server"%p.name)
                return
                
            moduleName = os.path.splitext(os.path.basename(dllfile))[0].replace("_%d"%EvaluatorBuildParams["version"], "")

            if ev:
                setEvaluatorLock(ev, False)
                ev._module.set("$EV_SERVER_PATH/nodes/"+moduleName)

                userData = {"patch": p.loadedFrom.replace("\\", "/"),
                            "creator": os.path.expandvars("%USERNAME%"),
                            "datetime": time.strftime("%Y/%m/%d %H:%M", time.localtime(time.time())),
                            "server": True}

                ev._userData.set(json.dumps(userData))
                setEvaluatorLock(ev, True)

            else:
                QMessageBox.critical(self, "Patch", "Attach evaluator!")

    def viewCodeBtnClicked(self):
        if not self.treeWidget.selectedItems():
            return

        p = self.treeWidget.currentItem().patch

        code = ""
        try:
            file = p.buildCpp()
        except Exception as e:
            logWidget = PatchMainWindow.browserWidget.rightBrowserWidget.logWidget
            logWidget.update(str(e), False)
            return

        with open(file, "r") as f:
            code = f.read()

        ViewCodeWidget(code, parent=PatchMainWindow).exec_()

class ViewCodeWidget(QDialog):
    def __init__(self, text, **kwargs):
        super(ViewCodeWidget, self).__init__(**kwargs)

        self.setWindowFlags(self.windowFlags() | Qt.WindowSystemMenuHint | Qt.WindowMinMaxButtonsHint)

        self.setWindowTitle("View code")
        self.setGeometry(400,300, 600, 500)

        layout = QVBoxLayout()
        self.setLayout(layout)
        layout.setMargin(0)

        self.cppEditorWidget = CppEditorWidget()
        self.cppEditorWidget.setText(text)
        self.cppEditorWidget.setReadOnly(True)

        layout.addWidget(self.cppEditorWidget)

    def setText(self, text):
        self.cppEditorWidget.setText(formatCpp(text))

class PatchSettingsWidget(QWidget):
    def __init__(self, patch, **kwargs):
        super(PatchSettingsWidget, self).__init__(**kwargs)

        self.patch = patch
        self.isUpdating = False

        layout = QVBoxLayout()
        self.setLayout(layout)
        layout.setMargin(0)

        gridLayout = QGridLayout()
        gridLayout.setDefaultPositioning(2, Qt.Horizontal)
        gridLayout.setMargin(5)
        self.namespaceWidget = QCheckBox("Namespace")
        self.namespaceWidget.stateChanged.connect(self.stateChanged)

        self.customNamespaceWidget = QLineEdit()
        self.customNamespaceWidget.editingFinished.connect(self.stateChanged)

        self.parallelRunWidget = QCheckBox("Parallel run")
        self.parallelRunWidget.setStyleSheet("QCheckBox:enabled { color: #C85A5A }")

        self.parallelDeformInitWidget = QCheckBox("Parallel deform init")
        self.parallelDeformInitWidget.setStyleSheet("QCheckBox:enabled { color: #4477CC }")

        self.parallelDeformWidget = QCheckBox("Parallel deform")
        self.parallelDeformWidget.setStyleSheet("QCheckBox:enabled { color: #5AC85A }")

        self.debugOpenCLWidget = QCheckBox("Debug OpenCL (via draw)")
        self.blockWidget = QCheckBox("Block")

        self.parallelRunWidget.stateChanged.connect(self.stateChanged)
        self.parallelDeformWidget.stateChanged.connect(self.stateChanged)
        self.parallelDeformInitWidget.stateChanged.connect(self.stateChanged)
        self.debugOpenCLWidget.stateChanged.connect(self.stateChanged)
        self.blockWidget.stateChanged.connect(self.stateChanged)

        self.codeBlocksWidget = QCheckBox("Code blocks")
        self.codeBlocksWidget.stateChanged.connect(self.stateChanged)

        self.embedCodeWidget = QCheckBox("Embed code recursively")
        self.embedCodeWidget.stateChanged.connect(self.embedCodeStateChanged)

        self.descriptionWidget = QTextEdit()
        self.descriptionWidget.textChanged.connect(self.descriptionChanged)

        gridLayout.addWidget(self.namespaceWidget)
        gridLayout.addWidget(self.customNamespaceWidget)
        gridLayout.addWidget(self.codeBlocksWidget)
        gridLayout.addWidget(self.blockWidget)
        gridLayout.addWidget(self.parallelRunWidget)
        gridLayout.addWidget(self.debugOpenCLWidget)
        gridLayout.addWidget(self.parallelDeformInitWidget)
        gridLayout.addWidget(self.embedCodeWidget)
        gridLayout.addWidget(self.parallelDeformWidget)

        gridLayout.setColumnStretch(2, 1)

        layout.addLayout(gridLayout)
        layout.addWidget(self.descriptionWidget)

        self.update(patch)

    def descriptionChanged(self):
        if self.isUpdating or not self.patch:
            return

        text = unicode(self.descriptionWidget.toPlainText())
        self.patch.description = text

    def embedCodeChildren(self, patch, v):
        patch.embedCode = v
        for ch in patch.children:
            self.embedCodeChildren(ch, v)

    def embedCodeStateChanged(self, v):
        if self.isUpdating or not self.patch:
            return

        embedCode = self.embedCodeWidget.isChecked()
        self.embedCodeChildren(self.patch, embedCode)

    def stateChanged(self):
        if self.isUpdating or not self.patch:
            return

        self.patch.hasNamespace = self.namespaceWidget.isChecked()
        self.patch.customNamespace = self.customNamespaceWidget.text()
        self.customNamespaceWidget.setEnabled(bool(self.patch.hasNamespace))

        self.patch.parallelRun = self.parallelRunWidget.isChecked()
        self.patch.codeBlocks = self.codeBlocksWidget.isChecked()
        self.patch.parallelDeform = self.parallelDeformWidget.isChecked()
        self.patch.parallelDeformInit = self.parallelDeformInitWidget.isChecked()
        self.patch.debugOpenCL = self.debugOpenCLWidget.isChecked()
        self.patch.block = self.blockWidget.isChecked()

    def update(self, patch):
        self.patch = patch

        if not self.patch:
            self.setEnabled(False)
            return

        self.setEnabled(True)

        self.isUpdating = True

        self.namespaceWidget.setChecked(bool(self.patch.hasNamespace))
        self.customNamespaceWidget.setText(self.patch.customNamespace)
        self.customNamespaceWidget.setEnabled(bool(self.patch.hasNamespace))

        self.codeBlocksWidget.setChecked(bool(self.patch.codeBlocks))

        self.parallelRunWidget.setChecked(bool(self.patch.parallelRun))
        self.parallelDeformWidget.setChecked(bool(self.patch.parallelDeform))
        self.parallelDeformInitWidget.setChecked(bool(self.patch.parallelDeformInit))
        self.debugOpenCLWidget.setChecked(bool(self.patch.debugOpenCL))
        self.blockWidget.setChecked(bool(self.patch.block))

        self.embedCodeWidget.setChecked(bool(self.patch.embedCode))

        self.descriptionWidget.setPlainText(self.patch.description)

        self.isUpdating = False

def clearLayout(layout):
     if layout is not None:
         while layout.count():
             item = layout.takeAt(0)
             widget = item.widget()
             if widget is not None:
                 widget.setParent(None)
             else:
                 clearLayout(item.layout())

class LogHighligher(QSyntaxHighlighter):
    def __init__(self, parent=None):
        super(LogHighligher, self).__init__(parent)

        self.highlightingRules = []

        errorFormat = QTextCharFormat()
        errorFormat.setForeground(QColor(200, 100, 160))
        errorFormat.setFontUnderline(True)
        self.highlightingRules.append((QRegExp(LogWidget.ErrorRegexp), errorFormat))

    def highlightBlock(self, text):
        for pattern, format in self.highlightingRules:
            if not pattern:
                continue

            expression = QRegExp(pattern)
            index = expression.indexIn(text)
            while index >= 0:
                length = expression.matchedLength()
                self.setFormat(index, length, format)
                index = expression.indexIn(text, index + length)

        self.setCurrentBlockState(0)

class LogWidget(QTextEdit):
    ErrorRegexp = "([^:]+(?:defines|run|deformInit|deform|deformParallel|draw|commands)\\.cpp):(\\d+):(\\d+): error:"
    def __init__(self, **kwargs):
        super(LogWidget, self).__init__(**kwargs)

        self.setReadOnly(True)
        self.syntax = LogHighligher(self)

    def showEvent(self, event):
        self.makeVisible(False)

    def mouseDoubleClickEvent(self, event):
        if event.button() == Qt.LeftButton:
            cursor = self.cursorForPosition(event.pos())
            cursor.select(QTextCursor.LineUnderCursor)

            r = re.search(LogWidget.ErrorRegexp, unicode(cursor.selectedText()))
            if not r:
                return

            file, line, column = r.groups("")

            file = os.path.basename(file)
            fileItems = file.split("__")

            codeType = os.path.splitext(fileItems[-1])[0]

            tw = PatchMainWindow.browserWidget.leftBrowserWidget.treeWidget
            root = None
            for p in fileItems[:-1]:
                item = tw.findItemByName(p, root, walk=False)
                if not item:
                    QMessageBox.critical(self, "Patch", "Cannot find '%s' patch"%("/".join(fileItems[:-1])))
                    return

                root = item

            tw.setCurrentItem(item)
            tw.treeItemClicked(item)

            patchEditor = PatchMainWindow.browserWidget.rightBrowserWidget.patchCppEditorWidget

            idx = patchEditor.codeTypes.index(codeType+"Code")
            patchEditor.tabWidget.setCurrentIndex(idx)

            editor = patchEditor.tabWidget.widget(idx).editorWidget
            cursor = QTextCursor(editor.document().findBlockByLineNumber(int(line)-1))

            editor.setTextCursor(cursor)
        else:
            QTextEdit.mouseDoubleClickEvent(self, event)

    def makeVisible(self, v):
        size = 300 if v else 0

        splitter = self.parent()
        sizes = splitter.sizes()
        sizes[2] = size
        splitter.setSizes(sizes)

    def update(self, text, ok):
        logtime = time.strftime("%Y/%m/%d %H:%M", time.localtime())
        self.setText("\n".join([logtime, text]))

        self.makeVisible(True)

def color2hex(qcolor):
    return "%.2x%.2x%.2x"%(qcolor.red(), qcolor.green(), qcolor.blue())

class EditAttributeDialog(QDialog):
    def __init__(self, attribute, **kwargs):
        super(EditAttributeDialog, self).__init__(**kwargs)

        self.attribute = attribute

        self.setWindowTitle("Edit attribute")

        layout = QVBoxLayout()
        self.setLayout(layout)

        gridLayout = QGridLayout()
        gridLayout.setDefaultPositioning(2, Qt.Horizontal)

        self.nameWidget = QLineEdit()
        self.typeWidget = QComboBox()
        self.typeWidget.setEditable(True)
        self.typeWidget.addItems(Attribute.EvTypes)
        self.typeWidget.lineEdit().editingFinished.connect(self.typeChanged)

        self.valueWidget = QLineEdit()
        self.valueWidget.setCompleter(QCompleter())
        self.valueWidget.keyPressEvent = self.valueKeyPressEvent

        self.minValueWidget = QLineEdit()
        self.maxValueWidget = QLineEdit()

        minMaxLayoutWidget = QWidget()
        minMaxLayoutWidget.setLayout(QHBoxLayout())
        minMaxLayoutWidget.layout().setMargin(0)
        minMaxLayoutWidget.layout().addWidget(self.minValueWidget)
        minMaxLayoutWidget.layout().addWidget(self.maxValueWidget)

        self.clWidget = QComboBox()
        self.clWidget.addItems(["(not used)", "global", "constant", "global alloc (must be vector)", "local alloc (must be vector)"])

        self.itemsWidget = QLineEdit()

        gridLayout.addWidget(QLabel("Name"))
        gridLayout.addWidget(self.nameWidget)
        gridLayout.addWidget(QLabel("Type"))
        gridLayout.addWidget(self.typeWidget)
        gridLayout.addWidget(QLabel("Value"))
        gridLayout.addWidget(self.valueWidget)
        gridLayout.addWidget(QLabel("Min/Max"))
        gridLayout.addWidget(minMaxLayoutWidget)
        gridLayout.addWidget(QLabel("OpenCL"))
        gridLayout.addWidget(self.clWidget)
        gridLayout.addWidget(QLabel("Items"))
        gridLayout.addWidget(self.itemsWidget)

        self.outputWidget = QCheckBox("output")
        self.keyableWidget = QCheckBox("keyable")
        self.arrayWidget = QCheckBox("array")
        self.cachedWidget = QCheckBox("cached")
        self.hiddenWidget = QCheckBox("hidden")
        self.connectedOnlyWidget = QCheckBox("connected only")

        aw = PatchMainWindow.browserWidget.rightBrowserWidget.attributesWidget
        '''
        self.outputWidget.setStyleSheet("background-color: #%s"%color2hex(aw.colorsByHeader["Output"]))
        self.keyableWidget.setStyleSheet("background-color: #%s"%color2hex(aw.colorsByHeader["Keyable"]))
        self.arrayWidget.setStyleSheet("background-color: #%s"%color2hex(aw.colorsByHeader["Array"]))
        self.cachedWidget.setStyleSheet("background-color: #%s"%color2hex(aw.colorsByHeader["Cached"]))
        self.hiddenWidget.setStyleSheet("background-color: #%s"%color2hex(aw.colorsByHeader["Hidden"]))
        self.connectedOnlyWidget.setStyleSheet("background-color: #%s"%color2hex(aw.colorsByHeader["Connected"]))
        '''

        boolLayout = QGridLayout()
        boolLayout.setDefaultPositioning(3, Qt.Horizontal)
        boolLayout.addWidget(self.outputWidget)
        boolLayout.addWidget(self.keyableWidget)
        boolLayout.addWidget(self.arrayWidget)
        boolLayout.addWidget(self.cachedWidget)
        boolLayout.addWidget(self.hiddenWidget)
        boolLayout.addWidget(self.connectedOnlyWidget)

        okBtn = QPushButton("OK")
        okBtn.clicked.connect(self.okClicked)
        okBtn.setAutoDefault(False)

        layout.addLayout(gridLayout)
        layout.addLayout(boolLayout)

        layout.addWidget(okBtn)

        layout.addStretch()

        self.update()

    def typeChanged(self):
        newType = unicode(self.typeWidget.currentText())
        isMaya = newType in Attribute.EvTypes

        self.valueWidget.setVisible(newType not in ["EvMesh", "EvNurbsCurve", "EvNurbsSurface", "EvCompound", "EvMessage"])
        self.minValueWidget.setVisible(newType in ["EvInt", "EvFloat", "EvDouble", "EvVector"])
        self.maxValueWidget.setVisible(newType in ["EvInt", "EvFloat", "EvDouble", "EvVector"])

        self.outputWidget.setVisible(isMaya)
        self.keyableWidget.setVisible(isMaya and newType in ["EvBoolean", "EvInt", "EvFloat", "EvDouble", "EvVector", "EvEnum"])
        self.hiddenWidget.setVisible(isMaya)
        self.arrayWidget.setVisible(isMaya)
        self.cachedWidget.setVisible(True if newType else False)
        self.connectedOnlyWidget.setVisible(isMaya)
        self.itemsWidget.setVisible(isMaya and newType in ["EvEnum"])

        self.typeWidget.setStyleSheet("background-color: #6633aa" if isMaya else "")

    def valueKeyPressEvent(self, event):
        def chs_and_attrs(p, stopp, path=""):
            if p == stopp:
                return []

            elements = [path+"/"+a.name for a in p.attributes]
            for ch in p.children:
                items = chs_and_attrs(ch, stopp, path+"/"+ch.name)
                if items:
                    elements += items
                else:
                    break
            return elements
        
        QLineEdit.keyPressEvent(self.valueWidget, event)

        value = unicode(self.valueWidget.text())

        if event.key() == Qt.Key_Slash:
            if len(value) == 1: # just at first character
                tw = PatchMainWindow.browserWidget.leftBrowserWidget.treeWidget
                tw_item = tw.currentItem()

                elements = []
                if tw_item.parent():
                    elements = chs_and_attrs(tw_item.parent().patch, tw_item.patch)

                self.valueWidget.completer().setModel(QStringListModel(elements))
        
    def okClicked(self):
        self.attribute.name = unicode(self.nameWidget.text())
        self.attribute.type = unicode(self.typeWidget.currentText())

        if self.attribute.type in ["EvInt", "EvFloat", "EvDouble", "EvVector"]:
            self.attribute.minValue = unicode(self.minValueWidget.text())
            self.attribute.maxValue = unicode(self.maxValueWidget.text())
        else:
            self.attribute.minValue = ""
            self.attribute.maxValue = ""

        if self.attribute.type in ["EvMatrix", "EvMesh", "EvNurbsCurve", "EvNurbsSurface"]:
            self.attribute.defaultValue = ""
        else:
            self.attribute.defaultValue = unicode(self.valueWidget.text())

        self.attribute.cl = self.clWidget.currentIndex()

        if self.attribute.type in ["EvEnum", "EvCompound"]:
            self.attribute.items = unicode(self.itemsWidget.text())
        else:
            self.attribute.items = ""

        self.attribute.output = self.outputWidget.isChecked() and self.attribute.isMaya()
        self.attribute.keyable = self.keyableWidget.isChecked() and self.attribute.isMaya()
        self.attribute.array = self.arrayWidget.isChecked() and self.attribute.isMaya()
        self.attribute.cached = self.cachedWidget.isChecked()
        self.attribute.hidden = self.hiddenWidget.isChecked() and self.attribute.isMaya()
        self.attribute.connectedOnly = self.connectedOnlyWidget.isChecked() and self.attribute.isMaya()

        self.accept()

    def update(self):
        self.nameWidget.setText(self.attribute.name)
        self.typeWidget.setEditText(self.attribute.type)
        self.valueWidget.setText(self.attribute.defaultValue)
        self.minValueWidget.setText(self.attribute.minValue)
        self.maxValueWidget.setText(self.attribute.maxValue)
        self.clWidget.setCurrentIndex(self.attribute.cl)
        self.itemsWidget.setText(self.attribute.items)

        self.outputWidget.setChecked(self.attribute.output)
        self.keyableWidget.setChecked(self.attribute.isKeyable())
        self.arrayWidget.setChecked(self.attribute.array)
        self.cachedWidget.setChecked(self.attribute.cached)
        self.hiddenWidget.setChecked(self.attribute.hidden)
        self.connectedOnlyWidget.setChecked(self.attribute.connectedOnly)

        self.typeChanged()

class AttributesWidget(QTreeWidget):
    def __init__(self, patch, **kwargs):
        super(AttributesWidget, self).__init__(**kwargs)

        self.patch = patch

        self.itemUnderMouse = None
        self.clipboard = []

        self.attributeNames = ["Name",
                               "Type",
                               "Value",
                               "Min",
                               "Max",
                               "Output",
                               "OpenCL",
                               "Keyable",
                               "Array",
                               "Cached",
                               "Hidden",
                               "Connected",
                               "Items"]

        self.colorsByHeader = {
            "Output": QColor(40, 100, 40),
            "Keyable": QColor(140, 70, 40),
            "Array": QColor(110, 110, 40),
            "Cached": QColor(60, 80, 140),
            "Hidden": QColor(100, 100, 100),
            "Connected": QColor(70, 90, 60)
        }

        self.setHeaderLabels(self.attributeNames)

        if "setSectionResizeMode" in dir(self.header()):
            self.header().setSectionResizeMode(QHeaderView.ResizeToContents) # Qt5
        else:
            self.header().setResizeMode(QHeaderView.ResizeToContents) # Qt4

        self.setSelectionMode(QAbstractItemView.ExtendedSelection) # ExtendedSelection

        self.setDragEnabled(True)
        self.setDragDropMode(QAbstractItemView.InternalMove)
        self.setDropIndicatorShown(True)

        self.setContextMenuPolicy(Qt.DefaultContextMenu)

        self.setIndentation(30)
        # self.setSortingEnabled(True)
        # self.sortItems(0, Qt.AscendingOrder)

        self.setMouseTracking(True)

        self.entered.connect(self.itemEntered)

        self.update()

    def contextMenuEvent(self, event):
        if not self.patch:
            return

        menu = QMenu(self)

        editAction = QAction("Edit\tENTER", self)
        editAction.triggered.connect(lambda: self.editItem(self.currentItem()))
        menu.addAction(editAction)

        replaceValueAction = QAction("Replace value\tR", self)
        replaceValueAction.triggered.connect(self.replaceValue)
        replaceValueAction.setEnabled(len(self.selectedItems()) > 0)
        menu.addAction(replaceValueAction)
        
        addAction = QAction("Add\tINSERT", self)
        addAction.triggered.connect(lambda: self.addTopLevelItem(self.makeItemFromAttribute(Attribute("attr", ""))))
        menu.addAction(addAction)

        removeAction = QAction("Remove\tDEL", self)
        removeAction.triggered.connect(self.removeAttributes)
        removeAction.setEnabled(len(self.selectedItems()) > 0)
        menu.addAction(removeAction)

        menu.addSeparator()
        copyAction = QAction("Copy\tCtrl-C", self)
        copyAction.triggered.connect(self.copyClipboard)
        copyAction.setEnabled(len(self.selectedItems()) > 0)
        menu.addAction(copyAction)

        cutAction = QAction("Cut\tCtrl-X", self)
        cutAction.triggered.connect(self.cutClipboard)
        cutAction.setEnabled(len(self.selectedItems()) > 0)
        menu.addAction(cutAction)

        pasteAction = QAction("Paste\tCtrl-V", self)
        pasteAction.triggered.connect(self.pasteClipboard)
        pasteAction.setEnabled(len(self.clipboard) > 0)
        menu.addAction(pasteAction)

        menu.addSeparator()

        updateAction = QAction("Update\tU", self)
        updateAction.triggered.connect(self.updateWithClipboard)
        updateAction.setEnabled(len(self.clipboard) > 0 and len(self.selectedItems()) > 0)
        menu.addAction(updateAction)

        promoteAction = QAction("Promote\tP", self)
        promoteAction.triggered.connect(self.promoteAttribute)
        promoteAction.setEnabled(len(self.selectedItems()) > 0)
        menu.addAction(promoteAction)

        menu.popup(event.globalPos())

    def replaceValue(self):
        expr, ok = QInputDialog.getText(self, "Patch", "Replace value expression (old=new)", QLineEdit.Normal, "L_=R_")
        if ok:
            exprItems = expr.split("=")
            if len(exprItems) == 2:
                old, new = exprItems
                for item in self.selectedItems():
                    item.attribute.defaultValue = item.attribute.defaultValue.replace(old, new)
            else:
                QMessageBox.critical(self, "Patch", "Invalid expression")    

    def updateWithClipboard(self):
        for item in self.selectedItems():
            oldName = item.attribute.name
            item.attribute = self.clipboard[0].copy()
            item.attribute.name = oldName
            self.updateItem(item)

    def editItem(self, item):
        if not item:
            return

        oldName = item.attribute.name

        shouldRenameVar = len(self.findItems(oldName, Qt.MatchExactly | Qt.MatchRecursive)) == 1

        EditAttributeDialog(item.attribute).exec_()
        self.updateItem(item)

        newName = item.attribute.name

        if not shouldRenameVar:
            return

        # replace variables
        for ct in ["runCode", "deformInitCode", "deformCode", "deformParallelCode", "drawCode", "clCode", "commandsCode"]:
            text = re.sub("@\\b%s\\b"%oldName, "@%s"%newName, self.patch.__getattribute__(ct))
            text = re.sub("@\\b%s_indices\\b"%oldName, "@%s_indices"%newName, text)
            
            self.patch.__setattr__(ct, text)

        PatchMainWindow.browserWidget.rightBrowserWidget.patchCppEditorWidget.update(self.patch)

        tw = PatchMainWindow.browserWidget.leftBrowserWidget.treeWidget
        tw_item = tw.currentItem()

        topLevelItem = tw_item
        while topLevelItem.parent():
            topLevelItem = topLevelItem.parent()

            topLevelItem.patch.renameAbsoluteLinks("%s/%s"%(self.patch.name, oldName), "%s/%s"%(self.patch.name, newName))
            tw_item.patch.renameRelativeLinks(oldName, newName)

            if tw_item.parent():
                tw_item.parent().patch.renameRelativeLinks("%s/%s"%(self.patch.name, oldName), "%s/%s"%(self.patch.name, newName)) # resolves parent relative links

    def updateItem(self, item):
        cl2text = {Attribute.CL_none:"",
                   Attribute.CL_Global:"global",
                   Attribute.CL_Constant:"constant",
                   Attribute.CL_GlobalAlloc:"global alloc",
                   Attribute.CL_LocalAlloc:"local alloc"}

        attr = item.attribute
        item.setText(self.attributeNames.index("Name"), attr.name)
        item.setText(self.attributeNames.index("Type"), attr.type)
        item.setText(self.attributeNames.index("Value"), attr.defaultValue)
        item.setText(self.attributeNames.index("Min"), attr.minValue)
        item.setText(self.attributeNames.index("Max"), attr.maxValue)
        item.setText(self.attributeNames.index("OpenCL"), cl2text[attr.cl])

        for n, a in [("Output", attr.output),
                     ("Keyable", attr.isKeyable()),
                     ("Array", attr.array),
                     ("Cached", attr.cached),
                     ("Hidden", attr.hidden),
                     ("Connected", attr.connectedOnly)]:
            item.setText(self.attributeNames.index(n), n.lower() if a else "")

        flags = Qt.ItemIsSelectable | Qt.ItemIsEditable | Qt.ItemIsEnabled | Qt.ItemIsDragEnabled
        if attr.type == "EvCompound":
            flags |= Qt.ItemIsDropEnabled

        item.setFlags(flags)

        fontMetrics = self.fontMetrics()
        for a in self.attributeNames:
            idx = self.attributeNames.index(a)
            tx = item.text(idx)
            width = fontMetrics.width(tx)

            item.setSizeHint(idx, QSize(width + 5, fontMetrics.height() + 5))

    def makeItemFromAttribute(self, attr):
        item = QTreeWidgetItem()
        item.attribute = attr
        self.updateItem(item)

        if attr.type == "EvCompound":
            for ch in re.split("[ ,]*", attr.items):
                found_attrs = self.patch.findAttribute(ch)
                if found_attrs:
                    ch_item = self.makeItemFromAttribute(found_attrs[0])
                    item.addChild(ch_item)

        else: # EvEnum
            item.setText(self.attributeNames.index("Items"), attr.items)

        return item

    def update(self, patch=None):
        self.patch = patch

        self.clear()

        if not self.patch:
            return

        for a in self.patch.attributes:
            if a.child:
                continue

            item = self.makeItemFromAttribute(a)
            self.addTopLevelItem(item)
            item.setExpanded(True)

    def findItemByAttributeName(self, name, root=None, walk=True):
        if not root:
            root = self.invisibleRootItem()
        elif root.attribute.name == name:
            return root

        for i in range(root.childCount()):
            ch = root.child(i)
            if ch.attribute.name == name:
                return ch
            elif walk:
                res = self.findItemByAttributeName(name, ch)
                if res:
                    return res

    def itemEntered(self, modelIdx):
        item = self.itemFromIndex(modelIdx)
        self.itemUnderMouse = item
        self.repaint()

    def leaveEvent(self, event):
        self.itemUnderMouse = None
        self.repaint()

    def dragEnterEvent(self, event):
        if event.mouseButtons() == Qt.MiddleButton:
            QTreeWidget.dragEnterEvent(self, event)
            self.dragItems = self.selectedItems()

    def dragMoveEvent(self, event):
        QTreeWidget.dragMoveEvent(self, event)

    def dropEvent(self, event):
        QTreeWidget.dropEvent(self, event)

        idx = self.indexFromItem(self.dragItems[0]).row()
        for item in self.dragItems:
            parent = item.parent()
            if parent:
                parent.removeChild(item)
                parent.insertChild(idx, item)
            else:
                self.invisibleRootItem().removeChild(item)
                self.insertTopLevelItem(idx, item)

        self.dragItems = []

    def drawRow(self, painter, options, modelIdx):
        def drawText(rect, text, foreground=None, background=None, align=Qt.AlignCenter):
            painter.save()
            if foreground:
                painter.setPen(foreground)
            if background:
                brush = QBrush(background)
                brush.setStyle(Qt.SolidPattern)
                painter.fillRect(rect, brush)
            painter.drawText(rect, align, text)
            painter.restore()

        painter.save()

        rect = self.visualRect(modelIdx)
        item = self.itemFromIndex(modelIdx)

        indent = self.indentation()

        if rect.width() < 0:
            return

        painter.setPen(QPen(QBrush(QColor(60, 60, 60)), 1, Qt.SolidLine))
        numberBranch = rect.x() / indent
        if numberBranch > 1:
            for i in range(1, numberBranch):
                plusInt = i * indent + 10
                x = rect.x() - plusInt
                painter.drawLine(x, rect.y(), x, rect.y() + rect.height())

        parentIndex = modelIdx.parent()
        parentItem = self.itemFromIndex(parentIndex)

        if item.childCount() and rect.x() + rect.width() > rect.x():
            painter.setPen(QPen(QBrush(QColor(100, 100, 100)), 1, Qt.SolidLine))
            painter.fillRect(QRect(rect.x() - 16, rect.y() + 4, 12, 12), QColor(45, 45, 45))
            painter.drawRect(rect.x() - 16, rect.y() + 4, 12, 12)
            painter.setPen(QPen(QBrush(QColor(120, 120, 120)), 1, Qt.SolidLine))
            if item.isExpanded():
                painter.drawLine(rect.x() - 7, rect.y() + 10, rect.x() - 13, rect.y() + 10)
            else:
                painter.drawLine(rect.x() - 10, rect.y() + 7, rect.x() - 10, rect.y() + 14)
                painter.drawLine(rect.x() - 7, rect.y() + 10, rect.x() - 13, rect.y() + 10)

        attr = item.attribute

        if self.itemUnderMouse is item:
            painter.setPen(QColor(255, 255, 255))
        else:
            painter.setPen(QColor(210, 210, 210))

        if attr.muted:
            painter.setPen(QColor(100, 100, 100))

        background = None
        idx = modelIdx.sibling(modelIdx.row(), self.attributeNames.index("Name"))
        rect = self.visualRect(idx)
        if not attr.isEmpty() and (not re.match("^[_a-zA-Z]\\w*$", attr.name) or
                                   len([_item for _item in self.patch.findAttribute(attr.name)]) > 1):
            background = QColor(130, 40, 40)
        drawText(rect, attr.name, None, background, Qt.AlignLeft | Qt.AlignVCenter)

        idx = modelIdx.sibling(modelIdx.row(), self.attributeNames.index("Type"))
        rect = self.visualRect(idx)
        background = None
        if not attr.muted:
            if attr.isMaya():
                painter.fillRect(rect, QBrush(QColor(90, 50, 140)))
            else: # c++
                r = re.search(Attribute.TypeRefExp, attr.type)
                if r:
                    link, typ = r.groups("")
                    if link:
                        tw = PatchMainWindow.browserWidget.leftBrowserWidget.treeWidget
                        tw_item = tw.currentItem()

                        topLevelItem = tw_item
                        while topLevelItem.parent():
                            topLevelItem = topLevelItem.parent()

                        matchedPatches = None

                        if link.startswith("/"): # parent relative link
                            if tw_item.parent():
                                matchedPatches = tw_item.parent().patch.resolveLink(link, attribute=False)

                        elif link.startswith("$"): # absolute link
                            matchedPatches = topLevelItem.patch.resolveLink(link, attribute=False)

                        background = QColor(30, 80, 30) if matchedPatches else QColor(80, 30, 30)

        drawText(rect, attr.type, None, background)

        background = None
        foreground = None
        idx = modelIdx.sibling(modelIdx.row(), self.attributeNames.index("Value"))
        rect = self.visualRect(idx)
        if not attr.muted and not attr.isEmpty():
            if attr.isReference():
                tw = PatchMainWindow.browserWidget.leftBrowserWidget.treeWidget
                tw_item = tw.currentItem()

                if tw_item:
                    topLevelItem = tw_item
                    while topLevelItem.parent():
                        topLevelItem = topLevelItem.parent()

                    matchedAttrs = []

                    if attr.defaultValue.startswith("/"): # parent relative link
                        if tw_item.parent():
                            _, matchedAttrs = tw_item.parent().patch.resolveLink(attr.defaultValue)

                    elif attr.defaultValue.startswith("$"): # absolute link
                        _, matchedAttrs = topLevelItem.patch.resolveLink(attr.defaultValue)

                    if attr.defaultValue.startswith("/") or attr.defaultValue.startswith("$"):
                        if matchedAttrs:
                            background = QColor(30, 80, 30)
                            tooltip = "%s (%s) = '%s'"%(matchedAttrs[0].name, matchedAttrs[0].type, matchedAttrs[0].defaultValue)
                            item.setToolTip(self.attributeNames.index("Value"),tooltip)
                        else:
                            background = QColor(80, 30, 30)

            if attr.defaultValue and attr.defaultValue.startswith("\""):
                foreground = QColor(110, 180, 110)
        drawText(rect, attr.defaultValue, foreground, background)

        idx = modelIdx.sibling(modelIdx.row(), self.attributeNames.index("Min"))
        rect = self.visualRect(idx)
        drawText(rect, attr.minValue)

        idx = modelIdx.sibling(modelIdx.row(), self.attributeNames.index("Max"))
        rect = self.visualRect(idx)
        drawText(rect, attr.maxValue)

        painter.setPen(QColor(200, 200, 200))
        for a in ["Output",
                  "Keyable",
                  "Array",
                  "Cached",
                  "Hidden",
                  "Connected"]:
            n = self.attributeNames.index(a)
            idx = modelIdx.sibling(modelIdx.row(), self.attributeNames.index(a))
            rect = self.visualRect(idx)

            background = self.colorsByHeader[a] if not attr.muted and item.text(n) else None
            drawText(rect, item.text(n), None, background)

        idx = modelIdx.sibling(modelIdx.row(), self.attributeNames.index("Items"))
        rect = self.visualRect(idx)
        painter.drawText(rect, Qt.AlignCenter, item.text(self.attributeNames.index("Items")))

        idx = modelIdx.sibling(modelIdx.row(), self.attributeNames.index("OpenCL"))
        rect = self.visualRect(idx)

        if attr.cl != Attribute.CL_none:
            if not attr.muted:
                painter.fillRect(rect, QBrush(QColor(50, 100, 170)))
            painter.drawText(rect, Qt.AlignCenter, item.text(self.attributeNames.index("OpenCL")))

        painter.setPen(QColor(70, 70, 70))
        for a in self.attributeNames:
            idx = modelIdx.sibling(modelIdx.row(), self.attributeNames.index(a))
            rect = self.visualRect(idx)
            painter.drawLine(rect.left(), rect.bottom(), rect.right(), rect.bottom())
            painter.drawLine(rect.right(), rect.top(), rect.right(), rect.bottom())

            if idx.row() % 2 == 0:
                painter.fillRect(rect, QBrush(QColor(100, 100, 100, 30)))

        # set selected style
        painter.setPen(QColor(130, 130, 130))
        if modelIdx in self.selectedIndexes():
            for a in self.attributeNames:
                idx = modelIdx.sibling(modelIdx.row(), self.attributeNames.index(a))
                rect = self.visualRect(idx)
                painter.drawLine(rect.left(), rect.top(), rect.right(), rect.top())
                painter.drawLine(rect.left(), rect.bottom(), rect.right(), rect.bottom())
                painter.fillRect(rect.x()-1, rect.y(), rect.width(), rect.height()-1, QColor(100, 105, 145, 50))

        painter.restore()

    def mouseDoubleClickEvent(self, event):
        index = self.indexAt(event.pos())
        if not index.isValid():
            return

        item = self.itemFromIndex(index)
        column = index.column()

        if column in [self.attributeNames.index("Name"),
                      self.attributeNames.index("Type"),
                      self.attributeNames.index("Value"),
                      self.attributeNames.index("Min"),
                      self.attributeNames.index("Max"),
                      self.attributeNames.index("OpenCL"),
                      self.attributeNames.index("Items")]:
            self.editItem(item)

        else:
            if column == self.attributeNames.index("Cached"):
                item.attribute.cached = not item.attribute.cached

            if item.attribute.type in Attribute.EvTypes:
                if column == self.attributeNames.index("Output"):
                    item.attribute.output = not item.attribute.output
                if column == self.attributeNames.index("Keyable"):
                    item.attribute.keyable = not item.attribute.keyable
                if column == self.attributeNames.index("Array"):
                    item.attribute.array = not item.attribute.array
                if column == self.attributeNames.index("Hidden"):
                    item.attribute.hidden = not item.attribute.hidden
                if column == self.attributeNames.index("Connected"):
                    item.attribute.connectedOnly = not item.attribute.connectedOnly

            self.updateItem(item)

    def bakeAttributes(self, root):
        for i in range(root.childCount()):
            ch = root.child(i)
            parent = ch.parent()

            isChild = 1 if parent and parent.attribute.type == "EvCompound" else 0
            ch.attribute.child = isChild

            if ch.attribute.type == "EvCompound":
                childrenNames = []
                for k in range(ch.childCount()):
                    childrenNames.append(ch.child(k).attribute.name)
                ch.attribute.items = ", ".join(childrenNames)

                self.bakeAttributes(ch)

            self.patch.attributes.append(ch.attribute)

    def saveAttributes(self):
        if not self.patch:
            return

        self.patch.attributes = []
        self.bakeAttributes(self.invisibleRootItem())

    def focusOutEvent(self, event):
        self.saveAttributes()

    def promoteAttribute(self):
        tw = PatchMainWindow.browserWidget.leftBrowserWidget.treeWidget
        tw_item = tw.currentItem()
        if tw_item.parent():
            attr = self.currentItem().attribute

            pubAttrName = "%s_%s"%(self.patch.name, attr.name)

            if len(tw_item.parent().patch.findAttribute(pubAttrName)) == 0: # promote
                ok = QMessageBox.question(self, "Patch", "Promote '%s' attribute?"%attr.name, QMessageBox.Yes and QMessageBox.No, QMessageBox.Yes) == QMessageBox.Yes
                if ok:
                    pubAttr = attr.copy()
                    pubAttr.name = pubAttrName

                    if attr.isMaya(): # invert type to c++
                        attr.type = Attribute.EvTypesToCpp.get(attr.type)

                    if attr.isReference():
                        pubAttr.defaultValue = ""

                    tw_item.parent().patch.attributes.append(pubAttr)

                    attr.defaultValue = "/%s"%pubAttr.name

            else: # unpromote
                ok = QMessageBox.question(self, "Patch", "Found promoted attribute! Unpromote it?", QMessageBox.Yes and QMessageBox.No, QMessageBox.Yes) == QMessageBox.Yes
                if ok:
                    pubAttr = tw_item.parent().patch.findAttribute(pubAttrName)[0]
                    tw_item.parent().patch.removeAttribute(pubAttrName)
                    attr.defaultValue = pubAttr.defaultValue
                    attr.type = pubAttr.type
        else:
            QMessageBox.critical(self, "Patch", "Cannot promote. There is no parent found")

    def removeAttributes(self):
        names = ", ".join([item.attribute.name for item in self.selectedItems()])
        ok = QMessageBox.question(self, "Patch", "Remove selected attributes?\n%s"%names, QMessageBox.Yes and QMessageBox.No, QMessageBox.Yes) == QMessageBox.Yes
        if ok:
            for item in self.selectedItems():
                (item.parent() or self.invisibleRootItem()).removeChild(item)

    def expandRecursive(self, item, v):
        item.setExpanded(v)
        for i in range(item.childCount()):
            ch = item.child(i)
            ch.setExpanded(v)
            self.expandRecursive(ch, v)

    def copyClipboard(self):
        self.clipboard = []
        for item in self.selectedItems():
            self.clipboard.append(item.attribute.copy())

    def pasteClipboard(self):
        for a in self.clipboard:
            item = self.makeItemFromAttribute(a.copy())
            self.addTopLevelItem(item)

        self.clipboard = []

    def cutClipboard(self):
        self.clipboard = []
        for item in self.selectedItems():
            self.clipboard.append(item.attribute.copy())
            (item.parent() or self.invisibleRootItem()).removeChild(item)

    def keyPressEvent(self, event):
        shift = event.modifiers() & Qt.ShiftModifier
        ctrl = event.modifiers() & Qt.ControlModifier
        alt = event.modifiers() & Qt.AltModifier
        key = event.key()

        if key in [Qt.Key_N, Qt.Key_Insert]: # new
            self.addTopLevelItem(self.makeItemFromAttribute(Attribute("attr", "")))

        elif ctrl and key == Qt.Key_C: # copy
            self.copyClipboard()

        elif ctrl and key == Qt.Key_V: # paste
            self.pasteClipboard()

        elif ctrl and key == Qt.Key_X: # cut
            self.cutClipboard()

        elif ctrl and key == Qt.Key_D: # duplicate
            for item in self.selectedItems():
                newItem = self.makeItemFromAttribute(item.attribute.copy())
                parent = item.parent()

                if parent:
                    parent.insertChild(parent.indexOfChild(item)+1, newItem)
                else:
                    self.insertTopLevelItem(self.indexOfTopLevelItem(item)+1, newItem)

        elif ctrl and event.key() == Qt.Key_Space:
            for item in self.selectedItems():
                v = not item.isExpanded()
                self.expandRecursive(item, v)

        elif event.key() == Qt.Key_Space:
            for item in self.selectedItems():
                item.setExpanded(not item.isExpanded())

        elif key == Qt.Key_Return:
            self.editItem(self.currentItem())

        elif key == Qt.Key_Delete: # remove
            self.removeAttributes()

        elif key == Qt.Key_M: # mute
            for item in self.selectedItems():
                item.attribute.muted = not item.attribute.muted

        elif key == Qt.Key_U: # update
            self.updateWithClipboard()

        elif key == Qt.Key_P: # promote
            self.promoteAttribute()

        elif key == Qt.Key_R: # replace value
            self.replaceValue()
            
        elif key == Qt.Key_F12: # full screen
            PatchMainWindow.browserWidget.leftBrowserWidget.setVisible(not PatchMainWindow.browserWidget.leftBrowserWidget.isVisible())
            PatchMainWindow.browserWidget.rightBrowserWidget.patchCppEditorWidget.setVisible(not PatchMainWindow.browserWidget.rightBrowserWidget.patchCppEditorWidget.isVisible())

class RightBrowserWidget(QWidget):
    def __init__(self, **kwargs):
        super(RightBrowserWidget, self).__init__(**kwargs)

        self.clipboard = []
        self.undoQueue = []
        self.keepHistory = True

        self.patch = None
        self.currentFontPointSize = 14

        layout = QVBoxLayout()
        self.setLayout(layout)
        layout.setMargin(0)

        self.patchCppEditorWidget = PatchCppEditorWidget(self.patch)

        self.attributesWidget = AttributesWidget(self.patch)

        self.logWidget = LogWidget()

        splitter = QSplitter(Qt.Vertical)
        splitter.addWidget(self.attributesWidget)
        splitter.addWidget(self.patchCppEditorWidget)
        splitter.addWidget(self.logWidget)

        splitter.setSizes([500, 400, 200])

        layout.addWidget(splitter)

        self.update()

    def update(self, patch=None):
        if patch:
            self.patch = patch

        self.patchCppEditorWidget.update(self.patch)
        self.attributesWidget.update(self.patch)

class BrowserWidget(QWidget):
    def __init__(self, **kwargs):
        super(BrowserWidget, self).__init__(**kwargs)

        layout = QHBoxLayout()
        self.setLayout(layout)
        layout.setMargin(0)

        self.leftBrowserWidget = LeftBrowserWidget()
        self.rightBrowserWidget = RightBrowserWidget()

        splitter = QSplitter(Qt.Horizontal)
        splitter.addWidget(self.leftBrowserWidget)
        splitter.addWidget(self.rightBrowserWidget)

        splitter.setSizes([100, 900])

        layout.addWidget(splitter)

class MainWindow(QFrame):
    def __init__(self, **kwargs):
        super(MainWindow, self).__init__(**kwargs)
        self.setWindowTitle("Patch")
        self.setGeometry(400, 100, 1200, 800)

        # self.setWindowFlags(QFrame().windowFlags() | Qt.WindowStaysOnTopHint)

        layout = QVBoxLayout()
        self.setLayout(layout)

        self.browserWidget = BrowserWidget()

        layout.addWidget(self.browserWidget)

    def closeEvent(self, event):
        if self.browserWidget.leftBrowserWidget.treeWidget.topLevelItemCount() > 0:
            ok = QMessageBox.question(self, "Patch", "Really exit?", QMessageBox.Yes and QMessageBox.No, QMessageBox.Yes) == QMessageBox.Yes
            if not ok:
                event.ignore()

if not os.path.exists(EvaluatorLocalPath):
    os.makedirs(EvaluatorLocalPath+"/patch/patches/__backup__")
    os.makedirs(EvaluatorLocalPath+"/patch/temp")
    os.makedirs(EvaluatorLocalPath+"/nodes")

    shutil.copytree(EvaluatorServerPath+"/clang", EvaluatorLocalPath+"/clang")
    shutil.copyfile(EvaluatorServerPath+"/.clang-format", EvaluatorLocalPath+"/.clang-format")

if __name__ == "__main__":
    app = QApplication([])

    with open(EvaluatorServerPath+"/patch/qss/qstyle.qss", "r") as f:
        iconsDir = (EvaluatorServerPath+"/patch/qss/icons/").replace("\\","/")
        style = f.read().replace("icons/", iconsDir)
        app.setStyleSheet(style)

    PatchMainWindow = MainWindow()
    PatchMainWindow.show()
    app.exec_()

else:
    PatchMainWindow = MainWindow()

    with open(EvaluatorServerPath+"/patch/qss/qstyle.qss", "r") as f:
        iconsDir = (EvaluatorServerPath+"/patch/qss/icons/").replace("\\","/")
        style = f.read().replace("icons/", iconsDir)
        PatchMainWindow.setStyleSheet(style)

    # PatchMainWindow.show()
