
import pymel.core as core

class Network:
    """
    A class for working with a network - a network of attributes linked to scene elements.

    Represents a network node with various attributes.
    Used to store connections between different objects in the scene and additional information.
    """
    __network = None

    def __init__(self, name, createAnyway=False):
        """
        Initialization.

        Parameters:
        name (str): The name of the network.
        createAnyway (bool): Whether to create the network if it does not exist.
        """
        if not core.objExists(name) and createAnyway:
            self.create(name)
        else:
            self.set(name)

    def set(self, name):
        """
        Set the network to work with.

        Parameters:
        name (str): The name of the network.
        """
        if not core.objExists(name):
            core.displayError("Network: set() cannot find '" + name + "' network")
        else:
            self.__network = core.PyNode(name)

    def get(self):
        """
        Return the network as a PyNode.

        Returns:
        core.PyNode: The network node.
        """
        return self.__network

    def name(self):
        """
        Return the name of the network.

        Returns:
        str: The name of the network node.
        """
        if not core.objExists(self.__network):
            core.displayError("Network: name() cannot find '" + self.__network + "' network")
        else:
            return self.__network.name()

    def create(self, name):
        """
        Create a network.

        Parameters:
        name (str): The name of the network.
        """
        if core.objExists(name):
            core.displayError("Network: create() cannot create '" + name + "' network. Object with this name already exists")
        else:
            self.__network = core.createNode("network", n=name)

    def delete(self):
        """
        Delete the network.
        """
        if not core.objExists(self.__network):
            core.displayError("Network: delete() cannot delete '" + self.__network + "' network. Object does not exist")
        else:
            core.delete(self.__network)

    def setAttr(self, name, value="", typ="connection", lockState=False):
        """
        Add or set an attribute.

        Parameters:
        name (str): The name of the attribute.
        value (str, int, float, bool): The value of the attribute.
        typ (str): The type of the attribute ("connection" for a connection, "value" for a value).
        lockState (bool): The lock state (True to lock, False to unlock).
        """
        if not core.objExists(self.__network):
            return

        if not self.__network.hasAttr(name):
            if typ == "connection":
                self.__network.addAttr(name, at="message")

            elif typ == "value":
                if isinstance(value, str):
                    self.__network.addAttr(name, dt="string")

                elif isinstance(value, bool):
                    self.__network.addAttr(name, at="bool")

                elif isinstance(value, int):
                    self.__network.addAttr(name, at="long")

                elif isinstance(value, float):
                    self.__network.addAttr(name, at="float")
            else:
                return

        if self.__network.attr(name).type() == "message":
            if core.objExists(value):
                self.__network.attr(name).disconnect()
                core.PyNode(value) >> self.__network.attr(name)
            else:
                return
        else:
            self.__network.attr(name).set(value)

        self.__network.attr(name).setLocked(lockState)

    def getAttr(self, name):
        """
        Get the value of an attribute.

        Parameters:
        name (str): The name of the attribute.

        Returns:
        The node and attribute if the attribute has a connection.
        The value if the attribute holds just a value.
        """
        if not core.objExists(self.__network):
            return

        if not self.__network.hasAttr(name):
            return

        lst = self.__network.attr(name).listConnections(p=True)

        return lst[0] if lst else self.__network.attr(name).get()

    def deleteAttr(self, name):
        """
        Delete an attribute.

        Parameters:
        name (str): The name of the attribute.
        """
        if not core.objExists(self.__network):
            return

        if not self.__network.hasAttr(name):
            return

        self.__network.attr(name).delete()

    def listAttrs(self):
        """
        List all attributes of the network.

        Returns:
        list: A list of attributes.
        """
        return self.__network.listAttr(ud=True) if core.objExists(self.__network) else []

def getNetwork(mayaObjName):
    """
    Get the network for the specified Maya object.

    Parameters:
    mayaObjName (str): The object to query.

    Returns:
    Network or None: The network object or None if not found.
    """
    node = core.PyNode(mayaObjName)
    if node.hasAttr("message"):
        messages = node.attr("message").connections(type='network', plugs=True)
        if messages:
            return Network(core.PyNode(messages[0]).node())
