from maya.OpenMayaUI import MQtUtil
import shiboken2
from Qt.QtCore import QEvent, Qt, QObject, QMargins, Slot
from Qt.QtWidgets import QApplication, QWidget, QMenu, QPushButton, QAction

import pymel.core as pm
import enum

from . import dynamicParent
# import wizart.cef_picker as cef_picker
from . import network
from . import mirror
from . import bezierRig
from . import general
from . import kinematicSwitch


def undoable(fn):
    def wrapped(*args, **kwargs):
        pm.cmds.undoInfo(openChunk=True)
        try:
            return fn(*args, **kwargs)
        finally:
            pm.cmds.undoInfo(closeChunk=True)

    wrapped.__name__ = fn.__name__
    wrapped.__doc__ = fn.__doc__
    return wrapped


class AppType(enum.Enum):
    Qt = 0
    Maya = 1

class MouseButtonReleaseEventEater(QObject):
    def __init__(self, parent=None):
        super(MouseButtonReleaseEventEater, self).__init__(parent)

    def eventFilter(self, obj, event):
        if event.type() == event.MouseButtonRelease and event.button() == Qt.RightButton:
            obj.click()

            return True

        return QObject.eventFilter(self, obj, event)


class AbstractMenuFactory(object):
    def addAction(self, parent, name, cmd, btn=False, btnCmd=None, boldFont=False):
        return None

    def addMenu(self, parent, name):
        return None

    def addSeparator(self, parent):
        pass


class QtMenuFactory(AbstractMenuFactory):
    def __init__(self):
        super(QtMenuFactory, self).__init__()

    def addAction(self, parent, name, cmd, btn=False, btnCmd=None, boldFont=False):
        cmd_fn = undoable(cmd) if cmd else None
        act = parent.addAction(name, cmd_fn)
        if btn:
            rect = parent.actionGeometry(act)
            rect.setLeft(rect.width() - rect.height())

            btn = QPushButton("X", parent)
            btn.setStyleSheet("color: black; background-color: rgb(177, 76, 73);")
            btn.setGeometry(rect.marginsRemoved(QMargins(2, 2, 2, 2)))
            btn.released.connect(parent.hide)
            btn.released.connect(undoable(btnCmd))
            btn.installEventFilter(MouseButtonReleaseEventEater(btn))

        if boldFont:
            font = act.font()
            font.setBold(True)
            act.setFont(font)

        return act

    def addMenu(self, parent, name):
        return parent.addMenu(name)

    def addSeparator(self, parent):
        parent.addSeparator()


class MayaMenuFactory(AbstractMenuFactory):
    def __init__(self):
        super(MayaMenuFactory, self).__init__()

    def addAction(self, parent, name, cmd, btn=False, btnCmd=None, boldFont=False):
        pm.setParent(parent, menu=True)
        if boldFont:
            if cmd:
                act = pm.menuItem(label=name, command=lambda x: cmd(), boldFont=boldFont)
            else:
                act = pm.menuItem(label=name, boldFont=boldFont)
        else:
            if cmd:
                act = pm.menuItem(label=name, command=lambda x: cmd())
            else:
                act = pm.menuItem(label=name)

        if btn and btnCmd:
            pm.menuItem(optionBox=True, command=lambda x: btnCmd())

        return act

    def addMenu(self, parent, name):
        pm.setParent(parent, menu=True)
        return pm.menuItem(label=name, subMenu=True, allowOptionBoxes=True)

    def addSeparator(self, parent):
        pm.setParent(parent, menu=True)
        pm.menuItem(divider=True)


def addDynamicActions(menuFactory, contextMenu, objNode):
    objName = objNode.name()

    subMenu = menuFactory.addMenu(contextMenu, "Dynamic parents")
    menuFactory.addAction(subMenu, "(no parent)", lambda: dynamicParent.dynamicParentSwitch(objName, ''))

    dyn = dynamicParent.DynamicParent(objNode.dynParent.get(), objNode)

    pc = dyn.getParentConstraint()
    if pc:
        for tg in pc.getTargetList():
            menuItemCommand = lambda: dynamicParent.dynamicParentSwitch(objName, tg)
            menuItemOptionBoxCommand = lambda: dynamicParent.deleteFromDynamicParent(objName, tg)
            menuFactory.addAction(subMenu, tg.nodeName(), menuItemCommand, True, menuItemOptionBoxCommand)

    if pm.ls(sl=True):
        menuFactory.addAction(subMenu, "Add selected", lambda: dynamicParent.addSelectedToDynamicParent(objName))

    menuFactory.addAction(contextMenu, "Dynamic Parent Bake", lambda: dynamicParent.dynamicParentBake(objName))

def showPicker(objName):
    exec("import wizart.cef_picker;wizart.cef_picker.show_picker_for_node('%s')" % objName) in globals()
    # cef_picker.show_picker_for_node(objName)

def menuByType(menuFactory, contextMenu, typ, objNode, networkNode):
    namespace = objNode.namespace()
    objName = objNode.name()

    if typ == "main":
        menuFactory.addAction(contextMenu, "Reset controls", lambda: general.resetControls(namespace))
        menuFactory.addAction(contextMenu, "Select controls", lambda: general.selectControls(namespace))

    elif typ in ["arm", "wing", "dragonWing", "arm_hier", "biped_rig_arm", "arm_1", "arm_2", "arm_3", "arm_4"]:
        menuFactory.addAction(contextMenu, "IK/FK Switch", lambda: kinematicSwitch.kinematicSwitchMulti(objName))
        menuFactory.addAction(contextMenu, "Mirror", lambda: mirror.mirrorByNetwork(networkNode.name()))
        menuFactory.addAction(contextMenu, "Flip", lambda: mirror.mirrorByNetwork(networkNode.name(), True))

    elif typ in ["leg", "frontLeg", "backLeg", "midLeg","leg_hier", "biped_rig_leg", "leg_1", "leg_2", "leg_3", "leg_4"]:
        menuFactory.addAction(contextMenu, "IK/FK Switch", lambda: kinematicSwitch.kinematicSwitchMulti(objName))
        menuFactory.addAction(contextMenu, "Mirror", lambda: mirror.mirrorByNetwork(networkNode.name()))
        menuFactory.addAction(contextMenu, "Flip", lambda: mirror.mirrorByNetwork(networkNode.name(), True))

    elif typ == "spine":
        menuFactory.addAction(contextMenu, "Flip", lambda: mirror.mirrorByNetwork(networkNode.name()))

    elif typ in ["head", "head_hier", "biped_rig_head"]:
        menuFactory.addAction(contextMenu, "IK/FK Switch", lambda: kinematicSwitch.kinematicSwitchMulti(objName))
        menuFactory.addAction(contextMenu, "FollowNeck Switch", lambda: general.seamlessSwitchAttr(objName, 'followNeck'))
        menuFactory.addAction(contextMenu, "FollowBody Switch", lambda: general.seamlessSwitchAttr(objName, 'followBody'))
        menuFactory.addAction(contextMenu, "Flip", lambda: mirror.mirrorByNetwork(networkNode.name()))

    elif typ == "fingers":
        menuFactory.addAction(contextMenu, "Mirror", lambda: mirror.mirrorByNetwork(networkNode.name()))
        menuFactory.addAction(contextMenu, "Flip", lambda: mirror.mirrorByNetwork(networkNode.name(), True))

    elif typ in ["finger", "finger_hier", "biped_rig_fingers"]:
        menuFactory.addAction(contextMenu, "IK/FK Switch", lambda: kinematicSwitch.kinematicSwitchMulti(objName))

    elif typ == "bezierRig":
        subMenu = menuFactory.addMenu(contextMenu, "Dynamic IK/FK")
        menuFactory.addAction(subMenu, "To FK selected", lambda: bezierRig.bezierRig_switchToSelected(objName, True, False))
        menuFactory.addAction(subMenu, "To FK down", lambda: bezierRig.bezierRig_switchToSelected(objName, True, True))
        menuFactory.addSeparator(subMenu)
        menuFactory.addAction(subMenu, "To IK selected", lambda: bezierRig.bezierRig_switchToSelected(objName, False, False))
        menuFactory.addAction(subMenu, "To IK down", lambda: bezierRig.bezierRig_switchToSelected(objName, False, True))

def fillMenu(menuFactory, contextMenu, mayaObjName, inPicker=False):
    objNode = pm.PyNode(mayaObjName)
    networkNode = network.getNetwork(objNode)

    typ = networkNode.getAttr("type") if networkNode else ""

    if networkNode:
        menuByType(menuFactory, contextMenu, typ, objNode, networkNode)

    if typ == "main":
        visibleSets = pm.ls("%s*_visible_set" % objNode.namespace(), type="objectSet")
        if visibleSets:
            subMenu = menuFactory.addMenu(contextMenu, "Visible geometry")
            menuFactory.addAction(subMenu, "Show all", lambda: general.showAllVisibleSets(visibleSets))
            menuFactory.addAction(subMenu, "Hide all", lambda: general.hideAllVisibleSets(visibleSets))
            menuFactory.addSeparator(subMenu)

            for visibleSet in visibleSets:
                menuFactory.addAction(subMenu, visibleSet.stripNamespace().replace("_visible_set", ""), (lambda j: lambda: general.toggleVisible(j))(visibleSet))

    if objNode.hasAttr("dynParent"):
        addDynamicActions(menuFactory, contextMenu, objNode)

    if pm.objExists(objNode.namespace() + "rig"):
        menuFactory.addSeparator(contextMenu)
        menuFactory.addAction(contextMenu, "Reset Translation", lambda: general.reset_selected_controls(general.ResetControlMode.TRANSLATE))
        menuFactory.addAction(contextMenu, "Reset Rotation", lambda: general.reset_selected_controls(general.ResetControlMode.ROTATE))
        menuFactory.addAction(contextMenu, "Reset Scale", lambda: general.reset_selected_controls(general.ResetControlMode.SCALE))
        menuFactory.addAction(contextMenu, "Reset Translation Rotation Scale", lambda: general.reset_selected_controls(general.ResetControlMode.TRANSLATE | general.ResetControlMode.ROTATE | general.ResetControlMode.SCALE))
        menuFactory.addSeparator(contextMenu)
    if pm.objExists(objNode.namespace() + "rig") and not inPicker:
        menuFactory.addAction(contextMenu, "Show Picker", lambda: showPicker(mayaObjName))


def createContextMenu(contextMenu, mayaObjName, appType, inPicker=False):
    if appType is AppType.Qt:
        menuFactory = QtMenuFactory()
    elif appType is AppType.Maya:
        menuFactory = MayaMenuFactory()
    else:
        return

    # create title item
    fullSelection = set(pm.ls(sl=True) + [pm.PyNode(mayaObjName)])
    title = "[ %s... (MULTI) ]" if len(fullSelection) > 1 else "[ %s ]"

    menuFactory.addAction(contextMenu, title % mayaObjName, None, boldFont=True)

    # add others actions
    fillMenu(menuFactory, contextMenu, mayaObjName, inPicker)

