import pymel.core as core


import enum
class ResetControlMode(enum.IntEnum):
      NONE = 0
      TRANSLATE = 1 << 0
      ROTATE = 1 << 1
      SCALE = 1 << 2
      OTHER = 1 << 3
      ALL = TRANSLATE | ROTATE |SCALE | OTHER

def resetControl(control, reset_mode = ResetControlMode.ALL):
    """
    Resets the specified control's attributes to their default values.

    Parameters:
    control (core.PyNode): The control object whose attributes need to be reset.
    reset_mode (ResetControlMode): bitwise flag that can be used to affect TRANSLATE, ROTATE, SCALE, OTHER channels. ALL by default

    """
    if core.objExists(control):
        control = core.PyNode(control)

        for a in control.listAttr(k=True, w=True, u=True, s=True):
            if a.longName() in ["visibility", "rotateOrder"]:
                continue
            
            if a.isDynamic() and reset_mode & ResetControlMode.OTHER:
                a.set(core.addAttr(a, q=True, dv=True))  # Reset dynamic attributes to their default values
            else:
                if a.shortName() in ('tx', 'ty', 'tz') and not reset_mode & ResetControlMode.TRANSLATE:
                    continue
                elif a.shortName() in ('rx', 'ry', 'rz') and not reset_mode & ResetControlMode.ROTATE:
                    continue
                elif a.shortName() in ('sx', 'sy', 'sz') and not reset_mode & ResetControlMode.SCALE:
                    continue

                if a.shortName() in ["sx", "sy", "sz"]:
                    a.set(1)  # Set scale attributes to 1
                else:
                    a.set(0)  # Set static attributes to 0
    else:
        core.displayError("resetControl: '" + control + "' is an invalid node")

def reset_selected_controls(reset_mode = ResetControlMode.ALL):
    """
    convienience wrapper for the ui tools
    """
    for ctrl in core.ls(sl=True):
        resetControl(ctrl, reset_mode)

def resetControls(namespace, skip=["main_control"]):
    """
    Resets all control attributes within the specified namespace to their default values,
    except for those controls specified in the skip list.

    Parameters:
    namespace (str): The namespace containing the controls to be reset.
    skip (list): A list of control names to be skipped during the reset process. Defaults to ["main_control"].

    """
    ls = core.ls(namespace + "*_control")
    skip = [core.PyNode(namespace + o) for o in skip if core.objExists(namespace + o)]

    for ctrl in set(ls) - set(skip):
        resetControl(ctrl)


def selectControls(namespace, skip=[]):
    """
    Selects all controls within the specified namespace, excluding those specified in the skip list.

    Parameters:
    namespace (str): The namespace containing the controls to be selected.
    skip (list): A list of control names to be skipped during the selection process. Defaults to an empty list.

    """
    ls = core.ls(namespace + "*_control")
    skip = [core.PyNode(namespace + o) for o in skip if core.objExists(namespace + o)]

    core.select(set(ls) - set(skip))


def lockAttr(attr, val):
    """
    Lock or hide an attribute in the channel box.

    Parameters:
    attr (core.PyNode): The attribute to be modified.
    val (int): The value indicating the desired state of the attribute.
               - 1: Fully lock (lock, hide, and make non-keyable).
               - 0.5: Make non-keyable, unlock, and show.
               - 0: Make keyable and unlock.

    """
    if val == 1:  # Fully lock (lock, hide, and make non-keyable)
        attr.setKeyable(False)
        attr.setLocked(True)
        attr.showInChannelBox(False)

    elif val == 0.5:  # Make non-keyable, unlock, and show
        attr.setKeyable(False)
        attr.setLocked(False)
        attr.showInChannelBox(True)

    elif val == 0:  # Make keyable and unlock
        attr.setLocked(False)
        attr.showInChannelBox(True)
        attr.setKeyable(True)


def lockAttrs(object, t, r, s, v):
    """
    Lock or hide transformation attributes of a transform object.

    Parameters:
    object (core.nt.Transform): The selected object.
    t (list of 3 elements): Lock values for each of the translate channels (tx, ty, tz).
    r (list of 3 elements): Lock values for each of the rotate channels (rx, ry, rz).
    s (list of 3 elements): Lock values for each of the scale channels (sx, sy, sz).
    v (int): Lock value for the visibility channel (0, 0.5, 1).

    """
    if not core.objExists(object):
        return

    if not (isinstance(object, core.nt.Transform) or isinstance(object, core.nt.Joint)):
        return  # Skip all nodes except transforms and joints

    translates = ["tx", "ty", "tz"]
    rotates = ["rx", "ry", "rz"]
    scales = ["sx", "sy", "sz"]
    visibility = ["v"]

    listAttrs = ([translates, t], [rotates, r], [scales, s], [visibility, [v]])  # Corresponding attributes and values

    for attrs, vals in listAttrs:  # Iterate over attributes and values
        for attr, val in zip(attrs, vals):
            lockAttr(object.attr(attr), val)


def seamlessSwitchAttr(ctrl, attr):
    """
    Seamlessly changes the specified attribute of the control while preserving the control's world translation and rotation.

    Parameters:
    ctrl (str): The control to be modified.
    attr (str): The attribute to be toggled (e.g., 'follow', 'followNeck').

    """
    if not core.objExists(ctrl + "." + attr):
        core.warning("seamlessSwitchAttr: %s.%s doesn't exist" % (ctrl, attr))
        return
    
    ctrl = core.PyNode(ctrl)
    attr = ctrl.attr(attr)

    mat = core.xform(ctrl, q=True, ws=True, m=True)

    if attr.get() < 0.5:
        attr.set(1)
    else:
        attr.set(0)

    core.xform(ctrl, ws=True, m=mat)


def toggleVisible(obj):
    isHidden = core.hide(obj, tv=True)
    if isHidden == 1: # hidden
        core.showHidden(obj)
    elif isHidden == 2: # visible
        core.hide(obj)
    elif isHidden == 0:
        if core.objectType(obj) == "objectSet":
            v = core.sets(obj, q=True)[0].v.get()
        else:
            v = core.PyNode(obj).v.get()

        if v:
            core.hide(obj)
        else:
            core.showHidden(obj)

def showAllVisibleSets(visibleSets):
    for visibleSet in visibleSets:
        core.showHidden(visibleSet)

def hideAllVisibleSets(visibleSets):
    for visibleSet in visibleSets:
        core.hide(visibleSet)
