import pymel.core as core
import maya.cmds as cmds
from .mirror import *
import maya.mel as mel

class DynamicParent:
    '''
    A class for handling dynamic parenting.
    '''

    def __init__(self, dynGroup, ctrl):
        '''
        Initialize with a group and control.

        Parameters:
        dynGroup (core.nt.Transform) - This group is the top for the dynamic parent (dynGroup -> null -> ctrl)
        ctrl (core.nt.Transform) - The control managed by the dynamic parent
        '''
        self.controlDynGroup = core.PyNode(dynGroup)
        self.control = core.PyNode(ctrl)

    def getParentConstraint(self):
        '''
        Get the parent constraint node.

        Returns:
        core.nt.ParentConstraint - The parent constraint node
        '''
        lst = self.controlDynGroup.listRelatives(type="parentConstraint")
        return lst[0] if lst else None

    def findTarget(self, target):
        '''
        Find the index of the target in the constraint.

        Parameters:
        target (core.nt.Transform) - The target to find

        Returns:
        int - The index of the target or -1 if not found
        '''
        constraint = self.getParentConstraint()
        if not constraint:
            return -1

        targets = constraint.getTargetList()

        return targets.index(target.name()) if target.name() in targets else -1

    def setDrivenKeys(self):
        '''
        Automatically set driven keys for targets by the parent attribute.
        '''
        constraint = self.getParentConstraint()
        if not constraint or not self.control.hasAttr("parent"):
            return

        weights = constraint.getWeightAliasList()
        for i in range(len(weights)):
            attr = weights[i].split(".")[1]

            core.setDrivenKeyframe(constraint, at=attr, v=1, dv=i + 1, cd=self.control.attr("parent"))

            if i < len(weights) - 1:
                core.setDrivenKeyframe(constraint, at=attr, v=0, dv=i + 2, cd=self.control.attr("parent"))

            core.setDrivenKeyframe(constraint, at=attr, v=0, dv=i, cd=self.control.attr("parent"))

    def addTarget(self, target):
        '''
        Add a target to the dynamic parent.

        Parameters:
        target (core.nt.Transform) - The target to add
        '''
        if self.findTarget(target) != -1:  # if the target already exists
            core.displayWarning("addTarget: Target '" + str(target) + "' already exists in this dynamic parent")
            return

        core.undoInfo(openChunk=True)
        if not self.control.hasAttr("parent"):
            self.control.addAttr("parent", at="enum", enumName="(no parent)", k=True)

        if not self.getParentConstraint():  # for the first parent
            pos = core.xform(self.control, q=True, ws=True, t=True)
            rot = core.xform(self.control, q=True, ws=True, ro=True)

            self.controlDynGroup.t.set([0, 0, 0])
            self.controlDynGroup.r.set([0, 0, 0])

            core.xform(self.control, ws=True, t=pos)
            core.xform(self.control, ws=True, ro=rot)

        pc = core.parentConstraint(target, self.controlDynGroup, mo=True)

        id = len(pc.getTargetList()) - 1
        pc.target[id].targetScale.disconnect()

        if core.cycleCheck(pc):  # check for potential cycles
            core.undoInfo(closeChunk=True)
            core.undo()
            core.displayWarning("addTarget: cannot add '" + str(target) + "' as a target because of a cycle")
            return

        pc.restTranslate.set([0, 0, 0])
        pc.restRotate.set([0, 0, 0])

        names = cmds.addAttr(self.control.attr("parent").name(), q=True, en=True)
        names = names + ":" + target.name().replace(":", "_")
        cmds.addAttr(self.control.attr("parent").name(), e=True, en=names)

        self.setDrivenKeys()
        core.undoInfo(closeChunk=True)

    def removeTarget(self, itemToRemove):
        '''
        Remove a target from the dynamic parent.

        Parameters:
        itemToRemove (str) - The item to remove (associated with the scene object)
        '''
        constraint = self.getParentConstraint()
        if not constraint:
            return

        if core.keyframe(self.control.attr("parent"), q=True, n=True):
            core.displayWarning("removeTarget: cannot delete a target because 'parent' attribute has an animation")
            return

        names = cmds.addAttr(self.control.attr("parent").name(), q=True, en=True)  # list of elements in parent

        itemToRemove_ = itemToRemove.replace(":", "_")
        if itemToRemove_ in names.split(":"):
            for w in constraint.getWeightAliasList():
                core.delete(core.keyframe(w, q=True, name=True))  # delete all drivenKeys

            newNames = names.replace(":" + itemToRemove_, "")
            cmds.addAttr(self.control.attr("parent").name(), e=True, en=newNames)

            i = names.split(":").index(itemToRemove_) - 1  # index of the item in the parent attribute
            target = constraint.getTargetList()[i]
            core.parentConstraint(target, self.controlDynGroup, remove=True)

            self.setDrivenKeys()

            return True
        else:
            core.displayWarning("removeTarget: cannot find '" + itemToRemove + "' in '" + self.control.attr("parent").name() + "'")

def hasDynamicParent(control):
    '''
    Check if the object has a dynamic parent system.

    Parameters:
    control (core.nt.Transform) - The control to check

    Returns:
    bool - True if it has a dynamic parent, False otherwise
    '''
    return core.PyNode(control).hasAttr("dynParent")

def dynamicParentSwitch_local(control, parent):
    '''
    Local switch for dynamic parent.

    Parameters:
    control (core.nt.Transform) - The control to switch
    parent (str) - The new parent
    '''
    items = core.attributeQuery("parent", node=control, le=True)[0].split(":")  # get the list of enum attribute values

    parent_ = parent.replace(":", "_")
    index = items.index(parent_) if parent_ in items else 0

    ls = core.ls(sl=True)  # save the list of selected objects

    tmp = core.nt.Transform()
    core.delete(core.parentConstraint(control, tmp))
    control.attr("parent").set(index)  # set the parent

    ctrlTmp = control.duplicate(po=True, rc=True)[0]

    for a in ["tx", "ty", "tz", "rx", "ry", "rz"]:
        ctrlTmp.attr(a).set(l=False, k=True, cb=False)  # unlock transform attributes for parentConstraint

    core.delete(core.parentConstraint(tmp, ctrlTmp))

    safeCopyAttrs(ctrlTmp, control, ["tx", "ty", "tz", "rx", "ry", "rz"])  # copy values

    core.delete([tmp, ctrlTmp])  # delete temporary objects
    core.select(ls)  # restore previous selection

def dynamicParentSwitch(control, parent):
    '''
    Switch dynamic parent.

    Parameters:
    control (str) - The control to switch
    parent (str) - The new parent
    '''
    cmds.undoInfo(openChunk=True)
    try:
        nodes = set(core.ls(sl=True) + [core.PyNode(control)])

        mayaPlayBackSlider = mel.eval('$tmpVar=$gPlayBackSlider')
        selectedRange = cmds.timeControl(mayaPlayBackSlider, q=True, ra=True)

        for control in nodes:
            if not control.hasAttr("parent"):
                continue

            if selectedRange[1] - selectedRange[0] > 1:  # selected keys

                animCurves = control.attr("parent").listConnections(s=True, d=False, type="animCurve")
                for ac in animCurves:  # len(animCurves) = 1
                    for i in range(core.keyframe(ac, q=True, kc=True)):
                        time = core.keyframe(ac, index=i, q=True)
                        if time and time[0] >= selectedRange[0] and time[0] <= selectedRange[1]:
                            core.currentTime(time[0])  # go to specific frame and make kinematic switch
                            dynamicParentSwitch_local(control, parent)

            else:
                dynamicParentSwitch_local(control, parent)
    except:
        pass

    cmds.undoInfo(closeChunk=True)

def dynamicParentBake(control):
    '''
    Bake dynamic parent.

    Parameters:
    control (core.nt.Transform) - The control to bake
    '''
    nodes = set(core.ls(sl=True) + [core.PyNode(control)])

    mayaPlayBackSlider = mel.eval('$tmpVar=$gPlayBackSlider')
    selectedRange = cmds.timeControl(mayaPlayBackSlider, q=True, ra=True)

    animCurves = {}
    for control in nodes:  # make temporary anim curves
        if not control.hasAttr("parent"):
            continue

        animCurves[control] = {}
        for a in ["parent", "tx", "ty", "tz", "rx", "ry", "rz"]:
            curves = control.attr(a).listConnections(s=True, d=False, type="animCurve")
            if curves:
                animCurves[control][a] = curves[0].duplicate()

    for f in range(int(selectedRange[0]), int(selectedRange[1]) + 1):
        core.currentTime(f)

        for control in nodes:
            if not animCurves.get(control):
                continue

            for a in ["parent", "tx", "ty", "tz", "rx", "ry", "rz"]:
                if animCurves[control].get(a):
                    value = core.keyframe(animCurves[control][a], t=(f, f), q=True, eval=True)[0]
                    control.attr(a).set(value)

            if control.attr("parent").get() != 0:
                dynamicParentSwitch_local(control, "")

    for k in animCurves:  # remove temporary curves
        for a in animCurves[k]:
            core.delete(animCurves[k][a])

    core.currentTime(selectedRange[0])

def addSelectedToDynamicParent(control):
    '''
    Add the selected object to the list of dynamic parents.

    Parameters:
    control (str) - The object (control) with the dynamic parent
    '''
    if not hasDynamicParent(control):
        return

    ls = core.ls(sl=True, type="transform")
    if not ls or len(ls) != 1:  # works only for 1 object
        return

    selected = ls[0]

    control = core.PyNode(control)
    if control != selected:  # exit if the object wants to add itself
        dyn = DynamicParent(control.dynParent.get(), control)
        dyn.addTarget(selected)

def deleteFromDynamicParent(control, itemToRemove):
    '''
    Remove an object from the list of dynamic parents.

    Parameters:
    control (str) - The object (control) with the dynamic parent
    itemToRemove (str) - The item to remove (associated with the scene object)
    '''
    nodes = set(core.ls(sl=True) + [core.PyNode(control)])
    for control in nodes:
        if not hasDynamicParent(control):
            continue

        dyn = DynamicParent(control.dynParent.get(), control)

        names = cmds.addAttr(dyn.control.attr("parent").name(), q=True, en=True).split(":")  # list of names in parent

        itemToRemove_ = itemToRemove.replace(":", "_")
        if itemToRemove_ in names:
            dynamicParentSwitch(control.name(), "")  # switch to 'no parent'

            if not dyn.removeTarget(itemToRemove):  # if there was an error removing
                dynamicParentSwitch(control.name(), itemToRemove)  # switch back
        else:
            core.displayWarning("deleteFromDynamicParent: cannot find '" + itemToRemove + "' in '" + dyn.control.attr("parent").name() + "'")
