from datetime import timedelta
from functools import wraps
import json
import pymongo
from flask import Response, request, make_response, Blueprint, jsonify
from flask_api import status
from wizart.fs_ctrl.utils import get_schema
from wizart.fs_ctrl import ldap_wrapper
from wizart.fs_ctrl.auth_utils import Session

SKIP_CMD_ARGS = ["variant"]

simple_page = Blueprint('simple_page', __name__,
                        template_folder='templates')


class FS_Session(Session):
    def __init__(self, session_token=None, expire_delta=timedelta(days=1)):
        from wizart.fs_ctrl.fs_start_service import app
        Session.__init__(self, session_token,
                         mongodb_host=app.config.get('MONGODB_HOST'),
                         mongodb_port=app.config.get('MONGODB_PORT'),
                         database_name=app.config.get('DATABASE_NAME'),
                         expire_delta=expire_delta)


def login_requested(func):
    """Decorator that validate token before running request"""
    @wraps(func)
    def wrapper(*args, **kwargs):
        token = request.cookies.get('session_token')
        session = FS_Session(token)
        if session.get('username', None):
            return func(*args, **kwargs)
        else:
            return make_response("Wrong password or login", status.HTTP_403_FORBIDDEN)
    return wrapper


def has_permissions_on_task(permissions, user_name, user_groups):
    if user_name in permissions:
        return True
    elif any([group in permissions for group in user_groups]):
        return True
    return False


@simple_page.route('/login', methods=['POST'])
def login():
    login = request.json.get('login')
    password = request.json.get('password')

    success, user_object = ldap_wrapper.LdapConnection.auth(login, password)
    if success is True:
        session_token = FS_Session.generate_token(user_object['givenName'])
        session_inst = FS_Session(session_token)
        session_inst["username"] = login
        session_inst["mail"] = user_object["mail"]
        session_inst["fullname"] = user_object["givenName"]
        session_inst["groups"] = ldap_wrapper.ldap_parse_groups(user_object["memberOf"])
        session_inst.flush()
        resp = make_response('Logged in successfully', status.HTTP_200_OK)
        resp.set_cookie('session_token', session_token)
        return resp
    else:
        return make_response("Wrong password or login", status.HTTP_403_FORBIDDEN)


@simple_page.route('/logout', methods=['POST'])
def logout():
    session_token = request.cookies.get('session_token')
    if session_token is None:
        return make_response('Was not logged in', status.HTTP_200_OK)

    session = FS_Session(session_token)
    session.forget()
    return make_response('Logged out successfully', status.HTTP_200_OK)


@simple_page.route('/api/command/<project>/<command_name>', methods=['POST'])
@login_requested
def post_command(project, command_name):
    from wizart.fs_ctrl.fs_start_service import app
    if not app.config.get('PROJECTS').get(project):
        return 'Project with name {} not found'.format(project), status.HTTP_404_NOT_FOUND
    else:
        project_name, project_path = app.config.get('PROJECTS').get(project)

    project_schema = get_schema(project_name, project_path)
    commands = project_schema.load_fs_ctrl_config().get('commands')

    for command in commands:
        if command['name'] == command_name:

            # check permissions. if no permissions specified - command permitted for everyone
            permissions = command.get('permissions')
            if permissions:
                session_token = request.cookies.get('session_token')
                session = FS_Session(session_token)
                permission_granted = has_permissions_on_task(permissions,
                                                             session.get('username'),
                                                             session.get('groups'))
                if not permission_granted:
                    return 'You do not have permissions to this command', status.HTTP_403_FORBIDDEN
            else:
                return 'You do not have permissions to this command', status.HTTP_403_FORBIDDEN

            # check arguments
            if "args" in command:
                for arg in command['args'].keys():
                    if arg not in request.json and arg not in SKIP_CMD_ARGS:
                        return 'Need argument "{}"'.format(arg), status.HTTP_405_METHOD_NOT_ALLOWED

            args = request.json
            if args is None:
                args = {}
            args['command_name'] = command_name
            if "variant" in command['args'].keys() and "variant" not in args:
                args["variant"] = "base"

            if app.config.get('VCS_PATH'):
                args['vcs_config'] = app.config.get('VCS_PATH')

            import wizart.fs_ctrl.celery_tasks
            if not app.config.get('DEBUG'):
                async_res = wizart.fs_ctrl.celery_tasks.run_task.delay(
                    project_name, project_path, args)
                return Response(json.dumps({"id": async_res.id}),
                                status=status.HTTP_200_OK, 
                                headers={"Content-type": "application/json"})
            else:
                wizart.fs_ctrl.celery_tasks.run_task(project_name, project_path, args)
                return Response('WARNING: need to start celery. Command run successfully.', status=status.HTTP_200_OK)
    return 'Wrong command for this project', status.HTTP_405_METHOD_NOT_ALLOWED


@simple_page.route('/api/task_info/<string:celery_task_id>', methods=['GET'])
@login_requested
def task_info(celery_task_id):
    from wizart.fs_ctrl.fs_start_service import app
    with pymongo.MongoClient(app.config.get('MONGODB_HOST'), app.config.get('MONGODB_PORT')) as client:
        celery_db = client["celery"]
        task_data = celery_db.celery_taskmeta.find_one({"_id": celery_task_id})

        if task_data is None:
            return make_response("Task not found", status.HTTP_404_NOT_FOUND)

    return jsonify(task_data)
