import ldap

__SERVER__ = "burut.org"

__BIND_USER__ = "cn=reader,ou=ServiceUsers,ou=AllUsers,dc=burut,dc=org"

__BIND_PWD__ = "readerpass"

__BIND_BASE_DN__ = "ou=AllUsers,dc=burut,dc=org"


import re


def ldap_parse_groups(raw_list):
    result = []
    for group_str in raw_list:
        group_str_list = group_str.decode().split(",")
        #CN is first
        match = re.search('CN=(\w+)', group_str_list[0])
        if match is None:
            raise(RuntimeError, "Cant parse LDAP group name '%s' " % group_str_list[0])
        result.append( match.group(1) )
    return result


class LdapConnection(object):
    def __init__(self, bind_user = __BIND_USER__, bind_password = __BIND_PWD__):
        self._connection  = None
        self.bind_user = bind_user
        self.bind_password = bind_password
    
    def __enter__(self):
        self.connect()
        return self 
    
    def __exit__(self, exc_type, exc_val, exc_tb):
        self.disconnect()

    def connect(self):
        self._connection = ldap.initialize("ldap://%s" % __SERVER__ )
        self._connection.set_option(ldap.OPT_REFERRALS, 0)
        self._connection.simple_bind_s(self.bind_user, self.bind_password)

    def get_user_by_login(self, login):
        result = self._connection.search_s(__BIND_BASE_DN__, ldap.SCOPE_SUBTREE, '(sAMAccountName=%s)' % login, ['givenName', 'memberOf','mail'])
        #print result
        dn, attr = result[0] if (result is not None and len(result)) else (__BIND_BASE_DN__, dict())
        
        return {"dn": dn, "givenName": attr.get("givenName",[None])[0] , "memberOf": attr.get("memberOf", []), "mail":  attr.get('mail',[None])[0]}
        
    @staticmethod
    def auth(login, password):
        with LdapConnection() as ldap_info:
            user_obj = ldap_info.get_user_by_login(login)

            auth_success = True

            auth_ldap = LdapConnection(user_obj['dn'],  password)
            try:
                auth_ldap.connect()
            except ldap.LDAPError:
                auth_success = False
            finally:
                auth_ldap.disconnect()
        
            return auth_success, user_obj


    def disconnect(self):
        self._connection.unbind_s()



if __name__ == "__main__":
    # with LdapConnection() as ldap_info:
    #     print ldap_info.get_user_by_login("xeash")
    auth_status, user_data = LdapConnection.auth('xeash', "bla")
    print(ldap_parse_groups( user_data['memberOf']))