import pymongo
import hashlib
from datetime import datetime
from datetime import timedelta

class Session(object):
    def __init__(self, session_token = None, mongodb_host = 'localhost', mongodb_port = 27017, database_name = 'sessions', expire_delta = timedelta(days=1)):
        assert( isinstance(expire_delta, timedelta) )
        self.expire_delta = expire_delta       
        self.database = pymongo.MongoClient(mongodb_host, mongodb_port)[database_name]
        self.session_token = session_token
        if session_token is not None:
            self.session_data = self.database.sessions.find_one({'token' : session_token}, projection = {'data': 1})
            if self.session_data is None:
                self.session_data = dict()
            else:
                self.session_data = self.session_data['data']
        else:
            self.session_data = dict()
        self.database.sessions.ensure_index([("expire_at", pymongo.ASCENDING)], expireAfterSeconds = 0, background = True)
    
    def __setitem__(self, key, value):
        self.session_data[key] = value

    def __getitem__(self, key):
        return self.session_data[key]

    def __delitem__(self, key):
        del self.session_data[key]

    def __len__(self):
        return len(self.session_data)

    def __contains__(self, key):
        return key in self.session_data

    def __iter__(self):
        return self.session_data.items()
    
    def get(self, key,  default_value = None):
        return self.session_data.get(key, default_value)

    def flush(self):
        if self.session_token is None:
            raise AttributeError('Session token is None')

        expire_date = datetime.utcnow() + self.expire_delta
        update_data = {'data' : self.session_data, 'token' : self.session_token, 'expire_at': expire_date}
        self.database.sessions.update_one({'token' : self.session_token}, {"$set": update_data}  , upsert = True)

    def forget(self):
        if self.session_token is None:
            raise AttributeError('Session token is None')		

        self.database.sessions.remove({'token' : self.session_token})

    @staticmethod
    def generate_token(username):
        hasher = hashlib.sha224()
        hasher.update(username)
        hasher.update(str(datetime.utcnow()).encode())
        return hasher.hexdigest()