import os
import json
import requests

default_path = os.path.abspath(r"../../config.json")
config_path = os.environ.get('FS_SERVICE_CONFIG', default_path)


class AssetVcsApiError(Exception):
    pass


class AssetVcsAPI(object):

    def __init__(self):
        self.fs_config = None
        self.__load_fs_config()

        self.default_headers = {
            "app_token": self.fs_config["ASSETVCS_APP_TOKEN"]
        }
        self.assetvcs_host = self.fs_config["ASSETVCS_HOST"]

    def __load_fs_config(self):
        try:
            with open(config_path) as config_file:
                self.fs_config = json.loads(config_file.read())
        except IOError:
            raise AssetVcsApiError(
                "Assetvcs: Can`t find config.json, path: {}".format(config_path))

    def __post(self, **kwargs):
        request_params = dict(**kwargs)
        request_params["headers"] = self.default_headers
        response = requests.post(**request_params)
        return response

    def create_asset(self, data):
        response = self.__post(
            url="{host}/asset".format(host=self.assetvcs_host), data=data)
        if response.status_code != requests.codes.created:
            raise AssetVcsApiError("Assetvcs: create asset error - '{}', status code: {}, reason: {}".format(
                data, response.status_code, response.reason))
        return response.json()

    def create_version(self, asset_id):
        response = self.__post(url="{host}/asset/{asset_id}/version".format(
            host=self.assetvcs_host, asset_id=asset_id))
        if response.status_code != requests.codes.ok:
            raise AssetVcsApiError("Assetvcs: create version error, status code: {}, reason: {}".format(
                response.status_code, response.reason))
        return response.json()

    def upload_version_files(self, asset_id, version_num, version_files):
        response = self.__post(url="{host}/asset/{asset_id}/version/{version_num}/upload".format(
            host=self.assetvcs_host, asset_id=asset_id, version_num=version_num), files=version_files)
        if response.status_code != requests.codes.created:
            raise AssetVcsApiError("Assetvcs: can`t upload version files, status code: {}, reason: {}".format(
                response.status_code, response.reason))

    def get_asset_versions(self, asset_id):
        response = requests.get("{host}/asset/{asset_id}/version".format(
            host=self.fs_config["ASSETVCS_HOST"], asset_id=asset_id))
        if response.status_code != requests.codes.ok:
            raise AssetVcsApiError("Assetvcs: can`t get versions list, status code: {}, reason: {}".format(
                response.status_code, response.reason))
        return response.json()

    def publish_version(self, asset_id, version_num, data):
        response = self.__post(url="{host}/asset/{asset_id}/version/{version_num}/publish".format(
            host=self.assetvcs_host, asset_id=asset_id, version_num=version_num), data=data)
        if response.status_code != requests.codes.ok:
            raise AssetVcsApiError("Assetvcs: can`t publish version, status code: {}, reason: {}".format(
                response.status_code, response.reason))
        return response.status_code
